{*******************************************************}
{                                                       }
{   Copyright (c) 1997,1998 Classic Software            }
{   All Rights Reserved                                 }
{                                                       }
{*******************************************************}

unit CSUpDown;

{$B-,P+,W-,X+}

interface

uses
  SysUtils, WinTypes, WinProcs, Messages, Classes, Graphics, Controls,
  Forms, Dialogs, Buttons, ExtCtrls;

type
  TcsArrowButtonStyle = (bsWin31, bsWin95);
  TcsArrowButtonDirection = (bdLeft, bdRight, bdUp, bdDown);
  TcsUpDownOrientation = (udHorizontal, udVertical);

  { TcsArrowButton is a special purpose speed button for use as the scroller
    for tab/page controls.  An up, down, left or right pointing triangle is
    painted on the button's face and the button auto-repeats (clicks) when
    held down.
    TcsArrowButton is currently not intended for general purpose use (and
    may not be supported outside of its use within tab/page controls) in that
    it ensures that the button's Width and Height are the same extent so that
    the button's arrow graphic is correctly sized and placed.
    The Timer used for the auto-repeat clicks is only created when a button is
    pressed and is destroyed as soon as the button is released.  This ensures
    that the minimum no. of timers are used at any one time on Win 3.1 systems
    which only allow 32 active timers.
  }
  TcsArrowButton = class(TSpeedButton)
  private
    FArrowColor: TColor;
    FBorderColor: TColor;
    FDirection: TcsArrowButtonDirection;
    FFaceColor: TColor;
    FHighlightColor: TColor;
    FShadowColor: TColor;
    FStyle: TcsArrowButtonStyle;
    FTimer: TTimer;
    procedure CreateTimer;
    procedure DestroyTimer;
    procedure DrawButton;
    procedure SetArrowColor(Value: TColor);
    procedure SetBorderColor(Value: TColor);
    procedure SetDirection(Value: TcsArrowButtonDirection);
    procedure SetFaceColor(Value: TColor);
    procedure SetHighlightColor(Value: TColor);
    procedure SetShadowColor(Value: TColor);
    procedure SetStyle(Value: TcsArrowButtonStyle);
    procedure TimerExpired(Sender: TObject);
  protected
    procedure MouseDown(Button: TMouseButton; Shift: TShiftState;
      X, Y: Integer); override;
    procedure MouseUp(Button: TMouseButton; Shift: TShiftState;
      X, Y: Integer); override;
    procedure Paint; override;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
  published
    property ArrowColor: TColor read FArrowColor write SetArrowColor default clBtnText;
    property BorderColor: TColor read FBorderColor write SetBorderColor default clWindowFrame;
    property Direction: TcsArrowButtonDirection read FDirection write SetDirection default bdLeft;
    property FaceColor: TColor read FFaceColor write SetFaceColor default clBtnFace;
    property HighlightColor: TColor read FHighlightColor write SetHighlightColor default clBtnHighlight;
    property ShadowColor: TColor read FShadowColor write SetShadowColor default clBtnShadow;
    property Style: TcsArrowButtonStyle read FStyle write SetStyle default bsWin95;
  end;

  TcsUpDown = class(TWinControl)
  private
    FArrowColor: TColor;
    FBorderColor: TColor;
    FDecBtn: TcsArrowButton;
    FFaceColor: TColor;
    FHighlightColor: TColor;
    FIncBtn: TcsArrowButton;
    FOrientation: TcsUpDownOrientation;
    FShadowColor: TColor;
    FStyle: TcsArrowButtonStyle;
    FOnClick: TNotifyEvent;
    procedure AdjustButtonSize (var W: Integer; var H: Integer);
    procedure CreateButtons;
    procedure DoClick(Sender: TObject);
    procedure SetArrowColor(Value: TColor);
    procedure SetBorderColor(Value: TColor);
    procedure SetFaceColor(Value: TColor);
    procedure SetHighlightColor(Value: TColor);
    procedure SetOrientation(Value: TcsUpDownOrientation);
    procedure SetShadowColor(Value: TColor);
    procedure SetStyle(Value: TcsArrowButtonStyle);
    procedure CMParentCtl3DChanged(var Message: TMessage); message CM_PARENTCTL3DCHANGED;
    procedure WMEraseBkgnd(var Message: TWMEraseBkgnd); message WM_ERASEBKGND;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    procedure SetBounds(ALeft, ATop, AWidth, AHeight: Integer); override;
  published
    property ArrowColor: TColor read FArrowColor write SetArrowColor default clBtnText;
    property BorderColor: TColor read FBorderColor write SetBorderColor default clWindowFrame;
    property Ctl3D;
    property FaceColor: TColor read FFaceColor write SetFaceColor default clBtnFace;
    property HighlightColor: TColor read FHighlightColor write SetHighlightColor default clBtnHighlight;
    property Orientation: TcsUpDownOrientation read FOrientation
      write SetOrientation default udHorizontal;
    property ShadowColor: TColor read FShadowColor write SetShadowColor default clBtnShadow;
    property Style: TcsArrowButtonStyle read FStyle write SetStyle default bsWin31;
    property OnClick: TNotifyEvent read FOnClick write FOnClick;
  end;

implementation

const
  InitRepeatPause = 400;  { pause before repeat timer kicks in (ms) }
  RepeatPause     = 100;  { pause before button auto-repeats (ms)}

type
  TWinControlProxy = class(TWinControl);

{ TcsUpDown }

constructor TcsUpDown.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FArrowColor := clBtnText;
  FBorderColor := clWindowFrame;
  FFaceColor := clBtnFace;
  FHighlightColor := clBtnHighlight;
  FOrientation := udHorizontal;
  FShadowColor := clBtnShadow;
  FStyle := bsWin95;
  CreateButtons;
  Height := 20;
  Width := 40;
end;

destructor TcsUpDown.Destroy;
begin
  FDecBtn.Free;
  FIncBtn.Free;
  inherited Destroy;
end;

procedure TcsUpDown.AdjustButtonSize(var W: Integer; var H: Integer);
var
  Overlap: Integer;
  BtnW, BtnH: Integer;
begin
  Overlap := 0;
  if FOrientation = udHorizontal then
  begin
    if (FStyle = bsWin31) then { allow for overlap of border between buttons }
    begin
      if W mod 2 = 0 then { width is even }
        Dec(W); { reduce width to allow for overlap of common border }
      Overlap := 1;
    end
    else { bsWin95 }
      if W mod 2 <> 0 then { width is odd }
        Dec(W); { so there won't be any remaining extra space }
    BtnW := (W + Overlap) div 2;
    BtnH := BtnW;
    H := BtnW;
  end
  else { udVertical }
  begin
    if (FStyle = bsWin31) then { allow for overlap of border between buttons }
    begin
      if H mod 2 = 0 then { height is even }
        Dec(H); { reduce width to allow for overlap of common border }
      Overlap := 1;
    end
    else { bsWin95 }
      if H mod 2 <> 0 then { height is odd }
        Dec(H); { so there won't be any remaining extra space }
    BtnH := (H + Overlap) div 2;
    BtnW := BtnH;
    W := BtnH;
  end;
  if FOrientation = udHorizontal then
  begin
    FDecBtn.SetBounds(0, 0, BtnW, BtnH);
    FIncBtn.SetBounds(FDecBtn.Width - Overlap, 0, BtnW, BtnH)
  end
  else
  begin
    FDecBtn.SetBounds(0, 0, BtnW, BtnH);
    FIncBtn.SetBounds(0, FDecBtn.Height - Overlap, BtnW, BtnH);
  end;
end;

procedure TcsUpDown.CreateButtons;
var
  Btns: Array[0..1] of TcsArrowButton;
  I, L, T, Extent: Integer;
begin
  FDecBtn := TcsArrowButton.Create(Self);
  FIncBtn := TcsArrowButton.Create(Self);
  if FOrientation = udHorizontal then
  begin
    FDecBtn.Direction := bdLeft;
    FIncBtn.Direction := bdRight;
    Extent := Height;
  end
  else
  begin
    FDecBtn.Direction := bdUp;
    FIncBtn.Direction := bdDown;
    Extent := Width;
  end;
  Btns[0] := FDecBtn;
  Btns[1] := FIncBtn;
  L := Left;
  T := Top;
  for I := Low(Btns) to High(Btns) do
  begin
    { size and position each button within self (UpDown) }
    Btns[I].SetBounds(L, T, Extent, Extent);
    Btns[I].Parent := Self;
    Btns[I].ArrowColor := ArrowColor;
    Btns[I].BorderColor := BorderColor;
    Btns[I].FaceColor := FaceColor;
    Btns[I].HighlightColor := HighlightColor;
    Btns[I].ShadowColor := ShadowColor;
    Btns[I].OnClick := DoClick;
    if FOrientation = udHorizontal then
      Inc(L, Extent)
    else
      Inc(T, Extent);
  end;
end;

{ Pass the button which was pressed (instead of Self) to the OnClick handler. }  
procedure TcsUpDown.DoClick(Sender: TObject);
begin
  if Assigned(FOnClick) then FOnClick(Sender);
end;

procedure TcsUpDown.SetArrowColor(Value: TColor);
begin
  if FArrowColor <> Value then
  begin
    FArrowColor := Value;
    FDecBtn.ArrowColor := FArrowColor;
    FIncBtn.ArrowColor := FArrowColor;
  end;
end;

procedure TcsUpDown.SetBorderColor(Value: TColor);
begin
  if FBorderColor <> Value then
  begin
    FBorderColor := Value;
    FDecBtn.BorderColor := FBorderColor;
    FIncBtn.BorderColor := FBorderColor;
  end;
end;

procedure TcsUpDown.SetBounds(ALeft, ATop, AWidth, AHeight: Integer);
var
  W, H: Integer;
begin
  W := AWidth;
  H := AHeight;
  AdjustButtonSize(W, H);
  inherited SetBounds(ALeft, ATop, W, H);
end;

procedure TcsUpDown.SetFaceColor(Value: TColor);
begin
  if FFaceColor <> Value then
  begin
    FFaceColor := Value;
    FDecBtn.FaceColor := FFaceColor;
    FIncBtn.FaceColor := FFaceColor;
  end;
end;

procedure TcsUpDown.SetHighlightColor(Value: TColor);
begin
  if FHighlightColor <> Value then
  begin
    FHighlightColor := Value;
    FDecBtn.HighlightColor := FHighlightColor;
    FIncBtn.HighlightColor := FHighlightColor;
  end;
end;

procedure TcsUpDown.SetOrientation(Value: TcsUpDownOrientation);
begin
  if FOrientation <> Value then
  begin
    FOrientation := Value;
    if FOrientation = udHorizontal then
    begin
      FDecBtn.Direction := bdLeft;
      FIncBtn.Direction := bdRight;
    end
    else
    begin
      FDecBtn.Direction := bdUp;
      FIncBtn.Direction := bdDown;
    end;
    { switch dimensions so each button's Width and Height remains equal }
    SetBounds(Left, Top, Height, Width); { Note Height and Width are switched }
  end;
end;

procedure TcsUpDown.SetShadowColor(Value: TColor);
begin
  if FShadowColor <> Value then
  begin
    FShadowColor := Value;
    FDecBtn.ShadowColor := FShadowColor;
    FIncBtn.ShadowColor := FShadowColor;
  end;
end;

procedure TcsUpDown.SetStyle(Value: TcsArrowButtonStyle);
begin
  if FStyle <> Value then
  begin
    FStyle := Value;
    FDecBtn.Style := FStyle;
    FIncBtn.Style := FStyle;
  end;
end;

procedure TcsUpDown.CMParentCtl3DChanged(var Message: TMessage);
begin
  Invalidate;
  inherited;
end;

{ Entire control is covered by the two buttons so prevent the background being
  painted in the parent's brush color.
}
procedure TcsUpDown.WMEraseBkgnd(var Message: TWMEraseBkgnd);
begin
  { let Paint take care of everything }
  Message.Result := 1;
end;

{ TcsArrowButton }

constructor TcsArrowButton.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FArrowColor := clBtnText;
  FBorderColor := clWindowFrame;
  FDirection := bdLeft;
  FFaceColor := clBtnFace;
  FHighlightColor := clBtnHighlight;
  FShadowColor := clBtnShadow;
  FStyle := bsWin95;
end;

destructor TcsArrowButton.Destroy;
begin
  DestroyTimer;
  inherited Destroy;
end;

procedure TcsArrowButton.CreateTimer;
begin
  if FTimer = nil then
    FTimer := TTimer.Create(Self);
  FTimer.OnTimer := TimerExpired;
  FTimer.Interval := InitRepeatPause;
  FTimer.Enabled  := True;
end;

procedure TcsArrowButton.DestroyTimer;
begin
  if FTimer <> nil then
  begin
    FTimer.Enabled := False;
    FTimer.Free;
    FTimer := nil;
  end;
end;

{ DrawButton draws the button in its current state (up/down) and style. }
procedure TcsArrowButton.DrawButton;

  { TcsArrowButton is a TGraphicControl and thus doesn't receive any messages,
    (i.e. WM_ERASEBKGND) and so erasure of background is done here.
  }
  procedure DrawBackground;
  begin
    with Canvas do
    begin
      Brush.Color := FFaceColor;
      FillRect(ClientRect);
    end;
  end;

  procedure DrawFrame;
  begin
    with Canvas do
    begin
      if FStyle = bsWin95 then
      begin
        if FState = bsUp then
        begin
          Pen.Color := FHighlightColor;
          MoveTo(1, Height - 2);
          LineTo(1, 1);
          LineTo(Width - 1, 1);
          Pen.Color := FBorderColor;
          MoveTo(Width - 1, 0);
          LineTo(Width - 1, Height - 1);
          LineTo(-1, Height - 1);
          Pen.Color := FShadowColor;
          MoveTo(Width - 2, 1);
          LineTo(Width - 2, Height - 2);
          LineTo(0, Height - 2);
        end
        else
        begin
          Pen.Color := FShadowColor;
          MoveTo(0, Height - 1);
          LineTo(0, 0);
          LineTo(Width, 0);
          Pen.Color := FBorderColor;
          MoveTo(1, Height - 3);
          LineTo(1, 1);
          LineTo(Width - 2, 1);
          Pen.Color := FHighlightColor;
          MoveTo(Width - 1, 0);
          LineTo(Width - 1, Height - 1);
          LineTo(-1, Height - 1);
        end;
      end
      else { bsWin31 }
      begin
        { draw the border }
        Pen.Color := FBorderColor;
        MoveTo(0, 0);
        LineTo(Width - 1, 0);
        LineTo(Width - 1, Height - 1);
        LineTo(0, Height - 1);
        LineTo(0, 0);
        if (Parent <> nil) and TWinControlProxy(Parent).Ctl3D then { make 3D }
        begin
          if FState = bsUp then
          begin
            Pen.Color := FHighlightColor;
            MoveTo(Width - 2, 1);
            LineTo(1, 1);
            LineTo(1, Height - 2);
            Pen.Color := FShadowColor;
            LineTo(Width - 2, Height - 2);
            LineTo(Width - 2, 1);
          end
          else
          begin
            Pen.Color := FShadowColor;
            MoveTo(1, Height - 2);
            LineTo(1, 1);
            LineTo(Width - 1, 1);
          end;
        end;
      end;
    end;
  end;

  { Draw the arrow on the button's face. }
  procedure DrawArrow;
  var
    Extent, OneThird, OneQuarter, OneHalf, Offset: Integer;
  begin
    with Canvas do
    begin
      Pen.Color := FArrowColor;
      Brush.Color := FArrowColor;
      { Note: Assumes button's Width and Height are the same extent (square) }
      Extent := Width;
      OneThird := Extent div 3;
      OneQuarter := Extent div 4;
      OneHalf := OneQuarter*2;
      if FState = bsDown then
        Offset := 1
      else
        Offset := 0;
      case FDirection of
        bdLeft:
          Polygon([Point(OneThird + Offset, OneHalf + Offset),
                   Point(OneThird + OneQuarter + Offset, OneHalf - OneQuarter + Offset),
                   Point(OneThird + OneQuarter + Offset, OneHalf + OneQuarter + Offset)]);
        bdRight:
          Polygon([Point(Extent - 1 - OneThird + Offset, OneHalf + Offset),
                   Point(Extent - 1 - OneThird - OneQuarter + Offset, OneHalf - OneQuarter + Offset),
                   Point(Extent - 1 - OneThird - OneQuarter + Offset, OneHalf + OneQuarter + Offset)]);
        bdUp:
          Polygon([Point(OneHalf + Offset, OneThird + Offset),
                   Point(OneHalf - OneQuarter + Offset, OneThird + OneQuarter + Offset),
                   Point(OneHalf + OneQuarter + Offset, OneThird + OneQuarter + Offset)]);
        bdDown:
          Polygon([Point(OneHalf + Offset, Extent - 1 - OneThird + Offset),
                   Point(OneHalf - OneQuarter + Offset, Extent - 1 - OneThird - OneQuarter + Offset),
                   Point(OneHalf + OneQuarter + Offset, Extent - 1 - OneThird - OneQuarter + Offset)]);
      end;
    end;
  end;

begin { DrawButton }
  DrawBackground;
  DrawFrame;
  DrawArrow;
end;

procedure TcsArrowButton.MouseDown(Button: TMouseButton; Shift: TShiftState;
  X, Y: Integer);
begin
  inherited MouseDown (Button, Shift, X, Y);
  CreateTimer;
end;

procedure TcsArrowButton.MouseUp(Button: TMouseButton; Shift: TShiftState;
                                  X, Y: Integer);
begin
  inherited MouseUp (Button, Shift, X, Y);
  DestroyTimer;
end;

procedure TcsArrowButton.Paint;
begin
  DrawButton;
end;

procedure TcsArrowButton.SetArrowColor(Value: TColor);
begin
  if FArrowColor <> Value then
  begin
    FArrowColor := Value;
    Invalidate;
  end;
end;

procedure TcsArrowButton.SetBorderColor(Value: TColor);
begin
  if FBorderColor <> Value then
  begin
    FBorderColor := Value;
    Invalidate;
  end;
end;

procedure TcsArrowButton.SetDirection(Value: TcsArrowButtonDirection);
begin
  if FDirection <> Value then
  begin
    FDirection := Value;
    Invalidate;
  end;
end;

procedure TcsArrowButton.SetFaceColor(Value: TColor);
begin
  if FFaceColor <> Value then
  begin
    FFaceColor := Value;
    Invalidate;
  end;
end;

procedure TcsArrowButton.SetHighlightColor(Value: TColor);
begin
  if FHighlightColor <> Value then
  begin
    FHighlightColor := Value;
    Invalidate;
  end;
end;

procedure TcsArrowButton.SetShadowColor(Value: TColor);
begin
  if FShadowColor <> Value then
  begin
    FShadowColor := Value;
    Invalidate;
  end;
end;

procedure TcsArrowButton.SetStyle(Value: TcsArrowButtonStyle);
begin
  if FStyle <> Value then
  begin
    FStyle := Value;
    Invalidate;
  end;
end;

procedure TcsArrowButton.TimerExpired(Sender: TObject);
begin
  FTimer.Interval := RepeatPause;
  if (FState = bsDown) and MouseCapture then
  begin
    try
      Click;
    except
      DestroyTimer;
      raise;
    end;
  end;
end;

end.
