{*******************************************************}
{                                                       }
{   Copyright (c) 1997 Classic Software                 }
{   All Rights Reserved                                 }
{                                                       }
{*******************************************************}

unit CSTCTExC; { Tab Control - Text Extent Cache classes }

{$B-,P+,W-,X+}

interface

uses
  SysUtils, WinTypes, WinProcs, Messages, Classes, Controls;

type
  { TcsTextExtentCache implements a cache of the text extents for strings.
    If the font on which the stored text extents are based is changed then
    it is up to the caller to clear the cache.
    It is assumed that the cache only contains extents for the same font style,
    e.g. all Normal or all Bold but not a mixture.
  }
  TcsTextExtentCache = class(TObject)
  private
    FCache: TStringList;
  public
    constructor Create;
    destructor Destroy; override;
    { Add adds a String->Extent association to the cache. }
    procedure Add(const AString: String; Extent: TSize);
    procedure Clear; { entire cache }
    function Find(const AString: String; var Extent: TSize): Boolean;
  end;

implementation

{ ----------------------------- }
{ TcsTextExtentCacheItem        }
{ ----------------------------- }

type
  { The undelying size of TSize is different under different Delphi versions
    (4 bytes in Delphi 1, 8 bytes in Delphi 2/3) so we can't just store the
    extent in the object pointer of each FCache item.
  }
  TcsTextExtentCacheItem = class(TObject)
  private
    FExtent: TSize;
  public
    constructor Create(Extent: TSize);
    property Extent: TSize read FExtent write FExtent;
  end;

constructor TcsTextExtentCacheItem.Create(Extent: TSize);
begin
  inherited Create;
  FExtent := Extent;
end;

{ ----------------------------- }
{ TcsTextExtentCache            }
{ ----------------------------- }

constructor TcsTextExtentCache.Create;
begin
  inherited Create;
  FCache := TStringList.Create;
end;

destructor TcsTextExtentCache.Destroy;
begin
  Clear;
  FCache.Free;
  inherited Destroy;
end;

procedure TcsTextExtentCache.Add(const AString: String; Extent: TSize);
begin
  if FCache.IndexOf(AString) = -1 then
    FCache.AddObject(AString, TcsTextExtentCacheItem.Create(Extent));
end;

procedure TcsTextExtentCache.Clear;
var
  I: Integer;
begin
  for I := 0 to FCache.Count - 1 do
    TcsTextExtentCacheItem(FCache.Objects[I]).Free;
  FCache.Clear;
end;

function TcsTextExtentCache.Find(const AString: String; var Extent: TSize): Boolean;
var
  Idx: Integer;
begin
  Result := FCache.Find(AString, Idx);
  if Result then
    Extent := TcsTextExtentCacheItem(FCache.Objects[Idx]).Extent;
end;

end.
