{*******************************************************}
{                                                       }
{   Copyright (c) 1997, 1999 Classic Software           }
{   All Rights Reserved                                 }
{                                                       }
{*******************************************************}

unit CSTCSubF;

{$B-,P+,W-,X+}

interface

uses
  SysUtils, WinTypes, WinProcs, Messages, Classes, Graphics, Controls,
  Forms, Dialogs, CSTCBase, Grids, ExtCtrls, StdCtrls, Buttons, DsgnIntf;

type
  TcsTabEditorSubForm = class(TForm)
    HeaderColumns: THeader;
    GridTabs: TDrawGrid;
    BtnAdd: TButton;
    BtnDelete: TButton;
    LabelCaption: TLabel;
    Label2: TLabel;
    EditNumGlyphs: TEdit;
    LabelNumGlyphs: TLabel;
    LabelColor: TLabel;
    BtnColor: TButton;
    CheckEnabled: TCheckBox;
    CheckVisible: TCheckBox;
    BtnMoveUp: TButton;
    BtnMoveDown: TButton;
    LabelEnabled: TLabel;
    LabelVisible: TLabel;
    Panel1: TPanel;
    ImageBitmap: TImage;
    Bevel2: TBevel;
    MemoCaption: TMemo;
    BtnNext: TButton;
    BtnPrevious: TButton;
    MemoHint: TMemo;
    Label1: TLabel;
    procedure HeaderColumnsSized(Sender: TObject; ASection,
      AWidth: Integer);
    procedure LabelCaptionClick(Sender: TObject);
    procedure GridTabsDblClick(Sender: TObject);
    procedure LabelEnabledClick(Sender: TObject);
    procedure LabelVisibleClick(Sender: TObject);
    procedure MemoCaptionEnter(Sender: TObject);
    procedure GridTabsDrawCell(Sender: TObject; ACol, ARow: Longint;
      ARect: TRect; AState: TGridDrawState);
    procedure FormShow(Sender: TObject);
    procedure EditExit(Sender: TObject);
    procedure BtnAddClick(Sender: TObject);
    procedure GridTabsClick(Sender: TObject);
    procedure BtnColorClick(Sender: TObject);
    procedure BtnDeleteClick(Sender: TObject);
    procedure LabelColorClick(Sender: TObject);
    procedure LabelNumGlyphsClick(Sender: TObject);
    procedure BtnNextClick(Sender: TObject);
    procedure BtnPreviousClick(Sender: TObject);
    procedure BtnMoveClick(Sender: TObject);
    procedure ImageBitmapClick(Sender: TObject);
    procedure CheckClick(Sender: TObject);
  private
    { Private declarations }
    FBitmapPropertyEditor: TPropertyEditor;
{$IFDEF VER130}
    FFormDesigner: IFormDesigner;
{$ELSE}
{$IFDEF VER120}
    FFormDesigner: IFormDesigner;
{$ELSE}
    FFormDesigner: TFormDesigner;
{$ENDIF}
{$ENDIF}
    FInFetch: Boolean;    
    FTabDataList: TcsTabDataList;
    procedure AddTab;
    procedure DeleteTab;
    procedure Fetch;
    procedure GetPropertyEditor(Editor: TPropertyEditor);
    procedure SetColumnWidths;
    procedure SetStatus;
    procedure SetTabDataList(Value: TcsTabDataList);
    procedure Store;
  public
    { Public declarations }
{$IFDEF VER130}
    property FormDesigner: IFormDesigner read FFormDesigner write FFormDesigner;
{$ELSE}
{$IFDEF VER120}
    property FormDesigner: IFormDesigner read FFormDesigner write FFormDesigner;
{$ELSE}
    property FormDesigner: TFormDesigner read FFormDesigner write FFormDesigner;
{$ENDIF}
{$ENDIF}    
    property TabDataList: TcsTabDataList read FTabDataList write SetTabDataList;
  end;

implementation

uses TypInfo;

{$R *.DFM}

procedure TcsTabEditorSubForm.FormShow(Sender: TObject);
begin
  SetColumnWidths;
  SetStatus;
  Fetch;
end;

{ Set widths of GridTabs columns to the width of the corresponding sections in
  HeaderColumns.
}
procedure TcsTabEditorSubForm.SetColumnWidths;
var
  I: Integer;
begin
  for I := GridTabs.FixedCols to GridTabs.ColCount - 1 do
    HeaderColumnsSized(Self, I - GridTabs.FixedCols,
      HeaderColumns.SectionWidth[I - GridTabs.FixedCols]);
end;

procedure TcsTabEditorSubForm.SetTabDataList(Value: TcsTabDataList);
begin
  FTabDataList := Value;
  GridTabs.RowCount := TabDataList.Count + GridTabs.FixedRows;
  GridTabs.Row := GridTabs.FixedRows;
  GridTabs.Invalidate;
  SetStatus;
  Fetch;
end;

procedure TcsTabEditorSubForm.HeaderColumnsSized(Sender: TObject; ASection,
  AWidth: Integer);
begin
  GridTabs.ColWidths[ASection] := AWidth - GridTabs.GridLineWidth;
end;

procedure TcsTabEditorSubForm.LabelCaptionClick(Sender: TObject);
begin
  if LabelCaption.FocusControl <> nil then
    LabelCaption.FocusControl.SetFocus;
end;

procedure TcsTabEditorSubForm.GridTabsDblClick(Sender: TObject);
begin
  case GridTabs.Col of
    0:  begin
          MemoCaption.SelectAll;
          MemoCaption.SetFocus;
        end;
    1:  ImageBitmapClick(Self);
    2:  EditNumGlyphs.SetFocus;
    3:  BtnColorClick(Self);
    4:  CheckEnabled.Checked := not CheckEnabled.Checked;
    5:  CheckVisible.Checked := not CheckVisible.Checked;
  end;
end;

procedure TcsTabEditorSubForm.LabelEnabledClick(Sender: TObject);
begin
  if (LabelEnabled.FocusControl <> nil) and
    (LabelEnabled.FocusControl is TCheckBox) then
    with LabelEnabled.FocusControl as TCheckBox do
      Checked := not Checked;
end;

procedure TcsTabEditorSubForm.LabelVisibleClick(Sender: TObject);
begin
  if (LabelVisible.FocusControl <> nil) and
    (LabelVisible.FocusControl is TCheckBox) then
    with LabelVisible.FocusControl as TCheckBox do
      Checked := not Checked;
end;

procedure TcsTabEditorSubForm.MemoCaptionEnter(Sender: TObject);
begin
  with Sender as TMemo do
    SelectAll;
end;

procedure TcsTabEditorSubForm.GridTabsDrawCell(Sender: TObject; ACol,
  ARow: Longint; ARect: TRect; AState: TGridDrawState);
var
  TextL, TextT: Integer;
  Idx: Integer;
  TabData: TcsTabData;

  procedure DrawTick;
  begin
    if (gdSelected in AState) then
      GridTabs.Canvas.Pen.Color := clHighlightText
    else
      GridTabs.Canvas.Pen.Color := clWindowText;
    GridTabs.Canvas.MoveTo(ARect.Left + 4, ARect.Bottom - 8);
    GridTabs.Canvas.LineTo(ARect.Left + 6, ARect.Bottom - 6);
    GridTabs.Canvas.LineTo(ARect.Left + 12, ARect.Bottom - 12);
    GridTabs.Canvas.MoveTo(ARect.Left + 4, ARect.Bottom - 7);
    GridTabs.Canvas.LineTo(ARect.Left + 6, ARect.Bottom - 5);
    GridTabs.Canvas.LineTo(ARect.Left + 12, ARect.Bottom - 11);
  end;

begin
  TextL := ARect.Left + 2;
  TextT := ARect.Top + 2;
  Idx := ARow - GridTabs.FixedRows;
  if Idx < TabDataList.Count then
  begin
    TabData := TabDataList[Idx];
    case ACol of
      0: GridTabs.Canvas.TextOut(TextL, TextT, TabData.Caption);
      1: if not TabData.Bitmap.Empty then
           GridTabs.Canvas.CopyRect(ARect, TabData.Bitmap.Canvas,
             Rect(0, 0, ARect.Right - ARect.Left, ARect.Bottom - ARect.Top));
      2: GridTabs.Canvas.TextOut(TextL, TextT, IntToStr(TabData.NumGlyphs));
      3:  with GridTabs.Canvas do
          begin
            Brush.Color := TabData.Color;
            FillRect(ARect);
          end;
      4:  begin
            if TabData.Enabled then
              DrawTick;
          end;
      5:  begin
            if TabData.Visible then
              DrawTick;
          end;
    end;
  end;
end;

procedure TcsTabEditorSubForm.AddTab;
var
  TabData: TcsTabData;
begin
  TabData := TcsTabData.Create;
  TabData.Caption := '<New>';
  TabDataList.Add(TabData);
  if GridTabs.RowCount - GridTabs.FixedRows < TabDataList.Count then
  begin
    GridTabs.RowCount := GridTabs.RowCount + 1;
    GridTabs.Row := GridTabs.RowCount - 1;
  end;
  GridTabs.Invalidate;
  SetStatus;
  Fetch;
end;

procedure TcsTabEditorSubForm.DeleteTab;
var
  Idx: Integer;
begin
  Idx := GridTabs.Row - GridTabs.FixedRows;
  TabDataList.Delete(Idx);
  if GridTabs.RowCount > GridTabs.FixedRows + 1 then
    GridTabs.RowCount := GridTabs.RowCount - 1;
  { grid will always have at least one non-fixed row -- clear it }
  GridTabs.Invalidate;
  SetStatus;
  Fetch;
end;

{ Fetch data for current tab from TabDataList into editing controls. }
procedure TcsTabEditorSubForm.Fetch;
var
  Idx: Integer;
  TabData: TcsTabData;
begin
  Idx := GridTabs.Row - GridTabs.FixedRows;
  if (Idx >= 0) and (Idx < TabDataList.Count) then
  begin
    TabData := TabDataList[Idx];
    MemoCaption.Text := TabData.Caption;
    ImageBitmap.Picture.Bitmap := TabData.Bitmap;
    EditNumGlyphs.Text := IntToStr(TabData.NumGlyphs);
    FInFetch := True; { so that checkbox OnClick handler doesn't call Store }
    CheckEnabled.Checked := TabData.Enabled;
    CheckVisible.Checked := TabData.Visible;
    FInFetch := False;
    MemoHint.Text := TabData.Hint;
  end
  else
  begin
    MemoCaption.Text := '';
    ImageBitmap.Picture.Bitmap := nil;
    EditNumGlyphs.Text := IntToStr(1);
    FInFetch := True; { so that checkbox OnClick handler doesn't call Store }
    CheckEnabled.Checked := True;
    CheckVisible.Checked := True;
    FInFetch := False;
    MemoHint.Text := '';
  end;
end;

{ Store data for current tab from editing controls into TabDataList. }
procedure TcsTabEditorSubForm.Store;
var
  Idx: Integer;
  TabData: TcsTabData;
begin
  Idx := GridTabs.Row - GridTabs.FixedRows;
  if (Idx >= 0) and (Idx < TabDataList.Count) then
  begin
    TabData := TabDataList[Idx];
    TabData.Caption := MemoCaption.Text;
    TabData.Bitmap := ImageBitmap.Picture.Bitmap;
    try
      TabData.NumGlyphs := StrToInt(EditNumGlyphs.Text);
    except
      TabData.NumGlyphs := 1;
    end;
    TabData.Enabled := CheckEnabled.Checked;
    TabData.Visible := CheckVisible.Checked;
    TabData.Hint := MemoHint.Text;
    GridTabs.Invalidate;
  end;
end;

procedure TcsTabEditorSubForm.EditExit(Sender: TObject);
begin
  Store;
end;

procedure TcsTabEditorSubForm.BtnAddClick(Sender: TObject);
begin
  Store;
  AddTab;
  MemoCaption.SelectAll;
  MemoCaption.SetFocus;
end;

procedure TcsTabEditorSubForm.GridTabsClick(Sender: TObject);
begin
  Fetch;
end;

procedure TcsTabEditorSubForm.BtnColorClick(Sender: TObject);
begin
  with TColorDialog.Create(Self) do
  begin
    Color := TabDataList[GridTabs.Row - GridTabs.FixedRows].Color;
    if Execute then
    begin
      TabDataList[GridTabs.Row - GridTabs.FixedRows].Color := Color;
      GridTabs.Invalidate;
    end;
    Free;
  end;
end;

procedure TcsTabEditorSubForm.BtnDeleteClick(Sender: TObject);
begin
  DeleteTab;
  Fetch;
end;

procedure TcsTabEditorSubForm.LabelColorClick(Sender: TObject);
begin
  if (LabelColor.FocusControl <> nil) and
    (LabelColor.FocusControl is TButton) then
    with LabelColor.FocusControl as TButton do
      Click;
end;

procedure TcsTabEditorSubForm.LabelNumGlyphsClick(Sender: TObject);
begin
  if LabelNumGlyphs.FocusControl <> nil then
    LabelNumGlyphs.FocusControl.SetFocus;
end;

procedure TcsTabEditorSubForm.SetStatus;
var
  HasTabs: Boolean;
begin
  HasTabs := (TabDataList.Count > 0);
  BtnDelete.Enabled := HasTabs;
  BtnNext.Enabled := HasTabs;
  BtnPrevious.Enabled := HasTabs;
  BtnMoveUp.Enabled := HasTabs;
  BtnMoveDown.Enabled := HasTabs;
  MemoCaption.Enabled := HasTabs;
  ImageBitmap.Enabled := HasTabs;
  EditNumGlyphs.Enabled := HasTabs;
  BtnColor.Enabled := HasTabs;
  CheckEnabled.Enabled := HasTabs;
  CheckVisible.Enabled := HasTabs;
  MemoHint.Enabled := HasTabs;
  GridTabs.Enabled := HasTabs;
end;

procedure TcsTabEditorSubForm.BtnNextClick(Sender: TObject);
begin
  Store;
  if GridTabs.Row < GridTabs.RowCount - GridTabs.FixedRows  - 1 then
    GridTabs.Row := GridTabs.Row + 1;
end;

procedure TcsTabEditorSubForm.BtnPreviousClick(Sender: TObject);
begin
  Store;
  if GridTabs.Row > GridTabs.FixedRows then
    GridTabs.Row := GridTabs.Row - 1;
end;

procedure TcsTabEditorSubForm.BtnMoveClick(Sender: TObject);
var
  Idx, NewIdx: Integer;
begin
  Store;
  Idx := GridTabs.Row - GridTabs.FixedRows;
  NewIdx := Idx;
  if (Sender = BtnMoveUp) then Dec(NewIdx)
  else Inc(NewIdx);
  if (NewIdx >= 0) and (NewIdx < TabDataList.Count) then
  begin
    TabDataList.Move(Idx, NewIdx);
    GridTabs.Row := NewIdx + GridTabs.FixedRows;
  end;
end;

{ TBitmapComponent is just a TComponent descendant with a published
  Bitmap property that is put into the list passed as the first argument
  to GetComponentProperties to get the currently installed property editor
  for TBitmap types.
}
type
  TBitmapComponent = class(TComponent)
  private
    FBitmap: TBitmap;
    procedure SetBitmap(Value: TBitmap);
  published
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    property Bitmap: TBitmap read FBitmap write SetBitmap;
  end;

constructor TBitmapComponent.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FBitmap := TBitmap.Create;
end;

destructor TBitmapComponent.Destroy;
begin
  FBitmap.Free;
  inherited Destroy;
end;

procedure TBitmapComponent.SetBitmap(Value: TBitmap);
begin
  FBitmap.Assign(Value);
end;

{ Call the currently installed property editor for type Bitmap to edit
  the bitmap value -- note that this isn't testable outside of design-mode
  due to use of form designer and property editors.
} 
procedure TcsTabEditorSubForm.ImageBitmapClick(Sender: TObject);
var
{$IFDEF VER130}
  ComponentList: TDesignerSelectionList;
{$ELSE}
  ComponentList: TComponentList;
{$ENDIF}
  BitmapComponent: TBitmapComponent;
begin
{$IFDEF VER130}
  ComponentList := TDesignerSelectionList.Create;
{$ELSE}
  ComponentList := TComponentList.Create;
{$ENDIF}
  try
    BitmapComponent := TBitmapComponent.Create(Self);
    try
      BitmapComponent.Bitmap := ImageBitmap.Picture.Bitmap;
      ComponentList.Add(BitmapComponent);
      GetComponentProperties(ComponentList, tkAny, FFormDesigner, GetPropertyEditor);
      if (FBitmapPropertyEditor <> nil) then
      begin
        FBitmapPropertyEditor.Edit;
        ImageBitmap.Picture.Bitmap := BitmapComponent.Bitmap;
        Store;
      end;
    finally
      BitmapComponent.Free;
    end;
  finally
    ComponentList.Free;
  end;
end;

procedure TcsTabEditorSubForm.CheckClick(Sender: TObject);
begin
  if (Sender is TCheckBox) then
    with (Sender as TCheckBox) do
      if Checked then
        Caption := 'Yes'
      else
        Caption := 'No';
  if not FInFetch then
    Store;
end;

{ Callback procedure for getting a property editor. }
procedure TcsTabEditorSubForm.GetPropertyEditor(Editor: TPropertyEditor);
begin
  if Editor.GetName = 'Bitmap' then
  begin
    Editor.Initialize;
    FBitmapPropertyEditor := Editor;
  end;
end;

end.
