{*******************************************************}
{                                                       }
{   Copyright (c) 1997 Classic Software                 }
{   All Rights Reserved                                 }
{                                                       }
{*******************************************************}

unit CSTCRgn;

{$B-,P+,W-,X+}

interface

uses
  SysUtils, WinTypes, WinProcs, Messages, Classes, Controls, Graphics;

const
  { Define 'optimum' no. of tab regions kept in cache; cannot be 0. }
  TcsOptimumTabRegionCacheSize = 1;

type
  { TcsTabRegionCache implements a MRU (most recently used) cache of recently
    created tab regions.  The first item in the cache is the MRU tab region.
    Because a region (and not just a rect) must be used for hit-testing when
    the mouse is moved over a tab it is too slow to dynamically create the
    tab's region each time the mouse is moved.  Conversely, creating
    and keeping the regions for all tabs would use too much of GDI resources
    (about 5% for 10 tabs in Delphi 1).  The default cache capacity of 1 (one)
    minimises GDI resource usage while at the same time having acceptable
    speed -- while the mouse is moved over a tab the cached region is the only
    one needed (and thus immediately available) until the mouse crosses over to
    another tab.
  }
  TcsTabRegionCache = class(TObject)
  private
    FCache: TList;
    FCapacity: Integer;
    function GetCount: Integer;
    procedure SetCapacity(Value: Integer);
  public
    constructor Create;
    destructor Destroy; override;
    { Add adds a Rect -> Region association to the cache, as the new
      first entry, if there is no region for Rect already present.
    }
    procedure Add(ARect: TRect; ARegion: hRgn); { to front of cache }
    procedure Clear; { entire cache }
    function Find(ARect: TRect): hRgn;
    { Capacity determines how many tab regions will be cached.
      Capacity can be:
        -1 to cache all tab regions (maximum GDI resource usage, maximum speed),
        +n, where n > 0, to cache up to n tab regions (compromise).
    }
    property Capacity: Integer read FCapacity write SetCapacity
      default TcsOptimumTabRegionCacheSize;
    property Count: Integer read GetCount;
  end;

implementation

{ ----------------------------- }
{ TcsTabRegionCacheItem         }
{ ----------------------------- }

type
  TcsTabRegionCacheItem = class(TObject)
  private
    FTabRect: TRect;
    FTabRegion: hRgn;
    procedure SetTabRegion(Value: hRgn);
  public
    constructor Create;
    destructor Destroy; override;
    property TabRect: TRect read FTabRect write FTabRect;
    property TabRegion: hRgn read FTabRegion write SetTabRegion;
  end;

constructor TcsTabRegionCacheItem.Create;
begin
  inherited Create;
  FTabRect := Rect(-1, -1, -1, -1);
end;

destructor TcsTabRegionCacheItem.Destroy;
begin
  SetTabRegion(0);
  inherited Destroy;
end;

procedure TcsTabRegionCacheItem.SetTabRegion(Value: hRgn);
begin
  if FTabRegion <> Value then
  begin
    { free memory used by previous region }
    if FTabRegion <> 0 then
      DeleteObject(FTabRegion);
    FTabRegion := Value;
  end;
end;

{ ----------------------------- }
{ TcsTabRegionCache             }
{ ----------------------------- }

constructor TcsTabRegionCache.Create;
begin
  inherited Create;
  FCache := TList.Create;
  FCapacity := TcsOptimumTabRegionCacheSize;
end;

destructor TcsTabRegionCache.Destroy;
var
  I: Integer;
begin
  for I := 0 to FCache.Count - 1 do
    TcsTabRegionCacheItem(FCache[I]).Free;
  FCache.Free;
  inherited Destroy;
end;

procedure TcsTabRegionCache.Add(ARect: TRect; ARegion: hRgn);
var
  Item: TcsTabRegionCacheItem;
  Region: hRgn;
  OldPos: Integer;
  I: Integer;
begin
  Region := Find(ARect);
  if Region = 0 then { no region for specified rect was found }
  begin { create new item and insert as most recent }
    Item := TcsTabRegionCacheItem.Create;
    Item.TabRect := ARect;
    Item.TabRegion := ARegion;
    FCache.Insert(0, Item);
  end
  else
  begin { a region for the specified rect was found; move to start }
    OldPos := -1;
    for I := 0 to FCache.Count - 1 do
    begin
      if TcsTabRegionCacheItem(FCache[I]).TabRegion = Region then
      begin
        OldPos := I;
        Break;
      end;
    end;
    if (OldPos >= 0) and (OldPos < FCache.Count) then
      FCache.Move(OldPos, 0);
  end;
  if (FCapacity <> -1) and (FCache.Count > FCapacity) then
  begin { remove oldest item from cache }
    I := FCache.Count - 1;
    TcsTabRegionCacheItem(FCache[I]).Free;
    FCache.Delete(I);
  end;
end;

procedure TcsTabRegionCache.Clear;
var
  I: Integer;
begin
  for I := 0 to FCache.Count - 1 do
    TcsTabRegionCacheItem(FCache[I]).Free;
  FCache.Clear;
end;

function TcsTabRegionCache.Find(ARect: TRect): hRgn;
var
  I: Integer;
begin
  Result := 0;
  for I := 0 to FCache.Count - 1 do
  begin
    if EqualRect(ARect, TcsTabRegionCacheItem(FCache[I]).TabRect) then
    begin
      Result := TcsTabRegionCacheItem(FCache[I]).TabRegion;
      Break;
    end;
  end;
end;

function TcsTabRegionCache.GetCount: Integer;
begin
  Result := FCache.Count;
end;

procedure TcsTabRegionCache.SetCapacity(Value: Integer);
var
  I: Integer;
begin
  if (Value < -1) or (Value = 0) then
    raise Exception.Create('Capacity must be -1 (to cache all regions) or > 0');
  if FCapacity <> Value then
  begin
    FCapacity := Value;
    if FCapacity <> -1 then
      while FCache.Count > FCapacity do
      begin
        I := FCache.Count - 1;
        TcsTabRegionCacheItem(FCache[I]).Free;
        FCache.Delete(I);
      end;
  end;
end;

end.
