{*******************************************************}
{                                                       }
{   Copyright (c) 1997, 1999 Classic Software           }
{   All Rights Reserved                                 }
{                                                       }
{*******************************************************}

unit CSTCBase;

{$IFDEF VER80}
{$B-,P+,W-,X+}
{$ELSE}
{$B-,P+,W-,X+,J+}
{$ENDIF}
{$DEFINE CSTC_TEXTEXTENTCACHE}

interface

uses
{$IFDEF CSTC_TEXTEXTENTCACHE}
  SysUtils, WinTypes, WinProcs, Messages, Classes, Controls, Graphics,
  StdCtrls, Forms, CSTCRgn, CSUpDown, CSTCTExC;
{$ELSE}
  SysUtils, WinTypes, WinProcs, Messages, Classes, Controls, Graphics,
  StdCtrls, Forms, CSTCRgn, CSUpDown;
{$ENDIF}

const
  TcsMaxTabRegionPts = 20; { Maximum no. of points for defining tab shape }

type
  TcsTabStyle =
    (tsTabSet,         { Win 3.1 TabSet style tabs }
     tsTabbedNotebook, { Win 3.1 Tabbed Notebook style tabs (a la TcsNotebook) }
     tsTabControl      { Win 95 Tab Control style tabs }
    );

  TcsBevelWidth = 0..2;
  TcsTabNumGlyphs = 1..3; { Selected, Disabled, Unselected }
  TcsTabSequence = (tsStandard, tsReverse);
  TcsTabOrientation = (toTop, toLeft, toBottom, toRight);
  { Glyph position is relative to the tab's caption,
    i.e. gaLeft indicates the glyph is displayed to the left of the caption.
    gaStretch will cause the glyph to be stretched to fit the entire
    _usable area_ of the tab face.  gaBack places the glyph 'behind' the text.
  }
  TcsGlyphPosition = (gpLeft, gpTop, gpRight, gpBottom, gpBack, gpStretch);
  TcsHorizontalAlignment = (haLeft, haCenter, haRight);
  TcsVerticalAlignment = (vaTop, vaCenter, vaBottom);
  TcsFrameItem = (fiTab, fiCard);
  TcsFrameFacet = (ffBorder, ffHighlight, ffShadow);

  { Theoretically, the tsHotTrack style should not be mutually exclusive to
    tsSelected or tsUnselected, but changing TcsTextStyle to a set to allow
    multiple values (e.g. [tsHotTrack, tsSelected]) would break existing user's
    code.  The user will have to add their own code to determine if the tab
    being drawn is selected or not.
  }
  TcsTextStyle = (tsHotTrack, tsSelected, tsUnselected, tsDisabledHighlight, tsDisabledShadow);

  TcsTabRegionPts = Array[0..TcsMaxTabRegionPts - 1] of TPoint;

  TcsTabData = class;
  TcsTabDataList = class;

  { A TabChangingEvent event is generated prior to selecting a different tab. }
  TcsTabChangingEvent = procedure(Sender: TObject; NewIndex: Integer;
                                  var AllowChange: Boolean) of object;

  { A PaintBackgroundEvent occurs when the background of the tab control is
    being textured/filled.
  }
  TcsPaintBackgroundEvent =
    procedure(Sender: TObject; ACanvas: TCanvas; const ARect: TRect;
      var Handled: Boolean) of object;

  { A PaintTabBackgroundEvent occurs when the background of each tab is being
    erased/painted.
  }
  TcsPaintTabBackgroundEvent =
    procedure(Sender: TObject; ACanvas: TCanvas; ATabIndex: Integer;
      const ARect: TRect; var Handled: Boolean) of object;

  { A PaintCardBackgroundEvent occurs when the background of each card is being
    erased/painted.  The front (current) card is in Row 0.
  }
  TcsPaintCardBackgroundEvent =
    procedure(Sender: TObject; ACanvas: TCanvas; ARow: Integer;
      const ARect: TRect; var Handled: Boolean) of object;

  { A GetTextColorEvent occurs when the text of a tab is being drawn. }
  TcsGetTextColorEvent =
    procedure(Sender: TObject; ATabIndex: Integer; AStyle: TcsTextStyle;
      var AColor: TColor; var Handled: Boolean) of object;

  { A GetFrameColorEvent occurs when the color of a certain facet of the frame
    is needed.
  }
  TcsGetFrameColorEvent =
    procedure(Sender: TObject; AItem: TcsFrameItem; AFacet: TcsFrameFacet;
      AIndex: Integer; var AColor: TColor; var Handled: Boolean) of object;

  TcsChameleonTabControl = class(TCustomControl)
  private
    { ===== Fields used for internal workings of the class: ===== }
    FBuffer: TBitmap;
    FCalcNeeded: Boolean;
    FCalcTextExtentLines: TStringList; { used by CalcTextExtent }
    FChangingDone: Boolean; { flag to indicate OnChanging already generated }
    FCommands: TList;
    FDoneTabIndexDefault: Boolean; { indicates if default tab index has been set }
    FDrawTabTextLines: TStringList; { used by DrawTabText }
    FFirstInView: Integer;
    FFixedDimension: Integer;
    FFocusRectBrushColor: TColor;
    FHFonts: TList; { Stack of font handles selected into buffer's canvas }
    FHitTest: TPoint;
    FHotTrackIndex: Integer;
    FOriginalHint: String; { original Hint for the control }
    FPrevDisplayRect: TRect; { DisplayRect used for previous AlignControls }
    FRowExtent: Integer;
    FScrollBtns: TcsUpDown;
    FScrollBtnsNeeded: Boolean;
    FTabDataList: TcsTabDataList; { internal tab data }
    FTabRegionCache: TcsTabRegionCache;
    FTabRegionCacheSize: Integer;
{$IFDEF CSTC_TEXTEXTENTCACHE}
    FTextExtentCacheNormal: TcsTextExtentCache;
    FTextExtentCacheBold: TcsTextExtentCache;
{$ENDIF}
    { FTextFont contains the same font as Self.Font and is used instead of
      Self.Font so that changes to the font style (of FTextFont) don't cause
      further (infinite) CMFontChanged events to self.
    }
    FTextFont: TFont;
    FTimerHandle: THandle;
    { ===== Fields used for properties: ===== }
    FAlignTabs: Boolean;
    FBackgroundColor: TColor;
    FBevelWidth: TcsBevelWidth;
    FBoldCurrentTab: Boolean;
    FClientArea: Boolean;
    FColoredTabs: Boolean;
    FCornerSize: Integer;
    FFrameBorderColor: TColor;
    FFrameHighlightColor: TColor;
    FFrameShadowColor: TColor;
    FGlyphPosition: TcsGlyphPosition;
    FGlyphHAlignment: TcsHorizontalAlignment;
    FGlyphMargin: Integer;
    FGlyphVAlignment: TcsVerticalAlignment;
    FHotTrack: Boolean;
    FMargin: Integer;
    FMultiLine: Boolean;
    FParentBackgroundColor: Boolean;
    FRowIndent: Integer;
    FRowOverlap: Integer; { amount each row of tabs overlaps the previous row }
    FScrollBtnArrowColor: TColor;
    FScrollBtnFaceColor: TColor;
    FSidewaysText: Boolean;
    FTabHeight: Integer;
    FTabHints: Boolean;
    FTabIndex: Integer;
    FTabIndexDefault: Integer;
    FTabOrientation: TcsTabOrientation;
    FTabSequence: TcsTabSequence;
    FTabStyle: TcsTabStyle;
    FTabWidth: Integer;
    FTextColorDisabledHighlight: TColor;
    FTextColorDisabledShadow: TColor;
    FTextColorHotTrack: TColor;
    FTextColorSelected: TColor;
    FTextColorUnselected: TColor;
    FTextHAlignment: TcsHorizontalAlignment;
    FTextVAlignment: TcsVerticalAlignment;
    { ===== Fields used for events: ===== }
    FOnChange: TNotifyEvent;
    FOnChanging: TcsTabChangingEvent;
    FOnGetFrameColor: TcsGetFrameColorEvent;
    FOnGetTextColor: TcsGetTextColorEvent;
    FOnPaintBackground: TcsPaintBackgroundEvent;
    FOnPaintCardBackground: TcsPaintCardBackgroundEvent;
    FOnPaintTabBackground: TcsPaintTabBackgroundEvent;
    FOnTabClick: TNotifyEvent;
    procedure AddCommand(const Command: Integer);
    procedure AddCommandPt(const Command: Integer; const APoint: TPoint);
    procedure AlignScrollBtns;
    procedure BringTabToFrontRow(TabData: TcsTabData);
    procedure CalcCardDrawCommands(ARow: Integer);
    function CalcDefaultTabFaceExtent(AFont: TFont): TSize;
    function CalcGlyphExtent(ABitmap: TBitmap;
      ANumGlyphs: TcsTabNumGlyphs): TSize;
    function CalcMappedCardRect(ARow: Integer): TRect;
    function CalcMappedTabRect(ATabIndex: Integer; TabData: TcsTabData): TRect;
    procedure CalcMappedTabRegionPts(ATabIndex: Integer;
      var Pts: TcsTabRegionPts; var NumPts: Integer);
    function CalcMapPoint(const RawPt: TPoint): TPoint;
    function CalcMapRect(const RawRect: TRect): TRect;
    procedure CalcMetrics;
    procedure CalcScrollBtnsNeeded;
    function CalcTabExtentFromTabFaceExtent(TabFaceExtent: TSize): TSize;
    function CalcTabFaceRect(ARect: TRect): TRect;
    function CalcTabRegion(ATabIndex: Integer;
      const ARect: TRect): hRgn;
    procedure CalcTabRegionPts(ATabIndex: Integer;
      var Pts: TcsTabRegionPts; var NumPts: Integer);
    function CalcTextExtent(const S: String; Horizontal: Boolean): TSize;
    procedure CalcTabDrawCommands(ATabIndex: Integer);
    function CalcTabRect(ATabIndex: Integer; TabData: TcsTabData): TRect;
    function CalcWholeRect: TRect;
    procedure CancelHotTrackTimer;
    function CanSelectTab(ATabIndex: Integer): Boolean;
    procedure CheckCalcNeeded;
    procedure CreateScrollBtns;
    procedure DeselectFont;
    procedure DoRealign;
    procedure DoTextOut(X, Y: Integer; const AString: String; ACanvas: TCanvas;
      Horizontal: Boolean; AColor: TColor);
    procedure DrawShape(Current: Boolean);
    procedure DrawTabBackground(ATabIndex: Integer; const ARect: TRect);
    procedure DrawTabFace(ATabIndex: Integer; const ARect: TRect);
    procedure DrawTabs;
    function GetDisplayRect: TRect;
    function GetExtraTopMargin: Integer;
    function GetFirstVisible: Integer;
    function GetIndexHeight: Integer;
    function GetIndexRect: TRect;
    function GetIndexWidth: Integer;
    function GetInitialTabOffset: Integer;
    function GetLastVisible: Integer;
    function GetScrollBtnSize: Integer;
    function GetTabOffset(ATabHeight: Integer): Integer;
{$IFDEF CSTC_TEXTEXTENTCACHE}
    function GetTextExtent(const AString: String; Horizontal: Boolean): TSize;
{$ENDIF}
    function IsBackgroundColorStored: Boolean;
    procedure ParseTextLines(const S: String; Lines: TStrings);
    function PopHFont: HFont;
    procedure PushHFont(Value: HFont);
    procedure ScrollBtnClick(Sender: TObject);
    procedure SelectFont;
    procedure SetAlignTabs(Value: Boolean);
    procedure SetBackgroundColor(Value: TColor);
    procedure SetBevelWidth(Value: TcsBevelWidth);
    procedure SetBoldCurrentTab(Value: Boolean);
    procedure SetCalcNeeded(Value: Boolean);
    procedure SetClientArea(Value: Boolean);
    procedure SetColoredTabs(Value: Boolean);
    procedure SetCornerSize(Value: Integer);
    procedure SetFrameBorderColor(Value: TColor);
    procedure SetFrameHighlightColor(Value: TColor);
    procedure SetFrameShadowColor(Value: TColor);
    procedure SetGlyphPosition(Value: TcsGlyphPosition);
    procedure SetGlyphHAlignment(Value: TcsHorizontalAlignment);
    procedure SetGlyphMargin(Value: Integer);
    procedure SetGlyphVAlignment(Value: TcsVerticalAlignment);
    procedure SetHotTrack(Value: Boolean);
    procedure SetMargin(Value: Integer);
    procedure SetMultiLine(Value: Boolean);
    procedure SetParentBackgroundColor(Value: Boolean);
    procedure SetRowExtent(Value: Integer);
    procedure SetRowIndent(Value: Integer);
    procedure SetRowOverlap(Value: Integer);
    procedure SetScrollBtnArrowColor(Value: TColor);
    procedure SetScrollBtnFaceColor(Value: TColor);
    procedure SetSidewaysText(Value: Boolean);
    procedure SetTabHeight(Value: Integer);
    procedure SetTabHints(Value: Boolean);
    procedure SetTabIndex(Value: Integer);
    procedure SetTabOrientation(Value: TcsTabOrientation);
    procedure SetTabRegionCacheSize(Value: Integer);
    procedure SetTabSequence(Value: TcsTabSequence);
    procedure SetTabStyle(Value: TcsTabStyle);
    procedure SetTabWidth(Value: Integer);
    procedure SetTextColorDisabledHighlight(Value: TColor);
    procedure SetTextColorDisabledShadow(Value: TColor);
    procedure SetTextColorHotTrack(Value: TColor);
    procedure SetTextColorSelected(Value: TColor);
    procedure SetTextColorUnselected(Value: TColor);
    procedure SetTextHAlignment(Value: TcsHorizontalAlignment);
    procedure SetTextVAlignment(Value: TcsVerticalAlignment);
    procedure StopHotTracking;    
    function TabInView(TabData: TcsTabData): Boolean;
    procedure CMColorChanged(var Message: TMessage); message CM_COLORCHANGED;
    procedure CMCtl3DChanged(var Message: TMessage); message CM_CTL3DCHANGED;
    procedure CMDesignHitTest(var Msg: TCMDesignHitTest); message CM_DESIGNHITTEST;
    procedure CMDialogChar(var Message: TCMDialogChar); message CM_DIALOGCHAR;
    procedure CMFontChanged(var Message: TMessage); message CM_FONTCHANGED;
    procedure CMParentColorChanged(var Message: TMessage); message CM_PARENTCOLORCHANGED;
    procedure WMEraseBkgnd(var Message: TWMEraseBkgnd); message WM_ERASEBKGND;
    procedure WMGetDlgCode(var Message: TWMGetDlgCode); message WM_GETDLGCODE;
    procedure WMKillFocus(var Msg: TWMKillFocus); message WM_KILLFOCUS;
    procedure WMNCHitTest(var Msg: TWMNCHitTest); message WM_NCHITTEST;
    procedure WMSetCursor(var Msg: TWMSetCursor); message WM_SETCURSOR;
    procedure WMSetFocus(var Msg: TWMSetFocus); message WM_SETFOCUS;
    procedure WMSize(var Message: TWMSize); message WM_SIZE;
    procedure WMTimer(var Msg: TWMTimer); message WM_TIMER;
  protected
    procedure AlignControls(AControl: TControl; var Rect: TRect); override;
    procedure Change; virtual;
    procedure Changing(NewIndex: Integer; var Allowed: Boolean); virtual;
    { GetTabDataList is used by descendant classes (in other units) and thus
      cannot be private.
    }
    function GetTabDataList: TcsTabDataList;
    procedure GetFrameColor(AItem: TcsFrameItem; AFacet: TcsFrameFacet;
      AIndex: Integer; var AColor: TColor; var Handled: Boolean); virtual;
    procedure GetTextColor(ATabIndex: Integer; AStyle: TcsTextStyle;
      var AColor: TColor; var Handled: Boolean); virtual;
    procedure KeyDown(var Key: Word; Shift: TShiftState); override;
    procedure Loaded; override;
    procedure MouseDown(Button: TMouseButton; Shift: TShiftState;
      X, Y: Integer); override;
    procedure MouseMove(Shift: TShiftState; X, Y: Integer); override;
    procedure Paint; override;
    procedure PaintBackground(ACanvas: TCanvas; const ARect: TRect;
      var Handled: Boolean); virtual;
    procedure PaintCardBackground(ACanvas: TCanvas; ARow: Integer;
      const ARect: TRect; var Handled: Boolean); virtual;
    procedure PaintTabBackground(ACanvas: TCanvas; ATabIndex: Integer;
      const ARect: TRect; var Handled: Boolean); virtual;
    procedure Rebuild; virtual;
    procedure TabClick; virtual;
    property AlignTabs: Boolean read FAlignTabs write SetAlignTabs default False;
    property BackgroundColor: TColor read FBackgroundColor
      write SetBackgroundColor stored IsBackgroundColorStored;
    { BevelWidth = 2 is only supported for tsTabbedNotebook style tabs
      and is equivalent to BevelWidth = 1 for other style tabs.
      BevelWidth has no effect for tsTabControl style tabs and
      only affects the current tab for tsTabSet style tabs.
      For those tab styles where BevelWidth is used, Ctl3D must also be True.
    }
    property BevelWidth: TcsBevelWidth read FBevelWidth write SetBevelWidth
      default 2;
    property BoldCurrentTab: Boolean read FBoldCurrentTab write SetBoldCurrentTab
      default True;
    { ClientArea determines whether an enclosed frame ('card') is drawn around
      the client area or just the top line of the frame is drawn.
    }
    property ClientArea: Boolean read FClientArea write SetClientArea
      default True;
    property ColoredTabs: Boolean read FColoredTabs write SetColoredTabs
      default False;
    property CornerSize: Integer read FCornerSize write SetCornerSize default 5;
    property DisplayRect: TRect read GetDisplayRect;
    property FrameBorderColor: TColor read FFrameBorderColor
      write SetFrameBorderColor default clWindowFrame;
    property FrameHighlightColor: TColor read FFrameHighlightColor
      write SetFrameHighlightColor default clBtnHighlight;
    property FrameShadowColor: TColor read FFrameShadowColor
      write SetFrameShadowColor default clBtnShadow;
    property GlyphHAlignment: TcsHorizontalAlignment read FGlyphHAlignment
      write SetGlyphHAlignment default haLeft;
    property GlyphMargin: Integer read FGlyphMargin write SetGlyphMargin
      default 2;
    property GlyphPosition: TcsGlyphPosition read FGlyphPosition
      write SetGlyphPosition default gpLeft;
    property GlyphVAlignment: TcsVerticalAlignment
      read FGlyphVAlignment write SetGlyphVAlignment default vaTop;
    property HotTrack: Boolean read FHotTrack write SetHotTrack default False;
    property Margin: Integer read FMargin write SetMargin default 0;
    property MultiLine: Boolean read FMultiLine write SetMultiLine
      default False;
    { ParentBackgroundColor determines whether the color of the parent will be
      used for BackgroundColor.
    }
    property ParentBackgroundColor: Boolean read FParentBackgroundColor
      write SetParentBackgroundColor default True;
    property RowIndent: Integer read FRowIndent write SetRowIndent
      default 5;
    property RowOverlap: Integer read FRowOverlap write SetRowOverlap
      default 5;
    property ScrollBtnArrowColor: TColor read FScrollBtnArrowColor
      write SetScrollBtnArrowColor default clBtnText;
    property ScrollBtnFaceColor: TColor read FScrollBtnFaceColor
      write SetScrollBtnFaceColor default clBtnFace;
    { ScrollBtns property is only provided for use by descendants so they can
      ignore the scroll buttons within their GetChildren and WriteComponents
      implementations.
    }
    property ScrollBtns: TcsUpDown read FScrollBtns stored False;
    property SidewaysText: Boolean read FSidewaysText
      write SetSidewaysText default False;
    property TabHeight: Integer read FTabHeight write SetTabHeight default 0;
    property TabHints: Boolean read FTabHints write SetTabHints default False;    
    property TabIndex: Integer read FTabIndex write SetTabIndex default -1;
    property TabIndexDefault: Integer read FTabIndexDefault
      write FTabIndexDefault default 0;
    property TabOrientation: TcsTabOrientation read FTabOrientation
      write SetTabOrientation default toTop;
    property TabRegionCacheSize: Integer read FTabRegionCacheSize
      write SetTabRegionCacheSize default TcsOptimumTabRegionCacheSize;
    property TabSequence: TcsTabSequence read FTabSequence
      write SetTabSequence default tsStandard;
    property TabStyle: TcsTabStyle read FTabStyle
      write SetTabStyle default tsTabbedNotebook;
    property TabWidth: Integer read FTabWidth write SetTabWidth default 0;
    property TextColorDisabledHighlight: TColor read FTextColorDisabledHighlight
      write SetTextColorDisabledHighlight default clBtnHighlight;
    property TextColorDisabledShadow: TColor read FTextColorDisabledShadow
      write SetTextColorDisabledShadow default clBtnShadow;
    property TextColorHotTrack: TColor read FTextColorHotTrack
      write SetTextColorHotTrack default clActiveCaption;      
    property TextColorSelected: TColor read FTextColorSelected
      write SetTextColorSelected default clBtnText;
    property TextColorUnselected: TColor read FTextColorUnselected
      write SetTextColorUnselected default clBtnText;
    property TextHAlignment: TcsHorizontalAlignment
      read FTextHAlignment write SetTextHAlignment default haLeft;
    property TextVAlignment: TcsVerticalAlignment
      read FTextVAlignment write SetTextVAlignment default vaTop;
    property OnChange: TNotifyEvent read FOnChange write FOnChange;
    property OnChanging: TcsTabChangingEvent read FOnChanging write FOnChanging;
    property OnGetFrameColor: TcsGetFrameColorEvent
      read FOnGetFrameColor write FOnGetFrameColor;
    property OnGetTextColor: TcsGetTextColorEvent
      read FOnGetTextColor write FOnGetTextColor;
    property OnPaintBackground: TcsPaintBackgroundEvent
      read FOnPaintBackground write FOnPaintBackground;
    property OnPaintCardBackground: TcsPaintCardBackgroundEvent
      read FOnPaintCardBackground write FOnPaintCardBackground;
    property OnPaintTabBackground: TcsPaintTabBackgroundEvent
      read FOnPaintTabBackground write FOnPaintTabBackground;
    property OnTabClick: TNotifyEvent read FOnTabClick write FOnTabClick;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    function TabAtPos(X, Y: Integer): Integer;
    { Ctl3D has no effect for tsTabControl style tabs (fixed 3D effect used),
      and only affects the current tab in tsTabSet style tabs.
    }
    property Ctl3D;
    property TabStop default True;
  end;

  { A TcsTabData object is used to hold the attributes of a single tab. }
  TcsTabData = class(TPersistent)
  private
    FBitmap: TBitmap;
    FCaption: String;
    FColor: TColor;
    FEnabled: Boolean;
    FHint: String;
    FNumGlyphs: TcsTabNumGlyphs;
    FRawRect: TRect;
    FVisible: Boolean;
    FRow: Integer;
    procedure SetBitmap(Value: TBitmap);
  public
    constructor Create;
    destructor Destroy; override;
    procedure Assign(Source: TPersistent); override;
    property Bitmap: TBitmap read FBitmap write SetBitmap;
    property Caption: String read FCaption write FCaption;
    property Color: TColor read FColor write FColor default clBtnFace;
    property Enabled: Boolean read FEnabled write FEnabled default True;
    property Hint: String read FHint write FHint;
    property NumGlyphs: TcsTabNumGlyphs read FNumGlyphs write FNumGlyphs
      default 1;
    property RawRect: TRect read FRawRect write FRawRect;
    property Row: Integer read FRow write FRow default 0;
    property Visible: Boolean read FVisible write FVisible default True;
  end;

  TcsTabDataList = class(TPersistent)
  private
    FTabList: TList;
    function GetCount: Integer;
    function GetItem(Index: Integer): TcsTabData;
    procedure SetItem(Index: Integer; Value: TcsTabData);
  public
    constructor Create;
    destructor Destroy; override;
    function Add(Item: TcsTabData): Integer;
    procedure Assign(Source: TPersistent); override;
    procedure Clear;
    procedure Delete(Index: Integer);
    function First: TcsTabData;
    function IndexOf(Item: TcsTabData): Integer;
    procedure Insert(Index: Integer; Item: TcsTabData);
    function Last: TcsTabData;
    procedure Move(CurIndex, NewIndex: Integer);
    function Remove(Item: TcsTabData): Integer;
    property Count: Integer read GetCount;
    property Items[Index: Integer]: TcsTabData read GetItem write SetItem; default;
  end;

implementation

uses CSCConst, CSGfx, CSMaxMin;

type
  TRectClass = class(TObject)
  public
    Rect: TRect;
  end;

var
  CursorLoaded: Boolean;

{ ----------------------------- }
{ TcsTabData                    }
{ ----------------------------- }

constructor TcsTabData.Create;
begin
  FBitmap := TBitmap.Create;
  FColor := clBtnFace;
  FEnabled := True;
  FNumGlyphs := 1;
  FVisible := True;
end;

destructor TcsTabData.Destroy;
begin
  FBitmap.Free;
  inherited Destroy;
end;

procedure TcsTabData.Assign(Source: TPersistent);
var
  TabData: TcsTabData;
begin
  if Source is TcsTabData then
  begin
    TabData := TcsTabData(Source);
    Caption := TabData.Caption;
    Bitmap := TabData.Bitmap;
    NumGlyphs := TabData.NumGlyphs;
    Color := TabData.Color;
    Enabled := TabData.Enabled;
    Visible := TabData.Visible;
    Hint := TabData.Hint;
    Exit;
  end;
  inherited Assign(Source);
end;

procedure TcsTabData.SetBitmap(Value: TBitmap);
begin
  FBitmap.Assign(Value);
end;

{ ----------------------------- }
{ TcsTabDataList                }
{ ----------------------------- }

constructor TcsTabDataList.Create;
begin
  inherited Create;
  FTabList := TList.Create;
end;

destructor TcsTabDataList.Destroy;
begin
  Clear;
  FTabList.Free;
  inherited Destroy;
end;

procedure TcsTabDataList.Assign(Source: TPersistent);
var
  AList: TcsTabDataList;
  TabData: TcsTabData;
  I: Integer;
begin
  if Source is TcsTabDataList then
  begin
    Clear;
    AList := TcsTabDataList(Source);
    for I := 0 to AList.Count - 1 do
    begin
      TabData := TcsTabData.Create;
      TabData.Assign(AList[I]);
      Add(TabData);
    end;
    Exit;
  end;
  inherited Assign(Source);
end;

function TcsTabDataList.Add(Item: TcsTabData): Integer;
begin
  Result := FTabList.Add(Item);
end;

procedure TcsTabDataList.Clear;
var
  I: Integer;
begin
  for I := 0 to FTabList.Count - 1 do
    Items[I].Free;
  FTabList.Clear;
end;

procedure TcsTabDataList.Delete(Index: Integer);
begin
  Items[Index].Free;
  FTabList.Delete(Index);
end;

function TcsTabDataList.First: TcsTabData;
begin
  Result := TcsTabData(FTabList.First);
end;

function TcsTabDataList.GetCount: Integer;
begin
  Result := FTabList.Count;
end;

function TcsTabDataList.GetItem(Index: Integer): TcsTabData;
begin
  Result := TcsTabData(FTabList[Index]);
end;

function TcsTabDataList.IndexOf(Item: TcsTabData): Integer;
begin
  Result := FTabList.IndexOf(Item);
end;

procedure TcsTabDataList.Insert(Index: Integer; Item: TcsTabData);
begin
  FTabList.Insert(Index, Item);
end;

function TcsTabDataList.Last: TcsTabData;
begin
  Result := TcsTabData(FTabList.Last);
end;

procedure TcsTabDataList.Move(CurIndex, NewIndex: Integer);
var
  Item: TcsTabData;
begin
  if CurIndex <> NewIndex then
  begin
    Item := Items[CurIndex];
    FTabList.Delete(CurIndex); { don't use Self.Delete because it frees item }
    Insert(NewIndex, Item);
  end;
end;

procedure TcsTabDataList.SetItem(Index: Integer; Value: TcsTabData);
begin
  Items[Index].Free;
  FTabList[Index] := Value;
end;

{ Note that Remove uses the non-virtual Delete method }
function TcsTabDataList.Remove(Item: TcsTabData): Integer;
begin
  Result := IndexOf(Item);
  if Result <> -1 then Delete(Result);
end;

{ ----------------------------- }
{ TcsChameleonTabControl        }
{ ----------------------------- }

const
  PolyFillMode = Integer(Alternate);  { DON'T use TFillMode values }
  MidRangeScrollBtnPosition = 1;
  ExtentAdjustment = 10;
  ScrollBtnMargin = 2;
  CRLF = #13#10;
  { These commands don't expect any data after the command }
  CMD_BORDER          = 0;
  CMD_HIGHLIGHT       = 1;
  CMD_SHADOW          = 2;
  CMD_CTL3D1          = 3; { First level of 3D }
  CMD_CTL3D2          = 4; { Second level of 3D }
  { These commands all expect data (2 additional items) after the command } 
  CMD_MOVETO          = 5;
  CMD_LINETO          = 6;
  CMD_BORDER_COLOR    = 7; { set frame border color }
  CMD_HIGHLIGHT_COLOR = 8; { set frame highlight color }
  CMD_SHADOW_COLOR    = 9; { set frame shadow color }

constructor TcsChameleonTabControl.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  Width := 300;
  Height := 100;
  ControlStyle := [csAcceptsControls, csCaptureMouse, csClickEvents, csDoubleClicks];
  FBuffer := TBitmap.Create;
  FBuffer.Width := Width;
  FBuffer.Height := Height;
  FCommands := TList.Create;
  FHotTrackIndex := -1;
  FDoneTabIndexDefault := False;
  FFocusRectBrushColor := clBtnFace;
  FHFonts := TList.Create;
  FRowExtent := 1;
  FTabDataList := TcsTabDataList.Create;
{$IFDEF CSTC_TEXTEXTENTCACHE}
  { For speed: Create caches for extents of text strings, rather than
               recalculating their extents over and over.
  }
  FTextExtentCacheNormal := TcsTextExtentCache.Create;
  FTextExtentCacheBold := TcsTextExtentCache.Create;
{$ENDIF}
  { For speed: Create FCalcTextExtentLines and FDrawTabTextLines in advance so
               we won't need to test if they are nil prior to use.
  }
  FCalcTextExtentLines := TStringList.Create;
  FDrawTabTextLines := TStringList.Create;
  FTextFont := TFont.Create;
  { Ctl3D is already True -- setting again/explicitly would mess up ParentCtl3D }
  TabStop := True;
  { declare default property values -- some are unecessary (e.g. where False or
    first enumerated value) but are included for clarity
  }
  FAlignTabs := False;
  FBackgroundColor := clBtnFace;
  FBevelWidth := 2;
  FBoldCurrentTab := True;
  FClientArea := True;
  FColoredTabs := False;
  FCornerSize := 5;
  FFrameBorderColor := clWindowFrame;
  FFrameHighlightColor := clBtnHighlight;
  FFrameShadowColor := clBtnShadow;
  FGlyphPosition := gpLeft;
  FGlyphHAlignment := haLeft;
  FGlyphMargin := 2;
  FGlyphVAlignment := vaTop;
  FHotTrack := False;
  FMargin := 0;
  FMultiLine := False;
  FParentBackgroundColor := True;
  FRowIndent := 5;
  FRowOverlap := 5;
  FScrollBtnArrowColor := clBtnText;
  FScrollBtnFaceColor := clBtnFace;
  FSidewaysText := False;
  FTabHeight := 0;
  FTabHints := False;
  FTabIndex := -1;
  FTabIndexDefault := 0;
  FTabOrientation := toTop;
  FTabRegionCacheSize := TcsOptimumTabRegionCacheSize;
  FTabSequence := tsStandard;
  FTabStyle := tsTabbedNotebook;
  FTabWidth := 0;
  FTextColorDisabledHighlight := clBtnHighlight;
  FTextColorDisabledShadow := clBtnShadow;
  FTextColorHotTrack := clActiveCaption;
  FTextColorSelected := clBtnText;
  FTextColorUnselected := clBtnText;
  FTextHAlignment := haLeft;
  FTextVAlignment := vaTop;
  { create tab region cache, now that FTabRegionCacheSize has been set }
  FTabRegionCache := TcsTabRegionCache.Create;
  FTabRegionCache.Capacity := FTabRegionCacheSize;
  SetCalcNeeded(True);
end;

destructor TcsChameleonTabControl.Destroy;
begin
  CancelHotTrackTimer;
  FTabDataList.Free;
  FTabRegionCache.Free;
  FCommands.Free;
  FBuffer.Free;
  FHFonts.Free;
{$IFDEF CSTC_TEXTEXTENTCACHE}
  FTextExtentCacheNormal.Free;
  FTextExtentCacheBold.Free;
{$ENDIF}
  FTextFont.Free;
  FCalcTextExtentLines.Free;
  FDrawTabTextLines.Free;
  inherited Destroy;
end;

function TcsChameleonTabControl.GetTabDataList: TcsTabDataList;
begin
  Result := FTabDataList;
end;

procedure TcsChameleonTabControl.Paint;
var
  Handled: Boolean;
  ARect: TRect;
  BrushHandle: HBrush;

begin
  FDoneTabIndexDefault := True;
  CheckCalcNeeded;
  if (Width <> FBuffer.Width) or
    (Height <> FBuffer.Height) then
  begin
    FBuffer.Width := Width;
    FBuffer.Height := Height;
  end;

  { It is necessary to init the buffer's brush to prevent 'silent' errors
    (which only show in a debug/log output window) from being caused *sometimes*
    when we assign new brush colors later.  (Error is raised by
    TResourceManager.FreeResource in Graphics, but don't know why?)
  }
  FBuffer.Canvas.Brush.Handle := GetStockObject(BLACK_BRUSH);

  { === Erase background === }
  ARect := CalcWholeRect;
  { expand right and bottom sides to include edge }
  Inc(ARect.Right);
  Inc(ARect.Bottom);
  Handled := False;
  PaintBackground(FBuffer.Canvas, ARect, Handled);
  if not Handled then
  begin
    BrushHandle := CreateSolidBrush(ColorToRGB(FBackgroundColor));
    FillRect(FBuffer.Canvas.Handle, ARect, BrushHandle);
    DeleteObject(BrushHandle);
  end;

  AlignScrollBtns;
  DrawTabs;
  { copy buffer bitmap to control's canvas }
  Canvas.CopyRect(ClientRect, FBuffer.Canvas, ClientRect);
end;

procedure TcsChameleonTabControl.Rebuild;
begin
  SetCalcNeeded(True);
end;

procedure TcsChameleonTabControl.SetBevelWidth(Value: TcsBevelWidth);
begin
  if FBevelWidth <> Value then
  begin
    FBevelWidth := Value;
    { bevel is only shown when Ctl3D is in effect }
    if Ctl3D then
      Rebuild;
  end;
end;

procedure TcsChameleonTabControl.SetCornerSize(Value: Integer);
begin
  if (Value >= 0) and (FCornerSize <> Value) then
  begin
    FCornerSize := Value;
    if FTabStyle = tsTabbedNotebook then
      Rebuild;
  end;
end;

procedure TcsChameleonTabControl.SetTabOrientation(Value: TcsTabOrientation);
begin
  if FTabOrientation <> Value then
  begin
    FTabOrientation := Value;
    if not ((csLoading in ComponentState) or (csReading in ComponentState)) then
      Rebuild;
  end;
end;

procedure TcsChameleonTabControl.SetTabRegionCacheSize(Value: Integer);
begin
  if (Value = -1) or (Value >= 1) and (FTabRegionCacheSize <> Value) then
  begin
    FTabRegionCacheSize := Value;
    FTabRegionCache.Capacity := Value;
  end;
end;

procedure TcsChameleonTabControl.SetTabStyle(Value: TcsTabStyle);
begin
  if FTabStyle <> Value then
  begin
    FTabStyle := Value;
    Rebuild;
  end;
end;

procedure TcsChameleonTabControl.SetSidewaysText(Value: Boolean);
begin
  if FSidewaysText <> Value then
  begin
    FSidewaysText := Value;
    Rebuild;
  end;
end;

procedure TcsChameleonTabControl.SetTabSequence(Value: TcsTabSequence);
begin
  if FTabSequence <> Value then
  begin
    FTabSequence := Value;
    Rebuild;
  end;
end;

procedure TcsChameleonTabControl.SetTabHeight(Value: Integer);
begin
  if (Value >= 0) and (FTabHeight <> Value) then
  begin
    FTabHeight := Value;
    Rebuild;
  end;
end;

procedure TcsChameleonTabControl.SetTabWidth(Value: Integer);
begin
  if (Value >= 0) and (FTabWidth <> Value) then
  begin
    FTabWidth := Value;
    Rebuild;
  end;
end;

{ The scroll buttons are only needed if the total width of all tabs
  is > the tab index width.
}
procedure TcsChameleonTabControl.CalcScrollBtnsNeeded;
var
  FirstTab, LastTab: TcsTabData;
  FirstVisibleIdx, IndexWidth: Integer;
begin
  FScrollBtnsNeeded := False;
  if FMultiLine then Exit;
  FirstVisibleIdx := GetFirstVisible;
  if (FirstVisibleIdx >= 0) then
  begin
    FirstTab := FTabDataList[FirstVisibleIdx];
    LastTab := FTabDataList[GetLastVisible];
    IndexWidth := GetIndexWidth - (FMargin*2);
    FScrollBtnsNeeded :=
      (FTabDataList.Count > 0) and
      ((FTabSequence = tsStandard) and
       (LastTab.RawRect.Right > IndexWidth)) or
      ((FTabSequence = tsReverse) and
       (FirstTab.RawRect.Right > IndexWidth));
  end;
end;

procedure TcsChameleonTabControl.AlignScrollBtns;
var
  R: TRect;
  L, T, W, H: Integer;
  BtnSize: Integer;
begin
  if not ((csLoading in ComponentState) or (csReading in ComponentState)) then
  begin
    if FScrollBtnsNeeded then
    begin
      if FScrollBtns = nil then
        CreateScrollBtns;
    end
    else
    begin
      if FScrollBtns <> nil then
      begin
        { using Hide doesn't work at design-time; use trickery! }
        W := FScrollBtns.Width;
        H := FScrollBtns.Height;
        FScrollBtns.SetBounds(-1-W, -1-H, W, H);
      end;
      Exit;
    end;

    if FTabStyle = tsTabControl then
      FScrollBtns.Style := bsWin95
    else
      FScrollBtns.Style := bsWin31;

    { scroll btns are needed; determine appropriate size and position }
    R := GetIndexRect;
    BtnSize := GetScrollBtnSize;
    if FTabOrientation in [toTop, toBottom] then
    begin
      H := BtnSize;
      W := BtnSize*2; { two buttons }
    end
    else
    begin
      H := BtnSize*2; { two buttons }
      W := BtnSize;
    end;
    L := 0; { suppress compiler warning }
    T := 0; { suppress compiler warning }
    if FTabSequence = tsStandard then
      case FTabOrientation of
        toTop:
          begin
            L := R.Right - W - ScrollBtnMargin;
            T := R.Bottom - H - ScrollBtnMargin;
          end;
        toBottom:
          begin
            L := R.Right - W - ScrollBtnMargin;
            T := R.Top + ScrollBtnMargin;
          end;
        toLeft:
          begin
            L := R.Right - W - ScrollBtnMargin;
            T := R.Top + ScrollBtnMargin;
          end;
        toRight:
          begin
            L := R.Left + ScrollBtnMargin;
            T := R.Bottom - H - ScrollBtnMargin;
          end;
      end
    else { tsReverse }
      case FTabOrientation of
        toTop:
          begin
            L := R.Left + ScrollBtnMargin;
            T := R.Bottom - H - ScrollBtnMargin;
          end;
        toBottom:
          begin
            L := R.Left + ScrollBtnMargin;
            T := R.Top + ScrollBtnMargin;
          end;
        toLeft:
          begin
            L := R.Right - W - ScrollBtnMargin;
            T := R.Bottom - H - ScrollBtnMargin;
          end;
        toRight:
          begin
            L := R.Left + ScrollBtnMargin;
            T := R.Top + ScrollBtnMargin;
          end;
      end;

    if FTabOrientation in [toTop, toBottom] then
      FScrollBtns.Orientation := udHorizontal
    else
      FScrollBtns.Orientation := udVertical;

    FScrollBtns.SetBounds(L, T, W, H);
    FScrollBtns.Visible := True;
  end;
end;

procedure TcsChameleonTabControl.AlignControls(AControl: TControl;
  var Rect: TRect);
begin
  AlignScrollBtns;
  Rect := DisplayRect;
  FPrevDisplayRect := Rect;
  inherited AlignControls(AControl, Rect);
end;

procedure TcsChameleonTabControl.Loaded;
begin
  inherited Loaded;
  FOriginalHint := Hint; { remember control's original Hint value }
  SetCalcNeeded(True);
  if FTabIndexDefault <> FTabIndex then
    FTabIndex := FTabIndexDefault;
  { now that tabs have been loaded, set TabIndex to check validity }
  SetTabIndex(FTabIndex);
end;

function TcsChameleonTabControl.GetLastVisible: Integer;
var
  TabData: TcsTabData;
begin
  Result := FTabDataList.Count - 1;
  while (Result >= 0) do
  begin
    TabData := FTabDataList[Result];
    if TabData.Visible then
      Break
    else
      Dec(Result);
  end;
end;

function TcsChameleonTabControl.GetFirstVisible: Integer;
var
  TabData: TcsTabData;
  Idx: Integer;
begin
  Result := -1;
  Idx := 0;
  while (Idx < FTabDataList.Count) do
  begin
    TabData := FTabDataList[Idx];
    if TabData.Visible then
    begin
      Result := Idx;
      Break;
    end;
    Inc(Idx);
  end;
end;

function TcsChameleonTabControl.GetExtraTopMargin: Integer;
begin
  if (FTabStyle = tsTabControl) then
    Result := 3
  else
    Result := 0;
end;

function TcsChameleonTabControl.GetIndexHeight;
var
  Adjustment: Integer;
begin
  if FClientArea then
    Result := FFixedDimension +
      ((FFixedDimension - FRowOverlap)*(FRowExtent - 1)) + GetExtraTopMargin
  else
  begin
    if (FTabOrientation in [toTop, toBottom]) then
      Result := Height - 1
    else
      Result := Width - 1;
    Adjustment := 0;
    case FTabStyle of
      tsTabSet:
        if Ctl3D then
          Inc(Adjustment, MinInt(1, FBevelWidth));
      tsTabControl:
        if FTabOrientation in [toBottom, toRight] then
          Inc(Adjustment, 1);
      tsTabbedNotebook:
        if Ctl3D then
          Inc(Adjustment, FBevelWidth);
    end;
    Dec(Result, Adjustment);
  end;
end;

function TcsChameleonTabControl.GetIndexWidth;
var
  IndexRect: TRect;
begin
  IndexRect := GetIndexRect;
  if FTabOrientation in [toTop, toBottom] then
    Result := IndexRect.Right - IndexRect.Left
  else
    Result := IndexRect.Bottom - IndexRect.Top;
end;

{ Return the rect for the area where the tabs are displayed. }
function TcsChameleonTabControl.GetIndexRect: TRect;
var
  WholeRect, IndexRect: TRect;
  IndexHeight: Integer;
  L, T, R, B, W, H: Integer;
begin
  WholeRect := Rect(0, 0, Width - 1, Height - 1);
  IndexHeight := GetIndexHeight;
  W := WholeRect.Right - WholeRect.Left;
  H := WholeRect.Bottom - WholeRect.Top;

  if FTabOrientation = toRight then L := W - IndexHeight else L := 0;
  if FTabOrientation = toBottom then T := H - IndexHeight else T := 0;
  if FTabOrientation = toLeft then R := IndexHeight else R := W;
  if FTabOrientation = toTop then B := IndexHeight else B := H;

  IndexRect := Rect(L, T, R, B);
  Result := IndexRect;
end;

function TcsChameleonTabControl.GetDisplayRect: TRect;
var
  Adjustment: Integer;
begin
  if FClientArea then
  begin
    Result := CalcMappedCardRect(0);
    case FTabStyle of
      tsTabSet:
        begin
          Inc(Result.Left, 1);
          Inc(Result.Top, 1);
          Adjustment := 0;
          if Ctl3D then
            Inc(Adjustment, MinInt(1, FBevelWidth));
          InflateRect(Result, -Adjustment, -Adjustment);
        end;
      tsTabControl:
        begin
          Inc(Result.Left, 1);
          Dec(Result.Right, 1);
          Inc(Result.Top, 1);
          Dec(Result.Bottom, 1);
        end;
      tsTabbedNotebook:
        begin
          Inc(Result.Left, 1);
          Inc(Result.Top, 1);
          Adjustment := 0;
          if Ctl3D then
            Inc(Adjustment, FBevelWidth);
          InflateRect(Result, -Adjustment, -Adjustment);
        end;
    end;
  end
  else { no client-area }
    { must return a rect which is not empty (0,0,0,0), otherwise
      AlignControls will just ignore the DisplayRect, so return
      a rect which is outside the visible area instead
    }
    Result := Rect(-2, -2, -1, -1);
end;

function TcsChameleonTabControl.CalcWholeRect: TRect;
var
  ARect: TRect;
begin
  { ClientRect cannot be used if control is being created
    and has no window handle yet.
  }
  if HandleAllocated then
    ARect := ClientRect
  else
    ARect := Rect(0, 0, Width, Height);
  Result := Rect(0, 0, ARect.Right - 1, ARect.Bottom - 1);
end;

{ Return the rect for the card corresponding to the specified row of tabs. }
function TcsChameleonTabControl.CalcMappedCardRect(ARow: Integer): TRect;
var
  ARect: TRect;
  L, T, R, B: Integer;
  TopAdj, BottomAdj, LeftAdj, RightAdj: Integer;

  { AdjustCardRect changes the specified rect so that the 'sides' and 'bottom'
    are beyond the control's boundaries and won't be seen.
    By faking the card rect so that the 'sides' and 'bottom' are actually
    outside the control's bounds we will only get the 'top' of the card drawn
    and don't need to redraw to eliminate the unwanted frame portions and also
    don't need to use additional drawing to fix up the joins on each end of the
    'top' line.
  }
  procedure AdjustCardRect(var ARect: TRect);
  const
    Adjustment: Integer = 3; { sufficient for max. bevel depth + border }
  var
    WholeRect: TRect;
  begin
    WholeRect := CalcWholeRect;
    case FTabOrientation of
      toTop:
        begin
          ARect.Left := WholeRect.Left - Adjustment;
          ARect.Right := WholeRect.Right + Adjustment;
          ARect.Bottom := WholeRect.Bottom + Adjustment;
        end;
      toBottom:
        begin
          ARect.Left := WholeRect.Left - Adjustment;
          ARect.Right := WholeRect.Right + Adjustment;
          ARect.Top := WholeRect.Top - Adjustment;
        end;
      toLeft:
        begin
          ARect.Right := WholeRect.Right + Adjustment;
          ARect.Top := WholeRect.Top - Adjustment;
          ARect.Bottom := WholeRect.Bottom + Adjustment;
        end;
      toRight:
        begin
          ARect.Left := WholeRect.Left - Adjustment;
          ARect.Top := WholeRect.Top - Adjustment;
          ARect.Bottom := WholeRect.Bottom + Adjustment;
        end;
    end;
  end; { AdjustCardRect }

begin { CalcMappedCardRect }
  ARect := CalcWholeRect;
  if (FTabDataList.Count = 0) and FClientArea then
  begin
    Result := ARect;
    Exit;
  end;
  L := ARect.Left;
  T := ARect.Top;
  R := ARect.Right;
  B := ARect.Bottom;
  { CalcMapRect (or more correctly, CalcMapPoint) can't be used to map the card
    rect, so we need to work out the rect for the appropriate orientation and
    tab sequence.
  }
  TopAdj := GetIndexHeight - ((FFixedDimension - FRowOverlap)*ARow);
  BottomAdj := (FFixedDimension - FRowOverlap)*ARow;
  LeftAdj := (FRowIndent*ARow);
  RightAdj := FRowIndent*(FRowExtent - ARow - 1);

  if FTabSequence = tsStandard then
  begin
    case FTabOrientation of
      toTop:
        begin
          Inc(T, TopAdj);
          Dec(B, BottomAdj);
          Inc(L, LeftAdj);
          Dec(R, RightAdj);
        end;
      toBottom:
        begin
          Inc(T, BottomAdj);
          Dec(B, TopAdj);
          Inc(L, LeftAdj);
          Dec(R, RightAdj);
        end;
      toLeft:
        begin
          Inc(T, RightAdj);
          Dec(B, LeftAdj);
          Inc(L, TopAdj);
          Dec(R, BottomAdj);
        end;
      toRight:
        begin
          Inc(T, LeftAdj);
          Dec(B, RightAdj);
          Inc(L, BottomAdj);
          Dec(R, TopAdj);
        end;
    end;
  end
  else
  begin
    case FTabOrientation of
      toTop:
        begin
          Inc(T, TopAdj);
          Dec(B, BottomAdj);
          Inc(L, RightAdj);
          Dec(R, LeftAdj);
        end;
      toBottom:
        begin
          Inc(T, BottomAdj);
          Dec(B, TopAdj);
          Inc(L, RightAdj);
          Dec(R, LeftAdj);
        end;
      toLeft:
        begin
          Inc(T, LeftAdj);
          Dec(B, RightAdj);
          Inc(L, TopAdj);
          Dec(R, BottomAdj);
        end;
      toRight:
        begin
          Inc(T, RightAdj);
          Dec(B, LeftAdj);
          Inc(L, BottomAdj);
          Dec(R, TopAdj);
        end;
    end;
  end;
  Result := Rect(L, T, R, B);
  if (L > R) or (T > B) then { tabs are oversized -- shrink card rect }
    Result := Rect(0, 0, 0, 0)
  else if not FClientArea then
    AdjustCardRect(Result);
end;

function TcsChameleonTabControl.GetScrollBtnSize: Integer;
begin
  SelectFont;
{$IFDEF CSTC_TEXTEXTENTCACHE}
  Result := GetTextExtent('A', True).cY;
{$ELSE}
  Result := CalcTextExtent('A', True).cY;
{$ENDIF}
  DeselectFont;
  Inc(Result, 2);
  { check if larger than fixed dimension }
  Result := MinInt(Result, FFixedDimension - 2);
end;

procedure TcsChameleonTabControl.DrawTabs;
var
  I: Integer;
  TabData: TcsTabData;
  FirstTabIdx: Integer;
  LastTabIdx: Integer;
  PrevRow: Integer;

  function FindFirstTabIdx: Integer;
  var
    I: Integer;
  begin
    Result := 0;
    for I := 0 to FTabDataList.Count - 1 do
    begin
      TabData := FTabDataList[I];
      if TabData.Visible and (TabData.Row = 0) then
      begin
        Result := I;
        Break;
      end;
    end;
  end;

  procedure DrawTab(ATabIndex: Integer);
  var
    R1, R2: TRect;
    Region: hRgn;
    TabData: TcsTabData;
    ARect: TRect;

    { draw tab outline }
    procedure DrawTabShape(ATabIndex: Integer);
    var
      APoint: TPoint;
    begin
      FCommands.Clear;
      { store information for frame item and aspect a point string }
      APoint.X := Ord(fiTab); { Item is tab }
      APoint.Y := ATabIndex;  { Aspect is the tab being drawn }
      AddCommandPt(CMD_BORDER_COLOR, APoint);
      AddCommandPt(CMD_HIGHLIGHT_COLOR, APoint);
      AddCommandPt(CMD_SHADOW_COLOR, APoint);
      CalcTabDrawCommands(ATabIndex {, Commands});
      DrawShape({Commands,} (ATabIndex = FTabIndex) { current tab? });
    end;

  begin { DrawTab }
    TabData := FTabDataList[ATabIndex];
    ARect := CalcMappedTabRect(ATabIndex, TabData);
    R1 := FBuffer.Canvas.ClipRect; { area of canvas being redrawn }
    R2 := ARect; { assume ARect already ordered (required for IntersectRect) }
    { check if this tab is inside the area being drawn, note that
      IntersectRect return value for Delphi 1 must be converted to Boolean
    }
    if not Bool(IntersectRect(R1, R1, R2)) then Exit; { no intersection; don't draw }

    { restrict tab drawing to within the tab's region }
    Region := CalcTabRegion(ATabIndex, ARect);
    if Region <> 0 then { region was created }
    begin
      { save DC so that region can be de-activated later }
      if SaveDC(FBuffer.Canvas.Handle) = 0 then
        raise Exception.Create('Couldn''t save device context');
      { Note: When a region is selected using SelectObject, the result is not
              the old region, i.e. don't save/restore value returned by
              SelectObject.
      }
      SelectObject(FBuffer.Canvas.Handle, Region);
      try
        { fill the background of the tab (inside the region) }
        DrawTabBackground(ATabIndex, ARect);
        { draw the tab caption & glyph }
        DrawTabFace(ATabIndex, ARect);
      finally
        { de-activate clipping region }
        if RestoreDC(FBuffer.Canvas.Handle, -1) = WordBool(0) then
          raise Exception.Create('Couldn''t restored device context');
      end;
    end;
    { now draw tab shape -- this is not done until the region is deselected so
      that the bottom and right edges, which are not included within regions,
      will be drawn
    }
    DrawTabShape(ATabIndex);
  end;

  procedure DrawCard(ARow: Integer);
  var
    ARect: TRect;
    Handled: Boolean;
    BrushColor: TColor;

    { ARow is the row of tabs, where row 0 is the first, row 1 the second etc. }
    procedure DrawCardShape(ARow: Integer);
    var
      APoint: TPoint;
    begin
      { draw card outline }
      FCommands.Clear;
      { store information for frame item and aspect as a point string }
      APoint.X := Ord(fiCard);
      APoint.Y := ARow;
      AddCommandPt(CMD_BORDER_COLOR, APoint);
      AddCommandPt(CMD_HIGHLIGHT_COLOR, APoint);
      AddCommandPt(CMD_SHADOW_COLOR, APoint);
      CalcCardDrawCommands(ARow);
      DrawShape((ARow = 0) { current card? });
    end;

  begin { DrawCard }
    ARect := CalcMappedCardRect(ARow);
    if not EmptyRect(ARect) then
    begin
      if FClientArea then
      begin
        { expand right and bottom edges (which aren't included by FillRect) }
        Inc(ARect.Right);
        Inc(ARect.Bottom);
      end;
      Handled := False;
      PaintCardBackground(FBuffer.Canvas, ARow, ARect, Handled);
      if not Handled then
      begin
        { fill the specified card with the appropriate color }
        if FColoredTabs and (FTabIndex >= 0) then
        begin
          BrushColor := FTabDataList[FTabIndex].Color;
          { change color of self to match the current tab -- this is OK to do
            within Paint processing because we have a special CMColorChanged
            handler which doesn't Invalidate (again)
          }
          Color := BrushColor;
        end
        else
          BrushColor := Color;
        FBuffer.Canvas.Brush.Color := BrushColor;
        FBuffer.Canvas.FillRect(ARect);
      end;

      { now draw card frame onto card's background }
      DrawCardShape(ARow);
    end;
  end;

begin { DrawTabs }
  if FTabDataList.Count > 1 then
  begin
    { draw tabs other than the current one; start with the last and go
      through to the first (painter's algorithm) -- this ensures they overlap
      each other correctly without having to worrying about explicitly
      drawing joins
    }
    FirstTabIdx := FindFirstTabIdx;
    if FirstTabIdx = 0 then
      LastTabIdx := FTabDataList.Count - 1
    else
      LastTabIdx := FirstTabIdx - 1;
    I := LastTabIdx;
    PrevRow := FRowExtent - 1;
    repeat
      TabData := FTabDataList[I];
      if TabData.Visible then
      begin
        if TabInView(TabData) and (I >= FFirstInView) and
          (I <> FTabIndex) then
        begin
          if (TabData.Row < PrevRow) and (PrevRow > 0) then
          begin
            { tab is in a new row, draw the card for previous row of tabs }
            DrawCard(PrevRow);
            PrevRow := TabData.Row;
          end;
          DrawTab(I);
        end;
      end;
      if I > 0 then
        Dec(I)
      else
        I := FTabDataList.Count - 1;
    until (I = LastTabIdx); { same tab that we started on }
    if (PrevRow = 1) then { penultimate card hasn't been drawn yet }
      DrawCard(1);
  end;

  { draw current tab if in view -- the current tab is drawn last so that it is
    always 'on top of' the other tabs
  }
  if (FTabIndex >= 0) and (FTabIndex >= FFirstInView) then
  begin
    TabData := FTabDataList[FTabIndex];
    if TabInView(TabData) then
    begin
      DrawTab(FTabIndex);
    end;
  end;
  DrawCard(0); { front card }
end;

{ Return the offset to the 'initial' edge of the initial tab's
  raw tab rect.
  Only relevant when MultiLine = False (i.e. initial tab may not be the
  first tab due to having scrolled).
  tsStandard: The left side of the initial tab's raw tab rect is
            aligned with the left side of the raw index rect.
  tsReverse: The right side of the initial tab's raw tab rect is
            aligned with the right side of the raw index rect.
}
function TcsChameleonTabControl.GetInitialTabOffset: Integer;
var
  TabData: TcsTabData;
  Adjust: Integer;
begin
  TabData := FTabDataList[FFirstInView];
  if FTabStyle = tsTabControl then
    Adjust := 2
  else
    Adjust := 0;
  Inc(Adjust, FMargin);
  if FTabSequence = tsStandard then
    Result := TabData.RawRect.Left - Adjust
  else
    Result := TabData.RawRect.Right + Adjust;
end;

{ Assume: Tab's clipping region has already been selected. }
procedure TcsChameleonTabControl.DrawTabBackground(ATabIndex: Integer;
  const ARect: TRect);
var
  Handled: Boolean;
  BrushColor: TColor;
begin
  Handled := False;
  PaintTabBackground(FBuffer.Canvas, ATabIndex, ARect, Handled);
  if not Handled then
  begin
    { fill the specified tab with the appropriate color }
    if FColoredTabs then
      BrushColor := FTabDataList[ATabIndex].Color
    else
      BrushColor := Color;
    FBuffer.Canvas.Brush.Color := BrushColor;
    FBuffer.Canvas.FillRect(ARect);
  end;
  { save color of brush for use later when drawing focus rect of current tab }
  FFocusRectBrushColor := FBuffer.Canvas.Brush.Color;
end;

{ Wrapper for TextOut call.
  This method is used to overcome the limitation of TextOut()
  (and ExtTextOut()) not automatically underlining the character prefixed
  by an ampersand in the string.
  Draws the text using the specified device context. If the text contains
  an ampersand (&) the following character will be underlined.
  If the text contains two successive ampersands (&&) then a single
  ampersand will be output.
}
procedure TcsChameleonTabControl.DoTextOut(X, Y: Integer; const AString: String;
  ACanvas: TCanvas; Horizontal: Boolean; AColor: TColor);
var
  P1, P2: Integer;
  Part: Array[0..2] of String;
  I, MaxPartIdx: Integer;
  Extent: TSize;
  RemoveUnderline: Boolean;
begin
  { Break the string up into parts: the part before the '&', the character
    immediately after the '&' (which will be underlined), and the remainder
  }
  MaxPartIdx := 2;
  P1 := Pos('&', AString);
  if (P1 > 0) then
  begin
    P2 := Pos('&', Copy(AString, P1 + 1, Length(AString)));
    if (P2 = 1) then Inc(P2, P1)
    else P2 := 0;
  end
  else P2 := 0;
  if (P1 > 0) and (P2 <> P1 + 1) then
  begin
    Part[0] := Copy(AString, 1, P1 - 1);
    Part[1] := Copy(AString, P1 + 1, 1);
    Part[2] := Copy(AString, P1 + 2, Length(AString));
  end
  else if (P1 > 0) and (P2 = P1 + 1) then { '&&' in string }
  begin
    Part[0] := Copy(AString, 1, P1);
    Part[1] := '';
    Part[2] := Copy(AString, P2 + 1, Length(AString));
  end
  else
  begin
    Part[0] := AString;
    Part[1] := '';
    Part[2] := '';
    MaxPartIdx := 0; { highest part index used }
  end;

  RemoveUnderline := False;
  { output each part in the appropriate font (plain or underlined) }
  for I := 0 to MaxPartIdx do
  begin
    if (I = 1) then
    begin
      { underlined part of text }
      if not (fsUnderline in FTextFont.Style) then
      begin
        FTextFont.Style := FTextFont.Style + [fsUnderline];
        RemoveUnderline := True;
      end
    end
    else
    begin
      if RemoveUnderline then
        FTextFont.Style := FTextFont.Style - [fsUnderline];
    end;

    SelectFont;
    { because we have selected our own font into the canvas we must explicitly
      set the text color and background mode for the selected font
    }
    SetTextColor(ACanvas.Handle, ColorToRGB(AColor));
    SetBkMode(ACanvas.Handle, TRANSPARENT);
    ACanvas.TextOut(X, Y, Part[I]);

    if MaxPartIdx > 0 then
    begin
      { determine extent of part just output so position can be updated }
      if Length(Part[I]) > 0 then
      begin
{$IFDEF CSTC_TEXTEXTENTCACHE}
        Extent := GetTextExtent(Part[I], Horizontal);
{$ELSE}
        Extent := CalcTextExtent(Part[I], Horizontal);
{$ENDIF}
        if not FSidewaysText then
          Inc(X, Extent.cX)
        else if FTabOrientation = toRight then
          Inc(Y, Extent.cY)
        else
          Dec(Y, Extent.cY);
      end;
    end;

    DeselectFont;
  end;
end; { DoTextOut }

{ Assume: Tab's clipping region has already been selected. }
procedure TcsChameleonTabControl.DrawTabFace(ATabIndex: Integer;
  const ARect: TRect);
var
  FaceRect: TRect;
  TabData: TcsTabData;
  GlyphSize: TSize;
  GlyphRect: TRect;
  TextRect: TRect;
  TextColor: TColor;
  Handled: Boolean;

  function GetTextRect: TRect;
  var
    Margin: Integer;
  begin
    Result := FaceRect;
    if TabData.Bitmap.Empty then
      Margin := 0
    else
      Margin := FGlyphMargin;
    { FGlyphPosition specifies where glyph will be relative to text }
    case FGlyphPosition of
      gpTop:     Result.Top := Result.Top + GlyphSize.cY + Margin;
      gpBottom:  Result.Bottom := Result.Bottom - GlyphSize.cY - Margin;
      gpLeft:    Result.Left := Result.Left + GlyphSize.cX + Margin;
      gpRight:   Result.Right := Result.Right - GlyphSize.cX - Margin;
    else
      { use all of FaceRect }
    end;
  end;

  function GetGlyphRect: TRect;
  begin
    Result := FaceRect;
    { FGlyphPosition specifies where glyph will be relative to text }
    case FGlyphPosition of
      gpTop:     Result.Bottom := Result.Top + GlyphSize.cY;
      gpBottom:  Result.Top := Result.Bottom - GlyphSize.cY;
      gpLeft:    Result.Right := Result.Left + GlyphSize.cX;
      gpRight:   Result.Left := Result.Right - GlyphSize.cX;
    else
      { use all of FaceRect }
    end;
  end;

  { Shrink TextRect to match the extent of the text in the direction
    perpendicular to the direction in which the text runs, i.e.
    if text is horizontal shrink TextRect in the vertical direction.
  }
  procedure AdjustTextRect;
  var
    Extent: TSize;
    Margin: Integer;
  begin
    { work out entire size of text and adjust text rect according to alignment }
    Extent.cX := 0;
    Extent.cY := 0;
    if Length(TabData.Caption) > 0 then
{$IFDEF CSTC_TEXTEXTENTCACHE}
      Extent := GetTextExtent(TabData.Caption, not FSidewaysText);
{$ELSE}
      Extent := CalcTextExtent(TabData.Caption, not FSidewaysText);
{$ENDIF}
    if FSidewaysText then
    begin
      if FTabOrientation = toRight then
      begin
        case TextHAlignment of
          haCenter:
            begin
              Margin := (TextRect.Right - TextRect.Left - Extent.cX) div 2;
              TextRect.Left := TextRect.Left + Margin;
              TextRect.Right := TextRect.Right - Margin;
            end;
          haLeft:
            TextRect.Right := TextRect.Left + Extent.cX;
        else { haRight }
          { do nothing }
        end;
      end
      else
      begin
        case TextHAlignment of
          haCenter:
            begin
              Margin := (TextRect.Right - TextRect.Left - Extent.cX) div 2;
              TextRect.Left := TextRect.Left + Margin;
              TextRect.Right := TextRect.Right - Margin;
            end;
          haRight:
            TextRect.Left := TextRect.Right - Extent.cX;
        else { haLeft }
          { do nothing }
        end;
      end
    end
    else
    begin
      case TextVAlignment of
        vaCenter:
          begin
            Margin := (TextRect.Bottom - TextRect.Top - Extent.cY) div 2;
            TextRect.Top := TextRect.Top + Margin;
            TextRect.Bottom := TextRect.Bottom - Margin;
          end;
        vaBottom:
          TextRect.Top := TextRect.Bottom - Extent.cY;
      else { vaTop }
        { do nothing }
      end;
    end;
  end;

  procedure DrawTabText(AColor: TColor; TextOffset: Integer);
  var
    OldStyle: TFontStyles;
    Lines: TStringList;
    I: Integer;
    Extent: TSize;
    XPos, YPos: Integer;
    XOffset, YOffset: Integer;
  begin
    OldStyle := FTextFont.Style;
    if (ATabIndex = FTabIndex) and FBoldCurrentTab then
      FTextFont.Style := FTextFont.Style + [fsBold];
    SelectFont;
    AdjustTextRect;

    { re-use same list object to speed things up by not recreating each time
      (which starts to add up when there are lots of tabs)
    }
    Lines := FDrawTabTextLines;
    Lines.Clear;
    ParseTextLines(TabData.Caption, Lines);
    if Lines.Count = 0 then Exit;
    XOffset := 0;
    YOffset := 0;
    I := 0;
    repeat
      { work out extent of current line and position within TextRect
        according to alignment
      }
      Extent.cX := 0;
      Extent.cY := 0;
      if Length(Lines[I]) > 0 then
{$IFDEF CSTC_TEXTEXTENTCACHE}
        Extent := GetTextExtent(Lines[I], not FSidewaysText);
{$ELSE}
        Extent := CalcTextExtent(Lines[I], not FSidewaysText);
{$ENDIF}
      if FSidewaysText then
      begin
        if FTabOrientation = toRight then
        begin
          case TextVAlignment of
            vaCenter:
              YOffset := (TextRect.Bottom - TextRect.Top - Extent.cY) div 2;
            vaBottom:
              YOffset := TextRect.Bottom - TextRect.Top - Extent.cY;
          else { vaTop }
            YOffset := 0;
          end;
          XPos := TextRect.Right - XOffset;
          YPos := TextRect.Top + YOffset;
        end
        else
        begin
          case TextVAlignment of
            vaTop:
              YOffset := TextRect.Bottom - TextRect.Top - Extent.cY;
            vaCenter:
              YOffset := (TextRect.Bottom - TextRect.Top - Extent.cY) div 2;
          else { vaBottom }
            YOffset := 0;
          end;
          XPos := TextRect.Left + XOffset;
          YPos := TextRect.Bottom - YOffset;
        end;
      end
      else
      begin
        case TextHAlignment of
          haCenter:
              XOffset := (TextRect.Right - TextRect.Left - Extent.cX) div 2;
          haRight:
            XOffset := TextRect.Right - TextRect.Left - Extent.cX;
        else { haLeft }
          XOffset := 0;
        end;
        XPos := TextRect.Left + XOffset;
        YPos := TextRect.Top + YOffset;
      end;
      Inc(XPos, TextOffset);
      Inc(YPos, TextOffset);
      DoTextOut(XPos, YPos, Lines[I],
        FBuffer.Canvas, not FSidewaysText, AColor);
      { update offset to be used for position of next line }
      if FSidewaysText then
      begin
        Inc(XOffset, Extent.cX);
        YOffset := 0;
      end
      else
      begin
        XOffset := 0;
        Inc(YOffset, Extent.cY);
      end;
      Inc(I);
    until I = Lines.Count;
    DeselectFont;
    FTextFont.Style := OldStyle;
  end;

  procedure DrawTabGlyph;
  var
    XOffset, YOffset: Integer;
    GlyphIdx: Integer;
    SrcRect: TRect;
    WorkW, WorkH: Integer;
    WorkRect: TRect;
    GlyphBmp, CompositeBmp, MaskBmp: TBitmap;

    { Fade the specified bitmap by applying a checkerboard brush pattern
      so that every second pixel is converted to the bitmap's transparent
      color.  The resultant bitmap is used when the tab is disabled.
    }
    procedure FadeBitmap(ABitmap: TBitmap);
    var
      BrushBmp: TBitmap;
      PrevColor, PixelColor: TColor;
      X, Y: Integer;
      PatternBrush: TBrush;
      OldTransparentColor: TColor;
      TempBmp: TBitmap;
      DstRect: TRect;
      BrushHandle: HBrush;
    begin
      OldTransparentColor := ABitmap.TransparentColor;
      DstRect := Rect(0, 0, ABitmap.Width, ABitmap.Height);
      TempBmp := TBitmap.Create;
      BrushBmp := TBitmap.Create;
      try
        with TempBmp do
        begin
          Width := ABitmap.Width;
          Height := ABitmap.Height;
          BrushHandle := CreateSolidBrush(ColorToRGB(OldTransparentColor));
          try
            FillRect(Canvas.Handle, DstRect, BrushHandle);
          finally
            DeleteObject(BrushHandle);
          end;
        end;

        { Make checkerboard brush pattern (alternating black and white) }
        BrushBmp.Width := 8;  { pattern brushes are 8x8 }
        BrushBmp.Height := 8;
        PrevColor := clWhite;
        for X := 0 to 7 do
        begin
          PixelColor := PrevColor;
          for Y := 0 to 7 do
          begin
            BrushBmp.Canvas.Pixels[X, Y] := ColorToRGB(PixelColor);
            PrevColor := PixelColor;
            if PixelColor = clWhite then
              PixelColor := clBlack
            else
              PixelColor := clWhite;
          end;
        end;

        { apply transparent color to target bitmap using brush pattern }
        PatternBrush := TBrush.Create;
        try
          PatternBrush.Bitmap := BrushBmp;
          ABitmap.Canvas.Brush := PatternBrush;
          BitBlt(ABitmap.Canvas.Handle, 0, 0, ABitmap.Width, ABitmap.Height,
                 TempBmp.Canvas.Handle, 0, 0, $CA0749); { DPSPxax }
        finally
          PatternBrush.Free;
        end;

      finally
        BrushBmp.Free;
        TempBmp.Free;
      end;
      { make sure bottom-left pixel retains original (transparent) color }
      ABitmap.Canvas.Pixels[0, ABitmap.Height - 1] := OldTransparentColor;
    end; { FadeBitmap }

  begin { DrawTabGlyph }
    if GlyphPosition <> gpStretch then
    begin
      case GlyphHAlignment of
        haCenter:
          XOffset := (GlyphRect.Right - GlyphRect.Left - GlyphSize.cX) div 2;
        haRight:
          XOffset := (GlyphRect.Right - GlyphRect.Left - GlyphSize.cX);
      else { haLeft }
          XOffset := 0;
      end;
      case GlyphVAlignment of
        vaCenter:
          YOffset := (GlyphRect.Bottom - GlyphRect.Top - GlyphSize.cY) div 2;
        vaBottom:
          YOffset := (GlyphRect.Bottom - GlyphRect.Top - GlyphSize.cY);
      else { vaTop }
          YOffset := 0;
      end;
      GlyphRect.Left := GlyphRect.Left + XOffset;
      GlyphRect.Top := GlyphRect.Top + YOffset;
      GlyphRect.Right := GlyphRect.Left + GlyphSize.cX;
      GlyphRect.Bottom := GlyphRect.Top + GlyphSize.cY;
    end;

    if TabData.Enabled then
    begin
      if (ATabIndex = FTabIndex) then
        GlyphIdx := 0
      else if (TabData.NumGlyphs >= 3) then
        GlyphIdx := 2
      else
        GlyphIdx := 0;
    end
    else
    begin
      if TabData.NumGlyphs >= 2 then
        GlyphIdx := 1
      else
        GlyphIdx := -1;
    end;
    if GlyphIdx >= 0 then
      SrcRect := Rect(GlyphIdx*GlyphSize.cX, 0,
        (GlyphIdx + 1)*GlyphSize.cX, GlyphSize.cY)
    else { use glyph 0 (Selected) }
      SrcRect := Rect(0, 0, GlyphSize.cX, GlyphSize.cY);

    WorkW := GlyphRect.Right - GlyphRect.Left + 1;
    WorkH := GlyphRect.Bottom - GlyphRect.Top + 1;
    WorkRect := Rect(0, 0, WorkW, WorkH);

    GlyphBmp := TBitmap.Create;
    CompositeBmp := TBitmap.Create;

    try
      { Copy the tab's existing background into CompositeBmp }
      with CompositeBmp do
      begin
        Width := WorkW;
        Height := WorkH;
        Canvas.CopyMode := cmSrcCopy;
        Canvas.CopyRect(WorkRect, FBuffer.Canvas, GlyphRect);
      end;

      { Make a copy of the tab's glyph, the same size as the destination area }
      with GlyphBmp do
      begin
        Width := WorkW;
        Height := WorkH;
        Canvas.CopyMode := cmSrcCopy;
        Canvas.CopyRect(WorkRect, TabData.Bitmap.Canvas, SrcRect);
      end;

      if GlyphIdx = -1 then
        FadeBitmap(GlyphBmp);

      { make GlyphBmp into a bitmap suitable for overlaying onto CompositeBmp }
      MaskBmp := CreateMonoMask(GlyphBmp, GlyphBmp.TransparentColor);
      try
        { combine the mask with the existing background; this will give
          the background with black ('holes') where the overlay will
          eventually be shown
        }
        CompositeBmp.Canvas.CopyMode := cmSrcAnd; { DSa }
        CompositeBmp.Canvas.CopyRect(WorkRect, MaskBmp.Canvas, WorkRect);

        { Generate the overlay image by combining the mask and the
          original image; this will give (courtesy of the appropriate
          ROP code) the image on a black background
        }
        GlyphBmp.Canvas.CopyMode := $00220326; { DSna }
        GlyphBmp.Canvas.CopyRect(WorkRect, MaskBmp.Canvas, WorkRect);

        { Now put the overlay image onto the background; this will
          fill in the black ('holes') with the overlay image, leaving
          the rest of the background as is
        }
        CompositeBmp.Canvas.CopyMode := cmSrcPaint; { DSo }
        CompositeBmp.Canvas.CopyRect(WorkRect, GlyphBmp.Canvas, WorkRect);
      finally
        MaskBmp.Free;
      end;

      { Now copy CompositeBmp back to position of tab in buffer }
      with FBuffer.Canvas do
      begin
        CopyMode := cmSrcCopy;
        CopyRect(GlyphRect, CompositeBmp.Canvas, WorkRect);
      end;

    finally
      GlyphBmp.Free;
      CompositeBmp.Free;
    end;
  end;

  procedure DrawTabFocusRect;
  var
    FocusRect: TRect;
  begin
    { inflate the face rect to get the focus rect }
    FocusRect := FaceRect;
    InflateRect(FocusRect, 1, 1);
    FBuffer.Canvas.DrawFocusRect(FocusRect);
  end;

begin { DrawTabFace }
  TabData := FTabDataList[ATabIndex];
  FaceRect := CalcTabFaceRect(ARect);
  GlyphSize := CalcGlyphExtent(TabData.Bitmap, TabData.NumGlyphs);
  TextRect := GetTextRect;
  GlyphRect := GetGlyphRect;
  if not TabData.Bitmap.Empty then
    DrawTabGlyph;
  Handled := False;

  { Note: OnGetTextColor events are performed even if caption is ''
    so that TextColor is initialised prior to calling DrawTabFocusRect.
  }
  if TabData.Enabled then
  begin
    if FHotTrack and (ATabIndex = FHotTrackIndex) then
    begin
      GetTextColor(ATabIndex, tsHotTrack, TextColor, Handled);
      if not Handled then
        TextColor := FTextColorHotTrack;
    end
    else if ATabIndex = FTabIndex then
    begin
      GetTextColor(ATabIndex, tsSelected, TextColor, Handled);
      if not Handled then
        TextColor := FTextColorSelected;
    end
    else
    begin
      GetTextColor(ATabIndex, tsUnselected, TextColor, Handled);
      if not Handled then
        TextColor := FTextColorUnselected;
    end;
    if TabData.Caption <> '' then
      DrawTabText(TextColor, 0);
  end
  else
  begin
    GetTextColor(ATabIndex, tsDisabledHighlight, TextColor, Handled);
    if not Handled then
      TextColor := FTextColorDisabledHighlight;
    if TabData.Caption <> '' then
      DrawTabText(TextColor, 1);
    GetTextColor(ATabIndex, tsDisabledShadow, TextColor, Handled);
    if not Handled then
      TextColor := FTextColorDisabledShadow;
    if TabData.Caption <> '' then
      DrawTabText(TextColor, 0);
  end;
  if (ATabIndex = FTabIndex) and Focused then
    DrawTabFocusRect;
end;

procedure TcsChameleonTabControl.DrawShape(Current: Boolean);
var
  I, Idx: Integer;
  Cmd: Integer;
  APoint: TPoint;
  BorderColor: TColor;    { state variable for current shadow color }
  HighlightColor: TColor; { state variable for current border color }
  ShadowColor: TColor;    { state variable for current highlight color }
  Item: TcsFrameItem;
  Facet: TcsFrameFacet;
  ItemColor: TColor;
  Handled: Boolean;

begin { DrawShape }
  BorderColor := FFrameBorderColor;
  HighlightColor := FFrameHighlightColor;
  ShadowColor := FFrameShadowColor;
  I := 0;
  while I < FCommands.Count do
  begin
    Cmd := Integer(FCommands[I]);
    Inc(I);
    if Cmd >= CMD_MOVETO then { data follows command }
    begin
      APoint.X := Integer(FCommands[I]);
      Inc(I);
      APoint.Y := Integer(FCommands[I]);
      Inc(I);
    end;
    case Cmd of
      CMD_BORDER:
        FBuffer.Canvas.Pen.Color := BorderColor;
      CMD_HIGHLIGHT:
        FBuffer.Canvas.Pen.Color := HighlightColor;
      CMD_SHADOW:
        FBuffer.Canvas.Pen.Color := ShadowColor;
      CMD_BORDER_COLOR, CMD_HIGHLIGHT_COLOR, CMD_SHADOW_COLOR:
      begin
        { set state variable for current color of specified facet }
        case Cmd of
          CMD_BORDER_COLOR: Facet := ffBorder;
          CMD_HIGHLIGHT_COLOR: Facet := ffHighlight
        else { CMD_SHADOW_COLOR }
          Facet := ffShadow;
        end;
        Item := TcsFrameItem(APoint.X);
        Idx := APoint.Y;
        Handled := False;
        GetFrameColor(Item, Facet, Idx, ItemColor, Handled);
        if not Handled then { use appropriate property values }
          case Cmd of
            CMD_BORDER_COLOR: ItemColor := FFrameBorderColor;
            CMD_HIGHLIGHT_COLOR: ItemColor := FFrameHighlightColor;
          else { CMD_SHADOW_COLOR }
            ItemColor := FFrameShadowColor;
          end;
        case Cmd of
          CMD_BORDER_COLOR: BorderColor := ItemColor;
          CMD_HIGHLIGHT_COLOR: HighlightColor := ItemColor;
        else { CMD_SHADOW_COLOR }
          ShadowColor := ItemColor;
        end;
      end;
      CMD_CTL3D1:
      begin
        if not (Ctl3D and (FBevelWidth > 0) and
          (((FTabStyle = tsTabSet) and Current) or
            (FTabStyle <> tsTabSet))) then
          Break;
      end;
      CMD_CTL3D2:
      begin
        if not (Ctl3D and (FBevelWidth > 1) and
          ((FTabStyle = tsTabbedNotebook) and Current)) then
          Break;
      end;
      CMD_MOVETO:
        FBuffer.Canvas.MoveTo(APoint.X, APoint.Y);
      CMD_LINETO:
        FBuffer.Canvas.LineTo(APoint.X, APoint.Y);
    end;
  end;
end;

procedure TcsChameleonTabControl.SetTabIndex(Value: Integer);
var
  OldTabIndex: Integer;
  Scrolled, AllowChange: Boolean;
  TabData: TcsTabData;
{$IFDEF VER130}
  Form: TCustomForm;
{$ELSE}
{$IFDEF VER120}
  Form: TCustomForm;
{$ELSE}
{$IFDEF VER100}
  Form: TCustomForm;
{$ELSE}
  Form: TForm;
{$ENDIF}
{$ENDIF}
{$ENDIF}
begin
  Scrolled := False;
  if (csLoading in ComponentState) or (csReading in ComponentState) then
  begin { tabs not loaded yet }
    FTabIndex := Value;
    Exit;
  end;
  if (Value < 0) or (Value >= FTabDataList.Count) then
  begin
    FTabIndex := -1;
    Invalidate;
  end
  else
  begin
    OldTabIndex := FTabIndex;
    { check that it is OK to change to the requested tab }
    AllowChange := True;
    if (FTabIndex <> Value) and not FChangingDone then
      AllowChange := CanSelectTab(Value);
    if AllowChange then
    begin
      FTabIndex := Value;
      if FScrollBtnsNeeded then
        if FTabIndex < FFirstInView then
        begin
          FFirstInView := FTabIndex;
          Invalidate;
        end
        else
        begin
          { check if necessary to 'scroll' so new tab is _fully_ in view }
          while (FFirstInView < FTabIndex) do
          begin
            TabData := FTabDataList[FTabIndex];
            if not TabInView(TabData) then
            begin
              Inc(FFirstInView);
              Scrolled := True;
            end
            else
              Break;
          end;
        end
      else
      begin
        TabData := FTabDataList[FTabIndex];
        BringTabToFrontRow(TabData);
      end;
    end;
    if Scrolled or (FTabIndex <> OldTabIndex) then
    begin
      Invalidate;
      if (FTabIndex <> OldTabIndex) then
      begin
        Change;
        if (csDesigning in ComponentState) and FDoneTabIndexDefault then
        begin
          Form := GetParentForm(Self);
          if (Form <> nil) and (Form.Designer <> nil) then
            Form.Designer.Modified;
        end;
      end;
    end;
  end;
end;

function TcsChameleonTabControl.CalcMapRect(const RawRect: TRect): TRect;
var
  NewOrigin, NewCorner: TPoint;
begin
  NewOrigin := CalcMapPoint(Point(RawRect.Left, RawRect.Top));
  NewCorner := CalcMapPoint(Point(RawRect.Right, RawRect.Bottom));
  Result.Left := NewOrigin.X;
  Result.Top := NewOrigin.Y;
  Result.Right := NewCorner.X;
  Result.Bottom := NewCorner.Y;
  OrderRectCorners(Result);
end;

{ Map the point from raw index rect coordinates to actual
  index rect coordinates. }
function TcsChameleonTabControl.CalcMapPoint(const RawPt: TPoint): TPoint;
var
  IndexRect: TRect;
  Offset: Integer;
begin
  Offset := GetInitialTabOffset;
  IndexRect := GetIndexRect;
  if FTabSequence = tsStandard then
    case FTabOrientation of
      toTop:
        begin
          Result.X := IndexRect.Left + (RawPt.X - Offset);
          Result.Y := IndexRect.Top + RawPt.Y;
        end;
      toBottom:
        begin
          Result.X := IndexRect.Left + (RawPt.X - Offset);
          Result.Y := IndexRect.Bottom - RawPt.Y;
        end;
      toLeft:
        begin
          Result.X := IndexRect.Left + RawPt.Y;
          Result.Y := IndexRect.Bottom - (RawPt.X - Offset);
        end;
      toRight:
        begin
          Result.X := IndexRect.Right - RawPt.Y;
          Result.Y := IndexRect.Top + (RawPt.X - Offset);
        end;
    end
    else { tsReverse }
    case FTabOrientation of
      toTop:
        begin
          Result.X := IndexRect.Right - (Offset - RawPt.X);
          Result.Y := IndexRect.Top + RawPt.Y;
        end;
      toBottom:
        begin
          Result.X := IndexRect.Right - (Offset - RawPt.X);
          Result.Y := IndexRect.Bottom - RawPt.Y;
        end;
      toLeft:
        begin
          Result.X := IndexRect.Left + RawPt.Y;
          Result.Y := IndexRect.Top + (Offset - RawPt.X);
        end;
      toRight:
        begin
          Result.X := IndexRect.Right - RawPt.Y;
          Result.Y := IndexRect.Bottom - (Offset - RawPt.X);
        end;
    end;
end;

function TcsChameleonTabControl.TabAtPos(X, Y: Integer): Integer;
var
  Pt: TPoint;
  I: Integer;
  Found: Boolean;

  { Return true if the specified point is inside the boundaries of the
    specified tab.
  }
  function PointInTab(APoint: TPoint; ATabIndex: Integer): Boolean;
  var
    TabData: TcsTabData;
    R: TRect;
    Region: hRgn;
  begin
    Result := False;
    TabData := FTabDataList[ATabIndex];
    if TabInView(TabData) then
    begin
      R := CalcMappedTabRect(ATabIndex, TabData);
      { First do a quick (but inexact) test using PointInRect, and if that
        succeeds then use an exact (but slower) PtInRegion test.
      }
      if PointInRect(R, APoint) then { now do exact test using region }
      begin
        Region := CalcTabRegion(ATabIndex, R);
        if (Region <> 0) and PtInRegion(Region, APoint.X, APoint.Y) then
          Result := True; { point is over specified tab }
      end;
    end;
  end;

begin
  Pt := Point(X, Y);
  Result := -1;
  if (FTabDataList.Count = 0) or (FFirstInView < 0) then Exit;
  Found := False;
  I := FTabIndex;
  { the current tab is 'in front of' all other tabs and so is checked first }
  if I >= 0 then
    Found := PointInTab(Pt, I);
  if not Found then
  begin
    { each tab (apart from current tab which has already been checked) --
      starting from the first in view -- is 'in front of' the following tab and
      so is checked in that order
    }
    I := FFirstInView;
    repeat
      if (I <> FTabIndex) then { this tab hasn't been checked yet }
        Found := PointInTab(Pt, I);
      if not Found then
        if I < FTabDataList.Count - 1 then
          Inc(I)
        else
          I := 0;
    until Found or (I = FFirstInView);
  end;

  if Found then
    Result := I;
end;

{ Select the tab on which the mouse pointer was clicked. }
procedure TcsChameleonTabControl.MouseDown(Button: TMouseButton;
  Shift: TShiftState; X, Y: Integer);
var
  NewTabIndex: Integer;
begin
  { In design-mode MouseDown is called after CMDesignHitTest,
    which will have already taken care of selecting the tab.
  }
  if (csDesigning in ComponentState) then Exit;

  inherited MouseDown(Button, Shift, X, Y);

  if (Button = mbLeft) and Enabled then
  begin
    NewTabIndex := TabAtPos(X, Y);

    if (NewTabIndex >= 0) and FTabDataList[NewTabIndex].Enabled then
    begin
      { only set focus to self if it has TabStop and the tab clicked
        is already the current one
      }
      if TabStop and (NewTabIndex = FTabIndex) then
      begin
        SetFocus; { to self }
        if not Focused then Exit; { OnExit event handlers have redirected focus }
      end;
      SetTabIndex(NewTabIndex);
      if FTabIndex = NewTabIndex then
        { tab change was allowed }
        TabClick;
    end;
  end;
end;

procedure TcsChameleonTabControl.MouseMove(Shift: TShiftState; X, Y: Integer);
var
  Idx: Integer;
  NewHint, SavedHint: String;
  IndexRect: TRect;
  InIndexRect: Boolean;
begin
  if (not (csDesigning in ComponentState)) and (FHotTrack or FTabHints) then
  begin
    Idx := TabAtPos(X, Y);
    if (Idx >= 0) then { mouse cursor is over a tab }
    begin
      if Idx <> FHotTrackIndex then { mouse cursor has been moved to different tab }
      begin
        FHotTrackIndex := Idx;
        if FHotTrack then { redraw tabs to indicate hot-tracked tab }
        begin
          IndexRect := GetIndexRect;
          InvalidateRect(Handle, @IndexRect, False);
        end;
        if FTabHints then { change control's Hint to hint for tab under mouse cursor }
        begin
          Application.CancelHint; { so hint will change even though we are on same control }
          NewHint := FTabDataList[Idx].Hint;
          if (NewHint <> '') then { replace control's Hint with tab's Hint }
            Hint := NewHint;
        end;
        { Create timer with minimum interval (max. freq is 18Hz ~= 60ms)
          so we can detect if the mouse cursor is moved quickly off of the
          tabs and ends up outside the control's bounds (and thus no longer
          sending mouse move messages to the control).
          This timer is used for both hot-tracking of tab captions and when
          doing hints for each tab.
        }
        if FTimerHandle = 0 then
          FTimerHandle := SetTimer(Handle, 0, 60, nil);
      end;
    end
    else { Idx < 0 }
    begin
      { mouse cursor may be either over the card area of the control, in which
        case we want to use the original hint value -- StopHotTracking takes
        care of this case -- or, over the index area but not over a tab, in
        which case we want to set the Hint to blank to stop if showing
      }
      IndexRect := GetIndexRect; { area in which tabs are displayed }
      InIndexRect := PointInRect(IndexRect, Point(X, Y));
      if FTabHints then
      begin
        { cancel hint if we were previously over a tab or are over the
          index area (but not over a tab)
        }
        if (FHotTrackIndex >= 0) or InIndexRect then
          Application.CancelHint;
        if InIndexRect then
        begin
          { mouse cursor is in index area but not over a tab; temporarily change
            FOriginalHint to '' before calling StopHotTracking (which uses
            the value of FOriginalHint) and then restore its value afterwards
            so that if the user subsequently moves into the card area of the
            control (but is still not over a tab) we will end up here again and
            StopHotTracking will be called and will use the FOriginalHint value
            we restored previously
          }
          SavedHint := FOriginalHint;
          FOriginalHint := '';
        end;
      end;
      StopHotTracking;
      if FTabHints and InIndexRect then { restore FOriginalHint }
        FOriginalHint := SavedHint;
    end;
  end;
  inherited MouseMove(Shift, X, Y);
end;

procedure TcsChameleonTabControl.StopHotTracking;
var
  IndexRect: TRect;
begin
  CancelHotTrackTimer;
  FHotTrackIndex := -1;
  IndexRect := GetIndexRect;
  InvalidateRect(Handle, @IndexRect, False);
  Hint := FOriginalHint;
end;

procedure TcsChameleonTabControl.CancelHotTrackTimer;
begin
  if FTimerHandle <> 0 then
  begin
    KillTimer(0, FTimerHandle);
    FTimerHandle := 0;
  end;
end;

procedure TcsChameleonTabControl.WMTimer(var Msg: TWMTimer);
var
  Point: TPoint;
  Idx: Integer;
begin
  { check for left-over messages generated prior to timer being killed }
  if FTimerHandle = 0 then Exit;
  { check if still over a tab }
  GetCursorPos(Point);
  Point := ScreenToClient(Point);
  Idx := TabAtPos(Point.X , Point.Y);
  if (Idx = -1) then { no longer over a tab }
    StopHotTracking;
end;

{ Return the (unmapped) rect for the tab with allowance for row indentation
  and/or enlarged current tab.
  Use CalcMappedTabRect if the mapped tab rect is required.
}
function TcsChameleonTabControl.CalcTabRect(ATabIndex: Integer;
  TabData: TcsTabData): TRect;
var
  R: TRect;
begin
  R := TabData.RawRect;
  if TabData.Row > 0 then { multiple rows of tabs }
    { add amount of row indent appropriate for row no. }
    if FTabSequence = tsStandard then
      OffsetRect(R, TabData.Row*FRowIndent, 0)
    else
      OffsetRect(R, -TabData.Row*FRowIndent, 0);
  if (FTabStyle = tsTabControl) and (ATabIndex = FTabIndex) then
  begin
    { Tab Control (Win95) style tabs have enlarged current tab }
    Dec(R.Left, 2);
    Inc(R.Right, 2);
    Dec(R.Top, 2);
  end;
  Result := R;
end;

function TcsChameleonTabControl.CalcMappedTabRect(ATabIndex: Integer;
  TabData: TcsTabData): TRect;
var
  R: TRect;
begin
  R := CalcTabRect(ATabIndex, TabData);
  Result := CalcMapRect(R);
end;

{ Return the region for the specified tab by seeing if it already exists in
  the region cache or otherwise creating it.
}
function TcsChameleonTabControl.CalcTabRegion(ATabIndex: Integer;
  const ARect: TRect): hRgn;
var
  R: TRect;
  Region: hRgn;
  Pts: TcsTabRegionPts;
  NumPts: Integer;
begin
  Result := 0;
  R := ARect;
  Region := FTabRegionCache.Find(R);
  if Region = 0 then { not in cache }
  begin
    CalcMappedTabRegionPts(ATabIndex, {R,} Pts, NumPts);
    if NumPts >= 3 then
    begin
      Region := CreatePolygonRgn(Pts, NumPts, PolyFillMode);
      if Region = 0 then
        raise Exception.Create('Couldn''t create region');
      { add new item to cache }
      FTabRegionCache.Add(R, Region);
      Result := Region;
    end;
  end
  else
    Result := Region;
end;

procedure TcsChameleonTabControl.CalcTabRegionPts(ATabIndex: Integer;
  var Pts: TcsTabRegionPts; var NumPts: Integer);
var
  L, T, R, B, H: Integer;
  Offset: Integer;
  RR: TRect;
  TabData: TcsTabData;

begin { CalcTabRegionPts }
  NumPts := 0;
  TabData := FTabDataList[ATabIndex];
  RR := CalcTabRect(ATabIndex, TabData);
  L := RR.Left;
  T := RR.Top;
  R := RR.Right;
  B := RR.Bottom;
  H := B - T;
  { now define points for raw rect }
  case FTabStyle of
    tsTabControl:
      begin
        NumPts := 6;
        Pts[0] := Point(L, B);
        Pts[1] := Point(L, T + 2);
        Pts[2] := Point(L + 2, T);
        Pts[3] := Point(R - 2, T);
        Pts[4] := Point(R, T + 2);
        Pts[5] := Point(R, B);
      end;
    tsTabSet:
      begin
        NumPts := 4;
        Offset := GetTabOffset(H);
        Pts[0] := Point(L, B);
        Pts[1] := Point(L + Offset, T);
        Pts[2] := Point(R - Offset, T);
        Pts[3] := Point(R, B);
      end;
    tsTabbedNotebook:
      begin
        NumPts := 6;
        Offset := FCornerSize;
        Pts[0] := Point(L, B);
        Pts[1] := Point(L, T + Offset);
        Pts[2] := Point(L + Offset, T);
        Pts[3] := Point(R - Offset, T);
        Pts[4] := Point(R, T + Offset);
        Pts[5] := Point(R, B);
      end;
  end;
end;

procedure TcsChameleonTabControl.CalcMappedTabRegionPts(ATabIndex: Integer;
  var Pts: TcsTabRegionPts; var NumPts: Integer);
var
  I: Integer;
begin { CalcMappedTabRegionPts }
  CalcTabRegionPts(ATabIndex, Pts, NumPts);
  { map points to actual orientation }
  for I := 0 to NumPts - 1 do
    Pts[I] := CalcMapPoint(Pts[I]);
end;

{ Build a list of commands which can be used to draw the specified tab.
  Commands like CMD_BORDER, CMD_HIGHLIGHT, etc. will result in the
  appropriate color being selected.
}
procedure TcsChameleonTabControl.CalcTabDrawCommands(ATabIndex: Integer);
var
  Pts: TcsTabRegionPts;
  MappedPts: Array[Low(TcsTabRegionPts)..High(TcsTabRegionPts)] of TPoint;
  I, NumPts: Integer;
  APoint: TPoint;

  procedure GenerateTabbedNotebookExtraDiagonalCommands;
  var
    I: Integer;
  begin
    for I := 1 to 4 do
      MappedPts[I] := CalcMapPoint(Pts[I]);
    case FTabOrientation of
      toTop:
        begin
          AddCommandPt(CMD_MOVETO, MappedPts[1]);
          AddCommand(CMD_HIGHLIGHT);
          AddCommandPt(CMD_LINETO, MappedPts[2]);
          AddCommandPt(CMD_MOVETO, MappedPts[3]);
          AddCommand(CMD_SHADOW);
          AddCommandPt(CMD_LINETO, MappedPts[4]);
        end;
      toBottom:
        begin
          AddCommandPt(CMD_MOVETO, MappedPts[1]);
          AddCommand(CMD_SHADOW);
          AddCommandPt(CMD_LINETO, MappedPts[2]);
          AddCommandPt(CMD_MOVETO, MappedPts[3]);
          AddCommandPt(CMD_LINETO, MappedPts[4]);
        end;
      toLeft:
        begin
          AddCommandPt(CMD_MOVETO, MappedPts[1]);
          AddCommand(CMD_SHADOW);
          AddCommandPt(CMD_LINETO, MappedPts[2]);
          AddCommandPt(CMD_MOVETO, MappedPts[3]);
          AddCommand(CMD_HIGHLIGHT);
          AddCommandPt(CMD_LINETO, MappedPts[4]);
        end;
      toRight:
        begin
          AddCommandPt(CMD_MOVETO, MappedPts[1]);
          AddCommand(CMD_SHADOW);
          AddCommandPt(CMD_LINETO, MappedPts[2]);
          AddCommandPt(CMD_MOVETO, MappedPts[3]);
          AddCommandPt(CMD_LINETO, MappedPts[4]);
        end;
    end;
  end; { GenerateTabbedNotebookExtraDiagonalCommands }

  procedure GenerateTabbedNotebookBevelCommands;
  var
    I: Integer;
  begin
    for I := 0 to 5 do
      MappedPts[I] := CalcMapPoint(Pts[I]);
    case FTabOrientation of
      toTop:
        begin
          AddCommandPt(CMD_MOVETO, MappedPts[0]);
          AddCommand(CMD_HIGHLIGHT);
          AddCommandPt(CMD_LINETO, MappedPts[1]);
          AddCommandPt(CMD_LINETO, MappedPts[2]);
          AddCommandPt(CMD_LINETO, MappedPts[3]);
          AddCommand(CMD_SHADOW);
          AddCommandPt(CMD_LINETO, MappedPts[4]);
          AddCommandPt(CMD_LINETO, MappedPts[5]);
        end;
      toBottom:
        begin
          AddCommandPt(CMD_MOVETO, MappedPts[0]);
          AddCommand(CMD_HIGHLIGHT);
          AddCommandPt(CMD_LINETO, MappedPts[1]);
          AddCommand(CMD_SHADOW);
          AddCommandPt(CMD_LINETO, MappedPts[2]);
          AddCommandPt(CMD_LINETO, MappedPts[3]);
          AddCommandPt(CMD_LINETO, MappedPts[4]);
          AddCommandPt(CMD_LINETO, MappedPts[5]);
        end;
      toLeft:
        begin
          AddCommandPt(CMD_MOVETO, MappedPts[0]);
          AddCommand(CMD_SHADOW);
          AddCommandPt(CMD_LINETO, MappedPts[1]);
          AddCommandPt(CMD_LINETO, MappedPts[2]);
          AddCommand(CMD_HIGHLIGHT);
          AddCommandPt(CMD_LINETO, MappedPts[3]);
          AddCommandPt(CMD_LINETO, MappedPts[4]);
          AddCommandPt(CMD_LINETO, MappedPts[5]);
        end;
      toRight:
        begin
          AddCommandPt(CMD_MOVETO, MappedPts[0]);
          AddCommand(CMD_HIGHLIGHT);
          AddCommandPt(CMD_LINETO, MappedPts[1]);
          AddCommand(CMD_SHADOW);
          AddCommandPt(CMD_LINETO, MappedPts[2]);
          AddCommandPt(CMD_LINETO, MappedPts[3]);
          AddCommandPt(CMD_LINETO, MappedPts[4]);
          AddCommandPt(CMD_LINETO, MappedPts[5]);
        end;
    end;
  end; { GenerateTabbedNotebookBevelCommands }

begin { CalcTabDrawCommands }
  { get tab points for unmapped tab outline }
  CalcTabRegionPts(ATabIndex, Pts, NumPts);
  { build list of commands needed to draw the tab shape }

  case FTabStyle of

    tsTabControl:
      begin
        if NumPts <> 6 then raise Exception.Create('wrong no. of points');
        for I := 0 to 5 do
          MappedPts[I] := CalcMapPoint(Pts[I]);
        case FTabOrientation of
          { Non-intuitive order of drawing line segments is so that appropriate
            segments are 'on top of' other segments and appropriate end points
            are included so the control matches the look of the native Windows
            tab control.
            Also note that some of the 3D highlighting is not strictly
            realistic (i.e. highlight used on some segments which would be
            shadowed) but has been changed to give the tabs a better appearance
            in bottom and right orientations.
          }
          toTop:
            begin
              AddCommand(CMD_BORDER);
              AddCommandPt(CMD_MOVETO, MappedPts[4]);
              AddCommandPt(CMD_LINETO, MappedPts[5]);
              AddCommandPt(CMD_MOVETO, MappedPts[4]);
              AddCommandPt(CMD_LINETO, MappedPts[3]);
              APoint := Point(Pts[4].X - 1, Pts[4].Y);
              AddCommandPt(CMD_MOVETO, CalcMapPoint(APoint));
              APoint := Point(Pts[5].X - 1, Pts[5].Y);
              AddCommand(CMD_SHADOW);
              AddCommandPt(CMD_LINETO, CalcMapPoint(APoint));
              AddCommand(CMD_HIGHLIGHT);
              AddCommandPt(CMD_MOVETO, MappedPts[3]);
              AddCommandPt(CMD_LINETO, MappedPts[2]);
              AddCommandPt(CMD_LINETO, MappedPts[1]);
              AddCommandPt(CMD_LINETO, MappedPts[0]);
            end;
          toBottom:
            begin
              AddCommand(CMD_BORDER);
              AddCommandPt(CMD_MOVETO, MappedPts[4]);
              AddCommandPt(CMD_LINETO, MappedPts[5]);
              AddCommandPt(CMD_MOVETO, MappedPts[4]);
              AddCommandPt(CMD_LINETO, MappedPts[3]);
              APoint := Point(Pts[4].X - 1, Pts[4].Y);
              AddCommandPt(CMD_MOVETO, CalcMapPoint(APoint));
              APoint := Point(Pts[5].X - 1, Pts[5].Y);
              AddCommand(CMD_SHADOW);
              AddCommandPt(CMD_LINETO, CalcMapPoint(APoint));
              AddCommandPt(CMD_MOVETO, MappedPts[3]);
              AddCommandPt(CMD_LINETO, MappedPts[2]);
              AddCommand(CMD_HIGHLIGHT);
              AddCommandPt(CMD_LINETO, MappedPts[1]);
              AddCommandPt(CMD_LINETO, MappedPts[0]);
            end;
          toLeft:
            begin
              AddCommand(CMD_BORDER);
              AddCommandPt(CMD_MOVETO, MappedPts[1]);
              AddCommandPt(CMD_LINETO, MappedPts[0]);
              AddCommandPt(CMD_MOVETO, MappedPts[1]);
              AddCommandPt(CMD_LINETO, MappedPts[2]);
              APoint := Point(Pts[1].X + 1, Pts[1].Y);
              AddCommandPt(CMD_MOVETO, CalcMapPoint(APoint));
              APoint := Point(Pts[0].X + 1, Pts[0].Y);
              AddCommand(CMD_SHADOW);
              AddCommandPt(CMD_LINETO, CalcMapPoint(APoint));
              AddCommandPt(CMD_MOVETO, MappedPts[2]);
              AddCommand(CMD_HIGHLIGHT);
              AddCommandPt(CMD_LINETO, MappedPts[3]);
              AddCommandPt(CMD_LINETO, MappedPts[4]);
              AddCommandPt(CMD_LINETO, MappedPts[5]);
            end;
          toRight:
            begin
              AddCommand(CMD_BORDER);
              AddCommandPt(CMD_MOVETO, MappedPts[4]);
              AddCommandPt(CMD_LINETO, MappedPts[5]);
              AddCommandPt(CMD_MOVETO, MappedPts[4]);
              AddCommandPt(CMD_LINETO, MappedPts[3]);
              APoint := Point(Pts[4].X - 1, Pts[4].Y);
              AddCommandPt(CMD_MOVETO, CalcMapPoint(APoint));
              APoint := Point(Pts[5].X - 1, Pts[5].Y);
              AddCommand(CMD_SHADOW);
              AddCommandPt(CMD_LINETO, CalcMapPoint(APoint));
              AddCommandPt(CMD_MOVETO, MappedPts[3]);
              AddCommandPt(CMD_LINETO, MappedPts[2]);
              AddCommand(CMD_HIGHLIGHT);
              AddCommandPt(CMD_LINETO, MappedPts[1]);
              AddCommandPt(CMD_LINETO, MappedPts[0]);
            end;
        end;
      end;

    tsTabSet:
      begin
        if NumPts <> 4 then raise Exception.Create('wrong no. of points');
        AddCommand(CMD_BORDER);
        AddCommandPt(CMD_MOVETO, CalcMapPoint(Pts[0]));
        AddCommandPt(CMD_LINETO, CalcMapPoint(Pts[1]));
        AddCommandPt(CMD_LINETO, CalcMapPoint(Pts[2]));
        AddCommandPt(CMD_LINETO, CalcMapPoint(Pts[3]));
        AddCommand(CMD_CTL3D1); { start of commands only used when Ctl3D on }
        { adjust unmapped points for bevel }
        Pts[0] := Point(Pts[0].X + 1, Pts[0].Y);
        Pts[1] := Point(Pts[1].X + 1, Pts[1].Y + 1);
        Pts[2] := Point(Pts[2].X - 1, Pts[2].Y + 1);
        Pts[3] := Point(Pts[3].X - 1, Pts[3].Y);
        case FTabOrientation of
          toTop, toBottom:
          begin
            AddCommand(CMD_HIGHLIGHT);
            AddCommandPt(CMD_MOVETO, CalcMapPoint(Pts[0]));
            { draw to Pts[1].Y - 1 instead of Pts[1].Y to get same angle as border }
            AddCommandPt(CMD_LINETO, CalcMapPoint(Point(Pts[1].X, Pts[1].Y - 1)));
            AddCommand(CMD_SHADOW);
            AddCommandPt(CMD_MOVETO, CalcMapPoint(Pts[3]));
            { draw to Pts[2].Y - 1 instead of Pts[2].Y to get same angle as border }
            AddCommandPt(CMD_LINETO, CalcMapPoint(Point(Pts[2].X, Pts[2].Y - 1)));
            AddCommand(CMD_HIGHLIGHT);
            AddCommandPt(CMD_MOVETO, CalcMapPoint(Pts[1]));
            AddCommandPt(CMD_LINETO, CalcMapPoint(Pts[2]));
          end;
          toLeft:
          begin
            AddCommand(CMD_SHADOW);
            AddCommandPt(CMD_MOVETO, CalcMapPoint(Pts[0]));
            { draw to Pts[1].Y - 1 instead of Pts[1].Y to get same angle as border }
            AddCommandPt(CMD_LINETO, CalcMapPoint(Point(Pts[1].X, Pts[1].Y - 1)));
            AddCommand(CMD_HIGHLIGHT);
            AddCommandPt(CMD_MOVETO, CalcMapPoint(Pts[3]));
            { draw to Pts[2].Y - 1 instead of Pts[2].Y to get same angle as border }
            AddCommandPt(CMD_LINETO, CalcMapPoint(Point(Pts[2].X, Pts[2].Y - 1)));
            AddCommand(CMD_SHADOW);
            AddCommandPt(CMD_MOVETO, CalcMapPoint(Pts[1]));
            AddCommandPt(CMD_LINETO, CalcMapPoint(Pts[2]));
          end;
          toRight:
          begin
            AddCommand(CMD_HIGHLIGHT);
            AddCommandPt(CMD_MOVETO, CalcMapPoint(Pts[0]));
            { draw to Pts[1].Y - 1 instead of Pts[1].Y to get same angle as border }
            AddCommandPt(CMD_LINETO, CalcMapPoint(Point(Pts[1].X, Pts[1].Y - 1)));
            AddCommand(CMD_SHADOW);
            AddCommandPt(CMD_MOVETO, CalcMapPoint(Pts[3]));
            { draw to Pts[2].Y - 1 instead of Pts[2].Y to get same angle as border }
            AddCommandPt(CMD_LINETO, CalcMapPoint(Point(Pts[2].X, Pts[2].Y - 1)));
            AddCommand(CMD_SHADOW);
            AddCommandPt(CMD_MOVETO, CalcMapPoint(Pts[1]));
            AddCommandPt(CMD_LINETO, CalcMapPoint(Pts[2]));
          end;
        end;
      end;

    tsTabbedNotebook:
      begin
        if NumPts <> 6 then raise Exception.Create('wrong no. of points');
        AddCommand(CMD_BORDER);
        AddCommandPt(CMD_MOVETO, CalcMapPoint(Pts[0]));
        AddCommandPt(CMD_LINETO, CalcMapPoint(Pts[1]));
        AddCommandPt(CMD_LINETO, CalcMapPoint(Pts[2]));
        AddCommandPt(CMD_LINETO, CalcMapPoint(Pts[3]));
        AddCommandPt(CMD_LINETO, CalcMapPoint(Pts[4]));
        AddCommandPt(CMD_LINETO, CalcMapPoint(Pts[5]));
        for I := 1 to 2 do
        begin
          { define unmapped points for bevel }
          if I = 1 then
            AddCommand(CMD_CTL3D1)
          else
            AddCommand(CMD_CTL3D2);
          Pts[0] := Point(Pts[0].X + 1, Pts[0].Y);
          Pts[1] := Point(Pts[1].X + 1, Pts[1].Y);
          Pts[2] := Point(Pts[2].X, Pts[2].Y + 1);
          Pts[3] := Point(Pts[3].X, Pts[3].Y + 1);
          Pts[4] := Point(Pts[4].X - 1, Pts[4].Y);
          Pts[5] := Point(Pts[5].X - 1, Pts[5].Y);
          GenerateTabbedNotebookBevelCommands;
        end;
        { adjust points for corners and include draw commands for extra
          diagonal lines so bevel thickness appears uniform on corners
        }
        Pts[1] := Point(Pts[1].X, Pts[1].Y + 1);
        Pts[2] := Point(Pts[2].X + 1, Pts[2].Y);
        Pts[3] := Point(Pts[3].X - 1, Pts[3].Y);
        Pts[4] := Point(Pts[4].X, Pts[4].Y + 1);
        GenerateTabbedNotebookExtraDiagonalCommands;
      end;
  end;
end;

{ Build a list of commands which can be used to draw the card for the
  specified row of tabs.
  Commands like CMD_BORDER, CMD_HIGHLIGHT, etc. will result in the
  appropriate color being selected.
}
procedure TcsChameleonTabControl.CalcCardDrawCommands(ARow: Integer);
var
  TabData: TcsTabData;
  CurrentTabInView: Boolean;
  FirstRawPt, LastRawPt, FirstPt, LastPt: TPoint;
  CardRect: TRect;
  Pts: TcsTabRegionPts;
  I, NumPts: Integer;

  procedure GenerateContinuousBorderCommands;
  begin
    AddCommand(CMD_BORDER);
    AddCommandPt(CMD_MOVETO, Pts[0]);
    AddCommandPt(CMD_LINETO, Pts[1]);
    AddCommandPt(CMD_LINETO, Pts[2]);
    AddCommandPt(CMD_LINETO, Pts[3]);
    AddCommandPt(CMD_LINETO, Pts[0]);
  end;

  procedure GenerateBrokenBorderCommands;
  begin
    case FTabOrientation of
      toTop:
        begin
          AddCommand(CMD_BORDER);
          AddCommandPt(CMD_MOVETO, Pts[0]);
          AddCommandPt(CMD_LINETO, FirstPt);
          AddCommandPt(CMD_MOVETO, LastPt);
          AddCommandPt(CMD_LINETO, Pts[1]);
          AddCommandPt(CMD_LINETO, Pts[2]);
          AddCommandPt(CMD_LINETO, Pts[3]);
          AddCommandPt(CMD_LINETO, Point(Pts[0].X, Pts[0].Y - 1));
        end;
      toBottom:
        begin
          AddCommand(CMD_BORDER);
          AddCommandPt(CMD_MOVETO, Pts[0]);
          AddCommandPt(CMD_LINETO, Pts[1]);
          AddCommandPt(CMD_LINETO, Point(Pts[2].X, Pts[2].Y + 1));
          AddCommandPt(CMD_MOVETO, Pts[2]);
          AddCommandPt(CMD_LINETO, Point(LastPt.X - 1, LastPt.Y));
          AddCommandPt(CMD_MOVETO, Point(FirstPt.X - 1, LastPt.Y));
          AddCommandPt(CMD_LINETO, Pts[3]);
          AddCommandPt(CMD_LINETO, Pts[0]);
        end;
      toLeft:
        begin
          AddCommand(CMD_BORDER);
          AddCommandPt(CMD_MOVETO, Pts[0]);
          AddCommandPt(CMD_LINETO, Pts[1]);
          AddCommandPt(CMD_LINETO, Pts[2]);
          AddCommandPt(CMD_LINETO, Point(Pts[3].X - 1, Pts[3].Y));
          AddCommandPt(CMD_MOVETO, Pts[3]);
          AddCommandPt(CMD_LINETO, FirstPt);
          AddCommandPt(CMD_MOVETO, LastPt);
          AddCommandPt(CMD_LINETO, Pts[0]);
        end;
      toRight:
        begin
          AddCommand(CMD_BORDER);
          AddCommandPt(CMD_MOVETO, Pts[0]);
          AddCommandPt(CMD_LINETO, Point(Pts[1].X + 1, Pts[1].Y));
          AddCommandPt(CMD_MOVETO, Pts[1]);
          AddCommandPt(CMD_LINETO, FirstPt);
          AddCommandPt(CMD_MOVETO, LastPt);
          AddCommandPt(CMD_LINETO, Pts[2]);
          AddCommandPt(CMD_LINETO, Pts[3]);
          AddCommandPt(CMD_LINETO, Pts[0]);
        end;
    end;
  end; { GenerateBrokenBorderCommands }

  procedure GenerateContinuousBevelCommands;
  begin
    AddCommandPt(CMD_MOVETO, Pts[0]);
    AddCommand(CMD_HIGHLIGHT);
    AddCommandPt(CMD_LINETO, Pts[1]);
    AddCommand(CMD_SHADOW);
    AddCommandPt(CMD_LINETO, Pts[2]);
    AddCommandPt(CMD_LINETO, Pts[3]);
    AddCommand(CMD_HIGHLIGHT);
    AddCommandPt(CMD_LINETO, Pts[0]);
  end; { GenerateContinuousBevelCommands }

  procedure GenerateBrokenBevelCommands;
  begin
    case FTabOrientation of
      toTop:
        begin
          AddCommandPt(CMD_MOVETO, Pts[0]);
          AddCommand(CMD_HIGHLIGHT);
          AddCommandPt(CMD_LINETO, FirstPt);
          if not (LastPt.X > Pts[1].X) then
          begin
            AddCommandPt(CMD_MOVETO, LastPt);
            AddCommandPt(CMD_LINETO, Pts[1]);
          end;
          AddCommandPt(CMD_MOVETO, Pts[1]);
          AddCommand(CMD_SHADOW);
          AddCommandPt(CMD_LINETO, Pts[2]);
          AddCommandPt(CMD_LINETO, Pts[3]);
          AddCommand(CMD_HIGHLIGHT);
          AddCommandPt(CMD_LINETO, Pts[0]);
        end;
      toBottom:
        begin
          AddCommandPt(CMD_MOVETO, Pts[0]);
          AddCommand(CMD_HIGHLIGHT);
          AddCommandPt(CMD_LINETO, Pts[1]);
          AddCommand(CMD_SHADOW);
          AddCommandPt(CMD_LINETO, Pts[2]);
          AddCommandPt(CMD_LINETO, LastPt);
          AddCommandPt(CMD_MOVETO, FirstPt);
          AddCommandPt(CMD_LINETO, Pts[3]);
          AddCommand(CMD_HIGHLIGHT);
          AddCommandPt(CMD_LINETO, Pts[0]);
        end;
      toLeft:
        begin
          AddCommandPt(CMD_MOVETO, Pts[0]);
          AddCommand(CMD_HIGHLIGHT);
          AddCommandPt(CMD_LINETO, Pts[1]);
          AddCommand(CMD_SHADOW);
          AddCommandPt(CMD_LINETO, Pts[2]);
          AddCommandPt(CMD_LINETO, Pts[3]);
          AddCommand(CMD_HIGHLIGHT);
          AddCommandPt(CMD_LINETO, FirstPt);
          AddCommandPt(CMD_MOVETO, LastPt);
          AddCommandPt(CMD_LINETO, Pts[0]);
        end;
      toRight:
        begin
          AddCommandPt(CMD_MOVETO, Pts[0]);
          AddCommand(CMD_HIGHLIGHT);
          AddCommandPt(CMD_LINETO, Pts[1]);
          AddCommand(CMD_SHADOW);
          AddCommandPt(CMD_LINETO, FirstPt);
          AddCommandPt(CMD_MOVETO, LastPt);
          AddCommandPt(CMD_LINETO, Pts[2]);
          AddCommandPt(CMD_LINETO, Pts[3]);
          AddCommand(CMD_HIGHLIGHT);
          AddCommandPt(CMD_LINETO, Pts[0]);
        end;
    end;
  end; { GenerateBrokenBevelCommands }

  procedure GenerateContinuousTCCommands; { Tab Control border/bevel commands }
  begin
    { draw outline in weird order to get same look as true Tab Control }
    AddCommandPt(CMD_MOVETO, Pts[3]);
    AddCommand(CMD_HIGHLIGHT);
    AddCommandPt(CMD_LINETO, Pts[0]);
    AddCommandPt(CMD_LINETO, Pts[1]);
    AddCommand(CMD_BORDER);
    AddCommandPt(CMD_LINETO, Pts[2]);
    AddCommandPt(CMD_LINETO, Point(Pts[3].X - 1, Pts[3].Y));
    Pts[1] := Point(Pts[1].X - 1, Pts[1].Y + 1);
    Pts[2] := Point(Pts[2].X - 1, Pts[2].Y - 1);
    Pts[3] := Point(Pts[3].X, Pts[3].Y - 1);
    { Tab Control only has bevel on right and bottom edges }
    AddCommandPt(CMD_MOVETO, Pts[1]);
    AddCommand(CMD_SHADOW);
    AddCommandPt(CMD_LINETO, Pts[2]);
    AddCommandPt(CMD_LINETO, Pts[3]);
  end; { GenerateContinuousTCCommands }

  procedure GenerateBrokenTCCommands; { Tab Control border/bevel commands }
  begin
    { The reason why some of these drawing commands start from Pts[3] is to
      ensure the final segment drawn overlaps the first segment drawn.
    }
    case FTabOrientation of
      toTop:
        begin
          AddCommandPt(CMD_MOVETO, Pts[3]);
          AddCommand(CMD_HIGHLIGHT);
          AddCommandPt(CMD_LINETO, Pts[0]);
          AddCommandPt(CMD_LINETO, FirstPt);
          AddCommandPt(CMD_MOVETO, LastPt);
          AddCommandPt(CMD_LINETO, Pts[1]);
          AddCommand(CMD_BORDER);
          AddCommandPt(CMD_LINETO, Pts[2]);
          AddCommandPt(CMD_LINETO, Point(Pts[3].X - 1, Pts[3].Y));
          Pts[1] := Point(Pts[1].X - 1, Pts[1].Y + 1);
          Pts[2] := Point(Pts[2].X - 1, Pts[2].Y - 1);
          Pts[3] := Point(Pts[3].X, Pts[3].Y - 1);
          { Tab Control only has bevel on right and bottom edges }
          AddCommandPt(CMD_MOVETO, Pts[1]);
          AddCommand(CMD_SHADOW);
          AddCommandPt(CMD_LINETO, Pts[2]);
          AddCommandPt(CMD_LINETO, Pts[3]);
          { fix up join between current tab and card }
          AddCommand(CMD_SHADOW);
          AddCommandPt(CMD_MOVETO, Point(LastPt.X - 1, LastPt.Y));
          AddCommandPt(CMD_LINETO, Point(LastPt.X, LastPt.Y));
          AddCommand(CMD_BORDER);
          AddCommandPt(CMD_LINETO, Point(LastPt.X + 1, LastPt.Y));
        end;
      toBottom:
        begin
          FirstPt.X := FirstPt.X - 1;
          LastPt.X := LastPt.X - 1;
          AddCommandPt(CMD_MOVETO, Pts[3]);
          AddCommand(CMD_HIGHLIGHT);
          AddCommandPt(CMD_LINETO, Pts[0]);
          AddCommandPt(CMD_LINETO, Pts[1]);
          AddCommand(CMD_BORDER);
          AddCommandPt(CMD_LINETO, Pts[2]);
          AddCommandPt(CMD_LINETO, LastPt);
          AddCommandPt(CMD_MOVETO, FirstPt);
          AddCommandPt(CMD_LINETO, Pts[3]);
          Pts[1] := Point(Pts[1].X - 1, Pts[1].Y + 1);
          Pts[2] := Point(Pts[2].X - 1, Pts[2].Y - 1);
          Pts[3] := Point(Pts[3].X, Pts[3].Y - 1);
          FirstPt.Y := FirstPt.Y - 1;
          LastPt.Y := LastPt.Y - 1;
          { Tab Control only has bevel on right and bottom edges }
          AddCommandPt(CMD_MOVETO, Pts[1]);
          AddCommand(CMD_SHADOW);
          AddCommandPt(CMD_LINETO, Pts[2]);
          AddCommandPt(CMD_LINETO, LastPt);
          AddCommandPt(CMD_MOVETO, FirstPt);
          AddCommandPt(CMD_LINETO, Pts[3]);
          { fix up join between current tab and card }
          AddCommand(CMD_HIGHLIGHT);
          AddCommandPt(CMD_MOVETO, Point(FirstPt.X, LastPt.Y));
          AddCommandPt(CMD_LINETO, Point(FirstPt.X, LastPt.Y + 3));
          AddCommand(CMD_SHADOW);
          AddCommandPt(CMD_MOVETO, Point(LastPt.X, LastPt.Y));
          AddCommandPt(CMD_LINETO, Point(LastPt.X, LastPt.Y + 3));
        end;
      toLeft:
        begin
          AddCommandPt(CMD_MOVETO, Pts[3]);
          AddCommand(CMD_HIGHLIGHT);
          AddCommandPt(CMD_LINETO, FirstPt);
          AddCommandPt(CMD_MOVETO, LastPt);
          AddCommandPt(CMD_LINETO, Pts[0]);
          AddCommandPt(CMD_LINETO, Pts[1]);
          AddCommand(CMD_BORDER);
          AddCommandPt(CMD_LINETO, Pts[2]);
          AddCommandPt(CMD_LINETO, Point(Pts[3].X - 1, Pts[3].Y));
          Pts[1] := Point(Pts[1].X - 1, Pts[1].Y + 1);
          Pts[2] := Point(Pts[2].X - 1, Pts[2].Y - 1);
          Pts[3] := Point(Pts[3].X, Pts[3].Y - 1);
          { Tab Control only has bevel on right and bottom edges }
          AddCommandPt(CMD_MOVETO, Pts[1]);
          AddCommand(CMD_SHADOW);
          AddCommandPt(CMD_LINETO, Pts[2]);
          AddCommandPt(CMD_LINETO, Pts[3]);
          { fix up join between current tab and card }
          AddCommand(CMD_BORDER);
          AddCommandPt(CMD_MOVETO, Point(FirstPt.X, FirstPt.Y + 1));
          AddCommandPt(CMD_LINETO, Point(FirstPt.X, FirstPt.Y));
          AddCommand(CMD_SHADOW);
          AddCommandPt(CMD_LINETO, Point(FirstPt.X, FirstPt.Y - 1));
        end;
      toRight:
        begin
          FirstPt.Y := FirstPt.Y - 1;
          AddCommandPt(CMD_MOVETO, Pts[3]);
          AddCommand(CMD_HIGHLIGHT);
          AddCommandPt(CMD_LINETO, Pts[0]);
          AddCommandPt(CMD_LINETO, Point(Pts[1].X + 1, Pts[1].Y));
          AddCommandPt(CMD_MOVETO, Pts[1]);
          AddCommand(CMD_BORDER);
          AddCommandPt(CMD_LINETO, FirstPt);
          AddCommandPt(CMD_MOVETO, LastPt);
          AddCommandPt(CMD_LINETO, Pts[2]);
          AddCommandPt(CMD_LINETO, Point(Pts[3].X - 1, Pts[3].Y));
          Pts[1] := Point(Pts[1].X - 1, Pts[1].Y + 1);
          Pts[2] := Point(Pts[2].X - 1, Pts[2].Y - 1);
          Pts[3] := Point(Pts[3].X, Pts[3].Y - 1);
          FirstPt.X := FirstPt.X - 1;
          FirstPt.Y := FirstPt.Y + 1;
          LastPt.X := LastPt.X - 1;
          LastPt.Y := LastPt.Y - 1;
          { Tab Control only has bevel on right and bottom edges }
          AddCommandPt(CMD_MOVETO, Pts[1]);
          AddCommand(CMD_SHADOW);
          AddCommandPt(CMD_LINETO, FirstPt);
          AddCommandPt(CMD_MOVETO, LastPt);
          AddCommandPt(CMD_LINETO, Pts[2]);
          AddCommandPt(CMD_LINETO, Pts[3]);
          { fix up join between current tab and card }
          AddCommand(CMD_HIGHLIGHT);
          AddCommandPt(CMD_MOVETO, Point(FirstPt.X, FirstPt.Y - 1));
          AddCommandPt(CMD_LINETO, Point(FirstPt.X + 2, FirstPt.Y - 1));
          AddCommand(CMD_SHADOW);
          AddCommandPt(CMD_MOVETO, Point(LastPt.X, LastPt.Y));
          AddCommandPt(CMD_LINETO, Point(LastPt.X + 2, LastPt.Y));
        end;
    end;
  end; { GenerateBrokenTCBorderCommands }

  procedure GenerateTabSetJoinCommands;
  begin
    case FTabOrientation of
      toTop:
        begin
          AddCommand(CMD_HIGHLIGHT);
          AddCommandPt(CMD_MOVETO, FirstPt);
          AddCommandPt(CMD_LINETO, Point(FirstPt.X, FirstPt.Y - 2));
          AddCommand(CMD_SHADOW);
          AddCommandPt(CMD_MOVETO, Point(LastPt.X - 1, LastPt.Y - 1));
          AddCommandPt(CMD_LINETO, Point(LastPt.X, FirstPt.Y - 1));
        end;
      toBottom:
        begin
          AddCommand(CMD_HIGHLIGHT);
          AddCommandPt(CMD_MOVETO, Point(FirstPt.X, FirstPt.Y));
          AddCommandPt(CMD_LINETO, Point(FirstPt.X, FirstPt.Y + 2));
          AddCommand(CMD_SHADOW);
          if FTabSequence = tsStandard then
          begin
            AddCommandPt(CMD_MOVETO, Point(LastPt.X, LastPt.Y));
            AddCommandPt(CMD_LINETO, Point(LastPt.X, LastPt.Y + 2));
          end
          else
          begin
            AddCommandPt(CMD_MOVETO, Point(LastPt.X, LastPt.Y + 1));
            AddCommandPt(CMD_LINETO, Point(LastPt.X, LastPt.Y - 1));
          end;
          if (LastPt.X = Pts[2].X + 1) then
          begin
            { tab is flush with right hand edge of card }
            AddCommand(CMD_BORDER);
            AddCommandPt(CMD_MOVETO, Point(Pts[2].X + 1, Pts[2].Y + 1));
            AddCommandPt(CMD_LINETO, Point(Pts[2].X + 1, Pts[2].Y - 1));
            AddCommand(CMD_SHADOW);
            AddCommandPt(CMD_MOVETO, Point(Pts[2].X, Pts[2].Y + 1));
            AddCommandPt(CMD_LINETO, Point(Pts[2].X, Pts[2].Y - 1));
          end;
        end;
      toLeft:
        begin
          AddCommand(CMD_SHADOW);
          AddCommandPt(CMD_MOVETO, Point(FirstPt.X - 1, FirstPt.Y));
          AddCommandPt(CMD_LINETO, Point(FirstPt.X + 1, FirstPt.Y));
          AddCommand(CMD_HIGHLIGHT);
          AddCommandPt(CMD_MOVETO, Point(LastPt.X - 1, LastPt.Y + 1));
          AddCommandPt(CMD_LINETO, Point(LastPt.X + 1, LastPt.Y + 1));
          if (LastPt.Y = Pts[0].Y - 1) then
          begin
            { tab is flush with top edge of card }
            AddCommand(CMD_BORDER);
            AddCommandPt(CMD_MOVETO, Point(Pts[0].X, Pts[0].Y - 1));
            AddCommandPt(CMD_LINETO, Point(Pts[0].X + 1, Pts[0].Y - 1));
            AddCommand(CMD_HIGHLIGHT);
            AddCommandPt(CMD_MOVETO, Point(Pts[0].X - 1, Pts[0].Y));
            AddCommandPt(CMD_LINETO, Point(Pts[0].X, Pts[0].Y));
          end;
        end;
      toRight:
        begin
          AddCommand(CMD_HIGHLIGHT);
          AddCommandPt(CMD_MOVETO, Point(FirstPt.X, FirstPt.Y));
          AddCommandPt(CMD_LINETO, Point(FirstPt.X + 2, FirstPt.Y));
          AddCommand(CMD_SHADOW);
          AddCommandPt(CMD_MOVETO, Point(LastPt.X, LastPt.Y - 1));
          AddCommandPt(CMD_LINETO, Point(LastPt.X + 2, LastPt.Y - 1));
          if (LastPt.Y = Pts[2].Y + 1) then
          begin
            { tab is flush with bottom edge of card }
            AddCommand(CMD_BORDER);
            AddCommandPt(CMD_MOVETO, Point(Pts[2].X, Pts[2].Y + 1));
            AddCommandPt(CMD_LINETO, Point(Pts[2].X - 1, Pts[2].Y + 1));
          end;
        end;
    end;
  end;

  { Generate TabbedNotebook join commands }
  procedure GenerateTNBJoinCommands(Depth: Integer);
  begin
    if Depth = 1 then { first bevel level }
      case FTabOrientation of
        toTop:
          begin
            AddCommand(CMD_HIGHLIGHT);
            AddCommandPt(CMD_MOVETO, Point(FirstPt.X, FirstPt.Y - 1));
            AddCommandPt(CMD_LINETO, Point(FirstPt.X, FirstPt.Y));
            AddCommandPt(CMD_LINETO, Point(FirstPt.X + 1, FirstPt.Y));
            AddCommand(CMD_SHADOW);
            AddCommandPt(CMD_MOVETO, Point(LastPt.X - 1, LastPt.Y - 1));
            AddCommandPt(CMD_LINETO, Point(LastPt.X, LastPt.Y + 1));
          end;
        toBottom:
          begin
            AddCommand(CMD_HIGHLIGHT);
            AddCommandPt(CMD_MOVETO, Point(FirstPt.X, FirstPt.Y + 1));
            AddCommandPt(CMD_LINETO, Point(FirstPt.X, FirstPt.Y));
            AddCommand(CMD_SHADOW);
            AddCommandPt(CMD_MOVETO, Point(LastPt.X - 1, LastPt.Y + 1));
            AddCommandPt(CMD_LINETO, Point(LastPt.X - 1, LastPt.Y - 1));
            AddCommandPt(CMD_MOVETO, Point(LastPt.X - 1, LastPt.Y));
            AddCommandPt(CMD_LINETO, Point(LastPt.X + 1, LastPt.Y));
            if (LastPt.X = Pts[2].X + 1) then
            begin
              { tab is flush with right hand edge of card }
              AddCommand(CMD_BORDER);
              AddCommandPt(CMD_MOVETO, Point(Pts[2].X + 1, Pts[2].Y));
              AddCommandPt(CMD_LINETO, Point(Pts[2].X + 1, Pts[2].Y + 1));
            end;
          end;
        toLeft:
          begin
            AddCommand(CMD_SHADOW);
            AddCommandPt(CMD_MOVETO, Point(FirstPt.X - 1, FirstPt.Y));
            AddCommandPt(CMD_LINETO, Point(FirstPt.X + 1, FirstPt.Y));
            AddCommand(CMD_HIGHLIGHT);
            AddCommandPt(CMD_MOVETO, Point(LastPt.X - 1, LastPt.Y + 1));
            AddCommandPt(CMD_LINETO, Point(LastPt.X + 1, LastPt.Y + 1));
            if (LastPt.Y = Pts[0].Y - 1) then
            begin
              { tab is flush with top edge of card }
              AddCommand(CMD_BORDER);
              AddCommandPt(CMD_MOVETO, Point(Pts[0].X, Pts[0].Y - 1));
              AddCommandPt(CMD_LINETO, Point(Pts[0].X + 1, Pts[0].Y - 1));
            end;
          end;
        toRight:
          begin
            AddCommand(CMD_HIGHLIGHT);
            AddCommandPt(CMD_MOVETO, Point(FirstPt.X, FirstPt.Y));
            AddCommandPt(CMD_LINETO, Point(FirstPt.X + 2, FirstPt.Y));
            AddCommand(CMD_SHADOW);
            AddCommandPt(CMD_MOVETO, Point(LastPt.X, LastPt.Y - 1));
            AddCommandPt(CMD_LINETO, Point(LastPt.X + 2, LastPt.Y - 1));
            if (LastPt.Y = Pts[2].Y + 1) then
            begin
              { tab is flush with bottom edge of card }
              AddCommand(CMD_BORDER);
              AddCommandPt(CMD_MOVETO, Point(Pts[2].X, Pts[2].Y + 1));
              AddCommandPt(CMD_LINETO, Point(Pts[2].X + 1, Pts[2].Y + 1));
            end;
          end;
      end
    else { second bevel level }
      case FTabOrientation of
        toTop:
          begin
            AddCommand(CMD_HIGHLIGHT);
            AddCommandPt(CMD_MOVETO, Point(FirstPt.X + 1, FirstPt.Y - 2));
            AddCommandPt(CMD_LINETO, Point(FirstPt.X + 1, FirstPt.Y));
            AddCommandPt(CMD_LINETO, Point(FirstPt.X - 1, FirstPt.Y));
            AddCommand(CMD_SHADOW);
            AddCommandPt(CMD_MOVETO, Point(LastPt.X - 2, LastPt.Y - 2));
            AddCommandPt(CMD_LINETO, Point(LastPt.X - 2, LastPt.Y + 1));
            AddCommand(CMD_HIGHLIGHT);
            AddCommandPt(CMD_MOVETO, Point(LastPt.X - 1, LastPt.Y));
            AddCommandPt(CMD_LINETO, Point(LastPt.X + 1, LastPt.Y));
            if (LastPt.X = Pts[1].X + 2) then
            begin
              { tab is flush with right hand edge of card }
              AddCommand(CMD_BORDER);
              AddCommandPt(CMD_MOVETO, Point(Pts[1].X + 2, Pts[1].Y));
              AddCommandPt(CMD_LINETO, Point(Pts[1].X + 2, Pts[1].Y + 1));
              AddCommand(CMD_SHADOW);
              AddCommandPt(CMD_MOVETO, Point(Pts[1].X + 1, Pts[1].Y));
              AddCommandPt(CMD_LINETO, Point(Pts[1].X + 1, Pts[1].Y + 1));
            end;
          end;
        toBottom:
          begin
            AddCommand(CMD_HIGHLIGHT);
            AddCommandPt(CMD_MOVETO, Point(FirstPt.X + 1, FirstPt.Y + 2));
            AddCommandPt(CMD_LINETO, Point(FirstPt.X + 1, FirstPt.Y - 1));
            AddCommand(CMD_SHADOW);
            AddCommandPt(CMD_MOVETO, Point(FirstPt.X + 1, FirstPt.Y));
            AddCommandPt(CMD_LINETO, Point(FirstPt.X, FirstPt.Y));
            AddCommand(CMD_SHADOW);
            AddCommandPt(CMD_MOVETO, Point(LastPt.X - 2, LastPt.Y + 2));
            AddCommandPt(CMD_LINETO, Point(LastPt.X - 2, LastPt.Y - 1));
            AddCommandPt(CMD_MOVETO, Point(LastPt.X - 2, LastPt.Y));
            AddCommandPt(CMD_LINETO, Point(LastPt.X + 1, LastPt.Y));
            { final touch up to correct anomolies at join of current tab when
              flush with card edge
            }
            AddCommand(CMD_HIGHLIGHT);
            AddCommandPt(CMD_MOVETO, Point(Pts[3].X - 1, Pts[3].Y));
            AddCommandPt(CMD_LINETO, Point(Pts[3].X + 1, Pts[3].Y));
            AddCommand(CMD_BORDER);
            AddCommandPt(CMD_MOVETO, Point(Pts[2].X + 2, Pts[2].Y));
            AddCommandPt(CMD_LINETO, Point(Pts[2].X + 2, Pts[2].Y + 1));
          end;
        toLeft:
          begin
            AddCommand(CMD_SHADOW);
            AddCommandPt(CMD_MOVETO, Point(FirstPt.X - 2, FirstPt.Y - 1));
            AddCommandPt(CMD_LINETO, Point(FirstPt.X + 1, FirstPt.Y - 1));
            if not (FirstPt.Y = Pts[3].Y + 1) then
            begin
              AddCommand(CMD_HIGHLIGHT);
              AddCommandPt(CMD_MOVETO, Point(FirstPt.X, FirstPt.Y));
              AddCommandPt(CMD_LINETO, Point(FirstPt.X, FirstPt.Y + 1));
            end;
            AddCommand(CMD_HIGHLIGHT);
            AddCommandPt(CMD_MOVETO, Point(LastPt.X - 2, LastPt.Y + 2));
            AddCommandPt(CMD_LINETO, Point(LastPt.X + 1, LastPt.Y + 2));
            AddCommandPt(CMD_MOVETO, Point(LastPt.X, LastPt.Y + 2));
            AddCommandPt(CMD_LINETO, Point(LastPt.X, LastPt.Y));
            if (LastPt.Y = Pts[0].Y - 2) then
            begin
              { tab is flush with top edge of card }
              AddCommand(CMD_BORDER);
              AddCommandPt(CMD_MOVETO, Point(Pts[0].X, Pts[0].Y - 2));
              AddCommandPt(CMD_LINETO, Point(Pts[0].X + 1, Pts[0].Y - 2));
            end;
          end;
        toRight:
          begin
            if (FirstPt.Y = Pts[1].Y - 1) then
            begin
              AddCommand(CMD_HIGHLIGHT);
              AddCommandPt(CMD_MOVETO, Point(FirstPt.X, FirstPt.Y + 1));
              AddCommandPt(CMD_LINETO, Point(FirstPt.X + 3, FirstPt.Y + 1));
            end
            else
            begin
              AddCommand(CMD_SHADOW);
              AddCommandPt(CMD_MOVETO, Point(FirstPt.X, FirstPt.Y));
              AddCommandPt(CMD_LINETO, Point(FirstPt.X + 2, FirstPt.Y));
              AddCommandPt(CMD_MOVETO, Point(FirstPt.X, FirstPt.Y + 1));
              AddCommandPt(CMD_LINETO, Point(FirstPt.X + 1, FirstPt.Y + 1));
              AddCommand(CMD_HIGHLIGHT);
              AddCommandPt(CMD_LINETO, Point(FirstPt.X + 3, FirstPt.Y + 1));
            end;
            AddCommand(CMD_SHADOW);
            AddCommandPt(CMD_MOVETO, Point(LastPt.X, LastPt.Y));
            AddCommandPt(CMD_LINETO, Point(LastPt.X, LastPt.Y - 2));
            AddCommandPt(CMD_LINETO, Point(LastPt.X + 3, LastPt.Y - 2));
            if (LastPt.Y = Pts[2].Y + 2) then
            begin
              { tab is flush with bottom edge of card }
              AddCommand(CMD_BORDER);
              AddCommandPt(CMD_MOVETO, Point(Pts[2].X, Pts[2].Y + 2));
              AddCommandPt(CMD_LINETO, Point(Pts[2].X + 1, Pts[2].Y + 2));
            end;
          end;
      end;
  end;

begin { CalcCardDrawCommands }
  if FTabIndex < 0 then
  begin
    { no tabs are visible/enabled/selectable, draw appropriate continuous card }

    { put corners of mapped rect points into Pts[] }
    CardRect := CalcMappedCardRect(ARow);
    NumPts := 4;
    Pts[0] := Point(CardRect.Left, CardRect.Top);
    Pts[1] := Point(CardRect.Right, CardRect.Top);
    Pts[2] := Point(CardRect.Right, CardRect.Bottom);
    Pts[3] := Point(CardRect.Left, CardRect.Bottom);

    { build list of commands needed to draw the tab shape }
    case FTabStyle of
      tsTabControl:
        GenerateContinuousTCCommands;
      tsTabSet:
        begin
          GenerateContinuousBorderCommands;
          Pts[0] := Point(Pts[0].X + 1, Pts[0].Y + 1);
          Pts[1] := Point(Pts[1].X - 1, Pts[1].Y + 1);
          Pts[2] := Point(Pts[2].X - 1, Pts[2].Y - 1);
          Pts[3] := Point(Pts[3].X + 1, Pts[3].Y - 1);
          AddCommand(CMD_CTL3D1);
          GenerateContinuousBevelCommands;
        end;
    tsTabbedNotebook:
      begin
        GenerateContinuousBorderCommands;
        for I := 1 to 2 do
        begin
          { define points for bevel }
          if I = 1 then
            AddCommand(CMD_CTL3D1)
          else
            AddCommand(CMD_CTL3D2);
          Pts[0] := Point(Pts[0].X + 1, Pts[0].Y + 1);
          Pts[1] := Point(Pts[1].X - 1, Pts[1].Y + 1);
          Pts[2] := Point(Pts[2].X - 1, Pts[2].Y - 1);
          Pts[3] := Point(Pts[3].X + 1, Pts[3].Y - 1);
          GenerateContinuousBevelCommands
        end;
      end;
    end;

  end
  else
  begin

    TabData := FTabDataList[FTabIndex];
    CurrentTabInView := TabInView(TabData);

    { get first and last points of outline, for use in calculating the opening
      where the current tab joins the front card }
    CalcTabRegionPts(FTabIndex, Pts, NumPts);
    FirstRawPt := Point(Pts[0].X + 1, Pts[0].Y);
    LastRawPt := Pts[NumPts - 1];
    FirstPt := CalcMapPoint(FirstRawPt);
    LastPt := CalcMapPoint(LastRawPt);

    { put corners of mapped rect points into Pts[] }
    CardRect := CalcMappedCardRect(ARow);
    NumPts := 4;
    Pts[0] := Point(CardRect.Left, CardRect.Top);
    Pts[1] := Point(CardRect.Right, CardRect.Top);
    Pts[2] := Point(CardRect.Right, CardRect.Bottom);
    Pts[3] := Point(CardRect.Left, CardRect.Bottom);

    { build list of commands needed to draw the tab shape }
    case FTabStyle of
      tsTabControl:
        begin
          if (ARow > 0) or not CurrentTabInView then { continuous card border }
            GenerateContinuousTCCommands
          else
            GenerateBrokenTCCommands
        end;

      tsTabSet:
        begin
          if (ARow > 0) or not CurrentTabInView then { continuous card border }
            GenerateContinuousBorderCommands
          else { card border has gap where current tab meets card }
            GenerateBrokenBorderCommands;
          Pts[0] := Point(Pts[0].X + 1, Pts[0].Y + 1);
          Pts[1] := Point(Pts[1].X - 1, Pts[1].Y + 1);
          Pts[2] := Point(Pts[2].X - 1, Pts[2].Y - 1);
          Pts[3] := Point(Pts[3].X + 1, Pts[3].Y - 1);
          FirstRawPt.Y := FirstRawPt.Y + 1;
          LastRawPt.Y := LastRawPt.Y + 1;
          FirstPt := CalcMapPoint(FirstRawPt);
          LastPt := CalcMapPoint(LastRawPt);
          AddCommand(CMD_CTL3D1);
          if (ARow > 0) or not CurrentTabInView then
            GenerateContinuousBevelCommands
          else
          begin
            GenerateBrokenBevelCommands;
            GenerateTabSetJoinCommands;
          end;
        end;

      tsTabbedNotebook:
        begin
          if (ARow > 0) or not CurrentTabInView then { continuous card border }
            GenerateContinuousBorderCommands
          else { card has gap where current tab meets card }
            GenerateBrokenBorderCommands;
          for I := 1 to 2 do
          begin
            { define points for bevel }
            if I = 1 then
              AddCommand(CMD_CTL3D1)
            else
              AddCommand(CMD_CTL3D2);
            Pts[0] := Point(Pts[0].X + 1, Pts[0].Y + 1);
            Pts[1] := Point(Pts[1].X - 1, Pts[1].Y + 1);
            Pts[2] := Point(Pts[2].X - 1, Pts[2].Y - 1);
            Pts[3] := Point(Pts[3].X + 1, Pts[3].Y - 1);
            Inc(FirstRawPt.Y);
            Inc(LastRawPt.Y);
            FirstPt := CalcMapPoint(FirstRawPt);
            LastPt := CalcMapPoint(LastRawPt);
            if (ARow > 0) or not CurrentTabInView then { continuous card border }
              GenerateContinuousBevelCommands
            else
            begin
              GenerateBrokenBevelCommands;
              GenerateTNBJoinCommands(I);
            end;
          end;
        end;
    end;
  end;
end;

procedure TcsChameleonTabControl.PaintBackground(ACanvas: TCanvas;
  const ARect: TRect; var Handled: Boolean);
begin
  if Assigned(FOnPaintBackground) then
    FOnPaintBackground(Self, ACanvas, ARect, Handled);
end;

procedure TcsChameleonTabControl.PaintCardBackground(ACanvas: TCanvas;
  ARow: Integer; const ARect: TRect; var Handled: Boolean);
begin
  if Assigned(FOnPaintCardBackground) then
    FOnPaintCardBackground(Self, ACanvas, ARow, ARect, Handled);
end;

procedure TcsChameleonTabControl.PaintTabBackground(ACanvas: TCanvas;
  ATabIndex: Integer; const ARect: TRect; var Handled: Boolean);
begin
  if Assigned(FOnPaintTabBackground) then
    FOnPaintTabBackground(Self, ACanvas, ATabIndex, ARect, Handled);
end;

function TcsChameleonTabControl.CalcGlyphExtent(ABitmap: TBitmap;
  ANumGlyphs: TcsTabNumGlyphs): TSize;
begin
  { calculate size needed for glyph }
  Result.cX := 0;
  Result.cY := 0;
  if ABitmap <> nil then
  begin
    Result.cX := ABitmap.Width div ANumGlyphs;
    Result.cY := ABitmap.Height;
  end;
end;

{ CalcTextExtent will determine the extent (X, Y) of the specified
  string. The string can be multi-line (CRLF at end of each line) and can
  contain a single ampersand character to indicate underlining or double
  ampersand characters to indicate the inclusion of a single ampersand in the
  text. The Horizontal parameter indicates if the text runs horizontally or
  vertically (i.e. sideways).
}
function TcsChameleonTabControl.CalcTextExtent(const S: String;
  Horizontal: Boolean): TSize;
var
  NextLine: String;
  Extent: TSize;
  DC: hDC;
  Lines: TStringList;
  I: Integer;

  { CalcExtent will determine the extent (width & height) of the NextLine
    string; single and double ampersand character sequences in the string
    are supported.
  }
  procedure CalcExtent;
  var
    Pos1, Pos2: Integer;
    PlainText: String;
    PText: Array[0..255] of Char;
    ActualY: Integer;
  begin
    Pos1 := Pos('&', NextLine);
    if (Pos1 = Length(NextLine)) then
    begin
      { If the string contains a '&' as the last character it will
        usually be because DoTextOut is determining the length of the
        first part of a caption which contains '&&' indicating a
        literal '&' is to actually be displayed.  In this case
        no underlining will occur so the length of the whole
        string (including the '&') should be calculated.
      }
      Pos1 := 0;
      Pos2 := 0;
    end
    else if (Pos1 > 0) then
    begin
      Pos2 := Pos('&', Copy(NextLine, Pos1 + 1, Length(NextLine)));
      if (Pos2 = 1) then Inc(Pos2, Pos1)
      else Pos2 := 0;
    end
    else Pos2 := 0;
    { check for special case of trailing single '&' which will occur
      when getting the length of the first part of a string which
      contained '&&', e.g. 'A&&A'
    }
    if (Pos1 > 0) and (Pos2 <> Pos1 + 1) then
      PlainText := Copy(NextLine, 1, Pos1 - 1) +
        Copy(NextLine, Pos1 + 1, Length(NextLine))
    else if (Pos1 > 0) and (Pos2 = Pos1 + 1) then
      PlainText := Copy(NextLine, 1, Pos1) +
        Copy(NextLine, Pos2 + 1, Length(NextLine))
    else
      PlainText := NextLine;
    StrPCopy(PText, PlainText);
{$IFDEF WIN32}
    if not (((Win32Platform = VER_PLATFORM_WIN32s) and
        GetTextExtentPoint(DC, PText, StrLen(PText), Extent)) or
      GetTextExtentPoint32(DC, PText, StrLen(PText), Extent)) then
{$ELSE}
    if not GetTextExtentPoint(DC, PText, StrLen(PText), Extent) then
{$ENDIF}
    begin
      Extent.cX := 0;
      Extent.cY := 0;
    end;
    if not Horizontal then
    begin
      { The GetTextExtentPoint* functions don't seem to allow for when using
        sideways text and thus still return the width and height AS IF THE
        TEXT WERE HORIZONTAL, so we need to switch x and y extents.
      }
      ActualY := Extent.cX;
      Extent.cX := Extent.cY;
      Extent.cY := ActualY;
    end;
  end; { CalcExtent }

begin { CalcTextExtent }
  Result.cX := 0;
  Result.cY := 0;
  { re-use same list object to speed things up by not recreating each time
    (which starts to add up when there are lots of tabs)
  }
  Lines := FCalcTextExtentLines;
  Lines.Clear;
  DC := FBuffer.Canvas.Handle;
  ParseTextLines(S, Lines);
  for I := 0 to Lines.Count - 1 do
  begin
    NextLine := Lines[I];
    CalcExtent;
    if Horizontal then
    begin
      { total X extent is maximum of X extents of all lines,
        total Y extent is sum of Y extents of each line
      }
      if Extent.cX > Result.cX then
        Result.cX := Extent.cX;
      Inc(Result.cY, Extent.cY);
    end
    else
    begin
      { total X extent is sum of X extents of each line,
        total Y extent is maximum of Y extents of all lines
      }
      Inc(Result.cX, Extent.cX);
      if Extent.cY > Result.cY then
        Result.cY := Extent.cY;
    end;
  end;
  { update text extent caches }
{$IFDEF CSTC_TEXTEXTENTCACHE}
  if fsBold in FTextFont.Style then
    FTextExtentCacheBold.Add(S, Result)
  else
    FTextExtentCacheNormal.Add(S, Result);
{$ENDIF}
end; { CalcTextExtent }

function TcsChameleonTabControl.CalcTabExtentFromTabFaceExtent(
  TabFaceExtent: TSize): TSize;
begin
  { TabFaceExtent will be the size of just the tab's caption -- allow for
    1 pixel border on all sides for focus rect
  }
  Inc(TabFaceExtent.cX, 2);
  Inc(TabFaceExtent.cY, 2);
  { now work out tab size needed to accomodate the tab face size }
  case FTabStyle of
    tsTabControl:
      begin
        Result := TabFaceExtent;
        if FSidewaysText then
        begin
          Inc(Result.cX, 4);
          Inc(Result.cY, 14);
        end
        else
        begin
          Inc(Result.cX, 14);
          Inc(Result.cY, 4);
        end;
      end;
    tsTabSet:
      begin
        if FTabOrientation in [toTop, toBottom] then
        begin
            Result.cX := TabFaceExtent.cX +
              (GetTabOffset(TabFaceExtent.cY + 4)*2) + 4;
            Result.cY := TabFaceExtent.cY + 4;
        end
        else
        begin
          Result.cX := TabFaceExtent.cX + 4;
          Result.cY := TabFaceExtent.cY +
            (GetTabOffset(TabFaceExtent.cX + 4)*2) + 4;
        end;
      end;
    tsTabbedNotebook:
      if FTabOrientation in [toTop, toBottom] then
      begin
        if FSidewaysText then
        begin
          Result.cX := TabFaceExtent.cX + 8;
          Result.cY := TabFaceExtent.cY + FCornerSize + 5;
        end
        else
        begin
          Result.cX := TabFaceExtent.cX + (FCornerSize*2) + 6;
          Result.cY := TabFaceExtent.cY + 5;
        end
      end
      else { toLeft, toRight }
      begin
        if FSidewaysText then
        begin
          Result.cX := TabFaceExtent.cX + 5;
          Result.cY := TabFaceExtent.cY + (FCornerSize*2) + 6;
        end
        else
        begin
          Result.cX := TabFaceExtent.cX + FCornerSize + 5;
          Result.cY := TabFaceExtent.cY + 8;
        end;
      end;
  else
      begin
        Result.cX := TabFaceExtent.cX;
        Result.cY := TabFaceExtent.cY;
      end;
  end;
end;

{ Return rect of tab face for specified raw rect.
  Note: Implicitly related to CalcTabExtentFromTabFaceExtent.
}
function TcsChameleonTabControl.CalcTabFaceRect(ARect: TRect): TRect;
begin
  Result := ARect;
  case FTabStyle of
    tsTabControl:
      begin
        if FSidewaysText then
          InflateRect(Result, -2, -4)
        else
          InflateRect(Result, -4, -2);
      end;
    tsTabSet:
      if FTabOrientation in [toTop, toBottom] then
      begin
        InflateRect(Result,
          -(GetTabOffset(ARect.Bottom - ARect.Top) + 2), -1);
        OffsetRect(Result, 1, 0);
        if FTabOrientation = toTop then
          Inc(Result.Top, 2)
        else
          Dec(Result.Bottom, 2);
        if FTabOrientation = toBottom then
          OffsetRect(Result, 0, 1);
      end
      else
      begin
        InflateRect(Result,
          -1, -(GetTabOffset(ARect.Right - ARect.Left) + 2));
        OffsetRect(Result, 0, 1);
        if FTabOrientation = toLeft then
          Inc(Result.Left, 2)
        else
          Dec(Result.Right, 2);
        if FTabOrientation = toRight then
          OffsetRect(Result, 1, 0);
      end;
    tsTabbedNotebook:
      if FTabOrientation in [toTop, toBottom] then
      begin
        if FSidewaysText then
        begin
          InflateRect(Result, -4, -1);
          if FTabOrientation = toTop then
            Inc(Result.Top, FCornerSize + 3)
          else
            Dec(Result.Bottom, FCornerSize + 3);
        end
        else
        begin
          InflateRect(Result, -(FCornerSize + 3), -1);
          if FTabOrientation = toTop then
            Inc(Result.Top, 3)
          else
            Dec(Result.Bottom, 3);
        end;
        if FTabOrientation = toBottom then
          OffsetRect(Result, 0, 1);
      end
      else { toLeft, toRight }
      begin
        if FSidewaysText then
        begin
          InflateRect(Result, -1, -(FCornerSize + 3));
          if FTabOrientation = toLeft then
            Inc(Result.Left, 5)
          else
            Dec(Result.Right, 5);
        end
        else
        begin
          InflateRect(Result, -1, -4);
          if FTabOrientation = toLeft then
            Inc(Result.Left, FCornerSize + 5)
          else
            Dec(Result.Right, FCornerSize + 5);
        end;
        if FTabOrientation = toRight then
          OffsetRect(Result, 1, 0);
      end;
  end;
  { size of Result includes the 1 pixel border for the focus rect -- deflate
    to return just the text rect
  }
  InflateRect(Result, -1, -1); 
end;

{ Calculate the default size tab face to be used.  The tab face would, for
  example, be this size when the tab caption is a single character and has no
  glyph.  This ensures that tabs with short captions aren't all scrunched up.
}
function TcsChameleonTabControl.CalcDefaultTabFaceExtent(AFont: TFont): TSize;
var
  MinExtent: TSize;
  Horizontal: Boolean; { horizontal tab text }
  OldStyle: TFontStyles;

  function AdjustmentForFont: Integer;
  var
    S, H: Integer;
    Margin: Integer;
  begin
    { -ve font heights are used for font heights which don't include
      the Internal Leading amount
    }
    S := Abs(AFont.Size);
    H := Abs(AFont.Height);
    { Microsoft probably use some algorithm to determine this but
      the closest I can get is to use ranges.
    }
    case S of
      0..5: Margin := 8;
      6,7,10..18: Margin := 10;
      8, 9: Margin := 11;
      19..29: Margin := 12;
      30..35: Margin := 13;
      36..39: Margin := 14;
      40..47: Margin := 15;
      else Margin := 16;
    end;
    if FTabStyle = tsTabControl then
      Margin := Margin - 4;
    Margin := Margin div 2;
    Result := H + Margin
  end; { AdjustmentForFont }

 begin { CalcDefaultTabFaceExtent }
  Horizontal := not FSidewaysText;
  OldStyle := FTextFont.Style;
  if FBoldCurrentTab then { calculate all extents using bold font }
    FTextFont.Style := FTextFont.Style + [fsBold];
  SelectFont;
  if FTabStyle = tsTabControl then
{$IFDEF CSTC_TEXTEXTENTCACHE}
    MinExtent := GetTextExtent('AAA', Horizontal)
{$ELSE}
    MinExtent := CalcTextExtent('AAA', Horizontal)
{$ENDIF}
  else
{$IFDEF CSTC_TEXTEXTENTCACHE}
    MinExtent := GetTextExtent('A', Horizontal);
{$ELSE}
    MinExtent := CalcTextExtent('A', Horizontal);
{$ENDIF}
  DeselectFont;
  FTextFont.Style := OldStyle;
  if FSidewaysText then
  begin
    Result.cX := AdjustmentForFont;
    Result.cY := MinExtent.cY;
  end
  else
  begin
    Result.cX := MinExtent.cX;
    Result.cY := AdjustmentForFont;
  end;
end;

procedure TcsChameleonTabControl.CalcMetrics;
var
  MaxTabFaceExtent: TSize;
  MaxTabExtent: TSize;

  { Calculate the extent (X, Y) of the minimum sized rect
    needed to bound the tab's caption and glyph.
  }
  function CalcTabFaceExtent(ATabIndex: Integer; TabData: TcsTabData): TSize;
  var
    GlyphSize: TSize;
    TextSize: TSize;
    MinSize: TSize;
    OldStyle: TFontStyles;
  begin
    Result.cX := 0;
    Result.cY := 0;
    if not TabData.Visible then Exit;

    { calculate size needed for glyph }
    GlyphSize := CalcGlyphExtent(TabData.Bitmap, TabData.NumGlyphs);

    OldStyle := FTextFont.Style;
    if FBoldCurrentTab then { select bold font to allow for extra size needed }
      FTextFont.Style := FTextFont.Style + [fsBold];
    SelectFont;
    TextSize.cX := 0;
    TextSize.cY := 0;
    if Length(TabData.Caption) > 0 then
{$IFDEF CSTC_TEXTEXTENTCACHE}
      TextSize := GetTextExtent(TabData.Caption, not FSidewaysText);
{$ELSE}
      TextSize := CalcTextExtent(TabData.Caption, not FSidewaysText);
{$ENDIF}
    DeselectFont;
    FTextFont.Style := OldStyle;

    { now calculate total size required }
    case FGlyphPosition of
      gpTop, gpBottom:
        begin
          Result.cX := MaxInt(GlyphSize.cX, TextSize.cX);
          Result.cY := GlyphSize.cY + FGlyphMargin + TextSize.cY;
        end;
      gpLeft, gpRight:
        begin
          Result.cX := GlyphSize.cX + FGlyphMargin + TextSize.cX;
          Result.cY := MaxInt(GlyphSize.cY, TextSize.cY);
        end;
      gpBack: { use larger of glyph and text sizes in each dimension }
        begin
          Result.cX := MaxInt(GlyphSize.cX, TextSize.cX);
          Result.cY := MaxInt(GlyphSize.cY, TextSize.cY);
        end;
      gpStretch: { stretch size of glyph to match text size }
        begin
          Result := TextSize;
        end;
    end;
    { check that tab face is not less than default size when using
      automatic sizing (TabWidth or TabHeight is zero)
    }
    MinSize := CalcDefaultTabFaceExtent(FTextFont);
    if (FTabWidth = 0) and (Result.cX < MinSize.cX) then
      Result.cX := MinSize.cX;
    if (FTabHeight = 0) and (Result.cY < MinSize.cY) then
      Result.cY := MinSize.cY;
  end; { CalcTabFaceExtent }

  { Calculate the maximum tab face extent, i.e. the minimum size rect
    within which each of the tab's captions + glyphs would fit.
  }
  function CalcMaxTabFaceExtent: TSize;
  var
    I: Integer;
    TabData: TcsTabData;
    Extent: TSize;
  begin
    Result.cX := 0;
    Result.cY := 0;
    for I := 0 to FTabDataList.Count - 1 do
    begin
      TabData := FTabDataList[I];
      if TabData.Visible then
      begin
        Extent := CalcTabFaceExtent(I, TabData);
        if Extent.cX > Result.cX then
          Result.cX := Extent.cX;
        if Extent.cY > Result.cY then
          Result.cY := Extent.cY;
      end;
    end;
  end;

  function CalcMaxTabExtent: TSize;
  var
    I: Integer;
    TabData: TcsTabData;
    Extent: TSize;
    FaceExtent: TSize;
    MinExtent: TSize;
    OldStyle: TFontStyles;
  begin
    Result.cX := 0;
    Result.cY := 0;
    OldStyle := FTextFont.Style;
    if FBoldCurrentTab then { select bold font to allow for extra size needed }
      FTextFont.Style := FTextFont.Style + [fsBold];
    SelectFont;
    MinExtent := CalcDefaultTabFaceExtent(Font);
    for I := 0 to FTabDataList.Count - 1 do
    begin
      TabData := FTabDataList[I];
      if TabData.Visible then
      begin
        FaceExtent := CalcTabFaceExtent(I, TabData);
        if FaceExtent.cX < MinExtent.cX then
          FaceExtent.cX := MinExtent.cX;
        if FaceExtent.cY < MinExtent.cY then
          FaceExtent.cY := MinExtent.cY;
        Extent := CalcTabExtentFromTabFaceExtent(FaceExtent);
        if Extent.cX > Result.cX then
          Result.cX := Extent.cX;
        if Extent.cY > Result.cY then
          Result.cY := Extent.cY;
      end;
    end;
    DeselectFont;
    FTextFont.Style := OldStyle;
    { override if fixed width or fixed height specified }
    if FTabWidth > 0 then
      Result.cX := FTabWidth;
    if FTabHeight > 0 then
      Result.cY := FTabHeight;
  end;

  procedure CheckWidth(var TabWidth: Integer; IndexWidth: Integer);
  var
    Adjustment: Integer;
  begin
    if (FTabStyle = tsTabControl) then
      Adjustment := 4 { Tab Control style inflates current tab }
    else
      Adjustment := 0;
    if (TabWidth > IndexWidth - Adjustment) then { truncate width }
      TabWidth := IndexWidth - Adjustment;
  end;

  function CalcActualTabExtent(ATabIndex: Integer; TabData: TcsTabData): TSize;
  var
    FaceExtent: TSize;
  begin
    FaceExtent := CalcTabFaceExtent(ATabIndex, TabData);
    if (FTabOrientation in [toTop, toBottom]) then
    begin
      if FaceExtent.cY < MaxTabFaceExtent.cY then
        FaceExtent.cY := MaxTabFaceExtent.cY;
    end
    else
    begin
      if FaceExtent.cX < MaxTabFaceExtent.cX then
        FaceExtent.cX := MaxTabFaceExtent.cX;
    end;
    Result := CalcTabExtentFromTabFaceExtent(FaceExtent);
    if FTabWidth > 0 then
      Result.cX := FTabWidth;
    if FTabHeight > 0 then
      Result.cY := FTabHeight;
  end;

  { Calculate no. of rows of tabs needed and return the result.
    This is an iterative process because each time the row count increases
    it means the available tab index width (for all rows) needs to be reduced
    (by an amount of RowIndent) and the process started again.
  }
  function CalcRowsNeeded: Integer;
  var
    I, L, H, W: Integer;
    TabData: TcsTabData;
    TabExtent: TSize;
    MaxIdx, MaxIndexWidth: Integer;
    RowCount, TabCount, MinRowCount: Integer;
  begin
    Result := 1;
    if FMultiLine then
    begin
      H := FFixedDimension;
      L := FMargin;
      MaxIndexWidth := GetIndexWidth - (FMargin*2);
      RowCount := 1;
      MinRowCount := 1;
      TabCount := FTabDataList.Count;
      MaxIdx := GetLastVisible;
      I := 0;
      while I < TabCount do
      begin
        TabData := FTabDataList[I];
        if TabData.Visible then
        begin
          TabExtent := CalcActualTabExtent(I, TabData);
          if (FTabOrientation in [toTop, toBottom]) then
            W := TabExtent.cX
          else
            W := TabExtent.cY;
          CheckWidth(W, MaxIndexWidth);
          if (L + W >= MaxIndexWidth) then { full width of tab won't fit }
          begin
            if (RowCount < MinRowCount) then
            begin
              Inc(RowCount);
              if (L = FMargin) then { only tab in row and W = MaxIndexWidth }
                Inc(I);
            end
            else if (L > FMargin) or
              ((L = FMargin) and (I < MaxIdx)) then
            begin
              Inc(MinRowCount);
              Dec(MaxIndexWidth, FRowIndent);
              I := 0; { restart using reduced MaxIndexWidth }
              RowCount := 1;
            end
            else
              { L = 0 and I = MaxIdx, i.e. last tab }
              Inc(I);
            L := FMargin;
          end
          else
          begin
            Inc(I);
            L := L + W - GetTabOffset(H) + 1;
          end;
        end
        else
          Inc(I);
      end;
      Result := RowCount;
    end;
  end;

  { Calculate the 'raw' tab rects for all tabs.  The 'raw' rect is the rect the
    tab would have if the tabset was drawn in its actual orientation and the
    entire tabset then rotated (if necessary) to top orientation.
    tsStandard tabs are like this: [ A ][ B ][ C ]...
    tsReverse tabs are like this: ...[ C ][ B ][ A ]
    The raw tab rects only need to be recalculated when anything to do with an
    individual tab is changed (caption, font, visible).
    Other actions such as scrolling the tabs do not require the raw tab rects
    to be recalculated.
  }
  procedure CalcRawTabRects;
  var
    I, H, W, L, R, T, Offset, RowIdx, MaxIndexWidth: Integer;
    Box: TRect;
    TabData: TcsTabData;
    TabExtent: TSize;
  begin
    SetRowExtent(CalcRowsNeeded);
    H := FFixedDimension;
    T := GetIndexHeight - H;
    RowIdx := 0;
    MaxIndexWidth := GetIndexWidth - (FMargin*2);
    if FMultiLine then
      MaxIndexWidth := MaxIndexWidth - ((FRowExtent - 1)*FRowIndent)
    else
      MaxIndexWidth := MaxIndexWidth - (GetScrollBtnSize*2) - ScrollBtnMargin;

    { actual row indent (which is *not* the same thing as margin) for each row
      is added later when displaying because it will depend on the actual order
      in which the rows are displayed at that time
    }
    L := FMargin;
    R := -FMargin;
    I := 0;
    while I < FTabDataList.Count do
    begin
      TabData := FTabDataList[I];
      if TabData.Visible then
      begin
        TabExtent := CalcActualTabExtent(I, TabData);
        if (FTabOrientation in [toTop, toBottom]) then
          W := TabExtent.cX
        else
          W := TabExtent.cY;
        CheckWidth(W, MaxIndexWidth);
        if (FTabSequence = tsStandard) then
        begin
          if FMultiLine then
          begin
            if (L + W >= MaxIndexWidth) then
            begin
              if (L = FMargin) then { only tab in row }
              begin
                TabData.Row := RowIdx;
                Box := Rect(L, T, L + W, T + H);
                Inc(I);
              end
              else { process this tab again in next row }
                L := FMargin;
              Dec(T, H - FRowOverlap);
              Inc(RowIdx);
            end
            else
            begin
              TabData.Row := RowIdx;
              Box := Rect(L, T, L + W, T + H);
              L := L + W - GetTabOffset(H) + 1;
              Inc(I);
            end
          end
          else
          begin
            TabData.Row := RowIdx;
            Box := Rect(L, T, L + W, T + H);
            L := L + W - GetTabOffset(H) + 1;
            Inc(I);
          end;
          TabData.RawRect := Box;
        end
        else { == tsReverse === }
        begin
          if FMultiLine then
          begin
            if (R - W <= - MaxIndexWidth) then
            begin
              if (R = -FMargin) then { only tab in row }
              begin
                TabData.Row := RowIdx;
                Box := Rect(R - W, T, R, T + H);
                Inc(I);
              end
              else { process this tab again in next row }
                R := -FMargin;
              Dec(T, H - FRowOverlap);
              Inc(RowIdx);
            end
            else
            begin
              TabData.Row := RowIdx;
              Box := Rect(R - W, T, R, T + H);
              R := R - W + GetTabOffset(H) - 1;
              Inc(I);
            end
          end
          else
          begin
            TabData.Row := RowIdx;
            Box := Rect(R - W, T, R, T + H);
            R := R - W + GetTabOffset(H) - 1;
            Inc(I);
          end;
          TabData.RawRect := Box;
        end;
      end
      else
        Inc(I);
    end;
    if (FTabSequence = tsReverse) then
    begin
      { offset all raw tab rects so leftmost is at 0,0 }
      if FMultiLine then { current value of R not relevant }
        Offset := Width
      else
        Offset := Abs(R);
      for I := 0 to FTabDataList.Count - 1 do
      begin
        TabData := FTabDataList[I];
        if TabData.Visible then
        begin
          Box := TabData.RawRect;
          Inc(Box.Left, Offset);
          Inc(Box.Right, Offset);
          TabData.RawRect := Box;
        end;
      end;
    end;

  end; { CalcRawTabRects }

  { SetFixedDimension determines the extent of the 'fixed dimension', which is
    the height for top/bottom tab orientation and the width for left/right
    tab orientation.
    In top/bottom orientation all tabs will always have the same height.
    In left/right orientation all tabs will always have the same width.
  }
  procedure SetFixedDimension(Extent: TSize);
  begin
  if (FTabOrientation = toTop) or (FTabOrientation = toBottom) then
    FFixedDimension := Extent.cY
  else
    FFixedDimension := Extent.cX;
  end;

  { Divides any extra space at the end of each row amongst the tabs in that
    row so they are aligned with both the left and right margins of the cards.
   The no. of rows of tabs is unchanged.
  }
  procedure AlignRawTabRects;
  var
    I, MaxIdx, FirstIdxInRow, TabsInRow: Integer;
    TabData, PrevTabData: TcsTabData;

    { Align the tabs with indexes FirstIdx..LastIdx so they
      fill the row (from margin to margin).
    }
    procedure AlignRow(FirstIdx, LastIdx, TabCount: Integer);
    var
      I, MaxIndexWidth, RowWidth, Extra, Quotient, Remainder, Offset: Integer;
      TabToAlign: TcsTabData;
      Box: TRect;
    begin
      MaxIndexWidth := GetIndexWidth - (FMargin*2);
      MaxIndexWidth := MaxIndexWidth - ((FRowExtent - 1)*FRowIndent);
      if FTabSequence = tsStandard then
        RowWidth := FTabDataList[LastIdx].RawRect.Right -
                    FTabDataList[FirstIdx].RawRect.Left
      else
        RowWidth := FTabDataList[FirstIdx].RawRect.Right -
                    FTabDataList[LastIdx].RawRect.Left;
      if FTabStyle = tsTabControl then
        Inc(RowWidth, 4); { +2 on either side allowed for enlarged current tab }
      Extra := MaxIndexWidth - RowWidth;
      if Extra <= 0 then Exit; { should never happen (tabs already fit) }
      Quotient := Extra div TabCount;
      Remainder := Extra mod TabCount;
      Offset := 0;
      for I := FirstIdx to LastIdx do
      begin
        TabToAlign := FTabDataList[I];
        if TabToAlign.Visible then
        begin
          Box := TabToAlign.RawRect;
          Extra := Quotient; { start with whole amount }
          if Remainder > 0 then
          begin
            Inc(Extra); { plus one from remainder }
            Dec(Remainder);
          end;
          { offset each raw rect (to allow for lengthening of previous tabs
            in row) and then extend
          }
          if FTabSequence = tsStandard then
          begin
            Inc(Box.Left, Offset);
            Inc(Box.Right, Offset + Extra);
          end
          else
          begin
            Dec(Box.Right, Offset);
            Dec(Box.Left, Offset + Extra);
          end;
          TabToAlign.RawRect := Box;
          Inc(Offset, Extra);
        end;
      end;
    end; { AlignRow }

  begin { AlignRawTabRects }
    { Only perform alignment if necessary prerequisites met. }
    if not (FMultiLine and FAlignTabs and
      (((FTabOrientation in [toTop, toBottom]) and (FTabWidth = 0)) or
       ((FTabOrientation in [toLeft, toRight]) and (FTabHeight = 0)))) then
      Exit;
    PrevTabData := nil;
    MaxIdx := FTabDataList.Count - 1;
    FirstIdxInRow := -1;
    TabsInRow := 0;
    for I := 0 to MaxIdx do
    begin
      TabData := FTabDataList[I];
      if TabData.Visible then
      begin
        if FirstIdxInRow = -1 then
          FirstIdxInRow := TabData.Row;
        if PrevTabData = nil then
          PrevTabData := TabData;
        if (TabData.Row <> PrevTabData.Row) or (I = MaxIdx) then { new/last row }
        begin
          if (TabData.Row <> PrevTabData.Row) then { align previous tab }
          begin
            AlignRow(FirstIdxInRow, I - 1, TabsInRow);
            FirstIdxInRow := I;
            TabsInRow := 0;
          end;
          if (I = MaxIdx) then { align very last tab }
          begin
            Inc(TabsInRow);
            AlignRow(FirstIdxInRow, I, TabsInRow);
          end;
          FirstIdxInRow := I;
        end;
        Inc(TabsInRow);
        PrevTabData := TabData;
      end;
    end;
  end; { AlignRawTabRects }

begin { CalcMetrics }
{$IFDEF CSTC_TEXTEXTENTCACHE}
  FTextExtentCacheNormal.Clear;
  FTextExtentCacheBold.Clear;
{$ENDIF}
  MaxTabFaceExtent := CalcMaxTabFaceExtent;
  MaxTabExtent := CalcMaxTabExtent;
  SetFixedDimension(MaxTabExtent);
  CalcRawTabRects;
  AlignRawTabRects;
end;

function TcsChameleonTabControl.GetTabOffset(ATabHeight: Integer): Integer;
begin
  case FTabStyle of
    tsTabSet: Result := ATabHeight div 4;
    tsTabbedNotebook: Result := 1; { so abutting tabs re-use same border }
  else
    Result := 0;
  end;
end;

procedure TcsChameleonTabControl.SetMultiLine(Value: Boolean);
begin
  if FMultiLine <> Value then
  begin
    FMultiLine := Value;
    Rebuild;
  end;
end;

procedure TcsChameleonTabControl.SetRowExtent(Value: Integer);
begin
  FRowExtent := Value;
end;

procedure TcsChameleonTabControl.SetRowOverlap(Value: Integer);
begin
  if (Value >= 0) and (FRowOverlap <> Value) then
  begin
    FRowOverlap := Value;
    Rebuild;
  end;
end;

procedure TcsChameleonTabControl.SetRowIndent(Value: Integer);
begin
  if (Value >= 0) and (FRowIndent <> Value) then
  begin
    FRowIndent := Value;
    Rebuild;
  end;
end;

procedure TcsChameleonTabControl.CreateScrollBtns;
begin
  FScrollBtns := TcsUpDown.Create(Self); { Self will free FScrollBtns }
  with FScrollBtns do
  begin
    Parent := Self;
    ArrowColor := FScrollBtnArrowColor;
    BorderColor := FFrameBorderColor;
    FaceColor := FScrollBtnFaceColor;
    HighlightColor := FFrameHighlightColor;
    ShadowColor := FFrameShadowColor;
    OnClick := ScrollBtnClick;
  end;
end;

{ Whereas TcsTabData.Visible indicates that the tab can be shown,
  TabInView indicates that the tab is currently showing which depends on
  the scrolled position of all the tabs.
}
function TcsChameleonTabControl.TabInView(TabData: TcsTabData): Boolean;
var
  ScrollerWidth, IndexWidth: Integer;
begin
  if TabData.Visible then
  begin
    if FScrollBtnsNeeded then
    begin
      ScrollerWidth := GetScrollBtnSize*2;
      IndexWidth := GetIndexWidth - ScrollerWidth - (FMargin*2);
      Result := ((FTabSequence = tsStandard) and
                 (TabData.RawRect.Left >= GetInitialTabOffset) and
                 (TabData.RawRect.Right - GetInitialTabOffset <= IndexWidth)) or
                ((FTabSequence = tsReverse) and
                 (TabData.RawRect.Right <= GetInitialTabOffset) and
                 (GetInitialTabOffset - TabData.RawRect.Left <= IndexWidth));
    end
    else
      Result := True;
  end
  else
    Result := False;
end;

procedure TcsChameleonTabControl.ScrollBtnClick(Sender: TObject);
var
  Next: Boolean;
  OldFirstInView: Integer;
  Direction: TcsArrowButtonDirection;
begin
  Next := False;
  Direction := (Sender as TcsArrowButton).Direction;
  case FTabOrientation of
    toTop, toBottom:
      if (FTabSequence = tsStandard) then Next := (Direction = bdRight)
      else Next := (Direction = bdLeft);
    toLeft:
      if (FTabSequence = tsStandard) then Next := (Direction = bdUp)
      else Next := (Direction = bdDown);
    toRight:
      if (FTabSequence = tsStandard) then Next := (Direction = bdDown)
      else Next := (Direction = bdUp);
  end;
  OldFirstInView := FFirstInView;
  if Next and (FFirstInView < FTabDataList.Count - 1) and
    not TabInView(FTabDataList[GetLastVisible]) then
  begin
    repeat
      Inc(FFirstInView);
    until (FFirstInView = FTabDataList.Count) or
      TabInView(FTabDataList[FFirstInView]);
    if FFirstInView = FTabDataList.Count then
      FFirstInView := OldFirstInView
    else
      Invalidate;
  end
  else if (not Next) and (FFirstInView > 0) and
    not TabInView(FTabDataList[GetFirstVisible]) then
  begin
    repeat
      Dec(FFirstInView);
    until (FFirstInView < 0) or TabInView(FTabDataList[FFirstInView]);
    if FFirstInView < 0 then
      FFirstInView := OldFirstInView
    else
      Invalidate;
  end;
end;

procedure TcsChameleonTabControl.SetCalcNeeded(Value: Boolean);
begin
  if (FCalcNeeded <> Value) then
  begin
    FCalcNeeded := Value;
    if FCalcNeeded then
      Invalidate;
  end;
end;

{ WMEraseBkgdn does nothing -- all 'erasing' is handled by Paint so that
  an off-screen bitmap can be used (i.e. double buffering).
}
procedure TcsChameleonTabControl.WMEraseBkgnd(var Message: TWMEraseBkgnd);
begin
  { let Paint take care of everything }
  Message.Result := 1;
end;

procedure TcsChameleonTabControl.CheckCalcNeeded;
var
  NewTabIndex: Integer;
  OldFirstInView: Integer;
  FirstVisibleIdx: Integer;
begin
  if FCalcNeeded then
  begin
    SetCalcNeeded(False);
    CalcMetrics;
    FTabRegionCache.Clear;
    OldFirstInView := FFirstInView;
    FirstVisibleIdx := GetFirstVisible; { *not* the same as first in view }
    CalcScrollBtnsNeeded;
    if FScrollBtnsNeeded then
    begin
      { check in forwards direction for first tab in view }
      while (FFirstInView >= 0) and (FFirstInView < FTabDataList.Count) and
        not TabInView(FTabDataList[FFirstInView]) do
          Inc(FFirstInView);
      if FFirstInView = FTabDataList.Count then
      begin
       { check backwards from initial position }
        FFirstInView := OldFirstInView - 1;
        while (FFirstInView >= 0) and
          not TabInView(FTabDataList[FFirstInView]) do
            Dec(FFirstInView);
        { FFirstInView will be -1 if no tabs are visible }
      end;
      if FFirstInView < 0 then
        FFirstInView := FirstVisibleIdx;
    end
    else
      FFirstInView := FirstVisibleIdx;

    if FirstVisibleIdx >= 0 then
    begin
      if FTabIndex < 0 then
        { When FTabIndex is -1 don't automatically select another tab; this
          is necessary for page controls to allow the current page to remain
          active when its tab is deleted by setting the TabIndex to -1 prior to
          doing Rebuild (to remove the deleted tab).
        }
        NewTabIndex := -1
      else
        NewTabIndex := FTabIndex;
    end
    else
      NewTabIndex := -1;
    { check in forwards direction for first tab that is visible and enabled
      (but not necessarily in view)
    }
    while (NewTabIndex >= 0) and (NewTabIndex < FTabDataList.Count) and
      not CanSelectTab(NewTabIndex) do
        Inc(NewTabIndex);
    if NewTabIndex = FTabDataList.Count then
    begin
      { check backwards from initial position }
      NewTabIndex := FTabIndex;
      if NewTabIndex = FTabDataList.Count then { current tab was just deleted }
        Dec(NewTabIndex);
      while (NewTabIndex >= 0) and not CanSelectTab(NewTabIndex) do
        Dec(NewTabIndex);
      { NewTabIndex will be -1 if no tabs are visible }
    end;
    if (NewTabIndex <> FTabIndex) then
      { OnChanging event will have already been generated by CanSelectTab --
        set flag to prevent duplicate event being generated by SetTabIndex
      }
      FChangingDone := True;
    { SetTabIndex is called so that even if NewTabIndex = TabIndex the current
      tab will be brought into view and/or the first row.
    }
    SetTabIndex(NewTabIndex);
    FChangingDone := False;
    DoRealign;
  end;
end;

procedure TcsChameleonTabControl.BringTabToFrontRow(TabData: TcsTabData);
var
  I, NewFrontRow, MaxRow, NumRows, NumShifts: Integer;
  ATabData: TcsTabData;
  RectList: TList;
  Item: TRectClass;
  ARect: TRect;
begin
  if not FMultiLine then Exit;
  RectList := TList.Create;
  try
    NewFrontRow := TabData.Row;
    MaxRow := 0;
    for I := 0 to FTabDataList.Count - 1 do
    begin
      ATabData := FTabDataList[I];
      if ATabData.Row > MaxRow then
        MaxRow := ATabData.Row;
    end;
    NumRows := MaxRow + 1;
    { add enough items to list to allow for all rows -- the index of each item
      in the list then corresponds to the row no.
    }
    for I := 0 to NumRows - 1 do
      RectList.Add(nil);
    { save the rects of _a_ tab for each row -- since we are only
      interested in the Top and Bottom values in the rects it doesn't matter
      which tab from a row is used as they all have the same Top and Bottom
    }
    for I := 0 to FTabDataList.Count - 1 do
    begin
      ATabData := FTabDataList[I];
      if ATabData.Visible then
      begin
        if RectList[ATabData.Row] = nil then
        begin
          Item := TRectClass.Create;
          Item.Rect := ATabData.RawRect;
          RectList[ATabData.Row] := Item;
        end;
      end;
    end;
    { rows are shifted in a circular order, i.e. they stay in the same relative
      order and we just change which row is the first row
    }
    NumShifts := (NumRows - NewFrontRow) mod NumRows;
    for I := 0 to FTabDataList.Count - 1 do
    begin
      ATabData := FTabDataList[I];
      if ATabData.Visible then
      begin
        { determine new row position for this tab }
        ATabData.Row := (ATabData.Row + NumShifts) mod NumRows;
        { get old rect for row from list and adjust Top and Bottom of RawRect }
        ARect := ATabData.RawRect;
        Item := TRectClass(RectList[ATabData.Row]);
        ARect.Top := Item.Rect.Top;
        ARect.Bottom := Item.Rect.Bottom;
        ATabData.RawRect := ARect;
      end;
    end;
  finally
    for I := 0 to RectList.Count - 1 do
      TRectClass(RectList[I]).Free;
    RectList.Free;
  end;
  Invalidate;
end;

{ SelectFont will create a new font handle for a font with the required
  metrics (taken from FTextFont) and select it into the buffer's canvas in
  preparation for use during text metric calculations or text output.
}
procedure TcsChameleonTabControl.SelectFont;
var
  Escapement: Integer;
  LogFont: TLogFont;
  NewHFont: HFont;
begin
  case FTabOrientation of
    toTop, toBottom:  Escapement := iifInt(FSidewaysText, 900, 0);
    toLeft: Escapement := iifInt(FSidewaysText, 900, 0);
    toRight: Escapement := iifInt(FSidewaysText, 2700, 0);
  else
    Escapement := 0;
  end;

 { Must initialise all fields of the record structure! }
  with LogFont do
  begin
    lfHeight := FTextFont.Height;
    lfWidth := 0; { have font mapper choose }
    lfEscapement := Escapement;
    lfOrientation := 0; { no rotation }
    if (fsBold in FTextFont.Style) then
      lfWeight := FW_BOLD
    else
      lfWeight := FW_NORMAL;
    lfItalic := Byte(fsItalic in FTextFont.Style);
    lfUnderline := Byte(fsUnderline in FTextFont.Style);
    lfStrikeOut := Byte(fsStrikeOut in FTextFont.Style);
    lfCharSet := DEFAULT_CHARSET;
    StrPCopy(lfFaceName, FTextFont.Name);
    lfQuality := DEFAULT_QUALITY;
    { Everything else as default }
    lfOutPrecision := OUT_DEFAULT_PRECIS;
    lfClipPrecision := CLIP_DEFAULT_PRECIS;
    lfPitchAndFamily := DEFAULT_PITCH;
  end;
  NewHFont := CreateFontIndirect(LogFont);
  { save the current DC state so the current font object can be restored }
  if SaveDC(FBuffer.Canvas.Handle) = 0 then
    raise Exception.Create('Couldn''t save device context');
  { save the handle of the font object on our 'stack' }
  PushHFont(NewHFont);
  { and select the new font into the buffer's canvas }
  SelectObject(FBuffer.Canvas.Handle, NewHFont);
end; { SelectTextFont }

{ Restore a previously selected font in the buffer's canvas. }
procedure TcsChameleonTabControl.DeselectFont;
begin
  if RestoreDC(FBuffer.Canvas.Handle, -1) = WordBool(0) then
    raise Exception.Create('Couldn''t restore device context');
  { delete the deselected font object }
  DeleteObject(PopHFont);
end;

{ Push the specified value onto our 'stack' of font handles.
  The last value in the list corresponds to the top of the stock
  and the value at the top of the stack corresponds to the currently
  selected font object in the buffer's canvas.
}
procedure TcsChameleonTabControl.PushHFont(Value: HFont);
begin
  FHFonts.Add(TObject(Value));
end;

{ Pop the top most (i.e. last) value from the 'stack' of font handles. }
function TcsChameleonTabControl.PopHFont: HFont;
begin
  Result := 0;
  if FHFonts.Count > 0 then
  begin
    Result := HFont(FHFonts[FHFonts.Count - 1]);
    FHFonts.Delete(FHFonts.Count - 1);
  end;
end;

procedure TcsChameleonTabControl.CMFontChanged(var Message: TMessage);
begin
  { Copy the current font into the FTextFont field }
  FTextFont.Assign(Self.Font);
  Rebuild;
  inherited;
end;

procedure TcsChameleonTabControl.SetGlyphHAlignment(Value: TcsHorizontalAlignment);
begin
  if FGlyphHAlignment <> Value then
  begin
    FGlyphHAlignment := Value;
    Rebuild;
  end;
end;

procedure TcsChameleonTabControl.SetGlyphVAlignment(Value: TcsVerticalAlignment);
begin
  if FGlyphVAlignment <> Value then
  begin
    FGlyphVAlignment := Value;
    Rebuild;
  end;
end;

procedure TcsChameleonTabControl.SetTextHAlignment(Value: TcsHorizontalAlignment);
begin
  if FTextHAlignment <> Value then
  begin
    FTextHAlignment := Value;
    Rebuild;
  end;
end;

procedure TcsChameleonTabControl.SetTextVAlignment(Value: TcsVerticalAlignment);
begin
  if FTextVAlignment <> Value then
  begin
    FTextVAlignment := Value;
    Rebuild;
  end;
end;

procedure TcsChameleonTabControl.ParseTextLines(const S: String; Lines: TStrings);
var
  WorkText: String;
  NextLine: String;

  { ParseNextLine will get the next line of text (minus the trailing CRLF) from
    WorkText and will then set the work string to the remainder
    (minus the leading CRLF).
  }
  procedure ParseNextLine;
  var
    CRLFPos: Integer;
  begin
    CRLFPos := Pos(CRLF, WorkText);
    if CRLFPos = 0 then
    begin
      NextLine := Copy(WorkText, 1, Length(WorkText));
      WorkText := '';
    end
    else
    begin
      NextLine := Copy(WorkText, 1, CRLFPos - 1);
      WorkText := Copy(WorkText, CRLFPos + 2, Length(WorkText));
    end;
  end; { ParseNextLine }
begin
  Lines.Clear;
  WorkText := Copy(S, 1, Length(S));
  while Length(WorkText) > 0 do
  begin
    ParseNextLine;
    Lines.Add(NextLine);
  end;
end;

procedure TcsChameleonTabControl.SetGlyphPosition(Value: TcsGlyphPosition);
begin
  if FGlyphPosition <> Value then
  begin
    FGlyphPosition := Value;
    Rebuild;
  end;
end;

procedure TcsChameleonTabControl.SetGlyphMargin(Value: Integer);
begin
  if (Value >= 0) and (FGlyphMargin <> Value) then
  begin
    FGlyphMargin := Value;
    Rebuild;
  end;
end;

procedure TcsChameleonTabControl.SetClientArea(Value: Boolean);
begin
  if FClientArea <> Value then
  begin
    FClientArea := Value;
    Rebuild;
  end;
end;

procedure TcsChameleonTabControl.SetFrameBorderColor(Value: TColor);
begin
  if FFrameBorderColor <> Value then
  begin
    FFrameBorderColor := Value;
    Invalidate;
    if FScrollBtns <> nil then
      FScrollBtns.BorderColor := FFrameBorderColor;
  end;
end;

procedure TcsChameleonTabControl.SetFrameHighlightColor(Value: TColor);
begin
  if FFrameHighlightColor <> Value then
  begin
    FFrameHighlightColor := Value;
    Invalidate;
    if FScrollBtns <> nil then
      FScrollBtns.HighlightColor := FFrameHighlightColor;
  end;
end;

procedure TcsChameleonTabControl.SetFrameShadowColor(Value: TColor);
begin
  if FFrameShadowColor <> Value then
  begin
    FFrameShadowColor := Value;
    Invalidate;
    if FScrollBtns <> nil then
      FScrollBtns.ShadowColor := FFrameShadowColor;
  end;
end;

procedure TcsChameleonTabControl.GetFrameColor(AItem: TcsFrameItem;
  AFacet: TcsFrameFacet; AIndex: Integer; var AColor: TColor;
  var Handled: Boolean);
begin
  if Assigned(FOnGetFrameColor) then
    FOnGetFrameColor(Self, AItem, AFacet, AIndex, AColor, Handled);
end;

procedure TcsChameleonTabControl.GetTextColor(ATabIndex: Integer;
  AStyle: TcsTextStyle; var AColor: TColor; var Handled: Boolean);
begin
  if Assigned(FOnGetTextColor) then
    FOnGetTextColor(Self, ATabIndex, AStyle, AColor, Handled);
end;

procedure TcsChameleonTabControl.SetTextColorSelected(Value: TColor);
begin
  if FTextColorSelected <> Value then
  begin
    FTextColorSelected := Value;
    Invalidate;
  end;
end;

procedure TcsChameleonTabControl.SetTextColorUnselected(Value: TColor);
begin
  if FTextColorUnselected <> Value then
  begin
    FTextColorUnselected := Value;
    Invalidate;
  end;
end;

procedure TcsChameleonTabControl.SetTextColorDisabledHighlight(Value: TColor);
begin
  if FTextColorDisabledHighlight <> Value then
  begin
    FTextColorDisabledHighlight := Value;
    Invalidate;
  end;
end;

procedure TcsChameleonTabControl.SetTextColorDisabledShadow(Value: TColor);
begin
  if FTextColorDisabledShadow <> Value then
  begin
    FTextColorDisabledShadow := Value;
    Invalidate;
  end;
end;

procedure TcsChameleonTabControl.SetTextColorHotTrack(Value: TColor);
begin
  if FTextColorHotTrack <> Value then
  begin
    FTextColorHotTrack := Value;
    Invalidate;
  end;
end;

procedure TcsChameleonTabControl.Change;
begin
  if Assigned(FOnChange) then
    FOnChange(Self);
end;

procedure TcsChameleonTabControl.Changing(NewIndex: Integer;
  var Allowed: Boolean);
begin
  if Assigned(FOnChanging) then
    FOnChanging(Self, NewIndex, Allowed);
end;

{ CanSelectTab returns true if OK to select the tab with the specified index }
function TcsChameleonTabControl.CanSelectTab(ATabIndex: Integer): Boolean;
begin
  Result :=
    TcsTabData(FTabDataList[ATabIndex]).Visible and
    (TcsTabData(FTabDataList[ATabIndex]).Enabled or
     (csDesigning in ComponentState));
  if Result and (ATabIndex <> FTabIndex) then
    Changing(ATabIndex, Result);
end;

procedure TcsChameleonTabControl.SetMargin(Value: Integer);
begin
  if (Value >= 0) and (FMargin <> Value) then
  begin
    FMargin := Value;
    Rebuild;
  end;
end;

procedure TcsChameleonTabControl.WMSize(var Message: TWMSize);
begin
  inherited;
  Rebuild;
end;

procedure TcsChameleonTabControl.SetColoredTabs(Value: Boolean);
begin
  if FColoredTabs <> Value then
  begin
    FColoredTabs := Value;
    Invalidate;
  end;
end;

procedure TcsChameleonTabControl.CMDialogChar(var Message: TCMDialogChar);
var
  I: Integer;
  TabData: TcsTabData;
begin
  if CanFocus then
    with Message do
    begin
      for I := 0 to FTabDataList.Count - 1 do
      begin
        TabData := FTabDataList[I];
        if IsAccel(CharCode, TabData.Caption) and
          TabData.Enabled and TabData.Visible then
        begin { select appropriate Tab and give it focus }
          Result := 1;  { accelerator key is valid (don't beep) }
          SetFocus; { to self }
          if not Focused then Exit; { OnExit event handlers redirected focus }
          SetTabIndex(I);
          if (FTabIndex = I) then
            { page change was allowed }
            TabClick;
          Exit;
        end;
      end;
    end;
  inherited;
end;

{ Draw the Tab's focus rect and perform user-assigned OnClick event method }
procedure TcsChameleonTabControl.TabClick;
begin
  if Assigned(FOnTabClick) then FOnTabClick(Self);
end;

procedure TcsChameleonTabControl.WMGetDlgCode(var Message: TWMGetDlgCode);
begin
  Message.Result := DLGC_WANTARROWS;
end;

procedure TcsChameleonTabControl.KeyDown(var Key: Word; Shift: TShiftState);
var
  NewTabIndex: Integer;
begin
  NewTabIndex := FTabIndex;
  case Key of
    VK_RIGHT, VK_DOWN, VK_LEFT, VK_UP:
    begin
      repeat
        if ((FTabSequence = tsStandard) and ((Key = VK_RIGHT) or (Key = VK_DOWN))) or
          ((FTabSequence = tsReverse) and ((Key = VK_LEFT) or (Key = VK_UP))) then
          { determine 'next' tab }
          if (NewTabIndex = FTabDataList.Count - 1) then
            NewTabIndex := 0
          else
            Inc(NewTabIndex)
        else
          { determine 'previous' tab }
          if (NewTabIndex <= 0) then { first tab or -1 }
            NewTabIndex := FTabDataList.Count - 1
          else
            Dec(NewTabIndex);
        { continue until we find a selectable tab or
          are back where we started
        }
      until (CanSelectTab(NewTabIndex) or (NewTabIndex = FTabIndex));
    end;

    VK_HOME, VK_END:
    begin
      if Key = VK_HOME then
        NewTabIndex := 0
      else { VK_END }
        NewTabIndex := FTabDataList.Count - 1;
      while not ((NewTabIndex = FTabIndex) or CanSelectTab(NewTabIndex)) do
      begin
        if Key = VK_HOME then
          Inc(NewTabIndex)
        else { VK_END }
          Dec(NewTabIndex);
        if (NewTabIndex < 0) or (NewTabIndex = FTabDataList.Count) then
          NewTabIndex := FTabIndex; { must have been -1 }
      end;
    end;
  end;

  if (NewTabIndex <> FTabIndex) then
  begin
    { OnChanging event will have already been generated by CanSelectTab --
      set flag to prevent duplicate event being generated by SetTabIndex
    }
    FChangingDone := True;
    { SetTabIndex is called so that even if NewTabIndex = TabIndex the current
      tab will be brought into view and/or the first row.
    }
    SetTabIndex(NewTabIndex);
    FChangingDone := False;
    TabClick;
  end;
end;

procedure TcsChameleonTabControl.WMSetFocus(var Msg: TWMSetFocus);
begin
  Invalidate;
  inherited;
end;

procedure TcsChameleonTabControl.WMKillFocus(var Msg: TWMKillFocus);
begin
  Invalidate;
  inherited;
end;

procedure TcsChameleonTabControl.CMDesignHitTest(var Msg: TCMDesignHitTest);
const
  TabChosen: Boolean = False;
var
  ShiftState: TShiftState;
  NewTabIndex: Integer;
begin
  ShiftState := KeysToShiftState(Msg.Keys);
  if (ssLeft in ShiftState) then
  begin
    if TabChosen then { tab already chosen, button not released yet }
      Msg.Result := 1 { message handled }
    else
    begin
      NewTabIndex := TabAtPos(Msg.Pos.X, Msg.Pos.Y);
      if (NewTabIndex >= 0) then
      begin
        SetTabIndex(NewTabIndex);
        TabChosen := True;
        Msg.Result := 1; { message handled }
        { Note: Even though the message result is set to indicate that
                the message has been handled, MouseDown still ends up
                being called and must check if the component is in
                design-mode.
        }
      end;
    end;
  end
  else if TabChosen then
  begin
    TabChosen := False; { button released; reset flag }
    Msg.Result := 1; { message handled }
  end;
end;

procedure TcsChameleonTabControl.WMNCHitTest(var Msg: TWMNCHitTest);
begin
  DefaultHandler(Msg);
  FHitTest := SmallPointToPoint(Msg.Pos);
end;

procedure TcsChameleonTabControl.WMSetCursor(var Msg: TWMSetCursor);
var
  TabCursor: HCURSOR;
begin
  TabCursor := 0;
  if (csDesigning in ComponentState) then
    with Msg do
    begin
      if HitTest = HTCLIENT then
      begin
        FHitTest := ScreenToClient(FHitTest);
        if TabAtPos(FHitTest.X, FHitTest.Y) >= 0 then
        begin
          { Note: The CursorLoaded flag is used instead of just testing
                  if Screen.Cursors[crCSTabCursorID] = 0 because for some
                  unknown reason the latter never returns 0.
          }
          if not CursorLoaded then { cursor hasn't been loaded yet }
          begin
            { the cursor is only loaded here, i.e. when in design-mode,
              so that LoadCursor isn't performed by user's apps, at which
              time the resource won't be present
            }
            Screen.Cursors[crCSTabCursorID] := LoadCursor(HInstance, CSTabCursorName);
            CursorLoaded := True;
          end;
          TabCursor := Screen.Cursors[crCSTabCursorID];
        end;
      end;
    end;
  if TabCursor <> 0 then SetCursor(TabCursor)
  else inherited;
end;

procedure TcsChameleonTabControl.CMCtl3DChanged(var Message: TMessage);
begin
  inherited;
  Rebuild;
end;

procedure TcsChameleonTabControl.SetBackgroundColor(Value: TColor);
begin
  if FBackgroundColor <> Value then
  begin
    FBackgroundColor := Value;
    FParentBackgroundColor := False;
    Invalidate;
  end;
end;

procedure TcsChameleonTabControl.SetParentBackgroundColor(Value: Boolean);
begin
  if FParentBackgroundColor <> Value then
  begin
    FParentBackgroundColor := Value;
    if Parent <> nil then Perform(CM_PARENTCOLORCHANGED, 0, 0);
  end;
end;

procedure TcsChameleonTabControl.CMParentColorChanged(var Message: TMessage);
begin
  inherited;
  if FParentBackgroundColor and (Parent <> nil) then
  begin
    { set BackgroundColor to the parent's Color -- this disables
     ParentBackgroundColor so we must re-enable afterwards
    }
    SetBackgroundColor(TGroupBox(Parent).Color); { typecast needed to access }
    FParentBackgroundColor := True;
  end;
end;

procedure TcsChameleonTabControl.CMColorChanged(var Message: TMessage);
begin
  if FColoredTabs then
  begin
    { When tracking the current tab color, self's Color property is set
      (by DrawCard) to the color of the current tab.  Normally, changes to the
      Color property of a TWinControl would cause the control to be Invalidated
      and all its children to be sent a CM_PARENTCOLORCHANGED control message.
      Because self's Color is changed automatically during our Paint processing
      we don't want to Invalidate the control (again) which would cause a
      second paint iteration.
    }
    NotifyControls(CM_PARENTCOLORCHANGED);
  end
  else
    inherited;
end;

procedure TcsChameleonTabControl.DoRealign;
begin
  if not EqualRect(DisplayRect, FPrevDisplayRect) then
    { alignment needed for child controls }
    Realign;
end;

procedure TcsChameleonTabControl.SetAlignTabs(Value: Boolean);
begin
  if FAlignTabs <> Value then
  begin
    FAlignTabs := Value;
    if FMultiLine then
      Rebuild;
  end;
end;

procedure TcsChameleonTabControl.AddCommand(const Command: Integer);
begin
  FCommands.Add(Pointer(Command));
end;

procedure TcsChameleonTabControl.AddCommandPt(const Command: Integer;
  const APoint: TPoint);
begin
  FCommands.Add(Pointer(Command));
  FCommands.Add(Pointer(APoint.X));
  FCommands.Add(Pointer(APoint.Y));
end;

function TcsChameleonTabControl.IsBackgroundColorStored: Boolean;
begin
  Result := not ParentBackgroundColor;
end;

procedure TcsChameleonTabControl.SetBoldCurrentTab(Value: Boolean);
begin
  if FBoldCurrentTab <> Value then
  begin
    FBoldCurrentTab := Value;
    Rebuild;
  end;
end;

procedure TcsChameleonTabControl.SetScrollBtnArrowColor(Value: TColor);
begin
  if FScrollBtnArrowColor <> Value then
  begin
    FScrollBtnArrowColor := Value;
    if FScrollBtns <> nil then
      FScrollBtns.ArrowColor := FScrollBtnArrowColor;
  end;
end;

procedure TcsChameleonTabControl.SetScrollBtnFaceColor(Value: TColor);
begin
  if FScrollBtnFaceColor <> Value then
  begin
    FScrollBtnFaceColor := Value;
    if FScrollBtns <> nil then
      FScrollBtns.FaceColor := FScrollBtnFaceColor;
  end;
end;

{$IFDEF CSTC_TEXTEXTENTCACHE}
function TcsChameleonTabControl.GetTextExtent(const AString: String;
  Horizontal: Boolean): TSize;
begin
  if fsBold in FTextFont.Style then
  begin
    if not FTextExtentCacheBold.Find(AString, Result) then
      Result := CalcTextExtent(AString, Horizontal);
  end
  else
  begin
    if not FTextExtentCacheNormal.Find(AString, Result) then
      Result := CalcTextExtent(AString, Horizontal);
  end;
end;
{$ENDIF}

procedure TcsChameleonTabControl.SetHotTrack(Value: Boolean);
begin
  if (FHotTrack <> Value) then
  begin
    if FHotTrack and not FTabHints then
      CancelHotTrackTimer;
    FHotTrack := Value;
  end;
end;

procedure TcsChameleonTabControl.SetTabHints(Value: Boolean);
begin
  if (FTabHints <> Value) then
  begin
    if FTabHints and not FHotTrack then
      CancelHotTrackTimer;
    FTabHints := Value;
  end;
end;

end.
