{*******************************************************}
{                                                       }
{   Copyright (c) 1997, 1999 Classic Software           }
{   All Rights Reserved                                 }
{                                                       }
{*******************************************************}

unit CSTC32;

{$B-,P+,W-,X+}

interface

uses Classes, Controls, Graphics, CSTCBase;

type

  TcsCustomTabControl = class;

  { A TcsTabCollectionItem object is used to hold the attributes of a single tab. }
  TcsTabCollectionItem = class(TCollectionItem)
  private
    FBitmap: TBitmap;
    FCaption: String;
    FColor: TColor;
    FEnabled: Boolean;
    FHint: String;
    FNumGlyphs: TcsTabNumGlyphs;
    FVisible: Boolean;
    procedure AssignToTabData(Dest: TPersistent);
    procedure BitmapChanged(Sender: TObject);
    procedure SetBitmap(Value: TBitmap);
    procedure SetCaption(const Value: String);
    procedure SetColor(Value: TColor);
    procedure SetEnabled(Value: Boolean);
    procedure SetHint(const Value: String);
    procedure SetNumGlyphs(Value: TcsTabNumGlyphs);
    procedure SetVisible(Value: Boolean);
  protected
    procedure AssignTo(Dest: TPersistent); override;
  public
    constructor Create(Collection: TCollection); override;
    destructor Destroy; override;
    procedure Assign(Source: TPersistent); override;
  published
    property Bitmap: TBitmap read FBitmap write SetBitmap;
    property Caption: String read FCaption write SetCaption;
    property Color: TColor read FColor write SetColor default clBtnFace;
    property Enabled: Boolean read FEnabled write SetEnabled default True;
    property Hint: String read FHint write SetHint;
    property NumGlyphs: TcsTabNumGlyphs read FNumGlyphs write SetNumGlyphs
      default 1;
    property Visible: Boolean read FVisible write SetVisible default True;
  end;

  TcsTabCollection = class(TCollection)
  private
    FTabControl: TcsCustomTabControl;
    function GetItem(Index: Integer): TcsTabCollectionItem;
    procedure SetItem(Index: Integer; Value: TcsTabCollectionItem);
  protected
{$IFDEF VER100}
    function GetOwner: TPersistent; override;
{$ENDIF}
    procedure Update(Item: TCollectionItem); override;
  public
    { Note: Argument to Create is TabControl, *not* owner (so creator is
      responsible for freeing).
    }
    constructor Create(TabControl: TcsCustomTabControl);
    function Add: TcsTabCollectionItem;
    procedure Delete(Index: Integer);
    procedure Move(CurIndex, NewIndex: Integer);
    property Items[Index: Integer]: TcsTabCollectionItem read GetItem write SetItem; default;
  end;

  TcsCustomTabControl = class(TcsChameleonTabControl)
  private
    FTabCollection: TcsTabCollection;
    function GetTabCollection: TcsTabCollection;
    procedure SetTabCollection(Value: TcsTabCollection);
  protected
    procedure Rebuild; override;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    property TabIndex;
    property Tabs: TcsTabCollection read GetTabCollection
      write SetTabCollection;
  end;

  TcsTabControl = class(TcsCustomTabControl)
  public
    property DisplayRect;
  published
    property Align;
    property AlignTabs;
{$IFDEF VER130}
    property Anchors;
{$ENDIF}
{$IFDEF VER120}
    property Anchors;
{$ENDIF}    
    property BackgroundColor;
    property BevelWidth;
    property BoldCurrentTab;
    property ClientArea;
    property Color;
    property ColoredTabs;
{$IFDEF VER130}
    property Constraints;
{$ENDIF}
{$IFDEF VER120}
    property Constraints;
{$ENDIF}
    property CornerSize;
    property Ctl3D;
    property Enabled;
    property Font;
    property FrameBorderColor;
    property FrameHighlightColor;
    property FrameShadowColor;
    property GlyphHAlignment;
    property GlyphMargin;
    property GlyphPosition;
    property GlyphVAlignment;
    property HelpContext;
    property Hint;
    property HotTrack;
    property Margin;
    property MultiLine;
    property ParentBackgroundColor;
    property ParentColor;
    property ParentCtl3D;
    property ParentFont;
    property ParentShowHint;
    property PopupMenu;
    property RowIndent;
    property RowOverlap;
    property ScrollBtnArrowColor;
    property ScrollBtnFaceColor;
    property ShowHint;
    property SidewaysText;
    property TabHeight;
    property TabHints;
    property TabIndex;
    property TabIndexDefault;
    property TabOrder;
    property TabOrientation;
    property Tabs;
    property TabSequence;
    property TabStop;
    property TabStyle;
    property TabWidth;
    property TextColorDisabledHighlight;
    property TextColorDisabledShadow;
    property TextColorHotTrack;    
    property TextColorSelected;
    property TextColorUnselected;
    property TextHAlignment;
    property TextVAlignment;
    property Visible;
    property OnChange;
    property OnChanging;
    property OnClick;
    property OnDblClick;
    property OnDragDrop;
    property OnDragOver;
    property OnEndDrag;
    property OnEnter;
    property OnExit;
    property OnGetFrameColor;  
    property OnGetTextColor;
    property OnMouseDown;
    property OnMouseMove;
    property OnMouseUp;
    property OnPaintBackground;
    property OnPaintCardBackground;
    property OnPaintTabBackground;
    property OnTabClick;
  end;

implementation

{$IFDEF EVALUATION}
uses CSEval;
{$ENDIF}

constructor TcsCustomTabControl.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  { Note: Argument to TcsTabCollection.Create is TabControl, *not* owner. }
  FTabCollection := TcsTabCollection.Create(Self);
end;

destructor TcsCustomTabControl.Destroy;
begin
  FTabCollection.Free;
  inherited Destroy;
end;

function TcsCustomTabControl.GetTabCollection: TcsTabCollection;
begin
  Result := FTabCollection;
end;

procedure TcsCustomTabControl.SetTabCollection(Value: TcsTabCollection);
begin
  GetTabCollection.Assign(Value);
  Rebuild;
end;

procedure TcsCustomTabControl.Rebuild;
var
  I: Integer;
  Tab: TcsTabData;
begin
  { mirror the persistent tab definitions in the internal tab data structures }
  GetTabDataList.Clear;
  for I := 0 to FTabCollection.Count - 1 do
  begin
    Tab := TcsTabData.Create;
    Tab.Assign(FTabCollection[I]); { gets passed to AssignTo }
    GetTabDataList.Add(Tab);
  end;
  inherited Rebuild;
end;

{ ----------------------------- }
{ TcsTabCollectionItem          }
{ ----------------------------- }

constructor TcsTabCollectionItem.Create(Collection: TCollection);
begin
  FBitmap := TBitmap.Create;
  FBitmap.OnChange := BitmapChanged;
  FColor := clBtnFace;
  FEnabled := True;
  FNumGlyphs := 1;
  FVisible := True;
  inherited Create(Collection);
end;

destructor TcsTabCollectionItem.Destroy;
begin
  FBitmap.Free;
  inherited Destroy;
end;

procedure TcsTabCollectionItem.Assign(Source: TPersistent);
begin
  if Source is TcsTabCollectionItem then
  begin
    Caption := TcsTabCollectionItem(Source).Caption;
    Bitmap := TcsTabCollectionItem(Source).Bitmap;
    NumGlyphs := TcsTabCollectionItem(Source).NumGlyphs;
    Color := TcsTabCollectionItem(Source).Color;
    Enabled := TcsTabCollectionItem(Source).Enabled;
    Visible := TcsTabCollectionItem(Source).Visible;
    Hint := TcsTabCollectionItem(Source).Hint;
    Changed(True);
    Exit;
  end;
  inherited Assign(Source); { eventually gets to AssignTo }
end;

procedure TcsTabCollectionItem.AssignTo(Dest: TPersistent);
begin
  if Dest is TcsTabData then
    AssignToTabData(TcsTabData(Dest))
  else
    inherited AssignTo(Dest);
end;

procedure TcsTabCollectionItem.AssignToTabData(Dest: TPersistent);
begin
  with TcsTabData(Dest) do
  begin
    { copy relevant properties of self into TcsTabData destination }
    Caption := Self.Caption;
    Bitmap := Self.Bitmap;
    NumGlyphs := Self.NumGlyphs;
    Color := Self.Color;
    Enabled := Self.Enabled;
    Visible := Self.Visible;
    Hint := Self.Hint;    
  end;
end;

procedure TcsTabCollectionItem.BitmapChanged(Sender: TObject);
begin
  Changed(True);
end;

procedure TcsTabCollectionItem.SetBitmap(Value: TBitmap);
begin
  FBitmap.Assign(Value);
  Changed(True);
end;

procedure TcsTabCollectionItem.SetCaption(const Value: String);
begin
  if FCaption <> Value then
  begin
    FCaption := Value;
    Changed(True);
  end;
end;

procedure TcsTabCollectionItem.SetColor(Value: TColor);
begin
  if FColor <> Value then
  begin
    FColor := Value;
    Changed(True);
  end;
end;

procedure TcsTabCollectionItem.SetEnabled(Value: Boolean);
begin
  if FEnabled <> Value then
  begin
    FEnabled := Value;
    Changed(True);
  end;
end;

procedure TcsTabCollectionItem.SetHint(const Value: String);
begin
  if FHint <> Value then
  begin
    FHint := Value;
    Changed(True);
  end;
end;

procedure TcsTabCollectionItem.SetNumGlyphs(Value: TcsTabNumGlyphs);
begin
  if FNumGlyphs <> Value then
  begin
    FNumGlyphs := Value;
    Changed(True);
  end;
end;

procedure TcsTabCollectionItem.SetVisible(Value: Boolean);
begin
  if FVisible <> Value then
  begin
    FVisible := Value;
    Changed(True);
  end;
end;

{ ----------------------------- }
{ TcsTabCollection              }
{ ----------------------------- }

constructor TcsTabCollection.Create(TabControl: TcsCustomTabControl);
begin
  inherited Create(TcsTabCollectionItem);
  FTabControl := TabControl;
end;

function TcsTabCollection.Add: TcsTabCollectionItem;
begin
  Result := TcsTabCollectionItem(inherited Add);
end;

procedure TcsTabCollection.Delete(Index: Integer);
begin
  { setting an item's Collection property to nil will delete it }
  Items[Index].Collection := nil;
end;

function TcsTabCollection.GetItem(Index: Integer): TcsTabCollectionItem;
begin
  Result := TcsTabCollectionItem(inherited GetItem(Index));
end;

{$IFDEF VER100}
function TcsTabCollection.GetOwner: TPersistent;
begin
  Result := FTabControl;
end;
{$ENDIF}

procedure TcsTabCollection.Move(CurIndex, NewIndex: Integer);
begin
  Items[CurIndex].Index := NewIndex;
end;

procedure TcsTabCollection.SetItem(Index: Integer; Value: TcsTabCollectionItem);
begin
  inherited SetItem(Index, Value);
end;

procedure TcsTabCollection.Update(Item: TCollectionItem);
begin
  FTabControl.Rebuild;
end;

end.
