{*******************************************************}
{                                                       }
{   Copyright (c) 1997, 1999 Classic Software           }
{   All Rights Reserved                                 }
{                                                       }
{*******************************************************}

unit CSTC16;

{$B-,P+,W-,X+}

interface

uses WinTypes, Classes, Controls, Graphics, CSTCBase, Menus;

type

  TcsCustomTabControl16 = class;

  { A TcsTabComponent object is used to hold the attributes of a single tab. }
  TcsTabComponent = class(TComponent)
  private
    FBitmap: TBitmap;
    FCaption: String;
    FColor: TColor;
    FEnabled: Boolean;
    FHint: String;
    FNumGlyphs: TcsTabNumGlyphs;
    FOnChange: TNotifyEvent;
    FVisible: Boolean;
    procedure AssignToTabData(Dest: TPersistent);
    procedure Changed;
    procedure BitmapChanged(Sender: TObject);
    procedure SetBitmap(Value: TBitmap);
    procedure SetCaption(const Value: String);
    procedure SetColor(Value: TColor);
    procedure SetEnabled(Value: Boolean);
    procedure SetHint(const Value: String);
    procedure SetNumGlyphs(Value: TcsTabNumGlyphs);
    procedure SetVisible(Value: Boolean);
    { OnChange property is used by TcsTabComponentList so it will know when
      changes are made to individual tabs; this property is only made
      available to other classes within this unit.
    }
    property OnChange: TNotifyEvent read FOnChange write FOnChange;
  protected
    procedure AssignTo(Dest: TPersistent); override;
    procedure ReadState(Reader: TReader); override;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    procedure Assign(Source: TPersistent); override;
  published
    property Bitmap: TBitmap read FBitmap write SetBitmap;
    property Caption: String read FCaption write SetCaption;
    property Color: TColor read FColor write SetColor default clBtnFace;
    property Enabled: Boolean read FEnabled write SetEnabled default True;
    property Hint: String read FHint write SetHint;
    property NumGlyphs: TcsTabNumGlyphs read FNumGlyphs write SetNumGlyphs
      default 1;
    property Visible: Boolean read FVisible write SetVisible default True;
  end;

  TcsTabComponentList = class(TComponent)
  private
    FTabList: TList;
    FTabControl: TcsCustomTabControl16;
    procedure Changed;
    function GetCount: Integer;
    function GetItem(Index: Integer): TcsTabComponent;
    procedure SetItem(Index: Integer; Value: TcsTabComponent);
    procedure SetTabControl(Value: TcsCustomTabControl16);
    procedure TabChanged(Sender: TObject);
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    function Add(Item: TcsTabComponent): Integer;
    procedure Assign(Source: TPersistent); override;
    procedure Clear;
    procedure Delete(Index: Integer);
    function First: TcsTabComponent;
    function IndexOf(Item: TcsTabComponent): Integer;
    procedure Insert(Index: Integer; Item: TcsTabComponent);
    function Last: TcsTabComponent;
    procedure Move(CurIndex, NewIndex: Integer);
    function Remove(Item: TcsTabComponent): Integer;
    property Count: Integer read GetCount;
    property Items[Index: Integer]: TcsTabComponent read GetItem write SetItem; default;
    property TabControl: TcsCustomTabControl16 read FTabControl
      write SetTabControl;
  end;

  TcsCustomTabControl16 = class(TcsChameleonTabControl)
  private
    FFixupList: TList;
    FTabComponentList: TcsTabComponentList;
    { AddToFixupList is used by the TcsTabComponent class to add a reference
      to a tab component so it can be removed from the owner form's Components
      array after reading in the other components owned by the form (e.g.
      child controls).
    }
    procedure AddToFixupList(ATabComponent: TcsTabComponent);
    function GetTabComponentList: TcsTabComponentList;
    procedure SetTabComponentList(Value: TcsTabComponentList);
  protected
{$IFDEF WIN32}
{$IFDEF VER130}
    procedure GetChildren(Proc: TGetChildProc; Root: TComponent); override;
{$ELSE}
{$IFDEF VER120}
    procedure GetChildren(Proc: TGetChildProc; Root: TComponent); override;
{$ELSE}
{$IFDEF VER100}
    procedure GetChildren(Proc: TGetChildProc; Root: TComponent); override;
{$ELSE}
    procedure GetChildren(Proc: TGetChildProc); override;
{$ENDIF}
{$ENDIF}
{$ENDIF}
{$ENDIF}
    procedure Loaded; override;
    procedure ReadState(Reader: TReader); override;
    procedure Rebuild; override;
{$IFNDEF WIN32}
    procedure WriteComponents(Writer: TWriter); override;
{$ENDIF}
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    property TabIndex;
    property Tabs: TcsTabComponentList read GetTabComponentList
      write SetTabComponentList stored False;
  end;

  TcsTabControl16 = class(TcsCustomTabControl16)
  public
    property DisplayRect;
  published
    property Align;
    property AlignTabs;
{$IFDEF VER130}
    property Anchors;
{$ENDIF}
{$IFDEF VER120}
    property Anchors;
{$ENDIF}
    property BackgroundColor;
    property BevelWidth;
    property BoldCurrentTab;
    property ClientArea;
    property Color;
    property ColoredTabs;
{$IFDEF VER130}
    property Constraints;
{$ENDIF}
{$IFDEF VER120}
    property Constraints;
{$ENDIF}
    property CornerSize;
    property Ctl3D;
    property Enabled;
    property Font;
    property FrameBorderColor;
    property FrameHighlightColor;
    property FrameShadowColor;
    property GlyphHAlignment;
    property GlyphMargin;
    property GlyphPosition;
    property GlyphVAlignment;
    property HelpContext;
    property Hint;
    property HotTrack;
    property Margin;
    property MultiLine;
    property ParentBackgroundColor;
    property ParentColor;
    property ParentCtl3D;
    property ParentFont;
    property ParentShowHint;
    property PopupMenu;
    property RowIndent;
    property RowOverlap;
    property ScrollBtnArrowColor;
    property ScrollBtnFaceColor;
    property ShowHint;
    property SidewaysText;
    property TabHeight;
    property TabHints;
    property TabIndex;
    property TabIndexDefault;
    property TabOrder;
    property TabOrientation;
    property Tabs;
    property TabSequence;
    property TabStop;
    property TabStyle;
    property TabWidth;
    property TextColorDisabledHighlight;
    property TextColorDisabledShadow;
    property TextColorHotTrack;
    property TextColorSelected;
    property TextColorUnselected;
    property TextHAlignment;
    property TextVAlignment;
    property Visible;
    property OnChange;
    property OnChanging;
    property OnClick;
    property OnDblClick;
    property OnDragDrop;
    property OnDragOver;
    property OnEndDrag;
    property OnEnter;
    property OnExit;
    property OnGetFrameColor;
    property OnGetTextColor;
    property OnMouseDown;
    property OnMouseMove;
    property OnMouseUp;
    property OnPaintBackground;
    property OnPaintCardBackground;
    property OnPaintTabBackground;
{$IFDEF WIN32}
    property OnStartDrag;
{$ENDIF}
    property OnTabClick;
  end;

implementation

{$IFDEF EVALUATION}
uses Forms, CSEval;
{$ELSE}
uses Forms;
{$ENDIF}

var
  Registered: Boolean;

{ ----------------------------- }
{ TcsCustomTabControl16         }
{ ----------------------------- }

constructor TcsCustomTabControl16.Create(AOwner: TComponent);
begin
  if not Registered then
  begin
    Classes.RegisterClasses([TcsTabComponent]);
    Registered := True;
  end;
  inherited Create(AOwner);
{$IFDEF WIN32}
  Exclude(FComponentStyle, csInheritable);
{$ENDIF}
  FTabComponentList := TcsTabComponentList.Create(Self);
  FTabComponentList.TabControl := Self;
  FFixupList := TList.Create;
end;

destructor TcsCustomTabControl16.Destroy;
begin
  FFixupList.Free;
  { TabComponentList will be automatically freed by Self (the list's Owner) }
  inherited Destroy;
end;

function TcsCustomTabControl16.GetTabComponentList: TcsTabComponentList;
begin
  Result := FTabComponentList;
end;

procedure TcsCustomTabControl16.SetTabComponentList(Value: TcsTabComponentList);
begin
  GetTabComponentList.Assign(Value);
  Rebuild;
end;

procedure TcsCustomTabControl16.Rebuild;
var
  I: Integer;
  Tab: TcsTabData;
begin
  { mirror the persistent tab definitions in the internal tab data structures }
  GetTabDataList.Clear;
  for I := 0 to FTabComponentList.Count - 1 do
  begin
    Tab := TcsTabData.Create;
    Tab.Assign(FTabComponentList[I]); { gets passed to AssignTo }
    GetTabDataList.Add(Tab);
  end;
  inherited Rebuild;
end;

procedure TcsCustomTabControl16.AddToFixupList(ATabComponent: TcsTabComponent);
begin
  FFixupList.Add(ATabComponent);
end;

procedure TcsCustomTabControl16.ReadState(Reader: TReader);
var
{$IFDEF VER130}
  Form: TCustomForm;
{$ELSE}
{$IFDEF VER120}
  Form: TCustomForm;
{$ELSE}
{$IFDEF VER100}
  Form: TCustomForm;
{$ELSE}
  Form: TForm;
{$ENDIF}
{$ENDIF}
{$ENDIF}
  I: Integer;
  OldTabComponent, NewTabComponent: TcsTabComponent;
begin
  FFixupList.Clear;
  inherited ReadState(Reader);
  { At this point the tab components (TcsTabComponent objects) for this
    tab control will be in the form's Components array.  This isn't the
    desired ownership relationship (we want the Tabs list to own the tab
    components) so we need to create new tab components and add them to the
    Tabs list and then delete the original tab components from the form's
    Components array.  References to the tab components in the form's
    Components array will have been added to FFixupList when they were
    streamed in (by TcsTabComponent.ReadState).
  }
  GetTabComponentList.Clear;
  Form := GetParentForm(Self);
  I := Form.ComponentCount - 1; { start at end to allow deletes as we go }
  while I >= 0 do
  begin
    if (FFixupList.IndexOf(Form.Components[I]) >= 0) then
    begin
      { create new tab component which has Self as owner }
      OldTabComponent := TcsTabComponent(Form.Components[I]);
      NewTabComponent := TcsTabComponent.Create(Self);
      NewTabComponent.Assign(OldTabComponent);
      { Need to use insert to get original order }
      GetTabComponentList.Insert(0, NewTabComponent);
      { Now remove the old tab component from the form's Components list.
        Note that the old tab component can't be freed at this point because
        it is still referred to by the form's FLoaded list -- it will be freed
        later in by the overridden Loaded method.
      }
      Form.RemoveComponent(OldTabComponent);
    end;
    Dec(I);
  end;
  if not ((TabIndex >= 0) and (TabIndex < GetTabComponentList.Count)) then
    TabIndex := -1;
end;

{$IFDEF WIN32}

{$IFDEF VER130}
procedure TcsCustomTabControl16.GetChildren(Proc: TGetChildProc;
  Root: TComponent);
{$ELSE}
{$IFDEF VER120}
procedure TcsCustomTabControl16.GetChildren(Proc: TGetChildProc;
  Root: TComponent);
{$ELSE}
{$IFDEF VER100}
procedure TcsCustomTabControl16.GetChildren(Proc: TGetChildProc;
  Root: TComponent);
{$ELSE}
procedure TcsCustomTabControl16.GetChildren(Proc: TGetChildProc);
{$ENDIF}
{$ENDIF}
{$ENDIF}
var
  I: Integer;
begin
  { Write out the tab components -- when read back in they will initially be
    created with the form as their owner but this is fixed up by ReadState.
  }
  for I := 0 to GetTabComponentList.Count - 1 do
    Proc(GetTabComponentList[I]);
  { Now write out other child controls, e.g. panels, etc. but ignore the
    scroll buttons control created by self.
  }
  for I := 0 to ControlCount - 1 do
    if (Controls[I] <> ScrollBtns) then
      Proc(Controls[I]);
end;

{$ELSE}

procedure TcsCustomTabControl16.WriteComponents(Writer: TWriter);
var
  I: Integer;
begin
  { Write out the tab components -- when read back in they will initially be
    created with the form as their owner but this is fixed up by ReadState.
  }
  for I := 0 to GetTabComponentList.Count - 1 do
  begin
    Writer.WriteComponent(GetTabComponentList[I]);
  end;
  { Now write out other child controls, e.g. panels, etc. but ignore the
    scroll buttons control created by self.
  }
  for I := 0 to ControlCount - 1 do
    if (Controls[I] <> ScrollBtns) then
      Writer.WriteComponent(Controls[I]);
end;

{$ENDIF}

procedure TcsCustomTabControl16.Loaded;
var
  I: Integer;
begin
  inherited Loaded;
  { OK to now free the old tab components }
  for I := 0 to FFixupList.Count - 1 do
    TcsTabComponent(FFixupList[I]).Free;
end;

{ ----------------------------- }
{ TcsTabComponent               }
{ ----------------------------- }

constructor TcsTabComponent.Create(AOwner: TComponent);
var
{$IFDEF VER130}
  OldDesigner: IDesigner;
{$ELSE}
{$IFDEF VER120}
  OldDesigner: IDesigner;
{$ELSE}
  OldDesigner: TDesigner;
{$ENDIF}
{$ENDIF}
begin
  OldDesigner := nil;
  { Do a special check for a Paste operation in design-mode (using
    ComponentState of AOwner because self hasn't been created yet).
  }
  if (csDesigning in AOwner.ComponentState) and
    (not (csReading in AOwner.ComponentState)) and
{$IFDEF VER130}
    (AOwner is TCustomForm) and (TCustomForm(AOwner).Designer <> nil) then
{$ELSE}
{$IFDEF VER120}
    (AOwner is TCustomForm) and (TCustomForm(AOwner).Designer <> nil) then
{$ELSE}
{$IFDEF VER100}
    (AOwner is TCustomForm) and (TCustomForm(AOwner).Designer <> nil) then
{$ELSE}
    (AOwner is TForm) and (TForm(AOwner).Designer <> nil) then
{$ENDIF}
{$ENDIF}
{$ENDIF}
  begin
    { During a Paste operation in design-mode something goes wrong with the
      Designer of the form. The opInsert notification -- caused by creation of
      self with the form as its owner, and sent to the form -- causes a call to
      FDesigner.Notification which crashes.
      [At this stage the exact cause of this problem is unknown.  It probably
       has something to do with the special handling of the tab components by
       the tab control but I'm not sure what exactly.  It doesn't seem to be
       related to the naming of the tab components because the problem occurs
       whether they are given names or not.]
      To prevent the crash, the Designer is temporarily disabled during the
      Paste operation and then restored afterwards.
    }
{$IFDEF VER130}
    OldDesigner := TCustomForm(AOwner).Designer; { save Designer }
    TCustomForm(AOwner).Designer := nil; { disable Designer }
{$ELSE}
{$IFDEF VER120}
    OldDesigner := TCustomForm(AOwner).Designer; { save Designer }
    TCustomForm(AOwner).Designer := nil; { disable Designer }
{$ELSE}
{$IFDEF VER100}
    OldDesigner := TCustomForm(AOwner).Designer; { save Designer }
    TCustomForm(AOwner).Designer := nil; { disable Designer }
{$ELSE}
    OldDesigner := TForm(AOwner).Designer; { save Designer }
    TForm(AOwner).Designer := nil; { disable Designer }
{$ENDIF}
{$ENDIF}
{$ENDIF}
  end;
  inherited Create(AOwner);
  if OldDesigner <> nil then { restore Designer }
{$IFDEF VER130}
    TCustomForm(AOwner).Designer := OldDesigner;
{$ELSE}
{$IFDEF VER120}
    TCustomForm(AOwner).Designer := OldDesigner;
{$ELSE}
{$IFDEF VER100}
    TCustomForm(AOwner).Designer := OldDesigner;
{$ELSE}
    TForm(AOwner).Designer := OldDesigner;
{$ENDIF}
{$ENDIF}
{$ENDIF}
  FBitmap := TBitmap.Create;
  FBitmap.OnChange := BitmapChanged;
  FColor := clBtnFace;
  FEnabled := True;
  FNumGlyphs := 1;
  FVisible := True;
end;

destructor TcsTabComponent.Destroy;
begin
  FBitmap.Free;
  inherited Destroy;
end;

procedure TcsTabComponent.Assign(Source: TPersistent);
var
  TabComponent: TcsTabComponent;
begin
  if Source is TcsTabComponent then
  begin
    TabComponent := TcsTabComponent(Source);
    Caption := TabComponent.Caption;
    Bitmap := TabComponent.Bitmap;
    NumGlyphs := TabComponent.NumGlyphs;
    Color := TabComponent.Color;
    Enabled := TabComponent.Enabled;
    Visible := TabComponent.Visible;
    Hint := TabComponent.Hint;
    Changed;
    Exit;
  end;
  inherited Assign(Source); { eventually gets to AssignTo }
end;

procedure TcsTabComponent.AssignTo(Dest: TPersistent);
begin
  if Dest is TcsTabData then
    AssignToTabData(TcsTabData(Dest))
  else
    inherited AssignTo(Dest);
end;

procedure TcsTabComponent.AssignToTabData(Dest: TPersistent);
begin
  with TcsTabData(Dest) do
  begin
    { copy relevant properties of self into TcsTabData destination }
    Caption := Self.Caption;
    Bitmap := Self.Bitmap;
    NumGlyphs := Self.NumGlyphs;
    Color := Self.Color;
    Enabled := Self.Enabled;
    Visible := Self.Visible;
    Hint := Self.Hint;
  end;
end;

procedure TcsTabComponent.ReadState(Reader: TReader);
begin
  { Reader.Parent is the enclosing component of Self in the form file }
  if (Reader.Parent is TcsCustomTabControl16) then
    { add a reference to self into the tab control's fixup list }
    TcsCustomTabControl16(Reader.Parent).AddToFixupList(Self);
  inherited ReadState(Reader);
end;

procedure TcsTabComponent.BitmapChanged(Sender: TObject);
begin
  Changed;
end;

procedure TcsTabComponent.SetBitmap(Value: TBitmap);
begin
  FBitmap.Assign(Value);
  Changed;
end;

procedure TcsTabComponent.SetCaption(const Value: String);
begin
  if FCaption <> Value then
  begin
    FCaption := Value;
    Changed;
  end;
end;

procedure TcsTabComponent.SetColor(Value: TColor);
begin
  if FColor <> Value then
  begin
    FColor := Value;
    Changed;
  end;
end;

procedure TcsTabComponent.SetEnabled(Value: Boolean);
begin
  if FEnabled <> Value then
  begin
    FEnabled := Value;
    Changed;
  end;
end;

procedure TcsTabComponent.SetHint(const Value: String);
begin
  if FHint <> Value then
  begin
    FHint := Value;
    Changed;
  end;
end;

procedure TcsTabComponent.SetNumGlyphs(Value: TcsTabNumGlyphs);
begin
  if FNumGlyphs <> Value then
  begin
    FNumGlyphs := Value;
    Changed;
  end;
end;

procedure TcsTabComponent.SetVisible(Value: Boolean);
begin
  if FVisible <> Value then
  begin
    FVisible := Value;
    Changed;
  end;
end;

procedure TcsTabComponent.Changed;
begin
  if Assigned(FOnChange) then FOnChange(Self);
end;

{ ----------------------------- }
{ TcsTabComponentList           }
{ ----------------------------- }

constructor TcsTabComponentList.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FTabList := TList.Create;
end;

destructor TcsTabComponentList.Destroy;
begin
  { Items in list will be destroyed automatically by their Owner (the form or
    the tab control) because they are in their Owner's Components list;
    hence, shouldn't free each item here (via Clear), just free the list.
  }
  FTabList.Free;
  inherited Destroy;
end;

procedure TcsTabComponentList.Assign(Source: TPersistent);
var
  AList: TcsTabComponentList;
  I: Integer;
  TabComponent: TcsTabComponent;
begin
  if (Source is TcsTabComponentList) and (FTabControl <> nil) then
  begin
    Clear;
    AList := TcsTabComponentList(Source);
    for I := 0 to AList.Count - 1 do
    begin
      TabComponent := TcsTabComponent.Create(FTabControl);
      TabComponent.Assign(AList[I]);
      Add(TabComponent);
    end;
    Exit;
  end;
  inherited Assign(Source);
end;

function TcsTabComponentList.Add(Item: TcsTabComponent): Integer;
begin
  Item.OnChange := TabChanged;
  Result := FTabList.Add(Item);
  Changed;
end;

procedure TcsTabComponentList.Clear;
var
  I: Integer;
begin
  for I := 0 to FTabList.Count - 1 do
    Items[I].Free;
  FTabList.Clear;
  Changed;
end;

procedure TcsTabComponentList.Delete(Index: Integer);
begin
  Items[Index].Free;
  FTabList.Delete(Index);
  Changed;
end;

function TcsTabComponentList.First: TcsTabComponent;
begin
  Result := TcsTabComponent(FTabList.First);
end;

function TcsTabComponentList.GetCount: Integer;
begin
  Result := FTabList.Count;
end;

function TcsTabComponentList.GetItem(Index: Integer): TcsTabComponent;
begin
  Result := TcsTabComponent(FTabList[Index]);
end;

function TcsTabComponentList.IndexOf(Item: TcsTabComponent): Integer;
begin
  Result := FTabList.IndexOf(Item);
end;

procedure TcsTabComponentList.Insert(Index: Integer; Item: TcsTabComponent);
begin
  Item.OnChange := TabChanged;
  FTabList.Insert(Index, Item);
  Changed;
end;

function TcsTabComponentList.Last: TcsTabComponent;
begin
  Result := TcsTabComponent(FTabList.Last);
end;

procedure TcsTabComponentList.Move(CurIndex, NewIndex: Integer);
var
  Item: TcsTabComponent;
begin
  if CurIndex <> NewIndex then
  begin
    Item := Items[CurIndex];
    { don't use Self.Delete because it frees item }
    FTabList.Delete(CurIndex);
    Insert(NewIndex, Item);
    Changed;
  end;
end;

procedure TcsTabComponentList.SetItem(Index: Integer; Value: TcsTabComponent);
begin
  Items[Index].Free;
  Value.OnChange := TabChanged;
  Items[Index] := Value;
end;

function TcsTabComponentList.Remove(Item: TcsTabComponent): Integer;
begin
  Result := IndexOf(Item);
  if Result <> -1 then
    Delete(Result);
end;

procedure TcsTabComponentList.SetTabControl(Value: TcsCustomTabControl16);
var
  I: Integer;
begin
  if Value <> FTabControl then
  begin
    if FTabControl <> nil then
      { detach OnChange handler for each item }
      for I := 0 to FTabList.Count - 1 do
        Items[I].OnChange := nil;
    FTabControl := Value;
    if FTabControl <> nil then
      { attach OnChange handler for each item }
      for I := 0 to FTabList.Count - 1 do
        Items[I].OnChange := TabChanged;
    Changed;
  end;
end;

procedure TcsTabComponentList.TabChanged;
begin
  if FTabControl <> nil then
    FTabControl.Rebuild;
end;

procedure TcsTabComponentList.Changed;
begin
  TabChanged(Self);
end;

initialization
  Registered := False;

end.
