{*******************************************************}
{                                                       }
{   Copyright (c) 1996 Classic Software                 }
{   All rights reserved                                 }
{                                                       }
{*******************************************************}

unit CSStrTbl;

{$B-,P+,W-,X+}

interface

uses
  SysUtils, WinTypes, WinProcs, Messages, Classes, Graphics, Controls,
  Forms, Dialogs;

type

  TcsStringTable = class(TObject)
  private
    FColumnList: TList;
    FOnChange: TNotifyEvent;
    FRowExtent: Integer; { only used when creating new columns }
    FSelectedRows: TStringList; { each item is IntToStr() of row no. }
    function GetCell(ACol, ARow: Integer): String;
    function GetColCount: Integer;
    function GetObject(ACol, ARow: Integer): TObject;
    function GetRowCount: Integer;
    procedure SetCell(ACol, ARow: Integer; Value: String);
    procedure SetObject(ACol, ARow: Integer; Value: TObject);
  protected
    procedure Changed; virtual;
    function NewColumn: TStringList; virtual;
    function GetSelected(ARow: Integer): Boolean; virtual;
    procedure SetSelected(ARow: Integer; Value: Boolean); virtual;
  public
    constructor Create(NumCols, NumRows: Integer); virtual;
    destructor Destroy; override;
    procedure AddColumn; virtual;
    procedure AddRow; virtual;
    procedure ChangeSize(NewColCount, NewRowCount: LongInt); virtual;
    procedure Clear; virtual;
    procedure ClearSelected; virtual;
    procedure DeleteColumn(ACol: Integer); virtual;
    procedure DeleteRow(ARow: Integer); virtual;
    procedure InsertColumn(ACol: Integer); virtual;
    procedure InsertRow(ARow: Integer); virtual;
    procedure MoveColumn(FromIndex, ToIndex: LongInt); virtual;
    procedure MoveRow(FromIndex, ToIndex: LongInt); virtual;
    property Cells[ACol, ARow: Integer]: String read GetCell write SetCell;
    property ColCount: Integer read GetColCount;
    property Objects[ACol, ARow: Integer]: TObject read GetObject write SetObject;
    property RowCount: Integer read GetRowCount;
    property Selected[ARow: Integer]: Boolean read GetSelected write SetSelected;
    property SelectedRows: TStringList read FSelectedRows;
    property OnChange: TNotifyEvent read FOnChange write FOnChange;
  end;

implementation

{$IFDEF EVALUATION} uses CSEval; {$ENDIF}

{ TcsStringTable }

constructor TcsStringTable.Create(NumCols, NumRows: Integer);
var I: Integer;
begin
  inherited Create;
  FOnChange := nil;
  FSelectedRows := TStringList.Create;
  FColumnList := TList.Create;
  if NumCols < 1 then NumCols := 1;
  if NumRows < 1 then NumRows := 1;
  FRowExtent := NumRows;
  for I := 0 to NumCols - 1 do
    AddColumn;
end;

destructor TcsStringTable.Destroy;
var I: Integer;
begin
  FOnChange := nil;
  FSelectedRows.Free;
  for I := 0 to FColumnList.Count - 1 do
    TStringList(FColumnList[I]).Free;
  FColumnList.Free;
  inherited Destroy;
end;

function TcsStringTable.GetCell(ACol, ARow: Integer): String;
var ColumnCells: TStringList;
begin
  try
    ColumnCells := TStringList(FColumnList[ACol]);
    Result := ColumnCells[ARow];
  except
    Result := '';
  end;
end;

procedure TcsStringTable.SetCell(ACol, ARow: Integer; Value: String);
var ColumnCells: TStringList;
begin
  if (FColumnList <> nil) and (ACol < FColumnList.Count) then
  begin
    ColumnCells := TStringList(FColumnList[ACol]);
    if (ColumnCells <> nil) and (ARow < ColumnCells.Count) then
    begin
      ColumnCells[ARow] := Value;
      Changed;
    end;
  end;
end;

function TcsStringTable.GetObject(ACol, ARow: Integer): TObject;
var ColumnCells: TStringList;
begin
  try
    ColumnCells := TStringList(FColumnList[ACol]);
    Result := ColumnCells.Objects[ARow];
  except
    Result := nil;
  end;
end;

procedure TcsStringTable.SetObject(ACol, ARow: Integer; Value: TObject);
var ColumnCells: TStringList;
begin
  if (FColumnList <> nil) and (ACol < FColumnList.Count) then
  begin
    ColumnCells := TStringList(FColumnList[ACol]);
    if (ColumnCells <> nil) and (ARow < ColumnCells.Count) then
    begin
      ColumnCells.Objects[ARow] := Value;
      Changed;
    end;
  end;
end;

procedure TcsStringTable.Changed;
begin
  if Assigned(FOnChange) then FOnChange(Self);
end;

{ Add a new row to the end of the existing rows. }
procedure TcsStringTable.AddRow;
var I: Integer;
begin
  for I := 0 to FColumnList.Count - 1 do
    TStringList(FColumnList[I]).Add('');
  Changed;
end;

{ Delete the specified row.
  Adjust the keys in the list of selected rows to reflect the deletion.
}
procedure TcsStringTable.DeleteRow(ARow: Integer);
var I: Integer;
    NewKey: Integer;
begin
  if (ARow < 0) or (ARow >= RowCount) then
    Exit;

  for I := 0 to FSelectedRows.Count - 1 do
  begin
    { If row to be deleted is in selected rows then remove its
      index from selected rows first.
      Decrement index values of all other selected rows.
    }
    if (StrToInt(FSelectedRows[I]) = ARow) then
    begin
      FSelectedRows.Delete(I);
    end
    else if (StrToInt(FSelectedRows[I]) > ARow) then
    begin
      NewKey := StrToInt(FSelectedRows[I]);
      Dec(NewKey);
      FSelectedRows[I] := IntToStr(NewKey);
    end;
  end;

  { delete the cell for the specified row in each column }
  for I := 0 to FColumnList.Count - 1 do
    TStringList(FColumnList[I]).Delete(ARow);
  Changed;
end;

procedure TcsStringTable.Clear;
var I, J: Integer;
begin
  { remove all selected row info }
  FSelectedRows.Clear;
  { blank all cells }
  for I := 0 to ColCount - 1 do
    for J := 0 to RowCount - 1 do
      SetCell(I, J, '');
  Changed;
end;

procedure TcsStringTable.ChangeSize(NewColCount, NewRowCount: LongInt);
var I, OldColCount, OldRowCount: Integer;
begin
  OldColCount := FColumnList.Count;
  { Assert: FColumnList.Count > 0 }
  OldRowCount := TStringList(FColumnList[0]).Count;
  FRowExtent := NewRowCount;

  { adjust column count }
  if (NewColCount > OldColCount) then
    for I := OldColCount to NewColCount - 1 do
      AddColumn
  else if (NewColCount < OldColCount) then
    { loop from end of list when deleting to ensure proper coverage }
    for I := OldColCount - 1 downto NewColCount do
      DeleteColumn(I);

  { adjust row count }
  if (NewRowCount > OldRowCount) then
    for I := OldRowCount to NewRowCount - 1 do
      AddRow
  else if (NewRowCount < OldRowCount) then
    for I := OldRowCount - 1 downto NewRowCount do
      DeleteRow(I);
end;

procedure TcsStringTable.DeleteColumn(ACol: Integer);
begin
  if (ACol >= 0) and (ACol < FColumnList.Count) then
  begin
    TStringList(FColumnList[ACol]).Free;
    FColumnList.Delete(ACol);
    Changed;
  end;
end;

procedure TcsStringTable.MoveColumn(FromIndex, ToIndex: LongInt);
var ColumnCells: TStringList;
begin
  if (FromIndex < 0) or (FromIndex >= ColCount) or
    (ToIndex < 0) or (ToIndex >= ColCount) then
    Exit;
  ColumnCells := FColumnList[FromIndex];
  FColumnList.Delete(FromIndex);
  FColumnList.Insert(ToIndex, ColumnCells);
  Changed;
end;

procedure TcsStringTable.MoveRow(FromIndex, ToIndex: LongInt);
var I, OldKey: Integer;
begin
  if (FromIndex < 0) or (FromIndex >= RowCount) or
    (ToIndex < 0) or (ToIndex >= RowCount) then
    Exit;

  { adjust selected row list }
  for I := 0 to FSelectedRows.Count - 1 do
  begin
    OldKey := StrToInt(FSelectedRows[I]);
    if (OldKey = FromIndex) then
      OldKey := ToIndex
    else if (OldKey = ToIndex) and (OldKey < FromIndex) then
      Inc(OldKey)
    else if (OldKey = ToIndex) and (OldKey > FromIndex) then
      Dec(OldKey)
    else if (OldKey > FromIndex) and (OldKey < ToIndex) then
      Dec(OldKey)
    else if (OldKey < FromIndex) and (OldKey > ToIndex) then
      Inc(OldKey);
    FSelectedRows[I] := IntToStr(OldKey);
  end;

  { move the row in each column }
  for I := 0 to FColumnList.Count - 1 do
    TStringList(FColumnList[I]).Move(FromIndex, ToIndex);
  Changed;
end;

{ Similar to AddRow but uses Insert/InsertObject
  instead of Add/AddObject.
}
procedure TcsStringTable.InsertRow(ARow: Integer);
var I, OldKey: Integer;
begin
  { adjust selected row list }
  for I := 0 to FSelectedRows.Count - 1 do
  begin
    OldKey := StrToInt(FSelectedRows[I]);
    if (OldKey >= ARow) then
      Inc(OldKey);
    FSelectedRows[I] := IntToStr(OldKey);
  end;

  for I := 0 to FColumnList.Count - 1 do
    TStringList(FColumnList[I]).Insert(ARow, '');
  Changed;
end;

{ Common function used by AddColumn and InsertColumn }
function TcsStringTable.NewColumn: TStringList;
var I: Integer;
begin
  Result := TStringList.Create;
  for I := 0 to FRowExtent - 1 do
    Result.Add('');
end;

procedure TcsStringTable.AddColumn;
begin
  FColumnList.Add(NewColumn);
  Changed;
end;

procedure TcsStringTable.InsertColumn(ACol: Integer);
begin
  FColumnList.Insert(ACol, NewColumn);
  Changed;
end;

{ Selected/deselect the specified row. }
procedure TcsStringTable.SetSelected(ARow: Integer; Value: Boolean);
  { Find the position (index) of the specified row (key) in the list
    of selected rows.  Currently uses sequential search; OK for small
    number of selections.  If a large number of selections were
    expected then the list of selected rows should perhaps be kept
    sorted and a binary search used.
  }
  function IndexOfKey: Integer;
  var I: Integer;
  begin
    Result := -1;
    for I := 0 to FSelectedRows.Count - 1 do
      if (StrToInt(FSelectedRows[I]) = ARow) then
      begin
        Result := I;
        Exit;
      end;
  end;

begin
  if (IndexOfKey >= 0) and not Value then
    { row is already present and is being deselected; remove from list }
    FSelectedRows.Delete(IndexOfKey)
  else if (IndexOfKey < 0) and Value then
    { row is not in list yet and is being selected; add to list }
    FSelectedRows.Add(IntToStr(ARow));
end;

{ Uses a sequential search to determine if the specified row is
  in the list of selected rows.
}
function TcsStringTable.GetSelected(ARow: Integer): Boolean;
var I: Integer;
begin
  Result := False;
  for I := 0 to FSelectedRows.Count - 1 do
    if (StrToInt(FSelectedRows[I]) = ARow) then
    begin
      Result := True;
      Exit;
    end;
end;

function TcsStringTable.GetColCount: Integer;
begin
  Result := FColumnList.Count;
end;

function TcsStringTable.GetRowCount: Integer;
begin
  if FColumnList.Count = 0 then
    Result := 0
  else
    Result := TStringList(FColumnList[0]).Count;
end;

procedure TcsStringTable.ClearSelected;
begin
  FSelectedRows.Clear;
end;

end.
