{*******************************************************}
{                                                       }
{   Copyright (c) 1996, 1999 Classic Software           }
{   All rights reserved                                 }
{                                                       }
{*******************************************************}

unit CSSclptB;

{$B-,P+,W-,X+}

interface

uses
  SysUtils, WinTypes, WinProcs, Messages, Classes, Graphics, Controls,
  Forms, Buttons;

type
  TcsBevelWidth = 0..2;
  TcsTextPosition = (tpCentered, tpXY);

  TcsSculptButton = class(TGraphicControl)
  private
    FAutoSize: Boolean;
    FBevelWidth: TcsBevelWidth;
    FBevelHighlightColor: TColor;
    FBevelShadowColor: TColor;
    FBitmap: TBitmap;
    FBitmapUp: TBitmap;
    FBitmapDown: TBitmap;
    FBorderStyle: TBorderStyle;
    FBorderColor: TColor;
    FDragging: Boolean;
    FHitTestMask: TBitmap;
    FPrevCursorSaved: Boolean;
    FPrevCursor: TCursor;
    FPrevShowHintSaved: Boolean;
    FPrevShowHint: Boolean;
    FPrevParentShowHint: Boolean;
    FPreciseClick: Boolean;
    FPreciseShowHint: Boolean;
    FSpeckled: Boolean;
    FSpeckleOpaqueColor: TColor;
    FSpeckleTransparentColor: TColor;
    FTextPosition: TcsTextPosition;
    FTextX: Integer;
    FTextY: Integer;
    procedure AddSpeckle(Source: TBitmap; TransparentColor: TColor);
    procedure AdjustBounds;
    procedure AdjustButtonSize(var W, H: Integer);
    function BevelColor(const AState: TButtonState; const TopLeft: Boolean): TColor;
    procedure BitmapChanged(Sender: TObject);
    procedure Create3DBitmap(Source: TBitmap; const AState: TButtonState; Target: TBitmap);
    procedure InitPalette(DC: HDC);
    procedure SetAutoSize(Value: Boolean);
    procedure SetBevelHighlightColor(Value: TColor);
    procedure SetBevelShadowColor(Value: TColor);
    procedure SetBevelWidth(Value: TcsBevelWidth);
    procedure SetBitmap(Value: TBitmap);
    procedure SetBitmapDown(Value: TBitmap);
    procedure SetBitmapUp(Value: TBitmap);
    procedure SetBorderColor(Value: TColor);
    procedure SetBorderStyle(Value: TBorderStyle);
    procedure SetSpeckled(Value: Boolean);
    procedure SetSpeckleOpaqueColor(Value: TColor);
    procedure SetSpeckleTransparentColor(Value: TColor);
    procedure SetTextPosition(Value: TcsTextPosition);
    procedure SetTextX(Value: Integer);
    procedure SetTextY(Value: Integer);
    procedure CMDialogChar(var Message: TCMDialogChar); message CM_DIALOGCHAR;
    procedure CMFontChanged(var Message: TMessage); message CM_FONTCHANGED;
    procedure CMTextChanged(var Message: TMessage); message CM_TEXTCHANGED;
    procedure CMSysColorChange(var Message: TMessage); message CM_SYSCOLORCHANGE;
  protected
    FState: TButtonState;
    procedure DefineProperties(Filer: TFiler); override;
    procedure DrawButtonText(Canvas: TCanvas; const Caption: String;
      TextBounds: TRect; State: TButtonState); virtual;
    function GetPalette: HPALETTE; override;
    function GetTextRect(Canvas: TCanvas; const Caption: String): TRect; virtual;
    procedure Loaded; override;
    procedure MouseDown(Button: TMouseButton; Shift: TShiftState;
      X, Y: Integer); override;
    procedure MouseMove(Shift: TShiftState; X, Y: Integer); override;
    procedure MouseUp(Button: TMouseButton; Shift: TShiftState;
      X, Y: Integer); override;
    procedure Paint; override;
    procedure ReadBitmapDownData(Stream: TStream); virtual;
    procedure ReadBitmapUpData(Stream: TStream); virtual;
    procedure WriteBitmapDownData(Stream: TStream); virtual;
    procedure WriteBitmapUpData(Stream: TStream); virtual;
    { BorderStyle should be bsSingle to get the best effect between
      the button's up and down 3D images.  However, it can be changed
      without any negative side-effects.
    }
    property BorderStyle: TBorderStyle read FBorderStyle write SetBorderStyle default bsSingle;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    procedure Click; override;
    procedure Invalidate; override;
    function PtInMask(const X, Y: Integer): Boolean; virtual;
    procedure SetBounds(ALeft, ATop, AWidth, AHeight: Integer); override;
    procedure SetTextXY(const X, Y: Integer); virtual;
    property BitmapUp: TBitmap read FBitmapUp;
    property BitmapDown: TBitmap read FBitmapDown;
  published
{$IFDEF VER130}
    property Anchors;
{$ENDIF}
{$IFDEF VER120}
    property Anchors;
{$ENDIF}
    property AutoSize: Boolean read FAutoSize write SetAutoSize default True;
    property BevelHighlightColor: TColor read FBevelHighlightColor write SetBevelHighlightColor default clBtnHighlight;
    property BevelShadowColor: TColor read FBevelShadowColor write SetBevelShadowColor default clBtnShadow;
    property BevelWidth: TcsBevelWidth read FBevelWidth write SetBevelWidth default 2;
    property Bitmap: TBitmap read FBitmap write SetBitmap;
    property BorderColor: TColor read FBorderColor write SetBorderColor default clBlack;
    property Caption;
{$IFDEF VER130}
    property Constraints;
{$ENDIF}
{$IFDEF VER120}
    property Constraints;
{$ENDIF}
    property DragCursor;
    property DragMode;
    property Enabled;
    property Font;
    property ParentFont;
    property ParentShowHint;
    property PreciseClick: Boolean read FPreciseClick write FPreciseClick default True;
    property PreciseShowHint: Boolean read FPreciseShowHint write FPreciseShowHint default True;
    property ShowHint;
    property Speckled: Boolean read FSpeckled write SetSpeckled default False;
    property SpeckleOpaqueColor: TColor read FSpeckleOpaqueColor write SetSpeckleOpaqueColor default clWhite;
    property SpeckleTransparentColor: TColor read FSpeckleTransparentColor write SetSpeckleTransparentColor default clBlack;
    property TextPosition: TcsTextPosition read FTextPosition write SetTextPosition default tpCentered;
    property TextX: Integer read FTextX write SetTextX default 0;
    property TextY: Integer read FTextY write SetTextY default 0;
    property Visible;
    property OnClick;
    property OnDragDrop;
    property OnDragOver;
    property OnEndDrag;
    property OnMouseDown;
    property OnMouseMove;
    property OnMouseUp;
  end;

implementation

type
  PointArray = Array[0..1] of Integer;

{ Make a copy of a logical palette, returning the handle of the new palette. }
function CopyPalette(SrcPalette: HPalette): HPalette;
var
  Count: Cardinal;
  LogPal: PLogPalette;
begin
  Result := 0;
  Count := 0; { must init. because GetObject only passes back a 16-bit value }
  { Is there a source palette? If not, then return zero. }
  if SrcPalette = 0 then Exit;

  { Get the number of entries in the source palette. }
  if GetObject(SrcPalette, SizeOf(Count), @Count) = 0 then
    raise Exception.Create('Invalid palette in CopyPalette');
  if Count = 0 then
  begin
    { No entries is the equivalent of no palette. }
    Result := 0;
    Exit;
  end;

  { TLogPalette already has room for one TPaletteEntry, so allocate
    memory for an additional Count-1 entries. }
  GetMem(LogPal, SizeOf(TLogPalette) + (Count-1) * SizeOf(TPaletteEntry));
  try
    { Get the palette entries from the source palette. }
    if GetPaletteEntries(SrcPalette, 0, Count, LogPal^.palPalEntry) <> Count
    then
      raise Exception.Create('Cannot get palette entries in CopyPalette');
    LogPal^.palVersion := $300;
    LogPal^.palNumEntries := Count;
    { Create a new palette. }
    Result := CreatePalette(LogPal^);
    if Result = 0 then
      raise EOutOFResources.Create('Cannot create palette in CopyPalette');
  finally
    FreeMem(LogPal, SizeOf(TLogPalette) + (Count-1) * SizeOf(TPaletteEntry));
  end;
end;

{ Create a monochrome bitmap mask for use when overlaying images or
  when performing hit-testing.
}
function CreateMonoMask(ColorBmp: TBitmap; TransparentColor: TColor): TBitmap;
var R: TRect;
    OldBkColor: TColorRef;
begin
  Result := TBitmap.Create;
  try
    Result.Monochrome := True;
    Result.Width := ColorBmp.Width;
    Result.Height := ColorBmp.Height;
    { Set background color for source bitmap -- this will be used
      when copying to convert from a color bitmap to a mono bitmap
    }
    OldBkColor := SetBkColor(ColorBmp.Canvas.Handle, ColorToRGB(TransparentColor));
    R := Rect(0, 0, ColorBmp.Width, ColorBmp.Height);
    { Now copy to monochrome bitmap; all pixels in source bitmap that
      were the transparent color will be white in the destination bitmap,
      all other pixels will be black
    }
    Result.Canvas.CopyMode := cmSrcCopy;
    Result.Canvas.CopyRect(R, ColorBmp.Canvas, R);
    SetBkColor(ColorBmp.Canvas.Handle, OldBkColor);
  except
    Result.Free;
    Raise;
  end;

end;

function CreateMonoOutlineMask(Source, NewSource: TBitmap; const OffsetPts: Array of PointArray;
  TransparentColor: TColor): TBitmap;
var I, W, H: Integer;
    R, NewR: TRect;
    SmallMask, BigMask, NewSourceMask: TBitmap;
begin
  Result := TBitmap.Create;
  try
    W := Source.Width;
    H := Source.Height;
    R := Rect(0, 0, W, H);

    Result.Monochrome := True;
    Result.Width := W;
    Result.Height := H;

    SmallMask := CreateMonoMask(Source, TransparentColor);
    NewSourceMask := CreateMonoMask(NewSource, TransparentColor);
    BigMask := CreateMonoMask(NewSourceMask, TransparentColor);

    try

      BigMask.Canvas.CopyMode := cmSrcCopy;
      BigMask.Canvas.CopyRect(R, NewSourceMask.Canvas, R);

      for I := Low(OffsetPts) to High(OffsetPts) do
      begin
        if (OffsetPts[I, 0] = 0) and (OffsetPts[I, 1] = 0) then
          Break;
        NewR := R;
        OffsetRect(NewR, OffsetPts[I, 0], OffsetPts[I, 1]);
        BigMask.Canvas.CopyMode := cmSrcAnd; { DSa }
        BigMask.Canvas.CopyRect(NewR, SmallMask.Canvas, R);
      end;
      BigMask.Canvas.CopyMode := cmSrcCopy;

      with Result do
      begin
        Canvas.CopyMode := cmSrcCopy;
        Canvas.CopyRect(R, NewSourceMask.Canvas, R);
        Canvas.CopyMode := $00DD0228; { SDno }
        Canvas.CopyRect(R, BigMask.Canvas, R);
        Canvas.CopyMode := cmSrcCopy;
      end;

    finally
      SmallMask.Free;
      NewSourceMask.Free;
      BigMask.Free;
    end;

  except
    Result.Free;
    Raise;
  end;

end;

{ TcsSculptButton }
constructor TcsSculptButton.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  SetBounds(0, 0, 80, 80);
  ControlStyle := [csCaptureMouse, csOpaque];
  FAutoSize := True;
  FBitmap := TBitmap.Create;
  FBitmap.OnChange := BitmapChanged;
  FBitmapUp := TBitmap.Create;
  FBitmapDown := TBitmap.Create;
  FHitTestMask := nil;
  ParentFont := True;
  FBevelWidth := 2;
  FBorderStyle := bsSingle;
  FState := bsUp;
  FPreciseClick := True;
  FPreciseShowHint := True;
  FBorderColor := clBlack;
  FBevelHighlightColor := clBtnHighlight;
  FBevelShadowColor := clBtnShadow;
  FSpeckleOpaqueColor := clWhite;
  FSpeckleTransparentColor := clBlack;
  FTextPosition := tpCentered;
end;

destructor TcsSculptButton.Destroy;
begin
  FBitmap.Free;
  FBitmapUp.Free;
  FBitmapDown.Free;
  FHitTestMask.Free;
  inherited Destroy;
end;

procedure TcsSculptButton.Paint;
var W, H: Integer;
    Composite, Mask, Overlay, CurrentBmp: TBitmap;
    R, NewR: TRect;
    BrushHandle: hBrush;
begin
  if csDesigning in ComponentState then
    with Canvas do
    begin
      Pen.Style := psDash;
      Brush.Style := bsClear;
      Rectangle(0, 0, Width, Height);
    end;

  if (csDesigning in ComponentState) or
    (FState in [bsDisabled, bsExclusive]) then
    FState := bsUp;

  if (FState = bsUp) then CurrentBmp := FBitmapUp
  else CurrentBmp := FBitmapDown;

  if not CurrentBmp.Empty then
  begin

    W := Width;
    H := Height;
    R := ClientRect;
    NewR := R;

    Composite := TBitmap.Create;
    Overlay := TBitmap.Create;

    { When not using a palette (4, 16 or 24 bit color) CopyPalette returns
      without doing anything, and thus doesn't impact performance on these
      systems.
    }
    Composite.Palette := CopyPalette(FBitmap.Palette);
    Overlay.Palette := CopyPalette(FBitmap.Palette);
    InitPalette(Composite.Canvas.Handle);
    InitPalette(Overlay.Canvas.Handle);
    InitPalette(Canvas.Handle);

    try
      with Composite do
      begin
        Width := W;
        Height := H;
        Canvas.CopyMode := cmSrcCopy;
        Canvas.CopyRect(R, Self.Canvas, R); { start with existing background }
      end;

      with Overlay do
      begin
        Width := W;
        Height := H;
        Canvas.CopyMode := cmSrcCopy;
        BrushHandle := CreateSolidBrush(ColorToRGB(FBitmap.TransparentColor));
        try
          FillRect(Canvas.Handle, R, BrushHandle);
        finally
          DeleteObject(BrushHandle);
        end;
        if FState = bsDown then
          OffsetRect(NewR, 1, 1);
        Canvas.CopyRect(NewR, CurrentBmp.Canvas, R);
      end;

      Mask := CreateMonoMask(Overlay, FBitmap.TransparentColor);
      try
        { Combine the mask with the existing background; this will give
          the background with black ('holes') where the overlay will
          eventually be shown
        }
        Composite.Canvas.CopyMode := cmSrcAnd; { DSa }
        Composite.Canvas.CopyRect(R, Mask.Canvas, R);

        { Generate the overlay image by combining the mask and the
          original image; this will give (courtesy of the appropriate
          ROP code) the image on a black background
        }
        Overlay.Canvas.CopyMode := $00220326; { DSna }
        Overlay.Canvas.CopyRect(R, Mask.Canvas, R);

        { Now put the overlay image onto the background; this will
          fill in the black ('holes') with the overlay image, leaving
          the rest of the background as is
        }
        Composite.Canvas.CopyMode := cmSrcPaint; { DSo }
        Composite.Canvas.CopyRect(R, Overlay.Canvas, R);

        { now copy the composite image back }
        Canvas.CopyMode := cmSrcCopy;
        Canvas.CopyRect(R, Composite.Canvas, R);

      finally
        Mask.Free;
      end;

    finally
      Composite.Free;
      Overlay.Free;
    end;

  end;

  if Length(Caption) > 0 then
  begin
    { draw the button caption }
    Canvas.Font := Self.Font;
    R := GetTextRect(Canvas, Caption);
    DrawButtonText(Canvas, Caption, R, FState);
  end;

end;

function TcsSculptButton.PtInMask(const X, Y: Integer): Boolean;
begin
  Result := True;
  if FHitTestMask <> nil then
    Result := (FHitTestMask.Canvas.Pixels[X, Y] = clBlack);
end;

procedure TcsSculptButton.MouseDown(Button: TMouseButton; Shift: TShiftState;
  X, Y: Integer);
var Clicked: Boolean;
begin
  inherited MouseDown(Button, Shift, X, Y);
  if (Button = mbLeft) and Enabled then
  begin
    if FPreciseClick then
      Clicked := PtInMask(X, Y)
    else
      Clicked := True;

    if Clicked then
    begin
      FState := bsDown;
      Repaint;
    end;
    FDragging := True;
  end;
end;

procedure TcsSculptButton.MouseMove(Shift: TShiftState; X, Y: Integer);
var NewState: TButtonState;
    InMask: Boolean;
begin
  inherited MouseMove(Shift, X, Y);
  InMask := PtInMask(X, Y);

  if FPreciseShowHint and not InMask then
  begin
    { The outcome of PreciseShowHint being True may not be quite
      what the user/developer expects because the Application
      may still display the hint for the parent (if the parent
      has ShowHint = True).  Consider the situation where the
      button has been placed over a TImage and the button, image and
      form all have ShowHint = True.  In this case PreciseShowHint
      will result in the hint for the form being shown (because it is
      the parent of the button) rather than the hint for the image
      when the cursor is not positioned inside the masked area.
    }
    if not FPrevShowHintSaved then
    begin
      { must save ParentShowHint before changing ShowHint }
      FPrevParentShowHint := ParentShowHint;
      ParentShowHint := False;
      FPrevShowHint := ShowHint;
      ShowHint := False;
      FPrevShowHintSaved := True;
    end;
  end
  else if FPreciseClick and not InMask then
  begin
    if not FPrevCursorSaved then
    begin
      FPrevCursor := Cursor;
      Cursor := crDefault;
      FPrevCursorSaved := True;
    end;
  end
  else
  begin
    if FPrevShowHintSaved then
    begin
      { must set ShowHint before changing ParentShowHint }
      ShowHint := FPrevShowHint;
      ParentShowHint := FPrevParentShowHint;
      FPrevShowHintSaved := False;
    end;
    if FPrevCursorSaved then
    begin
      Cursor := FPrevCursor;
      FPrevCursorSaved := False;
    end;
  end;

  if FDragging then
  begin
    if FPreciseClick then
      if InMask then
        NewState := bsDown
      else
        NewState := bsUp
    else
      if (X >= 0) and (X < ClientWidth) and (Y >= 0) and (Y <= ClientHeight) then
        NewState := bsDown
      else
        NewState := bsUp;

    if (NewState <> FState) then
    begin
      FState := NewState;
      Repaint;
    end;
  end;

end;

procedure TcsSculptButton.MouseUp(Button: TMouseButton; Shift: TShiftState;
  X, Y: Integer);
var
  DoClick: Boolean;
begin
  inherited MouseUp(Button, Shift, X, Y);
  if FDragging then
  begin
    FDragging := False;
    if FPreciseClick then
      { determine if mouse released while on masked area }
      DoClick := PtInMask(X, Y)
    else
      DoClick := (X >= 0) and (X < ClientWidth) and (Y >= 0) and (Y <= ClientHeight);

    if (FState = bsDown) then
    begin
      FState := bsUp;
      Repaint;
    end;
    if DoClick then Click;
  end;

end;

procedure TcsSculptButton.Click;
begin
  inherited Click;
end;

function TcsSculptButton.GetPalette: HPALETTE;
begin
  Result := FBitmap.Palette;
end;

procedure TcsSculptButton.SetBitmap(Value: TBitmap);
begin
  FBitmap.Assign(Value);
end;

procedure TcsSculptButton.SetBitmapUp(Value: TBitmap);
begin
  FBitmapUp.Assign(Value);
end;

procedure TcsSculptButton.SetBitmapDown(Value: TBitmap);
begin
  FBitmapDown.Assign(Value);
end;

procedure TcsSculptButton.BitmapChanged(Sender: TObject);
var OldCursor: TCursor;
    W, H: Integer;
begin
  AdjustBounds;

  if not ((csReading in ComponentState) or (csLoading in ComponentState)) then
  begin
    if FBitmap.Empty then
    begin
      { Bitmap has been cleared, also clear up & down images } 
      SetBitmapUp(nil);
      SetBitmapDown(nil);
    end
    else
    begin
      W := FBitmap.Width;
      H := FBitmap.Height;
      OldCursor := Screen.Cursor;
      Screen.Cursor := crHourGlass;
      try
        if (FBitmapUp.Width <> W) or (FBitmapUp.Height <> H) or
          (FBitmapDown.Width <> W) or (FBitmapDown.Height <> H) then
        begin
          FBitmapUp.Width := W;
          FBitmapUp.Height := H;
          FBitmapDown.Width := W;
          FBitmapDown.Height := H;
        end;
        Create3DBitmap(FBitmap, bsUp, FBitmapUp);
        Create3DBitmap(FBitmap, bsDown, FBitmapDown);
        { Use the non-speckled bitmap to create a mono mask to be used
          to check if mouse clicks are on non-transparent areas;
          the speckled bitmaps have a transparent pixel in every
          2nd pixel so mouse clicks would be (incorrectly) ignored
          if these were used.
          Use TransparentColor of FBitmap rather than that of
          the 3D bitmap because the bevel/outline may have been drawn into
          the pixel previously used to indicate the transparent color.
        }
        FHitTestMask.Free;
        FHitTestMask := CreateMonoMask(FBitmapUp, FBitmap.TransparentColor);
        if FSpeckled then
        begin
          AddSpeckle(FBitmapUp, FBitmap.TransparentColor);
          AddSpeckle(FBitmapDown, FBitmap.TransparentColor);
        end;
      finally
        Screen.Cursor := OldCursor;
      end;
    end;
  end;
  Invalidate;

end;

procedure TcsSculptButton.CMDialogChar(var Message: TCMDialogChar);
begin
  with Message do
    if IsAccel(CharCode, Caption) and Enabled then
    begin
      Click;
      Result := 1;
    end else
      inherited;
end;

procedure TcsSculptButton.CMFontChanged(var Message: TMessage);
begin
  Invalidate;
end;

procedure TcsSculptButton.CMTextChanged(var Message: TMessage);
begin
  Invalidate;
end;

procedure TcsSculptButton.CMSysColorChange(var Message: TMessage);
begin
  BitmapChanged(Self);
end;

function TcsSculptButton.BevelColor(const AState: TButtonState; const TopLeft: Boolean): TColor;
begin
  if (AState = bsUp) then
  begin
    if TopLeft then Result := FBevelHighlightColor
    else Result := FBevelShadowColor;
  end
  else { bsDown }
  begin
    if TopLeft then Result := FBevelShadowColor
    else Result := FBevelHighlightColor;
  end;
end;

{ Create3DBitmap:
  The source bitmap is converted to a 3D bitmap by adding successive
  borders (outlines) around each successive image using the appropriate
  color to get the 3D shading effect.  Masks are used to just add each
  successive outline without affecting the existing image.  The new outline
  for each layer is generated by offsetting the original image to
  successive different positions so as to enlarge its 'footprint'.
  Up to 3 layers of outlines are possible depending on BevelWidth (0..2)
  and BorderStyle (bsNone, bsSingle).  Each bevel outline can consist
  of two parts each of which can be a different color (the border is all
  one color).  When the image is in the 'up' state the first
  part, consisting of the top-right corner to the bottom-right corner
  to the bottom-left corner, will (by default) be dark grey in color
  and the second part, consisting of the bottom-left corner to the
  top-left corner to the top-right corner will be (by default) be white.
  Each outline (getting further away from the origin) requires more
  points to define its path than the previous outline.  The reverse
  colors will apply when the button is in the 'down' state.
  The OutlineOffsetPts type is used to define the points for each
  successive outline.  The first subscript is the outline level (1..3),
  the next subscript is for the part, dark-grey or white (0..1),
  and the final subscript is for each of the points.  Unused points
  are specified as (0,0).  The correct sequence for processing the points
  is necessary to get the correct 3D shading effect; this is why the
  the points don't just start from the top-left corner but always start
  from the top-right and proceed clockwise from there.  The 3D image is
  also built up from the inside out so as to be able to extract each
  succesive outline so it can be combined with the original image. 

  The points are derived from the following grids, where B = Black (border)
  W = White and G = Dk Grey.  X is the origin of the original image.
  Each character represents one pixel.

  Up: BBBBBBB   Down: BBBBBBB
      BWWWWGB         BGGGGWB
      BWWWGGB         BGGGWWB
      BWWXGGB         BGGXWWB
      BWGGGGB         BGWWWWB
      BGGGGGB         BWWWWWB
      BBBBBBB         BBBBBBB

}
procedure TcsSculptButton.Create3DBitmap(Source: TBitmap; const AState: TButtonState; Target: TBitmap);
type OutlineOffsetPts = Array[1..3, 0..1, 0..12] of PointArray;
const
  OutlinePts: OutlineOffsetPts =
    ( (((1,-1),(1,0),(1,1),(0,1),(-1,1),(0,0),(0,0),(0,0),(0,0),(0,0),(0,0),(0,0),(0,0)),
       ((-1,0),(-1,-1),(0,-1),(0,0),(0,0),(0,0),(0,0),(0,0),(0,0),(0,0),(0,0),(0,0),(0,0))),
      (((2,-2),(2,-1),(2, 0),(2, 1),(2, 2),(1, 2),(0, 2),(-1,2),(-2,2),(0,0),(0,0),(0,0),(0,0)),
       ((-2,1),(-2,0),(-2,-1),(-2,-2),(-1,-2),(0,-2),(1,-2),(0,0),(0,0),(0,0),(0,0),(0,0),(0,0))),
      (((3,-3),(3,-2),(3,-1),(3,0),(3,1),(3,2),(3,3),(2,3),(1,3),(0,3),(-1,3),(-2,3),(-3,3)),
       ((-3,2),(-3,1),(-3,0),(-3,-1),(-3,-2),(-3,-3),(-2,-3),(-1,-3),(0,-3),(1,-3),(2,-3),(0,0),(0,0)))
    );
var I, J, W, H, Outlines: Integer;
    R: TRect;
    OutlineMask, Overlay, NewSource: TBitmap;
    BrushHandle: hBrush;
    OldBrushHandle: hBrush;
begin
  if (Source = nil) or (Target = nil) then
    Exit;

  W := Source.Width;
  H := Source.Height;
  R := Rect(0, 0, W, H);

  Overlay := TBitmap.Create;
  NewSource := TBitmap.Create;

  { The following lines may look strange -- they are just used to force the
    bitmap and canvas handles for Source to be created before doing anything
    with Source. (The handles are only assigned back to themselves so it will
    compile under Delphi 1 -- under Delphi 2 we could just reference each
    Handle property without assigning to itself.)
    I don't know why but if the handles aren't initialised like this for
    256 color systems then changing the bitmap at design-time won't always
    'take' and changing the bitmap at design or run-time causes memory losses
    (palettes not freed by Graphics unit).
  }
  Source.Handle := Source.Handle;
  Source.Canvas.Handle := Source.Canvas.Handle;

  Overlay.Palette := CopyPalette(Source.Palette);
  NewSource.Palette := CopyPalette(Source.Palette);
  Target.Palette := CopyPalette(Source.Palette);
  InitPalette(Overlay.Canvas.Handle);
  InitPalette(NewSource.Canvas.Handle);
  InitPalette(Target.Canvas.Handle);

  try

    NewSource.Width := W;
    NewSource.Height := H;

    { Copy source to target }
    Target.Canvas.CopyMode := cmSrcCopy;
    Target.Canvas.CopyRect(R, Source.Canvas, R);

    Overlay.Width := W;
    Overlay.Height := H;

    Outlines := FBevelWidth;
    if (FBorderStyle = bsSingle) then
      Inc(Outlines);

    for I := 1 to Outlines do
    begin
      { use the target bitmap as the basis for the new outline }
      with NewSource.Canvas do
      begin
        CopyMode := cmSrcCopy;
        CopyRect(R, Target.Canvas, R);
      end;

      for J := 0 to 1 do
      begin
        if (AState = bsDown) and (I = Outlines) and (J = 0) then
          Continue; { no shadow outline for final border is used }
        { Use TransparentColor of FBitmap rather than that of
          the 3D bitmap because the bevel/outline may have been drawn into
          the pixel previously used to indicate the transparent color.
        }
        OutlineMask := CreateMonoOutlineMask(Source, NewSource, OutlinePts[I, J],
                        FBitmap.TransparentColor);
        try
          with Overlay.Canvas do
          begin
            { create our own brush rather than using the canvas's -- not sure
              if this is absolutely necessary but you never know when dealing
              with palette colours!
            }
            if (I = Outlines) and (FBorderStyle = bsSingle) then
              BrushHandle := CreateSolidBrush(ColorToRGB(FBorderColor))
            else
              BrushHandle := CreateSolidBrush(ColorToRGB(BevelColor(AState, (J = 1))));
            OldBrushHandle := SelectObject(Handle, BrushHandle);
            try
              CopyMode := $0030032A; { PSna }
              CopyRect(R, OutlineMask.Canvas, R);
            finally
              SelectObject(Handle, OldBrushHandle);
              DeleteObject(BrushHandle);
            end;
          end;

          with Target.Canvas do
          begin
            { Create black outline in target where colored outline is to go }
            CopyMode := cmSrcAnd; { DSa }
            CopyRect(R, OutlineMask.Canvas, R);
            { Copy colored outline into black outline area }
            CopyMode := cmSrcPaint; { DSo }
            CopyRect(R, Overlay.Canvas, R);
            CopyMode := cmSrcCopy;
          end;

        finally
          OutlineMask.Free;
        end;

      end;
    end;

  finally
    Overlay.Free;
    NewSource.Free;

  end;
end;

procedure TcsSculptButton.SetBorderStyle(Value: TBorderStyle);
begin
  if Value <> FBorderStyle then
  begin
    FBorderStyle := Value;
    BitmapChanged(Self);
  end;
end;

procedure TcsSculptButton.SetBorderColor(Value: TColor);
begin
  if Value <> FBorderColor then
  begin
    FBorderColor := Value;
    BitmapChanged(Self);
  end;
end;

procedure TcsSculptButton.SetBevelWidth(Value: TcsBevelWidth);
begin
  if Value > 2 then Value := 2;
  if Value <> FBevelWidth then
  begin
    FBevelWidth := Value;
    BitmapChanged(Self);
  end;
end;

procedure TcsSculptButton.SetBevelHighlightColor(Value: TColor);
begin
  if Value <> FBevelHighlightColor then
  begin
    FBevelHighlightColor := Value;
    BitmapChanged(Self);
  end;
end;

procedure TcsSculptButton.SetBevelShadowColor(Value: TColor);
begin
  if Value <> FBevelShadowColor then
  begin
    FBevelShadowColor := Value;
    BitmapChanged(Self);
  end;
end;

procedure TcsSculptButton.SetSpeckled(Value: Boolean);
begin
  if Value <> FSpeckled then
  begin
    FSpeckled := Value;
    BitmapChanged(Self);
  end;
end;

procedure TcsSculptButton.SetSpeckleTransparentColor(Value: TColor);
begin
  if Value <> FSpeckleTransparentColor then
  begin
    FSpeckleTransparentColor := Value;
    if FSpeckled then
      BitmapChanged(Self);
  end;
end;

procedure TcsSculptButton.SetSpeckleOpaqueColor(Value: TColor);
begin
  if Value <> FSpeckleOpaqueColor then
  begin
    FSpeckleOpaqueColor := Value;
    if FSpeckled then
      BitmapChanged(Self);
  end;
end;

procedure TcsSculptButton.SetTextPosition(Value: TcsTextPosition);
begin
  if Value <> FTextPosition then
  begin
    FTextPosition := Value;
    Invalidate;
  end;
end;

procedure TcsSculptButton.SetTextX(Value: Integer);
begin
  SetTextXY(Value, FTextY);
end;

procedure TcsSculptButton.SetTextY(Value: Integer);
begin
  SetTextXY(FTextX, Value);
end;

procedure TcsSculptButton.SetTextXY(const X, Y: Integer);
var Moved: Boolean;
begin
  Moved := False;
  if X <> FTextX then
  begin
    FTextX := X;
    Moved := True;
  end;
  if Y <> FTextY then
  begin
    FTextY := Y;
    Moved := True;
  end;
  if Moved and (FTextPosition = tpXY) then
    Invalidate;
end;

function TcsSculptButton.GetTextRect(Canvas: TCanvas; const Caption: String): TRect;
var CString: array[0..255] of Char;
begin
  if FTextPosition = tpCentered then
    Result := ClientRect
  else
  begin
    Result := Rect(0, 0, ClientRect.Right - ClientRect.Left, 0);
    DrawText(Canvas.Handle, StrPCopy(CString, Caption), -1, Result,
      DT_CALCRECT);
    OffsetRect(Result, FTextX, FTextY);
  end;
end;

procedure TcsSculptButton.DrawButtonText(Canvas: TCanvas; const Caption: String;
  TextBounds: TRect; State: TButtonState);
var
  CString: array[0..255] of Char;
begin
  StrPCopy(CString, Caption);
  Canvas.Brush.Style := bsClear;
  if State = bsDown then OffsetRect(TextBounds, 1, 1);
  DrawText(Canvas.Handle, CString, -1, TextBounds,
      DT_CENTER or DT_VCENTER or DT_SINGLELINE);
end;

procedure TcsSculptButton.AddSpeckle(Source: TBitmap; TransparentColor: TColor);
var BrushBmp, SpeckledBmp: TBitmap;
    PatternBrush: TBrush;
    PrevColor, PixelColor: TColor;
    X, Y: Integer;
    R: TRect;
    OldBrushHandle: hBrush;
begin
  BrushBmp := TBitmap.Create;
  SpeckledBmp := TBitmap.Create;
  PatternBrush := TBrush.Create;
  BrushBmp.Palette := CopyPalette(Source.Palette);
  SpeckledBmp.Palette := CopyPalette(Source.Palette);
  InitPalette(BrushBmp.Canvas.Handle);
  InitPalette(SpeckledBmp.Canvas.Handle);
  InitPalette(Source.Canvas.Handle);
  try
    BrushBmp.Width := 8;  { brush pattern bitmaps are 8x8 pixels }
    BrushBmp.Height := 8;

    { make a checkerboard brush pattern (alternating colors) }
    PrevColor := FSpeckleOpaqueColor;
    for X := 0 to 7 do
    begin
      PixelColor := PrevColor;
      for Y := 0 to 7 do
      begin
        BrushBmp.Canvas.Pixels[X, Y] := PixelColor;
        PrevColor := PixelColor;
        if PixelColor = FSpeckleOpaqueColor then
          PixelColor := TransparentColor
        else
          PixelColor := FSpeckleOpaqueColor;
      end;
    end;

    PatternBrush.Bitmap := BrushBmp;
    PatternBrush.Bitmap.Palette := CopyPalette(Source.Palette);
    InitPalette(PatternBrush.Bitmap.Canvas.Handle);

    R := Rect(0, 0, Source.Width, Source.Height);

    { Note: Can't do a BrushCopy where the source and destination
            are the same bitmap,
            e.g. Source.Canvas.BrushCopy(R, Source, ...),
            hence use of SpeckledBmp bitmap.
    }
    SpeckledBmp.Width := Source.Width;
    SpeckledBmp.Height := Source.Height;

    with SpeckledBmp.Canvas do
    begin
      OldBrushHandle := SelectObject(Handle, PatternBrush.Handle);
      { Apply pattern to source bitmap; areas of the source bitmap
        which are FSpeckleTransparentColor color will be set to
        Source.TransparentColor color or FSpeckleOpaqueColor color
        in alternate pixels (as defined by the brush pattern).
        When this bitmap is eventually displayed every second pixel
        (in the area which was originally the FSpeckleTransparentColor
        color) will be transparent.
      }
      BrushCopy(R, Source, R, FSpeckleTransparentColor);
      SelectObject(Handle, OldBrushHandle);
    end;

    { Now copy the speckled bitmap back to the source }
    Source.Canvas.CopyMode := cmSrcCopy;
    Source.Canvas.CopyRect(R, SpeckledBmp.Canvas, R);

  finally
    BrushBmp.Free;
    SpeckledBmp.Free;
    PatternBrush.Free;
  end;

end;

procedure TcsSculptButton.Loaded;
var BigMask: TBitmap;
    R: TRect;
begin
  inherited Loaded;
  if (FBitmap <> nil) and (FBitmap.Width > 0) and (FBitmap.Height > 0) then
  begin
    { Combine the mask for the original image with one the mask of one
      of the 'enlarged' images; this will remove any speckling inside
      the image so that hit-testing will work correctly.
      Use TransparentColor of FBitmap rather than that of
      the 3D bitmap because the bevel/outline may have been drawn into
      the pixel previously used to indicate the transparent color.
    }
    FHitTestMask.Free;
    FHitTestMask := CreateMonoMask(FBitmap, FBitmap.TransparentColor);
    BigMask := CreateMonoMask(FBitmapUp, FBitmap.TransparentColor);
    try
      R := Rect(0, 0, FBitmap.Width, FBitmap.Height);
      FHitTestMask.Canvas.CopyMode := cmSrcAnd;
      FHitTestMask.Canvas.CopyRect(R, BigMask.Canvas, R);
    finally
      BigMask.Free;
    end;
  end;
end;

{ Fake BitmapUp and BitmapDown properties are defined so that
  the bitmaps for the button's up and down states are stored.
}
procedure TcsSculptButton.DefineProperties(Filer: TFiler);
begin
  inherited DefineProperties(Filer);
  Filer.DefineBinaryProperty('BitmapUp', ReadBitmapUpData, WriteBitmapUpData, not FBitmapUp.Empty);
  Filer.DefineBinaryProperty('BitmapDown', ReadBitmapDownData, WriteBitmapDownData, not FBitmapDown.Empty)
end;

procedure TcsSculptButton.ReadBitmapUpData(Stream: TStream);
begin
  FBitmapUp.LoadFromStream(Stream);
end;

procedure TcsSculptButton.WriteBitmapUpData(Stream: TStream);
begin
  FBitmapUp.SaveToStream(Stream);
end;

procedure TcsSculptButton.ReadBitmapDownData(Stream: TStream);
begin
  FBitmapDown.LoadFromStream(Stream);
end;

procedure TcsSculptButton.WriteBitmapDownData(Stream: TStream);
begin
  FBitmapDown.SaveToStream(Stream);
end;

procedure TcsSculptButton.AdjustBounds;
begin
  SetBounds(Left, Top, Width, Height);
end;

procedure TcsSculptButton.AdjustButtonSize(var W, H: Integer);
begin
  if not (csReading in ComponentState) and FAutoSize and not FBitmap.Empty then
  begin
    W := FBitmap.Width;
    H := FBitmap.Height;
  end;
end;

procedure TcsSculptButton.SetAutoSize(Value: Boolean);
begin
  if FAutoSize <> Value then
  begin
    FAutoSize := Value;
    AdjustBounds;
  end;
end;

procedure TcsSculptButton.SetBounds(ALeft, ATop, AWidth, AHeight: Integer);
var W, H: Integer;
begin
  W := AWidth;
  H := AHeight;
  AdjustButtonSize(W, H);
  inherited SetBounds(ALeft, ATop, W, H);
end;

procedure TcsSculptButton.Invalidate;
var R: TRect;
begin
  if (Visible or (csDesigning in ComponentState)) and
    (Parent <> nil) and Parent.HandleAllocated then
  begin
    R := BoundsRect;
    InvalidateRect(Parent.Handle, @R, True);
  end;
end;

{ Select and realize the control's palette }
procedure TcsSculptButton.InitPalette(DC: HDC);
var Palette: HPALETTE;
begin
  Palette := GetPalette;
  if (Palette <> 0) then
  begin
    SelectPalette(DC, Palette, False);
    RealizePalette(DC);
  end;
end;

end.
