{*******************************************************}
{                                                       }
{   Copyright (c) 1995, 1999 Classic Software           }
{   All rights reserved                                 }
{                                                       }
{*******************************************************}

unit CSProper;

{$B-,P+,W-,X+}

interface

uses
  SysUtils, WinTypes, WinProcs, Messages, Classes, Graphics, Controls,
  Forms, Dialogs, StdCtrls, DBCtrls;

type

  { Define new method pointer type for Convert event. }
  TcsConvertEvent = procedure(var AString: String; var Handled: Boolean) of object;

  TcsProperEdit = class(TEdit)
  private
    { The FLoadedProperCase flag indicates whether the
      ProperCase property has been loaded yet.
      The Text value is not converted unless the ProperCase
      property has been loaded; this is done to prevent the
      default value for ProperCase being used before the
      actual value has been read.
    }
    FLoadedProperCase: Boolean;
    FProperCase: Boolean;
    FOnConvert: TcsConvertEvent;
{$IFNDEF VER130}
    procedure Change; override;
{$ENDIF}    
    procedure SetProperCase(Value: Boolean);
  protected
{$IFDEF VER130}
    procedure Change; override;
{$ENDIF}
    procedure Convert; virtual;
  public
    constructor Create(AOwner: TComponent); override;
  published
    { ProperCase must always be stored; this allows Text to be
      initialised correctly.
    }
    property ProperCase: Boolean read FProperCase write SetProperCase stored True;
    property OnConvert: TcsConvertEvent read FOnConvert write FOnConvert;
  end;

  { TcsDBProperEdit is nearly the same as TcsProperEdit, just descended
    from a different ancestor class and the Convert method differs.
  }
  TcsDBProperEdit = class(TDBEdit)
  private
    FLoadedProperCase: Boolean;
    FProperCase: Boolean;
    FOnConvert: TcsConvertEvent;
{$IFNDEF VER130}
    procedure Change; override;
{$ENDIF}
    procedure SetProperCase(Value: Boolean);
  protected
{$IFDEF VER130}
    procedure Change; override;
{$ENDIF}    
    procedure Convert; virtual;
  public
    constructor Create(AOwner: TComponent); override;
  published
    { ProperCase must always be stored; this allows Text to be
      initialised correctly.
    }
    property ProperCase: Boolean read FProperCase write SetProperCase stored True;
    property OnConvert: TcsConvertEvent read FOnConvert write FOnConvert;
  end;

implementation

uses CSString, DB, DBTables {$IFDEF EVALUATION}, CSEval {$ENDIF};

{ TcsProperEdit }

constructor TcsProperEdit.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FLoadedProperCase := False;
  FProperCase := True;
end;

procedure TcsProperEdit.Change;
var Pos: Integer;
begin
  if FLoadedProperCase and FProperCase and
    not ((csDesigning in ComponentState) and (Name = Text)) then
  begin
    Pos := SelStart;
    Convert;
    SelStart := Pos;
  end;
  inherited Change;
end;

procedure TcsProperEdit.SetProperCase(Value: Boolean);
begin
  { The ProperCase property is not loaded until after the Text property,
    i.e. when the Text property is loaded the default ProperCase
    value is still current, not the stored ProperCase value.
    Setting FLoadedProperCase to True allows the text to be converted.
  }
  if (Value <> FProperCase) or not FLoadedProperCase then
  begin
    FProperCase := Value;
    FLoadedProperCase := True;  { ProperCase property has now been initialised }
    if FProperCase then
      Change;
  end;
end;

procedure TcsProperEdit.Convert;
var Handled: Boolean;
    NewText: String;
begin
  Handled := False;
  NewText := Text;
  if Assigned(FOnConvert) then FOnConvert(NewText, Handled);
  if not Handled then
    { Prefix ProperCase _function_ with unit name (CSString) to prevent
      conflict with like-named property in this class.
    }
    NewText := CSString.ProperCase(NewText); { default handling }
  Text := NewText;
end;


{ TcsDBProperEdit (code is same as for TcsProperEdit except for Convert) }

constructor TcsDBProperEdit.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FLoadedProperCase := False;
  FProperCase := True;
end;

procedure TcsDBProperEdit.Change;
var Pos: Integer;
begin
  if FLoadedProperCase and FProperCase and
    not ((csDesigning in ComponentState) and (Name = Text)) then
  begin
    Pos := SelStart;
    Convert;
    SelStart := Pos;
  end;
  inherited Change;
end;

procedure TcsDBProperEdit.SetProperCase(Value: Boolean);
begin
  { The ProperCase property is not loaded until after the Text property,
    i.e. when the Text property is loaded the default ProperCase
    value is still current, not the stored ProperCase value.
    Setting FLoadedProperCase to True allows the text to be converted.
  }
  if (Value <> FProperCase) or not FLoadedProperCase then
  begin
    FProperCase := Value;
    FLoadedProperCase := True;  { ProperCase property has now been initialised }
    if FProperCase then
      Change;
  end;
end;

{ Convert text and assign new value to the field. This method differs
  from the one in TcsProperEdit to allow for the data-link.
}
procedure TcsDBProperEdit.Convert;
var Handled: Boolean;
    NewText: String;
begin
  Handled := False;
  NewText := Text;
  if Assigned(FOnConvert) then FOnConvert(NewText, Handled);
  if not Handled then
    { Prefix ProperCase _function_ with unit name (CSString) to prevent
      conflict with like-named property in this class.
    }
    NewText := CSString.ProperCase(NewText); { default handling }

  {  Do not assign the new value if it is the same as the
    current value; doing so causes trailing spaces to be removed
    (which we don't want) by the Field object.
  }
  if (NewText <> Text) and (Field <> nil) and (Field.DataSet <> nil) and
    (Field.DataSet.State in [dsEdit, dsInsert]) then
    { Must actually change the field's data, not the Text or EditText
      properties as these are only a reflection of the field's data.
    }
    (Field as TStringField).Value := NewText;
end;

end.
