{*******************************************************}
{                                                       }
{   Copyright (c) 1997, 1999 Classic Software           }
{   All Rights Reserved                                 }
{                                                       }
{*******************************************************}

unit CSPCTE;

{$B-,P+,W-,X+}

interface

uses
  SysUtils, WinTypes, WinProcs, Messages, Classes, Graphics, Controls,
  Forms, ExtCtrls, CSFrmPnl, StdCtrls, Buttons, DsgnIntf, CSTCBase, CSPC;

type
  TcsPCTabEditorDlg = class(TForm)
    BtnOK: TBitBtn;
    BtnCancel: TBitBtn;
    PanelSubForm: TcsFormPanel;
    procedure FormCreate(Sender: TObject);
    procedure FormShow(Sender: TObject);
    procedure BtnCancelClick(Sender: TObject);
    procedure BtnOKClick(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
  private
    { Private declarations }
{$IFDEF VER130}
    FFormDesigner: IFormDesigner;
{$ELSE}
{$IFDEF VER120}
    FFormDesigner: IFormDesigner;
{$ELSE}
    FFormDesigner: TFormDesigner;
{$ENDIF}
{$ENDIF}
    { FMasterList is used to keep a list of references to the original
      tab data items in FTabDataList prior to editing.  This is necessary
      because we must maintain the contents (child controls) of existing
      pages.
    }
    FMasterList: TList;
    FModified: Boolean;
    FPageControl: TcsPageControl;
    FTabDataList: TcsTabDataList;
    procedure Load;
    procedure Save;
  public
    { Public declarations }
    { Forms already have a Designer property, so call it FormDesigner.
      Used to pass form designer to sub-form.
    }
{$IFDEF VER130}
    property FormDesigner: IFormDesigner read FFormDesigner write FFormDesigner;
{$ELSE}
{$IFDEF VER120}
    property FormDesigner: IFormDesigner read FFormDesigner write FFormDesigner;
{$ELSE}
    property FormDesigner: TFormDesigner read FFormDesigner write FFormDesigner;
{$ENDIF}
{$ENDIF}
    property Modified: Boolean read FModified write FModified;
    property PageControl: TcsPageControl read FPageControl write FPageControl;
  end;

implementation

{$IFDEF WIN32}
uses CSTCSubF, CSPCPr, Dialogs;
{$ELSE}
uses CSTCSubF, CSPCPr, Dialogs, LibMain;
{$ENDIF}

{ The subform variable cannot be a field in the form class because in Delphi 1
  a single instance of the subform needs to be created and reused so as to
  prevent the resource losses associated with dynamically creating the
  subform over and over (due to problems in the VCL Forms and Grids units).
}
var
  SubForm: TcsTabEditorSubForm;

{$R *.DFM}

procedure TcsPCTabEditorDlg.FormCreate(Sender: TObject);
begin
{$IFDEF WIN32}
  SubForm := TcsTabEditorSubForm.Create(Self); { Self will free subform }
{$ELSE}
  if SubForm = nil then { single instance of subform not created yet }
    SubForm := TcsTabEditorSubForm.Create(nil);
{$ENDIF}
  FMasterList := TList.Create;
  FTabDataList := TcsTabDataList.Create;
end;

procedure TcsPCTabEditorDlg.FormDestroy(Sender: TObject);
begin
{$IFNDEF WIN32}
  { For Delphi1, prevent subform from being freed by removing it from its
    parent's Controls list; because the subform's Parent is set to the
    FormPanel, it would normally be destroyed when the FormPanel is destroyed.
  }
  SubForm.Parent := nil;
{$ENDIF}
  FMasterList.Free;
  FTabDataList.Free;
end;

procedure TcsPCTabEditorDlg.FormShow(Sender: TObject);
begin
  SubForm.FormDesigner := FormDesigner;
  Load;
  FModified := False;
  PanelSubForm.Form := SubForm;
end;

procedure TcsPCTabEditorDlg.Load;
var
  I: Integer;
  TabData: TcsTabData;
  Page: TcsTabSheet;
begin
  if (PageControl <> nil) then
  begin
    { copy data from PageControl.Pages into FTabDataList and then assign to
      the sub-form's TabDataList property
    }
    FTabDataList.Clear;
    FMasterList.Clear;
    for I := 0 to PageControl.PageCount - 1 do
    begin
      Page := PageControl.Pages[I];
      TabData := TcsTabData.Create;
      TabData.Bitmap := Page.Bitmap;
      TabData.Caption := Page.Caption;
      TabData.Color := Page.Color;
      TabData.Enabled := Page.TabEnabled;
      TabData.NumGlyphs := Page.NumGlyphs;
      TabData.Visible := Page.TabVisible;
      TabData.Hint := Page.Hint;
      FTabDataList.Add(TabData);
      { keep a copy of what items are in FTabDataList for comparison later }
      FMasterList.Add(TabData);
    end;
    SubForm.TabDataList := FTabDataList;
  end;
end;

{ Save performs a series of actions:
  1. Remove any pages which were deleted in the tab editor dialog.
  2. Append any new pages which were added in the tab editor dialog.
  3. Update and match the position of any pages which are still present.
}
procedure TcsPCTabEditorDlg.Save;

  procedure RemoveDeletedPages;
  var
    I: Integer;
    Page, NextPage: TcsTabSheet;
    Selection: TComponent;
  begin
    { process in reverse order to maintain correct position after deleting }
    I := FMasterList.Count - 1;
    while I >= 0 do
    begin
      { check if master list item is still in tab data list }
      if FTabDataList.IndexOf(FMasterList[I]) = -1 then { item not found }
      begin
        Page := PageControl.Pages[I];
{$IFDEF WIN32}
        { check if the page was defined in an ancestor form }
        if (csAncestor in Page.ComponentState) then
          MessageDlg('Cannot delete ' + Page.Name +
            '; this page was defined in an ancestor form.',
            mtError, [mbOK], 0)
        else
{$ENDIF}
        begin
          if Page = PageControl.ActivePage then { deselect in designer }
          begin
            NextPage := PageControl.FindNextPage(PageControl.ActivePage,
              True, False);
            if (NextPage <> nil) and (NextPage <> PageControl.ActivePage) then
            begin
              PageControl.ActivePage := NextPage;
              Selection := NextPage;
            end
            else
              Selection := PageControl;
{$IFDEF WIN32}
            FormDesigner.SelectComponent(Selection);
{$ELSE}
            CompLib.GetActiveForm.SetSelection(Selection.Name);
{$ENDIF}
          end;
          { remove page from form }
          Page.Free;
          { remove associated item from master list }
          FMasterList.Delete(I);
        end;
      end;
      Dec(I);
    end;
  end;

  procedure AddAndUpdatePages;
  var
    I, Idx: Integer;
    TabData: TcsTabData;
    Page: TcsTabSheet;
  begin
    for I := 0 to FTabDataList.Count - 1 do
    begin
      { check if tab data list item is already in master list }
      Idx := FMasterList.IndexOf(FTabDataList[I]);
      if Idx = -1 then { item not found }
      begin
        { create new page:
          owner of page has to be the form so the new page will be selectable
          (i.e. listed in the form's declaration) at design time
        }
        Page := TcsTabSheet.Create(FormDesigner.Form);
        try
          Page.Name := UniqueName(Page);
          Page.PageControl := PageControl;
        except
          Page.Free;
          raise;
        end;
        PageControl.ActivePage := Page;
{$IFDEF WIN32}
        FormDesigner.SelectComponent(Page);
{$ELSE}
        CompLib.GetActiveForm.SetSelection(Page.Name);
{$ENDIF}
      end
      else
        { update existing page }
        Page := PageControl.Pages[Idx];
      { update the page's properties before adding to page control (when new page) }
      TabData := FTabDataList[I];
      Page.Bitmap := TabData.Bitmap;
      Page.Caption := TabData.Caption;
      Page.Color := TabData.Color;
      Page.NumGlyphs := TabData.NumGlyphs;
      Page.TabEnabled := TabData.Enabled;
      Page.TabVisible := TabData.Visible;
      Page.Hint := TabData.Hint;
      if Idx = -1 then { add new page }
      begin
        { add an item to the master list for the sorting process later }
        FMasterList.Add(TabData);
      end;
    end;
  end;

  { The order of the pages in PageControl.Pages will initially be the same as
    the order of tab references in FMasterList. We need to re-arrange the
    order of pages in PageControl.Pages so that it instead matches the order
    of the tab references in FTabDataList (as ordered in the tab editor).
    Pages for tabs which the user attempted to delete which are inherited from
    an ancestor form will end up at the end of the list.
  }
  procedure MatchPageOrder;
  var
    Continue: Boolean;
    I, Idx: Integer;
  begin
    Continue := True;
    while Continue do
    begin
      Continue := False;
      for I := 0 to FTabDataList.Count - 1 do
      begin
        Idx := FMasterList.IndexOf(FTabDataList[I]);
        if Idx <> I then
        begin
          FMasterList.Move(Idx, I);
          PageControl.Pages[Idx].PageIndex := I;
          { restart }
          Continue := True;
          Break;
        end;
      end;
    end;
  end;

begin
  if (PageControl <> nil) then
  begin
    RemoveDeletedPages;
    AddAndUpdatePages;
    MatchPageOrder;
  end;
end;

procedure TcsPCTabEditorDlg.BtnCancelClick(Sender: TObject);
begin
  Modified := False;
end;

procedure TcsPCTabEditorDlg.BtnOKClick(Sender: TObject);
begin
  PanelSubForm.Form.Close;
  Save;
  Modified := True;
end;

{$IFNDEF WIN32}
procedure DestroySubForm; far;
begin
  if SubForm <> nil then
    SubForm.Free;
end;

initialization
  SubForm := nil;
  AddExitProc(DestroySubForm);
{$ENDIF}

end.
