{**********************************************************}
{                                                          }
{   Copyright (c) 1997, 1999 Classic Software              }
{   All Rights Reserved                                    }
{                                                          }
{**********************************************************}

unit CSPCPr;

{$B-,P+,W-,X+}

interface

uses
  DsgnIntf, Classes, Controls;

type
  TcsActivePageProperty = class(TComponentProperty)
  public
    function GetAttributes: TPropertyAttributes; override;
    procedure GetValues(Proc: TGetStrProc); override;
  end;

  TcsPageControlEditor = class(TDefaultEditor)
  private
    function EditTabs: Boolean;
  public
    procedure ExecuteVerb(Index: Integer); override;
    function GetVerb(Index: Integer): String; override;
    function GetVerbCount: Integer; override;
  end;

function UniqueName(AComponent: TComponent): String;

implementation

uses
  SysUtils, WinTypes, WinProcs, Messages, Forms, CSPC, CSPCTE
  {$IFNDEF WIN32} , LibMain {$ENDIF};

 { There is no UniqueName method for TFormDesigner in Delphi 1, so we need our
   own equivalent.  The local UniqueName function is also used for Delphi 2/3
   because it makes the names nicer by removing the 'cs' prefix normally
   included (by TFormDesigner.UniqueName) for objects of type TcsTabSheet.
 }

{ Test a component name for uniqueness and return True if it is unique or False
  if there is another component with the same name.
}
function TryName(const AName: String; AComponent: TComponent): Boolean;
var
  I: Integer;
begin
  Result := False;
  for I := 0 to AComponent.ComponentCount - 1 do
    if CompareText(AComponent.Components[I].Name, AName) = 0 then
      Exit;
  Result := True;
end;

{ Generate a unique name for a component.  Use the standard Delphi rules,
  e.g., <type><number>, where <type> is the component's class name without
  a leading 'T', and <number> is an integer to make the name unique.
}
function UniqueName(AComponent: TComponent): String;
var
  I: Integer;
  Fmt: String;
begin
  { Create a Format string to use as a name template. }
  if CompareText(Copy(AComponent.ClassName, 1, 3), 'Tcs') = 0 then
    Fmt := Copy(AComponent.ClassName, 4, 255) + '%d'
  else
    Fmt := AComponent.ClassName + '%d';

  if AComponent.Owner = nil then
  begin
    { No owner; any name is unique. Use 1. }
    Result := Format(Fmt, [1]);
    Exit;
  end
  else
  begin
    { Try all possible numbers until we find a unique name. }
    for I := 1 to High(Integer) do
    begin
      Result := Format(Fmt, [I]);
      if TryName(Result, AComponent.Owner) then
        Exit;
    end;
  end;

  { This should never happen, but just in case... }
  raise Exception.CreateFmt('Cannot create unique name for %s.',
    [AComponent.ClassName]);
end;

{ TcsActivePageProperty }

function TcsActivePageProperty.GetAttributes: TPropertyAttributes;
begin
  Result := [paValueList];
end;

procedure TcsActivePageProperty.GetValues(Proc: TGetStrProc);
var
  I: Integer;
  APageControl: TcsPageControl;
begin
  APageControl := TcsPageControl(GetComponent(0));
  for I := 0 to APageControl.PageCount - 1 do
    if APageControl.Pages[I].Name <> '' then
      Proc(APageControl.Pages[I].Name);
end;

{ TcsPageControlEditor }

type
  TVerbList = Array[0..3] of String;
const
  PageControlVerbs: TVerbList = ('New Page', 'Next Page', 'Previous Page',
    'Edit Tabs...');

procedure TcsPageControlEditor.ExecuteVerb(Index: Integer);
var
  APageControl: TcsPageControl;
  Page: TcsTabSheet;
{$IFDEF VER130}
  Designer: IFormDesigner;
{$ELSE}
{$IFDEF VER120}
  Designer: IFormDesigner;
{$ELSE}
  Designer: TFormDesigner;
{$ENDIF}
{$ENDIF}
begin
  if Component is TcsTabSheet then
    APageControl := TcsTabSheet(Component).PageControl
  else
    APageControl := TcsPageControl(Component);
  if APageControl <> nil then
  begin
    Designer := Self.Designer;
    if Index = 0 then { New Page }
    begin
      Page := TcsTabSheet.Create(Designer.Form);
      try
        { Generate unique name; no public UniqueName method in Delphi 1 }
        Page.Name := UniqueName(Page);
        Page.PageControl := APageControl;
      except
        Page.Free;
        raise;
      end;
      APageControl.ActivePage := Page;
{$IFDEF WIN32}
      Designer.SelectComponent(Page);
{$ELSE}
      CompLib.GetActiveForm.SetSelection(Page.Name);
{$ENDIF}
      Designer.Modified;
    end
    else if Index < 3 then { Next/Previous Page }
    begin
      Page := APageControl.FindNextPage(APageControl.ActivePage,
        Index = 1 {Next}, False);
      if (Page <> nil) and (Page <> APageControl.ActivePage) then
      begin
        APageControl.ActivePage := Page;
        if Component is TcsTabSheet then
{$IFDEF WIN32}
          Designer.SelectComponent(Page);
{$ELSE}
          CompLib.GetActiveForm.SetSelection(Page.Name);
{$ENDIF}
        Designer.Modified;
      end;
    end
    else { Edit Pages... }
      if EditTabs then
        Designer.Modified;
  end;
end;

function TcsPageControlEditor.GetVerb(Index: Integer): string;
begin
  Result := PageControlVerbs[Index];
end;

function TcsPageControlEditor.GetVerbCount: Integer;
begin
  Result := High(PageControlVerbs) + 1;
end;

function TcsPageControlEditor.EditTabs: Boolean;
var
  APageControl: TcsPageControl;
begin
  if Component is TcsTabSheet then
    APageControl := TcsTabSheet(Component).PageControl
  else
    APageControl := TcsPageControl(Component);
  with TcsPCTabEditorDlg.Create(Application) do
  begin
    FormDesigner := Self.Designer;
    PageControl := APageControl;
    ShowModal;
    { changing Designer.Modified sometimes crashes when done here so return
      Modified status via Result instead
    }
    Result := Modified;
    Free;
  end;
end;

end.
