{**********************************************************}
{                                                          }
{   Copyright (c) 1997, 1999 Classic Software              }
{   All Rights Reserved                                    }
{                                                          }
{   Portions Copyright (c) 1996,97 Borland International   }
{                                                          }
{**********************************************************}

unit CSPC;

{$B-,P+,W-,X+}

interface

uses
  SysUtils, WinTypes, WinProcs, Messages, Classes, Controls, Graphics, Forms,
  Menus, CSTCBase;

type
  TcsPageControl = class;

  TcsTabSheet = class(TCustomControl)
  private
    FBitmap: TBitmap;
    FNumGlyphs: TcsTabNumGlyphs;
    FPageControl: TcsPageControl;
    { The PageOrder property is needed so we can maintain the correct
      page order when reloading Delphi 1 forms (into any version of Delphi).
    }
    FPageOrder: Integer;
    FTabEnabled: Boolean;
    FTabVisible: Boolean;
    FOnPaintBackground: TcsPaintBackgroundEvent;
    procedure AssignToTabData(Dest: TPersistent);
    procedure BitmapChanged(Sender: TObject);
    procedure Changed;
    function GetColor: TColor;
    function GetPageIndex: Integer;
    function GetPageOrder: Integer; { Delphi 1 forms }
    function GetTabIndex: Integer;
    function GetVisible: Boolean;
    procedure InternalSetVisible(Value: Boolean);
    procedure ReadPageOrder(Reader: TReader); { Delphi 1 forms }
    procedure SetBitmap(Value: TBitmap);
    procedure SetColor(Value: TColor);
    procedure SetNumGlyphs(Value: TcsTabNumGlyphs);
    procedure SetPageControl(APageControl: TcsPageControl);
    procedure SetPageIndex(Value: Integer);
    procedure SetPageOrder(Value: Integer); { Delphi 1 forms }
    procedure SetTabEnabled(Value: Boolean);
    procedure SetTabVisible(Value: Boolean);
    procedure SetVisible(Value: Boolean);
{$IFNDEF WIN32}
    procedure WritePageOrder(Writer: TWriter); { Delphi 1 forms }
{$ENDIF}
    procedure CMTextChanged(var Message: TMessage); message CM_TEXTCHANGED;
    procedure WMEraseBkgnd(var Message: TWmEraseBkgnd); message WM_ERASEBKGND;
    { PageOrder property is only needed by TcsPageControl and is thus private. }
    property PageOrder: Integer read GetPageOrder write SetPageOrder;
  protected
    procedure AssignTo(Dest: TPersistent); override;
    procedure DefineProperties(Filer: TFiler); override; { Delphi 1 forms }
    procedure DestroyHandle;
    procedure Paint; override;
    procedure PaintBackground(ACanvas: TCanvas; const ARect: TRect;
      var Handled: Boolean);
    procedure ReadState(Reader: TReader); override;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    property PageControl: TcsPageControl read FPageControl write SetPageControl;
    property TabIndex: Integer read GetTabIndex;
    property Visible: Boolean read GetVisible write SetVisible;
  published
    property Bitmap: TBitmap read FBitmap write SetBitmap;
    property Caption;
    property Color: TColor read GetColor write SetColor;
{$IFDEF VER130}
    property Constraints;
{$ENDIF}
{$IFDEF VER120}
    property Constraints;
{$ENDIF}    
    property Enabled;
    property Font;
    property Height stored False;
    property Left stored False;
    property NumGlyphs: TcsTabNumGlyphs read FNumGlyphs write SetNumGlyphs
      default 1;
    property PageIndex: Integer read GetPageIndex write SetPageIndex stored False;
    property ParentFont;
    property ParentShowHint;
    property PopupMenu;
    property ShowHint;
    property TabEnabled: Boolean read FTabEnabled write SetTabEnabled default True;
    property TabVisible: Boolean read FTabVisible write SetTabVisible default True;
    property Top stored False;
    property Width stored False;
    property OnClick;
    property OnDblClick;
    property OnDragDrop;
    property OnDragOver;
    property OnEnter;
    property OnExit;
    property OnMouseDown;
    property OnMouseMove;
    property OnMouseUp;
    property OnPaintBackground: TcsPaintBackgroundEvent
      read FOnPaintBackground write FOnPaintBackground;
  end;

  TcsPageControl = class(TcsChameleonTabControl)
  private
    FActivePage: TcsTabSheet;
    FActivePageDefault: TcsTabSheet;
    FPages: TList;
    FSaveResources: Boolean;
    procedure ChangeActivePage(Page: TcsTabSheet);
    function GetColoredTabs: Boolean;
    function GetPage(Index: Integer): TcsTabSheet;
    function GetPageCount: Integer;
    procedure InsertPage(Page: TcsTabSheet);
    procedure RemovePage(Page: TcsTabSheet);
    procedure SetActivePage(Page: TcsTabSheet);
    procedure SetColoredTabs(Value: Boolean);
    procedure SetSaveResources(Value: Boolean);
    procedure CMColorChanged(var Message: TMessage); message CM_COLORCHANGED;
    procedure CMDialogKey(var Message: TCMDialogKey); message CM_DIALOGKEY;
  protected
    procedure Change; override;
    procedure Changing(NewIndex: Integer; var Allowed: Boolean); override;
    procedure CreateParams(var Params: TCreateParams); override;
    procedure Loaded; override;
    procedure Notification(AComponent: TComponent; Operation: TOperation); override;
    procedure PaintCardBackground(ACanvas: TCanvas; ARow: Integer;
      const ARect: TRect; var Handled: Boolean); override;
    procedure Rebuild; override;
{$IFDEF WIN32}
{$IFDEF VER130}
    procedure GetChildren(Proc: TGetChildProc; Root: TComponent); override;
{$ELSE}
{$IFDEF VER120}
    procedure GetChildren(Proc: TGetChildProc; Root: TComponent); override;
{$ELSE}
{$IFDEF VER100}
    procedure GetChildren(Proc: TGetChildProc; Root: TComponent); override;
{$ELSE}
    procedure GetChildren(Proc: TGetChildProc); override;
{$ENDIF}
{$ENDIF}
{$ENDIF}
    procedure SetChildOrder(Child: TComponent; Order: Integer); override;
{$ENDIF}
    procedure ShowControl(AControl: TControl); override;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    function FindNextPage(CurPage: TcsTabSheet;
      GoForward, CheckTabVisible: Boolean): TcsTabSheet;
    function PageForTab(ATabIndex: Integer): TcsTabSheet;
    procedure SelectNextPage(GoForward: Boolean);
    property PageCount: Integer read GetPageCount;
    property Pages[Index: Integer]: TcsTabSheet read GetPage;
  published
    property ActivePage: TcsTabSheet read FActivePage write SetActivePage;
    property ActivePageDefault: TcsTabSheet read FActivePageDefault write
      FActivePageDefault;
    property Align;
    property AlignTabs;
{$IFDEF VER130}
    property Anchors;
{$ENDIF}
{$IFDEF VER120}
    property Anchors;
{$ENDIF}
    property BackgroundColor;
    property BevelWidth;
    property BoldCurrentTab;
    property Color;
    property ColoredTabs: Boolean read GetColoredTabs write SetColoredTabs
      default False;
{$IFDEF VER130}
    property Constraints;
{$ENDIF}
{$IFDEF VER120}
    property Constraints;
{$ENDIF}
    property CornerSize;
    property Ctl3D;
    property DragCursor;
    property DragMode;
    property Enabled;
    property Font;
    property FrameBorderColor;
    property FrameHighlightColor;
    property FrameShadowColor;
    property GlyphHAlignment;
    property GlyphMargin;
    property GlyphPosition;
    property GlyphVAlignment;
    property HelpContext;
    property Hint;
    property HotTrack;
    property Margin;
    property MultiLine;
    property ParentBackgroundColor;
    property ParentColor;
    property ParentCtl3D;
    property ParentFont;
    property ParentShowHint;
    property PopupMenu;
    property RowIndent;
    property RowOverlap;
    property SaveResources: Boolean read FSaveResources write SetSaveResources
      default False;
    property ScrollBtnArrowColor;
    property ScrollBtnFaceColor;
    property ShowHint;
    property SidewaysText;
    property TabHeight;
    property TabHints;    
    property TabOrder;
    property TabOrientation;
    property TabSequence;
    property TabStop;
    property TabStyle;
    property TabWidth;
    property TextColorDisabledHighlight;
    property TextColorDisabledShadow;
    property TextColorHotTrack;
    property TextColorSelected;
    property TextColorUnselected;
    property TextHAlignment;
    property TextVAlignment;
    property Visible;
    property OnChange;
    property OnChanging;
    property OnClick;
    property OnDblClick;
    property OnDragDrop;
    property OnDragOver;
    property OnEndDrag;
    property OnEnter;
    property OnExit;
    property OnGetFrameColor;
    property OnGetTextColor;
    property OnMouseDown;
    property OnMouseMove;
    property OnMouseUp;
    property OnPaintBackground; { background behind tabs }
    { Even though each page of a page control has an OnPaintBackground event,
      the OnPaintCardBackground event is still needed to paint the 'gap' where
      the card shows through between the current tab and its page.
    }
    property OnPaintCardBackground;
    property OnPaintTabBackground;
    property OnTabClick;
  end;

implementation

{$IFDEF EVALUATION}
uses CSEval;
{$ENDIF}

var
  Registered: Boolean;

{ TcsTabSheet }

constructor TcsTabSheet.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  InternalSetVisible(False);
{$IFDEF WIN32}
  ControlStyle := ControlStyle + [csAcceptsControls, csOpaque, csNoDesignVisible];
{$ELSE}
  ControlStyle := ControlStyle + [csAcceptsControls, csOpaque];
{$ENDIF}
  Align := alClient;
  FBitmap := TBitmap.Create;
  FBitmap.OnChange := BitmapChanged;
  FNumGlyphs := 1;
  FTabEnabled := True;
  FTabVisible := True;
end;

destructor TcsTabSheet.Destroy;
begin
  if FPageControl <> nil then
    FPageControl.RemovePage(Self);
  FBitmap.Free;
  inherited Destroy;
end;

procedure TcsTabSheet.AssignTo(Dest: TPersistent);
begin
  if Dest is TcsTabData then
    AssignToTabData(TcsTabData(Dest))
  else
    inherited AssignTo(Dest);
end;

procedure TcsTabSheet.AssignToTabData(Dest: TPersistent);
begin
  with TcsTabData(Dest) do
  begin
    { copy relevant properties of self into TcsTabData destination }
    Caption := Self.Caption;
    Bitmap := Self.Bitmap;
    NumGlyphs := Self.NumGlyphs;
    Color := Self.Color;
    Enabled := Self.TabEnabled; { Note use of TabEnabled, instead of Enabled }
    Visible := Self.TabVisible; { Note use of TabVisible, instead of Visible }
    Hint := Self.Hint;    
  end;
end;

procedure TcsTabSheet.DefineProperties(Filer: TFiler);
begin
  inherited DefineProperties(Filer);
{$IFDEF WIN32}
  { Only need capability to read PageOrder property which has been written
    in a form saved in Delphi 1; any form saved by a 32-bit version of
    Delphi will not be readable in Delphi 1 (due to inclusion of extra form
    properties which are not recognised by Delphi 1) so we never need to write
    the PageOrder property in a 32-bit version.
  }
  Filer.DefineProperty('PageOrder', ReadPageOrder, nil, False);
{$ELSE}
  Filer.DefineProperty('PageOrder', ReadPageOrder, WritePageOrder, True);
{$ENDIF}
end;

procedure TcsTabSheet.BitmapChanged(Sender: TObject);
begin
  Changed;
end;

procedure TcsTabSheet.SetBitmap(Value: TBitmap);
begin
  FBitmap.Assign(Value);
  Changed;
end;

function TcsTabSheet.GetPageIndex: Integer;
begin
  if FPageControl <> nil then
    Result := FPageControl.FPages.IndexOf(Self)
  else
    Result := -1;
end;

function TcsTabSheet.GetPageOrder: Integer;
begin
  if (csWriting in ComponentState) then
    Result := GetPageIndex
  else
    Result := FPageOrder;
end;

function TcsTabSheet.GetTabIndex: Integer;
var
  I: Integer;
begin
  Result := 0;
  if not FTabVisible then Dec(Result) else
    for I := 0 to PageIndex - 1 do
      if FPageControl.Pages[I].TabVisible then
        Inc(Result);
end;

procedure TcsTabSheet.ReadPageOrder(Reader: TReader);
begin
  PageOrder := Reader.ReadInteger;
end;

procedure TcsTabSheet.ReadState(Reader: TReader);
begin
  inherited ReadState(Reader);
  if Reader.Parent is TcsPageControl then
    PageControl := TcsPageControl(Reader.Parent);
end;

procedure TcsTabSheet.SetNumGlyphs(Value: TcsTabNumGlyphs);
begin
  if FNumGlyphs <> Value then
  begin
    FNumGlyphs := Value;
    Changed;
  end;
end;

procedure TcsTabSheet.SetPageControl(APageControl: TcsPageControl);
begin
  if FPageControl <> APageControl then
  begin
    if FPageControl <> nil then FPageControl.RemovePage(Self);
    Parent := APageControl;
    if APageControl <> nil then APageControl.InsertPage(Self);
  end;
end;

procedure TcsTabSheet.SetPageIndex(Value: Integer);
var
  MaxPageIndex: Integer;
begin
  if FPageControl <> nil then
  begin
    MaxPageIndex := FPageControl.FPages.Count - 1;
    if Value > MaxPageIndex then
      raise Exception.CreateFmt('%d is an invalid PageIndexValue. ' +
        'PageIndex must be between 0 and %d', [Value, MaxPageIndex]);
    FPageControl.FPages.Move(PageIndex, Value);
    Changed;
  end;
end;

{ PageOrder is only set prior to loading (when page is read); thus only need to
  change FPageOrder field.
}
procedure TcsTabSheet.SetPageOrder(Value: Integer);
begin
  FPageOrder := Value;
end;

procedure TcsTabSheet.SetTabEnabled(Value: Boolean);
begin
  if FTabEnabled <> Value then
  begin
    FTabEnabled := Value;
    Changed;
  end;
end;

procedure TcsTabSheet.SetTabVisible(Value: Boolean);
begin
  if FTabVisible <> Value then
  begin
    FTabVisible := Value;
    Changed;
  end;
end;

{ Note: Caption is updated as each character is typed/changed in the O.I. }
procedure TcsTabSheet.CMTextChanged(var Message: TMessage);
begin
  Changed;
end;

procedure TcsTabSheet.Changed;
begin
  if FPageControl <> nil then
    FPageControl.Rebuild;
end;

{$IFNDEF WIN32}
procedure TcsTabSheet.WritePageOrder(Writer: TWriter);
begin
  Writer.WriteInteger(PageOrder);
end;
{$ENDIF}

procedure TcsTabSheet.DestroyHandle;
begin
  inherited DestroyHandle;
end;

procedure TcsTabSheet.Paint;
var
  ARect: TRect;
  Handled: Boolean;
begin
  ARect := Rect(0, 0, Width, Height);
  Handled := False;
  PaintBackground(Canvas, ARect, Handled);
  if not Handled then
  begin
    with Canvas do
    begin
      Brush.Color := Color;
      FillRect(ARect);
    end;
    Handled := True;
  end;  
end;

procedure TcsTabSheet.PaintBackground(ACanvas: TCanvas;
  const ARect: TRect; var Handled: Boolean);
begin
  if Assigned(FOnPaintBackground) then
    FOnPaintBackground(Self, ACanvas, ARect, Handled);
end;

procedure TcsTabSheet.WMEraseBkgnd(var Message: TWmEraseBkgnd);
begin
  { Paint takes care of everything -- ignore erase to prevent flicker }
  Message.Result := 1; { done }
end;

function TcsTabSheet.GetVisible: Boolean;
begin
  Result := inherited Visible;
end;

procedure TcsTabSheet.SetVisible(Value: Boolean);
begin
  { Visibility of pages can only be changed via the InternalSetVisible method;
    this prevents attempts to change the Visible property programmatically.
  }
  raise Exception.Create('You cannot change the Visible property of a page');
end;

procedure TcsTabSheet.InternalSetVisible(Value: Boolean);
begin
  inherited Visible := Value;
end;

function TcsTabSheet.GetColor: TColor;
begin
  Result := inherited Color;
end;

{ Prevent change to color of page unless using ColoredTabs or the page's color
  is being changed (automatically) to match that of the page control.
}
procedure TcsTabSheet.SetColor(Value: TColor);
begin
  if (GetColor <> Value) then
    if ((FPageControl <> nil) and (FPageControl.ColoredTabs or
      ((GetColor <> FPageControl.Color) and (Value = FPageControl.Color)))) or
      (FPageControl = nil) then
    begin
      inherited Color := Value;
      Changed;
    end;
end;

{ TcsPageControl }

constructor TcsPageControl.Create(AOwner: TComponent);
begin
  if not Registered then
  begin
    Classes.RegisterClasses([TcsTabSheet]);
    Registered := True;
  end;
  inherited Create(AOwner);
  ControlStyle := ControlStyle - [csAcceptsControls] + [csOpaque];
  Height := 150; { make page controls larger than tab controls }
  ColoredTabs := False;
  FActivePage := nil;
  FActivePageDefault := nil;
  FPages := TList.Create;
end;

destructor TcsPageControl.Destroy;
var
  I: Integer;
begin
  for I := 0 to FPages.Count - 1 do
    TcsTabSheet(FPages[I]).FPageControl := nil;
  FPages.Free;
  inherited Destroy;
end;

procedure TcsPageControl.Change;
begin
  inherited Change;
end;

procedure TcsPageControl.ChangeActivePage(Page: TcsTabSheet);
var
{$IFDEF VER130}
  Form: TCustomForm;
{$ELSE}
{$IFDEF VER120}
  Form: TCustomForm;
{$ELSE}
{$IFDEF VER100}
  Form: TCustomForm;
{$ELSE}
  Form: TForm;
{$ENDIF}
{$ENDIF}
{$ENDIF}
begin
  if FActivePage <> Page then
  begin
    Form := GetParentForm(Self);
    if (Form <> nil) and (FActivePage <> nil) and
      FActivePage.ContainsControl(Form.ActiveControl) then
      Form.ActiveControl := FActivePage;
    if Page <> nil then
    begin
      Page.BringToFront;
      Page.InternalSetVisible(True);
      if (Form <> nil) and (FActivePage <> nil) and
        (Form.ActiveControl = FActivePage) then
        if Page.CanFocus then
          Form.ActiveControl := Page else
          Form.ActiveControl := Self;
    end;
    if FActivePage <> nil then
    begin
      FActivePage.InternalSetVisible(False);
      if FSaveResources and not (csDesigning in ComponentState) then
        FActivePage.DestroyHandle;
    end;
    FActivePage := Page;
    if (Form <> nil) and (FActivePage <> nil) and
      (Form.ActiveControl = FActivePage) then
      FActivePage.SelectFirst;
  end;
end;

function TcsPageControl.FindNextPage(CurPage: TcsTabSheet;
  GoForward, CheckTabVisible: Boolean): TcsTabSheet;
var
  I, StartIndex: Integer;
begin
  if FPages.Count <> 0 then
  begin
    StartIndex := FPages.IndexOf(CurPage);
    if StartIndex = -1 then
      if GoForward then StartIndex := FPages.Count - 1 else StartIndex := 0;
    I := StartIndex;
    repeat
      if GoForward then
      begin
        Inc(I);
        if I = FPages.Count then I := 0;
      end else
      begin
        if I = 0 then I := FPages.Count;
        Dec(I);
      end;
      Result := FPages[I];
      if not CheckTabVisible or Result.TabVisible then Exit;
    until I = StartIndex;
  end;
  Result := nil;
end;

{$IFDEF WIN32}
{$IFDEF VER130}
procedure TcsPageControl.GetChildren(Proc: TGetChildProc; Root: TComponent);
{$ELSE}
{$IFDEF VER120}
procedure TcsPageControl.GetChildren(Proc: TGetChildProc; Root: TComponent);
{$ELSE}
{$IFDEF VER100}
procedure TcsPageControl.GetChildren(Proc: TGetChildProc; Root: TComponent);
{$ELSE}
procedure TcsPageControl.GetChildren(Proc: TGetChildProc);
{$ENDIF}
{$ENDIF}
{$ENDIF}
var
  I: Integer;
begin
  for I := 0 to FPages.Count - 1 do Proc(TComponent(FPages[I]));
end;
{$ENDIF}

function TcsPageControl.GetPage(Index: Integer): TcsTabSheet;
begin
  Result := FPages[Index];
end;

function TcsPageControl.GetPageCount: Integer;
begin
  Result := FPages.Count;
end;

procedure TcsPageControl.InsertPage(Page: TcsTabSheet);
begin
  FPages.Add(Page);
  Page.FPageControl := Self;
  Rebuild;
end;

procedure TcsPageControl.RemovePage(Page: TcsTabSheet);
begin
  if FActivePage = Page then SetActivePage(nil);
  { remove page before doing Rebuild to ensure correct state used }
  FPages.Remove(Page);
  Rebuild;
  Page.FPageControl := nil; { disconnect page from control }
end;

procedure TcsPageControl.SelectNextPage(GoForward: Boolean);
var
  Page: TcsTabSheet;
begin
  Page := FindNextPage(ActivePage, GoForward, True);
  SetActivePage(Page);
end;

procedure TcsPageControl.SetActivePage(Page: TcsTabSheet);
var
  NewTabIndex: Integer;
begin
  if (Page <> nil) and (Page.PageControl <> Self) then Exit;
  if (csLoading in ComponentState) or (csReading in ComponentState) then
  begin
    { pages haven't been re-ordered by Loaded yet }
    FActivePage := Page;
    Exit;
  end;
  { When the destination page has a TabIndex of -1 (no tab visible)
    we cannot rely on the setting of (self's) TabIndex to cause the
    required page change (via Changing call) and must instead explicitly
    change pages.
  }
  if Page <> nil then
    NewTabIndex := Page.TabIndex { could be -1 if page has no tab }
  else
    NewTabIndex := -1;
  if NewTabIndex < 0 then { explicitly change pages }
    ChangeActivePage(Page);
  { If NewTabIndex is -1, setting TabIndex will always succeed.
    If NewTabIndex is >= 0, setting TabIndex will generate Changing call to
    check if OK to change to a different tab.  If the change is allowed,
    Changing will perform the necessary page change too.
  }
  TabIndex := NewTabIndex;
end;

{$IFDEF WIN32}
procedure TcsPageControl.SetChildOrder(Child: TComponent; Order: Integer);
begin
  TcsTabSheet(Child).PageIndex := Order;
end;
{$ENDIF}

procedure TcsPageControl.ShowControl(AControl: TControl);
begin
  if (AControl is TcsTabSheet) and (TcsTabSheet(AControl).PageControl = Self) then
    SetActivePage(TcsTabSheet(AControl));
  inherited ShowControl(AControl);
end;

procedure TcsPageControl.CMDialogKey(var Message: TCMDialogKey);
var
{$IFDEF VER130}
  Form: TCustomForm;
{$ELSE}
{$IFDEF VER120}
  Form: TCustomForm;
{$ELSE}
{$IFDEF VER100}
  Form: TCustomForm;
{$ELSE}
  Form: TForm;
{$ENDIF}
{$ENDIF}
{$ENDIF}
begin
  { Broadcast the message to any child controls (e.g. a nested page control)
    before attempting to handle in self.
  }
  inherited;
  if Message.Result <> 0 then Exit; { handled by a child control }
  { Now see if self is in the parent hierarchy for the active control --
    this ensures that the page control which is the most immediate parent of
    the active control handles the message.
  }
  Form := GetParentForm(Self);
  if (Form <> nil) and ContainsControl(Form.ActiveControl) and
    (Message.CharCode = VK_TAB) and (GetKeyState(VK_CONTROL) < 0) then
  begin
    SelectNextPage(GetKeyState(VK_SHIFT) >= 0);
    Message.Result := 1;
  end;
end;

procedure TcsPageControl.CreateParams(var Params: TCreateParams);
begin
  inherited CreateParams(Params);
  with Params do Style := Style or WS_CLIPCHILDREN;
end;

procedure TcsPageControl.Loaded;
var
  I: Integer;
  PageList: TList;
  Page: TcsTabSheet;
  OrderPages: Boolean;
begin
  inherited Loaded;
  { Check if the pages need to be re-ordered.  Since the PageOrder property of
    each page is only present in forms saved by Delphi 1, we need to re-order
    if any pages have a PageOrder <> 0.
  }
  OrderPages := False;
  for I := 0 to FPages.Count - 1 do
    if TcsTabSheet(FPages[I]).PageOrder <> 0 then
    begin
      OrderPages := True;
      Break;
    end;
  if OrderPages then
  begin
    { fix up order of pages so they are in FPages in the order of
      their PageOrder values (SetChildOrder handles this in Delphi 2/3)
    }
    PageList := TList.Create;
    try
      PageList.Capacity := FPages.Count;
      for I := 0 to FPages.Count - 1 do
        PageList.Add(nil);
      { add each page into PageList in PageOrder position }
      for I := 0 to FPages.Count - 1 do
      begin
        Page := TcsTabSheet(FPages[I]);
        PageList[Page.PageOrder] := Page;
      end;
      { copy back to FPages in PageOrder order }
      for I := 0 to FPages.Count - 1 do
        FPages[I] := PageList[I];
    finally
      PageList.Free;
    end;
  end;
  TabIndex := -1; { to ensure a Changing event occurs (if the page has a tab) }
  if (FActivePageDefault <> nil) and (FActivePageDefault <> FActivePage) then
    Page := FActivePageDefault
  else
    Page := FActivePage;
  FActivePage := nil;
  Rebuild;
  { Call SetActivePage again (first call occurs when ActivePage property is
    streamed in, i.e. before pages have been reordered).
  }
  SetActivePage(Page);
end;

procedure TcsPageControl.Rebuild;
var
  I: Integer;
  Tab: TcsTabData;
  Page: TcsTabSheet;
begin
  { mirror the persistent page definitions in the internal tab data structures }
  GetTabDataList.Clear;
  for I := 0 to FPages.Count - 1 do
  begin
    Page := TcsTabSheet(FPages[I]);
    { don't add a tab for pages with TabVisible = False; this also implies
      that all internal TcsTabData objects for a page control will have
      TcsTabData.Visible = True
    }
    if Page.TabVisible then
    begin
      Tab := TcsTabData.Create;
      Tab.Assign(Page); { gets passed to AssignTo }
      GetTabDataList.Add(Tab);
    end;
  end;
  { handle special cases after rebuild of tabs/pages }
  if FActivePage <> nil then
    if not FActivePage.TabVisible then
      TabIndex := -1
    else if TabIndex <> FActivePage.TabIndex then
      TabIndex := FActivePage.TabIndex;
  inherited Rebuild;
end;

procedure TcsPageControl.Changing(NewIndex: Integer; var Allowed: Boolean);
begin
  inherited Changing(NewIndex, Allowed);
  if Allowed then
  begin
    { NewIndex is the TabIndex of the destination page.  It will always be >= 0
      because changing to a page (via ActivePage property) which has no tab
      (TabIndex = -1) will not generate a Changing (tab) call.
    }
    ChangeActivePage(PageForTab(NewIndex));
  end;
end;

function TcsPageControl.PageForTab(ATabIndex: Integer): TcsTabSheet;
var
  I: Integer;
  Page: TcsTabSheet;
begin
  Result := nil;
  if ATabIndex < 0 then Exit;
  for I := 0 to FPages.Count - 1 do
  begin
    Page := TcsTabSheet(FPages[I]);
    if Page.TabIndex = ATabIndex then
    begin
      Result := Page;
      Exit;
    end;
  end;
end;

procedure TcsPageControl.SetSaveResources(Value: Boolean);
var
  I: Integer;
begin
  if (Value <> FSaveResources) then
  begin
    FSaveResources := Value;
    if FSaveResources and not (csDesigning in ComponentState) then
      { release existing resources for all pages which are not visible }
      for I := 0 to FPages.Count - 1 do
        if FPages[I] <> FActivePage then
          TcsTabSheet(FPages[I]).DestroyHandle;
  end;
end;

{ The 'card' still has to be painted to fill in the gap between the current tab
  and the associated page; the page is actually smaller than the entire
  'card' size.
}
procedure TcsPageControl.PaintCardBackground(ACanvas: TCanvas;
  ARow: Integer; const ARect: TRect; var Handled: Boolean);
begin
  { call inherited method to allow user's OnPaintCardBackground to fill gap }
  inherited PaintCardBackground(ACanvas, ARow, ARect, Handled);
  if (not Handled) and (ARow = 0) and (TabIndex >= 0) and (ActivePage <> nil) then
  begin
    with ACanvas do
    begin
      Brush.Color := ActivePage.Color;
      FillRect(ARect);
    end;
    Handled := True;
  end;
end;

{ When not using ColoredTabs we have to manually change the color of each page
  to match the color of the page control.  This is necessary because the
  ParentColor property of each page will be False due to changes to each
  page's Color property and thus each page's Color won't be updated via the
  normal process (CM_PARENTCOLORCHANGED message).
}
procedure TcsPageControl.CMColorChanged(var Message: TMessage);
var
  I: Integer;
begin
  if not ColoredTabs then
    for I := 0 to FPages.Count - 1 do
      TcsTabSheet(FPages[I]).Color := Color;
  inherited;
end;

function TcsPageControl.GetColoredTabs: Boolean;
begin
  Result := inherited ColoredTabs;
end;

{ Update the ColoredTabs property.  If necessary, a CM_COLORCHANGED message is
  generated to ensure the page's colors are changed to match.
}
procedure TcsPageControl.SetColoredTabs(Value: Boolean);
begin
  inherited ColoredTabs := Value;
  if (Parent <> nil) and not GetColoredTabs then { change color of pages to match }
    Perform(CM_COLORCHANGED, 0, 0);
end;

procedure TcsPageControl.Notification(AComponent: TComponent; Operation: TOperation);
begin
  inherited Notification(AComponent, Operation);
  if (Operation = opRemove) and (AComponent = FActivePageDefault) then
    FActivePageDefault := nil;
end;

initialization
  Registered := False;

end.
