{*******************************************************}
{                                                       }
{   Copyright (c) 1995, 1999 Classic Software           }
{   All Rights Reserved                                 }
{                                                       }
{*******************************************************}

unit CSNoteBk;

{$IFDEF VER80}
{$B-,P+,W-,X+}
{$ELSE}
{$B-,P+,W-,X+,J+}
{$ENDIF}
interface

uses
  SysUtils, WinTypes, WinProcs, Messages, Classes, Graphics, Controls,
  Forms, StdCtrls, ExtCtrls, Menus;

const
  CM_TABFONTCHANGED = CM_BASE + 100;
  CM_PAGEENABLEDCHANGED = CM_BASE + 101;
  CM_PAGEVISIBLECHANGED = CM_BASE + 102;

type

  TTabNumGlyphs = 1..3; { Selected, Disabled, Unselected }

  TcsPage = class(TPage)
  private
    FBitmap: TBitmap;
    FPageEnabled: Boolean;
    FPageVisible: Boolean;
    FNumGlyphs: TTabNumGlyphs;
    procedure BitmapChanged(Sender: TObject);
    procedure InitPalette(DC: HDC);
    procedure SetBitmap(Value: TBitmap);
    procedure SetNumGlyphs(Value: TTabNumGlyphs);
    procedure SetPageEnabled(Value: Boolean);
    procedure SetPageVisible(Value: Boolean);
    procedure CMTextChanged(var Message: TMessage); message CM_TEXTCHANGED;
    procedure CMPageEnabledChanged(var Message: TMessage); message CM_PAGEENABLEDCHANGED;
    procedure CMPageVisibleChanged(var Message: TMessage); message CM_PAGEVISIBLECHANGED;
    procedure WMEraseBkgnd(var Message: TWmEraseBkgnd); message WM_ERASEBKGND;
  protected
    function GetPalette: HPALETTE; override;
    procedure ReadState(Reader: TReader); override;
    procedure Paint; override;
    function PaletteChanged(Foreground: Boolean): Boolean; override;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    procedure DestroyHandle;
  published
    property Bitmap: TBitmap read FBitmap write SetBitmap;
    property Enabled stored False;
    property NumGlyphs: TTabNumGlyphs read FNumGlyphs write SetNumGlyphs default 1;
    property PageEnabled: Boolean read FPageEnabled write SetPageEnabled default True;
    property PageVisible: Boolean read FPageVisible write SetPageVisible default True;
  end;

  TTabOrientation = (toTop, toLeft, toBottom, toRight);
  TTextAlignment = (taLeftTop, taCentreTop, taCenterTop, taRightTop,
                    taLeftMiddle, taCentreMiddle, taCenterMiddle,
                    taRightMiddle, taLeftBottom, taCentreBottom,
                    taCenterBottom, taRightBottom, taInvisible);
  TBitmapAlignment = (baLeftTop, baCentreTop, baCenterTop, baRightTop,
                      baLeftMiddle, baCentreMiddle, baCenterMiddle,
                      baRightMiddle, baLeftBottom, baCentreBottom,
                      baCenterBottom, baRightBottom, baFit, baInvisible);
  TTabRects = (trAll, trSelected, trUnselected);

  TcsTabPoints = Array[0..5] of TPoint;

  { A PageChangingEvent event will occur _before_ changing to a new
    page; this allows, for example, validation of the current page
    before switching to another page.
  }
  TPageChangingEvent = procedure(Sender: TObject; NewIndex: Integer;
                                  var AllowChange: Boolean) of object;

  { A PaintBackgroundEvent occurs when the background is being
    erased and can be used to copy an image which is 'behind' the
    notebook to give the appearance that the area outside the tabs
    is transparent. (Only works when ParentColor is False or
    UseUnselectedTabColor is True).
  }
  TPaintBackgroundEvent =
    procedure(Sender: TObject; ACanvas: TCanvas; ARect: TRect;
      var Handled: Boolean) of object;

  { A TGetPalette event occurs when the GetPalette method needs the
    handle of the palette to be used for the notebook and its pages.
  }
  TGetPaletteEvent =
    procedure(Sender: TObject; var Handle: HPALETTE) of object;

  TcsNotebook = class(TCustomControl)
  private
    FPageList: TList;
    FAccess: TStrings;
    FPageIndex: Integer;
    FOnPageChanging: TPageChangingEvent;
    FOnPageChanged: TNotifyEvent;
    FAnchoredTabs: Boolean;
    FRowExtent: Integer;  { internal use only }
    FColExtent: Integer;
    FRowIndent: Integer;
    FCornerSize: Integer;
    FTabOrientation: TTabOrientation;
    FSidewaysText: Boolean;
    FMaxTabWidth: Integer;
    FTabHeight: Integer;
    FTextAlignment: TTextAlignment;
    FBitmapAlignment: TBitmapAlignment;
    FRowIdentities: TList;
    FTabIndex: TStringList;
    FHasLoaded: Boolean;
    FHitTest: TPoint;
    FTabFont: TFont;
    FHasFocusRect: Boolean; { does notebook tab already have focus rect? }
    FOnTabClick: TNotifyEvent;
    FSaveResources: Boolean;
    FSelectedColour: TColor;
    FUnselectedColour: TColor;
    FParentTabFont: Boolean;
    FUnselectedTabColour: TColor;
    FUseUnselectedTabColour: Boolean;
    FPageIndexDefault: Integer;
    FOnPaintBackground: TPaintBackgroundEvent;
    FOnGetPalette: TGetPaletteEvent;
    procedure AddBackCardRHPoints(var Pts: TList; CardRect: TRect; Offset: Integer);
    procedure AlignChildRect(var childRect: TRect; Row, Col: Integer);
    function AnyPagesEnabled: Boolean;
    function AnyPagesVisible: Boolean;
    function BackCardPenColour(Segment: Integer): TColor;
    procedure BuildBackCardPoints(var Pts: TList; BaseRect: TRect; Offset, Row: Integer);
    procedure BuildFrontCardPoints(var Pts: Array of TPoint;
                                   BaseRect: TRect; Offset: Integer);
    procedure BuildRowIdentities;
    procedure BuildTabPoints(var Pts: TcsTabPoints; TargetRect: TRect; Offset, Row, Col: Integer);
    procedure CalcAndBuild;
    function CalcBitmapRect(TabIdentity: String): TRect;
    function CalcBitmapSize(TabIdentity: String): TPoint;
    function CalcCardRect(Row: Integer): TRect;
    function CalcFrameRect: TRect;
    function CalcIndexHeight: Integer;
    function CalcIndexRect: TRect;
    function CalcPageRect: TRect;
    function CalcRowExtent: Integer;
    function CalcTabDescent: Integer;
    function CalcTabTextRect(TabIdentity: String): TRect;
    function CalcTextExtent(Str: String; DC: HDC): Integer;
    function CalcTextOrigin(Str: String; BaseRect: TRect; DC: HDC): TPoint;
    function CalcWholeRect: TRect;
    procedure CheckSidewaysText;
    procedure ClearTabIndex;
    procedure DoTextOut(x, y: Integer; Str: String; DC: HDC; PlainFont, UnderlinedFont: HFont);
    procedure DrawBackCard(Rect: TRect; Row: Integer);
    procedure DrawBevelBottom(Pts: TcsTabPoints; Offset: Integer);
    procedure DrawBevelLeft(Pts: TcsTabPoints; Offset: Integer);
    procedure DrawBevelRight(Pts: TcsTabPoints; Offset: Integer);
    procedure DrawBevels;
    procedure DrawBevelTop(Pts: TcsTabPoints; Offset: Integer);
    procedure DrawCards;
    procedure DrawFrontCard(Rect: TRect);
    procedure DrawTabFaces;
    procedure DrawTabFocusRect;
    procedure DrawTabFrames;
    function FindNextPageIdx(StartIdx: Integer): Integer;
    function FindTabCol(const TabIdentity: String): Integer;
    function FindTabRow(const TabIdentity: String): Integer;
    function FrontCardPenColour(Segment: Integer): TColor;
    function GetActivePage: string;
    function GetBitmap: TBitmap; { bitmap for current tab }
    function GetNumGlyphs: TTabNumGlyphs; { NumGlyphs for current tab }
    function GetTabBitmap(Index: Integer): TBitmap;
    function GetTabCaption(Index: Integer): String;
    function GetTabNumGlyphs(Index: Integer): TTabNumGlyphs;
    function GetTabPageEnabled(Index: Integer): Boolean;
    function GetTabPageIndex(const TabIdentity: String): Integer;
    function GetTabPageVisible(Index: Integer): Boolean;
    function HasTab(Row, Col: Integer): Boolean;
    procedure InitPalette(DC: HDC);
    procedure InvalidateTab(Index: Integer); { Used by TcsPage }
    procedure InvalidateTabTextRects(Which: TTabRects);
    procedure MakeFirstRow(TabIdentity: String);
    procedure MapPoints(var Pts: Array of TPoint; const SourceRect, TargetRect: TRect);
{$IFNDEF VER130}
    { Versions of the compiler earlier than Delphi 5 didn't
      hint that MouseDown was actually declared as protected
      in the ancestor class.
    }
    procedure MouseDown(Button: TMouseButton; Shift: TShiftState;
                        X, Y: Integer); override;
{$ENDIF}
    procedure Reconfigure; { Used by TcsPageAccess }
    function RotateTabRect(BaseRect, TabRect, VirtualRect: TRect): TRect;
    procedure SetActivePage(const Value: string);
    procedure SetAnchoredTabs(Value: Boolean);
    procedure SetBitmap(Value: TBitmap); { for current tab }
    procedure SetBitmapAlignment(Value: TBitmapAlignment);
    procedure SetColExtent(Value: Integer);
    procedure SetCornerSize(Value: Integer);
    procedure SetInitialPage; { Also used by TcsPage }
    procedure SetMaxTabWidth(Value: Integer);
    procedure SetNumGlyphs(Value: TTabNumGlyphs);
    procedure SetPageIndex(Value: Integer);
    procedure SetPageIndexDefault(Value: Integer);
    procedure SetPages(Value: TStrings);
    procedure SetParentTabFont(Value: Boolean);
    procedure SetRowIndent(Value: Integer);
    procedure SetSaveResources(Value: Boolean);
    procedure SetSelectedColour(Value: TColor);
    procedure SetSidewaysText(Value: Boolean);
    procedure SetTabBitmap(Index: Integer; Value: TBitmap);
    procedure SetTabCaption(Index: Integer; Value: String);
    procedure SetTabFont(Value: TFont);
    procedure SetTabHeight(Value: Integer);
    procedure SetTabNumGlyphs(Index: Integer; Value: TTabNumGlyphs);
    procedure SetTabOrientation(Value: TTabOrientation);
    procedure SetTabPageEnabled(Index: Integer; Value: Boolean);
    procedure SetTabPageVisible(Index: Integer; Value: Boolean);
    procedure SetTextAlignment(Value: TTextAlignment);
    procedure SetUnselectedColour(Value: TColor);
    procedure SetUnselectedTabColour(Value: TColor);
    procedure SetupTabs;
    procedure SetUseUnselectedTabColour(Value: Boolean);
    procedure TabFontChanged(Sender: TObject);
    procedure UpdateTabCaption(Index: Integer; Value: String); { Used by TcsPage }
    function VirtualRect(const RealRect: TRect): TRect;
    procedure WMEraseBkgnd(var Message: TWmEraseBkgnd); message WM_ERASEBKGND;
    procedure WMGetDlgCode(var Message: TWMGetDlgCode); message WM_GETDLGCODE;
    procedure WMNCHitTest(var Msg: TWMNCHitTest); message WM_NCHITTEST;
    procedure WMSetCursor(var Msg: TWMSetCursor); message WM_SETCURSOR;
    procedure CMFocusChanged(var Message: TCMFocusChanged); message CM_FOCUSCHANGED;
    procedure CMDesignHitTest(var Msg: TCMDesignHitTest); message CM_DESIGNHITTEST;
    procedure CMDialogChar(var Message: TCMDialogChar); message CM_DIALOGCHAR;
    procedure CMParentColorChanged(var Message: TMessage); message CM_PARENTCOLORCHANGED;
    procedure CMParentFontChanged(var Message: TMessage); message CM_PARENTFONTCHANGED;
    property HasLoaded: Boolean read FHasLoaded; { Used by TcsPageAccess }
  protected
    procedure AlignControls(AControl: TControl; var Rect: TRect); override;
    procedure CreateParams(var Params: TCreateParams); override;
    procedure DoExit; override;
{$IFDEF WIN32}
    function GetChildOwner: TComponent; override;
{$IFDEF VER130}
    procedure GetChildren(Proc: TGetChildProc; Root: TComponent); override;
{$ELSE}
{$IFDEF VER120}
    procedure GetChildren(Proc: TGetChildProc; Root: TComponent); override;
{$ELSE}
{$IFDEF VER100}
    procedure GetChildren(Proc: TGetChildProc; Root: TComponent); override;
{$ELSE}
    procedure GetChildren(Proc: TGetChildProc); override;
{$ENDIF}
{$ENDIF}
{$ENDIF}
{$ENDIF}
    function GetPalette: HPALETTE; override;
    procedure Loaded; override;
    procedure KeyDown(var Key: Word; Shift: TShiftState); override;
{$IFDEF VER130}
    procedure MouseDown(Button: TMouseButton; Shift: TShiftState;
                        X, Y: Integer); override;
{$ENDIF}
    procedure Paint; override;
    function PaletteChanged(Foreground: Boolean): Boolean; override;
    procedure ReadState(Reader: TReader); override;
    procedure ShowControl(AControl: TControl); override;
    procedure TabClick; dynamic;
    function TabRect(const TabIdentity: String): TRect;
{$IFNDEF WIN32}
    procedure WriteComponents(Writer: TWriter); override;
{$ENDIF}
    procedure CMTabFontChanged(var Message: TMessage); message CM_TABFONTCHANGED;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    function TabAtPos(X, Y: Integer): Integer; { Index of tab at X,Y }
    { The Tab* properties (TabBitmap, TabCaption, etc.) are provided
      as shorthand ways of programmatically accessing the pages/tabs.
    }
    property TabBitmap[Index: Integer]: TBitmap read GetTabBitmap write SetTabBitmap;
    property TabCaption[Index: Integer]: String read GetTabCaption write SetTabCaption;
    property TabNumGlyphs[Index: Integer]: TTabNumGlyphs read GetTabNumGlyphs write SetTabNumGlyphs;
    property TabPageEnabled[Index: Integer]: Boolean read GetTabPageEnabled write SetTabPageEnabled;
    property TabPageIndex[const TabIdentity: String]: Integer read GetTabPageIndex;
    property TabPageVisible[Index: Integer]: Boolean read GetTabPageVisible write SetTabPageVisible;
  published
    property Bitmap: TBitmap read GetBitmap write SetBitmap stored False;
    property ActivePage: string read GetActivePage write SetActivePage stored False;
    property Align;
    property AnchoredTabs: Boolean read FAnchoredTabs write SetAnchoredTabs default False;
{$IFDEF VER130}
    property Anchors;
{$ENDIF}
{$IFDEF VER120}
    property Anchors;
{$ENDIF}
    property BitmapAlignment: TBitmapAlignment read FBitmapAlignment write SetBitmapAlignment default baRightMiddle;
    property Color;
{$IFDEF VER130}
    property Constraints;
{$ENDIF}
{$IFDEF VER120}
    property Constraints;
{$ENDIF}
    property CornerSize: Integer read FCornerSize write SetCornerSize default 5;
    property Ctl3D;
    property DragCursor;
    property DragMode;
    property Font;
    property Enabled;
    property MaxTabWidth: Integer read FMaxTabWidth write SetMaxTabWidth default 0;
    property NumGlyphs: TTabNumGlyphs read GetNumGlyphs write SetNumGlyphs stored False;
    property PageIndex: Integer read FPageIndex write SetPageIndex default 0;
    { PageIndexDefault defines the page to display when the notebook
      is initially loaded.  It saves you from having to set the initial
      page in source code (to override whatever page you left current
      when in design-mode).
    }
    property PageIndexDefault: Integer read FPageIndexDefault write SetPageIndexDefault default 0;
    property Pages: TStrings read FAccess write SetPages stored False;
    property ParentColor;
    property ParentCtl3D;
    property ParentFont;
    property ParentShowHint;
    property ParentTabFont: Boolean read FParentTabFont write SetParentTabFont default True;
    property PopupMenu;
    property RowExtent: Integer read CalcRowExtent;
    property RowIndent: Integer read FRowIndent write SetRowIndent default 5;
    property SaveResources: Boolean read FSaveResources write SetSaveResources default False;
    property SelectedColor: TColor read FSelectedColour write SetSelectedColour default clBtnText;
    property ShowHint;
    property SidewaysText: Boolean read FSidewaysText write SetSidewaysText default False;
    property TabFont: TFont read FTabFont write SetTabFont;
    property TabHeight: Integer read FTabHeight write SetTabHeight default 30;
    property TabOrder;
    property TabOrientation: TTabOrientation read FTabOrientation write SetTabOrientation default toTop;
    property TabsPerRow: Integer read FColExtent write SetColExtent default 3;
    property TabStop default True;
    property TextAlignment: TTextAlignment read FTextAlignment write SetTextAlignment default taLeftMiddle;
    property UnselectedColor: TColor read FUnselectedColour write SetUnselectedColour default clBtnText;
    property UnselectedTabColor: TColor read FUnselectedTabColour write SetUnselectedTabColour default clBtnFace;
    property UseUnselectedTabColor: Boolean read FUseUnselectedTabColour write SetUseUnselectedTabColour default False;
    property Visible;
    property OnClick;
    property OnDblClick;
    property OnDragDrop;
    property OnDragOver;
    property OnEndDrag;
    property OnEnter;
    property OnExit;
    property OnGetPalette: TGetPaletteEvent read FOnGetPalette write FOnGetPalette;
    property OnMouseDown;
    property OnMouseMove;
    property OnMouseUp;
    property OnPaintBackground: TPaintBackgroundEvent
      read FOnPaintBackground write FOnPaintBackground;
    property OnPageChanged: TNotifyEvent read FOnPageChanged write FOnPageChanged;
    property OnPageChanging: TPageChangingEvent read FOnPageChanging write FOnPageChanging;
    property OnTabClick: TNotifyEvent read FOnTabClick write FOnTabClick;
  end;

implementation

uses Consts, Dialogs {$IFDEF EVALUATION}, CSEval {$ENDIF}, CSCConst;

{ Distance from top of one row of tabs to bottom of next row of tabs
   (behind the current row).
}
const
  CardDescent = 5;
  PolyFillMode = Integer(Alternate);  { DON'T use TFillMode values }

var
  CursorLoaded: Boolean;

function MaxInt(Expr1, Expr2: Integer): Integer;
begin
  if Expr1 >= Expr2 then
    Result := Expr1
  else
    Result := Expr2;
end;


function MinInt(Expr1, Expr2: Integer): Integer;
begin
  if Expr1 <= Expr2 then
    Result := Expr1
  else
    Result := Expr2
end;


function iifInt(Condition: Boolean; TrueValue, FalseValue: Integer): Integer;
begin
  if Condition then
    Result := TrueValue
  else
    Result := FalseValue;
end;


{ TcsPage }

constructor TcsPage.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FBitmap := TBitmap.Create;
  FBitmap.OnChange := BitmapChanged;
  Enabled := False;
  FPageEnabled := True;
  FPageVisible := True;
  FNumGlyphs := 1;
end;


destructor TcsPage.Destroy;
begin
  FBitmap.Free;
  inherited Destroy;
end;


procedure TcsPage.DestroyHandle;
begin
  inherited DestroyHandle;
end;


{ Pages will use the same palette as the notebook }
function TcsPage.GetPalette: HPALETTE;
begin
  if (Parent is TcsNotebook) then
    Result := TcsNotebook(Parent).GetPalette
  else
    Result := inherited GetPalette;
end;


{ Select and realize the control's palette }
procedure TcsPage.InitPalette(DC: HDC);
var Palette: HPALETTE;
begin
  Palette := GetPalette;
  if (Palette <> 0) then
  begin
    SelectPalette(DC, Palette, False);
    RealizePalette(DC);
  end;
end;


function TcsPage.PaletteChanged(Foreground: Boolean): Boolean;
begin
  if Showing then
    Invalidate;
  Result := inherited PaletteChanged(Foreground);
end;


procedure TcsPage.Paint;
var
  NoteBook: TcsNotebook;
  ColourOfParent: TColor;
  BrushHandle: HBRUSH;
begin
  Notebook := TcsNotebook(Parent);
  { notebook's parent can be a form, panel or scroll box }
  if (Notebook.Parent <> nil) and Notebook.ParentColor then
    { typecast as panel to allow access to the colour property }
    ColourOfParent := TPanel(Notebook.Parent).Color
  else
    ColourOfParent := Notebook.Color;
  BrushHandle := CreateSolidBrush(ColorToRGB(ColourOfParent));
  try
    InitPalette(Canvas.Handle);
    FillRect(Canvas.Handle, Canvas.ClipRect, BrushHandle);
  finally
    DeleteObject(BrushHandle);
  end;
  if (csDesigning in ComponentState) then
    with Canvas do
    begin
      Pen.Style := psDash;
      Brush.Style := bsClear;
      Rectangle(0, 0, Width, Height);
    end;
end;


procedure TcsPage.WMEraseBkgnd(var Message: TWmEraseBkgnd);
begin
  { let Paint take care of everything }
  Message.Result := 1;
end;


procedure TcsPage.ReadState(Reader: TReader);
var OldOwner: TComponent;
begin
  if Reader.Parent is TcsNotebook then
    TcsNotebook(Reader.Parent).FPageList.Add(Self);
  OldOwner := Reader.Owner;
  Reader.Owner := Reader.Root;
  try
    inherited ReadState(Reader);
  finally
    Reader.Owner := OldOwner;
  end;
end;


procedure TcsPage.SetBitmap(Value: TBitmap);
begin
  FBitmap.Assign(Value);
end;


procedure TcsPage.BitmapChanged(Sender: TObject);
var Notebook: TcsNotebook;
    Index: Integer;
begin
  { Invalidate the tab whose bitmap has been changed. }
  if (Parent is TcsNotebook) and
    not ((csReading in ComponentState) or (csLoading in ComponentState)) then
  begin
    Notebook := TcsNotebook(Parent);
    Index := Notebook.TabPageIndex[Caption];
    Notebook.InvalidateTab(Index);
  end;
end;


{ Perform additional processing after caption for page has been changed. }
procedure TcsPage.CMTextChanged(var Message: TMessage);
var Notebook: TcsNotebook;
    Index: Integer;
begin
  inherited;
  if (Parent is TcsNotebook) then
  begin
    Notebook := TcsNotebook(Parent);
    Index := Notebook.TabPageIndex[Caption];
    Notebook.UpdateTabCaption(Index, Caption);
  end;
end;


{ Change a page's PageEnabled value }
procedure TcsPage.SetPageEnabled(Value: Boolean);
begin
  if (Value <> FPageEnabled) then
  begin
    FPageEnabled := Value;
    Perform(CM_PAGEENABLEDCHANGED, 0, 0);
  end;
end;


{ Perform additional processing after PageEnabled has been changed. }
procedure TcsPage.CMPageEnabledChanged(var Message: TMessage);
var Notebook: TcsNotebook;
    Index: Integer;
    OldEnabled: Boolean;
begin
  if (Parent is TcsNotebook) and
    not ((csReading in ComponentState) or (csLoading in ComponentState)) then
  begin
    Notebook := TcsNotebook(Parent);
    Index := Notebook.TabPageIndex[Caption];
    { Invalidate the tab which has been enabled/disabled. }
    Notebook.InvalidateTab(Index);
    if not (csDesigning in ComponentState) then
    begin
      { check if any notebook pages are enabled }
      OldEnabled := Notebook.Enabled;
      Notebook.Enabled := Notebook.AnyPagesEnabled;
      { Don't change page's Enabled property unless it is the current
        page or the only PageEnabled page.
      }
      if (Caption = Notebook.ActivePage) or
        (Notebook.Enabled and not OldEnabled) then
      begin
        Enabled := FPageEnabled;
        Notebook.SetInitialPage;
      end;
    end;
  end;
end;


{ Change a page's PageVisible value }
procedure TcsPage.SetPageVisible(Value: Boolean);
begin
  if (Value <> FPageVisible) then
  begin
    FPageVisible := Value;
    Perform(CM_PAGEVISIBLECHANGED, 0, 0);
  end;
end;


{ Perform additional processing after PageVisible has been changed. }
procedure TcsPage.CMPageVisibleChanged(var Message: TMessage);
var Notebook: TcsNotebook;
begin
  if (Parent is TcsNotebook) and
    not ((csReading in ComponentState) or (csLoading in ComponentState)) then
  begin
    Notebook := TcsNotebook(Parent);
    { Note: Notebook.Visible is not automatically changed based on
            whether any pages are visible; this would interfere with
            any hiding/showing logic added by the user (developer).
    }
    Notebook.Reconfigure;
    Notebook.SetInitialPage; { move to an enabled/visible page }
    if not (csDesigning in ComponentState) then
      Notebook.Enabled := Notebook.AnyPagesEnabled;
  end;
end;


procedure TcsPage.SetNumGlyphs(Value: TTabNumGlyphs);
begin
  if (Value <> FNumGlyphs) then
  begin
    FNumGlyphs := Value;
    BitmapChanged(Self);
  end;
end;


{ TcsTab }

type
  TcsTab = class(TObject)
  private
    FRect: TRect;
  public
    constructor Create(Rect: TRect);
    property Rect: TRect read FRect write FRect;
  end;

constructor TcsTab.Create(Rect: TRect);
begin
  inherited Create;
  FRect := Rect;
end;


{ TcsPointObj }

type
  TcsPointObj = class(TObject)
  private
    FPoint: TPoint;
  public
    constructor Create(Point: TPoint);
    property Point: TPoint read FPoint write FPoint;
  end;

constructor TcsPointObj.Create(Point: TPoint);
begin
  inherited Create;
  FPoint := Point;
end;


{ TcsPageAccess }

type
  TcsPageAccess = class(TStrings)
  private
    PageList: TList;
    Notebook: TcsNotebook;
    procedure Modified;
    procedure SynchroniseNotebook(Index: Integer);
    procedure MsgDlg(const Msg: String);
  protected
    function GetCount: Integer; override;
    function Get(Index: Integer): string; override;
    procedure Put(Index: Integer; const S: string); override;
    function GetObject(Index: Integer): TObject; override;
    procedure SetUpdateState(Updating: Boolean); override;
  public
    constructor Create(APageList: TList; ANotebook: TcsNotebook);
    function Add(const S: String): Integer; override;
    procedure Clear; override;
    procedure Delete(Index: Integer); override;
    procedure Insert(Index: Integer; const S: string); override;
    procedure Exchange(Index1, Index2: Integer); override;
    procedure Move(CurIndex, NewIndex: Integer); override;
  end;

const { used by TcsPageAccess only }
  PAMsgAlreadyExists1 = 'Tab with Caption ''';
  PAMsgAlreadyExists2 = ''' already exists. Tab Captions must be unique.';
  PAMsgCannotClear = 'Cannot clear all Notebook pages.  Notebooks with no pages are not allowed.';
  PAMsgCannotDelete = 'Cannot delete last remaining Tab.  Notebooks must have at least one Tab.';

constructor TcsPageAccess.Create(APageList: TList; ANotebook: TcsNotebook);
begin
  inherited Create;
  PageList := APageList;
  Notebook := ANotebook;
end;


function TcsPageAccess.GetCount: Integer;
begin
  Result := PageList.Count;
end;


function TcsPageAccess.Get(Index: Integer): string;
begin
  Result := TcsPage(PageList[Index]).Caption;
end;


procedure TcsPageAccess.Put(Index: Integer; const S: string);
begin
  if (IndexOf(S) >= 0) and (IndexOf(S) <> Index) then
    { caption already exists and is not for same tab; tabs must be unique }
    MsgDlg(PAMsgAlreadyExists1 + S + PAMsgAlreadyExists2)
  else { Not found }
  begin
    TcsPage(PageList[Index]).Caption := S;
    Modified;
    { No need to do SynchroniseNotebook for caption change. }
  end;
end;


function TcsPageAccess.GetObject(Index: Integer): TObject;
begin
  { return page at specified index }
  Result := PageList[Index];
end;


procedure TcsPageAccess.SetUpdateState(Updating: Boolean);
begin
  { do nothing; method is needed because it is virtual in TStrings }
end;


procedure TcsPageAccess.Clear;
var I: Integer;
begin
  if Notebook.HasLoaded then
    MsgDlg(PAMsgCannotClear)
  else
  begin
    { component hasn't been loaded yet; OK to clear }
    for I := 0 to PageList.Count - 1 do
      TcsPage(PageList[I]).Free;
    PageList.Clear;
  end;
end;


procedure TcsPageAccess.Delete(Index: Integer);
var OldPage: String;
    OldIndex: Integer;
begin
  if PageList.Count <= 1 then
    MsgDlg(PAMsgCannotDelete)
  else { there will still be a Tab after delete }
  begin
    { because deleting will cause the indexes of each remaining
      page (after the deleted one) to change we must remember
      the current page by it's caption
    }
    OldPage := Notebook.ActivePage;
    TcsPage(PageList[Index]).Free;
    PageList.Delete(Index);
    Modified;
    { Must move PageIndex off deleted page before doing
      SynchroniseNotebook; otherwise FPageIndex (and thus
      ActivePage) will be invalid and will cause errors when
      used by CalcAndBuild etc.
    }
    Notebook.PageIndex := 0; { always at least one page }
    OldIndex := IndexOf(OldPage);
    if (OldIndex < 0) then
      { current page was deleted; use first as new current }
      OldIndex := 0;
    SynchroniseNotebook(OldIndex);
  end;
end;

procedure TcsPageAccess.Insert(Index: Integer; const S: string);
var Page: TcsPage;
begin
  if (IndexOf(S) >= 0) then
    MsgDlg(PAMsgAlreadyExists1 + S + PAMsgAlreadyExists2)
  else { Not found; tab caption is unique. }
  begin
    Page := TcsPage.Create(Notebook);
    with Page do
    begin
      Parent := Notebook;
      Caption := S;
    end;
    PageList.Insert(Index, Page);
    Modified;
    SynchroniseNotebook(Index);
  end;
end;


procedure TcsPageAccess.Exchange(Index1, Index2: Integer);
var AObject: TObject;
    OldIndex, TempIndex: Integer;
begin
  if Index1 <> Index2 then
  begin
    AObject := PageList[Index1];
    PageList[Index1] := PageList[Index2];
    PageList[Index2] := AObject;
    Modified;
    OldIndex := Notebook.PageIndex;
    if ((OldIndex = Index1) or (OldIndex = Index2)) then
    begin
      { exchanging from or to current page's position }
      if (Index1 = OldIndex) then
        TempIndex := Index2
      else
        TempIndex := Index1;
      { Synchronise with a different page as the temporary current page... }
      SynchroniseNotebook(TempIndex);
      { ... and then return to original page so controls on current page
        are shown.
      }
      Notebook.PageIndex := OldIndex;
    end
    else { neither of the pages exchanged are the current page }
      SynchroniseNotebook(-1); { no page change necessary }
  end;
end;


{ Move the page at position CurIndex to position NewIndex.
  The same active page is retained.  Note that the active page is
  not necessarily the page which was at position CurIndex.
}
procedure TcsPageAccess.Move(CurIndex, NewIndex: Integer);
var OldActivePage: String;
    TempObject: TObject;
    NewActiveIndex: Integer;
begin
  if (CurIndex <> NewIndex) and
    (CurIndex >= 0) and (CurIndex < PageList.Count) and
    (NewIndex >= 0) and (NewIndex < PageList.Count) then
  begin
    { Deleting will cause the indexes of each remaining
      page after the deleted one to change (by -1) so we must
      remember the _caption_ of the active page, not its index
      position, so we can return to the same active page.
    }
    OldActivePage := Notebook.ActivePage;  { caption of current page }
    TempObject := GetObject(CurIndex);     { actual page being moved }
    PageList.Delete(CurIndex);             { delete page being moved }
    PageList.Insert(NewIndex, TempObject); { insert page at new pos. }
    Modified;
    NewActiveIndex := IndexOf(OldActivePage);
    SynchroniseNotebook(NewActiveIndex);   { retain same active page }
  end;
end;


function TcsPageAccess.Add(const S: String): Integer;
var Page: TcsPage;
begin
  if (IndexOf(S) >= 0) then
  begin
    Result := -1;
    MsgDlg(PAMsgAlreadyExists1 + S + PAMsgAlreadyExists2);
  end
  else { Not found; tab caption is unique. }
  begin
    Page := TcsPage.Create(Notebook);
    with Page do
    begin
      Parent := Notebook;
      Caption := S;
    end;
    Result := PageList.Add(Page);
    Modified;
    SynchroniseNotebook(PageList.Count - 1);
  end;
end;

procedure TcsPageAccess.Modified;
var
{$IFDEF VER130}
  Form: TCustomForm;
{$ELSE}
{$IFDEF VER120}
  Form: TCustomForm;
{$ELSE}
{$IFDEF VER100}
  Form: TCustomForm;
{$ELSE}
  Form: TForm;
{$ENDIF}
{$ENDIF}
{$ENDIF}
begin
  if csDesigning in NoteBook.ComponentState then
  begin
    Form := GetParentForm(NoteBook);
    if (Form <> nil) and (Form.Designer <> nil) then
      Form.Designer.Modified;
  end;
end;


procedure TcsPageAccess.SynchroniseNotebook(Index: Integer);
var OldIndex: Integer;
begin
  with Notebook do
    if HasLoaded then
    begin { Rebuild data structures to match contents of PageList }
      OldIndex := PageIndex;
      Reconfigure;
      if Index >= 0 then
        PageIndex := Index
      else { -1 }
        PageIndex := OldIndex;
    end;
end;

procedure TcsPageAccess.MsgDlg(const Msg: String);
begin
  MessageDlg(Msg, mtError, [mbOK], 0);
end;


{ TcsNotebook }

constructor TcsNotebook.Create(AOwner: TComponent);
const
  Registered: Boolean = False;
begin
  inherited Create(AOwner);
{$IFDEF WIN32}
  Exclude(FComponentStyle, csInheritable);
{$ENDIF}
  Width := 250;
  Height := 150;
  FHasLoaded := False;
  FHasFocusRect := False;
  FPageList := TList.Create;
  FAccess := TcsPageAccess.Create(FPageList, Self);
  FPageIndex := -1;
{$IFDEF VER130}
  FAccess.Add(SDefault);
{$ELSE}
{$IFDEF VER120}
  FAccess.Add(SDefault);
{$ELSE}
{$IFDEF VER100}
  FAccess.Add(SDefault);
{$ELSE}
  FAccess.Add(LoadStr(SDefault));
{$ENDIF}
{$ENDIF}
{$ENDIF}
  PageIndex := 0;
  if not Registered then
  begin
    Classes.RegisterClasses([TPage, TcsPage]);
    Registered := True;
  end;

  { define default property values }
  FAnchoredTabs := False;
  FColExtent := 3;
  FRowIndent := 5;
  FCornerSize := 5;
  FTabOrientation := toTop;
  FSidewaysText := False;
  FMaxTabWidth := 0;
  FTabHeight := 25;
  FTextAlignment := taLeftMiddle;
  FBitmapAlignment := baRightMiddle;
  FRowIdentities := TList.Create;
  FTabIndex := TStringList.Create;
  FTabFont := TFont.Create;
  FTabFont.OnChange := TabFontChanged;
  FSaveResources := False;
  TabStop := True;
  FSelectedColour := clBtnText;
  FUnselectedColour := FSelectedColour;
  FParentTabFont := True;
  FUnselectedTabColour := clBtnFace;
  FUseUnselectedTabColour := False;

  CalcAndBuild; { for 'Default' page only }
end;


destructor TcsNotebook.Destroy;
var I: Integer;
begin
  for I := 0 to FRowIdentities.Count - 1 do
  begin
    TStringList(FRowIdentities.Items[I]).Clear;
    TStringList(FRowIdentities.Items[I]).Free; { free the item }
  end;
  FRowIdentities.Free; { free the list }

  for I := 0 to FTabIndex.Count - 1 do
    TcsTab(FTabIndex.Objects[I]).Free;
  FTabIndex.Free;

  FTabFont.Free;
  FAccess.Free;
  FPageList.Free;
  inherited Destroy;
end;


procedure TcsNotebook.CreateParams(var Params: TCreateParams);
begin
  inherited CreateParams(Params);
  with Params do Style := Style or WS_CLIPCHILDREN;
end;


procedure TcsNotebook.ReadState(Reader: TReader);
var OldOwner: TComponent;
begin
  Pages.Clear;
  OldOwner := Reader.Owner;
  Reader.Owner := Self;
  try
    inherited ReadState(Reader);
  finally
    Reader.Owner := OldOwner;
  end;
  if (FPageIndex <> -1) and (FPageIndex >= 0) and (FPageIndex < FPageList.Count) then
    with TcsPage(FPageList[FPageIndex]) do
    begin
      BringToFront;
      Enabled := True;
      Visible := True;
      Align := alClient;
    end
  else FPageIndex := -1;
end;


procedure TcsNotebook.ShowControl(AControl: TControl);
var I: Integer;
begin
  for I := 0 to FPageList.Count - 1 do
    if FPageList[I] = AControl then
    begin
      SetPageIndex(I);
      Exit;
    end;
  inherited ShowControl(AControl);
end;


{$IFDEF WIN32}
function TcsNotebook.GetChildOwner: TComponent;
begin
  Result := Self;
end;

{$IFDEF VER130}
procedure TcsNotebook.GetChildren(Proc: TGetChildProc;
  Root: TComponent);
{$ELSE}
{$IFDEF VER120}
procedure TcsNotebook.GetChildren(Proc: TGetChildProc;
  Root: TComponent);
{$ELSE}
{$IFDEF VER100}
procedure TcsNotebook.GetChildren(Proc: TGetChildProc;
  Root: TComponent);
{$ELSE}
procedure TcsNotebook.GetChildren(Proc: TGetChildProc);
{$ENDIF}
{$ENDIF}
{$ENDIF}
var
  I: Integer;
begin
  for I := 0 to FPageList.Count - 1 do Proc(TControl(FPageList[I]));
end;

{$ELSE}
procedure TcsNotebook.WriteComponents(Writer: TWriter);
var I: Integer;
begin
  for I := 0 to ControlCount - 1 do
    Writer.WriteComponent(FPageList[I]);
end;
{$ENDIF}


procedure TcsNotebook.SetPages(Value: TStrings);
begin
  FAccess.Assign(Value);
end;


procedure TcsNotebook.SetPageIndex(Value: Integer);
var TabIdentity: String;
{$IFDEF VER130}
    ParentForm: TCustomForm;
{$ELSE}
{$IFDEF VER120}
    ParentForm: TCustomForm;
{$ELSE}
{$IFDEF VER100}
    ParentForm: TCustomForm;
{$ELSE}
    ParentForm: TForm;
{$ENDIF}
{$ENDIF}
{$ENDIF}
    AllowChange: Boolean;
    OldPageIndex: Integer;

  procedure InvalidateTabAreas(OldIndex, NewIndex: Integer);
  var OldIdentity, NewIdentity: String;
      OldRow, OldCol, NewRow, NewCol: Integer;
      Pts: TcsTabPoints;
      TabRegion: HRGN;
      Rect: TRect;

    { Extend the first and last segments of the tab frames (corresponding
      to left and right hand edges if viewing in top orientation) to
      include the top bevel of the front card.
    }
    procedure ExtendRgn(var Pts: TcsTabPoints);
    begin
      if not FAnchoredTabs then
        case FTabOrientation of
          toTop:
            begin
              Pts[0].Y := Pts[0].Y + 4;
              Pts[5].Y := Pts[0].Y;
            end;
          toBottom:
            begin
              Pts[0].Y := Pts[0].Y - 3;
              Pts[5].Y := Pts[0].Y;
            end;
          toLeft:
            begin
              Pts[0].X := Pts[0].X + 4;
              Pts[5].X := Pts[0].X;
            end;
          toRight:
            begin
              Pts[0].X := Pts[0].X - 3;
              Pts[5].X := Pts[0].X;
            end;
        end;
    end; {ExtendRgn }

    { Adjust the rect to be invalidated so the edges of the
      top of front card are also included (current tab is joined
      to card); edge to be adjusted varies with orientation
    }
    procedure ExtendRect(var Rect: TRect);
    begin
      if not FAnchoredTabs then
      begin
        with Rect do
          case FTabOrientation of
            toTop:    begin
                        Right := Right + 1;
                        Bottom := Bottom + 3;
                      end;
            toBottom: begin
                        Top := Top - 2;
                        Right := Right + 1;
                        Bottom := Bottom + 1;
                      end;
            toLeft:   begin
                        Right := Right + 3;
                        Bottom := Bottom + 1;
                      end;
            toRight:  begin
                        Left := Left - 2;
                        Right := Right + 1;
                        Bottom := Bottom + 1;
                      end;
          end;
      end;
    end; { ExtendRect }

  { Where no swapping of rows of tabs is to occur, only the previous
    and current tabs' regions will be invalidated to improve the visual
    appearance (by reducing flicker) of the subsequent redraw.
    Otherwise, the whole index rect area is invalidated.
  }
  begin { InvalidateTabAreas }
    if (OldIndex < FAccess.Count) and HandleAllocated then
    begin
      OldIdentity := FAccess[OldIndex];
      NewIdentity := FAccess[NewIndex];
      OldRow := FindTabRow(OldIdentity);
      NewRow := FindTabRow(NewIdentity);

      if ((FRowExtent = 1) or (OldRow = NewRow) or FAnchoredTabs) and
        TcsPage(FPageList[OldIndex]).PageVisible and
        TcsPage(FPageList[NewIndex]).PageVisible then
      begin { no row swapping; only invalidate old and new tab areas }

        OldCol := FindTabCol(OldIdentity);
        NewCol := FindTabCol(NewIdentity);

        { invalidate region of new tab }
        Rect := TabRect(NewIdentity);
        BuildTabPoints(Pts, Rect, 0, NewRow, NewCol);
        ExtendRgn(Pts);
        TabRegion := CreatePolygonRgn(Pts, Integer(High(Pts) + 1), PolyFillMode);
        InvalidateRgn(Handle, TabRegion, True);
        DeleteObject(TabRegion);

        { invalidate region of old tab }
        Rect := TabRect(OldIdentity);
        BuildTabPoints(Pts, Rect, 0, OldRow, OldCol);
        ExtendRgn(Pts);
        TabRegion := CreatePolygonRgn(Pts, Integer(High(Pts) + 1), PolyFillMode);
        InvalidateRgn(Handle, TabRegion, True);
        DeleteObject(TabRegion);

      end
      else
      begin { rows have been swapped }
        Rect := CalcIndexRect;
        ExtendRect(Rect);
        InvalidateRect(Handle, @Rect, True);
      end;
    end;
  end; { InvalidateTabAreas }

begin
  if csLoading in ComponentState then
  begin
    FPageIndex := Value;
    Exit;
  end;
  { The inclusion of "not TcsPage(FPageList[FPageIndex]).PageVisible" is a
    special test necessary to allow changing (to a possibly disabled page)
    from a current page that has been made invisible.
  }
  if (Value <> FPageIndex) and (Value >= 0) and (Value < FPageList.Count) and
     ((csDesigning in ComponentState) or
      TcsPage(FPageList[Value]).PageEnabled or
      not TcsPage(FPageList[FPageIndex]).PageVisible) and
     TcsPage(FPageList[Value]).PageVisible then
  begin
    if not (csDesigning in ComponentState) then
    begin
      ParentForm := GetParentForm(Self);
      if (ParentForm <> nil) and
        (ParentForm.ActiveControl <> Self) and
        ContainsControl(ParentForm.ActiveControl) then
        ParentForm.ActiveControl := Self;
    end;

    AllowChange := True;
    if Assigned(FOnPageChanging) then
      FOnPageChanging(Self, Value, AllowChange);

    if AllowChange then
    begin
      if FHasLoaded then
      begin
        TabIdentity := FAccess[Value];
        if (not FAnchoredTabs) then
          MakeFirstRow(TabIdentity);
      end;

      with TcsPage(FPageList[Value]) do
      begin { show new page }
        BringToFront;
        Enabled := True;
        Visible := True;
        Align := alClient;
      end;
      if (FPageIndex >= 0) and (FPageIndex < FPageList.Count) then
        with TcsPage(FPageList[FPageIndex]) do
        begin { hide old page and release resources }
          Enabled := False;
          Visible := False;
          if FSaveResources and not (csDesigning in ComponentState) then
            DestroyHandle;
        end;

      OldPageIndex := FPageIndex;
      FPageIndex := Value;
      FHasFocusRect := False;

      { force appropriate tabs to be redrawn }
      InvalidateTabAreas(OldPageIndex, FPageIndex);

      if Assigned(FOnPageChanged) then
        FOnPageChanged(Self);

    end;
  end;
end;


procedure TcsNotebook.SetActivePage(const Value: string);
begin
  SetPageIndex(FAccess.IndexOf(Value));
end;


function TcsNotebook.GetActivePage: string;
begin
  Result := FAccess[FPageIndex];
end;


procedure TcsNotebook.AlignControls(AControl: TControl; var Rect: TRect);
begin
  Rect := CalcPageRect;
  inherited AlignControls(AControl, Rect);
end;


procedure TcsNotebook.Paint;
begin
  if AnyPagesVisible then
  begin
    InitPalette(Canvas.Handle);
    DrawTabFrames;
    DrawBevels;
    DrawCards;
    DrawTabFaces;
    if FHasFocusRect then
      { only redrawing, not changing tabs (so the Focus Change event
        which is normally used to draw the focus rect will not occur);
        have to explicitly draw focus rect.
      }
      DrawTabFocusRect;
  end;
end;


function TcsNotebook.CalcWholeRect;
var WholeRect: TRect;
begin
  { ClientRect cannot be used if control is being created
    and has no window handle yet.
  }
  if HandleAllocated then
    WholeRect := ClientRect
  else
    WholeRect := Rect(0, 0, Width, Height);
  Result := Rect(0, 0, WholeRect.Right - 1, WholeRect.Bottom - 1);
end;


function TcsNotebook.CalcIndexHeight;
begin
  Result := (FTabHeight * FRowExtent);
end;


{ --------------------------------------------------------------------- }
{ Return the coordinates for the index component (the area where        }
{ the tabs are displayed) of the window.                                }
{ --------------------------------------------------------------------- }
function TcsNotebook.CalcIndexRect;
var WholeRect, IndexRect: TRect;
    IndexHeight, l, t, r, b, w, h: Integer;
begin
  WholeRect := CalcWholeRect;
  IndexHeight := CalcIndexHeight;
  w := WholeRect.Right - WholeRect.Left;
  h := WholeRect.Bottom - WholeRect.Top;

  if FTabOrientation = toRight then l := w - IndexHeight else l := 0;
  if FTabOrientation = toBottom then t := h - IndexHeight else t := 0;
  if FTabOrientation = toLeft then r := IndexHeight else r := w;
  if FTabOrientation = toTop then b := IndexHeight else b := h;

  IndexRect := Rect(l, t, r, b);

  w := r - l;
  h := b - t;

  if FMaxTabWidth > 0 then
  begin
    if (FTabOrientation = toTop) or (FTabOrientation = toBottom) then
    begin
      if w > (FColExtent*FMaxTabWidth) then
        IndexRect.Right := (FColExtent*FMaxTabWidth) - 1;
    end
    else
      if h > (FColExtent*FMaxTabWidth) then
        if FTabOrientation = toLeft then
          IndexRect.Top := b - ((FColExtent*FMaxTabWidth) - 1)
        else
          { toRight }
          IndexRect.Bottom := (FColExtent*FMaxTabWidth) - 1;
  end;

  Result := IndexRect;

end;


{ Return the coordinates for the area to be used by the pages
  of the notebook (the area where the tabs are not displayed).
}
function TcsNotebook.CalcPageRect;
var CardRect: TRect;
begin
  CardRect := CalcCardRect(0);  { Row 0 is front card }
  CardRect.Left := CardRect.Left + 3;
  CardRect.Top := CardRect.Top + 3;
  CardRect.Right := CardRect.Right - 2;
  CardRect.Bottom := CardRect.Bottom - 2;

  { Inflate left and top by -3 and right and bottom by -2 }
  Result := CardRect;

end;


function TcsNotebook.CalcRowExtent;
var PageCount, I: Integer;
begin
  if FColExtent = 0 then
    Result := 1
  else
  begin
    PageCount := 0;
    for I := 0 to FPageList.Count - 1 do
      if TcsPage(FPageList.Items[I]).PageVisible then
        Inc(PageCount);
    Result := ((PageCount - 1) div FColExtent) + 1;
  end;
end;


procedure TcsNotebook.SetTabOrientation(Value: TTabOrientation);
var OldTabOrientation: TTabOrientation;
begin
  if Value <> FTabOrientation then
  begin
    OldTabOrientation := FTabOrientation;
    FTabOrientation := Value;
    if not ((csLoading in ComponentState) or (csReading in ComponentState)) then
    begin
      if ((FTabOrientation in [toLeft, toRight]) and (OldTabOrientation in [toTop, toBottom])) or
         ((FTabOrientation in [toTop, toBottom]) and (OldTabOrientation in [toLeft, toRight])) then
        FSidewaysText := (FTabOrientation in [toLeft, toRight]);
      CalcAndBuild;
      Invalidate;
    end;
  end;
end;


procedure TcsNotebook.BuildRowIdentities;
var I, Col, Index: Integer;
    Cols: TStringList;
    Str: String;
begin
  { clear any existing entries }
  for I := 0 to FRowIdentities.Count - 1 do
  begin
    TStringList(FRowIdentities.Items[I]).Clear;
    TStringList(FRowIdentities.Items[I]).Free; { free the item }
  end;
  FRowIdentities.Clear;

  Col := 0;
  Index := 0;

  while Index < FPageList.Count do
  begin
    { add tabs for this row }
    Cols := TStringList.Create;
    repeat
      Str := FAccess[Index];
      if TcsPage(FPageList[Index]).PageVisible then
      begin
        Cols.Add(Str);
        Col := Col + 1;
      end;
      Index := Index + 1;
    until (Index = FPageList.Count) or (Col = FColExtent);
    if (Cols.Count > 0) then
    begin
      FRowIdentities.Add(Cols);
      Col := 0;
    end;
  end;

end;


procedure TcsNotebook.SetupTabs;
var IndexRect, TbRect: TRect;
    l, t, w, h, xInc, yInc, MaxY, Row, Col, Pos: Integer;
    TabIdentity: String;
begin
  IndexRect := CalcIndexRect;

  { Note: Row and Col are 0 based.
          RowExtent and ColExtent are 1 based. Also note that row/col
          are switched for left/right oriented tabs, i.e. as if
          you were viewing at 90 degrees rotation.
  }
  Row := 0;
  Col := 0;

  w := IndexRect.Right - IndexRect.Left;
  h := IndexRect.Bottom - IndexRect.Top;
  MaxY := h;

  if (FTabOrientation = toTop) or (FTabOrientation = toBottom) then
  begin
    xInc := ((w - ((FRowExtent - 1)*FRowIndent)) div FColExtent);
    yInc := (h div FRowExtent);
  end
  else
  begin
    xInc := (w div FRowExtent);
    yInc := ((h - ((FRowExtent - 1)*FRowIndent)) div FColExtent);
  end;

  { assign required origin coordinates }
  if FTabOrientation = toLeft then l := ((FRowExtent - 1)*xInc) else l := 0;
  if FTabOrientation = toTop then
    t := ((FRowExtent - 1)*yInc)
  else
    if FTabOrientation = toLeft then t := MaxY - yInc else t := 0;

  { setup the index tabs }
  while Row < FRowExtent do
  begin

    if HasTab(Row, Col) then
    begin
      { Note: Addition/subtraction of one (1) to some of the coords
               is done to prevent gaps between tabs.
      }
      TabIdentity := TStringList(FRowIdentities.Items[Row]).Strings[Col];
      TbRect := Rect(l, t, l + xInc, t + yInc);

      { stretch tab in last column (if necessary) }
      AlignChildRect(TbRect, Row, Col);

      { set the child rect for the tab }
      Pos := FTabIndex.IndexOf(TabIdentity);
      if Pos < 0 then
        FTabIndex.AddObject(TabIdentity, TcsTab.Create(TbRect))
      else
        TcsTab(FTabIndex.Objects[Pos]).Rect := TbRect;

      case FTabOrientation of
        toTop, toBottom: l := l + xInc;
        toLeft: t := t - yInc;
        toRight: t := t + yInc;
      end;

      Col := Col + 1;
    end
    else
      { the rest of this row is empty }
      Col := FColExtent;

    if Col = FColExtent then  { start new row }
    begin
      Row := Row + 1;
      Col := 0;
      case FTabOrientation of
        toTop:
        begin
          l := ((Row)*FRowIndent);
          t := ((FRowExtent - Row - 1)*yInc);
        end;
        toBottom:
        begin
          l := ((Row)*FRowIndent);
          t := t + yInc;
        end;
        toLeft:
        begin
          l := ((FRowExtent - Row - 1)*xInc);
          t := ((FColExtent - 1)*yInc) + ((FRowExtent - Row - 1)*FRowIndent);
        end;
        toRight:
        begin
          l := ((Row)*xInc);
          t := ((Row)*FRowIndent);
        end;
      end;
    end;
  end;
end;


{ Aligns the child rect for the last tab in a row (when no maximum tab
  width specified) so that it's side will line up with the card.
  ------------------------------ Note --------------------------------
  If you want the tabs (especially when only one row) to be flush with
  the right hand side then use FMaxTabWidth = 0.  Only if you want
  a blank section between the last tab in the row and the edge of the
  card should FMaxTabWidth be > 0.  Otherwise specify FMaxTabWidth = 0
  to prevent a jagged join on the edge.
}
procedure TcsNotebook.AlignChildRect(var childRect: TRect; Row, Col: Integer);
var WholeRect: TRect;
    Margin: Integer;
begin
  if ((FMaxTabWidth = 0) and (Col = FColExtent - 1)) then
  begin
    { last column for row; clip or stretch to boundary }
    WholeRect := CalcWholeRect;
    Margin := ((FRowExtent - Row - 1)*FRowIndent);

    case FTabOrientation of
      toTop, toBottom:
        childRect.Right := WholeRect.Right - Margin;
      toLeft:
        childRect.Top := Margin;
      toRight:
        childRect.Bottom := WholeRect.Bottom - Margin;
    end;

  end;
end;


procedure TcsNotebook.SetColExtent(Value: Integer);
begin
  if (Value > 0) then
    FColExtent := Value
  else
    FColExtent := 3; { default value }
  if not ((csReading in ComponentState) or (csLoading in ComponentState)) then
  begin
    CalcAndBuild;
    Invalidate;
  end;
end;


{ Return the child rect, RELATIVE TO THE INDEX RECT, for the tab
   with the specified identity.
}
function TcsNotebook.TabRect(const TabIdentity: String): TRect;
var IRect: TRect;
    Index: Integer;
begin
  IRect := CalcIndexRect;
  Index := FTabIndex.IndexOf(TabIdentity);
  Result := TcsTab(FTabIndex.Objects[Index]).Rect;
  OffsetRect(Result, IRect.Left, IRect.Top);
end;


{ Determine the row containing the tab with the specified identity.
   Note: The first row is row 0, 2nd row is row 1, etc.
}
function TcsNotebook.FindTabRow(const TabIdentity: String): Integer;
var I: Integer;
begin
  Result := 0;

  for I := 0 to FRowIdentities.Count - 1 do
  begin
    if TStringList(FRowIdentities.Items[I]).IndexOf(TabIdentity) >= 0 then
    begin
      Result := I;
      Exit;
    end;
  end;

end;


{ Determine the column containing the tab with the specified identity.
   Note: The first column is column 0, 2nd column is column 1 etc.
}
function TcsNotebook.FindTabCol(const TabIdentity: String): Integer;
var Row: Integer;
begin
  Row := FindTabRow(TabIdentity);
  Result := TStringList(FRowIdentities.Items[Row]).IndexOf(TabIdentity);
end;


{ Returns a 'virtual' rectangle which is used when calculating
  tab & card points which are to subsequently be mapped according
  to the current orientation before being drawn.
  For #left and #right orientations the width and height are
  swapped. The origin is unaffected.
}
function TcsNotebook.VirtualRect(const RealRect: TRect): TRect;
begin
  if (FTabOrientation = toTop) or (FTabOrientation = toBottom) then
    Result := RealRect
  else
    Result := Rect(RealRect.Left,
                   RealRect.Top,
                   RealRect.Left + (RealRect.Bottom - RealRect.Top),
                   RealRect.Top + (RealRect.Right - RealRect.Left));
end;


{ Calculate how far the tab frame must descend to the tab below it.
  Note that in certain orientations (e.g. bottom) the distance is actually
  an ascention.
}
function TcsNotebook.CalcTabDescent: Integer;
var RealIndent, RealCornerSize: Real;
begin
  RealIndent := FRowIndent;
  RealCornerSize := FCornerSize;
  if FRowIndent < FCornerSize then
  begin
    Result := Integer(Trunc((1.0 - (RealIndent/RealCornerSize))*RealCornerSize));
    if Result < 0 then
      Result := 0;
  end
  else
    Result := 0;
end;


{ Map the specified points from top orientation in the source rectangle
   to the current orientation in the target rectangle.
   No scaling is done (1:1 mapping performed).
}
procedure TcsNotebook.MapPoints(var Pts: Array of TPoint; const SourceRect, TargetRect: TRect);
var pt: TPoint;
    I: Integer;
begin
  for I := Low(Pts) to High(Pts) do
  begin
    pt := Pts[I];

    case FTabOrientation of
      { toTop: no mapping necessary }
      toBottom:
        pt := Point(pt.X, TargetRect.Bottom - (pt.Y - SourceRect.Top));
      toLeft:
        pt := Point(TargetRect.Left + (pt.Y - SourceRect.Top),
                    TargetRect.Bottom - (pt.X - SourceRect.Left));
      toRight:
        pt := Point(TargetRect.Left + (SourceRect.Bottom - pt.Y),
                    TargetRect.Top + (pt.X - SourceRect.Left));
    end;

    { assign the new point value }
    Pts[I] := pt;
  end;

end;


{ Builds an array of points outlining the shape of a tab.
  Offset denotes the offset inside the tab's border.
  Note that because these points are used with the Windows lineTo()
  function the y values for the very first and very last points are not
  the same because lines are drawn up to but not including the last point;
  hence the last y value has to extend 1 unit further.
  The points refer to a tab in top orientation.
  The argument row refers to the 0 based row no. (starting from front
  card) that contains the tab.
}
procedure TcsNotebook.BuildTabPoints(var Pts: TcsTabPoints; TargetRect: TRect; Offset, Row, Col: Integer);
var VRect: TRect;
    l, t, r, b, Descent, Adjust: Integer;
begin
  VRect := VirtualRect(TargetRect);
  Descent := CalcTabDescent;

  l := VRect.Left;
  t := VRect.Top;
  r := VRect.Right;
  b := VRect.Bottom;

  if Row = 0 then
    Pts[0] := point(l + Offset, b - 1)
  else
  begin
    if Col = 0 then
      Pts[0] := point(l + Offset, b + maxInt(0, Descent - Offset) - 1)
    else
    begin
      if HasTab(Row - 1, Col) then
        Pts[0] := point(l + Offset, b + minInt(maxInt(0, Descent - Offset), CardDescent) - 1)
      else
        { no tab in same column in row below; extend down }
        Pts[0] := point(l + Offset, b + CardDescent);
    end;
  end;

  Pts[1] := point(l + Offset, t + FCornerSize);
  Pts[2] := point(l + FCornerSize, t + Offset);
  Pts[3] := point(r - FCornerSize, t + Offset);
  Pts[4] := point(r - Offset, t + FCornerSize);

  if Row = 0 then
    Pts[5] := point(r - Offset, b)
  else
  begin
    if (Col = FColExtent - 1) or
       ((Col < FColExtent - 1) and
        not HasTab(Row - 1, Col + 1)) then
      { last column or no tab in next column in row below }
      Pts[5] := point(r - Offset, b + CardDescent)
    else
      { Determine length based on where between the tabs the edge descends. }
      begin
        Pts[5] := point(r - Offset, 0);
        case FRowIndent of
          0:
            Pts[5].Y := b + minInt(CardDescent, maxInt(0, FCornerSize - Offset));
          1:
            begin
              if Offset = 1 then
                Adjust := 1
              else
                Adjust := 0;
              Pts[5].Y := b + minInt(CardDescent, FCornerSize - 1 + Adjust);
            end;
          else
            if (FRowIndent >= 2) and (FRowIndent <= FCornerSize) then
              Pts[5].Y := b + minInt(CardDescent, Descent + Offset)
            else
              if FRowIndent = FCornerSize + 1 then
              begin
                if Offset = 2 then
                  Adjust := 1
                else
                  Adjust := 0;
                Pts[5].Y := b + Adjust;
              end
              else
                Pts[5].Y := b;
        end;
      end;

  end;

  MapPoints(Pts, VRect, TargetRect);
end;


procedure TcsNotebook.DrawTabFrames;
var TabIdentity: String;
    Row, Col: Integer;
    TbRect: TRect;
    Pts: TcsTabPoints;
    I, J: Integer;
begin
  for I := 0 to FTabIndex.Count - 1 do
  begin
    TabIdentity := FTabIndex[I];
    Row := FindTabRow(TabIdentity);
    Col := FindTabCol(TabIdentity);
    TbRect := TabRect(TabIdentity);
    BuildTabPoints(Pts, TbRect, 0, Row, Col);
    for J := 0 to High(Pts) - 1 do
      begin
        with Canvas do
        begin
          Pen.Color := clWindowFrame;
          Pen.Style := psSolid;
          MoveTo(Pts[J].X, Pts[J].Y);
          LineTo(Pts[J + 1].X, Pts[J + 1].Y);
        end;
      end;
  end;
end;


procedure TcsNotebook.DrawBevels;
var I: Integer;
    TabIdentity: String;
    TbRect: TRect;
    Pts1, Pts2: TcsTabPoints;
    Row, Col: Integer;
begin

  for I := 0 to FTabIndex.Count - 1 do
  begin
    TabIdentity := FTabIndex.Strings[I];
    Row := FindTabRow(TabIdentity);
    Col := FindTabCol(TabIdentity);
    TbRect := TabRect(TabIdentity);
    BuildTabPoints(Pts1, TbRect, 1, Row, Col);
    BuildTabPoints(Pts2, TbRect, 2, Row, Col);

    case FTabOrientation of
      toTop:
        begin
          DrawBevelTop(Pts1, 1);
          if TabIdentity = ActivePage then
            DrawBevelTop(Pts2, 2);
        end;
      toBottom:
        begin
          DrawBevelBottom(Pts1, 1);
          if TabIdentity = ActivePage then
            DrawBevelBottom(Pts2, 2);
        end;
      toLeft:
        begin
          DrawBevelLeft(Pts1, 1);
          if TabIdentity = ActivePage then
            DrawBevelLeft(Pts2, 2);
        end;
      toRight:
        begin
          DrawBevelRight(Pts1, 1);
          if TabIdentity = ActivePage then
            DrawBevelRight(Pts2, 2);
        end;
    end;
  end;
end;


procedure TcsNotebook.DrawBevelTop(Pts: TcsTabPoints; Offset: Integer);
begin
  with Canvas do
  begin
    Pen.Color := clBtnHighlight;
    MoveTo(Pts[0].X, Pts[0].Y);
    LineTo(Pts[1].X, Pts[1].Y);
    LineTo(Pts[2].X, Pts[2].Y);
    if Offset = 2 then
    begin
      { draw additional diagonal line for correct thickness }
      MoveTo(Pts[1].X, Pts[1].Y + 1);
      LineTo(Pts[2].X + 1, Pts[2].Y);
    end;
    MoveTo(Pts[2].X, Pts[2].Y);
    LineTo(Pts[3].X, Pts[3].Y);
    Pen.Color := clBtnShadow;
    LineTo(Pts[4].X, Pts[4].Y);
    if Offset = 2 then
    begin
      { draw additional diagonal line for correct thickness }
      MoveTo(Pts[3].X - 1, Pts[3].Y);
      LineTo(Pts[4].X, Pts[4].Y + 1);
    end;
    MoveTo(Pts[4].X, Pts[4].Y);
    LineTo(Pts[5].X, Pts[5].Y);
  end;
end;


procedure TcsNotebook.DrawBevelBottom(Pts: TcsTabPoints; Offset: Integer);
begin
  with Canvas do
  begin
    Pen.Color := clBtnHighlight;
    MoveTo(Pts[0].X, Pts[0].Y);
    LineTo(Pts[1].X, Pts[1].Y);
    Pen.Color := clBtnShadow;
    LineTo(Pts[2].X, Pts[2].Y);
  if Offset = 2 then
  begin
    { draw additional diagonal line for correct thickness }
    MoveTo(Pts[1].X, Pts[1].Y - 1);
    LineTo(Pts[2].X + 1, Pts[2].Y);
  end;
  MoveTo(Pts[2].X, Pts[2].Y);
  LineTo(Pts[3].X, Pts[3].Y);
  LineTo(Pts[4].X, Pts[4].Y);
  if Offset = 2 then
  begin
    { draw additional diagonal line for correct thickness }
    MoveTo(Pts[3].X - 1, Pts[3].Y);
    LineTo(Pts[4].X, Pts[4].Y - 1);
  end;
  MoveTo(Pts[4].X, Pts[4].Y);
  LineTo(Pts[5].X, Pts[5].Y);
  end;
end;


procedure TcsNotebook.DrawBevelLeft(Pts: TcsTabPoints; Offset: Integer);
begin
  with Canvas do
  begin
    Pen.Color := clBtnShadow;
    MoveTo(Pts[0].X, Pts[0].Y);
    LineTo(Pts[1].X, Pts[1].Y);
    LineTo(Pts[2].X, Pts[2].Y);
    if Offset = 2 then
    begin
      { draw additional diagonal line for correct thickness }
      MoveTo(Pts[1].X + 1, Pts[1].Y);
      LineTo(Pts[2].X, Pts[2].Y - 1);
    end;
    Pen.Color := clBtnHighlight;
    MoveTo(Pts[2].X, Pts[2].Y);
    LineTo(Pts[3].X, Pts[3].Y);
    LineTo(Pts[4].X, Pts[4].Y);
    if Offset = 2 then
    begin
      { draw additional diagonal line for correct thickness }
      MoveTo(Pts[3].X, Pts[3].Y + 1);
      LineTo(Pts[4].X + 1, Pts[4].Y);
    end;
    MoveTo(Pts[4].X, Pts[4].Y);
    LineTo(Pts[5].X, Pts[5].Y);
  end;
end;


procedure TcsNotebook.DrawBevelRight(Pts: TcsTabPoints; Offset: Integer);
begin
  with Canvas do
  begin
    Pen.Color := clBtnHighlight;
    MoveTo(Pts[0].X, Pts[0].Y);
    LineTo(Pts[1].X, Pts[1].Y);
    Pen.Color := clBtnShadow;
    MoveTo(Pts[1].X, Pts[1].Y);
    LineTo(Pts[2].X, Pts[2].Y);
    if Offset = 2 then
    begin
      { draw additional diagonal line for correct thickness }
      MoveTo(Pts[1].X - 1, Pts[1].Y);
      LineTo(Pts[2].X, Pts[2].Y + 1);
    end;
    MoveTo(Pts[2].X, Pts[2].Y);
    LineTo(Pts[3].X, Pts[3].Y);
    LineTo(Pts[4].X, Pts[4].Y);
    if Offset = 2 then
    begin
      { draw additional diagonal line for correct thickness }
      MoveTo(Pts[3].X, Pts[3].Y - 1);
      LineTo(Pts[4].X - 1, Pts[4].Y);
    end;
    MoveTo(Pts[4].X, Pts[4].Y);
    LineTo(Pts[5].X, Pts[5].Y);
  end;
end;


function TcsNotebook.CalcCardRect(Row: Integer): TRect;
var Rect: TRect;
    l, t, r, b: Integer;
begin
  Rect := CalcFrameRect;
  l := Rect.Left;
  t := Rect.Top;
  r := Rect.Right;
  b := Rect.Bottom;

  case FTabOrientation of
    toTop:
      begin
        Rect.Left := l + (Row*FRowIndent);
        Rect.Top := t - (Row*FTabHeight);
        Rect.Right := r - ((FRowExtent - Row - 1)*FRowIndent);
        Rect.Bottom := b - (Row*FTabHeight);
      end;
    toBottom:
      begin
        Rect.Left := l + (Row*FRowIndent);
        Rect.Top := t + (Row*FTabHeight);
        Rect.Right := r - ((FRowExtent - Row - 1)*FRowIndent);
        Rect.Bottom := b + (Row*FTabHeight);
      end;
    toLeft:
      begin
        Rect.Left := l - (Row*FTabHeight);
        Rect.Top := t + ((FRowExtent - Row - 1)*FRowIndent);
        Rect.Right := r - (Row*FTabHeight);
        Rect.Bottom := b - (Row*FRowIndent);
      end;
    toRight:
      begin
        Rect.Left := l + (Row*FTabHeight);
        Rect.Top := t + (Row*FRowIndent);
        Rect.Right := r + (Row*FTabHeight);
        Rect.Bottom := b - ((FRowExtent - Row - 1)*FRowIndent);
      end;
    end;

  Result := Rect;

end;


{ Calculate the coordinates for the frame component of the window,
   i.e. excluding the index-tabs component.
}
function TcsNotebook.CalcFrameRect: TRect;
var WholeRect: TRect;
    h: Integer;
begin
  WholeRect := CalcWholeRect;
  h := CalcIndexHeight;

  case FTabOrientation of
    toTop:    WholeRect.Top := WholeRect.Top + h;
    toBottom: WholeRect.Bottom := WholeRect.Bottom - h;
    toLeft:   WholeRect.Left := WholeRect.Left + h;
    toRight:  WholeRect.Right := WholeRect.Right - h;
  end;

  Result := WholeRect;

end;


{ Returns an array of points outlining the shape of a
  front card.
  Offset denotes the offset inside the card's border.
  The argument BaseRect will be the rectangle for the card.
}
procedure TcsNotebook.BuildFrontCardPoints(var Pts: Array of TPoint;
         BaseRect: TRect; Offset: Integer);
var l, t, r, b: Integer;
    Rect, TbRect: TRect;
    TabL, TabR: Integer;
begin

  if FAnchoredTabs then
  begin
    { Pts is of size 5 [0..4] }
    Rect := VirtualRect(BaseRect);
    l := Rect.Left;
    t := Rect.Top;
    r := Rect.Right;
    b := Rect.Bottom;

    Pts[0] := point(r - Offset, t + Offset);
    Pts[1] := point(l + Offset, t + Offset);
    Pts[2] := point(l + Offset, b - Offset);
    Pts[3] := point(r - Offset, b - Offset);
    Pts[4] := point(r - Offset, t + Offset - 1);

    MapPoints(Pts, VirtualRect(BaseRect), BaseRect);

  end
  else
  begin
    { Pts is of size 9 [0..8] }
    Rect := VirtualRect(BaseRect);
    l := Rect.Left;
    t := Rect.Top;
    r := Rect.Right;
    b := Rect.Bottom;

    TbRect := TabRect(ActivePage);
    TabL := TbRect.Left;
    TabR := TbRect.Right;

    case FTabOrientation of
      toTop:
        begin
          TabL := TbRect.Left;
          TabR := TbRect.Right;
        end;
      toBottom:
        begin
          TabL := TbRect.Left;
          TabR := TbRect.Right;
        end;
      toLeft:
        begin
          TabL := l + (BaseRect.Bottom - TbRect.Bottom);
          TabR := l + (BaseRect.Bottom - TbRect.Top);
        end;
      toRight:
        begin
          TabL := l + (TbRect.Top - BaseRect.Top);
          TabR := l + (TbRect.Bottom - BaseRect.Top);
        end;
    end;

    Pts[0] := point((r - Offset), t + Offset);
    Pts[1] := point((TabR - Offset), t + Offset);
    Pts[2] := point((TabR - Offset), t - 1);
    Pts[3] := point((TabL + Offset), t);
    Pts[4] := point((TabL + Offset), t + Offset);
    Pts[5] := point((l + Offset), t + Offset);
    Pts[6] := point((l + Offset), b - Offset);
    Pts[7] := point((r - Offset), b - Offset);
    Pts[8] := point((r - Offset), t + Offset);

    MapPoints(Pts, VirtualRect(BaseRect), BaseRect);
  end;
end;


procedure TcsNotebook.DrawFrontCard(Rect: TRect);
var PtsA: Array[0..4] of TPoint;
    Pts: Array[0..8] of TPoint;
    I, Offset: Integer;
begin
  with Canvas do
  begin

    if FAnchoredTabs then
    begin
      BuildFrontCardPoints(PtsA, Rect, 0);
      Pen.Color := clWindowFrame;
      MoveTo(PtsA[0].X, PtsA[0].Y);
      for I := 1 to High(PtsA) do
        LineTo(PtsA[I].X, PtsA[I].Y);

      BuildFrontCardPoints(PtsA, Rect, 1);

      Pen.Color := FrontCardPenColour(0);
      MoveTo(PtsA[0].X, PtsA[0].Y);
      LineTo(PtsA[1].X, PtsA[1].Y);
      Pen.Color := FrontCardPenColour(1);
      LineTo(PtsA[2].X, PtsA[2].Y);
      Pen.Color := FrontCardPenColour(2);
      LineTo(PtsA[3].X, PtsA[3].Y);
      Pen.Color := FrontCardPenColour(3);
      LineTo(PtsA[4].X, PtsA[4].Y);

      BuildFrontCardPoints(PtsA, Rect, 2);

      Pen.Color := FrontCardPenColour(0);
      MoveTo(PtsA[0].X, PtsA[0].Y);
      LineTo(PtsA[1].X, PtsA[1].Y);
      Pen.Color := FrontCardPenColour(1);
      LineTo(PtsA[2].X, PtsA[2].Y);
      Pen.Color := FrontCardPenColour(2);
      LineTo(PtsA[3].X, PtsA[3].Y);
      Pen.Color := FrontCardPenColour(3);
      LineTo(PtsA[4].X, PtsA[4].Y);

    end
    else
    begin
      BuildFrontCardPoints(Pts, Rect, 0);

      Pen.Color := clWindowFrame;
      MoveTo(Pts[0].X, Pts[0].Y);
      LineTo(Pts[1].X, Pts[1].Y);
      LineTo(Pts[2].X, Pts[2].Y);

      for I := 3 to High(Pts) - 1 do
      begin
        MoveTo(Pts[I].X, Pts[I].Y);
        LineTo(Pts[I + 1].X, Pts[I + 1].Y);
      end;

      for Offset := 1 to 2 do
      begin
        BuildFrontCardPoints(Pts, Rect, Offset);

        Pen.Color := FrontCardPenColour(0);
        MoveTo(Pts[0].X, Pts[0].Y);
        LineTo(Pts[1].X, Pts[1].Y);
        Pen.Color := FrontCardPenColour(1);
        LineTo(Pts[2].X, Pts[2].Y);
        { Note: Pts[2] and Pts[3] are not connected. }
        Pen.Color := FrontCardPenColour(3);
        MoveTo(Pts[3].X, Pts[3].Y);
        LineTo(Pts[4].X, Pts[4].Y);

        for I := 4 to High(Pts) - 1 do
        begin
          Pen.Color := FrontCardPenColour(I);
          MoveTo(Pts[I].X, Pts[I].Y);
          LineTo(Pts[I + 1].X, Pts[I + 1].Y);
        end;

      end;
    end;
  end;
end;


{ Return appropriate bevel colour for each side of the
   front card.
}
function TcsNotebook.FrontCardPenColour(Segment: Integer): TColor;
type AMap = array[0..3] of TColor; { map for Anchored tabs/cards }
      Map = array[0..7] of TColor;
const ATopMap: AMap = (clBtnHighlight, clBtnHighlight, clBtnShadow, clBtnShadow);
      ABottomMap: AMap = (clBtnShadow, clBtnHighlight, clBtnHighlight, clBtnShadow);
      ALeftMap: AMap = (clBtnHighlight, clBtnShadow, clBtnShadow, clBtnHighlight);
      ARightMap: AMap = (clBtnShadow, clBtnHighlight, clBtnHighlight, clBtnShadow);
      { Note: positions which contain clBlack are never used, clBlack is
              just used as a place-holder
      }
      TopMap: Map = (clBtnHighlight, clBtnShadow, clBlack, clBtnHighlight,
                      clBtnHighlight, clBtnHighlight, clBtnShadow, clBtnShadow);
      BottomMap: Map = (clBtnShadow, clBtnShadow, clBlack, clBtnHighlight,
                        clBtnShadow, clBtnHighlight, clBtnHighlight, clBtnShadow);
      LeftMap: Map = (clBtnHighlight, clBtnHighlight, clBlack, clBtnShadow,
                      clBtnHighlight, clBtnShadow, clBtnShadow, clBtnHighlight);
      RightMap: Map = (clBtnShadow, clBtnShadow, clBlack, clBtnHighlight,
                        clBtnShadow, clBtnHighlight, clBtnHighlight, clBtnShadow);
var Colour: TColor;
begin
  { the next line of code is only included to prevent the compiler
    warning: "'Colour' might not have been initialized" in Delphi 2
  }
  Colour := clBtnHighlight;

  if FAnchoredTabs then
  begin
    case FTabOrientation of
      toTop:    Colour := ATopMap[Segment];
      toBottom: Colour := ABottomMap[Segment];
      toLeft:   Colour := ALeftMap[Segment];
      toRight:  Colour := ARightMap[Segment];
    end;
  end
  else
  begin
    case FTabOrientation of
      toTop:    Colour := TopMap[Segment];
      toBottom: Colour := BottomMap[Segment];
      toLeft:   Colour := LeftMap[Segment];
      toRight:  Colour := RightMap[Segment];
    end;
  end;

  Result := Colour;

end;


function TcsNotebook.BackCardPenColour(Segment: Integer): TColor;
type Map = array[0..2] of TColor;
const TopMap: Map = (clBtnHighlight, clBtnShadow, clBtnShadow);
      BottomMap: Map = (clBtnShadow, clBtnShadow, clBtnHighlight);
      LeftMap: Map = (clBtnHighlight, clBtnHighlight, clBtnShadow);
      RightMap: Map = (clBtnShadow, clBtnShadow, clBtnHighlight);
var Colour: TColor;
begin
  { the next line of code is only included to prevent the compiler
    warning: "'Colour' might not have been initialized" in Delphi 2
  }
  Colour := clBtnHighlight;

  case FTabOrientation of
    toTop:    Colour := TopMap[Segment];
    toBottom: Colour := BottomMap[Segment];
    toLeft:   Colour := LeftMap[Segment];
    toRight:  Colour := RightMap[Segment];
  end;

  Result := Colour;

end;


procedure TcsNotebook.CalcAndBuild;
var Rect: TRect;
begin
  FRowExtent := CalcRowExtent;
  BuildRowIdentities;
  { reset FPageIndexDefault if now out of range due to add/delete of pages }
  if (FPageIndexDefault < 0) or (FPageIndexDefault >= FPageList.Count) then
    FPageIndexDefault := 0;
  if not FAnchoredTabs then
    MakeFirstRow(ActivePage);
  Rect := CalcPageRect;
  AlignControls(nil, Rect);
end;


procedure TcsNotebook.DrawCards;
var Row: Integer;
    Rect: TRect;
begin
  with Canvas do
  begin
    for Row := 0 to FRowExtent - 1 do
    begin
      Rect := CalcCardRect(Row);
      if Row = 0 then
        { front card }
        DrawFrontCard(Rect)
      else
      begin
        { partially covered (back) card }
        DrawBackCard(Rect, Row);
      end;
    end;
  end;
end;


{ FMaxTabWidth and FCornerSize must satisfy the equation:
      FMaxTabWidth = 0 or >= 40
     FMaxTabWidth >= (FCornerSize*2) + 6
}
procedure TcsNotebook.SetMaxTabWidth(Value: Integer);
var OldMaxTabWidth: Integer;
begin
  OldMaxTabWidth := FMaxTabWidth;
  if Value > 0 then
  begin
    if (Value > 0) and (Value < 15) then
      Value := 15;
    FMaxTabWidth := Value;
    FCornerSize := minInt(FCornerSize, maxInt(1, (FMaxTabWidth div 2) - 6));
  end
  else
    FMaxTabWidth := Value;
  if (FMaxTabWidth <> OldMaxTabWidth) and
    not ((csReading in ComponentState) or (csLoading in ComponentState)) then
    { no need to do CalcAndBuild }
    Invalidate;
end;


{ Corner size must always be:
          >= 1
          < tabHeight
  MaxTabWidth must be:
          >= ((cornerSize)*2) + 6
}
procedure TcsNotebook.SetCornerSize(Value: Integer);
var OldCornerSize: Integer;
begin
  OldCornerSize := FCornerSize;
  if FTabHeight > 0 then
    FCornerSize := maxInt(1, minInt(Value, FTabHeight - 1))
  else
    FCornerSize := maxInt(1, Value);

  { Ensure maxTabWidth is appropriate for new corner size. }
  if FMaxTabWidth > 0 then
    FMaxTabWidth := maxInt(FMaxTabWidth, ((FCornerSize)*2) + 6);

  if (FCornerSize <> OldCornerSize) and
    not ((csReading in ComponentState) or (csLoading in ComponentState)) then
    { no need to do CalcAndBuild }
    Invalidate;
end;


{ Sets the tab height and also ensures that the corner size is appropriate
   for the new tab height.
}
procedure TcsNotebook.SetTabHeight(Value: Integer);
begin
  if (Value <> FTabHeight) then
  begin
    FTabHeight := Value;
    { adjust corner size (if necessary) }
    SetCornerSize(FCornerSize);
    if not ((csReading in ComponentState) or (csLoading in ComponentState)) then
    begin
      CalcAndBuild;
      Invalidate;
    end;
  end;
end;


procedure TcsNotebook.SetAnchoredTabs(Value: Boolean);
begin
  if Value <> FAnchoredTabs then
  begin
    FAnchoredTabs := Value;
    if not ((csReading in ComponentState) or (csLoading in ComponentState)) then
    begin
      CalcAndBuild; { required to make sure 1st tab is in front row }
      Invalidate;
    end;
  end;
end;


procedure TcsNotebook.SetRowIndent(Value: Integer);
begin
  if (Value >= 0) and (Value <> FRowIndent) then
  begin
    FRowIndent := Value;
    if not ((csReading in ComponentState) or (csLoading in ComponentState)) then
    begin
      CalcAndBuild;
      Invalidate;
    end;
  end;
end;


{ Used when row > 0 }
procedure TcsNotebook.DrawBackCard(Rect: TRect; Row: Integer);
var Pts: TList;
    I: Integer;
begin
  Pts := TList.Create;
  try
    BuildBackCardPoints(Pts, Rect, 0, Row);
    with Canvas do
    begin
      Pen.Color := clWindowFrame;
      I := 0;
      while I < Pts.Count do
      begin
        MoveTo(TcsPointObj(Pts.Items[I]).Point.X, TcsPointObj(Pts.Items[I]).Point.Y);
        LineTo(TcsPointObj(Pts.Items[I + 1]).Point.X, TcsPointObj(Pts.Items[I + 1]).Point.Y);
        I := I + 2;
      end;

      BuildBackCardPoints(Pts, Rect, 1, Row);

      I := 0;
      while I < Pts.Count do
      begin
        if I < (FColExtent*2) + 2 then
          Pen.Color := backCardPenColour(0)
        else
          if I < Pts.Count - 2 then
            Pen.Color := backCardPenColour(1)
          else
            Pen.Color := backCardPenColour(2);
        MoveTo(TcsPointObj(Pts.Items[I]).Point.X, TcsPointObj(Pts.Items[I]).Point.Y);
        LineTo(TcsPointObj(Pts.Items[I + 1]).Point.X, TcsPointObj(Pts.Items[I + 1]).Point.Y);
        I := I + 2;
      end;
    end;

  finally
    for I := 0 to Pts.Count - 1 do
      TcsPointObj(Pts.Items[I]).Free;
    Pts.Free;
  end;

end;


function TcsNotebook.RotateTabRect(BaseRect, TabRect, VirtualRect: TRect): TRect;
begin
  if FTabOrientation = toLeft then
  begin
    TabRect.Left := VirtualRect.Left + BaseRect.Bottom - TabRect.Bottom;
    TabRect.Right := VirtualRect.Left + BaseRect.Bottom - TabRect.Top;
  end
  else
    if FTabOrientation = toRight then
    begin
      TabRect.Left := VirtualRect.Left + TabRect.Top - BaseRect.Top;
      TabRect.Right := VirtualRect.Left + TabRect.Bottom - BaseRect.Top;
    end;
  Result := TabRect;
end;


{ Builds an array of points which describe the lines outlining
   the shape of a back-card (a back-card is any card other than the
   front card), i.e. row will be in the range 1..rowExtent - 1
   Offset denotes the offset inside the card's border.
}
procedure TcsNotebook.BuildBackCardPoints(var Pts: TList; BaseRect: TRect; Offset, Row: Integer);
var VRect, TbRect: TRect;
    l, t, r, b, Col, I, Adjust: Integer;
    ColHasTab: Boolean;
    ptObj: TcsPointObj;
    ptArray: Array[0..0] of TPoint;
begin
  VRect := VirtualRect(BaseRect);

  l := VRect.Left;
  t := VRect.Top + CardDescent;
  r := VRect.Right;
  b := VRect.Bottom;

  for I := 0 to (FColExtent*2) + 5 do
  begin
    ptObj := TcsPointObj.Create(Point(0, 0));
    Pts.Add(ptObj);
  end;

  { determine points along top of card }
  Col := 0;
  I := 0;

  while Col < FColExtent do
  begin
    { Note: There can only ever be one (if any) row with less than
            colExtent columns.  Because of this the other rows are
            guaranteed to have tabs in all their columns; therefore
            tabIdentities[row - 1][col] will _not_ be nil if
            tabIdentities[row][col] is nil.
    }
    ColHasTab := HasTab(Row, Col);
    if ColHasTab then
      TbRect := TabRect(TStringList(FRowIdentities.Items[Row])[Col])
    else
      TbRect := TabRect(TStringList(FRowIdentities.Items[Row - 1])[Col]);

    TbRect := RotateTabRect(BaseRect, TbRect, VRect);
    if not ColHasTab then
      { move across to position of row above (top & bottom not used so
         no need to change them)
      }
      OffsetRect(TbRect, FRowIndent, 0);

    if Col = 0 then
    begin
      if (FCornerSize > CardDescent) and
         (FRowIndent <= (FCornerSize - CardDescent)) then
      begin
        TcsPointObj(Pts[I]).Point := point(TbRect.Left + Offset, t + Offset);
        TcsPointObj(Pts[I + 1]).Point := point(TbRect.Left + maxInt(Offset, FCornerSize - CardDescent - FRowIndent - Offset),
                                               t + Offset);
      end
    end
    else
    begin
      if HasTab(Row - 1, Col) then
      begin
        { there is a tab in the same column in the row below }
        if FCornerSize > CardDescent then
        begin
          TcsPointObj(Pts[I]).Point := point(TbRect.Left - FRowIndent - maxInt(0, FCornerSize - CardDescent - Offset - 1),
                                             t + Offset);
          TcsPointObj(Pts[I + 1]).Point := point(TbRect.Left - FRowIndent + maxInt(0, FCornerSize - CardDescent - Offset),
                                           t + Offset);
        end
      end
      else
      begin
        { there is no tab in the same column in the row below,
           extend line across
        }
        if FCornerSize <= CardDescent then
          Adjust := -1
        else
          Adjust := FCornerSize - CardDescent - Offset - 1;
        TcsPointObj(Pts[I]).Point := point(TbRect.Left - FRowIndent - Adjust,
                                           t + Offset);
        TcsPointObj(Pts[I + 1]).Point := point(TbRect.Right - Offset, t + Offset);
      end;
    end;

    Col := Col + 1;
    I := I + 2;
  end;

  { Now add points for the right hand side of the card. }
  AddBackCardRHPoints(Pts, Rect(l, t, r, b), Offset);

  { map each point in the list }
  for I := 0 to Pts.Count - 1 do
  begin
    ptArray[0] := TcsPointObj(Pts.Items[I]).Point;
    MapPoints(ptArray, VRect, BaseRect);
    TcsPointObj(Pts.Items[I]).Point := ptArray[0];
  end;
end;


{ Add points for the right-hand side (with respect to top orientation)
   of the back-card.
}
procedure TcsNotebook.AddBackCardRHPoints(var Pts: TList; CardRect: TRect; Offset: Integer);
var t, r, b, w, I, ExtraLength, RMinusOffset: Integer;
    IRect, VCIRect: TRect;
    Adjust1, Adjust2: Integer;
begin
  t := CardRect.Top;
  r := CardRect.Right;
  b := CardRect.Bottom;
  w := CardRect.Right - CardRect.Left;
  IRect := CalcIndexRect;
  VCIRect := VirtualRect(IRect);

  { this is the extra line length appropriate when maxTabWidth is in effect }
  ExtraLength := maxInt(0, w - ((VCIRect.Right - VCIRect.Left) -
                  ((FRowExtent - 1)*FRowIndent)));

  { Pts[0..i-1] are used for lines (2 points per line) between each tab. }
  I := (FColExtent*2);
  RMinusOffset := (r - Offset);

  if (FRowIndent < 2) and (FCornerSize <= CardDescent + 2) and (ExtraLength = 0) then
  begin
    if Offset = 0 then
    begin
      TcsPointObj(Pts.Items[I]).Point := point(r - FRowIndent, t);
      TcsPointObj(Pts.Items[I + 1]).Point := point(r - 1, t);
    end
    else
    begin
      TcsPointObj(Pts.Items[I]).Point := Point(0, 0);
      TcsPointObj(Pts.Items[I + 1]).Point := Point(0, 0);
    end;
  end
  else
  begin
    if FCornerSize > CardDescent then
      Adjust1 := Offset + 2
    else
      Adjust1 := Offset + 1;
    if ExtraLength = 0 then
      Adjust2 := 0
    else
    begin
      Adjust2 := 1;
    end;
    TcsPointObj(Pts.Items[I]).Point := point(r - FRowIndent - maxInt(0, FCornerSize - CardDescent) +
                     Adjust1 - (ExtraLength + Adjust2),
                    t + Offset);
    TcsPointObj(Pts.Items[I + 1]).Point := point(RMinusOffset, t + Offset);
  end;

  TcsPointObj(Pts.Items[I + 2]).Point := point(RMinusOffset, t + Offset);
  TcsPointObj(Pts.Items[I + 3]).Point := point(RMinusOffset, b - Offset);
  TcsPointObj(Pts.Items[I + 4]).Point := point(RMinusOffset, b - Offset);
  TcsPointObj(Pts.Items[I + 5]).Point := point(r - FRowIndent, b - Offset);

end;


{ Select the tab on which the mouse pointer was clicked. }
procedure TcsNotebook.MouseDown(Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
var Pt: TPoint;
    I: Integer;
    TabIdentity: String;
    Found: Boolean;
    Rect: TRect;
begin
  { In design-mode MouseDown will be called after CMDesignHitTest
    which will have already taken care of selecting the tab.
  }
  if (csDesigning in ComponentState) then Exit;

  inherited MouseDown(Button, Shift, X, Y);

  if (Button = mbLeft) and Enabled then
  begin
    Pt := Point(X, Y);
    I := 0;
    Found := False;

    while (I < FAccess.Count) and not Found do
    begin
      TabIdentity := FAccess[I];
      if TcsPage(FPageList[I]).PageVisible then
      begin
        Rect := TabRect(TabIdentity);
        if PtInRect(Rect, Pt) then
          Found := True;
      end;
      if not Found then
        Inc(I);
    end;

    if Found and TcsPage(FPageList.Items[I]).PageEnabled then
    begin
      SetFocus; { to self }
      if not Focused then Exit; { OnExit event handlers redirected focus }
      SetActivePage(TabIdentity);
      if (ActivePage = TabIdentity) then
        { page change was allowed }
        TabClick;
    end;

  end;
end;


{ Arranges FRowIdentities so that the row which contains
  the specified index-tab identity is made the first row.
  Note that the order (if you imagine the array as a
  continuous loop) of the rows isn't changed,
  just which is first.
}
procedure TcsNotebook.MakeFirstRow(TabIdentity: String);
var Row, Tally, Index, I: Integer;
    NewRowIdents: TList;
begin
  Row := FindTabRow(TabIdentity);
  if Row > 0 then
  begin
    NewRowIdents := TList.Create;
    try
      Tally := 0;
      Index := Row;

      while Tally < FRowExtent do
      begin
        NewRowIdents.Add(FRowIdentities.Items[Index]);
        Tally := Tally + 1;
        Index := Index + 1;
        if Index > FRowExtent - 1 then
          { go back to beginning }
          Index := 0;
      end;

      for I := 0 to NewRowIdents.Count - 1 do
        FRowIdentities.Items[I] := NewRowIdents.Items[I];

    finally
      NewRowIdents.Free;
    end;
  end;

end;


{ Return the area within which text can be drawn on the tab-button. }
function TcsNotebook.CalcTabTextRect(TabIdentity: String): TRect;
var TbRect: TRect;
    l, t, r, b, Margin: Integer;
begin
  TbRect := TabRect(TabIdentity);
  Margin := maxInt(3, FCornerSize);

  l := TbRect.Left + iifInt(FTabOrientation = toRight, 1, Margin);
  t := TbRect.Top + iifInt(FTabOrientation = toBottom, 1, Margin);
  r := TbRect.Right - iifInt(FTabOrientation = toLeft, 1, Margin);
  b := TbRect.Bottom - iifInt(FTabOrientation = toTop, 1, Margin);

  Result := Rect(l, t, r, b);
  InflateRect(Result, -1, -1);
end;


{ Wrapper for TextOut().
  This method is used to overcome the limitation of TextOut()
  (and ExtTextOut()) not automatically underlining the character prefixed
  by an ampersand in the string.
  Draws the text using the specified device context. If the text contains
  an ampersand (&) the following character will be underlined.
  If the text contains two successive ampersands (&&) then a single
  ampersand will be output.
}
procedure TcsNotebook.DoTextOut(x, y: Integer; Str: String; DC: HDC; PlainFont, UnderlinedFont: HFont);
var P1, P2: Integer;
    Part: Array[0..2] of String;
    Extent, I: Integer;
    PText: Array[0..255] of Char;
    NewFont, OldFont: HFont;
begin
  P1 := Pos('&', Str);
  if (P1 > 0) then
  begin
    P2 := Pos('&', Copy(Str, P1 + 1, Length(Str)));
    if (P2 = 1) then Inc(P2, P1)
    else P2 := 0;
  end
  else P2 := 0;
  if (P1 > 0) and (P2 <> P1 + 1) then
  begin
    Part[0] := Copy(Str, 1, P1 - 1);
    Part[1] := Copy(Str, P1 + 1, 1);
    Part[2] := Copy(Str, P1 + 2, Length(Str));
  end
  else if (P1 > 0) and (P2 = P1 + 1) then { '&&' in string }
  begin
    Part[0] := Copy(Str, 1, P1);
    Part[1] := '';
    Part[2] := Copy(Str, P2 + 1, Length(Str));
  end
  else
  begin
    Part[0] := Str;
    Part[1] := '';
    Part[2] := '';
  end;

  for I := 0 to 2 do
  begin

    { get font handle for this part of the text }
    if (I = 1) then
      { underlined part of text }
      NewFont := UnderlinedFont
    else
      NewFont := PlainFont;
    OldFont := SelectObject(DC, NewFont);

    { prevent BoundsChecker from reporting false error when empty string }
    if Part[I] <> '' then
       StrPCopy(PText, Part[I])
    else
        PText[0] := #0; { array subscript needed for Delphi 1 }
    TextOut(DC, x, y, PText, StrLen(PText));

    { determine extent of part just output; needed to update position }
    Extent := CalcTextExtent(Part[I], DC);

    if (not FSidewaysText) or (FTabOrientation = toTop) or (FTabOrientation = toBottom) then
      x := x + Extent
    else
      case FTabOrientation of
        toLeft:   y := y - Extent;
        toRight:  y := y + Extent;
      end;

    SelectObject(DC, OldFont);
  end;

end;


procedure TcsNotebook.SetTextAlignment(Value: TTextAlignment);
begin
  if Value <> FTextAlignment then
  begin
    FTextAlignment := Value;
    if not ((csReading in ComponentState) or (csLoading in ComponentState)) then
      InvalidateTabTextRects(trAll);
  end;
end;


procedure TcsNotebook.SetBitmapAlignment(Value: TBitmapAlignment);
begin
  if Value <> FBitmapAlignment then
  begin
    FBitmapAlignment := Value;
    if not ((csReading in ComponentState) or (csLoading in ComponentState)) then
      InvalidateTabTextRects(trAll);
  end;
end;


procedure TcsNotebook.SetSidewaysText(Value: Boolean);
begin
  if (Value <> FSidewaysText) then
  begin
    if ((csLoading in ComponentState) or (csReading in ComponentState)) then
      FSidewaysText := Value
    else
      if not (Value and not (FTabOrientation in [toLeft, toRight])) then
      begin
        FSidewaysText := Value;
        InvalidateTabTextRects(trAll);
      end;
  end;
end;


function TcsNotebook.HasTab(Row, Col: Integer): Boolean;
begin
  Result := False;
  if Row < FRowIdentities.Count then
    if Col < TStringList(FRowIdentities.Items[Row]).Count then
      Result := True;
end;


{ Calculates the extent of the string; ignores '&' character if present
  (and not '&&') when calculating the actual extent of the string.
}
function TcsNotebook.CalcTextExtent(Str: String; DC: hDC): Integer;
var P1, P2: Integer;
    PlainText: String;
    PText: Array[0..255] of Char;
    Extent: TSize;
begin
  P1 := Pos('&', Str);
  if (P1 = Length(Str)) then
  begin
    { If the string contains a '&' as the last character it will
      usually be because DoTextOut is determining the length of the
      first part of a caption which contains '&&' indicating a
      literal '&' is to actually be displayed.  In this case
      no underlining will occur so the length of the whole
      string (including the '&') should be calculated.
    }
    P1 := 0;
    P2 := 0;
  end
  else if (P1 > 0) then
  begin
    P2 := Pos('&', Copy(Str, P1 + 1, Length(Str)));
    if (P2 = 1) then Inc(P2, P1)
    else P2 := 0;
  end
  else P2 := 0;
  { check for special case of trailing single '&' which will occur
    when getting the length of the first part of a string which
    contained '&&', e.g. 'A&&A'
  }
  if (P1 > 0) and (P2 <> P1 + 1) then
    PlainText := Copy(Str, 1, P1 - 1) + Copy(Str, P1 + 1, Length(Str))
  else if (P1 > 0) and (P2 = P1 + 1) then
    PlainText := Copy(Str, 1, P1) + Copy(Str, P2 + 1, Length(Str))
  else
    PlainText := Str;
  { prevent BoundsChecker from reporting false error when empty string }
  if PlainText <> '' then
    StrPCopy(PText, PlainText)
  else
    PText[0] := #0; { array subscript needed for Delphi 1 }
  StrPCopy(PText, PlainText);
{$IFDEF WIN32}
  if ((Win32Platform = VER_PLATFORM_WIN32s) and
      GetTextExtentPoint(DC, PText, StrLen(PText), Extent)) or
    GetTextExtentPoint32(DC, PText, StrLen(PText), Extent) then
{$ELSE}
  if GetTextExtentPoint(DC, PText, StrLen(PText), Extent) then
{$ENDIF}
    Result := Extent.cX
  else
    Result := 0;
end;


{ The argument str is assumed to have already been clipped so that
  it is <= the width (in logical units, NOT no. of characters) of
  the specified rectangle.
}
function TcsNotebook.CalcTextOrigin(Str: String; BaseRect: TRect; DC: HDC): TPoint;
var Metrics: TTextMetric;
    SourceRect: TRect;
    Align: Integer;
    x, y, TextHeight, TextWidth: Integer;
    ptArray: Array[0..0] of TPoint;
begin
  if FSidewaysText and ((FTabOrientation = toLeft) or (FTabOrientation = toRight)) then
    SourceRect := VirtualRect(BaseRect)
  else
    SourceRect := BaseRect;

  TextWidth := CalcTextExtent(Str, DC);

  GetTextMetrics(DC, Metrics);
  TextHeight := Metrics.tmHeight + Metrics.tmExternalLeading;
  Align := TA_TOP + TA_LEFT;

  { X origin }
  case FTextAlignment of
    taLeftTop, taLeftMiddle, taLeftBottom:
      x := SourceRect.Left;
    taCentreTop, taCenterTop, taCentreMiddle, taCenterMiddle,
    taCentreBottom, taCenterBottom:
      x := (SourceRect.Left + SourceRect.Right - TextWidth) div 2;
    taRightTop, taRightMiddle, taRightBottom:
      x := SourceRect.Right - TextWidth;
  else
      x := SourceRect.Left;
  end;

  { Y origin }
  case FTextAlignment of
    taLeftTop, taCentreTop, taCenterTop, taRightTop:
      y := SourceRect.Top;
    taLeftMiddle, taCentreMiddle, taCenterMiddle, taRightMiddle:
      y := (SourceRect.Top + SourceRect.Bottom - TextHeight) div 2;
    taLeftBottom, taCentreBottom, taCenterBottom, taRightBottom:
    begin
      y := SourceRect.Bottom;
      Align := TA_BOTTOM;
    end;
  else
      y := SourceRect.Top;
  end;

  SetTextAlign(DC, Align);

  { Don't use MapPoints() for toBottom orientation; unlike the drawing
    of the bottom tabs (which are upside down) we do not need to map
    the text origin, it is still right way up.
  }
  if FSidewaysText and ((FTabOrientation = toLeft) or (FTabOrientation = toRight)) then
  begin
    ptArray[0] := Point(x, y);
    MapPoints(ptArray, SourceRect, BaseRect);
    Result := ptArray[0];
  end
  else
    Result := Point(x, y);

end;


{ The bitmap's x alignment can be left, centre or right.
  The bitmap's y alignment can be top, middle or bottom.
  bitmapAlignment is specified as a x-y combination,
  e.g. baLeftTop, and always has x alignment first.
  The alignment is relative to the button's orientation and thus
  may need swapping around to get the true x-y alignment.
}
function TcsNotebook.CalcBitmapRect(TabIdentity: String): TRect;
type  TXAlign = (xaLeft, xaCentre, xaRight);
      TYAlign = (yaTop, yaMiddle, yaBottom);
var xAlign: TXAlign;
    yAlign: TYAlign;
    TabRect: TRect;
    x, y, w, h: Integer;
    pt: TPoint;
begin
  { the next two lines of code are only included to prevent the compiler
    warning that x & y "might not have been initialized" in Delphi 2
  }
  xAlign := xaLeft;
  yAlign := yaTop;

  case FBitmapAlignment of
    baLeftTop:
      begin
        if FTabOrientation = toRight then xAlign := xaRight else xAlign := xaLeft;
        if FTabOrientation = toLeft then yAlign := yaBottom else yAlign := yaTop;
      end;
    baCentreTop, baCenterTop:
      begin
        case FTabOrientation of
          toTop, toBottom: xAlign := xaCentre;
          toLeft: xAlign := xaLeft;
          toRight: xAlign := xaRight;
        end;
        if (FTabOrientation in [toTop, toBottom]) then
          yAlign := yaTop
        else
          yAlign := yaMiddle;
      end;
    baRightTop:
      begin
       if FTabOrientation = toLeft then xAlign := xaLeft else xAlign := xaRight;
       if FTabOrientation = toRight then yAlign := yaBottom else yAlign := yaTop;
      end;
    baLeftMiddle:
      begin
        if (FTabOrientation in [toTop, toBottom]) then
          xAlign := xaLeft
        else
          xAlign := xaCentre;
        case FTabOrientation of
          toTop, toBottom:  yAlign := yaMiddle;
          toLeft: yAlign := yaBottom;
          toRight: yAlign := yaTop;
        end;
      end;
    baCentreMiddle, baCenterMiddle:
      begin
        xAlign := xaCentre;
        yAlign := yaMiddle;
      end;
    baRightMiddle:
      begin
        if (FTabOrientation in [toTop, toBottom]) then
          xAlign := xaRight
        else
          xAlign := xaCentre;
        case FTabOrientation of
          toTop, toBottom: yAlign := yaMiddle;
          toLeft: yAlign := yaTop;
          toRight: yAlign := yaBottom;
        end;
      end;
    baLeftBottom:
      begin
        if FTabOrientation = toLeft then xAlign := xaRight else xAlign := xaLeft;
        if FTabOrientation = toRight then yAlign := yaTop else yAlign := yaBottom;
      end;
    baCentreBottom, baCenterBottom:
      begin
        case FTabOrientation of
          toTop, toBottom: xAlign := xaCentre;
          toLeft: xAlign := xaRight;
          toRight: xAlign := xaLeft;
        end;
        if (FTabOrientation in [toTop, toBottom]) then
          yAlign := yaBottom
        else
          yAlign := yaMiddle;
      end;
    baRightBottom:
      begin
        if FTabOrientation = toRight then xAlign := xaLeft else xAlign := xaRight;
        if FTabOrientation = toLeft then yAlign := yaTop else yAlign := yaBottom;
      end;
  else
      begin
        xAlign := xaLeft;
        yAlign := yaTop;
      end;
  end;

  TabRect := CalcTabTextRect(TabIdentity);
  if FBitmapAlignment = baFit then
  begin
    w := TabRect.Right - TabRect.Left;
    h := TabRect.Bottom - TabRect.Top;
  end
  else
  begin
    pt := CalcBitmapSize(TabIdentity);
    w := minInt(pt.X, TabRect.Right - TabRect.Left);
    h := minInt(pt.Y, TabRect.Bottom - TabRect.Top);
  end;

  { the next two lines of code are only included to prevent the compiler
    warning that x & y "might not have been initialized" in Delphi 2
  }
  x := 0;
  y := 0;

  case xAlign of
    xaLeft: x := TabRect.Left;
    xaCentre: x := TabRect.Left + ((TabRect.Right - TabRect.Left) div 2) - (w div 2);
    xaRight: x := TabRect.Right - w + 1;
  end;

  case yAlign of
    yaTop: y := TabRect.Top;
    yaMiddle: y := TabRect.Top + ((TabRect.Bottom - TabRect.Top) div 2) - (h div 2);
    yaBottom: y := TabRect.Bottom - h + 1;
  end;

  Result := Rect(x, y, x + w - 1, y + h - 1);

end;


{ Return size of bitmap as a point where: x = Width, y = Height. }
function TcsNotebook.CalcBitmapSize(TabIdentity: String): TPoint;
var Index, W, H: Integer;
    Bitmap: TBitmap;
begin
  W := 0;
  H := 0;
  Index := FAccess.IndexOf(TabIdentity);
  if (Index >= 0) then
  begin
    Bitmap := TcsPage(FPageList.Items[Index]).Bitmap;
    if (Bitmap <> nil) and (Bitmap.Handle <> 0) then
    begin
      W := Bitmap.Width div TabNumGlyphs[Index];
      H := Bitmap.Height;
    end;
  end;
  Result := Point(W, H);
end;


procedure TcsNotebook.DrawTabFocusRect;
var TabIdentity: String;
    Rect: TRect;
begin
  if FHasLoaded then
  begin
    TabIdentity := ActivePage;
    Rect := CalcTabTextRect(TabIdentity);
    InflateRect(Rect, 1, 1);
    with Canvas do
      DrawFocusRect(Rect);
  end;
end;


procedure TcsNotebook.SetTabFont(Value: TFont);
begin
  FTabFont.Assign(Value);
end;


procedure TcsNotebook.SetParentTabFont(Value: Boolean);
begin
  if (Value <> FParentTabFont) then
  begin
    FParentTabFont := Value;
    if Parent <> nil then Perform(CM_PARENTFONTCHANGED, 0, 0);
  end;
end;


procedure TcsNotebook.TabFontChanged(Sender: TObject);
begin
  InvalidateTabTextRects(trAll);
  FParentTabFont := False;
  Perform(CM_TABFONTCHANGED, 0, 0);
end;


procedure TcsNotebook.CMTabFontChanged(var Message: TMessage);
begin
  inherited;
  if HandleAllocated then Perform(WM_SETFONT, FTabFont.Handle, 0);
end;


procedure TcsNotebook.CMParentFontChanged(var Message: TMessage);
begin
  inherited;
  if FParentTabFont then
  begin
    { notebook's parent can be a form, panel or scroll box }
    if (Parent <> nil) then
      { typecast as panel to allow access to the font property }
      SetTabFont(TPanel(Parent).Font);
    FParentTabFont := True;
  end;
end;


procedure TcsNotebook.CMFocusChanged(var Message: TCMFocusChanged);
begin
  if AnyPagesVisible then
  begin
    if Message.Sender = Self then
    begin
      if not FHasFocusRect then
      begin
        DrawTabFocusRect;
        FHasFocusRect := True;
      end;
    end
    else
    begin
      if FHasFocusRect then
      begin
        { undraw (XOR) }
        DrawTabFocusRect;
        FHasFocusRect := False;
      end;
    end;
  end;
  inherited;
end;


{ Draw the Tab's focus rect and perform user-assigned OnClick event method }
procedure TcsNotebook.TabClick;
begin
  { Focus may have been changed from self to something else
    during processing of the OnPageChanged event handler,
    for example to set focus to a specific control on the
    new page; only draw focus rect if self still has focus.
  }
  Update; { so that Paint has been performed and FHasFocusRect is correct }
  if Focused and not FHasFocusRect then
  begin
    DrawTabFocusRect;
    FHasFocusRect := True;
  end;
  if Assigned(FOnTabClick) then FOnTabClick(Self); { do user-assigned event method }
end;


{ Handles processing needed for OnExit event. }
procedure TcsNotebook.DoExit;
begin
  inherited DoExit;
  if FHasFocusRect then
  begin
    DrawTabFocusRect; { undraw }
    FHasFocusRect := False;
  end;
end;


procedure TcsNotebook.ClearTabIndex;
var I: Integer;
begin
  for I := 0 to FTabIndex.Count - 1 do
    TcsTab(FTabIndex.Objects[I]).Free;
  FTabIndex.Clear;
end;


procedure TcsNotebook.CMDialogChar(var Message: TCMDialogChar);
var I: Integer;
    TabIdentity: String;
begin
  if CanFocus then
    with Message do
    begin
      for I := 0 to FPageList.Count - 1 do
      begin
        TabIdentity := FAccess[I];
        if IsAccel(CharCode, TabIdentity) and
          TcsPage(FPageList[I]).PageEnabled and
          TcsPage(FPageList[I]).PageVisible then
        begin { select appropriate Tab and give it focus }
          Result := 1;  { accelerator key is valid (don't beep) }
          SetFocus; { to self }
          if not Focused then Exit; { OnExit event handlers redirected focus }
          SetActivePage(TabIdentity);
          if (ActivePage = TabIdentity) then
            { page change was allowed }
            TabClick;
          Exit;
        end;
      end;
    end;
  inherited;
end;


procedure TcsNotebook.WMGetDlgCode(var Message: TWMGetDlgCode);
begin
  Message.Result := DLGC_WANTARROWS;
end;


procedure TcsNotebook.KeyDown(var Key: Word; Shift: TShiftState);
var NewPageIndex: Integer;
begin
  case Key of
    VK_RIGHT, VK_DOWN, VK_LEFT, VK_UP:
    begin
      NewPageIndex := FPageIndex;
      repeat
        if ((Key = VK_RIGHT) or (Key = VK_DOWN)) then
          { determine next page }
          if (NewPageIndex = FPageList.Count - 1) then
            NewPageIndex := 0
          else
            NewPageIndex := NewPageIndex + 1
        else
          { determine previous page }
          if (NewPageIndex = 0) then
            NewPageIndex := FPageList.Count - 1
          else
            NewPageIndex := NewPageIndex - 1;
      { continue until we find a select-able page or
        are back where we started
      }
      until ((TcsPage(FPageList[NewPageIndex]).PageEnabled and
              TcsPage(FPageList[NewPageIndex]).PageVisible) or
             (NewPageIndex = FPageIndex));

      if (NewPageIndex <> FPageIndex) then
      begin
        SetPageIndex(NewPageIndex);
        if (PageIndex = NewPageIndex) then
          { page change was allowed }
          TabClick;
      end;
    end;
  end;
end;


procedure TcsNotebook.SetSaveResources(Value: Boolean);
var I: Integer;
begin
  if (Value <> FSaveResources) then
  begin
    FSaveResources := Value;
    if FSaveResources and not (csDesigning in ComponentState) then
      { release existing resources for all pages which are not visible }
      for I := 0 to FPageList.Count - 1 do
      begin
        if (I <> FPageIndex) then
          TcsPage(FPageList[I]).DestroyHandle;
      end;
  end;
end;


{ Reconfigure allows TcsPageAccess to rebuild the Notebook tabs
  after making changes, e.g. adding, editing, deleting of tabs.
  It is intended for use by TcsPageAccess in design-mode.
}
procedure TcsNotebook.Reconfigure;
begin
  if FHasLoaded then
  begin
    ClearTabIndex;
    CalcAndBuild;
    { It is necessary to explicitly do SetupTabs (normally done by
      WMEraseBkgnd) because if the notebook is obscured (by the
      Notebook Tabs editor dialog box) then no WM_ERASEBKGND message
      is generated and thus SetupTabs won't have been performed
      again before the Paint method is called.
    }
    SetupTabs;
    Invalidate;
  end;
end;


procedure TcsNotebook.SetSelectedColour(Value: TColor);
begin
  if (Value <> FSelectedColour) then
  begin
    FSelectedColour := Value;
    InvalidateTabTextRects(trSelected);
  end;
end;


procedure TcsNotebook.SetUnselectedColour(Value: TColor);
begin
  if (Value <> FUnselectedColour) then
  begin
    FUnselectedColour := Value;
    InvalidateTabTextRects(trUnselected);
  end;
end;


{ Note: The inherited CMParentColorChanged method will only invalidate
        when ParentColor is true which is not sufficient for the notebook
        because it can have two different areas of (background) colour
        when ParentColor is false.
}
procedure TcsNotebook.CMParentColorChanged(var Message: TMessage);
begin
  if ParentColor then
    inherited { update colour }
  else
    { area outside tabs/cards needs redrawing }
    Invalidate;
end;


{ Erase the background of the notebook and surrounding areas using
  the appropriate colours (based on ParentColor) for each area.
}
procedure TcsNotebook.WMEraseBkgnd(var Message: TWMEraseBkgnd);
var Rect, TbRect: TRect;
    Pts: TcsTabPoints;
    I, Row, Col: Integer;
    ColourOfParent: TColor;
    TabIdentity: String;
    BigRegion, TabRegion, CardRegion, DiffRegion: HRGN;
    FrontRegion: HRGN;
    BkgndHandled: Boolean;
    OldRgn: HRGN;
    ACanvas: TCanvas;
    BrushHandle: HBRUSH;

  { Calculate the area occupied by a (back) card which needs to be erased
    when redrawing.  Necessary due to Tab Descent; back cards are not as
    tall as the front card.
  }
  function CalcCardEraseRect(Row: Integer): TRect;
  begin
    Result := CalcCardRect(Row);
    if (Row > 0) then
      case FTabOrientation of
        toTop:    Result.Top := minInt(Result.Top + CardDescent, Result.Bottom);
        toBottom: Result.Bottom := maxInt(Result.Bottom - CardDescent, Result.Top);
        toLeft:   Result.Left := minInt(Result.Left + CardDescent, Result.Right);
        toRight:  Result.Right := maxInt(Result.Right - CardDescent, Result.Left);
      end;
  end;

  { Extend the first and last segments of the tab frames (corresponding
    to left and right hand edges if viewing in top orientation) to
    eliminate gaps (due to corner cut-offs and tab descents)
    when erasing the background.
  }
  procedure ExtendTabPoints(var Pts: TcsTabPoints);
  var CardRect: TRect;
  begin
    CardRect := CalcCardRect(0);
    case FTabOrientation of
      toTop:
        begin
          Pts[0].Y := minInt(Pts[0].Y + CardDescent + 1, CardRect.Top);
          Pts[5].Y := Pts[0].Y;
        end;
      toBottom:
        begin
          Pts[0].Y := maxInt(Pts[0].Y - CardDescent - 1, CardRect.Bottom);
          Pts[5].Y := Pts[0].Y;
        end;
      toLeft:
        begin
          Pts[0].X := minInt(Pts[0].X + CardDescent + 1, CardRect.Left);
          Pts[5].X := Pts[0].X;
        end;
      toRight:
        begin
          Pts[0].X := maxInt(Pts[0].X - CardDescent - 1, CardRect.Right);
          Pts[5].X := Pts[0].X;
        end;
    end;
  end; {ExtendTabPoints }

begin { WMEraseBkgnd }
  if not FHasLoaded then
  begin
    { Can't rely on Loaded (only) to set FHasLoaded because when a new
      component is added (from palette) or pasted to a form at design-time
      or created dynamically (in source code) no Loaded event will
      be generated.
    }
    CalcAndBuild; { needed especially for pasted component }
    FHasLoaded := True;
  end;
  SetupTabs;  { so that correct/current sizes are used }

  InitPalette(Message.DC);
  { notebook's parent can be a form, panel or scroll box }
  if (Parent <> nil) then
    { typecast as panel to allow access to the colour property }
    ColourOfParent := TPanel(Parent).Color
  else
    ColourOfParent := Brush.Color;

  if ParentColor and not FUseUnselectedTabColour then
  begin
    { notebook and parent are same colour; can just fill client area }
    BrushHandle := CreateSolidBrush(ColorToRGB(ColourOfParent));
    try
      FillRect(Message.DC, ClientRect, BrushHandle);
    finally
      DeleteObject(BrushHandle);
    end;
  end
  else
  begin
    { Notebook is different colour to parent:
      The area to erase (and the colour used to fill the area) is
      calculated by creating regions for the area occupied by the cards
      and tabs and combining them into one region.  The area outside this
      region but inside the notebook's client rect is then erased
      separately (to prevent 'flicker' when selecting tabs).
    }

    { include the area occupied by the front card; this area overlaps
      with the area handled by the page itself but the gap between the
      card and the page (the page is smaller than the card) must be
      included to ensure all the 'gaps' are filled
    }
    Rect := CalcCardRect(0);
    BigRegion := CreateRectRgnIndirect(Rect);

    for I := 0 to FTabIndex.Count - 1 do
    begin
      TabIdentity := FTabIndex[I];
      Row := FindTabRow(TabIdentity);
      Col := FindTabCol(TabIdentity);
      TbRect := TabRect(TabIdentity);
      BuildTabPoints(Pts, TbRect, 0, Row, Col);
      ExtendTabPoints(Pts);
      TabRegion := CreatePolygonRgn(Pts, Integer(High(Pts) + 1), PolyFillMode);
      CombineRgn(BigRegion, BigRegion, TabRegion, Rgn_Or);
      DeleteObject(TabRegion);
    end;

    { now do back card regions (if any) }
    for I := 1 to FRowExtent - 1 do
    begin
      Rect := CalcCardEraseRect(I);
      CardRegion := CreateRectRgnIndirect(Rect);
      CombineRgn(BigRegion, BigRegion, CardRegion, Rgn_Or);
      DeleteObject(CardRegion);
    end;

    { Calculate region outside tabs/cards before changing BigRegion }
    Rect := ClientRect;
    DiffRegion := CreateRectRgnIndirect(Rect);
    CombineRgn(DiffRegion, DiffRegion, BigRegion, Rgn_Diff);

    if FUseUnselectedTabColour and ((FRowExtent = 1) or not FAnchoredTabs) then
    begin
      { Create a region for the selected tab and front card.
        This will be used if unselected tabs are to be a different
        colour to the selected tab/card.
      }
      Rect := CalcCardRect(0);
      FrontRegion := CreateRectRgnIndirect(Rect);

      TabIdentity := ActivePage;
      Row := FindTabRow(TabIdentity);
      Col := FindTabCol(TabIdentity);
      TbRect := TabRect(TabIdentity);
      BuildTabPoints(Pts, TbRect, 0, Row, Col);
      ExtendTabPoints(Pts);
      TabRegion := CreatePolygonRgn(Pts, Integer(High(Pts) + 1), PolyFillMode);
      CombineRgn(FrontRegion, FrontRegion, TabRegion, Rgn_Or);
      DeleteObject(TabRegion);

      { Fill region for selected tab and front card. }
      BrushHandle := CreateSolidBrush(ColorToRGB(Color));
      try
        FillRgn(Message.DC, FrontRegion, BrushHandle);
      finally
        DeleteObject(BrushHandle);
      end;

      { Subtract FrontRegion from BigRegion so that all the
        unselected tabs and the back cards can be coloured using
        UnselectedTabColor.
      }
      CombineRgn(BigRegion, BigRegion, FrontRegion, Rgn_Diff);
      DeleteObject(FrontRegion);
    end;

    { fill in region inside tabs/cards }
    if FUseUnselectedTabColour and ((FRowExtent = 1) or not FAnchoredTabs) then
      BrushHandle := CreateSolidBrush(ColorToRGB(FUnselectedTabColour))
    else
      BrushHandle := CreateSolidBrush(ColorToRGB(Color));
    try
      FillRgn(Message.DC, BigRegion, BrushHandle);
    finally
      DeleteObject(BrushHandle);
    end;

    { now fill in region outside notebook tabs/cards }
    BkgndHandled := False;

    if Assigned(FOnPaintBackground) then
    begin
      SaveDC(Message.DC); { so that region can be de-activated }
      OldRgn := SelectObject(Message.DC, DiffRegion);
      ACanvas := TCanvas.Create;
      try
        Rect := CalcWholeRect;
        Rect.Right := Rect.Right + 1;
        Rect.Bottom := Rect.Bottom + 1;
        ACanvas.Handle := Message.DC;
        FOnPaintBackground(Self, ACanvas, Rect, BkgndHandled);
      finally
        ACanvas.Free;
        SelectObject(Message.DC, OldRgn);
        RestoreDC(Message.DC, -1); { de-activate clipping region }
      end;
    end;

    if not BkgndHandled then
    begin
      BrushHandle := CreateSolidBrush(ColorToRGB(ColourOfParent));
      FillRgn(Message.DC, DiffRegion, BrushHandle);
      DeleteObject(BrushHandle);
    end;

    DeleteObject(DiffRegion);
    DeleteObject(BigRegion);

    Message.Result := 1; { erasing done }
  end;

end; { WMEraseBkgnd }


{ Invalidate only the area used by the caption and bitmap for the
  tab at the specified index position.
}
procedure TcsNotebook.InvalidateTab(Index: Integer);
var Rect: TRect;
begin
  if (Index >= 0) and (Index < FAccess.Count) and
    TcsPage(FPageList[Index]).PageVisible and
    HandleAllocated then
  begin
    Rect := CalcTabTextRect(FAccess[Index]);
    InvalidateRect(Handle, @Rect, True);
  end;
end;


{ Shorthand way of accessing the Bitmap for a page's tab. }
function TcsNotebook.GetTabBitmap(Index: Integer): TBitmap;
begin
  if (Index >= 0) and (Index < FPageList.Count) then
    Result := TcsPage(FPageList.Items[Index]).Bitmap
  else
    Result := nil;
end;

{ Shorthand way of assigning the Bitmap for a page's tab. }
procedure TcsNotebook.SetTabBitmap(Index: Integer; Value: TBitmap);
begin
  if (Index >= 0) and (Index < FPageList.Count) then
    TcsPage(FPageList.Items[Index]).Bitmap := Value;
end;


{ Return the page no. of the tab with the specified caption. }
function TcsNotebook.GetTabPageIndex(const TabIdentity: String): Integer;
begin
  Result := FAccess.IndexOf(TabIdentity);
end;


{ Shorthand way of accessing the caption of the specified page;
  equivalent to using:
    TcsPage(<notebook>.Pages.Objects[Index]).Caption;
}
function TcsNotebook.GetTabCaption(Index: Integer): String;
begin
  if (Index >= 0) and (Index < FAccess.Count) then
    Result := FAccess[Index]
  else
    Result := '';
end;


{ Change the caption of the tab for the specified page. }
procedure TcsNotebook.UpdateTabCaption(Index: Integer; Value: String);
var OldValue: String;
    Orphaned: Boolean;
    I, Row, Col: Integer;

  { Do a CASE SENSITIVE search in the list Strings for the string S.
    Used instead of TStrings.IndexOf which does a case insensitive search
    and treats strings differing only in capitalisation as being the same.
  }
  function CaseSensitiveIndexOf(Strings: TStrings; const S: String): Integer;
  begin
    for Result := 0 to Strings.Count - 1 do
      if AnsiCompareStr(Strings[Result], S) = 0 then Exit;
    Result := -1;
  end;

begin
  if (Index >= 0) and (Index < FAccess.Count) then
  begin
    { Find previous caption value by looking for an orphaned value
      in FTabIndex.
    }
    I := 0;
    Orphaned := False;

    while (I < FTabIndex.Count) and not Orphaned do
    begin
      { must do case sensitive search to catch changes in caption's capitalisation }
      Orphaned := (CaseSensitiveIndexOf(FAccess, FTabIndex[I]) < 0);
      if Orphaned then
        OldValue := FTabIndex[I]
      else
        Inc(I);
    end;

    if Orphaned then
    begin
      { Update internal structures with new caption value. }
      { Note: Tab info in FTabIndex is not in page no. order but in
              order tabs were initially displayed.
      }
      FTabIndex[I] := Value;
      Row := FindTabRow(OldValue);
      Col := FindTabCol(OldValue);
      TStringList(FRowIdentities.Items[Row]).Strings[Col] := Value;
      InvalidateTab(Index);
    end;
  end;
end;


{ Shorthand way of assigning the caption of the specified page;
  equivalent to using:
    TcsPage(<notebook>.Pages.Objects[Index]).Caption := Value;
}
procedure TcsNotebook.SetTabCaption(Index: Integer; Value: String);
begin
  FAccess[Index] := Value;
end;


{ Invalidate the tab text rects of the specified tabs (All, Selected or
  Unselected).
}
procedure TcsNotebook.InvalidateTabTextRects(Which: TTabRects);
var I: Integer;
    Rect: TRect;
    Page: TcsPage;
begin
  { The check for FHasLoaded is necessary because when using Large Fonts
    under Windows95 the font is changed when the form is loaded but
    before the notebook/tabs have been loaded.
  }
  if HandleAllocated and FHasLoaded then
  begin
    if (Which = trSelected) then
    begin { only invalidate tab text rect for current tab }
      Rect := CalcTabTextRect(ActivePage);
      InvalidateRect(Handle, @Rect, True);
    end
    else
    begin
      for I := 0 to FPageList.Count - 1 do
      begin
        Page := TcsPage(FPageList.Items[I]);
        if Page.PageVisible and
          ((Which = trAll) or ((Which = trUnselected) and (I <> FPageIndex))) then
        begin
          Rect := CalcTabTextRect(Page.Caption);
          InvalidateRect(Handle, @Rect, True);
        end;
      end;
    end;
  end;
end;


procedure TcsNotebook.CheckSidewaysText;
begin
  if FSidewaysText and (FTabOrientation in [toTop, toBottom]) then
    { SidewaysText property must have been manually edited in the
      .DFM file (in ASCII mode).
      Ensure that SidewaysText property value is appropriate for
      current TabOrientation.
    }
    FSidewaysText := False;
end;


procedure TcsNotebook.Loaded;
begin
  inherited Loaded;
  CheckSidewaysText;
  if not FHasLoaded then
  begin
    CalcAndBuild; { for all pages loaded from resource }
    FHasLoaded := True;
  end;
  SetupTabs;
  { set the default initial page if within range }
  if (FPageIndex <> FPageIndexDefault) and (FPageIndexDefault >= 0) and
    (FPageIndexDefault < FPageList.Count) then
    SetPageIndex(FPageIndexDefault);
  if not (csDesigning in ComponentState) then
  begin
    SetInitialPage;
    { disable notebook if no pages are enabled }
    Enabled := AnyPagesEnabled;
  end;
end;


procedure TcsNotebook.SetUnselectedTabColour(Value: TColor);
begin
  if (Value <> FUnselectedTabColour) then
  begin
    FUnselectedTabColour := Value;
    if FUseUnselectedTabColour then
      Invalidate;
  end;
end;


procedure TcsNotebook.SetUseUnselectedTabColour(Value: Boolean);
begin
  if (Value <> FUseUnselectedTabColour) then
  begin
    FUseUnselectedTabColour := Value;
    if (FUnselectedTabColour <> Color) then
      Invalidate;
  end;
end;


{ Draw the caption and bitmap for each tab, greying if tab is disabled. }
procedure TcsNotebook.DrawTabFaces;
var DC: THandle;
    I, N, L, T, W, H, Index, GlyphWidth, GlyphHeight: Integer;
    SizeAsPt: TPoint;
    TabIdentity, TabText: String;
    TabBitmap, TmpBmp: TBitmap;
    DstRect, TextRect, BMRect, TmpRect, GlyphRect: TRect;
    NormalFont, NormalUnderlineFont, BoldFont, BoldUnderlineFont: HFont;
    OldFont, PlainFont, UnderlinedFont: HFont;
    GreyFace: Boolean;
    BrushHandle: HBrush;
    OldBrushHandle: HBrush;

  { GetFontHandle returns a font handle to be used for text output.
    The current orientation is used to create a rotated font
    if necessary.
    The sender should destroy the returned font object when it
    is no longer needed.
  }
  function GetFontHandle(flags: TFontStyles): HFont;
  var Escapement: Integer;
      LogFont: TLogFont;
  begin
    case FTabOrientation of
      toTop, toBottom:  Escapement := 0;
      toLeft: Escapement := iifInt(FSidewaysText, 900, 0);
      toRight: Escapement := iifInt(FSidewaysText, 2700, 0);
    else
      Escapement := 0;
    end;

    { Must initialise all fields of the record structure! }
    with LogFont do
    begin
      lfHeight := TabFont.Height;
      lfWidth := 0; { have font mapper choose }
      lfEscapement := Escapement;
      lfOrientation := 0; { no rotation }
      if (fsBold in TabFont.Style) or (fsBold in flags) then
        lfWeight := FW_BOLD
      else
        lfWeight := FW_NORMAL;
      lfItalic := Byte(fsItalic in TabFont.Style);
      lfUnderline := Byte((fsUnderline in TabFont.Style) or (fsUnderline in flags));
      lfStrikeOut := Byte(fsStrikeOut in TabFont.Style);
      lfCharSet := DEFAULT_CHARSET;
      StrPCopy(lfFaceName, TabFont.Name);
      lfQuality := DEFAULT_QUALITY;
      { Everything else as default }
      lfOutPrecision := OUT_DEFAULT_PRECIS;
      lfClipPrecision := CLIP_DEFAULT_PRECIS;
      lfPitchAndFamily := DEFAULT_PITCH;
    end;

    Result := CreateFontIndirect(LogFont);

  end; { GetFontHandle }

  { Return the appropriate background colour for the specified tab. }
  function TabBkgndColour: TColor;
  begin
    if FUseUnselectedTabColour then
      if (TabIdentity = ActivePage) then
        Result := Color
      else
        Result := FUnselectedTabColour
    else
        Result := Color;
  end; { TabBkgndColour }

  { Draws the tab's bitmap onto the specified canvas. }
  procedure DrawTabFaceBitmap(ACanvas: TCanvas);
  var
    XOffset, YOffset: Integer;
  begin
    { create and use our own brush so that if using a palette the colors work }
    InitPalette(ACanvas.Handle);
    BrushHandle := CreateSolidBrush(ColorToRGB(TabBkgndColour));
    try
      OldBrushHandle := SelectObject(ACanvas.Handle, BrushHandle);
      if (FBitmapAlignment = baFit) then
        { stretch (or shrink) to fit }
        ACanvas.BrushCopy(BMRect, TabBitmap, GlyphRect, TabBitmap.TransparentColor)
      else
      begin
        { determine portion of glyph which should be visible
          and adjust it's rect to clip accordingly (BrushCopy uses
          a StretchBlt so this is necessary to prevent shrinking
          to fit)
        }
        XOffset := 0;
        YOffset := 0;
        if (GlyphWidth > (BMRect.Right - BMRect.Left)) then
          XOffset := (GlyphWidth - (BMRect.Right - BMRect.Left)) div 2;
        if (GlyphHeight > (BMRect.Bottom - BMRect.Top)) then
          YOffset := (GlyphHeight - (BMRect.Bottom - BMRect.Top)) div 2;
        if ((XOffset > 0) or (YOffset > 0)) then
          InflateRect(GlyphRect, -XOffset, -YOffset);
        ACanvas.BrushCopy(BMRect, TabBitmap, GlyphRect, TabBitmap.TransparentColor);
      end;
    finally
      SelectObject(ACanvas.Handle, OldBrushHandle);
      DeleteObject(BrushHandle);
    end;
  end; { DrawTabFaceBitmap }

  { Draw the tab's text onto the specified memory DC. }
  procedure DrawTabFaceText(DC: THandle);
  var OldBkMode: Integer;
  begin
    if (TabIdentity = ActivePage) then
    begin
      PlainFont := BoldFont;
      UnderlinedFont := BoldUnderlineFont;
      SetTextColor(DC, ColorToRGB(FSelectedColour));
    end
    else
    begin
      PlainFont := NormalFont;
      UnderlinedFont := NormalUnderlineFont;
      SetTextColor(DC, ColorToRGB(FUnselectedColour));
    end;

    { Select font that will be used for tab text so that text extent
      calculations (using the current DC) will be correct.
    }
    OldFont := SelectObject(DC, PlainFont);
    OldBkMode := SetBkMode(DC, TRANSPARENT);
    try
      SizeAsPt := CalcTextOrigin(TabText, TextRect, DC);
      DoTextOut(SizeAsPt.X, SizeAsPt.Y, TabText, DC, PlainFont, UnderlinedFont);
    finally
      SetBkMode(DC, OldBkMode);
      SelectObject(DC, OldFont);
    end;
  end; { DrawTabFaceText }

  { Simulate 'greying' of the specified bitmap by applying a
    checkerboard brush pattern (as used for disabled buttons).
    Using this technique of greying (as opposed to converting to a
    monochrome bitmap and then adding a white 'shadow') will work
    even if there is no black used in the bitmap and regardless of
    the actual colours of the caption text, notebook and bitmaps.
  }
  procedure GreyBitmap(ABitmap: TBitmap);
  var BrushBmp, BkgndBmp: TBitmap;
      PrevColour, PixelColour: TColor;
      X, Y, BmpW, BmpH: Integer;
      PatternBrush: TBrush;
      BmpRect: TRect;
  begin
    BrushBmp := TBitmap.Create;
    BkgndBmp := TBitmap.Create;
    try
      BrushBmp.Width := 8;  { pattern brushes are 8x8 }
      BrushBmp.Height := 8;

      { Make checkerboard brush pattern (alternating black and white) }
      PrevColour := clWhite;
      for X := 0 to 7 do
      begin
        PixelColour := PrevColour;
        for Y := 0 to 7 do
        begin
          BrushBmp.Canvas.Pixels[X, Y] := ColorToRGB(PixelColour);
          PrevColour := PixelColour;
          if PixelColour = clWhite then
            PixelColour := clBlack
          else
            PixelColour := clWhite;
        end;
      end;

      BmpW := ABitmap.Width;
      BmpH := ABitmap.Height;
      BmpRect := Rect(0, 0, BmpW, BmpH);

      { prepare 2nd bitmap which is to be filled with background colour }
      BkgndBmp.Width := BmpW;
      BkgndBmp.Height := BmpH;

      InitPalette(BkgndBmp.Canvas.Handle);
      BrushHandle := CreateSolidBrush(ColorToRGB(TabBkgndColour));
      try
        FillRect(BkgndBmp.Canvas.Handle, BmpRect, BrushHandle);
      finally
        DeleteObject(BrushHandle);
      end;

      PatternBrush := TBrush.Create;
      try
        PatternBrush.Bitmap := BrushBmp;
        ABitmap.Canvas.Brush := PatternBrush;
        { apply background bitmap to target bitmap using brush pattern;
          this is equivalent to putting a fly-screen over the target
          and then spraying the background colour through the fly-screen
        }
        BitBlt(ABitmap.Canvas.Handle, 0, 0, W, H,
               BkgndBmp.Canvas.Handle, 0, 0, $CA0749); { DPSPxax }
      finally
        PatternBrush.Free;
      end;
    finally
      BrushBmp.Free;
      BkgndBmp.Free;
    end;
  end; { GreyBitmap }


  { Overlay the canvas (already containing the tab's bitmap/glyph) with
    the greyed text in the specified bitmap.  This is done to allow
    greyed text along with a specific Disabled glyph.
  }
  procedure OverlayBitmap(ACanvas: TCanvas; ABitmap: TBitmap);
  var MaskBmp, TempBmp: TBitmap;
      BmpW, BmpH: Integer;
      BmpRect: TRect;
  begin
    { create monochrome bitmap }
    BmpW := ABitmap.Width;
    BmpH := ABitmap.Height;
    BmpRect := Rect(0, 0, BmpW, BmpH);

    MaskBmp := TBitmap.Create;
    TempBmp := TBitmap.Create;

    try
      MaskBmp.Monochrome := True; { must be mono to achieve mask effect }
      MaskBmp.Width := BmpW;
      MaskBmp.Height := BmpH;
      TempBmp.Width := BmpW;
      TempBmp.Height := BmpH;

      { set brush (background) colour for colour to monochrome conversion }
      ABitmap.Canvas.Brush.Color := TabBkgndColour;
      BitBlt(MaskBmp.Canvas.Handle, 0, 0, BmpW, BmpH,
             ABitmap.Canvas.Handle, 0, 0, SrcCopy);

      { invert 1s and 0s to get valid mask }
      InvertRect(MaskBmp.Canvas.Handle, BmpRect);

      { copy what's on the screen first }
      BitBlt(TempBmp.Canvas.Handle, 0, 0, BmpW, BmpH,
             ACanvas.Handle, L, T, SrcCopy);

      BitBlt(TempBmp.Canvas.Handle, 0, 0, BmpW, BmpH,
             ABitmap.Canvas.Handle, 0, 0, SrcInvert);

      BitBlt(TempBmp.Canvas.Handle, 0, 0, BmpW, BmpH,
             MaskBmp.Canvas.Handle, 0, 0, $220326); { DSna }

      BitBlt(TempBmp.Canvas.Handle, 0, 0, BmpW, BmpH,
             ABitmap.Canvas.Handle, 0, 0, SrcInvert);

      { display in appropriate position on tab }
      BitBlt(ACanvas.Handle, L, T, W, H,
             TempBmp.Canvas.Handle, 0, 0, SrcCopy);
    finally
      MaskBmp.Free;
      TempBmp.Free;
    end;
  end; { OverlayBitmap }

begin
  { Create all necessary font handles to be used for the various
    parts of the tabs' text.
  }
  DC := Canvas.Handle;
  SaveDC(DC);

  NormalFont := GetFontHandle([]);
  NormalUnderlineFont := GetFontHandle([fsUnderline]);
  BoldFont := GetFontHandle([fsBold]);
  BoldUnderlineFont := GetFontHandle([fsBold, fsUnderline]);

  try
    for I := 0 to FTabIndex.Count - 1 do
    begin
      TabIdentity := FTabIndex.Strings[I];
      TabText := TabIdentity;
      DstRect := CalcTabTextRect(TabIdentity);
      L := DstRect.Left;
      T := DstRect.Top;
      W := DstRect.Right - DstRect.Left;
      H := DstRect.Bottom - DstRect.Top;
      if (H <= 0) or (W <= 0) then
        Continue;
      TextRect := DstRect;
      Index := FAccess.IndexOf(TabIdentity);
      TabBitmap := TcsPage(FPageList.Items[Index]).Bitmap;

      if (FBitmapAlignment <> baInvisible) and (TabBitmap <> nil) and
        (TabBitmap.Handle <> 0) then
      begin
        { draw the tab's bitmap }
        GreyFace := False;
        { Glyphs No.s: 0=Selected; 1=Disabled; 2=Unselected }
        if TcsPage(FPageList.Items[Index]).PageEnabled then
          if (TabIdentity = ActivePage) then N := 0
          else if (TabNumGlyphs[Index] = 3) then N := 2
          else N := 0
        else
          if (TabNumGlyphs[Index] > 1) then N := 1
          else
          begin
            { use 1st glyph and grey it }
            N := 0;
            GreyFace := True;
          end;
        GlyphWidth := TabBitmap.Width div TabNumGlyphs[Index];
        GlyphHeight := TabBitmap.Height;
        GlyphRect := Rect(N * GlyphWidth, 0, (N + 1) * GlyphWidth, TabBitmap.Height);

        if GreyFace then
        begin
          { Create temporary bitmap so that drawing of the tab's bitmap and
            caption can be done off-screen so that when it has to be
            'greyed' there will be no visible flicker.  The temp. bitmap
            is equivalent to the tab's text area (which also happens to
            include the tab's bitmap area).
            Because the temp. bitmap is only as large as the
            tab's text rect. there is no need to define a clipping region
            when drawing the text.
          }
          TmpBmp := TBitmap.Create;
          try
            TmpBmp.Width := W;
            TmpBmp.Height := H;
            TmpRect := Rect(0, 0, W, H);

            { start with appropriate background }
            InitPalette(TmpBmp.Canvas.Handle);
            BrushHandle := CreateSolidBrush(ColorToRGB(TabBkgndColour));
            try
              FillRect(TmpBmp.Canvas.Handle, TmpRect, BrushHandle);
            finally
              DeleteObject(BrushHandle);
            end;

            BMRect := CalcBitmapRect(TabIdentity);
            { adjust BMRect so it is relative to the temp. bitmap's origin }
            OffsetRect(BMRect, -L, -T);
            DrawTabFaceBitmap(TmpBmp.Canvas);
            GreyBitmap(TmpBmp);

            { now copy the bitmap back to the visible canvas }
            Canvas.CopyRect(Bounds(L, T, W, H), TmpBmp.Canvas, TmpRect);

          finally
            TmpBmp.Free;
          end;
        end
        else
        begin
          { draw directly onto screen canvas for better speed }
          BMRect := CalcBitmapRect(TabIdentity);
          DrawTabFaceBitmap(Canvas);
        end;
      end;

      if (FTextAlignment <> taInvisible) then
      begin
        { draw the tab's text }
        if TabPageEnabled[Index] then
        begin
          { draw the tab's text into a bitmap so it will be clipped --
            an actual clipping region isn't used because the text
            won't show when the control is printed
          }
          TmpBmp := TBitmap.Create;
          try
            TmpBmp.Width := W;
            TmpBmp.Height := H;
            TmpRect := Rect(0, 0, W, H);
            { start with appropriate background }
            InitPalette(TmpBmp.Canvas.Handle);
            BrushHandle := CreateSolidBrush(ColorToRGB(TabBkgndColour));
            try
              FillRect(TmpBmp.Canvas.Handle, TmpRect, BrushHandle);
            finally
              DeleteObject(BrushHandle);
            end;
            { adjust TextRect to TmpBmp origin }
            OffsetRect(TextRect, -L, -T);
            DrawTabFaceText(TmpBmp.Canvas.Handle);
            OverlayBitmap(Canvas, TmpBmp);
          finally
            TmpBmp.Free;
          end;
        end
        else
        begin
          { tab is disabled; draw greyed text and show on top of tab's glyph }
          TmpBmp := TBitmap.Create;
          try
            TmpBmp.Width := W;
            TmpBmp.Height := H;
            TmpRect := Rect(0, 0, W, H);

            { start with appropriate background }
            InitPalette(TmpBmp.Canvas.Handle);
            BrushHandle := CreateSolidBrush(ColorToRGB(TabBkgndColour));
            try
              FillRect(TmpBmp.Canvas.Handle, TmpRect, BrushHandle);
            finally
              DeleteObject(BrushHandle);
            end;

            OffsetRect(TextRect, -L, -T);  { adjust TextRect to TmpBmp origin }
            DrawTabFaceText(TmpBmp.Canvas.Handle);
            GreyBitmap(TmpBmp);
            OverlayBitmap(Canvas, TmpBmp);
          finally
            TmpBmp.Free;
          end;
        end;
      end;

    end; { for I }

  finally
    DeleteObject(NormalFont);
    DeleteObject(NormalUnderlineFont);
    DeleteObject(BoldFont);
    DeleteObject(BoldUnderlineFont);
    RestoreDC(DC, -1);
  end;
end; { DrawTabFaces }


{ Select a new (initial) page if the current page/tab is
  disabled or not visible.
}
procedure TcsNotebook.SetInitialPage;
var InitialPage: Integer;
begin
    if not (TcsPage(FPageList[FPageIndex]).PageEnabled and
            TcsPage(FPageList[FPageIndex]).PageVisible) then
    begin
      InitialPage := FPageIndex;
      repeat
        if InitialPage = FPageList.Count - 1 then
          InitialPage := 0
        else
          InitialPage := InitialPage + 1;
      until (TcsPage(FPageList[InitialPage]).PageEnabled and
             TcsPage(FPageList[InitialPage]).PageVisible) or
             (InitialPage = FPageIndex);

      if (InitialPage = FPageIndex) then
      begin
        { All tabs are disabled and/or invisible.
          Ensure Enabled and PageEnabled are in synch.
        }
        with TcsPage(FPageList[InitialPage]) do
          if Enabled and not PageEnabled then
            Enabled := False;

        { Also ensure that current page is a visible page when there
          are no enabled pages. }
        if AnyPagesVisible and not AnyPagesEnabled then
        begin
          repeat
            if InitialPage = FPageList.Count - 1 then
              InitialPage := 0
            else
              InitialPage := InitialPage + 1;
          until (TcsPage(FPageList[InitialPage]).PageVisible);
          SetPageIndex(InitialPage);
        end;
      end
      else
        SetPageIndex(InitialPage);
    end;
end;


{ Shorthand way of accessing the PageEnabled property of a notebook page. }
function TcsNotebook.GetTabPageEnabled(Index: Integer): Boolean;
begin
  if (Index >= 0) and (Index < FPageList.Count) then
    Result := TcsPage(FPageList[Index]).PageEnabled
  else
    Result := False;
end;


{ Shorthand way of assigning the PageEnabled property of a notebook page. }
procedure TcsNotebook.SetTabPageEnabled(Index: Integer; Value: Boolean);
begin
  if (Index >= 0) and (Index < FPageList.Count) then
    TcsPage(FPageList[Index]).PageEnabled := Value;
end;


{ Shorthand way of accessing the PageVisible property of a notebook page. }
function TcsNotebook.GetTabPageVisible(Index: Integer): Boolean;
begin
  if (Index >= 0) and (Index < FPageList.Count) then
    Result := TcsPage(FPageList[Index]).PageVisible
  else
    Result := False;
end;


{ Shorthand way of assigning the PageVisible property of a notebook page. }
procedure TcsNotebook.SetTabPageVisible(Index: Integer; Value: Boolean);
var
  NewIdx: Integer;
begin
  if (Index >= 0) and (Index < FPageList.Count) then
  begin
    { check if user is trying to hide the current page/tab }
    if (Index = FPageIndex) and not Value then
    begin
      { prevent focus problems which occur if current page/tab is hidden }
      NewIdx := FindNextPageIdx(FPageIndex);
      if (NewIdx >= 0) and (Index < FPageList.Count) then
        SetPageIndex(NewIdx);
    end;
    TcsPage(FPageList[Index]).PageVisible := Value;
  end;
end;


{ Check if any notebook tabs/pages are enabled and return the result. }
function TcsNotebook.AnyPagesEnabled: Boolean;
var Found: Boolean;
    I: Integer;
begin
  if (csDesigning in ComponentState) then
    Result := True
  else
  begin
    I := 0;
    repeat
      with TcsPage(FPageList[I]) do
        Found := (PageEnabled and PageVisible);
      Inc(I);
    until Found or (I = FPageList.Count);
    Result := Found;
  end;
end;


{ Check if any notebook tabs/pages are visible and return the result. }
function TcsNotebook.AnyPagesVisible: Boolean;
var Found: Boolean;
    I: Integer;
begin
  I := 0;
  repeat
    Found := TcsPage(FPageList[I]).PageVisible;
    Inc(I);
  until Found or (I = FPageList.Count);
  Result := Found;
end;

{ Get Bitmap for current page. }
function TcsNotebook.GetBitmap: TBitmap;
begin
  Result := TabBitmap[FPageIndex];
end;


{ Set Bitmap for current page. }
procedure TcsNotebook.SetBitmap(Value: TBitmap);
begin
  TabBitmap[FPageIndex] := Value;
end;


{ Shorthand way of accessing the NumGlyphs property of a notebook page. }
function TcsNotebook.GetTabNumGlyphs(Index: Integer): TTabNumGlyphs;
begin
  if (Index >= 0) and (Index < FPageList.Count) then
    Result := TcsPage(FPageList[Index]).NumGlyphs
  else
    Result := 1;
end;


{ Shorthand way of assigning the NumGlyphs property of a notebook page. }
procedure TcsNotebook.SetTabNumGlyphs(Index: Integer; Value: TTabNumGlyphs);
begin
  if (Index >= 0) and (Index < FPageList.Count) then
    TcsPage(FPageList[Index]).NumGlyphs := Value;
end;


{ Get NumGlyphs for current page. }
function TcsNotebook.GetNumGlyphs: TTabNumGlyphs;
begin
  Result := TabNumGlyphs[FPageIndex];
end;


{ Set NumGlyphs for current page. }
procedure TcsNotebook.SetNumGlyphs(Value: TTabNumGlyphs);
begin
  TabNumGlyphs[FPageIndex] := Value;
end;


{ Return the index of the tab at the specified location or -1 if
  no tab is found at the specified location.
}
function TcsNotebook.TabAtPos(X, Y: Integer): Integer;
var Pt: TPoint;
    I: Integer;
    TabIdentity: String;
    Found: Boolean;
    Rect: TRect;
begin
  Pt := Point(X, Y);
  I := 0;
  Found := False;

  while (I < FAccess.Count) and not Found do
  begin
    TabIdentity := FAccess[I];
    if TcsPage(FPageList[I]).PageVisible then
    begin
      Rect := TabRect(TabIdentity);
      if PtInRect(Rect, Pt) then
        Found := True;
    end;
    if not Found then
      Inc(I);
  end;

  if Found then
    Result := I
  else
    Result := -1;
end;


procedure TcsNotebook.CMDesignHitTest(var Msg: TCMDesignHitTest);
const TabChosen: Boolean = False;
var ShiftState: TShiftState;
    NewIndex: Integer;
{$IFDEF VER130}
    Form: TCustomForm;
{$ELSE}
{$IFDEF VER120}
    Form: TCustomForm;
{$ELSE}
{$IFDEF VER100}
    Form: TCustomForm;
{$ELSE}
    Form: TForm;
{$ENDIF}
{$ENDIF}
{$ENDIF}
begin
  ShiftState := KeysToShiftState(Msg.Keys);
  if (ssLeft in ShiftState) then
  begin
    if TabChosen then { tab already chosen, button not released yet }
      Msg.Result := 1 { message handled }
    else
    begin
      NewIndex := TabAtPos(Msg.Pos.X, Msg.Pos.Y);
      if (NewIndex >= 0) then
      begin
        SetPageIndex(NewIndex);
        TabChosen := True;
        { Set the form designer's Modified flag so the
          Object Inspector shows the new PageIndex value
        }
        Form := GetParentForm(Self);
        if (Form <> nil) and (Form.Designer <> nil) then
          Form.Designer.Modified;
        Msg.Result := 1; { message handled }
        { Note: Even though the message result is set to indicate that
                the message has been handled, MouseDown still ends up
                being called and must check if the component is in
                design-mode.
        }
      end;
    end;
  end
  else if TabChosen then
  begin
    TabChosen := False; { button released; reset flag }
    Msg.Result := 1; { message handled }
  end;
end;


procedure TcsNotebook.WMNCHitTest(var Msg: TWMNCHitTest);
begin
  DefaultHandler(Msg);
  FHitTest := SmallPointToPoint(Msg.Pos);
end;


procedure TcsNotebook.WMSetCursor(var Msg: TWMSetCursor);
var TabCursor: HCURSOR;
begin
  TabCursor := 0;
  if (csDesigning in ComponentState) then
    with Msg do
    begin
      if HitTest = HTCLIENT then
      begin
        FHitTest := ScreenToClient(FHitTest);
        if TabAtPos(FHitTest.X, FHitTest.Y) >= 0 then
        begin
          { Note: The CursorLoaded flag is used instead of just testing
                  if Screen.Cursors[crCSTabCursorID] = 0 because for some
                  unknown reason the latter never returns 0.
          }
          if not CursorLoaded then { cursor hasn't been loaded yet }
          begin
            { the cursor is only loaded here, i.e. when in design-mode,
              so that LoadCursor isn't performed by user's apps, at which
              time the resource won't be present
            }
            Screen.Cursors[crCSTabCursorID] := LoadCursor(HInstance, CSTabCursorName);
            CursorLoaded := True;
          end;
          TabCursor := Screen.Cursors[crCSTabCursorID];
        end;

      end;
    end;
  if TabCursor <> 0 then SetCursor(TabCursor)
  else inherited;
end;


{ GetPalette returns the handle of the palette to use for painting & erasing
  the control.
  The handle is obtained via the response to an OnGetPalette event.
}
function TcsNotebook.GetPalette: HPALETTE;
var Handle: HPALETTE;
begin
  Handle := inherited GetPalette; { zero }
  if Assigned(FOnGetPalette) then FOnGetPalette(Self, Handle);
  Result := Handle;
end;


{ Select and realize the control's palette }
procedure TcsNotebook.InitPalette(DC: HDC);
var Palette: HPALETTE;
begin
  Palette := GetPalette;
  if (Palette <> 0) then
  begin
    SelectPalette(DC, Palette, False);
    RealizePalette(DC);
  end;
end;


{ PaletteChanged has been overridden so that when the user switches to another
  window which uses a different palette, the page will be repainted in the same
  colour as the rest of the notebook.  This PaletteChanged method has no bearing
  on how the other aspects (painting and erasing) of the palette handling work.
}
function TcsNotebook.PaletteChanged(Foreground: Boolean): Boolean;
begin
  Invalidate;
  Result := inherited PaletteChanged(Foreground);
end;


procedure TcsNotebook.SetPageIndexDefault(Value: Integer);
begin
  { When the PageIndexProperty is being read from the form resource the
    pages haven't been read yet and so we can't range check then.
  }
  if (FPageIndexDefault <> Value) and
    (((csReading in ComponentState) or (csLoading in ComponentState)) or
    ((Value >= 0) and (Value < FPageList.Count))) then
    FPageIndexDefault := Value;
end;


{ Find the index of the next enabled and visible page;
  returns the index of the next page or -1 if no enabled and visible page found.
}
function TcsNotebook.FindNextPageIdx(StartIdx: Integer): Integer;
var
  NewIdx: Integer;
begin
  NewIdx := StartIdx;
  repeat
    if NewIdx = FPageList.Count - 1 then
      NewIdx := 0
    else
      Inc(NewIdx);
  until (TcsPage(FPageList[NewIdx]).PageEnabled and
         TcsPage(FPageList[NewIdx]).PageVisible) or
        (NewIdx = StartIdx);

  if (NewIdx = StartIdx) then
    Result := -1
  else
    Result := NewIdx;
end;

end.

