{*******************************************************}
{                                                       }
{   Copyright (c) 1996 Classic Software                 }
{   All rights reserved                                 }
{                                                       }
{*******************************************************}

unit CSADMain;

{$B-,P+,W-,X+}

interface

uses
  SysUtils, WinTypes, WinProcs, Messages, Classes, Graphics, Controls,
  Forms, Dialogs, StdCtrls, IniFiles, DsgnIntf;

const
  WM_ApplyDefaults = WM_USER + 0;

type
  TcsAutoDefaults = class(TComponent)
  private
    FFilename: String;
    FActive: Boolean;
    FWindowHandle: HWND;
    procedure BumpRefCount(Amount: Integer);
    function GetShowHint: Boolean;
    function InstancesOnForm: Integer;
    procedure SetFilename(const Value: String);
    procedure SetShowHint(Value: Boolean);
    procedure WndProc(var Msg: TMessage);
  protected
    procedure Notification(AComponent: TComponent; Operation: TOperation); override;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    procedure ApplyDefaults(Component: TComponent);
    procedure ClearDefaults(ComponentClass: TClass);
    procedure GetClassesWithDefaults(List: TStrings);
    procedure GetIgnoreOptions(List: TStrings);
    function HasDefaultsFor(Component: TComponent): Boolean;
    procedure SaveAsDefault(Component: TComponent);
    procedure SetIgnoreOptions(Value: TStrings);
  published
    property Active: Boolean read FActive write FActive default True;
    property ShowHint: Boolean read GetShowHint write SetShowHint default True;
    property Filename: String read FFilename write SetFilename;
  end;

  EcsADStreamError = class(EStreamError);

implementation

uses TypInfo {$IFDEF EVALUATION}, CSEval {$ENDIF};

type

  TDefaultsItem = class;

  TDefaultsTable = class(TObject)
  private
    FFileList: TStringList;
    FParentForm: TForm;  { used as Parent for default components }
    FShowHint: Boolean; { Show hints in component editor at design-time? }
    FUnsupportedClassList: TStringList; { classes which can't be saved }
    function GetItemCount: Integer;
    function GetShowHint: Boolean;
    procedure InitUnsupportedClassList;
    procedure SetShowHint(Value: Boolean);
  public
    constructor Create;
    destructor Destroy; override;
    procedure AddItem(const Filename: String; Item: TDefaultsItem);
    procedure DeleteItem(Item: TDefaultsItem);
    function FindFilename(const Filename: String; var Index: Integer): Boolean;
    function GetDefaultComponent(const Filename: String; ComponentClass: TClass): TComponent;
    function GetItem(const Filename: String): TDefaultsItem;
    function GetItemFilename(Item: TDefaultsItem): String;
    procedure PutDefaultComponent(const Filename: String; Component: TComponent);
    property ItemCount: Integer read GetItemCount;
    property ParentForm: TForm read FParentForm;
    property ShowHint: Boolean read GetShowHint write SetShowHint;
    property UnsupportedClassList: TStringList read FUnsupportedClassList;
  end;

  TDefaultsItem = class(TObject)
  private
    FIgnoreList: TStringList; { Property Name --> Class Ref. associations }
    FDefaultComponents: TList;
    FRefCount: Integer; { no. of TcsAutoDefaults objects using this item }
    procedure InitIgnoreList;
    procedure SetRefCount(Value: Integer);
  public
    constructor Create;
    destructor Destroy; override;
    procedure GetIgnoreOptions(List: TStrings);
    function IgnoreProperty(const PropertyName: String; Component: TComponent): Boolean;
    procedure LoadFromFile(const Filename: String);
    procedure LoadFromStream(Stream: TStream);
    procedure SaveToFile(const Filename: String);
    procedure SaveToStream(Stream: TStream);
    procedure SetIgnoreOptions(Value: TStrings);
    property DefaultComponents: TList read FDefaultComponents;
    property RefCount: Integer read FRefCount write SetRefCount;
  end;

  TClassRefObj = class(TObject)
  public
    ClassRef: TClass;
  end;

  { While some of the fields of a TADHeader record do not really need
    the range of values allowed by a LongInt (specifically Version and
    ComponentCount), all are declared as LongInt so that the record
    size will be a multiple of 4 bytes to allow for Delphi 2 aligning
    all records on a 32-bit boundary (i.e. adding extra bytes to the
    actual record's length if not a multiple of 4).
  }
  TADHeader = record
    Signature: Array[0..3] of Char;
    Version: LongInt;
    OptionsOffset: LongInt;
    ComponentCount: LongInt;
  end;

const AD_HEADER: TADHeader =
        (Signature: 'CSAD'; Version: 1; OptionsOffset: 0; ComponentCount: 0);

{ Variables used for management of internal data structures }

var DefaultsTable: TDefaultsTable;

{$IFNDEF WIN32}
{ Compatibility function for Delphi 1.0 16-bit }
procedure SetLength(var S: String; NewLength: Integer);
begin
  S[0] := Char(NewLength);
end;
{$ENDIF}

{ Cloning routines that use RTTI; CloneObjectProp and CloneMethodProp
  may do nothing.
}
procedure CloneProperty(SrcInstance, DstInstance: TPersistent; PropInfo: Pointer; Item: TDefaultsItem);
var
  PropType: PTypeInfo;
  PropName: String;

  procedure CloneOrdProp;
  begin
    SetOrdProp(DstInstance, PropInfo, GetOrdProp(SrcInstance, PropInfo));
  end;

  procedure CloneFloatProp;
  begin
    SetFloatProp(DstInstance, PropInfo, GetFloatProp(SrcInstance, PropInfo));
  end;

  procedure CloneStrProp;
  begin
    SetStrProp(DstInstance, PropInfo, GetStrProp(SrcInstance, PropInfo));
  end;

  procedure CloneObjectProp;
  begin
    SetOrdProp(DstInstance, PropInfo, GetOrdProp(SrcInstance, PropInfo));
  end;

  procedure CloneMethodProp;
  begin
    SetMethodProp(DstInstance, PropInfo, GetMethodProp(SrcInstance, PropInfo));
  end;

begin
  if PPropInfo(PropInfo)^.SetProc <> nil then
  begin
    PropName := PPropInfo(PropInfo)^.Name;
    if not ((PropName = 'Name') or Item.IgnoreProperty(PropName, TComponent(SrcInstance))) then
    begin
      PropType := PPropInfo(PropInfo)^.PropType;
      case PropType^.Kind of
        tkInteger, tkChar, tkEnumeration, tkSet: CloneOrdProp;
        tkFloat: CloneFloatProp;
        tkString: CloneStrProp;
        tkClass: CloneObjectProp;
        tkMethod: CloneMethodProp;
      end;
    end;
  end;
end;

procedure CloneProperties(SrcInstance, DstInstance: TPersistent; Item: TDefaultsItem);
var
  I, Count: Integer;
  PropInfo: PPropInfo;
  PropList: PPropList;
begin
  Count := GetTypeData(SrcInstance.ClassInfo)^.PropCount;
  if Count > 0 then
  begin
    GetMem(PropList, Count * SizeOf(Pointer));
    try
      GetPropInfos(SrcInstance.ClassInfo, PropList);
      for I := 0 to Count - 1 do
      begin
        PropInfo := PropList^[I];
          CloneProperty(SrcInstance, DstInstance, PropInfo, Item);
      end;
    finally
      FreeMem(PropList, Count * SizeOf(Pointer));
    end;
  end;
end;

function CloneComponent(Component: TComponent; AOwner: TComponent; Item: TDefaultsItem): TComponent;
begin
  Result := TComponentClass(Component.ClassType).Create(AOwner);
  if (Component is TControl) then
    TControl(Result).Parent := DefaultsTable.ParentForm;
  CloneProperties(Component, Result, Item);
end;

{ TDefaultsTable }

constructor TDefaultsTable.Create;
begin
  inherited Create;
  FFileList := TStringList.Create;
  FParentForm := TForm.Create(Application); { form remains hidden }
  FParentForm.Hide;
  FShowHint := True;
  InitUnsupportedClassList;
end;

destructor TDefaultsTable.Destroy;
begin
  { There is no need to free each item in FFileList.Objects[]
    (a TDefaultsItem) because these are freed as a result of their
    RefCount property reaching 0 when all the TcsAutoDefaults
    objects are destroyed (as part of Application shutdown).
  }
  FFileList.Free;
  { Because DefaultsTable is destroyed within an exit procedure,
    FParentForm will have already been freed because it is owned
    by the application.
  }
  FUnsupportedClassList.Free;
  inherited Destroy;
end;

{ Read unsupported classes from INI file }
procedure TDefaultsTable.InitUnsupportedClassList;
const DefaultStr: String = 'TMainMenu;TPopupMenu;TTabSet;' +
                            'TDBNavigator;TDBLookupCombo;' +
                            'TSpinButton;TSpinEdit';
var IniFile: TIniFile;
    Str, ClassName: String;
    SemiPos: Integer;
begin
  FUnsupportedClassList := TStringList.Create;
  IniFile := TIniFile.Create('delphi.ini');
  try
    Str := IniFile.ReadString('ClassicSoftware.AutoDefaults',
                               'UnsupportedClasses',
                               DefaultStr);
    while Length(Str) > 0 do
    begin
      SemiPos := Pos(';', Str);
      if (SemiPos > 0) then
      begin
        ClassName := Copy(Str, 1, SemiPos - 1);
        Str := Copy(Str, SemiPos + 1, Length(Str) - SemiPos);
      end
      else
      begin
        ClassName := Str;
        Str := '';
      end;
      FUnsupportedClassList.Add(ClassName);
    end;
  finally
    IniFile.Free;
  end;
end;

function TDefaultsTable.FindFilename(const Filename: String; var Index: Integer): Boolean;
begin
  Index := FFileList.IndexOf(Filename);
  Result := (Index >= 0);
end;

function TDefaultsTable.GetItem(const Filename: String): TDefaultsItem;
var Pos: Integer;
begin
  Result := nil;
  if FindFilename(Filename, Pos) then
    Result := TDefaultsItem(FFileList.Objects[Pos]);
end;

procedure TDefaultsTable.AddItem(const Filename: String; Item: TDefaultsItem);
begin
  FFileList.AddObject(Filename, Item);
end;

procedure TDefaultsTable.DeleteItem(Item: TDefaultsItem);
var ItemPos: Integer;
begin
  ItemPos := FFileList.IndexOfObject(Item);
  if (ItemPos >= 0) then
  begin
    Item.Free;
    FFileList.Delete(ItemPos);
  end;
end;

function TDefaultsTable.GetItemFilename(Item: TDefaultsItem): String;
var ItemPos: Integer;
begin
  ItemPos := FFileList.IndexOfObject(Item);
  if (ItemPos >= 0) then
    Result := FFileList[ItemPos]
  else
    raise Exception.Create('TcsAutoDefaults Internal Error');
end;

procedure TDefaultsTable.PutDefaultComponent(const Filename: String; Component: TComponent);
var Pos, I: Integer;
    Found: Boolean;
    Item: TDefaultsItem;
    List: TList;
begin
  if not FindFilename(Filename, Pos) then
    raise Exception.Create('Filename is blank or invalid.' +
                           #13 + 'Unable to save');

  { Assert: Pos >= 0 }

  { look for existing defaults for class of component }
  Item := DefaultsTable.GetItem(Filename);
  if (Item <> nil) then
  begin
    List := Item.DefaultComponents;
    Found := False;
    I := 0;
    while (I < List.Count) and not Found do
    begin
      Found := (Component.ClassType = TComponent(List[I]).ClassType);
      if not Found then
        Inc(I);
    end;

    if Found then
    begin
      { remove existing defaults for the class of the component }
      TComponent(List[I]).Free;
      List.Delete(I);
    end;

    { add component defaults }
    List.Add(CloneComponent(Component, Application, Item));

    { Save changes }
    Item.SaveToFile(Filename);
  end;
end;

function TDefaultsTable.GetDefaultComponent(const Filename: String; ComponentClass: TClass): TComponent;
var Item: TDefaultsItem;
    List: TList;
    I: Integer;
    Found: Boolean;
begin
  Result := nil;
  Item := DefaultsTable.GetItem(Filename);
  if (Item <> nil) then
  begin
    List := Item.DefaultComponents;
    repeat
      { look for a default component for the current class }
      I := 0;
      Found := False;
      while (I < List.Count) and not Found do
      begin
        Found := (TComponent(List[I]).ClassType = ComponentClass);
        if Found then
          Result := List[I]
        else
          Inc(I);
      end;
      if Result = nil then { no default found, go to class ancestor }
        ComponentClass := ComponentClass.ClassParent;
    until (Result <> nil) or (ComponentClass = TPersistent);
  end;
end;

function TDefaultsTable.GetItemCount: Integer;
begin
  if FFileList = nil then Result := 0
  else Result := FFileList.Count;
end;

function TDefaultsTable.GetShowHint: Boolean;
begin
  Result := FShowHint;
end;

procedure TDefaultsTable.SetShowHint(Value: Boolean);
begin
  if FShowHint <> Value then
    FShowHint := Value;
end;

{ TDefaultsItem }

constructor TDefaultsItem.Create;
begin
  inherited Create;
  InitIgnoreList;
  FDefaultComponents := TList.Create;
  FRefCount := 0;
end;

destructor TDefaultsItem.Destroy;
var I: Integer;
begin
  for I := 0 to FIgnoreList.Count - 1 do
    TClassRefObj(FIgnoreList.Objects[I]).Free;
  FIgnoreList.Free;
  for I := 0 to FDefaultComponents.Count - 1 do
    { free each default component }
    TComponent(FDefaultComponents[I]).Free;
  FDefaultComponents.Free;
  inherited Destroy;
end;

procedure TDefaultsItem.SetRefCount(Value: Integer);
begin
  FRefCount := Value;
  if (FRefCount = 0) then
    DefaultsTable.DeleteItem(Self);
end;

{ Read default ignore list from INI file }
procedure TDefaultsItem.InitIgnoreList;
var IniFile: TIniFile;
    ShortList, LongList: TStringList;
    I: Integer;
    Line: String;
    EqualsPos, DotPos: Integer;
begin
  FIgnoreList := TStringList.Create;
  IniFile := TIniFile.Create('delphi.ini');
  try
    ShortList := TStringList.Create;
    LongList := TStringList.Create;
    try
      IniFile.ReadSectionValues('ClassicSoftware.AutoDefaults', LongList);
      for I := 0 to LongList.Count - 1 do
      begin
        Line := LongList[I];
        if (UpperCase(Copy(Line, 1, 6)) = 'IGNORE') then
        begin
          { Line is of format "Ignore<n>=<setting>",
            e.g. "Ignore0=TControl.Left"
          }
          EqualsPos := Pos('=', Line);
          DotPos := Pos('.', Line);
          if (EqualsPos > 0) and (DotPos > 0) and (Length(Line) > DotPos) then
            ShortList.Add(Copy(Line, EqualsPos + 1, Length(Line) - EqualsPos));
        end;
      end;
      SetIgnoreOptions(ShortList);
    finally
      ShortList.Free;
      LongList.Free;
    end;
  finally
    IniFile.Free;
  end;
end;

{ Save the defaults for the specified file. }
procedure TDefaultsItem.SaveToFile(const Filename: String);
var Stream: TStream;
begin
  Stream := TFileStream.Create(Filename, fmCreate);
  try
    SaveToStream(Stream);
  finally
    Stream.Free;
  end;
end;

procedure TDefaultsItem.SaveToStream(Stream: TStream);
const Filler: LongInt = 0;
var Header: TADHeader;
    I: Integer;
    List: TStringList;
    ClassName: String;
    NameLength, OldPosition, EndPosition: LongInt;
begin
  Header := AD_HEADER;
  Header.ComponentCount := FDefaultComponents.Count;
  Stream.Write(Header, SizeOf(Header));
  { write each default component, prefixed by its class name;
    also write the position of the end of each component so
    an individual default component can be skipped if a problem
    when reloading it (because it is an unsupported class)
  }
  for I := 0 to FDefaultComponents.Count - 1 do
  begin
    ClassName := TComponent(FDefaultComponents[I]).ClassName;
    { allow for different string types in Delphi 1 & 2 (can't refer
      to length byte ([0]) for Delphi 2 long strings);
      also use LongInt instead of Integer for length to maintain
      same storage size
    }
    NameLength := Length(ClassName);
    Stream.Write(NameLength, SizeOf(NameLength));
    Stream.Write(ClassName[1], NameLength);

    OldPosition := Stream.Position;
    { reserve space for end position of this component }
    Stream.Write(Filler, SizeOf(Filler));
    Stream.WriteComponent(FDefaultComponents[I]);
    { remember end position }
    EndPosition := Stream.Position;
    { move back to reserved space and write end position }
    Stream.Position := OldPosition;
    Stream.Write(EndPosition, SizeOf(EndPosition));
    { move back to end of component again }
    Stream.Position := EndPosition;
  end;
  { set offset of start of Ignore Options data in header (written later) }
  Header.OptionsOffset := Stream.Position;
  { write Ignore Options data }
  List := TStringList.Create;
  try
    GetIgnoreOptions(List);
    List.SaveToStream(Stream);
  finally
    List.Free;
  end;
  { write offset of start of Ignore Options data into header }
  Stream.Position := SizeOf(Header.Signature) + SizeOf(Header.Version);
  Stream.Write(Header.OptionsOffset, SizeOf(Header.OptionsOffset));
end;

{ Load the defaults from the specified file. }
procedure TDefaultsItem.LoadFromFile(const Filename: String);
var Stream: TStream;
begin
  if (Filename = '') or not FileExists(Filename) then
    { nothing to load }
    Exit;
  Stream := TFileStream.Create(FileName, fmOpenRead);
  try
    LoadFromStream(Stream);
  finally
    Stream.Free;
  end;
end;


procedure TDefaultsItem.LoadFromStream(Stream: TStream);
var I: Integer;
    Header: TADHeader;
    SampleSignature, ErrorMsg: PChar;
    List: TStringList;
    Component: TComponent;
    ClassName: String;
    NameLength, EndPosition: LongInt;
begin
  SampleSignature := StrAlloc(SizeOf(Header.Signature) + 1);
  try
    Stream.Read(Header, SizeOf(Header));
    StrMove(SampleSignature, Header.Signature, SizeOf(Header.Signature));
    SampleSignature[SizeOf(Header.Signature)] := #0;
    if StrPas(SampleSignature) <> AD_HEADER.Signature then
      raise EcsADStreamError.Create('Invalid AutoDefaults file');
    { free existing items in FDefaultComponents before loading new items }
    for I := 0 to FDefaultComponents.Count - 1 do
      TComponent(FDefaultComponents[I]).Free;
    FDefaultComponents.Clear;
    for I := 1 to Header.ComponentCount do
    begin
      { allow for different string types in Delphi 1 & 2 (can't refer
        to length byte ([0]) for Delphi 2 long strings);
      }
      Stream.Read(NameLength, SizeOf(NameLength));
      SetLength(ClassName, NameLength);
      Stream.Read(ClassName[1], NameLength);
      { read end position of this component (used if need to skip) }
      Stream.Read(EndPosition, SizeOf(EndPosition));
      try
        Component := TComponentClass(FindClass(ClassName)).Create(Application);
        if (Component is TControl) then
          { Parent needs to be set before reading the component for
            some components (esp. for properties which are TStrings)
          }
          TControl(Component).Parent := DefaultsTable.ParentForm;
{ Currently the next line of code will produce an exception
  under Delphi 2 for the 2nd and subsequent default
  components read from the ADF file (regardless
  of the actual class of the component).
}
        Component := Stream.ReadComponent(Component);
        FDefaultComponents.Add(Component);
      except
        { skip the 'bad' component/class and let the user know }
        Stream.Position := EndPosition;
        ErrorMsg := StrAlloc(512);
        try
          StrPCopy(ErrorMsg,  'The ' + ClassName + ' default component' + #13 +
                              'could not be loaded.' + #13 + #13 +
                              'You should add this class to the' + #13 +
                              'UnSupportedClasses setting in the' + #13 +
                              '[ClassicSoftware.AutoDefaults]' + #13 +
                              'section of your DELPHI.INI file.');
          Application.MessageBox(ErrorMsg, 'TcsAutoDefaults',
                                  MB_ICONSTOP + MB_OK);
        finally
          StrDispose(ErrorMsg);
        end;
      end;
    end;
    List := TStringList.Create;
    try
      if Header.OptionsOffset <> 0 then
      begin
        { move to start of Ignore Options data and read }
        Stream.Position := Header.OptionsOffset;
        List.LoadFromStream(Stream);
      end;
      { only replace default Ignore list (read from INI file) if
        items were loaded from the AutoDefaults file
      }
      if List.Count > 0 then
        SetIgnoreOptions(List);
    finally
      List.Free;
    end;
  finally
    StrDispose(SampleSignature);
  end;
end;

{ The strings in Value will be in "<class>.<property>" format,
  e.g. "TControl.Left".
  These will be converted into "property" --> <class> associations
  and stored in FIgnoreList.  The Strings in FIgnoreList will be
  the property names and the associated Objects will be a
  ClassRefObj object.
}
procedure TDefaultsItem.SetIgnoreOptions(Value: TStrings);
var I, DotPos: Integer;
    Str, ClassName, PropertyName: String;
    ClassRef: TClass;
    ClassRefObj: TClassRefObj;
begin
  { free any existing entries in Ignore list's objects }
  for I := 0 to FIgnoreList.Count - 1 do
    TClassRefObj(FIgnoreList.Objects[I]).Free;

  FIgnoreList.Clear;

  for I := 0 to Value.Count - 1 do
  begin
    Str := Value[I];
    DotPos := Pos('.', Str);
    if DotPos > 0 then
    begin
      ClassName := Copy(Str, 1, DotPos - 1);
      PropertyName := Copy(Str, DotPos + 1, Length(Str) - DotPos);
      if (Length(ClassName) > 0) and (Length(PropertyName) > 0) then
      begin
        ClassRef := GetClass(ClassName);
        if (ClassRef <> nil) then
        begin
          ClassRefObj := TClassRefObj.Create;
          ClassRefObj.ClassRef := ClassRef;
          FIgnoreList.AddObject(PropertyName, ClassRefObj);
        end;
      end;
    end;
  end;
end;

{ Return IgnoreOptions list of TStrings. }
procedure TDefaultsItem.GetIgnoreOptions(List: TStrings);
var I: Integer;
begin
  List.BeginUpdate;
  try
    List.Clear;
    for I := 0 to FIgnoreList.Count - 1 do
      List.Add(TClassRefObj(FIgnoreList.Objects[I]).ClassRef.ClassName + '.' +
               FIgnoreList[I]);
  finally
    List.EndUpdate;
  end;
end;

function TDefaultsItem.IgnoreProperty(const PropertyName: String; Component: TComponent): Boolean;
var I: Integer;
begin
  Result := False;
  for I := 0 to FIgnoreList.Count - 1 do
  begin
    if (UpperCase(FIgnoreList[I]) = UpperCase(PropertyName)) and
      (Component is TClassRefObj(FIgnoreList.Objects[I]).ClassRef) then
    begin
      Result := True;
      Exit;
    end;
  end;
end;

{ TcsAutoDefaults }

constructor TcsAutoDefaults.Create(AOwner: TComponent);
var IniFile: TIniFile;
begin
  inherited Create(AOwner);
  FWindowHandle := 0;
  FFilename := '';
  if not (csDesigning in ComponentState) then
    Exit;
  if (InstancesOnForm > 1) then
  begin
    { Let the user know.  Even though it is OK to have more than one
      AutoDefaults component on the same form the results may not be
      what they expect.
    }
    Application.MessageBox('There is now more than one TcsAutoDefaults' + #13 +
                           'component on this form.', 'TcsAutoDefaults',
                            MB_ICONINFORMATION + MB_OK);
  end;
  FWindowHandle := AllocateHWnd(WndProc);
  IniFile := TIniFile.Create('delphi.ini');
  try
    { Initialise default Filename.  Can't rely on Loaded to do this
      because it isn't called when adding the component to the form
      at design-time.
      SetFilename must be used so that the reference count is
      updated.
    }
    SetFilename(IniFile.ReadString('ClassicSoftware.AutoDefaults',
                                   'DefaultFilename', ''));
  finally
    IniFile.Free;
  end;
  FActive := True;
end;

destructor TcsAutoDefaults.Destroy;
begin
  if FWindowHandle <> 0 then
    DeallocateHWnd(FWindowHandle);
  BumpRefCount(-1);
  inherited Destroy;
end;

procedure TcsAutoDefaults.WndProc(var Msg: TMessage);
begin
  with Msg do
    if Msg = WM_ApplyDefaults then
      try
        ApplyDefaults(TComponent(lParam));
      except
        Application.HandleException(Self);
      end
    else
      Result := DefWindowProc(FWindowHandle, Msg, wParam, lParam);
end;

procedure TcsAutoDefaults.Notification(AComponent: TComponent; Operation: TOperation);
begin
  inherited Notification(AComponent, Operation);
  if FActive and (Operation = opInsert) and (csDesigning in ComponentState) then
  begin
    { The Create method for the component which generated the
      notification won't have been completed at this point;
      post a message so that creation is completed before
      attempting to set the defaults.
    }
    if FWindowHandle <> 0 then
      PostMessage(FWindowHandle, WM_ApplyDefaults, 0, LongInt(AComponent));
  end;
end;

{ search for existing defaults for same class as component }
procedure TcsAutoDefaults.SaveAsDefault(Component: TComponent);
var Supported: Boolean;
    I: Integer;
    TestClass: TClass;
    Msg: PChar;
begin
  { check that the class of the component is supported by TcsAutoDefaults }
  Supported := True;
  for I := 0 to DefaultsTable.UnsupportedClassList.Count - 1 do
  begin
    TestClass := GetClass(DefaultsTable.UnsupportedClassList[I]);
    if (TestClass <> nil) and (Component is TestClass) then
    begin
      Msg := StrAlloc(512);
      try
        StrPCopy(Msg, 'TcsAutoDefaults cannot save ' +
                      DefaultsTable.UnsupportedClassList[I] +
                      ' components.');
        Application.MessageBox(Msg, 'TcsAutoDefaults',
                               MB_ICONINFORMATION + MB_OK);
      finally
        StrDispose(Msg);
      end;
      Supported := False;
      Break;
    end;
  end;

  if Supported then
    DefaultsTable.PutDefaultComponent(FFilename, Component);
end;

{ Return true if there are defaults for the specified component or for
  any of its ancestors' classes.
}
function TcsAutoDefaults.HasDefaultsFor(Component: TComponent): Boolean;
var I: Integer;
    Item: TDefaultsItem;
    List: TList;
    Found: Boolean;
begin
  Result := False;
  { look for defaults for class (or ancestor) of component }
  Item := DefaultsTable.GetItem(FFilename);
  if (Item <> nil) then
  begin
    List := Item.DefaultComponents;
    Found := False;
    I := 0;
    while (I < List.Count) and not Found do
    begin
      Found := (Component is TComponent(List[I]).ClassType);
      if not Found then
        Inc(I);
    end;
    Result := Found;
  end;
end;

procedure TcsAutoDefaults.SetFilename(const Value: String);
var Pos: Integer;
    NewItem: TDefaultsItem;
begin
  if (csDesigning in ComponentState) and (Value <> FFilename) then
  begin
    BumpRefCount(-1);
    FFilename := Value;
    if (FFilename <> '') then
    begin
      if DefaultsTable.FindFilename(FFilename, Pos) then
        { specified file is already in the index }
        BumpRefCount(+1)
      else
      begin
        { specified file is not in the index yet }
        NewItem := TDefaultsItem.Create;
        try
          NewItem.LoadFromFile(FFilename);
          DefaultsTable.AddItem(FFilename, NewItem);
          BumpRefCount(+1)
        except
          on EcsADStreamError do
          begin
            { LoadFromFile failed; free unused NewItem }
            NewItem.Free;
            raise;
          end;
        end;
      end;
    end;
  end;
end;

procedure TcsAutoDefaults.ApplyDefaults(Component: TComponent);
var DfltComponent: TComponent;
    Item: TDefaultsItem;
begin
  DfltComponent := DefaultsTable.GetDefaultComponent(FFilename, Component.ClassType);
  if (DfltComponent <> nil) then
  begin
    Item := DefaultsTable.GetItem(FFilename);
    CloneProperties(DfltComponent, Component, Item);
  end;
end;

procedure TcsAutoDefaults.ClearDefaults(ComponentClass: TClass);
var Item: TDefaultsItem;
    I: Integer;
    Found: Boolean;
    List: TList;
begin
  Item := DefaultsTable.GetItem(FFilename);
  if (Item <> nil) then
  begin
    List := Item.DefaultComponents;
    if ComponentClass = nil then
    begin
      { clear defaults for all classes }
      for I := 0 to List.Count - 1 do
        TComponent(List[I]).Free;
      List.Clear;
    end
    else
    begin
      Found := False;
      I := 0;
      while (I < List.Count) and not Found do
      begin
        Found := (ComponentClass = TComponent(List[I]).ClassType);
        if not Found then
          Inc(I);
      end;

      if Found then
      begin
        { remove existing defaults for the class of the component }
        TComponent(List[I]).Free;
        List.Delete(I);
      end;
    end;
    { Save changes }
    Item.SaveToFile(FFilename);
  end;
end;

{ Count how many instances of TcsAutoDefaults components are on
  the form.
}
function TcsAutoDefaults.InstancesOnForm: Integer;
var Form: TForm;
    I: Integer;
begin
  Result := 0;
  Form := TForm(Owner);
  for I := 0 to Form.ComponentCount - 1 do
    if (Form.Components[I] is TcsAutoDefaults) then Inc(Result);
end;

procedure TcsAutoDefaults.BumpRefCount(Amount: Integer);
var Item: TDefaultsItem;
begin
  Item := DefaultsTable.GetItem(FFilename);
  if (Item <> nil) then
    Item.RefCount := Item.RefCount + Amount;
end;

procedure TcsAutoDefaults.GetIgnoreOptions(List: TStrings);
var Item: TDefaultsItem;
begin
  Item := DefaultsTable.GetItem(FFilename);
  if (Item <> nil) then
    Item.GetIgnoreOptions(List);
end;

procedure TcsAutoDefaults.SetIgnoreOptions(Value: TStrings);
var Item: TDefaultsItem;
begin
  Item := DefaultsTable.GetItem(FFilename);
  if (Item <> nil) then
  begin
    Item.SetIgnoreOptions(Value);
    Item.SaveToFile(FFilename);
  end;
end;

{ Build a string list of the names of classes which have defaults. }
procedure TcsAutoDefaults.GetClassesWithDefaults(List: TStrings);
var Item: TDefaultsItem;
    I: Integer;
begin
  List.Clear;
  Item := DefaultsTable.GetItem(FFilename);
  if (Item <> nil) then
    for I := 0 to Item.DefaultComponents.Count - 1 do
      List.Add(TComponent(Item.DefaultComponents[I]).ClassName);
end;

function TcsAutoDefaults.GetShowHint: Boolean;
begin
  Result := DefaultsTable.ShowHint;
end;

procedure TcsAutoDefaults.SetShowHint(Value: Boolean);
begin
  DefaultsTable.ShowHint := Value;
end;

procedure DoneDefaults; far;
begin
  DefaultsTable.Free;
end;

initialization
  DefaultsTable := TDefaultsTable.Create;
  AddExitProc(DoneDefaults);
end.
