unit gs6_dbsy;
{-----------------------------------------------------------------------------
                          dBase III/IV File Handler

       gs6_dbsy Copyright (c) 1996 Griffin Solutions, Inc.

       Date
          4 Jun 1998

       Programmer:
          Richard F. Griffin                     tel: (912) 953-2680
          Griffin Solutions, Inc.             e-mail: grifsolu@hom.net
          102 Molded Stone Pl
          Warner Robins, GA  31088

       -------------------------------------------------------------
       This unit contains the objects to manipulate the data, index, and
       memo files that constitute a database.

   Changes:

------------------------------------------------------------------------------}
{$I gs6_flag.pas}
interface
uses
   Windows,
   SysUtils,
   gs6_eror,
   gs6_cnst,
   gs6_date,
   gs6_tool,
   gs6_sort,
   gs6_disk,
   gs6_dbf,
   gs6_indx,
   {$IFDEF FOXOK}
   gs6_cdx,
   gs6_mix,
   {$ENDIF}
   {$IFDEF DBASE4OK}
   gs6_mdx,
   {$ENDIF}
   {$IFDEF DBASE3OK}
   gs6_ndx,
   {$ENDIF}
   {$IFDEF CLIPOK}
   gs6_ntx,
   {$ENDIF}
   gs6_Memo,
   gs6_sql,
   gs6_glbl;

{private}

const
   IndexesAvail    = 63;
   DBFCacheSize    : word = 32768;

type

   GSobjSortDB = class;

   GSO_dBHandler = class(GSO_dBaseFld)
      IndexMaster : GSobjIndexTag;
      PrimaryTagName: array[0..15] of char;
      IndexStack  : array[0..IndexesAvail] of GSobjIndexFile;
      CacheFirst  : Longint;
      CacheLast   : Longint;
      CachePtr    : GSptrByteArray;
      CacheRecs   : integer;
      CacheSize   : LongInt;
      CacheRead   : boolean;
      CacheAllowed: boolean;
      gsvFound    : boolean;
      gsvFindNear : boolean;
      gsvRecRead  : boolean;
      DoingSort   : boolean;
      curMemo     : GSobjMemo;
      newMemo     : GSobjMemo;
      ResyncIndex : boolean;
      ActiveIndexChg: boolean;
      FFilterSet  : TgsHugeSet;
      FFilterHndlr: TgsExpHandler;
      FFilterExpr : string;
      FFilterLive : boolean;
      FFilterOn   : boolean;
      SearchMode  : word; {Mode flags for gsSearchDBF.  Default value is zero}
                          {Values are summed together for multiple conditions}
                          {1=Case Insensitive; 2=Use Exact; 4=Use Wildcards  }
                          {8=Use Filter; 16=Use Index; 32=Match at Start Only}

      constructor Create(const FName, APassword: String; ReadWrite, Shared: boolean);
      destructor  Destroy; override;
      procedure   gsAppend; override;
      procedure   gsCopyFile(const filname, apassword: String);
      procedure   gsCopyFromIndex(ixColl: GSobjSortDB; const filname, apassword: String);
      procedure   gsCopyMemoRecord(df : GSO_dBHandler);
      procedure   gsCopyRecord(filobj: GSO_dBHandler);
      procedure   gsCopyStructure(const filname: String);
      function    gsDBFEvent(Event: dbFileEvent; Action: longint): boolean; override;
      Function    gsFind(const st : String) : boolean; virtual;
      function    gsGetKey(RecNum : Longint; var keyst: String): longint;
      Procedure   gsGetRec(RecNum : LongInt); override;
      Function    gsHuntDuplicate(const st, ky: String) : longint; virtual;
      Function    gsIndex(const INames, Tag: String): integer;   {!!RFG 091297}
      Procedure   gsIndexClear;
      function    gsIndexFileExtend(const IName: string): string;
      function    gsIndexFileIsOpen(const IName: string): integer;
      function    gsIndexFileRemove(const IName: String): integer;
      function    gsIndexFileKill(const IName: String): integer;
      function    gsIndexKeyValue(Value: integer; ExpVar: TgsVariant): boolean;
      function    gsIndexTagRemove(const IName, Tag: String): integer;
      procedure   gsIndexOrder(Value: integer);
      function    gsIndexPointer(Value: integer): GSobjIndexTag;
      function    gsIndexRoute(const IName: String): integer;
      function    gsIndexInsert(ix : GSobjIndexFile) : integer;
      Function    gsIndexTo(const IName, tag, keyexpr, forexpr: String;
                          uniq: GSsetIndexUnique;
                          ascnd: GSsetSortStatus): integer;
      Procedure   gsLoadToIndex(ixColl: GSobjSortDB; zfld: PChar);
      Function    gsMemoryIndexAdd(const tag, keyexpr, forexpr: String;
                     uniq: GSsetIndexUnique; ascnd: GSsetSortStatus): boolean;
      Function    gsSetTagTo(const TName: String; SameRec: boolean): integer;
      Procedure   gsPack;
      Procedure   gsPutRec(RecNum : LongInt); override;
      Procedure   gsRefresh;
      Procedure   gsReIndex;
      Function    gsRename(const NewName: string): boolean; override;
      function    gsRead(blk: gsuint32; var dat; len: gsuint32): integer; override;
      procedure   gsSetDBFCacheAllowed(tf: boolean);
      procedure   gsSetDBFCache(tf: boolean);
      procedure   gsResetRange;
      procedure   gsRefreshFilter;
      procedure   gsSetFilterExpr(const FiltExpr: string; Insensitive, Wildcards: boolean);
      procedure   gsSetFilterActive(Value: boolean);
      Procedure   gsSetRange(const RLo, RHi: String; LoIn, HiIn, Partial: boolean);
      procedure   gsSetRangeEmpty;
      procedure   gsSkip(RecCnt : LongInt); virtual;
      procedure   gsSortFile(const filname, apassword, zfld: string; isascend : GSsetSortStatus);
      function    gsTestFilter : boolean; virtual;
      Function    gsWithIndex: boolean; override;
      Procedure   gsWrite(blk: gsuint32; var dat; len: gsuint32); override;
      Procedure   gsZap;
      Function    gsSearchDBF(const s: String; var FNum : word;
                            var fromrec: longint; torec: longint): word;
      Procedure   gsSetLockProtocol(LokProtocol: GSsetLokProtocol); override;
      Function    gsLockIndexes: boolean; virtual;
      Function    gsUnLockIndexes: boolean;
      Procedure   gsSetPassword(const APassword: string); virtual;
      Procedure   gsRewriteMemoRecord;
   end;

   GSobjSortDB = class(TgsSort)
      curFile: GSO_dBHandler;
      newFile: GSO_dBHandler;
      KeyCnt: longint;
      procedure    OutputWord;
   end;

implementation

const
   {$IFDEF DBASE3OK}
      IndexExt : string[4] = '.NDX';
   {$ELSE}
      {$IFDEF DBASE4OK}
         IndexExt : string[4] = '.MDX';
      {$ELSE}
         {$IFDEF FOXOK}
            IndexExt : string[4]= '.CDX';
         {$ELSE}
            {$IFDEF CLIPOK}
                IndexExt : string[4] = '.NTX';
            {$ELSE}
                IndexExt : string[4] = '.NDX';
            {$ENDIF}
         {$ENDIF}
      {$ENDIF}
   {$ENDIF}

constructor GSO_dBHandler.Create(const FName, APassword: String; ReadWrite, Shared: boolean);
var
   i : integer;
begin
   inherited Create(FName, APassword, ReadWrite, Shared);
   CacheRead := false;
   CacheAllowed := true;
   CachePtr := nil;
   CacheFirst := -1;
   CacheLast := 0;
   gsvFound := false;
   gsvFindNear := false;
   gsvRecRead := false;
   IndexMaster := nil;
   PrimaryTagName[0] := #0;
   for i := 1 to IndexesAvail do IndexStack[i] := nil;
   DoingSort := false;
   ResyncIndex := false;
   SearchMode := 1;
   FFilterSet := nil;
   FFilterHndlr := nil;
   FFilterLive := false;
   FFilterOn := false;
   FFilterExpr := '';
end;

destructor GSO_dBHandler.Destroy;
var
   i : integer;
begin
   if FFilterSet <> nil then FFilterSet.Free;
   if FFilterHndlr <> nil then FFilterHndlr.Free;
   for i := 1 to IndexesAvail do
      if IndexStack[i] <> nil then
      begin
         IndexStack[i].Free;
         IndexStack[i] := nil;
      end;
   IndexMaster := nil;               {Set index active flag to false}
   if CachePtr <> nil then FreeMem(CachePtr, CacheSize);
   CachePtr := nil;
   CacheSize := 0;
   inherited Destroy;
end;

{------------------------------------------------------------------------------
                              Record Processing
------------------------------------------------------------------------------}

Function GSO_dBHandler.gsLockIndexes: boolean;
VAR
   i: integer;
   j: integer;
   r: boolean;
BEGIN
   gsLockIndexes := false;
   for i := 1 to IndexesAvail do
      if (IndexStack[i] <> nil) and (IndexStack[i].DiskFile <> nil) then
      begin
         r := IndexStack[i].IndexLock;
         if not r then
         begin
            for j := 1 to i-1 do
               if IndexStack[j] <> nil then
               begin
                  IndexStack[j].DiskFile.gsUnLock;
               end;
            SetLastError(dosLockViolated);
            exit;
         end;
      end;
   gsLockIndexes := true;
end;

Function GSO_dBHandler.gsUnlockIndexes: boolean;
VAR
   i: integer;
BEGIN
   for i := 1 to IndexesAvail do
      if (IndexStack[i] <> nil) and (IndexStack[i].DiskFile <> nil) then
      begin
         IndexStack[i].DiskFile.gsUnLock;
      end;
   gsUnlockIndexes := true;
end;

Procedure GSO_dBHandler.gsAppend;
BEGIN
   if gsLockIndexes then
   begin
      try
         inherited gsAppend;
      finally
         gsUnlockIndexes;
      end;
   end;
end;

function GSO_dBHandler.gsDBFEvent(Event: dbFileEvent; Action: longint): boolean;
var
   i: integer;
   q: boolean;
   r: integer;
begin
   gsDBFEvent := inherited gsDBFEvent(Event, Action);
   case Event of
      dbPostWrite     : begin
                           if (FFilterHndlr <> nil) and (FFilterSet <> nil) then
                           begin
                              FFilterHndlr.ExpressionResult(@q);
                              if q then
                                 r := 1
                              else
                                 r := 0;
                              FFilterSet.ChangeBit(RecNumber,r);
                           end;
                           ActiveIndexChg := false;
                           for i := 1 to IndexesAvail do
                           begin
                              if (IndexStack[i] <> nil) then
                              begin
                                 IndexStack[i].TagUpdate(RecNumber,Action = 0);
                              end;
                           end;
                           ResyncIndex := false;
                           if IndexMaster <> nil then
                           begin
                              ActiveIndexChg := IndexMaster.KeyUpdated;
                              ResyncIndex := IndexMaster.KeySync(RecNumber) <> RecNumber;
                              File_TOF := IndexMaster.TagBOF;
                              File_EOF := IndexMaster.TagEOF;
                           end;
                           if FFilterOn and (not ResyncIndex) then
                           begin
                              ResyncIndex := FFilterSet.BitValue(RecNumber);
                           end;
                           if (IndexMaster <> nil) and (not FFilterOn) then
                           begin
                              File_EOF := (Action = 0) or   {!!RFG 082097}
                                          (Action = NumRecs);
                              File_TOF := Action = 1;   {!!RFG 082097}
                           end;
                        end;
      dbFlush         : begin
                           for i := 1 to IndexesAvail do
                              if (IndexStack[i] <> nil) and
                                 (IndexStack[i].DiskFile <> nil) then
                                 IndexStack[i].DiskFile.gsFlush;
                        end;
   end;
end;

Function GSO_dBHandler.gsFind(const st : String) : boolean;
var
   RNum   : longint;
   ps: GSobjIndexKeyData;
   dv: longint;
begin
   gsvFound := false;
   gsFind := gsvFound;
   if NumRecs = 0 then exit;                    {!!RFG 022198}
   if dfLockStyle = ClipLock then
   begin
      if not gsLockIndexes then exit;
   end;
   if (IndexMaster <> nil) then
   begin
      ps := GSobjIndexKeyData.Create(256);
      ps.PutString(st);
      ps.Tag := IgnoreRecNum;
      RNum := IndexMaster.KeyFind(ps);
      if RNum > 0 then                {RNum = 0 if no match, otherwise}
                                      {it holds the valid record number}
      begin
         gsvFound := false;
         inherited gsGetRec(RNum);    {If match found, read the record}
         while ((not gsTestFilter) or
               (gsDelFlag and (not UseDeletedRec))) and
               (not File_EOF) do gsSkip(1);
         if not File_EOF then
         begin
            with IndexMaster do
            begin
               AdjustValue(ps,true);
               gsvFound := ps.Compare(CurKeyInfo,dv,MatchIsExact,Owner.Owner.CollateTable) = 0;
               if not gsvFound then
                  if (not gsvExactMatch) and (dv > ps.SizeOfVar) then   {!RFG 080597}
                     gsvFound := true;  {Non-Exact match test}
            end;
            if (not gsvFound) and (not gsvFindNear) then
            begin
               gsGetRec(Bttm_Record);
               File_EOF := True;
            end;
         end;
      end else
      begin                           {If no matching index key, then}
         gsvFound := False;              {Set Match Found Flag False}
         if (IndexMaster.TagEOF) or (not gsvFindNear) then
         begin
            gsGetRec(Bttm_Record);
            File_EOF := True;
         end
         else
         begin
            RNum := IndexMaster.KeyRead(-5);  {Read current index pos}
            gsGetRec(RNum);                        {read the record}
         end;
      end;
      ps.Free;
   end else                           {If there is no index file, then}
   begin
      gsvFound := False;                 {Set Match Found Flag False}
      gsGetRec(Bttm_Record);
      File_EOF := True;
   end;
   if dfLockStyle = ClipLock then
   begin
      gsUnlockIndexes;
   end;
   gsFind := gsvFound;
end;                  {Find}

Function GSO_dBHandler.gsHuntDuplicate(const st, ky: String) : longint;
var
   im: GSobjIndexTag;
   ps: GSobjIndexKeyData;
   i: integer;
   pc: array[0..79] of char;
   em: boolean;
begin                                 {!!RFG 082097}
   gsHuntDuplicate := -1;
   if dfLockStyle = ClipLock then
   begin
      if not gsLockIndexes then exit;
   end;
   im := nil;
   StrPCopy(pc,st);
   i := 0;
   while (i < IndexesAvail) and (im = nil) do
   begin
      inc(i);
      if IndexStack[i] <> nil then
         im := IndexStack[i].TagByName(pc);
   end;
   if (im <> nil) then
   begin
      ps := GSobjIndexKeyData.Create(256);
      ps.PutString(ky);
      ps.Tag := IgnoreRecNum;
      em := gsvExactMatch;
      gsvExactMatch := true;
      gsHuntDuplicate := im.HuntDuplicate(ps);
      gsvExactMatch := em;
      ps.Free;
   end;
   if dfLockStyle = ClipLock then
   begin
      gsUnlockIndexes;
   end;
end;

function GSO_dBHandler.gsGetKey(RecNum : Longint; var keyst: String): longint;
begin
   if IndexMaster <> nil then
   begin
      gsGetKey := IndexMaster.KeyRead(RecNum);
      keyst := IndexMaster.CurKeyInfo.GetString;
   end
   else
   begin
      keyst := '';
      gsGetKey := 0;
   end;
end;

procedure GSO_dBHandler.gsGetRec(RecNum : LongInt);
var
   inum  : longint;
   rnum  : longint;
   knum  : longint;
   cread : boolean;
   okread: boolean;
begin
   ResyncIndex := false;
   if dState <> dbBrowse then
   begin
      inherited gsGetRec(RecNum);
      exit;
   end;
   gsvRecRead := false;
{
   if dfLockStyle = ClipLock then
   begin
      if not gsLockIndexes then exit;
   end;
}
   inum := 0;
   cread := CacheRead;
   okread := false;
   File_EOF := false;
   File_TOF := false;
   rnum := RecNum;
   knum := RecNum;
   if (knum = Top_Record) or (knum = Same_Record) then knum := Next_Record
      else if knum = Bttm_Record then knum := Prev_Record;
   repeat
      if (IndexMaster <> nil) and (knum < 0) then
      begin
         repeat
            CacheRead := false;
            rnum := IndexMaster.KeyRead(rnum);
            inum := rnum;
            File_EOF := IndexMaster.TagEOF;
            File_TOF := IndexMaster.TagBOF;
            if FFilterOn then
            begin
               okread := FFilterSet.BitValue(rnum);
               if not okread then rnum := knum;
            end
            else
               okread := true;
         until okread or File_EOF or File_TOF;
      end
      else
      if FFilterOn and (rnum < 0) then
      begin
         case rnum of
            Top_Record   : begin
                              File_TOF := FFilterSet.Highest < 1;
                              if not File_TOF then
                              begin
                                 rnum := 1;
                                 repeat
                                    okread := FFilterSet.BitValue(rnum);
                                    if not okread then inc(rnum);
                                 until okread or (rnum > FFilterSet.Highest);
                                 if not okread then File_EOF := true;
                              end
                              else
                                 File_EOF := true;
                           end;
            Next_Record  : begin
                              File_TOF := FFilterSet.Highest < 1;
                              if not File_TOF then
                              begin
                                 rnum := RecNumber+1;
                                 repeat
                                    okread := FFilterSet.BitValue(rnum);
                                    if not okread then inc(rnum);
                                 until okread or (rnum > FFilterSet.Highest);
                                 if not okread then File_EOF := true;
                              end
                              else
                                 File_EOF := true;
                           end;
            Prev_Record  : begin
                              File_TOF := FFilterSet.Highest < 1;
                              if not File_TOF then
                              begin
                                 rnum := RecNumber-1;
                                 okread := false;
                                 while (rnum > 0) and (not okread) do
                                 begin                                  {%FIX 00001}
                                    okread := FFilterSet.BitValue(rnum);
                                    if not okread then dec(rnum);
                                 end;;
                                 if not okread then File_TOF := true;
                              end
                              else
                                 File_EOF := true;
                           end;
            Bttm_Record  : begin
                              File_TOF := FFilterSet.Highest < 1;
                              if not File_TOF then
                              begin
                                 rnum := FFilterSet.Highest;
                                 repeat
                                    okread := FFilterSet.BitValue(rnum);
                                    if not okread then dec(rnum);
                                 until okread or (rnum = 0);
                                 if not okread then File_TOF := true;
                              end
                              else
                                 File_EOF := true;
                           end;
         end;
         inum := rnum;
      end;
      if (not File_EOF) and (not File_TOF) then   {Destroy if EOF reached}
      begin
         inherited gsGetRec(rnum);
         okread := not((rnum = 0) or (rnum > numrecs));
         if not okread then
         begin
            gsBlank;
            exit;
         end;
         if knum < 0 then                   {ignore if physical record access}
            okread := (not (gsDelFlag and (not UseDeletedRec))) and
                       gsTestFilter;
         rnum := knum;
      end;
   until okread or File_EOF or File_TOF;
   CacheRead := cread;
(*
   if File_TOF then
   begin
      if (RecNum <> Top_Record) and (RecNum <> Bttm_Record) then
         gsGetRec(Top_Record);   {Recursion for first filtered record}
      if File_TOF then
         gsBlank;
      File_TOF := True;
   end
   else
   if File_EOF then
   begin
      if (RecNum <> Top_Record) and (RecNum <> Bttm_Record) and
         (not gsvIndexState) then                               {!!RFG 083097}
         gsGetRec(Bttm_Record);  {Recursion for last filtered record}
      if File_EOF then
         gsBlank;
      File_EOF := True;
   end;
*)
   gsvRecRead := not (File_EOF or File_TOF);
   if IndexMaster <> nil then     {Resync index if necesary}
   begin
      if (RecNumber <> inum) and (gsvRecRead) then
         ResyncIndex := IndexMaster.KeySync(RecNumber) <> RecNumber;
   end;
   if FFilterOn and (not ResyncIndex) then
   begin
      if (RecNumber <> inum) and (gsvRecRead) then
         ResyncIndex := FFilterSet.BitValue(RecNumber);
   end;
   if dfLockStyle = ClipLock then
   begin
      gsUnlockIndexes;
   end;
end;

Procedure GSO_dBHandler.gsPutRec(RecNum : LongInt);
begin
   ResyncIndex := false;
   if gsLockIndexes then
   begin
      inherited gsPutRec(RecNum);
      gsUnlockIndexes;
   end;
end;

function GSO_dBHandler.gsRead(blk: gsuint32; var dat; len: gsuint32): integer;
var
   goodrec: integer;
   blkpsn: integer;
begin
   blkpsn := blk;
   if (not CacheRead) or (blkpsn < HeadLen) then
      gsRead := inherited gsRead(blk,dat,len)
   else
   begin
      if (CacheFirst = -1) or
         (blkpsn < CacheFirst) or
         (blkpsn >= CacheLast-RecLen) then
      begin
         goodrec := inherited gsRead(blk,CachePtr^,CacheSize);
         FileError := 0;
         CacheFirst := blk;
         CacheLast := (blkpsn + goodrec);
      end;
      if blkpsn >= CacheLast then goodrec := 0
      else
      begin
         goodrec := RecLen;
         if DoingSort then
            CurRecord := @CachePtr^[blkpsn - CacheFirst]
         else
            Move(CachePtr^[blkpsn - CacheFirst],dat,RecLen);
      end;
      gsRead := goodrec;
   end;
end;

procedure GSO_dBHandler.gsRefresh;
begin
   gsGetRec(Same_Record);
end;

Function GSO_dBHandler.gsRename(const NewName: string): boolean;
var
   i: integer;
   dbnamestring: string;
   ixnamestring: string;
begin
   dbnamestring := UpperCase(ExtractFileNameOnly(StrPas(FileName)));
   inherited gsRename(NewName);
   for i := 1 to IndexesAvail do
   begin
      if (IndexStack[i] <> nil) and (IndexStack[i].DiskFile <> nil) then
      begin
         ixnamestring := UpperCase(ExtractFileNameOnly(StrPas(IndexStack[i].DiskFile.FileName)));
         if dbnamestring = ixnamestring then
            IndexStack[i].Rename(NewName);
      end;
   end;
   Result := true;
end;

Procedure GSO_dBHandler.gsSetDBFCacheAllowed(tf: boolean);
begin
   CacheAllowed := tf;
   if not tf then
      gsSetDBFCache(false);
end;

Procedure GSO_dBHandler.gsSetDBFCache(tf: boolean);
begin
   if not CacheAllowed then tf := false;
   if tf and CacheRead then exit;
   CacheRead := tf;
   if not tf then
   begin
      if CachePtr <> nil then FreeMem(CachePtr, CacheSize);
      CachePtr := nil;
      CacheSize := 0;
   end
   else
   begin
      {$IFDEF WIN32}
         CacheSize := GetFreeSpace(0);
      {$ELSE}
         CacheSize := MemAvail;
      {$ENDIF}
      if CacheSize > DBFCacheSize then
         CacheSize := DBFCacheSize
      else CacheSize := CacheSize - 16384;
      CacheSize := CacheSize - (CacheSize mod RecLen);
      if CacheSize < RecLen then CacheSize := RecLen;
      GetMem(CachePtr, CacheSize);
      CacheFirst := -1;
      CacheRecs := CacheSize div RecLen;
   end;
end;

Procedure GSO_dBHandler.gsResetRange;
begin
   if IndexMaster = nil then exit;
   IndexMaster.SetRange(nil,false,nil,false,false);
end;

Procedure GSO_dBHandler.gsSetLockProtocol(LokProtocol: GSsetLokProtocol);
var
   i: integer;
begin
   inherited gsSetLockProtocol(LokProtocol);
   for i := 1 to IndexesAvail do
      if (IndexStack[i] <> nil) and (IndexStack[i].DiskFile <> nil) then
          IndexStack[i].ixSetLockProtocol(LokProtocol);
end;

procedure GSO_dBHandler.gsRefreshFilter;
var
   rn: longint;
   ix: gsObjIndexTag;
   cr: boolean;
   qb: boolean;
begin
   if FFilterOn then
   begin
      FFilterOn := false;
      FFilterSet.Clear;
      rn := RecNumber;
      ix := IndexMaster;
      cr := CacheRead;
      if (ix = nil) then
      begin
         gsSetDBFCache(true);
      end;
      gsGetRec(Top_Record);
      while not File_EOF do
      begin
         FFilterHndlr.ExpressionResult(@qb);
         if qb then
             FFilterSet.ChangeBit(RecNumber,1);
         gsGetRec(Next_Record);
      end;
      gsSetDBFCache(cr);
      IndexMaster := ix;
      gsGetRec(rn);
      gsSetFilterActive(true);
   end;
end;

procedure GSO_dBHandler.gsSetFilterExpr(const FiltExpr: string; Insensitive, Wildcards: boolean);
var
   fa: boolean;
   rn: longint;
   ix: gsObjIndexTag;
   cr: boolean;
   qb: boolean;
   bf: array[0..511] of byte;
begin
   fa := FFilterLive;
   FFilterOn := false;
   if FFilterSet = nil then
      FFilterSet := TgsHugeSet.Create;
   if FFilterHndlr = nil then
      FFilterHndlr := TgsExpHandler.Create(DBFExpLink,nil,false);
   if (FiltExpr <> FFilterExpr) or (FFilterHndlr.CaseInsensitive <> Insensitive) or
      (FFilterHndlr.UseWildCards <> WildCards) then
   begin
      FFilterExpr := FiltExpr;
      FFilterSet.Clear;
      FFilterHndlr.CaseInsensitive := Insensitive;
      FFilterHndlr.UseWildCards := WildCards;
      FFilterHndlr.Expression := FiltExpr;
      FFilterHndlr.ExpressionResult(@bf);   {Execute expression to get result type}
      if FFilterHndlr.ResultType <> rtBoolean then
         raise EHalcyonError.Create(gsErrFilterExpression);
      rn := RecNumber;
      ix := IndexMaster;
      cr := CacheRead;
      if (ix = nil) then
      begin
         gsSetDBFCache(true);
      end;
      gsGetRec(Top_Record);
      while not File_EOF do
      begin
         FFilterHndlr.ExpressionResult(@qb);
         if qb then
             FFilterSet.ChangeBit(RecNumber,1);
         gsGetRec(Next_Record);
      end;
      gsSetDBFCache(cr);
      IndexMaster := ix;
      gsGetRec(rn);
   end;
   gsSetFilterActive(fa);
end;

procedure GSO_dBHandler.gsSetFilterActive(Value: boolean);
begin
   FFilterLive := Value;
   FFilterOn := FFilterLive and (FFilterHndlr <> nil) and (FFilterHndlr.ArgCount > 0);
end;

Procedure GSO_dBHandler.gsSetRange(const RLo, RHi: String; LoIn, HiIn, Partial: boolean);
var
   s1: TgsVariant;
   s2: TgsVariant;
begin
   if IndexMaster = nil then exit;
   s1 := TgsVariant.Create(256);
   s2 := TgsVariant.Create(256);
   s1.PutString(RLo);
   s2.PutString(RHi);
   IndexMaster.SetRange(s1,LoIn,s2,HiIn, Partial);
   s2.Free;
   s1.Free;
   gsGetRec(Top_Record);
   if File_TOF then gsBlank;
end;

Procedure GSO_dBHandler.gsSetRangeEmpty;
begin
   if IndexMaster = nil then exit;
   IndexMaster.SetRangeEmpty;
   gsGetRec(Top_Record);
   if File_TOF then gsBlank;
end;

PROCEDURE GSO_dBHandler.gsSkip(RecCnt : LongInt);
VAR
   i  : integer;
   rn : longint;
   de : longint;
   dr : longint;
   rl : longint;
   rc : longint;
   im : pointer;

   procedure SkipFromDBF;
   begin
      rl := Recnumber + RecCnt;
      rc := rl;
      if (rl > NumRecs) then
      begin                         {flag out of file range}
         rc := 0;
         rl := NumRecs;
      end
      else
      if (rl < 1) then
      begin
         rc := 0;                   {flag out of file range}
         rl := 1;
      end;
   end;

   procedure SkipFromIndex;
   begin
      i := 1;
      repeat
         rc := IndexMaster.KeyRead(dr);
         if rc > 0 then
            rl := rc
         else
            rl := IndexMaster.KeyRead(de);
         inc(i);
      until (i > rn) or (rc = 0);
   end;


begin
   If RecCnt <> 0 then
   begin
      if RecCnt < 0 then de := Top_Record else de := Bttm_Record;
      rl := RecNumber;
      rn := abs(RecCnt);
      if RecCnt > 0 then dr := Next_Record else dr := Prev_Record;
      if (not FFilterLive) and UseDeletedRec then
      begin                                  {do fast skip}
         if (IndexMaster <> nil) then
         begin
            SkipFromIndex;
            if rl <> 0 then
            begin
               im := IndexMaster;
               IndexMaster := nil;
               gsGetRec(rl);
               IndexMaster := im;
            end;
         end
         else
         begin
            SkipFromDBF;
            gsGetRec(rl);
         end;
         if rc = 0 then
         begin
            File_EOF := de = Bttm_Record;
            File_TOF := de = Top_Record;
         end;
      end
      else
      begin
         repeat
            gsGetRec(dr);
            dec(rn);
         until (rn = 0) or File_EOF or File_TOF;
      end;
   end
   else gsGetRec(Same_Record);
end;

function GSO_dBHandler.gsTestFilter: boolean;
begin
   gsTestFilter := true;
end;

Function  GSO_dBHandler.gsWithIndex: boolean;
begin
   gsWithIndex := IndexStack[1] <> nil;
end;

Procedure GSO_dBHandler.gsWrite(blk: gsuint32; var dat; len: gsuint32);
var
   blkpsn: integer;
begin
   blkpsn := blk;
   inherited gsWrite(blk,dat,len);
   if blk = 0 then exit;
   if (CacheRead) then
   begin
      if (CacheFirst = -1) or
         (blkpsn < CacheFirst) or
         (blkpsn >= CacheLast-RecLen) then
      begin
      end
      else
         Move(dat,CachePtr^[blkpsn - CacheFirst],len);
   end;
end;

Procedure GSO_dBHandler.gsSetPassword(const APassword: string);
var
   rn: longint;
   ix: gsObjIndexTag;
   cr: boolean;
   fl: integer;
   flds: PByteArray;
   i: integer;
   TmpState: dbFileState;
begin
   TmpState := dState;
   if FileShared then
   begin
      gsTestForOk(dosAccessDenied, dbsPackError);
      exit;
   end;
   if gsvIsEncrypted and (APassword = gsvPasswordIn) then exit;
   if (not gsvIsEncrypted) and (APassword='') then exit;
 try
   dState := dbCopy;
   if not gsvIsEncrypted then gsvPasswordIn := '';
   gsvPasswordOut := APassword;
   gsvIsEncrypted := true;
   if WithMemo then
   begin
      curMemo := gsMemoObject;
      if CurMemo <> nil then
      begin
         curMemo.IsEncrypted := true;
         curMemo.PasswordIn := gsvPasswordIn;
         curMemo.PasswordOut := gsvPasswordOut;
      end;
   end;
   rn := RecNumber;
   ix := IndexMaster;
   IndexMaster := nil;
   cr := CacheRead;
   gsSetDBFCache(true);
   gsGetRec(Top_Record);
   while not File_EOF do
   begin
      if WithMemo then gsRewriteMemoRecord;
      gsPutRec(RecNumber);
      gsGetRec(Next_Record);
   end;
   fl := HeadLen-33;               {Size of field descriptors}
   if FileVers = VFP3File then
      fl := fl - 263;
   GetMem(flds, fl);             {Allocate memory for fields buffer.}
   DBEncryption(gsvPasswordOut,PByteArray(Fields),flds,1,fl);
   inherited gsWrite(32, flds^, fl);          {Store field data}
   FreeMem(flds, fl);             {Allocate memory for fields buffer.}
   gsSetDBFCache(cr);
   IndexMaster := ix;
   for i := 1 to IndexesAvail do
   begin
      if IndexStack[i] <> nil then
         IndexStack[i].ixEncrypt;
   end;
   gsvPasswordIn := APassword;
   gsvIsEncrypted := APassword <> '';
   if WithMemo and (CurMemo <> nil) then
   begin
      curMemo.IsEncrypted := gsvIsEncrypted;
      curMemo.PasswordIn := gsvPasswordIn;
   end;
   dStatus := Updated;
   gsHdrWrite;
   gsGetRec(rn);
 finally
   dState := TmpState;
 end;  
end;

procedure GSO_dBHandler.gsRewriteMemoRecord;
var
   fp     : integer;
   mbuf   : PChar;
   vcnt   : longint;
   rcnt   : longint;
   blk    : longint;
begin
   for fp := 1 to NumFields do
   begin
      if Fields^[fp].dbFieldType in ['B','G','M'] then
      begin
         blk := Trunc(gsNumberGetN(fp));
         if (blk <> 0) then
         begin
            vcnt := curMemo.moMemoSize(blk) + 16;
            rcnt := vcnt;
            GetMem(mbuf, vcnt);
            curMemo.moMemoRead(mbuf, blk, rcnt);
            curMemo.moMemoWrite(mbuf, blk, rcnt);
            FreeMem(mbuf, vcnt);
         end;
      end;
   end;
end;

{------------------------------------------------------------------------------
                              Index Processing
------------------------------------------------------------------------------}

function GSO_dBHandler.gsIndex(const INames, Tag: String): integer;  {!!RFG 091297}
var
   NameList: PChar;
   IName: PChar;
   NameListBegin: PChar;
   NameListEnd: PChar;
   NoLongName: boolean;
   rsl: integer;
begin
   rsl := 1;
   gsIndex := rsl;                                         {!!RFG 091297}
   gsIndexClear;
   if INames = '' then exit;
   GetMem(NameList,260);
   GetMem(IName,260);
   StrPCopy(NameList,INames);
   NameListBegin := NameList;
   NameListEnd := StrEnd(NameList);
   dec(NameListEnd);
   while (NameListEnd[0] in [' ',',',';']) and (NameListEnd >= NameListBegin) do
   begin
      NameListEnd[0] := #0;
      dec(NameListEnd);
   end;
   if StrLen(NameListBegin) > 0 then
   begin
      NameListEnd := NameListBegin;
      NoLongName := true;
      while NameListEnd[0] <> #0 do
      begin
         if NameListEnd[0] = '"' then NoLongName := not NoLongName;
         if NoLongName and (NameListEnd[0] in [' ',',',';']) then
            NameListEnd[0] := #9;
         inc(NameListEnd);
      end;
      repeat
         while NameListBegin[0] in [#9,'"'] do inc(NameListBegin);
         NameListEnd := NameListBegin;
         while not (NameListEnd[0] in [#9,'"',#0]) do inc(NameListEnd);
         StrLCopy(IName,NameListBegin,NameListEnd-NameListBegin);
         if StrLen(NameListBegin) > 0 then
            rsl := gsIndexRoute(StrPas(IName));    {!!RFG 091297}
         NameListBegin := NameListEnd;
      until (rsl <> 0) or (NameListEnd[0] = #0);   {!!RFG 091297}
      if rsl = 0 then                              {!!RFG 091297}
         gsSetTagTo(Tag,true);
   end;
   gsIndex := rsl;                                 {!!RFG 091297}
   FreeMem(NameList,260);
   FreeMem(IName,260);
end;

function GSO_dBHandler.gsIndexFileExtend(const IName: string): string;
var
   IFile: String;
begin
   IFile := AnsiUpperCase(IName);
   if ExtractFilePath(IFile) = '' then
      IFile := ExtractFilePath(StrPas(FileName))+ IFile;
   gsIndexFileExtend := ChangeFileExtEmpty(IFile,IndexExt);
end;

function GSO_dBHandler.gsIndexFileIsOpen(const IName: string): integer;
var
   i: integer;
begin
   gsIndexFileIsOpen := 0;;
   for i := 1 to IndexesAvail do
   begin
      if IndexStack[i] <> nil then
         if IndexStack[i].IsFileName(IName) then
            gsIndexFileIsOpen := i;
   end;
end;

function GSO_dBHandler.gsIndexFileRemove(const IName: String): integer;
var
   iz: integer;
begin
   gsIndexFileRemove := 1;
   if IName = '' then exit;
   iz := gsIndexFileIsOpen(IName);
   if iz = 0 then exit;
   if (IndexMaster <> nil) and (IndexMaster.Owner = IndexStack[iz]) then
      gsSetTagTo('',true);
   IndexStack[iz].Free;
   IndexStack[iz] := nil;
   gsIndexFileRemove := 0;
end;

function GSO_dBHandler.gsIndexFileKill(const IName: String): integer;  {!!RFG 091397}
var
   iz: integer;
   b: boolean;
   pc: array[0..MAX_PATH] of char;
begin
   gsIndexFileKill := 1;
   if IName = '' then exit;
   StrPCopy(pc,IName);
   if  GetFileAttributes(pc) = $FFFFFFFF then exit;
   iz := gsIndexFileIsOpen(IName);
   if iz > 0 then
   begin
      if (IndexMaster <> nil) and (IndexMaster.Owner = IndexStack[iz]) then
         gsSetTagTo('',true);
      IndexStack[iz].Free;
      IndexStack[iz] := nil;
   end;
   if not DeleteFile(IName) then
   begin
      iz := FileOpen(IName,fmOpenReadWrite+fmShareDenyNone);
      if iz > 0 then
      begin
         b := false;
         if FileSeek(iz, 0, 0) <> -1 then
             b := SetEndOfFile(iz);
         FileClose(iz);
         if not b then exit;
      end
      else exit;
   end;
   gsIndexFileKill := 0;
end;

function GSO_dBHandler.gsIndexKeyValue(Value: integer; ExpVar: TgsVariant): boolean;
var
   p: GSobjIndexTag;
begin
   p := gsIndexPointer(Value);
   Result := p <> nil;
   if Result then
   begin
      p.ExprHandlr.ExpressionAsVariant(expvar);
   end;
end;

Procedure GSO_dBHandler.gsIndexOrder(Value: integer);
var
   p: GSobjIndexTag;
begin
   p := gsIndexPointer(Value);
   if p <> nil then
      gsSetTagTo(StrPas(p.TagName),true)
   else
      gsSetTagTo('',true);
end;

Function GSO_dBHandler.gsIndexPointer(Value: integer): GSobjIndexTag;
var
   i: integer;
   n: integer;
   n1: integer;
   p: GSobjIndexTag;
begin
   p := nil;
   if Value > 0 then
   begin
      n := 0;
      n1 := 0;
      i := 1;
      while (n < Value) and (i <= IndexesAvail) do
      begin
         if IndexStack[i] <> nil then
         begin
            n1 := n1 + IndexStack[i].TagCount;
            if n1 >= Value then
            begin
               n1 := Value - n;
               n := Value;
               p := IndexStack[i].TagByNumber(pred(n1));
            end
            else
               n := n1;
         end;
         inc(i);
      end;
   end;
   gsIndexPointer := p;
end;

function GSO_dBHandler.gsIndexTagRemove(const IName, Tag: String): integer;
var
   i   : integer;                     {Local working variable  }
   pc: array [0..16] of char;
   px: GSobjIndexTag;
   ix: GSobjIndexFile;
begin
   gsIndexTagRemove := 1;
   if Tag = '' then exit;
   StrPCopy(pc,UpperCase(Trim(Tag)));
   ix := nil;
   px := nil;
   i := 0;
   while (i < IndexesAvail) and (px = nil) do
   begin
      inc(i);
      if IndexStack[i] <> nil then
      begin
         ix := IndexStack[i];
         px := ix.TagByName(pc);
      end;
   end;
   if (px <> nil) and (px = IndexMaster) then
      gsSetTagTo('',true);
   if px <> nil then
   begin
      gsIndexTagRemove := -1;
      if ix.DeleteTag(pc) then
      begin
         gsIndexTagRemove := 0;
         if ix.TagCount = 0 then
            gsIndexFileRemove(ix.GetFileName);
      end;
   end;
end;

function GSO_dBHandler.gsIndexRoute(const IName: String): integer;
var
   IExt: String[4];
   IFile: String;
   ix: GSobjIndexFile;
   IPC: PChar;
begin
   gsIndexRoute := -1;
   if IName = '' then exit;
   IFile := gsIndexFileExtend(IName);
   if (gsIndexFileIsOpen(IFile) <> 0)then exit;
   IExt := ExtractFileExt(IFile);
   ix := nil;
{$IFDEF FOXOK}
   if IExt = '.CDX' then
   begin
      ix := GSobjCDXFile.Create(Self,IName,FileReadWrite,FileShared,false,false);
   end;
{$ENDIF}
{$IFDEF CLIPOK}
   if IExt = '.NTX' then
   begin
      ix := GSobjNTXFile.Create(Self,IName,FileReadWrite,FileShared,false,false);
   end;
{$ENDIF}
{$IFDEF DBASE4OK}
   if IExt = '.MDX' then
   begin
      ix := GSobjMDXFile.Create(Self,IName,FileReadWrite,FileShared,false,false);
   end;
{$ENDIF}
{$IFDEF DBASE3OK}
   if IExt = '.NDX' then
   begin
      ix := GSobjNDXFile.Create(Self,IName,FileReadWrite,FileShared,false,false);
   end;
{$ENDIF}

   if (ix = nil) or (not ix.CreateOK) then           {!!RFG 091297}
   begin
      GetMem(IPC, length(IFile)+1);
      StrPCopy(IPC, IFile);
      if (ix <> nil) and (ix.Corrupted) then          {!!RFG 091297}
      begin
         FoundError(dbsIndexFileBad,gsfMsgOnly,IPC); {!!RFG 091297}
         gsIndexRoute := dbsIndexFileBad;              {!!RFG 091297}
      end
      else
      begin
         FoundError(dosFileNotFound,gsfMsgOnly,IPC);
         gsIndexRoute := dosFileNotFound;              {!!RFG 091297}
      end;
      FreeMem(IPC,length(IFile)+1);
      if ix <> nil then ix.Free;                      {!!RFG 091297}
   end
   else
   begin
      gsIndexRoute := 0;
      gsIndexInsert(ix);
      ix.ixSetLockProtocol(dfLockStyle);
   end;
end;

Procedure GSO_dBHandler.gsIndexClear;
var
   i: integer;
begin
   for i := 1 to IndexesAvail do
      if IndexStack[i] <> nil then
      begin
         IndexStack[i].Free;
         IndexStack[i] := nil;
      end;
   IndexMaster := nil;               {Set index active flag to false}
   PrimaryTagName[0] := #0;
{   IndexHandle := -1;}
end;

Function GSO_dBHandler.gsIndexInsert(ix : GSobjIndexFile) : integer;
var
   i   : integer;                     {Local working variable  }
begin
   i := 1;
   while (IndexStack[i] <> nil) and (i <= IndexesAvail) do inc(i);
   if i <= IndexesAvail then
   begin
      IndexStack[i] := ix;
      gsIndexInsert := i;
   end else gsIndexInsert := -1;
end;

Function GSO_dBHandler.gsIndexTo(const IName, tag, keyexpr, forexpr: String;
                               uniq: GSsetIndexUnique;
                               ascnd: GSsetSortStatus): integer;
var
   IFile: array[0..259] of char;
   IWork: string;
   ITag: array[0..32] of char;
   IExt: String[4];
   FormK: array[0..255] of char;
   FormF: array[0..255] of char;
   crd: boolean;
   ix: GSobjIndexFile;
begin
   crd := CacheRead;
   gsIndexTo := 0;
   gsIndexClear;
   gsSetDBFCache(true);
   IWork := UpperCase(Trim(IName));
   IWork := gsIndexFileExtend(IWork);
   StrPCopy(IFile,IWork);
   ITag[0] := #0;
   if (length(Tag) < 32) and (length(Tag) > 0) then
      StrPCopy(ITag, Tag);
   IExt := AnsiUpperCase(ExtractFileExt(IName));
   if IExt = '' then IExt := IndexExt;
   StrPCopy(FormK,keyexpr);
   if length(forexpr) > 0 then
      StrPCopy(FormF,forexpr)
   else
      FormF[0] := #0;
   ix := nil;
{$IFDEF FOXOK}
   if IExt = '.CDX' then
   begin
      if (StrLen(ITag) = 0) then
      begin
         FoundError(cdxNoSuchTag,cdxInitError,'Tag Field is Empty');
         exit;           {Exit if formula is no good}
      end;
      ix := GSobjCDXFile.Create(Self, StrPas(IFile), FileReadWrite,FileShared,true,false);
      if ix <> nil then
      begin
         if ascnd in [AscendingGeneral, DescendingGeneral] then
         begin
            {$IFNDEF FOXGENERAL}
            FoundError(cdxNoCollateGen,cdxInitError,'General Collate Invalid');
            {$ENDIF}
            GSbytCDXCollateInfo := General;
         end
         else
            GSbytCDXCollateInfo := Machine;
      end;
   end;
{$ENDIF}
{$IFDEF DBASE4OK}
   if IExt = '.MDX' then
   begin
      if (StrLen(ITag) = 0) then
      begin
         FoundError(cdxNoSuchTag,cdxInitError,'Tag Field is Empty');
         exit;           {Exit if formula is no good}
      end;
      ix := GSobjMDXFile.Create(Self, StrPas(IFile), FileReadWrite,FileShared,true,false);
   end;
{$ENDIF}
{$IFDEF CLIPOK}
   if IExt = '.NTX' then
   begin
      StrPCopy(ITag,ExtractFileNameOnly(IName));
      ix := GSobjNTXFile.Create(Self, StrPas(IFile), FileReadWrite,FileShared,true,true);
   end;
{$ENDIF}
{$IFDEF DBASE3OK}
   if IExt = '.NDX' then
   begin
      StrPCopy(ITag,ExtractFileNameOnly(IName));
      ix := GSobjNDXFile.Create(Self, StrPas(IFile),FileReadWrite,FileShared,true,true);
   end;
{$ENDIF}
   if (ix <> nil) and (ix.CreateOK) then
   begin
      ix.Dictionary := ascnd in [AscendingGeneral, DescendingGeneral];
      if ascnd = AscendingGeneral then
         ascnd := Ascending
      else
         if ascnd = DescendingGeneral then
            ascnd := Descending;
      dState := dbIndex;
      {$IFDEF DELPHI}try{$ENDIF}
      ix.AddTag(ITag, FormK, FormF, Ascnd=Ascending, Uniq=Unique);
      {$IFDEF DELPHI}finally{$ENDIF}
      ix.Free;
      dState := dbBrowse;
      {$IFDEF DELPHI}end;{$ENDIF}
   end
   else
   begin
      if (ix <> nil) and (ix.Corrupted) then            {!!RFG 091297}
         FoundError(dbsIndexFileBad,gsfMsgOnly,IFile)  {!!RFG 091297}
      else
         FoundError(dosFileNotFound,gsfMsgOnly,IFile);
      if ix <> nil then ix.Free;
   end;
   gsSetDBFCache(crd);
end;

Function GSO_dBHandler.gsMemoryIndexAdd(const tag, keyexpr, forexpr: String;
                                        uniq: GSsetIndexUnique;
                                        ascnd: GSsetSortStatus): boolean;
var
   ITag: array[0..32] of char;
   FormK: array[0..255] of char;
   FormF: array[0..255] of char;
   ix: GSobjIndexFile;
begin
   gsMemoryIndexAdd := false;
   if (tag = '') or (keyexpr = '') then exit;
   gsIndexFileRemove(Tag);
   if (length(Tag) < 32) and (length(Tag) > 0) then
      StrPCopy(ITag, AnsiUpperCase(Tag));
   StrPCopy(FormK,keyexpr);
   if length(forexpr) > 0 then
      StrPCopy(FormF,forexpr)
   else
      FormF[0] := #0;
   ix := GSobjMIXFile.Create(Self, Tag);
   if (ix <> nil) and (ix.CreateOK) then
   begin
      if ascnd in [AscendingGeneral, DescendingGeneral] then
      begin
         {$IFNDEF FOXGENERAL}
         FoundError(cdxNoCollateGen,cdxInitError,'General Collate Invalid');
         {$ENDIF}
         GSbytCDXCollateInfo := General;
      end
      else
         GSbytCDXCollateInfo := Machine;
      ix.Dictionary := ascnd in [AscendingGeneral, DescendingGeneral];
      if ascnd = AscendingGeneral then
         ascnd := Ascending
      else
         if ascnd = DescendingGeneral then
            ascnd := Descending;
      ix.AddTag(ITag, FormK, FormF, Ascnd=Ascending, Uniq=Unique);
      gsIndexInsert(ix);
      gsSetTagTo(tag,false);
      gsMemoryIndexAdd := true;
   end
   else
   begin
      if ix <> nil then ix.Free;
      FoundError(dosFileNotFound,dbsIndexFileBad,ITag);
   end;
end;

Function GSO_dBHandler.gsSetTagTo(const TName: String; SameRec: boolean): integer;
var
   i   : integer;                     {Local working variable  }
   pc: array [0..16] of char;
   px: GSobjIndexTag;
begin                                      {!!RFG 043098}
   if TName = '' then
   begin
      gsResetRange;
      if IndexMaster <> nil then IndexMaster.TagClose;
      IndexMaster := nil;
      gsSetDBFCache(true);
      if not SameRec then gsGetRec(Top_Record);
      gsSetTagTo := 0;
      exit;
   end;
   StrPCopy(pc,UpperCase(Trim(TName)));
   px := nil;
   i := 0;
   while (i < IndexesAvail) and (px = nil) do
   begin
      inc(i);
      if IndexStack[i] <> nil then
         px := IndexStack[i].TagByName(pc);
   end;
   if px = nil then
   begin
      raise EHalcyonError.CreateFmt(gsErrIndexTagMissing,[TName]);  {%FIX0016 raise exception for missing index tag}
   end;
   gsResetRange;
   if IndexMaster <> nil then IndexMaster.TagClose;
   IndexMaster := nil;
   gsSetDBFCache(false);
   StrCopy(PrimaryTagName,pc);
   IndexMaster := px;
   IndexMaster.TagOpen(1);
   if not SameRec then
      gsGetRec(Top_Record)
   else
      gsGetRec(RecNumber);       {%FIX 0002 Keep same record when changing tags}
   gsSetTagTo := i;
end;

Procedure GSO_dBHandler.gsReIndex;
var
   rxIndexMaster : GSobjIndexTag;
   i   : integer;
   crd: boolean;
begin
   rxIndexMaster := IndexMaster;
   IndexMaster := nil;
   crd := CacheRead;             {!!RFG 041398}
   dState := dbIndex;
   gsSetDBFCache(true);          {!!RFG 041398}
  {$IFDEF DELPHI}try{$ENDIF}
   for i := 1 to IndexesAvail do
   begin
      if IndexStack[i] <> nil then
         IndexStack[i].ReIndex;
   end;
  {$IFDEF DELPHI}finally{$ENDIF}
   dState := dbBrowse;
   gsSetDBFCache(crd);           {!!RFG 041398}
   {$IFDEF DELPHI}end;{$ENDIF}
   IndexMaster := rxIndexMaster;
   if IndexMaster <> nil then IndexMaster.TagOpen(0);
   gsGetRec(Top_Record);
end;


{------------------------------------------------------------------------------
                  File Modifying Routine (Sort, Copy, Pack, Zap)
------------------------------------------------------------------------------}

Procedure GSO_dBHandler.gsCopyFile(const filname, apassword: String);
var
   FCopy  : GSO_dBHandler;
   RecPos : longint;

BEGIN
   repeat until gsLockFile;
   gsHdrRead;
   gsStatusUpdate(StatusStart,StatusCopy,NumRecs);
   RecPos := RecNumber;
   gsCopyStructure(filname);
   FCopy := GSO_dBHandler.Create(filname, '',true,false);
   FCopy.gsSetPassword(apassword);
   FCopy.gsLockFile;
   if WithMemo then
   begin
      curMemo := gsMemoObject;
      newMemo := FCopy.gsMemoObject;
   end;
   gsGetRec(Top_Record);
   while not File_EOF do           {Read .DBF sequentially}
   begin
      gsStatusUpdate(StatusCopy,RecNumber,0);
      move(CurRecord^,FCopy.CurRecord^,RecLen+1);
      if WithMemo then gsCopyMemoRecord(FCopy);
      FCopy.gsAppend;
      gsGetRec(Next_Record);
   end;
   FCopy.gsLockOff;
   FCopy.Free;
   gsStatusUpdate(StatusStop,0,0);
   if WithMemo then
   begin
      curMemo.Free;
      newMemo.Free;
   end;
   gsGetRec(RecPos);
   gsLockOff;
END;                        { CopyFile }

Procedure GSO_dBHandler.gsCopyRecord;  {(filobj : GSO_dBHandler);}
BEGIN
   move(CurRecord^,filobj.CurRecord^,RecLen+1);
   if WithMemo then
   begin
      curMemo := gsMemoObject;
      newMemo := filobj.gsMemoObject;
      gsCopyMemoRecord(filobj);
{ 04/20/99             Removed - the MemoObject must be set nil if doing this}
{                      Better to leave it assigned since it may be in use}
{      curMemo.Free;}
{      newMemo.Free;}
   end;
   filobj.gsAppend;
END;                        { CopyRecord }

procedure GSO_dBHandler.gsCopyMemoRecord(df : GSO_dbHandler);
var
   fp     : integer;
   mbuf   : PChar;
   rl     : FloatNum;
   tcnt   : longint;
   vcnt   : longint;
   rcnt   : longint;
   blk    : longint;
begin
   for fp := 1 to NumFields do
   begin
      if Fields^[fp].dbFieldType in ['B','G','M'] then
      begin
         blk := Trunc(gsNumberGetN(fp));
         if (blk <> 0) then
         begin
            vcnt := curMemo.moMemoSize(blk) + 16;
            rcnt := vcnt;
            GetMem(mbuf, vcnt);
            curMemo.moMemoRead(mbuf, blk, rcnt);
            tcnt := 0;
            newMemo.moMemoWrite(mbuf, tcnt, rcnt);
            rl := tcnt;
            df.gsNumberPutN(fp,rl);
            FreeMem(mbuf, vcnt);
         end;
      end;
   end;
end;

procedure GSO_dBHandler.gsCopyStructure(const filname: string);
var
   NuFile : GSO_DBFBuild;
   fp     : integer;
BEGIN
   case FileTarget of
      DB4WithMemo  : NuFile := GSO_DB4Build.Create(filname);
      FxPWithMemo  : NuFile := GSO_DBFoxBuild.Create(filname);
      else NuFile := GSO_DB3Build.Create(filname);
   end;
   for fp := 1 to NumFields do
      NuFile.InsertField(gsFieldName(fp),Fields^[fp].dbFieldType,
                          Fields^[fp].dbFieldLgth,Fields^[fp].dbFieldDec);
   NuFile.Free;
END;

Procedure GSO_dBHandler.gsPack;
var
   rxIndexMaster : GSobjIndexTag;
   fp            : integer;
   i, j          : longint;
   eofm: char;
   dbfchgd: boolean;
   crd: boolean;
begin      {Pack}
   eofm := EOFMark;
   if FileShared then
   begin
      gsTestForOk(dosAccessDenied, dbsPackError);
      exit;
   end;
   rxIndexMaster := IndexMaster;
   crd := CacheRead;
   if WithMemo then
      curMemo := gsMemoObject;
   IndexMaster := nil;               {Set index active flag to false}
   gsSetDBFCache(false);  {!!RGF 040698}   {Clear any current cached records}
   gsSetDBFCache(true);
   gsStatusUpdate(StatusStart,StatusPack,NumRecs);
   j := 0;
   for i := 1 to NumRecs do           {Read .DBF sequentially}
   begin
      gsRead(HeadLen+((i-1) * RecLen), CurRecord^, RecLen);
      if gsvIsEncrypted then
         DBEncryption(gsvPasswordIn,PByteArray(CurRecord),PByteArray(CurRecord),0,RecLen);
      RecNumber := i;
      if not gsDelFlag then             {Write to work file if not deleted}
      begin
         inc(j);                      {Increment record count for packed file }
         if j <> i then inherited gsPutRec(j);
      end
      else
         if WithMemo then
         begin
            for fp := 1 to NumFields do
            begin
               if Fields^[fp].dbFieldType in ['B','G','M'] then
               begin
                  curMemo.MemoLocation := Trunc(gsNumberGetN(fp));
                  if (curMemo.MemoLocation <> 0) then
                     CurMemo.moMemoBlockRelease(curMemo.MemoLocation);
               end;
            end;
         end;
      gsStatusUpdate(StatusPack,i,0);
   end;
   dbfchgd := NumRecs > j;         {If records were deleted then...}
   if dbfchgd then
   begin
      NumRecs := j;                   {Store new record count in objectname}
      gsWrite(HeadLen+(j*RecLen), eofm, 1);
                                      {Write End of File byte at file end}
      gsTruncate(HeadLen+(j*RecLen)+1);
                                      {Set new file size for dBase file};
      dStatus := Updated;
      gsHdrWrite;
   end;
   gsStatusUpdate(StatusStop,0,0);
{ 03/28/99             Removed - the MemoObject must be set nil if doing this}
{                      Better to leave it assigned since it may be in use}
{
   if WithMemo then
      curMemo.Free;
}
   if dbfchgd then gsReIndex;
   IndexMaster := rxIndexMaster;
   gsSetDBFCache(crd);
   if IndexMaster <> nil then
   begin
      IndexMaster.TagOpen(0);
   end;
   gsGetRec(Top_Record);
END;                        { Pack }

                     {-------------------------------}

{-----------------------------------------------------------------------------
                               File Sorting Routines
-----------------------------------------------------------------------------}

Procedure GSO_dBHandler.gsLoadToIndex(ixColl: GSobjSortDB; zfld: PChar);
var
   exph: TgsExpHandler;
   Rsl: TgsVariant;
begin
   exph := TgsExpHandler.Create(DBFExpLink,zfld,false);
   Rsl := TgsVariant.Create(256);
   try
      gsStatusUpdate(StatusStart,StatusSort,NumRecs);
      gsGetRec(Top_Record);             {Read all dBase file records}
      while not File_EOF do
      begin
         exph.ExpressionAsVariant(rsl);
         ixColl.AddKey(Rsl,RecNumber);
         gsStatusUpdate(StatusSort,RecNumber,0);
         gsGetRec(Next_Record);
      end;
      gsStatusUpdate(StatusStop,0,0);
   finally
      Rsl.Free;
      exph.Free;
   end;
end;

procedure GSO_dBHandler.gsCopyFromIndex(ixColl: GSobjSortDB; const filname, apassword: String);
var
   FCopy  : GSO_dBHandler;
BEGIN
   gsStatusUpdate(StatusStart,StatusCopy,ixColl.Count);
   ixColl.KeyCnt := 0;
   gsCopyStructure(filname);
   FCopy := GSO_dBHandler.Create(filname,'',true,false);
   FCopy.gsSetPassword(apassword);
   if WithMemo then
   begin
      curMemo := gsMemoObject;
      newMemo := FCopy.gsMemoObject;
   end;
   ixColl.curFile := Self;
   ixColl.newFile := FCopy;
   ixColl.OutputWord;
   FCopy.Free;
   if WithMemo then                {!!RFG 111897}
   begin                           {!!RFG 111897}
      curMemo.Free;               {!!RFG 111897}
      newMemo.Free;               {!!RFG 111897}
   end;                            {!!RFG 111897}
   gsStatusUpdate(StatusStop,0,0);
end;

Procedure GSO_dBHandler.gsSortFile(const filname, apassword, zfld: string; isascend : GSsetSortStatus);
var
   pckey : PChar;
   ixColl: GSobjSortDB;
   rn    : longint;
   ps    : PChar;
   frslt : integer;
begin
   frslt := FileOpen(filname+'.DBF',fmOpenReadWrite + fmShareExclusive);
   if frslt > 0 then
   begin
      FileClose(frslt);
   end
   else
   begin
      frslt := GetLastError;
      if frslt <> 2 then
      begin
         GetMem(ps, 261);
         StrPCopy(ps, filname+'.DBF');
         FoundError(gsFileAlreadyOpen, dbsSortFile, ps);
         FreeMem(ps, 261);
         exit;
      end;
   end;

   if zfld <> '' then
   begin
      GetMem(pckey,255);
      StrPCopy(pckey,zfld);
      rn := RecNumber;
      if rn = 0 then rn := Top_Record;

      ixColl := GSobjSortDB.Create(240,false,isascend=SortUp,gsvTempDir);
      gsLoadToIndex(ixColl, pckey);
      gsCopyFromIndex(ixColl, filname, apassword);
      FreeMem(pckey,255);
      ixColl.Free;
      gsGetRec(rn);
   end;
end;

                     {-------------------------------}

Procedure GSO_dBHandler.gsZap;
var
   mbuf : array[0..dBaseMemoSize] of byte;
   i : longint;                    {Local variables   }
   ib: byte;
   eofm: char;
   MemoFile: GSobjMemo;
begin              {Zap}
   eofm := EOFMark;
   if FileShared then
   begin
      gsTestForOk(dosAccessDenied, dbsZapError);
      exit;
   end;
   if WithMemo then
   begin
      MemoFile := gsMemoObject;
      MemoFile.gsRead(0,mbuf,512);
      i := 0;
      move(i,mbuf[0],SizeOf(i));
      if (MemoFile.TypeMemo = FXPWithMemo) or (MemoFile.TypeMemo = VFP3File) then
      begin
         ib := 512 div FoxMemoSize;
         if (512 mod FoxMemoSize) <> 0 then inc(ib);
         mbuf[3] := ib;
      end
      else
      begin
         ib := 1;
         mbuf[0] := ib;
      end;
      MemoFile.gsWrite(0,mbuf,512);
      MemoFile.gsTruncate(512);
{ 03/28/99             Removed - the MemoObject must be set nil if doing this}
{                      Better to leave it assigned since it may be in use}
{      MemoFile.Free;}
   end;
   NumRecs := 0;                   {Store new record count in objectname}
   RecNumber := 0;
   dStatus := Updated;
   gsWrite(HeadLen, eofm, 1);
   gsTruncate(HeadLen+1);
   gsHdrWrite;
   gsGetRec(Top_Record);
   gsReIndex;
END;                        { Zap }

{------------------------------------------------------------------------------
                           Database Search Routine
------------------------------------------------------------------------------}

Function GSO_dBHandler.gsSearchDBF(const s : string; var FNum : word;
                          var fromrec: longint; toRec: longint): word;
var
   BTable: string[255];
   MTable: string[255];
   crd : boolean;
   ia : pointer;
   lr : longint;
   sloc: word;
   i   : integer;
   Strt: word;
   Size: word;
   rnum: longint;
   rsl : integer;
   ns  : real;
   rs  : real;
   di  : longint;
   dr  : longint;
   li  : boolean;
   lv  : boolean;
   mp  : integer;
   ml  : integer;
   mc  : integer;
   fstrt: integer;
   ffini: integer;
   floc: integer;
   multifld: boolean;
   caseinsensitive: boolean;
   matchexact: boolean;
   usewildcards: boolean;
   useindex: boolean;
   usefilter: boolean;
   astrbegin: boolean;
   astrend: boolean;
   hasqmarks: boolean;
   startonly: boolean;

   function MatchField: integer;
   var
      mfi: integer;
      mfe: integer;
   begin
      MatchField := 0;
      if length(MTable) < length(BTable) then exit;
      if caseinsensitive then
         MTable := AnsiUpperCase(MTable);
      if hasqmarks then
         for mfi := 1 to length(BTable) do
            if BTable[mfi] = '?' then MTable[mfi] := '?';
      if matchexact then
      begin
         while (length(MTable) > length(BTable)) and   {!!RFG 032498}
               (MTable[length(Mtable)] = ' ') do
                MTable[0] := pred(MTable[0]);
         if BTable = MTable then MatchField := 1;
      end
      else
      begin
         mfi := pos(BTable,MTable);
         if (mfi > 1) and (not astrbegin) then mfi := 0;
         if (mfi > 0) and (not astrend) then
         begin
            MTable := TrimRight(MTable);
            mfe := mfi + length(BTable) - 1;
            if length(MTable) > mfe then mfi := 0;
         end;
         MatchField := mfi;
      end;
   end;


begin
   rnum := 1;
   sloc := 0;
   if (FNum > NumFields) or (length(s) = 0) then
   begin
      gsSearchDBF := 0;
      exit;
   end;

   caseinsensitive := (SearchMode and 1) <> 0;
   matchexact := (SearchMode and 2) <> 0;
   usewildcards := (SearchMode and 4) <> 0;
   usefilter := (SearchMode and 8) <> 0;
   useindex := (SearchMode and 16) <> 0;
   startonly := (SearchMode and 32) <> 0;

{   if toRec = 0 then toRec := NumRecs;}
   fstrt := FNum;
   ffini := FNum;
   multifld := FNum = 0;
   if multifld then
   begin
      fstrt := 1;
      ffini := numFields;
   end;
   if caseinsensitive then
      BTable := AnsiUpperCase(s)
   else
      BTable := s;
   if usewildcards then
   begin
      astrbegin := BTable[1] = '*';
      astrend := BTable[length(BTable)] = '*';
      if astrbegin then system.delete(BTable,1,1);
      if astrend and (length(BTable) > 0) then
         system.delete(BTable,length(BTable),1);
      if astrbegin then
         hasqmarks := false
      else
         hasqmarks := pos('?',BTable) <> 0;
      if astrbegin or astrend or hasqmarks then
         matchexact := false;
   end
   else
   begin
      astrbegin := not startonly;
      astrend := not matchexact;
      hasqmarks := false;
   end;
   di := -2;
   val(s,ns,rsl);
   if rsl <> 0 then ns := 0.0;
   li := pos(s[1],LogicalTrue) > 0;
   gsStatusUpdate(StatusStart,StatusSearch,NumRecs);
   lr := RecNumber;
   ia := IndexMaster;

   if (not usefilter) and UseDeletedRec then
      dState := dbIndex;

   if (not useindex) then
      IndexMaster := nil;
   crd := CacheRead;
   if IndexMaster = nil then
      gsSetDBFCache(true);
   {$IFDEF DELPHI}try{$ENDIF}
   if fromrec = 0 then
      gsGetRec(Top_Record)
   else
   begin
      gsGetRec(fromrec);
      gsSkip(1);
   end;
   while (not File_EOF) and ((RecNumber <= toRec) or (torec = 0)) and
         (sloc = 0) do
   begin
      floc := fstrt;
      repeat
         FNum := floc;
         Strt := 1;
         if FNum > 1 then
            for i := 1 to FNum-1 do
               Strt := Strt + gsFieldLength(i);
         Size := gsFieldLength(FNum);
         if sloc = 0 then
         case gsFieldType(FNum) of
         'C' : begin
                  move(CurRecord^[Strt],MTable[1],Size);
                  MTable[0] := chr(Size);
                  sloc := MatchField;
               end;
         'F',
         'N' : begin
                  sloc := 0;
                  if rsl = 0 then
                  begin
                     rs := gsNumberGetN(FNum);
                     if rs = ns then sloc := 1;
                  end;
               end;
         'D' : begin
                  if di = -2 then
                  try
                     di := DBFDate.CTOD(s);
                  except
                     di := -1;
                  end;
                  sloc := 0;
                  dr := gsDateGetN(FNum);
                  if di = dr then sloc := 1;
               end;
         'L' : begin
                  sloc := 0;
                  if not multifld then
                  begin
                     lv := gsLogicGetN(FNum);
                     if li = lv then sloc := 1;
                  end;
               end;
         (*
         'M' : begin
                  sloc := 0;
                  mp := 0;
                  gsMemoGetN(FNum);
                  ml := gsMemoLinesN(FNum);
                  if ml > 0 then
                  begin
                     mc := 1;
                     while (mc <= ml) and (sloc = 0) do
                     begin
                        MTable := gsMemoGetLineN(FNum,mc);
                        sloc := MatchField;
                        if sloc > 0 then
                           sloc := sloc + mp
                        else
                           mp := mp + length(MTable) + 2;
                        inc(mc);
                     end;
                  end;
               end;
               *)
         end;
         inc(floc);
      until (floc > ffini) or (sloc > 0);
       if sloc = 0 then
      begin
         inc(rnum);
         gsStatusUpdate(StatusSearch,rnum,0);
         gsGetRec(Next_Record);
      end;
   end;
   {$IFDEF DELPHI}finally{$ENDIF}
   dState := dbBrowse;
   gsSetDBFCache(crd);
   IndexMaster := ia;
   if sloc > 0 then
   begin
      fromrec := RecNumber;
      gsGetRec(fromrec);            {Reset for index}
   end
   else
      if lr > 0 then gsGetRec(lr);
   gsSearchDBF := sloc;
   gsStatusUpdate(StatusStop,rnum,0);
   {$IFDEF DELPHI}end;{$ENDIF}
end;

{-----------------------------------------------------------------------------
                               GSobjSortDB
-----------------------------------------------------------------------------}

procedure GSobjSortDB.OutputWord;
var
  kw: TgsVariant;
   tag: longint;
begin
   kw := TgsVariant.Create(256);
   while GetNextKey(kw,tag) do
   begin
      curFile.gsGetRec(Tag);
      inc(KeyCnt);
      curFile.gsStatusUpdate(StatusCopy,KeyCnt,0);
      move(curFile.CurRecord^,newFile.CurRecord^,curFile.RecLen);
      if curFile.WithMemo then curFile.gsCopyMemoRecord(newFile);
      newFile.gsAppend;
   end;
   kw.Free;
end;

{-----------------------------------------------------------------------------
                                 Initialization
-----------------------------------------------------------------------------}

end.


