
{*******************************************************}
{                                                       }
{       AnimatedMenus/2000                              }
{       T_AM2000_EditboxOptions                         }
{                                                       }
{       Copyright (c) 1997-99 AnimatedMenus.com         }
{       All rights reserved.                            }
{                                                       }
{*******************************************************}


unit am2000editbox;

interface

uses
  Windows, Classes, Graphics, Menus, Controls,
  {$IFDEF VER120} ImgList, {$ENDIF}
  am2000options;

type

  // editbox options
  T_AM2000_EditboxOptions = class(T_AM2000_ControlOptions)
  private
    LastCanvas    : TCanvas;
    LastWidth     : Integer;
    LastText      : String;

//    FAutoSelect   : Boolean;
//    FColor        : TColor;
    FMaxLength    : Integer;
    FReadOnly     : Boolean;
    FText         : String;
    FSelStart     : Integer;
    FSelLength     : Integer;
    FCaretPos     : Integer;
    FFirstPos     : Integer;
    FMinWidth     : Integer;

    procedure SetCaretPos(const Value: Integer);

    procedure CheckPos;
    function ClearSel: Boolean;

  public
    constructor Create(AParent: TMenuItem); override;

    procedure Draw(DrawRect: P_AM2000_DrawMenuItemRect); override;

    function GetHeight(ItemHeight: Integer): Integer; override;
    function GetWidth(Canvas: TCanvas): Integer; override;

    procedure KeyDown(var Key: Word; Shift: TShiftState); override;
    procedure KeyPress(var Key: Char); override;

    procedure BeginEdit(X, Y: Integer);
    procedure CancelEdit(AcceptChanges: Boolean);

  published
//    property AutoSelect   : Boolean
//      read FAutoSelect write FAutoSelect default True;
//    property Color        : TColor
//      read FColor write FColor default clWindow;
    property MaxLength    : Integer
      read FMaxLength write FMaxLength default 0;
    property ReadOnly     : Boolean
      read FReadOnly write FReadOnly default False;
    property Text         : String
      read FText write FText;
    property MinWidth     : Integer
      read FMinWidth write FMinWidth default 0;

  end;


implementation

uses
  SysUtils,
  am2000menuitem, am2000utils;


{ T_AM2000_EditboxOptions }

constructor T_AM2000_EditboxOptions.Create(AParent: TMenuItem);
begin
  inherited;
//  FAutoSelect:= True;
//  FColor:= clWindow;
end;


procedure T_AM2000_EditboxOptions.Draw(DrawRect: P_AM2000_DrawMenuItemRect);
var
  X, DX: Integer;
  S: String;
  R: TRect;
begin
  with DrawRect^, Canvas do begin
    LastCanvas:= Canvas;
    DrawBackground(Canvas, State, Options, mir.LineRect);

    // draw the caption
    if isSelected in State
    then Font.Color:= Options.Colors.HighlightText
    else Font.Color:= Options.Colors.MenuText;

    R:= mir.LineRect;
    Inc(R.Left, 2);
    DrawCaption(Canvas, Item.Caption, taLeftJustify, R);

    // draw edit box
    if Brush.Style <> bsSolid
    then Brush.Style:= bsSolid;

    R.Left:= AmpTextWidth(Canvas, Item.Caption) +5;
    if R.Left < mir.ItemLeft -1 then R.Left:= mir.ItemLeft -2;
    InflateRect(R, -2, -2);

    if isSelected in State
    then begin
      DrawEdge(Handle, R, bdr_SunkenOuter, bf_Rect);
      InflateRect(R, -1, -1);
    end;

    Brush.Color:= clWindow;
    FillRect(R);

    // draw text
    if isSelected in State
    then InflateRect(R, -2, 0)
    else InflateRect(R, -3, 0);

    // set edit width
    LastWidth:= R.Right - R.Left;

    // draw text
    Font.Color:= clWindowText;
    DrawText(Handle, PChar(@FText[FFirstPos +1]), -1, R, dt_VCenter + dt_SingleLine);

    // draw caret
    if isActivated in State then begin
      S:= Copy(FText, FFirstPos +1, FCaretPos - FFirstPos);
      X:= R.Left + TextWidth(S);

      Pen.Style:= psSolid;
      Pen.Color:= clWindowText;
      PolyLine([Point(X, R.Top +1), Point(X, R.Bottom -1)]);
    end;

    // draw selection
    if FSelLength > 0 then begin
      InflateRect(R, 0, -1);

      // sel start
      if FSelStart > FFirstPos then begin
        S:= Copy(FText, FFirstPos +1, FSelStart - FFirstPos);
        Inc(R.Left, TextWidth(S));
        X:= FSelStart;
        DX:= FSelLength;
      end
      else begin
        X:= FFirstPos;
        DX:= FSelLength - (FFirstPos - FSelStart);
      end;

      // get visible selection
      repeat
        S:= Copy(FText, X +1, DX);
        if TextWidth(S) > LastWidth
        then Dec(DX)
        else Break;
      until DX = 0;

      R.Right:= R.Left + TextWidth(S);

      // draw selection
      Brush.Color:= clHighlight;
      Font.Color:= clHighlightText;
      DrawText(Handle, PChar(S), -1, R, dt_VCenter + dt_SingleLine);

    end;
  end;
end;

function T_AM2000_EditboxOptions.GetHeight(ItemHeight: Integer): Integer;
begin
  Result:= ItemHeight + 4;
end;

function T_AM2000_EditboxOptions.GetWidth(Canvas: TCanvas): Integer;
begin
  Result:= AmpTextWidth(Canvas, Parent.Caption) + FMinWidth +5;
end;


procedure T_AM2000_EditboxOptions.KeyDown(var Key: Word;
  Shift: TShiftState);
  // symbol keys
var
  S: String;
  L: Integer;
begin
  // errors
  if ((FCaretPos = 0) and (FSelLength = 0) and ((Key = vk_Left) or (Key = vk_Back) or (Key = vk_Home)))
  or ((FCaretPos = Length(FText)) and (FSelLength = 0) and ((Key = vk_Right) or (Key = vk_Delete) or (Key = vk_End)))
  or (FReadOnly and ((Key = vk_Back) or (Key = vk_Delete)))
  then begin
    MessageBeep(0);
    Key:= 0;
    Exit;
  end;

  // key
  case Key of
    vk_Left:
      begin
        if ssShift in Shift then
          if (FSelStart = FCaretPos) or (FSelLength = 0)
          then begin
            FSelStart:= FCaretPos -1;
            Inc(FSelLength);
          end
          else
            Dec(FSelLength)

        else
          FSelLength:= 0;

        SetCaretPos(FCaretPos -1);
      end;

    vk_Right:
      begin
        // add selection
        if ssShift in Shift then begin
          if FSelLength = 0
          then FSelStart:= FCaretPos;

          if (FSelStart = FCaretPos) and (FSelLength <> 0)
          then begin
            FSelStart:= FCaretPos +1;
            Dec(FSelLength);
          end
          else
            Inc(FSelLength);

        end
        else
          FSelLength:= 0;

        SetCaretPos(FCaretPos +1);
      end;

    vk_Back:
      begin
        if not ClearSel then Delete(FText, FCaretPos, 1);
        SetCaretPos(FCaretPos -1);
      end;

    vk_Delete:
      if ssShift in Shift then begin
        // cut and place to clipboard
        CopyToClipboard(Copy(FText, FSelStart +1, FSelLength));
        ClearSel;
        Repaint;
      end
      else
      if ssCtrl in Shift then begin
        // clear selection
        ClearSel;
        Repaint;
      end
      else begin
        if not ClearSel then Delete(FText, FCaretPos +1, 1);
        Repaint;
      end;

    vk_Home:
      if ssShift in Shift then begin
        if (FSelLength = 0) or (FSelStart >= FCaretPos)
        then Inc(FSelLength, FCaretPos)
        else FSelLength:= FSelStart;

        FSelStart:= 0;
        SetCaretPos(0);
      end
      else
      begin
        FSelLength:= 0;
        SetCaretPos(0);
      end;

    vk_End:
      if ssShift in Shift then begin
        L:= Length(FText);
        if (FSelLength = 0) or (FSelStart < FCaretPos)
        then Inc(FSelLength, L - FCaretPos)
        else begin
          L:= L - FSelStart - FSelLength;
          FSelStart:= FSelStart + FSelLength;
          FSelLength:= L;
        end;

        SetCaretPos(Length(FText));
      end
      else
      begin
        FSelLength:= 0;
        SetCaretPos(Length(FText));
      end;

    vk_Insert:
      if (ssShift in Shift) and IsClipboardFormatAvailable(cf_Text) then begin
        // insert from clipboard
        S:= PasteFromClipboard;
        if S <> '' then begin
          ClearSel;
          Insert(S, FText, FCaretPos +1);
          SetCaretPos(FCaretPos + Length(S));
        end;
      end
      else
      if ssCtrl in Shift then begin
        // copy to clipboard
        CopyToClipboard(Copy(FText, FSelStart +1, FSelLength));
      end;

    else
      Exit;
  end;

  Key:= 0;
end;

procedure T_AM2000_EditboxOptions.KeyPress(var Key: Char);
  // character keys
begin
  if FReadOnly
    or ((FMaxLength > 0)
    and (Length(FText) >= FMaxLength))
  then
    MessageBeep(0)
  else begin
    // clear selection
    if ClearSel
    then FCaretPos:= FCaretPos -1;

    // insert key
    Insert(Key, FText, FCaretPos +1);
    SetCaretPos(FCaretPos +1);
  end;

  Key:= #0;
end;

function T_AM2000_EditboxOptions.ClearSel: Boolean;
  // clear selection
begin
  if FSelLength > 0
  then begin
    Delete(FText, FSelStart +1, FSelLength);
    FSelLength:= 0;
    FCaretPos:= FSelStart;
    Result:= True;
  end
  else
    Result:= False;
end;


procedure T_AM2000_EditboxOptions.CheckPos;
var
  W: Integer;
begin
  // correct left bound
  if FCaretPos < 0 then FCaretPos:= 0;
  if FCaretPos <= FFirstPos then FFirstPos:= FCaretPos -6;
  if FFirstPos < 0 then FFirstPos:= 0;

  // correct right bound
  if FCaretPos > Length(FText) then FCaretPos:= Length(FText);

  // correct width
  if
    (LastCanvas <> nil) and (LastWidth <> 0)
  then
    repeat
      W:= LastCanvas.TextWidth(Copy(FText, FFirstPos +1, FCaretPos - FFirstPos));
      if W > LastWidth then Inc(FFirstPos);
    until W <= LastWidth;
end;

procedure T_AM2000_EditboxOptions.SetCaretPos(const Value: Integer);
begin
  FCaretPos:= Value;
  CheckPos;
  Repaint;
end;

procedure T_AM2000_EditboxOptions.BeginEdit(X, Y: Integer);
begin
  LastText:= FText;
  Repaint;
end;

procedure T_AM2000_EditboxOptions.CancelEdit(AcceptChanges: Boolean);
begin
  if not AcceptChanges then begin
    FText:= LastText;
    LastText:= '';
  end;
  FSelLength:= 0;
  Repaint;
end;

end.
