
{*******************************************************}
{                                                       }
{       AnimatedMenus/2000                              }
{       T_AM2000_MenuItemCache Unit                     }
{       English Version                                 }
{                                                       }
{       Copyright (c) 1997-99 AnimatedMenus.com         }
{       All rights reserved.                            }
{                                                       }
{*******************************************************}


unit am2000cache;

interface

uses
  Windows, Classes, Graphics;

type
  T_AM2000_MenuItemCacheItem = class
  private
    FBitmap   : HBitmap;
    FBmpRes   : Integer;

    function GetBitmap: HBitmap;

  public
    ShortCuts : String;
    Hint      : String;
    IsDefault : Boolean;

    property Bitmap : HBitmap
      read GetBitmap write FBitmap;

    destructor Destroy; override;

  end;

  T_AM2000_MenuItemCache = class(TStringList)
  private
    function GetItem(const Caption: String): T_AM2000_MenuItemCacheItem;

  public
    property Items    [const Caption: String]: T_AM2000_MenuItemCacheItem read GetItem; default;

    constructor Create;
    destructor Destroy; override;
    function AddCache(ACaption, AShortCuts, AHint: String; ABitmap: HBitmap; ADefault: Boolean): T_AM2000_MenuItemCacheItem;
    function AddCacheRes(ACaption, AShortCuts, AHint: String; ABitmapResource: Integer; ADefault: Boolean): T_AM2000_MenuItemCacheItem;
    function GetDefaultBitmap(Index: Integer): HBitmap;

  end;

var
  MenuItemCache: T_AM2000_MenuItemCache;

implementation

uses
  SysUtils,
  am2000const;

{ Utilities }

function GetStandardCaption(S: String): String;
var
  J: Integer;
begin
  Result:= '';
  S:= AnsiUpperCase(S);
  for J:= 1 to Length(S) do
    if IsCharAlpha(S[J])
    then AppendStr(Result, S[J]);
end;


{ T_AM2000_MenuItemCacheItem }

destructor T_AM2000_MenuItemCacheItem.Destroy;
begin
  if Bitmap <> 0
  then DeleteObject(Bitmap);

  inherited;
end;

function T_AM2000_MenuItemCacheItem.GetBitmap: HBitmap;
begin
  if (FBitmap = 0) and (FBmpRes <> 0)
  then FBitmap:= LoadBitmap(HInstance, MakeIntResource(FBmpRes));
  Result:= FBitmap;
end;

{ T_AM2000_MenuItemCache }

constructor T_AM2000_MenuItemCache.Create;
begin
  inherited;
  Sorted:= True;

  // initialize default menu items
  AddCacheRes('ADDRESSBOOK', '', 'Displays the list of contacts', BMP_AM2000_ADDRESSBOOK, True);
  AddCacheRes('ALIGNCENTER', SCtrl+'+E', 'Centers the paragraph between the margins', BMP_AM2000_ALIGNCENTER, True);
  AddCacheRes('ALIGNLEFT', SCtrl+'+L', 'Aligns the paragraph at the left margin', BMP_AM2000_ALIGNLEFT, True);
  AddCacheRes('ALIGNRIGHT', SCtrl+'+R', 'Aligns the paragraph at the right margin', BMP_AM2000_ALIGNRIGHT, True);
  AddCacheRes('AUTOPREVIEW', '', 'Shows or hides the autopreview pane', BMP_AM2000_AUTOPREVIEW, True);
  AddCacheRes('BOLD', SCtrl+'+B', 'Bold Text', BMP_AM2000_BOLD, True);
  AddCacheRes('BOOKMARKS', '', 'Insert a bookmark at the cursor', BMP_AM2000_BOOKMARKS, True);
  AddCacheRes('BULLETS', '', 'Inserts a bullet on this line', BMP_AM2000_BULLETS, True);
  AddCacheRes('CASCADE', '', 'Arranges the windows as overlapping tiles', BMP_AM2000_CASCADE, True);
  AddCacheRes('CONTENTS', '', 'Displays help topics', BMP_AM2000_CONTENTS, False);
  AddCacheRes('CONTENTSANDINDEX', 'F1', 'Displays help topics', BMP_AM2000_CONTENTSANDINDEX, True);
  AddCacheRes('COPY', SCtrl+'+C;'+SCtrl+'+'+SIns, 'Copies the selection to the clipboard', BMP_AM2000_COPY, True);
  AddCacheRes('CUT', SCtrl+'+X;'+SShift+'+'+SDel, 'Cuts the selection and moves it to the clipboard', BMP_AM2000_CUT, True);
  AddCacheRes('DATE', '', 'Inserts today''s date', BMP_AM2000_SETDATE, False);
  AddCacheRes('DATEANDTIME', '', 'Inserts today''s date and/or time', BMP_AM2000_DATEANDTIME, True);
  AddCacheRes('DATETIME', '', 'Inserts today''s date and/or time', BMP_AM2000_DATEANDTIME, False);
  AddCacheRes('DECREASEINDENT', '', 'Decrease indent', BMP_AM2000_DECREASEINDENT, True);
  AddCacheRes('DELETE', ''+SDel+';'+SCtrl+'+'+SDel, 'Erases the selection', BMP_AM2000_DELETE, True);
  AddCacheRes('FIND', SCtrl+'+F;F3', 'Finds the specified text in the active document', BMP_AM2000_FIND, True);
  AddCacheRes('FINDNEXT', SShift+'+F3', 'Repeats the last find', BMP_AM2000_FINDNEXT, True);
  AddCacheRes('FONT', '', 'View or edit the character attributes of the selected text', BMP_AM2000_SETFONT, False);
  AddCacheRes('FONTSIZE', '', 'View or edit the font size of the selected text', BMP_AM2000_FONTSIZE, True);
  AddCacheRes('FONTSTYLE', '', 'View or edit the font style of the selected text', BMP_AM2000_FONTSTYLE, True);
  AddCacheRes('FULLSCREEN', 'F11', 'Zooms the active document to the full screen', BMP_AM2000_FULLSCREEN, True);
  AddCacheRes('GOTOBOOKMARK', '', 'Moves the cursor to the specified bookmark', BMP_AM2000_SETBOOKMARK, False);
  AddCacheRes('HELPONHELP', 'F1', 'Display instructions about how to use help', BMP_AM2000_HELP, True);
  AddCacheRes('HOME', '', 'Browses the web', BMP_AM2000_HOME, True);
  AddCacheRes('HYPERLINK', '', 'Insert a new hyperlink', BMP_AM2000_HYPERLINK, True);
  AddCacheRes('INCREASEINDENT', '', 'Increase indent', BMP_AM2000_INCREASEINDENT, True);
  AddCacheRes('INDENT', '', 'Increase Indent', BMP_AM2000_INCREASEINDENT, False);
  AddCacheRes('ITALIC', SCtrl+'+I', 'Italic Text', BMP_AM2000_ITALIC, True);
  AddCacheRes('ALIGNJUSTIFY', SCtrl+'+J', 'Justifies the paragraph between the margins', BMP_AM2000_JUSTIFY, True);
  AddCacheRes('JUSTIFY', SCtrl+'+J', 'Justifies the paragraph between the margins', BMP_AM2000_JUSTIFY, False);
  AddCacheRes('MARKASREAD', '', 'Marks the selected messages as read', BMP_AM2000_MARKASREAD, True);
  AddCacheRes('MARKASUNREAD', '', 'Marks the selected messages as unread', BMP_AM2000_MARKASUNREAD, True);
  AddCacheRes('MOVETOFOLDER', '', 'Moves selected items to another folder', BMP_AM2000_MOVETOFOLDER, True);
  AddCacheRes('NEW', SCtrl+'+N', 'Create a new document', BMP_AM2000_NEW, True);
  AddCacheRes('NEWFOLDER', '', 'Create a new folder', BMP_AM2000_NEWFOLDER, True);
  AddCacheRes('NEWMAIL', '', 'Create a new email message', BMP_AM2000_NEWMAIL, True);
  AddCacheRes('NEWMAILMESSAGE', '', 'Create a new email message', BMP_AM2000_NEWMAIL, False);
  AddCacheRes('NEWNOTE', '', 'Create a new note', BMP_AM2000_NEWNOTE, True);
  AddCacheRes('NEWTASK', '', 'Create a new task', BMP_AM2000_NEWTASK, True);
  AddCacheRes('NOTE', '', 'Create a new note', BMP_AM2000_NEWNOTE, False);
  AddCacheRes('NUMBERING', '', 'Inserts a numbering on this line', BMP_AM2000_NUMBERING, True);
  AddCacheRes('OPEN', SCtrl+'+O', 'Opens an existing document from a file', BMP_AM2000_OPEN, True);
  AddCacheRes('PASTE', SCtrl+'+V;'+SShift+'+'+SIns, 'Inserts the clipboard contents at the insertion point', BMP_AM2000_PASTE, True);
  AddCacheRes('PRINT', SCtrl+'+P', 'Print the active document', BMP_AM2000_PRINT, True);
  AddCacheRes('PRINTPREVIEW', '', 'Display full pages', BMP_AM2000_PRINTPREVIEW, True);
  AddCacheRes('PROPERTIES', '', 'Display properties for the active document', BMP_AM2000_PROPERTIES, True);
  AddCacheRes('REDO', SCtrl+'+'+SShift+'+Z', 'Redoes the last action that was undone', BMP_AM2000_REDO, True);
  AddCacheRes('REFRESH', '', 'Updates the display to reflect any changes', BMP_AM2000_REFRESH, True);
  AddCacheRes('RELOAD', '', 'Reloads the content of the current page', BMP_AM2000_REFRESH, False);
  AddCacheRes('REPEAT', '', 'Repeats the last action', BMP_AM2000_REPEAT, True);
  AddCacheRes('REPLACE', SCtrl+'+H', 'Replaces specific text with different text', BMP_AM2000_REPLACE, True);
  AddCacheRes('SAVE', SCtrl+'+S;'+SShift+'+F12;'+SAlt+'+'+SShift+'+F2', 'Saves the active document', BMP_AM2000_SAVE, True);
  AddCacheRes('SAVEALL', '', 'Saves all opened documents', BMP_AM2000_SAVEALL, True);
  AddCacheRes('SEARCHTHEWEB', '', 'Searches the Word Wide Web for the specified resource', BMP_AM2000_SEARCHTHEWEB, True);
  AddCacheRes('SENDMAIL', '', 'Compose and send new mail message', BMP_AM2000_SENDMAIL, True);
  AddCacheRes('SETBOOKMARK', '', 'Insert a bookmark at the cursor', BMP_AM2000_SETBOOKMARK, True);
  AddCacheRes('SETDATE', '', 'Set today''s date', BMP_AM2000_SETDATE, True);
  AddCacheRes('SETFONT', '', 'View or edit the character attributes of the selected text', BMP_AM2000_SETFONT, True);
  AddCacheRes('SETTIME', '', 'Set current time', BMP_AM2000_SETTIME, True);
  AddCacheRes('SORT', '', 'Arranges items in the window', BMP_AM2000_SORT, True);
  AddCacheRes('SPELLCHECK', 'F7', 'Checks the spelling in the active document', BMP_AM2000_SPELLCHECK, True);
  AddCacheRes('CHECKSPELLING', 'F7', 'Checks the spelling in the active document', BMP_AM2000_SPELLCHECK, False);
  AddCacheRes('SPLITWINDOW', '', 'Splits the active document window horizontally', BMP_AM2000_SPLITWINDOW, True);
  AddCacheRes('TASK', '', 'Create a new task', BMP_AM2000_NEWTASK, False);
  AddCacheRes('TILE', '', 'Arranges the windows as nonoverlapping tiles', BMP_AM2000_TILE, True);
  AddCacheRes('TIME', '', 'Inserts current time', BMP_AM2000_SETTIME, False);
  AddCacheRes('UNDERLINE', SCtrl+'+U', 'Underlined text', BMP_AM2000_UNDERLINE, True);
  AddCacheRes('UNDO', SCtrl+'+Z', 'Reverses the last action', BMP_AM2000_UNDO, True);
  AddCacheRes('UNINDENT', '', 'Decrease indent', BMP_AM2000_DECREASEINDENT, False);
  AddCacheRes('WHATSTHIS', SShift+'+F1', 'Shows help about the selected element', BMP_AM2000_WHATSTHIS, True);
  AddCacheRes('PARAGRAPH', '', 'View or edit the paragraph attributes of the selected text', BMP_AM2000_PARAGRAPH, True);
  AddCacheRes('MOVEUP', '', 'Moves the selection up', BMP_AM2000_MOVEUP, True);
  AddCacheRes('MOVEDOWN', '', 'Moves the selection down', BMP_AM2000_MOVEDOWN, True);
  AddCacheRes('MOVELEFT', '', 'Moves the selection left', BMP_AM2000_MOVELEFT, True);
  AddCacheRes('MOVERIGHT', '', 'Moves the selection right', BMP_AM2000_MOVERIGHT, True);
  AddCacheRes('UPONELEVEL', '', 'Up one level', BMP_AM2000_UPONELEVEL, True);
  AddCacheRes('RULER', '', 'Shows or hides the ruler', BMP_AM2000_RULER, True);
  AddCacheRes('DELETE_B', '', 'Erases the selection', BMP_AM2000_DELETE2, True);
  AddCacheRes('NEWFOLDER_B', '', 'Create a new folder', BMP_AM2000_NEWFOLDER2, True);
  AddCacheRes('CREATE', '', 'Create a new document', BMP_AM2000_CREATE, True);
  AddCacheRes('CREATE_B', '', 'Create a new document', BMP_AM2000_CREATE2, True);
  AddCacheRes('BACKWARD', '', 'Return to the previous page in the hyperlink history list', BMP_AM2000_BACKWARD, True);
  AddCacheRes('BACK', '', 'Return to the previous page in the hyperlink history list', BMP_AM2000_BACKWARD, False);
  AddCacheRes('FORWARD', '', 'Go to the next page in the hyperlink history list', BMP_AM2000_FORWARD, True);

  // system icons
  AddCache('SYSTEMRESTORE', '', 'Restore the window to normal size', LoadBitmap(HInstance, 'AM2000_SYSTEMRESTORE'), False);
  AddCache('SYSTEMMOVE', '', 'Change the window position', LoadBitmap(HInstance, 'AM2000_SYSTEMMOVE'), True);
  AddCache('SYSTEMSIZE', '', 'Change the window size', LoadBitmap(HInstance, 'AM2000_SYSTEMSIZE'), True);
  AddCache('SYSTEMMINIMIZE', '', 'Reduce the window to an icon', LoadBitmap(HInstance, 'AM2000_SYSTEMMINIMIZE'), False);
  AddCache('SYSTEMMAXIMIZE', '', 'Enlarge the window to full size', LoadBitmap(HInstance, 'AM2000_SYSTEMMAXIMIZE'), False);
  AddCache('SYSTEMCLOSE', '', 'Close the active window and prompts to save the pages', LoadBitmap(HInstance, 'AM2000_SYSTEMCLOSE'), True);

  AddCache('CLEAR', SDel+';'+SShift+'+'+SDel, 'Erases the selection', 0, False);
  AddCache('SELECTALL', SCtrl+'+A', 'Select the entire document', 0, False);
  AddCache('GOTO', SCtrl+'+G', 'Moves the cursor to the specified line', 0, False);
  AddCache('PAGESETUP', '', 'Change the printing options', 0, False);
  AddCache('PRINTERSETUP', '', 'Change the printer and printing options', 0, False);
  AddCache('SAVEAS', 'F12;'+SCtrl+'+'+SShift+'+S', 'Saves a copy of the document in a separate file', 0, False);
  AddCache('PASTELINK', '', 'Insert Clipboard contents and a link to its source', 0, False);
  AddCache('PASTESPECIAL', '', 'Insert Clipboard contents with options', 0, False);
  AddCache('NEWWINDOW', '', 'Open another window for the active document', 0, False);
  AddCache('ARRANGEICONS', '', 'Arrange icons at the bottom of the window', 0, False);
  AddCache('EDIT', '', 'Contains commands for the clipboard, finding text, and editing links', 0, False);
  AddCache('VIEW', '', 'Contains commands for controlling the display of your document', 0, False);
  AddCache('FILE', '', 'Contains commands for controlling the display of your document', 0, False);
  AddCache('SEARCH', '', 'Contains commands for controlling the display of your document', 0, False);
  AddCache('TOOLS', '', 'Contains commands for controlling the display of your document', 0, False);
  AddCache('WINDOW', '', 'Contains commands for controlling the display of your document', 0, False);
  AddCache('HELP', '', 'Contains commands for controlling the display of your document', 0, False);
end;

destructor T_AM2000_MenuItemCache.Destroy;
var
  I: Integer;
//  Item: T_AM2000_MenuItemCacheItem;
begin
  for I:= 0 to Count -1 do
    Objects[I].Free;

  inherited;
end;

function T_AM2000_MenuItemCache.AddCache(ACaption, AShortCuts, AHint: String;
  ABitmap: HBitmap; ADefault: Boolean): T_AM2000_MenuItemCacheItem;
var
  Index: Integer;
begin
  ACaption:= GetStandardCaption(ACaption);
  if not Find(ACaption, Index) then begin
    Result:= T_AM2000_MenuItemCacheItem.Create;
    AddObject(ACaption, Result);
  end
  else
    Result:= T_AM2000_MenuItemCacheItem(Objects[Index]);

//  Result.Caption:=   ACaption;
  Result.ShortCuts:= AShortCuts;
  Result.Hint:=      AHint;
  Result.Bitmap:=    ABitmap;
  Result.IsDefault:= ADefault;
end;

function T_AM2000_MenuItemCache.AddCacheRes(ACaption, AShortCuts, AHint: String;
  ABitmapResource: Integer; ADefault: Boolean): T_AM2000_MenuItemCacheItem;
begin
  Result:= AddCache(ACaption, AShortCuts, AHint, 0, ADefault);
  Result.FBmpRes:= ABitmapResource;
end;

function T_AM2000_MenuItemCache.GetItem(const Caption: String): T_AM2000_MenuItemCacheItem;
var
  Index: Integer;
  S: String;
begin
  S:= GetStandardCaption(Caption);
  if Find(S, Index)
  then Result:= T_AM2000_MenuItemCacheItem(Objects[Index])
  else Result:= AddCache(S, '', '', LoadBitmap(HInstance, PChar('BMP_AM2000_' + S)), False);
end;

function T_AM2000_MenuItemCache.GetDefaultBitmap(Index: Integer): HBitmap;
var
  I: Integer;
begin
  Result:= 0;
  for I:= 0 to Count -1 do
    with T_AM2000_MenuItemCacheItem(Objects[I]) do begin
      if IsDefault
      then Dec(Index);

      if Index < 0 then begin
        Result:= Bitmap;
        Exit;
      end;
    end;
end;

initialization
  MenuItemCache:= T_AM2000_MenuItemCache.Create;

finalization
  MenuItemCache.Free;

end.
