unit QSRulers;

{{$DEFINE EVALUATION}

(*******************************************************************************
Horizontal and Vertical Ruler Components
for Delphi 1, Delphi 2 and Delphi 3
Copyright by Q-Systems Engineering 1997
8959 N. Calle Buena Vista
Tucson Arizona 85704
qsystems@flash.net
http://www.flash.net/~qsystems

******************************************************************************
******************************************************************************
COPYRIGHT NOTICE:
THE SOURCE OF THIS COMPONENT IS PROVIDED AS A PERSONAL COPY.  Q-SYSTEMS ENGINEERING
CLAIMS NO ROYALTIES FOR THE USE OF QSRulers IN YOUR PROGRAMS.  HOWEVER, YOU MAY
NOT DISTRIBUTE THE SOURCE -IN PART OR WHOLE-, THE .DCU, THE .OBJ, the .BPL
or .LIB OF THIS FILE.

******************************************************************************
******************************************************************************



Modified March 31, 1998
Version 2.0
Added
  Properties
     ShowBorder
     DivisionPlacement
     LengthUnits
Changed
  Default Interface

Modified August 11, 1998
Version 2 maintenance release 1
Added
  Property
    TicColor


Modified August 11, 1998
Version 2 maintenance release 2
Added
  Property
    ParentFont : false by default

Modified December 13, 1998; December 16
Version 3

Definition:
Adjustable colors:
  Color : color of main body of ruler
  BevelColor : color of bevel
  TicColor   : color of tics
  Font.Color : color of numbers

Added
  Properties
    BevelInner
    BevelOuter
    BevelWidth
    AutoColor : if true and ParentColor is true then set automatically all adjustable colors
    BevelColor : Base color from which the Highlight and shadow colors of the bevel are found.

  functions
    Function  HighLightColor(color:TColor): TColor; : High color for bevel
    Function  ShadowColor(color:TColor): TColor;    : Shadow color for bevel
    Function  ProperColor(color:TColor): TColor;    : Color for tics and numbers.
  properties  

Publishized
  Property
    ParentColor

Will add
  AutoDecimals: To choose the decimals automatically based on range.
  AutoFormat  : To choose automatically between fixed and scientific notation
  NumberFormat : nfFixed, nfScientific

KNOWN PROBLEMS: NONE
*******************************************************************************)



interface

uses
  SysUtils, WinTypes, WinProcs, Messages, Classes, Graphics, Controls,forms,printers,dialogs,
  extCtrls;

{$ifdef VER110} {CBUILDER 3}
{$ObjExportAll On}
{$ENDIF}

(*******************************************************************************
                      Declarations for Horizontal Ruler
*******************************************************************************)
  type
    THRDivisionPlacement = (dpTop,dpBottom);
    TVRDivisionPlacement = (dpLeft,dpRight);
    TLengthUnits         = (luActualInches,luActualMM,luCustom);
    TMyControl           = class(TControl);



  type
  THRuler = class(TGraphicControl)

  private
    fCurrentCanvas                   : TCanvas;
    fLeftToRight                     : boolean;
    fXMin,fXMax                      : double;
    fDecimals                        : byte;
    fpLeft,fpTop                     : integer; (*left,bottom corner to start print*)
    fXDiv                            : double; (*x Divisions*)
    fShowBorder                      : boolean;
    fDivisionPlacement               : THRDivisionPlacement;
    fLengthUnits                     : TLengthUnits;
    fTicColor                        : TColor;
    fAutoColor                       : Boolean;

    fBevelInner                      : TPanelBevel;(*bvNone, bvLowered, bvRaised*)
    fBevelOuter                      : TPanelBevel;(*bvNone, bvLowered, bvRaised*)
    fBevelWidth                      : integer;
    fBevelColor                      : TColor;
    fAutoDecimals                    : Boolean;

    procedure SetXMin(Value:double);
    procedure SetXMax(Value:double);
    procedure SetXDiv(Value:double);
    procedure SetDecimals(Value:byte);
    procedure SetLefttoRight(Value:boolean);
    Function  DoubleToString(d:double):string;
    procedure SetBorderBoolean(value:boolean);
    procedure SetDivisionPlacement(value:THRDivisionPlacement);
    procedure SetLengthUnits(value:TLengthUnits);
    procedure SetTicColor(value:TColor);
    procedure SetAutoColor(value:Boolean);
    procedure SetBevelInner(value : TPanelBevel);
    procedure SetBevelOuter(value : TPanelBevel);
    procedure SetBevelWidth(value : integer);
    procedure SetBevel(bI,bO:TPanelBevel;bW:integer);
    procedure SetBevelColor(value : TColor);
    procedure SetAutoDecimals(value:Boolean);


  protected
    Function  HighLightColor(color:TColor): TColor;
    Function  ShadowColor(color:TColor): TColor;
    Function  ProperColor(color:TColor): TColor;
    Function  SetProperDecimals : integer;
    procedure Paint; override;

  public
    PrintMagnification : Double;
    constructor Create(AOwner: TComponent); override;
    procedure   Print(Left,Top:double;StartPrint,EndPrint:boolean);
  published
    property Align default alNone;
    property Color;
    property Decimals:byte read fDecimals write SetDecimals default 0;
    property Height default 33;
    property XMin:double read fXMin write SetXMin;
    property XMax:double read fXMax write SetXMax;
    property XDiv:double read fXDiv write SetXDiv;
    property LeftToRight:boolean read fLeftToRight write SetLeftToRight default true;
    property ShowBorder: boolean read fShowBorder write SetBorderBoolean default true;
    property DivisionPlacement:THRDivisionPlacement read fDivisionPlacement write SetDivisionPlacement
             default dpTop;
    property LengthUnits :TLengthUnits read fLengthUnits write SetLengthUnits default luCustom;
    property Width default 768;
    property Visible;
    property Font;
    property ParentFont;
    property ParentColor;
    property TicColor : TColor read fTicColor write SetTicColor default clBlack;
    property AutoColor : Boolean read fAutoColor write SetAutoColor default true;
    property BevelInner:TPanelBevel read fBevelInner write SetBevelInner;
    property BevelOuter:TPanelBevel read fBevelOuter write SetBevelOuter;
    property BevelWidth:Integer read fBevelWidth write SetBevelWidth;
    property BevelColor:TColor read fBevelColor write SetBevelColor;
    property AutoDecimals:Boolean read fAutoDecimals write SetAutoDecimals;

  end;

(*******************************************************************************
                      Declarations for Vertical Ruler
*******************************************************************************)


  type
  TVRuler = class(TGraphicControl)
  private
    { Private declarations }
    fYMin,fYMax : double;
    fDecimals   : byte;
    fBottomToTop : boolean;
    fCurrentCanvas  : TCanvas;
    fpLeft,fpTop    : integer; (*left,Top corner to start print*)
    fYDiv           : double;
    fShowBorder     : boolean;
    fDivisionPlacement : TVRDivisionPlacement;
    fLengthUnits       : TLengthUnits;
    fTicColor          : TColor;
    fAutoColor                       : Boolean;

    fBevelInner                      : TPanelBevel;(*bvNone, bvLowered, bvRaised*)
    fBevelOuter                      : TPanelBevel;(*bvNone, bvLowered, bvRaised*)
    fBevelWidth                      : integer;
    fBevelColor                      : TColor;
    fAutoDecimals                    : Boolean;


    procedure SetYMin(Value:double);
    procedure SetYMax(Value:double);
    procedure SetYDiv(Value:double);
    procedure SetDecimals(Value:byte);
    procedure SetBottomToTop(Value:boolean);
    Function DoubleToString(d:double):string;
    procedure SetBorderBoolean(value:boolean);
    procedure SetDivisionPlacement(value:TVRDivisionPlacement);
    procedure SetLengthUnits(value:TLengthUnits);
    procedure SetTicColor(value:TColor);
    procedure SetAutoColor(value:Boolean);
    procedure SetBevelInner(value : TPanelBevel);
    procedure SetBevelOuter(value : TPanelBevel);
    procedure SetBevelWidth(value : integer);
    procedure SetBevel(bI,bO:TPanelBevel;bW:integer);
    procedure SetBevelColor(value : TColor);
    procedure SetAutoDecimals(value:Boolean);


  protected
    procedure Paint; override;
    Function HighLightColor(color:TColor): TColor;
    Function ShadowColor(color:TColor): TColor;
    Function  ProperColor(color:TColor): TColor;
    Function  SetProperDecimals : integer;

  public
    PrintMagnification : double;
    constructor Create(AOwner: TComponent); override;
    procedure Print(Left,Top:double;StartPrint,EndPrint:boolean);
  published
    property Align default alNone;
    property Color;
    property Decimals:byte read fDecimals write SetDecimals default 0;
    property Height default 200;
    property YMin:double read FYMin write SetYMin;
    property YMax:double read FYMax write SetYMax;
    property YDiv:double read fYDiv write SetYDiv;
    property BottomToTop : boolean read fBottomToTop write SetBottomToTop default true;
    property ShowBorder : boolean read fShowBorder write SetBorderBoolean default true;
    property DivisionPlacement:TVRDivisionPlacement read fDivisionPlacement write SetDivisionPlacement
             default dpLeft;
    property LengthUnits :TLengthUnits read fLengthUnits write SetLengthUnits default luCustom;
    property Width default 33;
    property Visible;
    property Font;
    property ParentFont;
    property ParentColor;
    property TicColor:TColor read fTicColor write SetTicColor default clBlack;
    property AutoColor : Boolean read fAutoColor write SetAutoColor default true;
    property BevelInner:TPanelBevel read fBevelInner write SetBevelInner;
    property BevelOuter:TPanelBevel read fBevelOuter write SetBevelOuter;
    property BevelWidth:Integer read fBevelWidth write SetBevelWidth;
    property BevelColor:TColor read fBevelColor write SetBevelColor;
    property AutoDecimals:Boolean read fAutoDecimals write SetAutoDecimals;

  end;

var
  Evaluation : byte;


(*******************************************************************************)
(*******************************************************************************)

implementation


(*******************************************************************************)


function Power (a,x:double):double;  { a^x }         {checked 6 FEB 88}
begin
  if a < 0 then                     {a is a negative double number}
     if frac(x) = 0 then            {x is an integer}
        if odd(Trunc(x)) then       {x is a odd integer}
           Power := -Exp(x*Ln(-a))
        else                        {x is a even integer}
           Power := Exp(x*Ln(-a))
     else                           {x is a double number}
        begin
          writeln ('Error: Negative double number was raised to a ',
                    'non-integer power');
          halt
        end
  else if a = 0 then                {a is zero}
    if x = 0 then                   {a=0 and x=0}
      begin
        writeln ('Error: Zero raised to the zero power is undefined');
        halt
      end
    else if x < 0 then               {a=0 and x<0}
      begin
        writeln ('Error: Zero raised to a negative power is undefined');
        halt
      end
    else                            {x is positive}
      Power := 0
  else if x = 0 then                {x is zero}
    Power := 1
  else
    Power := Exp(x*Ln(a))             {a is a positive double, x is double}
end;

(*******************************************************************************)

procedure THRuler.SetXMin(Value:double);
begin
  if fXMin <> Value then
    begin
      fXMin := Value;
      if visible then
        paint;
    end;
end;

(*******************************************************************************)

procedure THRuler.SetXMax(Value:double);

begin
  if fXMax <> Value then
    begin
      fXMax := Value;
      if visible then
        paint;
    end;
end;

(*******************************************************************************)

procedure THRuler.SetXDiv(Value:double);

begin
  if fXDiv <> Value then
    begin
      fXDiv := Value;
      if visible then
        paint;
    end;
end;

(*******************************************************************************)

procedure THRuler.SetDecimals(Value:byte);
begin
  if fDecimals <> Value then
    begin
      fDecimals := Value;
      if visible then
        paint;
    end;
end;

(*******************************************************************************)

procedure THRuler.SetLeftToRight(Value:boolean);
begin
  if fLeftToRight <> Value then
    begin
      fLeftToRight := Value;
      if visible then
        paint;
    end;
end;

(*******************************************************************************)

procedure THRuler.SetBorderBoolean(value:boolean);
begin
  if fShowBorder <> Value then
    begin
      fShowBorder := Value;
      if visible then
        paint;
    end;
end;

(*******************************************************************************)

procedure THRuler.SetDivisionPlacement(value:THRDivisionPlacement);
begin
  if fDivisionPlacement <> value then
      begin
        fDivisionPlacement := value;
        if visible then
          paint;
      end;
end;

(*******************************************************************************)

procedure THRuler.SetLengthUnits(value:TLengthUnits);
begin
  if fLengthUnits <> value then
      begin
        fLengthUnits := value;
        if visible then
          paint;
      end;
end;

(*******************************************************************************)

procedure THRuler.SetTicColor(value:TColor);
begin
  if fTicColor <> value then
    begin
      fTicColor := value;
      if visible then
        paint;
    end;
end;

procedure THRuler.SetAutoColor(value:Boolean);

begin
  if fAutoColor <> value then
    begin
      fAutoColor := value;
      if visible then
        paint;
    end;
end;

procedure THRuler.SetBevel(bI,bO:TPanelBevel;bW:integer);

var
  Rect : TRect;

begin
  Rect := ClientRect;
{$ifdef ver120 then}
  if bI = bvSpace then bI := bvRaised;
  if bO = bvSpace then bO := bvRaised;
{$endif}

  if (bI = bvNone) and (bO = bvNone) then exit;

  if (bI = bvNone) and (bO = bvRaised) then Frame3D(canvas,Rect,HighLightColor(fBevelColor),ShadowColor(fBevelColor),bW);
  if (bI = bvNone) and (bO = bvLowered) then Frame3D(canvas,Rect,ShadowColor(fBevelColor),HighLightColor(fBevelColor),bW);

  if (bO = bvNone) and (bI = bvRaised) then Frame3D(canvas,Rect,HighLightColor(fBevelColor),ShadowColor(fBevelColor),bW);
  if (bO = bvNone) and (bI = bvLowered) then Frame3D(canvas,Rect,ShadowColor(fBevelColor),HighLightColor(fBevelColor),bW);

  if (bO = bvRaised) and (bI = bvRaised) then Frame3D(canvas,Rect,HighLightColor(fBevelColor),ShadowColor(fBevelColor),2*bW);
  if (bO = bvLowered) and (bI = bvLowered) then Frame3D(canvas,Rect,ShadowColor(fBevelColor),HighLightColor(fBevelColor),2*bW);

  if (bO = bvRaised) and (bI = bvLowered) then
    begin
      Frame3D(canvas,Rect,HighLightColor(fBevelColor),ShadowColor(fBevelColor),bW);
      Frame3D(canvas,Rect,ShadowColor(fBevelColor),HighLightColor(fBevelColor),bW);
    end;
  if (bO = bvLowered) and (bI = bvRaised) then
    begin
      Frame3D(canvas,Rect,ShadowColor(fBevelColor),HighLightColor(fBevelColor),bW);
      Frame3D(canvas,Rect,HighLightColor(fBevelColor),ShadowColor(fBevelColor),bW);
    end;


end;

procedure THRuler.SetBevelOuter(value:TPanelBevel);

begin
  if value <> fBevelOuter then
    begin
      fBevelOuter := value;
      paint;
    end;
end;

procedure THRuler.SetBevelInner(value:TPanelBevel);

begin
  if value <> fBevelInner then
    begin
      fBevelInner := value;
      paint;
    end;
end;

procedure THRuler.SetBevelWidth(value:integer);

begin
  if value <> fBevelWidth then
    begin
      fBevelWidth := value;
      paint;
    end;
end;

procedure THRuler.SetBevelColor(value:TColor);

begin
  if value <> fBevelColor then
    begin
      fBevelColor := value;
      paint;
    end;
end;

procedure THRuler.SetAutoDecimals(value:Boolean);

begin
  if value <> fAutoDecimals then
    begin
      fAutoDecimals := value;
      paint;
    end;
end;

Function THRuler.DoubleToString(d:double):string;

begin
  str(d:0:fDecimals,result);
end;

(*******************************************************************************)

constructor THRuler.Create(AOwner: TComponent);

begin
  inherited Create(AOwner);
  inc(evaluation);
  parent := TWinControl(AOwner);
  fCurrentCanvas := canvas;
  fShowBorder := true;
  Color := TMyControl(AOwner).color;
  fBevelColor := Color;
  fAutoColor := True;
  XMin := 0;
  XMax := 100;
  XDiv := 0;
  Height := 33;
  Width := 200;
  decimals := 0;
  fAutoDecimals := false;
  font.name := 'arial';
  font.size := 6;
  font.color := ProperColor(color);
  fLeftToRight := true;
  PrintMagnification := 1;
  fDivisionPlacement := dpTop;
  fLengthUnits := luCustom;
  fTicColor := ProperColor(color);

  fBevelInner := bvNone;
  fBevelOuter := bvRaised;
  fBevelWidth := 1;
  {$IFDEF EVALUATION THEN}
  if not(csDesigning in ComponentState) and (evaluation < 2) then
    MessageDlg('This program uses an evaluation copy of QSRulers by Q-Systems Engineering',
               mtInformation,[mbOK],0);
  {$ENDIF}
end;


(******************************************************************************)

procedure THRuler.Paint;

var N           : integer;
    NX          : double;
    X           : double;
    DeltaX      : double;
    ratio       : double;
    i           : word;
    range       : double;
    Interval    : double;
    a16th       : double;
    a5mm        : double;
    dpi         : integer;

begin

  if (fAutoColor and ParentColor) then
    begin
      Font.Color := ProperColor(color);
      fTicColor := ProperColor(color);
      fBevelColor := color;
    end;

  if fAutoDecimals then
    fDecimals := SetProperDecimals;

  if fCurrentCanvas = printer.canvas then
    begin
      ratio := 1.0*Printer.PageWidth / Screen.width;
      dpi := GetDeviceCaps(printer.handle,LOGPIXELSX);
    end
  else
    begin
      ratio := 1;
      fpLeft := 0;
      fpTop := 0;
      PrintMagnification := 1;
      dpi  := Screen.PixelsPerInch;
    end;

  if fXMax <= fXMin then fXMax := fXMin + 1;

  ratio := PrintMagnification*ratio;

  range := fXMax - fXMin;


(*******************************************************************************
                         AUTOMATIC DIVISION
The goal in automatic division is to generate approximately 10 divisions and
force divisions to be in intervals of 1,2, or 5 multiplied by the appropriate
power of 10.  For example if Xmin = 0 and XMax = 3 it shall generate XDiv = 0.5
if xMin = 1.2 and xMax = 10.1 it will generate XDiv = 1 as follows:
1.2, 2, 3, 4, 5, 6, 7, 8, 9, 10, 10.1 and so on.
*******************************************************************************)
  if fXDiv <= 0 then (*Automatic division*)
    begin
      interval := range / 10;

      if interval < 1 then
        begin
          i := 0;
          repeat
            inc(i);
            interval := interval*10;
          until trunc(interval) > 0; (*See how many powers of ten below 1 it is*)

         interval := trunc(interval);

         if not (interval = 1) and (not (interval = 2)) then
           begin
             if interval < 5 then interval := 5
             else interval := 10;
           end;
         interval := interval/power(10,i);
        end

      else if interval < 10 then
        begin
          interval := trunc(interval);
          if not (interval = 1) and (not (interval = 2)) then
            begin
              if interval < 5 then interval := 5
              else interval := 10;
            end;
        end

      else if interval >= 10 then
        begin
          i := 0;
          repeat
            inc(i);
            interval := interval*0.1;
          until trunc(interval) < 10;
          interval := trunc(interval);
          if not (interval = 1) and (not (interval = 2)) then
            begin
              if interval < 5 then interval := 5
              else interval := 10;
            end;
          interval := interval*power(10,i);
        end;

      DeltaX := interval;
      N := 0;

      (*Find the first value x from which we start adding xDiv.  If the range
        is 1.2 to 9.8, we wish to generate a XDiv = 1 with values 2,3,4,..9,
        surrounded by the values of 1.2 and 9.8 that do not fall on any division*)
      if fXMin >= 0 then
        begin
          i := 0;
          repeat
            x := i*interval;
            inc(i);
          until x >= fXMin
        end
      else
        begin
          i := 0;
          repeat
            x := -i*interval;
            inc(i);
          until x < fXMin;
          x := x+interval;
        end;
      try
        if fLeftToRight then
          NX := round(width*(x-fXMin)/Range)
        else
          NX := width - round(width*(x-fXMin)/Range);
      except
      end;
    end  (* if fXDiv <= 0*)

(*******************************************************************************
                         MANUAL DIVISION
Here the division is specified by the user.  The only concern is that it will occur
on numbers that are easy to follow.  For example, if
XMin = 0.6, XMax = 8.2, and XDiv = 1, it makes no sense to have divisions at 0.6,
1.6, 2.6, and so on.  In stead it makes a lot of sence to have:
0.6, 1, 2, 3,..7,8,8.2!
*******************************************************************************)

  else  (* The x interval is as specified*)
    begin
      interval := fXDiv;
      DeltaX := interval;
      N := 0;

      (*Find the first X to start the divisions as explained in the comment above*)
      if fXMin >= 0 then
        begin
          i := 0;
          repeat
            x := i*interval;
            inc(i);
          until x >= fXMin
        end
      else
        begin
          i := 0;
          repeat
            x := -i*interval;
            inc(i);
          until x < fXMin;
          x := x+interval;
        end;

      try
        if fLeftToRight then
          NX := round(width*(x-fXMin)/Range)
        else
          NX := width - round(width*(x-fXMin)/Range);
      except
      end;
    end;

  fCurrentCanvas.font := font; 
  fCurrentCanvas.font.size := round(font.size*Ratio);

  with fCurrentCanvas do
    begin
      Brush.Color := Color;
      if fShowBorder then pen.Color := clBlack
      else pen.Color := Color;
      with ClientRect do
        begin
          Rectangle(fpLeft + round(ratio*Left), fpTop + round(ratio*Top),
                   fpLeft+round(ratio*Right), fpTop+round(ratio*Bottom));
          Pen.Color := fTicColor;
          pen.style := psSolid;
        end;

      if fLengthUnits = luCustom then
        begin
          while X <= fXMax do
            begin
              if fDivisionPlacement = dpTop then
                begin
                  MoveTo(fpLeft + round(ratio*NX), fpTop + round(ratio*1)-byte(not fShowBorder));
                  LineTo(fpLeft + round(ratio*NX),
                         fpTop + round(ratio*((height div 5)*(1 + byte(N mod 2 = 0) + byte(N mod 10 = 0)))));
                end
              else
                begin
                  MoveTo(fpLeft + round(ratio*NX), fpTop + fpTop+round(ratio*ClientRect.Bottom));
                  LineTo(fpLeft + round(ratio*NX),
                         fpTop+round(ratio*ClientRect.Bottom) - round(ratio*((height div 5)*(1 + byte(N mod 2 = 0) +
                         byte(N mod 10 = 0)))));
                end;

              brush.style := bsClear;
              if (N >= 0) and (N mod 2 = 0) then
                textout(penPos.X+3,(ClientRect.Bottom-TextHeight('2')) div 2,DoubleToString(X));

              brush.style := bsSolid;
              N := N + 1;
              X := X + DeltaX;

              try
                if fLeftToRight then
                  NX := round(width*(x-fXMin)/Range)
                else
                  NX := width - round(width*(x-fXMin)/Range);
              except
              end;
            end;
        end
      else if fLengthUnits = luActualInches then
        begin
          if fLeftToRight then
            begin
              a16th := round(dpi/16.0);
              N := 1;  (*Skip the zero*)
              NX := 1; (*Skip the zero*)
              while nx < ratio*width do
                begin
                  if fDivisionPlacement = dpTop then
                    begin
                      Moveto(fpLeft+round(nx),fpTop+0);
                      (*multiplier round(height/2/5) forces the longest division to be half height*)
                      Lineto(fpLeft+round(nx),fpTop+round(ratio*height/2.0/5.0)*(1+(byte(n mod 2 = 0)+
                                     byte(N mod 4 = 0)+
                                     byte(N mod 8 = 0)+
                                     byte(N mod 16 = 0))));
                    end
                  else
                    begin
                      Moveto(fpLeft+round(nx),fpTop+round(ratio*(clientRect.bottom-1)));
                      Lineto(fpLeft+round(nx), fpTop+
                            round(ratio*(ClientRect.bottom-1-round(height/2.0/5.0)*
                                  (1+(byte(n mod 2 = 0)+
                                     byte(N mod 4 = 0)+
                                     byte(N mod 8 = 0)+
                                     byte(N mod 16 = 0))))));
                    end;
                  if (N > 0) and (n mod 16 = 0) then
                    textout(penPos.X+round(3*Ratio),fpTop+(round(ratio*ClientRect.Bottom)-TextHeight('2')) div 2,
                                   InttoStr(N Div 16));
                  n := n+1;
                  nx := nx+a16th;
                end;
            end
          else
            begin
              a16th := round(dpi/16.0);
              N := 1;
              NX := Ratio*Width;
              while nx > 0 do
                begin
                  if fDivisionPlacement = dpTop then
                    begin
                      Moveto(fpLeft+round(nx),fpTop+0);
                      (*multiplier round(height/2/5) forces the longest division to be half height*)
                      Lineto(fpLeft+round(nx),fpTop+round(ratio*height/2.0/5.0)*(1+(byte(n mod 2 = 0)+
                                     byte(N mod 4 = 0)+
                                     byte(N mod 8 = 0)+
                                     byte(N mod 16 = 0))));
                    end
                  else
                    begin
                      Moveto(fpLeft+round(nx),fpTop+round(ratio*(clientRect.bottom-1)));
                      Lineto(fpLeft+round(nx), fpTop+
                            round(ratio*(ClientRect.bottom-1-round(height/2.0/5.0)*
                                  (1+(byte(n mod 2 = 0)+
                                     byte(N mod 4 = 0)+
                                     byte(N mod 8 = 0)+
                                     byte(N mod 16 = 0))))));
                    end;
                  if (N > 0) and (n mod 16 = 0) then
                    textout(penPos.X+round(3*Ratio),fpTop+(round(ratio*ClientRect.Bottom)-TextHeight('2')) div 2,
                                   InttoStr(N Div 16));
                  n := n+1;
                  nx := nx-a16th;
                end;
            end
        end
      else if fLengthUnits = luActualMM then
        begin
          if fLeftToRight then
            begin
              a5mm := round(dpi/(2.54*2));
              N := 0;
              NX := 0;
              while nx < ratio*width do
                begin
                  if fDivisionPlacement = dpTop then
                    begin
                      Moveto(fpLeft+round(nx),fpTop+0);
                      Lineto(fpLeft+round(nx),fpTop+round(ratio*height/2.0/2.0)*((byte(n mod 1 = 0)+
                                     byte(N mod 2 = 0))));
                    end
                  else
                    begin
                      Moveto(fpLeft+round(nx),fpTop+round(ratio*(clientRect.bottom-1)));
                      Lineto(fpLeft+round(nx),
                             fpTop+round(ratio*(ClientRect.bottom-1-round(height/2.0/2.0)*
                             ((byte(n mod 1 = 0)+
                                     byte(N mod 2 = 0))))));
                    end;
                  if (N > 0) and (n mod 2 = 0)  then
                    textout(penPos.X+round(3*ratio),
                      fpTop+(round(ratio*ClientRect.Bottom)-TextHeight('2')) div 2,
                      InttoStr(N Div 2));
                  n := n+1;
                  nx := nx+a5mm;
                end;
            end
          else
            begin
              a5mm := round(dpi/(2.54*2));
              N := 0;
              NX := ratio*width;
              while nx > 0 do
                begin
                  if fDivisionPlacement = dpTop then
                    begin
                      Moveto(fpLeft+round(nx),fpTop+0);
                      Lineto(fpLeft+round(nx),fpTop+round(ratio*height/2.0/2.0)*((byte(n mod 1 = 0)+
                                     byte(N mod 2 = 0))));
                    end
                  else
                    begin
                      Moveto(fpLeft+round(nx),fpTop+round(ratio*(clientRect.bottom-1)));
                      Lineto(fpLeft+round(nx), fpTop+round(ratio*(ClientRect.bottom-1-round(height/2.0/2.0)*
                             ((byte(n mod 1 = 0)+
                                     byte(N mod 2 = 0))))));
                    end;
                  if (N > 0) and (n mod 2 = 0) then
                    textout(penPos.X+round(3*ratio),
                      fpTop+(round(ratio*ClientRect.Bottom)-TextHeight('2')) div 2,
                      InttoStr(N Div 2));
                  n := n+1;
                  nx := nx-a5mm;
                end;
            end;
        end;
    end;
  SetBevel(fBevelInner,fBevelOuter,fBevelWidth);  
end;


{*********************************************}

Function THRuler.HighLightColor(color:TColor): TColor;

var
  ir,ig,ib : integer;
  sr,sg,sb : string;
  i        : integer;
  s        : string;

begin
  if Color = clBlack then
    begin
      result := rgb(50,50,50);
      exit;
    end;
  i := ColorToRGB(Color);
  s := IntToHex(i,6);
  sb := Copy(s,1,2);
  sg := Copy(s,3,2);
  sr := Copy(s,5,2);
  ib := StrToInt('$'+sb);
  ig := StrToInt('$'+sg);
  ir := strToInt('$'+sr);
  if round(1.2*ir) > 255 then ir := 255 else ir := round(1.2*ir);
  if ir < 100 then ir := 100;
  if round(1.2*ig) > 255 then ig := 255 else ig := round(1.2*ig);
  if ig < 100 then ig := 100;
  if round(1.2*ib) > 255 then ib := 255 else ib := round(1.2*ib);
  if ib < 100 then ib := 100;
  result := rgb(ir,ig,ib);
end;

{*********************************************}

Function THRuler.ShadowColor(color:TColor): TColor;

var
  ir,ig,ib : integer;
  sr,sg,sb : string;
  i        : integer;
  s        : string;

begin
  i := ColorToRGB(Color);
  s := IntToHex(i,6);
  sb := Copy(s,1,2);
  sg := Copy(s,3,2);
  sr := Copy(s,5,2);
  ib := StrToInt('$'+sb);
  ig := StrToInt('$'+sg);
  ir := strToInt('$'+sr);
  ir := round(0.8*ir);
  if ir > 160 then ir := 160;
  ig := round(0.8*ig);
  if ig > 160 then ig := 160;
  ib := round(0.8*ib);
  if ib > 160 then ib := 160;
  result := rgb(ir,ig,ib);
end;

{*********************************************}

Function THRuler.ProperColor(color:TColor): TColor;

var
  ir,ig,ib : integer;
  sr,sg,sb : string;
  i        : integer;
  s        : string;

begin
  i := ColorToRGB(Color);
  s := IntToHex(i,6);
  sb := Copy(s,1,2);
  sg := Copy(s,3,2);
  sr := Copy(s,5,2);
  ib := StrToInt('$'+sb);
  ig := StrToInt('$'+sg);
  ir := strToInt('$'+sr);
  if (ir+ig+ib) > 1.5*255 then (*light enought*)
    result := clBlack
  else
    result := clWhite;
end;

{*********************************************}

Function  THRuler.SetProperDecimals : integer;

var
  r : double;

begin
  r := abs(fXMax - fXMin);

  if r > 5 then result := 0
  else if r > 0.5 then result := 1
  else if r > 0.05 then result := 2
  else result := 3;
end;

{*********************************************}

procedure THRuler.print(Left,Top:double;StartPrint,EndPrint:boolean);

(*Left and Top is the left and Top beginning of printing in inches
  within this procedure they will be changed to printer pixels*)

var
  MMprinterWidth : integer;
  InchprinterWidth : double;
  HorizontalDots : integer;
  dpi    : integer;

begin
  MMPrinterWidth := GetDeviceCaps(printer.handle,HorzSize); (* in mm  *)
  InchPrinterWidth := MMPrinterWidth/25.4;
  HorizontalDots := GetDeviceCaps(printer.handle,HorzRes); (* total dots  *)
  dpi := round(1.0*HorizontalDots/InchPrinterWidth); (* dots per inch;  *)


  fpLeft := round(Left*dpi);
  fpTop := round(Top*dpi);

  fCurrentCanvas := printer.canvas;
  if StartPrint then
    printer.beginDoc;
  paint;
  if EndPrint then
    printer.endDoc;
  fCurrentCanvas := canvas;
end;


procedure TVRuler.SetYMin(Value:double);
begin
  if fYMin <> Value then
    begin
      fYMin := Value;
      if visible then
        paint;
    end;
end;

procedure TVRuler.SetYMax(Value:double);

begin
  if fYMax <> Value then
    begin
      fYMax := Value;
      if visible then
        paint;
    end;
end;


procedure TVRuler.SetYDiv(Value:double);

begin
  if fYDiv <> Value then
    begin
      fYDiv := Value;
      if visible then
        paint;
    end;
end;

procedure TVRuler.SetDecimals(Value:byte);
begin
  if fDecimals <> Value then
    begin
      fDecimals := Value;
      if visible then
        paint;
    end;
end;

procedure TVRuler.SetBottomToTop(Value:boolean);
begin
  if fBottomToTop <> Value then
    begin
      fBottomToTop := Value;
      if visible then
        paint;
    end;
end;

procedure TVRuler.SetBorderBoolean(value:boolean);
begin
  if fShowBorder <> Value then
    begin
      fShowBorder := Value;
      if visible then
        paint;
    end;
end;


(*******************************************************************************)

procedure TVRuler.SetDivisionPlacement(value:TVRDivisionPlacement);
begin
  if fDivisionPlacement <> value then
      begin
        fDivisionPlacement := value;
        if visible then
          paint;
      end;
end;

(*******************************************************************************)

procedure TVRuler.SetLengthUnits(value:TLengthUnits);
begin
  if fLengthUnits <> value then
      begin
        fLengthUnits := value;
        if visible then
          paint;
      end;
end;

(*******************************************************************************)

procedure TVRuler.SetTicColor(value:TColor);
begin
  if fTicColor <> value then
    begin
      fTicColor := value;
      if visible then
        paint;
    end;
end;


procedure TVRuler.SetAutoColor(value:Boolean);

begin
  if fAutoColor <> value then
    begin
      fAutoColor := value;
      if visible then
        paint;
    end;
end;

procedure TVRuler.SetBevel(bI,bO:TPanelBevel;bW:integer);

var
  Rect : TRect;

begin
  Rect := ClientRect;

{$ifdef ver120 then}
  if bI = bvSpace then bI := bvRaised;
  if bO = bvSpace then bO := bvRaised;
{$endif}

  if (bI = bvNone) and (bO = bvNone) then exit;

  if (bI = bvNone) and (bO = bvRaised) then Frame3D(canvas,Rect,HighLightColor(fBevelColor),ShadowColor(fBevelColor),bW);
  if (bI = bvNone) and (bO = bvLowered) then Frame3D(canvas,Rect,ShadowColor(fBevelColor),HighLightColor(fBevelColor),bW);

  if (bO = bvNone) and (bI = bvRaised) then Frame3D(canvas,Rect,HighLightColor(fBevelColor),ShadowColor(fBevelColor),bW);
  if (bO = bvNone) and (bI = bvLowered) then Frame3D(canvas,Rect,ShadowColor(fBevelColor),HighLightColor(fBevelColor),bW);

  if (bO = bvRaised) and (bI = bvRaised) then Frame3D(canvas,Rect,HighLightColor(fBevelColor),ShadowColor(fBevelColor),2*bW);
  if (bO = bvLowered) and (bI = bvLowered) then Frame3D(canvas,Rect,ShadowColor(fBevelColor),HighLightColor(fBevelColor),2*bW);

  if (bO = bvRaised) and (bI = bvLowered) then
    begin
      Frame3D(canvas,Rect,HighLightColor(fBevelColor),ShadowColor(fBevelColor),bW);
      Frame3D(canvas,Rect,ShadowColor(fBevelColor),HighLightColor(fBevelColor),bW);
    end;
  if (bO = bvLowered) and (bI = bvRaised) then
    begin
      Frame3D(canvas,Rect,ShadowColor(fBevelColor),HighLightColor(fBevelColor),bW);
      Frame3D(canvas,Rect,HighLightColor(fBevelColor),ShadowColor(fBevelColor),bW);
    end;
end;

(**********************************************************************)

procedure TVRuler.SetBevelOuter(value:TPanelBevel);

begin
  if value <> fBevelOuter then
    begin
      fBevelOuter := value;
      paint;
    end;
end;

procedure TVRuler.SetBevelInner(value:TPanelBevel);

begin
  if value <> fBevelInner then
    begin
      fBevelInner := value;
      paint;
    end;
end;

(**********************************************************************)

procedure TVRuler.SetBevelWidth(value:integer);

begin
  if value <> fBevelWidth then
    begin
      fBevelWidth := value;
      paint;
    end;
end;

(**********************************************************************)

procedure TVRuler.SetBevelColor(value:TColor);

begin
  if value <> fBevelColor then
    begin
      fBevelColor := value;
      paint;
    end;
end;

(**********************************************************************)

procedure TVRuler.SetAutoDecimals(value:Boolean);

begin
  if value <> fAutoDecimals then
    begin
      fAutoDecimals := value;
      paint;
    end;
end;

(**********************************************************************)

constructor TVRuler.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  inc(evaluation);
  parent := TWinControl(AOwner);
  fCurrentCanvas := Canvas;
  fShowBorder := true;
  Color := TMyControl(AOwner).color;
  fBevelColor := Color;
  fAutoColor := True;
  YMin := 0;
  YMax := 100;
  YDiv := 0;
  Height := 200;
  Width := 33;
  decimals := 0;
  fAutoDecimals := false;
  font.color := clBlack;
  font.color := ProperColor(color);
  font.name := 'arial';
  font.size := 6;
  fBottomToTop := true;
  PrintMagnification := 1;
  fDivisionPlacement := dpLeft;
  fLengthUnits := luCustom;
  fTicColor := ProperColor(color);
  fBevelInner := bvNone;
  fBevelOuter := bvRaised;
  fBevelWidth := 1;

  {$IFDEF EVALUATION THEN}
  if not(csDesigning in ComponentState) and (evaluation < 2) then
    MessageDlg('This program uses an evaluation copy of QSRulers by Q-Systems Engineering',
                mtInformation,[mbOK],0);
  {$ENDIF}
end;

(*******************************************************************************)

Function TVRuler.DoubleToString(d:double):string;

begin
  str(d:0:fDecimals,result);
end;

(*******************************************************************************)

procedure TVRuler.Paint;


var N           : longInt;
    NY          : double;
    Y           : double;
    DeltaY       : double;
    ratio       : double;

    range       : double;
    Interval    : double;
    i           : word;
    a16th       : double;
    a5MM        : double;
    dpi         : integer;

begin

  if (fAutoColor and ParentColor) then
    begin
      Font.Color := ProperColor(color);
      fTicColor := ProperColor(color);
      fBevelColor := color;
    end;

  if fAutoDecimals then
    fDecimals := SetProperDecimals;
    
  if fCurrentCanvas = printer.canvas then
    begin
      ratio := Printer.PageWidth / Screen.Width;
      dpi   := GetDeviceCaps(printer.handle,LOGPIXELSX);
    end
  else
    begin
      ratio := 1;
      fpLeft := 0;
      fpTop := 0;
      PrintMagnification := 1;
      dpi := Screen.pixelsPerInch;
    end;

  ratio := PrintMagnification*ratio;

  range := fYMax - fYMin;

(*  fCurrentCanvas.Font.Size := round(Font.Size*Ratio); *)

  if fYDiv <= 0 then
    begin
      interval := range/10;

      if interval < 1 then
        begin
          i := 0;
          repeat
            inc(i);
            interval := interval*10;
          until trunc(interval) > 0;
         interval := trunc(interval);
         if not (interval = 1) and (not (interval = 2)) then
           begin
             if interval < 5 then interval := 5
             else interval := 10;
           end;
         interval := interval/power(10,i);
        end

      else if interval < 10 then
        begin
          interval := trunc(interval);
          if not (interval = 1) and (not (interval = 2)) then
            begin
              if interval < 5 then interval := 5
              else interval := 10;
            end;
        end

      else if interval >= 10 then
        begin
          i := 0;
          repeat
            inc(i);
            interval := interval*0.1;
          until trunc(interval) < 10;
          interval := trunc(interval);
          if not (interval = 1) and (not (interval = 2)) then
            begin
              if interval < 5 then interval := 5
              else interval := 10;
            end;
          interval := interval*power(10,i);
        end;


      DeltaY := interval;
      N := 0;

      if FYMin >= 0 then
        begin
          i := 0;
          repeat
            y := i*interval;
            inc(i);
          until y >= FYMin
        end
      else
        begin
          i := 0;
          repeat
            y := -i*interval;
            inc(i);
          until y < FYMin;
          y := y+interval;
        end;


      if fBottomToTop then
        NY := height - round(height*(y-FYMin)/Range)
      else
        NY := round(height*(y-FYMin)/Range);
    end (* if fYDiv <= 0*)
  else
    begin
      interval := fYDiv;
      DeltaY := interval;
      N := 0;

      if FYMin >= 0 then
        begin
          i := 0;
          repeat
            y := i*interval;
            inc(i);
          until y >= FYMin
        end
      else
        begin
          i := 0;
          repeat
            y := -i*interval;
            inc(i);
          until y < FYMin;
          y := y+interval;
        end;

      if fBottomToTop then
        NY := height - round(height*(y-FYMin)/Range)
      else
        NY := round(height*(y-FYMin)/Range);
    end;

  fCurrentCanvas.Font := font;
  fCurrentCanvas.Font.Size := round(Font.Size*Ratio);

  with fCurrentCanvas do
  begin
    Brush.Color := Color;
    if fShowBorder then pen.Color := clBlack
      else pen.Color := Color;
    with ClientRect do
      begin
        Rectangle(fpLeft + round(ratio*Left), fpTop + round(ratio*Top),
                  fpLeft+round(ratio*Right), fpTop+round(ratio*Bottom));
        Pen.Color := fTicColor;
        pen.style := psSolid;
      end;

    if fLengthUnits = luCustom then
      begin
        while Y <= FYMax do
          begin
            if fDivisionPlacement = dpRight then
              begin
                MoveTo(fpLeft + round(ratio*ClientRect.Right), fpTop + round(ratio*NY));
                LineTo(fpLeft + round(ratio*ClientRect.Right) - round(ratio*((width div 5)*(2 + byte(N mod 2 = 0) +
                byte(N mod 10 = 0)))),
                        fpTop + round(ratio*NY));
              end
            else
              begin
                MoveTo(fpLeft + round(ratio*1)-byte(not fShowBorder), fpTop + round(ratio*NY));
                LineTo(fpLeft + round(ratio*((width div 5)*(2 + byte(N mod 2 = 0) + byte(N mod 10 = 0)))),
                        fpTop + round(ratio*NY));
              end;

            brush.style := bsClear;
            if (N > 0) and (N mod 2 = 0) then
              TextOut(fpLeft + round(ratio*(width div 3)), PenPos.Y-round(ratio*font.size*2), DoubleToString(Y));
            brush.style := bsSolid;
            N := N + 1;
            Y := Y + DeltaY;
            if fBottomToTop then
              NY := height - round(height*(y-FYMin)/Range)
            else
              NY := round(height*(y-FYMin)/Range);
          end;
      end
    else if fLengthUnits = luActualInches then
      begin
        if bottomToTop then
          begin
            a16th := round(dpi/16.0);
            N := 0;
            NY := ratio*height;
            while NY > 0 do
              begin
                if fDivisionPlacement = dpRight then
                  begin
                    MoveTo(fpLeft + round(ratio*ClientRect.Right), fpTop + round(NY));
                    LineTo(fpLeft + round(ratio*ClientRect.Right) -
                    round(ratio*width/2.0/5.0)*
                     (1+(byte(n mod 2 = 0)+byte(N mod 4 = 0)+byte(N mod 8 = 0)
                       +byte(n mod 16 = 0))),
                            fpTop + round(NY));
                  end
                else
                  begin
                    MoveTo(fpLeft, fpTop + round(NY));
                    LineTo(fpLeft +
                    round(ratio*width/2.0/5.0)*
                     (1+(byte(n mod 2 = 0)+byte(N mod 4 = 0)+byte(N mod 8 = 0)
                       +byte(n mod 16 = 0))),
                            fpTop + round(NY));
                  end;

                brush.style := bsClear;
                if (N > 0) and (N mod 16 = 0) then
                  TextOut(fpLeft + (round(Ratio*ClientRect.Right)-TextWidth(IntToStr(N div 16))) div 2 ,
                            PenPos.Y-round(font.size*2),
                            IntToStr(N div 16));
                brush.style := bsSolid;
                N := N + 1;
                NY := NY - a16th;
              end;
          end
        else
          begin
            a16th := round(dpi/16.0);
            N := 0;
            NY := 0;
            while NY < ratio*height do
              begin
                if fDivisionPlacement = dpRight then
                  begin
                    MoveTo(fpLeft + round(ratio*ClientRect.Right), fpTop + round(NY));
                    LineTo(fpLeft + round(ratio*ClientRect.Right) -
                    round(ratio*width/2.0/5.0)*
                     (1+(byte(n mod 2 = 0)+byte(N mod 4 = 0)+byte(N mod 8 = 0)
                       +byte(n mod 16 = 0))),
                            fpTop + round(NY));
                  end
                else
                  begin
                    MoveTo(fpLeft, fpTop + round(NY));
                    LineTo(fpLeft +
                    round(ratio*width/2.0/5.0)*
                     (1+(byte(n mod 2 = 0)+byte(N mod 4 = 0)+byte(N mod 8 = 0)
                       +byte(n mod 16 = 0))),
                            fpTop + round(NY));
                  end;

                brush.style := bsClear;
                if (N > 0) and (N mod 16 = 0)  then
                  TextOut(fpLeft + (round(Ratio*ClientRect.Right)-TextWidth(IntToStr(N div 16))) div 2 ,
                            PenPos.Y-round(font.size*2),
                            IntToStr(N div 16));
                brush.style := bsSolid;
                N := N + 1;
                NY := NY + a16th;
              end;
          end
      end
    else if fLengthUnits = luActualMM then
      begin
        if BottomToTop then
          begin
            a5mm := round(dpi/(2.54*2));
            N := 0;
            NY := ratio*height;
            while NY > 0 do
              begin
                if fDivisionPlacement = dpRight then
                  begin
                    MoveTo(fpLeft + round(ratio*ClientRect.Right), fpTop + round(NY));
                    LineTo(fpLeft + round(ratio*ClientRect.Right) -
                    round(ratio*width/2.0/2.0)*
                     ((byte(n mod 1 = 0)+byte(N mod 2 = 0))),
                            fpTop + round(NY));
                  end
                else
                  begin
                    MoveTo(fpLeft, fpTop + round(NY));
                    LineTo(fpLeft +
                    round(ratio*width/2.0/2.0)*
                     ((byte(n mod 1 = 0)+byte(N mod 2 = 0))),
                            fpTop + round(NY));
                  end;
                brush.style := bsClear;
                if (N > 0) and (N mod 2 = 0) then
                  TextOut(fpLeft + (round(Ratio*ClientRect.Right)-TextWidth(IntToStr(N div 2))) div 2 ,
                            PenPos.Y-round(font.size*2),
                            IntToStr(N div 2));
                brush.style := bsSolid;
                N := N + 1;
                NY := NY - a5mm;
              end;
          end
        else
          begin
            a5mm := round(dpi/(2.54*2));
            N := 0;
            NY := 0;
            while NY < ratio*Height do
              begin
                if fDivisionPlacement = dpRight then
                  begin
                    MoveTo(fpLeft + round(ratio*ClientRect.Right), fpTop + round(NY));
                    LineTo(fpLeft + round(ratio*ClientRect.Right) -
                    round(ratio*width/2.0/2.0)*
                     ((byte(n mod 1 = 0)+byte(N mod 2 = 0))),
                            fpTop + round(NY));
                  end
                else
                  begin
                    MoveTo(fpLeft, fpTop + round(NY));
                    LineTo(fpLeft +
                    round(ratio*width/2.0/2.0)*
                     ((byte(n mod 1 = 0)+byte(N mod 2 = 0))),
                            fpTop + round(NY));
                  end;
                brush.style := bsClear;
                if (N > 0) and (N mod 2 = 0) then
                  TextOut(fpLeft + (round(Ratio*ClientRect.Right)-TextWidth(IntToStr(N div 2))) div 2 ,
                            PenPos.Y-round(font.size*2),
                            IntToStr(N div 2));
                brush.style := bsSolid;
                N := N + 1;
                NY := NY + a5mm;
              end;
          end
      end;
  end;
  SetBevel(fBevelInner,fBevelOuter,fBevelWidth);
end;

{*********************************************}

Function TVRuler.HighLightColor(color:TColor): TColor;

var
  ir,ig,ib : integer;
  sr,sg,sb : string;
  i        : integer;
  s        : string;

begin
  if Color = clBlack then
    begin
      result := rgb(50,50,50);
      exit;
    end;
  i := ColorToRGB(Color);
  s := IntToHex(i,6);
  sb := Copy(s,1,2);
  sg := Copy(s,3,2);
  sr := Copy(s,5,2);
  ib := StrToInt('$'+sb);
  ig := StrToInt('$'+sg);
  ir := strToInt('$'+sr);
  if round(1.2*ir) > 255 then ir := 255 else ir := round(1.2*ir);
  if ir < 100 then ir := 100;
  if round(1.2*ig) > 255 then ig := 255 else ig := round(1.2*ig);
  if ig < 100 then ig := 100;
  if round(1.2*ib) > 255 then ib := 255 else ib := round(1.2*ib);
  if ib < 100 then ib := 100;
  result := rgb(ir,ig,ib);
end;

{*********************************************}

Function TVRuler.ShadowColor(color:TColor): TColor;

var
  ir,ig,ib : integer;
  sr,sg,sb : string;
  i        : integer;
  s        : string;

begin
  i := ColorToRGB(Color);
  s := IntToHex(i,6);
  sb := Copy(s,1,2);
  sg := Copy(s,3,2);
  sr := Copy(s,5,2);
  ib := StrToInt('$'+sb);
  ig := StrToInt('$'+sg);
  ir := strToInt('$'+sr);
  ir := round(0.8*ir);
  if ir > 160 then ir := 160;
  ig := round(0.8*ig);
  if ig > 160 then ig := 160;
  ib := round(0.8*ib);
  if ib > 160 then ib := 160;
  result := rgb(ir,ig,ib);
end;


{*********************************************}

Function TVRuler.ProperColor(color:TColor): TColor;

var
  ir,ig,ib : integer;
  sr,sg,sb : string;
  i        : integer;
  s        : string;

begin
  i := ColorToRGB(Color);
  s := IntToHex(i,6);
  sb := Copy(s,1,2);
  sg := Copy(s,3,2);
  sr := Copy(s,5,2);
  ib := StrToInt('$'+sb);
  ig := StrToInt('$'+sg);
  ir := strToInt('$'+sr);
  if (ir+ig+ib) > 1.5*255 then (*light enought*)
    result := clBlack
  else
    result := clWhite;
end;

{*********************************************}

Function  TVRuler.SetProperDecimals : integer;

var
  r : double;

begin
  r := abs(fYMax - fYMin);
  if r > 5 then result := 0
  else if r > 0.5 then result := 1
  else if r > 0.05 then result := 2
  else result := 3;
end;

{*********************************************}
procedure TVRuler.print(Left,Top:double;StartPrint,EndPrint:boolean);

(*Left and Top is the left and Top beginning of printing in inches
  within this procedure they will be changed to printer pixels*)

var
  dpi    : integer;
  MMprinterHeight : integer;
  InchprinterHeight : double;
  VerticalDots : integer;


begin

  MMPrinterHeight := GetDeviceCaps(printer.handle,VertSize); (* in mm *)
  InchPrinterHeight := MMPrinterHeight/25.4;
  VerticalDots := GetDeviceCaps(printer.handle,VertRes); (* total dots *)
  dpi := round(1.0*VerticalDots/InchPrinterHeight); (* dots per inch; *)

  fpLeft := round(Left*dpi);
  fpTop := round(Top*dpi);
  fCurrentCanvas := printer.canvas;
  if StartPrint then
    printer.beginDoc;
  paint;
  if EndPrint then
    printer.endDoc;
  fCurrentCanvas := canvas;
end;

initialization
  evaluation := 0;
end.
