unit WGraph;

interface

Uses
  Graphics,WinTypes,ExtCtrls,SysUtils,winProcs,Dialogs,alloc;

{$ifdef VER110} {CBUILDER 3}
{$ObjExportAll On}
{$ENDIF}

const
  LIMIT = 32767;


Type

  TAxis = record
    ShiftX,ShiftY                    : Integer;
    xmin,ymin,xmax,ymax              : double;
    TicSpacing                       : double;
    SecTics,Labels,Grid              : boolean;
    AxisWidth,GridWidth              : integer;
    AxisColor,GridColor              : TColor;
    FontColor                        : TColor;
    FontName                         : TFontName;
    FontSize                         : integer;
    FontStyle                        : TFontStyles;
    decimals                         : integer;
    Title                            : string;
    TitleFontSize                    : integer;
  end;

(*****************************************************************************)

  pointType = (square,triangle,circle,emptysquare,emptytriangle,emptycircle,crosspoint,nothing);

  TLineFormat = record
    Points      : word;
    PType       : pointType;
    PColor      : TColor;
    PSize       : integer;
    LineColor   : TColor;
    LineWidth   : integer;
  end;
(*****************************************************************************)

  TPCCoords  = Record
    Left      : integer;
    Right     : integer;
    Top       : integer;
    Bottom    : integer;
  end;
(*****************************************************************************)

  TWorldCoords   = Record
    Left        : double;
    Right       : double;
    Top         : double;
    Bottom      : double;
    LPercent    : double;
    RPercent    : double;
    TPercent    : double;
    BPercent    : double;
  end;
 (****************************************************************************)

  THighLow  = record
    high      : double;
    Low       : double;
  end;

  THighLowArray  = array[0..($FFFF div Sizeof(THighLow)-1)] of THighLow;
  PTHighLowArray = ^THighLowArray;
 (****************************************************************************)



var

  aa, bb, cc, dd: double;         { Internal values set by}
  paa,pbb,pcc,pdd:double;         { Same as above, saved for printing}


  PCLeft, PCRight, PCTop, PCBottom: integer; { set Set_Window and Set_ViewPort }
  WorldLeft, WorldRight, WorldTop, WorldBottom : double;
                                                { to map to screen coordinates }
  AdjustedWorldLeft,AdjustedWorldRight,AdjustedWorldTop,AdjustedWorldBottom :
                                                 double;

  QSGMagnification : double; (* Magnification that should be 1 for screen printing
                              and a desired value for actual printing *)

procedure WorldToPC(xw, yw: double; var xpc, ypc: integer);
procedure PCToWorld(xpc, ypc: integer; var xw, yw: double);
procedure WorldSystem(WorldCoords:TWorldCoords;PCCoords:TPCCoords);
procedure SetWorldWindow(xmin, xmax, ymin, ymax: double;
           leftPercent,rightPercent,bottomPercent,TopPercent:double);
procedure SetPCWindow(xmin, xmax, ymin, ymax: integer);
procedure WLineTo (myCanvas:TCanvas;Wx,Wy:double);
procedure WMoveTo (myCanvas:TCanvas;Wx,Wy:double);
procedure WArc(myCanvas:TCanvas;WX1, WY1, WX2, WY2, WX3, WY3, WX4, WY4: Double);
procedure WEllipse(myCanvas:TCanvas;WX1, WY1, WX2, WY2: Double);
procedure wquad4 (myCanvas:TCanvas;Wx1,Wy1,Wx2,Wy2,Wx3,Wy3,Wx4,Wy4:double;
                   color:TColor;bShowBoundary:boolean);
procedure wquad8 (myCanvas:TCanvas;Wx1,Wy1,Wx2,Wy2,Wx3,Wy3,Wx4,Wy4:double;
                                   Wx5,Wy5,Wx6,Wy6,Wx7,Wy7,Wx8,Wy8:double;
                                   color:TColor;bShowBoundary:boolean);

procedure DrawXAxis(myCanvas:TCanvas;AxisRec:TAxis);
procedure DrawYAxis(myCanvas:TCanvas;AxisRec:TAxis);
procedure ScatterPlotData(myCanvas:TCanvas;LineData:PTPointArray;LineInfo:TLineFormat);
procedure DrawSquarePoint(myCanvas:TCanvas;PointSize:integer;x,y:double;Color:TColor);
procedure DrawTrianglePoint(myCanvas:TCanvas;PointSize:integer;x,y:double;color:TColor);
procedure DrawCirclePoint(myCanvas:TCanvas;PointSize:integer;x,y:double;color:TColor);
procedure DrawEmptySquarePoint(myCanvas:TCanvas;PointSize:integer;x,y:double;color:TColor);
procedure DrawEmptyTrianglePoint(myCanvas:TCanvas;PointSize:integer;x,y:double;color:TColor);
procedure DrawEmptyCirclePoint(myCanvas:TCanvas;PointSize:integer;x,y:double;color:TColor);
procedure DrawCrossPoint(myCanvas:TCanvas;PointSize:integer;x,y:double;color:TColor);


implementation

procedure WorldToPC(xw, yw: double; var xpc, ypc: integer);
(* Converts a world coordinate to a screen coordinate *)

var
  expc,eypc : extended;

begin
  expc := aa * xw + bb;
  eypc := cc * yw + dd;
  if expc > High(integer) then
    expc := High(integer);
  if expc < Low(integer) then
    expc := Low(integer);
  if eypc > High(integer) then
    eypc := High(integer);
  if eypc < Low(integer) then
    eypc := Low(integer);

                              
  xpc := Round(expc);
  ypc := Round(eypc);
  if xpc > LIMIT then
    repeat
      xpc := 2*LIMIT div 3;
    until xpc <= LIMIT;
  if ypc > LIMIT then
    repeat
      ypc := 2*LIMIT div 3;
    until ypc <= LIMIT;
  if xpc < -LIMIT then
    repeat
      xpc := -2*LIMIT div 3;
    until xpc >= -LIMIT;
  if ypc < -LIMIT then
    repeat
      ypc := -2*LIMIT div 3;
    until ypc >= -LIMIT;  
end;

procedure PCToWorld(xpc, ypc: integer; var xw, yw: double);
{ Converts a screen coordinate to a world coordinate }
begin
  xw := (xpc - bb) / aa;
  yw := (ypc - dd) / cc;
end;

procedure WorldSystem(WorldCoords:TWorldCoords;PCCoords:TPCCoords);

(* Take the window definded by PCCoords and give it world coordinates
   difined by WCoords *)


begin
  with worldCoords do
    begin
      Worldleft := left;
      Worldright := right;
      Worldbottom := bottom;
      Worldtop := top;
    end;

  with PCCoords do
    begin
      PCLeft := left;    PCRight := right;
      PCBottom := Bottom;    PCTop := top;
    end;

  aa := (PCRight - PCLeft) / (WorldRight - WorldLeft);
  bb := PCLeft - aa * WorldLeft;
  cc := (PCTop - PCBottom) / (WorldTop - WorldBottom);
  dd := PCBottom - cc * WorldBottom;

end;

procedure SetWorldWindow(xmin, xmax, ymin, ymax: double;
           leftPercent,rightPercent,bottomPercent,TopPercent:double);
{ Defines the window used in double world coordinates }
var
  DxL,DxR,DyB,DyT : double;
begin
  DxL := (xmax-xmin)*leftPercent;
  DxR := (xmax-xmin)*rightPercent;
  DyB := (ymax-ymin)*bottomPercent;
  DyT := (ymax-ymin)*TopPercent;
  WorldLeft := xmin-DxL;
  WorldRight := xmax+DxR;
  WorldBottom := ymin-DyB;
  WorldTop := ymax+DyT;
end;

procedure SetPCWindow(xmin, xmax, ymin, ymax: integer);
{ Defines the region on the screen of which world objects are mapped to }
begin
  PCLeft := xmin;
  PCRight := xmax;
  PCBottom := ymin;
  PCTop := ymax;
  aa := (PCRight - PCLeft) / (WorldRight - WorldLeft);
  bb := PCLeft - aa * WorldLeft;
  cc := (PCTop - PCBottom) / (WorldTop - WorldBottom);
  dd := PCBottom - cc * WorldBottom;
end;

procedure WLineTo (myCanvas:TCanvas;Wx,Wy:double);

var
  x,y  : integer;

begin
  WorldToPC(Wx,Wy,x,y);
  MyCanvas.LineTo(x,y);
end;

procedure WMoveTo (myCanvas:TCanvas;Wx,Wy:double);

var
  x,y  : integer;

begin
  WorldToPC(Wx,Wy,x,y);
  MyCanvas.MoveTo(x,y);
end;

procedure WArc(myCanvas:TCanvas;WX1, WY1, WX2, WY2, WX3, WY3, WX4, WY4: Double);

var
  x1,y1,x2,y2,x3,y3,x4,y4 : integer;

begin
  WorldToPC(WX1,WY1,x1,y1);
  WorldToPC(WX2,WY2,x2,y2);
  WorldToPC(WX3,WY3,x3,y3);
  WorldToPC(WX4,WY4,x4,y4);
  MyCanvas.Arc(x1,y1,x2,y2,x3,y3,x4,y4);
end;

procedure WEllipse(myCanvas:TCanvas;WX1, WY1, WX2, WY2: Double);

var
  x1,y1,x2,y2 : integer;

begin
  WorldToPC(WX1,WY1,x1,y1);
  WorldToPC(WX2,WY2,x2,y2);
  MyCanvas.Ellipse(x1,y1,x2,y2);
end;

procedure wquad4 (myCanvas:TCanvas;Wx1,Wy1,Wx2,Wy2,Wx3,Wy3,Wx4,Wy4:double;
                   color:TColor;bShowBoundary:boolean);
var
  x,y  : integer;
  points : array[1..4] of Tpoint;
  oldBrushColor : TColor;
  oldPenColor   : TColor;

begin
  WorldToPC(wx1,wy1,x,y);
  points[1].x := x;
  points[1].y := y;
  WorldToPC(wx2,wy2,x,y);
  points[2].x := x;
  points[2].y := y;
  WorldToPC(wx3,wy3,x,y);
  points[3].x := x;
  points[3].y := y;
  WorldToPC(wx4,wy4,x,y);
  points[4].x := x;
  points[4].y := y;
  if not bShowBoundary then
    begin
      OldPenColor := myCanvas.pen.color;
      myCanvas.Pen.Color := color;
    end;
  OldBrushColor := myCanvas.brush.color;
  mycanvas.Brush.Color := color;
  myCanvas.polygon(points);
  myCanvas.Brush.Color := oldBrushColor;
  if not bShowBoundary then
    myCanvas.Pen.Color := oldPenColor;
end;





procedure wquad8 (myCanvas:TCanvas;Wx1,Wy1,Wx2,Wy2,Wx3,Wy3,Wx4,Wy4:double;
                                   Wx5,Wy5,Wx6,Wy6,Wx7,Wy7,Wx8,Wy8:double;
                                   color:TColor;bShowBoundary:boolean);

(*Quad8 is generated by producing a polygon of 160 points - 41 points per side*)
(*
   1---8---7
   |       |
   2       6
   |       |
   3---4---5
*)

type
  TwPoint41 = Array[1..41] of TwPoint;

var
  Gpoints : array[1..8] of Twpoint; (*the eight points that make up the quad8*)
  Points : array[1..160] of TPoint;
  wPoints : array[1..160] of TwPoint;
  LinePoints : TwPoint41;
  oldBrushColor : TColor;
  OldPenColor   : TColor;
  i             : integer;



procedure PointsOfCurve(p1,p2,p3:TwPoint; var NewPoints:TwPoint41);

var
  c,s : double;
  Dx,Dy,Ds : double;
  pp1,pp2,pp3 : double;
  pq1,pq2,pq3 : double;
  pxi1,pxi2,pxi3 : double;
  peta1,peta2,peta3 : double;
  DeltaXi           : double;
  xi,eta            : array[1..41] of double;
  p,q               : array[1..41] of double;
  i                 : integer;
  xi2               : double;  { xi value from -1 to 1}

(*Line interpolation functions N1,N2,N3*) (* 1----2----3 *)
function N1(x:double):double;
begin
  result := 0.5*(-x+x*x);
end;

function N2(x:double):double;
begin
  result := 1-x*x;
end;

function N3(x:double):double;

begin
  result := 0.5*(x+x*x);
end;


begin
  Dx := p3.x-p1.x;
  Dy := p3.y-p1.y;
  if Dx = 0 then {90 degrees}
    begin
      c := 0;
      s := 1;
    end
  else if Dy = 0 then { 0 degrees}
    begin
      c := 1;
      s := 0;
    end
  else
    begin
      Ds := sqrt(sqr(Dx)+Sqr(Dy));
      c := Dx/Ds;
      s := Dy/Ds;
    end;
  { Shift x-y coordinate origin at first point;}
  pp1 := 0;
  pq1 := 0;
  pp2 := p2.x - p1.x;
  pq2 := p2.y - p1.y;
  pp3 := p3.x - p1.x;
  pq3 := p3.y - p1.y;
  { Rotate coordinates so that new x axis (xi) is along the 1-3 points}
  pxi1 := pp1;
  peta1 := pq1;
  pxi2 := c*pp2+s*pq2;
  peta2 := -s*pp2+c*pq2;
  pxi3 := c*pp3+s*pq3;
  peta3 := -s*pp3+c*pq3;
  { Find array of 41 points on xi-eta system}
  DeltaXi := 2.0/40; { 40 intervals 41 points}
  xi[1] := 0;
  eta[1] := 0;
  xi2 := -1;
  for i := 2 to 41 do
    begin
      xi2 := xi2+DeltaXi;
      xi[i] := N1(xi2)*pxi1+N2(xi2)*pxi2+N3(xi2)*pxi3;
      eta[i] := N1(xi2)*peta1+N2(xi2)*pEta2+N3(xi2)*pEta3;
    end;

  { Now derotate the points}
  p[1] := 0;
  q[1] := 0;
  for i := 2 to 41 do
    begin
      p[i] := c*xi[i]-s*eta[i];
      q[i] := s*xi[i]+c*eta[i];
    end;

  { Now calculate the desired points}
  for i := 1 to 41 do
    begin
      NewPoints[i].x := p[i]+p1.x;
      NewPoints[i].y := q[i]+p1.y;
    end;
end;




begin

  Gpoints[1].x := wx1;
  Gpoints[1].y := wy1;
  Gpoints[2].x := wx5;
  Gpoints[2].y := wy5;
  Gpoints[3].x := wx2;
  Gpoints[3].y := wy2;
  Gpoints[4].x := wx6;
  Gpoints[4].y := wy6;
  Gpoints[5].x := wx3;
  Gpoints[5].y := wy3;
  Gpoints[6].x := wx7;
  Gpoints[6].y := wy7;
  Gpoints[7].x := wx4;
  Gpoints[7].y := wy4;
  Gpoints[8].x := wx8;
  Gpoints[8].y := wy8;

  { Now generate points 1 through 41;}
  PointsOfCurve(GPoints[1],GPoints[2],GPoints[3],LinePoints);
  for i := 1 to 41 do
    wPoints[i] := LinePoints[i];

  { Now generate points 42 through 81;}
  PointsOfCurve(GPoints[3],GPoints[4],GPoints[5],LinePoints);
  for i := 2 to 41 do (*first point does not need be reevaluated*)
    wPoints[40+i] := LinePoints[i];

  { Now generate points 82 through 121;}
  PointsOfCurve(GPoints[5],GPoints[6],GPoints[7],LinePoints);
  for i := 2 to 41 do (*first point does not need be reevaluated*)
    wPoints[80+i] := LinePoints[i];

  { Now generate points 122 through 160;}
  PointsOfCurve(GPoints[7],GPoints[8],GPoints[1],LinePoints);
  for i := 2 to 40 do (*first and last points do not need be reevaluated*)
    wPoints[120+i] := LinePoints[i];

  for i := 1 to 160 do
   WorldToPC(wPoints[i].x,wPoints[i].y,integer(Points[i].x),integer(Points[i].y));

  if not BShowBoundary then
    begin
      OldPenColor := myCanvas.pen.color;
      myCanvas.Pen.Color := color;
    end;

  OldBrushColor := myCanvas.brush.color;
  mycanvas.Brush.Color := color;
  myCanvas.polygon(points);
  myCanvas.Brush.Color := oldBrushColor;
    if not bShowBoundary then
    myCanvas.Pen.Color := OldPenColor;
end;


procedure DrawXAxis(myCanvas:TCanvas;AxisRec:TAxis);

const

  zero = 1e-5;

var
  x  : double;
  ix,iy : integer;
  xstr : string;
  strlen : integer;
  strHeight : integer;

begin
  MyCanvas.Brush.Style := bsClear; (*So that there is no background behind the letters*)
  with AxisRec do
    begin
      if abs(TicSpacing)>zero then (*draw primery x tics*)
        begin
          WorldToPC(xmin,ymin,ix,iy);
          ix := ix+shiftX;
          iy := iy + shiftY;
          myCanvas.MoveTo(ix,iy);
          x := xmin;
          while x <= xmax+zero do
            begin
              MyCanvas.Pen.Color := GridColor;
              MyCanvas.Pen.Width := GridWidth;
              WorldToPC(x,ymin,ix,iy);
              ix := ix+shiftX;
              iy := iy + shiftY;
              MyCanvas.MoveTo(ix,iy);
              if x > xmin then
                MyCanvas.LineTo(ix,iy-6);
              if Labels then
                begin
                  xstr := FloatToStrF(x,ffFixed,3,decimals);
                  strLen := myCanvas.textWidth(xstr);
                  MyCanvas.Font.Size := FontSize;
                  MyCanvas.Font.Name := FontName;
                  MyCanvas.Font.Style := FontStyle;
                  MYCanvas.Font.Color := FontColor;
                  if abs(x)> zero then
                    MyCanvas.TextOut(ix-strLen div 2,iy + 2,xstr);
                end;
              if Grid and (x>xmin) then
                begin
                  MyCanvas.pen.color := GridColor;
                  MyCanvas.pen.width := GridWidth;
                  WorldToPC(x,ymin,ix,iy);
                  ix := ix+shiftX;
                  iy := iy + shiftY;
                  MyCanvas.MoveTo(ix,iy);
                  WorldToPC(x,ymax,ix,iy);
                  ix := ix+shiftX;
                  iy := iy + shiftY;
                  MyCanvas.LineTo(ix,iy);
                  MyCanvas.pen.color := AxisColor;
                  MyCanvas.pen.width := AxisWidth;
                end;
              x := x + TicSpacing;
            end;
        end;
      if SecTics and (ticSpacing > zero) then (*draw secondary x tics*)
        begin
          WorldToPC(xmin,ymin,ix,iy);
          ix := ix+shiftX;
          iy := iy + shiftY;
          MyCanvas.MoveTo(ix,iy);
          x := xmin + TicSpacing/2;
          MyCanvas.pen.color := gridColor;
          MyCanvas.pen.width := gridWidth;
          while x <= xmax+zero do
            begin
              WorldToPC(x,ymin,ix,iy);
              ix := ix+shiftX;
              iy := iy + shiftY;
              MyCanvas.MoveTo(ix,iy);
              WorldToPC(x,ymin,ix,iy);
              ix := ix+shiftX;
              iy := iy+shiftY;
              MyCanvas.LineTo(ix,iy-4);
              x := x + TicSpacing;
            end;
        end;
      (* Draw X Axis *)
      WorldToPC(xmin,ymin,ix,iy);
      ix := ix+shiftX;
      iy := iy + shiftY;
      MyCanvas.MoveTo(ix,iy);
      mycanvas.Pen.Color := AxisColor;
      myCanvas.pen.width := AxisWidth;
      WorldToPC(xmax,ymin,ix,iy);
      ix := ix+shiftX;
      iy := iy + shiftY;
      MyCanvas.LineTo(ix,iy);

      if title <> '' then
        begin
          MyCanvas.Font.Size := TitleFontSize;
          MyCanvas.Font.Style := [fsBold];
          strLen := myCanvas.textWidth(title);
          strHeight := myCanvas.textHeight(title);
          WorldToPC((xmin+xmax)/2,ymin,ix,iy);
          ix := ix + shiftX;
          iy := iy + shiftY;
          myCanvas.TextOut(ix-strLen div 2, iy+strHeight+3,title);
        end;

  end; (* with AxisRec *)
end;

procedure DrawYAxis(myCanvas:TCanvas;AxisRec:TAxis);

const
  zero = 1e-5;

var
  y  : double;
  ix,iy : integer;
  ystr : string;
  strlen : integer;
  strHeight : integer;

  LogRec    : TLogFont;
  OldFont,yFont : HFONT;

begin
  MyCanvas.Brush.Style := bsClear; (*So that there is no background behind the letters*)
  with AxisRec do
    begin
      if abs(TicSpacing)>zero then (*draw primery  tics*)
        begin
          WorldToPC(xmin,ymin,ix,iy);
          ix := ix+shiftX;
          iy := iy + shiftY;
          MyCanvas.MoveTo(ix,iy);
          y := ymin;
          while y <= ymax + zero do
            begin
              MyCanvas.Pen.Color := GridColor;
              MyCanvas.Pen.Width := GridWidth;
              WorldToPC(xmin,y,ix,iy);
              ix := ix+shiftX;
              iy := iy + shiftY;
              MyCanvas.MoveTo(ix,iy);
              WorldToPC(xmin,y,ix,iy);
              ix := ix+shiftX;
              iy := iy+shiftY;
              if y > ymin then
                MyCanvas.LineTo(ix+6,iy);
              if Labels then
                begin
                  ystr := FloatToStrF(y,ffFixed,7,decimals);
                  strLen := myCanvas.textWidth(ystr);
                  strHeight := myCanvas.textHeight(ystr);
                  MyCanvas.Font.Size := FontSize;
                  MyCanvas.Font.Name := FontName;
                  MyCanvas.Font.Style := FontStyle;
                  MYCanvas.Font.Color := FontColor;
                  if abs(y)>0.0001 then
                    MyCanvas.TextOut(ix-strLen-2,iy-strHeight div 2,ystr);
                end;
              if Grid and (y > ymin)then
                begin
                  MyCanvas.pen.color := GridColor;
                  MyCanvas.pen.width := GridWidth;
                  WorldToPC(xmin,y,ix,iy);
                  ix := ix+shiftX;
                  iy := iy + shiftY;
                  MyCanvas.MoveTo(ix,iy);
                  WorldToPC(xmax,y,ix,iy);
                  ix := ix+shiftX;
                  iy := iy + shiftY;
                  MyCanvas.LineTo(ix,iy);
                  MyCanvas.pen.color := AxisColor;
                  MyCanvas.pen.width := AxisWidth;
                end;
              y := y + TicSpacing;
            end;
        end;
      if SecTics and (TicSpacing > zero) then (*draw secondary x tics*)
        begin
          WorldToPC(xmin,ymin,ix,iy);
          ix := ix+shiftX;
          iy := iy + shiftY;
          MyCanvas.MoveTo(ix,iy);
          y := ymin + TicSpacing/2;
          MyCanvas.Pen.Color := GridColor;
          MyCanvas.Pen.Width := GridWidth;
          while y <= ymax+zero do
            begin
              WorldToPC(xmin,y,ix,iy);
              ix := ix+shiftX;
              iy := iy + shiftY;
              MyCanvas.MoveTo(ix,iy);
              MyCanvas.LineTo(ix+3,iy);
              y := y + TicSpacing;
            end;
        end;
      (*Draw Y Axis*)

      WorldToPC(xmin,ymin,ix,iy);
      ix := ix+shiftX;
      iy := iy + shiftY;
      MyCanvas.MoveTo(ix,iy);
      MyCanvas.Pen.Color := AxisColor;
      MyCanvas.Pen.Width := AxisWidth;

      WorldToPC(xmin,ymax,ix,iy);
      ix := ix+shiftX;
      iy := iy + shiftY;
      MyCanvas.LineTo(ix,iy);


      if title <> '' then
        begin
          MyCanvas.font.name := 'arial';
          MyCanvas.Font.Size := TitleFontSize;
          MyCanvas.Font.Style := [fsbold];
          GetObject(myCanvas.Font.Handle,sizeOf(LogRec),@LogRec);
          LogRec.lfEscapement := 900;
          LogRec.lfOutPrecision := OUT_TT_ONLY_PRECIS;
          yfont := CreateFontIndirect(LogRec);

          strLen := myCanvas.textWidth(title);
          WorldToPC(xmin,(ymin+ymax)/2,ix,iy);
          ix := ix;
          iy := iy + shiftY + strLen div 2;

          oldFont :=SelectObject(myCanvas.Handle,yFont);
          myCanvas.TextOut(ix,iy,title);
          yfont := selectObject(myCanvas.Handle,OldFont);
          DeleteObject(yFont);
        end;
  end; (* with AxisRec *)

end;


procedure ScatterPlotData(myCanvas:TCanvas;LineData:PTPointArray;LineInfo:TLineFormat);

var
  i : word;


begin
  myCanvas.pen.color := LineInfo.LineColor;
  myCanvas.pen.width := LineInfo.Linewidth;
  WMoveTo(myCanvas,LineData^[1].x,LineData^[1].y);

  for i := 2 to LineInfo.points do
    WLineTo(myCanvas,LineData^[i].x,LineData^[i].y);

  myCanvas.pen.color := LineInfo.LineColor;
  myCanvas.pen.width := 1;
  WMoveTo(myCanvas,LineData^[1].x,LineData^[1].y);

  for i := 1 to LineInfo.points  do
    begin
      case LineInfo.ptype of
        square : DrawSquarePoint(myCanvas,LineInfo.psize,LineData^[i].x,LineData^[i].y,LineInfo.PColor);
        triangle : DrawTrianglePoint(myCanvas,LineInfo.psize,LineData^[i].x,LineData^[i].y,LineInfo.PColor);
        circle: DrawcirclePoint(myCanvas,LineInfo.psize,LineData^[i].x,LineData^[i].y,LineInfo.PColor);
        EmptySquare : DrawEmptySquarePoint(myCanvas,LineInfo.psize,LineData^[i].x,LineData^[i].y,LineInfo.PColor);
        Emptytriangle : DrawEmptyTrianglePoint(myCanvas,LineInfo.psize,LineData^[i].x,LineData^[i].y,LineInfo.PColor);
        Emptycircle: DrawEmptycirclePoint(myCanvas,LineInfo.psize,LineData^[i].x,LineData^[i].y,LineInfo.PColor);
        CrossPoint :DrawCrossPoint(myCanvas,LineInfo.PSize,LineData^[i].x,LineData^[i].y,LineInfo.PColor);
        nothing : begin end;
      end;
    end;
end;



procedure DrawSquarePoint(myCanvas:TCanvas;PointSize:integer;x,y:double;color:Tcolor);

var
  ix,iy  : integer;
  x1,y1  : integer;
  points : array[1..4] of Tpoint;
  oldPenColor : TColor;
  oldBrushColor : TColor;
  oldPenWidth : integer;

begin
  with myCanvas do
    begin
      OldPenWidth := Pen.Width;
      Pen.Width := 1;

      OldPenColor := Pen.Color;
      Pen.Color := color;

      worldtoPC(x,y,ix,iy);
      x1 := (ix - pointsize div 2);
      y1 := (iy-pointsize div 2);
      points[1].x := x1;
      points[1].y := y1;
      points[2].x := x1+pointsize;
      points[2].y := y1;
      points[3].x := x1+pointsize;
      points[3].y := y1+pointsize;
      points[4].x := x1;
      points[4].y := y1+pointsize;

      OldBrushColor := brush.color;
      Brush.Color := color;
      polygon(points);
      Pen.Color   := oldPenColor;
      Pen.Width   := oldPenWidth;
      Brush.Color := oldBrushColor;
    end;
end;

procedure DrawTrianglePoint(myCanvas:TCanvas;PointSize:integer;x,y:double;color:TColor);
var
  ix,iy  : integer;
  x1,y1  : integer;
  points : array[1..3] of Tpoint;
  oldPenColor : TColor;
  oldBrushColor : TColor;
  oldPenWidth : integer;

begin
  with myCanvas do
    begin
      OldPenWidth := Pen.Width;
      Pen.Width := 1;

      OldPenColor := Pen.Color;
      Pen.Color := color;

      worldtoPC(x,y,ix,iy);
      x1 := ix;
      y1 := iy-pointsize div 2;
      points[1].x := x1;
      points[1].y := y1;

      x1 := ix - round(PointSize div 2 * cos(pi/6));
      y1 := iy + round(PointSize div 2 * sin(pi/6));
      points[2].x := x1;
      points[2].y := y1;
      x1 := ix + round(PointSize div 2 * cos(pi/6));
      y1 := iy + round(PointSize div 2 * sin(pi/6));
      points[3].x := x1;
      points[3].y := y1;

      OldBrushColor := brush.Color;
      Brush.Color := Color;
      polygon(points);
      Pen.Color   := oldPenColor;
      Pen.Width   := oldPenWidth;
      Brush.color := oldBrushColor;
    end;
end;


procedure DrawCirclePoint(myCanvas:TCanvas;PointSize:integer;x,y:double;color:TColor);

var
  ix,iy  : integer;
  x1,y1  : integer;
  x2,y2  : integer;
  oldPenColor : TColor;
  oldBrushColor : TColor;
  oldPenWidth : integer;


begin
  with myCanvas do
    begin
      OldPenWidth := Pen.Width;
      Pen.Width := 1;

      OldPenColor := Pen.Color;
      Pen.Color := color;

      worldtoPC(x,y,ix,iy);
      x1 := (ix - pointsize div 2);
      y1 := (iy-pointsize div 2);
      x2 := (ix + pointsize div 2);
      y2 := (iy + pointsize div 2);

      OldBrushColor := myCanvas.brush.color;

      mycanvas.Brush.Color := color;
      if PointSize > 1 then
        myCanvas.ellipse(x1,y1,x2,y2)
      else
        myCanvas.Pixels[x1,y1]:= color;

      myCanvas.Brush.Color := oldBrushColor;
      Pen.Width := OldPenWidth;
      Pen.Color := OldPenColor;
    end;
end;

procedure DrawEmptySquarePoint(myCanvas:TCanvas;PointSize:integer;x,y:double;color:TColor);

var
  ix,iy  : integer;
  x1,y1  : integer;
  points : array[1..4] of Tpoint;
  oldPenColor : TColor;
  oldBrushStyle : TBrushStyle;
  oldPenWidth : integer;

begin
  with myCanvas do
    begin
      OldPenWidth := Pen.Width;
      Pen.Width := 1;

      OldPenColor := Pen.Color;
      Pen.Color := color;

      worldtoPC(x,y,ix,iy);
      x1 := (ix - pointsize div 2);
      y1 := (iy-pointsize div 2);
      points[1].x := x1;
      points[1].y := y1;
      points[2].x := x1+pointsize;
      points[2].y := y1;
      points[3].x := x1+pointsize;
      points[3].y := y1+pointsize;
      points[4].x := x1;
      points[4].y := y1+pointsize;

      OldBrushStyle := brush.Style;
      Brush.Style := bsClear;
      polygon(points);
      Pen.Color   := oldPenColor;
      Pen.Width   := oldPenWidth;
      Brush.Style := oldBrushStyle;
    end;
end;

procedure DrawEmptyTrianglePoint(myCanvas:TCanvas;PointSize:integer;x,y:double;color:TColor);

var
  ix,iy  : integer;
  x1,y1  : integer;
  points : array[1..3] of Tpoint;
  oldPenColor : TColor;
  oldBrushStyle : TBrushStyle;
  oldPenWidth : integer;

begin
  with myCanvas do
    begin
      OldPenWidth := Pen.Width;
      Pen.Width := 1;

      OldPenColor := Pen.Color;
      Pen.Color := color;

      worldtoPC(x,y,ix,iy);
      x1 := ix;
      y1 := iy-pointsize div 2;
      points[1].x := x1;
      points[1].y := y1;

      x1 := ix - round(PointSize div 2 * cos(pi/6));
      y1 := iy + round(PointSize div 2 * sin(pi/6));
      points[2].x := x1;
      points[2].y := y1;
      x1 := ix + round(PointSize div 2 * cos(pi/6));
      y1 := iy + round(PointSize div 2 * sin(pi/6));
      points[3].x := x1;
      points[3].y := y1;

      OldBrushStyle := brush.Style;
      Brush.Style := bsClear;
      polygon(points);
      Pen.Color   := oldPenColor;
      Pen.Width   := oldPenWidth;
      Brush.Style := oldBrushStyle;
    end;
end;

procedure DrawEmptyCirclePoint(myCanvas:TCanvas;PointSize:integer;x,y:double;color:TColor);
var
  ix,iy  : integer;
  x1,y1  : integer;
  x2,y2  : integer;
  oldPenColor : TColor;
  oldBrushStyle : TBrushStyle;
  oldPenWidth : integer;


begin
  with myCanvas do
    begin
      OldPenWidth := Pen.Width;
      Pen.Width := 1;

      OldPenColor := Pen.Color;
      Pen.Color := color;

      worldtoPC(x,y,ix,iy);
      x1 := (ix - pointsize div 2);
      y1 := (iy-pointsize div 2);
      x2 := (ix + pointsize div 2);
      y2 := (iy + pointsize div 2);
      if (pointsize mod 2) = 1 then
        begin
          x2 := x2+1;
          y2 := y2+1;
        end;  

      OldBrushStyle := brush.style;
      Brush.Style := bsClear;
      ellipse(x1,y1,x2,y2);

      Pen.Color := OldPenColor;
      Pen.Width := OldPenWidth;
      Brush.Style := oldBrushStyle;
    end;
end;


procedure DrawCrossPoint(myCanvas:TCanvas;PointSize:integer;x,y:double;color:TColor);

var
  ix,iy  : integer;
  x1,y1  : integer;
  x2,y2  : integer;
  oldPenColor : TColor;
  oldPenWidth : integer;


begin
  with myCanvas do
    begin
      OldPenWidth := Pen.Width;
      Pen.Width := 1;

      OldPenColor := Pen.Color;
      Pen.Color := color;

      worldtoPC(x,y,ix,iy);

      x1 := (ix - pointsize div 2);
      y1 := (iy);
      x2 := (ix + pointsize div 2);
      y2 := (iy);

      if (pointsize mod 2) = 1 then
        begin
          x2 := x2+1;
        end;

      myCanvas.MoveTo(x1,y1);
      myCanvas.LineTo(x2,y2);

      x1 := (ix);
      y1 := (iy - pointsize div 2);
      x2 := (ix);
      y2 := (iy + pointsize div 2);

      if (pointsize mod 2) = 1 then
        begin
          y2 := y2+1;
        end;

      myCanvas.MoveTo(x1,y1);
      myCanvas.LineTo(x2,y2);

      Pen.Color := OldPenColor;
      Pen.Width := OldPenWidth;

    end;
end;

end.

