unit QSGraph;


(******************************************************************************
    TQSWorldGraph Component.
    Version 1.0.  Last Update November 30, 1998
    Revision 1    Last Update December 10, 1998
    Revision 2    Last Update December 14, 1998
    Revision 3    Last Update December 31, 1998

    Version 2     Last Update March 15, 1999
    Copyright : Q-Systems Engineering. 8950 N. Calle Buena Vista. Tucson AZ


******************************************************************************)

(*Comment out the next line if this is not an evaluation copy*)
{{$DEFINE EVALUATION}

(******************************************************************************)

(*
    ******************************************************************************
    ******************************************************************************
    COPYRIGHT NOTICE:
    THE SOURCE OF THIS COMPONENT IS PROVIDED AS A PERSONAL COPY.  Q-SYSTEMS ENGINEERING
    CLAIMS NO ROYALTIES FOR THE USE OF TQSWorldGraph IN YOUR PROGRAMS.  HOWEVER, YOU MAY
    NOT DISTRIBUTE THE SOURCE -IN PART OR WHOLE-, THE .DCU, OR THE .OBJ OF THIS FILE.

    ******************************************************************************
    ******************************************************************************
*)


(*
    KNOWN PROBLEMS: NONE

    Functions of this component:
    1) creates a paint box where you can define distances in world coordinates
    2) adds a horizontal and vertical ruler to provide visual comprehension of the position
       of any point.
    3) simplifies zooming.

    UPDATES:

    version 1.01 - November 20 1998; November 30 1998; December 10 1998
    Added methods
        procedure    SetPen(PenColor:TColor;PenMode:TPenMode;PenStyle:TPenStyle;PenWidth:integer);
        procedure    ResetPen;
    Renamed properties and public variables and functions to start with capital leters.

    *****************
    Added variables  fHFontRelativeSize; fVFontRelativeSize; and bFirstTime
    Added property Font

    The above are necessary to make the component fully compatible with ElasticForm.

    *****************

    version 1.02 - December 14 1998
      Fixed error related to the first definition of fHFontRelativeSize and fVFontRelativeSize
      Connected the component's Canvas with the fCurrentCanvas

    version 1.03 - December 27 1998
      Added
        SetPen, ResetPen;
        WorldArc
        BeginDistance, ContinueDistance, EndDistance
        to facilitate measuring the distance between two points


    Version 2 - January 20-March 14, 1998
      Added
        Big cross to point out cursor position.
          method RemoveDetailedPosition
          property DetailedPosition

        method SetCursorTo
        method CrossPoint
        correstion of EmptyCirclePoint to respond correctly for even and odd point size.
        objects are not drawn if parts of them are not included in the visual frame
        Improved zoomming.  --WorldQuad8 still need fixing--
        ellipse and arc are drawn using Windows functions only if they are completely
        within the viewport.  Otherwise they are drawn using their equations.


*)


{$ifdef ver100 then}
{$define newCompiler}
{$endif}

{$ifdef ver120 then}
{$define newCompiler}
{$endif}

{$ifdef ver125 then}
{$define newCompiler}
{$endif}

interface

uses
  {$IFDEF WIN32}Windows,{$ifdef newCompiler then} jpeg,{$endif}{$ELSE}
  WinTypes, WinProcs,{$ENDIF} Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  extctrls,wGraph,QSPaint,PBRulers,printers;

{$ifdef VER110} {CBUILDER 3}
{$ObjExportAll On}
{$ENDIF}

{$ifdef VER125} {CBUILDER 4}
{$ObjExportAll On}
{$ENDIF}

const
  gVersion = 200;

type

  TWorldPoint = record
    x  : double;
    y  : double;
  end;

  TPixelsPoint = record
    x  : integer;
    y  : integer;
  end;

  TWLine = record
    x1 : double;
    y1 : double;
    x2 : double;
    y2 : double;
  end;


  (* Data structures for clipping *)
  TEdgeKind = (ekBottom,ekRight,ekTop,ekLeft);
  TEdge = record
    edge : TEdgeKind;
    xMin,xMax,yMin,yMax : double;
  end;

  TWorldPointArray = array[1..MAX_POLYGON_POINTS] of TWorldPoint;
  THRulerPosition = (hrpTop,hrpBottom,hrpHidden);
  TVRulerPosition = (vrpLeft,vrpRight,vrpHidden);


  TQSWorldGraph = class(TCustomPanel)
  private
    fXMinZoom     : double;
    fXMaxZoom     : double;
    fYMinZoom     : double;
    fYMaxZoom     : double;
    bZoom         : boolean;
    fCurrentCanvas : TCanvas;

    fHRuler       : TQSPBHRuler;
    fVRuler       : TQSPBVRuler;
    fPaintBox     : TQSPaintBox;

    fVersion      : Integer;

    fOnPaint      : TNotifyEvent;
    fLengthUnits  : TLengthUnits;
    fHRulerPosition : tHRulerPosition;
    fVRulerPosition : tVRulerPosition;

    PreviousX     : integer;
    PreviousY     : integer;

    (*variables for full geometric definition.  Allow up to 100 zoom-ins*)
    GeomXMax      : array[0..100] of double;
    GeomXMin      : array[0..100] of double;
    GeomYMax      : array[0..100] of double;
    GeomYMin      : array[0..100] of double;
    ZoomLevel     : integer;

    fOnMouseMove  : TMouseMoveEvent;
    fOnMouseDown  : TMouseEvent;
    fOnMouseUp    : TMouseEvent;

    fHFont,fVFont : TFont;
    fHFontRelativeSize, fVFontRelativeSize : double;
    bFirstTime    : boolean;

    fAvailableWidth,fAvailableHeight : integer; (*Do nothing, Just force owner form to keep their values*)

    fPrintMagnification : double;
    fBasicPrintMagnification : double;
    bPrinting           : boolean;
    DistanceFirst       : TWorldPoint;
    DistanceLast        : TWorldPoint;
    fDetailedPosition   : boolean;
    DetailedX,DetailedY : Integer;
    WLine               : TWLine;


    function  GetXMin:double;
    procedure SetXMin(value:double);
    function  GetXMax:double;
    procedure SetXMax(value:double);
    function  GetYMin:double;
    procedure SetYMin(value:double);
    function  GetYMax:double;
    procedure SetYMax(value:double);
    procedure SetProportional(value:boolean);
    function  GetProportional:boolean;
    function  CreateHRuler:TQSPBHRuler;
    function  CreateVRuler:TQSPBVRuler;
    function  CreatePaintBox : TQSPaintBox;
    procedure SetXIncrease(value:TXIncrease);
    function  GetXIncrease:TXIncrease;
    procedure SetYIncrease(value:TYIncrease);
    function  GetYIncrease:TYIncrease;
    procedure SetLengthUnits(value:TLengthUnits);
    procedure SetHRulerPosition(value:tHRulerPosition);
    procedure SetVRulerPosition(value:tVRulerPosition);
    procedure SetHRulerDecimals(value:integer);
    function  GetHRulerDecimals:integer;
    procedure SetVRulerDecimals(value:integer);
    function  GetVRulerDecimals:integer;

    procedure DoOnMouseMove(Sender: TObject; Shift: TShiftState; X, Y: Integer);
    procedure DoOnMouseDown(Sender: TObject;Button: TMouseButton;
                             Shift: TShiftState; X, Y: Integer);
    procedure DoOnMouseUp(Sender: TObject;Button: TMouseButton;
                             Shift: TShiftState; X, Y: Integer);

    procedure SetBackgroundColor(value:TColor);
    function  GetBackGroundColor:TColor;
    procedure SetHRulerFont(value:TFont);
    procedure SetVRulerFont(value:TFont);
    function  GetAvailableWidth:integer;
    function  GetAvailableHeight :integer;

    procedure SutherlandHodgmanPolygonClip(
     inVertexArray : TWorldPointArray;
     var outVertexArray : TWorldPointArray;
     inLength : integer;
     var outLength : integer;
     clipBoundary : TEdge);
    procedure Output(newVertex : TWorldPoint;
         var outLength : integer; var outVertexArray : TWorldPointArray);
    function Inside( testVertex: TWorldPoint; clipBoundary : TEdge):boolean;
    function Intersect(first,second : TWorldPoint;clipBoundary:TEdge):TWorldPoint;
    procedure SetVersion(value:Integer);



  protected
    StartX        : integer;
    StartY        : integer;
    EndX          : integer;
    EndY          : integer;
    procedure DrawLine;
    function InFrame(x1,y1:double):boolean;
    function LineSectionInterceptsYMinLine(var x0:double):boolean;
    function LineSectionInterceptsYMaxLine(var x0:double):boolean;
    function LineSectionInterceptsXMinLine(var y0:double):boolean;
    function LineSectionInterceptsXMaxLine(var y0:double):boolean;
    function PartOfFrameShows(wx1,wy1,wx2,wy2:double):boolean;
    function AllOfFrameShows(wx1,wy1,wx2,wy2:double):boolean;

  public
    (*variables for selection of region*)
    IPBx,IPBy     : integer;
    RPBx,RPBy     : double;

    JPEGPicture   : TPicture;
    BMPPicture    : TPicture;
    WMFPicture    : TPicture;
    GIFPicture    : TPicture;
    XOffSetStart  : Integer;

    procedure    Paint; override;
    constructor  Create(AOwner: TComponent); override;
    destructor   Destroy; override;
    function     LoadBMP(FileName : String):boolean;
    function     LoadWMF(FileName : String):boolean;
    {$IFDEF WIN32}
    function     LoadJPeg(FileName : String):boolean;
    {$ENDIF}
    procedure    BeginSelect(bx,by:integer);
    procedure    ContinueSelect(cx,cy:integer);
    function     EndSelect(ex,ey:integer):boolean;

    procedure    BeginDistance(bx,by:integer;bRePaint:boolean);
    function     ContinueDistance(cx,cy:integer):double;
    function     EndDistance(ex,ey:integer;bDrawDistanceLine:boolean):double;

    procedure    IncreaseZoom(x1,y1,x2,y2:double);
    procedure    DecreaseZoom;
    procedure    ResetZoom;
    function     Wxy(PCx,PCy:integer):TWorldPoint;
    function     PCxy(wx,wy:double):TPixelsPoint;

    procedure    SetCursorTo(wx,wy:double);

    procedure    BeginPrinting;
    procedure    EndPrinting;

    procedure    WorldLineTo (Wx,Wy:double);
    procedure    WorldMoveTo (Wx,Wy:double);

    procedure    WorldArc(WX1, WY1, WX2, WY2, WX3, WY3, WX4, WY4: Double);
    procedure    WorldEllipse(WX1, WY1, WX2, WY2: Double);

    procedure    WorldQuad4 (Wx1,Wy1,Wx2,Wy2,Wx3,Wy3,Wx4,Wy4:Double;
                   color:TColor;bShowBoundary:boolean);
    procedure    WorldQuad8 (Wx1,Wy1,Wx2,Wy2,Wx3,Wy3,Wx4,Wy4:Double;
                                   Wx5,Wy5,Wx6,Wy6,Wx7,Wy7,Wx8,Wy8:Double;
                                   color:TColor;bShowBoundary:boolean);

    procedure    WorldPolygon(NumberOfPoints:word; Points: TWorldPointArray;
                           PenColor,FillColor:TColor;bShowBoundary:boolean);

    procedure    SquarePoint(PointSize:integer;x,y:Double;color:TColor);
    procedure    TrianglePoint(PointSize:integer;x,y:Double;color:TColor);
    procedure    CirclePoint(PointSize:integer;x,y:Double;color:TColor);
    procedure    EmptySquarePoint(PointSize:integer;x,y:Double;color:TColor);
    procedure    EmptyTrianglePoint(PointSize:integer;x,y:Double;color:TColor);
    procedure    EmptyCirclePoint(PointSize:integer;x,y:Double;color:TColor);
    procedure    CrossPoint(PointSize:integer;x,y:Double;color:TColor);

    procedure    SetPen(PenColor:TColor;PenMode:TPenMode;PenStyle:TPenStyle;PenWidth:integer);
    procedure    ResetPen;
    procedure    RemoveDetailedPosition;

    procedure    DrawBitMap(bm:TGraphic);
    procedure    StretchDrawBitMap(bm:TGraphic);
    procedure    StretchPrintBitMap(bm:TGraphic;Magnification:double);
    procedure    PrintBitmap(Bitmap: TBitmap; X, Y: Integer;Magnification:double);
    procedure    PrintUserGraphic(Magnification:double);
    procedure    ClipPolygon(inVertexArray : TWorldPointArray; inLength: integer;
                     var outVertexArray : TWorldPointArray; var outLength : integer);


    property     Canvas read fCurrentCanvas write fCurrentCanvas;

  published

    property Align;
    property Color default clWhite;
    property Height default 100;
    property Width default 100;
    property Top   default 10;
    property Left  default 10;
    property Visible default true;

    property XMin:double read GetXMin write SetXMin;
    property XMax:double read GetXMax write SetXMax;
    property YMin:double read GetYMin write SetYMin;
    property YMax:double read GetYMax write SetYMax;

    property Proportional:boolean read GetProportional write SetProportional default true;

    property XIncrease : TXIncrease read GetXIncrease write SetXIncrease;
    property YIncrease : TYIncrease read GetYIncrease write SetYIncrease default BottomToTop;
    property LengthUnits : TLengthUnits read fLengthUnits write SetLengthUnits default luCustom;

    property HRulerPosition : tHRulerPosition  read fHRulerPosition write SetHRulerPosition default hrpTop;
    property VRulerPosition : tVRulerPosition  read fVRulerPosition write SetVRulerPosition default vrpLeft;

    property HRulerDecimals : integer read GetHRulerDecimals write SetHRulerDecimals default 0;
    property VRulerDecimals : integer read GetVRulerDecimals write SetVRulerDecimals default 0;

    property BackgroundColor:TColor read GetBackGroundColor write SetBackGroundColor default clWhite;
    property HRulerFont : TFont read fHFont write SetHRulerFont;
    property VRulerFont : TFont read fVFont write SetVRulerFont;

    property AvailableWidth : integer read GetAvailableWidth write fAvailableWidth;
    property AvailableHeight : integer read GetAvailableHeight write fAvailableHeight;

    property DetailedPosition : boolean read fDetailedPosition write fDetailedPosition;
    property Version : Integer read fVersion write SetVersion;


    property Font;
    property OnClick;
    property OnDblClick;
    property OnDragDrop;
    property OnDragOver;
    property OnEndDrag;
    property OnMouseDown read fOnMouseDown write fOnMouseDown;
    property OnMouseMove  read fOnMouseMove write fOnMouseMove;
    property OnMouseUp read fOnMouseUp write fOnMouseUp;
    property OnPaint : TNotifyEvent read fOnPaint write fOnPaint;

  end;

var
  RegisteredID : integer;

(******************************************************************************)
(******************************************************************************)

implementation


(******************************************************************************)


constructor TQSWorldGraph.Create(AOwner: TComponent);

begin
  inherited Create(AOwner);
  parent := TWinControl(AOwner);

  bFirstTime := true;

  SetVersion(gVersion);

  BevelWidth := 1;
  BevelInner := bvRaised;
  BevelOuter := bvRaised;

  Left     := 10;
  Top      := 10;
  width    := 300;
  height   := 300;

  if fHFont = nil then
    fHFont  := TFont.Create;
  if fVFont = nil then
    fVFont  := TFont.Create;

  with fHFont do
    begin
      name := 'arial';
      size := 8;
    end;

  with fVFont do
    begin
      name := 'arial';
      size := 8;
    end;

  fHRuler := CreateHRuler;
  fVRuler := CreateVRuler;

  fPaintBox  := CreatePaintBox; (*No need to destroy this on exit*)
  fCurrentCanvas := fPaintBox.canvas; (*this must change to printer.canvas for print*)

  fPaintBox.left := 30;
  fPaintBox.top  := 30;
  fPaintBox.width := width - fPaintBox.left - 5;
  fPaintBox.height := height - fPaintBox.top - 5;

  fPaintBox.OnMouseMove := DoOnMouseMove;
  fPaintBox.OnMouseDown := DoOnMouseDown;
  fPaintBox.OnMouseUp   := DoOnMouseUp;

  fCurrentCanvas.Pen.Color := clBlack;

  SetLengthUnits(luCustom);

  SetXIncrease(LeftToRight);

  with fHRuler do
    begin
      left := fPaintBox.left;
      top  := 2;
      width := fPaintBox.width;
      height := 28;
    end;

  SetYIncrease(BottomToTop);

  with fVRuler do
    begin
      left := 2;
      top  := fPaintBox.top;
      width := 28;
      height := fPaintBox.height;
    end;

  SetxMin(0);
  SetxMax(10);
  SetyMin(0);
  SetyMax(10);


  color := clWhite;
  SetBackGroundColor(clWhite);
  SetProportional(true);


  bZoom := false;
  ZoomLevel := 0;
  bPrinting := false;

  {$IFDEF EVALUATION THEN}
  if not(csDesigning in ComponentState) then

    MessageDlg('This program uses an evaluation copy of QSWorldGraph by Q-Systems Engineering',
                mtInformation,[mbOK],0);

  {$UNDEF EVALUATION}
  {$ENDIF}

  fDetailedPosition := false;
  DetailedX := -9999;
  DetailedY := -9999;

  bFirstTime := true;
end;

(******************************************************************************)

destructor TQSWorldGraph.Destroy;

begin
  if fHRuler <> nil then
    fHRuler.free;
  if fVRuler <> nil then
    fVRuler.free;
  if fPaintBox <> nil then
    fPaintBox.free;

  if fHFont <> nil then
    fHFont.free;
  if fVFont <> nil then
    fVFont.free;

  if BMPPicture <> nil then  BMPPicture.free;
  if WMFPicture <> nil then  WMFPicture.free;
  if JPEGPicture <> nil then JPEGPicture.free;

  inherited destroy;
end;


(******************************************************************************)

procedure TQSWorldGraph.SetVersion;

begin
  fVersion := gVersion;
end;

(******************************************************************************)

procedure TQSWorldGraph.PrintBitmap(Bitmap: TBitmap; X, Y: Integer;Magnification:double);

var
  Info: PBitmapInfo;
  InfoSize: {$ifdef win32 then}DWORD;{$else} integer;{$endif}
  Image: Pointer;
  ImageSize: {$ifdef win32 then}DWORD;{$else} LongInt;{$endif}
  a,b      : double;


begin
  if magnification = 0 then (*print same size as on screen*)
    begin
      a := 1.0*printer.pagewidth/screen.width;
      b := 1.0*printer.pageHeight/screen.Height;
      if a < b then  magnification := a
      else magnification := b;
    end
  else
    magnification := magnification*fBasicPrintMagnification;

  with Bitmap do
    begin
      GetDIBSizes(Handle, InfoSize, ImageSize);
      Info := (*MemAlloc(InfoSize)*) AllocMem(InfoSize);
      try
        Image := (*MemAlloc(ImageSize)*) AllocMem(ImageSize);
        try
          GetDIB(Handle, Palette, Info^, Image^);
          with Info^.bmiHeader do
            StretchDIBits(Printer.Canvas.Handle, X, Y, round(magnification*Width),
              round(magnification*Height), 0, 0, biWidth, biHeight, Image, Info^,
              DIB_RGB_COLORS, SRCCOPY);
        finally
          FreeMem(Image, ImageSize);
        end;
      finally
        FreeMem(Info, InfoSize);
      end;
    end;
end;

(******************************************************************************)

procedure TQSWorldGraph.StretchPrintBitMap(bm:TGraphic;Magnification:double);

var
  rect : TRect;
  xMin,xMax,yMin,yMax : double;
  a,b : double;

begin
  xMin := GeomXMin[0];
  xMax := GeomXMax[0];
  yMin := GeomYMin[0];
  yMax := GeomYMax[0];

  if magnification = 0 then (*print same size as on screen*)
    begin
      a := 1.0*printer.pagewidth/screen.width;
      b := 1.0*printer.pageHeight/screen.Height;
      if a < b then  magnification := a
      else magnification := b;
    end
  else
    magnification := magnification*fBasicPrintMagnification;

  with Rect do
    begin
      WorldToPC(xMin,yMin,left,bottom);
      WorldToPC(xMax,yMax,right,top);
      left := round(left*magnification);
      right := round(right*magnification);
      top := round(top*magnification);
      bottom := round(bottom*magnification);
    end;
  printer.canvas.stretchdraw(rect,bm);
end;

(******************************************************************************)

procedure TQSWorldGraph.PrintUserGraphic(Magnification:double);

var
  a, b  : double;
begin
  if Magnification = 0 then (*print same size as on screen*)
    begin
      a := 1.0*printer.pagewidth/screen.width;
      b := 1.0*printer.pageHeight/screen.Height;
      if a < b then  magnification := a
      else magnification := b;
      magnification := magnification/fBasicPrintMagnification;
    end;

  fPrintMagnification := Magnification;
  fPaintBox.fCurrentCanvas := printer.canvas;
  fPaintBox.SetPrintProportional(true,fPrintMagnification*fBasicPrintMagnification);
  paint;
  bPrinting := false;
  fPaintBox.fCurrentCanvas := fPaintBox.canvas;
  SetProportional(true);
  paint;
end;
(******************************************************************************)

function TQSWorldGraph.LoadBMP(FileName : String):boolean;

var
  ValidName : boolean;

begin

  result := false;

  ValidName := FileExists(FileName);

  if ValidName then
    try
      if BMPPicture = nil then  BMPPicture := TPicture.Create;
      BMPPicture.LoadFromFile(FileName);
    except
      on EInvalidGraphic do BMPPicture.Graphic := nil;
    end;
  if ValidName then
    if BMPPicture.Graphic is TBitMap then
      begin
        result := true;
      end;
end;

(******************************************************************************)
function TQSWorldGraph.LoadWMF(FileName : String):boolean;

var
  ValidName : boolean;

begin

  result := false;

  ValidName := FileExists(FileName);

  if ValidName then
    try
      if WMFPicture = nil then  WMFPicture := TPicture.Create;
      WMFPicture.LoadFromFile(FileName);
    except
      on EInvalidGraphic do WMFPicture.Graphic := nil;
    end;

  if ValidName then
    if WMFPicture.Graphic is TMetafile then
      begin
        result := true;
      end;

end;

(******************************************************************************)

{$IFDEF WIN32}
function  TQSWorldGraph.LoadJPeg(FileName : String):boolean;

var
  ValidName : boolean;

begin

  result := false;

  ValidName := FileExists(FileName);

  if ValidName then
    try
      if JPEGPicture = nil then  JPEGPicture := TPicture.Create;
      JPEGPicture.LoadFromFile(FileName);
    except
      on EInvalidGraphic do JPegPicture.Graphic := nil;
    end;
  {$ifdef newCompiler then}
  if JPEGPicture.Graphic is TJPEGImage then
    begin
      result := true;
    end;
  {$endif}  

end;
{$ENDIF}

(******************************************************************************)

function  TQSWorldGraph.CreateHRuler:TQSPBHRuler;
begin
  Result := TQSPBHRuler.Create(Self);
  Result.parent := self;
  Result.visible := true;
  Result.font := HRulerFont;
end;
(******************************************************************************)

function  TQSWorldGraph.CreateVRuler:TQSPBVRuler;
begin
  Result := TQSPBVRuler.Create(Self);
  Result.parent := self;
  Result.visible := true;
end;

(******************************************************************************)

function  TQSWorldGraph.CreatePaintBox : TQSPaintBox;
begin
  Result := TQSPaintBox.Create(Self);
  Result.parent := self;
  Result.visible := true;
end;


(******************************************************************************)

procedure TQSWorldGraph.Paint;


var
  h,w           : integer;

begin

  if not (csDesigning in ComponentState) then
    begin
      if bFirstTime then
        begin
          fHFontRelativeSize := 1.0*fHFont.size/Font.size;
          fVFontRelativeSize := 1.0*fVFont.size/Font.size;
          bFirstTime := false;
        end;

      fHFont.size := round(Font.size*fHFontRelativeSize);
      fVFont.size := round(Font.size*fVFontRelativeSize);

    end;
    
  if not bPrinting then
    begin
    (*  if Assigned(fOnPaint) then fOnPaint(self);*)

      fPaintBox.bRedraw := true;  (*I would think that this is needed.  However, there is
                                    No difference if I remove it.*)

      h := height;
      w := width;
      fPaintBox.XIncrease := GetXIncrease;
      fPaintBox.YIncrease := GetYIncrease;


      if Proportional then
        begin
          SetProportional(true);
          fHRuler.XMin := fPaintBox.fXMin;
          fHRuler.XMax := fPaintBox.fXMax;
          fVRuler.YMin := fPaintBox.fYMin;
          fVRuler.YMax := fPaintBox.fYMax;
        end
      else
        begin
          fPaintBox.fXMin := fHRuler.XMin;
          fPaintBox.fXMax := fHRuler.XMax;
          fPaintBox.fYMin := fVRuler.YMin;
          fPaintBox.fYMax := fVRuler.YMax;
          SetProportional(false);
        end;


      if fHRulerPosition = hrpTop then
        begin
          with fHRuler do
            begin
              DivisionPlacement := dpBottom;
              left := fPaintBox.left;
              top  := 0;
              width := fPaintBox.width;
              height := round(0.08*h);
              visible := true;
              paint;
            end;

          fPaintBox.top  := fHRuler.height;
          fPaintBox.height := height - fPaintBox.top;
        end
      else if fHRulerPosition = hrpBottom then
        begin
          with fHRuler do
            begin
              DivisionPlacement := dptop;
              left := fPaintBox.left;
              width := fPaintBox.width;
              height := round(0.08*h);
              top  := h - height;
              visible := true;
              paint;
            end;
          fPaintBox.top  :=  0;
          fPaintBox.height := h - fHRuler.height;
        end
      else if fHRulerPosition = hrpHidden then
        begin
          fHRuler.hide;
          fHRuler.top := -50;
          fPaintBox.top  := 0;
          fPaintBox.height := height;
        end;

      if fVRulerPosition = vrpLeft then
        begin
          with fVRuler do
            begin
              DivisionPlacement := dpRight;
              left := 0;
              top  := fPaintBox.top;
              width := round(w*0.07);
              height := fPaintBox.height;
              visible := true;
              paint;
            end;
          fPaintBox.left := fVRuler.width;
          fPaintBox.width := width - fPaintBox.left;
        end
      else if fVRulerPosition = vrpRight then
        begin
          with fVRuler do
            begin
              DivisionPlacement := dpLeft;
              left := fPaintBox.width;
              top  := fPaintBox.top;
              width := round(w*0.07);
              height := fPaintBox.height;
              visible := true;
              paint;
            end;
          fPaintBox.left := 0;
          fPaintBox.width := width - fVRuler.width;
        end
      else if fVRulerPosition = vrpHidden then
        begin
          fVRuler.hide;
          fVRuler.left := -50;
          fPaintBox.Left := 0;
          fPaintBox.width := width;
        end;

      if fHRuler.visible then  SetHRulerFont(fHFont);
      if fVRuler.visible then SetVRulerFont(fVFont);

      (*setup the world coordinate system*)
      with fPaintBox do
        begin
          SetWorldWindow(fxmin, fxmax, fymin, fymax, 0, 0, 0, 0);
          if (XIncrease = LeftToRight) and (YIncrease = BottomToTop) then
            SetPCWindow(0, width, height,0)
          else if (XIncrease = LeftToRight) and (YIncrease = TopToBottom) then
            SetPCWindow(0, width, 0,height)
          else if (XIncrease = RightToLeft) and (YIncrease = TopToBottom) then
            SetPCWindow(width,0, 0,height)
          else if (XIncrease = RightToLeft) and (YIncrease = BottomToTop) then
            SetPCWindow(width,0, height,0);
        end;

      if ZoomLevel = 0 then
        begin
          GeomXMin[ZoomLevel] := xMin;
          GeomXMax[ZoomLevel] := xMax;
          GeomYMin[ZoomLevel] := yMin;
          GeomYMax[ZoomLevel] := yMax;
        end;
    end;


  (*Draw whatever is assigned on the OnPaint event of the component*)
  if Assigned(fOnPaint) then
    begin
      fOnPaint(self);
    end;
  fPaintBox.bRedraw := false;
end;


(******************************************************************************)

procedure TQSWorldGraph.SetXIncrease(value:TXIncrease);

begin
  if value = LeftToRight then
    fHRuler.LeftToRight := true
  else
    fHRuler.LeftToRight := false;
end;

(******************************************************************************)

function  TQSWorldGraph.GetXIncrease:TXIncrease;

begin
  if fHRuler.LeftToRight then
    result := LeftToRight
  else
    result := RightToLeft;
end;

(******************************************************************************)

procedure TQSWorldGraph.SetYIncrease(value:TYIncrease);

begin
  if value = BottomToTop then
    fVRuler.BottomToTop := true
  else
    fVRuler.BottomToTop := false;
end;

(******************************************************************************)

function  TQSWorldGraph.GetYIncrease:TYIncrease;

begin
  if fVRuler.BottomtoTop then
    result := BottomtoTop
  else
    result := TopToBottom;
end;

(******************************************************************************)


procedure TQSWorldGraph.SetLengthUnits(value:TLengthUnits);


begin
  if fLengthUnits <> value then
    begin
      fLengthUnits := value;
      fHRuler.LengthUnits := value;
      fVRuler.LengthUnits := value;
      if value <> luCustom then
        SetProportional(false);
      if visible then
        paint;
    end;
end;

(******************************************************************************)

procedure TQSWorldGraph.SetHRulerPosition(value:tHRulerPosition);

begin
  if fHRulerPosition <> value then
    begin
      fHRulerPosition := value;
      if visible then
        paint;
    end;
end;

(******************************************************************************)
procedure TQSWorldGraph.SetVRulerPosition(value:tVRulerPosition);

begin
  if fVRulerPosition <> value then
    begin
      fVRulerPosition := value;
      if visible then
        paint;
    end;
end;

(******************************************************************************)

procedure TQSWorldGraph.SetHRulerDecimals(value:integer);
begin
  if fHRuler.Decimals <> value then
    begin
      fHRuler.Decimals := value;
    end;
end;

(******************************************************************************)
function  TQSWorldGraph.GetHRulerDecimals:integer;

begin
  result := fHRuler.Decimals;
end;

(******************************************************************************)
procedure TQSWorldGraph.SetVRulerDecimals(value:integer);

begin
  if fVRuler.Decimals <> value then
    begin
      fVRuler.Decimals := value;
    end;
end;

(******************************************************************************)
function  TQSWorldGraph.GetVRulerDecimals:integer;

begin
  result := fVRuler.Decimals;
end;

(******************************************************************************)

procedure TQSWorldGraph.DoOnMouseMove(Sender: TObject; Shift: TShiftState; X, Y: Integer);

var
  r : TWorldPoint;
begin
  if fDetailedPosition then
    begin
      if (DetailedX <> -9999) and (DetailedY <> -9999) then
        fPaintBox.DrawCrossAt(pmNot,DetailedX,DetailedY);
      fPaintBox.DrawCrossAt(pmNot,x,y);
      DetailedX := x;
      DetailedY := y;
    end;

  if assigned(fOnMouseMove) then
    begin
      IPBx := x;
      IPBy := y;
      r := Wxy(x,y);
      RPBx := r.x;
      RPBy := r.y;
      x := x + fPaintBox.left;
      y := y + fPaintBox.top;
      fOnMouseMove(Sender,Shift,X, Y);
    end;
end;

procedure TQSWorldGraph.RemoveDetailedPosition;

begin
  if (DetailedX <> -9999) and (DetailedY <> -9999) then
    fPaintBox.DrawCrossAt(pmNot,DetailedX,DetailedY);
  DetailedX := -9999;
  DetailedY := -9999;
end;

procedure TQSWorldGraph.DoOnMouseDown(Sender: TObject;Button: TMouseButton;
                             Shift: TShiftState; X, Y: Integer);

var
  r : TWorldPoint;

begin
  if assigned(fOnMouseDown) then
    begin
      IPBx := x;
      IPBy := y;
      r := Wxy(x,y);
      RPBx := r.x;
      RPBy := r.y;
      x := x + fPaintBox.left;
      y := y + fPaintBox.top;
      fOnMouseDown(Sender,Button,Shift,X, Y);
    end;
end;

(******************************************************************************)

procedure TQSWorldGraph.DoOnMouseUp(Sender: TObject;Button: TMouseButton;
                             Shift: TShiftState; X, Y: Integer);

var
  r : TWorldPoint;
                               
begin
  if assigned(fOnMouseUp) then
    begin
      IPBx := x;
      IPBy := y;
      r := Wxy(x,y);
      RPBx := r.x;
      RPBy := r.y;
      x := x + fPaintBox.left;
      y := y + fPaintBox.top;
      fOnMouseUp(Sender,Button,Shift,X, Y);
    end;
end;

(******************************************************************************)

procedure TQSWorldGraph.SetBackgroundColor(value:TColor);
begin
  if fPaintBox.color <> value then
    fPaintBox.color := value;
end;

(******************************************************************************)

function  TQSWorldGraph.GetBackGroundColor:TColor;

begin
  result := fPaintBox.color;
end;

(******************************************************************************)

procedure TQSWorldGraph.SetHRulerFont(value:TFont);

begin
  if fHRuler.font <> value then
    begin
      fHRuler.font := value;

      if fHFont = nil then
        fHFont := TFont.create;

      fHFont.name := value.name;
      fHFont.size := value.size;
      fHFont.color := value.color;
      fHFont.style := value.style;
    end;
end;


(******************************************************************************)


procedure TQSWorldGraph.SetVRulerFont(value:TFont);
begin
  if fVRuler.font <> value then
    begin
      fVRuler.font := value;
      if fVFont = nil then
        fVFont := TFont.create;
      fVFont.name := value.name;
      fVFont.size := value.size;
      fVFont.color := value.color;
    end;
end;

(******************************************************************************)

function TQSWorldGraph.GetAvailableWidth:integer;

begin
  result := fPaintBox.width;
end;

(******************************************************************************)

function TQSWorldGraph.GetAvailableHeight :integer;

begin
  result := fPaintBox.height;
end;

(******************************************************************************)

procedure TQSWorldGraph.DrawLine;

(*
  Draw the line (WLine.x1,WLine.y1) (WLine.x2,WLine.y2) by making certain that
  all its points are within the drawing frame
*)

var
  x0,y0 : double;
  x,y   : array[1..2] of double;
  counter : integer;

begin
  (*Case: Line within the visible frame*)
  with WLine do
    begin

      if InFrame(x1,y1) and InFrame(x2,y2) then
        begin
          fPaintBox.WorldMoveTo(x1,y1);
          fPaintBox.WorldLineTo(x2,y2);
        end

         (*Case: Only first point of Line within the visible frame*)
      else if InFrame(x1,y1) and (not InFrame(x2,y2)) then
        begin
          fPaintBox.WorldMoveTo(x1,y1);
          if LineSectionInterceptsYMinLine(x0) then
             fPaintBox.WorldLineTo(x0,YMin)
          else if LineSectionInterceptsYMaxLine(x0) then
             fPaintBox.WorldLineTo(x0,YMax)
          else if LineSectionInterceptsXMinLine(y0) then
             fPaintBox.WorldLineTo(xMin,y0)
          else if LineSectionInterceptsXMaxLine(y0) then
             fPaintBox.WorldLineTo(xMax,y0);
        end
          (*Case: Only second point of Line within the visible frame*)
      else if (not InFrame(x1,y1) and InFrame(x2,y2)) then
        begin
          fPaintBox.WorldMoveTo(x2,y2);
          if LineSectionInterceptsYMinLine(x0) then
             fPaintBox.WorldLineTo(x0,YMin)
          else if LineSectionInterceptsYMaxLine(x0) then
             fPaintBox.WorldLineTo(x0,YMax)
          else if LineSectionInterceptsXMinLine(y0) then
             fPaintBox.WorldLineTo(xMin,y0)
          else if LineSectionInterceptsXMaxLine(y0) then
             fPaintBox.WorldLineTo(xMax,y0);
        end
      (*Case: No end point of Line is within the visible frame*)
      else if (not InFrame(x1,y1)) and (not InFrame(x2,y2)) then
        begin
          counter := 0;
          if LineSectionInterceptsYMinLine(x0) then
            begin
              inc(counter);
              x[counter] := x0;
              y[counter] := YMin;
            end;
          if LineSectionInterceptsYMaxLine(x0) then
            begin
              inc(counter);
              x[counter] := x0;
              y[counter] := YMax;
            end;
          if LineSectionInterceptsXMinLine(y0) then
            begin
              inc(counter);
              x[counter] := XMin;
              y[counter] := y0;
            end;
          if LineSectionInterceptsXMaxLine(y0) then
            begin
              inc(counter);
              x[counter] := XMax;
              y[counter] := y0;
            end;
          if counter = 2 then
            begin
              fPaintBox.WorldMoveTo(x[1],y[1]);
              fPaintBox.WorldLineTo(x[2],y[2]);
            end;
        end;
      end;
end;

(******************************************************************************)

function TQSWorldGraph.InFrame(x1,y1:double):boolean;

(*
  Check if point (x1,y1) is in the WorldGraph visible frame
*)
begin
  result := (x1>=XMin) and (x1<=XMax) and (y1>=YMin) and (y1<=YMax);
end;

(******************************************************************************)

function TQSWorldGraph.LineSectionInterceptsYMinLine(var x0:double):boolean;
(*
  Find intersection of WLine with bottom frame boundary
*)

var
  A,B: double; (*a line is y = Ax+B*)
  y0 : double; (*y coordinate of x0*)

begin
  with WLine do
    begin
      if (y2=y1) then
        begin
          result := false;
          exit;
        end;
      if (x2 = x1) then x2 := x1+1e-7; 
      A := (y2-y1)/(x2-x1);
      B := (y1*x2-y2*x1)/(x2-x1);
      x0 := (YMin - B)/A;
      y0 := YMin;
      if ((y0-y1)*(y2-y0) >= 0) and ((x0-XMin)*(XMax-x0) >= 0) then result := true
      else result := false;
    end;

end;

(******************************************************************************)
function TQSWorldGraph.LineSectionInterceptsYMaxLine(var x0:double):boolean;
(*
  Find intersection of WLine with top frame boundary
*)

var
  A,B: double; (*a line is y = Ax+B*)
  y0 : double; (*y coordinate of x0*)

begin
  with WLine do
    begin
      if (y2=y1) then
        begin
          result := false;
          exit;
        end;

      A := (y2-y1)/(x2-x1);
      B := (y1*x2-y2*x1)/(x2-x1);
      x0 := (YMax - B)/A;
      y0 := YMax;
      if ((y0-y1)*(y2-y0) >= 0) and ((x0-XMin)*(XMax-x0) >= 0) then result := true
      else result := false;
    end;

end;

(******************************************************************************)
function TQSWorldGraph.LineSectionInterceptsXMinLine(var y0:double):boolean;
(*
  Find intersection of WLine with left frame boundary
*)

var
  A,B: double; (*a line is y = Ax+B*)
  x0 : double; (*x coordinate of y0*)

begin
  with WLine do
    begin
      if (x2=x1) then
        begin
          result := false;
          exit;
        end;
      A := (y2-y1)/(x2-x1);
      B := (y1*x2-y2*x1)/(x2-x1);

      y0 := A*XMin + B;
      x0 := XMin;
      if ((x0-x1)*(x2-x0) >= 0) and ((y0-YMin)*(YMax-y0) >= 0) then result := true
      else result := false;
    end;
end;

(******************************************************************************)
function TQSWorldGraph.LineSectionInterceptsXMaxLine(var y0:double):boolean;
(*
  Find intersection of WLine with right frame boundary
*)

var
  A,B: double; (*a line is y = Ax+B*)
  x0 : double; (*x coordinate of y0*)

begin
  with WLine do
    begin
      if (x2=x1) then
        begin
          result := false;
          exit;
        end;

      A := (y2-y1)/(x2-x1);
      B := (y1*x2-y2*x1)/(x2-x1);

      y0 := A*XMax + B;
      x0 := XMax;
      if ((x0-x1)*(x2-x0) >= 0) and ((y0-YMin)*(YMax-y0) >= 0) then result := true
      else result := false;
    end;
end;

(******************************************************************************)
function TQSWorldGraph.PartOfFrameShows(wx1,wy1,wx2,wy2:double):boolean;

var
  temp : double;

begin
  if wx1 > wx2 then
    begin
      temp := wx1;
      wx1 := wx2;
      wx2 := temp;
    end;
  if wy1 > wy2 then
    begin
      temp := wy1;
      wy1 := wy2;
      wy2 := temp;
    end;
  result := false;
  result := result or InFrame(wx1,wy1);
  result := result or InFrame(wx2,wy1);
  result := result or InFrame(wx2,wy2);
  result := result or InFrame(wx1,wy2);

  (*frame completely encompases viewframe*)
  result := result or ((wx1 <= XMin)and(wx2 >= XMax)and(wy1 <= YMin)and(wy2 >= YMax));

  (*frame completely encompases upper part of viewframe*)
  result := result or ((wx1 <= XMin)and(wx2 >= XMax)and((wy1 >= YMin) and (wy1 <= YMax) and (wy2 >= YMax)));

  (*frame completely encompases lower part of viewframe*)
  result := result or ((wx1 <= XMin)and(wx2 >= XMax)and((wy1 <= YMin)and(wy2<=YMax)and(wy2>=YMin)));

  (*frame completel encompases the left part of viewframe*)
  result := result or ((wy1 <= YMin)and(wy2 >= YMax)and((wx1 <= XMin)and(wx2>=XMin)and(wx2 <= XMax)));

  (*frame completel encompases the right part of viewframe*)
  result := result or ((wy1 <= YMin)and(wy2 >= YMax)and((wx1 >= XMin)and(wx1<=XMax)and(wx2>= XMax)));

  (*frame completely encompases the horizontal middle of the viewframe*)
  result := result or ((wx1 <= XMin)and(wx2 >= XMax)and((wy1 >= YMin)and(wy2 <= YMax)));

  (*frame completely encompases the vertical middle of the viewframe*)
  result := result or ((wy1 <= YMin)and(wy2 >= YMax)and((wx1 >= XMin)and(wx2 <= XMax)));

end;

(******************************************************************************)


function TQSWorldGraph.AllOfFrameShows(wx1,wy1,wx2,wy2:double):boolean;

begin
  result := ((wx1 >= XMin)and(wx2 <= XMax)and(wy1 >= YMin)and(wy2 <= YMax));
end;
(******************************************************************************)

procedure TQSWorldGraph.DrawBitMap(bm:TGraphic);

begin
  fPaintBox.canvas.draw(0,0,bm);
end;


procedure TQSWorldGraph.StretchDrawBitMap(bm:TGraphic);

var
  rect : TRect;
  xMin,xMax,yMin,yMax : double;


begin
  xMin := GeomXMin[0];
  xMax := GeomXMax[0];
  yMin := GeomYMin[0];
  yMax := GeomYMax[0];
  with Rect do
    begin
      WorldToPC(xMin,yMin,left,bottom);
      WorldToPC(xMax,yMax,right,top);
    end;
  fPaintBox.canvas.stretchdraw(rect,bm);
end;


(******************************************************************************)

procedure TQSWorldGraph.SetXMin(value:double);
begin
  fHRuler.xMin := value;
  fPaintBox.xMin := value;
end;

(******************************************************************************)

function TQSWorldGraph.GetXMin:double;
begin
  result := fHRuler.xMin;
end;

(******************************************************************************)

procedure TQSWorldGraph.SetXMax(value:double);
begin
  fHRuler.xMax := value;
  fPaintBox.xMax := value;
end;

(******************************************************************************)

function TQSWorldGraph.GetXMax:double;
begin
  result := fHRuler.xMax;
end;

(******************************************************************************)

procedure TQSWorldGraph.SetYMin(value:double);
begin
  fVRuler.yMin := value;
  fPaintBox.yMin := value;
end;

(******************************************************************************)

function TQSWorldGraph.GetYMin:double;
begin
  result := fVRuler.yMin;
end;

(******************************************************************************)

procedure TQSWorldGraph.SetYMax(value:double);
begin
  fVRuler.yMax := value;
  fPaintBox.yMax := value;
end;

(******************************************************************************)

function TQSWorldGraph.GetYMax:double;
begin
  result := fVRuler.yMax;
end;

(******************************************************************************)


procedure TQSWorldGraph.SetProportional(value:boolean);

(*if false, then the world coordinates are attached to the PC coordinates as are.
  If true then PC and World coordinates must maintain the same proportions.  For
  example, if the X and Y PC ranges are 100 and 100 while the X and Y world
  ranges are 10 and 5 then the world range should not cover the entire canvas of
  the component, or the drawn shapes will be unrealistic.*)


begin
  fPaintBox.SetProportional(value);
end;

(******************************************************************************)

function TQSWorldGraph.GetProportional:boolean;

begin
  result := fPaintBox.Proportional;
end;

(******************************************************************************)
procedure TQSWorldGraph.IncreaseZoom(x1,y1,x2,y2:double);

var
  temp : double;

begin

  GeomXMin[ZoomLevel] := xMin;
  GeomXMax[ZoomLevel] := xMax;
  GeomYMin[ZoomLevel] := yMin;
  GeomYMax[ZoomLevel] := yMax;

  (*Set the ZOOM frame coordinates*)
  fXMinZoom := x1;
  fYMinZoom := y1;
  fXMaxZoom := x2;
  fYMaxZoom := y2;

  (* if min > max then switch values*)
  if fXMinZoom > fXMaxZoom then
    begin
      temp := fXMinZoom;
      fXMinZoom := fXMaxZoom;
      fXMaxZoom := temp;
    end;
  if fYMinZoom > fYMaxZoom then
    begin
      temp := fYMinZoom;
      fYMinZoom := fYMaxZoom;
      fYMaxZoom := temp;
    end;

  xMin := fXMinZoom;
  xMax := fXMaxZoom;
  yMin := fYMinZoom;
  yMax := fYMaxZoom;

  inc(ZoomLevel);
  bZoom := (ZoomLevel > 0);
  paint;
end;

(******************************************************************************)
procedure TQSWorldGraph.DecreaseZoom;


begin

  if ZoomLevel = 0 then exit;

  Dec(ZoomLevel);

  xMin := GeomXMin[ZoomLevel];
  xMax := GeomXMax[ZoomLevel];
  yMin := GeomYMin[ZoomLevel];
  yMax := GeomYMax[ZoomLevel];

  bZoom := (ZoomLevel > 0);
  paint;
end;

(******************************************************************************)

procedure TQSWorldGraph.ResetZoom;

begin
  if ZoomLevel = 0 then exit;

  ZoomLevel := 0;

  xMin := GeomXMin[ZoomLevel];
  xMax := GeomXMax[ZoomLevel];
  yMin := GeomYMin[ZoomLevel];
  yMax := GeomYMax[ZoomLevel];

  bZoom := (ZoomLevel > 0);
  paint;
end;

(******************************************************************************)

procedure TQSWorldGraph.BeginSelect(bx,by:integer);

begin
  fPaintBox.canvas.pen.mode := pmNot;
  StartX := bx;
  StartY := by;
  PreviousX := bx;
  PreviousY := by;
end;

(******************************************************************************)

procedure TQSWorldGraph.ContinueSelect(cx,cy:integer);

begin
  with fPaintBox.canvas do
    begin
      MoveTo(StartX,StartY);
      LineTo(PreviousX,StartY);
      LineTo(PreviousX,PreviousY);
      LineTo(StartX,PreviousY);
      LineTo(StartX,StartY);
      MoveTo(StartX,StartY);
      LineTo(cX,StartY);
      LineTo(cx,cy);
      LineTo(StartX,cy);
      LineTo(StartX,StartY);
      PreviousX := cx;
      PreviousY := cy;
    end;
end;

(******************************************************************************)

function  TQSWorldGraph.EndSelect(ex,ey:integer):boolean;

var
  iTemp : integer;

begin
  with fPaintBox.canvas do
    begin
      MoveTo(StartX,StartY);
      LineTo(PreviousX,StartY);
      LineTo(PreviousX,PreviousY);
      LineTo(StartX,PreviousY);
      LineTo(StartX,StartY);
      pen.mode := pmcopy;

      if (StartX = ex) or (StartY = ey) then
        begin
          result := false;
          exit;
        end
      else
        result := true;

      (*Make certain that StartX,StartY is the top left end of selection*)
      if StartX > eX then
       begin
         ITemp := StartX;
         StartX := ex;
         ex := iTemp;
       end;
      if StartY  > ey then
        begin
          iTemp := StartY;
          StartY := eY;
          ey := iTemp;
        end;
    end;

  EndX := ex;
  EndY := ey;


  if (XIncrease = LeftToRight) and (YIncrease = BottomToTop) then
    begin
      PCToWorld(StartX,StartY,fXMinZoom,fYMaxZoom);
      PCToWorld(EndX,EndY,fXMaxZoom,fYMinZoom);
    end
  else if (XIncrease = LeftToRight) and (YIncrease = TopToBottom) then
    begin
      PCToWorld(StartX,StartY,fXMinZoom,fYMinZoom);
      PCToWorld(EndX,EndY,fXMaxZoom,fYMaxZoom);
    end
  else if (XIncrease = RightToLeft) and (YIncrease = TopToBottom) then
    begin
      PCToWorld(StartX,StartY,fXMaxZoom,fYMinZoom);
      PCToWorld(EndX,EndY,fXMinZoom,fYMaxZoom);
    end
  else if (XIncrease = RightToLeft) and (YIncrease = BottomToTop) then
    begin
      PCToWorld(StartX,StartY,fXMaxZoom,fYMaxZoom);
      PCToWorld(EndX,EndY,fXMinZoom,fYMinZoom);
    end;
  IncreaseZoom(fXMinZoom,fYMinZoom,fXMaxZoom,fYMaxZoom);
end;


(******************************************************************************)


procedure TQSWorldGraph.BeginDistance(bx,by:integer;bRePaint:boolean);

begin
  if bRePaint then Repaint;
  fPaintBox.canvas.pen.mode := pmNot;
  StartX := bx;
  StartY := by;
  DistanceFirst := Wxy(bx,by);
  PreviousX := bx;
  PreviousY := by;
end;

(******************************************************************************)

function TQSWorldGraph.ContinueDistance(cx,cy:integer):double;

begin
  with fPaintBox.canvas do
    begin
      MoveTo(StartX,StartY);
      LineTo(PreviousX,PreviousY);
      MoveTo(StartX,StartY);
      LineTo(cx,cy);
      PreviousX := cx;
      PreviousY := cy;
      DistanceLast := Wxy(cx,cy);
      result := sqrt(sqr(DistanceLast.X-DistanceFirst.X)+
                     sqr(DistanceLast.Y-DistanceFirst.Y));
    end;
end;

(******************************************************************************)

function  TQSWorldGraph.EndDistance(ex,ey:integer;bDrawDistanceLine:boolean):double;

begin
  with fPaintBox.canvas do
    begin
      if not bDrawDistanceLine then
        begin
          MoveTo(StartX,StartY);
          LineTo(PreviousX,PreviousY);
        end;
      pen.mode := pmCopy;
      result := sqrt(sqr(DistanceLast.X-DistanceFirst.X)+
                     sqr(DistanceLast.Y-DistanceFirst.Y));
    end;
end;


(******************************************************************************)

function TQSWorldGraph.Wxy(PCx,PCy:integer):TWorldPoint;

(*
   This function returns the world coordinates of a point within
   the Q-Systems WorldGraph component for a point of local
   screen coordinates PCx, PCy, such as the ones returned by the
   OnMouseMove, OnMouseUp and OnMouseDown events of this component
*)

var
  wx,wy : double;

begin
  if (aa = 0) or (cc = 0) then exit; (*aa and cc are denominator values in PCtoWorld*)
  PCToWorld(PCx,PCy, wx, wy);
  with result do
    begin
      x := wx;
      y := wy;
    end;
end;

(******************************************************************************)

function TQSWorldGraph.PCxy(wx,wy:double):TPixelsPoint;

(*
   This function returns the local screen coordinates of a point within
   the Q-Systems WorldGraph component for a point of World Coordinates wx, wy
*)

var
  PCx, PCy : integer;

begin
  WorldToPC(wx, wy, PCx, PCy);
  with result do
    begin
      x := PCx;
      y := PCy;
    end;

end;

(******************************************************************************)


procedure TQSWorldGraph.SetCursorTo(wx,wy:double);

var
  PCx, PCy : integer;
  ps, pc   : TPoint; (*screen point and client point*)

begin
  WorldToPC(wx, wy, PCx, PCy);
  pc.x := PCx;
  pc.y := PCy;
  ps := fPaintBox.ClientToScreen(pc);
  SetCursorPos(ps.x,ps.y);
end;

(******************************************************************************)


procedure TQSWorldGraph.BeginPrinting;

begin
  bPrinting := true;

  fBasicPrintMagnification := 1.0*printer.PageWidth/width;
  if 1.0*printer.PageHeight/Height < fBasicPrintMagnification then
    fBasicPrintMagnification := 1.0*printer.PageHeight/Height;

  printer.BeginDoc;
end;

(******************************************************************************)

procedure TQSWorldGraph.EndPrinting;

begin
  bPrinting := false;
  printer.EndDoc;
end;

(******************************************************************************)

procedure TQSWorldGraph.WorldLineTo (Wx,Wy:double);
begin
  WLine.x2 := Wx;
  WLine.y2 := Wy;
  DrawLine;
  WLine.x1 := Wx;
  WLine.y1 := Wy;

(*  /// fPaintBox.WorldLineTo(wx,wy); *)
end;

(******************************************************************************)

procedure TQSWorldGraph.WorldMoveTo (Wx,Wy:double);
begin
  WLine.x1 := Wx;
  WLine.y1 := Wy;
  fPaintBox.WorldMoveTo(Wx,Wy);
end;

(******************************************************************************)

procedure  TQSWorldGraph.WorldArc(WX1, WY1, WX2, WY2, WX3, WY3, WX4, WY4: Double);

const
  DIVISIONS = 1000;

var
  a,b              : double;
  cx,cy            : double;
  PhiStart, PhiEnd : double;
  DeltaPhi         : double;
  phi,r            : double;
  p                : TWorldPoint;
  i                : integer;

(*****************included *****************)
function PhiOf(WX1, WY1, WX2, WY2, WX3, WY3:double):double;
begin
  if WX3 = 0 then WX3 := 1e-10;
  if WY3 = 0 then WY3 := 1e-10;
  result := 0;
  if (WX3>=0) and (WY3>=0) then
      result := arcTan(abs(WY3/WX3))
  else if (WX3<=0) and (WY3>=0) then
    result := 0.5*pi+arcTan(abs(WX3/WY3))
  else if (WX3<=0) and (WY3<=0) then
    result := pi+arcTan(abs(WY3/WX3))
  else if (WX3>=0) and (WY3<=0) then
    result := 1.5*pi+arcTan(abs(WX3/WY3))
end;
(************end of included PhiOf **********)

function arcPoint(a,b,phi:double):TWorldPoint;
begin
  r := a*b/sqrt(sqr(b*cos(Phi))+sqr(a*sin(Phi)));
  result.x := r*cos(Phi);
  result.y := r*sin(Phi);
end;

begin
  if  AllOfFrameShows(WX1,WY1,WX2,WY2) then
    begin
      fPaintBox.WorldArc(WX1, WY1, WX2, WY2, WX3, WY3, WX4, WY4);
      exit;
    end;  

  if PartOfFrameShows(WX1,WY1,WX2,WY2) then
    begin
      cx := (WX2+WX1)*0.5;
      cy := (WY2+WY1)*0.5;
      WX1 := WX1-cx;
      WX2 := WX2-cx;
      WX3 := WX3-cx;
      WX4 := WX4-cx;

      WY1 := WY1-cy;
      WY2 := WY2-cy;
      WY3 := WY3-cy;
      WY4 := WY4-cy;

      a := abs(WX2-WX1)*0.5;
      b := abs(WY2-WY1)*0.5;
      PhiStart := PhiOf(WX1, WY1, WX2, WY2, WX3, WY3);
      PhiEnd := PhiOf(WX1, WY1, WX2, WY2, WX4, WY4);
      if PhiEnd < PhiStart then PhiEnd := PhiEnd+2*pi;
      DeltaPhi := (PhiEnd-PhiStart)/DIVISIONS;
      Phi := PhiStart;
      p := arcPoint(a,b,phi);
      WorldMoveTo(p.x+cx,p.y+cy);
      for i := 1 to DIVISIONS do
        begin
          Phi := Phi+DeltaPhi;
          p := arcPoint(a,b,phi);
          WorldLineTo(p.x+cx,p.y+cy);
        end;
    end;
end;

(******************************************************************************)

procedure  TQSWorldGraph.WorldEllipse(WX1, WY1, WX2, WY2: Double);

const
  DIVISIONS = MAX_POLYGON_POINTS DIV 2;

var
  ellipsePoints : TWorldPointArray;
  clippedEllipsePoints : TWorldPointArray;
  numberOfClipppedPoints : integer;
  i : integer;
  a,b : double;
  xc,yc : double;
  Dx : double;
  x  : double;

begin
  if not PartOfFrameShows(WX1,WY1,WX2,WY2) then
    exit;

  if  AllOfFrameShows(WX1,WY1,WX2,WY2) then
    begin
      fPaintBox.WorldEllipse(WX1,WY1,WX2,WY2);
      exit;
    end;

    
  xc := 0.5*(wx1+wx2);
  yc := 0.5*(wy1+wy2);

  a := abs(wx2-wx1)*0.5;
  b := abs(wy2-wy1)*0.5;
  Dx := 2*a/DIVISIONS;
  x := -a;

  for i := 1 to DIVISIONS+1 do
    begin
      ellipsePoints[i].y := -sqrt(abs(1-sqr(x/a)))*b;
      ellipsePoints[i].x := x;
      x := x+Dx;
    end;
  for i := 1 to DIVISIONS-1 do
    begin
      ellipsePoints[DIVISIONS+1+i].y := -ellipsePoints[DIVISIONS+1-i].y;
      ellipsePoints[DIVISIONS+1+i].x := ellipsePoints[DIVISIONS+1-i].x;
    end;
  for i := 1 to 2*DIVISIONS DO
    begin
      ellipsePoints[i].y := ellipsePoints[i].y + yc;
      ellipsePoints[i].x := ellipsePoints[i].x + xc
    end;

  ClipPolygon(ellipsePoints,2*DIVISIONS,clippedEllipsePoints,numberOfClipppedPoints);

  WorldPolygon(2*DIVISIONS, ellipsePoints, fCurrentcanvas.Pen.Color,fCurrentCanvas.Brush.Color,true);

end;

{
procedure  TQSWorldGraph.WorldEllipse(WX1, WY1, WX2, WY2: Double);
begin
  if PartOfFrameShows(WX1,WY1,WX2,WY2) then
    fPaintBox.WorldEllipse(WX1,WY1,WX2,WY2);
end;
}
(******************************************************************************)

procedure TQSWorldGraph.WorldQuad4 (Wx1,Wy1,Wx2,Wy2,Wx3,Wy3,Wx4,Wy4:Double;
                      color:TColor;bShowBoundary:boolean);

var
  i      : integer;
  Points: TWorldPointArray;
  outPoints    : TWorldPointArray;
  myOutPoints : TWPointArray;
  outNumberOfPoints : integer;

begin
  Points[1].x := Wx1;
  Points[2].x := Wx2;
  Points[3].x := Wx3;
  Points[4].x := Wx4;
  Points[1].y := Wy1;
  Points[2].y := Wy2;
  Points[3].y := Wy3;
  Points[4].y := Wy4;


  ClipPolygon(Points,4,outPoints,outNumberOfPoints);
  for i := 1 to outNumberOfPoints do
    myOutPoints[i] := TWPoint(outPoints[i]);
  fPaintBox.WorldPolygon(outNumberOfPoints,myOutPoints,clBlack,Color,bShowBoundary);

end;

{
procedure TQSWorldGraph.WorldQuad4 (Wx1,Wy1,Wx2,Wy2,Wx3,Wy3,Wx4,Wy4:Double;
                      color:TColor;bShowBoundary:boolean);
begin
  if PartOfFrameShows(WX1,WY1,WX3,WY3) or PartOfFrameShows(WX2,WY2,WX4,WY4) then
    fPaintBox.WorldQuad4 (Wx1,Wy1,Wx2,Wy2,Wx3,Wy3,Wx4,Wy4,color,bShowBoundary);
end;
}
(******************************************************************************)

procedure TQSWorldGraph.WorldQuad8 (Wx1,Wy1,Wx2,Wy2,Wx3,Wy3,Wx4,Wy4:Double;
                      Wx5,Wy5,Wx6,Wy6,Wx7,Wy7,Wx8,Wy8:Double;
                      color:TColor;bShowBoundary:boolean);
begin
  if PartOfFrameShows(WX1,WY1,WX3,WY3) or PartOfFrameShows(WX2,WY2,WX4,WY4) then
    fPaintBox.WorldQuad8 (Wx1,Wy1,Wx2,Wy2,Wx3,Wy3,Wx4,Wy4,
                      Wx5,Wy5,Wx6,Wy6,Wx7,Wy7,Wx8,Wy8,
                      color,bShowBoundary);
end;

(******************************************************************************)

procedure TQSWorldGraph.WorldPolygon(NumberOfPoints:word; Points: TWorldPointArray;
                                   PenColor,FillColor:TColor;bShowBoundary:boolean);

var
  i                 : integer;
  outPoints         : TWorldPointArray;
  myOutPoints       : TWPointArray;
  outNumberOfPoints : integer;

begin
  ClipPolygon(Points,NumberOfPoints,outPoints,outNumberOfPoints);
  if OutNumberOfPoints > 0 then
    begin
      for i := 1 to outNumberOfPoints do
        myOutPoints[i] := TWPoint(outPoints[i]);
      fPaintBox.WorldPolygon(outNumberOfPoints,myOutPoints,PenColor,FillColor,bShowBoundary);
    end;
end;
(******************************************************************************)

procedure TQSWorldGraph.SquarePoint(PointSize:integer;x,y:Double;color:TColor);

begin
  if bPrinting then PointSize := round(PointSize*fPrintMagnification*fBasicPrintMagnification);
  if inFrame(x,y) then
    fPaintBox.SquarePoint(PointSize,x,y,color);
end;

(******************************************************************************)

procedure TQSWorldGraph.TrianglePoint(PointSize:integer;x,y:Double;color:TColor);

begin
  if bPrinting then PointSize := round(PointSize*fPrintMagnification*fBasicPrintMagnification);
  if inFrame(x,y) then
    fPaintBox.TrianglePoint(PointSize,x,y,color);
end;

(******************************************************************************)
procedure TQSWorldGraph.CirclePoint(PointSize:integer;x,y:Double;color:TColor);

begin
  if bPrinting then PointSize := round(PointSize*fPrintMagnification*fBasicPrintMagnification);
  if inFrame(x,y) then
    fPaintBox.CirclePoint(PointSize,x,y,color);
end;

(******************************************************************************)

procedure TQSWorldGraph.EmptySquarePoint(PointSize:integer;x,y:Double;color:TColor);

begin
  if bPrinting then PointSize := round(PointSize*fPrintMagnification*fBasicPrintMagnification);
  if inFrame(x,y) then
    fPaintBox.EmptySquarePoint(PointSize,x,y,color);
end;

(******************************************************************************)

procedure TQSWorldGraph.EmptyTrianglePoint(PointSize:integer;x,y:Double;color:TColor);

begin
  if bPrinting then PointSize := round(PointSize*fPrintMagnification*fBasicPrintMagnification);
  if inFrame(x,y) then
    fPaintBox.EmptyTrianglePoint(PointSize,x,y,color);
end;

(******************************************************************************)

procedure TQSWorldGraph.EmptyCirclePoint(PointSize:integer;x,y:Double;color:TColor);

begin
  if bPrinting then PointSize := round(PointSize*fPrintMagnification*fBasicPrintMagnification);
  if inFrame(x,y) then
    fPaintBox.EmptyCirclePoint(PointSize,x,y,color);
end;

procedure TQSWorldGraph.CrossPoint(PointSize:integer;x,y:Double;color:TColor);
begin
  if bPrinting then PointSize := round(PointSize*fPrintMagnification*fBasicPrintMagnification);
  if inFrame(x,y) then
    fPaintBox.CrossPoint(PointSize,x,y,color);
end;

Procedure TQSWorldGraph.SetPen(PenColor:TColor;PenMode:TPenMode;PenStyle:TPenStyle;PenWidth:integer);

begin
  fPaintBox.SetPen(PenColor,PenMode,PenStyle,PenWidth);
end;
(******************************************************************************)
Procedure TQSWorldGraph.ResetPen;

begin
  fPaintBox.ResetPen;
end;

(******************************************************************************)

procedure TQSWorldGraph.Output(newVertex : TWorldPoint;
         var outLength : integer; var outVertexArray : TWorldPointArray);

begin
  inc(outLength);
  outVertexArray[outLength] := newVertex;
end;

(******************************************************************************)
function TQSWorldGraph.Inside( testVertex: TWorldPoint; clipBoundary : TEdge):boolean;

begin
  result := false;
  case clipBoundary.edge of
    ekLeft    : if testVertex.x >= clipBoundary.xMin then result := true;
    ekBottom  : if testVertex.y >= clipBoundary.yMin then result := true;
    ekRight   : if testVertex.x <= clipBoundary.xMax then result := true;
    ekTop   : if testVertex.y <= clipBoundary.yMax then result := true;
  end;
end;

(******************************************************************************)

function TQSWorldGraph.Intersect(first,second : TWorldPoint;clipBoundary:TEdge):TWorldPoint;

var
  A,B : double;

begin
  if first.x = second.x then second.x := second.x+1e-10;
  if first.y = second.y then second.y := second.y+1e-10;
  A := (second.y - first.y)/(second.x-first.x);
  B := (first.y*second.x-first.x*second.y)/(second.x-first.x);
  case clipBoundary.edge of
    ekleft : begin
               result.x := clipBoundary.xMin;
               result.y := A*result.x+B;
             end;
     ekright : begin
                 result.x := clipBoundary.xMax;
                 result.y := A*result.x + B;
               end;
     ekbottom : begin
                  result.y := clipBoundary.yMin;
                  result.x := (result.y-B)/A;
                end;
     ektop    : begin
                  result.y := clipBoundary.yMax;
                  result.x := (result.y-B)/A;
                end;
  end;
end;

(******************************************************************************)

procedure TQSWorldGraph.SutherlandHodgmanPolygonClip(
     inVertexArray : TWorldPointArray;
     var outVertexArray : TWorldPointArray;
     inLength : integer;
     var outLength : integer;
     clipBoundary : TEdge);

var
  s,p    : TWorldPoint;
  j      : integer;
begin
  if inLength = 0 then
    exit;
  outLength := 0;
  s := inVertexArray[inLength];
  (*Start with the last vertex in inVertexArray*)
  for j := 1 to inLength do
    begin
      p := inVertexArray[j];
      if Inside(p,clipBoundary) then
        if Inside(s,clipBoundary) then
          Output(p,outLength,outVertexArray)
        else
          begin
            Output(Intersect(s,p,clipBoundary),outLength,outVertexArray);
            Output(p,outLength,outVertexArray);
          end
      else
        if Inside(s,clipBoundary) then
          Output(Intersect(s,p,clipBoundary),outLength,outVertexArray);
      s := p;
    end;
end;

(******************************************************************************)

procedure TQSWorldGraph.ClipPolygon(inVertexArray : TWorldPointArray; inLength: integer;
                             var outVertexArray : TWorldPointArray; var outLength : integer);
var
  edgeOutVertexArray :TWorldPointArray;
  i                  : integer;
  clipBoundary       : TEdge;
  edgeOutLength      : integer;

begin
  (*clip left boundary*)
  clipBoundary.edge := ekLeft;
  clipBoundary.xMin := xMin;
  clipBoundary.xMax := xMax;
  clipBoundary.yMin := yMin;
  clipBoundary.yMax := yMax;
  SutherlandHodgmanPolygonClip(inVertexArray,
                               EdgeOutVertexArray,
                               InLength,
                               EdgeOutLength,ClipBoundary);
  OutLength := EdgeOutLength;
  for i := 1 to OutLength do
    OutVertexArray[i] := edgeOutVertexArray[i];

(*clip bottom boundary*)

  clipBoundary.edge := ekBottom;
  SutherlandHodgmanPolygonClip(OutVertexArray,
                               EdgeOutVertexArray,
                               OutLength,
                               EdgeOutLength,ClipBoundary);

  OutLength := EdgeOutLength;
  for i := 1 to outLength do
    OutVertexArray[i] := edgeOutVertexArray[i];



 (*clip right boundary*)
  clipBoundary.edge := ekRight;
  SutherlandHodgmanPolygonClip(OutVertexArray,
                               EdgeOutVertexArray,
                               OutLength,
                               EdgeOutLength,ClipBoundary);

  OutLength := EdgeOutLength;
  for i := 1 to outLength do
    OutVertexArray[i] := edgeOutVertexArray[i];

 (*clip top boundary*)
  clipBoundary.edge := ekTop;
  SutherlandHodgmanPolygonClip(OutVertexArray,
                               EdgeOutVertexArray,
                               OutLength,
                               EdgeOutLength,ClipBoundary);

  OutLength := EdgeOutLength;
  for i := 1 to outLength do
    OutVertexArray[i] := edgeOutVertexArray[i];

end;


(******************************************************************************)
(******************************************************************************)

end.
