/*
 * Service Launcher
 * (part of JObjects Java System Toolkit)
 *
 * Copyright (c) 1998 JObjects, All Rights Reserved.
 *
 * This software is the confidential and proprietary information 
 * of JObjects ("Confidential Information").  You shall not 
 * disclose such Confidential Information and shall use it only in 
 * accordance with the terms of the license agreement you entered 
 * into with JObjects.
 *
 * JOBJECTS MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE
 * SUITABILITY OF THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING
 * BUT NOT LIMITED TO THE IMPLIED WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE, OR NON-INFRINGEMENT.
 * JOBJECTS SHALL NOT BE LIABLE FOR ANY DAMAGES SUFFERED BY
 * LICENSEE AS A RESULT OF USING, MODIFYING OR DISTRIBUTING THIS
 * SOFTWARE OR ITS DERIVATIVES.
 *
 * JObjects
 * Web: http://www.jobjects.com
 * Email: contact@jobjects.com
 */
    
// launcher-win32.cpp

#include <windows.h>
#include <stdio.h>
#include <iostream.h>
#include <stdlib.h>

#include <fstream.h>

#include <jni.h>

// simple error logging macro
#define LOGMSG( msg ) {ofstream fout; fout.open("jstlog.txt",ios::ate); fout << msg << endl; fout.close();}


//#define DEBUG_BEEPING


//*** Win95 specific

VOID ServiceMain_95(DWORD argc, LPTSTR *argv);
BOOL ConsoleHandlerRoutine_95( DWORD dwCtrlType );

//*** end of Win95 specific


// Settings...


// Class for a connection with the Java System Toolkit library

#define MAIN_JAVA_CLASS "com/jobjects/jst/win32/ControlMessagePeer"

// JST's location in a registry

#define JST_REG_ROOT "Software\\JObjects\\JavaSystemToolkit"

// How many JVMs can we accept? Anyway, the first one is used.

#define MAX_JVMs 3

// Beeping service (for testing)
#define DEFAULT_BEEP_DELAY 2000


// Global variables


// The name of the service

char *SERVICE_NAME = "";

// Full service class name

char *serviceClassName = "";

// Java CLASSPATH

char *serviceClassPath = "";

// Service working directory

char *serviceWrkDir = "";

// Event used to hold ServiceMain from completing

HANDLE terminateEvent = NULL;

// Handle used to communicate status info with
// the SCM. Created by RegisterServiceCtrlHandler

SERVICE_STATUS_HANDLE serviceStatusHandle;

// Flags holding current state of service

BOOL pauseService = FALSE;

BOOL runningService = FALSE;

BOOL sentShutdownSignal = FALSE;

// Thread for the actual work

HANDLE threadHandle = 0;

// The beep interval in ms.
int beepDelay = DEFAULT_BEEP_DELAY;


/** Reads service data from the registry. */
int readServiceSettings()
{
  HKEY key;
  DWORD res;

  if( RegOpenKeyEx( HKEY_LOCAL_MACHINE, JST_REG_ROOT, 0, KEY_READ, &key ) == 0)
  {
    HKEY skey;

    if( RegOpenKeyEx( key, SERVICE_NAME, 0, KEY_READ, &skey ) == 0 )
	{

	  DWORD dwSize, dwType;

	  // get service class name
      if( RegQueryValueEx( skey, "ServiceClassName", NULL, &dwType, NULL, &dwSize) == 0 )
	  {
        serviceClassName = (char *) malloc(dwSize);
        RegQueryValueEx( skey, "ServiceClassName", NULL, &dwType, 
		  (unsigned char *) serviceClassName, &dwSize);
      }
	  else  {
       LOGMSG( "Can't get service class name. Err " << GetLastError() )
	  }

	  // get classpath
      if( RegQueryValueEx( skey, "ClassPath", NULL, &dwType, NULL, &dwSize) == 0 )
	  {
        serviceClassPath = (char *) malloc(dwSize);
        RegQueryValueEx( skey, "ClassPath", NULL, &dwType, 
		  (unsigned char *) serviceClassPath, &dwSize);
      }
	  else  {
       LOGMSG( "Can't get classpath. Err " << GetLastError() )
	  }

	  // get working dir.
      if( RegQueryValueEx( skey, "WorkingDir", NULL, &dwType, NULL, &dwSize) == 0 )
	  {
        serviceWrkDir = (char *) malloc(dwSize);
        RegQueryValueEx( skey, "WorkingDir", NULL, &dwType, 
		  (unsigned char *) serviceWrkDir, &dwSize);
      }
	  else  {
       // LOGMSG( "Can't get service class name. Err " << GetLastError() )
	  }
	
      RegCloseKey(skey);
	}
	else  {
      LOGMSG( "Can't open registry key for" << SERVICE_NAME )
      RegCloseKey(key);
	  return -1;
	}
	  
    RegCloseKey(key);
  }
  else  {
    LOGMSG( "Can't open registry key for" << JST_REG_ROOT )
    return -1;
  }

  return 0;
}


int startJavaService()
{
  JavaVM *vmBuf[MAX_JVMs];
  jsize vmSize;
  jint res;
  JNIEnv *env;
  jclass cls;
  jmethodID mid;
  jstring jstr;
  jobjectArray args;


  // get created / active JVMs
  res = JNI_GetCreatedJavaVMs( vmBuf, MAX_JVMs, &vmSize );
  if( res )  {
    LOGMSG( "Can't get created JVMs." )
  	return 1;
  }

  // attach to an existing JVM
  res = vmBuf[0]->AttachCurrentThread( &env, NULL );
  if( res )  {
    LOGMSG( "Can't atach current thread to JVM." )
  	return 1;
  }
  
	
  cls = env->FindClass( MAIN_JAVA_CLASS );
  if (cls == 0) {
      LOGMSG( "Can't find class " << MAIN_JAVA_CLASS )
      LOGMSG( "ClassPath = " << serviceClassPath )
      return 1;
  }
 
  mid = env->GetStaticMethodID( cls, "onStart", "(Ljava/lang/String;)V");
  if (mid == 0) {
      LOGMSG( "Can't find method " << MAIN_JAVA_CLASS << ".onStart" );
      return 1;
  }

  env->CallStaticVoidMethod( cls, mid, env->NewStringUTF( serviceClassName ) );


  res = vmBuf[0]->DetachCurrentThread();
  if( res )  {
    LOGMSG( "Can't detach current thread." );
  	return 1;
  }
	
  return 0;
}


int callStaticHandlerMethod( char *method, char *sign )
{
  JNIEnv *env;
  JavaVM *vmBuf[MAX_JVMs];
  jsize vmSize;
  jint res;

  res = JNI_GetCreatedJavaVMs( vmBuf, MAX_JVMs, &vmSize );
  if( res )  {
    LOGMSG( "Can't get created JVMs." );
  	return 1;
  }

  res = vmBuf[0]->AttachCurrentThread( &env, NULL );
  if( res )  {
    LOGMSG( "Can't atach current thread to JVM." );
  	return 1;
  }
  
  jclass cls = env->FindClass( MAIN_JAVA_CLASS );
  if (cls == 0) {
    LOGMSG( "Can't find class " << MAIN_JAVA_CLASS )
    LOGMSG( "ClassPath = " << serviceClassPath )
    return 1;
  }

  jmethodID mid = env->GetStaticMethodID( cls, method, sign );
  if (mid == 0) {
    LOGMSG( "Can't find " << method << " " << sign << " method." );
    return 1;
  }

  
  env->CallStaticVoidMethod( cls, mid );


  res = vmBuf[0]->DetachCurrentThread();
  if( res )  {
    LOGMSG( "Can't detach current thread." )
  	return 1;
  }
	
  return 0;
}


void ErrorHandler(char *s, DWORD err)
{
  cout <<  s <<  endl;
  cout <<  "Error number: " <<  err <<  endl;

  LOGMSG( "Error number: " <<  err )
  
  ExitProcess(err);
}

    

// This function consolidates the activities of 
// updating the service status with
// SetServiceStatus

BOOL SendStatusToSCM (DWORD dwCurrentState,
                      DWORD dwWin32ExitCode, 
                      DWORD dwServiceSpecificExitCode,
                      DWORD dwCheckPoint,
                      DWORD dwWaitHint )
{
  BOOL success;
  SERVICE_STATUS serviceStatus;

  // Fill in all of the SERVICE_STATUS fields

  serviceStatus.dwServiceType = SERVICE_WIN32_OWN_PROCESS;

  serviceStatus.dwCurrentState = dwCurrentState;

  // If in the process of doing something, then accept
  // no control events, else accept anything

  if (dwCurrentState == SERVICE_START_PENDING)
    serviceStatus.dwControlsAccepted = 0;
  else
    serviceStatus.dwControlsAccepted = 
                           SERVICE_ACCEPT_STOP |
                           SERVICE_ACCEPT_PAUSE_CONTINUE |
                           SERVICE_ACCEPT_SHUTDOWN;

  // if a specific exit code is defined, set up
  // the win32 exit code properly

  if (dwServiceSpecificExitCode == 0)
    serviceStatus.dwWin32ExitCode = dwWin32ExitCode;
  else
    serviceStatus.dwWin32ExitCode = ERROR_SERVICE_SPECIFIC_ERROR;

  serviceStatus.dwServiceSpecificExitCode = dwServiceSpecificExitCode;

  serviceStatus.dwCheckPoint = dwCheckPoint;

  serviceStatus.dwWaitHint = dwWaitHint;

  // Pass the status record to the SCM

  success = SetServiceStatus (serviceStatusHandle, &serviceStatus);

  return success;
}

    

// Dispatches events received from the SCM

VOID Handler (DWORD controlCode) 
{
  DWORD currentState = 0;
  BOOL success;

  switch(controlCode)
  {

    // There is no START option because
    // ServiceMain gets called on a start


    // Stop the service
	
    case SERVICE_CONTROL_STOP:

         // Tell the SCM what's happening

         success = SendStatusToSCM(SERVICE_STOP_PENDING, NO_ERROR, 0, 1, 5000);

         // send Stop message to JST
						   
         callStaticHandlerMethod( "onStop", "()V" );
						   
         runningService=FALSE;

         // Set the event that is holding ServiceMain
         // so that ServiceMain can return

         SetEvent(terminateEvent);

         return;

    

    // Pause the service

    case SERVICE_CONTROL_PAUSE:

         if (runningService && !pauseService)
         {
           // Tell the SCM what's happening

           success = SendStatusToSCM( SERVICE_PAUSE_PENDING, NO_ERROR, 0, 1, 2000);

           // send Pause message to JST
  						   
           callStaticHandlerMethod( "onPause", "()V" );

           // suspend working thread
		   
           pauseService = TRUE;

           SuspendThread(threadHandle);

           currentState = SERVICE_PAUSED;

         }

         break;


    // Resume from a pause

    case SERVICE_CONTROL_CONTINUE:

         if (runningService && pauseService)
         {
           // Tell the SCM what's happening

           success = SendStatusToSCM( SERVICE_CONTINUE_PENDING, NO_ERROR, 0, 1, 1000);

           pauseService=FALSE;

           ResumeThread(threadHandle);

           // send Continue message to JST
  						   
           callStaticHandlerMethod( "onContinue", "()V" );

           currentState = SERVICE_RUNNING;

         }
         break;

    
    // Update current status

    case SERVICE_CONTROL_INTERROGATE:

         // it will fall to bottom and send status

         break;


    // Do nothing in a shutdown. Could do cleanup
    // here but it must be very quick.

    case SERVICE_CONTROL_SHUTDOWN:

         if( !sentShutdownSignal )
		 {
           // send Shutdown message to JST
  						   
           callStaticHandlerMethod( "onShutdown", "()V" );

           sentShutdownSignal = TRUE;
		 }

         return;


    default:
         break;
  }

  SendStatusToSCM(currentState, NO_ERROR, 0, 0, 0);

}


BOOL ConsoleHandlerRoutine_NT( DWORD dwCtrlType )
{
  if( dwCtrlType == CTRL_SHUTDOWN_EVENT && !sentShutdownSignal )
  {
    // send Shutdown message to JST
  						   
    callStaticHandlerMethod( "onShutdown", "()V" );
  
    sentShutdownSignal = TRUE;
  }
  
  return FALSE;
}


// Handle an error from ServiceMain by cleaning up
// and telling SCM that the service didn't start.

VOID terminate(DWORD error)
{
  // if terminateEvent has been created, close it.

  if (terminateEvent) CloseHandle(terminateEvent);

  // Send a message to the scm to tell about stopage

  if (serviceStatusHandle)
    SendStatusToSCM(SERVICE_STOPPED, error, 0, 0, 0);

  // If the thread has started, kill it off

  if (threadHandle)
    CloseHandle(threadHandle);

  // Do not need to close serviceStatusHandle
}

    
DWORD ServiceThread(LPDWORD param)
{
  if( serviceClassName == NULL || !strlen( serviceClassName ) )
    return 0;

  startJavaService();

#ifdef DEBUG_BEEPING

  while (1)
  {
    Beep(200,200);

    Sleep(beepDelay);
  }

#endif

  return 0;
}


// Initializes the service by starting its thread

BOOL InitService()
{
  DWORD id;

  // Start the service's thread

  threadHandle = CreateThread( 0, 0, 
                 (LPTHREAD_START_ROUTINE) ServiceThread,
                 0, 0, &id);

  if (threadHandle==0)
    return FALSE;

  else
    runningService = TRUE;

  return TRUE;
}

    
// ServiceMain is called when the SCM wants to
// start the service. When it returns, the service
// has stopped. It therefore waits on an event
// just before the end of the function, and
// that event gets set when it is time to stop. 
// It also returns on any error because the
// service cannot start if there is an eror.

VOID ServiceMain(DWORD argc, LPTSTR *argv) 
{
  BOOL success;

  // get service name
  
  SERVICE_NAME = argv[0];

  // immediately call Registration function
  
  serviceStatusHandle = RegisterServiceCtrlHandler(
                      SERVICE_NAME, (LPHANDLER_FUNCTION) Handler );
  if (!serviceStatusHandle)  {
    terminate(GetLastError());
	return;
  }

  // Notify SCM of progress
  
  success = SendStatusToSCM(SERVICE_START_PENDING, NO_ERROR, 0, 1, 5000);
  if (!success)  {
    terminate(GetLastError());
	return;
  }

  // create the termination event
  
  terminateEvent = CreateEvent (0, TRUE, FALSE, 0);
  if (!terminateEvent) {
    terminate(GetLastError());
	return;
  }

  // get service parameters from the system registry

  if( readServiceSettings() )  {
    LOGMSG( "Can't read service settings from the registry" )
    terminate(GetLastError());
	return;
  }

  // Notify SCM of progress

  success = SendStatusToSCM(SERVICE_START_PENDING, NO_ERROR, 0, 2, 5000);
  if (!success)  {
    terminate(GetLastError());
	return;
  }

  // set current directory (if required by service)
  
  if( serviceWrkDir != NULL && strlen( serviceWrkDir ) )
    SetCurrentDirectory( serviceWrkDir );

  // register console handler to catch shutdown event on time
  // due to shutdown queue problem on NT
  
  if( !SetConsoleCtrlHandler( 
      (PHANDLER_ROUTINE) ConsoleHandlerRoutine_NT, TRUE ) )
  {
    LOGMSG( "Couldn't set ConsoleCtrlHandler." )
  }

  // Notify SCM of progress

  success = SendStatusToSCM(SERVICE_START_PENDING, NO_ERROR, 0, 3, 5000);
  if (!success)  {
    terminate(GetLastError());
	return;
  }

  // Create Java VM
  
  JNIEnv *env;
  JavaVM *jvm;
  JDK1_1InitArgs vm_args;
  jint res;

  /* IMPORTANT: specify vm_args version # if you use JDK1.1.2 and beyond */
  vm_args.version = 0x00010001;

  if( JNI_GetDefaultJavaVMInitArgs(&vm_args) < 0 )  {
    LOGMSG( "Requested JVM not supported." );
    terminate(GetLastError());
	return;
  }

  if( serviceClassPath != NULL && strlen(serviceClassPath) > 0)
    vm_args.classpath = serviceClassPath;
	
  res = JNI_CreateJavaVM( &jvm, &env, &vm_args );
  if (res < 0) {
    LOGMSG( "Can't create JVM." );
    terminate(GetLastError());
	return;
  }

  // Notify SCM of progress

  success = SendStatusToSCM(SERVICE_START_PENDING, NO_ERROR, 0, 4, 1000);
  if (!success)  {
    terminate(GetLastError());
	return;
  }

  // Start the service itself

  success = InitService();
  if (!success) {
    terminate(GetLastError());
	return;
  }

  // The service is now running. 

  // Notify SCM of progress

  success = SendStatusToSCM(SERVICE_RUNNING, NO_ERROR, 0, 0, 0);
  if (!success)  {
    terminate(GetLastError());
	return;
  }

  // Wait for stop signal, and then terminate

  WaitForSingleObject (terminateEvent, INFINITE);

  // Shutdown Java VM
  
  jvm->DestroyJavaVM();
  
  terminate(0);
}

    

VOID main(int argc, char *argv[])
{
  // check OS version
  
  OSVERSIONINFO osver;
  osver.dwOSVersionInfoSize = sizeof( OSVERSIONINFO );

  if( GetVersionEx( &osver ) )  // start depends on OS
  {
    if( osver.dwPlatformId == VER_PLATFORM_WIN32_NT )  {  // Windows NT

      // register service main method
  
      SERVICE_TABLE_ENTRY serviceTable[] = 
      { 
        { SERVICE_NAME, (LPSERVICE_MAIN_FUNCTION) ServiceMain},
        { NULL, NULL }
      };

      BOOL success;

      // Register with the SCM

      success = StartServiceCtrlDispatcher(serviceTable);
      if (!success)
        ErrorHandler( "Error in StartServiceCtrlDispatcher", GetLastError());

	}
	if( osver.dwPlatformId == VER_PLATFORM_WIN32_WINDOWS )  { // Windows 95

      printf("Windows 95 services are not supported yet.");
	  return;
      
	  /*
	  LOGMSG( "Win 95 section" );
	  
	  if( argc != 2 )  {
        LOGMSG( "Bad number of parameters" );
	    printf( "Usage: srvcl service_name" );
	    return;
	  }

      printf("Win 95 2");
      LOGMSG( "call ServiceMain_95" );

      // call service main
      ServiceMain_95( argc, argv );
	  
      printf("Win 95 3");
      LOGMSG( "ServiceMain_95 finished" );
	  */
	}
  }
}


//************* Windows 95 specific
/* Not functional yet. 
 */

VOID ServiceMain_95(DWORD argc, LPTSTR *argv) 
{
  BOOL success;

  // get service name
  
  SERVICE_NAME = argv[1];

  // create the termination event
  
  terminateEvent = CreateEvent (0, TRUE, FALSE, 0);
  if (!terminateEvent) {
    //terminate(GetLastError());
	return;
  }

  // get service parameters from the system registry

  if( readServiceSettings() )  {
    LOGMSG( "Can't read service settings from the registry" )
    // terminate(GetLastError());
	return;
  }

  LOGMSG( "WrkDir " << serviceWrkDir )
  LOGMSG( "ClassPath " << serviceClassPath )
  
  // set current directory (if required by service)
  
  if( serviceWrkDir != NULL && strlen( serviceWrkDir ) )
    SetCurrentDirectory( serviceWrkDir );

  // register console handler to catch shutdown event on time
  // due to shutdown queue problem on NT
  
  if( !SetConsoleCtrlHandler( 
      (PHANDLER_ROUTINE) ConsoleHandlerRoutine_95, TRUE ) )
  {
    LOGMSG( "Couldn't set ConsoleCtrlHandler (Win95)." )
  }

  // Create Java VM
  
  JNIEnv *env;
  JavaVM *jvm;
  JDK1_1InitArgs vm_args;
  jint res;

  /* IMPORTANT: specify vm_args version # if you use JDK1.1.2 and beyond */
  vm_args.version = 0x00010001;

  if( JNI_GetDefaultJavaVMInitArgs(&vm_args) < 0 ) {
    LOGMSG( "Requested JVM not supported." );
    //terminate(GetLastError());
	return;
  }

  LOGMSG( "Default cp " << vm_args.classpath );
  
  if( serviceClassPath != NULL && strlen(serviceClassPath) > 0)
    vm_args.classpath = serviceClassPath;
	
  res = JNI_CreateJavaVM( &jvm, &env, &vm_args );
  if (res < 0) {
    LOGMSG( "Can't create JVM." );
    //terminate(GetLastError());
	return;
  }

  LOGMSG( "JVM created" );
  
  // Start the service itself

  success = InitService();
  if (!success) {
    //terminate(GetLastError());
	return;
  }

  // Wait for stop signal, and then terminate

  WaitForSingleObject (terminateEvent, INFINITE);

  // Shutdown Java VM
  
  jvm->DestroyJavaVM();
  
  //terminate(0);
}


BOOL ConsoleHandlerRoutine_95( DWORD dwCtrlType )
{
  if( dwCtrlType == CTRL_SHUTDOWN_EVENT && !sentShutdownSignal )
  {
    // send Shutdown message to JST
  						   
    callStaticHandlerMethod( "onShutdown", "()V" );
  
    sentShutdownSignal = TRUE;

    // Set the event that is holding ServiceMain
    // so that ServiceMain can return

    SetEvent(terminateEvent);

  }
  
  return FALSE;
}

