/*
 * Copyright (c) 1998 JObjects, All Rights Reserved.
 *
 * This software is the confidential and proprietary information 
 * of JObjects ("Confidential Information").  You shall not 
 * disclose such Confidential Information and shall use it only in 
 * accordance with the terms of the license agreement you entered 
 * into with JObjects.
 *
 * JOBJECTS MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE
 * SUITABILITY OF THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING
 * BUT NOT LIMITED TO THE IMPLIED WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE, OR NON-INFRINGEMENT.
 * JOBJECTS SHALL NOT BE LIABLE FOR ANY DAMAGES SUFFERED BY
 * LICENSEE AS A RESULT OF USING, MODIFYING OR DISTRIBUTING THIS
 * SOFTWARE OR ITS DERIVATIVES.
 *
 * JObjects
 * Web: http://www.jobjects.com
 * Email: contact@jobjects.com
 */

package com.jobjects.jst.win32;

import com.jobjects.jst.*;
import java.util.*;
import java.io.File;

/**
   Windows NT implementation of ServiceControlManager.
   Some methods can be mapped for Windows 95.
*/
public class SCM implements ServiceControlManager
{
  private Vector listeners = new Vector();

  private boolean isNT;

  public SCM( boolean nt )
  {
    isNT = nt;
  }

  /** load native library */
  static {

    String fullpath = lookupForFile( "jst.dll", 1 );
    if( fullpath != null )  {
      //System.out.println( "full path found" );
      System.load( fullpath );
    }
    else  {
      //System.out.println( "full path not found" );
      System.loadLibrary( "jst" ); // Can't rely on default path
    }
  }

  /**
     Adds service control listener.
  */
  public synchronized void
         addServiceControlListener( ServiceControlListener lis )
  {
    listeners.insertElementAt( lis, 0 );
  }

  /**
     Removes service control listener.
  */
  public synchronized void
         removeServiceControlListener( ServiceControlListener lis )
  {
    listeners.removeElement( lis );
  }

  /**
     Sends "stop" message to listeneres.
  */
  protected synchronized void notifyOnStop()
  {
    for( Enumeration e = listeners.elements(); e.hasMoreElements(); )  {
      ((ServiceControlListener) e.nextElement()).stopService();
    }
  }

  /**
     Sends "pause" message to listeneres.
  */
  protected synchronized void notifyOnPause()
  {
    for( Enumeration e = listeners.elements(); e.hasMoreElements(); )  {
      ((ServiceControlListener) e.nextElement()).pauseService();
    }
  }

  /**
     Sends "continue" message to listeneres.
  */
  protected synchronized void notifyOnContinue()
  {
    for( Enumeration e = listeners.elements(); e.hasMoreElements(); )  {
      ((ServiceControlListener) e.nextElement()).continueService();
    }
  }

  /**
     Sends "shutdown" message to listeneres.
  */
  protected synchronized void notifyOnShutdown()
  {
    // Shutdown should be as fast as possible, so no threading here
    for( Enumeration e = listeners.elements(); e.hasMoreElements(); )  {
      ((ServiceControlListener) e.nextElement()).shutdownService();
    }
  }

  public void createService( ServiceSettings srvset )
         throws ServiceControlException
  {
    // check class path
    if( srvset.getClasspath() == null )
      srvset.setClasspath( System.getProperty( "java.class.path" ) );

    // create service
    if( isNT )  {
      if( ! _createService( srvset ) )
       throw new ServiceControlException( _getLastErrorDescription() );
    }
    else  {
      if( ! _createService95( srvset ) )
       throw new ServiceControlException( _getLastErrorDescription() );
    }
  }

  public void deleteService( String name )
         throws ServiceControlException
  {
    if( name == null || name.length() == 0 )
      throw new IllegalAccessError();

    if( isNT )  {
      if( ! _deleteService( name ) )
        throw new ServiceControlException( _getLastErrorDescription() );
    }
    else  {
      if( ! _deleteService95( name ) )
        throw new ServiceControlException( _getLastErrorDescription() );
    }
  }

  /**
     @param name file name
     @param where where to look
            (0x01 -> classpath, 0x02 -> path)
     @return full file path or null if not found
  */
  protected static String lookupForFile( String name, int where )
  {
    // get path...

    String path = "";
    if( (where & 0x01) != 0 )
      path += System.getProperty( "java.class.path" ); // get java CLASSPATH
    if( (where & 0x02) != 0 )  {
      path += File.pathSeparator + "." +
              File.pathSeparator + _getPathVariable(); // get system PATH
    }

    StringTokenizer tok = new StringTokenizer( path, File.pathSeparator );
    while( tok.hasMoreTokens() )  {

      String t = tok.nextToken().toLowerCase();  // Windows is case insensitive
      if( t.equals( "." ) )
        t = System.getProperty( "user.dir" ); // get current directory

      File f = new File( t );
      if( f.isFile() || t.endsWith( ".zip" ) || t.endsWith( ".jar" ) ) {
        t = f.getParent();
        f = new File( t );
      }

      // check in directory
      File lf = new File( f, name );
      if( lf.exists() && lf.isFile() )
        return lf.getAbsolutePath();

      // check in ..\lib
      lf = new File( new File( f, "..\\lib" ), name );
      if( lf.exists() && lf.isFile() )
        return lf.getAbsolutePath();

      // check in ..\bin
      lf = new File( new File( f, "..\\bin" ), name );
      if( lf.exists() && lf.isFile() )
        return lf.getAbsolutePath();
    }

    return null;
  }

  private native boolean _createService( ServiceSettings srvset );

  private native boolean _deleteService( String name );

  private native String _getLastErrorDescription();

  private native static String _getPathVariable();

  private native boolean _createService95( ServiceSettings srvset );

  private native boolean _deleteService95( String name );

}

