{ based on rfc1034 and rfc1035}


unit msmxcls;

interface

uses
  agWsock, Classes;

type
(*
    +--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+
    |QR|   Opcode  |AA|TC|RD|RA|   Z    |   RCODE   |
    +--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+
*)
  TQuerySpec = class
  private
    function GetQR : boolean;
    procedure SetQR(Value : boolean);
    function GetOpCode : byte;
    procedure SetOpCode(Value : byte);
    function GetAA : boolean;
    function GetTC : boolean;
    function GetRD : boolean;
    procedure SetRD(Value : boolean);
    function GetRA : boolean;
    function GetRCode : byte;
  protected
  public
    SpecWord : word;
    property QR : boolean read GetQR write SetQR;
    { false - query, true - answer}
    property OpCode : byte read GetOpCode write SetOpCode;
    { 0 - Standard query (QUERY)
      1 - inverse query (IQUERY)
      2 - a server status request (STATUS)
      3-15 - reserved}
    property AA : boolean read GetAA;
    {  Authoritative Answer - this bit is valid in responses,
       and specifies that the responding name server is an
       authority for the domain name in question section.}
    property TC : boolean read GetTC;
    { TrunCation - specifies that this message was truncated
      due to length greater than that permitted on the
      transmission channel.}
    property RD : boolean read GetRD write SetRD;
    { Recursion Desired - this bit may be set in a query and
      is copied into the response.  If RD is set, it directs
      the name server to pursue the query recursively.
      Recursive query support is optional.}
    property RA : boolean read GetRA;
    { Recursion Available - this be is set or cleared in a
      response, and denotes whether recursive query support is
      available in the name server.}
    property RCode : byte read GetRCode;
    { Response code - this 4 bit field is set as part of
      responses.  The values have the following
      interpretation:
      0              No error condition
      1              Format error - The name server was
                     unable to interpret the query.
      2              Server failure - The name server was
                     unable to process this query due to a
                     problem with the name server.
      3              Name Error - Meaningful only for
                     responses from an authoritative name
                     server, this code signifies that the
                     domain name referenced in the query does
                     not exist.
      4              Not Implemented - The name server does
                     not support the requested kind of query.
      5               Refused - The name server refuses to
                     perform the specified operation for
                     policy reasons.
      6-15           Reserved for future use.}
  end;

{ More records and classes}
  TQueryHeader = record
    ID : word;
    Spec : word;
    QDCount : word;
    AnCount : word;
    NsCount : word;
    ArCount : word;
  end;

  TQuestRec = record
    Domain : string;
    QType : word;  { $0F for MX}
    QClass : word; { $01 for IN}
  end;

  TAnswerRec = record
    Name : string;
    QType : word;
    QClass : word;
    TTL : u_long;
    RDLen : word;
  end;

  TMXRec = class
    Preference : word;
    Exchanger : string;
  end;

  TMXList = class(TPersistent)
  private
    FList : TList;
    function Get(Index : Integer) : TMXRec;
    procedure Put(Index : Integer; Value : TMXRec);
    function GetCount : Integer;
  public
    constructor Create;
    destructor Destroy; override;
    procedure Clear;
    function Add(Value : TMXRec) : Integer;
    procedure Delete(Index : Integer);
    procedure Sort;
    property Items[Index : Integer] : TMXRec read Get write Put; default;
    property Count : Integer read GetCount;
  end;

implementation

{TQuerySpec}
function TQuerySpec.GetQR : boolean;
var
  boo : boolean;
begin
  boo:=(SpecWord and $8000) = $8000;
  Result:=boo;
end;

procedure TQuerySpec.SetQR(Value : boolean);
begin
  if Value then
    SpecWord:=SpecWord or $8000
  else
    SpecWord:=SpecWord and $7fff;
end;

function TQuerySpec.GetOpCode : byte;
begin
  Result:=Lo(SpecWord shr 11);
end;

procedure TQuerySpec.SetOpCode(Value : byte);
var
  TempWord : word;
begin
  TempWord:=Value;
  TempWord:=TempWord shl 11;
  SpecWord:=SpecWord or TempWord;
end;

function TQuerySpec.GetAA : boolean;
begin
  Result:=(SpecWord and $0400) = $0400;
end;

function TQuerySpec.GetTC : boolean;
begin
  Result:=(SpecWord and $0200) = $0200;
end;

function TQuerySpec.GetRD : boolean;
begin
  Result:=(SpecWord and $0100) = $0100;
end;

procedure TQuerySpec.SetRD(Value : boolean);
begin
  if Value then
    SpecWord:=SpecWord or $0100
  else
    SpecWord:=SpecWord and $FEFF;
end;

function TQuerySpec.GetRA : boolean;
begin
  Result:=(SpecWord and $0080) = $0080;
end;

function TQuerySpec.GetRCode : byte;
begin
  Result:=SpecWord and $000F;
end;

{TMXList}
constructor TMXList.Create;
begin
  inherited Create;
  FList:=TList.Create;
end;

destructor TMXList.Destroy;
var
  i : Integer;
begin
  for i:=Count-1 DownTo 0 do
    Items[i].Free;
  FList.Free;
  inherited Destroy;
end;

function TMXList.Get(Index : Integer) : TMXRec;
begin
  Result:=FList[Index];
end;

procedure TMXList.Put(Index : Integer; Value : TMXRec);
begin
  FList[Index]:=Value;
end;

function TMXList.GetCount : Integer;
begin
  Result:=FList.Count;
end;

procedure TMXList.Clear;
var
  i : Integer;
begin
  for i:=Count-1 DownTo 0 do
    Items[i].Free;
  FList.Clear;
end;

function TMXList.Add(Value : TMXRec) : Integer;
begin
  Result:=FList.Add(Value);
end;

procedure TMXList.Delete(Index : Integer);
begin
  Items[Index].Free;
  FList.Delete(Index);
end;

procedure TMXList.Sort;
var
  Min : word;
  i,j : Integer;
begin
  for i:=0 to Count-1 do
  begin
    Min:=Items[i].Preference;
    for j:=i+1 to Count-1 do
    begin
      if Items[j].Preference<Min then
      begin
        Min:=Items[j].Preference;
        FList.Exchange(i,j);
      end;
    end;
  end;
end;

end.


