unit msmx;

interface

uses
{$IFDEF WIN32}
  Windows,
{$ELSE}
  WinTypes, WinProcs,
{$ENDIF}
  Messages, SysUtils, Classes, Forms, Dialogs, agWSock, ExtCtrls,
  msMxCls, agTypes, msCls, agSocket, agConst;

const
  BufSize = 1024;

type
  TmsMXResolver = class(TComponent)
  private
    FDomain : string;
    FDNSAddress : string;
    FQueryHeader : TQueryHeader;
    FID : word;
    FLog : TStringList;
    FLogFileName : string;
    FPort : word;
    FWindow : THandle;
    FReadyToRead : boolean;
    FCanceled : boolean;
    FMXList : TMXList;
    FOnSendingQuery : TNotifyEvent;
    FOnWaitingForResponse : TNotifyEvent;
    FOnClosingConnection : TNotifyEvent;
    function GetTimeOut : Integer;
    procedure SetTimeOut(Value : Integer);
  protected
    Stream : TMemoryStream;
    TheSocket : TSocket;
    RemoteAddress : TSockAddr;
    TC : TTimeCounter;
    rc : Integer;
    procedure Error(Ident : word);
    procedure DNSError(Ident : word);
    procedure WndProc(var Msg : TMessage);
    function ntohDomain : string;  {recursive!!!}
    procedure ConstructQuery;
    procedure ParseHeader;
    procedure ParseQuestion;
    procedure ParseAnswer;
    procedure AddToLog(const s : string);
    procedure ReInit;
    procedure OpenConnection;
    procedure CloseConnection;
    procedure SendQuery;
    procedure RecvData;
    procedure GetResults;
  public
    constructor Create(AOwner : TComponent); override;
    destructor Destroy; override;
    procedure Cancel;
    procedure Resolve;
    property LogFileName : string read FLogFileName write FLogFileName;
    property MXList : TMXList read FMXList;
  published
    property DNSAddress : string read FDNSAddress write FDNSAddress;
    property Domain : string read FDomain write FDomain;
    property TimeOut : Integer read GetTimeOut write SetTimeOut default 60;
    property Port : word read FPort write FPort default 53;
    property OnSendingQuery : TNotifyEvent read FOnSendingQuery write FOnSendingQuery;
    property OnWaitingForResponse : TNotifyEvent read FOnWaitingForResponse
      write FOnWaitingForResponse;
    property OnClosingConnection : TNotifyEvent read FOnClosingConnection
      write FOnClosingConnection;
  end;

  EResolverError=class(Exception);

  procedure Register;

implementation

procedure Register;
begin
  RegisterComponents('Internet Mail Suite',[TmsMXResolver]);
end;

var
  WSAData : TWSAData;

constructor TmsMXResolver.Create(AOwner : TComponent);
begin
  inherited Create(AOwner);
  agWinsock.LoadWinsock;
  rc:=agWinsock.WSAStartUp($0101,WSAData);
  if rc<>0 then
    Error(rc);
  FDNSAddress:='192.153.156.22';
  FPort:=53;
  TC:=TTimeCounter.Create;
  TimeOut:=60;
  Stream:=TMemoryStream.Create;
  FWindow:=AllocateHWnd(WndProc);
  FLog:=TStringList.Create;
  FMXList:=TMXList.Create;
end;

destructor TmsMXResolver.Destroy;
begin
  MXList.Free;
  if FLogFileName<>'' then
    FLog.SaveToFile(FLogFileName);
  DeallocateHWnd(FWindow);
  FLog.Free;
  Stream.Free;
  rc:=agWinsock.WSACleanup;
  if rc=SOCKET_ERROR then
    Error(agWinsock.WSAGetLastError);
  inherited Destroy;
end;

function TmsMXResolver.GetTimeOut : Integer;
begin
  Result:=TC.TimeOut;
end;

procedure TmsMXResolver.SetTimeOut(Value : Integer);
begin
  TC.TimeOut:=Value;
end;

procedure TmsMXResolver.Error(Ident : word);
var
{$IFDEF USEWSERRDLL}
  LibHandle : THandle;
  Buf : PChar;
{$ENDIF}
  s : string;
begin
  if Ident=0 then Exit;
  TC.TimerOff;
  s:='';
{$IFDEF USEWSERRDLL}
  if (Ident>=WSABASEERR) and (Ident<=WSANO_DATA) then
  begin
{$IFDEF WIN32}
    LibHandle:=LoadLibrary('wserr32.dll');
{$ELSE}
    LibHandle:=LoadLibrary('wserr16.dll');
{$ENDIF}
    if LibHandle>0 then
    begin
      Buf:=StrAlloc(255);
      LoadString(LibHandle,Ident,Buf,255);
      s:=StrPas(Buf);
      StrDispose(Buf);
      FreeLibrary(LibHandle);
    end;
  end
  else
    s:=LoadStr(Ident);
{$ELSE}
  s:=LoadStr(Ident);
{$ENDIF}
  if s='' then
    s:=Concat('Error #',IntToStr(Ident));
  raise ESocketError.Create(s);
end;

procedure TmsMXResolver.WndProc(var Msg : TMessage);
begin
  with Msg do
  if Msg=WM_SOCKETACTIVITY then
  begin
    rc:=agWinsock.WSAGetAsyncError(lParam);
    if (rc=0) and (agWinsock.WSAGetSelectEvent(lParam)=FD_READ) then
      FReadyToRead:=true;
    Result:=0;
  end
  else
    Result:=DefWindowProc(FWindow,Msg,wParam,lParam);
end;

procedure TmsMXResolver.AddToLog(const s : string);
begin
  if FLogFileName<>'' then
    FLog.Add(s);
end;

procedure TmsMXResolver.Cancel;
begin
  FCanceled:=true;
end;

procedure TmsMXResolver.OpenConnection;
var
  DNSIP : u_long;
  Buf : array[0..255] of Char;
begin
  AddToLog('Opening connection');
  StrPCopy(@Buf,FDNSAddress);
  DNSIP:=agWinsock.Inet_Addr(@Buf);
  TheSocket:=agWinsock.Socket(PF_INET,SOCK_DGRAM,IPPROTO_UDP);
  if TheSocket=INVALID_SOCKET then
    Error(agWinsock.WSAGetLastError);
  with RemoteAddress do
  begin
    Sin_Family:=PF_INET;
    Sin_Port:=agWinsock.htons(FPort);
    Sin_addr:=TInAddr(DNSIP);
  end;
  rc:=agWinsock.WSAAsyncSelect(TheSocket,FWindow,WM_SOCKETACTIVITY,FD_READ);
  if rc=SOCKET_ERROR then
    Error(agWinsock.WSAGetLastError);
  rc:=agWinSock.Connect(TheSocket,RemoteAddress,SizeOf(RemoteAddress));
  if rc=SOCKET_ERROR then
    Error(agWinsock.WSAGetLastError);
end;

procedure TmsMXResolver.ConstructQuery;
var
  i,j : Integer;
  QuerySpec : TQuerySpec;
begin
  AddToLog('Constructing query');
  with FQueryHeader do
  begin
    FID:=$1234;
    ID:=agWinsock.htons(FID);
    QuerySpec:=TQuerySpec.Create;
    with QuerySpec do
    try
      QR:=false;
      OpCode:=0;
      RD:=true;   {Recursion desired}
      Spec:=agWinsock.htons(QuerySpec.SpecWord);
    finally
      QuerySpec.Free;
    end;
    QDCount:=agWinsock.htons(1);
    AnCount:=0;
    NsCount:=0;
    ArCount:=0;
  end;
  Stream.Write(FQueryHeader,12);
{Go thru FDomain and replace '.'s by numbers}
  FDomain:=Concat('.',FDomain);
  j:=0;
  for i:=Length(FDomain) downto 1 do
  begin
    if FDomain[i]='.' then
    begin
      FDomain[i]:=Chr(j);
      j:=0;
    end
    else
      Inc(j);
  end;
  FDomain:=Concat(FDomain,#0,#0,#15{MX},#0,#1{IN});
  Stream.Write(FDomain[1],Length(FDomain));
end;

procedure TmsMXResolver.CloseConnection;
begin
  agWinsock.CloseSocket(TheSocket);
end;

procedure TmsMXResolver.DNSError(Ident : word);
var
  s : string;
begin
  case Ident of
    1 : s:=LoadStr(agSDNSFormatError);
    2 : s:=LoadStr(agSDNSServerFailure);
    3 : s:=LoadStr(agSDNSBadDomain);
    4 : s:=LoadStr(agSDNSNotImplemented);
    5 : s:=LoadStr(agSDNSRefused);
    else
      s:='';
  end;
  if s='' then s:='Domain server Error #'+IntToStr(Ident);
  raise EResolverError.Create(s);
end;

procedure TmsMXResolver.SendQuery;
var
  Buf : PChar;
begin
  ConstructQuery;
  AddToLog('Sending query');
  Buf:=StrAlloc(Stream.Size+1);
  try
    Stream.Position:=0;
    Stream.Read(Buf^,Stream.Size);
    rc:=agWinsock.send(TheSocket,Buf,Stream.Size,0);
    if rc=SOCKET_ERROR then
      Error(agWinsock.WSAGetLastError);
  finally
    StrDispose(Buf);
  end;
end;

procedure TmsMXResolver.RecvData;
var
  Buf : PChar;
  Done : boolean;
  LastError : Integer;
begin
  TC.TimerOn;
  repeat
    Application.ProcessMessages;
  until FReadyToRead or TC.TimedOut or FCanceled;
  TC.TimerOff;
  if TC.TimedOut then
    raise ETimedOutError.Create;
  if FCanceled then
    raise ECanceledError.Create;
  Buf:=StrAlloc(BufSize);
  try
    TC.TimerOn;
    repeat
      rc:=agWinsock.Recv(TheSocket,Buf,BufSize,0);
      if rc=SOCKET_ERROR then
      begin
        LastError:=agWinsock.WSAGetLastError;
        if LastError<>WSAEWOULDBLOCK then
          Error(LastError);
      end;
      Done:=rc>0;
      Application.ProcessMessages;
    until Done or TC.TimedOut;
    TC.TimerOff;
    if TC.TimedOut then
      raise ETimedOutError.Create;
    Stream.Clear;
    Stream.Write(Buf^,rc);
  finally
    StrDispose(Buf);
  end;
end;

function TmsMXResolver.ntohDomain : string;
var
  c,i : byte;
  Ch : char;
  SaveStreamPos : LongInt;
begin
  Result:='';
  repeat
    Stream.Read(c,1);
    if c=$C0 then
    begin
      Stream.Read(c,1);
      SaveStreamPos:=Stream.Position;
      Stream.Position:=c;
      Result:=Result+ntohDomain;
      Stream.Position:=SaveStreamPos;
      c:=0;
    end
    else
    begin
      for i:=1 to c do
      begin
        Stream.Read(Ch,1);
        Result:=Concat(Result,Ch);
      end;
      if c<>0 then
        Result:=Concat(Result,'.');
    end;
  until c=0;
end;

procedure TmsMXResolver.ParseHeader;
var
  QuerySpec : TQuerySpec;
begin
  Stream.Position:=0;
  Stream.Read(FQueryHeader,12);
  with FQueryHeader do
  begin
    ID:=agWinsock.ntohs(ID);
    Spec:=agWinsock.ntohs(Spec);
    QDCount:=agWinsock.ntohs(QDCount);
    AnCount:=agWinsock.ntohs(AnCount);
    NsCount:=agWinsock.ntohs(NsCount);
    ArCount:=agWinsock.ntohs(ArCount);
  end;
  if FQueryHeader.ID<>FID then
    raise EResolverError.CreateRes(agSDNSDifferentIDs);
  QuerySpec:=TQuerySpec.Create;
  try
    QuerySpec.SpecWord:=FQueryHeader.Spec;
    if not QuerySpec.QR then
      raise EResolverError.CreateRes(agSDNSNoResponse);
    if QuerySpec.AA then
      AddToLog('Authoritative Answer');
    if QuerySpec.TC then
      AddToLog('Response truncated');
    if QuerySpec.RA then
      AddToLog('Recursion available');
    if QuerySpec.RCode>0 then
    begin
      AddToLog('Error code '+IntToStr(QuerySpec.RCode));
      DNSError(QuerySpec.RCode);
    end;
  finally
    QuerySpec.Free;
  end;
end;

procedure TmsMXResolver.ParseQuestion;
var
  QuestRec : TQuestRec;
begin
{ Skip the question part - it should be same as
  we sent}
  with QuestRec do
  begin
    Domain:=ntohDomain;
    Stream.Read(QType,2);
    QType:=agWinsock.ntohs(QType);
    Stream.Read(QClass,2);
    QClass:=agWinsock.ntohs(QClass);
  end;
end;

procedure TmsMXResolver.ParseAnswer;
var
  AnswerRec : TAnswerRec;
  MXRec : TMXRec;
begin
  with AnswerRec do
  begin
    Name:=ntohDomain;
    Stream.Read(QType,2);
    QType:=agWinsock.ntohs(QType);
    Stream.Read(QClass,2);
    QClass:=agWinsock.ntohs(QClass);
    Stream.Read(TTL,4);
    TTL:=agWinsock.ntohl(TTL);
    Stream.Read(RDLen,2);
    RDLen:=agWinsock.ntohs(RDLen);
    if QType=15 then
    begin
      MXRec:=TMXRec.Create;
      with MXRec do
      try
        Stream.Read(Preference,2);
        Preference:=agWinsock.ntohs(Preference);
        Exchanger:=ntohDomain;
      except
        free;
        raise;
      end;
      FMXList.Add(MXRec);
    end;
  end;
end;

procedure TmsMXResolver.GetResults;
var
  i : Integer;
begin
  ParseHeader;
  ParseQuestion;
  for i:=1 to FQueryHeader.AnCount do
    ParseAnswer;
end;

procedure TmsMXResolver.Reinit;
begin
  FCanceled:=false;
  FReadyToRead:=false;
  TC.TimedOut:=false;
  Stream.Clear;
end;

procedure TmsMXResolver.Resolve;
begin
  Reinit;
  FMXList.Clear;
  if Assigned(FOnSendingQuery) then
    FOnSendingQuery(Self);
  OpenConnection;
  try
    SendQuery;
    if Assigned(FOnWaitingForResponse) then
      FOnWaitingForResponse(Self);
    RecvData;
    GetResults;
    FMXList.Sort;
  finally
    CloseConnection;
    if Assigned(FOnClosingConnection) then
      FOnClosingConnection(Self);
  end;
end;

end.
