(**********************************************)
(*  The Internet Mail Suite  1.9.2            *)
(*  (c) ArGo Software Design, 1996,1997,1998. *)
(**********************************************)
unit mspop3;

{$I msDef.inc}

interface

uses
{$IFDEF WIN32}
  Windows,
{$ELSE}
  WinProcs,
{$ENDIF}
  SysUtils, Messages, Classes, Graphics, Controls,
  Forms, Dialogs, ExtCtrls, msmb, msUtils, msMsgCls, msMsg, msCls,
  mspop3cl, msConst, agConst, agTypes, agSocket, msMIME;

type
  TmsCustomPOP = class(TmsMailBase)
  private
    { Private declarations }
    FUserName : string;
    FPassword : string;
    FTotalMessages : Integer;
    FTotalOctets : LongInt;
    FOnRetrieving : TmsNumNotifyEvent;
    FOnRetrieved : TmsNumNotifyEvent;
    FOnDeleting : TmsNumNotifyEvent;
    procedure DoRetrieving(Sender : TObject; Num : Integer);
    procedure DoRetrieved(Sender : TObject; Num : Integer);
    procedure DoDeleting(Sender : TObject; Num : Integer);
  protected
    { Protected declarations }
    WSInitCount : Integer;
    procedure OutputMsgStream(MsgNo : Integer;
              MsgStream : TMemoryStream); virtual; abstract;
    procedure LogIn; override;
    procedure LogOut; override;
    procedure Verify; virtual;
    function GetServerReplyCode(const s : string) : Integer; override;
    procedure GetStatistics;
    procedure ReInit; override;
  public
    { Public declarations }
    constructor Create(AOwner : TComponent); override;
    destructor Destroy; override;
    procedure Open; override;
    procedure RetrieveMessage(MsgNo : Integer);
    procedure DeleteMessage(MsgNo : Integer);
    property TotalMessages : Integer read FTotalMessages;
    property TotalOctets : LongInt read FTotalOctets;
  published
    { Published declarations }
    property UserName : string read FUserName write FUserName;
    property Password : string read FPassword write FPassword;
    property DefaultPort;
    property OnProgress;
    property OnRetrieving : TmsNumNotifyEvent read FOnRetrieving write FOnRetrieving;
    property OnRetrieved : TmsNumNotifyEvent read FOnRetrieved write FOnRetrieved;
    property OnDeleting : TmsNumNotifyEvent read FOnDeleting write FOnDeleting;
  end;

  TmsPOP = class(TmsCustomPOP)
  private
    { Private declarations }
    FMessageList : TmsMessageList;
  protected
    { Protected declarations }
    procedure OutputMsgStream(MsgNo : Integer; MsgStream : TMemoryStream); override;
  public
    { Public declarations }
    constructor Create(AOwner : TComponent); override;
    destructor Destroy; override;
    property MessageList : TmsMessageList read FMessageList;
    procedure RetrieveAllMessages(LeaveOnServer : boolean);
  published
    { Published declarations }
  end;

  TmsRemotePOP = class(TmsCustomPOP)
  private
    { Private declarations }
    FMessageDirectory : string;
    FRemoteInfo : TmsRemoteInfo;
    FKeepAlive : boolean;
    FOnUpdatingInfo : TNotifyEvent;
    FOnInfoUpdated : TNotifyEvent;
    FOnTransferringData : TNotifyEvent;
    FOnDataTransfered : TNotifyEvent;
    FOnNextInfoUpdated : TmsNumNotifyEvent;
    FOnUserMsgSave : TmsUserMsgSaveEvent;
    procedure SetMessageDirectory(Value : string);
    procedure DoUpdatingInfo(Sender : TObject);
    procedure DoInfoUpdated(Sender : TObject);
    procedure DoTransferringData(Sender : TObject);
    procedure DoDataTransfered(Sender : TObject);
    procedure DoNextInfoUpdated(Sender : TObject; Num : Integer);
  protected
    { Protected declarations }
    procedure OutputMsgStream(MsgNo : Integer; MsgStream : TMemoryStream); override;
    procedure FillRemoteInfoItem(MsgNo : Integer; RII : TmsRemoteInfoItem;
                                 UIDL, OtherInfo : boolean; var UIDLSupported : boolean);
    function CheckIdentity(MsgNo : Integer) : boolean;
    function IDsAreSame(RII1,RII2 : TmsRemoteInfoItem) : boolean;
  public
    { Public declarations }
    constructor Create(AOwner : TComponent); override;
    destructor Destroy; override;
    procedure UpdateRemoteInfo;
    procedure UpdateUIDL;
    procedure UpdateRemoteInfoAndUIDL;
    procedure MarkToDelete(Index : Integer);
    procedure MarkToRetrieve(Index : Integer);
    procedure Reset(Index : Integer);
    procedure Transfer;
    property RemoteInfo : TmsRemoteInfo read FRemoteInfo;
    property KeepAlive : boolean read FKeepAlive write FKeepAlive;
  published
    { Published declarations }
    property MessageDirectory : string read FMessageDirectory write
                  SetMessageDirectory;
    property OnUpdatingInfo : TNotifyEvent read FOnUpdatingInfo
               write FOnUpdatingInfo;
    property OnInfoUpdated : TNotifyEvent read FOnInfoUpdated
               write FOnInfoUpdated;
    property OnTransferringData : TNotifyEvent read FOnTransferringData
               write FOnTransferringData;
    property OnDataTransfered : TNotifyEvent read FOnDataTransfered
               write FOnDataTransfered;
    property OnNextInfoUpdated : TmsNumNotifyEvent read FOnNextInfoUpdated
               write FOnNextInfoUpdated;
    property OnUserMessageSave : TmsUserMsgSaveEvent read FOnUserMsgSave
               write FOnUserMsgSave;
  end;

implementation

{TmsCustomPOP}
constructor TmsCustomPOP.Create(AOwner : TComponent);
begin
  inherited Create(AOwner);
  ServiceName:='pop3';
  DefaultPort:=110;
  FTotalMessages:=-1;
  FTotalOctets:=-1;
end;

destructor TmsCustomPOP.Destroy;
begin
  inherited Destroy;
end;

procedure TmsCustomPOP.Verify;
begin
  if UserName='' then
    Error(msSUserNameRequired)
  else
  if Password='' then
    Error(msSPasswordRequired);
end;

procedure TmsCustomPOP.DoRetrieving(Sender : TObject; Num : Integer);
begin
  if Assigned(FOnRetrieving) then
    FOnRetrieving(Sender,Num);
end;

procedure TmsCustomPOP.DoRetrieved(Sender : TObject; Num : Integer);
begin
  if Assigned(FOnRetrieved) then
    FOnRetrieved(Sender,Num);
end;

procedure TmsCustomPOP.DoDeleting(Sender : TObject; Num : Integer);
begin
  if Assigned(FOnDeleting) then
    FOnDeleting(Sender,Num);
end;

procedure TmsCustomPOP.Open;
begin
  Verify;
  ReInit;
  inherited Open;
end;

function TmsCustomPOP.GetServerReplyCode(const s : string) : Integer;
{1=+, -1=-, 0=?}
begin
  Result:=0;
  if Length(s)>0 then
  begin
    case s[1] of
     '+' : Result:=1;
     '-' : Result:=-1;
     else
       Result:=0;
    end;
  end;
end;

procedure TmsCustomPOP.LogIn;
var
  rs : Integer;
begin
  ReInit;
  try
    Connect;
    rs:=RecvLine(TempS);
    if rs<>1 then
      raise EServerError.Create(TempS);
    SendLine('USER '+FUserName);
    rs:=RecvLine(TempS);
    if rs<>1 then
      raise EServerError.Create(TempS);
    SendLine('PASS '+FPassword);
    rs:=RecvLine(TempS);
    if rs<>1 then
      raise EServerError.Create(TempS);
    GetStatistics;
  except
    on ESocketError do
    begin
      Close;
      raise;
    end
    else
      raise;
  end;
end;

procedure TmsCustomPOP.LogOut;
var
  rs : Integer;
begin
  if OnLine then
  begin
    SendLine('QUIT');
    rs:=RecvLine(TempS);
    if rs<>1 then
      raise EServerError.Create(TempS);
  end;
end;

procedure TmsCustomPOP.ReInit;
begin
  inherited ReInit;
end;

procedure TmsCustomPOP.GetStatistics;
var
  i : byte;
  rs : Integer;
begin
  SendLine('STAT');
  rs:=RecvLine(TempS);
  if rs<>1 then
    raise EServerError.Create(TempS);
  Delete(TempS,1,4);
  i:=Pos(' ',TempS);
  if i>0 then
  begin
    FTotalMessages:=StrToInt(Copy(TempS,1,i-1));
    Delete(TempS,1,i);
    i:=Pos(' ',TempS);
    if i>0 then
      FTotalOctets:=StrToInt(Copy(TempS,1,i-1))
    else
      FTotalOctets:=StrToInt(TempS);
  end;
end;

procedure TmsCustomPOP.RetrieveMessage(MsgNo : Integer);
var
  TempStream : TMemoryStream;
  MsgSize : LongInt;
  j : byte;
  rs : Integer;
begin
  SendLine('LIST '+IntToStr(MsgNo+1));
  rs:=RecvLine(TempS);
  if rs<>1 then
    raise EServerError.Create(TempS);
  Delete(TempS,1,4);
  j:=Pos(' ',TempS);
  MsgSize:=StrToIntDef(Copy(TempS,j+1,Length(TempS)-j),0);
  DoRetrieving(Self,MsgNo);
  SendLine('RETR '+IntToStr(MsgNo+1));
  rs:=RecvLine(TempS);
  if rs<>1 then
    raise EServerError.Create(TempS);
  TempStream:=TMemoryStream.Create;
  try
    RecvLineStream(TempStream,MsgSize);
    OutputMsgStream(MsgNo,TempStream);
  finally
    TempStream.Free;
  end;
  DoRetrieved(Self,MsgNo);
end;

procedure TmsCustomPOP.DeleteMessage(MsgNo : Integer);
begin
  DoDeleting(Self,MsgNo);
  SendLine('DELE '+IntToStr(MsgNo+1));
  RecvLine(TempS);
end;

{TmsPOP}
constructor TmsPOP.Create(AOwner : TComponent);
begin
  inherited Create(AOwner);
  FMessageList:=TmsMessageList.Create(Self);
end;

destructor TmsPOP.Destroy;
begin
  FMessageList.Free;
  inherited Destroy;
end;

procedure TmsPOP.OutputMsgStream(MsgNo : Integer; MsgStream : TMemoryStream);
var
  TempMessage : TmsMessage;
begin
  TempMessage:=TmsListMessage.Create(FMessageList);
  TempMessage.LoadFromStream(MsgStream);
end;

procedure TmsPOP.RetrieveAllMessages(LeaveOnServer : boolean);
var
  i : Integer;
begin
  OpenConnection;
  try
    for i:=0 to TotalMessages-1 do
    begin
      RetrieveMessage(i);
      if not LeaveOnServer then
        DeleteMessage(i);
    end;
  finally
    CloseConnection;
  end;
end;

{TmsRemotePOP}
constructor TmsRemotePOP.Create(AOwner : TComponent);
begin
  inherited Create(AOwner);
  FRemoteInfo:=TmsRemoteInfo.Create;
end;

destructor TmsRemotePOP.Destroy;
begin
  FRemoteInfo.Free;
  inherited Destroy;
end;

procedure TmsRemotePOP.SetMessageDirectory(Value : string);
begin
  if (Value<>'') and (Value[Length(Value)]<>'\') then
    FMessageDirectory:=Concat(Value,'\')
  else
    FMessageDirectory:=Value;
end;

procedure TmsRemotePOP.DoUpdatingInfo(Sender : TObject);
begin
  if Assigned(FOnUpdatingInfo) then
    OnUpdatingInfo(Sender);
end;

procedure TmsRemotePOP.DoInfoUpdated(Sender : TObject);
begin
  if Assigned(FOnInfoUpdated) then
    OnInfoUpdated(Sender);
end;

procedure TmsRemotePOP.DoTransferringData(Sender : TObject);
begin
  if Assigned(FOnTransferringData) then
    FOnTransferringData(Sender);
end;

procedure TmsRemotePOP.DoDataTransfered(Sender : TObject);
begin
  if Assigned(FOnDataTransfered) then
    FOnDataTransfered(Sender);
end;

procedure TmsRemotePOP.DoNextInfoUpdated(Sender : TObject; Num : Integer);
begin
  if Assigned(FOnNextInfoUpdated) then
    FOnNextInfoUpdated(Sender,Num);
end;

procedure TmsRemotePOP.FillRemoteInfoItem(MsgNo : Integer;
          RII : TmsRemoteInfoItem; UIDL, OtherInfo : boolean; var UIDLSupported : boolean);
var
  TempStream : TMemoryStream;
  j,rs : Integer;
begin
  if UIDL then
  begin
    RII.UIDL:='';
    SendLine('UIDL '+IntToStr(MsgNo+1));
    rs:=RecvLine(TempS);
    if rs=1 then
      RII.UIDL:=PickString(TempS,2,true)
    else
      UIDLSupported:=false;
  end;
  if OtherInfo then
  begin
    SendLine('LIST '+IntToStr(MsgNo+1));
    rs:=RecvLine(TempS);
    if rs<>1 then
      raise EServerError.Create(TempS);
    Delete(TempS,1,4);
    j:=Pos(' ',TempS);
    RII.Size:=StrToIntDef(Copy(TempS,j+1,Length(TempS)-j),0);
    SendLine('TOP '+IntToStr(MsgNo+1)+' 1');
    rs:=RecvLine(TempS);
    if rs<>1 then
      raise EServerError.Create(TempS);
    TempStream:=TMemoryStream.Create;
    try
      RecvLineStream(TempStream,-1);  { -1 retrieving just headers}
      TempStream.Position:=0;
      RII.Headers.LoadFromStream(TempStream);
      RII.Headers.CleanUp;
    finally
      TempStream.Free;
    end;
  end;
end;

procedure TmsRemotePOP.UpdateUIDL;
var
  TempInfoItem : TmsRemoteInfoItem;
  i : Integer;
  UIDLSupported : boolean;
begin
  OpenConnection;
  DoUpdatingInfo(Self);
  FRemoteInfo.Clear;
  UIDLSupported:=true;
  for i:=0 to FTotalMessages-1 do
  begin
    TempInfoItem:=TmsRemoteInfoItem.Create;
    try
      FillRemoteInfoItem(i,TempInfoItem,UIDLSupported,false,UIDLSupported);
      FRemoteInfo.Add(TempInfoItem);
      DoNextInfoUpdated(Self,FRemoteInfo.Count-1);
    except
      TempInfoItem.Free;
      raise;
    end;
  end;
  if not FKeepAlive then
    CloseConnection;
  DoInfoUpdated(Self);
end;

procedure TmsRemotePOP.UpdateRemoteInfo;
var
  TempInfoItem : TmsRemoteInfoItem;
  i : Integer;
  UIDLSupported : boolean;
begin
  OpenConnection;
  DoUpdatingInfo(Self);
  FRemoteInfo.Clear;
  for i:=0 to FTotalMessages-1 do
  begin
    TempInfoItem:=TmsRemoteInfoItem.Create;
    try
      FillRemoteInfoItem(i,TempInfoItem,false,true,UIDLSupported);
      FRemoteInfo.Add(TempInfoItem);
      DoNextInfoUpdated(Self,FRemoteInfo.Count-1);
    except
      TempInfoItem.Free;
      raise;
    end;
  end;
  if not FKeepAlive then
    CloseConnection;
  DoInfoUpdated(Self);
end;

procedure TmsRemotePOP.UpdateRemoteInfoAndUIDL;
var
  TempInfoItem : TmsRemoteInfoItem;
  i : Integer;
  UIDLSupported : boolean;
begin
  OpenConnection;
  DoUpdatingInfo(Self);
  FRemoteInfo.Clear;
  UIDLSupported:=true;
  for i:=0 to FTotalMessages-1 do
  begin
    TempInfoItem:=TmsRemoteInfoItem.Create;
    try
      FillRemoteInfoItem(i,TempInfoItem,UIDLSupported,true,UIDLSupported);
      FRemoteInfo.Add(TempInfoItem);
      DoNextInfoUpdated(Self,FRemoteInfo.Count-1);
    except
      TempInfoItem.Free;
      raise;
    end;
  end;
  if not FKeepAlive then
    CloseConnection;
  DoInfoUpdated(Self);
end;

procedure TmsRemotePOP.MarkToDelete(Index : Integer);
begin
  FRemoteInfo[Index].MarkedToDelete:=true;
end;

procedure TmsRemotePOP.MarkToRetrieve(Index : Integer);
begin
  FRemoteInfo[Index].MarkedToRetrieve:=true;
end;

procedure TmsRemotePOP.Reset(Index : Integer);
begin
  FRemoteInfo[Index].MarkedToDelete:=false;
  FRemoteInfo[Index].MarkedToRetrieve:=false;
end;

function TmsRemotePOP.IDsAreSame(RII1,RII2 : TmsRemoteInfoItem) : boolean;
var
  ID1,ID2 : string;
  i,Count : Integer;
begin
  Result:=true;
  Exit;
  ID1:=RII1.Headers.GetFieldBody('Message-ID');
  ID2:=RII2.Headers.GetFieldBody('Message-ID');
  if (ID1<>'') and (ID2<>'') then
  begin
    Result:=ID1=ID2;
    Exit;
  end;
  if RII1.Headers.Count<RII1.Headers.Count then Count:=RII1.Headers.Count
    else Count:=RII2.Headers.Count;
  for i:=0 to Count-1 do
  begin
    Result:=RII1.Headers[i]=RII2.Headers[i];
    if not Result then Break;
  end;
end;

function TmsRemotePOP.CheckIdentity(MsgNo : Integer) : boolean;
var
  TempRII : TmsRemoteInfoItem;
  UIDLSupported : boolean;
begin
  TempRII:=TmsRemoteInfoItem.Create;
  try
    FillRemoteInfoItem(MsgNo,TempRII,true,true,UIDLSupported);
{Compare with existing}
    Result:=IDsAreSame(TempRII,FRemoteInfo[MsgNo]);
    if not Result then
      Error(msSIDsDiffer);  {This should not happen}
  finally
    TempRII.Free;
  end;
end;

procedure TmsRemotePOP.Transfer;
var
  i : Integer;
begin
  if FRemoteInfo.Count<>0 then
  begin
    OpenConnection;
    DoTransferringData(Self);
    for i:=0 to FRemoteInfo.Count-1 do
    begin
      if FRemoteInfo[i].MarkedToRetrieve then
      begin
        if CheckIdentity(i) then
          RetrieveMessage(i);
      end;
      if FRemoteInfo[i].MarkedToDelete then
      begin
        if CheckIdentity(i) then
          DeleteMessage(i);
      end;
    end;
    DoDataTransfered(Self);
    CloseConnection;
  end;
end;

procedure TmsRemotePOP.OutputMsgStream(MsgNo : Integer; MsgStream : TMemoryStream);
var
  FileName : string;
begin
  if not Assigned(FOnUserMsgSave) then
  begin
    FileName:=Concat(FMessageDirectory,'msg_',IntToStr(MsgNo),'.eml');
    MsgStream.SaveToFile(FileName);
  end
  else
    FOnUserMsgSave(Self,MsgNo,MsgStream);
end;

end.
