(**********************************************)
(*  The Internet Mail Suite  1.9.2            *)
(*  (c) ArGo Software Design, 1996,1997,1998. *)
(**********************************************)

{ Modified by: Bruce W. Caron                }
{ Excelsoft Design Limited                   }
{ 22 Clearview Drive                         }
{ Bedford, NS, Canada, B4A3C8                }
{ February 19,1998                           }
{ Added: RFC977-NNTP-Ext 3.1 AUTHINFO        }

unit msnntp;

interface

uses
  SysUtils, WinTypes, WinProcs, Messages, Classes, Graphics, Controls,
  Forms, Dialogs, agSocket, msMb, agTypes, msnwscls, msMsg, agConst, msCls,
  msConst, msMsgCls, msUtils;

const
  AuthRequest1 = 480;                      {bwc}
  AuthRequest2 = 450;

type

  TmsNNTP = class(TmsMailBase)
  private
    { Private declarations }
    FNewsGroup : string;
    FTotalArticles : LongInt;
    FFirstArticle : LongInt;
    FLastArticle : LongInt;
    FCurrentArticle : LongInt;
    FLastHeaderNumber : LongInt;
    FNewsInfoList : TmsNgInfoList;
    FNewsGroupList : TmsNewsGroupList;

    FAuthenticate : Boolean;               {bwc}
    FUsername     : string;                {bwc}
    FPassword     : string;                {bwc}

    FOnNextHeaderRetrieved : TmsNumNotifyEvent;
    FArticle : TmsArticle;
    FOnPosted : TNotifyEvent;
    FOnPosting : TNotifyEvent;
    FOnArticleRetrieved : TNotifyEvent;
    FOnGroupSelected : TNotifyEvent;
    FOnLastHeaderRetrieved : TNotifyEvent;
    FOnRetrievingNewsGroupList : TNotifyEvent;
    FOnNewsGroupListRetrieved : TNotifyEvent;
    procedure SetNewsGroup(const Value : string);
    procedure SetCurrentArticle(Value : LongInt);
    function GetCurrentArticle : LongInt;
    procedure SetLastHeaderNumber(Value : LongInt);
    procedure DoNextHeaderRetrieved(Sender : TObject; Num : LongInt);
    procedure DoArticleRetrieved(Sender : TObject);
    procedure DoPosted(Sender : TObject);
    procedure DoPosting(Sender : TObject);
    procedure DoGroupSelected(Sender : TObject);
    procedure DoLastHeaderRetrieved(Sender : TObject);
    procedure DoRetrievingNewsGroupList(Sender : TObject);
    procedure DoNewsGroupListRetrieved(Sender : TObject);
    function SendLineHook( Cmd : string ) : Integer;                     {bwc}
  protected
    { Protected declarations }
    procedure Notification(AComponent : TComponent;
               Operation : TOperation); override;
    procedure LogIn; override;
    procedure LogOut; override;
    function ConcatIfNotBlank(const s1,s2 : string) : string;
    procedure RetrieveCurrentHeader(NewsHeaders : TmsNgInfoItem);
  public
    { Public declarations }
    constructor Create(AOwner : TComponent); override;
    destructor Destroy; override;
    procedure Open; override;
    procedure RetrieveNewsGroupList;
    procedure LoadNewsGroupList(const FileName : string);
    procedure SaveNewsGroupList(const FileName : string);
    procedure RetrieveCurrentArticle;
    procedure RetrieveArticleByNumber(Num : LongInt);
    procedure RetrieveArticleById(const ID : string);
    function SetNextArticle : boolean;
    procedure RetrieveAllHeaders;
    procedure RetrieveNextNHeaders(N : Integer; Append : boolean);
    procedure PostArticle;
    property NewsHeadersList : TmsNgInfoList read FNewsInfoList;
    property FirstArticle : LongInt read FFirstArticle;
    property LastArticle : LongInt read FLastArticle;
    property TotalArticles : LongInt read FTotalArticles;
    property CurrentArticle : LongInt read GetCurrentArticle write SetCurrentArticle;
    property LastHeaderNumber : LongInt read FLastHeaderNumber write SetLastHeaderNumber;
    property NewsGroupList : TmsNewsGroupList read FNewsGroupList write FNewsGroupList;
  published
    { Published declarations }
    property Article : TmsArticle read FArticle write FArticle;
    property NewsGroup : string read FNewsGroup write SetNewsGroup;

    property Authenticate: Boolean read FAuthenticate write FAuthenticate;   {bwc}
    property Username    : string  read FUsername write FUsername;           {bwc}
    property Password    : string  read FPassword write FPassword;           {bwc}

    property OnNextHeaderRetrieved : TmsNumNotifyEvent read FOnNextHeaderRetrieved
                        write FOnNextHeaderRetrieved;
    property OnProgress;
    property OnArticleRetrieved : TNotifyEvent read FOnArticleRetrieved
                        write FOnArticleRetrieved;
    property OnPosted : TNotifyEvent read FOnPosted write FOnPosted;
    property OnPosting : TNotifyEvent read FOnPosting write FOnPosting;
    property OnGroupSelected : TNotifyEvent read FOnGroupSelected write FOnGroupSelected;
    property OnLastHeaderRetrieved : TNotifyEvent read FOnLastHeaderRetrieved
                        write FOnLastHeaderRetrieved;
    property OnRetrievingNewsGroupList : TNotifyEvent read FOnRetrievingNewsGroupList
                        write FOnRetrievingNewsGroupList;
    property OnNewsGroupListRetrieved : TNotifyEvent read FOnNewsGroupListRetrieved
                        write FOnNewsGroupListRetrieved;
  end;

implementation

constructor TmsNNTP.Create(AOwner : TComponent);
begin
  inherited Create(AOwner);
  FNewsInfoList:=TmsNgInfoList.Create;
  FNewsGroupList:=TmsNewsGroupList.Create;
  FLastHeaderNumber:=-1;
  ServiceName:='nntp';
  DefaultPort:=119;
  Units:=utLines;
end;

destructor TmsNNTP.Destroy;
begin
  CloseConnection;
  FNewsGroupList.Free;
  FNewsInfoList.Free;
  inherited Destroy;
end;

function TmsNNTP.SendLineHook( Cmd : string ) : Integer;     {bwc}
var
  rs : Integer;
begin
  SendLine(Cmd);
  rs := RecvLine(TempS);
  if TempS='.' then
    rs:=RecvLine(TempS);
  Result := rs;

  if FAuthenticate and ((rs = AuthRequest1) or (rs = AuthRequest2)) then
  begin
    SendLine('AUTHINFO USER ' + FUsername );
    rs:=RecvLine(TempS);
    if (rs=381) then
    begin
      SendLine('AUTHINFO PASS ' + FPassword );
      rs:=RecvLine(TempS);
      if (rs=281) or (rs=250) then
      begin
        SendLine(Cmd);
        rs := RecvLine(TempS);
        Result := rs;
      end;
    end else
      Result := rs;
  end;
end;

procedure TmsNNTP.DoNextHeaderRetrieved(Sender : TObject; Num : LongInt);
begin
  if Assigned(FOnNextHeaderRetrieved) then
    FOnNextHeaderRetrieved(Sender,Num);
end;

procedure TmsNNTP.DoArticleRetrieved(Sender : TObject);
begin
  if Assigned(FOnArticleRetrieved) then
    FOnArticleRetrieved(Sender);
end;

procedure TmsNNTP.Notification(AComponent : TComponent;
             Operation : TOperation);
begin
  inherited Notification(AComponent, Operation);
  if (Operation=opRemove) and (AComponent=FArticle) then
    FArticle:=nil;
end;

procedure TmsNNTP.SetNewsGroup(const Value : string);
var
  rs : Integer;
begin
  if (UpperCase(Value)<>UpperCase(FNewsGroup)) then
  begin
    OpenConnection;
    rs:=SendLineHook('GROUP '+Value);  {bwc}
    if (rs<>211) then
      raise EServerError.Create(TempS);
    FTotalArticles:=PickNumber(TempS,1);
    FFirstArticle:=PickNumber(TempS,2);
    FLastArticle:=PickNumber(TempS,3);
    FCurrentArticle:=FFirstArticle;
    FLastHeaderNumber:=-1;
    FNewsGroup:=Value;
    DoGroupSelected(Self);
  end;
end;

procedure TmsNNTP.SetCurrentArticle(Value : LongInt);
var
  rs : Integer;
begin
  if FNewsGroup='' then
    Error(msSNewsGroupNotSelected);
  rs := SendLineHook('STAT '+IntToStr(Value));   {bwc}
  if (rs>223) or (rs<220) then
    raise EServerError.Create(TempS);
  FCurrentArticle:=PickNumber(TempS,1);
end;

function TmsNNTP.GetCurrentArticle : LongInt;
var
  rs : Integer;
begin
  if FNewsGroup='' then
    Error(msSNewsGroupNotSelected);
  rs := SendLineHook('STAT');                    {bwc}
  if (rs>223) or (rs<220) then
    raise EServerError.Create(TempS);
  FCurrentArticle:=PickNumber(TempS,1);
  Result:=FCurrentArticle;
end;

procedure TmsNNTP.SetLastHeaderNumber(Value : LongInt);
begin
  if Value<>-1 then
  begin
    if FNewsGroup='' then
      Error(msSNewsGroupNotSelected);
    if (Value<FirstArticle) and (Value>LastArticle) then
      Error(msSBadArticleNumber);
  end;
  FLastHeaderNumber:=Value;
end;

procedure TmsNNTP.DoPosted(Sender : TObject);
begin
  if Assigned(FOnPosted) then
    FOnPosted(Sender);
end;

procedure TmsNNTP.DoPosting(Sender : TObject);
begin
  if Assigned(FOnPosting) then
    FOnPosting(Sender);
end;

procedure TmsNNTP.DoGroupSelected(Sender : TObject);
begin
  if Assigned(FOnGroupSelected) then
    FOnGroupSelected(Sender);
end;

procedure TmsNNTP.DoLastHeaderRetrieved(Sender : TObject);
begin
  if Assigned(FOnLastHeaderRetrieved) then
    FOnLastHeaderRetrieved(Sender);
end;

procedure TmsNNTP.DoRetrievingNewsGroupList(Sender : TObject);
begin
  if Assigned(FOnRetrievingNewsGroupList) then
    FOnRetrievingNewsGroupList(Sender);
end;

procedure TmsNNTP.DoNewsGroupListRetrieved(Sender : TObject);
begin
  if Assigned(FOnNewsGroupListRetrieved) then
    FOnNewsGroupListRetrieved(Sender);
end;

procedure TmsNNTP.Open;
begin
  ReInit;
  Verify;
  inherited Open;
end;

procedure TmsNNTP.LogIn;
var
  rs : Integer;
begin
  Connect;
  rs:=RecvLine(TempS);
  if (rs<>200) and (rs<>201) then
    raise EServerError.Create(TempS);
end;

procedure TmsNNTP.LogOut;
var
  rs : Integer;
begin
  FNewsGroup:='';
  SendLine('QUIT');
  rs:=RecvLine(TempS);
  if rs<>205 then
    raise EServerError.Create(TempS);
  Close;
end;

function TmsNNTP.ConcatIfNotBlank(const s1,s2 : string) : string;
begin
  Result:=s1;
  if s2<>'' then
    Result:=Concat(s1,' ',s2);
end;

procedure TmsNNTP.RetrieveArticleById(const ID : string);
var
  TempStream : TMemoryStream;
  NewsHeaders : TmsHeaders;
  Lines : LongInt;
  rs : Integer;
begin
  TempStream:=TMemoryStream.Create;
  try
    rs := SendLineHook(ConcatIfNotBlank('HEAD',ID));   {bwc}
    if (rs>223) or (rs<220) then
      raise EServerError.Create(TempS);
    RecvLineStream(TempStream,-1);
    TempStream.Position:=0;
    NewsHeaders:=TmsHeaders.Create;
    try
      NewsHeaders.LoadFromStream(TempStream);
      try
        Lines:=StrToInt(NewsHeaders.GetFieldBody('LINES'));
      except
        Lines:=-1;
      end;
    finally
      NewsHeaders.Free;
    end;
    FArticle.Clear;
    TempStream.Clear;
    rs := SendLineHook(ConcatIfNotBlank('ARTICLE',ID));  {bwc}
    if (rs>223) or (rs<220) then
      raise EServerError.Create(TempS);
    RecvLineStream(TempStream,Lines);
    TempStream.Position:=0;
    FArticle.LoadFromStream(TempStream);
  finally
    TempStream.Free;
  end;
  DoArticleRetrieved(Self);
end;

procedure TmsNNTP.RetrieveNewsGroupList;
var
  TempStream : TStream;
  rs : Integer;
begin
  OpenConnection;
  FNewsGroupList.Clear;
  DoRetrievingNewsGroupList(Self);
  rs := SendLineHook('LIST');                     {bwc}
  if rs<>215 then
    raise EServerError.Create(TempS);
  TempStream:=TMemoryStream.Create;
  try
    RecvLineStream(TempStream,-1);
    TempStream.Position:=0;
    FNewsGroupList.LoadFromStream(TempStream);
  finally
    TempStream.Free;
  end;
  DoNewsGroupListRetrieved(Self);
end;

procedure TmsNNTP.LoadNewsGroupList(const FileName : string);
begin
  FNewsGroupList.LoadFromFile(FileName);
end;

procedure TmsNNTP.SaveNewsGroupList(const FileName : string);
begin
  FNewsGroupList.SaveToFile(FileName);
end;

procedure TmsNNTP.RetrieveCurrentArticle;
begin
  if FNewsGroup='' then
    Error(msSNewsGroupNotSelected);
  RetrieveArticleById('');
end;

procedure TmsNNTP.RetrieveArticleByNumber(Num : LongInt);
begin
  CurrentArticle:=Num;
  RetrieveCurrentArticle;
end;

function TmsNNTP.SetNextArticle : boolean;
var
  rs : Integer;
begin
  if FNewsGroup='' then
    Error(msSNewsGroupNotSelected);
  rs := SendLineHook('NEXT');                      {bwc}
  if (rs<>421) and (rs<>223) then
    raise EServerError.Create(TempS);
  Result:=rs<>421;
  if Result then
    FCurrentArticle:=PickNumber(TempS,1);
end;

procedure TmsNNTP.RetrieveCurrentHeader(NewsHeaders : TmsNgInfoItem);
var
  TempStream : TStream;
  rs : Integer;
begin
  if FNewsGroup='' then
    Error(msSNewsGroupNotSelected);
  rs := SendLineHook('HEAD');                      {bwc}
  if (rs>223) or (rs<220) then
    raise EServerError.Create(TempS);
  NewsHeaders.Number:=PickNumber(TempS,1);
  TempStream:=TMemoryStream.Create;
  try
    RecvLineStream(TempStream,-1);
    TempStream.Position:=0;
    NewsHeaders.LoadFromStream(TempStream);
  finally
    TempStream.Free;
  end;
end;

procedure TmsNNTP.RetrieveAllHeaders;
begin
  RetrieveNextNHeaders(FTotalArticles,false);
end;

procedure TmsNNTP.RetrieveNextNHeaders(N : Integer; Append : boolean);
var
  SetOk : boolean;
  NewsInfoItem : TmsNgInfoItem;
  ct : Integer;
begin
  if FNewsGroup='' then
    Error(msSNewsGroupNotSelected);
  if not Append then
    FNewsInfoList.Clear;
  if FLastHeaderNumber<>-1 then
  begin
    CurrentArticle:=FLastHeaderNumber;
    if not SetNextArticle then
    begin
      DoLastHeaderRetrieved(Self);
      Exit;
    end;
  end
  else
  begin
    try
      CurrentArticle:=FirstArticle;
    except
      on E:EServerError do
      begin
        if Pos('423',E.Message)>0 then
        begin
          SetNextArticle;
          FFirstArticle:=FCurrentArticle;
        end
        else
          raise;
      end
      else
        raise;
    end;
  end;
  ct:=0;
  repeat
    NewsInfoItem:=TmsNgInfoItem.Create;
    RetrieveCurrentHeader(NewsInfoItem);
    FNewsInfoList.Add(NewsInfoItem);
    DoNextHeaderRetrieved(Self,FNewsInfoList.Count-1);
    SetOk:=SetNextArticle;
    Inc(ct);
  until (not SetOk) or Canceled or (ct=N);
  FLastHeaderNumber:=FNewsInfoList[FNewsInfoList.Count-1].Number;
  if Canceled then
    raise ECanceledError.Create;
  if not SetOk then
    DoLastHeaderRetrieved(Self);
end;

procedure TmsNNTP.PostArticle;
var
  TempFileStream : TTempFileStream;
  rs : Integer;
begin
  if not Assigned(FArticle) then
    Error(msSArticleRequired);
  Article.Verify;
  DoPosting(Self);
  TempFileStream:=TTempFileStream.Create;
  try
    Article.SaveToStream(TempFileStream);
    OpenConnection;
    rs := SendLineHook('POST');                      {bwc}
    if rs<>340 then
      raise EServerError.Create(TempS);
    SendStream(TempFileStream);
    rs := SendLineHook('.');
    if rs<>240 then
      raise EServerError.Create(TempS);
    DoPosted(Self);
  finally
    TempFileStream.Free;
  end;
end;

end.

