(**********************************************)
(*  The Internet Mail Suite  1.9.2            *)
(*  (c) ArGo Software Design, 1996,1997,1998. *)
(**********************************************)
unit msCls;

{$I msdef.inc}

interface

uses
{$IFDEF WIN32}
  Windows,
{$ELSE}
  WinProcs,
  ms16utls,
{$ENDIF}
  Classes,
  ExtCtrls;

const
  BadString = #255#255#255#255;

type

  TTempFileStream = class(TFileStream)
  private
    FFileName : string;
    FFile : file;
  protected
    function MakeTempFileName : string;
  public
    constructor Create;
    destructor Destroy; override;
    procedure LoadFromStream(Stream : TStream);
    procedure LoadFromFile(const FileName : string);
    procedure SaveToStream(Stream : TSTream);
    procedure SaveToFile(const FileName : string);
  end;

  TLineStream = class(TMemoryStream)
  private
    FLines : TStringList;
    FCurLine : Integer;
    FBufSize : Word;
    FEOS : boolean;
    procedure SetBufSize(Value : Word);
  protected
    Finished : boolean;
    procedure LoadNextPart;
  public
    constructor Create;
    destructor Destroy; override;
    procedure Reset;
    function GetNextLine : string;
    property BufSize : Word read FBufSize write SetBufSize
                         default $4000;
    property EOS : boolean read FEOS;
 end;

 TTimeCounter = class
 private
   FTimer : TTimer;
   FCount : Integer;
   FTimeOut : Integer;
   FTimedOut : boolean;
 protected
   procedure TimerOnTimer(Sender : TObject);
 public
   constructor Create;
   destructor Destroy; override;
   procedure TimerOn;
   procedure TimerOff;
   property TimedOut : boolean read FTimedOut write FTimedOut;
   property TimeOut : Integer read FTimeOut write FTimeOut;
 end;

implementation

uses
  msUtils,SysUtils;

constructor TTempFileStream.Create;
begin
  FFileName:=MakeTempFileName;
  inherited Create(FFileName,fmCreate or fmOpenReadWrite);
end;

destructor TTempFileStream.Destroy;
begin
  inherited Destroy;
  AssignFile(FFile,FFileName);
  try
    Erase(FFile);
  except
    {Do Nothing}
  end;
end;

function TTempFileStream.MakeTempFileName : string;
var
  TempPath : array[0..255] of char;
  TempFileName : array[0..255] of char;
begin
{$IFDEF WIN32}
  GetTempPath(255,@TempPath);
  GetTempFileName(@TempPath,PChar('ms'),0,@TempFileName);
{$ELSE}
  GetTempFileName(#0,'ms',0,@TempFileName);
{$ENDIF}
  Result:=StrPas(@TempFileName);
end;

procedure TTempFileStream.LoadFromStream(Stream : TStream);
var
  Buf : PChar;
  rd : LongInt;
begin
  Buf:=StrAlloc(2048);
  try
    Position:=0;
    Stream.Position:=0;
    repeat
      rd:=Stream.Read(Buf^,2048);
      Write(Buf^,rd);
    until rd<2048;
  finally
    StrDispose(Buf);
  end;
end;

procedure TTempFileStream.LoadFromFile(const FileName : string);
var
  FileStream : TFileStream;
begin
  FileStream:=TFileStream.Create(FileName, fmOpenRead);
  try
    LoadFromStream(FileStream);
  finally
    FileStream.Free;
  end;
end;

procedure TTempFileStream.SaveToStream(Stream : TStream);
var
  Buf : PChar;
  rd : LongInt;
begin
  Buf:=StrAlloc(2048);
  try
    Position:=0;
    Stream.Position:=0;
    repeat
      rd:=Read(Buf^,2048);
      Stream.Write(Buf^,rd);
    until rd<2048;
  finally
    StrDispose(Buf);
  end;
end;

procedure TTempFileStream.SaveToFile(const FileName : string);
var
  FileStream : TFileStream;
begin
  FileStream:=TFileStream.Create(FileName, fmOpenWrite or fmCreate);
  try
    SaveToStream(FileStream);
  finally
    FileStream.Free;
  end;
end;

constructor TLineStream.Create;
begin
  inherited Create;
  FLines:=TStringList.Create;
  Finished:=false;
  FCurLine:=-1;
  FEOS:=false;
  FBufSize:=$4000;
end;

destructor TLineStream.Destroy;
begin
  FLines.Free;
  inherited Destroy;
end;

procedure TLineStream.SetBufSize(Value : word);
begin
  if FBufSize<>Value then
    FBufSize:=Value;
end;

procedure TLineStream.LoadNextPart;
var
  Buf,BufEnd : PChar;
  rb : LongInt;
begin
  Buf:=StrAlloc(FBufSize);
  try
    FillChar(Buf^,FBufSize,0);
    rb:=Read(Buf^,FBufSize);
    Finished:=rb<FBufSize;
    BufEnd:=LineStart(Buf,Buf+FBufSize);
    Position:=Position-(Buf+FBufSize-BufEnd);
    if not Finished then
      BufEnd^:=#0;
{$IFDEF WIN32}
    FLines.SetText(Buf);
{$ELSE}
    BufToLines(Buf,FLines,255);
{$ENDIF}
  finally
    StrDispose(Buf);
  end;
end;

procedure TLineStream.Reset;
begin
  FCurLine:=-1;
  FLines.Clear;
  FEOS:=false;
  Position:=0;
  Finished:=false;
end;

function TLineStream.GetNextLine : string;
begin
  Inc(FCurLine);
  if FCurLine>=FLines.Count then
  begin
    if not Finished then
    begin
      LoadNextPart;
      FCurLine:=0;
      if FLines.Count>0 then
        Result:=FLines[FCurLine]
      else
      begin
        Result:=BadString;
        FEOS:=true;
      end;
    end
    else
    begin
      Result:=BadString;
      FEOS:=true;
    end;
  end
  else
    Result:=FLines[FCurLine];
end;

{TTimeCounter}
constructor TTimeCounter.Create;
begin
  inherited Create;
  FTimer:=TTimer.Create(nil);
  FTimer.Enabled:=false;
  FTimeOut:=60;
  FTimer.OnTimer:=TimerOnTimer;
end;

destructor TTimeCounter.Destroy;
begin
  FTimer.Free;
  inherited Destroy;
end;

procedure TTimeCounter.TimerOnTimer(Sender : TObject);
begin
  Dec(FCount);
  if FCount<=0 then
    FTimedOut:=true;
end;

procedure TTimeCounter.TimerOn;
begin
  FCount:=FTimeOut;
  FTimer.Enabled:=true;
end;

procedure TTimeCounter.TimerOff;
begin
  FTimer.Enabled:=false;
end;

end.
