(**********************************************)
(*  The Internet Mail Suite  1.9.2            *)
(*  (c) ArGo Software Design, 1996,1997,1998. *)
(**********************************************)
unit agsocket;

{$I msdef.inc}

interface

uses
{$IFDEF EMBEDRAS}
  agRasCmp,
{$ENDIF}
{$IFDEF WIN32}
  Windows,
{$IFDEF STATICWINSOCK}
  Winsock,
{$ELSE}
  agWsock,
{$ENDIF}
{$ELSE}
  WinProcs,
  WinTypes,
  agWsock,
{$ENDIF}
  Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  ExtCtrls, agConst, agTypes, msUtils, msCls;

type
  TagSocket = class(TComponent)
  private
    { Private declarations }
    FServer : string;
    FTimeOut : Integer;
    FLogFileName : string;
    FDefaultPort : word;
    FUnits : TUnits;
    FServiceName : string;
    FProtocol : TmsProtocol;
    FUseBlockingSockets : boolean;
    FOnOpen : TNotifyEvent;
    FOnConnected : TNotifyEvent;
    FOnClosing : TNotifyEvent;
    FOnClosed : TNotifyEvent;
    FOnCancel : TNotifyEvent;
    FOnProgress : TmsProgressEvent;
    FOnError : TmsErrorEvent;
    FOnLineSent : TmsLineTransmitEvent;
    FOnLineReceived : TmsLineTransmitEvent;
{$IFDEF EMBEDRAS}
    FRAS : TagRas;
{$ENDIF}
    InvWnd : THandle;
    function GetOnLine : boolean;
    function JustServer : string;  {new, if port is forced}
    function PickPort : word; { -"-}
{$IFNDEF STATICWINSOCK}
    function GetWinsockLoaded : boolean;
    procedure SetWinsockLoaded(Value : boolean);
    function GetWinsockAvailable : boolean;
{$ENDIF}
    function GetTimeOut : Integer;
    procedure SetTimeOut(Value : Integer);
    procedure DoOpen(Sender : TObject);
    procedure DoConnected(Sender : TObject);
    procedure DoClosing(Sender : TObject);
    procedure DoClosed(Sender : TObject);
    procedure DoCancel(Sender : TObject);
    procedure DoProgress(Sender : TObject; Perc : Integer; Transferred : LongInt);
    procedure DoLineSent(Sender : TObject; const TheLine : string);
    procedure DoLineReceived(Sender : TObject; const TheLine : string);
  protected
    { Protected declarations }
    MyWSAData : TWSAData;
    TheSocket : TSocket;
    AsyncHandle : Integer;
    TC : TTimeCounter;
    ServerInAddr : u_Long;
    ServerIPAddr : string;
    LocalHost : string;
    ThePort : word;
    WsInitCount : Integer;
    Log : TStrings;
    Canceled : boolean;
    HostFound : boolean;
    ServiceFound : boolean;
    Connected : boolean;
    DataHasArrived : boolean;
    ReadyToSend : boolean;
    ConnectionClosed : boolean;
    AsyncError : boolean;
    ErrorNo : word;
    Flags : word;
    TempS : string;
{$IFDEF EMBEDRAS}
    procedure Notification(AComponent : TComponent;
                Operation : TOperation); override;
{$ENDIF}
    procedure Error(Ident : word);
    procedure TimeOutError;
    procedure CanceledError;
    procedure WndProc(var Msg : TMessage);
    procedure ReInit; virtual;
    procedure ResolveRemoteHost;
    procedure GetLocalHost;
    procedure GetService;
    procedure OpenSocket;
    procedure Connect;
    procedure Verify;
    function GetServerReplyCode(const s : string) : Integer; virtual; {11.07}
    function RecvLine(var s : string) : Integer;
    function RecvLines : Integer;
    procedure SendLine(const s : string);
    procedure RecvLineStream(SL : TStream; FullSize : LongInt);
    procedure RecvStream(SL : TStream; FullSize : LongInt);
    procedure SendStream(SL : TStream);
    procedure DoError(Sender : TObject; var Msg : string);
    property Server : string read FServer write FServer;
    property ServiceName : string read FServiceName write FServiceName;
    property Protocol : TmsProtocol read FProtocol write FProtocol;
    property DefaultPort : word read FDefaultPort write FDefaultPort;
    property TimeOut : Integer read GetTimeOut write SetTimeOut default 60;
    property OnOpen : TNotifyEvent read FOnOpen write FOnOpen;
    property OnConnected : TNotifyEvent read FOnConnected write FOnConnected;
    property OnClosing : TNotifyEvent read FOnClosing write FOnClosing;
    property OnClosed : TNotifyEvent read FonClosed write FOnClosed;
    property OnCancel : TNotifyEvent read FOnCancel write FOnCancel;
    property OnProgress : TmsProgressEvent read FOnProgress write FOnProgress;
    property OnError : TmsErrorEvent read FOnError write FOnError;
    property OnLineSent : TmsLineTransmitEvent read FOnLineSent write FOnLineSent;
    property OnLineReceived : TmsLineTransmitEvent read FOnLineReceived write FOnLineReceived;
  public
    { Public declarations }
    constructor Create(AOwner : TComponent); override;
    destructor Destroy; override;
    procedure Open; virtual; {11.4}
    procedure Close; virtual;
    procedure Cancel; virtual;
    procedure WriteLogFile;
    property LogFileName : string read FLogFileName write FLogFileName;
    property OnLine : boolean read GetOnLine;
    property Units : TUnits read FUnits write FUnits; {For progress in NNTP}
    property UseBlockingSockets : boolean read FUseBlockingSockets
       write FUseBlockingSockets default false;
{$IFNDEF STATICWINSOCK}
    property WinsockAvailable : boolean read GetWinsockAvailable;
{$ENDIF}
  published
    { Published declarations }
{$IFNDEF STATICWINSOCK}
    property WinsockLoaded : boolean read GetWinsockLoaded write SetWinsockLoaded
         default false;
{$ENDIF}
{$IFDEF EMBEDRAS}
    property RAS : TagRAS read FRAS write FRAS;
{$ENDIF}
end;

const
  WM_MSBASE = WM_USER+147;
  WM_HOSTFOUND = WM_MSBASE+1;
  WM_SERVICEFOUND = WM_MSBASE+2;
  WM_SOCKETACTIVITY = WM_MSBASE+3;

implementation

{$IFDEF WIN32}
{$R agconst.res}
{$ELSE}
{$R agcons16.res}
{$ENDIF}

{$IFNDEF USEWSERRDLL}
{$IFDEF WIN32}
{$R wserr32.res}
{$ELSE}
{$R wserr16.res}
{$ENDIF}
{$ENDIF}

{$IFDEF UNREGISTERED}
uses msNag;
{$ENDIF}

constructor TagSocket.Create(AOwner : TComponent);
begin
  inherited Create(AOwner);
  TheSocket:=INVALID_SOCKET;
  FTimeOut:=60;
  Canceled:=false;
  TC:=TTimeCounter.Create;
  Log:=TStringList.Create;
  if not (csDesigning in ComponentState) then
    InvWnd:=AllocateHWnd(WndProc);
  WsInitCount:=0;
  AsyncHandle:=0;
  AsyncError:=false;
  HostFound:=false;
  ServiceFound:=false;
  Connected:=false;
  DataHasArrived:=false;
  ReadyToSend:=false;
  FUnits:=utOctets;
  FProtocol:=ptTCP;
{$IFNDEF STATICWINSOCK}
  WinsockLoaded:=false;
{$ENDIF}
  FUseBlockingSockets:=false;
  Flags:=0;
end;

destructor TagSocket.Destroy;
var
  i : Integer;
begin
  if not (csDesigning in ComponentState) then
    DeallocateHWnd(InvWnd);
  for i:=1 to WsInitCount do
{$IFDEF STATICWINSOCK}
    Winsock.WSACleanup;
{$ELSE}
    agWinsock.WSACleanUp;
{$ENDIF}
  WriteLogFile;
  Log.Free;
  TC.Free;
  inherited Destroy;
end;

procedure TagSocket.WndProc(var Msg : TMessage);
begin
  with Msg do
  if Msg=WM_HOSTFOUND then
  begin
    HostFound:=true;
{$IFDEF STATICWINSOCK}
    ErrorNo:=Winsock.WSAGetAsyncError(lParam);
{$ELSE}
    ErrorNo:=agWinsock.WSAGetAsyncError(lParam);
{$ENDIF}
    AsyncError:=ErrorNo<>0;
    Result:=0;
  end
  else
  if Msg=WM_SERVICEFOUND then
  begin
    ServiceFound:=true;
{$IFDEF STATICWINSOCK}
    ErrorNo:=Winsock.WSAGetAsyncError(lParam);
{$ELSE}
    ErrorNo:=agWinsock.WSAGetAsyncError(lParam);
{$ENDIF}
    AsyncError:=ErrorNo<>0;
    Result:=0;
  end
  else
  if Msg=WM_SOCKETACTIVITY then
  begin
{$IFDEF STATICWINSOCK}
    ErrorNo:=Winsock.WSAGetAsyncError(lParam);
{$ELSE}
    ErrorNo:=agWinsock.WSAGetAsyncError(lParam);
{$ENDIF}
    AsyncError:=ErrorNo<>0;
    if not AsyncError then
    begin
{$IFDEF STATICWINSOCK}
      case Winsock.WSAGetSelectEvent(lParam) of
{$ELSE}
      case agWinsock.WSAGetSelectEvent(lParam) of
{$ENDIF}
      FD_CONNECT :
        Connected:=true;
      FD_READ :
        DataHasArrived:=true;
      FD_WRITE :
        ReadyToSend:=true;
      FD_CLOSE :
        ConnectionClosed:=true;
      end;
    end;
    Result:=0;
  end
  else
    Result:=DefWindowProc(InvWnd,Msg,wParam,lParam);
end;

function TagSocket.GetOnline : boolean;
begin
  Result:=TheSocket<>INVALID_SOCKET;
end;

procedure TagSocket.DoOpen(Sender : TObject);
begin
  if Assigned(FOnOpen) then
    FOnOpen(Sender);
end;

procedure TagSocket.DoConnected(Sender : TObject);
begin
  if Assigned(FOnConnected) then
    FOnConnected(Sender);
end;

procedure TagSocket.DoClosing(Sender : TObject);
begin
  if Assigned(FOnClosing) then
    FOnClosing(Sender);
end;

procedure TagSocket.DoClosed(Sender : TObject);
begin
  if Assigned(FOnClosed) then
    FOnClosed(Sender);
end;

procedure TagSocket.DoCancel(Sender : TObject);
begin
  if Assigned(FOnCancel) then
    FOnCancel(Sender);
end;

procedure TagSocket.DoProgress(Sender : TObject; Perc : Integer;
    Transferred : LongInt);
begin
  if Assigned(FOnProgress) then
    FOnProgress(Sender,Perc,Transferred);
end;

procedure TagSocket.DoLineSent(Sender : TObject; const TheLine : string);
begin
  if Assigned(FOnLineSent) then
    FOnLineSent(Sender,TheLine);
end;

procedure TagSocket.DoLineReceived(Sender : TObject; const TheLine : string);
begin
  if Assigned(FOnLineReceived) then
    FOnLineReceived(Sender,TheLine);
end;

procedure TagSocket.DoError(Sender : TObject; var Msg : string);
begin
  if Assigned(FOnError) then
    FOnError(Sender,Msg);
end;

procedure TagSocket.WriteLogFile;
begin
  if FLogFileName<>'' then
{$IFDEF WIN32}
    SLSaveToFile(Log,FLogFileName);
{$ELSE}
    Log.SaveToFile(FLogFileName);
{$ENDIF}
end;

{$IFDEF EMBEDRAS}
procedure TagSocket.Notification(AComponent : TComponent;
             Operation : TOperation);
begin
  inherited Notification(AComponent,Operation);
  if (Operation=opRemove) and (AComponent=FRAS) then
    FRAS:=nil;
end;
{$ENDIF}

procedure TagSocket.Error(Ident : word);
var
{$IFDEF USEWSERRDLL}
  LibHandle : THandle;
  Buf : PChar;
{$ENDIF}
  s : string;
begin
  if Ident=0 then Exit;
  TC.TimerOff;
  s:='';
{$IFDEF USEWSERRDLL}
  if (Ident>=WSABASEERR) and (Ident<=WSANO_DATA) then
  begin
{$IFDEF WIN32}
    LibHandle:=LoadLibrary('wserr32.dll');
{$ELSE}
    LibHandle:=LoadLibrary('wserr16.dll');
{$ENDIF}
    if LibHandle>0 then
    begin
      Buf:=StrAlloc(255);
      LoadString(LibHandle,Ident,Buf,255);
      s:=StrPas(Buf);
      StrDispose(Buf);
      FreeLibrary(LibHandle);
    end;
  end
  else
    s:=LoadStr(Ident);
{$ELSE}
  s:=LoadStr(Ident);
{$ENDIF}
  if s='' then
    s:=Concat('Error #',IntToStr(Ident));
  DoError(Self,s);
  if Ident=WSAECONNRESET then
    Close;
  raise ESocketError.Create(s);
end;

procedure TagSocket.TimeOutError;
var
  s : string;
begin
  TC.TimerOff;
  s:=LoadStr(agSTimedOut);
  DoError(Self,s);
  if AsyncHandle<>0 then
  begin
{$IFDEF STATICWINSOCK}
    Winsock.WSACancelAsyncRequest(AsyncHandle);
{$ELSE}
    agWinsock.WSACancelAsyncRequest(AsyncHandle);
{$ENDIF}
    AsyncHandle:=0;
  end;
{$IFDEF STATICWINSOCK}
  if Winsock.WSAIsBlocking then
    Winsock.WSACancelBlockingCall;
{$ELSE}
  if agWinsock.WSAIsBlocking then
    agWinsock.WSACancelBlockingCall;
{$ENDIF}
  Close;
  raise ETimedOutError.Create;
end;

procedure TagSocket.CanceledError;
var
  s : string;
begin
  TC.TimerOff;
  s:=LoadStr(agSCanceled);
  DoError(Self,s);
  raise ECanceledError.Create;
end;

procedure TagSocket.ReInit;
begin
  Canceled:=false;
  TC.TimedOut:=false;
  AsyncHandle:=0;
  AsyncError:=false;
  HostFound:=false;
  ServiceFound:=false;
  Connected:=false;
  DataHasArrived:=false;
  ReadyToSend:=false;
  ConnectionClosed:=false;
end;

function TagSocket.JustServer : string;
{returns Server if server forces the port, e.g. smtp.blah.com:120}
var
  i : Integer;
begin
  Result:=FServer;
  i:=Pos(':',Result);
  if i>0 then
    Result:=Copy(Result,1,i-1);
end;

function TagSocket.PickPort : word;
var
  s : string;
  i : Integer;
begin
  i:=Pos(':',FServer);
  if i>0 then
  begin
    s:=Copy(FServer,i+1,Length(FServer)-i);
    try
      Result:=StrToInt(s);
    except
      Result:=0;
    end;
  end
  else
    Result:=0;
end;

{$IFNDEF STATICWINSOCK}
function TagSocket.GetWinsockLoaded : boolean;
begin
  Result:=agWinsock.Loaded;
end;

procedure TagSocket.SetWinsockLoaded(Value : boolean);
begin
  if Value then
  begin
    if not agWinsock.Loaded then
      agWinsock.LoadWinsock;
  end
  else
   agWinsock.FreeWinsock;
end;

function TagSocket.GetWinsockAvailable : boolean;
begin
  Result:=agWinsock.Available;
end;
{$ENDIF}

function TagSocket.GetTimeOut : Integer;
begin
  Result:=TC.TimeOut;
end;

procedure TagSocket.SetTimeOut(Value : Integer);
begin
  TC.TimeOut:=Value;
end;

procedure TagSocket.Verify;
begin
  if FServer='' then
    Error(agSServerRequired);
end;

function TagSocket.GetServerReplyCode(const s : string) : Integer;
begin
  Result:=StrToIntDef(Copy(s,1,3),-1);
end;

function TagSocket.RecvLine(var s : string) : Integer;
var
  InBuf : array[0..1024] of char;
  rc,i : Integer;
  Finished : boolean;
  LastError : Integer;
begin
  Result:=-1;
    if not OnLine then
      Error(agSNotConnected);
  if not FUseBlockingSockets then
  begin
    TC.TimerOn;
    repeat
      Application.ProcessMessages;
    until DataHasArrived or Canceled or TC.TimedOut or AsyncError;
    TC.TimerOff;
    if Canceled then
      CanceledError
    else
    if TC.TimedOut then
      TimeOutError
    else
    if AsyncError then
      Error(ErrorNo);
    TC.TimerOn;
    DataHasArrived:=false;
    Application.ProcessMessages;
  end;
  FillChar(InBuf,SizeOf(InBuf),0);
  i:=0; s:='';
  try
    repeat
      if not FUseBlockingSockets then
        Application.ProcessMessages;
{receive characters one by one}
{$IFDEF STATICWINSOCK}
      rc:=Winsock.recv(TheSocket,InBuf[i],1,0);
{$ELSE}
      rc:=agWinsock.recv(TheSocket,@InBuf[i],1,0);
{$ENDIF}
      if rc=SOCKET_ERROR then
      begin
{$IFDEF STATICWINSOCK}
        LastError:=Winsock.WSAGetLastError;
{$ELSE}
        LastError:=agWinsock.WSAGetLastError;
{$ENDIF}
        if FUseBlockingSockets or ((LastError<>WSAEWOULDBLOCK) and not Canceled) then
          Error(LastError);
      end
      else
        Inc(i,rc);
      Finished:=StrPos(@InBuf,^J)<>nil;
      if not FUseBlockingSockets then
        Application.ProcessMessages;
    until Finished or Canceled or TC.TimedOut or (ConnectionClosed and (rc=0));
    TC.TimerOff;
    if Finished then
    begin
      s:=StrPas(@InBuf);
      while (Length(s)>0) and (s[Length(s)] in [^M,^J]) do
        Delete(s,Length(s),1);
      Result:=GetServerReplyCode(s);
    end;
    if not FUseBlockingSockets then
    begin
      if TC.TimedOut then
        TimeOutError
      else
      if Canceled then
        CanceledError;
    end;
  finally
    if LogFileName<>'' then
      Log.Add(s);
    DoLineReceived(Self,s);
  end;
end;

function TagSocket.RecvLines : Integer;
var
  Code : Integer;
begin
  Code:=RecvLine(TempS);
  Result:=Code;
  if (Code>0) and (Pos('-',TempS)=4) then {there are more lines}
  begin
    repeat
      Code:=RecvLine(TempS);
    until ConnectionClosed or ((Code=Result) and (Pos(' ',TempS)=4));
  end;
end;

procedure TagSocket.SendLine(const s : string);
var
  Buf : string;
  sd, i : Integer;
  LastError : Integer;
  Finished : boolean;
begin
  if not OnLine then
    Error(agSNotConnected);
  FillChar(Buf,SizeOf(Buf),0);
  Buf:=Concat(s,^M^J);
    TC.TimerOn;
  i:=1;
  repeat
    if not FUseBlockingSockets then
      Application.ProcessMessages;
{$IFDEF STATICWINSOCK}
    sd:=Winsock.Send(TheSocket,Buf[i],Length(Buf)-i+1,Flags);
{$ELSE}
    sd:=agWinsock.Send(TheSocket,@Buf[i],Length(Buf)-i+1,Flags);
{$ENDIF}
    if sd=SOCKET_ERROR then
    begin
{$IFDEF STATICWINSOCK}
      LastError:=Winsock.WSAGetLastError;
{$ELSE}
      LastError:=agWinsock.WSAGetLastError;
{$ENDIF}
      if FUseBlockingSockets or ((LastError<>WSAEWOULDBLOCK) and not Canceled) then
        Error(LastError);
    end
    else
      Inc(i,sd);
    Finished:=i>Length(Buf);
    if not FUseBlockingSockets then
      Application.ProcessMessages;
  until Finished or TC.TimedOut or Canceled;
  if not FUseBlockingSockets then
  begin
    TC.TimerOff;
    if TC.TimedOut then
      TimeOutError
    else
    if Canceled then
      CanceledError;
  end;
  if LogFileName<>'' then
    Log.Add(s);
  DoLineSent(Self,s);
end;

procedure TagSocket.RecvLineStream(SL : TStream; FullSize : LongInt);
{ Recieves the stream.  Stops when CRLF.CRLF arrives}
const
  BufEndSign : PChar = ^M^J'.'^M^J;
var
  Buf : PChar;
  BufEnd : array[0..5] of Char;
  rc, LastError : Integer;
  Finished : boolean;
begin
  if not OnLine then
    Error(agSNotConnected);
{for NNTP - estimated size in octets}
  if (FUnits=utLines) and (FullSize<>-1) then
    FullSize:=FullSize*76;
  if not FUseBlockingSockets then
  begin
  {Wait until data arrives}
    TC.TimerOn;
    repeat
      Application.ProcessMessages;
    until DataHasArrived or Canceled or TC.TimedOut or AsyncError;
    TC.TimerOff;
    if Canceled then
      CanceledError
    else
    if TC.TimedOut then
      TimeOutError
    else
    if AsyncError then
      Error(ErrorNo);
    DataHasArrived:=false;
  end;
  Buf:=StrAlloc(MaxBufSize);
  try
    FillChar(Buf^,StrBufSize(Buf),0);
    Finished:=false;
    if not FUseBlockingSockets then
      TC.TimerOn;
    repeat
{$IFDEF STATICWINSOCK}
      rc:=Winsock.recv(TheSocket,Buf,MaxBufSize,0);
{$ELSE}
      rc:=agWinsock.recv(TheSocket,Buf,MaxBufSize,0);
{$ENDIF}
      if rc=SOCKET_ERROR then
      begin
{$IFDEF STATICWINSOCK}
        LastError:=Winsock.WSAGetLastError;
{$ELSE}
        LastError:=agWinsock.WSAGetLastError;
{$ENDIF}
        if FUseBlockingSockets or ((LastError<>WSAEWOULDBLOCK) and not Canceled) then
          Error(LastError);
      end;
      if FullSize>0 then
        DoProgress(Self,Round(100*SL.Position/FullSize),SL.Position)
      else
        DoProgress(Self,-1,SL.Position);
      if rc>0 then
      begin
        SL.Write(Buf^,rc);
        if SL.Position>=5 then
        begin
          SL.Seek(-5,2);
          FillChar(BufEnd,6,0);
          SL.Read(BufEnd,5);
          Finished:=StrComp(@BufEnd,BufEndSign)=0;
        end;
        FillChar(Buf^,StrBufSize(Buf),0);
        TC.TimerOn;
      end;
      if not FUseBlockingSockets then
        Application.ProcessMessages;
    until Finished or Canceled or TC.TimedOut;
    if not FUseBlockingSockets then
    begin
      TC.TimerOff;
      if TC.TimedOut then
        TimeOutError
      else
      if Canceled then
        CanceledError;
    end;
  finally
    StrDispose(Buf);
  end;
end;

procedure TagSocket.RecvStream(SL : TStream; FullSize : LongInt);
{Receives the stream.  Stops when exactly Size bytes received or connection
 closed from the server side}
var
  Finished : boolean;
  LastError : Integer;
  rc, rs : LongInt{Integer};
  Total : LongInt;
  RecvBuf : array[1..$2001] of Char;
begin
  DoProgress(Self,0,0);
  Total:=0;
  if not FUseBlockingSockets then
  begin
    TC.TimerOn;
    repeat
      Application.ProcessMessages;
    until DataHasArrived or Canceled or TC.TimedOut or AsyncError;
    TC.TimerOff;
    if Canceled then
      CanceledError
    else
    if TC.TimedOut then
      TimeOutError
    else
    if AsyncError then
      Error(ErrorNo);
    DataHasArrived:=false;
  end;
  repeat
    if not FUseBlockingSockets then
    begin
      TC.TimerOn;
      Application.ProcessMessages;
    end;
    if FullSize>-1 then
    begin
      rs:=FullSize-Total;
      if rs>$2000 then
        rs:=$2000;
    end
    else
     rs:=$2000;
{$IFDEF STATICWINSOCK}
    rc:=Winsock.recv(TheSocket,RecvBuf,rs,0);
{$ELSE}
    rc:=agWinsock.recv(TheSocket,@RecvBuf,rs,0);
{$ENDIF}
    if rc=SOCKET_ERROR then
    begin
{$IFDEF STATICWINSOCK}
      LastError:=Winsock.WSAGetLastError;
{$ELSE}
      LastError:=agWinsock.WSAGetLastError;
{$ENDIF}
      if FUseBlockingSockets or ((LastError<>WSAEWOULDBLOCK) and not Canceled) then
        Error(LastError)
    end
    else
    begin
      Total:=Total+rc;
      SL.Write(RecvBuf,rc);
      if FullSize>0 then
        DoProgress(Self,Round(100*SL.Position/FullSize),Total)
      else
        DoProgress(Self,-1,Total);
      if not FUseBlockingSockets then
        TC.TimerOff;
      FillChar(RecvBuf,SizeOf(RecvBuf),0);
    end;
    if not FUseBlockingSockets then
      Application.ProcessMessages;
    Finished:=(ConnectionClosed and (rc=0)) or (Total=FullSize);
  until Finished or Canceled or TC.TimedOut;
  if not FUseBlockingSockets then
  begin
    TC.TimerOff;
  {  SL.Position:=0;}
    if Finished then
      DoClosed(Self)
    else
    if Canceled then
      CanceledError
    else
    if TC.TimedOut then
      TimeOutError;
  end;
end;

procedure TagSocket.SendStream(SL : TStream);
var
  Buf : array[1..2048] of char;
  rd,sd,i : Integer;
  Done : boolean;
  LastError : Integer;
begin
  SL.Position:=0;
  if LogFileName<>'' then
    Log.Add('Suppressing the message body');
  repeat
    FillChar(Buf,SizeOf(Buf),0);
    rd:=SL.Read(Buf,2047);
    i:=1;
    if not FUseBlockingSockets then
      TC.TimerOn;
    repeat
{$IFDEF STATICWINSOCK}
      sd:=Winsock.Send(TheSocket,Buf[i],rd-i+1,0);
{$ELSE}
      sd:=agWinsock.Send(TheSocket,@Buf[i],rd-i+1,0);
{$ENDIF}
      if sd=SOCKET_ERROR then
      begin
{$IFDEF STATICWINSOCK}
        LastError:=Winsock.WSAGetLastError;
{$ELSE}
        LastError:=agWinsock.WSAGetLastError;
{$ENDIF}
        if FUseBlockingSockets or ((LastError<>WSAEWOULDBLOCK) and not Canceled) then
          Error(LastError);
        sd:=0;
      end;
      if sd>0 then
      begin
        TC.TimerOn;
        Inc(i,sd);
      end;
      Done:=i>rd;
      if not FUseBlockingSockets then
        Application.ProcessMessages;
    until Done or TC.TimedOut or Canceled;
    if not FUseBlockingSockets then
      TC.TimerOff;
    Done:=rd<2047;
    DoProgress(Self,Round(100*SL.Position/SL.Size),SL.Position);
  until Done or TC.TimedOut or Canceled;
  if not FUseBlockingSockets then
  begin
    if TC.TimedOut then
      TimeOutError
    else
    if Canceled then
      CanceledError;
  end;
  DoProgress(Self,0,0);
end;

procedure TagSocket.Open;  {11.4}
begin
{$IFDEF EMBEDRAS}
  if assigned(FRAS) then
    FRAS.Dial;
{$ENDIF}
  DoOpen(Self);
  Verify;
{$IFDEF STATICWINSOCK}
  if Winsock.WSAStartUp($0101,MyWSADATA)<>0 then
{$ELSE}
  if agWinsock.WSAStartUp($0101,MyWSADATA)<>0 then
{$ENDIF}
    Error(agSBadVersion);
  Inc(wsInitCount); {11.4}
  GetLocalHost;
  ResolveRemoteHost;
  GetService;
  OpenSocket;
end;

procedure TagSocket.Close;
begin
  if TheSocket<>INVALID_SOCKET then
  begin
    DoClosing(Self);
{$IFDEF STATICWINSOCK}
    Winsock.CloseSocket(TheSocket);
{$ELSE}
    agWinsock.CloseSocket(TheSocket);
{$ENDIF}
    TheSocket:=INVALID_SOCKET;
    DoClosed(Self);
  end;
{$IFDEF EMBEDRAS}
  if Assigned(FRAS) then
    FRAS.Cancel;
{$ENDIF}
end;

procedure TagSocket.ResolveRemoteHost;
var
  RemoteHost : PHostEnt;
  Buf : array[0..MAXGETHOSTSTRUCT] of char;
  a : array[0..3] of byte;
  PBuf : array[0..127] of byte;
  i : byte;
begin
  FillChar(PBuf,SizeOf(PBuf),0);
  StrPCopy(@PBuf,JustServer);
{$IFDEF STATICWINSOCK}
  ServerInAddr:=Winsock.Inet_Addr(@PBuf);
{$ELSE}
  ServerInAddr:=agWinsock.Inet_Addr(@PBuf);
{$ENDIF}
  if ServerInAddr=SOCKET_ERROR then
  begin
    if FUseBlockingSockets then
    begin
{$IFDEF STATICWINSOCK}
      RemoteHost:=Winsock.GetHostByName(@PBuf);
      if RemoteHost=nil then
        Error(Winsock.WSAGetLastError);
{$ELSE}
      RemoteHost:=agWinsock.GetHostByName(@PBuf);
      if RemoteHost=nil then
        Error(agWinsock.WSAGetLastError);
{$ENDIF}
    end
    else
    begin
{$IFDEF STATICWINSOCK}
      AsyncHandle:=Winsock.WSAAsyncGetHostByName(InvWnd,WM_HOSTFOUND,@PBuf,
                   @Buf,MAXGETHOSTSTRUCT);
{$ELSE}
      AsyncHandle:=agWinsock.WSAAsyncGetHostByName(InvWnd,WM_HOSTFOUND,@PBuf,
                   @Buf,MAXGETHOSTSTRUCT);
{$ENDIF}
      if AsyncHandle=0 then
        Error(ErrorNo);
      TC.TimerOn;
      repeat
        Application.ProcessMessages;
      until HostFound or TC.TimedOut or Canceled;
      TC.TimerOff;
      AsyncHandle:=0;
      if AsyncError then
        Error(ErrorNo)
      else
      if TC.TimedOut then TimeOutError
      else
      if Canceled then CanceledError;
      Application.ProcessMessages;
      RemoteHost:=PHostEnt(@Buf);
    end;
    for i:=0 to 3 do
      a[i]:=byte(RemoteHost^.h_addr_list^[i]);
    ServerIPAddr:=IntToStr(a[0])+'.'+IntToStr(a[1])+
      '.'+IntToStr(a[2])+'.'+IntToStr(a[3]);
    FillChar(PBuf,SizeOf(PBuf),0);
    StrPCopy(@PBuf,ServerIPAddr);
{$IFDEF STATICWINSOCK}
    ServerInAddr:=Winsock.Inet_Addr(@PBuf);
{$ELSE}
    ServerInAddr:=agWinsock.Inet_Addr(@PBuf);
{$ENDIF}
    if ServerInAddr=SOCKET_ERROR then
      Error(agSResolving);
  end;
end;

procedure TagSocket.GetLocalHost;
var
  Buf : PChar;
  rc : Integer;
begin
  Buf:=StrAlloc(512);
  try
{$IFDEF STATICWINSOCK}
    rc:=Winsock.gethostname(Buf,512);
{$ELSE}
    rc:=agWinsock.gethostname(Buf,512);
{$ENDIF}
    if rc=SOCKET_ERROR then
      LocalHost:=JustServer
    else
      LocalHost:=StrPas(Buf);
  finally
    StrDispose(Buf);
  end;
end;

procedure TagSocket.GetService;
{Need to handle this}
var
  Buf : array[0..MAXGETHOSTSTRUCT] of char;
  PSE : PServEnt;
  ProtoStr : array[0..3] of char;
  PBuf : array[0..63] of char;
begin
  if PickPort>0 then
  begin
    ThePort:=PickPort;  {if predefined port, then exit}
    Exit;
  end
  else
  if FServiceName='' then
    Error(agSServiceRequired)
  else
  if Canceled then
    CanceledError;
  if FProtocol=ptTCP then StrCopy(ProtoStr,'tcp')
    else StrCopy(ProtoStr,'udp');
  FillChar(PBuf,SizeOf(PBuf),0);
  StrPCopy(@PBuf,FServiceName);
  if FUseBlockingSockets then
  begin
{$IFDEF STATICWINSOCK}
    PSE:=Winsock.GetServByName(@PBuf,@ProtoStr);
    if PSE=nil then
      Error(Winsock.WSAGetLastError);
    ThePort:=Winsock.htons(PSE^.s_port);
{$ELSE}
    PSE:=agWinsock.GetServByName(@PBuf,@ProtoStr);
    if PSE=nil then
      Error(agWinsock.WSAGetLastError);
    ThePort:=agWinsock.htons(PSE^.s_port);
{$ENDIF}
  end
  else
  begin
{$IFDEF STATICWINSOCK}
    AsyncHandle:=Winsock.WSAAsyncGetServByName(InvWnd,WM_SERVICEFOUND,@PBuf,
                  @ProtoStr,@Buf,MAXGETHOSTSTRUCT);
{$ELSE}
    AsyncHandle:=agWinsock.WSAAsyncGetServByName(InvWnd,WM_SERVICEFOUND,@PBuf,
                  @ProtoStr,@Buf,MAXGETHOSTSTRUCT);
{$ENDIF}
    if AsyncHandle=0 then
      ThePort:=FDefaultPort
    else
    begin
      TC.TimerOn;
      repeat
        Application.ProcessMessages
      until ServiceFound or Canceled or TC.TimedOut;
      TC.TimerOff;
      AsyncHandle:=0;
      if AsyncError or TC.TimedOut then
      begin
        ThePort:=FDefaultPort;
      end
      else
      if Canceled then CanceledError
      else
      begin
        PSE:=PServEnt(@Buf);
{$IFDEF STATICWINSOCK}
        ThePort:=Winsock.htons(PSE^.s_port);
{$ELSE}
        ThePort:=agWinsock.htons(PSE^.s_port);
{$ENDIF}
      end;
    end;
  end;
end;

procedure TagSocket.OpenSocket;
var
  Proto : Integer;
begin
  if FProtocol=ptUDP then
    Proto:=IPPROTO_UDP
  else
    Proto:=IPPROTO_IP;
{$IFDEF STATICWINSOCK}
  TheSocket:=Winsock.Socket(PF_INET,SOCK_STREAM,Proto);
  if TheSocket=INVALID_SOCKET then
    Error(Winsock.WSAGetLastError);
{$ELSE}
  TheSocket:=agWinsock.Socket(PF_INET,SOCK_STREAM,Proto);
  if TheSocket=INVALID_SOCKET then
    Error(agWinsock.WSAGetLastError);
{$ENDIF}
end;

procedure TagSocket.Connect;
var
  RemoteAddress : TSockAddr;
  LastError : Integer;
begin
  if Canceled then CanceledError;
  with RemoteAddress do
  begin
    Sin_Family:=PF_INET;
{$IFDEF STATICWINSOCK}
    Sin_Port:=Winsock.htons(ThePort);
{$ELSE}
    Sin_Port:=agWinsock.htons(ThePort);
{$ENDIF}
    Sin_addr:=TInAddr(ServerInAddr);
  end;
  if not FUseBlockingSockets then
  begin
{$IFDEF STATICWINSOCK}
    AsyncHandle:=Winsock.WSAAsyncSelect(TheSocket,InvWnd,WM_SOCKETACTIVITY,
                                FD_CONNECT or FD_READ or FD_WRITE or FD_CLOSE);
    if AsyncHandle=SOCKET_ERROR then
      Error(Winsock.WSAGetLastError);
{$ELSE}
    AsyncHandle:=agWinsock.WSAAsyncSelect(TheSocket,InvWnd,WM_SOCKETACTIVITY,
                                FD_CONNECT or FD_READ or FD_WRITE or FD_CLOSE);
    if AsyncHandle=SOCKET_ERROR then
      Error(agWinsock.WSAGetLastError);
{$ENDIF}
    TC.TimerOn;
{$IFDEF STATICWINSOCK}
    if WinSock.Connect(TheSocket,RemoteAddress,
               SizeOf(RemoteAddress))=SOCKET_ERROR then
{$ELSE}
    if agWinSock.Connect(TheSocket,RemoteAddress,
               SizeOf(RemoteAddress))=SOCKET_ERROR then
{$ENDIF}
    begin
{$IFDEF STATICWINSOCK}
      LastError:=Winsock.WSAGetLastError;
{$ELSE}
      LastError:=agWinsock.WSAGetLastError;
{$ENDIF}
      if LastError<>WSAEWOULDBLOCK then
        Error(LastError);
    end;
    repeat
      Application.ProcessMessages
    until Connected or Canceled or TC.TimedOut;
    TC.TimerOff;
    if Canceled then
      CanceledError
    else
    if TC.TimedOut then
      TimeOutError;
  end
  else
  begin
{$IFDEF STATICWINSOCK}
    if Winsock.connect(TheSocket,RemoteAddress,
               SizeOf(RemoteAddress))=SOCKET_ERROR then
      Error(Winsock.WSAGetLastError);
{$ELSE}
    if agWinsock.connect(TheSocket,RemoteAddress,
               SizeOf(RemoteAddress))=SOCKET_ERROR then
      Error(agWinsock.WSAGetLastError);
{$ENDIF}
  end;
  DoConnected(Self);
end;

procedure TagSocket.Cancel;
begin
  Canceled:=true;
  DoCancel(Self);
  if AsyncHandle<>0 then
  begin
{$IFDEF STATICWINSOCK}
    Winsock.WSACancelAsyncRequest(AsyncHandle);
{$ELSE}
    agWinsock.WSACancelAsyncRequest(AsyncHandle);
{$ENDIF}
    AsyncHandle:=0;
  end;
  Close;
end;

initialization
{$IFDEF UNREGISTERED}
  Nag;
{$ENDIF}
end.
