(**********************************************)
(*  The Internet Mail Suite  1.9.2            *)
(*  (c) ArGo Software Design, 1996,1997,1998. *)
(**********************************************)
unit aghttp;

{$I msdef.inc}

{.$DEFINE UseFileStream}

interface

uses
{$IFDEF WIN32}
  Windows,
{$ELSE}
  WinProcs,
{$ENDIF}
  Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  agSocket, agConst, agTypes, msMIME, msUtils,
{$IFDEF UseFileStream}
  msCls,
{$ENDIF}
  msMsgCls;

type
  TagHTTP = class(TagSocket)
  private
    { Private declarations }
    FURL : string;
    FFileName : string;
    FOnSending : TNotifyEvent;
    FHeaders : TmsHeaders;
{$IFDEF UseFileStream}
    FInStream : TTempFileStream;
{$ELSE}
    FInStream : TMemoryStream;
{$ENDIF}
    FOutStream : TMemoryStream;
    FProxy : string;
    FProxyUserName : string;
    FProxyPassword : string;
    FPostContentType : string;
    FUserAgent : string;
    procedure SetURL(Value : string);
    procedure SetProxy(Value : string);
    procedure SetPostContentType(Value : string);
    procedure DoSending(Sender : TObject);
    function PickHost(const s: string): string;
  protected
    { Protected declarations }
    AddInfo : string;
    procedure StringToStream(Stream : TStream; const s : string);
    procedure SendQuery(const Query, Authorization : string);
    procedure SendPostQuery(const Authorization : string);
    function RecvHeaders : Integer;
    procedure RecvData;
    function Chunked : boolean;
  public
    { Public declarations }
    constructor Create(AOwner : TComponent); override;
    destructor Destroy; override;
    function Get : Integer;
    function GetAuthorized(const UserName, Password : string) : Integer;
    function Head : Integer;
    function HeadAuthorized(const UserName, Password : string) : Integer;
    function Post : Integer;
    function PostAuthorized(const UserName, Password : string) : Integer;
    property FileName : string read FFileName;
{$IFDEF UseFileStream}
    property InStream : TTempFileStream read FInStream write FInStream;
{$ELSE}
    property InStream : TMemoryStream read FInStream write FInStream;
{$ENDIF}
    property OutStream : TMemoryStream read FOutStream write FOutStream;
    property Headers : TmsHeaders read FHeaders;
    property PostContentType : string read FPostContentType write
                SetPostContentType;
  published
    { Published declarations }
    property URL : string read FURL write SetURL;
    property Proxy : string read FProxy write SetProxy;
    property ProxyUserName : string read FProxyUserName write FProxyUserName;
    property ProxyPassword : string read FProxyPassword write FProxyPassword;
    property UserAgent : string read FUserAgent write FUserAgent;
    property DefaultPort;
    property TimeOut;
    property OnOpen;
    property OnProgress;
    property OnClosing;
    property OnClosed;
    property OnCancel;
    property OnError;
    property OnSending : TNotifyEvent read FOnSending write FOnSending;
  end;

implementation

constructor TagHTTP.Create(AOwner : TComponent);
begin
  inherited Create(AOwner);
  DefaultPort:=80;
  ServiceName:='http';
  FHeaders:=TmsHeaders.Create;
  FPostContentType:='application/x-www-form-urlencoded';
{$IFDEF UseFileStream}
  FInStream:=TTempFileStream.Create;
{$ELSE}
  FInStream:=TMemoryStream.Create;
{$ENDIF}
  FOutStream:=TMemoryStream.Create;
end;

destructor TagHTTP.Destroy;
begin
  FOutStream.Free;
  FInStream.Free;
  FHeaders.Free;
  inherited Destroy;
end;

procedure TagHTTP.SetURL(Value : string);
var
  i : byte;
begin
  if UpperCase(Copy(Value,1,7))<>'HTTP://' then
    Error(agSInvalidURL);
  FURL:=Value;
  FFileName:='';
  Delete(Value,1,7);
  i:=Pos('/',Value);
  if i=0 then
  begin
    Server:=Value;
    AddInfo:='/';
  end
  else
  begin
    Server:=Copy(Value,1,i-1);
    Delete(Value,1,i-1);
    AddInfo:=Value;
    i:=Pos('.',Value);
    if i<>0 then
{if the filename is specified}
    begin
{find last '/'}
      i:=Length(Value);
      while (i>0) and (Value[i]<>'/') do Dec(i);
      if i=0 then
        Error(agSInvalidURL);
      FFileName:=Copy(Value,i+1,Length(Value)-i);
      if FFileName='' then FFileName:='index.html';
    end;
  end;
  if FProxy<>'' then
    SetProxy(FProxy);
end;

procedure TagHTTP.SetProxy(Value : string);
begin
  if (Pos(':',Value)=0) and (Value<>'') then
    Error(agSProxyPortRequired);
  FProxy:=Value;
  Server:=FProxy;
end;

procedure TagHTTP.SetPostContentType(Value : string);
begin
  Value:=LowerCase(Value);
  if (Value='application/x-www-form-urlencoded') or
     (Value='multipart/form-data') then
    FPostContentType:=Value
  else
    MessageDlg('Invalid content-type',mtError,[mbOk],0);
end;

procedure TagHTTP.DoSending(Sender : TObject);
begin
  if Assigned(FOnSending) then
    FOnSending(Sender);
end;

procedure TagHTTP.StringToStream(Stream : TStream; const s : string);
var
  Buf : PChar;
begin
  Buf:=StrAlloc(255);
  try
    StrPCopy(Buf,Concat(s,^M^J));
    Stream.Write(Buf^,StrLen(Buf));
  finally
    StrDispose(Buf);
  end;
end;

function TagHTTP.PickHost(const s: string) : string;
var
  s1: string;
  i: Integer;
begin
  s1:=Copy(s,8,Length(s));
  i:=Pos('/',s1);
  if i>0 then
    s1:=Copy(s1,1,i-1);
  Result:=s1;
end;

procedure TagHTTP.SendQuery(const Query, Authorization : string);
var
{$IFDEF WIN32}
  LineToSend : ShortString;
{$ELSE}
  LineToSend : string;
{$ENDIF}
  s : string;
  TempStream : TMemoryStream;
begin
  DoSending(Self);
  TempStream:=TMemoryStream.Create;
  try
    if FProxy<>'' then LineToSend:=FURL
      else LineToSend:=AddInfo;
    if LineToSend='' then LineToSend:='/';  {Request root}
    if (FProxy<>'') and (Pos('/',Copy(LineToSend,8,Length(LineToSend)))=0) then
      LineToSend:=Concat(LineToSend,'/');
    StringToStream(TempStream,Concat(UpperCase(Query),' ',LineToSend,' HTTP/1.0'));
    if FProxy='' then
      StringToStream(TempStream,Concat('Host: ',Server))
    else
      StringToStream(TempStream,Concat('Host: ',PickHost(LineToSend)));
    if Authorization<>'' then
    begin
      MakeTextData(Authorization,LineToSend);
      StringToStream(TempStream,'Authorization: Basic '+LineToSend);
    end;
    if FProxyUserName<>'' then
    begin
      s:=Concat(FProxyUserName,':',FProxyPassword);
      MakeTextData(s,LineToSend);
      StringToStream(TempStream,'Proxy-Authorization: Basic '+LineToSend);
    end;
    StringToStream(TempStream,'Accept: */*');
    if FUserAgent<>'' then
      StringToStream(TempStream,Concat('User-Agent: ',FUserAgent));
    StringToStream(TempStream,'Connection: close');
    StringToStream(TempStream,'');
    TempStream.Position:=0;
    SendStream(TempStream);
  finally
    TempStream.Free;
  end;
end;

function TagHTTP.RecvHeaders : Integer;
var
  s : string;
begin
  Result:=-1;
  FHeaders.Clear;
  repeat
    Application.ProcessMessages;
    {if not (ConnectionClosed and (s='')) then}
    begin
      RecvLine(s);
      FHeaders.Add(s);
    end;
  until (s='') {or (ConnectionClosed and (s=''))};
  if FHeaders.Count>0 then
    Result:=PickNumber(FHeaders[0],1);
end;

function TagHTTP.Chunked : boolean;
var
  s : string;
begin
  s:=LowerCase(Headers.GetFieldBody('Transfer-Encoding'));
  Result:=Pos('chunked',s)>0;
end;

procedure TagHTTP.RecvData;
var
  TempS : string;
  N : LongInt;
begin
{$IFDEF UseFileStream}
  FInStream.Position:=0;
{$IFDEF V10}
  FInStream.Size:=0;
{$ENDIF}
{$ELSE}
  FInStream.Clear;
{$ENDIF}
  if Chunked then  {New v1.7}
  begin
    repeat
      RecvLine(TempS);
      if TempS='' then
        RecvLine(TempS);
      TempS:=Concat('$',UpperCase(TempS));
      N:=StrToIntDef(Trim(TempS),-1);
      if N>0 then
        RecvStream(FInStream,N);
    until N<=0;
  end
  else
  begin
    N:=StrToIntDef(Headers.GetFieldBody('Content-Length'),-1);
    RecvStream(FInStream,N);
  end;
  FInStream.Position:=0;
end;

function TagHTTP.Get : Integer;
begin
  Reinit;
  Open;
  try
    Connect;
    SendQuery('GET','');
    Result:=RecvHeaders;
    RecvData;
  finally
    Close;
  end;
end;

function TagHTTP.HeadAuthorized(const UserName, Password : string) : Integer;
begin
  Reinit;
  Open;
  try
    Connect;
    SendQuery('GET',Concat(UserName,':',Password));
    Result:=RecvHeaders;
  finally
    Close;
  end;
end;

function TagHTTP.GetAuthorized(const UserName, Password : string) : Integer;
begin
  Reinit;
  Open;
  try
    Connect;
    SendQuery('GET',Concat(UserName,':',Password));
    Result:=RecvHeaders;
    RecvData;
  finally
    Close;
  end;
end;

function TagHTTP.Head : Integer;
begin
  Reinit;
  Open;
  try
    Connect;
    SendQuery('HEAD','');
    Result:=RecvHeaders;
  finally
    Close;
  end;
end;

procedure TagHTTP.SendPostQuery(const Authorization : string);
var
{$IFDEF WIN32}
  LineToSend : ShortString;
{$ELSE}
  LineToSend : string;
{$ENDIF}
  s : string;
  TempStream : TStream;
begin
  Connect;
  DoSending(Self);
  TempStream:=TMemoryStream.Create;
  try
    if FProxy<>'' then LineToSend:=FURL
      else LineToSend:=AddInfo;
    if LineToSend='' then LineToSend:='/';  {Request root}
    if (FProxy<>'') and (Pos('/',Copy(LineToSend,8,Length(LineToSend)))=0) then
      LineToSend:=Concat(LineToSend,'/');
    StringToStream(TempStream,Concat('POST ',LineToSend,' HTTP/1.1'));
    if FProxy='' then
      StringToStream(TempStream,Concat('Host: ',Server))
    else
      StringToStream(TempStream,Concat('Host: ',PickHost(LineToSend)));
    if Authorization<>'' then
    begin
      MakeTextData(Authorization,LineToSend);
      StringToStream(TempStream,'Authorization: Basic '+LineToSend);
    end;
    if FProxyUserName<>'' then
    begin
      s:=Concat(FProxyUserName,':',FProxyPassword);
      MakeTextData(s,LineToSend);
      StringToStream(TempStream,'Proxy-Authorization: Basic '+LineToSend);
    end;
    if FUserAgent<>'' then
      StringToStream(TempStream,Concat('User-Agent: ',FUserAgent));
    StringToStream(TempStream,'Connection: close');
  {PostContentType can be 'application/x-www-form-urlencoded' for simple
   forms or 'multipart/form-data' for complex forms, such as file
   uploads and so on.  See rfc1867.}
    StringToStream(TempStream,Concat('Content-Type: ',FPostContentType));
    StringToStream(TempStream,Concat('Content-Length: ',IntToStr(FOutStream.Size))+^M^J);
    TempStream.Position:=0;
    SendStream(TempStream);
  finally
    TempStream.Free;
  end;
end;

function TagHTTP.Post : Integer;
begin
  ReInit;
  Open;
  try
    SendPostQuery('');
    FOutStream.Position:=0;
    SendStream(FOutStream);
    Result:=RecvHeaders;
    if Result=100 then {if the reply is Continue}
    begin
      SendLine('');
      Result:=RecvHeaders;
    end;
    RecvData;
  finally
    Close;
  end;
end;

function TagHTTP.PostAuthorized(const UserName, Password: string) : Integer;
begin
  ReInit;
  Open;
  try
    SendPostQuery(Concat(UserName,':',Password));
    FOutStream.Position:=0;
    SendStream(FOutStream);
    Result:=RecvHeaders;
    if Result=100 then {if the reply is Continue}
    begin
      SendLine('');
      Result:=RecvHeaders;
    end;
    RecvData;
  finally
    Close;
  end;
end;

end.


