(**********************************************)
(*  The Internet Mail Suite  1.9.2            *)
(*  (c) ArGo Software Design, 1996,1997,1998. *)
(**********************************************)
unit agftp;

{$I msDef.Inc}

interface

uses
{$IFDEF WIN32}
  Windows,
{$IFDEF STATICWINSOCK}
  Winsock,
{$ELSE}
  agWsock,
{$ENDIF}
{$ELSE}
  WinTypes,
  WinProcs,
  agWSock,
{$ENDIF}
  Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  agsocket, agTypes, agFtpCls, agConst;

type

  TTransferType = (ftBinary, ftASCII);

  TagFTP = class(TagSocket)
  private
    { Private declarations }
    FDataSocket : TSocket;
    FListenerSocket : TSocket;
    FUserName : string;
    FPassword : string;
    FTransferType : TTransferType;
    FOnTransferStart : TNotifyEvent;
    FOnTransferProgress : TmsProgressEvent;
    FOnTransferEnd : TNotifyEvent;
    FTransferStream : TStream;
    FDirectory : TagFTPDirList;
    FTransferBufferSize : Cardinal;
    FRetrievingList : boolean;
    FTransferringFile : boolean;
    FUseProxy : boolean;
    FProxy : string;
    FProxyType : TFTPProxyType;
    function GetServerType : TServerType;
    procedure SetServerType(Value : TServerType);
    function GetCurrentDirectory : string;
    procedure SetCurrentDirectory(Value : string);
    procedure SetProxy(Value : string);
    procedure DoTransferStart(Sender : TObject);
    procedure DoTransferProgress(Sender : TObject; Perc : Integer;
       Transferred : LongInt);
    procedure DoTransferEnd(Sender : TObject);
  protected
    { Protected declarations }
    DataWndHandle : THandle;
    DataConnectionClosed : boolean;
    DataTransferCanceled : boolean;
    RealServer : string;
    procedure WndProc(var Msg : TMessage);
    procedure ReInit; override;
    procedure OpenListenerSocket;
    procedure CloseListenerSocket;
    procedure RequestDataConnection;
    procedure AcceptDataConnection;
    procedure CloseDataConnection;
    procedure RecvFTPStream(FullSize : LongInt);
    procedure SendFTPStream(FullSize : LongInt);
    procedure FillDirList;
  public
    { Public declarations }
    constructor Create(AOwner : TComponent); override;
    destructor Destroy; override;
    procedure Open; override;
    procedure OpenConnection;
    procedure CloseConnection;
    procedure LogIn;
    procedure LogOut;
    procedure RetrieveFile(const RemoteFilePath, LocalFilePath : string);
    procedure ResumeRetrieveFile(const RemoteFilePath, LocalFilePath : string;
       Marker : LongInt);
    procedure StoreFile(const LocalFilePath, RemoteFilePath : string);
    procedure ResumeStoreFile(const LocalFilePath, RemoteFilePath : string;
       Marker : LongInt);
    procedure AppendStoreFile(const LocalFilePath, RemoteFilePath : string);
    procedure StoreUnique(const LocalFilePath : string; var RemoteFilePath : string);
    procedure GetDirectory;
    procedure Delete_File(const FileName : string);
    procedure RenameFile(const OldFileName, NewFileName : string);
    procedure ChangeDirectory(const Path : string);
    procedure ChangeToUpperDirectory;
    procedure MakeDirectory(const Path : string);
    procedure DeleteDirectory(const Path : string);
    procedure AbortTransfer;
    property Directory : TagFTPDirList read FDirectory write FDirectory;
    property RetrievingList : boolean read FRetrievingList;
    property TransferringFile : boolean read FTransferringFile;
    property CurrentDirectory : string read GetCurrentDirectory write SetCurrentDirectory;
  published
    { Published declarations }
    property Server;
    property DefaultPort;
    property TimeOut;
    property OnOpen;
    property OnConnected;
    property OnClosing;
    property OnClosed;
    property OnCancel;
    property OnError;
    property OnLineSent;
    property OnLineReceived;
    property UserName : string read FUserName write FUserName;
    property Password : string read FPassword write FPassword;
    property ServerType : TServerType read GetServerType write SetServerType;
    property UseProxy : boolean read FUseProxy write FUseProxy default false;
    property Proxy : string read FProxy write SetProxy;
    property ProxyType : TFTPProxyType read FProxyType write FProxyType;
    property TrasferBufferSize : Cardinal read FTransferBufferSize write
                 FTransferBufferSize default 2048;
    property TransferType : TTransferType read FTransferType write FTransferType
                 default ftBinary;
    property OnTransferStart : TNotifyEvent read FOnTransferStart
                 write FOnTransferStart;
    property OnTransferProgress : TmsProgressEvent read FOnTransferProgress
                 write FOnTransferProgress;
    property OnTransferEnd : TNotifyEvent read FOnTransferEnd
                 write FOnTransferEnd;
  end;

implementation

uses
  msUtils;

constructor TagFTP.Create(AOwner : TComponent);
begin
  inherited Create(AOwner);
  ServiceName:='ftp';
  DefaultPort:=21;
  FUseProxy:=false;
  FTransferType:=ftBinary;
  FTransferBufferSize:=2048;
  FDirectory:=TagFTPDirList.Create;
end;

destructor TagFTP.Destroy;
begin
  FDirectory.Free;
  inherited Destroy;
end;

procedure TagFTP.DoTransferStart(Sender : TObject);
begin
  if Assigned(FOnTransferStart) then
    FOnTransferStart(Sender);
end;

procedure TagFTP.DoTransferProgress(Sender : TObject; Perc : Integer;
   Transferred : LongInt);
begin
  if Assigned(FOnTransferProgress) then
    FOnTransferProgress(Sender, Perc, Transferred);
end;

procedure TagFTP.DoTransferEnd(Sender : TObject);
begin
  if Assigned(FOnTransferEnd) then
    FOnTransferEnd(Sender);
end;

function TagFTP.GetServerType : TServerType;
begin
  Result:=FDirectory.ServerType;
end;

procedure TagFTP.SetServerType(Value : TServerType);
begin
  FDirectory.ServerType:=Value;
end;

procedure TagFTP.ReInit;
begin
  inherited ReInit;
  DataConnectionClosed:=false;
  DataTransferCanceled:=false;
  FTransferringFile:=false;
  FRetrievingList:=false;
end;

function TagFTP.GetCurrentDirectory : string;
var
  rs : Integer;
begin
  Result:='';
  if OnLine then
  begin
    SendLine('PWD');
    rs:=RecvLines;
    if (rs div 100)<>2 then
      raise EServerError.Create(TempS);
    Result:=TrimQuotes(PickString(TempS,1,true));
  end
end;

procedure TagFTP.SetCurrentDirectory(Value : string);
begin
  ChangeDirectory(Value);
end;

procedure TagFTP.SetProxy(Value : string);
begin
  if (Pos(':',Value)=0) and (Value<>'') then
    Error(agSProxyPortRequired);
  FProxy:=Value;
end;

procedure TagFTP.Open;
begin
  Verify;
  Reinit;
  inherited Open;
end;

procedure TagFTP.LogIn;
var
  rs : Integer;
begin
  Connect;
  rs:=RecvLines;
  if rs<>220 then
    raise EServerError.Create(TempS);
  if FUseProxy then
  begin
    if FProxyType=fpUserNoLogon then
      SendLine('USER '+FUserName+'@'+RealServer)
    else
    begin
      SendLine('OPEN '+RealServer);
      rs:=RecvLines;
      if rs<>220 then
        raise EServerError.Create(TempS);
      SendLine('USER '+FUserName);
    end;
  end
  else
    SendLine('USER '+FUserName);
  rs:=RecvLines;
  if (rs<>230) and (rs<>331) and (rs<>332) then
    raise EServerError.Create(TempS);
  SendLine('PASS '+FPassword);
  rs:=RecvLines;
  if (rs<>230) and (rs<>202) then
    raise EServerError.Create(TempS);
end;

procedure TagFTP.LogOut;
var
  rs : Integer;
begin
  SendLine('QUIT');
  rs:=RecvLines;
  if (rs div 100)<>2 then
    raise EServerError.Create(TempS);
end;

procedure TagFTP.OpenConnection;
begin
  if not OnLine then
  try
    if FUseProxy then
    begin
      RealServer:=Server;
      Server:=FProxy;
    end;
    Open;
    LogIn;
  finally
    if FUseProxy then
      Server:=RealServer;
  end;
end;

procedure TagFTP.CloseConnection;
begin
  if OnLine then
  begin
    LogOut;
    Close;
  end;
end;

procedure TagFTP.OpenListenerSocket;
var
  SockAddr : TSockAddr;
begin
{$IFDEF STATICWINSOCK}
  FListenerSocket:=Winsock.socket(PF_INET,SOCK_STREAM,IPPROTO_TCP);
{$ELSE}
  FListenerSocket:=agWinsock.socket(PF_INET,SOCK_STREAM,IPPROTO_TCP);
{$ENDIF}
  if FListenerSocket=INVALID_SOCKET then
{$IFDEF STATICWINSOCK}
    Error(Winsock.WSAGetLastError);
{$ELSE}
    Error(agWinsock.WSAGetLastError);
{$ENDIF}
  with SockAddr do
  begin
    sin_family:=AF_INET;
{$IFDEF STATICWINSOCK}
    sin_port:=Winsock.htons(0);  {Will return available port}
{$ELSE}
    sin_port:=agWinsock.htons(0);  {Will return available port}
{$ENDIF}
    sin_addr.s_addr:=INADDR_ANY;
  end;
{$IFDEF STATICWINSOCK}
  if Winsock.bind(FListenerSocket,SockAddr,SizeOf(SockAddr))=SOCKET_ERROR then
    Error(Winsock.WSAGetLastError);
  if Winsock.listen(FListenerSocket,SOMAXCONN)=SOCKET_ERROR then
    Error(Winsock.WSAGetLastError);
{$ELSE}
  if agWinsock.bind(FListenerSocket,SockAddr,SizeOf(SockAddr))=SOCKET_ERROR then
    Error(agWinsock.WSAGetLastError);
  if agWinsock.listen(FListenerSocket,SOMAXCONN)=SOCKET_ERROR then
    Error(agWinsock.WSAGetLastError);
{$ENDIF}
end;

procedure TagFTP.CloseListenerSocket;
begin
{$IFDEF STATICWINSOCK}
  if Winsock.CloseSocket(FListenerSocket)=SOCKET_ERROR then
    Error(Winsock.WSAGetLastError);
{$ELSE}
  if agWinsock.CloseSocket(FListenerSocket)=SOCKET_ERROR then
    Error(agWinsock.WSAGetLastError);
{$ENDIF}
end;

procedure TagFTP.WndProc(var Msg : TMessage);
begin
  with Msg do
  if Msg=WM_SOCKETACTIVITY then
  begin
{$IFDEF STATICWINSOCK}
    ErrorNo:=Winsock.WSAGetAsyncError(lParam);
{$ELSE}
    ErrorNo:=agWinsock.WSAGetAsyncError(lParam);
{$ENDIF}
    AsyncError:=ErrorNo<>0;
    if not AsyncError then
{$IFDEF STATICWINSOCK}
      if Winsock.WSAGetSelectEvent(lParam)=FD_CLOSE
{$ELSE}
      if agWinsock.WSAGetSelectEvent(lParam)=FD_CLOSE
{$ENDIF}
        then DataConnectionClosed:=true;
    Result:=0;
  end
  else
    Result:=DefWindowProc(DataWndHandle,Msg,wParam,lParam);
end;

procedure TagFTP.RequestDataConnection;
var
  SockAddr : TSockAddr;
  NameLen, rs : Integer;
  LocalPort : Integer;
  s : string;
begin
  NameLen:=SizeOf(SockAddr);
{$IFDEF STATICWINSOCK}
  if Winsock.getsockname(FListenerSocket,SockAddr,NameLen)=SOCKET_ERROR then
    Error(Winsock.WSAGetLastError);
{$ELSE}
  if agWinsock.getsockname(FListenerSocket,SockAddr,NameLen)=SOCKET_ERROR then
    Error(agWinsock.WSAGetLastError);
{$ENDIF}
  LocalPort:=SockAddr.sin_port;
{$IFDEF STATICWINSOCK}
  if Winsock.getsockname(TheSocket,SockAddr,NameLen)=SOCKET_ERROR then
    Error(Winsock.WSAGetLastError);
{$ELSE}
  if agWinsock.getsockname(TheSocket,SockAddr,NameLen)=SOCKET_ERROR then
    Error(agWinsock.WSAGetLastError);
{$ENDIF}
  with SockAddr.sin_addr.S_un_b do
    s:=Format('%d,%d,%d,%d,%d,%d',[byte(s_b1),byte(s_b2),byte(s_b3),byte(s_b4),
              LocalPort and $ff,LocalPort shr 8]);
  SendLine('PORT '+s);
  rs:=RecvLines;
  if (rs div 100)<>2 then
    raise EServerError.Create(TempS);
end;

procedure TagFTP.AcceptDataConnection;
var
  SockAddr : TSockAddr;
  NameLen : Integer;
begin
  NameLen:=SizeOf(SockAddr);
{$IFDEF STATICWINSOCK}
  FDataSocket:=Winsock.accept(FListenerSocket,@SockAddr,@NameLen);
  if FDataSocket=INVALID_SOCKET then
    Error(Winsock.WSAGetLastError);
{$ELSE}
  FDataSocket:=agWinsock.accept(FListenerSocket,SockAddr,NameLen);
  if FDataSocket=INVALID_SOCKET then
    Error(agWinsock.WSAGetLastError);
{$ENDIF}
  DataWndHandle:=AllocateHWnd(WndProc);
{$IFDEF STATICWINSOCK}
  if Winsock.WSAAsyncSelect(FDataSocket,DataWndHandle,WM_SOCKETACTIVITY,{FD_READ or} FD_CLOSE)=SOCKET_ERROR
    then Error(Winsock.WSAGetLastError);
{$ELSE}
  if agWinsock.WSAAsyncSelect(FDataSocket,DataWndHandle,WM_SOCKETACTIVITY,{FD_READ or} FD_CLOSE)=SOCKET_ERROR
    then Error(agWinsock.WSAGetLastError);
{$ENDIF}
end;

procedure TagFTP.CloseDataConnection;
begin
  DeallocateHWnd(DataWndHandle);
{$IFDEF STATICWINSOCK}
  Winsock.closesocket(FDataSocket);
{$ELSE}
  agWinsock.closesocket(FDataSocket);
{$ENDIF}
end;

procedure TagFTP.RecvFTPStream(FullSize : LongInt);
{Receives the stream via data socket.  Stops when connection closed}
var
  Finished : boolean;
  LastError : Integer;
  rc : Integer;
  Total : LongInt;
  Buf : PChar;
begin
  Buf:=StrAlloc(FTransferBufferSize+5);
  try
    DoTransferProgress(Self,0,0);
    Total:=0;
    if Canceled then
      CanceledError
    else
    if TC.TimedOut then
      TimeOutError
    else
    if AsyncError then
      Error(ErrorNo);
    DoTransferStart(Self);
    TC.TimerOn;
    repeat
      Application.ProcessMessages;
{$IFDEF STATICWINSOCK}
      rc:=Winsock.recv(FDataSocket,Buf^,FTransferBufferSize,0);
{$ELSE}
      rc:=agWinsock.recv(FDataSocket,Buf,FTransferBufferSize,0);
{$ENDIF}
      if rc=SOCKET_ERROR then
      begin
{$IFDEF STATICWINSOCK}
        LastError:=Winsock.WSAGetLastError;
{$ELSE}
        LastError:=agWinsock.WSAGetLastError;
{$ENDIF}
        if (LastError<>WSAEWOULDBLOCK) and not (Canceled or DataTransferCanceled) then
          Error(LastError)
      end
      else
      begin
        TC.TimerOn;
        Total:=Total+rc;
        FTransferStream.Write(Buf^,rc);
        if FullSize>0 then
          DoTransferProgress(Self,Round(100*Total/FullSize),Total)
        else
          DoTransferProgress(Self,-1,Total);
      end;
      Application.ProcessMessages;
      Finished:=(DataConnectionClosed and (rc=0));
    until Finished or Canceled or TC.TimedOut or DataTransferCanceled;
    TC.TimerOff;
    FTransferStream.Position:=0;
    if Finished then
      DoTransferEnd(Self)
    else
    if Canceled then
      CanceledError
    else
    if TC.TimedOut then
      TimeOutError;
    DataTransferCanceled:=false;
  finally
    StrDispose(Buf);
  end;
end;

procedure TagFTP.SendFTPStream(FullSize : LongInt);
{Sends a data via Data socket}
var
  Buf : PChar;
  rd,sd,i : Integer;
  Done : boolean;
  LastError : Integer;
  Total : LongInt;
begin
  DataTransferCanceled:=false;
  Buf:=StrAlloc(FTransferBufferSize+5);  {Add 5, just in case}
  try
    DoTransferStart(Self);
{    FTransferStream.Position:=0;}
    Total:=0;
    repeat
      FillChar(Buf^,StrBufSize(Buf),0);
      rd:=FTransferStream.Read(Buf^,FTransferBufferSize);
      i:=0;
      TC.TimerOn;
      repeat
{$IFDEF STATICWINSOCK}
        sd:=Winsock.Send(FDataSocket,(Buf+i)^,rd-i,0);
{$ELSE}
        sd:=agWinsock.Send(FDataSocket,Buf+i,rd-i,0);
{$ENDIF}
        if sd=SOCKET_ERROR then
        begin
{$IFDEF STATICWINSOCK}
          LastError:=Winsock.WSAGetLastError;
{$ELSE}
          LastError:=agWinsock.WSAGetLastError;
{$ENDIF}
          if (LastError<>WSAEWOULDBLOCK) and not (Canceled or DataTransferCanceled) then
            Error(LastError);
          sd:=0;
        end;
        if sd>0 then
        begin
          TC.TimerOn;
          Inc(i,sd);
          Inc(Total,sd);
        end;
        Done:=i>rd-1;
        Application.ProcessMessages;
      until Done or TC.TimedOut or Canceled or DataTransferCanceled;
      TC.TimerOff;
      Done:=FTransferStream.Position=FTransferStream.Size;
      if FullSize>0 then
        DoTransferProgress(Self,Round(100*Total/FullSize),Total)
      else
        DoTransferProgress(Self,-1,Total);
    until Done or TC.TimedOut or Canceled or DataTransferCanceled;
    if TC.TimedOut then
      TimeOutError
    else
    if Canceled then
      CanceledError;
    DataTransferCanceled:=false;
  finally
    DoTransferEnd(Self);
    StrDispose(Buf);
  end;
end;

procedure TagFTP.RetrieveFile(const RemoteFilePath, LocalFilePath : string);
var
  rs : Integer;
begin
  FTransferringFile:=true;
  FTransferStream:=TFileStream.Create(LocalFilePath,fmOpenWrite or fmCreate);
  try
    if FTransferType=ftBinary then
      SendLine('TYPE I')
    else
      SendLine('TYPE A');
    rs:=RecvLines;
    if (rs div 100)<>2 then
      raise EServerError.Create(TempS);
    OpenListenerSocket;
    try
      RequestDataConnection;
      SendLine('RETR '+RemoteFilePath);
      rs:=RecvLines;
      if (rs div 100)<>1 then
        raise EServerError.Create(TempS);
      AcceptDataConnection;
    finally
      CloseListenerSocket;
    end;
    RecvFTPStream(-1);
    CloseDataConnection;
    rs:=RecvLines;
    if (rs div 100)<>2 then
      raise EServerError.Create(TempS);
  finally
    FTransferStream.Free;
    FTransferringFile:=false;
  end;
end;

procedure TagFTP.ResumeRetrieveFile(const RemoteFilePath, LocalFilePath : string;
  Marker : LongInt);
var
  rs : Integer;
begin
  FTransferringFile:=true;
  FTransferStream:=TFileStream.Create(LocalFilePath,fmOpenWrite);
  FTransferStream.Seek(0,soFromEnd);
  try
    if FTransferType=ftBinary then
      SendLine('TYPE I')
    else
      SendLine('TYPE A');
    rs:=RecvLines;
    if (rs div 100)<>2 then
      raise EServerError.Create(TempS);
    OpenListenerSocket;
    try
      RequestDataConnection;
      SendLine('REST '+IntToStr(Marker));
      rs:=RecvLines;
      if (rs div 100)<>3 then
        raise EServerError.Create(TempS);
      SendLine('RETR '+RemoteFilePath);
      rs:=RecvLines;
      if (rs div 100)<>1 then
        raise EServerError.Create(TempS);
      AcceptDataConnection;
    finally
      CloseListenerSocket;
    end;
    RecvFTPStream(-1);
    CloseDataConnection;
    rs:=RecvLines;
    if (rs div 100)<>2 then
      raise EServerError.Create(TempS);
  finally
    FTransferStream.Free;
    FTransferringFile:=false;
  end;
end;

procedure TagFTP.StoreFile(const LocalFilePath, RemoteFilePath : string);
var
  rs : Integer;
begin
  FTransferringFile:=true;
  FTransferStream:=TFileStream.Create(LocalFilePath,fmOpenRead);
  try
    if FTransferType=ftBinary then
      SendLine('TYPE I')
    else
      SendLine('TYPE A');
    rs:=RecvLines;
    if (rs div 100)<>2 then
      raise EServerError.Create(TempS);
    OpenListenerSocket;
    try
      RequestDataConnection;
      SendLine('STOR '+RemoteFilePath);
      rs:=RecvLines;
      if (rs div 100)<>1 then
        raise EServerError.Create(TempS);
      AcceptDataConnection;
    finally
      CloseListenerSocket;
    end;
    FTransferStream.Position:=0;
    SendFTPStream(FTransferStream.Size);
    CloseDataConnection;
    rs:=RecvLines;
    if (rs div 100)<>2 then
      raise EServerError.Create(TempS);
  finally
    FTransferStream.Free;
    FTransferringFile:=false;
  end;
end;

procedure TagFTP.ResumeStoreFile(const LocalFilePath, RemoteFilePath : string;
  Marker : LongInt);
var
  rs : Integer;
begin
  FTransferringFile:=true;
  FTransferStream:=TFileStream.Create(LocalFilePath,fmOpenRead);
{  FTransferStream.Seek(soFromBeginning, Marker);}
  FTransferStream.Position:=Marker;
  try
    if FTransferType=ftBinary then
      SendLine('TYPE I')
    else
      SendLine('TYPE A');
    rs:=RecvLines;
    if (rs div 100)<>2 then
      raise EServerError.Create(TempS);
    OpenListenerSocket;
    try
      RequestDataConnection;
      SendLine('REST '+IntToStr(Marker));
      rs:=RecvLines;
      if (rs div 100)<>3 then
        raise EServerError.Create(TempS);
      SendLine('STOR '+RemoteFilePath);
      rs:=RecvLines;
      if (rs div 100)<>1 then
        raise EServerError.Create(TempS);
      AcceptDataConnection;
    finally
      CloseListenerSocket;
    end;
    SendFTPStream(FTransferStream.Size-Marker);
    CloseDataConnection;
    rs:=RecvLines;
    if (rs div 100)<>2 then
      raise EServerError.Create(TempS);
  finally
    FTransferStream.Free;
    FTransferringFile:=false;
  end;
end;

procedure TagFTP.AppendStoreFile(const LocalFilePath, RemoteFilePath : string);
var
  rs : Integer;
begin
  FTransferringFile:=true;
  FTransferStream:=TFileStream.Create(LocalFilePath,fmOpenRead);
  try
    if FTransferType=ftBinary then
      SendLine('TYPE I')
    else
      SendLine('TYPE A');
    rs:=RecvLines;
    if (rs div 100)<>2 then
      raise EServerError.Create(TempS);
    OpenListenerSocket;
    try
      RequestDataConnection;
      SendLine('APPE '+RemoteFilePath);
      rs:=RecvLines;
      if (rs div 100)<>1 then
        raise EServerError.Create(TempS);
      AcceptDataConnection;
    finally
      CloseListenerSocket;
    end;
    FTransferStream.Position:=0;
    SendFTPStream(FTransferStream.Size);
    CloseDataConnection;
    rs:=RecvLines;
    if (rs div 100)<>2 then
      raise EServerError.Create(TempS);
  finally
    FTransferStream.Free;
    FTransferringFile:=false;
  end;
end;

procedure TagFTP.StoreUnique(const LocalFilePath : string; var RemoteFilePath : string);
var
  rs : Integer;
begin
  FTransferringFile:=true;
  FTransferStream:=TFileStream.Create(LocalFilePath,fmOpenRead);
  try
    if FTransferType=ftBinary then
      SendLine('TYPE I')
    else
      SendLine('TYPE A');
    rs:=RecvLines;
    if (rs div 100)<>2 then
      raise EServerError.Create(TempS);
    OpenListenerSocket;
    try
      RequestDataConnection;
      SendLine('STOU');
      rs:=RecvLines;
      if (rs div 100)<>1 then
        raise EServerError.Create(TempS);
      RemoteFilePath:=PickString(TempS,7,true);
      if Pos('.',RemoteFilePath)=Length(RemoteFilePath)
        then Delete(RemoteFilePath,Length(RemoteFilePath),1);
      AcceptDataConnection;
    finally
      CloseListenerSocket;
    end;
    FTransferStream.Position:=0;
    SendFTPStream(FTransferStream.Size);
    CloseDataConnection;
    rs:=RecvLines;
    if (rs div 100)<>2 then
      raise EServerError.Create(TempS);
  finally
    FTransferStream.Free;
    FTransferringFile:=false;
  end;
end;

procedure TagFTP.FillDirList;
var
  SL : TStrings;
  s : string;
  i : Integer;
begin
  SL:=TStringList.Create;
  try
    SL.LoadFromStream(FTransferStream);
    TrimStrings(SL);
    FDirectory.Clear;
    if SL.Count>0 then
    begin
      s:=SL[0];
      if LowerCase(Copy(s,1,5))='total' then
      begin
        SL.Delete(0);
        FDirectory.ServerType:=stUnix;
      end;
      if FDirectory.ServerType=stAuto then
      begin
        if (s[1] in ['0'..'9']) then
        begin
          FDirectory.ServerType:=stDos;
        end
        else
        if (Pos('PATH',s)>0) then
        begin
          FDirectory.ServerType:=stHP3000;
          repeat
            SL.Delete(0);
            if SL.Count>0 then
              s:=SL[0];
          until (SL.Count=0) or ((Length(s)>16) and (s[16] in ['F','D']));
          if SL.Count>0 then
            SL.Insert(0,'');
        end
        else
        if (Copy(s,41,4)='*FLR') or (Copy(s,41,4)='*DOC')
           or (Copy(s,41,4)='*DIR') or (Copy(s,41,4)='*LIB')
           or (Copy(s,41,5)='*DDIR') then
        begin
          FDirectory.ServerType:=stAS400;
          if SL.Count>0 then
            SL.Insert(0,'');
        end
        else
          FDirectory.ServerType:=stUnix;
      end;
    end;
    for i:=0 to SL.Count-1 do
      FDirectory.AddString(SL[i]);
  finally
    SL.Free;
  end;
end;

procedure TagFTP.GetDirectory;
var
  rs : Integer;
begin
  FRetrievingList:=true;
  FTransferStream:=TMemoryStream.Create;
  try
    SendLine('TYPE A');
    rs:=RecvLines;
    if (rs div 100)<>2 then
      raise EServerError.Create(TempS);
    OpenListenerSocket;
    try
      RequestDataConnection;
      SendLine('LIST');
      rs:=RecvLines;
      if (rs div 100)<>1 then
        raise EServerError.Create(TempS);
      AcceptDataConnection;
    finally
      CloseListenerSocket;
    end;
    RecvFTPStream(-1);
    CloseDataConnection;
    rs:=RecvLines;
    if (rs div 100)<>2 then
      raise EServerError.Create(TempS);
    FillDirList;
  finally
    FTransferStream.Free;
    FRetrievingList:=false;
  end;
end;

procedure TagFTP.Delete_File(const FileName : string);
var
  rs : Integer;
begin
  SendLine('DELE '+FileName);
  rs:=RecvLines;
  if (rs div 100)<>2 then
    raise EServerError.Create(TempS);
end;

procedure TagFTP.RenameFile(const OldFileName, NewFileName : string);
var
  rs : Integer;
begin
  SendLine('RNFR '+OldFileName);
  rs:=RecvLines;
  if (rs div 100)<>3 then
    raise EServerError.Create(TempS);
  SendLine('RNTO '+NewFileName);
  rs:=RecvLines;
  if (rs div 100)<>2 then
    raise EServerError.Create(TempS);
end;

procedure TagFTP.ChangeDirectory(const Path : string);
var
  rs : Integer;
begin
  SendLine('CWD '+Path);
  rs:=RecvLines;
  if (rs div 100)<>2 then
    raise EServerError.Create(TempS);
end;

procedure TagFTP.ChangeToUpperDirectory;
var
  rs : Integer;
begin
  SendLine('CDUP');
  rs:=RecvLines;
  if (rs div 100)<>2 then
    raise EServerError.Create(TempS);
end;

procedure TagFTP.MakeDirectory(const Path : string);
var
  rs : Integer;
begin
  SendLine('MKD '+Path);
  rs:=RecvLines;
  if (rs div 100)<>2 then
    raise EServerError.Create(TempS);
end;

procedure TagFTP.DeleteDirectory(const Path : string);
var
  rs : Integer;
begin
  SendLine('RMD '+Path);
  rs:=RecvLines;
  if (rs div 100)<>2 then
    raise EServerError.Create(TempS);
end;

procedure TagFTP.AbortTransfer;
var
  rs : Integer;
begin
  DataTransferCanceled:=true;
  try
    Flags:=MSG_OOB;  {Out of bound data}
    SendLine('ABOR');
    Flags:=0;
    rs:=RecvLines;
    if (rs div 100)<>2 then
      raise EServerError.Create(TempS);
  finally
    FTransferringFile:=false;
  end;
end;

end.
