/*

edd.c

Author: Antti Huima <huima@ssh.fi>

Copyright (c) 1996 SSH Communications Security Oy, Espoo, Finland <info@ssh.fi>
                   All rights reserved

  Encrypting Data Dumper.
*/

/*
 * $Id: edd.c,v 1.3 1997/05/08 03:00:05 kivinen Exp $
 * $Log: edd.c,v $
 * Revision 1.3  1997/05/08 03:00:05  kivinen
 * 	Added cvs log. Removed O_EXLOCK and O_SHLOCK options from
 * 	open.
 *
 * $Endlog$
 */

#include "includes.h"
#include "ssh.h"
#include "cipher.h"
#include "buffer.h"
#include "bufaux.h"
#include "gmp.h"
#include "edd.h"
#include "xmalloc.h"
#include "randoms.h"
#include "userfile.h"
#include "mpaux.h"

/* Some definitions. */

#define ENCRYPT 0
#define DECRYPT 1
#define QUERY 2

#ifndef TRUE
#define TRUE 1
#endif

#ifndef FALSE
#define FALSE 0
#endif

#ifndef STDIN_FILENO
#define STDIN_FILENO 0
#endif

#ifndef STDOUT_FILENO
#define STDOUT_FILENO 1
#endif

#ifndef STDERR_FILENO
#define STDERR_FILENO 2
#endif

/* Usage */
#define USAGE_REPORT \
"edd version " EDD_VERSION "\n\
Encryption: edd [options] [bits exp mod [comment]] [keyfile]\n\
Decryption: edd [options] [keyfile]\n\
Edd info:   edd -I\n\
Query:      edd [options] -q\n\
Options:\n\
            [-B] [-C comment] [-c cipher] [-d] [-D diagnostics_flags]\n\
            [-e] [-f input_file] [-F magical_input_file]\n\
            [-o output_file] [-P passphrase] [-s block_size]\n\
Diagnostics flags:\n\
            c, e, f, n, t or all. Use q for none.\n\
"

/* Typedef of structure containing the fixed-length header parsed */
typedef struct
{
  char *version_string;
  int version, flags, cipher;
  int bits;
  MP_INT *exponent, *modulus;
  char *RSA_key_comment, *file_comment;
  MP_INT *encrypted_key;
} Header;

typedef int mode_of_operation;
typedef int bool;

/***************** Global variables. **********************/

uid_t original_real_uid = 0;

char *cp                 = NULL;   /* current program  */
char *av0                = NULL;   /* argv[0]          */

char *output_file        = NULL;
char *input_file         = NULL;

/* String indicating which diagnostics messages should be printed to
   stderr. Print fatal errors (f) and "normal" messages (n) by default. */
char *diagnostics        = DEFAULT_DIAGNOSTICS;

/* Flags which the user has already given. This array may be fixed-length
   because only flags understood will get to the array and no flag will
   be there twice. */
char given_flags[30];

/* Encrypt by default (use -d to decrypt) */
mode_of_operation mode   = ENCRYPT;

/* Was mode explicitly set? */
bool forced_mode         = FALSE;

/* Passphrase will not be asked if in batch mode */
bool batch_mode          = FALSE;

/* Default input/output block size */
int requested_block_size = DEFAULT_BLOCK_SIZE;

/* Some ciphers want to crypt only quantized length data */
int block_threshold      = 8;

/* Cipher is IDEA by default; BlowFish if IDEA isn't allowed */
#ifndef WITHOUT_IDEA
int cipher               = SSH_CIPHER_IDEA;
#else
int cipher               = SSH_CIPHER_BLOWFISH;
#endif

/* User's uid and password entry */
uid_t my_uid;
struct passwd *pw;

/* Session key, for either encrypting or decrypting */
unsigned char session_key[SSH_SESSION_KEY_LENGTH];

/* Random state */
RandomState state;

/* file descriptors */
int input_stream = STDIN_FILENO;
int output_stream = STDOUT_FILENO;

char *block = 0;
char *cursor;

int block_size = 0;
int block_length = 0;

/******************* Diagnostics **********************/

#define APPEND_NEWLINE 1

/* Diagnostics printing function used indirectly. */
void vdiagnose(char flag, char *label, const char *fmt,
	       va_list args, int flags)
{
  static bool new_line = TRUE;
  if (strchr(diagnostics, flag))
    {
      if (new_line)
	{
	  fprintf(stderr, "%s %s: ", av0, label);
	  new_line = FALSE;
	}
      vfprintf(stderr, fmt, args);
      if (flags & APPEND_NEWLINE)
	{
	  fprintf(stderr, "\n");
	  new_line = TRUE;
	}
      if (fmt[strlen(fmt) - 1] == '\n') new_line = TRUE;
      va_end(args);
    }
  if(flag == 'f') exit(255);
}

/* Macro for helping to make some functions. */
#define REPLACE_WITH_DIAGNOSE(name,label,flag) \
void name##(const char *fmt, ...) \
{ \
  va_list args; \
  va_start(args, fmt); \
  vdiagnose(flag, label, fmt, args, APPEND_NEWLINE); \
  va_end(args); \
}

/* Macro for helping to make some functions. */
#define REPLACE_WITH_DNO_NEWLINE(name,label,flag) \
void name##(const char *fmt, ...) \
{ \
  va_list args; \
  va_start(args, fmt); \
  vdiagnose(flag, label, fmt, args, 0); \
  va_end(args); \
}

/* Replacements for functions in log-client.c. These use
   vdiagnose function. */
REPLACE_WITH_DIAGNOSE (log_msg, "log", 'n'); 
REPLACE_WITH_DIAGNOSE (debug  , "debug", 't');
REPLACE_WITH_DIAGNOSE (fatal  , "fatal", 'f');
REPLACE_WITH_DIAGNOSE (error  , "error", 'e');

REPLACE_WITH_DNO_NEWLINE (nlog_msg, "log", 'n');
REPLACE_WITH_DNO_NEWLINE (ndebug  , "debug", 't');
REPLACE_WITH_DNO_NEWLINE (nfatal  , "fatal", 'f');
REPLACE_WITH_DNO_NEWLINE (nerror  , "error", 'e');
REPLACE_WITH_DNO_NEWLINE (print_comment, "comment", 'c');


/**************** Block handling ********************/

/* Allocate memory for block */
int allocate_block(int size)
{
  if (block) xfree(block);
  block = xmalloc(size);
  cursor = block;
  block_size = size;
  return 0;
}

int free_block()
{
  if (block) xfree(block);
  return 0;
}

/* Read data from fdes to block, return amount of characters read
   or negative if block is already full */
int read_into_block(int fdes)
{
  int read_bytes;
  int room = block_size - (((int) cursor)  - ((int) block));

  if (room < 1) return -1;

  read_bytes = read(fdes, cursor, room);
  if (read_bytes < 0)
    {
      nfatal( "Input stream corrupted\n");
      exit(1);
    }

  cursor += read_bytes;
  block_length += read_bytes;
  return read_bytes;
}

/* Write to fdes at most length bytes from the beginning of the block.
   Return number of bytes written. */
int write_from_block(int fdes, int length)
{
  int status;

  if(block_length < length)
    {
      nfatal("Out of characters (internal error)\n");
    }

  status = write(fdes, block, length);

  if(status < 0)
    {
      nfatal( "Output stream corrupted\n");
    }

  block_length -= status;

  return status;
}

/* Shift remaining data in the block to the beginning of the block. */
int shift_block()
{
  if(block_length) 
    {
      memmove(block, (cursor - block_length), block_length);
    }
  cursor = block + block_length;
  return(0);
}

/**************** Stream handling *******************/

/* Open the wanted output stream. Do nothing if output_file == NULL.
   (output is then to stdout) */
int open_output_stream()
{
  FILE *file;
  /* buf is better to be static because we might just set
     output_file = buf */
  static char buf[200];

  /* If -F was given and the output file name couldn't be
     guessed, ask for one. */
  if (output_file && (output_file == input_file))
    {
      if (batch_mode)
	nfatal("You can't supply an output file in batch mode, "
	       "bailing out.\n");
      nlog_msg("You must supply an output file name:\n");
      
      file = fopen("/dev/tty", "r");
      if (!file)
	nfatal("No controlling tty.\n");
      
      if (!fgets(buf, sizeof(buf), file))
	nfatal("End of file, exiting.\n");
     
      /* Remove the newline character. */
      if (strchr(buf, '\n'))
	*strchr(buf, '\n') = '\0';

      output_file = buf;
    }
  
  if (output_file)
    {
      output_stream = open(output_file, O_WRONLY | O_CREAT |
			   O_TRUNC | O_NONBLOCK,
			   0666);
      if(output_stream < 0)
	{
	  nfatal("Cannot open output file %s.\n", output_file);
	}    
    }
}

/* Generate a new session key to session_key[]. Random state is
   initialized the same way it's done in ssh. */
int generate_session_key()
{
  int i;
  struct stat st;
  char buf[1024];

  /* Initialize the random number generator. */
  sprintf(buf, "%.500s/%.200s", pw->pw_dir, SSH_CLIENT_SEEDFILE);
  if (userfile_stat(pw->pw_uid, buf, &st) < 0)
    nlog_msg("Creating random seed file ~/%.200s.  This may take a while.", 
	 SSH_CLIENT_SEEDFILE);
  else
    ndebug("Initializing random; seed file %.900s\n", buf);
  random_initialize(&state, pw->pw_uid, buf);
  
  /* Generate an encryption key for the session.   The key is a 256 bit
     random number, interpreted as a 32-byte key, with the least significant
     8 bits being the first byte of the key. */
  for (i = 0; i < SSH_SESSION_KEY_LENGTH; i++)
    session_key[i] = random_get_byte(&state);
  
  /* Save the new random state. */
  random_save(&state, pw->pw_uid, buf);

  /* Everything went all right. */
  return 0;
}

/* Read edd header from input and parse it to struct which is
   pointed by header_ptr. */
int read_header(Header *header_ptr)
{
  Buffer buffer; int length;
  char read_buf[HEADER_LENGTH];
  buffer_init(&buffer);

  ndebug("Reading header... ");

  /* Read the whole header to read_buf */
  length = read(input_stream, read_buf, HEADER_LENGTH);

  if (length < 0)
    nfatal("Cannot read input stream.\n");

  if (length < HEADER_LENGTH)
    nfatal("Cannot read enough data (input is not an edd file).\n");
  
  if (strncmp(read_buf, VERSION_HEADER, VERSION_HEADER_LENGTH))
    nfatal("Edd version mismatch or non-edd input.\n");
  
  ndebug("parsing... ");

  /* Append read_buf's contents to an "ssh" buffer, as to enable parsing. */
  buffer_append(&buffer, read_buf, HEADER_LENGTH);

  /* Initialize MP integers before using them. */
  mpz_init(header_ptr->exponent        = xmalloc(sizeof(MP_INT)));
  mpz_init(header_ptr->modulus         = xmalloc(sizeof(MP_INT)));
  mpz_init(header_ptr->encrypted_key   = xmalloc(sizeof(MP_INT)));

  /* Transfer (valid) version header from the beginning of the buffer
     to the header. */
  buffer_get(&buffer, read_buf, VERSION_HEADER_LENGTH);
  header_ptr->version_string           = strdup(read_buf);

  /* Read the rest of the header. */
  header_ptr->version                  = buffer_get_int(&buffer);
  header_ptr->flags                    = buffer_get_int(&buffer);
  header_ptr->cipher                   = buffer_get_int(&buffer);

  header_ptr->bits                     = buffer_get_int(&buffer);
  buffer_get_mp_int(&buffer, header_ptr->exponent);
  buffer_get_mp_int(&buffer, header_ptr->modulus);

  header_ptr->RSA_key_comment          = buffer_get_string(&buffer, NULL);
  header_ptr->file_comment             = buffer_get_string(&buffer, NULL);
  buffer_get_mp_int(&buffer, header_ptr->encrypted_key);

  /* Everything is read -- the buffer may be destroyed. */
  buffer_free(&buffer);

  ndebug("done.\n");

  /* Everything went all right. */
  return 0;
}

/* Write header from a header struct to the output stream. */
int write_header(Header *header_ptr)
{
  Buffer buffer; int len;

  /* Initialize a "ssh" buffer. */
  buffer_init(&buffer);

  ndebug("Writing header: packing... ");

  /* Write header to the buffer. */
  buffer_append    (&buffer, VERSION_HEADER, VERSION_HEADER_LENGTH);
  buffer_put_int   (&buffer, header_ptr->version);
  buffer_put_int   (&buffer, header_ptr->flags);
  buffer_put_int   (&buffer, header_ptr->cipher);
  buffer_put_int   (&buffer, header_ptr->bits);
  buffer_put_mp_int(&buffer, header_ptr->exponent);
  buffer_put_mp_int(&buffer, header_ptr->modulus);

  len = strlen(header_ptr->RSA_key_comment);
  if (len > KEY_COMMENT_MAX_LENGTH)
    {
      nerror("Key comment too long, truncated.\n");
      len = KEY_COMMENT_MAX_LENGTH;
    }
  buffer_put_string(&buffer, header_ptr->RSA_key_comment, len);

  len = strlen(header_ptr->file_comment);
  if (len > STREAM_COMMENT_MAX_LENGTH)
    {
      nerror("Stream coment too long, truncated.\n");
    }
  buffer_put_string(&buffer, header_ptr->file_comment, len);
  buffer_put_mp_int(&buffer, header_ptr->encrypted_key);

  ndebug("writing... ");

  if (buffer_len(&buffer) > HEADER_LENGTH)
    {
      nfatal("Header doesn't fit (fatal error), bailing out.\n");
    }
  
  /* Buffer is probably less than HEADER_LENGTH bytes long.
     Pad it with zero bytes for maximum compatibility with
     future versions. */
  while (buffer_len(&buffer) < HEADER_LENGTH)
    buffer_append(&buffer, "\0\0\0\0\0", 5);

  /* Write the buffer out. */
  write(output_stream, buffer_ptr(&buffer), HEADER_LENGTH);

  /* It may now be disposed. */
  buffer_free(&buffer);

  ndebug("done.\n");

  /* Everything went all right. */
  return 0;
}

/* Clear header data structures.
   This should be called only after when header is read
   from input stream with read_header. */

void clear_header(Header *header)
{
  mpz_clear(header->exponent);
  mpz_clear(header->modulus);
  mpz_clear(header->encrypted_key);
  xfree(header->version_string);
  xfree(header->RSA_key_comment);
  xfree(header->file_comment);
}

/* Read an edd header from the input stream and print it's interesting
   contents out. */
void query()
{
  Header header;
  char *mod, *exp;
  
  read_header(&header);
  
  exp = mpz_get_str(NULL, 10, header.exponent);
  mod = mpz_get_str(NULL, 10, header.modulus);

  printf("Version: %s\n",header.version_string);

  printf("RSA key:\n");

  printf(" Exponent: %s\n Modulus: %s\n", exp, mod);

  if(header.RSA_key_comment[0])
    printf(" Comment: %s\n", header.RSA_key_comment);

  if(header.file_comment[0])
    printf("Stream comment:\n %s\n",header.file_comment);

  xfree(exp);
  xfree(mod);
  clear_header(&header);
}

/* View some (non)interesting info about the current edd software. */
void edd_info()
{
  int i = 0;
  unsigned int mask = cipher_mask();
  
  printf("Edd version " EDD_VERSION "\n"
	 "Data quantization:  %d bytes\n"
	 "Default block size: %d bytes\n"
	 "Block size between  %d and %d bytes\n"
	 "Default cipher:     %s\n",
	 block_threshold,
	 requested_block_size,
	 MIN_BLOCK_SIZE,
	 MAX_BLOCK_SIZE,
	 cipher_name(cipher));
  
  /* Seek for supported ciphers. */
  printf("Supported ciphers:  ");
  while (mask) {
    if (mask & 1)
      {
	printf("%s ",cipher_name(i));
      }
    mask >>= 1;
    i++;
  }
  printf("\n"
	 "Header size:        %d bytes\n"
	 "Max key comment:    %d bytes long\n"
	 "Max stream comment: %d bytes long\n",
	 HEADER_LENGTH,
	 KEY_COMMENT_MAX_LENGTH,
	 STREAM_COMMENT_MAX_LENGTH);
}
 
void parse_public_key(char *bits, char *exp, char *mod,
		      RSAPublicKey *public_key)
{
  public_key->bits = atoi(bits);

  if(!public_key->bits)
    nfatal("Bad bits count for public key: %s\n", bits);

  mpz_init(&public_key->e);
  mpz_init(&public_key->n);

  if(mpz_set_str(&public_key->e, exp, 10) ||
     mpz_set_str(&public_key->n, mod, 10))
    nfatal("Bad public key data:\n%s\n%s\n", exp, mod);
}

/* Encrypt the input stream with a random session key which will
   be encrypted with an RSA key, composed of bits, exp and mod.
   key_comment and comment will be appended to the header. */
void encrypt_stream(RSAPublicKey *public_key, char *key_comment,
		    char *comment)
{
  MP_INT session_key_mp_int;
  int read_bytes, consume, padding;
  unsigned char export_padding;
  CipherContext context;
  Header header;

  ndebug("I will encrypt this stream with %s.\n",
	 cipher_name(cipher));

  mpz_init(&session_key_mp_int);

  ndebug("Generating session key.\n");

  generate_session_key();

  mp_unlinearize_msb_first(&session_key_mp_int,
			   session_key, SSH_SESSION_KEY_LENGTH);

  ndebug("Encrypting session key... ");
  rsa_public_encrypt(&session_key_mp_int,
		     &session_key_mp_int, public_key, &state);
  ndebug("done.\n");

  /* Write header */
  header.version_string = VERSION_HEADER;
  header.version = VERSION_NUMBER;
  header.flags = 0;
  header.cipher = cipher;
  header.bits = public_key->bits;
  header.exponent = &public_key->e;
  header.modulus = &public_key->n;
  header.RSA_key_comment = key_comment;
  header.file_comment = comment;
  header.encrypted_key = &session_key_mp_int;

  /* Open and write the header. */
  open_output_stream();

  write_header(&header);

  /* Initialize input/output block */
  allocate_block(requested_block_size);

  cipher_set_key(&context, cipher, session_key,
		 SSH_SESSION_KEY_LENGTH, 1);

  /* Original session key is no longer needed */
  memset(session_key, 0, SSH_SESSION_KEY_LENGTH *
	 sizeof(unsigned char));

  /* encryption loop */
  read_bytes = 1;
  ndebug("Entering encryption loop.\n");
  while (read_bytes)
    {
      read_bytes = read_into_block(input_stream);
      if (block_length >= block_threshold)
	{
	  consume = block_length - (block_length % block_threshold);
	  cipher_encrypt(&context, block, block, consume);
	  write_from_block(output_stream, consume);
	  shift_block();
	}
    }

  /* Perform padding if needed */
  padding = (block_threshold - block_length) % block_threshold;
  if (padding)
    {
      cipher_encrypt(&context, block, block, block_threshold);
      block_length += padding;
      write_from_block(output_stream, block_threshold);
    }

  /* Write out the amount of padding */
  export_padding = (unsigned char) padding;
  write(output_stream, &export_padding, sizeof(char));
  
  ndebug("Stream encrypted.\n");
  
  assert(block_length == 0);
  
  free_block();
  mpz_clear(&session_key_mp_int);
}

/* Decrypt the input stream using an RSA key which should be found
   from the given file. Optional passphrase may be used. */
void decrypt_stream(char *file, char *passphrase)
{
  char *comment;

#ifdef LOAD_PUBLIC_KEY
  RSAPublicKey public_key;
#endif

  RSAPrivateKey private_key;
  MP_INT session_key_mp_int;
  int read_bytes, padding;
  unsigned char import_padding;
  int consume;
  CipherContext context;
  Header header;

  if (read_header(&header))
    {
      nfatal("Fatal error. Bailing out.\n");
    }

#ifdef LOAD_PUBLIC_KEY
  mpz_init(&public_key.e);
  mpz_init(&public_key.n);

  mpz_set(&public_key.e, header.exponent);
  mpz_set(&public_key.n, header.modulus);
  public_key.bits = header.bits;
#endif

  ndebug("Loading private key...\n");

  /* Query passphrase once if requested but not given yet
     and the program not in the batch mode. */
  while (!load_private_key(my_uid, file, passphrase, &private_key, &comment))
    {
      if(passphrase[0])
	{
	  nfatal("Illegal passphrase, bailing out.\n");
	}
      else
	{
	  if(batch_mode)
	    {
	      nfatal("Key file requires passphrase, batch mode, "
		       "phrase not given, bailing out.\n");
	    }
	  else 
	    {
	      passphrase = read_passphrase(my_uid, "Passphrase: ", 0);
	      if(!passphrase[0])
		{
		  nfatal("No passphrase given, bailing out.\n");
		}
	    }
	}
    }
  
  /* View comments if user wants so. */
  if(header.file_comment[0])
    print_comment("[stream] %s\n",header.file_comment);
  if(header.RSA_key_comment[0])
    print_comment("[public key] %s\n",header.RSA_key_comment);
  if(comment[0])
    print_comment("[private key] %s\n",comment);

  /* Decrypt the session key. */
  mpz_init(&session_key_mp_int);
  rsa_private_decrypt(&session_key_mp_int, header.encrypted_key,
		      &private_key);
  mp_linearize_msb_first(session_key, SSH_SESSION_KEY_LENGTH,
			 &session_key_mp_int);
  ndebug("Got session key.\n");

  /* Set cipher context. */
  cipher = header.cipher;
  cipher_set_key(&context, cipher, session_key,
		 SSH_SESSION_KEY_LENGTH, 0);
  
  /* Original session key is no longer needed, it is kept in context
     (probably as a key schedule). */
  memset(session_key, 0, SSH_SESSION_KEY_LENGTH *
	 sizeof(unsigned char));
    
  /* Decryption loop */
  ndebug("Entering decryption loop.\n");

  allocate_block(requested_block_size);

  read_bytes = 1;
  
  open_output_stream();

  while (read_bytes)
    {
      read_bytes = read_into_block(input_stream);
      if(block_length >= (block_threshold * 2))
	{
	  consume = block_length - block_threshold -
	    (block_length % block_threshold);
	  cipher_decrypt(&context, block, block, consume);
	  write_from_block(output_stream, consume);
	  shift_block();
	}
    }
  
  assert (block_length);
  padding = ((int) (*(cursor - 1)));

  ndebug("Padding is %d characters.\n", padding);
  cipher_decrypt(&context, block, block, block_threshold);
  write_from_block(output_stream, block_length - 1 - padding);

  /* Cleaning */
  mpz_clear(&session_key_mp_int);

#ifdef LOAD_PUBLIC_KEY
  rsa_clear_public_key(&public_key);
#endif

  free_block();
  rsa_clear_private_key(&private_key);
  xfree(comment);
}

/* Give short usage report for an uncertain user. */
void usage()
{
  fprintf(stderr, USAGE_REPORT);
  exit(1);
}

/* Disable some command line options, exit if current one is
   conflicting. */
void disable(const char flag, char *exclusion)
{
  static char *ptr = given_flags;
  char *temp;

  if (strchr(given_flags, flag))
    {
      nlog_msg("Conflicting or duplicated option '-%c'.\n", flag);
      usage();
    }
  
  if (!strchr(given_flags, flag))
    *ptr++ = flag;
  
  if (exclusion)
    {     
      while (*exclusion)
	{
	  if (!strchr(given_flags, *exclusion))
	    {
	      *ptr++ = *exclusion;
	    }
	  *exclusion++;
	}
    }
}

/* Read next argument, if any left. Otherways return NULL. */
char *safe_arg(int ac, char **av, int *optind)
{
  if(*optind >= ac) return NULL;
  return av[(*optind)++];
}

/* Read public key. This is a separate function from
   load_public_key, because in future we want to read ascii
   key files too. */
int read_public_key(uid_t my_uid, char *filename, RSAPublicKey *pub,
		     char **comment_return)
{
  return !(load_public_key(my_uid, filename, pub, comment_return));
}

int main(int ac, char **av)
{
  int i, shift;

  /* option character */
  char opt;

  /* index which we run through the options table */
  int optind;

  /* cipher type requested */
  int type;

  /* difference between the given block size and the next
     multiple of block_threshold */
  int block_diff;

  /* argument for an option */
  char *optarg;

  /* used in parsing numerical options */
  char *temp;

  /* public key structure, filled only when encrypting */
  RSAPublicKey pub;

  /* pointers to strings representing the bits count, exponent,
     modulus and comment of the public key */
  char *bits, *exp, *mod, *key_comment = "";

  /* pointers to keyfile name and stream comment string */
  char *keyfile = NULL, *comment = "";

  /* passphrase */
  char *passphrase = "";

  /* Read the name of the program to cp */
  av0 = av[0];

  if  (strchr(av0, '/'))
    cp = strrchr(av0, '/') + 1;
  else
    cp = av0;

  /* Run through the options */
  shift = 1;

  for (optind = 1; optind < ac; optind += ((shift == 1) ? 1 : 0))
    {
      /* End when first non-switch is encountered */
      if (av[optind][0] != '-')
	{
	  break;
	}
      
      opt = av[optind][shift];
      
      /* Hyphen alone is not a valid option for us */
      if (!opt) usage();
      
      /* Check if the option requires an argument */
      if (strchr("cCPsDfoF", opt))
	{
	  /* Read argument */
	  if (*(av[optind] + shift + 1) != 0) {
	    optarg = av[optind] + shift + 1;
	  }
	  else
	    {
	      if  (optind >= ac - 1)
		usage();
	      optarg = av[++optind];
	    }
	  shift = 1;
	}
      else
	{
	  /* Argument is not required.
	     If it is given, will do something magical. */
	  if  (av[optind][shift + 1])
	    shift++;
	  else
	    shift = 1;
	  optarg = NULL;
	}
      
      /* Parse the given option (and the possible argument) */
      switch (opt)
	{
	  
	  /* c = select cipher */
	case 'c':
	  disable('c', NULL);
	  cipher = cipher_number(optarg);
	  if (cipher < 0)
	    {
	      nfatal("Unknown cipher type '%s'.\n", optarg);
	    }
	  break;
	  
	  /* I = print info and exit */
	case 'I':
	  disable('I',"BCcdeFfoPs");
	  edd_info();
	  exit(0);
	  break;
	  
	  /* P = give passphrase */
	case 'P':
	  disable('P', "e");
	  passphrase = strdup(optarg);
	  break;
	  
	  /* e = choose encryption */
	case 'e':
	  disable('e', "Pdq");
	  mode = ENCRYPT;
	  forced_mode = TRUE;
	  break;
	  
	  /* d = choose decryption */
	case 'd':
	  disable('d', "eq");
	  mode = DECRYPT;
	  forced_mode = TRUE;
	  break;
	  
	  /* q = choose query */
	case 'q':
	  disable('q', "Pedo");
	  mode = QUERY;
	  break;
	  
	  /* B = select batch mode */
	case 'B':
	  disable('B', NULL);
	  batch_mode = TRUE;
	  break;
	  
	  /* f = set input file */
	case 'f':
	  disable('f', "F");
	  input_file = optarg;
	  break;
	  
	  /* F = set magical input file */
	case 'F':
	  disable('F', "fo");
	  input_file = optarg;
	  output_file = input_file;
	  break;
	  
	  /* o = set output file */
	case 'o':
	  disable('o', "F");
	  output_file = optarg;
	  break;
	  
	  /* C = give stream comment */
	case 'C':
	  disable('C', "d");
	  comment = strdup(optarg);
	  break;
	  
	  /* D = set diagnostics flags */
	case 'D':
	  disable('D', NULL);
	  if(!strcmp(optarg,"all"))
	    diagnostics = "efcnt";
	  else
	    diagnostics = optarg;
	  break;
	  
	  /* s = choose block size */
	case 's':
	  disable('s', NULL);
	  requested_block_size = strtol(optarg, &temp, 0);
	  if (!(*optarg) && (*temp)) {
	    nfatal("Malformed block size '%s'.\n", optarg);
	  }
	  break;
	  
	  /* invalid argument */
	default:
	  usage();
	  break; /* never reached */
	}
    }
  
  ndebug("Checking input and output streams.\n");

  /* if we are querying, no output file is used */
  if (mode == QUERY) output_file = NULL;

  /* Check input and output files */

  /* If input file is given we'll first open it. */
  if (input_file)
    {
      input_stream = open(input_file, O_RDONLY, 0);
      if(input_stream < 0)
	{
	  nfatal("Cannot open input file %s.\n", input_file);
	}
      
      ndebug("Input file open.\n");
      
      /* If input_file has the suffix FILE_SUFFIX and encryption is
	 not forced, choose decryption mode. */
      if ((strlen(input_file) > 4) &&
	  !(strcmp(input_file + (strlen(input_file) - 4), FILE_SUFFIX)) &&
	  (mode == DECRYPT || (mode == ENCRYPT && !forced_mode)))
	{
	  if (mode == ENCRYPT)
	    {
	      nlog_msg("I guess this file should be decrypted.\n");
	      mode = DECRYPT;
	    }

	  /* This is true if -F was used. Then strip the SUFFIX off
	     and use the stripped name as the output file name. */
	  if (input_file == output_file)
	    {
	      output_file = xmalloc(sizeof(char) * (strlen(input_file) - 3));
	      strncpy(output_file, input_file, strlen(input_file) - 4);
	      output_file[strlen(input_file) - 4] = '\0';
	    }
	}

      /* If -F was used but suffix wasn't FILE_SUFFIX or encryption
	 was forced, and decryption wasn't selected, append the
	 suffix to the input file name and use that string as the
	 output file name. */
      if (output_file == input_file && (mode == ENCRYPT))
	{
	  output_file = xmalloc(sizeof(char) * (strlen(input_file) + 5));
	  strcpy(output_file, input_file);
	  strcat(output_file, FILE_SUFFIX);
	}
    }
  
  /* Get user information */
  my_uid = getuid();
  pw = getpwuid(my_uid);
  
  if(!pw)
    {
      nfatal("You are non-existent, bailing out.\n");
    }
  
  ndebug("Reading rest arguments.\n");
  
  /* take rest arguments */
  switch(mode)
    {
      /* Encryption mode */
    case ENCRYPT:
      
      /* Read three arguments */
      bits = safe_arg(ac, av, &optind);
      exp = safe_arg(ac, av, &optind);
      mod = safe_arg(ac, av, &optind);
      
      /* If there are still more arguments, read one. It is the
	 key comment. */
      if (optind < ac)
	key_comment = strdup(safe_arg(ac, av, &optind));
      
      /* If there were no arguments after flags
	 (i.e. bits == NULL), bits will contain the key file name
	 (this is a bit obscure, don't mind :) */
      if (!bits) {
	bits = getenv(PUB_KEYFILE_ENV);
	if (bits && !strlen(bits)) bits = NULL;
      }
      
      /* If bits != NULL but exp == NULL, there was only one
	 argument after flags -- or bits was just set
	 succeedingly from an environment variable.

	 Then bits is interpreted as a key file name. */
      if(bits && !exp)
	{
	  ndebug("Reading key file %s.\n", bits);
	  if(read_public_key(my_uid, bits, &pub, &key_comment))
	    nfatal( "Can't find public key from file '%s'.\n", bits);
	}
      else
	{
	  /* If either bits == NULL or exp != NULL. Either used
	     has given at least two arguments or getenv() returned
	     NULL. */
	  if(!mod) /* less than three arguments */
	    {
	      nerror( "Too few arguments for encryption.\n");
	      usage();
	    }
	}

      /* Public key will be parsed from the args if there were
	 three of them. Otherways we have already read, in success,
	 the public key with read_public_key. */
      if(mod)
	parse_public_key(bits, exp, mod, &pub);
      break;
      
      /* Decryption mode */
    case DECRYPT:

      /* Key file is the only argument. */
      keyfile = safe_arg(ac, av, &optind);

      /* If key file wasn't given, consult an environment
	 variable. */
      if(!keyfile)
	keyfile = getenv(PR_KEYFILE_ENV);
      
      /* If we still don't know about the key file, can't help. */
      if(!keyfile || !(strlen(keyfile)))
	{
	  nerror("Keyfile missing for decryption.\n");
	  usage();
	}
    }

  /* Now all arguments are parsed. If there are still more of them,
     something is wrong and we do not dare to continue. */
  if(optind < ac)
    {
      nlog_msg("Too many arguments.\n", optind, ac);
      usage();
    }

  /* Check that the cipher is supported. */
  if (!((cipher_mask()) & (1 << cipher)))
    {
      nfatal("Selected cipher (%s) not supported.\n", 
	      cipher_name(cipher));
    }

  /* We will use quantized block sizes. */
  if (block_diff = (requested_block_size % block_threshold))
    {
      requested_block_size += (block_threshold - block_diff);
      nlog_msg("Raising block size to %d.\n", requested_block_size);
    }

  /* Check that the block size is legal. */
  if ((requested_block_size < MIN_BLOCK_SIZE) ||
      (requested_block_size > MAX_BLOCK_SIZE))
    {
      nfatal("Block size must reside between %d and %d bytes.\n", 
	      MIN_BLOCK_SIZE, MAX_BLOCK_SIZE);
    }

  /* Perform switch  */
  switch (mode)
    {
      /* Encryption */
    case ENCRYPT:      
      encrypt_stream(&pub, key_comment, comment);
      rsa_clear_public_key(&pub);
      break;

      /* Decryption */
    case DECRYPT:
      decrypt_stream(keyfile, passphrase);
      break;

      /* Query */
    case QUERY:
      query();
      break;

      /* Imaginary */
    default: /* never reached, hope so... */
      exit(2);
      break;
    }

  /* If key comment was set, free it. */
  if(key_comment[0])
    xfree(key_comment);

  /* Closing input and output streams. */
  close(input_stream);
  close(output_stream);

  /* C ya! */
  exit(0);
}
