/*

Copyright (C) 1997 By Activated Intelligence Inc.

Redistribution and use in binary forms, without modification, are permitted provided
that the following conditions are met:

1. Redistributions in binary form must reproduce the above copyright
    notice, this list of conditions and the following disclaimer in the
    documentation and/or other materials provided with the distribution.
2. All advertising materials mentioning features or use of this software must display the following acknowledgement:
		This product includes software developed and copyrighted by
			Activated Intelligence, Inc. TM (http://www.activated.com, mailto://info@activated.com)
		Activated Intelligence, Inc. TM is guided by the strong conviction that there is a better
			way to make computers work for people.

THIS SOFTWARE IS PROVIDED BY THE AUTHOR `AS IS'' AND ANY EXPRESS OR
IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

Activated Intelligence, Inc. TM (http://www.activated.com, mailto://info@activated.com)

*/

import java.awt.*;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.ItemEvent;
import java.awt.event.ItemListener;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;

import java.io.File;
import java.io.IOException;

import java.util.Hashtable;
import java.util.Enumeration;

import com.activated.jimi.Jimi;
import com.activated.jimi.JimiReader;
import com.activated.jimi.JimiException;
import com.activated.jimi.UnsupportedFormatException;
import com.activated.jimi.component.JimiCanvas;

/**
  * An image diplayer and directory browser which uses <strong>J.I.M.I.</strong> to
  * display images within a directory, and to save those images in any other format.
  * The images are cached to make reloading faster, however there is a problem with
  * this in that it is easy to run out of memory with cached images.
  *
  * @author Christian Lucas
  * @author Luke Gorrie
  * @version 1.$Revision: 1.2 $
  * @since Jimi1.0
  */
public final class JimiBrowserApp implements ActionListener, ItemListener {

		//CLASS FIELDS

	/** The maximum size of the browser when it is first loaded.
	  */
	static final Dimension MAX_BROWSER_INIT_SIZE = new Dimension(600, 600);


		//PUBLIC FIELDS



		//PROTECTED FIELDS

	/** The frame in which everything resides.
	  */
	protected Frame displayFrame;

	/** The canvas in which we display the images.
	  */
	protected JimiCanvas jCanvas;
	/** What is the current status of the operations taking place in this
	  * application?
	  */
	protected Label statusBar;

	//protected ScrollPane scrollpane;

	/** The names displayed in the list aren't the fully qualified path names, this maps
	  * between the two.
	  */
	protected Hashtable imageNameMap = new Hashtable(2);

	/** MediaTracker for blocking on image loading.
	 */
	protected MediaTracker mediaTracker;

	/** Where we last loaded an image from
	  */
	protected String currentDirectory;

	/** Checkable MenuItems we need to keep track of
	  */
	protected CheckboxMenuItem menuFaster;
	protected CheckboxMenuItem menuSmoother;


	/** The frame size starts out as 3/4 of the screen real estate when the program is
	  * started
	  *
	  * @see DEFAULT_FRAME_LOCATION
	  */
	protected Dimension DEFAULT_FRAME_SIZE;

	/** The frame is initially centered on the screen, based upon the screen size when
	  * the program is started
	  *
	  * @see DEFAULT_FRAME_SIZE
	  */
	protected Point	 DEFAULT_FRAME_LOCATION;

	/** The menu which lists of all the images currently in the cache.
	  */
	protected Menu imageListMenu;

	/** The <code>ActionListener</code> which monitors the <code>imageListMenu</code>
	  * to determine which image to load.
	  */
	protected final LoadImageListenerClass LoadImageListener = new LoadImageListenerClass();


		//PRIVATE FIELDS

	/** Am I resetting my picture lock?
	  */
	private final LoadingLock lock = new LoadingLock();


    {

		Dimension temp = Toolkit.getDefaultToolkit().getScreenSize();

    	DEFAULT_FRAME_SIZE = new Dimension(temp.width * 3 / 4, temp.height * 3 / 4);

		if(DEFAULT_FRAME_SIZE.width > MAX_BROWSER_INIT_SIZE.width) {

			DEFAULT_FRAME_SIZE.width = MAX_BROWSER_INIT_SIZE.width;

		}

		if(DEFAULT_FRAME_SIZE.height > MAX_BROWSER_INIT_SIZE.height) {

			DEFAULT_FRAME_SIZE.height = MAX_BROWSER_INIT_SIZE.height;
		}

		DEFAULT_FRAME_LOCATION = new Point((temp.width - DEFAULT_FRAME_SIZE.width) / 2,
											(temp.height - DEFAULT_FRAME_SIZE.height) / 2);


	}


	/** To run the program you may either specify a directory, or rely upon
	  * the default directory
	  *
	  * @param args	The directory to browse
	  */
	public static void main(String[] args) {

		new JimiBrowserApp(args);

	}


		//CONSTRUCTORS

	/** Create a new browser using the command line arguments invoked when the class was
	  * "run"
	  *
	  * @see main
	  * @param args The directory to browse
	  */
	public JimiBrowserApp(String[] args) {

		if(args.length != 0) {

			currentDirectory = args[0];

		}


		displayFrame = new Frame("JIMI Image Browser");

		mediaTracker = new MediaTracker(displayFrame);

		displayFrame.addWindowListener(new WindowClosingListener());
		displayFrame.setLayout(new BorderLayout());
		displayFrame.setForeground(SystemColor.window);
		displayFrame.setBackground(SystemColor.desktop);

		//Create the display area for the Images to be displayed on
		jCanvas = new JimiCanvas();
		jCanvas.setBackground(SystemColor.window);
		jCanvas.setForeground(SystemColor.desktop);
		jCanvas.setJustificationPolicy(JimiCanvas.CENTER);
		jCanvas.setResizePolicy(JimiCanvas.CROP_AS_NECESSARY);
		jCanvas.setWillSizeToFit(true);

		ScrollPane sp = new ScrollPane(ScrollPane.SCROLLBARS_AS_NEEDED);
		sp.add(jCanvas);

		displayFrame.add(sp, BorderLayout.CENTER);


		statusBar = new Label("Ready", Label.CENTER);
		statusBar.setForeground(SystemColor.activeCaptionText);
		statusBar.setBackground(SystemColor.activeCaption);
		displayFrame.add(statusBar, BorderLayout.NORTH);

		//Create the Menu's
		MenuBar mBar = createMenuBar();
		displayFrame.setMenuBar(mBar);

		//Pack it, size it, show it
		displayFrame.pack();
		displayFrame.setSize(DEFAULT_FRAME_SIZE);
		displayFrame.setLocation(DEFAULT_FRAME_LOCATION);
		displayFrame.show();

	}


		//STATIC METHODS


		//PRIVATE METHODS


	/** Provides status reporting through a Label on the viewframe.
	  *
	  * @param aString What do we want to report?
	  */
	private void setStatus(String aString) {

		statusBar.setText(aString);

	}

	/** Provides the user with a dialog to select the image they wish to load.  Then
	  * loads that image, and displays it in the <code>JimiCanvas</code>
	  *
	  * @see loadImage
	  */
	protected void openImage() {

		// find out which image to load
		//popup a file dialog
		FileDialog dialog = new FileDialog(displayFrame, "Load Image");
		dialog.setMode(FileDialog.LOAD);

		if(currentDirectory != null) {

			dialog.setDirectory(currentDirectory);

		}

		dialog.show();

		//get the selection
		String shortName = dialog.getFile();

		if(shortName == null) {

			//load cancelled
			return;

		}

		currentDirectory = dialog.getDirectory();
		String fileName = currentDirectory + File.separatorChar + shortName;

		synchronized(lock) {

			lock.addLoad();
			setStatus("Loading " + shortName + "...");
			displayFrame.setCursor(Cursor.getPredefinedCursor(Cursor.WAIT_CURSOR));


		}

		try {

			//create the image
			Image image = loadImage(fileName);

			synchronized(lock) {

				lock.removeLoad();

				//show it on the jCanvas
				jCanvas.setImage(image);
				setStatus(shortName);

			}

			//add it to the listing of images
			MenuItem mi = new MenuItem(shortName);
			mi.addActionListener(LoadImageListener);
			imageListMenu.add(mi);

			imageNameMap.put(shortName, fileName);

		} catch (UnsupportedFormatException ufe) {
		
			setStatus("The format you requested requires JIMI Pro!");
			
			synchronized(lock) {
			
				lock.removeLoad();

			}
			
        } catch (JimiException je) {

            setStatus(je.toString());

            synchronized(lock) {

                lock.removeLoad();

            }

		} catch (IOException e) {

			setStatus(e.toString());

			synchronized(lock) {

				lock.removeLoad();

			}

		} finally {

			synchronized(lock) {

				if(!lock.hasLoaders()) {

					displayFrame.setCursor(Cursor.getDefaultCursor());

				}

			}

		}

	}

	/**
	 * Opens an image and makes sure it is fully loaded.  Caches it for the next time.
	 *
	 * @see openImage
	 * @exception If something goes wrong while loading the image.
	 * @return The image.
	 */
	protected Image loadImage(String filename) throws IOException, JimiException {

		// read entire image
		JimiReader jr = Jimi.createJimiReader(filename);
		Image image = jr.getImage();
		mediaTracker.addImage(image, 0);

		try {

			mediaTracker.waitForAll();

		} catch(InterruptedException e) {

			//Ignore

		}

		mediaTracker.removeImage(image);

		// was there an error in image loading?
		if ((image.getWidth(null) < 0) || (image.getHeight(null) < 0)) {

			throw new IOException("An exception was raised while opening the image.");

		}

		return image;

	}

	/** Saves the image which is currently displayed on the canvas as a new file name.
	  *
	  * @see openImage
	  */
	protected void saveImage() {

		Image image;

		synchronized(lock) {

			image = jCanvas.getImage();

		}

		if(image == null) {

			setStatus("No image");
			return;

		}

		FileDialog dialog = new FileDialog(displayFrame, "Save Image");
		dialog.setMode(FileDialog.SAVE);
		//dialog.setFilenameFilter(SaveFilesFilter);
		dialog.setDirectory(currentDirectory);
		dialog.show();

		String directory = dialog.getDirectory();
		String shortName = dialog.getFile();
		String fileName  = directory + File.separatorChar + shortName;

		if(shortName == null) {

			setStatus("Save Cancelled");
			return;

		}

		currentDirectory = directory;

		setStatus("Saving Image as " + shortName);

		try {

			displayFrame.setCursor(Cursor.getPredefinedCursor(Cursor.WAIT_CURSOR));
			Jimi.putImage(image, fileName);
			jCanvas.setImage(loadImage(fileName));

			//add it to the listing of images
			MenuItem mi = new MenuItem(shortName);
			mi.addActionListener(LoadImageListener);
			imageListMenu.add(mi);

			imageNameMap.put(shortName, fileName);
			setStatus(shortName);

		} catch(JimiException je) {

			System.out.println(je);
			setStatus("Save failed.");

		} catch(IOException ioe) {

			setStatus(ioe.toString());

		} finally {

			displayFrame.setCursor(Cursor.getDefaultCursor());

		}

	}


	/** To simplify the code the menu is created in this subroutine.  <br>
	  * The menu of images we've already loaded is created here, but not populated
	  * until we add those images.
	  *
	  * @returns MenuBar What can we do with this application?
	  */
	protected MenuBar createMenuBar() {

		MenuBar mBar = new MenuBar();

		Menu menu = new Menu("File");
		MenuItem mi = new MenuItem("Open");
		mi.addActionListener(this);
		menu.add(mi);

		mi = new MenuItem("Save");
		mi.addActionListener(this);
		menu.add(mi);

		menu.addSeparator();

		mi = new MenuItem("Exit");
		mi.addActionListener(this);
		menu.add(mi);

		mBar.add(menu);

		menu = new Menu("Options");

		Menu innerMenu = new Menu("Image Scaling");
		menu.add(innerMenu);

		mi = new MenuItem("Crop Images");
		mi.addActionListener(this);
		innerMenu.add(mi);

		mi = new MenuItem("Fit Images");
		mi.addActionListener(this);
		innerMenu.add(mi);

		mi = new MenuItem("Scale Images");
		mi.addActionListener(this);
		innerMenu.add(mi);

		mi = new MenuItem("Scroll");
		mi.addActionListener(this);
		innerMenu.add(mi);

        /*
         * Initialize the "scaling mode" menu
         */
		innerMenu = new Menu("Scaling Mode");
		menu.add(innerMenu);

		menuSmoother = new CheckboxMenuItem("Smoother");
		menuSmoother.addItemListener(this);
		innerMenu.add(menuSmoother);

		menuFaster = new CheckboxMenuItem("Faster");
		menuFaster.addItemListener(this);
		innerMenu.add(menuFaster);

		innerMenu = new Menu("Justification");

		menu.add(innerMenu);

		String[] directions = new String[] {
			"North",
			"Northeast",
			"East",
			"Southeast",
			"South",
			"Southwest",
			"West",
			"Northwest",
			"Center" };


		for(int i = 0; i < directions.length; i++) {

			mi = new MenuItem(directions[i]);
			mi.addActionListener(this);
			innerMenu.add(mi);

		}

		menu.addSeparator();

		mi = new MenuItem("Refresh");
		mi.addActionListener(this);
		menu.add(mi);

		mi = new MenuItem("Garbage Collect");
		mi.addActionListener(this);
		menu.add(mi);

		mBar.add(menu);

		imageListMenu = new Menu("Images");
		mBar.add(imageListMenu);

		return mBar;

	}


		//PUBLIC METHODS


	/** To simplify things, we listen to our own action events that occur in the
	  * main menus.  The list of loaded images is handled by a separate <code>
	  * ActionListener</code>
	  *
	  * @see ImageListListenerClass
	  * @see ImageListListener
	  * @param e	One of our menus was selected
	  */
   	public void actionPerformed(ActionEvent e) {

		//System.out.println(e);

		String command = e.getActionCommand();

		if (command.equals("Open"))	{

			openImage();

		} else if(command.equals("Save")) {

			saveImage();
			//if it is save, the we get the current image from the jCanvas
			//popup a file dialog
			//get the toSave local
			//ensure we have the right format
			//use Jimi to save it
			//reload it from the save location into the jCanvas

		} else if(command.equals("Exit")) {

			synchronized(this) {

				//we dispose of the frame
				displayFrame.setVisible(false);
				displayFrame.dispose();

				//we exit
				System.exit(0);

			}

		} else if(command.equals("Refresh")) {

			jCanvas.setImage(jCanvas.getImage());

		} else if(command.equals("Garbage Collect")) {

			java.lang.Runtime.getRuntime().gc();

		} else if(command.equals("Crop Images")) {

			if(jCanvas.getParent() != displayFrame) {

				Image temp = jCanvas.getImage();

				displayFrame.remove(jCanvas.getParent());
				jCanvas.getParent().remove(jCanvas);

				jCanvas.setWillSizeToFit(false);
				jCanvas.setImage(null);

				displayFrame.add(jCanvas, BorderLayout.CENTER);
				displayFrame.invalidate();
				displayFrame.doLayout();

				jCanvas.setResizePolicy(JimiCanvas.CROP_AS_NECESSARY);
				jCanvas.setImage(temp);


			} else {

				jCanvas.setResizePolicy(JimiCanvas.CROP_AS_NECESSARY);
				jCanvas.setImage(jCanvas.getImage());


			}

		} else if(command.equals("Fit Images")) {

			if(jCanvas.getParent() != displayFrame) {

				Image temp = jCanvas.getImage();

				displayFrame.remove(jCanvas.getParent());
				jCanvas.getParent().remove(jCanvas);

				jCanvas.setWillSizeToFit(false);
				jCanvas.setImage(null);

				displayFrame.add(jCanvas, BorderLayout.CENTER);
				displayFrame.invalidate();
				displayFrame.doLayout();

				jCanvas.setResizePolicy(JimiCanvas.BEST_FIT);
				jCanvas.setImage(temp);


			} else {

				jCanvas.setResizePolicy(JimiCanvas.BEST_FIT);
				jCanvas.setImage(jCanvas.getImage());

			}

		} else if(command.equals("Scale Images")) {

			if(jCanvas.getParent() != displayFrame) {

				Image temp = jCanvas.getImage();

				displayFrame.remove(jCanvas.getParent());
				jCanvas.getParent().remove(jCanvas);

				jCanvas.setWillSizeToFit(false);
				jCanvas.setImage(null);

				displayFrame.add(jCanvas, BorderLayout.CENTER);
				displayFrame.invalidate();

			} else {

				jCanvas.setResizePolicy(JimiCanvas.SCALE);
				jCanvas.setImage(jCanvas.getImage());

			}

		} else if(command.equals("Scroll")) {

			//Should I check if it's null? - Chris
			if(jCanvas.getParent() != displayFrame) {

				return;

			}

			displayFrame.remove(jCanvas);

			jCanvas.setResizePolicy(JimiCanvas.CROP_AS_NECESSARY);
			jCanvas.setJustificationPolicy(JimiCanvas.CENTER);
			jCanvas.setWillSizeToFit(true);

			ScrollPane sp = new ScrollPane(ScrollPane.SCROLLBARS_ALWAYS);
			sp.add(jCanvas);

			displayFrame.add(sp, BorderLayout.CENTER);
			displayFrame.invalidate();
			displayFrame.doLayout();
			// moved this call down to simulate a "Refresh"
			jCanvas.setImage(jCanvas.getImage());

		} else if(command.equals("Center")) {

			jCanvas.setJustificationPolicy(JimiCanvas.CENTER);
			jCanvas.setImage(jCanvas.getImage());

		} else if(command.equals("North")) {

			jCanvas.setJustificationPolicy(JimiCanvas.NORTH);
			jCanvas.setImage(jCanvas.getImage());

		} else if(command.equals("South")) {

			jCanvas.setJustificationPolicy(JimiCanvas.SOUTH);
			jCanvas.setImage(jCanvas.getImage());

		} else if(command.equals("East")) {

			jCanvas.setJustificationPolicy(JimiCanvas.EAST);
			jCanvas.setImage(jCanvas.getImage());

		} else if(command.equals("West")) {

			jCanvas.setJustificationPolicy(JimiCanvas.WEST);
			jCanvas.setImage(jCanvas.getImage());

		} else if(command.equals("Northeast")) {

			jCanvas.setJustificationPolicy(JimiCanvas.NORTHEAST);
			jCanvas.setImage(jCanvas.getImage());

		} else if(command.equals("Northwest")) {

			jCanvas.setJustificationPolicy(JimiCanvas.NORTHWEST);
			jCanvas.setImage(jCanvas.getImage());

		} else if(command.equals("Southeast")) {

			jCanvas.setJustificationPolicy(JimiCanvas.SOUTHEAST);
			jCanvas.setImage(jCanvas.getImage());

		} else if(command.equals("Southwest")) {

			jCanvas.setJustificationPolicy(JimiCanvas.SOUTHWEST);
			jCanvas.setImage(jCanvas.getImage());

		}
	}


    public void itemStateChanged( ItemEvent e )
    {

		if(e.getSource() == menuSmoother) {

		    jCanvas.setScalingPolicy( JimiCanvas.AREA_AVERAGING );
		    menuFaster.setState( false );
		    menuSmoother.setState( true );

		} else if(e.getSource() == menuFaster) {

		    jCanvas.setScalingPolicy( JimiCanvas.REPLICATE );
		    menuFaster.setState( true );
		    menuSmoother.setState( false );
		}

    }

	/** A helper class to listen for menu selection events in the Image list menu.
	  *
	  */
	private class LoadImageListenerClass implements ActionListener {

		public void actionPerformed(ActionEvent e) {

			String command = e.getActionCommand();

			try {

				Image image = JimiBrowserApp.this.loadImage((String)JimiBrowserApp.this.imageNameMap.get(command));

				synchronized(JimiBrowserApp.this.lock) {

					JimiBrowserApp.this.jCanvas.setImage(image);
					JimiBrowserApp.this.setStatus(command);

				}

		    } catch(JimiException je) {

		        JimiBrowserApp.this.setStatus("LoadListener : " + je.toString());

			} catch(IOException ioe) {

				//Image is in the cache, so this will never arise
				JimiBrowserApp.this.setStatus("LoadListener : " + ioe.toString());

			}

		}

	}

	/** Simply simulates an "Exit" menu selection
	  */
	private class WindowClosingListener extends WindowAdapter {

		public void windowClosing(WindowEvent we) {

			JimiBrowserApp.this.actionPerformed(new ActionEvent(JimiBrowserApp.this.displayFrame,
				ActionEvent.ACTION_PERFORMED, "Exit"));

		}

	}

	/** Synched externally
	  */
	protected static final class LoadingLock {

		protected int count = 0;

		LoadingLock() {

			super();

		}

		boolean hasLoaders() {

			return count > 0;

		}

		void addLoad() {

			count++;

		}

		void removeLoad() {

			count--;

		}

	}


}
