import com.activated.jimi.component.JimiCanvasLW;
import com.activated.jimi.util.GraphicsUtils;
import com.activated.jimi.util.ColorReducer;
import com.activated.jimi.JimiException;

import com.sun.java.swing.*;
import com.sun.java.swing.event.InternalFrameAdapter;
import com.sun.java.swing.event.InternalFrameEvent;
import java.awt.*;
import java.awt.event.*;
import java.awt.image.*;

/**
 * An InternalFrame for rendering an image and applying filters to it.
 * @author  Luke Gorrie
 * @author  Karl Avedal
 * @version $Revision: 1.2 $ $Date: 1998/07/22 08:52:55 $
 **/
public class ImageInternalFrame extends JInternalFrame implements ImageObserver
{

	// original image
	private Image originalImage_;
	// canvas to render image in
	private JimiCanvasLW jCanvas_;

	// frame title
	private String title_;

	// old image (added to save the old image before a filter is applied)
	private Image old_image_;

	public ImageInternalFrame(String title, Image image)
	{
		super(title);

		// store the title to be able to restore it
		title_ = title;

		originalImage_ = image;

		// initialize widgets
		initFrame();
		initCanvas();

		pack();
	}

	/**
	 * Set the image to be displayed.
	 * @param image The image to display.
	 **/
	public synchronized void setImage(Image image)
	{
		originalImage_ = image;
		setCanvasImage(image);
	}

	/**
	 * Get the Image being displayed.
	 * @return The Image.
	 **/
	public Image getImage()
	{
		return jCanvas_.getImage();
	}

	/**
	 * Applies a filter to the image current image and displays the result.
	 * @param filter The ImageFilter to apply.
	 **/
	public void applyFilter(ImageFilter filter)
	{
		setTitle("Applying..");
		old_image_ = getImage();

		ImageProducer source = new FilteredImageSource(old_image_.getSource(), filter);
		Image new_image = createImage(source);
		GraphicsUtils.waitForImage(this, new_image, this);

	}

	public boolean imageUpdate(Image img, int infoflasgs, int x, int y, int width, int height)
	{
		setCanvasImage(img);
		setTitle(title_);
		// if the previous image was the result of a filter, flush it
		if (old_image_ != originalImage_)
			old_image_.flush();
        return false;
	}

	/**
	 * Perform color reduction.
	 **/
	public void reduceColors(int colors, boolean dither)
	{
		ColorReducer reducer = new ColorReducer(colors, dither);
		Image img = null;
		try {
			img = reducer.getColorReducedImage(getImage());
		}
		catch (JimiException e)
		{
			// don't care
		}
		if (img != null)
			jCanvas_.setImage(img);
	}

	/**
	 * Revets to the original image, effectively undoing all filters.
	 **/
	public void revertImage()
	{
		setCanvasImage(originalImage_);
	}

	/**
	 * Flush out all data, forgetting references and rendering the frame inoperable.
	 */
	public void flush()
	{
		if (originalImage_ != null) {
			originalImage_.flush();
			originalImage_ = null;
		}
		if (old_image_ != null) {
			old_image_.flush();
			old_image_ = null;
		}
		jCanvas_ = null;
		removeAll();
	}

	/**
	 * Sets various properties in 'this' JInternalFrame.
	 **/
	protected void initFrame()
	{
		setResizable(false);
		setClosable(true);
		setIconifiable(false);
	}

	/**
	 * Initializes the JimiCanvas and adds it to 'this' Frame.
	 **/
	protected void initCanvas()
	{
		// create and configure canvas
		jCanvas_ = new JimiCanvasLW();
		jCanvas_.setJustificationPolicy(JimiCanvasLW.CENTER);
		jCanvas_.setResizePolicy(JimiCanvasLW.CROP_AS_NECESSARY);
		jCanvas_.setWillSizeToFit(true);
		jCanvas_.setImage(originalImage_);

		// add the canvas to the container
		Container content_pane = getContentPane();
		content_pane.setLayout(new BorderLayout());
		content_pane.add(BorderLayout.CENTER, jCanvas_);
	}

	/**
	 * Sets the image being displayed in the underlying JimiCanvas.
	 **/
	protected void setCanvasImage(Image image)
	{
		jCanvas_.setImage(image);
		invalidate();
		jCanvas_.repaint();
		pack();
	}

}

