program combine;
{
PURPOSE:       Combine NASA-TLX ratings and weights to
               produce overall weighted workload score.
               Overall workload score reflects the magnitude
               (rating) given each scale, multiplied times
               its importance (weight).  For any variable
               (subject, condition, replication) on which
               weights ARE conditional, only weights
               obtained for that variable are applied; if
               no weights were obtained for that variable,
               equal weights are applied.  If weights are
               NOT conditional on that variable, the first
               weights found will be applied.
               If .WWL file has less than 2000 lines of
               data, it will be sorted by subject
               (changes slowest), condition, and replication.
INPUT FILES:   User names a .RAT file with ratings and a
               .WGT file with weights.
OUTPUT FILE:   The ratings are written to a user-named
               .WWL file, and the weighted workload is
               appended to the end of each line of ratings.
ENVIRONMENT:   IBM-PC compatible microcomputer;
               Turbo Pascal 3.0
HISTORY:       v. 1.0  Sept. 1986.  Author:  Walter Johnson
}
type
 filetype = string[15];
 agetype = string[3];

Const
 blanks = '                                                                                ';

var
 ok,
 REP,
 COND,
 SUBJ           :    boolean;

 infofile,
 scalefile,
 weightfile,
 nscalefile     :    text;

 wfilename,
 nfilename      :    filetype;

 rating         :    array[1..8] of integer;
 weight         :    array[1..6] of real;

 scond,
 wcond          :    array[1..4] of char;

 overall_scale,
 i,
 ssubj,
 wsubj,
 srep,
 wrep           :    integer;

 temp           :    real;



Procedure Open(filename : filetype; age : agetype;
               var filevar : text; var status : boolean);
var
 old : boolean;

Begin
 Assign(filevar,filename);
 {$I-} Reset(filevar) {$I+};
 old := (IOresult = 0);
 if (old and (age='old')) then status := true else
 if (not old and (age='new')) then begin
  Rewrite(filevar);
  status := true;
 end else
 status := false;
End;


Procedure Open_Scalefile;
var
 filename : filetype;
 ok : boolean;


Begin
 ok := true;
 repeat
  if not ok then writeln(trm,'THIS FILE DOES NOT EXIST');
  writeln(trm,'Input the name of the file containing the ratings');
  write(trm,' (no extension): ');
  readln(con,filename);
  filename := Concat(filename,'.rat');
  Open(filename,'old',scalefile,ok);
 until ok;
End;


Procedure Open_NewScalefile;
var
 ok : boolean;

Begin
 ok := true;
 repeat
  if not ok then writeln(trm,'THIS FILE ALREADY EXISTS');
  writeln(trm,'Input the name of the file that will contain the weighted ');
  write(trm,' workload scores (no extension): ');
  readln(con,nfilename);
  nfilename := Concat(nfilename,'.wwl');
  Open(nfilename,'new',nscalefile,ok);
 until ok;
End;


Procedure Check_Weightfile;
var
 filename : filetype;
 ok : boolean;

Begin
 ok := true;
 repeat
   if not ok then writeln(trm,'THIS FILE DOES NOT EXIST');
   writeln(trm,'Input the name of the file containing the weights ');
   write(trm,' (no extension): ');
   readln(con,filename);
   wfilename := Concat(filename,'.wgt');
   Open(wfilename,'old',weightfile,ok);
  until ok;
 Close(weightfile);
End;



Procedure Make_Header;
var
 line : string[80];
 k,i : integer;

Begin
 for i := 1 to 3 do begin
  line := blanks;
  readln(scalefile,line);
  if i=1 then begin
   for k := 1 to 11 do write(nscalefile, line[k]);
   write(nscalefile,nfilename);
  end else write(nscalefile,line);
  if i=2 then write(nscalefile,' WWL');
  writeln(nscalefile);
 end;
End;

Procedure Get_Ids;
var
 i : integer;
 a : char;

 Begin
  { See if Overall Workload was used }
  assign (infofile, 'default');
  reset (infofile);
  for i := 1 to 9 do
    readln (infofile);
  readln (infofile, overall_scale);
  close (infofile);
  repeat
   Write(trm,'Are weights conditional on subjects (y or n): ');
   Readln(con,a);
  until a in ['y','Y','n','N'];
  if a in ['y','Y'] then SUBJ := true else SUBJ := false;

  repeat
   Write(trm,'Are weights conditional on conditions (y or n): ');
   Readln(con,a);
  until a in ['y','Y','n','N'];
  if a in ['y','Y'] then COND := true else COND := false;


  repeat
   Write(trm,'Are weights conditional on replications (y or n): ');
   Readln(con,a);
  until a in ['y','Y','n','N'];
  if a in ['y','Y'] then REP := true else REP := false;

 End;

PROCEDURE SORTOUTPUT;
{ PURPOSE:  Read up to 2000 lines of .WWL file created previously
  into dataarray, converting to save space.  Sort by subject,
  condition, and replication (subject changing slowest).  Rewrite
  to .WWL file.  If file has more than 2000 lines, a message
  is sent and no sorting is done.  (User should break .RAT files
  into smaller files and rerun COMBINE).  Bubble sort is used
  to sort indices, which are later used to rewrite dataarray. }

CONST
   maxrows = 2000;

TYPE
   dataline = record
      subj : byte;
      cond : string [3];
      replic : byte;
      values : array [1..8] of byte;  { 6 scales, room for OW, WWL }
   end;

VAR
   exit : boolean;  { T if > 2000 lines of data in .WWL file }
   dataarray : array [1..maxrows] of dataline;
   indices : array [1..maxrows] of integer;
   i,
   index,
   stopindex,
   numrows
      : integer;

PROCEDURE READWWLFILE;
VAR
   ch : string [1];
   i,j : integer;
begin
  reset (nscalefile);
  for i := 1 to 3 do
    readln (nscalefile);
  i := 1;
  while ( i <= maxrows ) and ( not EOF (nscalefile ) ) do
    begin
      with dataarray [i] do
        begin
          read (nscalefile, subj);
          read (nscalefile, ch, cond);
          read (nscalefile, replic);
          for j := 1 to 7 + overall_scale do
            read (nscalefile, values [j]);
          readln ( nscalefile);
        end;
      i := i +1;
    end;
  numrows := i-1;
  if i > maxrows then
    begin
      exit := true;
      writeln ( '.RAT and .WWL files have more than 2000 lines of data.' );
      writeln ( 'Output in .WWL file cannot be sorted.  If you want it' );
      writeln ( ' sorted, break .RAT files up so they are shorter than' );
      writeln ( ' 2000 lines and rerun COMBINE with each file.' );
    end;
end;

PROCEDURE REWRITEWWLFILE;
{ Write dataarray to .WWL file }
VAR
  i, j : integer;
  line : string [80];
begin
  rewrite (nscalefile);
  { Write header }
  writeln (nscalefile, 'FILENAME: ', nfilename);
  reset (scalefile);
  readln (scalefile);
  readln (scalefile, line);
  write (nscalefile, line);
  writeln (nscalefile, ' WWL' );
  writeln (nscalefile);
  for i:= 1 to numrows do
    with dataarray [ indices [i]] do
      begin
        write ( nscalefile, subj:4 );
        write ( nscalefile, ' ', cond );
        write ( nscalefile, replic:4 );
        for j := 1 to 7 + overall_scale do
          write ( nscalefile, values [j]:4 );
        writeln ( nscalefile );
      end;
end;

PROCEDURE FINDSAMESUBJS ( START : INTEGER; VAR STOP : INTEGER );
{ PURPOSE:  Find clumps of subjects which are the same, so can
  then sort by condition within the clump.  Enter with start=
  index of the first subject; exit with stop=index of the
  last subject which is the same.  If there is only one
  subject in that clump, stop=start. }
VAR
  i : integer;
begin
  i := start;
  repeat
     i := i + 1
  until ( i > numrows ) or
    (dataarray [indices[i]].subj <> dataarray [indices[start]].subj);
  stop := i-1;
end;

PROCEDURE FINDSAMECONDS ( START : INTEGER; VAR STOP : INTEGER );
{ PURPOSE:  Find clumps of conditions which are the same, so can
  then sort by replication within the clump.  Enter with start=
  index of the first conditions; exit with stop=index of the
  last condition which is the same.  If there is only one
  condition in that clump, stop=start. }
VAR
  i : integer;
begin
  i := start;
  repeat
     i := i + 1
  until ( i > numrows ) or
    (dataarray [indices[i]].cond <> dataarray [indices[start]].cond);
  stop := i-1;
end;

PROCEDURE SORTBYSUBJ ( START, STOP : INTEGER );
{ Use bubblesort to sort indices so subject numbers in rows referred
  to in dataarray are in increasing order. }
VAR
  inorder : boolean;
  temporary,
  i :
    integer;
begin
  repeat
    inorder := true;
    for i:= start to stop-1 do
      if dataarray [indices[i]].subj >
        dataarray [indices[i+1]].subj then
        begin
          temporary := indices [i];
          indices [i] := indices [i+1];
          indices [i+1] := temporary;
          inorder := false;
        end;
  until inorder;
end;

PROCEDURE SORTBYCOND ( START, STOP : INTEGER );
{ Use bubblesort to sort indices so conditions in rows referred
  to in dataarray are in alphabetical order. }
VAR
  inorder : boolean;
  temporary,
  i :
    integer;
begin
  repeat
    inorder := true;
    for i:= start to stop-1 do
      if dataarray [indices[i]].cond >
        dataarray [indices[i+1]].cond then
        begin
          temporary := indices [i];
          indices [i] := indices [i+1];
          indices [i+1] := temporary;
          inorder := false;
        end;
  until inorder;
end;

PROCEDURE SORTBYREPLIC ( START, STOP : INTEGER );
{ Use bubblesort to sort indices so replications in rows referred
  to in dataarray are in increasing order. }
VAR
  inorder : boolean;
  temporary,
  i :
    integer;
begin
  repeat
    inorder := true;
    for i:= start to stop-1 do
      if dataarray [indices[i]].replic >
        dataarray [indices[i+1]].replic then
        begin
          temporary := indices [i];
          indices [i] := indices [i+1];
          indices [i+1] := temporary;
          inorder := false;
        end;
  until inorder;
end;

{main of procedure sortoutput}
begin
  exit := false;
  readWWLfile;
  if not exit then
    begin
      writeln;
      writeln ( 'Please wait while output is sorted.' );
      { fill array of indices, to be sorted }
      for i := 1 to numrows do
        indices [i] := i;

      { sort by subject }
      sortbysubj ( 1, numrows );

      { sort by cond, within clumps of same subjects }
      index := 1;
      while index < numrows do
        begin
          findsamesubjs ( index, stopindex );
          if index < stopindex then
            sortbycond ( index, stopindex );
          index := stopindex + 1;
        end;

      { sort by replication, within clumps of same conditions }
      index := 1;
      while index < numrows do
        begin
          findsameconds ( index, stopindex );
          if index < stopindex then
            sortbyreplic ( index, stopindex );
          index := index + 1;
        end;

      rewritewwlfile;
    end;  { not exit }
 end;

{ main }
begin
 ClrScr;
 GotoXY(30,7);
 write('NASA Task Load Index');
 GotoXY(25,9);
 write('Combining Weights and Ratings');
 GotoXY(18,18);
 write('Enter "Cntrl C" at any time to abort program');
 GotoXY(29,20);
 write('Hit any key to continue');
 repeat until keypressed;
 ClrScr;
 Open_Scalefile;
 Check_Weightfile;
 Open_NewScalefile;
 Make_Header;
 Get_Ids;

 writeln;
 writeln ( 'Please wait while weighted workload scores are computed.' );
 repeat
  temp := 0;
  read(scalefile,ssubj,scond,srep);
  for i := 1 to 6 + overall_scale do read(scalefile,rating[i]); readln(scalefile);
  Begin
   Open(wfilename,'old',weightfile,ok);
   repeat
    read(weightfile,wsubj,wcond,wrep);
    if not REP then wrep := srep;
    if not COND then wcond := scond;
    if not SUBJ then wsubj := ssubj;
    for i := 1 to 6 do read(weightfile,weight[i]); readln(weightfile);
   until ( (ssubj=wsubj) and (scond=wcond) and (srep=wrep) ) or eof(weightfile);
   if not ( (ssubj=wsubj) and (scond=wcond) and (srep=wrep) ) and
   eof(weightfile) then for i := 1 to 6 do weight[i] := 0.1666;
   Close(weightfile);
  End;
  for i := 1 to 6 do temp := temp + weight[i]*rating[i];
  rating[7 + overall_scale] := round(temp);
  write(nscalefile,ssubj:4,scond,srep:4);
  for i := 1 to 7 + overall_scale do write(nscalefile,rating[i]:4);
  writeln(nscalefile);
 until eof(scalefile);

 { Sort output in nscalefile }
 sortoutput;
 close (scalefile);
 close (nscalefile);
end.