#include <stdlib.h>
#include <netdb.h>
#include <sys/time.h>
#include <sys/types.h>
#include <unistd.h>
#include <fcntl.h>
#include <string.h>
#include <stdio.h>
#include <signal.h>
#include <sys/ioctl.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <curses.h>
#include <net/if.h>
#include <errno.h>
#include "defines.h"
#include "tcp.h"

extern	bool use_own_ip;
extern	int	errno;
extern	char	okay_dir[256];

char own_ip[256];

bool DetermineOwnIP(char *device)
{
	static struct	ifreq	ifr;
	int	s;
		
	if ((s = socket(AF_INET, SOCK_DGRAM, 0)) < 0)
		return(FALSE);
	
	strcpy(ifr.ifr_name, device);
	ifr.ifr_addr.sa_family = AF_INET;
	if(ioctl(s, SIOCGIFADDR, &ifr) < 0)
		return(FALSE);
	
	strcpy(own_ip, inet_ntoa(((struct sockaddr_in *)&ifr.ifr_addr)->sin_addr));
	
	close(s);
	return(TRUE);
}

CTCP::CTCP()
{
	int	n;
	
	this->speed = 0.0;
	
	for(n = 0; n < MSG_STACK; n++)
		this->msg_stack[n] = -1;
		
	for(n = 0; n < LOG_LINES; n++)
		this->log[n] = NULL;
		
	this->control_connected = FALSE;
	this->haveIP = FALSE;
	this->have_accepted = FALSE;
		
	pthread_mutex_init(&(this->log_lock), NULL);
}

CTCP::~CTCP()
{
}

void CTCP::CloseControl(void)
{
	if(this->control_connected) {
		close(this->control_sock_fd);
		this->control_connected = FALSE;
	}
}

bool CTCP::GetIP(char *address, struct in_addr *ip_address)
{
	struct hostent		*host_ent;
		
	// first try dotted style
	if(inet_aton(address, ip_address)) {
		return(TRUE);
	}
	else {
		// use DNS lookup
		if((host_ent = gethostbyname(address)) == NULL) {
			this->error = E_CTRL_ADDRESS_RESOLVE;	
			return(FALSE);
		}

		bcopy(host_ent->h_addr, (char *)ip_address, host_ent->h_length);
	}

	return(TRUE);
}


bool CTCP::OpenControl(BOOKMARK *bm)
{
	struct	sockaddr_in	socket_in;
	struct	in_addr		ip_address;

	// open control socket (telnet)
	socket_in.sin_family = AF_INET;
	socket_in.sin_port = htons(bm->port);
	
	if(!this->GetIP(bm->host, &ip_address))
		return(FALSE);
	
	socket_in.sin_addr = ip_address;

	if((this->control_sock_fd = socket(AF_INET, SOCK_STREAM, 0)) < 0) {
		this->error = E_CTRL_SOCKET_CREATE;
		return(FALSE);
	}

	if(connect(this->control_sock_fd, (struct sockaddr *)&socket_in, sizeof(struct sockaddr))) {
		close(this->control_sock_fd);
		this->error = E_CTRL_SOCKET_CONNECT;
		return(FALSE);
	}

	// now we avoid select() calls since they got useless with this multithreading model
	// but we need timing information to not confuse our user, so... nonblocking
	fcntl(this->control_sock_fd, F_SETFL, O_NONBLOCK);
	
	this->control_connected = TRUE;
	return(TRUE);

}

bool CTCP::SendData(char *msg)
{
	if(this->control_connected) {
		if(write(this->control_sock_fd, msg, strlen(msg)) != -1) {
			sprintf(this->temp_string, ">>> %s", msg);
			this->AddLogLine(this->temp_string);
			return(TRUE);
		}
		else {
			this->control_connected = FALSE;
			return(FALSE);
		}
	}
	return(FALSE);
}

int CTCP::SearchStack(void)
{
	int	n;

	if(this->msg_stack[0] == -1)
		return(1);			// stack empty, load new stack
		
	if(((this->msg_stack[0] >= 200) && (this->msg_stack[0] < 300)) || (this->msg_stack[0] == 125) || (this->msg_stack[0] == 331) || (this->msg_stack[0] == 350) || (this->msg_stack[0] == 150)) {
		// *pheew* was the correct msg
		for(n = 1; n < MSG_STACK; n++)
			this->msg_stack[n-1] = this->msg_stack[n];
			
		this->msg_stack[MSG_STACK-1] = -1;
		return(2);			// found msg, removed
	}
	
	// bad luck, the wrong msg is on the stack...
	return(0);
}

void CTCP::ObtainLog(char *log_out[LOG_LINES])
{
	int	n;
	
	pthread_mutex_lock(&(this->log_lock));

	for(n = 0; n < LOG_LINES; n++) {
		if(this->log[n]) {
			log_out[n] = new(char[strlen(this->log[n]) + 1]);
			strcpy(log_out[n], this->log[n]);
		}
		else
			log_out[n] = NULL;
	}

	pthread_mutex_unlock(&(this->log_lock));
}

void CTCP::StripANSI(char *line)
{
	char	*m_pos, *position = line;
	// simply search for an ESC and then wait until we read the "m" (or end)

	while(*position) {
		if(*position == 27) {
			// found the possible start of an ESC code
			// now search for the "m"
			m_pos = position + 1;
			while(*m_pos && (*m_pos != 'm'))
				m_pos += 1;
			
			if(*m_pos) {
				// now copy the remaining string
				strcpy(position, m_pos + 1);
			}
			else {
				// else we reached '\0'
				position = m_pos;
			}
		}
		else
			position += 1;
	}	
}

void CTCP::AddLogLine(char *line)
{
	int	n;
	
	this->StripANSI(line);
	
	pthread_mutex_lock(&(this->log_lock));

	if(this->log[LOG_LINES-1]) {
		delete(this->log[LOG_LINES-1]);
		this->log[LOG_LINES-1] = NULL;
	}
	
	// "scroll" lines
	for(n = LOG_LINES - 1; n > 0; n--)
		this->log[n] = this->log[n-1];

	// add line
	this->log[0] = new(char[strlen(line) + 1]);
	strcpy(this->log[0], line);
	
	pthread_mutex_unlock(&(this->log_lock));
}

void CTCP::UpdateStack()
{
	char	*start = this->control_buffer, *end, temp, temp_stack[5];
	int	stack_cnt = 0, msg_nr;
	
	// analyze this->control_buffer and put entries in stack
	while((end = strchr(start, '\n'))) {		// hack!
		*end = '\0';
		this->AddLogLine(start);
		*end = '\n';
		
		temp = *(start+4);
		*(start + 4) = '\0';
		// can we use this msg?
		if(*(start+3) == ' ') {
			strncpy(temp_stack, start, 3);
			temp_stack[3] = '\0';
			msg_nr = atoi(temp_stack);
			if(msg_nr > 0) {		// i hate zipcheckers
				this->msg_stack[stack_cnt] = msg_nr;
				stack_cnt++;
			}
		}
		*(start+4) = temp;
		start = end + 1;			// hack!
	}
}

void CTCP::FlushStack(void)
{
	this->msg_stack[0] = -1;
	this->FlushSocket();
}

int CTCP::WaitForDataAndRead(bool control, int already_read)
{
	int	len, n;
	int	sock_fd = control? this->control_sock_fd : this->real_data_sock_fd;
	
	// give em a few tries tries within one second
	for(n = 0; n < 5; n++) {
		len = recv(sock_fd,  this->control_buffer + already_read, CONTROL_BUFFER_SIZE - already_read, 0);

		if(len == -1) {
			// hmmm, right now we just expect EWOULDBLOCK
			usleep(200000);		// 0.2 secs
		}
		else
			return(len);
	}
		
	return(-1);
}

bool CTCP::WaitForMessage(void)
{
	int	read_length, okay, time = 0;
	long	already_read = 0;
	char	*buffer_pos, *null_ptr;
	bool	full_lines;

	okay = this->SearchStack();
		
	if(okay == 2) {
		// found a positive msg in stack and removed it, okay
		return(TRUE);
	}
	else if(okay == 0) {
		// bad luck, we received a negative message
		this->FlushStack();
		return(FALSE);
	}
	
	// there is no message on the stack
	do {
		already_read = 0;
		full_lines = FALSE;
		
		// wait until one or more full lines are received
		do {
			while(((read_length = this->WaitForDataAndRead(TRUE, already_read)) == -1) && (time < CONTROL_TIMEOUT)) {
				time++;
			}
			
			if(time >= CONTROL_TIMEOUT) {
				this->error = E_CTRL_TIMEOUT;
				already_read = 0;
				return(FALSE);
			}
		
			time = 0;

			if(read_length == 0) {
				// it seems we got a wrong msg, but the server didn't send us an RFC-compliant msg
				*(this->control_buffer+already_read) = '\0';
				strcat(this->control_buffer, " \n");
				this->FlushStack();
				this->error = E_BAD_MSG;
				return(FALSE);
			}
			
			buffer_pos = this->control_buffer + already_read;

			// fix, first time occured at STH (thx _hawkeye for your buggy welcome.msg)
			// remove all occurances of '\0' b4 the real end
			null_ptr = buffer_pos;
			while(null_ptr < (buffer_pos+read_length)) {
				if(*null_ptr == '\0')
					*null_ptr = ' ';
				null_ptr += 1;
			}
			
			if((*(buffer_pos+read_length-2) == '\r') && (*(buffer_pos+read_length-1) == '\n'))
				full_lines = TRUE;

			already_read += read_length;
		} while(!full_lines);

		*(buffer_pos+read_length) = '\0';
	
		// add possible msg's to stack
		this->UpdateStack();
		okay = this->SearchStack();

		if(okay == 2) {			// found positive end-of-output msg
			return(TRUE);
		}
		else if(okay == 0) {		// stack contained a negative msg
			// recover
			this->FlushStack();
			this->error = E_BAD_MSG;
			return(FALSE);			
		}
				
		// the stack hadn't yet the end-of-output msg, give it another try (strange implementations of some servers)
	} while(TRUE);
}

void CTCP::FlushSocket(void)
{
	int	len;
	
	do {
		len = recv(this->control_sock_fd, this->control_buffer, CONTROL_BUFFER_SIZE, 0);
	} while(len > 0);
}

bool CTCP::OpenData(char *port_str)
{
	char		*start, myhost_str[256];
	struct in_addr	ip_address;
	int		sockaddr_len = sizeof(struct sockaddr_in);
	
	if((this->data_sock_fd = socket(AF_INET, SOCK_STREAM, 0)) < 0) {
		this->error = E_SOCKET_DATA_CREATE;
		return(FALSE);
	}

	if(!this->haveIP) {
		if(!use_own_ip)
			gethostname(myhost_str, 256);
		else
			strcpy(myhost_str, own_ip);
		
		if(!this->GetIP(myhost_str, &ip_address))
			return(FALSE);
			
		this->haveIP = TRUE;
		bcopy(&ip_address, &(this->stored_ip_address), sizeof(struct in_addr));
	}
	else {	// since the damn lookups lock when asking them fast (????) we have to overcome this	
		bcopy(&(this->stored_ip_address), (char *)&ip_address, sizeof(struct in_addr));
	}
	
	this->data_sock_in.sin_addr.s_addr = INADDR_ANY;
	this->data_sock_in.sin_family = AF_INET;
	this->data_sock_in.sin_port = 0;
	
	if(bind(this->data_sock_fd, (struct sockaddr *)&data_sock_in, sizeof(struct sockaddr_in)) < 0) {
		close(this->data_sock_fd);
		this->error = E_SOCKET_BIND;
		return(FALSE);
	}
	
	if(getsockname(this->data_sock_fd, (struct sockaddr *)&data_sock_in, &sockaddr_len) < 0) {
		close(this->data_sock_fd);
		this->error = E_SOCKET_BIND;
		return(FALSE);
	}
	
	this->data_port = ntohs(this->data_sock_in.sin_port);
	
	if(listen(this->data_sock_fd, 1) < 0) {
		close(this->data_sock_fd);
		this->error = E_SOCKET_LISTEN;
		return(FALSE);
	}

	// prepare PORT cmd string
	if(!use_own_ip) {
		ip_address.s_addr = (unsigned)gethostid();
		strcpy(myhost_str, inet_ntoa(ip_address));
		
		*(strchr(myhost_str, '.')) = ',';
		*(strchr(myhost_str, '.')) = ',';
		*(strchr(myhost_str, '.')) = ',';
		
		start = strchr(myhost_str, ',');
		start = strchr(start+1, ',') + 1;
		strcpy(port_str, start);
		strcat(port_str, ",");
		*(start-1) = '\0';
		strcat(port_str, myhost_str);
	}
	else {
		strcpy(myhost_str, own_ip);
		
		*(strchr(myhost_str, '.')) = ',';
		*(strchr(myhost_str, '.')) = ',';
		*(strchr(myhost_str, '.')) = ',';

		strcpy(port_str, myhost_str);
	}
	
	sprintf(myhost_str, "%d,%d", this->data_port / 256, this->data_port - this->data_port / 256 * 256);
	strcat(port_str, ",");
	strcat(port_str, myhost_str);

	// now we avoid select() calls since they got useless with this multithreading model
	// but we need timing information to not confuse our user, so... nonblocking
	fcntl(this->data_sock_fd, F_SETFL, O_NONBLOCK);

	return(TRUE);
}

bool CTCP::AcceptData(void)
{
	int	sockaddr_len = sizeof(struct sockaddr_in);
	int	n = 0;

	// give em 40 chances during 20 seconds
	do {
		this->real_data_sock_fd = accept(this->data_sock_fd, (struct sockaddr *)&(this->data_sock_in), &sockaddr_len);
		n++;
		if(this->real_data_sock_fd == -1)
			usleep(500000);
	}		
	while((this->real_data_sock_fd == -1) && (n < 40));
		
	if(this->real_data_sock_fd == -1) {
		close(this->data_sock_fd);
		this->error = E_SOCKET_ACCEPT_TIMEOUT;
		return(FALSE);
	}

	close(this->data_sock_fd);
	
	if(this->real_data_sock_fd < 0) {
		this->error = E_SOCKET_ACCEPT;
		return(FALSE);
	}
	
	fcntl(this->real_data_sock_fd, F_SETFL, O_NONBLOCK);

	this->have_accepted = TRUE;	
	return(TRUE);	
}

void CTCP::CloseData(void)
{
	// this fuck took 2 weeks of my life, a small bug... grrrrrrrrrrrrr
	if(this->have_accepted) {
		this->have_accepted = FALSE;
		close(this->real_data_sock_fd);
	}
	else
		close(this->data_sock_fd);
}

bool CTCP::WriteFile(char *read_name, bool no_ok)
{
	char		error_file[512], ok_file[512];
	int		block_wait = 0, write_length, infile, try_file = 0, sent, already_sent;
	bool		found = FALSE, error = FALSE, exists, finished = FALSE;
	struct stat	stat_out;
		
	this->size = 0;
	sprintf(error_file, "%s%s.error", okay_dir, strrchr(read_name, '/')+1);
	sprintf(ok_file, "%s%s.okay", okay_dir, strrchr(read_name, '/')+1);
	
	do {
		if(stat(read_name, &stat_out) == -1) {
			usleep(300000);
			try_file++;
		}
		else
			found = TRUE;
	} while(!found && (try_file < 120));

	if(!found) {
		this->error = E_WRITEFILE_TIMEOUT;
		return(FALSE);
	}

	if((infile = open(read_name, O_RDONLY)) == -1) {
		this->error = E_WRITEFILE_ERROR;
		return(FALSE);
	}
	

	// start uploading
	gettimeofday(&tv_before, NULL);
	exists = no_ok;
	do {
		// check if we already have our STOP-files
		if(!exists) {
			if(!stat(error_file, &stat_out) || !stat(ok_file, &stat_out))
				exists = TRUE;
		}
		
		write_length = read(infile, control_buffer, WRITE_SIZE);	// try to push max
		
		if(write_length > 0) {
			this->size += write_length;	
			// we have more bytes again, push em
			already_sent = 0;

			do {
				if((sent = write(this->real_data_sock_fd, control_buffer + already_sent, write_length - already_sent)) < 0) {
					if(errno != EAGAIN)
						error = TRUE;
					else {
						// blocking
						if(block_wait < (DATA_TIMEOUT*5)) {
							usleep(200000);         // 0.2 secs
							block_wait++;
						}
						else
							error = TRUE;
					}
				}
				else {
					already_sent += sent;
					block_wait = 0;
				}
			} while(!error && (already_sent < write_length));
			
			// calc transfer speed
			// between tv_before and tv_after we transferred write_length-bytes
			gettimeofday(&tv_after, NULL);
			seconds = tv_after.tv_sec - tv_before.tv_sec;
			micros = tv_after.tv_usec - tv_before.tv_usec;
			micros += seconds * 1000000 + 1;
			this->speed = ((float)(this->size) / ((float)(micros) / 1000000.0)) / 1024.0;

		}
		else {
			// no more data available (at least not now)
			if(exists)
				finished = TRUE;
			else
				usleep(200000);
		}
	} while(!error && !finished);
	
	this->speed = 0.0;
	
	close(infile);
	

	if(error) {
		if(block_wait >= (DATA_TIMEOUT*5))
			this->error = E_DATA_TIMEOUT;
		else
			this->error = E_WRITEFILE_ERROR;

		return(FALSE);
	}
	
	return(TRUE);
}

bool CTCP::ReadFile(char *save_name, long size_min)
{
	int		read_length = -1, outfile_fd, block_wait = 0;
	FILE		*ok_file;
	bool		write_error = FALSE, error = FALSE;
	
	this->size = 0;
	
	if((outfile_fd = open(save_name, O_CREAT | O_TRUNC | O_RDWR)) == -1) {
		this->error = E_BAD_LOCALFILE;
		return(FALSE);
	}
		
	fchmod(outfile_fd, S_IRUSR | S_IWUSR);
	
	gettimeofday(&tv_before, NULL);
	
	while(!write_error && !error && (read_length != 0)) {
		read_length = read(this->real_data_sock_fd,  this->control_buffer, WRITE_SIZE);
		if(read_length > 0) {
			// write data	
			this->size += read_length;
			if(write(outfile_fd, this->control_buffer, read_length) != read_length)
				write_error = TRUE;
		
			// calc transfer speed
			// between tv_before and tv_after we transferred read-bytes
			gettimeofday(&tv_after, NULL);
			seconds = tv_after.tv_sec - tv_before.tv_sec;
			micros = tv_after.tv_usec - tv_before.tv_usec;
			micros += seconds * 1000000 + 1;
			this->speed = ((float)(this->size) / ((float)(micros) / 1000000.0)) / 1024.0;
			block_wait = 0;
		}
		else if(read_length == -1) {
			// we had an error
			if(errno != EAGAIN)
				error = TRUE;
			else {
				// block, wait
				if(block_wait < (DATA_TIMEOUT*5)) {
					usleep(200000);         // 0.2 secs
					block_wait++;
				}
				else
					error = TRUE;
			}
		}
	}
	
	fsync(outfile_fd);
	close(outfile_fd);

	// write error?
	if(write_error) {
		this->error = E_BAD_LOCALFILE;
		if(size_min > -1) {
			sprintf(this->temp_string, "%s%s%s", okay_dir, strrchr(save_name, '/') + 1, ".error");
			ok_file = fopen(this->temp_string, "w");
			fclose(ok_file);
		}
		this->speed = 0.0;
		return(FALSE);	
	}
	
	// see if we have a read error
	if((read_length < 0) || (this->size < size_min)) {
		if(block_wait >= (DATA_TIMEOUT*5))
			this->error = E_DATA_TIMEOUT;
		else
			this->error = E_DATA_TCPERR;

		if(size_min > -1) {
			sprintf(this->temp_string, "%s%s%s", okay_dir, strrchr(save_name, '/') + 1, ".error");
			ok_file = fopen(this->temp_string, "w");
			fclose(ok_file);
		}
		this->speed = 0.0;
		return(FALSE);	
	}
	
	// all was okay
	if(size_min > -1) {
		sprintf(this->temp_string, "%s%s%s", okay_dir, strrchr(save_name, '/') + 1, ".okay");
		ok_file = fopen(this->temp_string, "w");
		fclose(ok_file);
	}
	this->speed = 0.0;
	return(TRUE);
}
