#include <curses.h>
#include <pthread.h>
#include <string.h>
#include <stdlib.h>
#include "defines.h"
#include "tcp.h"
#include "server.h"
#include "displayhandler.h"

extern	int		mode;
extern	SERVERLIST	*global_server;

bool FilterFilename(char *filename, char *filter);

void CDisplayHandler::UpdateMode(void)
{
	wattrset(this->window_command, COLOR_PAIR(STYLE_INVERSE) | inverse_mono);
	
	if(mode == MODE_FTP_CHAIN)
		mvwaddstr(this->window_command, 1, 74, "FTP +");
	else
		mvwaddstr(this->window_command, 1, 74, "  FTP");	
		
	wnoutrefresh(this->window_command);
	doupdate();
}

void CDisplayHandler::UpdateSpeed(bool left_win, bool force)
{
	SERVERLIST	*sv_temp = global_server;
	int		magic;
	char		*busy;
	float		*speed;
	bool		found = FALSE;
		
	if(left_win) {
		magic = this->leftwindow_magic;
		busy = this->leftwindow_busy;
		if(force)
			this->speed_left_old = this->speed_left;
		speed = &(this->speed_left);
	}
	else {
		magic = this->rightwindow_magic;
		busy = this->rightwindow_busy;
		if(force)
			this->speed_right_old = this->speed_right;
		speed = &(this->speed_right);
	}

	while(!found && sv_temp) {
		if(sv_temp->server->GetMagic() == magic)
			found = TRUE;
		else
			sv_temp = sv_temp->next;
	}
	
	if(found) {
		*speed = sv_temp->server->GetSpeed();
		if(*speed < 0.0)
			*speed = 0.0;
		else if(*speed > 9999.99)
			*speed = 9999.99;
			
		if(busy && (!strcmp("[LIST]", busy) || !strcmp("[RETR]", busy) || !strcmp("[STOR]", busy)))
			this->DrawSpeed(left_win, force);
	}
}

void CDisplayHandler::DrawSpeed(bool left_win, bool force)
{
	WINDOW	*window;
	char	speed_string[] = {"[                 "}, temp[20];
	int	width;
	float	speed, *old_speed, percent;
	
	if(left_win) {
		window = this->window_left;
		width = this->window_left_width;
		old_speed = &(this->speed_left_old);
		speed = this->speed_left;
	}
	else {
		window = this->window_right;
		width = this->window_right_width;
		old_speed = &(this->speed_right_old);
		speed = this->speed_right;
	}

	percent = *old_speed * 0.1;
	
	if(force || (speed >= (*old_speed + percent)) || (speed <= (*old_speed - percent))) {
		if(!force) {
			*old_speed = speed;
		}
		
		// draw speed
		sprintf(temp, "%4.02f kb/s]", speed);
		strcpy(speed_string+14-strlen(temp), temp);
	
		wattrset(window, COLOR_PAIR(STYLE_NORMAL) | A_NORMAL);
		mvwaddstr(window, 0, width - strlen(speed_string) - 8, speed_string);
		wnoutrefresh(window);
		doupdate();
	}
}

bool CDisplayHandler::OpenView(void)
{
	FILE	*file;
	int	n;
	
	// determine number of lines of the file
	if((file = fopen(this->view_filename, "r"))) {
		n = 0;
		do {
			fgets(this->view_buffer, VIEW_BUFFER_MAX, file);
			n++;
		} while(!feof(file));
		
		fclose(file);

		this->view_lines_max = n;
	}
	else {
		sprintf(this->temp_string, "error opening '%s' for reading", this->view_filename);
		this->AddStatusLine(this->temp_string, TRUE);
		return(FALSE);
	}
	
	this->window_log = newwin(this->terminal_max_y, this->terminal_max_x, 0, 0);
	leaveok(window_log, TRUE);

	wattrset(this->window_log, COLOR_PAIR(STYLE_NORMAL) | A_NORMAL);
	wbkgdset(this->window_log, ' ' | COLOR_PAIR(STYLE_NORMAL));
	werase(this->window_log);
	wbkgdset(this->window_log, ' ');

	this->view_line = 0;
	this->RefreshView();

	return(TRUE);
}

void CDisplayHandler::RefreshView(void)
{		
	FILE	*file;
	int	n, m;
	char	*c;
	
	wattrset(this->window_log, COLOR_PAIR(STYLE_NORMAL) | A_NORMAL);
	wbkgdset(this->window_log, ' ' | COLOR_PAIR(STYLE_NORMAL));
	werase(this->window_log);
	wbkgdset(this->window_log, ' ');
	
	// find position in file and display
	if((file = fopen(this->view_filename, "r"))) {
		// we rely on correct view_line
		for(n = 0; n < this->view_line; n++)
			fgets(this->view_buffer, VIEW_BUFFER_MAX, file);
			
		n = 0;
		do {
			fgets(this->view_buffer, VIEW_BUFFER_MAX, file);
			c = strrchr(this->view_buffer, '\r');
			if(c)
				*c = '\0';
			c = strrchr(this->view_buffer, '\n');
			if(c)
				*c = '\0';
				
			mvwaddnstr(this->window_log, n, 0, this->view_buffer, this->terminal_max_x);
			n++;
		} while((n < (this->terminal_max_y - 1)) && !feof(file));
		fclose(file);
	}
	else {
		sprintf(this->temp_string, "error opening '%s' for reading", this->view_filename);
		mvwaddnstr(this->window_log, 0, 0, this->temp_string, this->terminal_max_x);
	}
	
	wattrset(this->window_log, COLOR_PAIR(STYLE_INVERSE) | this->inverse_mono);
	for(m = 0; m < this->terminal_max_x; m++)
		mvwaddch(this->window_log, this->terminal_max_y - 1, m, ' ');
		
	mvwaddnstr(this->window_log, this->terminal_max_y - 1, 0, "ESC to close - CURSOR and PGUP/DN to navigate", this->terminal_max_x);
	wnoutrefresh(this->window_log);
	doupdate();
}

void CDisplayHandler::ScrollView(bool dir_up)
{
	bool	redraw = FALSE;
	
	// to be improved (real scroll)
	if(dir_up) {
		if((this->view_line + this->terminal_max_y - 2) < this->view_lines_max) {
			redraw = TRUE;
			this->view_line++;
		}
	}
	else {
		if(this->view_line > 0) {
			redraw = TRUE;
			this->view_line--;
		}
	}
	
	if(redraw)
		this->RefreshView();
}

void CDisplayHandler::PageMoveView(bool dir_up)
{
	bool	redraw = FALSE;
	
	if(dir_up) {
		if((this->view_line + this->terminal_max_y - 2) < this->view_lines_max) {
			redraw = TRUE;
			this->view_line += this->terminal_max_y / 2;
			if((this->view_line + this->terminal_max_y - 2) >= this->view_lines_max)
				this->view_line = this->view_lines_max - this->terminal_max_y + 2;
		}
	}
	else {
		if(this->view_line > 0) {
			redraw = TRUE;
			this->view_line -= this->terminal_max_y / 2;
			if(this->view_line < 0)
				this->view_line = 0;
		}
	}
	
	if(redraw)
		this->RefreshView();
}

void CDisplayHandler::CloseView(void)
{	
	delwin(this->window_log);
	redrawwin(this->window_command);
	redrawwin(this->window_status);
	redrawwin(this->window_left);
	redrawwin(this->window_right);
}

void CDisplayHandler::OpenLog(CServer *server)
{
	int	n;
	
	this->window_log = newwin(this->terminal_max_y, this->terminal_max_x, 0, 0);
	leaveok(window_log, TRUE);

	wattrset(this->window_log, COLOR_PAIR(STYLE_NORMAL) | A_NORMAL);
	wbkgdset(this->window_log, ' ' | COLOR_PAIR(STYLE_NORMAL));
	werase(this->window_log);
	wbkgdset(this->window_log, ' ');

	// obtain log from server
	server->GetTCP()->ObtainLog((this->log));
	
	// fixup lines
	for(n = 0; n < LOG_LINES; n++) {
		if(this->log[n]) {
			if(strrchr(this->log[n], '\r'))
				*(strrchr(this->log[n], '\r')) = '\0';
			else if(strrchr(this->log[n], '\n'))
				*(strrchr(this->log[n], '\n')) = '\0';
		}
	}
	
	this->log_start = 0;
	this->RefreshLog();
}

void CDisplayHandler::ScrollLog(bool dir_up)
{
	bool	redraw = FALSE;
	
	// to be improved (real scroll)
	if(dir_up) {
		if((this->log_start + this->terminal_max_y - 2) < (LOG_LINES-1)) {
			redraw = TRUE;
			this->log_start++;
		}
	}
	else {
		if(this->log_start > 0) {
			redraw = TRUE;
			this->log_start--;
		}
	}
	
	if(redraw)
		this->RefreshLog();
}

void CDisplayHandler::PageMoveLog(bool dir_up)
{
	bool	redraw = FALSE;
	
	if(dir_up) {
		if((this->log_start + this->terminal_max_y - 2) < (LOG_LINES-1)) {
			redraw = TRUE;
			this->log_start += this->terminal_max_y / 2;
			if((this->log_start + this->terminal_max_y - 2) >= (LOG_LINES-1))
				this->log_start = (LOG_LINES-1) - this->terminal_max_y + 2;
		}
	}
	else {
		if(this->log_start > 0) {
			redraw = TRUE;
			this->log_start -= this->terminal_max_y / 2;
			if(this->log_start < 0)
				this->log_start = 0;
		}
	}
	
	if(redraw)
		this->RefreshLog();
}

void CDisplayHandler::RefreshLog(void)
{
	int	m, n, index = this->log_start + this->terminal_max_y - 2;
		
	wattrset(this->window_log, COLOR_PAIR(STYLE_NORMAL) | A_NORMAL);
	wbkgdset(this->window_log, ' ' | COLOR_PAIR(STYLE_NORMAL));
	werase(this->window_log);
	wbkgdset(this->window_log, ' ');

	for(n = 0; n < (this->terminal_max_y - 1); n++) {
		if(this->log[index])
			mvwaddnstr(this->window_log, n, 0, this->log[index], this->terminal_max_x);
		
		index--;
	}
	
	wattrset(this->window_log, COLOR_PAIR(STYLE_INVERSE) | this->inverse_mono);
	for(m = 0; m < this->terminal_max_x; m++)
		mvwaddch(this->window_log, n, m, ' ');
		
	mvwaddnstr(this->window_log, n, 0, "ESC to close - CURSOR and PGUP/DN to navigate", this->terminal_max_x);
	wnoutrefresh(this->window_log);
	doupdate();
}

void CDisplayHandler::ShowWelcome(void)
{
	int	n;
	char	logo[][52] =   {"     __.                    $$$$$$",\
				"    /  |   ____               $$$$ $$$$$'",\
				"   /   |___\\__/________/\\_______$$.$$$'_______",\
				" _/   _|\\_____________   \\  ___/.$$$$'_______  \\",\
				":\\_   \\   |    |    /    / __|.$$$$ $$.   /____/:",\
				":::\\______|____|___/    /__|.$$$$$$ $$$$. ___\\:::",\
				"<---------------- /_____\\-.$$$$$$$$ $$$$$$.----->",\
				"    fear the l33tness   .$$$$$$$$$$ $$$$$$$$.",\
				"<-pl----------(linfXP II)---------- $$$$$$$$$$.->"};


	window_welcome = newwin(14, 60, (this->terminal_max_y - this->status_win_size - 2) / 2 - 7, this->terminal_max_x / 2 - 30);
	leaveok(window_welcome, TRUE);

	wattrset(this->window_welcome, COLOR_PAIR(STYLE_NORMAL) | A_NORMAL);
	wbkgdset(this->window_welcome, ' ' | COLOR_PAIR(STYLE_NORMAL));
	werase(this->window_welcome);
	wbkgdset(this->window_welcome, ' ');
	
	wborder(window_welcome, 0, 0, 0, 0, 0, 0, 0, 0);


	mvwaddstr(window_welcome, 12, 1, "(c) pSi, dedicated to EQUALiTY");
	mvwaddstr(window_welcome, 12, 59 - strlen(PFTP_VERSION), PFTP_VERSION);
		
	/* simply diplay the logo */
	for(n = 0; n < 9; n++)
		mvwaddstr(window_welcome, n+2, 5, logo[n]);
		
	mvwaddstr(window_welcome, 0, 1, "welcome to pftp II");

	wnoutrefresh(this->window_welcome);
	doupdate();
}

void CDisplayHandler::HideWelcome(void)
{
	delwin(this->window_welcome);
	redrawwin(this->window_command);
	redrawwin(this->window_status);
	redrawwin(this->window_left);
	redrawwin(this->window_right);
}

void CDisplayHandler::CloseLog(void)
{
	int	n;
	
	// free log
	for(n = 0; n < LOG_LINES; n++) {
		delete(this->log[n]);
		this->log[n] = NULL;
	}
	
	delwin(this->window_log);
	redrawwin(this->window_command);
	redrawwin(this->window_status);
	redrawwin(this->window_left);
	redrawwin(this->window_right);
}

FILELIST *CDisplayHandler::GetFilelistEntry(FILELIST *filelist, int magic)
{
	FILELIST	*fl_temp = filelist;
	int		n = 0;
	
	while((n < magic) && fl_temp) {		// no segf check
		fl_temp = fl_temp->next;
		n++;
	}

	if(!fl_temp)
AddStatusLine("SEG AT GETFILELISTENTRY", TRUE);

	return(fl_temp);
}

char *CDisplayHandler::GetFilelistString(FILELIST *filelist, int magic, int width, bool *is_marked, bool format)
{
	FILELIST	*fl_temp = filelist;
	int		n = 0, pos;
	char		*string, filler[200], temp_name[200], owner[9];
	
	while((n < magic) && fl_temp) {
		fl_temp = fl_temp->next;
		n++;
	}

	string = new(char[width + 1]);
	for(n = 0; n < width; n++)
		filler[n] = ' ';
	
	filler[n] = '\0';
	
	if(fl_temp) {
		*is_marked = fl_temp->is_marked;

		pos = width - strlen(fl_temp->name) - 22;

		if(pos >= 0) {
			filler[pos] = '\0';
			if(!format) {
				if(!fl_temp->is_dir)
					sprintf(string, "%s%s %8ld %s", fl_temp->name, filler, fl_temp->size, fl_temp->date);
				else
					sprintf(string, "%s%s      DIR %s", fl_temp->name, filler, fl_temp->date);
			}
			else {
				strcpy(owner, "        ");
				owner[8 - strlen(fl_temp->owner)] = '\0';
				strcat(owner, fl_temp->owner);
				sprintf(string, "%s%s %s   %s", fl_temp->name, filler, owner, fl_temp->mode);
			}
				
			filler[pos] = ' ';
		}
		else {
			strcpy(temp_name, fl_temp->name);
			temp_name[width - 22] = '\0';
			
			if(!format) {
				if(!fl_temp->is_dir)
					sprintf(string, "%s %8ld %s", temp_name, fl_temp->size, fl_temp->date);
				else
					sprintf(string, "%s      DIR %s", temp_name, fl_temp->date);
			}
			else {
				strcpy(owner, "        ");
				owner[8 - strlen(fl_temp->owner)] = '\0';
				strcat(owner, fl_temp->owner);
				sprintf(string, "%s %s   %s", temp_name, owner, fl_temp->mode);
			}
		}
	}
	else {
		// should never happen, just to avoid segfault
		*is_marked = FALSE;
		strcpy(string, "<INVALID ERROR>");
	}

	return(string);
}

void CDisplayHandler::UpdateTabBar(bool dont_remove)
{
	FILELIST	*filelist_remove = NULL, *filelist_draw = NULL;
	char		*remove_string, *draw_string;
	int		remove_magic = -1, draw_magic = -1, remove_ypos = -1, draw_ypos = -1, width_remove = -1, width_draw = -1;
	WINDOW		*window_remove = NULL, *window_draw = NULL;
	bool		is_marked = FALSE, format_remove = FALSE, format_draw = FALSE;
	
	if(this->window_tab == this->window_left) {
		if(this->filelist_left) {
			// it's okay to paint new tab-bar
			if(this->filelist_right) {
				// okay to remove old bar
				filelist_remove = this->filelist_right;
				remove_magic = this->filelist_right_magic;
				window_remove = this->window_right;
				remove_ypos = this->filelist_right_ypos;
				width_remove = this->window_right_width - 2;
				format_remove = this->filelist_right_format;
			}
			filelist_draw = this->filelist_left;
			draw_magic = this->filelist_left_magic;
			window_draw = this->window_left;
			draw_ypos = this->filelist_left_ypos;
			width_draw = this->window_left_width - 2;
			format_draw = this->filelist_left_format;
		}
	}
	else {
		if(this->filelist_right) {
			// it's okay to paint new tab-bar
			if(this->filelist_left) {
				// okay to remove old bar
				filelist_remove = this->filelist_left;
				remove_magic = this->filelist_left_magic;
				window_remove = this->window_left;
				remove_ypos = this->filelist_left_ypos;
				width_remove = this->window_left_width - 2;
				format_remove = this->filelist_left_format;
			}
			filelist_draw = this->filelist_right;
			draw_magic = this->filelist_right_magic;
			window_draw = this->window_right;
			draw_ypos = this->filelist_right_ypos;
			width_draw = this->window_right_width - 2;
			format_draw = this->filelist_right_format;
		}
	}

	// remove old bar
	if(filelist_remove && !dont_remove) {
		remove_string = this->GetFilelistString(filelist_remove, remove_magic, width_remove, &is_marked, format_remove);
		if(!is_marked)
			wattrset(window_remove, COLOR_PAIR(STYLE_NORMAL) | A_NORMAL);
		else
			wattrset(window_remove, COLOR_PAIR(STYLE_MARKED) | A_BOLD);
		
		mvwaddstr(window_remove, remove_ypos, 1, remove_string);
		mvwaddch(window_remove, remove_ypos, width_remove - 21, ACS_VLINE);
		mvwaddch(window_remove, remove_ypos, width_remove - 12, ACS_VLINE);
		delete(remove_string);
		wnoutrefresh(window_remove);
	}
	
	if(filelist_draw) {
		draw_string = this->GetFilelistString(filelist_draw, draw_magic, width_draw, &is_marked, format_draw);
		if(!is_marked)
			wattrset(window_draw, COLOR_PAIR(STYLE_INVERSE) | this->inverse_mono);
		else
			wattrset(window_draw, COLOR_PAIR(STYLE_MARKED_INVERSE) | A_BOLD | this->inverse_mono);
		
		mvwaddstr(window_draw, draw_ypos, 1, draw_string);
		mvwaddch(window_draw, draw_ypos, width_draw - 21, ACS_VLINE);
		mvwaddch(window_draw, draw_ypos, width_draw - 12, ACS_VLINE);
		delete(draw_string);
		wnoutrefresh(window_draw);
	}

	doupdate();
}

void CDisplayHandler::FetchBusy(CServer *server, WINDOW *window)
{
	char	*busy;
	int	n;
		
	busy = server->ObtainBusy();
	
	if(window == this->window_left) {
		if(this->leftwindow_busy)
			delete(this->leftwindow_busy);

		this->leftwindow_busy = NULL;
		
		// remove in every case
		wattrset(window, COLOR_PAIR(STYLE_NORMAL) | A_NORMAL);
		for(n = 0; n < 7; n++)
			mvwaddch(window, 0, this->window_left_width - 8 + n, ACS_HLINE);
		
		if(busy) {			
			sprintf(this->temp_string, "[%s]", busy);
			this->leftwindow_busy = new(char[strlen(this->temp_string) + 1]);
			strcpy(this->leftwindow_busy, this->temp_string);
			delete(busy);

			// draw on display
			mvwaddstr(window, 0, this->window_left_width - strlen(this->temp_string) - 1, this->temp_string);
		}
	}
	else {
		if(this->rightwindow_busy)
			delete(this->rightwindow_busy);
		
		this->rightwindow_busy = NULL;
		
		// remove in every case
		wattrset(window, COLOR_PAIR(STYLE_NORMAL) | A_NORMAL);
		for(n = 0; n < 7; n++)
			mvwaddch(window, 0, this->window_right_width - 8 + n, ACS_HLINE);
		
		if(busy) {			
			sprintf(this->temp_string, "[%s]", busy);
			this->rightwindow_busy = new(char[strlen(this->temp_string) + 1]);
			strcpy(this->rightwindow_busy, this->temp_string);
			delete(busy);

			// draw on display
			mvwaddstr(window, 0, window_right_width - strlen(this->temp_string) - 1, this->temp_string);
		}
	}

	wnoutrefresh(window);
	doupdate();
}

void CDisplayHandler::UpdateServerBusy(WINDOW *window, char *busy)
{
	char	*old_busy, *new_busy;
	int	n, width;
	
	if(window == this->window_left) {
		old_busy = this->leftwindow_busy;
		this->leftwindow_busy = NULL;
		width = this->window_left_width;
	}
	else {
		old_busy = this->rightwindow_busy;
		this->rightwindow_busy = NULL;
		width = this->window_right_width;
	}
	
	if(old_busy)
		delete(old_busy);
	
	wattrset(window, COLOR_PAIR(STYLE_NORMAL) | A_NORMAL);

	if(!busy) {
		// remove if not busy
		for(n = 0; n < 22; n++)
			mvwaddch(window, 0, width - 23 + n, ACS_HLINE);
	}
	else  {
		// set new busy reason		
		sprintf(this->temp_string, "[%s]", busy);
		new_busy = new(char[strlen(this->temp_string) + 1]);
		strcpy(new_busy, this->temp_string);
		
		if(window == this->window_left)
			this->leftwindow_busy = new_busy;
		else
			this->rightwindow_busy = new_busy;

		// draw on display
		mvwaddstr(window, 0, width - strlen(new_busy) - 1, new_busy);
	}

	wnoutrefresh(window);
	doupdate();
}

void CDisplayHandler::UpdateServerCWD(CServer *server, WINDOW *window)
{
	int	width;
	char	*cwd;
		
	// updates CWD
	if(window == this->window_left) {
		width = this->window_left_width;
		cwd = this->window_left_cwd;
	}
	else {
		width = this->window_right_width;
		cwd = this->window_right_cwd;
	}
		
	// obtain copy of new working-dir from server
	server->ObtainWorkingDir(cwd);
		
	// display cwd
	wattrset(window, COLOR_PAIR(STYLE_NORMAL) | A_NORMAL);

	if(strlen(cwd) <= (unsigned)(width - 2))
		mvwaddstr(window, this->terminal_max_y - this->status_win_size - 3, 1, cwd);
	else
		mvwaddstr(window, this->terminal_max_y - this->status_win_size - 3, 1, cwd + strlen(cwd) - (width - 2));

	wnoutrefresh(window);
	doupdate();
}

void CDisplayHandler::FileToggleMark(void)
{
	int		magic;
	FILELIST	*fl_temp;
	bool		found = FALSE;
	
	if(this->window_tab == this->window_left) {
		fl_temp = this->filelist_left;
		magic = this->filelist_left_magic;
	}
	else {
		fl_temp = this->filelist_right;
		magic = this->filelist_right_magic;
	}

	while(!found && fl_temp) {
		if(fl_temp->magic == magic)
			found = TRUE;
		else
			fl_temp = fl_temp->next;
	}
	
	if(found) {
		fl_temp->is_marked = !fl_temp->is_marked;
		this->UpdateFilelistScroll(TRUE, TRUE);
	}
}

void CDisplayHandler::UpdateFilelistScroll(bool dir_up, bool redraw_everytime)
{
	FILELIST	*fl_start;
	char		*cwd, *label, *string, *busy;
	int		n, width, height = this->terminal_max_y - this->status_win_size - 4, old_ypos, *ypos, old_magic, *magic, entries;
	bool		is_left, movetab = redraw_everytime, scroll = FALSE, is_marked, format;
	
	if(this->window_tab == this->window_left) {
		fl_start = this->filelist_left;
		entries = this->filelist_left_entries;
		magic = &(this->filelist_left_magic);
		ypos = &(this->filelist_left_ypos);
		cwd = this->window_left_cwd;
		busy = this->leftwindow_busy;
		label = this->window_left_label;
		width = this->window_left_width - 2;
		format = this->filelist_left_format;
		is_left = TRUE;
	}
	else {
		fl_start = this->filelist_right;
		entries = this->filelist_right_entries;
		magic = &(this->filelist_right_magic);
		ypos = &(this->filelist_right_ypos);
		cwd = this->window_right_cwd;
		busy = this->rightwindow_busy;
		label = this->window_right_label;
		width = this->window_right_width - 2;
		format = this->filelist_right_format;
		is_left = FALSE;
	}

	// determine direction
	old_ypos = *ypos;
	old_magic = *magic;
	
	if(dir_up) {
		if(*magic < (entries - 1)) {
			(*magic)++;
			if(*ypos < height) {
				movetab = TRUE;
				(*ypos)++;
			}
			else
				scroll = TRUE;
		}
	}
	else {
		if(*magic > 0) {
			(*magic)--;
			if(*ypos > 1) {
				movetab = TRUE;
				(*ypos)--;
			}
			else
				scroll = TRUE;
		}
	}

	// actually reflect the changings
	if(movetab || scroll) {
		// remove tabbar
		wattrset(this->window_tab, COLOR_PAIR(STYLE_NORMAL) | A_NORMAL);
		string = this->GetFilelistString(fl_start, old_magic, width, &is_marked, format);
		if(!is_marked)
			wattrset(this->window_tab, COLOR_PAIR(STYLE_NORMAL) | A_NORMAL);
		else
			wattrset(this->window_tab, COLOR_PAIR(STYLE_MARKED) | A_BOLD);
		
		mvwaddstr(this->window_tab,  old_ypos, 1, string);
		mvwaddch(this->window_tab, old_ypos, width - 21, ACS_VLINE);
		mvwaddch(this->window_tab, old_ypos, width - 12, ACS_VLINE);
		delete(string);
	}
	
	if(scroll) {
		wattrset(this->window_tab, COLOR_PAIR(STYLE_NORMAL) | A_NORMAL);
		wborder(this->window_tab, 0, 0, 0, 0, 0, 0, 0, 0);
		
		if(dir_up) {
			// scroll [2:height] one up
			wscrl(this->window_tab, 1);
		}
		else {
			// scroll [1:height-1] one down
			wscrl(this->window_tab, -1);
		}

		// we wiped some info, redraw
		wattrset(this->window_tab, COLOR_PAIR(STYLE_NORMAL) | A_NORMAL);
		wborder(this->window_tab, 0, 0, 0, 0, 0, 0, 0, 0);

		if(cwd) {
			wattrset(this->window_tab, COLOR_PAIR(STYLE_NORMAL) | A_NORMAL);

			if(strlen(cwd) <= (unsigned)width)
				mvwaddstr(this->window_tab, this->terminal_max_y - this->status_win_size - 3, 1, cwd);
			else
				mvwaddstr(this->window_tab, this->terminal_max_y - this->status_win_size - 3, 1, cwd + strlen(cwd) - width);
		}

		if(label) {
			wattrset(this->window_tab, COLOR_PAIR(STYLE_NORMAL) | A_NORMAL);

			if(strlen(label) <= (unsigned)width)
				mvwaddstr(this->window_tab, 0, 1, label);
			else
				mvwaddstr(this->window_tab, 0, 1, label + strlen(label) - width);
		}

		wattrset(this->window_tab, COLOR_PAIR(STYLE_NORMAL) | A_NORMAL);
		for(n = 0; n < 22; n++)
			mvwaddch(this->window_tab, 0, width + 2 - 23 + n, ACS_HLINE);

		if(busy) {
			mvwaddstr(this->window_tab, 0, width + 2 - strlen(busy) - 1, busy);
			this->UpdateSpeed(is_left, FALSE);
		}

	}

	if(movetab || scroll) {
		// draw actual line with bar
		string = this->GetFilelistString(fl_start, *magic, width, &is_marked, format);
		if(!is_marked)
			wattrset(this->window_tab, COLOR_PAIR(STYLE_INVERSE) | this->inverse_mono);
		else
			wattrset(this->window_tab, COLOR_PAIR(STYLE_MARKED_INVERSE) | A_BOLD | this->inverse_mono);
		
		mvwaddstr(this->window_tab,  *ypos, 1, string);
		mvwaddch(this->window_tab, *ypos, width - 21, ACS_VLINE);
		mvwaddch(this->window_tab, *ypos, width - 12, ACS_VLINE);
		delete(string);

		wnoutrefresh(this->window_tab);
		doupdate();
	}
}

void CDisplayHandler::UpdateFilelistPageMove(bool dir_up)
{
	WINDOW	*window;
	int	entries, *magic, jump = (this->terminal_max_y - this->status_win_size - 4) / 2;
	
	if(this->window_tab == this->window_left) {
		window = this->window_left;
		entries = this->filelist_left_entries;
		magic = &(this->filelist_left_magic);
	}
	else {
		window = this->window_right;
		entries = this->filelist_right_entries;
		magic = &(this->filelist_right_magic);
	}
	
	// determine how far we should jump
	if(dir_up) {
		if((*magic + jump) > (entries - 1))
			*magic = entries - 1;
		else
			*magic += jump;
	}
	else {
		if((*magic - jump) < 0)
			*magic = 0;
		else
			*magic -= jump;
	}
	
	this->UpdateFilelistNewPosition(window);
	this->UpdateTabBar(FALSE);
}

void CDisplayHandler::UpdateFilelistNewPosition(WINDOW *window)
{
	FILELIST	*fl_start, *fl_temp;
	char		filler[200], temp_name[200], owner[9], *busy, *cwd, *label;
	int		entries, magic;
	int		draw_ypos = 1, *ypos, pos, n, width, height = this->terminal_max_y - this->status_win_size - 4;
	bool		format, is_left;

	// refresh filelist from scratch, using actual magic

	// determine number of entries on filelist
	if(window == this->window_left) {
		fl_start = this->filelist_left;
		width = this->window_left_width - 2;
		entries = this->filelist_left_entries;
		magic = this->filelist_left_magic;
		ypos = &(this->filelist_left_ypos);
		cwd = this->window_left_cwd;
		busy = this->leftwindow_busy;
		label = this->window_left_label;
		format = this->filelist_left_format;
		is_left = TRUE;
	}
	else {
		fl_start = this->filelist_right;
		width = this->window_right_width - 2;
		entries = this->filelist_right_entries;
		magic = this->filelist_right_magic;
		ypos = &(this->filelist_right_ypos);
		cwd = this->window_right_cwd;
		busy = this->rightwindow_busy;
		label = this->window_right_label;
		format = this->filelist_right_format;
		is_left = FALSE; 
	}
		
	// now we got to determine how to display the filelist
	// strategie is as follows:
	// - entries <= height :	don't care, start from beginning and make *ypos = magic + 1
	// - entries > height  :	see if we can set *ypos to 1 and draw the remaining list until height reached
	//				otherwise draw that last entry is on last window_line and set *ypos to reflect the tab-height

	// erase screen
	wattrset(window, COLOR_PAIR(STYLE_NORMAL) | A_NORMAL);
	wbkgdset(window, ' ' | COLOR_PAIR(STYLE_NORMAL));
	werase(window);
	wbkgdset(window, ' ');
	wborder(window, 0, 0, 0, 0, 0, 0, 0, 0);

	for(n = 0; n < width; n++)
		filler[n] = ' ';
	
	filler[n] = '\0';

	if(entries <= height) {
		// the easy case, less entries than screenheight
		*ypos = magic + 1;
		
		fl_temp = fl_start;
	}
	else {
		if(magic <= (entries - height)) {
			// the not-so-easy-but-handleable-case
			*ypos = 1;
			
			fl_temp = fl_start;
			for(n = 0; n < magic; n++) {
				fl_temp = fl_temp->next;	// it cannot segfault here
			}
			
		}
		else {
			// worst case. as with murphy, i'm sure it's in here almost all the time :P
			fl_temp = fl_start;
			for(n = 0; n < (entries - height); n++)
				fl_temp = fl_temp->next;	// again, this is safe

			*ypos = magic - (entries - height) + 1;
		}
	}
		
	// actually draw the lines
	while((draw_ypos <= height) && fl_temp) {
		pos = width - strlen(fl_temp->name) - 22;

		if(pos >= 0) {
			filler[pos] = '\0';
			if(!format) {
				if(!fl_temp->is_dir)
					sprintf(this->temp_string, "%s%s %8ld %s", fl_temp->name, filler, fl_temp->size, fl_temp->date);
				else
					sprintf(this->temp_string, "%s%s      DIR %s", fl_temp->name, filler, fl_temp->date);
			}
			else {
				strcpy(owner, "        ");
				owner[8 - strlen(fl_temp->owner)] = '\0';
				strcat(owner, fl_temp->owner);
				sprintf(this->temp_string, "%s%s %s   %s", fl_temp->name, filler, owner, fl_temp->mode);
			}
			filler[pos] = ' ';
		}
		else {
			strcpy(temp_name, fl_temp->name);
			temp_name[width - 22] = '\0';
			
			if(!format) {
				if(!fl_temp->is_dir)
					sprintf(this->temp_string, "%s %8ld %s", temp_name, fl_temp->size, fl_temp->date);
				else
					sprintf(this->temp_string, "%s      DIR %s", temp_name, fl_temp->date);
			}
			else {
				strcpy(owner, "        ");
				owner[8 - strlen(fl_temp->owner)] = '\0';
				strcat(owner, fl_temp->owner);
				sprintf(this->temp_string, "%s %s   %s", temp_name, owner, fl_temp->mode);
			}
		}

		if(!fl_temp->is_marked)
			wattrset(window, COLOR_PAIR(STYLE_NORMAL) | A_NORMAL);
		else
			wattrset(window, COLOR_PAIR(STYLE_MARKED) | A_BOLD);

		mvwaddstr(window, draw_ypos, 1, this->temp_string);
		mvwaddch(window, draw_ypos, width - 21, ACS_VLINE);
		mvwaddch(window, draw_ypos, width - 12, ACS_VLINE);

		fl_temp = fl_temp->next;
		draw_ypos++;
	}

	// we wiped some information, redraw...
	if(cwd) {
		wattrset(window, COLOR_PAIR(STYLE_NORMAL) | A_NORMAL);

		if(strlen(cwd) <= (unsigned)width)
			mvwaddstr(window, this->terminal_max_y - this->status_win_size - 3, 1, cwd);
		else
			mvwaddstr(window, this->terminal_max_y - this->status_win_size - 3, 1, cwd + strlen(cwd) - width);
	}
		
	if(label) {
		wattrset(window, COLOR_PAIR(STYLE_NORMAL) | A_NORMAL);

		if(strlen(label) <= (unsigned)width)
			mvwaddstr(window, 0, 1, label);
		else
			mvwaddstr(window, 0, 1, label + strlen(label) - width);
	}
		
	wattrset(window, COLOR_PAIR(STYLE_NORMAL) | A_NORMAL);
	for(n = 0; n < 22; n++)
		mvwaddch(window, 0, width + 2 - 23 + n, ACS_HLINE);

	if(busy) {
		mvwaddstr(window, 0, width + 2 - strlen(busy) - 1, busy);
		this->UpdateSpeed(is_left, FALSE);
	}

	wnoutrefresh(window);
	doupdate();
	
}

void CDisplayHandler::UpdateServerFilelist(CServer *server, WINDOW *window)
{
	FILELIST	*fl_temp, *fl_temp1;
	int		n, *entries, *magic, *jump_stack_pos, *jump_stack, m;
	bool		use_jump, has_dirs = FALSE, found;
	time_t		elapsed_time, newest_time[JUMP_STACK];
	
	// free old list and obtain copy of new filelist
	if(window == this->window_left) {
		fl_temp = this->filelist_left;
		magic = &(this->filelist_left_magic);
		jump_stack_pos = &(this->jump_stack_left_pos);
		jump_stack = this->jump_stack_left;
		entries = &(this->filelist_left_entries);
	}
	else {
		fl_temp = this->filelist_right;
		magic = &(this->filelist_right_magic);
		jump_stack_pos = &(this->jump_stack_right_pos);
		jump_stack = this->jump_stack_right;
		entries = &(this->filelist_right_entries);
	}
	
	while(fl_temp) {
		fl_temp1 = fl_temp;
		fl_temp = fl_temp->next;
		delete(fl_temp1->name);
		delete(fl_temp1);
	}

	if(window == this->window_left)
		fl_temp = this->filelist_left = server->ObtainFilelist(&use_jump);
	else
		fl_temp = this->filelist_right = server->ObtainFilelist(&use_jump);

	// no seg-fault protection. it MUST work since noone modifies the list/magic other than this thread
	fl_temp1 = fl_temp;
	*entries = 0;
	while(fl_temp1) {
		if(fl_temp1->is_dir && strcmp(fl_temp1->name, ".."))
			has_dirs = TRUE;
			
		(*entries)++;
		fl_temp1 = fl_temp1->next;
	}
	
	for(n = 0; n < JUMP_STACK; n++)
		*(jump_stack+n) = -1;
		
	*jump_stack_pos = 0;

	if(use_jump) {
		for(n = 0; n < JUMP_STACK; n++)
			newest_time[n] = 0;
		
		// now determine the X newest dirs/files (used for cycle-through-newest)
		fl_temp1 = fl_temp;
		while(fl_temp1) {
			if(fl_temp1->is_dir == has_dirs) {
				if(FilterFilename(fl_temp1->name, server->GetFilter())) {
					elapsed_time = fl_temp1->time;

					// now see if we got a newer entry
					found = FALSE;
					n = 0;
					while(!found && (n < JUMP_STACK)) {
						if(elapsed_time >= newest_time[n])
							found = TRUE;
						else
							n++;
					}

					if(found) {
						// okiez, insert this one as 'n'
						for(m = JUMP_STACK-1; m > n; m--) {
							newest_time[m] = newest_time[m-1];
							*(jump_stack+m) = *(jump_stack+m-1);
						}
					
						newest_time[n] = elapsed_time;
						*(jump_stack+n) = fl_temp1->magic;
					}
				}
			}
			fl_temp1 = fl_temp1->next;
		}
				
		if(*jump_stack != -1)
			*magic = *jump_stack;
		else
			*magic = 0;
	}
	else
		*magic = 0;
		
	this->UpdateFilelistNewPosition(window);
	this->UpdateTabBar(TRUE);
}

void CDisplayHandler::ScrollStatusUp(void)
{
	if((this->status_line < (STATUS_LOG - this->status_win_size)) && (this->statuslog[this->status_line + 1] != NULL))
		this->status_line += 1;
		
	this->DisplayStatusLog();
}

void CDisplayHandler::ScrollStatusDown(void)
{
	if(this->status_line > 0)
		this->status_line -= 1;
		
	this->DisplayStatusLog();
}

void CDisplayHandler::DisplayStatusLog(void)
{
	int	n, ypos = this->status_win_size - 1;

	wattrset(this->window_status, COLOR_PAIR(STYLE_NORMAL) | A_NORMAL);
	wbkgdset(this->window_status, ' ' | COLOR_PAIR(STYLE_NORMAL));
	werase(this->window_status);
	wbkgdset(this->window_status, ' ');

	for(n = 0; n < this->status_win_size; n++) {
		if(this->statuslog_highlight[this->status_line + n])
			wattrset(this->window_status, COLOR_PAIR(STYLE_MARKED) | A_BOLD);
		else
			wattrset(this->window_status, COLOR_PAIR(STYLE_NORMAL) | A_NORMAL);

		mvwaddnstr(this->window_status, ypos, 0, this->statuslog[this->status_line + n], this->terminal_max_x);
		ypos--;
	}

	wnoutrefresh(this->window_status);
	doupdate();
}

void CDisplayHandler::AddStatusLine(char *line, bool highlight)
{
	char	*new_line;
	int	n;
	
	this->status_line = 0;
	new_line = new(char[strlen(line) + 1]);
	strcpy(new_line, line);

	// move log up	
	delete(this->statuslog[STATUS_LOG-1]);
	for(n = (STATUS_LOG - 2); n >= 0; n--) {
		this->statuslog[n+1] = this->statuslog[n];
		this->statuslog_highlight[n+1] = this->statuslog_highlight[n];
	}
	
	this->statuslog[0] = new_line;
	this->statuslog_highlight[0] = highlight;
	
	this->DisplayStatusLog();
}
