
#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <memory.h>
#include <sys/types.h>
#include <dirent.h>
#include <fcntl.h>
#include <utime.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <grp.h>
#include <pwd.h>
#include <signal.h>
#include <syslog.h>
#include "zanserv.h"

int		gsockfd = 0;
int		gdaemonsocket = 0;

time_t	glasttime = 0;

char	ProgramName[256];
char	HomePath[MAX_PATH];
char	HomePathEnv[MAX_PATH];

ULONG	ulMaxReaddirs = ZANNET_MAX_READDIRS;
ULONG	ulDefaultAttributes = S_IRUSR | S_IWUSR | S_IRGRP | S_IROTH;

int		fDebug = TRUE;

char	InBuff[ZANSERV_IN_BUFFER_SIZE];
char	OutBuff[ZANSERV_OUT_BUFFER_SIZE];

char	NativeSlash = '/';
char	NativeSlashStr[] = "/";
char	WindowsBSlash = '\\';

char magic_char = '~';
unsigned char dos_char_map[256];

char gkeepalivestr[] = ZANNET_KEEPALIVE_STR;

/* Windows to Unix error mappings */

USHORT WindowsErrors[ESTALE+1] = 
{
	NO_ERROR,					/* SUCCESS */
	ERROR_INVALID_FUNCTION,		/* EPERM   1  Operation not permitted              */
	ERROR_FILE_NOT_FOUND,		/* ENOENT  2  No such file or directory            */
	ERROR_INVALID_FUNCTION,		/* ESRCH   3  No such process                      */
	ERROR_UNEXP_NET_ERR,		/* EINTR   4  interrupted system call              */
	ERROR_UNEXP_NET_ERR,		/* EIO     5  I/O error                            */
	ERROR_UNEXP_NET_ERR,		/* ENXIO   6  No such device or address            */
	ERROR_UNEXP_NET_ERR,		/* E2BIG   7  Arg list too long                    */
	ERROR_UNEXP_NET_ERR,		/* ENOEXEC 8  Exec format error                    */
	ERROR_INVALID_HANDLE,		/* EBADF   9  Bad file descriptor                  */
	ERROR_UNEXP_NET_ERR,		/* ECHILD  10  No child processes                   */
	ERROR_DEV_NOT_EXIST,		/* EAGAIN  11  Resource temporarily unavailable     */
	ERROR_DISK_FULL,			/* ENOMEM  12  Not enough space                     */
	ERROR_ACCESS_DENIED,		/* EACCES  13  Permission denied                    */
	ERROR_INVALID_ADDRESS,		/* EFAULT  14  Bad address                          */
	ERROR_UNEXP_NET_ERR,		/* ENOTBLK 15  Block device required                */
	ERROR_NETWORK_BUSY,			/* EBUSY   16  Resource busy                        */
	ERROR_FILE_EXISTS,			/* EEXIST  17  File exists                          */
	ERROR_UNEXP_NET_ERR,		/* EXDEV   18  Improper link                        */
	ERROR_UNEXP_NET_ERR,		/* ENODEV  19  No such device                       */
	ERROR_PATH_NOT_FOUND,		/* ENOTDIR 20  Not a directory                      */
	ERROR_DIR_NOT_EMPTY,		/* EISDIR  21  Is a directory                       */
	ERROR_INVALID_PARAMETER,	/* EINVAL  22  Invalid argument                     */
	ERROR_TOO_MANY_OPEN_FILES,	/* ENFILE  23  Too many open files in system        */
	ERROR_TOO_MANY_OPEN_FILES,	/* EMFILE  24  Too many open files                  */
	ERROR_UNEXP_NET_ERR,		/* ENOTTY  25  Inappropriate I/O control operation  */
	ERROR_NETWORK_BUSY,			/* ETXTBSY 26  Text file busy                       */
	ERROR_DISK_FULL,			/* EFBIG   27  File too large                       */
	ERROR_DISK_FULL,			/* ENOSPC  28  No space left on device              */
	ERROR_SEEK_ON_DEVICE,		/* ESPIPE  29  Invalid seek                         */
	ERROR_ACCESS_DENIED,		/* EROFS   30  Read only file system                */
	ERROR_UNEXP_NET_ERR,		/* EMLINK  31  Too many links                       */
	ERROR_BROKEN_PIPE,			/* EPIPE   32  Broken pipe                          */
	ERROR_UNEXP_NET_ERR,		/* EDOM    33  Domain error within math function    */
	ERROR_UNEXP_NET_ERR,		/* ERANGE  34  Result too large                     */
	ERROR_UNEXP_NET_ERR,		/* ENOMSG  35  No message of desired type           */
	ERROR_UNEXP_NET_ERR,		/* EIDRM   36  Identifier removed                   */
	ERROR_UNEXP_NET_ERR,		/* ECHRNG  37  Channel number out of range          */
	ERROR_UNEXP_NET_ERR,		/* EL2NSYNC 38 Level 2 not synchronized             */
	ERROR_UNEXP_NET_ERR,		/* EL3HLT  39  Level 3 halted                       */
	ERROR_UNEXP_NET_ERR,		/* EL3RST  40  Level 3 reset                        */
	ERROR_UNEXP_NET_ERR,		/* ELNRNG  41  Link number out of range             */
	ERROR_UNEXP_NET_ERR,		/* EUNATCH 42  Protocol driver not attached         */
	ERROR_UNEXP_NET_ERR,		/* ENOCSI  43  No CSI structure available           */
	ERROR_UNEXP_NET_ERR,		/* EL2HLT  44  Level 2 halted                       */
	ERROR_POSSIBLE_DEADLOCK,	/* EDEADLK 45  Resource deadlock avoided            */
	ERROR_NOT_READY,			/* ENOTREADY   46  Device not ready					*/
	ERROR_WRITE_PROTECT,		/* EWRPROTECT      47 Write-protected media        */
	ERROR_NOT_DOS_DISK,			/* EFORMAT         48 Unformatted media            */
	ERROR_LOCK_VIOLATION,		/* ENOLCK          49 No locks available           */
	ERROR_UNEXP_NET_ERR,		/* ENOCONNECT      50 no connection                */
	0,							/* nothing         51  */
	ERROR_NOT_DOS_DISK			/* ESTALE          52 no filesystem                */
};


/* Global options */

int fOptMultiUser = FALSE;
int fOptGuestOnly = FALSE;

#ifdef ZANNET_DAEMON_MODE

int fOptDaemonMode = FALSE;
int fOptDebugMode = FALSE;

#endif

int fOptHomeDirChange = FALSE;
int fOptIP = FALSE;
int fOptPort = FALSE;

/* CONSIDER: int fOptBasicLogging = FALSE; */
/* CONSIDER: int fOptUserLogging = FALSE; */
/* CONSIDER: int fOptLogNewFiles = FALSE; */
/* CONSIDER: int fOptLogOpenedFiles = FALSE; */

/* Daemon stuff */

struct sockaddr_in peer_addr;
char RemoteHost[64];
char RemoteAddr[64];
char user[64];

int main (argc, argv)
int argc;
char *argv[];
{
	char	hostIP[64];
	int		winsock;
	int		port=0;
	int		i, on=1;
	char	*pEnv;
	char	*pOpt;
	struct	sigaction act;
#ifdef IPTOS_LOWDELAY
    int tos;
#endif

	if ( (sizeof(ULONG) != 4) || (sizeof(USHORT) != 2) || (sizeof(char) != 1) )
	{
		printf ("\nERROR! This code depends on 4 byte ints, 2 bytes shorts, and 1 bytes chars!\n");
		printf ("\nThis compilation is as follows:\n");
		printf ("\n    sizeof(int) = %d\n", sizeof(int)); 
		printf ("\n    sizeof(short) = %d\n", sizeof(short)); 
		printf ("\n    sizeof(char) = %d\n", sizeof(char)); 
		printf ("\nChange ULONG and USHORT in zanserv.h to 4 and 2 bytes values and recompile.\n\n");
		exit (1);
	}

	strcpy (ProgramName, argv[0]);

	hostIP[0] = 0;

	/* No parms--just display usage */

	if (argc < 2)
		Usage ();

	/* We allow for some ftpd options, but don't support them all */

	for (i = 1; i < argc; i++)
	{
		pOpt = argv[i];

		if (*pOpt != '-')
			Usage ();

		pOpt++;

		switch (*pOpt) 
		{
		case 'm': /* Multiuser mode */ 
			fOptMultiUser = TRUE;
			break;
		case 'g': /* Guest logins only (users: anonymous or ftp) */
			fOptGuestOnly = TRUE;
			break;

#ifdef ZANNET_DAEMON_MODE

		case 'd': /* Daemon mode (not recommended--use inetd) */
			fOptDaemonMode = TRUE;
			break;
		case 'b': /* Debug mode */
			fOptDebugMode = TRUE;
			break;
#endif

		case 'r': /* Allow home directory change */
			fOptHomeDirChange = TRUE;
			break;

		/* CONSIDER:
		case 'd': -- Write debug information to the syslog 
			fOptBasicLogging = TRUE;
			break;
		case 'l': -- Log each zannet session to the syslog 
			fOptUserLogging = TRUE;
			break;
		case 'i': -- Log new files created by the ZanNet server 
			fOptLogNewFiles = TRUE;
			break;
		case 'o': -- Log existing files opened by the ZanNet server 
			fOptLogOpenedFiles = TRUE;
			break;
		*/

		/* (Telnet only--hidden, do not document) */

		case 'I': /* IP address */
			fOptIP = TRUE;
			pOpt++;
			if (*pOpt != 0)
				strcpy (hostIP, pOpt);
			break;
		case 'P': /* Port */
			fOptPort = TRUE;
			pOpt++;
			if (*pOpt != 0)
				port = atoi (pOpt);
			break;

		default:
			printf("unknown flag -%c\n\n", *pOpt);
			Usage ();
			break;
		}
	}

	/* Don't allow any secret Telnet options for daemon */

#ifdef ZANNET_DAEMON_MODE

	if (fOptIP || fOptPort)
		if (fOptMultiUser || fOptGuestOnly || fOptDaemonMode || fOptDebugMode || fOptHomeDirChange)
				Usage ();

	/* Daemon mode must include multi user mode */

	if (fOptDaemonMode && !fOptMultiUser)
		Usage ();

#else

	if (fOptIP || fOptPort)
		if (fOptMultiUser || fOptGuestOnly || fOptHomeDirChange)
				Usage ();
#endif

	/* Wrong parms--just display usage */

	if (!fOptMultiUser && !(fOptIP || fOptPort))
		Usage ();

	InitDosCharMap ();

	/* Setup signal catching */

	act.sa_handler = SigHandler;
	act.sa_flags = 0;

#ifdef SIGHUP
    (void) sigaction (SIGHUP, &act, NULL);
#endif
#ifdef SIGINT
    (void) sigaction (SIGINT, &act, NULL);
#endif
#ifdef SIGQUIT
    (void) sigaction (SIGQUIT, &act, NULL);
#endif
#ifdef SIGILL
    (void) sigaction (SIGILL, &act, NULL);
#endif
#ifdef SIGTRAP
    (void) sigaction (SIGTRAP, &act, NULL);
#endif
#ifdef SIGIOT
    (void) sigaction (SIGIOT, &act, NULL);
#endif
#ifdef SIGEMT
    (void) sigaction (SIGEMT, &act, NULL);
#endif
#ifdef SIGFPE
    (void) sigaction (SIGFPE, &act, NULL);
#endif
#ifdef SIGKILL
    (void) sigaction (SIGKILL, &act, NULL);
#endif
#ifdef SIGBUS
    (void) sigaction (SIGBUS, &act, NULL);
#endif
#ifdef SIGSEGV
    (void) sigaction (SIGSEGV, &act, NULL);
#endif
#ifdef SIGSYS
    (void) sigaction (SIGSYS, &act, NULL);
#endif
#ifdef SIGALRM
    (void) sigaction (SIGALRM, &act, NULL);
#endif
#ifdef SIGSTOP
    (void) sigaction (SIGSTOP, &act, NULL);
#endif
#ifdef SIGTSTP
    (void) sigaction (SIGTSTP, &act, NULL);
#endif
#ifdef SIGTTIN
    (void) sigaction (SIGTTIN, &act, NULL);
#endif
#ifdef SIGTTOU
    (void) sigaction (SIGTTOU, &act, NULL);
#endif
#ifdef SIGIO
    (void) sigaction (SIGIO, &act, NULL);
#endif
#ifdef SIGXCPU
    (void) sigaction (SIGXCPU, &act, NULL);
#endif
#ifdef SIGXFSZ
    (void) sigaction (SIGXFSZ, &act, NULL);
#endif
#ifdef SIGWINCH
    (void) sigaction (SIGWINCH, &act, NULL);
#endif
#ifdef SIGVTALRM
    (void) sigaction (SIGVTALRM, &act, NULL);
#endif
#ifdef SIGPROF
    (void) sigaction (SIGPROF, &act, NULL);
#endif
#ifdef SIGUSR1
    (void) sigaction (SIGUSR1, &act, NULL);
#endif
#ifdef SIGUSR2
    (void) sigaction (SIGUSR2, &act, NULL);
#endif
#ifdef SIGPIPE
    (void) sigaction (SIGPIPE, &act, NULL);
#endif
#ifdef SIGCHLD
    (void) sigaction (SIGCHLD, &act, NULL);
#endif
#ifdef SIGURG
    (void) sigaction (SIGURG, &act, NULL);
#endif

	/* Use server daemon if needed */

	if (fOptMultiUser)
	{
		winsock = MultiUserMode (hostIP, &port);
	}
	else
	{
		fOptHomeDirChange = TRUE;
		winsock = 0;

		/* Ask for low delay */

#ifdef IPTOS_LOWDELAY
		tos = IPTOS_LOWDELAY;
		if (setsockopt (winsock, IPPROTO_IP, IP_TOS, (char *)&tos, sizeof(int)) < 0)
			if (fOptMultiUser)
				syslog(LOG_WARNING, "setsockopt (IP_TOS): %m");
#endif
	}

	/* Try to handle urgent data inline */

#ifdef SO_OOBINLINE
    if (setsockopt (winsock, SOL_SOCKET, SO_OOBINLINE, (char *)&on, sizeof(int)) < 0)
		if (fOptMultiUser)
			syslog(LOG_ERR, "setsockopt (SO_OOBINLINE): %m");
#endif

	/* Make sure we own our socket */

#ifdef  F_SETOWN
	if (fOptMultiUser)
		if (fcntl (winsock, F_SETOWN, getpid()) == -1)
	        syslog(LOG_ERR, "fcntl F_SETOWN: %m");
#endif
	
	/* Connect back to Windows for data connection */

	gsockfd = ConnectToWindows (hostIP, port);

	/* Tell Windows we have connected successfully */

	writen (winsock, ZANNET_CONNECT_STRING, sizeof(ZANNET_CONNECT_STRING)); 
	writen (winsock, "\n", 1); 

	/* Set default home path */

	if ((pEnv = (char *)getenv("HOME")))
		strcpy (HomePath, pEnv);
	else
		strcpy (HomePath, "/");

	time(&glasttime);

	/* We handle all permission bits so turn off file creation mask */

	umask (~(S_IRWXU | S_IRWXG | S_IRWXO));

	/* Install signal handler for read timeouts */

	act.sa_handler = SigReadTimeout;
	act.sa_flags = 0;

    (void) sigaction (SIGALRM, &act, NULL);

	/* Process mesages from Windows client */

	ProcessMessages (gsockfd);

	return 0;
}


/*
*  Connect to the Windows 95 machine 
*/
int ConnectToWindows (hostIP, port)
char hostIP[];
int port;
{
	int sockfd, on=1;
    struct	sockaddr_in	serv_addr;
#ifdef IPTOS_LOWDELAY
    int tos;
#endif

	/* Open the socket */

    memset ((char *)&serv_addr, 0, sizeof(serv_addr));

    serv_addr.sin_family = AF_INET;
    serv_addr.sin_addr.s_addr = inet_addr (hostIP);
    serv_addr.sin_port = htons (port);

	if ((sockfd = socket (AF_INET, SOCK_STREAM, 0)) < 0)
	{
		if (fDebug)
		{
			printf ("%s Socket Error:  socket(), rc = %d\n", ProgramName, errno);
			perror ("Unrecoverable ZanNet Error");
		}
		exit (1);
	}

    /* Connect to Windows */

    if (connect (sockfd, (struct sockaddr *)&serv_addr, sizeof(serv_addr)) < 0)
	{
		if (fDebug)
		{
			printf ("%s Socket Error:  connect(), port = %d, rc = %d, ip = %s\n", 
				ProgramName, port, errno, hostIP);
			perror ("Unrecoverable ZanNet Error");
		}
		exit (1);
	}

	/* Try to handle urgent data inline */

#ifdef SO_OOBINLINE
    if (setsockopt (sockfd, SOL_SOCKET, SO_OOBINLINE, (char *)&on, sizeof(int)) < 0)
		if (fOptMultiUser)
			syslog(LOG_ERR, "setsockopt (SO_OOBINLINE): %m");
#endif

	/* Ask for low delay */

#ifdef IPTOS_LOWDELAY
    tos = IPTOS_LOWDELAY;
    if (setsockopt (sockfd, IPPROTO_IP, IP_TOS, (char *)&tos, sizeof(int)) < 0)
		if (fOptMultiUser)
			syslog(LOG_WARNING, "setsockopt (IP_TOS): %m");
#endif

	return sockfd;
}


/*
*  Exit
*/
void ShutDown (rc)
int rc;
{
	if (gsockfd)
		close (gsockfd); 

	gsockfd = 0;

	if (gdaemonsocket)
		close (gdaemonsocket);

	gdaemonsocket = 0;
	
	exit (rc);
}


/*
*  Signal Handler
*/
void SigHandler (int sig)
{


	if (fOptMultiUser)
	{
		switch (sig)
		{
		case SIGPIPE:
			syslog(LOG_DEBUG, "lost connection to %s [%s]", RemoteHost, RemoteAddr);
			break;
		case SIGALRM:
			syslog(LOG_NOTICE, "LOGIN TIMEOUT from %s [%s], %s", RemoteHost, RemoteAddr, user);
			break;
		default:
			syslog(LOG_ERR, "exiting on signal %d", sig);
		}
	}

	chdir("/etc/tmp");

	ShutDown (1);
}


/*
*  Process Windows file system commands
*/
void ProcessMessages (sockfd)
int sockfd;
{
	winio_generic	*pwinGen;
	int				iBytesRead;
	time_t			currtime;
	pwinGen = (winio_generic *)InBuff;

	while ((iBytesRead = readn(sockfd, pwinGen, sizeof(winio_generic))))
	{
		/* First read the generic packet (above) and get the length */

		if (iBytesRead < sizeof(winio_generic))
		{
			!fDebug ? 0 : printf ("%s error. First readn() = %d\n", ProgramName, iBytesRead);
			ShutDown (1);
		}

		pwinGen->usType = ntohs(pwinGen->usType);
		pwinGen->usLen = ntohs(pwinGen->usLen);
		pwinGen->ulFlags = ntohl(pwinGen->ulFlags);

		/*
		!fDebug ? 0 : printf ("Got Generic, type = %d, reading len = %d\n", 
			pwinGen->usType, (int)(pwinGen->usLen - sizeof(winio_generic)));
		*/

		/* Now read the full packet */
		
		iBytesRead = readn (sockfd, &InBuff[sizeof(winio_generic)], 
			pwinGen->usLen - sizeof(winio_generic));

		if (iBytesRead < (pwinGen->usLen - sizeof(winio_generic)))
		{
			!fDebug ? 0 : printf ("%s error. Second readn() = %d\n", ProgramName, iBytesRead);
			ShutDown (1);
		}

		/* Keep the Telnet session alive with a message every 30 seconds (or more) */

		time(&currtime);

		if ((currtime - glasttime) > 30)
		{
			glasttime = currtime;
			writen (gdaemonsocket, gkeepalivestr, strlen(gkeepalivestr));
		}


		/* Process the packet */

		switch (pwinGen->usType)
		{
		case FS_CONFIGURE:
			fsConfigure (sockfd, InBuff);
			break;
		case FS_OPENFILE:
			fsOpenFile (sockfd, InBuff);
			break;
		case FS_CLOSEFILE:
			fsCloseFile (sockfd, InBuff);
			break;
		case FS_READFILE:
			fsReadFile (sockfd, InBuff);
			break;
		case FS_WRITEFILE:
			fsWriteFile (sockfd, InBuff);
			break;
		case FS_FILESEEK:
			fsFileSeek (sockfd, InBuff);
			break;
		case FS_FINDFIRSTFILE:
			fsFindFirstFile (sockfd, InBuff);
			break;
		case FS_FINDNEXTFILE:
			fsFindNextFile (sockfd, InBuff);
			break;
		case FS_FINDCLOSE:
			fsFindClose (sockfd, InBuff);
			break;
		case FS_FILEATTRIBUTES:
			fsFileAttributes (sockfd, InBuff);
			break;
		case FS_PERMISSIONS:
			fsFilePermissions (sockfd, InBuff);
			break;
		case FS_DIR:
			fsDir (sockfd, InBuff);
			break;
		case FS_DELETEFILE:
			fsDeleteFile (sockfd, InBuff);
			break;
		case FS_RENAMEFILE:
			fsRenameFile (sockfd, InBuff);
			break;
		case FS_GETDISKINFO:
			fsGetDiskInfo (sockfd, InBuff);
			break;
		default:
            !fDebug ? 0 : printf ("%s Unknown Command:  Command = %d\n", ProgramName, pwinGen->usType);
		}
		fflush (stdout);
		fflush (stderr);
	}
}


/*
*  Configuration and shutdown
*/
void fsConfigure (sockfd, pInBuff)
int sockfd;
char *pInBuff;
{
	fs_configure		*pConfig;
	fs_configureAck		*pConfigAck;
	struct stat			fStat;
	USHORT				usLen;
	char				*pEnv;

	pConfigAck = (fs_configureAck *)OutBuff;
	memset ((char *)pConfigAck, 0, sizeof(fs_configureAck));

	pConfigAck->usVersionMajor = htons(ZANNET_VERSION_MAJOR);
	pConfigAck->usVersionMinor = htons(ZANNET_VERSION_MINOR);

	pConfig = (fs_configure *)pInBuff;

	pConfig->ulServerDefaultAttribs = ntohl(pConfig->ulServerDefaultAttribs);
	pConfig->ulMaxReaddirs = ntohl(pConfig->ulMaxReaddirs);
	pConfig->sPathLen = ntohs(pConfig->sPathLen);

	if (pConfig->ulFlags & ZANSERV_DEBUG_ON)
	{
		fDebug = TRUE;
		!fDebug ? 0 : printf ("Debug On!\n");
	}

	if (pConfig->ulFlags & ZANSERV_DEBUG_OFF)
	{
		!fDebug ? 0 : printf ("Debug Off!\n");
		fDebug = FALSE;
	}

	if (pConfig->ulFlags & ZANSERV_MAX_READDIRS)
	{
		!fDebug ? 0 : printf ("ZANSERV_MAX_READDIRS = %d\n", pConfig->ulMaxReaddirs);
		fflush(stdout);

		ulMaxReaddirs = pConfig->ulMaxReaddirs;
	}

	if (pConfig->ulFlags & ZANSERV_NEW_HOME)
	{
		!fDebug ? 0 : printf ("ZANSERV_NEW_HOME: %s\n", pConfig->HomePath);

		if (fOptHomeDirChange)
		{
			if (!stat(pConfig->HomePath, &fStat))
			{
				strcpy (HomePath, pConfig->HomePath);
			}
			else
			{
				pConfigAck->rc = ERROR_FILE_NOT_FOUND;
				!fDebug ? 0 : printf ("***** Can't find new home directory: %s\n", pConfig->HomePath);
			}
		}
		else
		{
			pConfigAck->rc = ERROR_ACCESS_DENIED;
			!fDebug ? 0 : printf ("*** Home directory change denied by server! ***\n");
		}
	}

	if (pConfig->ulFlags & ZANSERV_DEFAULT_HOME)
	{
		!fDebug ? 0 : printf ("ZANSERV_DEFAULT_HOME\n");

		/* Set default home path */

		if ((pEnv = (char *)getenv("HOME")))
			strcpy (HomePath, pEnv);
		else
			strcpy (HomePath, "/");
	}

	if (pConfig->ulFlags & ZANSERV_ATTRIBS)
	{
		!fDebug ? 0 : printf ("ZANSERV_ATTRIBS: %x\n", pConfig->ulServerDefaultAttribs);

		ulDefaultAttributes = pConfig->ulServerDefaultAttribs;
	}

	if (pConfig->ulFlags & ZANSERV_KEEP_ALIVE)
	{
		writen (gdaemonsocket, gkeepalivestr, strlen(gkeepalivestr));
	}

	/* Write the results to Windows */

	if (pConfig->ulFlags & ZANSERV_PING)
	{
		usLen = pConfig->usLen;

		!fDebug ? 0 : printf ("ZANSERV_PING: Sending %d bytes\n", usLen);

		memcpy (&OutBuff[sizeof(fs_configureAck)], pConfig->HomePath, usLen);
	}
	else
		usLen = sizeof(fs_configureAck);

	pConfigAck->usLen = htons(usLen);

	pConfigAck->usType = htons(pConfig->usType);
	pConfigAck->rc = htonl(pConfigAck->rc);

	pConfigAck->ulKey = pConfig->ulKey;

	/* Shutdown if requested. This is ACKed by the Windows socket disconnect */

	if (pConfig->ulFlags & ZANSERV_SHUTDOWN)
	{
		!fDebug ? 0 : printf ("Normal Shutdown with flag: ZANSERV_SHUTDOWN\n");
		ShutDown(0);
	}

	writen (sockfd, pConfigAck, usLen);

	/* !fDebug ? 0 : printf ("FS_CONFIGURE: rc=%d\n", ntohl(pConfigAck->rc)); */
}


/*
*  Open file
*/
void fsOpenFile (sockfd, pInBuff)
int sockfd;
char *pInBuff;
{
	fs_openfile			*pOpen;
	fs_openfileAck		*pOpenAck;
	struct stat			fStat;
	int					iflags = 0; 
	mode_t				mode = 0;
	char				NATIVEPath[MAX_PATH];

	pOpen = (fs_openfile *)pInBuff;

	pOpen->ulAttribs = ntohl(pOpen->ulAttribs);
	pOpen->usOptions = ntohs(pOpen->usOptions);
	pOpen->sPathLen = ntohs(pOpen->sPathLen);

	!fDebug ? 0 : printf ("FS_OPENFILE: Flags=%x, Attribs=%x, Options=%x, Path=%s\n",
		pOpen->ulFlags, pOpen->ulAttribs, pOpen->usOptions, pOpen->Path);

	pOpenAck = (fs_openfileAck *)OutBuff;
	memset ((char *)pOpenAck, 0, sizeof(fs_openfileAck));

	/* Translate Windows flags to POSIX */

	switch (pOpen->ulFlags & ACCESS_MODE_MASK)
	{
	case ACCESS_READONLY:
		iflags |= O_RDONLY;
		break;
	case ACCESS_WRITEONLY:
		iflags |= O_WRONLY;
		break;
	case ACCESS_READWRITE:
		iflags |= O_RDWR;
		break;
	case ACCESS_EXECUTE:
		iflags |= O_RDONLY;
		break;
	default:
		iflags |= O_RDONLY;
	}

	/***

	Sharing flags are NOT yet supported. All files are considered
	read/write shareable at this time - 11/21/96 

  	switch (pOpen->ulFlags & SHARE_MODE_MASK)
	{
	case SHARE_COMPATIBILITY:
		break;
	case SHARE_DENYREADWRITE:
		break;
	case SHARE_DENYWRITE:
		break;
	case SHARE_DENYREAD:
		break;
	default:
	}

	***/

	if ((pOpenAck->rc = MakeNativePath (pOpen->Path, NATIVEPath)))
		goto ExitfsOpenFile;

	/* Next are the Windows options. Note: mode is ignored if file already
	   exists! */

	switch (pOpen->usOptions & ACTION_MASK)
	{
	case ACTION_CREATEALWAYS:
	{
		/* Create a new file. If the file already exists open it and set
		its new length and attributes. */

		iflags |= O_CREAT | O_TRUNC;
		
		if (!(stat(NATIVEPath, &fStat)))
		{
			/* File already exists, set new attributes */

			mode = fStat.st_mode;

			if (pOpen->ulAttribs & FILE_ATTRIBUTE_READONLY)
				mode = (mode & ~S_IWUSR) | S_IRUSR; 
			else
				mode = mode | S_IWUSR;     

			/* We can only chmod() if we own the file. We may be able to write
			   to the file without owning it. */

			if (getuid() == fStat.st_uid)
				if (chmod (NATIVEPath, mode) < 0)
					pOpenAck->rc = MapWindowsError (errno);

			if (!pOpenAck->rc)
				pOpenAck->usAction = ACTION_OPENED;

			mode = 0;
		}
		else
		{
			/* New file */

			if (pOpen->ulAttribs & FILE_ATTRIBUTE_READONLY)
				mode = ulDefaultAttributes & ~(S_IWUSR | S_IWGRP | S_IWOTH);
			else 
				mode = ulDefaultAttributes;

			pOpenAck->usAction = ACTION_CREATED;
		}

		break;
	}
	case ACTION_OPENALWAYS:
	{
		/* Open an existing file. If the file does not exist, create a new file. */

		if (stat(NATIVEPath, &fStat))
		{
			/* No existing file, set mode for file creation */

			if (pOpen->ulAttribs & FILE_ATTRIBUTE_READONLY)
				mode = ulDefaultAttributes & ~(S_IWUSR | S_IWGRP | S_IWOTH);
			else 
				mode = ulDefaultAttributes;

			pOpenAck->usAction = ACTION_CREATED;
		}
		else
		{
			pOpenAck->usAction = ACTION_OPENED;
		}

		iflags |= O_CREAT;
		break;
	}
	case ACTION_CREATENEW:
	{
		/* Create a new file. Fail if the file already exists. */

		if (pOpen->ulAttribs & FILE_ATTRIBUTE_READONLY)
			mode = ulDefaultAttributes & ~(S_IWUSR | S_IWGRP | S_IWOTH);
		else 
			mode = ulDefaultAttributes;

		iflags |= O_CREAT | O_EXCL;

		pOpenAck->usAction = ACTION_CREATED;
		break;
	}
	case ACTION_REPLACEEXISTING:
	{
		/* Open an existing file and set its new length. Fail if the file does not exist */

		iflags |= O_TRUNC;

		if (stat(NATIVEPath, &fStat))
		{
			/* No existing file, set error */

			pOpenAck->rc = ERROR_FILE_NOT_FOUND;
		}
		else
		{
			pOpenAck->usAction = ACTION_REPLACED;
		}

		break;
	}
	case ACTION_OPENEXISTING:
	{
		/* Open an existing file. Fail if the file does not exist */

		pOpenAck->usAction = ACTION_OPENED;
		break;
	}
	default:
	{
		/* Open an existing file. Fail if the file does not exist */

		pOpenAck->usAction = ACTION_OPENED;
		break;
	}
	}

	/* Make sure this is not a directory */

	if (!stat(NATIVEPath, &fStat))
	{
		if (S_ISDIR(fStat.st_mode))
			pOpenAck->rc = ERROR_ACCESS_DENIED;
	}

	if (!pOpenAck->rc)
	{
		if (mode)
			pOpenAck->iFileHandle = open (NATIVEPath, iflags, mode);
		else
			pOpenAck->iFileHandle = open (NATIVEPath, iflags);

		if (pOpenAck->iFileHandle > 0)
		{
			/* Get the length, last modification date/time and attributes */

			if (!(stat(NATIVEPath, &fStat)))
			{
				pOpenAck->ulSize = fStat.st_size;
				pOpenAck->LastAccess = fStat.st_mtime;

				if ((fStat.st_mode & S_IRUSR) && !(fStat.st_mode & S_IWUSR))
				{
					/* !fDebug ? 0 : printf ("Open file as read only\n"); */
					pOpenAck->ulAttribs = FILE_ATTRIBUTE_READONLY;
				}
				else
				{
					/* !fDebug ? 0 : printf ("Open file normal\n"); */
					/* pOpenAck->ulAttribs = FILE_ATTRIBUTE_ARCHIVE; */
					pOpenAck->ulAttribs = 0;
				}
			}
			else 
			{
				pOpenAck->usAction = 0;
				pOpenAck->rc = MapWindowsError (errno);
			}
		}
		else
		{
			pOpenAck->usAction = 0;
			pOpenAck->rc = MapWindowsError (errno);
		}
	}

ExitfsOpenFile:

	/* Write the results to Windows */

	pOpenAck->ulFlags = htonl(pOpenAck->ulFlags);
	pOpenAck->ulAttribs = htonl(pOpenAck->ulAttribs);
	pOpenAck->LastAccess = htonl(pOpenAck->LastAccess);
	pOpenAck->ulSize = htonl(pOpenAck->ulSize);
	pOpenAck->rc = htons(pOpenAck->rc);
	pOpenAck->usAction = htons(pOpenAck->usAction);

	pOpenAck->usLen = htons(sizeof(fs_openfileAck));
	pOpenAck->usType = htons(pOpen->usType);

	pOpenAck->ulKey = pOpen->ulKey;

	writen (sockfd, pOpenAck, sizeof(fs_openfileAck));

	!fDebug ? 0 : printf ("FS_OPENFILE: rc=%d, action=%d, size=%ld, attrib=%lx\n", 
		htons(pOpenAck->rc), htons(pOpenAck->usAction), htonl(pOpenAck->ulSize),
		htonl(pOpenAck->ulAttribs));
}


/*
*  Close file
*/
void fsCloseFile (sockfd, pInBuff)
int sockfd;
char *pInBuff;
{
	fs_close		*pClose;
	fs_closeAck		*pCloseAck;

	pClose = (fs_close *)pInBuff;

	!fDebug ? 0 : printf ("FS_CLOSEFILE: Handle=%d\n", pClose->iFileHandle); 
		
	pCloseAck = (fs_closeAck *)OutBuff;
	memset ((char *)pCloseAck, 0, sizeof(fs_closeAck));
	
	if (close (pClose->iFileHandle) < 0)
		pCloseAck->rc = MapWindowsError (errno);

	pCloseAck->usLen = htons(sizeof(fs_closeAck));
	pCloseAck->usType = htons(pClose->usType);
	pCloseAck->ulKey = pClose->ulKey;

	pCloseAck->rc = htons(pCloseAck->rc);

	writen (sockfd, pCloseAck, sizeof(fs_closeAck));

	!fDebug ? 0 : printf ("FS_CLOSEFILE: rc=%d\n", ntohs(pCloseAck->rc));
}


/*
*  Read file
*/
void fsReadFile (sockfd, pInBuff)
int sockfd;
char *pInBuff;
{
	fs_readfile		*pRead;
	fs_readfileAck	*pReadAck;
	USHORT			usWriteLen;

	pRead = (fs_readfile *)pInBuff;

	pRead->iLength = ntohl(pRead->iLength);
	pRead->lPosition = ntohl(pRead->lPosition);

	!fDebug ? 0 : printf ("FS_READFILE: Handle=%d, Length=%d, Position=%d\n",
		pRead->iFileHandle, pRead->iLength, pRead->lPosition);

	pReadAck = (fs_readfileAck *)OutBuff;
	memset ((char *)pReadAck, 0, sizeof(fs_readfileAck));

	/* Make sure we are in the correct position */

	pReadAck->rc = SetFilePosition (pRead->iFileHandle, pRead->lPosition);
	
	if (!pReadAck->rc)
	{
		pReadAck->lPosition = pRead->lPosition;

		/* Do the NATIVE file read */

		pReadAck->iLength = readn (pRead->iFileHandle, pReadAck->Data, pRead->iLength);
			
		if (pReadAck->iLength < 0)
		{
			pReadAck->rc = MapWindowsError (errno);
			pReadAck->iLength = 0;
		}
		else
			pReadAck->lPosition += pReadAck->iLength;
	}

	/* Write the packet and data to Windows */

	usWriteLen = sizeof(fs_readfileAck) + pReadAck->iLength - sizeof(pReadAck->Data);

	pReadAck->usLen = htons(usWriteLen);
	pReadAck->ulKey = pRead->ulKey;
	pReadAck->usType = htons(pRead->usType);

	pReadAck->iLength = htonl(pReadAck->iLength);
	pReadAck->lPosition = htonl(pReadAck->lPosition);
	pReadAck->rc = htons(pReadAck->rc);

	writen (sockfd, pReadAck, usWriteLen);

	!fDebug ? 0 : printf ("FS_READFILE: rc=%d, len=%ld, pos=%ld\n", 
		ntohs(pReadAck->rc), ntohl(pReadAck->iLength), ntohl(pReadAck->lPosition));
}


/*
*  Write file
*/
void fsWriteFile (sockfd, pInBuff)
int sockfd;
char *pInBuff;
{
	fs_writefile		*pWrite;
	fs_writefileAck		*pWriteAck;
	

	pWrite = (fs_writefile *)pInBuff;

	pWrite->iLength = ntohl(pWrite->iLength);
	pWrite->lPosition = ntohl(pWrite->lPosition);

	!fDebug ? 0 : printf ("FS_WRITEFILE: Handle=%d, Length=%d, Position=%d\n",
		pWrite->iFileHandle, pWrite->iLength, pWrite->lPosition);

	pWriteAck = (fs_writefileAck *)OutBuff;
	memset ((char *)pWriteAck, 0, sizeof(fs_writefileAck));

	if (pWrite->ulFlags & ZANSERV_FSWRITE_MOVE_EOF && !pWriteAck->rc)
	{
		fsWriteMoveEOF (pWrite, pWriteAck);
	}
	else if (!pWriteAck->rc)
	{
		if (!pWrite->iLength)
			pWriteAck->rc = ERROR_BAD_ARGUMENTS;

		/* Make sure we are in the correct position */

		pWriteAck->rc = SetFilePosition (pWrite->iFileHandle, pWrite->lPosition);

		if (!pWriteAck->rc)
		{
			pWriteAck->lPosition = pWrite->lPosition;

			/* Do the NATIVE file write */

			pWriteAck->iBytesWritten = writen (pWrite->iFileHandle, pWrite->WriteData, pWrite->iLength);
				
			if (pWriteAck->iBytesWritten < 0)
				pWriteAck->rc = MapWindowsError (errno);
			else	
				pWriteAck->lPosition += pWriteAck->iBytesWritten;
		}
	}

	/* Return status to Windows */

	pWriteAck->rc = htons(pWriteAck->rc);
	pWriteAck->usLen = htons(sizeof(fs_writefileAck));
	pWriteAck->ulKey = pWrite->ulKey;
	pWriteAck->usType = htons(pWrite->usType);

	pWriteAck->iBytesWritten = htonl(pWriteAck->iBytesWritten);
	pWriteAck->lPosition = htonl(pWriteAck->lPosition);

	writen (sockfd, pWriteAck, sizeof(fs_writefileAck));

	!fDebug ? 0 : printf ("FS_WRITEFILE: rc=%d, len=%ld, pos=%ld\n", 
		ntohs(pWriteAck->rc), ntohl(pWriteAck->iBytesWritten), ntohl(pWriteAck->lPosition));
}


/*
*  fsWriteMoveEOF
*/
void fsWriteMoveEOF (pWrite, pWriteAck)
fs_writefile *pWrite;
fs_writefileAck *pWriteAck;
{
	struct	stat	fStat;
	char	WritePad[5]; 
	int		lPos=0, lWritten=0;
	/*
	char	NATIVEPath[MAX_PATH];
	char	TmpFileName[MAX_PATH];
	int		lPos=0, lWritten=0, lDiff, lRead=0;
	FILE	*fpNew, *fpOld;
	int		NewDesc1, NewDesc2;
	*/

	/* A zero length write is used by Windows to set a new EOF position. */

	if (!(fstat (pWrite->iFileHandle, &fStat)))
	{
		if ((fStat.st_mode & S_IRUSR) && !(fStat.st_mode & S_IWUSR))
		{
			pWriteAck->rc = ERROR_ACCESS_DENIED;
			return;
		}

		/* Same position--just return */
		if (fStat.st_size == pWrite->lPosition)
		{
			pWriteAck->lPosition = pWrite->lPosition;
			return;
		}
		
		/* A truncation request, copy the file up to new EOF */
		if (fStat.st_size > pWrite->lPosition)
		{
			!fDebug ? 0 : printf ("FS_WRITEFILE: Truncation\n");

			if (ftruncate (pWrite->iFileHandle, pWrite->lPosition))
				pWriteAck->rc = MapWindowsError (errno);
			else
				pWriteAck->lPosition = pWrite->lPosition;
				
			/****
			if (pWriteAck->rc = MakeNativePath (pWrite->WriteData, NATIVEPath))
				return;

			close (pWrite->iFileHandle);

			!fDebug ? 0 : printf ("FS_WRITEFILE: Truncation - path = %s\n", NATIVEPath);

			// Rename it to something unique 

			strcpy (TmpFileName, NATIVEPath);
			strcat (TmpFileName, "_ZNet");

			if (rename (NATIVEPath, TmpFileName))
			{
				pWriteAck->rc = MapWindowsError (errno);
			}
			else
			{
				// Open new file and old file as streams 

				if (!(fpNew = fopen (NATIVEPath, "w+")))
				{
					pWriteAck->rc = MapWindowsError (errno);
				}
				else
				{
					if (!(fpOld = fopen (TmpFileName, "r")))
					{
						fclose (fpNew);
						pWriteAck->rc = MapWindowsError (errno);
					}
					else
					{
						// Copy up to the new EOF 

						while (lPos < pWrite->lPosition)
						{
							lDiff = pWrite->lPosition - lPos;

							lRead = fread (WritePad, 1, lDiff < sizeof(WritePad) ? lDiff : sizeof(WritePad), fpOld);

							if (lRead < 0)
							{
								pWriteAck->rc = MapWindowsError (errno);
								break;
							}

							lWritten = fwrite (WritePad, 1, lRead, fpNew);

							if (lWritten < 0)
							{
								pWriteAck->rc = MapWindowsError (errno);
								break;
							}
							else	
								lPos += lWritten;
						}

						// Close the streams 

						fclose (fpOld);
						fclose (fpNew);

						// Set the orginal file mode 

						chmod (NATIVEPath, fStat.st_mode);
						
						if (!pWriteAck->rc)
						{
							// Delete the tmp file 

							unlink (TmpFileName);

							// Get back our original file descriptor or else Windows will have a cow 

							if ((NewDesc1 = open (NATIVEPath, O_RDWR)) < 0)
							{
								pWriteAck->rc = MapWindowsError (errno);
							}
							else if (NewDesc1 != pWrite->iFileHandle)
							{
								if ((NewDesc2 = fcntl(NewDesc1, F_DUPFD, pWrite->iFileHandle)) < 0)
								{
									pWriteAck->rc = MapWindowsError (errno);
								}
								else if (NewDesc2 != pWrite->iFileHandle)
								{
									pWriteAck->rc = ERROR_WRITE_FAULT;
									close (NewDesc2);
								}
								close (NewDesc1);
							}
						}
					}
				}
			}
			if (!pWriteAck->rc)
				pWriteAck->lPosition = pWrite->lPosition;
			*******/
		}
		/* Grow the file */
		else 
		{
			!fDebug ? 0 : printf ("FS_WRITEFILE: Expansion\n");

			if ((lPos = lseek (pWrite->iFileHandle, pWrite->lPosition-1, SEEK_SET)) < 0)
				pWriteAck->rc = MapWindowsError (errno);
			else
			{
				WritePad[0] = 0;

				lWritten = writen (pWrite->iFileHandle, WritePad, 1);

				if (lWritten != 1)
					pWriteAck->rc = MapWindowsError (errno);
				else
					pWriteAck->lPosition = lPos + 1;
			}
		}
	}
	else
		pWriteAck->rc = ERROR_INVALID_HANDLE;
}


/*
*  Seek file - Note: A "seek" is really meaningless because all reads and writes 
*              come with length and offset. Just return the proper file size. The
*              seek is performed on the server for the sake of error checking. FILE_BEGIN
*              will never even get here because it is handled in Windows only.
*/
void fsFileSeek (sockfd, pInBuff)
int sockfd;
char *pInBuff;
{
	fs_fileseek		*pSeek;
	fs_fileseekAck	*pSeekAck;
	int				iwhence = -1, DoSeek=TRUE;
	struct			stat	fStat;

	pSeek = (fs_fileseek *)pInBuff;

	pSeek->lSeekOffset = ntohl(pSeek->lSeekOffset);

	!fDebug ? 0 : printf ("FS_FILESEEK: Handle=%d, Flags=%d, Offset=%d\n",
		pSeek->iFileHandle, pSeek->ulFlags, pSeek->lSeekOffset);

	pSeekAck = (fs_fileseekAck *)OutBuff;
	memset ((char *)pSeekAck, 0, sizeof(fs_fileseekAck));

	if (fstat(pSeek->iFileHandle, &fStat) < 0)
		pSeekAck->rc = MapWindowsError(errno);

	if (pSeek->ulFlags == FILE_BEGIN)
	{
		iwhence = SEEK_SET;
	}
	else if (pSeek->ulFlags == FILE_END)
	{
		if ((fStat.st_size + pSeek->lSeekOffset) < 0)
		{
			DoSeek = FALSE;
			pSeekAck->lSeekOffset = fStat.st_size + pSeek->lSeekOffset;
		}
		iwhence = SEEK_END;
	}
	else
		pSeekAck->rc = ERROR_SEEK_ON_DEVICE;

	if (!pSeekAck->rc)
	{
		if (DoSeek)
		{
			pSeekAck->lSeekOffset = lseek (pSeek->iFileHandle, pSeek->lSeekOffset, iwhence);

			if (pSeekAck->lSeekOffset < 0)
				pSeekAck->rc = MapWindowsError (errno);
		}
	}

	pSeekAck->usLen = htons(sizeof(fs_fileseekAck));
	pSeekAck->ulKey = pSeek->ulKey;
	pSeekAck->usType = htons(pSeek->usType);

	pSeekAck->lSeekOffset = htonl(pSeekAck->lSeekOffset);
	pSeekAck->rc = htons(pSeekAck->rc);

	writen (sockfd, pSeekAck, sizeof(fs_fileseekAck));

	!fDebug ? 0 : printf ("FS_FILESEEK: rc=%d, offset=%ld\n", 
		ntohs(pSeekAck->rc), ntohl(pSeekAck->lSeekOffset));
}

/* CONSIDER: Support search attributes for fsfindfirst/fsfindnext */

/*
* fsFindFirstFile
*/
void fsFindFirstFile (sockfd, pInBuff)
int sockfd;
char *pInBuff;
{
	fs_findfirstfile		*pFFirst;
	fs_findfirstfileAck		*pFFirstAck;
	DIR						*pDir=0;
	char					NATIVEPath[MAX_PATH];
	fs_findfileinfo			*pffInfo=0;
	char					*pSearchPattern=0, *pFile=0;
	int						isRoot=FALSE;
	USHORT					usWriteLen;

	/* Read the find first structure */

	pFFirst = (fs_findfirstfile *)pInBuff;

	pFFirst->ulRetBufLen = ntohl(pFFirst->ulRetBufLen);
	pFFirst->sPathLen = ntohs(pFFirst->sPathLen);

	!fDebug ? 0 : printf ("FS_FINDFIRSTFILE: Attribs=%x, BuffLen=%d, Path=%s\n",
		pFFirst->ulAttrib, pFFirst->ulRetBufLen, pFFirst->Path);

	pFFirstAck = (fs_findfirstfileAck *)OutBuff;
	memset ((char *)pFFirstAck, 0, sizeof(fs_findfirstfileAck));
	
	if (pFFirst->ulAttrib & FILE_FLAG_WILDCARDS || pFFirst->ulAttrib & FILE_FLAG_HAS_STAR)
	{
		/* This is a search pattern - strip it */

		/* !fDebug ? 0 : printf ("Found wildcard chars\n"); */

		pSearchPattern = StripFileFromPath (pFFirst->Path, WindowsBSlash);

		if (!pFFirst->Path[0])
		{
			/* !fDebug ? 0 : printf ("isRoot = TRUE\n"); */
			isRoot = TRUE;
		}
	}

	if (!(pFFirstAck->rc = MakeNativePath (pFFirst->Path, NATIVEPath)))
	{
		if (!pSearchPattern)
		{
			/* The last path element is a file/directory with no wildcards. 
			Just do a stat on it and return the results */

			/* !fDebug ? 0 : printf ("FF on single file: %s\n", NATIVEPath); */

			pffInfo	= (fs_findfileinfo *)((char *)&pFFirstAck->FindFiles);

			pffInfo->ulAttrib = pFFirst->ulAttrib;

			if (!(pFFirstAck->rc = FillFileStatus (NATIVEPath, pffInfo)))
			{
				pFFirstAck->sNumberOfFiles = 1;

				/* !fDebug ? 0 : printf ("NATIVEPath = %s\n", NATIVEPath); */

				if ((pFile = StripFileFromPath (NATIVEPath, NativeSlash)))
				{
					pffInfo->sFileNameLen = strlen(pFile) + 1;
					strcpy (pffInfo->FileName, pFile);
				}

				pFFirstAck->ulRetBufLen = (sizeof(fs_findfileinfo) + 
					pffInfo->sFileNameLen - sizeof(pffInfo->FileName));

				pffInfo->sFileNameLen = htons(pffInfo->sFileNameLen);

				/* !fDebug ? 0 : printf ("pFile = %s\n", pFile); */

				pFFirstAck->rc = ERROR_NO_MORE_FILES;
			}
			else
				pFFirstAck->rc = ERROR_FILE_NOT_FOUND;
		}
		else if ((pDir = opendir (NATIVEPath)))
		{
			/* Open the directory and return the handle to Windows */

			while (pFFirst->ulRetBufLen - (MAX_PATH + 1) >= pFFirstAck->ulRetBufLen)
			{
				/* Get as many files as our buffer size will allow */

				pffInfo	= (fs_findfileinfo *)(((char *)&pFFirstAck->FindFiles) + pFFirstAck->ulRetBufLen);

				pffInfo->ulAttrib = pFFirst->ulAttrib;

				if (!(pFFirstAck->rc = (short)ReadDirEntry (pDir, 
					pffInfo, 
					NATIVEPath,
					pSearchPattern,
					isRoot)))
				{
					pFFirstAck->sNumberOfFiles++;

					pFFirstAck->ulRetBufLen += sizeof(fs_findfileinfo) + 
						pffInfo->sFileNameLen - sizeof(pffInfo->FileName); 

					pffInfo->sFileNameLen = htons(pffInfo->sFileNameLen);
				}
				else if (pFFirstAck->rc == ERROR_NO_MORE_FILES)
					break;
			}
		}
		else
		{
			!fDebug ? 0 : printf ("Can't find/open file/directory - Path = %s\n", NATIVEPath);
			pFFirstAck->rc = MapWindowsError (errno);
		}
	}
	
	/* pFFirstAck->iFindContext = (int)pDir; */

	if (pDir)
		pFFirstAck->iFindContext = NewMemoryHandle (pDir);

	/* Write the Ack struct to Windows. We subtract the length of the array
	already included in fs_findfileinfo */

	usWriteLen = sizeof(fs_findfirstfileAck) + pFFirstAck->ulRetBufLen;

	pFFirstAck->usLen = htons(usWriteLen);

	pFFirstAck->ulKey = pFFirst->ulKey;
	pFFirstAck->usType = htons(pFFirst->usType);

	pFFirstAck->rc = htons(pFFirstAck->rc);
	pFFirstAck->sNumberOfFiles = htons(pFFirstAck->sNumberOfFiles);
	pFFirstAck->ulRetBufLen = htonl(pFFirstAck->ulRetBufLen);

	writen (sockfd, pFFirstAck, usWriteLen); 

	!fDebug ? 0 : printf ("FS_FINDFIRSTFILE: rc=%d, FilesFound=%d\n",
		ntohs(pFFirstAck->rc),
		ntohs(pFFirstAck->sNumberOfFiles));
}


/*
*  fsFindNextFile
*/
void fsFindNextFile (sockfd, pInBuff)
int sockfd;
char *pInBuff;
{
	fs_findnextfile		*pFNext;
	fs_findnextfileAck	*pFNextAck;
	fs_findfileinfo		*pffInfo;
	char				*pSearchPattern=0;
	int					isRoot=FALSE;
	char				NATIVEPath[MAX_PATH + 1];
	USHORT				usWriteLen;
	DIR					*pDir=0;

	pFNext = (fs_findnextfile *)pInBuff;

	pFNext->ulRetBufLen = ntohl(pFNext->ulRetBufLen);
	pFNext->sPathLen = ntohs(pFNext->sPathLen);

	!fDebug ? 0 : printf ("FS_FINDNEXTFILE: Flags=%x, SearchHandle=%d, BuffLen=%d, Path=%s\n",
		pFNext->ulFlags, pFNext->iFindHandle, pFNext->ulRetBufLen, pFNext->Path);

	pFNextAck = (fs_findnextfileAck *)OutBuff;
	memset ((char *)pFNextAck, 0, sizeof(fs_findnextfileAck));

	if (pFNext->ulFlags & FILE_FLAG_WILDCARDS || pFNext->ulFlags & FILE_FLAG_HAS_STAR)
	{
		/* This is a search pattern - strip it */

		/* !fDebug ? 0 : printf ("Found wildcard chars\n"); */

		pSearchPattern = StripFileFromPath (pFNext->Path, WindowsBSlash);

		if (!pFNext->Path[0])
			isRoot = TRUE;
	}

	if (!(pFNextAck->rc = MakeNativePath (pFNext->Path, NATIVEPath)))
	{
		pDir = (DIR *)GetMemoryByHandle (pFNext->iFindHandle);

		while (pFNext->ulRetBufLen - (MAX_PATH + 1) >= pFNextAck->ulRetBufLen)
		{
			/* Get as many files as our buffer size will allow */

			pffInfo	= (fs_findfileinfo *)(((char *)&pFNextAck->FindFiles) + pFNextAck->ulRetBufLen);

			pffInfo->ulAttrib = pFNext->ulFlags;

			if (!(pFNextAck->rc = (short)ReadDirEntry (pDir,
				pffInfo, 
				NATIVEPath,
				pSearchPattern,
				isRoot)))
			{
				pFNextAck->sNumberOfFiles++;

				pFNextAck->ulRetBufLen += (sizeof(fs_findfileinfo) + 
					pffInfo->sFileNameLen - sizeof(pffInfo->FileName));

				pffInfo->sFileNameLen = htons(pffInfo->sFileNameLen);
			}
			else if (pFNextAck->rc == ERROR_NO_MORE_FILES)
				break;
		}
	}
	else
		pFNextAck->rc = ERROR_NO_MORE_FILES;

	usWriteLen = sizeof(fs_findnextfileAck) + pFNextAck->ulRetBufLen;

	pFNextAck->usLen = htons(usWriteLen);
	pFNextAck->ulKey = pFNext->ulKey;
	pFNextAck->usType = htons(pFNext->usType);

	pFNextAck->rc = htons(pFNextAck->rc);
	pFNextAck->sNumberOfFiles = htons(pFNextAck->sNumberOfFiles);
	pFNextAck->ulRetBufLen = htonl(pFNextAck->ulRetBufLen);

	writen (sockfd, pFNextAck, usWriteLen);

	!fDebug ? 0 : printf ("FS_FINDNEXTFILE: rc=%d, FilesFound=%d\n",
		ntohs(pFNextAck->rc),
		ntohs(pFNextAck->sNumberOfFiles));
}


/*
*  ReadDirEntry
*/
int ReadDirEntry (pDir, pFFileInfo, pPath, pSearchPattern, isRoot)
DIR	*pDir;
fs_findfileinfo	*pFFileInfo;
char *pPath;
char *pSearchPattern;
int isRoot;
{
	/* CONSIDER: Portability issue with dirent */

	struct	dirent	*pDirEnt;

	char	NewPath[MAX_PATH];
	char	TempFile[MAX_PATH];
	int		fMatched = TRUE, rc, len;

	ULONG	ulAttrib = pFFileInfo->ulAttrib;

	/* isRoot = TRUE; */

	memset ((char *)pFFileInfo, 0, sizeof(fs_findfileinfo));

	pFFileInfo->ulAttrib = ulAttrib;

	if ((pDirEnt = readdir (pDir)))
	{
		if (isRoot && (!strcmp(pDirEnt->d_name, ".") || !strcmp(pDirEnt->d_name, "..")))
			fMatched = FALSE;
		else if (pSearchPattern)
		{
			fMatched = MetaMatch (pSearchPattern, pDirEnt->d_name);

			if (!fMatched)
			{
				/* Try again with a mangled name */

				strcpy (TempFile, pDirEnt->d_name);

				mangle_name_83 (TempFile);

				fMatched = MetaMatch (pSearchPattern, TempFile);
			}
		}

		if (fMatched)
		{
			len = strlen(pDirEnt->d_name) + 1;  /* Add one for the NULL */

			/* Pad to 4 byte boundries for pointer alignment sensitive machines */

			len += 4 - len%4;

			pFFileInfo->sFileNameLen = len;

			/*
			printf("pFFileInfo->sFileNameLen=%d\n", pFFileInfo->sFileNameLen);
			fflush(stdout);
			*/

			strcpy (pFFileInfo->FileName, pDirEnt->d_name);

			strcpy (NewPath, pPath);

			if (strcmp(NewPath, "/"))
				strcat (NewPath, "/");

			strcat (NewPath, pDirEnt->d_name);

			/*
			!fDebug ? 0 : printf ("Found file - pattern=%s, name=%s, NewPath=%s\n", 
				pSearchPattern, pDirEnt->d_name, NewPath);
			fflush(stdout);
			*/

			if ((rc = FillFileStatus (NewPath, pFFileInfo)))
				return rc;
		}
		else
		{
			return ERROR_FILE_NOT_FOUND;
		}
	}
	else
	{
		/* !fDebug ? 0 : printf ("readdir failed - handle = %d\n", pDir); */
		return ERROR_NO_MORE_FILES;
	}

	return 0;
}

/*
*  FillFileStatus
*/
int FillFileStatus (pPath, pFFInfo)
char *pPath;
fs_findfileinfo	*pFFInfo;
{
	struct	stat	fStat;
	ULONG	ulAttribTmp = 0;
	
	/* CONSIDER:

	#define FILE_ATTRIBUTE_MUSTMATCH	Must Match 
	#define FILE_ATTRIBUTE_EVERYTHING	Find Everything 

	*/

	if (!(stat (pPath, &fStat)))
	{
		/* !fDebug ? 0 : printf ("stat on %s\n", pPath); */

		if (fStat.st_mode & S_IRUSR && !(fStat.st_mode & S_IWUSR))
		{
			/* !fDebug ? 0 : printf ("FILE_ATTRIBUTE_READONLY on %s\n", pPath); */ 

			ulAttribTmp |= FILE_ATTRIBUTE_READONLY;
		}
		
		if (S_ISDIR(fStat.st_mode))
		{
			/* !fDebug ? 0 : printf ("FILE_ATTRIBUTE_DIRECTORY on %s\n", pPath); */

			if (!(pFFInfo->ulAttrib & FILE_ATTRIBUTE_DIRECTORY))
				return ERROR_FILE_NOT_FOUND;

			ulAttribTmp |= FILE_ATTRIBUTE_DIRECTORY;
		}
		
		pFFInfo->ulAttrib = htonl(ulAttribTmp);

		pFFInfo->tCreate = htonl(fStat.st_mtime);

		pFFInfo->tLastAccess = htonl(fStat.st_atime);
		pFFInfo->tLastWrite = htonl(fStat.st_mtime);

		if (ulAttribTmp & FILE_ATTRIBUTE_DIRECTORY)
			pFFInfo->ulFileSize = 0;
		else
			pFFInfo->ulFileSize = htonl(fStat.st_size);
	}
	else
	{
		!fDebug ? 0 : printf ("stat failed on %s\n", pPath);
		return MapWindowsError (errno);
	}

	return 0;
}


/*
*  fsFindClose
*/
void fsFindClose (sockfd, pInBuff)
int sockfd;
char *pInBuff;
{
	fs_findclose		*pFindClose;
	fs_findcloseAck		*pFindCloseAck;
	DIR					*pDir=0;

	pFindClose = (fs_findclose *)pInBuff;

	!fDebug ? 0 : printf ("FS_FINDCLOSE: Handle=%d\n", pFindClose->iFindHandle);

	pFindCloseAck = (fs_findcloseAck *)OutBuff;
	memset ((char *)pFindCloseAck, 0, sizeof(fs_findcloseAck));
	
	/* No find handle for single file finds */

	if (pFindClose->iFindHandle)
	{
		pDir = (DIR *)GetMemoryByHandle (pFindClose->iFindHandle);

		if (closedir (pDir) < 0)
			pFindCloseAck->rc = htons(MapWindowsError(errno));

		DeleteMemoryHandle (pFindClose->iFindHandle);
	}

	pFindCloseAck->usLen = htons(sizeof(fs_findcloseAck));
	pFindCloseAck->ulKey = pFindClose->ulKey;
	pFindCloseAck->usType = htons(pFindClose->usType);

	writen (sockfd, pFindCloseAck, sizeof(fs_findcloseAck));

	!fDebug ? 0 : printf ("FS_FINDCLOSE: rc=%d\n", ntohs(pFindCloseAck->rc));
}


/*
* fsFileAttributes
*/
void fsFileAttributes (sockfd, pInBuff)
int sockfd;
char *pInBuff;
{
	fs_fileattributes		*pFileAttrib;
	fs_fileattributesAck	*pFileAttribAck;
	struct stat				fStat;
	struct utimbuf			uTime;
	char					NATIVEPath[MAX_PATH + 1];

	pFileAttrib = (fs_fileattributes *)pInBuff;

	pFileAttrib->ulAttribs = ntohl(pFileAttrib->ulAttribs);
	pFileAttrib->NativeTime = ntohl(pFileAttrib->NativeTime);
	pFileAttrib->sPathLen = ntohs(pFileAttrib->sPathLen);

	!fDebug ? 0 : printf ("FS_FILEATTRIBUTES: Flags=%x, Attribs=%x, Time=%lx, Path=%s\n",
		pFileAttrib->ulFlags, pFileAttrib->ulAttribs, pFileAttrib->NativeTime, pFileAttrib->Path);

	pFileAttribAck = (fs_fileattributesAck *)OutBuff;
	memset ((char *)pFileAttribAck, 0, sizeof(fs_fileattributesAck));

	if ((pFileAttribAck->rc = MakeNativePath (pFileAttrib->Path, NATIVEPath)))
	{}
	else if (!(pFileAttribAck->rc = stat(NATIVEPath, &fStat)))
	{
		uTime.actime = fStat.st_atime;
		uTime.modtime = fStat.st_mtime;
									 
		/* What does windows want us to do? */

		switch (pFileAttrib->ulFlags)
		{
		case GET_ATTRIBUTES:
		{
			!fDebug ? 0 : printf ("GET_ATTRIBUTES - mode=%o\n", fStat.st_mode); 

			if (S_ISDIR(fStat.st_mode))
				pFileAttribAck->ulAttribs |= FILE_ATTRIBUTE_DIRECTORY;

  			if ((fStat.st_mode & S_IRUSR) && !(fStat.st_mode & S_IWUSR))
				pFileAttribAck->ulAttribs |= FILE_ATTRIBUTE_READONLY;

			break;
		}
		case SET_ATTRIBUTES:
		{
			mode_t mode;

			!fDebug ? 0 : printf ("SET_ATTRIBUTES - mode=%o\n", fStat.st_mode); 

			mode = fStat.st_mode;

			if (pFileAttrib->ulAttribs & FILE_ATTRIBUTE_READONLY)
				mode = (mode & ~S_IWUSR) | S_IRUSR; 
			else
				mode = mode | S_IWUSR;    

			if (chmod (NATIVEPath, mode) < 0)
				pFileAttribAck->rc = MapWindowsError (errno);

			break;
		}
		case GET_ATTRIB_COMP_FILESIZE:
		{
			!fDebug ? 0 : printf ("GET_ATTRIB_COMP_FILESIZE\n"); 
			pFileAttribAck->ulFileSize = fStat.st_size;
			break;
		}
		case GET_ATTRIB_MODIFY_DATETIME:
		{
			!fDebug ? 0 : printf ("GET_ATTRIB_MODIFY_DATETIME\n"); 
			pFileAttribAck->NativeTime = fStat.st_mtime;
			break;
		}
		case SET_ATTRIB_MODIFY_DATETIME:
		{
			!fDebug ? 0 : printf ("SET_ATTRIB_MODIFY_DATETIME\n"); 
			uTime.modtime = pFileAttrib->NativeTime;

			if (utime (NATIVEPath, &uTime) < 0)
				pFileAttribAck->rc = MapWindowsError (errno);

			break;
		}
		case GET_ATTRIB_LAST_ACCESS_DATETIME:
		{	
			!fDebug ? 0 : printf ("GET_ATTRIB_LAST_ACCESS_DATETIME\n"); 
			pFileAttribAck->NativeTime = fStat.st_atime;
			break;
		}
		case SET_ATTRIB_LAST_ACCESS_DATETIME:
		{
			!fDebug ? 0 : printf ("SET_ATTRIB_LAST_ACCESS_DATETIME\n");
			uTime.actime = pFileAttrib->NativeTime;

			if (utime (NATIVEPath, &uTime) < 0)
				pFileAttribAck->rc = MapWindowsError (errno);

			break;
		}
		case GET_ATTRIB_CREATION_DATETIME:
		{
			!fDebug ? 0 : printf ("GET_ATTRIB_CREATION_DATETIME\n"); 
			pFileAttribAck->NativeTime = fStat.st_mtime;
			break;
		}
		case SET_ATTRIB_CREATION_DATETIME:
		{
			!fDebug ? 0 : printf ("SET_ATTRIB_CREATION_DATETIME\n"); 
			uTime.modtime = pFileAttrib->NativeTime;

			if (utime (NATIVEPath, &uTime) < 0)
				pFileAttribAck->rc = MapWindowsError (errno);

			break;
		}
		default:
			!fDebug ? 0 : printf ("File Attributes: ERROR! Unknown Type=%d\n", pFileAttrib->ulFlags);
		}
	}
	else
	{
		pFileAttribAck->rc = MapWindowsError (errno);
		!fDebug ? 0 : printf ("fsFileAttributes: Stat failed on %s\n", NATIVEPath);
	}

	pFileAttribAck->usLen = htons(sizeof(fs_fileattributesAck));
	pFileAttribAck->ulKey = pFileAttrib->ulKey;
	pFileAttribAck->usType = htons(pFileAttrib->usType);

	pFileAttribAck->ulAttribs = htonl(pFileAttribAck->ulAttribs);
	pFileAttribAck->NativeTime = htonl(pFileAttribAck->NativeTime);
	pFileAttribAck->ulFileSize = htonl(pFileAttribAck->ulFileSize);
	pFileAttribAck->rc = htons(pFileAttribAck->rc);

	writen (sockfd, pFileAttribAck, sizeof(fs_fileattributesAck));

	!fDebug ? 0 : printf ("FS_FILEATTRIBUTES: rc=%d, attrib=%lx\n", 
		ntohs(pFileAttribAck->rc), ntohl(pFileAttribAck->ulAttribs));
}


/*
* fsFilePermissions
*/
void fsFilePermissions (sockfd, pInBuff)
int sockfd;
char *pInBuff;
{
	fs_filepermissions		*pFilePerm;
	fs_filepermissionsAck	*pFilePermAck;
	struct stat				fStat;
	struct group			*pGroup;
	struct passwd			*pPasswd;
	char					NATIVEPath[MAX_PATH + 1];
	short					rc = 0;
	USHORT					usPacketSize;
	int						fUnixText = FALSE;

	pFilePerm = (fs_filepermissions *)pInBuff;

	pFilePerm->ulPermissions = ntohl(pFilePerm->ulPermissions);
	pFilePerm->sPathLen = ntohs(pFilePerm->sPathLen);
	pFilePerm->sOwnerLen = ntohs(pFilePerm->sOwnerLen);
	pFilePerm->sGroupLen = ntohs(pFilePerm->sGroupLen);

	!fDebug ? 0 : printf ("FS_PERMISSIONS: Flags=%x, Permissions=%x, Path=%s\n",
		pFilePerm->ulFlags, pFilePerm->ulPermissions, pFilePerm->PathGroupOwner);

	pFilePermAck = (fs_filepermissionsAck *)OutBuff;
	memset ((char *)pFilePermAck, 0, sizeof(fs_filepermissionsAck));

	if ((pFilePermAck->rc = MakeNativePath (pFilePerm->PathGroupOwner, NATIVEPath)))
	{}
	else if (!(pFilePermAck->rc = stat(NATIVEPath, &fStat)))
	{
		/* Get permissions, group, and owner */

		if (pFilePerm->ulFlags & ZANSERV_GET_ALL_PERMS)
		{
			/* Get Permissions */

			pFilePermAck->ulPermissions = fStat.st_mode;

			/* Get Group */

			pGroup = getgrgid (fStat.st_gid);

			if (pGroup)
			{
				pFilePermAck->sGroupLen = strlen (pGroup->gr_name);
				strcpy (pFilePermAck->GroupOwner, pGroup->gr_name);

				/* Get Owner */

				pPasswd = getpwuid (fStat.st_uid);

				if (pPasswd)
				{
					

					pFilePermAck->sOwnerLen = strlen (pPasswd->pw_name);
					strcpy (&pFilePermAck->GroupOwner[pFilePermAck->sGroupLen+1], pPasswd->pw_name);

					/* Is this a text file ? */

					if (!S_ISDIR(fStat.st_mode))
					{
						if (IsTextFile (NATIVEPath, &fUnixText))
						{
							if (fUnixText)
								pFilePermAck->ulFlags |= ZANSERV_UNIX_TEXT;
							else
								pFilePermAck->ulFlags |= ZANSERV_WINDOWS_TEXT;
						}
					}
				}
				else
					rc = errno;
			}
			else
				rc = errno;

			if (rc)
				pFilePermAck->rc = MapWindowsError (rc);
		}

		/* Set permissions, group, and owner */

		else
		{
			/* Set Permissions */

			if (pFilePerm->ulFlags & ZANSERV_SET_PERMISSIONS)
			{
				pFilePermAck->ulFlags = ZANSERV_SET_PERMISSIONS;

				if (chmod (NATIVEPath, pFilePerm->ulPermissions) < 0)
					rc = errno;
			}

			/* Set Owner */

			if (pFilePerm->ulFlags & ZANSERV_SET_OWNER && !rc)
			{
				pFilePermAck->ulFlags = ZANSERV_SET_OWNER;

				!fDebug ? 0 : printf ("fsFilePermissions: Set new owner: %s\n", 
					&pFilePerm->PathGroupOwner[pFilePerm->sPathLen + pFilePerm->sGroupLen + 1]);

				pPasswd = getpwnam (&pFilePerm->PathGroupOwner[pFilePerm->sPathLen + pFilePerm->sGroupLen + 1]);

				if (pPasswd)
				{
					if (chown (NATIVEPath, pPasswd->pw_uid, fStat.st_gid) != 0)
						rc = errno;
					else
						fStat.st_uid = pPasswd->pw_uid; /* Update for below chown() */
				}
				else 
					rc = errno;

			}

			/* Set Group */

			if (pFilePerm->ulFlags & ZANSERV_SET_GROUP && !rc)
			{
				pFilePermAck->ulFlags = ZANSERV_SET_GROUP;

				!fDebug ? 0 : printf ("fsFilePermissions: Set new group: %s\n", 
					&pFilePerm->PathGroupOwner[pFilePerm->sPathLen]);

				pGroup = getgrnam (&pFilePerm->PathGroupOwner[pFilePerm->sPathLen]);

				if (pGroup)
				{
					if (chown (NATIVEPath, fStat.st_uid, pGroup->gr_gid) != 0)
						rc = errno;
				}
				else 
					rc = errno;
			}

			/* Translate Text */

			if (pFilePerm->ulFlags & ZANSERV_SET_TEXT && !rc)
			{
				if (pFilePerm->ulFlags & ZANSERV_UNIX_TEXT)
				{
					pFilePermAck->ulFlags = ZANSERV_UNIX_TEXT;
					rc = TranslateTextFile (NATIVEPath, TRUE);
				}
				else if (pFilePerm->ulFlags & ZANSERV_WINDOWS_TEXT)
				{
					pFilePermAck->ulFlags = ZANSERV_WINDOWS_TEXT;
					rc = TranslateTextFile (NATIVEPath, FALSE);
				}
				else
					rc = ERROR_BAD_ARGUMENTS;
			}

			if (rc)
			{
				pFilePermAck->rc = MapWindowsError (rc);
				!fDebug ? 0 : printf ("fsFilePermissions: error = %d\n", rc);
			}
		}
	}
	else
	{
		pFilePermAck->rc = MapWindowsError (errno);
		!fDebug ? 0 : printf ("fsFilePermissions: Stat failed on %s\n", NATIVEPath);
	}

	usPacketSize = sizeof(fs_filepermissionsAck) + pFilePermAck->sGroupLen + pFilePermAck->sOwnerLen - 2;

	pFilePermAck->usLen = htons(usPacketSize);
	pFilePermAck->sGroupLen = htons(pFilePermAck->sGroupLen);
	pFilePermAck->sOwnerLen = htons(pFilePermAck->sOwnerLen);
	pFilePermAck->ulKey = pFilePerm->ulKey;
	pFilePermAck->usType = htons(pFilePerm->usType);

	pFilePermAck->ulPermissions = htonl(pFilePermAck->ulPermissions);
	pFilePermAck->ulFlags = htonl(pFilePermAck->ulFlags);
	pFilePermAck->rc = htons(pFilePermAck->rc);

	writen (sockfd, pFilePermAck, usPacketSize);

	!fDebug ? 0 : printf ("FS_PERMISSIONS: rc=%d, permissions=%lx\n", 
		ntohs(pFilePermAck->rc), ntohl(pFilePermAck->ulPermissions));
}


/*
*  fsDir
*/
void fsDir (sockfd, pInBuff)
int sockfd;
char *pInBuff;
{
	fs_dir		*pDir;
	fs_dirAck	*pDirAck;
	struct stat	fStat;
	char		NATIVEPath[MAX_PATH];
	ULONG		ulDirAttribs=ulDefaultAttributes;
	int			len=0;


	pDir = (fs_dir *)pInBuff;

	pDir->ulAttribs = ntohl(pDir->ulAttribs);
	pDir->sPathLen = ntohs(pDir->sPathLen);

	!fDebug ? 0 : printf ("FS_DIR: Flags=%x, Path=%s\n", pDir->ulFlags, pDir->Path);

	pDirAck = (fs_dirAck *)OutBuff;
	memset ((char *)pDirAck, 0, sizeof(fs_dirAck));

	pDirAck->rc = MakeNativePath (pDir->Path, NATIVEPath);

	if (!pDirAck->rc)
	{
		switch (pDir->ulFlags)
		{
		case CREATE_DIR:
		{
			!fDebug ? 0 : printf ("CREATE_DIR: %s\n", NATIVEPath);

			if (ulDefaultAttributes & S_IRUSR)
				ulDirAttribs |= S_IXUSR;
			if (ulDefaultAttributes & S_IRGRP)
				ulDirAttribs |= S_IXGRP;
			if (ulDefaultAttributes & S_IROTH)
				ulDirAttribs |= S_IXOTH;
			
			if (mkdir(NATIVEPath, ulDirAttribs) < 0)
			{
				pDirAck->rc = MapWindowsError (errno);

				if (pDirAck->rc == ERROR_FILE_EXISTS)
					pDirAck->rc = ERROR_ALREADY_EXISTS;
			}
			break;
		}
		case DELETE_DIR:
		{
			!fDebug ? 0 : printf ("DELETE_DIR: %s\n", NATIVEPath);

			if (rmdir(NATIVEPath) < 0)
				pDirAck->rc = MapWindowsError (errno);
			break;
		}
		case QUERY83_DIR:
		case QUERYLONG_DIR:
		{
			!fDebug ? 0 : printf ("QUERY DIR %d: %s\n", pDir->ulFlags, NATIVEPath);

			pDirAck->rc = 0;

			/* Handle root case */

			if (HomePath[0] == '/' && HomePath[1] == 0)
				strcpy (pDirAck->Path, NATIVEPath);
			else
				strcpy (pDirAck->Path, &NATIVEPath[strlen(HomePath)]);

			pDirAck->sPathLen = strlen(pDirAck->Path) + 1;
			break;
		}
		case CHECK_DIR: 
		{
			!fDebug ? 0 : printf ("CHECK_DIR: %s\n", NATIVEPath);

			if (!(stat (NATIVEPath, &fStat)))
			{
				if (S_ISDIR(fStat.st_mode))
				{
					!fDebug ? 0 : printf ("Found Valid Directory\n");
					pDirAck->rc = 0;
				}
				else
					pDirAck->rc = ERROR_FILE_NOT_FOUND;
			}
			else
				pDirAck->rc = MapWindowsError (errno);

			break;
		}
		default:
			!fDebug ? 0 : printf ("FsDir: ERROR_INVALID_FUNCTION!!!!\n");
			pDirAck->rc = ERROR_INVALID_FUNCTION;
		}
	}
	else 
		pDirAck->rc = ERROR_PATH_NOT_FOUND;

	len = sizeof(fs_dirAck) + pDirAck->sPathLen;

	pDirAck->usLen = htons(len);
	pDirAck->ulKey = pDir->ulKey;
	pDirAck->usType = htons(pDir->usType);
	pDirAck->ulFlags = htonl(pDir->ulFlags);

	pDirAck->rc = htons(pDirAck->rc);

	writen (sockfd, pDirAck, len);

	!fDebug ? 0 : printf ("FS_DIR: rc=%d\n", ntohs(pDirAck->rc));
}


/*
*  fsDeleteFile
*/
void fsDeleteFile (sockfd, pInBuff)
int sockfd;
char *pInBuff;
{
	fs_deletefile		*pDelete;
	fs_deletefileAck	*pDeleteAck;



	char		NATIVEPath[MAX_PATH];

	struct stat				fStat;

	pDelete = (fs_deletefile *)pInBuff;

	pDelete->sPathLen = ntohs(pDelete->sPathLen);

	!fDebug ? 0 : printf ("FS_DELETEFILE: Attribs=%x, Path=%s\n", 
		pDelete->ulAttribs, pDelete->Path);

	pDeleteAck = (fs_deletefileAck *)OutBuff;
	memset ((char *)pDeleteAck, 0, sizeof(fs_deletefileAck));

	if (!(pDeleteAck->rc = MakeNativePath (pDelete->Path, NATIVEPath)))
	{
		if (!(stat(NATIVEPath, &fStat)))
		{
			if (!(fStat.st_mode & S_IWUSR))
			{
				/* Readonly */

				pDeleteAck->rc = ERROR_ACCESS_DENIED;
			}
			else if (!S_ISDIR(fStat.st_mode))
			{
				/* Delete a single file */

				!fDebug ? 0 : printf ("FS_DELETEFILE: Single file delete\n");

				if (unlink(NATIVEPath) < 0)
				{
					pDeleteAck->rc = MapWindowsError (errno);

					!fDebug ? 0 : printf ("FS_DELETEFILE: fail - rc = %d\n", errno);
				}
				else
					pDeleteAck->rc = 0;
			}
			else 
				pDeleteAck->rc = ERROR_FILE_NOT_FOUND;
		}
		else 
			pDeleteAck->rc = ERROR_FILE_NOT_FOUND;
	}

	pDeleteAck->usLen = htons(sizeof(fs_deletefileAck));
	pDeleteAck->ulKey = pDelete->ulKey;
	pDeleteAck->usType = htons(pDelete->usType);

	pDeleteAck->rc = htons(pDeleteAck->rc);

	writen (sockfd, pDeleteAck, sizeof(fs_deletefileAck));

	!fDebug ? 0 : printf ("FS_DELETEFILE: rc=%d\n", ntohs(pDeleteAck->rc));
}


/*
*  fsRenameFile
*/
void fsRenameFile (sockfd, pInBuff)
int sockfd;
char *pInBuff;
{
	fs_renamefile		*pRename;
	fs_renamefileAck	*pRenameAck;
	char				NATIVEPath1[MAX_PATH];
	char				NATIVEPath2[MAX_PATH];

	pRename = (fs_renamefile *)pInBuff;

	pRename->sPathLen1 = ntohs(pRename->sPathLen1);
	pRename->sPathLen2 = ntohs(pRename->sPathLen2);

	!fDebug ? 0 : printf ("FS_RENAMEFILE: Attribs=%x, From=%s, To=%s\n",
		pRename->ulAttribs, pRename->Paths, &pRename->Paths[pRename->sPathLen1]);
		
	pRenameAck = (fs_renamefileAck *)OutBuff;
	memset ((char *)pRenameAck, 0, sizeof(fs_renamefileAck));

	if (!(pRenameAck->rc = MakeNativePath (pRename->Paths, NATIVEPath1)))
	{
		pRenameAck->rc = MakeNativePath (&pRename->Paths[pRename->sPathLen1], NATIVEPath2);

		if (!pRenameAck->rc)
		{
			if (rename(NATIVEPath1, NATIVEPath2) < 0)
				pRenameAck->rc = MapWindowsError (errno);
		}
	}
	else
		pRenameAck->rc = ERROR_PATH_NOT_FOUND;

	pRenameAck->usLen = htons(sizeof(fs_renamefileAck));
	pRenameAck->ulKey = pRename->ulKey;
	pRenameAck->usType = htons(pRename->usType);

	pRenameAck->rc = htons(pRenameAck->rc);

	writen (sockfd, pRenameAck, sizeof(fs_renamefileAck));

	!fDebug ? 0 : printf ("FS_RENAMEFILE: rc=%d\n", ntohs(pRenameAck->rc));
}


/*
*  fsGetDiskInfo
*/
void fsGetDiskInfo (sockfd, pInBuff)
int sockfd;
char *pInBuff;
{
	fs_getdiskinfo		*pGetDiskInfo;
	fs_getdiskinfoAck	*pGetDiskInfoAck;
	ULONG				ulFileCount=0;

	/* Walk the tree starting from the directory where we were
	   started and not exceeding MAX readdir count */

	pGetDiskInfo = (fs_getdiskinfo *)pInBuff;

	!fDebug ? 0 : printf ("FS_GETDISKINFO: Starting Tree Walk...\n");

	pGetDiskInfoAck = (fs_getdiskinfoAck *)OutBuff;
	memset ((char *)pGetDiskInfoAck, 0, sizeof(fs_getdiskinfoAck));

	ProcessDir (HomePath, &pGetDiskInfoAck->ulDiskSpaceUsedBytes, &ulFileCount);

	if (ulFileCount >= ulMaxReaddirs)
	{
		!fDebug ? 0 : printf ("FS_GETDISKINFO: Maximum file count exceeded! Adjust \"Disk Space Calculation\".\n");
		pGetDiskInfoAck->rc = ERROR_NO_MORE_FILES;
		pGetDiskInfoAck->ulDiskSpaceUsedBytes = 0;
	}

	pGetDiskInfoAck->usLen = htons(sizeof(fs_getdiskinfoAck));
	pGetDiskInfoAck->ulKey = pGetDiskInfo->ulKey;
	pGetDiskInfoAck->usType = htons(pGetDiskInfo->usType);

	pGetDiskInfoAck->ulDiskSpaceUsedBytes = htonl(pGetDiskInfoAck->ulDiskSpaceUsedBytes);
	pGetDiskInfoAck->rc = htons(pGetDiskInfoAck->rc);

	writen (sockfd, pGetDiskInfoAck, sizeof(fs_getdiskinfoAck));

	!fDebug ? 0 : printf ("FS_GETDISKINFO: rc=%d, DiskSize=%ld\n", 
		ntohs(pGetDiskInfoAck->rc),
		ntohl(pGetDiskInfoAck->ulDiskSpaceUsedBytes));
}



/*
*  ProcessDir
*/
void ProcessDir (pPath, pulDiskSpace, pulFileCount)
char *pPath;
ULONG *pulDiskSpace;
ULONG *pulFileCount;
{
	DIR					*pCurrDir;
	struct	dirent		*pDirEntry;
	struct	stat		Status;
	char				Cwd[MAX_PATH];

	if ((pCurrDir = opendir (pPath)))
	{
		if (getcwd (Cwd,MAX_PATH))
		{
			if (!chdir(pPath))
			{
				while ((pDirEntry = readdir (pCurrDir)))
				{
					if (!stat(pDirEntry->d_name, &Status))
					{
						*pulDiskSpace += Status.st_size;
						(*pulFileCount)++;
	
						/*
							!fDebug ? 0 : printf ("ProcessDir: File=%s, Total Size=%d, Count=%d\n",
								pDirEntry->d_name, *pulDiskSpace, *pulFileCount);
						*/

						if (*pulFileCount >= ulMaxReaddirs)
							break;

						if (S_ISDIR(Status.st_mode))
						{
							if (strcmp(pDirEntry->d_name, "."))
							{
								if (strcmp(pDirEntry->d_name, ".."))
								{
									ProcessDir (pDirEntry->d_name, pulDiskSpace, pulFileCount);
								}
							}
						}
					}
				}
			}
			chdir (Cwd);
		}
		closedir (pCurrDir);
	}
}


/*****************************************************

  UTILITIES

******************************************************/

/*
* MakeNativePath  
*/
int MakeNativePath (pInPath, pOutPath)
char *pInPath;
char *pOutPath;
{
	char	*p, *pTok, *pLastSlash;
	int		i, rc=0;
	char	TempInPath[MAX_PATH];
	char	TempOutPath[MAX_PATH];

	/* !fDebug ? 0 : printf ("MakeNativePath - InPath = %s\n", pInPath); */ 

	if (!pInPath)
		return 0;

	if (!pInPath[0])
	{
		strcpy (pOutPath, HomePath); 	 
		return 0;
	}
	
	strcpy (TempOutPath, HomePath);

	if (pInPath[0])
	{
		/* Convert all the '\' to '/' */

		for (i=0, p=pInPath; i<strlen(pInPath); i++, p++)
		{
			if (*p == WindowsBSlash)
				*p = NativeSlash;
		}

		strcpy (TempInPath, pInPath);
	
		/* Parse the path (if any) and make sure we can find 
		all the sub-dirs in UPPERCASE or mangled for Windows  */

		pTok = TempInPath;

		pLastSlash = strrchr (pTok, NativeSlash);
		if (pLastSlash) pLastSlash++;

		while ((p = strtok (pTok, NativeSlashStr)))
		{
			rc = CaseMatchDirectory (p, TempOutPath);

			if (rc)
			{
				/* Error may be okay if this is the last element */

				if (p >= pLastSlash) 
					rc = 0;

				break;	
			}

			pTok = NULL;

			/* !fDebug ? 0 : printf ("MakeNativePath - adding: %s\n", TempOutPath); */
		}
	}

	strcpy (pOutPath, TempOutPath);

	!fDebug ? 0 : printf ("MakeNativePath - OutPath = %s\n", pOutPath);  

	return rc;
}


/*
*  CaseMatchDirectory
*/
int CaseMatchDirectory (pFile, pPath)
char *pFile;
char *pPath;
{
	DIR				*pDir;
	struct	dirent	*pDirEnt = 0;
	char			EntryName[MAX_PATH];
	char			EntryNameUpper[MAX_PATH];
	char			FileUpper[MAX_PATH];
	int				fMangled = FALSE;
	int				fFoundElement = FALSE;

	/* Is this a mangled path element? */

	if (is_mangled (pFile))
		fMangled = TRUE;

	/*
	else
		!fDebug ? 0 : printf ("CaseMatchDirectory: fMangled = FALSE\n");
	*/

	if ((pDir = opendir (pPath)))
	{
		while ((pDirEnt = readdir (pDir)))
		{
			strcpy (EntryName, pDirEnt->d_name);

			if (fMangled)
				fFoundElement = CompareAgainstMangled (pFile, EntryName);

			if (!fFoundElement)
			{
				strcpy (EntryNameUpper, EntryName);
				strcpy (FileUpper, pFile);

				if (!(strcmp(zstrupr(FileUpper), zstrupr(EntryNameUpper))))
				{
					fFoundElement = TRUE;
					break;
				}
			}
			else
				break;
		}

		/* Add the directory/file to path */

		if (strcmp(pPath, "/"))
			strcat (pPath, "/");

		/* !fDebug ? 0 : printf ("CaseMatchDirectory: Found = %s\n", EntryName); */

		if (fFoundElement)
			strcat (pPath, EntryName);
		else
			strcat (pPath, pFile);
	}
	else
	{
		!fDebug ? 0 : printf ("CaseMatchDirectory - Can't open dir: %s, File: %s\n", pPath, pFile);
		return MapWindowsError (errno);
	}

	closedir (pDir);

	if (!fFoundElement)
		return ERROR_FILE_NOT_FOUND;

	return 0;
}


/*
* CompareAgainstMangled
*/
int CompareAgainstMangled (pWinFile, pUnixFile)
char *pWinFile;
char *pUnixFile;
{
	char	TempFile[MAX_PATH];
	char	TmpWin[MAX_PATH];

	strcpy (TempFile, pUnixFile);
	strcpy (TmpWin, pWinFile);

	/* Mangle the Unix name */

	mangle_name_83 (TempFile);

	zstrupr (TmpWin);

	!fDebug ? 0 : printf ("CompareAgainstMangled - Mangled File: %s, InFile: %s\n", TempFile, TmpWin);

	/* Does it match the Windows name? */

	if (!strcmp(TempFile, TmpWin))
		return TRUE;
	else
		return FALSE;
}


/*
*  zstrupr
*/
char *zstrupr (pString)
char *pString;
{
	char	*p = pString;

	while (*p)
	{
		if (*p >= 'a' && *p <= 'z')
			*p -= 32;

		p++;
	}
	return pString;
}


#define		STAR_DOT_STAR	"*.*"
#define		STAR_DOT		"*."
#define		LONE_STAR		"*"

/*
* MetaMatch - Warning! You must convert to a native path before using this routine.
*             Also, you don't need to call this routine unless the FILE_FLAG_HAS_STAR
*             flag is set in the FS_FINDFIRSTFILE, FS_RENAMEFILE, and FS_DELETEFILE
*             file system calls.
*
* Note: Even with long filenames Windows 95 still supports "*.*" and "*." in 
*       the old 8.3 fashion. 
*
*  
*/
int MetaMatch (pWindowsFile, pNativeFile)
char *pWindowsFile;
char *pNativeFile;
{
	char	tmpWindowsPattern[MAX_PATH];
	char	tmpNativeFile[MAX_PATH];

	int		rc;

	/* In Windows long file name support the meta-character '*' indicates
	0 or more characters and '?' indicates one character. You can have
	multiple '*' and '?' */

	/* printf ("MetaMatch File=%s, Path=%s\n", pWindowsFile, pNativeFile); */  

	if (pWindowsFile[0] == 0)
		return FALSE;

	strcpy (tmpWindowsPattern, pWindowsFile);
	strcpy (tmpNativeFile, pNativeFile);

	zstrupr (tmpWindowsPattern);
	zstrupr (tmpNativeFile);

	/* No path names at this point */

	if (strrchr (tmpWindowsPattern, NativeSlash))
		return FALSE;

	/* Handle special Windows case: "*.*", "*." and regular "*" */

	if (!(strcmp(tmpWindowsPattern, LONE_STAR)))
		return TRUE;	/* This was a "*" */

	if (!(strcmp(tmpWindowsPattern, STAR_DOT_STAR)))
		return TRUE;	/* This was a "*.*" */

	rc = CheckForExtension (pNativeFile);

	if (!(strcmp(tmpWindowsPattern, STAR_DOT)))
		return 	rc ? FALSE : TRUE; /* This was a "*." */

	if (!rc) /* If no extension */
	{
		/* Handle "<pattern>.*" case for files without extensions */

		NukeDotStar (tmpWindowsPattern);
	}

	return MatchStarsAndQMs (tmpWindowsPattern, tmpNativeFile);
}


/*
*  MatchStarsAndQMs
*/
int MatchStarsAndQMs (pPattern, pNativeFile)
char *pPattern;
char *pNativeFile;
{
	char *pP=pPattern, *pF=pNativeFile;

	/*
	printf ("MatchStarsAndQMs - pPattern=%s, pNativeFile=%s\n", 
		pPattern, pNativeFile);
	*/

	while (*pF && *pP) /* Scan through file name */
	{
		if (*pP == '?')
		{
			pP++; pF++;
		}
		else if (*pP == '*')
		{
			pP++;

			if (*pP == 0)
				return TRUE;

			if (*pP == '*')
			{
				pP++;

				if (*pP == 0)
					return TRUE;

			}
			if (!(FindSubPattern (pP, pF, &pP, &pF)))
				pP--;
			else if (*pP == 0 && *pF != 0)
				return FALSE;
		}
		else if (*pP == *pF)
		{
			pP++; pF++;
		}
		else if (*pP != *pF)
		{
			return FALSE;
		}

	}
	if (*pP)
		return FALSE;
	
	return TRUE;
}


/*
*  FindSubPattern
*/
int FindSubPattern (pSubPat, pSubFile, pPatOut, pFileOut)
char *pSubPat;
char *pSubFile;
char **pPatOut;
char **pFileOut;
{
	char	*pSubTmp = pSubPat;

	/* printf ("FindSubPattern - pSubPat=%s, pSubFile=%s\n", pSubPat, pSubFile); */ 

	while (*pSubPat && *pSubFile)
	{
		/* Compare up to next '*' skip any '?' */

		if (*pSubPat == '*')
		{
			*pPatOut = pSubPat;
			*pFileOut = pSubFile;

			return TRUE;
		}

		if (*pSubPat == '?')
		{
			pSubPat++;
		}
		else if (*pSubPat == *pSubFile)
		{
			pSubPat++;
		}
		else
			pSubPat = pSubTmp;

		pSubFile++;
	}
	*pFileOut = pSubFile;

	if (*pSubPat == 0 && *pSubFile)
		return FALSE;

	if (*pSubPat)
		return TRUE;
	
	*pPatOut = pSubPat;

	return TRUE;
}



/*
*  CheckForExtension
*/
int CheckForExtension (pFile)
char *pFile;
{
	char	*p;
	int		i;

	/* This was a "*."  If any of the last four characters is a '.'
	we consider this a file with an 8.3 extenstion. Windows uses
	"*." to indicate files with no extension. */

	p = &pFile[strlen(pFile)-1];	/* Last char in string */

	for (i=0; i<4; i++, p--)
	{
		if (*p == '.')
			return TRUE;
	}

	return FALSE;
}


/*
*  NukeDotStar
*/
void NukeDotStar (pPattern)
char *pPattern;
{
	if (strlen(pPattern) >= 2)
	{
		if (pPattern[strlen(pPattern)-1] == '*')
		{
			if (pPattern[strlen(pPattern)-2] == '.')
			{
				pPattern[strlen(pPattern)-2] = 0;
			}
		}
	}
}


/*
*  StripFileFromPath - Puts a NULL in the last '\' and return ptr 
*                      to the file or search.
*/
char *StripFileFromPath (pPath, SlashType)
char *pPath;
char SlashType;
{
	char *p;

	/* Get the file name from the full Windows path. Windows ALWAYS provides
	a full path starting with '\' */

	if (!(p = strrchr (pPath, SlashType)))
		return 0;

	*p = 0;

	return p+1;
}


/*
*  Signal Handler for read timeouts
*/
void SigReadTimeout (int sig)
{


	if (sig == SIGALRM)
	{
		printf ("SigReadTimeout: SIGALRM\n");

		/* Windows client is gone. Time to exit. */
		
		if (!fOptMultiUser)
		{
			/* Kill the parent (the shell) for Telnet mode. We do this because
			   some Unix shells don't exit properly when we abnormally disconnect. */

			printf ("SigReadTimeout: kill (getppid(), SIGHUP)\n");

			kill (getppid(), SIGHUP);
		}
		ShutDown (1);
	}
}
	

/*
*  Read "n" bytes from a stream socket or file
*/
int readn (fd, ptr, nbytes)
int fd;
char *ptr;
int nbytes;
{
	int	nleft, nread;

	nleft = nbytes;

	while (nleft > 0)
	{
		(void) alarm (900); /* Check connection every 15 minutes */
		nread = read (fd, ptr, nleft);
		(void) alarm (0);

		if (nread < 0)
		{
			if (errno != EINTR)
			{
				!fDebug ? 0 : printf ("ZanNet error read(): rc = %d\n", errno);
				return nread;
			}
		}
		else if (nread == 0)
			break;		/* EOF */

		nleft -= nread;
		ptr += nread;
	}
	return (nbytes - nleft);	/* return >= 0 */
}


/*
*  Write "n" bytes to a stream socket 
*/
int writen (fd, ptr, nbytes)
int fd;
char *ptr;
int nbytes;
{
	int	nleft, nwritten;

	/* printf ("writen() fd=%d, ptr=%lx, nbytes=%d\n", fd, ptr, nbytes); */

	nleft = nbytes;

	while (nleft > 0)
	{
		nwritten = write (fd, ptr, nleft);

		if (nwritten <= 0)			/* error */
		{
            !fDebug ? 0 : printf ("ZanNet error write(): rc = %d\n", errno);
			return nwritten;
		}

		nleft -= nwritten;
		ptr += nwritten;
	}
	return (nbytes - nleft);
}


/*
*  MapWindowsError
*/
USHORT MapWindowsError (usPosixErr)
USHORT usPosixErr;
{
	if (usPosixErr > MAX_ERRORS)
		return ERROR_UNEXP_NET_ERR;

	return WindowsErrors[usPosixErr];
}


/*
*  SetFilePosition
*/
int SetFilePosition (iFileDesc, lNewPos)
int iFileDesc;
long lNewPos;
{
	struct stat	fStat;
	long lPosition;

	/* Get the length of this file */

	if (fstat(iFileDesc, &fStat) < 0)
		return MapWindowsError(errno);

	/* Make sure we are within the file */

	if (lNewPos > fStat.st_size)
		return ERROR_ACCESS_DENIED;

	/* Seek the file to new location */
	
	if ((lPosition = lseek (iFileDesc, lNewPos, SEEK_SET)) < 0)
	{
		!fDebug ? 0 : printf ("SetFilePosition: lseek failed, errno=%d, lPosition=%ld, iFileDesc=%d, lNewPos=%ld\n", 
			errno, lPosition, iFileDesc, lNewPos); 

		return MapWindowsError(errno);
	}

	if (lPosition != lNewPos)
		return ERROR_ACCESS_DENIED;

	return 0;
}


/*** Anything after this line is new for ZanNet Beta III. ***/


/*
* The following routines: NewMemoryHandle(), GetMemoryByHandle(), and DeleteMemoryHandle() are
* for memory handle to pointer conversions. We can refer to Unix memory pointers with 32 bit handles
* using these routines. 
*/

int	gHandleIndex = 0;
zn_memoryObject *pMemHead=0;


/*
*  NewMemoryHandle
*/
int NewMemoryHandle (pvMem)
void *pvMem;
{
	zn_memoryObject	*pObj=0;

	/* printf ("NewMemoryHandle: Entry - pvMem = %lx\n", pvMem); */

	if (pMemHead)
	{
		for (pObj=pMemHead; pObj->pNext; pObj=pObj->pNext);
		pObj->pNext = (zn_memoryObject *)malloc (sizeof(zn_memoryObject));
		pObj = pObj->pNext;
	}
	else
	{
		pMemHead = (zn_memoryObject *)malloc (sizeof(zn_memoryObject));
		pObj = pMemHead;
	}

	memset (pObj, 0, sizeof(zn_memoryObject));

	pObj->pvMem = pvMem;
	pObj->handle = ++gHandleIndex;

	return pObj->handle;
}


/*
*  GetMemoryByHandle
*/
void* GetMemoryByHandle (handle)
int handle;
{
	zn_memoryObject	*pObj=0;

	/* printf ("GetMemoryByHandle: Entry - handle = %d\n", handle); */

	for (pObj=pMemHead; pObj; pObj=pObj->pNext)
	{
		if (pObj->handle == handle)
			return pObj->pvMem;
	}
	return 0;
}


/*
*  DeleteMemoryHandle
*/
void DeleteMemoryHandle (handle)
int handle;
{
	zn_memoryObject	*pObj=0;

	/* printf ("DeleteMemoryHandle: Entry - handle = %d\n", handle); */

	for (pObj=pMemHead; pObj; pObj=pObj->pNext)
	{
		if (pObj->handle == handle)
		{
			UnChain (pObj);
			return;
		}
	}
}


/*
*  UnChain
*/
int UnChain (pRemove)
zn_memoryObject* pRemove;
{
	zn_memoryObject	*pCurr, *pPrev;

	/* printf ("UnChain: Entry - pRemove = %lx\n", pRemove); */

	pCurr = pPrev = pMemHead;

	while (pCurr)
	{
		if (pRemove == pCurr)
		{
			/* printf ("UnChain: Entry - Removing object\n"); */

			if (pCurr == pMemHead)
			{
				pMemHead = pCurr->pNext;
				/* printf ("UnChain: Entry - Removing Head\n"); */
				return 0;
			}
			pPrev->pNext = pCurr->pNext;
			return 0;
		}

		pPrev = pCurr;
		pCurr = pCurr->pNext;
	}
	return -1;
}

/*** 8.3 Name Support ***/

/*
* is_mangled
*/
int is_mangled(s)
char *s;
{
  char *m;
  
  m = strchr(s,magic_char);
  if (!m) return(FALSE);

  /* we use two base 36 chars before the extension */
  if (m[1] == '.' || m[1] == 0 ||
      m[2] == '.' || m[2] == 0 ||
      (m[3] != '.' && m[3] != 0))
    return(is_mangled(m+1));

  /* it could be */
  return(TRUE);
}


/*
* base36: return a base 36 character. v must be from 0 to 35.
*/
static char base36(v)
unsigned int v;
{
  static char basechars[] = "0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZ";
  return basechars[v % 36];
}


/*
* str_checksum: provide a checksum on a string
*/
int str_checksum(s)
char *s;
{
  int res = 0;
  int c;
  int i=0;
  while (*s)
    {
      c = *s;
      res ^= (c << (i % 15)) ^ (c >> (15-(i%15)));
      s++; i++;
    }
  return(res);
}


/*
* mangle_name_83
*/
void mangle_name_83 (s)
char *s;
{
  int csum = str_checksum(s);
  char *p;
  char extension[4];
  char base[9];
  int baselen = 0;
  int extlen = 0;

  extension[0]=0;
  base[0]=0;

  p = strrchr(s,'.'); 
 
  if (p && (strlen(p+1)<4) )
  {
    if (p[1] != 0)
	{
	  *p = 0;
	  csum = str_checksum(s);
	  *p = '.';
	}
  }
      
  zstrupr(s);

  if (p)
  {
	if (p == s)
		strcpy(extension,"___");
    else
	{
	  *p++ = 0;
	  while (*p && extlen < 3)
	  {

	    if (isdoschar(*p) && *p != '.')
			extension[extlen++] = *p;
	    p++;
	  }
	  extension[extlen] = 0;
	}
  }

  p = s;

  while (*p && baselen < 5)
  {

    if (isdoschar(*p) && *p != '.')
    	base[baselen++] = *p;
    p++;
  }

  base[baselen] = 0;

  csum = csum % (36*36);

  strcpy (s, base);
  s[baselen] = magic_char;
  s[baselen+1] = base36(csum/36);
  s[baselen+2] = base36(csum%36);
  s[baselen+3] = 0;
  
  /* sprintf(s,"%s%c%c%c",base,magic_char,base36(csum/36),base36(csum%36)); */

  if (*extension)
  {
     strcat(s,".");
     strcat(s,extension);
  }
}


void InitDosCharMap ()
{
	int i;

	for (i=0; i<256; i++)
	{
		if (i >= 'A' && i <= 'Z')
			dos_char_map[i] = 0xFF;
		else if (i >= 'a' && i <= 'z')
			dos_char_map[i] = 0xFF;
		else if (i >= '0' && i <= '9')
			dos_char_map[i] = 0xFF;
		else
		{
			switch (i)
			{
				case '.': case '_': case '^': case '$': case '~': 
				case '!': case '#': case '%': case '&': case '-':
				case '{': case '}':	case '(': case ')': case '@': 
                case '\'': case '`':
					dos_char_map[i] = 0xFF;
			}
		}
	}
}


/*** Text translation ***/

/*
* IsTextFile 
* Rules:
*         1) No char values over 127
*         2) No lines over 256
*         3) Assume TRUE after 64K scan of text
*         4) Windows newline is: 0x0D 0x0A (CR/LF) 
*         5) Unix newline is: 0x0A (LF)
*/
int IsTextFile (FileName, pfUnix)
char FileName[];
int *pfUnix;
{
	FILE	*fp;
	int		ch;
	int		chLast=0;
	int 	cnt=0;
	int		linecnt=0;
	int		fIsText = TRUE;

	*pfUnix = TRUE;

	if (!(fp = fopen (FileName, "r")))
		return FALSE;

	while (((ch = fgetc(fp)) != EOF) && (cnt++ < ZANNET_MAX_TEXTCHARS))
	{ 
		/* printf ("ch is %lx, EOF is %lx\n", ch, EOF); */

		if (ch > 127 || linecnt > 256)
		{
			fIsText = FALSE;
			break;
		}

		if (chLast == 0x0D)
		{
			if (ch == 0x0A)
				*pfUnix = FALSE;
		}

		chLast = ch;

		if (ch == 0x0A)
			linecnt = 0;
	}

	fclose (fp);

	!fDebug ? 0 : printf ("IsTextFile: fUnix is %d, rc = %d\n", *pfUnix, fIsText);
	
	return (fIsText);
}


/*
*  TranslateTextFile 
*/
int TranslateTextFile (FileName, fToUnix)
char FileName[];
int fToUnix;
{
	char MangledName[MAX_PATH];
	struct stat	fStat;
	int cntTry=0;
	int ch;
	FILE *fpsrc; 
	FILE *fptrg;
	int err=0;
	int fUnix;

	!fDebug ? 0 : printf ("TranslateTextFile: FileName is %s, fToUnix = %d\n", FileName, fToUnix);

	/* Verify caller. Performance hit is minor because it's interactive */

	if (!IsTextFile (FileName, &fUnix))
		return EINVAL; /* not text file */

	if (fToUnix && fUnix)
		return EINVAL; /* already is Unix format */

	if (!fToUnix && !fUnix)
		return EINVAL; /* already is Windows format */

	/* Make a tmp. file name */

	if (strlen(FileName)+3 > MAX_PATH)
		return ENOENT;

	while (cntTry++ < 20)
	{
		sprintf (MangledName, "%s.%2.2X", FileName, cntTry);

		if (stat(MangledName, &fStat))
		{
			if (errno == ENOENT)
				break;
		}
	}

	if (cntTry >= 20)
		return EEXIST;

	/* Rename the file */

	if (rename (FileName, MangledName))
		return errno;

	/* Open the original file name for writing */

	if (!(fptrg = fopen (FileName, "w")))
		return errno;

	if (!(fpsrc = fopen (MangledName, "r")))
		return errno;

	/* Translate */

	while (((ch = fgetc(fpsrc)) != EOF) && !err)
	{
		if (fToUnix) /* make unix text file */
		{
			if (ch != 0x000D)
				if (fputc(ch, fptrg) != ch)
					err = EBADF;
		}
		else /* make windows text file */
		{
			if (ch == 0x000A)
			{
				if (fputc(0x000D, fptrg) != 0x000D)
					err = EBADF;

				if (!err)
				{
					if (fputc(0x000A, fptrg) != 0x000A)
						err = EBADF;
				}
			}
			else if (fputc(ch, fptrg) != ch)
				err = EBADF;
		}
	}

	fclose (fptrg);
	fclose (fpsrc);

	/* Delete the temp. file */

	unlink (MangledName);

	return err;
}

#ifndef ZANNET_NO_MULTIUSER

/*** MultiUserMode mode support ***/

/*
*  MultiUserMode 
*/
int MultiUserMode (pip, pport)
char *pip;
int *pport;
{
	struct sockaddr_in client_addr;
	int addr_len;
	char tmpbuff[256];
	char password[128];
	int addrlen;
	int sock;
	struct hostent *hp;
#ifdef IPTOS_LOWDELAY
    int tos;
#endif

	if (getuid() != 0)
	{
		printf ("Only inetd as a root process can use this option.\n");
		exit(1);
	}

	/* Open the system log file */
		
#ifdef LOG_DAEMON
	openlog("ZanNetD", LOG_PID | LOG_NDELAY, LOG_DAEMON);
#else
	openlog("ZanNetD", LOG_PID | LOG_NDELAY, LOG_LOCAL5);
#endif
	
#ifdef ZANNET_DAEMON_MODE
	
	if (fOptDaemonMode || fOptDebugMode)
		sock = DaemonMode ();
	else

#endif

	sock = 0;

#ifdef IPTOS_LOWDELAY
    tos = IPTOS_LOWDELAY;
    if (setsockopt (sock, IPPROTO_IP, IP_TOS, (char *)&tos, sizeof(int)) < 0)
          syslog(LOG_WARNING, "setsockopt (IP_TOS): %m");
#endif

	/* Now we are a child whether from inetd or daemon mode.
	   File descriptor 0 (standard input) is our socket */

	/* Get the remote peer's IP address */

	addrlen = sizeof(peer_addr);

	if (getpeername (sock, (struct sockaddr *)&peer_addr, &addrlen) < 0)
	{
		printf ("Only inetd can use this option.\n");
		syslog(LOG_ERR, "getpeername failed: %m");
		exit(1);
	}

	/* Get the remote system name (if possible) */

	hp = gethostbyaddr((char *)&peer_addr.sin_addr, sizeof(struct in_addr), AF_INET);

	if (hp)
		(void) strncpy(RemoteHost, hp->h_name, sizeof(RemoteHost)-1);
	else
		(void) strncpy(RemoteHost, inet_ntoa(peer_addr.sin_addr), sizeof(RemoteHost)-1);

	RemoteHost[sizeof(RemoteHost)-1] = '\0';

	/* Get the remote IP address string */

	(void) strncpy(RemoteAddr, inet_ntoa(peer_addr.sin_addr), sizeof(RemoteAddr)-1);

	RemoteAddr[sizeof(RemoteAddr)-1] = '\0';

	addr_len = sizeof(client_addr);

	/* Get the local IP address */

	if (getsockname (sock, (struct sockaddr *)&client_addr, &addr_len))
	{
		syslog (LOG_ERR, "getsockname failed: %m");
		exit (1);
	}

	/* Start 60 second timer for login */
	(void) alarm (60);

	/* Write: "ZanNet daemon login: " */
	writen (sock, ZANNET_LOGIN_STRING, sizeof(ZANNET_LOGIN_STRING)); 

	/* Get the user name */
	GetLoginString (sock, user, sizeof(user));

	/* Echo the user name */
	writen (sock, user, strlen(user));
	writen (sock, "\n", 1); 

	/* Write: "Password: " */
	writen (sock, ZANNET_LOGIN_PASSWORD, sizeof(ZANNET_LOGIN_PASSWORD)); 

	/* Get the password */
	GetLoginString (sock, password, sizeof(password));

	/* Echo *'s for the password */
	strncpy (tmpbuff, "*************************************************", strlen(password));
	tmpbuff[strlen(password)] = 0;
	strcat (tmpbuff, "\n");
	writen (sock, tmpbuff, strlen(tmpbuff)); 

	/* Authenticate the user */
	AuthenticateUser (sock, user, password);

	/* End 60 second timer for login */
	(void) alarm (0);
	
	/* Write: "Welcome to the ZanNet server >" */
	writen (sock, ZANNET_LOGIN_WELCOME, sizeof(ZANNET_LOGIN_WELCOME));
	writen (sock, "\n", 1); 

	/* Get the Windows IP address and port: "IPADDRESS PORT" */
	GetLoginString (sock, tmpbuff, sizeof(tmpbuff));
	sscanf (tmpbuff, "%s %d", pip, pport);

	/* Echo string */
	writen (sock, tmpbuff, strlen(tmpbuff));
	writen (sock, "\n", 1); 

	return sock;
}


/*
*  GetLoginString -- Read up to the first new file char.
*/
int GetLoginString (s, buffer, len)
int s;
char buffer[];
int len;
{
	int cnt=0, rc;

	do
	{
		rc = read (s, &buffer[cnt], 1);

		if ((rc < 0) || !rc)
		{
			syslog (LOG_ERR, "read error: %m");
			ShutDown (1);
		} 
		if (buffer[cnt] == '\n')
		{
			buffer[cnt] = 0;
			return cnt+1;
		}
		cnt++;

	} while (cnt < len);

	syslog (LOG_ERR, "internal buffer overflow");
	ShutDown (1);

	return -1; /* prevents compiler warning */
}


/*
 * Check if a user is in the file "fname"
 */
int checkuser(fname, name)
char *fname;
char *name;
{
	FILE *fd;
	int found = 0;
	char *p, line[BUFSIZ];

	if ((fd = fopen(fname, "r")) != NULL) 
	{
		while (fgets(line, sizeof(line), fd) != NULL)
		{
			if ((p = strchr(line, '\n')) != NULL) 
			{
				*p = '\0';
				if (line[0] == '#')
					continue;
				if (strcmp(line, name) == 0) 
				{
					found = 1;
					break;
				}
			}
		}
		fclose(fd);
	}
	return (found);
}


/*
*  AuthenticateUser -- See if this is a valid user
*/
void AuthenticateUser (s, user, password)
int s;
char user[];
char password[];
{
	struct passwd *pw;
	char buff[BUFSIZ];

	/* Any users in _PATH_FTPUSERS are denied */

	if (checkuser (_PATH_FTPUSERS, user))
	{
		sprintf (buff, "User %s access denied.\n", user);
		writen (s, buff, strlen(buff)); 
		syslog (LOG_NOTICE, "LOGIN REFUSED from %s [%s], %s", RemoteHost, RemoteAddr, user);
		ShutDown (1);
	}

	/* Get this user's password entry */

	pw = getpwnam (user);

	if (pw == NULL)
	{
		sprintf (buff, "User %s unknown.\n", user);
		writen (s, buff, strlen(buff)); 
		ShutDown (1);
	}

	/* User's "ftp" and "anonymous" don't require passwords */

	if (strcmp(user, "ftp") == 0 || strcmp(user, "anonymous") == 0)
	{
	}
	else
	{
		if (fOptGuestOnly)
		{
			sprintf (buff, "Sorry, only anonymous ZanNet allowed.\n");
			writen (s, buff, strlen(buff));
			syslog (LOG_NOTICE, "NON-ANONYMOUS LOGIN REFUSED from %s [%s], %s", RemoteHost, RemoteAddr, user);
			ShutDown (1);
		}

		/* Validate the entered password */

		if (*pw->pw_passwd == '\0' || strcmp((char *)crypt(password, pw->pw_passwd), pw->pw_passwd))
		{
			sprintf (buff, "Login incorrect from %s.\n", user);
			writen (s, buff, strlen(buff));
			syslog (LOG_NOTICE, "LOGIN INCORRECT from %s [%s], %s", RemoteHost, RemoteAddr, user);
			ShutDown (1);
		}
	}

	/* New user must be okay. First, set the user's home directory */

	if (chdir (pw->pw_dir) < 0)
	{
		perror ("Can't change to home directory");
		syslog (LOG_ERR, "Can't set home directory for user: %s", user);
		ShutDown (1);
	}

	sprintf (HomePathEnv, "HOME=%s", pw->pw_dir);
	putenv(HomePathEnv);
	
	/* Set this user's group ID */

	if (setgid (pw->pw_gid) < 0)
	{
		perror ("Can't set group ID");
		syslog (LOG_ERR, "Can't set group ID for user: %s", user);
		ShutDown (1);
	}

	/* Set new user ID */

	if (setuid (pw->pw_uid) < 0)
	{
		perror ("Can't set user ID");
		syslog (LOG_ERR, "Can't set user ID for user: %s", user);
		ShutDown (1);
	}

	/* Double check--tighten up security */

	if (getuid() != pw->pw_uid)
	{
		perror ("Can't set user ID");
		syslog (LOG_ERR, "User ID check for user: %s failed", user);
		ShutDown (1);
	}

	/* We can relax now. This process can never get back to super user */

	syslog (LOG_NOTICE, "ZANNET LOGIN from %s [%s], %s", RemoteHost, RemoteAddr, user);
}

#else
/*
*  MultiUserMode -- Not compiled
*/
int MultiUserMode (pip, pport)
char *pip;
int *pport;
{
	printf ("This ZanNet server is not compiled for multi-user mode.\n");
	exit (1);

	return 0;
}
#endif


#ifdef ZANNET_DAEMON_MODE

/*
*  DaemonMode 
*/
int DaemonMode ()
{
	struct servent * pse;
	int sock_listen;
	int on = 1;
	int addr_len;
	struct sockaddr_in serv_addr;
	struct sockaddr_in client_addr;
	struct sigaction act; 
	struct sigaction oact;

#ifdef SA_NOCLDSTOP

	act.sa_handler = SIG_DFL;
	act.sa_flags = SA_NOCLDSTOP;

	sigaction (SIGCHLD, &act, &oact);

#endif

#ifdef SA_NOCLDWAIT

	act.sa_handler = SIG_DFL;
	act.sa_flags = SA_NOCLDWAIT;

	sigaction (SIGCHLD, &act, &oact);

#endif

	/* Create a new process for daemon mode only */

#ifdef ZANNET_DAEMON_MODE
	if (fOptDaemonMode && !fOptDebugMode)
		DetachProcess ();
#endif

	/* Get the port for ZanNet */

	errno = 0;

	pse = getservbyname ("zannet", "tcp");

	if (pse == NULL)
	{
		syslog(LOG_ERR, "getservbyname() failed: %m");
		exit (1);
	}

	/* Create a socket for listening */

	sock_listen = socket (AF_INET, SOCK_STREAM, 0);

	if (sock_listen < 0)
	{
		syslog(LOG_ERR, "socket() failed: %m");
		exit (1);
	}

	/* Re-use the port */

	if (setsockopt (sock_listen, SOL_SOCKET, SO_REUSEADDR, (char *)&on, sizeof(on)) < 0)
	{
		syslog(LOG_ERR, "setsockopt() SO_REUSEADDR failed: %m");
		exit (1);
	}

	/* Bind socket to that port */

	memset (&serv_addr, 0, sizeof(serv_addr));

	serv_addr.sin_family = AF_INET;
	serv_addr.sin_addr.s_addr = htonl(INADDR_ANY);
	serv_addr.sin_port = pse->s_port; 

	if (bind(sock_listen, (struct sockaddr *)&serv_addr, sizeof (serv_addr)) < 0)
	{
		syslog(LOG_ERR, "bind() failed: %m");
		perror ("bind failed--must be root!");
		exit (1);
	}

	/* Listen for connections */

	if (listen (sock_listen, 32))
	{
		syslog(LOG_ERR, "listen() failed: %m");
		exit (1);
	}

	/* Loop while accepting connections and forking children */

	for (;;)
	{	
		addr_len = sizeof(client_addr);

		gdaemonsocket = accept (sock_listen, (struct sockaddr *)&client_addr, &addr_len);

		if (gdaemonsocket == -1)
		{
			syslog(LOG_ERR, "accept() failed: %m");
			exit (1);
		}

		if (fork () == 0)
		{
			/* child */
			if (!fOptDebugMode)
			{
				dup2 (gdaemonsocket, 0);
				dup2 (gdaemonsocket, 1);
				dup2 (gdaemonsocket, 2);
			}
			close (sock_listen);
			break;
		}
		close (gdaemonsocket);
	}
	return gdaemonsocket;
}


/*
*  DetachProcess 
*/
int DetachProcess ()
{
	struct sigaction act; 
	struct sigaction oact;
	int fd;

	/* Ignore the terminal stop signals */

	act.sa_handler = SIG_IGN;
	act.sa_flags = 0;

	sigaction (SIGTTOU, &act, &oact);
	sigaction (SIGTTIN, &act, &oact);
	sigaction (SIGTSTP, &act, &oact);

	switch (fork()) 
	{
		case -1:
			syslog(LOG_ERR, "DetachProcess fork() error: %m");
			exit (1); 
		case 0:
			break; /* child */
		default:
			exit (0); /* parent exits to detach child */
	}
	
	/* Create a session and set the process group ID */

	if (setsid() == -1)
	{
		syslog(LOG_ERR, "DetachProcess setsid() error: %m");
		exit(1);
	}

	/* Close all the open files */

	for (fd = 0; fd < 20; fd++)
		close (fd);

	errno = 0;

	/* Open the NULL device and point stdin, stdout, and stderr to it */

	fd = open ("/dev/null", O_RDWR, 0);

	if (fd != -1) 
	{
		dup2 (fd, STDIN_FILENO);
		dup2 (fd, STDOUT_FILENO);
		dup2 (fd, STDERR_FILENO);
		if (fd > 2) 
			close (fd);
	}

	/* Change to root directory so sys. admins can unmount drives */

	chdir ("/");

	/* Clear the file mask */

	umask (0);
}

#endif


/* CONSIDER: Man page */

/*
*  Usage 
*/
void Usage ()
{
	char	hostName[MAX_PATH];
	struct	hostent *pHostEntry;
	struct	in_addr *pHostAddr;

	/* printf ("\nUsage: %s [-m] [-g] [-d] [-b] [-r]\n\n", ProgramName); */
	
	printf ("\nUsage: %s [-m] [-g] [-r]\n\n", ProgramName);

	printf ("Runs the ZanNet server as a daemon. This command should be\n");
	printf ("executed by a Unix system administrator only. Those using\n"); 
	printf ("ZanNet via Telnet should not directly run this command\n");
	printf ("as it will be started by the ZanNet client.\n\n");

	printf ("Dash options must be seperated.\n\n");
	
	/* CONSIDER: printf ("  -d  -- write basic debug info to the logfile\n"); */
	/* CONSIDER: printf ("  -l  -- write each session to the logfile\n"); */
	/* CONSIDER: printf ("  -i  -- log all new files created\n"); */
	/* UNDONN: printf ("  -o  -- log all existing files opened\n"); */

	printf ("  -m  -- run ZanNet in multiuser mode (required for inetd)\n");
	printf ("  -g  -- guest logins only (users: ftp and anonymous)\n");

#ifdef ZANNET_DAEMON_MODE
	printf ("  -d  -- daemon mode (not recommended, use inetd instead)\n");
	printf ("  -b  -- debug mode (foreground process)\n");
#endif

	printf ("  -r  -- allow user's home directory change\n\n"); 
	
	printf ("Configure ZanNet to run as a system wide server by adding it\n");
	printf ("to both /etc/services and /etc/inetd.conf. Be sure to use port 317\n");
	printf ("as assigned by IANA. ZanNet uses /etc/ftpusers and follows FTP user\n");
	printf ("validation conventions (see the FTPD man page).\n\n");

	printf ("Add the following line to /etc/inetd.conf:\n\n");

	printf ("    zannet stream tcp nowait root /usr/sbin/zanservd zanservd -m\n\n");

	printf ("Add the following line to /etc/services:\n\n");

	printf ("    zannet    317/tcp\n\n");

	printf ("The above example assumes that you have installed the ZanNet\n");
	printf ("binary to the /usr/sbin directory and that you have named the\n");
	printf ("binary: zanservd (yours is now called %s)\n\n", ProgramName);

	/* Print some system information */

	gethostname (hostName, sizeof(hostName));

	printf ("This machine's host name is: %s\n", hostName);

	if ((pHostEntry = gethostbyname (hostName)))
	{
		pHostAddr = (struct in_addr *)pHostEntry->h_addr;

		printf ("This machine's IP address is: %s\n\n", inet_ntoa(*pHostAddr));
	}

	printf ("Thank you for trying ZanNet! (Version %d, Build %d)\n\n",
		ZANNET_VERSION_MAJOR, ZANNET_VERSION_MINOR);

	exit (1);
}

