{D-}
(*

	 This is a demonstration on how to build an archive utility
	 using multiple compression / decompression components from the
   ZipTV component package for the Borland Delphi development language.

   The amount of code contained in this demo is very little when
   realized that this demo actually preforms the same fuctionality
   as 15 separate applications!  It may take a little time to
   understand the concept of ZipTV's object hierarchy when using
   multiple compression / decompression components, but once
   understood it becomes rather simple.

   ************************************************************

   There are no restrictions placed on the use of the code contained
   herein.  We have not attempted to build a full archive utility in
   this demonstration, only to provide the basic shell to prime ideas.

   ************************************************************

   Click on light when RED, to cancel the current operation.
   Click on column headers to sort.

   ************************************************************

   ZipTV COMPONENTS OBJECT HIERARCHY

   The most important feature on which to focus when using multiple
   compression / decompression components in ZipTV is the packages
   object hierarchy.

   *The TZipTV component is the visual front end archive manager.
   TCompBase & TUnBASE (parent classes for the compression /
   decompression components) do the actual work.


   The object heirarchy of ZipTV components is as follows:

   TZipCommon - | - TZipTV
                | - TZipSearch
                | - TZipCheck
                | - TUnSFX
   				  |
                | - TUnBase  - |  ( all decompression components & TZipKey )
                |              |
                |              | - TZipKey
                |              |
                |              | - TUnBh
                |              | - TUnZip
                |				  | - TUnArj
                |				  | - ...etc.
                |
                |
                |- TCompBase - |  ( all compression components )
                |              |
                |              | - TBlakHole
                |              | - TZip
                |              | - TLha
                |              | - ...etc.
                |


   To use all decompression components as a single component, assign
   the decompression components as type TUnBase as demonstrated in
   this program (with the use of the variable 'ExtractComponent').

   When assigning the ArchiveFile property a filename (in any component),
   the ArcType and LengthOfFile properties are automatically defined
   internally.  In the Extract1Click procedure below, we demonstrate how
   to use the ArcType property to type cast the desired decompression
   component as type TUnBase.

   ************************************************************

   HOW THE DEMO WORKS WITH MILTIPLE DECOMPRESSION COMPONENTS

	 The first thing of course is to open an archive (or create a new one).

   The 'Open' menu's OnClick event assigns the ArchiveFile property of
   the ZipTV1 component a filename (Open1Click procedure).  Internally,
   when the ArchiveFile property is assigned, the ArcType property is
   automatically defined as the type of archive the file is (ie..atZip,
   atBh,atArj...etc).

   To extract files from the opened archive the 'Extract' menu's
   'OnClick event' calls to the 'Extract1Click' procedure.  In the
   Extract1Click procedure, we use the ArcType property to determine
   which decompression component to use.  (ie.. if ArcType = atZIP then
   we want to use the UnZip1 component that was dropped on the form).  By
   typecasting the component as type 'TUnBase' (parent object to all
   decompression components), we eliminate continuous repeatitive coding.

   Follow the variable 'ExtractComponent' (type TUnBase) for an
   understanding of typecasting the decompression components.

   ************************************************************

   HOW THE DEMO WORKS WITH MILTIPLE COMPRESSION COMPONENTS

   Compression support works the same as decompression described
   above.  The base class for all compression components is TCompBase.

   In this demo we use the variable 'CompressComponent' (type TCompBase)
   to assign the proper compression component with the type of archive
   either in currently in use or the archive type we wish to create.

   When adding (or deleting) files to an already opened archive, we use
   the ZipTV's ArcType property to determine the desired compression
   component to assign to variable CompressComponent (type TCompBase).

   When creating a new archive, form 'frmAddFiles' contains an option to
   select which archive type to create.  Clicking on one of these option
   buttons, we typecast 'CompressComponent' as the proper compression
   component to use.

   ************************************************************

   NOTE: A ListView control is much slower than most other list controls.
   Due to the wide functionality of the ListView control, it was used
   in this demo to reduce required coding keeping the focal point on the
   usage of the ZipTV components themselves.

   Speed can be dramatically increased with the use of other list
   controls, however many features provided by ListView will require
   substituted coding to compare to the functionality of this demo
   program.

   ************************************************************

*)
UNIT Main;



INTERFACE


USES
	Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms,
  Dialogs,	Menus, ComCtrls, ExtCtrls, Gauges, FileCtrl,	StdCtrls,
  AwkTypes, AwkMain, AwkGbls, Err_Msgs, UnBH, UnARJ, UnARC, UnZIP,
  ZipTV, UnZoo, UnTar, UnLHA, UnGZIP, UnCAB, BlakHole, Zipper, LHA,
  UnRAR, ZipCheck, MakeCAB;


TYPE
	TfrmMain = CLASS( TForm )
		OpenDialog1		: TOpenDialog;

		MainMenu1		: TMainMenu;
		File1				: TMenuItem;
		Open1				: TMenuItem;
    	New1				: TMenuItem;
		Close1			: TMenuItem;
    	N1					: TMenuItem;
    	FileSpec1		: TMenuItem;
    	N2					: TMenuItem;
		Exit1				: TMenuItem;

     Action1			: TMenuItem;
     AddFiles1		: TMenuItem;
     DeleteSelected1: TMenuItem;
     Extract1			: TMenuItem;
    	N3					: TMenuItem;
    	Refresh1			: TMenuItem;

    	View1				: TMenuItem;
    	Report2			: TMenuItem;
    	List2				: TMenuItem;
    	SmallIcons2		: TMenuItem;
    	LargeIcons2		: TMenuItem;
    	N4					: TMenuItem;
    	BlankDirs1		: TMenuItem;
    	N5					: TMenuItem;
    	Sort1				: TMenuItem;

		pnlStatus		: TPanel;
		pbxLed			: TPaintBox;
		imgLed			: TImage;
		ListView1		: TListView;
		gauProgress		: TGauge;
		StatusBar1		: TStatusBar;

     (* Frontend to de/compression components *)
     ZipTV1			: TZipTV;

     (* Decompression components *)
     UnARC1			: TUnARC;
     UnARJ1			: TUnARJ;
     UnBH1				: TUnBH;
    	UnCAB1			: TUnCAB;
     UnGZIP1			: TUnGZIP;
    	UnLHA1			: TUnLHA;
     UnTAR1			: TUnTAR;
    	UnRAR1			: TUnRAR;
    	UnZIP1			: TUnZIP;
     UnZOO1			: TUnZOO;

     (* Compression Components *)
    	BlakHole1		: TBlakHole;
    	Zip1				: TZip;
    	Lha1				: TLha;
    MakeCAB1: TMakeCAB;


		PROCEDURE EnableMenus;
		PROCEDURE Open1Click( Sender: TObject );
		PROCEDURE Exit1Click( Sender: TObject );
		PROCEDURE Close1Click( Sender: TObject );
		FUNCTION  DefineCompressComponent: BOOLEAN;
     PROCEDURE ApplicationBusy;
     PROCEDURE ApplicationWaiting;

		PROCEDURE pbxLedPaint( Sender: TObject );
		PROCEDURE SetLedColor( lColor : TColor );

    	PROCEDURE FileSpec1Click(Sender: TObject);
		PROCEDURE FormCreate( Sender: TObject );
		PROCEDURE FormDestroy( Sender: TObject );
		PROCEDURE AddFiles1Click( Sender: TObject );
		PROCEDURE DeleteSelected1Click( Sender: TObject );
		PROCEDURE pbxLedClick(Sender: TObject);
		PROCEDURE DisplayTotals( Which: BYTE );
		PROCEDURE ListView1Click(Sender: TObject);
    	PROCEDURE StatusBar1Click(Sender: TObject);

    	PROCEDURE UnZIP1Progress(Sender: TObject; FN: STRING; Progress: BYTE);
    	PROCEDURE UnZIP1FileExists(Sender: TObject; FN: STRING; FileDate: TDateTime; VAR Accept: BOOLEAN);
    	PROCEDURE UnZIP1Begin(Sender: TObject; FN: STRING; RecNum: INTEGER; VAR Extract: BOOLEAN);
    	PROCEDURE UnZIP1End(Sender: TObject; FN: STRING; CRC_PASS: BOOLEAN);
    	PROCEDURE UnZIP1GetPassword(Sender: TObject; FN: STRING; VAR Password: STRING;
     		VAR TryAgain: BOOLEAN);

    	PROCEDURE Zip1Activate(Sender: TObject);
    	PROCEDURE Zip1Begin(Sender: TObject; FN: STRING; RecNum: INTEGER; VAR Extract: Boolean);
    	PROCEDURE Zip1Deactivate(Sender: TObject);
    	PROCEDURE Zip1End(Sender: TObject; FN: STRING; CRC_PASS: BOOLEAN);
    	PROCEDURE ZipTV1Error(Sender: TObject; FN, VolumeID: STRING; ECode: Integer);
    	PROCEDURE ZipTV1Read(Sender: TObject; Offset, Filenum: Integer);
    	PROCEDURE Zip1RecurseDir(Sender: TObject; Directory: STRING);
    	PROCEDURE Zip1ReplaceFile(Sender: TObject; FN: STRING; OldDate,
      		NewDate: TDateTime; OldFileSize, NewFileSize, OldAttr,
      		NewAttr: INTEGER; VAR Replace: BOOLEAN);
    	PROCEDURE ZipTV1NextVolume(Sender: TObject; VAR Dir, FN: STRING;
      		VolumeID: STRING; VAR Cancel: BOOLEAN);

    	PROCEDURE View1Click(Sender: TObject);
    	PROCEDURE Refresh1Click(Sender: TObject);
    	PROCEDURE Extract1Click(Sender: TObject);

     PROCEDURE ListView1ColumnClick(Sender: TObject; Column: TListColumn);
     PROCEDURE ListView1Compare(Sender: TObject; Item1, Item2: TListItem;
       		Data: INTEGER; VAR Compare: INTEGER);
    	PROCEDURE Sort1Click(Sender: TObject);
    	PROCEDURE BlankDirs1Click(Sender: TObject);
	PRIVATE
		FSortColumn: INTEGER;
		FSortForward: BOOLEAN;
	PUBLIC
	END;




VAR
	frmMain : TfrmMain;
	CompressComponent: TCompBase;		(* Base class for compression components *)
	ExtractComponent: TUnBase;			(* Base class for decompression components *)




IMPLEMENTATION



USES
	Unit1, Unit2, Unit3;

VAR
  ConfirmOverwrites: TModalResult;
  HoldDateSeparator: CHAR;


CONST
	TOTALS 	= 0;
	SELECTED = 1;


  (* ListView Column Subitems *)
  SUBITEM_DATETIME  = 0;
  SUBITEM_PACKED	   = 1;
  SUBITEM_UNPACKED  = 2;
  SUBITEM_RATIO 	   = 3;
  SUBITEM_METHOD    = 4;
  SUBITEM_ATTR	   = 5;
  SUBITEM_FOLDER    = 6;
  SUBITEM_FILETYPE  = 7;
  SUBITEM_CRC		   = 8;
  SUBITEM_OFFSET	   = 9;
  SUBITEM_ENCRYPTED = 10;
  SUBITEM_VOLUMENAME= 11;
  SUBITEM_COMMENT   = 12;


VAR
	TotalPackedSize,
	TotalUnpackedSize: LONGINT;


{$R *.DFM}


{-------------------------------------------------------------}
PROCEDURE TfrmMain.ApplicationBusy;
BEGIN
  ListView1.Cursor := crHourGlass;
  SetLedColor( clRed );
	ListView1.Items.BeginUpdate;
END;
{-------------------------------------------------------------}
PROCEDURE TfrmMain.ApplicationWaiting;
BEGIN
	ListView1.Items.EndUpdate;
  SetLedColor( clGreen );
  ListView1.Cursor := crDefault;
END;
{-------------------------------------------------------------}
(* Enable / Disable menu items *)
PROCEDURE TfrmMain.EnableMenus;
VAR
	CompressSupported,
  DecompressSupported: BOOLEAN;
BEGIN
  Action1.Visible := TRUE;

  CompressSupported   := ZipTV1.IsArcCompressable( ZipTV1.ArcType );
  DecompressSupported := ZipTV1.IsArcDecompressable( ZipTV1.ArcType );

  (* Currently can't add files to a CAB archive... only create new *)
  IF CompressSupported THEN
  	IF ZipTV1.ArcType = atCAB THEN
     	CompressSupported := False;

  AddFiles1.Enabled 		:= CompressSupported;
  DeleteSelected1.Enabled := CompressSupported;
  Extract1.Enabled 			:= DecompressSupported;
END;
{-------------------------------------------------------------}
(* Menu's Open & New share this event through their OnClick events *)
PROCEDURE TfrmMain.Open1Click( Sender: TObject );
CONST
	mOpen = 0;
  mNew  = 1;
BEGIN

                              (* load from Err_Msgs.Pas *)
  OpenDialog1.Filter := LoadStr( F_TZIPTV );

  CASE TMenuItem( Sender ).MenuIndex OF

  	mOpen:						(* Menu item Open *)
     	BEGIN
           OpenDialog1.Title := 'Open Archive...';
           OpenDialog1.Options := [ofHideReadOnly, ofFileMustExist, ofPathMustExist];
  		END;

     mNew:							(* Menu item New *)
     	BEGIN
           OpenDialog1.Title := 'New Archive...';
           OpenDialog1.Options := [ ofHideReadOnly ];
           OpenDialog1.Filename := '';
        END;

  END;


	IF OpenDialog1.Execute THEN
  BEGIN

     TotalPackedSize := 0;
  	TotalUnpackedSize := 0;

     TRY

     	ApplicationBusy;

        IF FileExists( OpenDialog1.Filename ) THEN
        BEGIN

           ListView1.Items.Clear;
           ZipTV1.ArchiveFile := OpenDialog1.Filename;
           ZipTV1.Activate;

        END ELSE BEGIN

           CompressComponent := NIL;
           IF frmAddFiles.ShowModal = mrCancel THEN
              EXIT;

           ListView1.Items.Clear;

           DisplayTotals( SELECTED );

           CompressComponent.ArchiveFile := OpenDialog1.Filename;
           CompressComponent.Activate;

                                (* Assign the newly created archive to *)
                                (*	ZipTV's ArchiveFile property for    *)
                                (* visual display                      *)
           ZipTV1.ArchiveFile := OpenDialog1.Filename;

        END;

        EnableMenus;
        Close1.Enabled := TRUE;
        DisplayTotals( TOTALS );
        Caption := OpenDialog1.Filename;

     FINALLY
     	ApplicationWaiting;
     END;

 	END;

END;
{-------------------------------------------------------------}
(* Exit menu item *)
PROCEDURE TfrmMain.Exit1Click( Sender: TObject );
BEGIN
	Close;
END;
{-------------------------------------------------------------}
(* Close menu item *)
PROCEDURE TfrmMain.Close1Click( Sender: TObject );
BEGIN

	ZipTV1.ArchiveFile := '';

  ApplicationBusy;
	ListView1.Items.Clear;
  ApplicationWaiting;

  Caption := '';
  StatusBar1.SimplePanel := TRUE;

  Action1.Visible := FALSE;
  Close1.Enabled := FALSE;
  									(* Set the ArcType property to atNA by *)
  									(*	blanking the ArchiveFile property   *)
  CompressComponent.ArchiveFile := '';

END;
{-------------------------------------------------------------}
(* Change colors of little light in corner of form *)
PROCEDURE TfrmMain.SetLedColor( lColor : TColor );
BEGIN

	WITH imgLed.Canvas DO
	BEGIN

		Brush.Color := lColor;
		FloodFill( 6, 6, Pixels[ 6, 6 ], fsSurface );

	END;

	pbxLed.Repaint;

END;
{-------------------------------------------------------------}
(* Little light in right corner of form *)
PROCEDURE TfrmMain.pbxLedPaint( Sender: TObject );
BEGIN

	WITH Sender AS TPaintBox DO
	 Canvas.Draw( ( Width - imgLed.Width ) DIV 2,
					( Height - imgLed.Height ) DIV 2,
              imgLed.Picture.Graphic );

END;
{-------------------------------------------------------------}
(* Display totals in bottom status bar *)
PROCEDURE TfrmMain.DisplayTotals( Which: BYTE );
VAR
	Ratio	: BYTE;
	i		: WORD;
  Files : INTEGER;
	SelectedPackedSize,
	SelectedUnpackedSize: LONGINT;
BEGIN

	IF ListView1.Selected = NIL THEN Which := TOTALS;

  IF Which = SELECTED THEN
  BEGIN

     Files := ListView1.SelCount;
  	SelectedPackedSize := 0;
     SelectedUnpackedSize := 0;

     FOR i := 0 TO Pred( ListView1.Items.Count ) DO
        IF ListView1.Items[i].Selected THEN
        BEGIN
           Inc( SelectedPackedSize,
           		StrToInt( ListView1.Items[i].SubItems[ SUBITEM_PACKED ] ) );
           Inc( SelectedUnpackedSize,
           		StrToInt( ListView1.Items[i].SubItems[ SUBITEM_UNPACKED ] ) );
        END;

  END ELSE BEGIN					(* Which = TOTALS *)

     Files := ListView1.Items.Count;
     SelectedPackedSize := TotalPackedSize;
     SelectedUnpackedSize := TotalUnpackedSize;

  END;

  Ratio := ZipTV1.CalcRatio( SelectedPackedSize, SelectedUnpackedSize );

  WITH StatusBar1 DO
  BEGIN

  	IF Files > 0 THEN
     	Panels[0].Text := ArcTypeNames[ ZipTV1.ArcType ]
     ELSE
     	Panels[0].Text := '';

     Panels[1].Text := 'Files: '   + IntToStr( Files );
     Panels[2].Text := 'Packed: '  + IntToStr( SelectedPackedSize );
     Panels[3].Text := 'Unpacked: '+ IntToStr( SelectedUnpackedSize );
     Panels[4].Text := 'Ratio: '   + IntToStr( Ratio ) + '%';
  END;

	StatusBar1.SimplePanel := FALSE;
	gauProgress.Visible := FALSE;
  ApplicationWaiting;

END;
{-------------------------------------------------------------}
(* Initialize system icons *)
PROCEDURE TfrmMain.FormCreate( Sender: TObject );
BEGIN

	HoldDateSeparator := DateSeparator;
  DateSeparator 		:= '-';

                             		(* Initialize systems image list *)
	InitializeImageList( Self, ListView1 );

  Action1.Visible := False;
  Close1.Visible := False;

  											(* If demo is called via registry *)
                                   (*	association load that archive  *)
  IF ( ParamCount = 1 ) THEN
  BEGIN
  	ZipTV1.ArchiveFile := ParamStr( 1 );
     ZipTV1.Activate;
     DisplayTotals( TOTALS );
		EnableMenus;
  END ELSE
  											(* Initialize CompressComponent with 	*)
                                	(*	BlakHole compression as default 		*)
  	CompressComponent := BlakHole1;

                             		(* Menu item - Show Empty dirs *)
  BlankDirs1Click( Sender );

END;
{-------------------------------------------------------------}
(* Destroy the system image list *)
PROCEDURE TfrmMain.FormDestroy( Sender: TObject );
BEGIN
	DestroyImageList( ListView1 );
  DateSeparator := HoldDateSeparator;
END;
{-------------------------------------------------------------}
FUNCTION TfrmMain.DefineCompressComponent: BOOLEAN;
BEGIN

	CompressComponent := NIL;

  WITH ZipTV1 DO
     IF NOT IsArcCompressable( ArcType ) THEN

        ShowMessage( 'Compression not supported for this archive' )

     ELSE

        CASE ArcType OF

           atBH		: CompressComponent := BlakHole1;
           
           atCAB		: CompressComponent := MakeCab1;

           atZip,
           atZip250	: CompressComponent := Zip1;

           atLha,
           atLzh		: CompressComponent := Lha1;

        END;

	RESULT := NOT ( CompressComponent = NIL );

END;
{-------------------------------------------------------------}
(* AddFiles menu item *)
PROCEDURE TfrmMain.AddFiles1Click( Sender: TObject );
BEGIN

	IF ZipTV1.ArchiveFile <> '' THEN
	BEGIN

     IF NOT DefineCompressComponent THEN
        EXIT;

     ListView1.Cursor := crHourGlass;

     CompressComponent.ArchiveFile := ZipTV1.ArchiveFile;
     IF frmAddFiles.ShowModal = mrCancel THEN EXIT;

     IF CompressComponent.FileSpec.Count > 0 THEN
        CompressComponent.Activate;

     ListView1.Cursor := crDefault;

	END ELSE

		ShowMessage( 'Open archive before adding files' );

END;
{-------------------------------------------------------------}
(* Delete selected menu item *)
PROCEDURE TfrmMain.DeleteSelected1Click( Sender: TObject );
VAR
	i: WORD;
BEGIN

	IF ( ListView1.Selected <> NIL ) AND ( ZipTV1.ArchiveFile <> '' ) THEN
	BEGIN

     IF MessageDlg( 'Delete selected file(s)?',
                       mtConfirmation,
                       [ mbYes, mbNo ],
                       0 ) = mrNO THEN
        Exit;

     (* DefineCompressComponent is a function defined in this module *)
		IF NOT DefineCompressComponent THEN
     	EXIT;

		ListView1.Cursor := crHourGlass;

		CompressComponent.ArchiveFile := ZipTV1.ArchiveFile;

		TotalPackedSize := 0;
		TotalUnpackedSize := 0;

		CompressComponent.FileSpec.Clear;
		//CompressComponent.ExcludeSpec.Clear;

     (* Add all selected rows in ListView to TBlakHole's filespec property *)
     FOR i := 0 to Pred( ListView1.Items.Count ) DO
			IF ListView1.Items[i].Selected THEN
           CompressComponent.FileSpec.Add( ListView1.Items[i].Caption );


     (* IF we have any filespecs, process file deletion *)
		CompressComponent.Switch := swDelete;
     ApplicationBusy;
     IF CompressComponent.FileSpec.Count > 0 THEN
        CompressComponent.Activate;

     ListView1.Items.Clear;
     ZipTV1.Activate;		(* Redisplay archive *)
     ApplicationWaiting;

     DisplayTotals( TOTALS );

		ListView1.Cursor := crDefault;

	END ELSE

  	ShowMessage( 'No files selected to delete' );
     
END;
{-------------------------------------------------------------}
(* TPaintBox ( pbxLed ) component OnClick event -
   Set the Cancel property to true *)
PROCEDURE TfrmMain.pbxLedClick(Sender: TObject);
BEGIN
	StatusBar1.SimpleText := 'Aborting...';
	IF CompressComponent <> NIL THEN	CompressComponent.Cancel := TRUE;
	IF ExtractComponent <> NIL THEN ExtractComponent.Cancel := TRUE;
END;
{-------------------------------------------------------------}
(* ListView's OnClick event -
	 Display totals in the status bar for the current select row *)
PROCEDURE TfrmMain.ListView1Click(Sender: TObject);
BEGIN
	DisplayTotals( SELECTED );
END;
{-------------------------------------------------------------}
(* StatusBar's OnClick event -
   Display totals in the status bar for all files in list *)
PROCEDURE TfrmMain.StatusBar1Click(Sender: TObject);
BEGIN
	DisplayTotals( TOTALS );
END;
{-------------------------------------------------------------}
(* OnRead Event

	 Shared by ZipTV1 and all compression components.

   * A ListView control is much slower to fill than other list
   	controls but more flexable to work with.  Speed can be
     increased with the reduction of subitems added.
   * ZipTV1 fills ListView with the compressed file info
   * The compression components access this OnRead event with
     compressed file info for each file 'added' to an archive.

*)
PROCEDURE TfrmMain.ZipTV1Read(Sender: TObject; Offset, Filenum: INTEGER);
VAR
	NewItem 	: TListItem;
	ZipCommon: TZipCommon;
BEGIN

  (* ZipCommon is the parent class to all ZipTV components.			*)
  (* Typecast the 'Sender' parameter as type TZipCommon to share		*)
  (* properties with all components.											*)
	ZipCommon := TZipCommon( Sender );

	NewItem := ListView1.Items.Add;                             (* Create a new row *)

	Inc( TotalPackedSize, ZipCommon.PackedSize );
	Inc( TotalUnpackedSize, ZipCommon.UnpackedSize );

	WITH NewItem DO
	BEGIN

		Data := POINTER( Offset );											(* For future use 		*)

										(* Add associated image to list.  First initialize 		*)
                             (* the list using	InitializeListImages. 					 	*)
		ImageIndex := ZipCommon.ImageIndex;


		Caption := ExtractFilename( ZipCommon.Filename );			(* Filename property		*)
		WITH SubItems DO
		BEGIN

			Add( FormatDateTime( 'mm' + DateSeparator +
        							'dd' + DateSeparator +
                             'yy hh:mm', ZipCommon.Date ) );	(* Date property 			*)

			Add( IntToStr( ZipCommon.PackedSize ) );					(* PackedSize property 	*)
			Add( IntToStr( ZipCommon.UnpackedSize ) );				(* UnpackedSize property*)
			Add( IntToStr( ZipCommon.Ratio ) + '%' );					(* Ratio						*)
			Add( ZipCommon.sCompressionMethod );						(* Method - string		*)
			//or
			//Add( IntToStr( ZipCommon.wCompressionMethod ) );		(* Method - word			*)
			Add( IntToStr( ZipCommon.ExternalAttr ) );				(* ExternalAttr property*)
			Add( AppendDirTail( ExtractFileDir( ZipCommon.Filename ) ) );
			Add( ZipCommon.GetFileType( ZipCommon.Filename ) );	(* Windows associated filetype *)
			Add( IntToHex( ZipCommon.CRC, 8 ) );						(* CRC Property *)

                                                              (* Beginning offset into*)
                                                              (* compressed file		*)
			Add( IntToStr( Offset ) );
        IF ZipCommon.Encrypted THEN									(* Encrypted property	*)
        	Add( 'Yes' )
        ELSE
        	Add( 'No' );
           																	(* VolumeID for Arj		*)
                                                              (* multi-volume archives*)
        IF ZipCommon.ArcType IN [ atArj, atCab ] THEN
        	Add( ZipCommon.VolumeName )
        ELSE
        	Add( '' );

			Add( StrPas( ZipTV1.FileComment ) );						(* FileComment property	*)

		END;
	END;
END;
{-------------------------------------------------------------}
(*
     OnError Event -

		All ZipTV components on dropped on this form share event.

   	See Err_Msgs.PAS & Err_Msgs.RC for error codes
     *)
PROCEDURE TfrmMain.ZipTV1Error(Sender: TObject; FN, VolumeID: STRING; ECode: INTEGER);

VAR
	Msg: STRING;

CONST
	CRLF = #13#13;

BEGIN

	Msg := FN + CRLF;

	CASE ECode OF

		E_BASE..E_WIN32:  			(* All error codes *)
				Msg := Msg + 'Error#: ' + IntToStr( Ecode ) + CRLF +
								 'Error: ' + LoadStr( Ecode ) + CRLF;

		M_BASE..M_PASSWORDFAILED:  (* All message codes *)
				Msg := Msg + 'General Message: ' + IntToStr( Ecode ) + CRLF +
								 'Messge: ' + LoadStr( Ecode ) + CRLF;
	ELSE
		EXIT;		(* <- no msg *)
	END;

	Msg := Msg + 'OnError event...';
  ShowMessage( Msg );

END;
{-------------------------------------------------------------}
(* OnProgress Event -

	-This event is shared by all compression and decompression components.

  -TZipCommon is the parent class for compression and decompression
   components.  TUnBase & TCompBase inherit Cancel from TZipCommon.
   'Sender' is typecast as type TZipCommon to check the value of the
   Cancel property for the current component in use.

  -When Cancel is true, compression / decompression is not interrupted
   until the current file is completed.  Accordingly, the active
   component will trigger their OnProgress events until the process
   on the current file is completed.

*)
PROCEDURE TfrmMain.UnZIP1Progress(Sender: TObject; FN: STRING; Progress: Byte);
BEGIN
	gauProgress.Progress := Progress;
END;
{-------------------------------------------------------------}
(* OnFileExists Event -
   -
   ConfirmOverwrites is a local global of type TModalResult. It
   is used to allow the user additional options to answer 'Yes
   to all' or "No to all' without having to be prompted for
   overwrites every time this event is activated.

   ConfirmOverwrites:
   	mrIgnore	= 'Yes to all'
   	mrCancel	= 'No to all'
     mrNo		= 'No'
     mrYes		= 'Yes'

   The buttons in frmOverwrite coincide with the above ModalResult
   values.
   -
	 All decompression components share this event.
   Triggered prior to extracting a file that was found to
   already exist on disk. *)
PROCEDURE TfrmMain.UnZIP1FileExists(Sender: TObject; FN: STRING; FileDate: TDateTime; VAR Accept: BOOLEAN);
BEGIN
	CASE ConfirmOverwrites OF
  	mrCancel	:            	(* No to all								*)
     	Accept := False;
     mrIgnore : ;  				(* Yes to all 								*)
     mrNo,
     mrYes,
     mrNone:
                             (* Overwrite existing selected in 	*)
                             (* form frmExtract? 						*)
        IF  frmExtract.RadioGroup2.ItemIndex = 1 THEN
        BEGIN
           frmOverwrite.Label1.Caption := FN;
           ConfirmOverwrites := frmOverwrite.ShowModal;

           IF ( ConfirmOverwrites = mrCancel ) OR
                 ( ConfirmOverwrites = mrNO		 ) THEN
              Accept := False;

        END;
  END;	(* case *)
END;
{-------------------------------------------------------------}
(* OnBegin Event -
	All decompression components share this event.
  Triggered prior to extracting individual files. *)
PROCEDURE TfrmMain.UnZIP1Begin(Sender: TObject; FN: STRING; RecNum: INTEGER; VAR Extract: BOOLEAN);
BEGIN
	StatusBar1.SimpleText := 'Extracting: ' + FN;
  Application.ProcessMessages;
END;
{-------------------------------------------------------------}
(* OnEnd Event -
	All decompression components share this event
  Triggered after extracting individual files. *)
PROCEDURE TfrmMain.UnZIP1End(Sender: TObject; FN: STRING; CRC_PASS: BOOLEAN);
BEGIN

  									(* reset visual values *)
	StatusBar1.SimpleText := '';
	gauProgress.Progress := 0;

	IF NOT CRC_PASS THEN
  	ShowMessage( FN + #13'Extract failed'#13#13'OnEnd Event...' );

END;
{-------------------------------------------------------------}
(* OnGetPassword event -
	All compression and decompression components share this event.
  Triggered when an archive requires a password for encryption or
  decryption. *)
PROCEDURE TfrmMain.UnZIP1GetPassword(Sender: TObject; FN: STRING; VAR Password: STRING; VAR TryAgain: BOOLEAN);
BEGIN

  (* InputQuery is a Delphi function *)
	IF NOT InputQuery( 'Encrypted file...', 'Enter password', Password ) THEN
  	TryAgain := False;

END;
{-------------------------------------------------------------}
(* File/FileSpec menu item -
	 Change the ZipTV1's FileSpec property and redisplay results *)
PROCEDURE TfrmMain.FileSpec1Click(Sender: TObject);
VAR
	NewString: STRING;
BEGIN

	NewString := ZipTV1.Filespec;

  (* InputQuery is a Delphi function *)
	IF NOT InputQuery( 'Change FileSpec property...', 'Example: *.*', NewString ) THEN
  	Exit;

  IF NewString <> '' THEN
  BEGIN
     ZipTV1.FileSpec := NewString;
     IF ZipTV1.ArchiveFile <> '' THEN
     BEGIN
        ApplicationBusy;

        ListView1.Items.Clear;
        ZipTV1.ArchiveFile := ZipTV1.ArchiveFile;
        ZipTV1.Activate;

        DisplayTotals( TOTALS );
        ApplicationWaiting;
     END;
  END;
END;
{-------------------------------------------------------------}
(* View/Report, View/List, View/Small Icons, View/Large Icons menus *)
PROCEDURE TfrmMain.View1Click(Sender: TObject);
BEGIN

	WITH ListView1 DO
		CASE TMenuItem( Sender ).MenuIndex OF
			0 : ViewStyle := vsReport;
			1 : ViewStyle := vsList;
			2 : ViewStyle := vsSmallIcon;
			3 : ViewStyle := vsIcon;
		END;

END;
{-------------------------------------------------------------}
(* Action/Refresh menu item *)
PROCEDURE TfrmMain.Refresh1Click(Sender: TObject);
BEGIN
  ApplicationBusy;
  ListView1.Items.Clear;
  ZipTV1.Activate;
  ApplicationWaiting;
END;
{-------------------------------------------------------------}
(* OnActivate event -
	 Shared by all compression components.
   Triggered just prior adding, moving, deleting compressed files *)
PROCEDURE TfrmMain.Zip1Activate(Sender: TObject);
BEGIN
	StatusBar1.SimplePanel := TRUE;
	gauProgress.Visible := TRUE;

  ApplicationBusy;
END;
{-------------------------------------------------------------}
(* OnBegin event -
   Shared by all compression components.
   Triggered just prior compressing individual files. *)
PROCEDURE TfrmMain.Zip1Begin(Sender: TObject; FN: STRING; RecNum: INTEGER;
  VAR Extract: BOOLEAN);
BEGIN
										(* CompBase is the parent class for all	*)
                             (* compression components 						*)
	IF TCompBase(Sender).Switch <> swDelete THEN
		StatusBar1.SimpleText := 'Compressing: ' + FN
	ELSE
		StatusBar1.SimpleText := 'Deleting: ' + ExtractFilename( FN );

										(* Give system a breath 						*)
	Application.ProcessMessages;

END;
{-------------------------------------------------------------}
(* OnDeactivate event -
	 Shared by all compression components.
   Triggered after all files matching the FileSpec property have
   been compressed. *)
PROCEDURE TfrmMain.Zip1Deactivate(Sender: TObject);
BEGIN
	ListView1.Items.EndUpdate;
	DisplayTotals( TOTALS );
	ListView1.Cursor := crDefault;
END;
{-------------------------------------------------------------}
(* OnEnd event -
	 Shared by all compression components.
   Triggered after compressing an individual file to an archive *)
PROCEDURE TfrmMain.Zip1End(Sender: TObject; FN: STRING; CRC_PASS: BOOLEAN);
BEGIN
	StatusBar1.SimpleText := '';
	gauProgress.Progress := 0;
END;
{-------------------------------------------------------------}
(* OnRecurseDir event -
	 Shared by all compression components.
   Triggered when a directory has changed when directory recursion
   was chosen. *)
PROCEDURE TfrmMain.Zip1RecurseDir(Sender: TObject; Directory: STRING);
BEGIN
	StatusBar1.SimpleText := 'Scanning directory: ' + Directory;
	Application.ProcessMessages;
END;
{-------------------------------------------------------------}
(* OnReplaceFile event -
	 Shared by all compression components.
   Triggered when an file was found to already exist in an archive. *)
PROCEDURE TfrmMain.Zip1ReplaceFile(Sender: TObject; FN: STRING; OldDate,
  NewDate: TDateTime; OldFileSize, NewFileSize, OldAttr, NewAttr: INTEGER;
  VAR Replace: BOOLEAN);
VAR
	Msg: STRING;
BEGIN

  (* In this demo, we use a simple dialog box.  It would be preferable to
     display a form with additional reponse buttons which might include
     'Yes to all', 'No to all'.  See example in procedure UnZIP1FileExists
     in this demo. *)

	Msg :=
  			'Replace:' + #13 +
           FN + #13 +
  			'Size: ' + IntToStr( OldFileSize  ) + #13 +
				'Date: ' + FormatDateTime( 'mm' + DateSeparator +
           									'dd' + DateSeparator +
                                      'yy' +
                                      'hh' + TimeSeparator +
                                      'mm', OldDate ) + #13 +
  			'Attr: ' + IntToStr( OldAttr ) + #13#13 +

           'With:' + #13 +
           FN + #13 +
				'Size: ' + IntToStr( NewFileSize  ) + #13 +
				'Date: ' + FormatDateTime( 'mm' + DateSeparator +
           									'dd' + DateSeparator +
                                      'yy' +
                                      'hh' + TimeSeparator +
                                      'mm', NewDate ) + #13 +
  			'Attr: ' + IntToStr( NewAttr );

	IF MessageDlg( Msg, mtConfirmation, [ mbYES, mbNO ] , 0 ) = mrNo THEN
  	Replace := FALSE;

END;
{-------------------------------------------------------------}
(*
   Extract files menu item

  *******************************************************************

   Example of using multiple decompression components to extract
   files for different types of archives.

  *******************************************************************

  -The ZipTV1.ArchiveFile has already been assigned a filename.
  -The ArcType property was assigned internally when ArchiveFile
   was assigned
  -Use the ArcType property to determine which decompression component
   to assign to variable 'ExtractComponent' (type TCompBase... parent
   object to all decompression components).
  -After assigning variable 'ExtractComponent', use its properties /
   methods as if it were the actual decompression component.

  *******************************************************************

  Always remember to use the ELSE statement with CASE ZipTV1.ArcType!
  The TZipTV component supports more archive types than the package
  currently has compression and decompression components.

*)
PROCEDURE TfrmMain.Extract1Click(Sender: TObject);
VAR
	i: WORD;
CONST
	EXTRACT_SELECTED = 0;
	EXTRACT_ALL = 1;
BEGIN

  IF ZipTV1.ArchiveFile <> '' THEN
  BEGIN

     (* Since frmExtract is auto-created, the FileListBox1 in that form 	*)
     (* sets the forms current directory as the current directory at      *)
     (* program load time.  Change it to current prior to activating the  *)
     (* form so the function GetCurrentDir will return the proper dir     *)
     (* instead of the dir which was current at program load time.        *)
  	IF ExtractComponent = NIL THEN
     	frmExtract.DirectoryListBox1.Directory := ExtractFileDir( ZipTV1.ArchiveFile );


     IF frmExtract.ShowModal = mrOK THEN
     BEGIN

        IF ( frmExtract.RadioGroup1.ItemIndex = EXTRACT_SELECTED ) AND
              ( ListView1.SelCount = 0 ) THEN
           EXIT;

        ConfirmOverwrites := mrNone;

                                (* Assign the TUn??? component *)

        CASE ZipTV1.ArcType OF

           atArc,
           atArcExe		: ExtractComponent := UnARC1;

           atArj,
           atArjExe		: ExtractComponent := UnARJ1;

           atBH,
           atBHExe		: ExtractComponent := UnBH1;

				(* Requires UnCAB.DLL in windows/systems directory *)
           atCAB			: ExtractComponent := UnCAB1;

           atGZIP		: ExtractComponent := UnGZIP1;

           atLHA,
           atLhaExe,
           atLZH,
           atLzhExe		: ExtractComponent := UnLHA1;

				(* Requires UnRAR.DLL in windows/systems directory *)
           atRar,
           atRarExe		: ExtractComponent := UnRAR1;

           atTar			: ExtractComponent := UnTAR1;

           atZip,
           atZipExe,
           atZip250,
           atZip250Exe	: BEGIN
                             ExtractComponent := UnZIP1;
                             ExtractComponent.ZipCmntBufSize := ZipTV1.ZipCmntBufSize;
                         END;


           atZoo			: ExtractComponent := UnZOO1;

        ELSE

           BEGIN

              ShowMessage( 'Extract not supported for this filetype.' );
              EXIT;

           END;

        END;

                                (* Assign filename & dir to extract to *)
        ExtractComponent.ArchiveFile 	:= ZipTV1.ArchiveFile;

                                (* Assign the ExtractDir property from the
                                   edit control in unit4.pas *)
        ExtractComponent.ExtractDir 	:= frmExtract.Edit1.Text;

                                (* 'Restore original folders' CheckBox
                                    control in unit4.pas *)
        ExtractComponent.UseStoredDirs:= frmExtract.CheckBox1.Checked;


        (*******************************)
        (* Form display controls setup *)
        (*******************************)
        StatusBar1.SimplePanel := TRUE;
        gauProgress.Visible := TRUE;

        ApplicationBusy;

        TRY

           CASE frmExtract.RadioGroup1.ItemIndex OF

              EXTRACT_ALL:

                 BEGIN

                    ExtractComponent.FileSpec := ZipTV1.FileSpec;
                    ExtractComponent.Extract;

                 END;

              EXTRACT_SELECTED:				(* Extract Selected *)

                 BEGIN

                    IF ZipTV1.ArcType = atCAB THEN
                    BEGIN

                       IF MessageDlg(
                             'If extracting more than a single file... because of CABs continuous ' +
                             'compressed stream method, it is much faster to decompress all ' +
                             'files from the archive and then delete from disk those you ' +
                             'do not wish to keep.  For more info on CAB files, see the ' +
                             'documentation on TUnCAB.'#13#13 + 'Cancel operation?',
                             mtWarning,
                             [mbYES, mbNO],
                             0 ) = mrYES THEN

                          EXIT;

                    END;

                    FOR i := 0 TO Pred( ListView1.Items.Count ) DO
                       IF ListView1.Items[i].Selected THEN
                       BEGIN

                          (* Retreive the full path stored in the archive
                             from the ListView Control *)
                          WITH ListView1.Items[i] DO
                          BEGIN
                             ExtractComponent.FileSpec := AppendDirTail( SubItems[ SUBITEM_FOLDER ] ) +
                                Caption;

                             IF ZipTV1.ArcType IN [ atARJ, atCAB ] THEN
                          		ExtractComponent.VolumeName := SubItems[ SUBITEM_VOLUMENAME ];
                          END;
                          ExtractComponent.Extract;

                       END;

                 END;

           END;

        FINALLY

           (*******************************)
           (* Form display controls reset *)
           (*******************************)
           gauProgress.Visible := FALSE;
           StatusBar1.SimplePanel := FALSE;

           ApplicationWaiting;

        END;
     END;
  END;
END;
{-------------------------------------------------------------}
(* OnNextVolume event -
	 Shared by ZipTV and UnARJ.
   Triggered when 'file not found' with ARJ, Cab, RAR multi-volume
   archives. *)
PROCEDURE TfrmMain.ZipTV1NextVolume(Sender: TObject; VAR Dir, FN: STRING;
  VolumeID: STRING; VAR Cancel: BOOLEAN);
VAR
  Prompt 	,
	Caption	: STRING;
BEGIN
	Caption := 'Enter new drv:\directory...';
  Prompt  := 'Filename: ' + FN;
  IF NOT InputQuery( Caption, Prompt, Dir ) THEN
  	Cancel := True;
END;
{-------------------------------------------------------------}
(* Click the column headers for sorting *)
PROCEDURE TfrmMain.ListView1ColumnClick( Sender: TObject; Column: TListColumn );
BEGIN

	IF Column.Index = FSortColumn THEN
		FSortForward := NOT FSortForward
	ELSE BEGIN
	  FSortColumn := Column.Index;
	  FSortForward := TRUE;
	END;

	ListView1.SortType := stData;
	ListView1.SortType := stNone;

END;
{-------------------------------------------------------------}
(* Sort columns ( not necessary, but nice ) *)
PROCEDURE TfrmMain.ListView1Compare(Sender: TObject; Item1,
  Item2: TListItem; Data: INTEGER; VAR Compare: INTEGER);
VAR
	size1, size2: INTEGER;
	date1, date2: TDateTime;
	Result: INTEGER;
	Str: STRING;

CONST
  (* ListView1 Column Headers *)
	COLUMN_FILENAME  	= 0;
  COLUMN_DATETIME  	= 1;
  COLUMN_PACKED	  	= 2;
  COLUMN_UNPACKED  	= 3;
  COLUMN_RATIO 	  	= 4;
  COLUMN_METHOD   	= 5;
  COLUMN_ATTR	   	= 6;
  COLUMN_FOLDER    	= 7;
  COLUMN_FILETYPE  	= 8;
  COLUMN_CRC		   = 9;
  COLUMN_OFFSET	   = 10;
  COLUMN_ENCRYPTED 	= 11;
  COLUMN_VOLUMENAME	= 12;
  COLUMN_COMMENT   	= 13;

BEGIN

	TRY

		CASE FSortColumn OF

                              (* Filename Column *)
			COLUMN_FILENAME:
        	Result := CompareText ( Item1.Caption, Item2.Caption );

                              (* Date/Time Column *)
			COLUMN_DATETIME:
        	BEGIN
              					(* Separate date from date & time *)
					Str := Item1.SubItems.Strings[ FSortColumn - 1 ];
              Str := Copy( Str, 1, Pos( ' ', Str ) - 1 );
					date1 := StrToDate( Str );

              					(* Separate date from date & time *)
					Str := Item2.SubItems.Strings[ FSortColumn - 1 ];
              Str := Copy( Str, 1, Pos( ' ', Str ) - 1 );
					date2 := StrToDate( Str );

					Result := Round( 1E5 * ( date1 - date2 ) );

				END;

                              (* Attr, Offset, Compressed, Uncompressed *)
        COLUMN_ATTR,
        COLUMN_OFFSET,
			COLUMN_PACKED,
        COLUMN_UNPACKED:
				BEGIN

					size1 := StrToInt( Item1.SubItems.Strings [ FSortColumn - 1 ] );
					size2 := StrToInt( Item2.SubItems.Strings [ FSortColumn - 1 ] );
					Result := ( size1 - size2 );

				END;
                              (* Ratio Column *)
			COLUMN_RATIO:
        	BEGIN
										 (* Copy string and remove % sign *)

					Str := Item1.SubItems.Strings [ FSortColumn - 1 ];
					SetLength( Str, LENGTH( Str ) - 1 );
					size1 := StrToInt( Str );

										 (* Copy string and remove % sign *)

					Str := Item2.SubItems.Strings [ FSortColumn - 1 ];
					SetLength( Str, LENGTH( Str ) - 1 );
					size2 := StrToInt( Str );

					Result := ( size1 - size2 );

				END;

		ELSE
										 (* All Other Columns *)

			Result := CompareText( Item1.SubItems.Strings [ FSortColumn - 1 ],
											Item2.SubItems.Strings [ FSortColumn - 1 ] );

		END;

	EXCEPT

		Result := 0;

	END;

	IF FSortForward THEN	Compare := Result ELSE Compare := -Result;

END;
{-------------------------------------------------------------}
(* Menu - View/Sort menu item... tell user how to sort *)
PROCEDURE TfrmMain.Sort1Click(Sender: TObject);
VAR
	Column: TListColumn;
BEGIN
  IF ListView1.ViewStyle = vsReport THEN
		ShowMessage( 'Click on the column headers' )
  ELSE BEGIN
     Column := ListView1.Column[ 0 ];
     ListView1ColumnClick( Sender, Column );
  END;
END;
{-------------------------------------------------------------}
(* Menu - View/Show Blank Dirs *)
PROCEDURE TfrmMain.BlankDirs1Click(Sender: TObject);
BEGIN
	BlankDirs1.Checked := BlankDirs1.Checked XOR True;
  ZipTV1.ShowBlankDirs := BlankDirs1.Checked;
END;
{-------------------------------------------------------------}

END.
