unit UPTTreeList; // Copyright  1997-1998 Plasmatech Software Design. All rights reserved.
{
 Shell Control Pack
 Version 1.3d

 Plasmatech's extensions to Delphi's implementation of the Windows tree and list view common controls.

 Implements the following components:
   TPTTreeView  - tree with IE4 styles, custom draw, per-item popup menus.
   TPTListView  - list with IE4 styles, custom draw, per-item popup menus.

 History
 ==============================================================
 V1.30d --TBA-- Fixed problem with TPTListView properties Checkboxes, GridLines, HotTrack and RowSelect not
                  being available at design time in C++Builder 3.
 V1.30c 16Mar98 C++Builder 3 support.
 V1.30b  7Feb98 No changes.
 V1.30a  7Jan98 Fixed problem with TPTCustomTreeView.CMFontChanged method.
 V1.30  28Nov97 Added fix to VCL's TListView.Color property. The background is now correctly set.
                Added fix to VCL's TTreeView.Color property - only for commctrl.dll v4.71 (IE4).
                Fixed some problems with custom draw. Foreground and background colors are available
                  now for both tree and list controls with IE3 and IE4.
 V1.20b 12Oct97 No changes.
 V1.20a  5Oct97 No significant changes.
 V1.20   6Sep97 Added GetImageIndex and GetSelectedIndex dynamic methods.
                Right-click popup menus now do not appear twice under Internet Explorer 4.
                Right-click popup menus now do not appear if a drag operation was initiated during the
                  mouse-down, mouse-up sequence.
                Fixed GDI leak when using the Font property of TPTCustomDraw. (*1)
                Fixed declaration of CDRF_SKIPDEFAULT so the NoDefaultDrawing property now works.
                Fixed bug in Delphi 2 by exposing protected InsertItem method.
 V1.10a  6Jul97 No changes.
 V1.10  26Jun97 This unit was added to the pack for this version.

(*1)
  The offending code was in the CNNotify method:
    obj := CustomDraw( ... ); // <-- user ends up implicitly creating font in an OnCustomDraw handler
    inherited; // <-- does nothing really
    obj.Free; // <-- font is destroyed here before it was used by the tree or list

  The problem is that it is not enough to simply do the inherited processing on the CNNotify handler before
  destroying the temporary TPTCustomDraw object (and its contained resources). You have to return the font
  before you destroy it. The real trick is that the leak won't happen if the Delphi font manager has decided
  to share the handle. If any other TFont object is using the same physical font handle then the 'obj.Free'
  call will simply reduce the reference count, not release the real HFONT and everything will work fine. Only
  in the case when you use a font not used by any other visible control will the leak occur.

  We get around the problem by using a temporary 'lastobj' member which we delete at the start of the next
  custom draw event. The new code is:
    mlastobj.Free; mlastObj:=nil;
    obj := CustomDraw(...);
    inherited;
    mlastobj := obj;
}
{$RANGECHECKS OFF} {$OVERFLOWCHECKS OFF} {$WRITEABLECONST OFF}
{$BOOLEVAL OFF}    {$EXTENDEDSYNTAX ON}  {$TYPEDADDRESS ON}

{$INCLUDE PTCompVer.inc}

interface
uses Windows, Messages, SysUtils, Classes, Graphics, Controls,
     Forms, Dialogs, Ole2, ComCtrls, CommCtrl, Menus;


{$IFNDEF VCL35PLUS}
{-- Declarations from comctrls.h ---------------------------------------------
    { ====== WM_NOTIFY codes (NMHDR.code values) ================== }
    const NM_CUSTOMDRAW        = NM_FIRST-12;

    //==================== CUSTOM DRAW ==========================================
    // custom draw return flags
    // values under 0x00010000 are reserved for global custom draw values.
    // above that are for specific controls
    const CDRF_DODEFAULT         = $00000000;
          CDRF_NEWFONT           = $00000002;
          CDRF_SKIPDEFAULT       = $00000004;

          CDRF_NOTIFYPOSTPAINT   = $00000010;
          CDRF_NOTIFYITEMDRAW    = $00000020;
          CDRF_NOTIFYSUBITEMDRAW = $00000020;  // flags are the same, we can distinguish by context
          CDRF_NOTIFYPOSTERASE   = $00000040;

    // drawstage flags
    // values under 0x00010000 are reserved for global custom draw values.
    // above that are for specific controls

    const CDDS_PREPAINT        = $000000001;
          CDDS_POSTPAINT       = $000000002;
    // the 0x000010000 bit means it's individual item specific
          CDDS_ITEM            = $000010000;
          CDDS_ITEMPREPAINT    = (CDDS_ITEM or CDDS_PREPAINT);
          CDDS_ITEMPOSTPAINT   = (CDDS_ITEM or CDDS_POSTPAINT);

    // itemState flags
    const CDIS_SELECTED        = $0001;
          CDIS_GRAYED          = $0002;
          CDIS_DISABLED        = $0004;
          CDIS_CHECKED         = $0008;
          CDIS_FOCUS           = $0010;
          CDIS_DEFAULT         = $0020;
          CDIS_HOT             = $0040;
          CDIS_MARKED          = $0080;
          CDIS_INDETERMINATE   = $0100;
{$ENDIF}

    type TNMCustomDraw = packed record
           hdr: TNmHdr;
           dwDrawStage: DWORD;
           hdc: HDC;
           rc: TRect; // This field seems to be always wrong.
           dwItemSpec: DWORD; // This is control specific, but it's how to specify an item. Valid only with CDDS_ITEM bit set
           uItemState: UINT;
           lParam: Integer;
         end;
         PNMCustomDraw = ^TNMCustomDraw;

         TNMLvCustomDraw = packed record
           nmcd: TNMCustomDraw;
           clrText: TColorRef;
           clrTextBk: TColorRef;
           subItem: Integer;
         end;
         PNMLvCustomDraw = ^TNMLvCustomDraw;

         TNMTvCustomDraw = packed record
           nmcd: TNMCustomDraw;
           clrText: TColorRef;
           clrTextBk: TColorRef;
           level: Integer;
         end;
         PNMTvCustomDraw = ^TNMTvCustomDraw;
{== END Declarations from comctrls.h =========================================}


{-- Wrapper types ------------------------------------------------------------}
type TPTCustomDrawStage = ( ptcdsUnknown,
                            ptcdsPrePaint,
                            ptcdsPostPaint,
                            ptcdsPreErase,
                            ptcdsItemPrePaint,
                            ptcdsItemPostPaint,
                            ptcdsItemPreErase,
                            ptcdsItemPostErase
                          );

     TPTCustomDraw = class(TObject)
       private
         mParentFont: TFont;
         mFont: TFont;
         mfFontChanged: Boolean;

         mBrush: TBrush;
         mfBrushChanged: Boolean;

         mfWantItems: Boolean;
         mfNoDefaultDrawing: Boolean;

         mpcd: ^TNMCustomDraw;
         mpResult: ^Integer;

         mfUserResult: Boolean;

         mCanvas: TCanvas;

         procedure OnFontChange( aSender: TObject );
         procedure OnBrushChange( aSender: TObject );

         function  GetRawDrawStage: DWORD;
         function  GetRect: TRect;
         function  GetDrawStage: TPTCustomDrawStage;
         function  GetFont: TFont;
         function  GetBrush: TBrush;
         function  GetCanvas: TCanvas;
         function  GetHandle: HDC;
         function  GetIsItem: Boolean;
         function  GetResult: DWORD;

         procedure SetFont( aValue: TFont );
         procedure SetBrush( aValue: TBrush );
         procedure SetResult( aValue: DWORD );
       protected
         procedure Reset( const acd: TNMCustomDraw;  var aResult: Integer ); virtual;
         function  CreateFont: TFont; virtual;
         function  CreateBrush: TBrush; virtual;

       public
         constructor Create( const acd: TNMCustomDraw;  var aResult: Integer;  const aParentFont: TFont );
         destructor  Destroy; override;

         procedure Apply; virtual; abstract;

         property RawDrawStage: DWORD read GetRawDrawStage;
         property DrawStage: TPTCustomDrawStage read GetDrawStage;

         property IsItem: Boolean read GetIsItem;
         property WantItems: Boolean read mfWantItems write mfWantItems;  // Only relevant when DrawStage = ptcdsPrePaint

         property Result: DWORD read GetResult write SetResult;  // !! Document
         property Rect: TRect read GetRect;
         property Canvas: TCanvas read GetCanvas;

         property Handle: HDC read GetHandle;
         property Font: TFont read GetFont write SetFont;
         property Brush: TBrush read GetBrush write SetBrush;

         property NoDefaultDrawing: Boolean read mfNoDefaultDrawing write mfNoDefaultDrawing; // Set TRUE if you do all the drawing yourself
     end; {TPTCustomDraw}


     TPTTvCustomDraw = class(TPTCustomDraw)
       public // private
         function GetMpTvCd: PNMTvCustomDraw;
         property mpTvCd: PNMTvCustomDraw read GetMpTvCd;
       protected
         constructor Create( const aCD: TNMCustomDraw;  var aResult: Integer;  const aParentFont: TFont );
         procedure Apply; override; // Fill in fgcolor/bgcolor fields instead of using settextcolor/setbkcolor 
     end; {TPTTvCustomDraw}


     TPTLvCustomDraw = class(TPTCustomDraw)
       public //private
         function GetMpLvCd: PNMLVCustomDraw;
         property mpLvCD: PNMLVCustomDraw read GetMpLvCD;

       protected
         constructor Create( const acd: TNMCustomDraw;  var aResult: Integer;  const aParentFont: TFont );
         procedure Apply; override;  // Fill in fgcolor/bgcolor fields instead of using settextcolor/setbkcolor
     end; {TPTCustomDrawListView}

{== END Wrapper types ========================================================}


type TPTTvOnNodeContextMenuEvent = procedure( aSender: TObject;
                                              aNode: TTreeNode;
                                              var aPos: TPoint;
                                              var aMenu: TPopupMenu ) of object;
     TPTTvCustomDrawEvent = procedure( aSender: TObject;  aCD: TPTCustomDraw;  aNode: TTreeNode ) of object;

     TPTCustomTreeView = class(TCustomTreeView)
       private
         mOnNodeContextMenuProc: TPTTvOnNodeContextMenuEvent;
         mfCustomDrawEnabled: Boolean;

         mOnCustomDraw: TPTTvCustomDrawEvent;
         mOnCustomDrawEx: TPTTvCustomDrawEvent;

         mfMenuAlreadyHandled: Boolean;
         mLastCustomDrawObj: TObject; // Deferred destruction

         procedure CMColorChanged( var aMsg: TMessage ); message CM_COLORCHANGED;
         procedure CMFontChanged( var aMsg: TMessage ); message CM_FONTCHANGED;
         procedure CNNotify( var aMsg: TWMNotify ); message CN_NOTIFY;
         procedure WMRButtonUp( var aMsg: TWMRButtonUp ); message WM_RBUTTONUP;
         procedure WMContextMenu( var aMsg: TMessage ); message WM_CONTEXTMENU;

       protected
         procedure CreateParams( var p: TCreateParams ); override;
         procedure CreateWnd; override;
         procedure DestroyWnd; override;

         function  CreateCustomDraw( const aNMHDR: TNMCustomDraw;  var aResult: Integer ): TPTCustomDraw; dynamic;
         function  CustomDraw( const aNMHDR: TNMCustomDraw;  var aResult: Integer ): TPTCustomDraw; virtual; // Virtual for speed

         procedure DeleteNode( aNode: TTreeNode ); dynamic;

         procedure DoPreNodeContextMenu; dynamic;
         procedure DoNodeContextMenu( aNode: TTreeNode; ap: TPoint ); dynamic;
{$IFNDEF VCL35PLUS}  // VCL 3.5+ already declares these methods.
         procedure GetImageIndex( aNode: TTreeNode ); dynamic;
         procedure GetSelectedIndex( aNode: TTreeNode ); dynamic;
{$ENDIF}
         procedure KeyDown( var key: Word;  aShiftState: TShiftState ); override;

         procedure NodeContextMenu( aNode: TTreeNode;  var aPos: TPoint;  var aMenu: TPopupMenu ); dynamic;

         property CustomDrawEnabled: Boolean read mfCustomDrawEnabled write mfCustomDrawEnabled default FALSE;
         property OnCustomDrawEx: TPTTvCustomDrawEvent read mOnCustomDrawEx write mOnCustomDrawEx;
         property OnCustomDraw: TPTTvCustomDrawEvent read mOnCustomDraw write mOnCustomDraw;
         property OnNodeContextMenu: TPTTvOnNodeContextMenuEvent read mOnNodeContextMenuProc write mOnNodeContextMenuProc;

       public
         destructor Destroy; override;
         procedure InvalidateNode( aNode: TTreeNode;  afTextOnly: Boolean;  afEraseBkgnd: Boolean );
     end; {TPTCustomTreeView}


     TPTTreeView = class(TPTCustomTreeView)
       private
       published
        // Properties published TTreeView
         property ShowButtons;
         property BorderStyle;
         property DragCursor;
         property ShowLines;
         property ShowRoot;
         property ReadOnly;
         property DragMode;
         property HideSelection;
         property Indent;
         property Items;
         property OnEditing;
         property OnEdited;
         property OnExpanding;
         property OnExpanded;
         property OnCollapsing;
         property OnCompare;
         property OnCollapsed;
         property OnChanging;
         property OnChange;
         property OnDeletion;
         property OnGetImageIndex;
         property OnGetSelectedIndex;
         property Align;
         property Enabled;
         property Font;
         property Color;
         property ParentColor;
         property ParentCtl3D;
         property Ctl3D;
         property SortType;
         property TabOrder;
         property TabStop default True;
         property Visible;
         property OnClick;
         property OnEnter;
         property OnExit;
         property OnDragDrop;
         property OnDragOver;
         property OnStartDrag;
         property OnEndDrag;
         property OnMouseDown;
         property OnMouseMove;
         property OnMouseUp;
         property OnDblClick;
         property OnKeyDown;
         property OnKeyPress;
         property OnKeyUp;
         property PopupMenu;
         property ParentFont;
         property ParentShowHint;
         property ShowHint;
         property Images;
         property StateImages;

        // New properties
         property OnCustomDraw;
         property OnCustomDrawEx;
         property OnNodeContextMenu;

     end; {TPTTreeView}


     TPTLvOnItemContextMenuEvent = procedure( aSender: TObject;
                                              aItem: TListItem;
                                              var aPos: TPoint;
                                              var aMenu: TPopupMenu ) of object;
     TPTLvCustomDrawEvent = procedure( aSender: TObject;  aCD: TPTCustomDraw;  aItem: TListItem ) of object;

     TPTCustomListView = class(TCustomListView)
       private
         mOnItemContextMenuProc: TPTLvOnItemContextMenuEvent;

         mOnCustomDraw: TPTLvCustomDrawEvent;
         mOnCustomDrawEx: TPTLvCustomDrawEvent;

         mfMenuAlreadyHandled: Boolean;
         mLastCustomDrawObj: TObject; // Deferred destruction
         mfDragStarted: Boolean;

         procedure CMColorChanged( var aMsg: TMessage ); message CM_COLORCHANGED;
         procedure CNNotify( var aMsg: TWMNotify ); message CN_NOTIFY;

         procedure WMContextMenu( var aMsg: TMessage ); message WM_CONTEXTMENU;
         procedure WMRButtonUp( var aMsg: TWMRButtonUp ); message WM_RBUTTONUP;

       protected
         procedure CreateWnd; override;
         procedure DestroyWnd; override;

         function  CreateCustomDraw( const aNMHDR: TNMCustomDraw;  var aResult: Integer ): TPTCustomDraw; dynamic;
         function  CustomDraw( const aNMHDR: TNMCustomDraw;  var aResult: Integer ): TPTCustomDraw; virtual; // Virtual for speed

         procedure DoPreItemContextMenu( pt: TPoint ); dynamic;
         procedure DoItemContextMenu( p: TPoint ); dynamic;

         procedure ItemContextMenu( aItem: TListItem;  var aPos: TPoint;  var aMenu: TPopupMenu ); dynamic;

         procedure KeyDown( var key: Word;  aShiftState: TShiftState ); override;

{$IFDEF VER9x}
         procedure InsertItem( aItem: TListItem ); dynamic;
{$ENDIF}         

         property OnCustomDraw: TPTLvCustomDrawEvent read mOnCustomDraw write mOnCustomDraw;
         property OnCustomDrawEx: TPTLvCustomDrawEvent read mOnCustomDrawEx write mOnCustomDrawEx;
         property OnItemContextMenu: TPTLvOnItemContextMenuEvent read mOnItemContextMenuProc write mOnItemContextMenuProc;
       public
         destructor  Destroy; override;
     end; {TPTCustomListView}

     TPTListView = class(TPTCustomListView)
       public
       published
        // Properties published by TListView
         property Align;
         property BorderStyle;
         property Color;
         property ColumnClick;
         property Columns;
         property OnClick;
         property OnDblClick;
         property Ctl3D;
         property DragMode;
         property ReadOnly;
         property Font;
         property HideSelection;
         property IconOptions;
         property Items;
         property AllocBy;
         property MultiSelect;
         property OnChange;
         property OnChanging;
         property OnColumnClick;
         property OnCompare;
         property OnDeletion;
         property OnEdited;
         property OnEditing;
         property OnEnter;
         property OnExit;
         property OnInsert;
         property OnDragDrop;
         property OnDragOver;
         property DragCursor;
         property OnStartDrag;
         property OnEndDrag;
         property OnMouseDown;
         property OnMouseMove;
         property OnMouseUp;
         property ParentShowHint;
         property ShowHint;
         property PopupMenu;
         property ShowColumnHeaders;
         property SortType;
         property TabOrder;
         property TabStop default True;
         property ViewStyle;
         property Visible;
         property OnKeyDown;
         property OnKeyPress;
         property OnKeyUp;
         property StateImages;
         property LargeImages;
         property SmallImages;
{$IFDEF VCL30PLUS}
         property Checkboxes;
         property GridLines;
         property HotTrack;
         property RowSelect;
{$ENDIF}
        // New properties
         property OnCustomDraw;
         property OnCustomDrawEx;
         property OnItemContextMenu;
     end; {TPTListView}

{*********************************************************}
implementation
uses ShellApi;


{**************************************
  TPTCustomDraw
**************************************}
constructor TPTCustomDraw.Create( const acd: TNMCustomDraw;  var aResult: Integer;  const aParentFont: TFont );
begin
  inherited Create;
  Reset( acd, aResult );
  mParentFont := aParentFont;
  mBrush := CreateBrush;
  mFont := CreateFont;
end;

destructor TPTCustomDraw.Destroy;
begin
  mFont.Free;  mFont:=nil;
  mBrush.Free; mBrush:=nil;
  mCanvas.Free; mCanvas:=nil;
  inherited;
end;

procedure TPTCustomDraw.OnFontChange( aSender: TObject );
  begin mfFontChanged := TRUE; end;

procedure TPTCustomDraw.OnBrushChange( aSender: TObject );
  begin mfBrushChanged := TRUE; end;

function TPTCustomDraw.GetRawDrawStage: DWORD;
  begin result := mpcd.dwDrawStage; end;

function TPTCustomDraw.GetRect: TRect;
  begin result := mpcd.rc; end;

function TPTCustomDraw.GetDrawStage: TPTCustomDrawStage;
begin
  case mpcd.dwDrawStage of
    CDDS_PREPAINT:      result := ptcdsPrePaint;
    CDDS_POSTPAINT:     result := ptcdsPostPaint;
    CDDS_ITEMPREPAINT:  result := ptcdsItemPrePaint;
    CDDS_ITEMPOSTPAINT: result := ptcdsItemPostPaint;
  else                  result := ptcdsUnknown;
  end;
end;

function TPTCustomDraw.GetFont: TFont;
begin
  if not Assigned(mFont) then mFont := CreateFont;
  result := mFont;
end;

function TPTCustomDraw.GetBrush: TBrush;
begin
  if not Assigned(mBrush) then mBrush := CreateBrush;
  result := mBrush;
end;

function TPTCustomDraw.GetCanvas: TCanvas;
begin
  if not Assigned(mCanvas) then
  begin
    mCanvas := TCanvas.Create;
    mCanvas.Handle := self.Handle;
  end;
  result := mCanvas;
end;

function TPTCustomDraw.GetHandle: HDC;
  begin result := mpcd.hdc; end;

function TPTCustomDraw.GetIsItem: Boolean;
  begin result := (mpcd.dwDrawStage and CDDS_ITEM)<>0; end;

function TPTCustomDraw.GetResult: DWORD;
  begin result := mpResult^; end;

procedure TPTCustomDraw.SetFont( aValue: TFont );
begin
  if not Assigned(mFont) then mFont := CreateFont;
  mFont.Assign( aValue );
end;

procedure TPTCustomDraw.SetBrush( aValue: TBrush );
begin
  if not Assigned(mBrush) then mBrush := CreateBrush;
  mBrush.Assign(aValue);
end;

procedure TPTCustomDraw.SetResult( aValue: DWORD );
begin
  mpResult^ := aValue;
  mfUserResult := TRUE;
end;

procedure TPTCustomDraw.Reset( const acd: TNMCustomDraw;  var aResult: Integer );
begin
  mpcd := @acd;
  mpResult := @aResult;
  mfUserResult := FALSE;
  mBrush.Free; mBrush:=nil;
  mFont.Free; mFont:=nil;
end;

function TPTCustomDraw.CreateFont: TFont;
begin
  result := TFont.Create;
  result.Assign( mParentFont );
  result.Color := TColor( GetTextColor(mpcd.hdc) );
  result.OnChange := OnFontChange;
end;

function TPTCustomDraw.CreateBrush: TBrush;
begin
  result := TBrush.Create;
  result.Style := bsSolid;
  result.Color := TColor( GetBkColor(mpcd.hdc) );
  result.OnChange := OnBrushChange;
end;

const _wantitems: array[Boolean] of Integer = (0, CDRF_NOTIFYITEMDRAW);
//      _wantpostpaint: array[Boolean] of Integer = (0, CDRF_NOTIFYPOSTPAINT);

(*
procedure TPTCustomDraw.Apply;
begin
  if not mfUserResult then
    mpResult^ := CDRF_DODEFAULT;

  case DrawStage of
    ptcdsPrePaint:
      if not mfUserResult then
        mpResult^ := mpResult^ or _wantitems[WantItems]; // or _wantpostpaint[WantPostPaint];

    ptcdsItemPrePaint:
      begin
        if NoDefaultDrawing then
        begin
          if not mfUserResult then
            mpResult^ := mpResult^ or CDRF_SKIPDEFAULT;
          Exit;
        end;

        if mfFontChanged and Assigned(mFont) then
        begin
          SelectObject( mpcd.hdc, mFont.Handle );
          mpResult^ := mpResult^ or CDRF_NEWFONT;
          if (mFont.Color <> clNone) then
            Windows.SetTextColor( mpcd.hdc, ColorToRGB(mFont.Color) );
        end;

        if mfBrushChanged and Assigned(mBrush) and (mBrush.Color <> clNone) then
          Windows.SetBkColor( mpcd.hdc, ColorToRGB(mBrush.Color) );
      end;

    ptcdsItemPostPaint:
      if not mfUserResult then
        mpResult^ := CDRF_DODEFAULT;
  end;
end;
*)


{***************************************
  TPTTvCustomDraw
***************************************}
constructor TPTTvCustomDraw.Create( const acd: TNMCustomDraw;  var aResult: Integer;  const aParentFont: TFont );
begin
  inherited;
end;

function TPTTvCustomDraw.GetMpTvCd: PNMTVCustomDraw;
begin
  result := PNMTVCustomDraw(mpcd);
end;

procedure TPTTvCustomDraw.Apply;
begin
  if not mfUserResult then
    mpResult^ := CDRF_DODEFAULT;
  case DrawStage of
    ptcdsPrePaint:
      if not mfUserResult then
        mpResult^ := mpResult^ or _wantitems[WantItems]; // or _wantpostpaint[WantPostPaint] or _wantposterase[WantPostErase]

    ptcdsItemPrePaint:
      begin
        if NoDefaultDrawing then
        begin
          if not mfUserResult then
            mpResult^ := mpResult^ or CDRF_SKIPDEFAULT;
          Exit;
        end;

        if mfFontChanged and Assigned(mFont) then
        begin
          if (mFont.Handle <> GetCurrentObject(mpcd.hdc, OBJ_FONT)) then
          begin
            SelectObject( mpcd.hdc, mFont.Handle );
            mpResult^ := mpResult^ or CDRF_NEWFONT;
          end;
          mpTvCd.clrText := ColorToRGB(mFont.Color);
        end;

        if mfBrushChanged and Assigned(mBrush) then
          mpTvCd.clrTextBk := ColorToRGB(mBrush.Color);
      end;
  end;
end;


{**************************************
  TPTLvCustomDraw
**************************************}
constructor TPTLvCustomDraw.Create( const acd: TNMCustomDraw;  var aResult: Integer;  const aParentFont: TFont );
begin
  inherited;
end;


function TPTLvCustomDraw.GetMpLvCd: PNMLVCustomDraw;
begin
  result := PNMLVCustomDraw(mpcd);
end;


procedure TPTLvCustomDraw.Apply;
begin
  mpResult^ := CDRF_DODEFAULT;
  case DrawStage of
    ptcdsPrePaint:
      mpResult^ := mpResult^ or _wantitems[WantItems]; // or _wantpostpaint[WantPostPaint];

    ptcdsItemPrePaint:
      begin
        if NoDefaultDrawing then
        begin
          mpResult^ := mpResult^ or CDRF_SKIPDEFAULT;
          Exit;
        end;

        if mfFontChanged and Assigned(mFont) then
        begin
          if (mFont.Handle <> GetCurrentObject(mpcd.hdc, OBJ_FONT)) then
          begin
            SelectObject( mpcd.hdc, mFont.Handle );
            mpResult^ := mpResult^ or CDRF_NEWFONT;
          end;
          mpLvCd.clrText := ColorToRGB(mFont.Color);
        end;

        if mfBrushChanged and Assigned(mBrush) then
          mpLvCd.clrTextBk := ColorToRGB(mBrush.Color);
      end;
  end;
end; {TPTLvCustomDraw.Apply}





{***************************************
  TPTCustomTreeView
***************************************}
destructor TPTCustomTreeView.Destroy;
begin
  mLastCustomDrawObj.Free; // Just in case
  inherited;
end;

procedure TPTCustomTreeView.InvalidateNode( aNode: TTreeNode;  afTextOnly: Boolean;  afEraseBkgnd: Boolean );
var r: TRect;
begin
  r := aNode.DisplayRect(afTextOnly);
  InvalidateRect( Handle, @r, afEraseBkgnd );
end;

procedure TPTCustomTreeView.CMColorChanged( var aMsg: TMessage );
const TVM_SETBKCOLOR = TV_FIRST + 29;
var cr: TColorRef;
begin
  inherited;
  if (Color=clNone) then cr := -1 else cr := ColorToRGB(Color);
  Perform( TVM_SETBKCOLOR, 0, cr ); // Only has any effect with commctrl.dll v4.71 (IE4)
end;

procedure TPTCustomTreeView.CMFontChanged( var aMsg: TMessage );
const TVM_SETTEXTCOLOR = TV_FIRST + 30;
var cr: TColorRef;
begin
  inherited;
  if (Font.Color=clNone) then cr := -1 else cr := ColorToRGB(Font.Color);
  Perform( TVM_SETTEXTCOLOR, 0, cr ); // Only has any effect with commctrl.dll v4.71 (IE4)
end;

procedure TPTCustomTreeView.CNNotify( var aMsg: TWMNotify );
{
  const TVN_GETINFOTIP = (TVN_FIRST - 13);  // comctl32.dll v4.71
        TVN_GETINFOTIPA = (TVN_FIRST - 13); // comctl32.dll v4.71
        TVN_GETINFOTIPW = (TVN_FIRST - 14); // comctl32.dll v4.71

  type  TNMTvGetInfoTipA = packed record
          hdr: TNMHdr;
          pszText: PChar;
          cchTextMax: Integer;
          hItem: HTreeItem;
          lParam: Integer;
        end;
        PNMTvGetInfoTipA = ^TNMTvGetInfoTipA;

        TNMTvGetInfoTip = TNMTvGetInfoTipA;
        PNMTvGetInfoTip = ^TNMTvGetInfoTip;
}
  function GetNodeFromItem(const Item: TTVItem): TTreeNode;
  begin
    with Item do
      if (state and TVIF_PARAM) <> 0 then Result := Pointer(lParam)
      else Result := Items.GetNode(hItem);
  end;
var node: TTreeNode;
    p: TPoint;
    obj: TPTCustomDraw;
    mnu: TPopupMenu;
    oldGetImageEvent: TTVExpandedEvent;
    oldGetSelectedImageEvent: TTVExpandedEvent;
begin {CNNotify}
  with aMsg.NMHdr^ do
    case code of
{     TVN_GETINFOTIP:
        begin
          messagebeep(uint(-1));
          with PNMTvGetInfoTip(Pointer(aMsg.NMHdr))^ do
          begin
            node := Items.GetNode(hItem);
            StrLCopy( pszText, PChar(Node.Text + '**'), cchTextMax );
          end;
        end; {TVN_GETINFOTIP}

      TVN_DELETEITEM:
        begin
          with PNMTreeView(Pointer(aMsg.NMHdr))^ do
            node := GetNodeFromItem(itemOld);

          if (node <> nil) then
            DeleteNode( node );
          inherited;
        end; {TVN_DELETEITEM}

      TVN_GETDISPINFO:
        with PTVDispInfo(Pointer(aMsg.NMHdr))^ do
        begin
          node := GetNodeFromItem(item);

          if Assigned(node) then
          begin
            if (item.mask and TVIF_IMAGE) <> 0 then
            begin
              GetImageIndex(Node);
              item.iImage := Node.ImageIndex;
            end;

            if (item.mask and TVIF_SELECTEDIMAGE) <> 0 then
            begin
              GetSelectedIndex(Node);
              item.iSelectedImage := Node.SelectedIndex;
            end;
          end;

          oldGetImageEvent := OnGetImageIndex;
          oldGetSelectedImageEvent := OnGetSelectedIndex;
          OnGetImageIndex:=nil;
          OnGetSelectedIndex:=nil;
          try
            inherited;
          finally
            OnGetImageIndex := oldGetImageEvent;
            OnGetSelectedIndex := oldGetSelectedImageEvent;
          end;
        end; {with, TVN_GETDISPINFO}

      NM_RCLICK:
        begin
          // Note: The RightClickSelect property introduced in Delphi 3 can do some of this. We don't use it
          //       in order to maintain Delphi 2 and C++Builder compatibility.
          if not (csDesigning in ComponentState) then
          begin
            GetCursorPos(p);
            p := ScreenToClient(p);
            node := GetNodeAt(p.x,p.y);
            if Assigned(node) then
            begin
              mnu := PopupMenu; // Default is normal popup
              NodeContextMenu( node, p, mnu );
              if Assigned(mnu) then
                with ClientToScreen(p) do
                begin
                  mnu.PopupComponent := self;
                  mnu.Popup(x,y);
                end;
              mfMenuAlreadyHandled := TRUE;
            end;
          end;
          inherited;
        end; {NM_RCLICK}

      NM_CUSTOMDRAW:
        if not (csDesigning in ComponentState) then
        begin
          if Assigned(mLastCustomDrawObj) then begin mLastCustomDrawObj.Free; mLastCustomDrawObj:=nil; end;
          obj := CustomDraw( PNMCustomDraw(aMsg.NMHdr)^, Integer(aMsg.result) );
          inherited;
          mLastCustomDrawObj := obj;
        end;
    else
      inherited;
    end; {case}
end; {TPTCustomTreeView.CNNotify}


procedure TPTCustomTreeView.WMRButtonUp( var aMsg: TWMRButtonUp );
var fOldAutoPopup: Boolean;
begin
  if mfMenuAlreadyHandled and Assigned(PopupMenu) then
  begin
    fOldAutoPopup := PopupMenu.AutoPopup;
    PopupMenu.AutoPopup := FALSE;
    try
      inherited;
    finally
      PopupMenu.AutoPopup := fOldAutoPopup;
      mfMenuAlreadyHandled := FALSE;
    end;
  end
  else
    inherited;
end; {TPTCustomTreeView.WMRButtonUp}


procedure TPTCustomTreeView.WMContextMenu( var aMsg: TMessage );
begin
  if not (csDesigning in ComponentState) and not Assigned(Selected) and not mfMenuAlreadyHandled then
  begin
    if (aMsg.lParam = -1) then
      DoPreNodeContextMenu
    else
      DoNodeContextMenu( Selected, ScreenToClient(Point(aMsg.lParamLo, aMsg.lParamHi)) );
  end;
end; {TPTCustomTreeView.WMContextMenu}



{ Work around a bug with tooltips in NT4. We just disable them. }
procedure TPTCustomTreeView.CreateParams( var p: TCreateParams );
const TVS_NOTOOLTIPS = $0080;  // comctl32.dll v4.70
      TVS_INFOTIP    = $0800;  // comctl32.dll v4.71
var v: TOSVersionInfo;
begin
  inherited;
  v.dwOSVersionInfoSize := Sizeof(TOSVersionInfo);
  GetVersionEx( v );
  if v.dwPlatformId = VER_PLATFORM_WIN32_NT then
    p.Style := p.Style or TVS_NOTOOLTIPS;
end;

procedure TPTCustomTreeView.CreateWnd;
begin
  inherited;
end;

procedure TPTCustomTreeView.DestroyWnd;
begin
  inherited;
end;

{ This builder method creates a TPTCustomDraw instance, initialised to the contents of the aMsg structure. }
function TPTCustomTreeView.CreateCustomDraw( const aNMHDR: TNMCustomDraw;  var aResult: Integer ): TPTCustomDraw;
begin
  result := TPTTvCustomDraw.Create( aNMHDR, aResult, Font );
  result.WantItems := TRUE;  
end;

{ This method is called in response to a WM_NOTIFY message with the NM_CUSTOMDRAW code. It determines if
  any event handlers are registered. If so, it calls OnCustomDraw if the draw state is item-pre-paint. If
  OnCustomDrawEx is assigned, then it is called for every state. If a TPTCustomDraw object is created,
  it is returned from the function. The CNNotify message handler will free it after performing the
  inherited CNNotify processing.

  We go to all this trouble to try and minimize the number of TNMCustomDraw objects that are created, and the
  number of 'dwItemSpec' lookups for the TTreeNode object. The effect of all this convolution is that no
  TPTCustomDraw object is created and no lookup is performed unless the OnCustomDraw or OnCustomDrawEx methods
  are set. Moreover, if OnCustomDraw is set then the object/lookup expense is incurred only for the item-pre-paint
  draw state. If OnCustomDrawEx is set then the object/lookup is performed for every notification (4 for the tree
  as a whole, 4 for each item!)

  If you want to derive from TPTCustomTreeView and do some sort of default custom draw handling, you will need
  to do some of the processing that's done in this method. Especially create the TPTCustomDraw object and do
  the node lookup. These operations have been broken out into methods so you can call them easily yourself
  from you own handlers.}
function TPTCustomTreeView.CustomDraw( const aNMHDR: TNMCustomDraw;  var aResult: Integer ): TPTCustomDraw;
var focda, focdexa: Boolean;
    node: TTreeNode;
  function FindNode: TTreeNode;
  begin
    if (aNMHDR.dwDrawStage and CDDS_ITEM) = 0 then
      result := nil
    else
      result := Items.GetNode(HTreeItem(aNMHdr.dwItemSpec));
  end;
begin
  result := nil;
  node := nil;
  focda := Assigned(OnCustomDraw);
  focdexa := Assigned(OnCustomDrawEx);

  if focdexa then
  begin
    result := CreateCustomDraw(aNMHDR, aResult);
    node := FindNode;
    OnCustomDrawEx( self, result, node );
  end;

  if focda and ((aNMHDR.dwDrawStage and CDDS_ITEMPREPAINT) = CDDS_ITEMPREPAINT) then
  begin
    if not Assigned(result) then result := CreateCustomDraw(aNMHDR, aResult);
    if not Assigned(node) then node := FindNode;
    OnCustomDraw( self, result, node );
  end;

  if Assigned(result) then
    result.Apply;

  if ((aNMHDR.dwDrawStage and CDDS_PREPAINT)<>0) and (focda or focdexa) then
    aResult := aResult or CDRF_NOTIFYITEMDRAW;
end; {TPTCustomTreeView.CustomDraw}

procedure TPTCustomTreeView.DeleteNode( aNode: TTreeNode );
  begin {OnDeletion is called by TTreeNodes, so we don't do it here} end;

procedure TPTCustomTreeView.DoPreNodeContextMenu;
  procedure DoDefault;
  begin
    if Assigned(PopupMenu) then
    begin
      PopupMenu.PopupComponent := self;
      with ClientToScreen(Point(0,0)) do
        PopupMenu.Popup( x,y );
    end;
  end;
var p: TPoint;
begin
  if Assigned(Selected) then
  begin
    with Selected.DisplayRect( True ) do
      p := Point( (left+right) div 2,  (bottom+top) div 2 )
  end
  else
  begin
    DoDefault;
    Exit;
  end;
  DoNodeContextMenu( Selected, p );
end; {TPTCustomTreeView.DoPreNodeContextMenu}

procedure TPTCustomTreeView.DoNodeContextMenu( aNode: TTreeNode;  ap: TPoint );
var mnu: TPopupMenu;
begin
  mnu := PopupMenu; // Default to normal popup
  NodeContextMenu( aNode, ap, mnu );
  if (mnu <> PopupMenu) then
    mfMenuAlreadyHandled := TRUE; //?
  if Assigned(mnu) then    
  begin
    mnu.PopupComponent := self;
    with ClientToScreen(ap) do
      mnu.Popup( x, y );
  end;
end; {TPTCustomTreeView.DoNodeContextMenu}



{$IFNDEF VCL35PLUS}
  procedure TPTCustomTreeView.GetImageIndex( aNode: TTreeNode );
  begin
    if Assigned(OnGetImageIndex) then OnGetImageIndex(self, aNode);
  end; {TPTCustomTreeView.GetImageIndex}
{$ENDIF}

{$IFNDEF VCL35PLUS}
  procedure TPTCustomTreeView.GetSelectedIndex( aNode: TTreeNode );
  begin
    if Assigned(OnGetSelectedIndex) then OnGetSelectedIndex(self, aNode);
  end; {TPTCustomTreeView.GetSelectedIndex}
{$ENDIF}

procedure TPTCustomTreeView.KeyDown( var key: Word;  aShiftState: TShiftState );
begin
  if ((key = VK_APPS) and (aShiftState = [])) or
     ((key = VK_F10) and (aShiftState = [ssShift]))
  then
  begin
    key := 0;
    DoPreNodeContextMenu;
  end;
  inherited;
end; {TPTCustomTreeView.KeyDown}


procedure TPTCustomTreeView.NodeContextMenu( aNode: TTreeNode;  var aPos: TPoint;  var aMenu: TPopupMenu );
  begin if Assigned(OnNodeContextMenu) then OnNodeContextMenu( self, aNode, aPos, aMenu ); end;



{***************************************
  TPTCustomListView
***************************************}
destructor TPTCustomListView.Destroy;
begin
  mLastCustomDrawObj.Free; // Just in case
  inherited;
end;

procedure TPTCustomListView.CMColorChanged( var aMsg: TMessage );
begin
  inherited;
  Perform( LVM_SETBKCOLOR, 0, ColorToRGB(Color) );
  Perform( LVM_SETTEXTBKCOLOR, 0, ColorToRGB(Color) );
end;

procedure TPTCustomListView.CNNotify( var aMsg: TWMNotify );
var obj: TPTCustomDraw;
    pt: TPoint;
begin
  with aMsg.NMHdr^ do
    case code of
      NM_CUSTOMDRAW:
        if not (csDesigning in ComponentState) then
        begin
          if Assigned(mlastCustomDrawObj) then begin mLastCustomDrawObj.Free; mLastCustomDrawObj:=nil; end;
          obj := CustomDraw( PNMCustomDraw(aMsg.NMHdr)^, Integer(aMsg.result) );
          inherited;
          mLastCustomDrawObj := obj;
        end;

      LVN_BEGINDRAG, LVN_BEGINRDRAG:
      begin
        mfDragStarted := TRUE;
        inherited;
      end;

{$IFDEF VER9x}
      LVN_INSERTITEM:
      begin
        with PNMListView(Pointer(aMsg.NMHdr))^ do
          InsertItem(Items[iItem]);
        inherited;
      end;
{$ENDIF}

      NM_RCLICK:
      begin
        Windows.GetCursorPos( pt );
        DoPreItemContextMenu( ScreenToClient(pt) );
      end

      else
       inherited;
    end;
end; {TPTCustomListView.CNNotify}

procedure TPTCustomListView.WMContextMenu( var aMsg: TMessage );
begin
  if not (csDesigning in ComponentState) and not mfMenuAlreadyHandled then
  begin
    if (SelCount=0) or not Assigned(ItemFocused) then
      Exit;

    if (aMsg.lParam = -1) then
      DoPreItemContextMenu( Point(0,0) )
    else
      DoItemContextMenu( ScreenToClient(Point(aMsg.lParamLo, aMsg.lParamHi)) );
  end
end; {TPTCustomListView.WMContextMenu}


{ Since we do all the popup menu handling ourselves, we need to defeat Delphi's default handling. We do this by
  setting the popup menu's AutoPopup property false before allowing Delphi a shot at it. }
procedure TPTCustomListView.WMRButtonUp( var aMsg: TWMRButtonUp );
var OldPopup: TPopupMenu;
begin
  try
    if mfMenuAlreadyHandled and Assigned(PopupMenu) then
    begin
      OldPopup := PopupMenu;
      PopupMenu := nil;
      try
        inherited;
      finally
        PopupMenu := OldPopup;
      end;
    end
    else
      inherited;
  finally
    mfMenuAlreadyHandled := FALSE;
  end;
end; {TPTCustomListView.WMRButtonUp}

{ Call this method when a menu is required, but you don't know if it should be the control or item one. If it should
  be the control one, then it is displayed. If it should be the item one, then the point at which it should popup is
  determined and passed to DoItemContextMenu.  'pt' is where the control menu should be placed if it is decided
  that it is the required menu. }
procedure TPTCustomListView.DoPreItemContextMenu( pt: TPoint );
  procedure DoDefault;
  begin
    if Assigned(PopupMenu) then
    begin
      PopupMenu.PopupComponent := self;
      with ClientToScreen(pt) do
        PopupMenu.Popup( x,y );
    end;
  end;
var p: TPoint;
begin
  if Assigned(ItemFocused) then
  begin // The following code to determine the popup coordinate is derived from observation of Explorer (95/NT)
    if ItemFocused.Selected then
      with ItemFocused.DisplayRect( drIcon ) do
        p := Point( (left+right) div 2,  (bottom+top) div 2 )
    else if (SelCount>0) and Assigned(Selected) then // Some selections, but the current focus isn't one of them, second clause just paranoid safety level
      with Selected.DisplayRect( drIcon ) do
        p := Point( (left+right) div 2,  (bottom+top) div 2 )
    else
    begin
      DoDefault;
      Exit;
    end;
  end
  else
  begin
    DoDefault;
    Exit;
  end;
  DoItemContextMenu( p );
end; {TPTCustomListView.DoPreItemContextMenu}

{ This method is called after it is known that the popup menu should be for an item. }
procedure TPTCustomListView.DoItemContextMenu( p: TPoint );
var mnu: TPopupMenu;
begin
  mnu := PopupMenu; // Default to normal popup
  ItemContextMenu( ItemFocused, p, mnu );
  mfMenuAlreadyHandled := TRUE;
  if Assigned(mnu) then
  begin
    mnu.PopupComponent := self;
    with ClientToScreen(p) do
      mnu.Popup( x, y );
  end;
end; {TPTCustomListView.DoItemContextMenu}

procedure TPTCustomListView.ItemContextMenu( aItem: TListItem;  var aPos: TPoint;  var aMenu: TPopupMenu );
begin
  if Assigned(OnItemContextMenu) then OnItemContextMenu( self, aItem, aPos, aMenu );
end; {TPTCustomListView.WMContextMenu}

procedure TPTCustomListView.KeyDown( var key: Word;  aShiftState: TShiftState );
begin
  if ((key = VK_APPS) and (aShiftState = [])) or
     ((key = VK_F10) and (aShiftState = [ssShift]))
  then
  begin
    key := 0;
    DoPreItemContextMenu(Point(0,0))
  end;
  inherited;
end; {TPTCustomListView.KeyDown}

{$IFDEF VER9x}
  procedure TPTCustomListView.InsertItem( aItem: TListItem );
  begin
    if Assigned(OnInsert) then OnInsert(self, aItem);
  end; {TPTCustomListView.InsertItem}
{$ENDIF}


procedure TPTCustomListView.CreateWnd;
begin
  inherited;
  Perform( LVM_SETTEXTBKCOLOR, 0, ColorToRGB(Color) );
  Perform( LVM_SETBKCOLOR, 0, ColorToRGB(Color) );
end;

procedure TPTCustomListView.DestroyWnd;
begin
  inherited;
end;

function TPTCustomListView.CreateCustomDraw( const aNMHDR: TNMCustomDraw;  var aResult: Integer ): TPTCustomDraw;
  begin result := TPTLvCustomDraw.Create( aNMHDR, aResult, Font ); end;

function TPTCustomListView.CustomDraw( const aNMHDR: TNMCustomDraw;  var aResult: Integer ): TPTCustomDraw;
var focda, focdexa: Boolean;
    item: TListItem;
  function FindItem: TListItem;
  begin
    if (aNMHDR.dwDrawStage and CDDS_ITEM) = 0 then
      result := nil
    else
      result := Items[aNMHDR.dwItemSpec];
  end;
begin
  result := nil;
  item := nil;
  focda := Assigned(OnCustomDraw);
  focdexa := Assigned(OnCustomDrawEx);

  if focdexa then
  begin
    result := CreateCustomDraw(aNMHDR, aResult);
    item := FindItem;
    OnCustomDrawEx( self, result, item );
  end;

  if focda and ((aNMHDR.dwDrawStage and CDDS_ITEMPREPAINT) = CDDS_ITEMPREPAINT) then
  begin
    if not Assigned(result) then result := CreateCustomDraw(aNMHDR, aResult);
    if not Assigned(item) then item := FindItem;
    OnCustomDraw( self, result, item );
  end;

  if Assigned(result) then
    result.Apply;

  if (aNMHDR.dwDrawStage and CDDS_PREPAINT)<>0 then
    aResult := aResult or CDRF_NOTIFYITEMDRAW;
end; {TPTCustomListView.CustomDraw}



end.

