unit UPTSysFolderDlg; // Copyright  1996-1998 Plasmatech Software Design. All rights reserved.
{
 Shell Control Pack
 Version 1.3d

 Delphi wrapper around the system browse for folder dialog, the SHBrowseForFolder function.

 History
 ==============================================================
 V1.30d --TBA-- No changes.
 V1.30c 16Mar98 C++Builder 3 support.
 V1.30b  7Feb98 No changes.
 V1.30a  7Jan98 No changes.
 V1.30  28Nov97 No changes.
 V1.20b 12Oct97 No changes.
 V1.20a  5Oct97 No changes.
 V1.20   6Sep97 Changed Bool types to Boolean for better C++Builder support.
 V1.10a  6Jul97 Removed HWND params for C++Builder support.
 V1.10  26Jun97 No changes.
 V1.00c 31May97 No changes.
 V1.00b 17May97 Delphi 3 support.
 V1.00  21Apr97 Released version 1.0
}
{$RANGECHECKS OFF} {$OVERFLOWCHECKS OFF} {$WRITEABLECONST OFF}
{$BOOLEVAL OFF}    {$EXTENDEDSYNTAX ON}  {$TYPEDADDRESS ON}

interface
uses Windows, Ole2, ShellApi, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
       UPTShell95, UPTShellUtils;

type TPTSysFolderDlgOption = ( fdoComputers, fdoPrinters, fdoDontGoBelowDomain, fdoReturnFSAncestors,
                               fdoReturnOnlyFSDirs, fdoStatusText );
     TPTSysFolderDlgOptions = set of TPTSysFolderDlgOption;

type TPTSysFolderDlg = class;
     TFDOnInitializedEvent = procedure( aSender: TPTSysFolderDlg;  hwnd: Integer ) of object;
     TFDOnSelChangedEvent  = procedure( aSender: TPTSysFolderDlg;  hwnd: Integer;  pidl: PItemIdList ) of object;

     TPTSysFolderDlg = class(TComponent)
       private
         mhLastHwnd: HWND;
       protected
         mTitle: String;
         mOptions: TPTSysFolderDlgOptions;
         mDomain: TCSIDL;
         mFolderIdList: PItemIdList;
         mOnInitializedProc: TFDOnInitializedEvent;
         mOnSelChangedProc: TFDOnSelChangedEvent;
         mfExecuting: Boolean;  // TRUE during an Execute call.  Some functionality is only available in this state.

        {Property helpers}
         function  GetFolderPath: String;
         function  GetFolderIdList: PItemIdList;

         procedure SetOkEnabled( aValue: Boolean );  virtual;
         procedure SetFolderIdList( aValue: PItemIdList );
         procedure SetFolderPath( aValue: String );
         procedure SetStatusText( aValue: String );  virtual;

        {Hook function and event handlers}
         procedure HookFn( h, umsg, lParam: Integer );

         procedure SelChanged( h: Integer;  pidl: PItemIdList ); dynamic;
         procedure Initialized( h: Integer ); dynamic;

       public
         constructor Create( aOwner: TComponent ); override;
         destructor  Destroy;  override;

         function  Execute: Boolean;  virtual;

         property  Executing: Boolean read mfExecuting;

       {Access these properties from within the OnInitialized or OnSelChanged event handlers}
         property  OkEnabled: Boolean write SetOkEnabled;
         property  StatusText: String write SetStatusText;

       {Access these properties after Execute returns TRUE}
         property  FolderPath: String read GetFolderPath write SetFolderPath;
         property  FolderIdList: PItemIdList read mFolderIdList write SetFolderIdList;

       published         
         property  Domain: TCSIDL read mDomain write mDomain default csidlNone;
         property  Options: TPTSysFolderDlgOptions read mOptions write mOptions;
         property  Title: String read mTitle write mTitle;

         property  OnInitialized: TFDOnInitializedEvent read mOnInitializedProc write mOnInitializedProc;
         property  OnSelChanged: TFDOnSelChangedEvent read mOnSelChangedProc write mOnSelChangedProc;
     end; {TPTSysFolderDlg}


{*****************************************************************************}
implementation
const BFFM_INITIALIZED        = 1;
      BFFM_SELCHANGED         = 2;
      BFFM_SETSTATUSTEXT      = (WM_USER + 100);
      BFFM_ENABLEOK           = (WM_USER + 101);
      BFFM_SETSELECTION       = (WM_USER + 102);


function FDCallback( h: HWND;  umsg: UINT;  lParam: LPARAM;  lData: Pointer ): Integer; stdcall;
begin
  TPTSysFolderDlg(lData).HookFn( h, umsg, lParam );
  result := 0;
end; {FDCallback}


constructor TPTSysFolderDlg.Create( aOwner: TComponent );
begin
  inherited Create( aOwner );
end; {TPTSysFolderDlg.Create}


destructor TPTSysFolderDlg.Destroy;
begin
  if Assigned(mFolderIdList) then ShellMemFree(mFolderIdList);
  inherited Destroy;
end; {TPTSysFolderDlg.Destroy}            


function TPTSysFolderDlg.GetFolderIdList: PItemIdList;
begin
  result := mFolderIdList;
end; {TPTSysFolderDlg.GetFolderIdList}


function TPTSysFolderDlg.GetFolderPath: String;
begin
  result := ShellGetPathFromIdList(mFolderIdList);
  if (Length(result)>0) and (result[ Length(result) ] <> '\') then result := result + '\';
end;


procedure TPTSysFolderDlg.SetOkEnabled( aValue: Boolean );
begin
  if not mfExecuting then raise Exception.CreateFmt( '%s: SetOkEnabled assignment only valid during Execute', [Name] );
  Windows.SendMessage( mhLastHWND, BFFM_ENABLEOK, 0, UINT(aValue) );
end; {TPTSysFolderDlg.SetOkEnabled}


procedure TPTSysFolderDlg.SetFolderIdList( aValue: PItemIdList );
begin
  if Assigned(mFolderIdList) then ShellMemFree(mFolderIdList);
  mFolderIdList := CopyIdList(nil, aValue);
end; {TPTSysFolderDlg.SetFolderIdList}


procedure TPTSysFolderDlg.SetFolderPath( aValue: String );
var p: PItemIdList;
begin
  if Ole2.Succeeded(ShellGetIdListFromPath( aValue, p )) then
  begin
    if Assigned(mFolderIdList) then ShellMemFree(mFolderIdList);
    mFolderIdList := p;
  end;
end; {TPTSysFolderDlg.SetFolderPath}


procedure TPTSysFolderDlg.SetStatusText( aValue: String );
begin
  if not mfExecuting then raise Exception.CreateFmt( '%s: Status text assignment only valid during Execute', [Name] );
  Windows.SendMessage( mhLastHWND, BFFM_SETSTATUSTEXT, 0, UINT( PChar(aValue) ) );
end; {TPTSysFolderDlg.SetStatusText}


procedure TPTSysFolderDlg.HookFn( h, umsg, lParam: Integer );
begin
  mhLastHWND := h;
  case umsg of
    BFFM_INITIALIZED: Initialized(h);
    BFFM_SELCHANGED:  SelChanged(h, PItemIdList(lParam));
  end;
end; {TPTSysFolderDlg.HookFn}


procedure TPTSysFolderDlg.SelChanged( h: Integer;  pidl: PItemIdList );
begin
  if Assigned(mFolderIdList) then ShellMemFree(mFolderIdList);
  mFolderIdList := CopyIdList(nil, pidl);
  if Assigned(mOnSelChangedProc) then mOnSelChangedProc(self, h, mFolderIdList);
end; {TPTSysFolderDlg.SelChanged}


procedure TPTSysFolderDlg.Initialized( h: Integer );
begin
  if Assigned(mOnInitializedProc) then mOnInitializedProc(self, h);
  if Assigned(FolderIdList) then SendMessage( h, BFFM_SETSELECTION, 0, Longint( FolderIdList ) );//PChar(Path_RemoveSlash(FolderPath)) ) );
end; {TPTSysFolderDlg.Initialized}


function TPTSysFolderDlg.Execute: Boolean;
const _Options: array[TPTSysFolderDlgOption] of UINT =
                (BIF_BROWSEFORCOMPUTER, BIF_BROWSEFORPRINTER, BIF_DONTGOBELOWDOMAIN,
                 BIF_RETURNFSANCESTORS, BIF_RETURNONLYFSDIRS, BIF_STATUSTEXT);
var igm: IMalloc;
    bi: TBrowseInfo;
    lpbuffer: array[0..MAX_PATH] of Char;
    pidlDomain: PItemIdList; // PIDL for domain
    pidlBrowse: PItemIdList; // PIDL selected by user
    opt: TPTSysFolderDlgOption;
    hparent: HWND;

    activeWindow: HWND;
    windowList: Pointer;
begin
  if (mfExecuting) then raise Exception.CreateFmt( '%s: Execute already called.', [Name] );
  result := FALSE;

  activeWindow:=0; igm:=nil; pidlDomain:=nil; pidlBrowse:=nil; windowlist:=nil;
  if UPTShell95.SHGetMalloc(igm) <> NOERROR then raise Exception.Create( SysErrorMessage(GetLastError) );
  try
    mfExecuting := TRUE;

    if Assigned(Owner) and (Owner is TWinControl) then
      hparent := TWinControl(Owner).Handle
    else
      hparent := 0;

    if SHGetSpecialFolderLocation( hparent, Integer(Domain), pidlDomain ) <> NOERROR then
      Exit;

    activeWindow := GetActiveWindow;
    windowList := DisableTaskWindows(0);

// Fill in the BROWSEINFO structure
    bi.hwndOwner := hparent;
    bi.pidlRoot := pidlDomain;
    bi.pszDisplayName := lpBuffer;
    bi.lpszTitle := PChar(Title);
    bi.ulFlags := 0;
    for opt := Low(TPTSysFolderDlgOption) to High(TPTSysFolderDlgOption) do
      if opt in Options then
        bi.ulFlags := bi.ulFlags or _Options[opt];
    bi.lpfn := FDCallback;
    bi.lParam := UINT(self);

    pidlBrowse := SHBrowseForFolder(bi);
    if Assigned(pidlBrowse) then
    begin
      // Store the display name and file system path.
      if Assigned(mFolderIdList) then ShellMemFree( mFolderIdList );
      mFolderIdList := CopyIdList(nil, pidlBrowse);
      result := TRUE;
    end
    else
      if Assigned(mFolderIdList) then begin ShellMemFree( mFolderIdList ); mFolderIdList:=nil; end;
  finally
    if Assigned(igm) then
    begin
      if Assigned(pidlDomain) then igm.Free( pidlDomain );
      if Assigned(pidlBrowse) then igm.Free( pidlBrowse );
      igm.Release;
      igm:=nil;
    end;
    mfExecuting := FALSE;
    if Assigned(windowList) then
    begin
      EnableTaskWindows(windowList);
      if activeWindow<>0 then SetActiveWindow( activeWindow );
    end;
  end;
end; {TPTSysFolderDlg.Execute}


end.

