unit UPTSplitter; // Copyright  1996-1998 Plasmatech Software Design. All rights reserved.
{
 Shell Control Pack
 Version 1.3d

 Design time interface:
   Right-click on the splitter to drag it at design time.
   Right-click on either pane to get the design menu.
   Holding shift and right-clicking on the splitter also shows the design menu - useful if panes are obscured.

 History
 ==============================================================
 V1.30d --TBA-- Fixed problem where splitter was in non-full drag mode, when the user dropped the splitter,
                  the OnSplitterDrop event would fire, but the OnChange event would not.
 V1.30c 16Mar98 Fixed problem with sizing when splitter was placed in MDI child window.
                Added Proportion property.
 V1.30b  7Feb98 Fixed problem where splitter position could 'creep' to the left while resizing in Proportional mode.
 V1.30a  7Jan98 Added FullDragMode property.
                Added Proportional property.
                Added OnChange property.
                Fixed redraw flicker (D3 only, see also UPTFrame).
 V1.30  28Nov97 Added public Pane1 and Pane2 properties.
 V1.20b 12Oct97 Added OnSplitterDrag and OnSplitterDrop events.
 V1.20a  5Oct97 No changes.
 V1.20   6Sep97 Fixed problem that prevented splitters from being used in visual form inheritance.
                Fixed problem with SwapPanes at design time.
                Improved splitter dragging with mDragOffset variable.
                Improved occasional paint glitch when dragging splitter.
 V1.10a  6Jul97 C++ Builder support with ifdef VER9x
 V1.10  26Jun97 No changes.
 V1.00c 31May97 No changes.
 V1.00b 17May97 Changes to support Delphi 3 -- look for ifdef ver90/ver100
 V1.00a  1May97 No changes.
 V1.00  21Apr97 Released version 1.0
}
{$RANGECHECKS OFF} {$OVERFLOWCHECKS OFF} {$WRITEABLECONST OFF}
{$BOOLEVAL OFF}    {$EXTENDEDSYNTAX ON}  {$TYPEDADDRESS ON}

{$I PTCompVer.inc}

interface
uses Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs, CommCtrl,
     UPTFrame, ExtCtrls, DsgnIntf;

type TPTSplitter = class;
     TPTPane = class;

     TPTSplitterStyle = (ptstVertical, ptstHorizontal);
     TPTSplitterDragEvent = procedure( aSender: TObject;  aPosition: Integer ) of object;

     TPTFullDragMode = (ptfdOff, ptfdOn, ptfdUser);

     TPTSplitter = class(TCustomControl)
       private
         mPanes: array[0..1] of TPTPane;
         procedure InitPanePos;
         procedure CreatePanes;

        {-- Property helpers --}
         function  GetPane( idx: Integer ): TPTPane;
         function  GetPaneColor( idx: Integer ): TColor;
         function  GetPaneStyle( idx: Integer ): TPTFrameStyle;
         function  GetPaneMinSize( idx: Integer ): Integer;

         procedure SetStyle( aValue: TPTSplitterStyle );
         procedure SetSplitterWidth( aValue: Integer );
         procedure SetPosition( aValue: Integer );
         procedure SetProportion( aValue: Single );
         procedure SetPaneColor( idx: Integer;  aValue: TColor );
         procedure SetPaneStyle( idx: Integer;  aValue: TPTFrameStyle );
         procedure SetPaneMinSize( idx: Integer;  aValue: Integer );
        {=====}

         procedure SetActivePane( paneIndex: Integer );

         procedure MouseDown( aButton: TMouseButton;  aShift: TShiftState;  x,y: Integer ); override;
         procedure MouseMove( aShift: TShiftState;  x, y: Integer ); override;
         procedure MouseUp( aButton: TMouseButton;  aShift: TShiftState;  x, y: Integer ); override;

         procedure CMColorChanged( var aMsg: TMessage ); message CM_COLORCHANGED;

         procedure WMCancelMode( var aMsg: TWMCancelMode ); message WM_CANCELMODE;
         procedure WMSize( var aMsg: TWMSize ); message WM_SIZE;
         procedure WMSetCursor( var aMsg: TWMSetCursor ); message WM_SETCURSOR;
         procedure WMParentNotify( var aMsg: TWMParentNotify ); message WM_PARENTNOTIFY;
         procedure WMCaptureChanged( var aMsg: TMessage ); message WM_CAPTURECHANGED;
       protected
         mStyle: TPTSplitterStyle;
         mSplitterWidth: Integer;
         mPosition: Integer;
         mFullDragMode: TPTFullDragMode;
         mProportional: Boolean;

         mOnResizeProc: TNotifyEvent;
         mOnSplitterDragProc: TPTSplitterDragEvent;
         mOnSplitterDropProc: TPTSplitterDragEvent;
         mOnChangeProc: TNotifyEvent;

         mSplitterProportion: Single; // ratio of left to right. eg .1 = 10% of width/height from left/top

        {-- Drag handling --}
         mfDragging: Boolean;
         mDragPosition: Integer;
         mDragOffset: Integer;
         mDragButton: TMouseButton;  // Button used to initiate dragging
         procedure PaintSizingLine; virtual;
         procedure BeginDrag( p: TPoint );
         procedure ContinueDrag( p: TPoint );
         procedure EndDrag;
         procedure SetDragPosition( pos: Integer );
        {======}

         function  IsFullDrag: Boolean;
         procedure FixProportion;
         function  Extent: Integer;
         procedure InternalSetPosition( aValue: Integer;  aFixProportion: Boolean );

         procedure Change; dynamic;
         procedure Resize; dynamic;
         procedure SplitterDrag( aPosition: Integer ); dynamic;
         procedure SplitterDrop( aPosition: Integer ); dynamic;

        {-- Load and store mechanism --}
{$IFDEF VCL20}  procedure GetChildren(Proc: TGetChildProc); override; {$ENDIF}
{$IFDEF VCL30PLUS} procedure GetChildren(Proc: TGetChildProc; root: TComponent); override; {$ENDIF}
         function  GetChildOwner: TComponent; override;
        {======}

         procedure CreateParams( var p: TCreateParams ); override;
         procedure CreateWnd; override;
         procedure Loaded; override;
         procedure SetName(const NewName: TComponentName); override;

       public
         constructor Create( aOwner: TComponent );  override;
         procedure SwapPanes( afSwapSplit: Bool );

         property Pane1: TPTPane index 0 read GetPane stored FALSE;
         property Pane2: TPTPane index 1 read GetPane stored FALSE;
         property Proportion: Single read mSplitterProportion write SetProportion stored FALSE;
       published
         property Align;
         property Visible;
         property Enabled;
         property Color;
         property ParentColor;
         property ShowHint;
         property ParentShowHint;

         property Width default 200;
         property Height default 140;

         property FullDragMode: TPTFullDragMode read mFullDragMode write mFullDragMode default ptfdUser;
         property Proportional: Boolean read mProportional write mProportional default TRUE;

         property Position: Integer read mPosition write SetPosition default 65;
         property Style: TPTSplitterStyle read mStyle write SetStyle default ptstVertical;
         property SplitterWidth: Integer read mSplitterWidth write SetSplitterWidth default 3;

         property Pane1Color: TColor index 0 read GetPaneColor write SetPaneColor stored FALSE default clBtnFace;
         property Pane1FrameStyle: TPTFrameStyle index 0 read GetPaneStyle write SetPaneStyle stored FALSE default ptfsLowered;
         property Pane1MinSize: Integer index 0 read GetPaneMinSize write SetPaneMinSize stored FALSE default 0;

         property Pane2Color: TColor index 1 read GetPaneColor write SetPaneColor stored FALSE default clBtnFace;
         property Pane2FrameStyle: TPTFrameStyle index 1 read GetPaneStyle write SetPaneStyle stored FALSE default ptfsLowered;
         property Pane2MinSize: Integer index 1 read GetPaneMinSize write SetPaneMinSize stored FALSE default 0;

         property OnChange: TNotifyEvent read mOnChangeProc write mOnChangeProc;
         property OnResize: TNotifyEvent read mOnResizeProc write mOnResizeProc;

         property OnSplitterDrag: TPTSplitterDragEvent read mOnSplitterDragProc write mOnSplitterDragProc;
         property OnSplitterDrop: TPTSplitterDragEvent read mOnSplitterDropProc write mOnSplitterDropProc;

         property OnMouseDown;
         property OnMouseUp;
         property OnMouseMove;
         property OnDragDrop;
         property OnDragOver;
         property TabOrder;
         property TabStop;
         property OnClick;
         property OnDblClick;
     end; {TPTSplitter}


     TPTPane = class(TPTGroup)
       private
         procedure CMDesignHitTest( var aMsg: TCMDesignHitTest ); message CM_DESIGNHITTEST;
       protected
         mMinSize: Integer;
         mIndex: Integer;
         mfDesignActive: Bool;
         procedure Paint; override;
       public
         constructor Create( aOwner: TComponent ); override;
       published
         property FrameStyle default ptfsLowered;
         property MinSize: Integer read mMinSize write mMinSize default 20;
         property Index: Integer read mIndex write mIndex;
     end; {TPTPane}


function Get50PctBmp: TBitmap;


{*****************************************************************************}
implementation

var g50PctBmp: TBitmap = nil;

{ g50PctBmp is an 8x8 monochrome bitmap used as a brush for drawing the splitter bar.
  We draw the image directly to a bitmap to save the hassle of using a resource.
  *.*.*.*.
  .*.*.*.*
  *.*.*.*.
  .*.*.*.*
  *.*.*.*.
  .*.*.*.*
  *.*.*.*.
  .*.*.*.*
}
function Get50PctBmp: TBitmap;
var x,y: Integer;
begin
  if not Assigned(g50PctBmp) then
  begin
    g50PctBmp := TBitmap.Create;
    with g50PctBmp do
    begin
      Monochrome := TRUE;
      Width := 8;
      Height := 8;
      for x := 0 to 7 do
        for y := 0 to 7 do
          if (x and 1 <> y and 1) then
            Canvas.Pixels[x,y] := clWhite
          else
            Canvas.Pixels[x,y] := clBlack;
    end;
  end;
  result := g50PctBmp
end; {Get50PctBmp}



{**************************************
  TPTSplitter
**************************************}
constructor TPTSplitter.Create( aOwner: TComponent );
begin
  inherited;
  Width := 200;
  Height := 140;
  ControlStyle := [csCaptureMouse, csDesignInteractive, csClickEvents];
  mPosition := 65;
  mSplitterWidth := 3;
  mFullDragMode := ptfdUser;
  mProportional := TRUE;
  mSplitterProportion := mPosition / Width;
end; {TPTSplitter.Create}

procedure TPTSplitter.SwapPanes( afSwapSplit: Bool );
var tmp: Pointer;
begin
  tmp := mPanes[0];
  mPanes[0] := mPanes[1];
  mPanes[1] := tmp;

  mPanes[0].Index := 0;
  mPanes[1].Index := 1;

  if afSwapSplit then
    case mStyle of
      ptstVertical:   Position := Width - Position;
      ptstHorizontal: Position := Height - Position;
    end;
  InitPanePos;
end; {TPTSplitter.SwapPanes}

{Assuming the panes have been creaed}
procedure TPTSplitter.InitPanePos;
  procedure DoVertical;
  begin
    mPanes[0].BoundsRect := Rect( 0,0, Position, ClientHeight );
    mPanes[1].BoundsRect := Rect( Position+mSplitterWidth,0, ClientWidth,ClientHeight );
  end; {DoVertical}

  procedure DoHorizontal;
  begin
    mPanes[0].BoundsRect := Rect( 0,0, ClientWidth, Position );
    mPanes[1].BoundsRect := Rect( 0,Position+mSplitterWidth, ClientWidth,ClientHeight );
  end; {DoHorizontal}
begin
  if not Assigned(mPanes[0]) or not HandleAllocated or (csLoading in ComponentState) then Exit; 
  case mStyle of
    ptstVertical:   DoVertical;
    ptstHorizontal: DoHorizontal;
  end;
end; {TPTSplitter.InitPanePos}

procedure TPTSplitter.CreatePanes;
  procedure CreateNew( index: Integer );
  var new: TPTPane;
  begin
    new := TPTPane.Create( Self );
    new.Parent := self;
    new.Index := index;
    new.Name := self.Name + '_pane' + IntToStr(index+1);  // Must name the panes, otherwise visual inheritance won't work.
    new.Caption := '';
    mPanes[index] := new;
  end;
begin
  CreateNew( 0 );
  CreateNew( 1 );
  SetActivePane( 0 );
end; {TPTSplitter.CreatePanes}

function TPTSplitter.GetPane(idx: Integer): TPTPane;
begin
  result := mPanes[idx];
end;

function TPTSplitter.GetPaneColor( idx: Integer ): TColor;
  begin result := mPanes[idx].Color; end;

function TPTSplitter.GetPaneStyle( idx: Integer ): TPTFrameStyle;
  begin result := mPanes[idx].FrameStyle; end;

function TPTSplitter.GetPaneMinSize( idx: Integer ): Integer;
  begin result := mPanes[idx].MinSize; end;

procedure TPTSplitter.SetStyle( aValue: TPTSplitterStyle );
begin
  if (mStyle <> aValue) then
  begin
    mStyle := aValue;
    InitPanePos;
  end;
end; {TPTSplitter.SetStyle}

procedure TPTSplitter.SetSplitterWidth( aValue: Integer );
begin
  if (mSplitterWidth <> aValue) then
  begin
    mSplitterWidth := aValue;
    InitPanePos;
  end;
end; {TPTSplitter.SetSplitterWidth}

procedure TPTSplitter.SetPosition( aValue: Integer );
begin
  InternalSetPosition( aValue, true );
end; {TPTSplitter.SetPosition}

procedure TPTSplitter.SetProportion( aValue: Single );
begin
  mSplitterProportion := aValue;
  Resize; 
end;

procedure TPTSplitter.SetPaneColor( idx: Integer;  aValue: TColor );
begin
  if Assigned(mPanes[idx]) then mPanes[idx].Color := aValue;
end;

procedure TPTSplitter.SetPaneStyle( idx: Integer;  aValue: TPTFrameStyle );
begin
  if Assigned(mPanes[idx]) then mPanes[idx].FrameStyle := aValue;
end;

procedure TPTSplitter.SetPaneMinSize( idx: Integer;  aValue: Integer );
begin
  if Assigned(mPanes[idx]) then mPanes[idx].MinSize := aValue;
  // !! check that the current size isn't now less than min size
end;


{ Selects the active pane. The highlighted active pane is only relevant at design time. It is the
  destination for pasted components and components added with a double-click of the palette. It
  has no other function. }
procedure TPTSplitter.SetActivePane( paneIndex: Integer );
begin
  if Assigned(mPanes[paneIndex]) and (not mPanes[paneIndex].mfDesignActive) then
  begin
    mPanes[paneIndex].mfDesignActive := TRUE;
    mPanes[paneIndex].Invalidate;
    mPanes[paneIndex].SetZOrder( FALSE );

    mPanes[ (paneIndex+1) and 1 ].mfDesignActive := FALSE;
    mPanes[ (paneIndex+1) and 1 ].Invalidate;
  end;
end; {TPTSplitter.SetActivePane}


procedure TPTSplitter.MouseDown( aButton: TMouseButton;  aShift: TShiftState;  x,y: Integer );
begin
  if (mfDragging) then
    EndDrag
  else if (aButton = mbLeft) or ( (csDesigning in ComponentState) and (aButton = mbRight) and not (ssShift in aShift) ) then
  begin
    if not MouseCapture then MouseCapture := TRUE;
    mDragButton := aButton;
    BeginDrag( Point(x,y) );
  end;
end; {TPTSplitter.MouseDown}


procedure TPTSplitter.MouseMove( aShift: TShiftState;  x, y: Integer );
begin
  if mfDragging then
    ContinueDrag( Point(x,y) );
end; {TPTSplitter.MouseMove}


procedure TPTSplitter.MouseUp( aButton: TMouseButton;  aShift: TShiftState;  x, y: Integer );
var
  oldpos: Integer;
begin
  if mfDragging and (aButton = mDragButton) then
  begin
    ContinueDrag( Point(x,y) );
    oldpos := mPosition;
    mPosition := mDragPosition - mDragOffset;
    EndDrag;
    InitPanePos;
    SplitterDrop( mPosition );
    if (oldpos <> mPosition) then
      Change;
  end;
end; {TPTSplitter.MouseUp}

procedure TPTSplitter.CMColorChanged( var aMsg: TMessage );
begin
  inherited;
  Invalidate;
end; {TPTSplitter.CMColorChanged}

procedure TPTSplitter.WMCancelMode( var aMsg: TWMCancelMode );
begin
  EndDrag;
  inherited;
end;

procedure TPTSplitter.WMSize( var aMsg: TWMSize );
begin
  inherited;
  if not (csLoading in ComponentState) then
  begin
    Resize;
    InitPanePos;
  end;
end; {TPTSplitter.WMSize}


procedure TPTSplitter.WMSetCursor( var aMsg: TWMSetCursor );
var p: TPoint;
    r1, r2: TRect;
begin
  Windows.GetCursorPos(p);
  p := ScreenToClient(p);

 // Adjust the hot rectangles for easier dragging
  r1 := mPanes[0].BoundsRect;   r2 := mPanes[1].BoundsRect;
{
  if mStyle = ptstHorizontal then
    begin Dec(r1.bottom,2); Inc(r2.top,2); end
  else // ptstVertical
    begin Dec(r1.right,2); Inc(r2.left,2); end;
}
 // Check mouse position against hot rectangles
  if not (PtInRect(r1, p) or PtInRect(r2, p)) then
  begin
    if mStyle = ptstHorizontal then
      Windows.SetCursor( Screen.Cursors[crVSplit] )
    else // ptstVertical
      Windows.SetCursor( Screen.Cursors[crHSplit] );
    aMsg.Result := 1;
  end
  else
    inherited;
end; {TPTSplitter.WMSetCursor}


procedure TPTSplitter.WMParentNotify( var aMsg: TWMParentNotify );
var c: TPTPane;
    p: TPoint;
begin
  if (csDesigning in ComponentState) then
    if (aMsg.Event = WM_LBUTTONDOWN) then
    begin
      p := Point(aMsg.xPos, aMsg.yPos);
      if PtInRect( mPanes[0].BoundsRect, p ) then
        c := mPanes[0]
      else if PtInRect( mPanes[1].BoundsRect, p ) then
        c := mPanes[1]
      else
      begin
        inherited;
        Exit;
      end;
      SetActivePane( c.Index );
      inherited;
    end
    else
      inherited
  else
    inherited;
end; {TPTSplitter.WMParentNotify}


procedure TPTSplitter.WMCaptureChanged( var aMsg: TMessage );
begin
  if aMsg.lParam <> 0 then EndDrag;
  inherited;
end; {TPTSplitter.WMCaptureChanged}


function TPTSplitter.IsFullDrag: Boolean;
var f: LongBool;
begin
  case mFullDragMode of
    ptfdOff:
      result := FALSE;

    ptfdUser:
      begin
        if SystemParametersInfo( SPI_GETDRAGFULLWINDOWS, 0,@f, 0 ) then
          result := f
        else
          result := FALSE;
      end;

    else // ptfdOn:
      result := TRUE;
  end;

  if mFullDragMode = ptfdOn then
    result := TRUE
  else
end; {TPTSplitter.IsFullDrag}


procedure TPTSplitter.FixProportion;
begin
  mSplitterProportion := Position / Extent;
end; {TPTSplitter.FixProportion}


function TPTSplitter.Extent;
begin
  if Style = ptstVertical then
    result := Width
  else
    result := Height;
end; {TPTSplitter.Extent}


procedure TPTSplitter.InternalSetPosition( aValue: Integer;  aFixProportion: Boolean );
begin
  if Assigned(mPanes[0]) then
  begin
    if aValue < Pane1MinSize then
      aValue := Pane1MinSize
    else if (aValue > (Extent - Pane2MinSize)) then
      aValue := Extent - Pane2MinSize;
  end;

  if (mPosition <> aValue) then
  begin
    mPosition := aValue;
    InitPanePos;
    if aFixProportion then
      FixProportion;
    Change;
  end;
end;


procedure TPTSplitter.Change;
begin
  if Assigned(mOnChangeProc) then
    mOnChangeProc(self);
end; {TPTSplitter.Change}


procedure TPTSplitter.Resize;
begin
  if Proportional then
    InternalSetPosition( Round(Extent * mSplitterProportion), false );
    
  if Assigned(mOnResizeProc) then
    mOnResizeProc(self);
end; {TPTSplitter.Resize}


procedure TPTSplitter.SplitterDrag( aPosition: Integer );
begin
  if Assigned(OnSplitterDrag) then OnSplitterDrag(self, aPosition);
end; {TPTSplitter.SplitterDrag}


procedure TPTSplitter.SplitterDrop( aPosition: Integer );
begin
  FixProportion;
  if Assigned(OnSplitterDrop) then OnSplitterDrop(self, aPosition);
end; {TPTSplitter.SplitterDrop}


procedure TPTSplitter.PaintSizingLine;
var dc: HDC;
    newbrush, oldbrush: HBRUSH;
{$IFDEF VCL20}
    oldp: TPoint;
{$ENDIF}
begin
  dc := GetDCEx(Handle, 0, DCX_LOCKWINDOWUPDATE or DCX_PARENTCLIP );
  try
    SetTextColor( dc, ColorToRGB(clWhite) );
    SetBkColor( dc, ColorToRGB(clBlack) );
    SetBkMode( dc, OPAQUE );
    SetRop2( dc, R2_COPYPEN );
    newbrush := CreatePatternBrush(Get50PctBmp.Handle);
    UnrealizeObject(newbrush);
{$IFDEF VCL20}
    SetBrushOrgEx( dc, 0,0, oldp );
{$ELSE}
    SetBrushOrgEx( dc, 0,0, nil );
{$ENDIF}
    oldbrush := SelectObject( dc, newbrush );

    case Style of
      ptstVertical:   PatBlt( dc, mDragPosition - mDragOffset, 0, mSplitterWidth, ClientHeight, PATINVERT );
      ptstHorizontal: PatBlt( dc, 0, mDragPosition - mDragOffset, ClientWidth, mSplitterWidth, PATINVERT );
    end;
    DeleteObject(SelectObject(dc,oldbrush));
  finally
    ReleaseDC(Handle, dc);
  end;
end; {TPTSplitter.PaintSizingLine}


procedure TPTSplitter.BeginDrag( p: TPoint );
var startpos: Integer;
begin
  if (Style=ptstVertical) then
  begin
    startpos := p.x;
    mDragOffset := p.x - mPanes[0].BoundsRect.Right;
  end
  else //ptstHorizontal
  begin
    startpos := p.y;
    mDragOffset := p.y - mPanes[0].BoundsRect.Bottom;
  end;

  if (startpos - mDragOffset) < mPanes[0].MinSize then startpos := mPanes[0].MinSize
  else if (startpos - mDragOffset) > (Extent - mPanes[1].MinSize) then startpos := (Extent - mPanes[1].MinSize);

  mfDragging := TRUE;
  mDragPosition := -1;
  SetDragPosition( startpos );
end; {TPTSplitter.BeginDrag}


procedure TPTSplitter.ContinueDrag( p: TPoint );
var newpos: Integer;
begin
  if (Style=ptstVertical) then
    newpos := p.x
  else
    newpos := p.y;

  if (newpos - mDragOffset) < mPanes[0].MinSize then newpos := mPanes[0].MinSize + mDragOffset
  else if (newpos - mDragOffset) > (Extent - mPanes[1].MinSize) then newpos := (Extent - mPanes[1].MinSize) + mDragOffset;
  if mDragPosition = newpos then Exit;

  SetDragPosition( newpos );
(* -- optimized paint routine to minimize flicker
var p: TPoint;
    ro, rn, intersect: TRect;
    fIntersection: Bool;
begin
  p := CtoC(FForm, FSplitControl, FSplit);
  ro := Bounds( p.x - (FSplitControl.Width+2) div 2, p.y, (FSplitControl.Width+2), FSplitControl.Height );

  if FVertical then FSplit.Y := Y else FSplit.X := X;

  p := CtoC(FForm, FSplitControl, FSplit);
  rn := Bounds( p.x - (FSplitControl.Width+2) div 2, p.y, (FSplitControl.Width+2), FSplitControl.Height );

  if IntersectRect( intersect, ro, rn ) then
  begin
    SubtractRect( ro, ro,intersect );
    SubtractRect( rn, rn,intersect );
  end;

  if mfFirst then
    mfFirst := FALSE
  else {Erase part}
    with ro do PatBlt( FForm.Canvas.Handle, left, top, right-left, bottom-top, PATINVERT );

  {Draw part}
  with rn do PatBlt( FForm.Canvas.Handle, left, top, right-left, bottom-top, PATINVERT );
*)
end; {TPTSplitter.ContinueDrag}


procedure TPTSplitter.EndDrag;
begin
  if not (mfDragging) then Exit;
  SetDragPosition( -1 );
  mfDragging := FALSE;
  if MouseCapture then MouseCapture := FALSE;
end; {TPTSplitter.EndDrag}


procedure TPTSplitter.SetDragPosition( pos: Integer );
begin
  if IsFullDrag then
  begin
    if pos >= 0 then
    begin
      mDragPosition := pos;
      Position := pos - mDragOffset;
    end;
  end
  else
  begin
    if mDragPosition >= 0 then
      PaintSizingLine;
    mDragPosition := pos;
    if pos >= 0 then
      PaintSizingLine;
  end;
  SplitterDrag( mDragPosition );
end; {TPTSplitter.SetDragPosition}


{$IFDEF VCL20}
procedure TPTSplitter.GetChildren(Proc: TGetChildProc);
var i: Integer;
begin
  for i := Low(mPanes) to High(mPanes) do
    Proc(mPanes[i]);
end; {TPTSplitter.GetChildren}
{$ENDIF VCL20}

{$IFDEF VCL30PLUS}
procedure TPTSplitter.GetChildren(Proc: TGetChildProc;  root: TComponent);
var i: Integer;
begin
  for i := Low(mPanes) to High(mPanes) do
    Proc(mPanes[i]);
end; {TPTSplitter.GetChildren}
{$ENDIF VCL30PLUS}

function TPTSplitter.GetChildOwner: TComponent;
begin
  if csLoading in ComponentState then
    result := Owner
  else
    result := self;
end;


procedure TPTSplitter.CreateParams( var p: TCreateParams );
begin
  inherited;
  p.style := p.style or WS_CLIPCHILDREN or WS_CLIPSIBLINGS;
  p.WindowClass.Style := p.WindowClass.Style and not (CS_HREDRAW or CS_VREDRAW);
end;


procedure TPTSplitter.CreateWnd;
begin
  inherited;
  if (csDesigning in ComponentState) and (ControlCount=0) then
  begin
    CreatePanes;
    InitPanePos;
  end;
end; {TPTSplitter.CreateWnd}


procedure TPTSplitter.Loaded;
var i: Integer;
begin
  inherited;

  for i := 0 to ControlCount-1 do
    if (Controls[i] is TPTPane) then
      mPanes[TPTPane(Controls[i]).Index] := Controls[i] as TPTPane;

  for i := 0 to 1 do
  begin
    mPanes[i].Owner.RemoveComponent( mPanes[i] );
    self.InsertComponent( mPanes[i] );
  end;

  if (csDesigning in ComponentState) then
    SetActivePane(0);

  InitPanePos;

  FixProportion;
end; {TPTSplitter.Loaded}


{ The child panels need to be named otherwise visual inheritance won't work. }
procedure TPTSplitter.SetName(const NewName: TComponentName);
var i: 0..1;
begin
  inherited SetName(NewName);
  if [csLoading, csReading] * ComponentState = [] then
  begin
    for i := Low(i) to High(i) do
      if Assigned(mPanes[i]) then mPanes[i].Name := NewName + '_pane' + IntToStr(i+1);
  end;
end; {TPTSplitter.SetName}




{**************************************
  TPTPane
**************************************}
constructor TPTPane.Create( aOwner: TComponent );
begin
  inherited;
  mMinSize := 20;
  mFrameStyle := ptfsLowered;
end; {TPTPane.Create}


procedure TPTPane.CMDesignHitTest( var aMsg: TCMDesignHitTest );
begin
  inherited;
  aMsg.result := 0;
end;


procedure TPTPane.Paint;
var p: TPen;
    b: TBrush;
begin
  inherited;
  if (csDesigning in ComponentState) and (mfDesignActive) then
    with Canvas do
    begin
      p := TPen.Create;    p.Assign( Pen );
      b := TBrush.Create;  b.Assign( Brush );
      Pen.Style := psDot;
      Pen.Color := clBlack;
      Pen.Mode := pmNot;
      Brush.Style := bsClear;
      with ClientRect do Rectangle( left,top,right,bottom );
      Pen.Assign( p );
      Brush.Assign( b );
      p.Free;
      b.Free;
    end;
end; {TPTPane.Paint}

initialization
  RegisterClass(TPTPane);
finalization
  if Assigned(g50PctBmp) then g50PctBmp.Free;
end.

