unit UPTShellUtils; // Copyright  1996-1998 Plasmatech Software Design. All rights reserved.
{
 Shell Control Pack
 Version 1.3d

 Implements low-level utilities useful for dealing with shell interfaces and structures.
 Also includes utilities for creating and resolving shortcuts.

 History
 ==============================================================
 V1.30d --TBA-- No changes.
 V1.30c 16Mar98 C++Builder 3 support.
 V1.30b  7Feb98 Added GetModuleVersion function and TPTModuleVersion type.
                Added constants, types and variable COMCTL32_VER for determining the version of
                  comctl32.dll at run-time.
                Changed ShellGetSystemImageList to support shell link and network share overlay
                  fix on WinNT/IE4.
 V1.30a  7Jan98 Added ShellGetIconIndexFromExt.
                Added ShellGetIconIndexFromPath.
 V1.30  28Nov97 No changes.
 V1.20b 12Oct97 No changes.
 V1.20a  5Oct97 No changes.
 V1.20   6Sep97 Added PTClsidFromFileType.
                ShellMemAlloc, ShellMemRealloc and ShellMemFree now used a cached IMalloc interface.
                ShellGetIconIndex method added.
 V1.10a  6Jul97 Removed HWND params for C++Builder support.
 V1.10  26Jun97 Fixed CompareAbsIdLists when Desktop folder used as either or both parameters.
 V1.00c 31May97 No significant changes.
 V1.00b 17May97 Delphi 3 support. Adjusted StrretFree to prevent exceptions under WinNT.
 V1.00a  1May97 Added IsWinNT, IsWin95, HasWin95Shell functions.
 V1.00  21Apr97 Released version 1.0
}
{$RANGECHECKS OFF} {$OVERFLOWCHECKS OFF} {$WRITEABLECONST OFF}
{$BOOLEVAL OFF}    {$EXTENDEDSYNTAX ON}  {$TYPEDADDRESS ON}

{$I PTCompVer.inc}

interface
uses Windows, SysUtils, ShellApi, Dialogs, Ole2, Classes,
       UPTShell95;


type TCSIDL = ( csidlDesktop,          // $0000
                  csidl_None1,           // $0001
                csidlPrograms,         // $0002
                csidlControls,         // $0003
                csidlPrinters,         // $0004
                csidlPersonal,         // $0005
                csidlFavorites,        // $0006
                csidlStartup,          // $0007
                csidlRecent,           // $0008
                csidlSendTo,           // $0009
                csidlBitBucket,        // $000A
                csidlStartMenu,        // $000B
                  csidl_None2,           // $000C
                  csidl_None3,           // $000D
                  csidl_None4,           // $000E
                  csidl_None5,           // $000F
                csidlDesktopDirectory, // $0010
                csidlDrives,           // $0011
                csidlNetwork,          // $0012
                csidlNethood,          // $0013
                csidlFonts,            // $0014
                csidlTemplates,        // $0015
                csidlCommonStartMenu,  // $0016
                csidlCommonPrograms,   // $0017
                csidlCommonStartup,    // $0018
                csidlCommonDesktopDirectory, //$0019
                csidlAppData,          // $001a
                csidlPrintHood,        // $001b
                  csidlNone );           // $001c

{-- General utilities. These are not necessarily shell related but are used by more than one Shell Control Pack unit. -- }
function IsWin95: Boolean;
function IsOSR2OrGreater: Boolean; // Returns TRUE if running Win95 OSR2 or higher.
function IsWinNT: Boolean;
function HasWin95Shell: Boolean;

type
  TPTModuleVersion = packed record
    case Integer of
      0: (w1, w2, w3, w4: Word);  // Higher number means more significant - w4=major, w3=minor etc.
      1: (dw1, dw2: Integer);
{$IFNDEF CBUILDER}
      2: (asComp: Comp);          // Treat as a single 64-bit integer
{$ENDIF}
      3: (_1, _2, minor, major: Word);
      4: (_3, version: Integer);
  end;
  PPTModuleVersion = ^TPTModuleVersion;
// Unless you are specifically interested in the build version (w2 or w1) then you would normally
// compare .version members.
                                                             
function GetModuleVersion( const aModuleName: String;  var {out} aVersion: TPTModuleVersion ): Boolean;

{-- Comctl32.dll support --}
const
  COMCTL32_VER471 = (4 shl 16) or 71;         // IE4 version of comctl32
  COMCTL32_VER470 = (4 shl 16) or 70;         // IE3 version of comctl32

var
  COMCTL32_VER: TPTModuleVersion;


{-- Utilities. There is virtually no performance penalty for using these ShellMem* routines compared
    to calling SHGetMalloc yourself - and you don't have to manage the IMalloc interface. }
function ShellMemAlloc(size: Cardinal): Pointer;
procedure ShellMemFree(p: Pointer);
function ShellMemRealloc(p: Pointer;  size: Cardinal): Pointer;

{-- Higher level conversion utils ----}
function ShellGetFolderFromIdList( p: PItemIdList;  var ish: IShellFolder ): HResult;
function ShellGetIdListFromPath( const path: String;  var p: PItemIdList ): HResult;
function ShellGetPathFromIdList( p: PItemIdList ): String;
function ShellGetDisplayPathName( aPathName: String ): String; // Returns the properly cased pathname
function ShellGetSpecialFolderPath( ahwnd: TPTHWND;  csidl: TCSIDL ): String;
function ShellGetSpecialFolderIdList( ahwnd: TPTHWND;  csidl: TCSIDL;  var idlist: PItemIdList ): HResult;
function ShellGetIconIndex( absIdList: PItemIdList;  uFlags: DWORD ): Integer;
function ShellGetIconIndexFromPath( const path: String;  uFlags: DWORD ): Integer;
function ShellGetIconIndexFromExt( const ext: String;  uFlags: DWORD ): Integer;  

type TPTFriendlyNameFlags = (ptfnNormal, ptfnInFolder, ptfnForParsing);
function ShellGetFriendlyNameFromIdList( ishf: IShellFolder;  pidl: PItemIdList;  flags: TPTFriendlyNameFlags ): String;
{
 If ishf=nil, then pidl is an absolute item id list. A temporary IShellFolder for the desktop will
 be created to get the name.
 flags can be any SHGNO constant.

                  File system path     Display name             Notes
                  -------------------- ------------------------ ----------------------------------------
   ptfnNormal     C:\Windows\File.txt  file                     If not showing extensions
                  \\Computer\Share     share on computer
                  C:\                  My Drive (C)             Where C has the volume name My Drive

   ptfnInFolder   C:\Windows\File.txt  file
                  \\Computer\Share     share
                  C:\                  My Drive (C)

   ptfnForParsing C:\Windows\File.txt  C:\Windows\File.txt
                  \\Computer\Share     \\Computer\Share
                  C:\                  C:\
}

type TPTShellIconSize = (ptsizSmall, ptsizLarge);
function ShellGetSystemImageList( aSize: TPTShellIconSize ): THandle;

{-- String utilities -----------------}
function StrretToString( pidl: PItemIdList;  const r: TStrRet ): String;
procedure StrretFree( const r: TStrRet );

function EnsureTrailingCharDB( const aSource: String;  aTrailingChar: Char ): String;

{-- Low-level Pidl Utilities ---------}
function CopyIdList( ishm: IMalloc;  pidl: PItemIdList ): PItemIdList;
function ConcatIdLists( ishm: IMalloc;  aFirst, aSecond: PItemIdList ): PItemIdList;
function IdListLen( pidl: PItemIdList ): Integer;
function CompareAbsIdLists( pidl1, pidl2: PItemIdList ): Integer;
  // Compare absolute (relative to desktop) pidls. Returns <0, 0 or >0. If result=MAXINT then function failed.

{The TPTIdListArray class treats an item id list as an array of items. You can easily process each
 element of the pidl.

 property Item[ idx: Integer ]: PItemIdList;
   The returned id is allocated from shell memory and returned. You don't have to free it. If you
   want to keep it you should use CopyIdList() to make a copy. Each call to Item invalidates the previous
   return value.

 Example:
   procedure DoWork( pa: TPTIdListArray ):
   var p1, p2: PItemIdList;
   begin
     p1 := pa.items[1];
     p2 := pa.items[2]; // !!BUG p1 is now invalid.
     // ... work ...
   end;

  You should instead do this:
   procedure DoWork( pa: TPTIdListArray ):
   var p1, p2: PItemIdList;
   begin
     p1:=nil; p2:=nil;
     try
       p1 := CopyIdList(pa.items[1]);
       p2 := CopyIdList(pa.items[2]);
       // ... work ...
     finally
       if Assigned(p1) then ShellMemFree(p1);
       if Assigned(p2) then ShellMemFree(p2);
     end;
   end;

   Since you will very rarely be processing more that one item at a time, you should very rarely need to
   go to this trouble.

   The GoUp method works in a similar way (it invalidates previous results of GoUp or Item[].
   GoUp(n) removes the last "n" items from the item id list and returns the result. The id list
   being operated on is not affected, hence GoUp() calls on a given TPTIdListArray are NOT cumulative.
 }
type TPTIdListArray = class(TObject)
       protected
         mPidl: PItemIdList;

         mLastItem: PItemIdList;
         function GetCount: Integer;
         function GetItem( idx: Integer ): PItemIdList;
       public
         constructor Create( p: PItemIdList );
         destructor  Destroy; override;

         function GoUp( items: Integer ): PItemIdList;

         property ItemCount: Integer read GetCount;
         property Item[idx: Integer]: PItemIdList read GetItem; default;
     end; {TPTIdListArray}


{-- Shortcuts Utilities --------------}
type  TLinkDataOption = (ldoUseDesc, ldoUseArgs, ldoUseIcon, ldoUseWorkDir, ldoUseHotKey, ldoUseShowCmd);
      TLinkDataOptions = set of TLinkDataOption;

      TLinkData = record
       // Mandatory members
        pathName: String;     // Pathname of original object
        options: TLinkDataOptions; // Set of flags indicating optional member usage

       // Optional members
        desc: String;         // Description of link file (its filename for example)
        args: String;         // Command-line arguments
        iconPath: String;     // Pathname of file containing the icon
        iconIndex: Integer;   // Index of icon in 'iconPath'.  -ve values are resource ids (i think?).
        workingDir: String;   // Working directory when process starts
        showCmd: Integer;     // How to show the initial window
        hotkey: Word;         // Hot key for the link

       // Output members - used by ResolveShortcut, not used by CreateShortcut or CreateQuickShortcut
        idList: PItemIdList;
        w32fd: TWin32FindData;
      end; {TLinkData}

function CreateShortcut( const linkPathName: String;  const linkData: TLinkData ): HResult;
function CreateQuickShortcut( const linkPathName, targetPathName: String ): HResult;

function ResolveShortcut( const linkPathName: String;  var linkData: TLinkData;  afWantIdList: Boolean ): HResult;

//--
function PTClsidFromFileType( aExtension: String;  var aCLSID: TGUID ): Boolean;

// You can pass these 'verbs' to TPTShellList.DoCommandForAllSelected and TPTShellTree.DoCommandForNode
// to execute the relevant menu command. These strings are never displayed and are language independent.

// -- These commands are available to most folders --
const PTSH_CMDS_DELETE = 'delete';
      PTSH_CMDS_PASTE  = 'paste';
      PTSH_CMDS_CUT    = 'cut';
      PTSH_CMDS_COPY   = 'copy';
      PTSH_CMDS_PROPERTIES = 'properties';
      PTSH_CMDS_EXPLORE = 'explore';  // Opens a Windows explorer
      PTSH_CMDS_OPEN   = 'open';      // Opens a Windows explorer folder-view
      PTSH_CMDS_FIND   = 'find';      // Open the find dialog
      PTSH_CMDS_LINK   = 'link';      // Same as 'Create Shortcut' menu item

// -- Commands used by Dialup Networking
const PTSH_CMDS_DUN_CREATE = 'create';    // Create new connection wizard
      PTSH_CMDS_DUN_CONNECT = 'connect';  // Connect

// -- These are commands that have no 'verb' but have tested to have the same ID under Win95, Win95OSR2, WinNT4 and IE4
// -- So the id is pretty reliable, but there a no promises!
// -- The other thing to remember is that the IDs are reused for different types of folders. So make sure you use the
// -- right command ID with the right kind of folder.
const PTSH_CMDID_FORMAT = PChar(35);    // Only on drive root directory folders


{ Substitutes strings of the form %1,%2 etc. into aFmtStr and returns the result. }
function FormatStrPos( aFmtStr: String;  data: array of String ): String;

{ Given a command line string 'ins' returns all the parameters, taking into account
  quotes and double-byte characters. }
procedure ParametizeCmdLineDB( const ins: String;  outs: TStrings );

{DBCS enabled TrimRight}
function TrimRightDB( s: String ): String;

{Copies possible DB char at 'aPos' from 'aSource' and appends to 'aDest', incrementing 'aPos' by 1 or 2.}
procedure CopyCharDB( var aPos: Integer;  aSource: String;  var aDest: String );


{*********************************************************}
implementation
uses CommCtrl, Registry;


{-- General Utilities ----------------}
var gOSVer: TOSVersionInfo;
    g_IShm: IMalloc = nil;

function IsWin95: Boolean;
  begin result := (gOSVer.dwPlatformId = VER_PLATFORM_WIN32_WINDOWS); end;

function IsOSR2OrGreater: Boolean; // Returns TRUE if running Win95 OSR2 or higher.
  begin result := IsWin95 and (LoWord(gOsVer.dwBuildNumber) > 1000); end;

function IsWinNT: Boolean;
  begin result := (gOSVer.dwPlatformId = VER_PLATFORM_WIN32_NT); end;

function HasWin95Shell: Boolean;
  begin result := IsWin95 or (IsWinNT and (gOSVer.dwMajorVersion >= 4)); end;


{ Returns the 64-bit version information for the given DLL in 'aVersion'. Returns true if the function
  succeeds, false if failed. }
function GetModuleVersion( const aModuleName: String;  var {out} aVersion: TPTModuleVersion ): Boolean;
var
  pinfo: Pointer;
  dummy: Integer;
  size: Integer;
  pffinfo: PVSFixedFileInfo;
begin
  result := false;
  pinfo := nil;
  try
    size := GetFileVersionInfoSize( PChar(aModuleName), dummy );
    if size=0 then Exit;

    GetMem( pinfo, size );
    if not GetFileVersionInfo( PChar(aModuleName), 0, size, pinfo ) then Exit;

    if not VerQueryValue( pinfo, '\', Pointer(pffinfo), dummy ) then Exit;
    aVersion.dw2 := pffinfo.dwFileVersionMS;
    aVersion.dw1 := pffinfo.dwFileVersionLS;
    result := true;
  finally
    FreeMem(pinfo);
  end;
end; {GetModuleVersion}



{-- Global Utilities -----------------}
function ShellMemAlloc(size: Cardinal): Pointer;
begin
  result := g_IShm.Alloc(size);
end; {ShellMemAlloc}


function ShellMemRealloc(p: Pointer;  size: Cardinal): Pointer;
begin
  if Assigned(p) then
    result := g_IShm.Realloc( p, size )
  else
    result := nil;
end; {ShellMemRealloc}


procedure ShellMemFree(p: Pointer);
begin
  if Assigned(p) then
    g_IShm.Free(p);
end; {ShellMemFree}


function ShellGetFolderFromIdList( p: PItemIdList;  var ish: IShellFolder ): HResult;
var idesk: IShellFolder;
begin
  ish := nil;
  result := SHGetDesktopFolder(idesk); if (result <> S_OK) then Exit;

  if (p = nil) or (PWord(p)^ = 0) then
    ish := idesk // If 'p' refers to the desktop, there is no binding required.
  else
    try
      result := idesk.BindToObject( p, nil, @IID_IShellFolder, Pointer(ish) );
    finally
      idesk.Release;
    end;
end; {ShellGetFolderFromIdList}


function ShellGetIdListFromPath( const path: String;  var p: PItemIdList ): HResult;
var ishf: IShellFolder;
    im: IMalloc;
    wtmp: array[0..MAX_PATH] of WideChar;
    chEaten: DWORD;
    dwAttributes: DWORD;
begin
  result := S_FALSE;
  im:=nil; ishf:=nil; p:=nil;
  try
    try
      result := SHGetMalloc(im);  if result <> S_OK then Exit;
      result := SHGetDesktopFolder(ishf);  if result <> S_OK then Exit;

      StringToWideChar( path, wtmp, High(wtmp) );
      result := ishf.ParseDisplayName(0,nil,wtmp,chEaten,p,dwAttributes);
      if result <> S_OK then Exit;
    except
      if Assigned(im) and Assigned(p) then im.Free(p);
      raise;
    end;
  finally
    if Assigned(im) then im.Release;
    if Assigned(ishf) then ishf.Release;
  end;
end; {ShellGetIdListFromPath}


function ShellGetPathFromIdList( p: PItemIdList ): String;
var sz: array[0..MAX_PATH] of Char;
    w: Word;
begin
  if (p=nil) then
  begin // If p is the desktop, do the 'make it work' hack
    w := 0;
    p := Pointer(@w);
  end;

  if SHGetPathFromIdList( p, @sz[0] ) then
    SetString( result, sz, Strlen(sz) )
  else
    result := '';
end; {ShellGetPathFromIdList}


// Returns the properly cased pathname
function ShellGetDisplayPathName( aPathName: String ): String;
var ish: IShellFolder;
    im: IMalloc;
    wtmp: array[0..MAX_PATH] of WideChar;
    chEaten: DWORD;
    dwAttributes: DWORD;
    pidl: PItemIdList;
begin
  result:='';
  im:=nil; ish:=nil; pidl:=nil;
  try
    if SHGetMalloc(im) <> S_OK then Exit;
    if SHGetDesktopFolder(ish) <> S_OK then Exit;

    StringToWideChar( aPathName, wtmp, High(wtmp) );
    if ish.ParseDisplayName(0,nil,wtmp,chEaten,pidl,dwAttributes) <> S_OK then Exit;
    SetLength( result, MAX_PATH );
    SHGetPathFromIdList(pidl, PChar(result));
    SetLength( result, StrLen(PChar(result)) );

  finally
    if Assigned(im) and Assigned(pidl) then im.Free(pidl);
    if Assigned(im) then im.Release;
    if Assigned(ish) then ish.Release;
  end;
end; {ShellGetDisplayPathname}


function ShellGetSpecialFolderPath( ahwnd: TPTHWND;  csidl: TCSIDL ): String;
var pidl: PItemIdList;
begin
  if Ole2.Succeeded(SHGetSpecialFolderLocation( ahwnd, Integer(csidl), pidl )) then
  begin
    result := ShellGetPathFromIdList(pidl);
    ShellMemFree(pidl);
  end
  else
    result := '';
end; {ShellGetSpecialFolderPathname}


function ShellGetSpecialFolderIdList( ahwnd: TPTHWND;  csidl: TCSIDL;  var idlist: PItemIdList ): HResult;
  begin result := SHGetSpecialFolderLocation(ahwnd, Integer(csidl), idlist ); end;


function ShellGetIconIndex( absIdList: PItemIdList;  uFlags: DWORD ): Integer;
var shfi: TSHFileInfo;
begin
  try
    SHGetFileInfo( PChar(absIdList),0, shfi, Sizeof(TSHFileInfo), SHGFI_PIDL or SHGFI_SYSICONINDEX or uflags );
    result := shfi.iIcon;
  except
    result := 0;  // Corrupt files or broken icon shell extensions can cause raise exceptions that need to be caught
  end;
end; {ShellGetIconIndex}


function ShellGetIconIndexFromPath( const path: String;  uFlags: DWORD ): Integer;
var shfi: TSHFileInfo;
begin
  try
    SHGetFileInfo( PChar(path),0, shfi, Sizeof(TSHFileInfo), SHGFI_SYSICONINDEX or uFlags );
    result := shfi.iIcon;
  except
    result := 0;  // Corrupt files or broken icon shell extensions can cause raise exceptions that need to be caught
  end;
end; {ShellGetIconIndexFromPath}


function ShellGetIconIndexFromExt( const ext: String;  uFlags: DWORD ): Integer;
var shfi: TSHFileInfo;
begin
  try
    SHGetFileInfo( PChar(ext),0, shfi, Sizeof(TSHFileInfo), SHGFI_SYSICONINDEX or SHGFI_USEFILEATTRIBUTES or uflags );
    result := shfi.iIcon;
  except
    result := 0;  // Corrupt files or broken icon shell extensions can cause raise exceptions that need to be caught
  end;
end; {ShellGetIconIndexFromExt}


function ShellGetFriendlyNameFromIdList( ishf: IShellFolder;  pidl: PItemIdList;  flags: TPTFriendlyNameFlags ): String;
const _F: array[TPTFriendlyNameFlags] of DWORD = (SHGDN_NORMAL, SHGDN_INFOLDER, SHGDN_FORPARSING);
var strret: TStrRet;
    dw: DWord;
  procedure DoDesktop;
  var idsk: IShellFolder;
  begin
    SHGetDesktopFolder(idsk);
    result := ShellGetFriendlyNameFromIdList(idsk, pidl, flags);
    idsk.Release;
  end;
begin
  if Assigned(ishf) then
  begin
    strret.uType := STRRET_CSTR;
    result := '';
    dw := ishf.GetDisplayNameOf( pidl, _F[flags], strret );
    if Ole2.Succeeded(dw) then
    begin
      result := StrretToString(pidl, strret);
      StrretFree(strret);
    end;
  end
  else
    DoDesktop;
end; {ShellGetFriendlyNameFromIdList}


var
  gOverlaysAppliedAlready: Boolean = false;

{ On WinNT/IE4 we get a completely private image list, with no overlays. We need to load up the shell
  link and network share images, add them to the list and make them overlays before we return. }
function ShellGetSystemImageList( aSize: TPTShellIconSize ): THandle;
{
  var r: TRegistry;
      sourceLink: String;
      sourceShare: String;
  begin
    r := TRegistry.Create;
    try
      r.RootKey := HKEY_LOCAL_MACHINE;
      if not r.Open('Software\Microsoft\Windows\CurrentVersion\explorer\Shell Icons',false) then Exit;
      sourceLink := r.ReadString( '29' );
//      sourceShare := r.ReadString( '32' ); ??
    finally
      r.Free;
    end;
}

  procedure ApplyOverlaysToSysImageLists;
  const
    SIC_SHARING_HAND = 29;
    SIC_SHORTCUT = 30;

    LR_SHARED = $8000;
  var
    h16, h32: THandle;
    hicon: THandle;
    hShell32: THandle;
    shfi: TSHFileInfo;
  begin
    hShell32 := Windows.LoadLibrary( 'shell32.dll' );
    if (hShell32=0) then Exit;
    try
      h16 := SHGetFileInfo( 'C:\',0, shfi,Sizeof(TSHFileInfo), SHGFI_SYSICONINDEX or SHGFI_SMALLICON );
      h32 := SHGetFileInfo( 'C:\',0, shfi,Sizeof(TSHFileInfo), SHGFI_SYSICONINDEX or SHGFI_LARGEICON );
        // Have to apply the icons to both image lists simultaneously

      hicon := Windows.LoadImage( hShell32, PChar(SIC_SHARING_HAND), IMAGE_ICON, 16,16, LR_SHARED );
      ImageList_SetOverlayImage( h16, ImageList_AddIcon(h16, hicon), 1 );
      DeleteObject(hicon);

      hicon := Windows.LoadImage( hShell32, PChar(SIC_SHARING_HAND), IMAGE_ICON, 32,32, LR_SHARED );
      ImageList_SetOverlayImage( h32, ImageList_AddIcon(h32, hicon), 1 );
      DeleteObject(hicon);

      hicon := Windows.LoadImage( hShell32, PChar(SIC_SHORTCUT), IMAGE_ICON, 16,16, LR_SHARED );
      ImageList_SetOverlayImage( h16, ImageList_AddIcon(h16, hicon), 2 );
      DeleteObject(hicon);

      hicon := Windows.LoadImage( hShell32, PChar(SIC_SHORTCUT), IMAGE_ICON, 32,32, LR_SHARED );
      ImageList_SetOverlayImage( h32, ImageList_AddIcon(h32, hicon), 2 );
      DeleteObject(hicon);

      gOverlaysAppliedAlready := true;
    finally
      if (hShell32<>0) then FreeLibrary(hShell32);
    end;
  end;

const _VALS: array[TPTShellIconSize] of DWORD = (SHGFI_SMALLICON, SHGFI_LARGEICON);
var shfi: TSHFileInfo;
begin
  if IsWinNT and (COMCTL32_VER.version = COMCTL32_VER471) and not gOverlaysAppliedAlready then
    ApplyOverlaysToSysImageLists;
  result := SHGetFileInfo( 'C:\',0, shfi,Sizeof(TSHFileInfo), SHGFI_SYSICONINDEX or _VALS[aSize] );
end; {ShellGetSystemImageList}


function StrretToString( pidl: PItemIdList;  const r: TStrRet ): String; //!!DOC
begin
  case r.uType of
    STRRET_CSTR:
      result := r.cstr;

    STRRET_OFFSET:
      if Assigned(pidl) then
        SetString( result, PChar( UINT(pidl)+r.uOffset ), StrLen(PChar(UINT(pidl)+r.uOffset)) );

    STRRET_WSTR:
      result := WideCharToString( r.pOleStr );
  end;
end; {StrretToString}


procedure StrretFree( const r: TStrRet ); //!!DOC
begin
  if (r.uType = STRRET_WSTR) and Assigned( r.pOleStr ) then CoTaskMemFree( r.pOleStr );
end; {StrretFree}


{ Ensures that 'aTrailingChar' is in fact the last character in the string.
  Exception: If aSource is empty then the trailing character is NOT appended.
  Double-byte safe under Delphi 3. } 
function EnsureTrailingCharDB( const aSource: String;  aTrailingChar: Char ): String;
begin
{$IFDEF VCL30PLUS}
  if (Length(aSource)>0) and (ByteType(aSource, Length(aSource)) = mbSingleByte) and (aSource[Length(aSource)] <> aTrailingChar) then
    result := aSource + aTrailingChar
  else
    result := aSource;
{$ELSE}
  if (Length(aSource)>0) and (aSource[Length(aSource)] <> aTrailingChar) then
    result := aSource + aTrailingChar
  else
    result := aSource;
{$ENDIF}
end;


function IdListLen( pidl: PItemIdList ): Integer;
var p: Pointer;
begin
  result := 0;
  if not Assigned(pidl) then Exit;
  p := pidl;
  while PSHItemID(p).cb <> 0 do
  begin
    Inc( result, PSHItemID(p).cb );
    Inc( PByte(p), PSHItemID(p).cb );
  end;
  Inc( result, 2 ); // terminator
end; {IdListLen}


function CompareAbsIdLists( pidl1, pidl2: PItemIdList ): Integer;
var ishf: IShellFolder;
    dw: DWORD;
begin
  result := MAXINT; // MAXINT means failure in this case

 // Special cases, IShellFolder.CompareIDs doesn't work well with empty pidls
  if not Assigned(pidl1) and not Assigned(pidl2) then
    result := 0
  else if not Assigned(pidl1) or not Assigned(pidl2) then
    result := -1
  else if (PWord(pidl1)^=0) and (PWord(pidl2)^=0) then
    result := 0
  else if (PWord(pidl1)^=0) or (PWord(pidl2)^=0) then
    result := -1
  else if Succeeded(SHGetDesktopFolder(ishf)) then
    try
      dw := ishf.CompareIDs( 0, pidl1, pidl2 );
      if Succeeded(dw) then
        result := Smallint(Ole2.ResultCode(dw));
    finally
      ishf.Release;
    end;
end; {CompareAbsIdLists}



{You must free the returned IDlist with the passed allocator - aFirst and/or aSecond can be nil.}
function ConcatIdLists( ishm: IMalloc;  aFirst, aSecond: PItemIdList ): PItemIdList;
var flen, slen: Integer;
begin
  flen := IdListLen(aFirst) -2; // not including terminator
  if (flen<0) then flen:=0;
  slen := IdListLen(aSecond) -2; // not including terminator
  if (slen<0) then slen:=0;
  if Assigned(ishm) then
    result := ishm.Alloc( flen+slen+2 ) // +2 for null-terminator
  else
    result := ShellMemAlloc( flen+slen+2 );
  if Assigned(result) then
  begin
    CopyMemory( result, aFirst, flen );
    CopyMemory( Pointer( UINT(result)+flen ), aSecond, slen );
    PWord( UINT(result)+flen+slen )^ := 0; // null-terminator at end the list
  end;
end; {ConcatIdLists}


function CopyIdList( ishm: IMalloc;  pidl: PItemIdList ): PItemIdList;
var len: Integer;
begin
  len := IdListLen(pidl) -2;
  if (len<0) then len:=0;
  if Assigned(ishm) then
    result := ishm.Alloc( len+2 )
  else
    result := ShellMemAlloc( len+2 );
  if Assigned(result) then
  begin
    CopyMemory( result, pidl, len );
    PWord( UINT(result)+len )^ := 0; // null-terminator to end the list
  end;
end; {CopyIdList}


{**************************************
  TPTIdListArray
**************************************}
constructor TPTIdListArray.Create( p: PItemIdList );
begin
  inherited Create;
  mPidl := p;
end; {TPTIdListArray.Create}

destructor TPTIdListArray.Destroy;
begin
  if Assigned(mLastItem) then ShellMemFree(mLastItem);
  inherited;
end; {TPTIdListArray.Destroy}

procedure RaiseOutOfRange(idx: Integer);
  begin raise Exception.Create( Format('TPTIdListArray - index %d out of range',[idx]) ); end;

function TPTIdListArray.GoUp( items: Integer ): PItemIdList;
var size: Integer;
    p: Pointer;
    curItem, endItem: Integer;
begin
  if Assigned(mLastItem) then begin ShellMemFree(mLastItem); mLastItem:=nil; end;

  p := mPidl;
  result := nil;
  if (p = nil) then Exit;

  endItem := (ItemCount - items);
  size:=0;
  curItem:=0;
  while (PWord(p)^ <> 0) and (curItem <> endItem) do
  begin
    Inc( size, PWord(p)^ );
    Inc( PByte(p), PWord(p)^ );
    Inc( curItem )
  end;
  
  if (curItem = enditem) then
  begin
    mLastItem := ShellMemAlloc(size+2);
    ZeroMemory( mLastItem, size+2 );
    Move( mPidl^, mLastItem^, size );
  end
  else // RaiseOutOfRange(curitem);
    mLastItem := nil;
  result := mLastItem;
end; {TPTIdListArray.GoUp}

function TPTIdListArray.GetCount: Integer;
var p: Pointer;
begin
  p := mPidl;
  result := 0;
  if (p = nil) then Exit;
  while PWord(p)^ <> 0 do
  begin
    Inc( PByte(p), PWord(p)^ );
    Inc( result );
  end;
end; {TPTIdListArray.GetCount}

function TPTIdListArray.GetItem( idx: Integer ): PItemIdList;
var p: Pointer;
    curidx: Integer;
begin
  if Assigned(mLastItem) then begin ShellMemFree(mLastItem); mLastItem:=nil; end;
  p := mPidl;

  if (p = nil) or (idx < 0) or ( (idx >= ItemCount) and (ItemCount<>0) ) then
    RaiseOutOfRange(idx);

  curidx := 0;
  while (PWord(p)^ <> 0) and (curidx <> idx) do
  begin
    Inc( PByte(p), PWord(p)^ );
    Inc( curidx );
  end;

  if (curidx = idx) then
  begin
    mLastItem := ShellMemAlloc( PWord(p)^ + 2 );
    Move( p^, mLastItem^, PWord(p)^ );
    PWord( UINT(mLastItem) + PWord(p)^ )^ := 0;
  end
  else
    mLastItem := nil;
  result := mLastItem;
end; {TPTIdListArray.GetItem}
// == END of TPTIdListArray ============ //


function CreateShortcut( const linkPathName: String;  const linkData: TLinkData ): HResult;
var ish: IShellLink;
    ips: IPersistFile;
    wsz: array[0..MAX_PATH] of WideChar;
begin
  ish:=nil;  ips:=nil;
  try
    result := CoCreateInstance( CLSID_ShellLink, nil, CLSCTX_INPROC_SERVER, IID_IShellLink, ish ); if Failed(result) then Exit;
    result := ish.QueryInterface( IID_IPersistFile, ips );  if Failed(result) then Exit;

   // Initialise the shortcut
    ish.SetPath( PChar(linkData.pathName) );
    if (ldoUseDesc in linkData.options) then ish.SetDescription( PChar(linkData.desc) );
    if (ldoUseArgs in linkData.options) then ish.SetArguments( PChar(linkData.args) );
    if (ldoUseIcon in linkData.options) then ish.SetIconLocation( PChar(linkData.iconPath), linkData.iconIndex );
    if (ldoUseWorkdir in linkData.options) then ish.SetWorkingDirectory( PChar(linkData.workingDir) );
    if (ldoUseHotKey in linkData.options) then ish.SetHotKey( linkData.hotkey );
    if (ldoUseShowCmd in linkData.options) then ish.SetShowCmd( linkData.showCmd );

   // Now save the shortcut to disk
    result := ips.Save( StringToWideChar(linkPathName, wsz, High(wsz)), TRUE );  if result<>S_OK then Exit;
  finally
    if Assigned(ips) then ips.Release;
    if Assigned(ish) then ish.Release;
  end;
end; {CreateShortcut}


function CreateQuickShortcut( const linkPathName, targetPathName: String ): HResult;
var ld: TLinkData;
begin
  ld.pathName := targetPathName;
  ld.options := [ldoUseDesc, ldoUseWorkDir];
  ld.desc := targetPathName;
  ld.workingDir := ExtractFilePath(targetPathName);
  result := CreateShortcut( linkPathName, ld );
end; {CreateQuickShortcut}


function ResolveShortcut( const linkPathName: String;  var linkData: TLinkData;  afWantIdList: Boolean ): HResult;
var ishl: IShellLink;
    ipf: IPersistFile;
    tmpsz: array[0..MAX_PATH] of Char;
    wsz: array[0..MAX_PATH] of WideChar;
begin
  ishl:=nil; ipf:=nil;
  try
    result := CoCreateInstance(CLSID_ShellLink, nil, CLSCTX_INPROC_SERVER, IID_IShellLink, ishl);  if Ole2.Failed(result) then Exit;
    result := ishl.QueryInterface( IID_IPersistFile, ipf );    if Ole2.Failed(result) then Exit;
    result := ipf.Load( StringToWideChar(linkPathName, @wsz[0], High(wsz)), STGM_READ );   if Ole2.Failed(result) then Exit;

    result := ishl.Resolve( 0, SLR_ANY_MATCH ); if Ole2.Failed(result) then Exit;

    result := ishl.GetPath( @tmpsz[0], sizeof(tmpsz), linkData.w32fd, 0 ); if Ole2.Failed(result) then Exit;
    linkData.pathName := String(tmpsz);

    result := ishl.GetArguments( @tmpsz[0], sizeof(tmpsz) ); if Ole2.Failed(result) then Exit;
    linkData.args := String(tmpsz);

    result := ishl.GetDescription( @tmpsz[0], sizeof(tmpsz) ); if Ole2.Failed(result) then Exit;
    linkData.desc := String(tmpsz);

    result := ishl.GetIconLocation( @tmpsz[0], sizeof(tmpsz), linkData.iconIndex ); if Ole2.Failed(result) then Exit;
    linkData.iconPath := String(tmpsz);

    result := ishl.GetWorkingDirectory( @tmpsz[0], sizeof(tmpsz) ); if Ole2.Failed(result) then Exit;
    linkData.workingDir := String(tmpsz);

    result := ishl.GetShowCmd( linkData.showCmd ); if Ole2.Failed(result) then Exit;
    result := ishl.GetHotKey( linkData.hotKey ); if Ole2.Failed(result) then Exit;

    if afWantIdList then
    begin
      result := ishl.GetIdList( linkData.idList ); if Ole2.Failed(result) then Exit;
    end
    else
      linkData.idList := nil;
  finally
    if Assigned(ipf) then ipf.Release;
    if Assigned(ishl) then ishl.Release;
  end;
end; {ResolveShortcut}


// Given a file extension (include the '.'), returns the CLSID of the associated object - if any.
function PTClsidFromFileType( aExtension: String;  var aCLSID: TGUID ): Boolean;
  const CLSID_KEY = 'CLSID';
var r: TRegistry;
    s: String;
    wca: array[0..79] of WideChar;
begin
  result := FALSE;
  r := TRegistry.Create;
  try
    r.RootKey := HKEY_CLASSES_ROOT;
    if not r.KeyExists( aExtension ) then Exit;
    if not r.OpenKey( aExtension, FALSE ) then Exit;

    s := r.ReadString( '' ); // Try default value first
    if AnsiCompareText( Copy(s,1,6), 'clsid\' )=0 then
    begin
      StringToWideChar( Copy(s, 7, $FF), @wca[0], High(wca) );
      result := Succeeded(CLSIDFromString( @wca[0], aCLSID ));
      Exit;
    end;

    if r.KeyExists( CLSID_KEY ) then
    begin
      if not r.OpenKey( CLSID_KEY, FALSE ) then Exit;
      StringToWideChar( r.ReadString(''), @wca[0], High(wca) );
      result := Succeeded(CLSIDFromString( @wca[0], aCLSID ));
      Exit;
    end;

    r.CloseKey;
    if not r.OpenKey( s, FALSE ) then Exit;
    if r.KeyExists( CLSID_KEY ) then
    begin
      if not r.OpenKey( CLSID_KEY, FALSE ) then Exit;
      StringToWideChar( r.ReadString(''), @wca[0], High(wca) );
      result := Succeeded(CLSIDFromString( @wca[0], aCLSID ));
      Exit;
    end;
  finally
    r.Free;
  end;
end; {PTClsidFromFileType}


function FormatStrPos( aFmtStr: String;  data: array of String ): String;
var i: Integer;
    params: array[0..49] of PChar;
    pBuff: PChar;
    len: UINT;
begin
  if (aFmtStr='') then begin result := ''; Exit; end;
  if High(data) > High(params) then raise Exception.Create( 'FormatStrPos: Too many substitution strings' );
  for i := 0 to High(data) do params[i] := PChar( data[i] );
  pbuff := nil;
  len := FormatMessage( FORMAT_MESSAGE_FROM_STRING or FORMAT_MESSAGE_ARGUMENT_ARRAY or FORMAT_MESSAGE_ALLOCATE_BUFFER,
                        PChar(aFmtStr), 0,0,
                        Pointer(@pBuff), 256 {MINIMUM size!},
                        @params[0] );
  try
//    if (len = 0) then raise Exception.Create( 'FormatStrPos: FormatMessage failed.  '+SysErrorMessage(GetLastError)+#13'"'+aFmtStr+'"' );
//    if (pBuff = nil) then raise Exception.Create( 'FormatStrPos: FormatMessage failed.  '+SysErrorMessage(GetLastError)+#13'"'+aFmtStr+'"' );
//    result := String(pBuff);
    if (len=0) or (pbuff=nil) then
      result := ''
    else
      result := String(pBuff);
  finally
    if len<>0 then LocalFree( UINT(pBuff) );
  end;
end; {FormatStrPos}


procedure ParametizeCmdLineDB( const ins: String;  outs: TStrings );
{$IFNDEF VCL30PLUS}
  function AnsiPos(const Substr, S: string): Integer;
  begin
    result := Pos(Substr, S);
  end;
{$ENDIF}
const WHITESPACE = [' ',#9];
var curs: String;
    state: (sNormal, sInQuotes, sInWhitespace);
    inpos: Integer;
    curchar: Char;
    fIsDBCS: Boolean;
begin
  curs := '';
  state := sInWhitespace;
  inpos := 1;
  while (inpos <= Length(ins)) do
  begin
    curchar := ins[inpos];
    fIsDBCS := IsDBCSLeadByte( Byte(curchar) );
    case state of
      sNormal:
        begin
          if (curchar = '"') or (curchar in WHITESPACE) then
          begin
            curs := TrimRightDB(curs);
            if Length(curs)>0 then
            begin
              outs.Add( curs );
              curs := '';
            end;
            if curchar = '"' then
              state := sInQuotes
            else
              state := sInWhitespace;
            Inc( inpos, 1 );
          end
          else
            CopyCharDB( inpos, ins, curs );
        end;

      sInQuotes:
        begin
          if not fIsDBCS and (curchar = '"') then
          begin
            curs := TrimRightDB(curs);
            if Length(curs)>0 then
            begin
              outs.Add( curs );
              curs := '';
            end;
            state := sInWhitespace;
            Inc( inpos );
          end
          else
            CopyCharDB( inpos, ins, curs );
        end;

      sInWhitespace:
        begin
          if not fIsDBCS then
          begin
            if (curchar = '"') then
            begin
              curs := '';
              state := sInQuotes;
            end
            else if not (curchar in WHITESPACE) then
            begin
              curs := curchar;
              state := sNormal;
            end;
            Inc( inpos, 1 );
          end
          else // fIsDBCS
          begin
            CopyCharDB( inpos, ins, curs );
            state := sNormal;
          end;
        end;
    end; {case}
  end; {while}

  curs := TrimRightDB(curs);
  if Length(curs)>0 then
    outs.Add( curs );
end; {ParametizeCmdLineDB}


{DBCS enabled TrimRight}
function TrimRightDB( s: String ): String;
const WHITESPACE_SET = [#0..#31, ' '];
var curpos, lastspace, endpos: Integer;
begin
  endpos := Length(s);
  lastspace := endpos;
  curpos := 1;
  while (curpos <= endpos) do
  begin
    if IsDBCSLeadByte( Byte(s[curpos]) ) then
    begin
      lastspace := endpos;
      Inc( curpos, 2 )
    end
    else if (s[curpos] in WHITESPACE_SET) then
    begin
      if lastspace = endpos then
        lastspace := curpos-1;
      Inc( curpos );
    end
    else
    begin
      lastspace := endpos;
      Inc( curpos );
    end;
  end;
  result := Copy(s, 1, lastspace);
end; {TrimRightDB}


procedure CopyCharDB( var aPos: Integer;  aSource: String;  var aDest: String );
begin
  if IsDBCSLeadByte(Byte(aSource[aPos])) then
  begin
    aDest := aDest + aSource[aPos] + aSource[aPos+1];
    Inc( aPos, 2 );
  end
  else
  begin
    aDest := aDest + aSource[aPos];
    Inc( aPos );
  end;
end; {CopyCharDB}






initialization
  gOSVer.dwOSVersionInfoSize := SizeOf(TOSVersionInfo);
  Windows.GetVersionEx( gOsVer );
  SHGetMalloc( g_IShm );
  if not GetModuleVersion( 'comctl32.dll', COMCTL32_VER ) then
    ZeroMemory( @COMCTL32_VER, Sizeof(COMCTL32_VER) );
  if Assigned(g_IShm) then g_IShm.Release;
end.

