unit UPTFrame; // Copyright  1996-1998 Plasmatech Software Design. All rights reserved.
{
 Shell Control Pack
 Version 1.3d

 Implements controls that frame a screen region with one of 11 different styles.
 TPTGroup is a windowed control that can be the parent of other controls. It works
 like a group box but with a choice of frame styles. TPTFrame is non-windowed and
 cannot be the parent of other controls - although sibling controls can be placed
 over TPTFrame.

 You should choose the control that is most applicable to each situation. If you need
 to contain other controls (as in a groupbox), use TPTGroup. If you are mearly framing
 a screen region then you might use TPTFrame. Being non-windowed TPTFrame uses much
 fewer resources (and no window handles) compared with TPTGroup.

 The Draw* functions used to draw the frames are exposed, so you can draw similar
 frames anywhere you want. Many Plasmatools components use these exposed Draw*
 functions to provide embedded frame functionality.

 History
 ==============================================================
 V1.30d --TBA-- No changes.
 V1.30c 16Mar98 C++Builder 3 support.
                Added DefaultDrawing property and OnPaint event to TPTGroup.
 V1.30b  7Feb98 Added compiler directives.
 V1.30a  7Jan98 Fixed flicker problem with TPTGroup (D3 only) - especially when used
                  in splitter controls.
                The border is now a true non-client area - see WM_NCCALCSIZE and WM_NCPAINT.
                Added TPTCustomGroup.EnableAllChildren method.
 V1.30  28Nov97 No changes.
 V1.20b 12Oct97 No changes.
 V1.20a  5Oct97 No changes.
 V1.20   6Sep97 Removed package warnings.
t                Added TPTCustomFrame and TPTCustomGroup base classes.
                Fixed OnClick and OnDblClick properties of TPTFrame and TPTGroup.
 V1.10a  6Jul97 No changes.
 V1.10  26Jun97 No changes.
 V1.00c 31May97 No changes.
 V1.00b 17May97 Delphi 3 support.
 V1.00a  1May97 No changes.
 V1.00  21Apr97 Released version 1.0
}
{$RANGECHECKS OFF} {$OVERFLOWCHECKS OFF} {$WRITEABLECONST OFF}
{$BOOLEVAL OFF}    {$EXTENDEDSYNTAX ON}  {$TYPEDADDRESS ON}

interface
uses Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs;

type TPTFrameStyle = ( ptfsNone, ptfsGroup, ptfsLowered, ptfsRaised, ptfsDint, ptfsBump,
                       ptfsSingle, ptfsHorzLine, ptfsHorzEdge, ptfsVertLine, ptfsVertEdge );

     TPTFramePaintEvent = procedure( aSender: TObject;  aCanvas: TCanvas ) of object; 

     TPTCustomGroup = class(TCustomControl)
       private
         procedure CMEnabledChanged( var msg: TMessage ); message CM_ENABLEDCHANGED;
         procedure CMTextChanged( var aMsg: TMessage); message CM_TEXTCHANGED;
         procedure CMColorChanged( var aMsg: TMessage ); message CM_COLORCHANGED;
         procedure CMFontChanged( var aMsg: TMessage ); message CM_FONTCHANGED;

         procedure SetAlignment( aValue: TAlignment );
         procedure SetFrameSpace( aValue: Integer );
         procedure SetFrameStyle( aValue: TPTFrameStyle );

         procedure WMEraseBkgnd( var aMsg: TWMEraseBkgnd ); message WM_ERASEBKGND;
         procedure WMNcCalcSize( var aMsg: TWMNCCalcSize ); message WM_NCCALCSIZE;
         procedure WMNcPaint( var aMsg: TWMNCPaint ); message WM_NCPAINT;
       protected
         mAlignment: TAlignment;
         mDefaultDrawing: Boolean;
         mFrameSpace: Integer;
         mFrameChanged: Boolean;
         mFrameStyle: TPTFrameStyle;
         mOnPaint: TPTFramePaintEvent;

         procedure CreateParams( var Params: TCreateParams );  override;

         procedure RepaintNonClient;

         procedure PaintNonClient( c: TCanvas; const aBoundsRect: TRect ); virtual;
         procedure Paint; override;

         property Alignment: TAlignment read mAlignment write SetAlignment default taCenter;
         property DefaultDrawing: Boolean read mDefaultDrawing write mDefaultDrawing default true;
         property FrameSpace: Integer read mFrameSpace write SetFrameSpace default 2;
         property FrameStyle: TPTFrameStyle read mFrameStyle write SetFrameStyle default ptfsGroup;
         property OnPaint: TPTFramePaintEvent read mOnPaint write mOnPaint;
         property Width default 64;
         property Height default 48;
       public
         constructor Create( aOwner: TComponent ); override;
         procedure EnableAllChildren( afEnable: Boolean );
     end; {TPTCustomGroup}


     TPTGroup = class(TPTCustomGroup)
       published
         property Alignment;
         property DefaultDrawing;
         property FrameSpace;
         property FrameStyle;
       //
         property Align;
         property Caption;
         property Color;
         property Font;
         property Enabled;
         property ParentColor;
         property ParentFont;
         property ParentShowHint;
         property PopupMenu;
         property ShowHint;
         property Visible;
         property Width;
         property Height;
       //
         property OnClick;
         property OnDblClick;
         property OnMouseDown;
         property OnMouseMove;
         property OnMouseUp;
         property OnEnter;
         property OnExit;
         property OnPaint;
     end; {TPTGroup}


     TPTCustomFrame = class(TGraphicControl)
       protected
         mAlignment: TAlignment;
         mDefaultDrawing: Boolean;
         mFrameSpace: Integer;
         mFrameStyle: TPTFrameStyle;
         mOnPaint: TPTFramePaintEvent;

         procedure SetAlignment( aValue: TAlignment );
         procedure SetFrameSpace( aValue: Integer );
         procedure SetFrameStyle( aValue: TPTFrameStyle );

         procedure CMTextChanged( var aMsg: TMessage); message CM_TEXTCHANGED;
         procedure CMColorChanged( var aMsg: TMessage ); message CM_COLORCHANGED;
         procedure CMFontChanged( var aMsg: TMessage ); message CM_FONTCHANGED;

         procedure InvalidateInterior;

         procedure Paint;  override;

         property Alignment: TAlignment read mAlignment write SetAlignment default taCenter;
         property DefaultDrawing: Boolean read mDefaultDrawing write mDefaultDrawing default true;
         property FrameSpace: Integer read mFrameSpace write SetFrameSpace default 2;
         property FrameStyle: TPTFrameStyle read mFrameStyle write SetFrameStyle default ptfsLowered;
         property Width default 64;
         property Height default 48;
         property OnPaint: TPTFramePaintEvent read mOnPaint write mOnPaint;
       public
         constructor Create( aOwner: TComponent ); override;
     end; {TPTCustomFrame}


     TPTFrame = class(TPTCustomFrame)
       published
         property Alignment;
         property DefaultDrawing;
         property FrameSpace;
         property FrameStyle;
         property OnPaint;
       //
         property Align;
         property Caption;
         property Color;
         property Font;
         property ParentColor;
         property ParentFont;
         property ParentShowHint;
         property PopupMenu;
         property ShowHint;
         property Visible;
         property Width;
         property Height;
       //
         property OnClick;
         property OnDblClick;
         property OnMouseDown;
         property OnMouseMove;
         property OnMouseUp;
     end;



procedure DrawBumpFrame( c: TCanvas;  const r: TRect );
procedure DrawDintFrame( c: TCanvas;  const r: TRect );
procedure DrawGroupFrame( c: TCanvas;  const ar: TRect;  const caption: String;  color: TColor;  enabled: Boolean );
procedure DrawHorzEdge( c: TCanvas;  const r: TRect;  c1, c2: TColor );
procedure DrawHorzLine( c: TCanvas;  const r: TRect;  c1, c2: TColor );
procedure DrawLoweredFrame( c: TCanvas;  const r: TRect );
procedure DrawRaisedFrame( c: TCanvas;  const r: TRect );
procedure DrawSingleFrame( c: TCanvas;  const r: TRect );
procedure DrawVertEdge( c: TCanvas;  const r: TRect;  c1, c2: TColor );
procedure DrawVertLine( c: TCanvas;  const r: TRect;  c1, c2: TColor );

procedure DrawFrame( fs: TPTFrameStyle; c: TCanvas;  const r: TRect );

procedure AdjustRectForFrame( fs: TPTFrameStyle;  var r: TRect );

{*********************************************************}
implementation

procedure DrawLoweredFrame( c: TCanvas;  const r: TRect );
begin
  with r do
  begin
    c.Pen.Color := clBtnShadow;
    c.Polyline( [ Point(left,bottom-1), Point(left,top), Point(right,top) ] );

    c.Pen.Color := cl3dDkShadow; {?}
    c.PolyLine( [ Point(left+1,bottom-2), Point(left+1,top+1), Point(right-1,top+1) ] );

    c.Pen.Color := clBtnFace;
    c.PolyLine( [ Point(left+1,bottom-1), Point(right-1,bottom-1), Point(right-1,top) ] );

    c.Pen.Color := clBtnHighlight;
    c.PolyLine( [ Point(left,bottom), Point(right,bottom), Point(right,top-1) ] );
  end;
end; {DrawLoweredFrame}

procedure DrawRaisedFrame( c: TCanvas;  const r: TRect );
begin
  with r do
  begin
    c.Pen.Color := clBtnHighlight;
    c.Polyline( [ Point(left,bottom-1), Point(left,top), Point(right,top) ] );

    c.Pen.Color := clBtnFace;
    c.Polyline( [ Point(left+1,bottom-2), Point(left+1,top+1), Point(right-1,top+1) ] );

    c.Pen.Color := clBtnShadow;
    c.Polyline( [ Point(left+1,bottom-1), Point(right-1,bottom-1), Point(right-1,top) ] );

    c.Pen.Color := cl3dDkShadow; {?}
    c.Polyline( [ Point(left,bottom), Point(right,bottom), Point(right,top-1) ] );
  end;
end; {DrawRaisedFrame}

procedure DrawDintFrame( c: TCanvas;  const r: TRect );
begin
  with r do
  begin
    c.Pen.Color := clBtnShadow;
    c.Polyline( [ Point(left,bottom-1), Point(left,top), Point(right,top) ] );

    c.Pen.Color := clBtnHighlight;
    c.Polyline( [ Point(left,bottom), Point(right,bottom), Point(right,top-1) ] );
  end;
end; {DrawDintFrame}

procedure DrawBumpFrame( c: TCanvas;  const r: TRect );
begin
  with r do
  begin
    c.Pen.Color := clBtnHighlight;
    c.Polyline( [ Point(left,bottom-1), Point(left,top), Point(right,top) ] );

    c.Pen.Color := clBtnShadow;
    c.Polyline( [ Point(left,bottom), Point(right,bottom), Point(right,top-1) ] );
  end;
end; {DrawBumpFrame}

procedure DrawSingleFrame( c: TCanvas;  const r: TRect );
begin
  with r do
  begin
    c.Pen.Color := clWindowFrame;
    c.Rectangle( left,top, right+1,bottom+1 );
  end;
end; {DrawSingleFrame}

procedure DrawGroupFrame( c: TCanvas;  const ar: TRect;  const caption: String;  color: TColor;  enabled: Boolean );
  procedure DoDrawText( var r: TRect );
  begin
    DrawTextEx(c.Handle, PChar(caption), Length(caption), R, DT_LEFT or DT_SINGLELINE or DT_END_ELLIPSIS, nil);
  end;
var H: Integer;
    R: TRect;
begin
  with c do
  begin
    Brush.Color := color;
    Fillrect( Rect(ar.left, ar.top, ar.right+1, ar.bottom+1) );

    if Caption <> '' then
      h := TextHeight('0')
    else
      h := 2;
    R := ar; //Rect(0, H div 2 - 1, Width, Height);
    Inc( r.top, h div 2 -1 );

    Brush.Style := bsClear;
    Inc(R.Left);
    Inc(R.Top);
    Pen.Color := clBtnHighlight;
    with r do Rectangle(left,top,right+1,bottom+1);
    OffsetRect(R, -1, -1);
    Pen.Color := clBtnShadow;
    with r do Rectangle(left,top,right+1,bottom+1);

    if caption <> '' then
    begin
      r.top := ar.top;  r.left := r.left+8;  r.bottom := r.top+h;  r.right := r.right-8;
//      DrawTextEx(Handle, PChar(caption), Length(caption), R, DT_LEFT or DT_SINGLELINE or DT_CALCRECT or DT_END_ELLIPSIS, nil);
      if enabled then
      begin
        Brush.Color := Color;
        DoDrawText(r);
      end
      else
      begin
        Brush.Color := Color;
        OffsetRect( r, +1,+1 );
        Font.Color := clBtnHighlight;
        DoDrawText(r);

        Brush.Style := bsClear;
        OffsetRect( r, -1,-1 );
        Font.Color := clBtnShadow;
        DoDrawText(r);
      end;
    end;
  end;
end; {DrawGroupFrame}

procedure DrawVertLine( c: TCanvas;  const r: TRect;  c1, c2: TColor );
begin
  with c,r do
  begin
    Pen.Color := c1;    Polyline( [ Point(left, top), Point(left, bottom+1) ] );
    Pen.Color := c2;    Polyline( [ Point(left+1, top), Point(left+1, bottom+1) ] );
  end;
end; {DrawVertLine}

procedure DrawVertEdge( c: TCanvas;  const r: TRect;  c1, c2: TColor );
begin
  with c,r do
  begin
    Pen.Color := c1;
      Polyline( [ Point(left, top), Point(left, bottom+1) ] );
      Polyline( [ Point(right-1, top), Point(right-1, bottom+1) ] );

    Pen.Color := c2;
      Polyline( [ Point(left+1, top), Point(left+1, bottom+1) ] );
      Polyline( [ Point(right, top), Point(right, bottom+1) ] );
  end;
end; {DrawVertEdge}

procedure DrawHorzLine( c: TCanvas;  const r: TRect;  c1, c2: TColor );
begin
  with c,r do
  begin
    Pen.Color := c1;  Polyline( [ Point(left,top), Point(right+1,top) ] );
    Pen.Color := c2;  Polyline( [ Point(left,top+1), Point(right+1,top+1) ] );
  end;
end; {DrawHorzLine}

procedure DrawHorzEdge( c: TCanvas;  const r: TRect;  c1, c2: TColor );
begin
  with c,r do
  begin
    Pen.Color := c1;
      Polyline( [ Point(left,top), Point(right+1,top) ] );
      Polyline( [ Point(left,bottom-1), Point(right+1,bottom-1) ] );

    Pen.Color := c2;
      Polyline( [ Point(left,top+1), Point(right+1,top+1) ] );
      Polyline( [ Point(left,bottom), Point(right+1,bottom) ] );
  end;
end; {DrawHorzEdge}

procedure DrawFrame( fs: TPTFrameStyle; c: TCanvas;  const r: TRect );
begin
  case fs of
    ptfsNone:    ;
    ptfsGroup:   DrawGroupFrame( c, r, '', c.Brush.Color, TRUE );
    ptfsLowered: DrawLoweredFrame( c, r );
    ptfsRaised:  DrawRaisedFrame( c, r );
    ptfsDint:    DrawDintFrame( c, r );
    ptfsBump:    DrawBumpFrame( c, r );
    ptfsSingle:  DrawSingleFrame( c, r );
    ptfsHorzLine: DrawHorzLine( c, r, clBtnShadow, clBtnHighlight );
    ptfsHorzEdge: DrawHorzEdge( c, r, clBtnShadow, clBtnHighlight );
    ptfsVertLine: DrawVertLine( c, r, clBtnShadow, clBtnHighlight );
    ptfsVertEdge: DrawVertEdge( c, r, clBtnShadow, clBtnHighlight );
  end;
end; {DrawFrame}

procedure AdjustRectForFrame( fs: TPTFrameStyle;  var r: TRect );
begin
  case fs of
    ptfsNone:     ;

{$IFDEF PTDEBUG}
    ptfsGroup: raise Exception.Create( 'UPTFrame.AdjustRectForFrame cannot be called for ptfsGroup style.' );
{$ENDIF}

    ptfsLowered, ptfsRaised:        InflateRect(r, -2, -2);
    ptfsDint, ptfsBump, ptfsSingle: InflateRect(r, -1, -1);
    ptfsHorzLine:                   Inc( r.top, 2 );
    ptfsHorzEdge:                   InflateRect( r, 0, -2 );
    ptfsVertLine:                   Inc( r.left, 2 );
    ptfsVertEdge:                   InflateRect( r, -2, 0 );
  end;
end;


procedure DoDrawText( c: TCanvas; var r: TRect;  flags: Word;  s: String );
begin
  if (flags and DT_CALCRECT <> 0) and ((s = '') or (flags and DT_NOPREFIX = 0) and
    (s[1] = '&') and (s[2] = #0)) then s := s + ' ';
  DrawTextEx(c.Handle, PChar(s), Length(s), r, flags, nil);
end;


const Alignments: array[TAlignment] of Word = (DT_LEFT, DT_RIGHT, DT_CENTER);


{**************************************
  TPTCustomGroup
**************************************}
constructor TPTCustomGroup.Create( aOwner: TComponent );
begin
  inherited;
  ControlStyle := [csAcceptsControls, csDoubleClicks, csClickEvents];
  Width := 64;
  Height := 48;
  mDefaultDrawing := true;
  mFrameStyle := ptfsGroup;
  mFrameSpace := 2;
  mAlignment := taCenter;
end; {TPTCustomGroup.Create}

procedure TPTCustomGroup.EnableAllChildren( afEnable: Boolean );
var i: Integer;
begin
  for i := 0 to ControlCount-1 do
    Controls[i].Enabled := afEnable;
end; {TPTCustomGroup.EnableAllChildren}

procedure TPTCustomGroup.WMEraseBkgnd( var aMsg: TWMEraseBkgnd );
begin
  if not (csDesigning in ComponentState) and not DefaultDrawing then
    aMsg.result := 1
  else
    inherited;
end;

procedure TPTCustomGroup.WMNcCalcSize( var aMsg: TWMNCCalcSize );
var r: TRect;
    c: TCanvas;
begin
  inherited;
  r := aMsg.CalcSize_Params.rgrc[0];
  if mFrameStyle = ptfsGroup then
  begin
    if Caption <> '' then
    begin
      c := TCanvas.Create;
      try
        c.Handle := GetDC(0);
        c.Font := Font;
        Inc( r.top, c.TextHeight('0') );
      finally
        c.Free;
      end
    end
    else
      Inc( r.top, 2 );
    Inc( r.left, 2 );
    Dec( r.bottom, 2 );
    Dec( r.right, 2 );
  end
  else
    AdjustRectForFrame( mFrameStyle, r );

  if (Caption<>'') or (csDesigning in ComponentState) or (mFrameChanged) then
  begin
    aMsg.Result := aMsg.Result or WVR_REDRAW;
    mFrameChanged := FALSE;
  end;

  aMsg.CalcSize_Params.rgrc[0] := r;
  if aMsg.CalcValidRects then
    aMsg.CalcSize_Params.rgrc[1] := r;
end;

procedure TPTCustomGroup.WMNcPaint( var aMsg: TWMNCPaint );
var oldh: THandle;
    rectClient, rectWindow: TRect;
begin
  inherited;
  oldh := Canvas.Handle;
  try
    Canvas.Handle := GetWindowDC(Handle);
    Canvas.Font := Font;
    Canvas.Brush := Brush;
    GetWindowRect( Handle, rectWindow );       // BoundsRect won't be updated yet, so we have to use Windows API
    Windows.GetClientRect( Handle, rectClient );
    MapWindowPoints( Handle, HWND_DESKTOP, rectClient, 2 );
    OffsetRect( rectClient, -rectWindow.left, -rectWindow.top );
    OffsetRect( rectWindow, -rectWindow.left, -rectWindow.top );
    with rectClient do ExcludeClipRect( Canvas.Handle, left, top, right, bottom );
    PaintNonClient( Canvas, rectWindow );
  finally
    ReleaseDC(Handle, Canvas.Handle);
    Canvas.Handle := oldh;
  end;
end;

procedure TPTCustomGroup.SetAlignment( aValue: TAlignment );
begin
  mAlignment := aValue;
  Invalidate;
end;

procedure TPTCustomGroup.SetFrameSpace( aValue: Integer );
begin
  mFrameSpace := aValue;
  Invalidate;
end;

procedure TPTCustomGroup.SetFrameStyle( aValue: TPTFrameStyle );
begin
  if (mFrameStyle <> aValue) then
  begin
    mFrameStyle := aValue;
    mFrameChanged := TRUE;
    RecreateWnd;
{
    RepaintNonClient;
    Invalidate;
    if HandleAllocated then
    begin
      Windows.GetClientRect( Handle, r );
      AlignControls(nil,r);
    end;
}
  end;
end;

procedure TPTCustomGroup.CMEnabledChanged( var msg: TMessage );
begin
  inherited;
  Invalidate;
  EnableAllChildren( Enabled );
  if (mFrameStyle=ptfsGroup) then
    RepaintNonClient;
end;

procedure TPTCustomGroup.CMTextChanged(var aMsg: TMessage);
begin
  inherited;
  Invalidate;
  if (mFrameStyle=ptfsGroup) then //and (Length(Caption) <= 1) then
    RepaintNonClient;
end;

procedure TPTCustomGroup.CMColorChanged( var aMsg: TMessage );
begin
  inherited;
  Invalidate;
  if (mFrameStyle=ptfsGroup) then
    RepaintNonClient;
end;

procedure TPTCustomGroup.CMFontChanged(var aMsg: TMessage);
begin
  inherited;
  if (mFrameStyle=ptfsGroup) then
    RepaintNonClient
  else
    Invalidate;
end;

procedure TPTCustomGroup.CreateParams( var Params: TCreateParams );
begin
  inherited;
  with Params.WindowClass do style := style and not (CS_HREDRAW or CS_VREDRAW);
  Params.ExStyle := Params.ExStyle or WS_EX_CONTROLPARENT;
end;

procedure TPTCustomGroup.RepaintNonClient;
begin
  if HandleAllocated then
    SetWindowPos( Handle, 0,0,0,0,0, SWP_FRAMECHANGED or SWP_DRAWFRAME or SWP_NOMOVE or SWP_NOZORDER or SWP_NOSIZE )
end;

procedure TPTCustomGroup.PaintNonClient( c: TCanvas; const aBoundsRect: TRect );
var r: TRect;
begin
  with c do //Canvas do
  begin
    Brush.Color := Color;
    Brush.Style := bsClear;
    r := aBoundsrect;
    Dec(r.right); Dec(r.bottom);
    Font := Self.Font;

    case mFrameStyle of
      ptfsNone: ;

      ptfsGroup:
        begin
          DrawGroupFrame( c, r, Caption, Color, Enabled );
          Exit;
        end;

      ptfsLowered: DrawLoweredFrame( c, r );
      ptfsRaised:  DrawRaisedFrame( c, r );
      ptfsDint:    DrawDintFrame( c, r );
      ptfsBump:    DrawBumpFrame( c, r );
      ptfsSingle:  DrawSingleFrame( c, r );
      ptfsHorzLine: DrawHorzLine( c, r, clBtnShadow, clBtnHighlight );
      ptfsHorzEdge: DrawHorzEdge( c, r, clBtnShadow, clBtnHighlight );
      ptfsVertLine: DrawVertLine( c, r, clBtnShadow, clBtnHighlight );
      ptfsVertEdge: DrawVertEdge( c, r, clBtnShadow, clBtnHighlight );
    end;
  end;
end;

procedure TPTCustomGroup.Paint;
var r: TRect;
begin
  with Canvas do
  begin
    Canvas.Font := self.Font;

    if (Color <> clNone) then
    begin
      Brush.Color := Self.Color;
      Brush.Style := bsSolid;
//      FillRect(ClientRect);
    end;

    if DefaultDrawing then
      if mFrameStyle <> ptfsGroup then
      begin
        r := ClientRect;
        InflateRect( r, -mFrameSpace, 0 );
        DoDrawText( Canvas, r, Alignments[mAlignment] or DT_VCENTER or DT_SINGLELINE or DT_END_ELLIPSIS, Caption );
      end;

    if Assigned(OnPaint) then
      OnPaint(self, Canvas);
  end;
end; {TPTCustomGroup.Paint}


{**************************************
  TPTCustomFrame
**************************************}
constructor TPTCustomFrame.Create( aOwner: TComponent );
begin
  inherited;
  ControlStyle := ControlStyle + [csDoubleClicks, csOpaque, csClickEvents];
  Width := 64;
  Height := 48;
  mDefaultDrawing := true;
  mFrameStyle := ptfsLowered;
  mFrameSpace := 2;
  mAlignment := taCenter;
end; {TPTCustomFrame.Create}

procedure TPTCustomFrame.SetAlignment( aValue: TAlignment );
  begin mAlignment := aValue; Invalidate; end;

procedure TPTCustomFrame.SetFrameSpace( aValue: Integer );
  begin mFrameSpace := aValue; Invalidate; end;

procedure TPTCustomFrame.SetFrameStyle( aValue: TPTFrameStyle );
  begin mFrameStyle := aValue; Invalidate; end;

procedure TPTCustomFrame.InvalidateInterior;
var r: TRect;
begin
  if Assigned(Parent) and Parent.HandleAllocated then
  begin
    r := BoundsRect;
    AdjustRectForFrame( mFrameStyle, r );
    InvalidateRect( Parent.Handle, @r, FALSE );
  end;
end;

procedure TPTCustomFrame.CMTextChanged(var aMsg: TMessage);
  begin inherited; InvalidateInterior; end;

procedure TPTCustomFrame.CMColorChanged( var aMsg: TMessage );
  begin inherited; InvalidateInterior; end;

procedure TPTCustomFrame.CMFontChanged(var aMsg: TMessage);
  begin inherited; InvalidateInterior; end;

procedure TPTCustomFrame.Paint;
var r: TRect;
begin
  with Canvas do
  begin
    if (Color <> clNone) then
    begin
      Brush.Color := Self.Color;
      Brush.Style := bsSolid;
      if DefaultDrawing or (csDesigning in ComponentState) then
        FillRect(ClientRect);
    end;

    Brush.Style := bsClear;
    r := ClientRect;
    Dec(r.right); Dec(r.bottom);

    Font := Self.Font;

    case mFrameStyle of
      ptfsNone: ;

      ptfsGroup:
        begin
          DrawGroupFrame( Canvas, r, Caption, Color, Enabled );
          Exit;
        end;

      ptfsLowered: DrawLoweredFrame( Canvas, r );
      ptfsRaised:  DrawRaisedFrame( Canvas, r );
      ptfsDint:    DrawDintFrame( Canvas, r );
      ptfsBump:    DrawBumpFrame( Canvas, r );
      ptfsSingle:  DrawSingleFrame( Canvas, r );
      ptfsHorzLine: DrawHorzLine( Canvas, r, clBtnShadow, clBtnHighlight );
      ptfsHorzEdge: DrawHorzEdge( Canvas, r, clBtnShadow, clBtnHighlight );
      ptfsVertLine: DrawVertLine( Canvas, r, clBtnShadow, clBtnHighlight );
      ptfsVertEdge: DrawVertEdge( Canvas, r, clBtnShadow, clBtnHighlight );
    end;

    if DefaultDrawing then
    begin
      InflateRect( r, -mFrameSpace, 0 );
      DoDrawText( Canvas, r, Alignments[mAlignment] or DT_VCENTER or DT_SINGLELINE or DT_END_ELLIPSIS, Caption );
    end;

    if Assigned(OnPaint) then
      OnPaint( self, Canvas );
  end;
end; {TPTCustomFrame.Paint}

end.

