(*

	 This is a demonstration on how to build your own archive utility
	 using multiple compression / decompression components from the
   ZipTV component package for the Borland Delphi development language.

   The amount of code contained in this demo is very little when
   realized that this demo actually preforms the same fuction
   as 13 seperate applications!  It may take a little time to
   understand the concept of ZipTV's object hierarchy when using
   multiple compression / decompression components, but once
   understood it becomes rather simple.

   The TZipTV component is used for visual archive management of
   multiple compression & decompression components.

   There are no restrictions placed on the use of the code contained
   herein.  We have not attempted to build a full archive utility in
   this demonstration, but have attempted to provide the basic shell
   to prime ideas.

   ************************************************************

   Click on light when RED, to cancel the current operation.

   ************************************************************

   ZipTV OBJECT HIERARCHY

   The most important feature on which to focus when using multiple
   compression / decompression components in ZipTV is the packages
   object hierarchy.

   The object heirarchy of ZipTV components is as follows:

   TZipCommon - | - TZipTV
                | - TZipSearch
                | - TUnSFX
   				  |
                | - TUnBase  - |  ( all decompression components & TZipKey )
                |              |
                |              | - TZipKey
                |              |
                |              | - TUnBh
                |              | - TUnZip
                |				  | - TUnArj
                |				  | - ...etc.
                |
                |
                |- TCompBase - |  ( all compression components )
                |              |
                |              | - TBlakHole
                |              | - TZip
                |              | - TLha
                |              | - ...etc.
                |


   To use all decompression components as a single component, assign
   the decompression components dropped on the form as type TUnBase
   as demonstrated in this program (with the use of the variable
   'ExtractComponent').

   When assigning the ArchiveFile property a filename, the ArcType
   property is automatically defined internally.  In the ExtractFiles
   procedure below, we demonstrate how to use the ArcType property to
   type cast the desired decompression component as type TUnBase.

   ************************************************************

   HOW THE DEMO WORKS WITH MILTIPLE DECOMPRESSION COMPONENTS

	 The first thing of course is to open an archive (or create a new one).

   The Open menu assigns the ArchiveFile property of the ZipTV1 component
   a filename (Open1Click procedure).  Internally, when the ArchiveFile
   property is assigned, the ArcType property is automatically assigned
   the type of archive ArchiveFile is (ie..atZip,atBh,atArj...etc).

   To extract files from this archive the Extract menu options 'OnClick
   event' points to the 'ExtractFiles' procedure below.  In the
   ExtractFiles procedure, we use the ArcType property to determine which
   decompression component to use.  (ie.. if ArcType = atZIP then we want
   to use the UnZip1 component that was dropped on the form).  By
   typecasting the component as type TUnBase, we eliminate continuous
   repeatitive coding.

   Follow the variable 'ExtractComponent' (type TUnBase) for an
   understanding of typecasting the decompression components.

   ************************************************************

   HOW THE DEMO WORKS WITH MILTIPLE COMPRESSION COMPONENTS

   Compression support works the same was as decompression described
   above.  The base class for all compression components is TCompBase.
   In this demo we use the variable 'CompressComponent' (type TCompBase)
   to assign the proper compression component with the type of archive in
   use.

   When adding (or deleting) files to an already opened archive, we use
   the ZipTV's ArcType property to assign the property component to the
   variable CompressComponent.

   When creating a new archive, form frmAddFiles contains an option to
   select which archive type to create.  Clicking on the selection
   assigns the CompressComponent variable the proper compression component
   to use.

   ************************************************************

   Version 1.27 - 1.31 of TBlakHole, TZip, & TZipSearch components are
   currently beta.  It is never advisable to use beta components in any
   commercial grade application.  Our time line with beta testing our
   components is generally 2 to 3 months from original release.  The
   date of release of these three components was Jan 11, 1998.

   ************************************************************
   IMPORTANT:

   If you do not have TUnRAR do the following:
   1. Main.PAS (this unit): rem UnRar from the uses clause.
   2. Main.PAS (this unit): rem all occurances of atRAR &
   	 atRarExe in the following code.
   3. Main.PAS (this unit) rem all occurances of UnRAR1.
   4. AwkGbls.PAS: rem atRar..atRar from Const "Decompress_ArcType"
   	 defined in AwkGbls.Pas.
   ************************************************************

*)
UNIT Main;



INTERFACE


USES
	Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms,
  Dialogs,	Menus, ComCtrls, ExtCtrls, Gauges, FileCtrl,	StdCtrls,
  AwkTypes, AwkMain, AwkGbls, Err_Msgs, UnBH, UnARJ, UnARC, UnZIP,
  ZipTV, UnZoo, UnTar, UnLHA, UnGZIP, UnCAB, BlakHole, Zipper,
  ZipSrch;


TYPE
	TfrmMain = CLASS( TForm )
		OpenDialog1		: TOpenDialog;

		MainMenu1		: TMainMenu;
		File1				: TMenuItem;
		Open1				: TMenuItem;
    	New1				: TMenuItem;
		Close1			: TMenuItem;
    	N1					: TMenuItem;
    	FileSpec1		: TMenuItem;
    	N2					: TMenuItem;
		Exit1				: TMenuItem;

     Action1			: TMenuItem;
     AddFiles1		: TMenuItem;
     DeleteSelected1: TMenuItem;
     Extract1			: TMenuItem;
     Selected1		: TMenuItem;
     AllFiles1		: TMenuItem;

		pnlStatus		: TPanel;
		pbxLed			: TPaintBox;
		imgLed			: TImage;
		ListView1		: TListView;
		gauProgress		: TGauge;
		StatusBar1		: TStatusBar;

     (* Frontend to de/compression components *)
     ZipTV1			: TZipTV;

     (* Decompression components *)
     UnARC1			: TUnARC;
     UnARJ1			: TUnARJ;
     UnBH1				: TUnBH;
     UnCAB1			: TUnCAB;
     UnGZIP1			: TUnGZIP;
     UnLHA1			: TUnLHA;
     UnTAR1			: TUnTAR;
    	UnZIP1			: TUnZIP;
     UnZOO1			: TUnZOO;

     (* Compression Components *)
    	BlakHole1		: TBlakHole;
    	Zip1				: TZip;

		PROCEDURE EnableMenus;
		PROCEDURE ExtractFiles( Sender: TObject );
		PROCEDURE Open1Click( Sender: TObject );
		PROCEDURE Exit1Click( Sender: TObject );
		PROCEDURE Close1Click( Sender: TObject );
		FUNCTION  DefineCompressComponent: BOOLEAN;

		PROCEDURE pbxLedPaint( Sender: TObject );
		PROCEDURE SetLedColor( lColor : TColor );

    	PROCEDURE FileSpec1Click(Sender: TObject);
		PROCEDURE FormCreate( Sender: TObject );
		PROCEDURE FormDestroy( Sender: TObject );
		PROCEDURE AddFiles1Click( Sender: TObject );
		PROCEDURE DeleteSelected1Click( Sender: TObject );
		PROCEDURE pbxLedClick(Sender: TObject);
		PROCEDURE DisplayTotals( Which: BYTE );
		PROCEDURE ListView1Click(Sender: TObject);
    	PROCEDURE StatusBar1Click(Sender: TObject);

    	PROCEDURE ZipTV1Read(Sender: TObject; Offset, Filenum: Integer);
    	PROCEDURE ZipTV1Error(Sender: TObject; FN, VolumeID: STRING; ECode: Integer);

		PROCEDURE BlakHole1Activate( Sender: TObject );
		PROCEDURE BlakHole1Begin( Sender: TObject; FN: STRING; RecNum: INTEGER;	VAR Extract: BOOLEAN );
		PROCEDURE BlakHole1End( Sender: TObject; FN: STRING; CRC_PASS: BOOLEAN );
		PROCEDURE BlakHole1Deactivate( Sender: TObject );
		PROCEDURE BlakHole1RecurseDir( Sender: TObject; Directory: STRING );

    	//PROCEDURE UnZIP1Activate(Sender: TObject);
    	//PROCEDURE UnZIP1Deactivate(Sender: TObject);
    	PROCEDURE UnZIP1Progress(Sender: TObject; FN: STRING; Progress: BYTE);
    	PROCEDURE UnZIP1FileExists(Sender: TObject; FN: STRING; FileDate: TDateTime; VAR Accept: BOOLEAN);
    	PROCEDURE UnZIP1Begin(Sender: TObject; FN: STRING; RecNum: INTEGER; VAR Extract: BOOLEAN);
    	PROCEDURE UnZIP1End(Sender: TObject; FN: STRING; CRC_PASS: BOOLEAN);
    	PROCEDURE UnZIP1GetPassword(Sender: TObject; FN: STRING; VAR Password: STRING;
     		VAR TryAgain: BOOLEAN);

	PRIVATE
	PUBLIC
	END;




VAR
	frmMain : TfrmMain;
  OldMenuIndex: INTEGER;
	CompressComponent: TCompBase;		(* Base class for compression components *)
	ExtractComponent: TUnBase;			(* Base class for decompression components *)




IMPLEMENTATION



USES
	AddFiles, Unit2, Unit3;



CONST
	TOTALS 	= 0;
	SELECTED = 1;

  (* ListView Column Subitems *)
  SUBITEM_DATETIME  = 0;
  SUBITEM_PACKED	   = 1;
  SUBITEM_UNPACKED  = 2;
  SUBITEM_RATIO 	   = 3;
  SUBITEM_METHOD    = 4;
  SUBITEM_ATTR	   = 5;
  SUBITEM_FOLDER    = 6;
  SUBITEM_FILETYPE  = 7;
  SUBITEM_CRC		   = 8;
  SUBITEM_OFFSET	   = 9;
  SUBITEM_ENCRYPTED = 10;
  SUBITEM_COMMENT   = 11;




VAR
	TotalPackedSize,
	TotalUnpackedSize,
	SelectedPackedSize,
	SelectedUnpackedSize: LONGINT;


{$R *.DFM}


{-------------------------------------------------------------}
(* Enable / Disable menu items *)
PROCEDURE TfrmMain.EnableMenus;

VAR
	CompressSupported,
  DecompressSupported: BOOLEAN;

BEGIN
  Action1.Visible := TRUE;

  CompressSupported   := ZipTV1.IsArcCompressable( ZipTV1.ArcType );
  DecompressSupported := ZipTV1.IsArcDecompressable( ZipTV1.ArcType );

  AddFiles1.Enabled 		:= CompressSupported;
  DeleteSelected1.Enabled := CompressSupported;
  Extract1.Enabled 			:= DecompressSupported;
END;
{-------------------------------------------------------------}
(* Menu's Open & New access this event procedure via their OnClick events *)
PROCEDURE TfrmMain.Open1Click( Sender: TObject );
CONST
	mOpen = 0;
  mNew  = 1;
BEGIN

  OpenDialog1.Filter := LoadStr( F_TZIPTV );		(* load from Err_Msgs.Pas *)

  CASE TMenuItem( Sender ).MenuIndex OF

  	mOpen:				(* Menu item Open *)
     	BEGIN
           OpenDialog1.Title := 'Open Archive...';
           OpenDialog1.Options := [ofHideReadOnly, ofFileMustExist, ofPathMustExist];
  		END;

     mNew:					(* Menu item New *)
     	BEGIN
           OpenDialog1.Title := 'New Archive...';
           OpenDialog1.Options := [ ofHideReadOnly ];
           OpenDialog1.Filename := '';
        END;

  END;


	IF OpenDialog1.Execute THEN
  BEGIN

     TotalPackedSize := 0;
  	TotalUnpackedSize := 0;

  	IF FileExists( OpenDialog1.Filename ) THEN
     BEGIN

     	ZipTV1.ArchiveFile := OpenDialog1.Filename;

        SetLedColor( clRed );
        ListView1.Cursor := crHourGlass;
        ListView1.Items.Clear;

        ZipTV1.Activate;

     END ELSE BEGIN

        CompressComponent := NIL;

        ListView1.Items.Clear;

        WITH StatusBar1 DO
        BEGIN
           Panels[0].Text := 'Files: 0';
           Panels[1].Text := 'Packed: 0';
           Panels[2].Text := 'Unpacked: 0';
           Panels[3].Text := 'Ratio: 0%';
        END;

     	IF frmAddFiles.ShowModal = mrCancel THEN EXIT;
        //IF frmAddFiles.ModalResult = mrCancel THEN EXIT;

        ListView1.Cursor := crHourGlass;
        ListView1.Items.Clear;

        SetLedColor( clRed );
        CompressComponent.ArchiveFile := OpenDialog1.Filename;
        CompressComponent.Activate;

        (* Assign the newly created archive to ZipTV's ArchiveFile property *)
     	ZipTV1.ArchiveFile := OpenDialog1.Filename;

     END;

     Caption := ZipTV1.ArchiveFile;

     EnableMenus;
     Close1.Enabled := TRUE;

     DisplayTotals( TOTALS );

     Caption := OpenDialog1.Filename;
     SetLedColor( clGreen );
     ListView1.Cursor := crDefault;

 	END;

END;
{-------------------------------------------------------------}
PROCEDURE TfrmMain.Exit1Click( Sender: TObject );
BEGIN
	Close;
END;
{-------------------------------------------------------------}
(* Close menu item *)
PROCEDURE TfrmMain.Close1Click( Sender: TObject );
BEGIN

	ZipTV1.ArchiveFile := '';
	ListView1.Items.Clear;
  Caption := '';
  StatusBar1.SimplePanel := TRUE;

  Action1.Visible := FALSE;
  Close1.Enabled := FALSE;
  									(* Set the ArcType property to atNA by
  										blanking the ArchiveFile property *)
  CompressComponent.ArchiveFile := '';

END;
{-------------------------------------------------------------}
(* Change colors of little light in corner of form *)
PROCEDURE TfrmMain.SetLedColor( lColor : TColor );
BEGIN

	WITH imgLed.Canvas DO
	BEGIN

		Brush.Color := lColor;
		FloodFill( 6, 6, Pixels[ 6, 6 ], fsSurface );

	END;

	pbxLed.Repaint;

end;
{-------------------------------------------------------------}
(* Little light in right corner of form *)
PROCEDURE TfrmMain.pbxLedPaint( Sender: TObject );
BEGIN

	WITH Sender AS TPaintBox DO
	 Canvas.Draw( ( Width - imgLed.Width ) DIV 2,
					( Height - imgLed.Height ) DIV 2, imgLed.Picture.Graphic );

END;
{-------------------------------------------------------------}
(* OnActivate event... activated when a compression component begins compressing
	 files *)
PROCEDURE TfrmMain.BlakHole1Activate( Sender: TObject );
BEGIN

	StatusBar1.SimplePanel := TRUE;
	gauProgress.Visible := TRUE;

	ListView1.Cursor := crHourGlass;
	ListView1.Items.BeginUpdate;
	SetLedColor( clRed );

END;
{-------------------------------------------------------------}
(* OnDeactivate event... activated when a compression component has completed its
	 task of compressing all files *)
PROCEDURE TfrmMain.BlakHole1Deactivate( Sender: TObject );
BEGIN

	ListView1.Items.EndUpdate;
	DisplayTotals( TOTALS );
	ListView1.Cursor := crDefault;

END;
{-------------------------------------------------------------}
(* OnBegin event... activates with each new compressed file found (matching the
	 FileSpec property) in an archive.

	 Display deleting/compressing message in bottom status bar
*)
PROCEDURE TfrmMain.BlakHole1Begin( Sender: TObject; FN: STRING; RecNum: INTEGER;
		VAR Extract: BOOLEAN );
BEGIN

	(* CompBase is the parent class for all 'compression' components *)
	IF TCompBase(Sender).Switch = swDelete THEN
		StatusBar1.SimpleText := 'Deleting: ' + ExtractFilename( FN )
	ELSE
		StatusBar1.SimpleText := 'Compressing: ' + FN;

	Application.ProcessMessages;

END;
{-------------------------------------------------------------}
(* Display totals in bottom status bar *)
PROCEDURE TfrmMain.DisplayTotals( Which: BYTE );

VAR
	Ratio: BYTE;
	i: WORD;

LABEL
	Tot;

BEGIN

	StatusBar1.SimplePanel := FALSE;
	gauProgress.Visible := FALSE;
	SetLedColor( clGreen );

	CASE Which OF
		TOTALS: 	 GOTO Tot;
		SELECTED: IF ListView1.Selected = NIL THEN
						 GOTO Tot;
	END;

	SelectedPackedSize := 0; SelectedUnpackedSize := 0;

	FOR i := 0 TO Pred( ListView1.Items.Count ) DO
	BEGIN
		IF ListView1.Items[i].Selected THEN
		BEGIN
			Inc( SelectedPackedSize, StrToInt( ListView1.Items[i].SubItems[1] ) );
			Inc( SelectedUnpackedSize, StrToInt( ListView1.Items[i].SubItems[2] ) );
		END;
	END;

	Ratio := ZipTV1.CalcRatio( SelectedPackedSize, SelectedUnpackedSize );

  WITH StatusBar1 DO
  BEGIN

     Panels[0].Text := 'Files: ' + IntToStr( ListView1.SelCount );
     Panels[1].Text := 'Packed: ' +  IntToStr( SelectedPackedSize );
     Panels[2].Text := 'Unpacked: ' + IntToStr( SelectedUnpackedSize );
     Panels[3].Text := 'Ratio: ' + IntToStr( Ratio ) + '%';

  END;
	EXIT;

Tot:
	Ratio := ZipTV1.CalcRatio( TotalPackedSize, TotalUnpackedSize );

	WITH StatusBar1 DO
	BEGIN

		Panels[0].Text := 'Files: ' + IntToStr( ListView1.Items.Count );
		Panels[1].Text := 'Packed: ' +  IntToStr( TotalPackedSize );
		Panels[2].Text := 'Unpacked: ' + IntToStr( TotalUnpackedSize );
		Panels[3].Text := 'Ratio: ' + IntToStr( Ratio ) + '%';

	END;

END;
{-------------------------------------------------------------}
(* OnEnd event... activated after compressing each individual file to an archive *)
PROCEDURE TfrmMain.BlakHole1End( Sender: TObject; FN: STRING;	CRC_PASS: BOOLEAN );
BEGIN

	StatusBar1.SimpleText := '';
	gauProgress.Progress := 0;

END;
{-------------------------------------------------------------}
(* Initialize system icons *)
PROCEDURE TfrmMain.FormCreate( Sender: TObject );
BEGIN
                             (* Initialize systems image list *)
	InitializeImageList( Self, ListView1 );

  Action1.Visible := FALSE;
  Close1.Enabled := FALSE;
  CompressComponent := BlakHole1;	(* Start program with BlakHole compression
                                      as default *)
END;
{-------------------------------------------------------------}
(* Destroy the system image list *)
PROCEDURE TfrmMain.FormDestroy( Sender: TObject );
BEGIN
	DestroyImageList( ListView1 );
END;
{-------------------------------------------------------------}
FUNCTION TfrmMain.DefineCompressComponent: BOOLEAN;
BEGIN

	CompressComponent := NIL;

  WITH ZipTV1 DO
     IF NOT IsArcCompressable( ArcType ) THEN

        MessageDlg( 'No compression support for this archive', mtInformation, [mbOK], 0 )

     ELSE

        CASE ArcType OF

           atBH		: CompressComponent := BlakHole1;

           atZip,
           atZip250	: CompressComponent := Zip1;

           //atLha,
           //atLzh		: CompressComponent := Lha1;

        END;

	RESULT := NOT ( CompressComponent = NIL );

END;
{-------------------------------------------------------------}
(* AddFiles menu item *)
PROCEDURE TfrmMain.AddFiles1Click( Sender: TObject );
BEGIN

	IF ZipTV1.ArchiveFile <> '' THEN
	BEGIN

     IF NOT DefineCompressComponent THEN
        EXIT;

     ListView1.Cursor := crHourGlass;

     CompressComponent.ArchiveFile := ZipTV1.ArchiveFile;
     IF frmAddFiles.ShowModal = mrCancel THEN EXIT;

     IF CompressComponent.FileSpec.Count > 0 THEN
        CompressComponent.Activate;

     ListView1.Cursor := crDefault;

	END ELSE

		MessageDlg( 'Open archive before adding files',
						mtInformation,
						[ mbOK ],
						0 );

END;
{-------------------------------------------------------------}

(*
   Extract files menu item

  *******************************************************************

   Example of using multiple decompression components to extract
   files for different types of archives.

  *******************************************************************

  -The ZipTV1.ArchiveFile has already been assigned a filename.
  -The ArcType property was assigned internally when ArchiveFile
   was assigned
  -Use the ArcType property to determine which decompression component
   to assign to variable 'ExtractComponent' (type TCompBase... parent
   object to all decompression components).
  -After assigning variable 'ExtractComponent', use its properties /
   methods as if it were the actual decompression component.

  *******************************************************************

  Always remember to use the ELSE statement with CASE ZipTV1.ArcType!
  The TZipTV component supports more archive types than the package
  currently has compression and decompression components.

*)
PROCEDURE TfrmMain.ExtractFiles( Sender: TObject );

VAR
	i: WORD;
	Dir: STRING;

CONST
	EXTRACT_SELECTED = 0;
	EXTRACT_ALL = 1;

BEGIN

  IF ZipTV1.ArchiveFile <> '' THEN
  BEGIN

     IF ( TMenuItem( Sender ).MenuIndex = EXTRACT_SELECTED ) AND
           ( ListView1.SelCount = 0 ) THEN
        EXIT;


     								(* If previous extraction, start with
                             	the same dir before *)

     IF ExtractComponent <> NIL THEN
     	Dir := ExtractComponent.ExtractDir;

                             (* Get dir from user to extract to *)

     IF NOT SelectDirectory( Dir,
                          [ sdAllowCreate,
                            sdPerformCreate,
                            sdPrompt ],
                            -1 ) THEN
        EXIT;



                             (* Assign the TUn??? component *)

     CASE ZipTV1.ArcType OF

        atArc,
        atArcExe		: ExtractComponent := UnARC1;

        atArj,
        atArjExe		: ExtractComponent := UnARJ1;

        atBH,
        atBHExe		: ExtractComponent := UnBH1;

        atCAB			: ExtractComponent := UnCAB1;

        atGZIP		: ExtractComponent := UnGZIP1;

        atLHA,
        atLhaExe,
        atLZH,
        atLzhExe		: ExtractComponent := UnLHA1;

        							(* If you don't have TUnRar component, rem
                                the following two lines.  Also rem atRAR
                                in the const "Decompress_ArcType" in
                                AwkGbls.PAS *)
        //atRar,
        //atRarExe		: ExtractComponent := UnRAR1;

        atTar			: ExtractComponent := UnTAR1;

        atZip,
        atZipExe,
        atZip250,
        atZip250Exe	: BEGIN
                          ExtractComponent := UnZIP1;
                          ExtractComponent.ZipCmntBufSize := ZipTV1.ZipCmntBufSize;
                      END;


        atZoo			: ExtractComponent := UnZOO1;

     ELSE

        BEGIN

           MessageDlg( 'Extract not supported for this filetype.', mtInformation, [mbOK], 0 );
           EXIT;

        END;

     END;

                             (* Assign filename & dir to extract to *)

     ExtractComponent.ArchiveFile := ZipTV1.ArchiveFile;
     ExtractComponent.ExtractDir := dir;


     (*******************************)
     (* Form display controls setup *)
     (*******************************)
     StatusBar1.SimplePanel := TRUE;
     gauProgress.Visible := TRUE;
     ListView1.Cursor := crHourGlass;
     SetLedColor( clRed );

     TRY

        CASE TMenuItem( Sender ).MenuIndex OF

           EXTRACT_ALL:
              BEGIN

                 ExtractComponent.FileSpec := ZipTV1.FileSpec;
                 ExtractComponent.Extract;

              END;

           EXTRACT_SELECTED:				(* Extract Selected *)
              BEGIN

                 IF ZipTV1.ArcType = atCAB THEN
                 BEGIN

                    IF MessageDlg(
                          'If extracting more than one file... because of CABs continuous ' +
                          'compressed stream method, it is much faster to decompress all ' +
                          'files from the archive and then delete from disk those you ' +
                          'do not wish to keep.  For more info on CAB files, see the ' +
                          'documentation on TUnCAB.'#13#13 + 'Cancel operation?',
                          mtWarning,
                          [mbYES, mbNO],
                          0 ) = mrYES THEN

                       EXIT;

                 END;

                 FOR i := 0 TO Pred( ListView1.Items.Count ) DO
                    IF ListView1.Items[i].Selected THEN
                    BEGIN

                       (* Retreive the full path stored in the archive
                          from the ListView Control *)
                       WITH ListView1.Items[i] DO
                          ExtractComponent.FileSpec := AppendDirTail( SubItems[ SUBITEM_FOLDER ] ) +
                             Caption;

                       ExtractComponent.Extract;

                    END;

              END;

        END;

     FINALLY

        (*******************************)
        (* Form display controls reset *)
        (*******************************)
        gauProgress.Visible := FALSE;
        StatusBar1.SimplePanel := FALSE;
        ListView1.Cursor := crDefault;
        SetLedColor( clGreen );

     END;

  END;

END;
{-------------------------------------------------------------}
(* Delete selected menu item *)
PROCEDURE TfrmMain.DeleteSelected1Click( Sender: TObject );

VAR
	i: WORD;

BEGIN

	IF ( ListView1.Selected <> NIL ) AND ( ZipTV1.ArchiveFile <> '' ) THEN
	BEGIN

		ListView1.Cursor := crHourGlass;

		IF NOT DefineCompressComponent THEN
     	EXIT;

		CompressComponent.ArchiveFile := ZipTV1.ArchiveFile;

		TotalPackedSize := 0;
		TotalUnpackedSize := 0;

		CompressComponent.FileSpec.Clear;
		//CompressComponent.ExcludeSpec.Clear;

     (* Add all selected rows in ListView to TBlakHole's filespec property *)
     FOR i := 0 to Pred( ListView1.Items.Count ) DO
			IF ListView1.Items[i].Selected THEN
           CompressComponent.FileSpec.Add( ListView1.Items[i].Caption );


     (* IF we have any filespecs, process file deletion *)
		CompressComponent.Switch := swDelete;
     IF CompressComponent.FileSpec.Count > 0 THEN
        CompressComponent.Activate;

     ListView1.Items.Clear;
     ZipTV1.Activate;		(* Redisplay archive *)
     DisplayTotals( TOTALS );

		ListView1.Cursor := crDefault;

	END ELSE

		MessageDlg( 'No files to delete',
						mtInformation,
						[ mbOK ],
						0 );

END;
{-------------------------------------------------------------}
(* OnRecurseDir Event Procedure (all compression components access
   this procedure) *)
PROCEDURE TfrmMain.BlakHole1RecurseDir( Sender: TObject; Directory: STRING );
BEGIN
	StatusBar1.SimpleText := 'Scanning directory: ' + Directory;
	Application.ProcessMessages;
END;
{-------------------------------------------------------------}
(* TPaintBox ( pbxLed ) Component OnClick Event Procedure.
   Set the Cancel property to true *)
PROCEDURE TfrmMain.pbxLedClick(Sender: TObject);
BEGIN
	IF CompressComponent <> NIL THEN	CompressComponent.Cancel := TRUE;
	IF ExtractComponent <> NIL THEN ExtractComponent.Cancel := TRUE;
	SetLedColor( clGreen );
END;
{-------------------------------------------------------------}
(* ListView Component OnClick Event Procedure *)
PROCEDURE TfrmMain.ListView1Click(Sender: TObject);
BEGIN
	DisplayTotals( SELECTED );
END;
{-------------------------------------------------------------}
(* StatusBar component OnClick Event Procedure *)
PROCEDURE TfrmMain.StatusBar1Click(Sender: TObject);
BEGIN
	DisplayTotals( TOTALS );
END;
{-------------------------------------------------------------}
(* OnRead Event Procedure (ZipTV1 & compression components
   access to this procedure)

   * ZipTV1 fills ListView with the compressed file info
   * The compression components access their OnRead event with
     compressed file info with every file added to an archive.
*)
PROCEDURE TfrmMain.ZipTV1Read(Sender: TObject; Offset, Filenum: Integer);
VAR
	NewItem : TListItem;
	ZipCommon: TZipCommon;
BEGIN

  (* ZipCommon is the parent object to both compression & decompression
     components - See object hierarchy at beginning of this unit *)
	ZipCommon := TZipCommon( Sender );

	NewItem := ListView1.Items.Add;

	Inc( TotalPackedSize, ZipCommon.PackedSize );
	Inc( TotalUnpackedSize, ZipCommon.UnpackedSize );

	WITH NewItem DO
	BEGIN
                               (* For future use *)
		Data := POINTER( Offset );

		(* Add associated image to list ( ...see also InitializeListImages *)
		ImageIndex := ZipCommon.ImageIndex;

		Caption := ExtractFilename( ZipCommon.Filename );		 (* filename *)
		WITH SubItems DO
		BEGIN

			Add( FormatDateTime( 'mm-dd-yy hh:mm', ZipCommon.Date ) ); (* Date/Time *)
			Add( IntToStr( ZipCommon.PackedSize ) );            (* PackedSize *)
			Add( IntToStr( ZipCommon.UnpackedSize ) );          (* UnpackedSize *)
			Add( IntToStr( ZipCommon.Ratio ) + '%' );           (* Ratio *)
			Add( ZipCommon.sCompressionMethod );                (* ( STRING ) method *)
			//or
			//Add( IntToStr( ZipCommon.wCompressionMethod ) );  (* ( WORD  ) method *)
			Add( IntToStr( ZipCommon.ExternalAttr ) ); 		    (* InternalAttr *)
			Add( AppendDirTail( ExtractFileDir( ZipCommon.Filename ) ) );
			Add( Zip1.GetFileType( ZipCommon.Filename ) ); 		 (* Windows associated filetype *)
			Add( IntToHex( ZipCommon.CRC, 8 ) );         		 (* 32 bit CRC *)
			Add( IntToStr( Offset ) );								 	 (* File beginning offset *)
        IF ZipCommon.Encrypted THEN                     	 (* File Encrypted *)
        	Add( 'Yes' )
        ELSE
        	Add( 'No' );

			Add( StrPas( ZipTV1.FileComment ) );	   	  		(* file comment *)

		END;

	END;

END;
{-------------------------------------------------------------}
(* OnActivate Event Procedure (all decompression components access
   this procedure) *)
{PROCEDURE TfrmMain.UnZIP1Activate(Sender: TObject);
BEGIN

	StatusBar1.SimplePanel := TRUE;
	gauProgress.Visible := TRUE;

	ListView1.Cursor := crHourGlass;
	SetLedColor( clRed );

END;}
{-------------------------------------------------------------}
(* OnDeactivate Event Procedure (all decompression components access
   this procedure) *)
{PROCEDURE TfrmMain.UnZIP1Deactivate(Sender: TObject);
BEGIN

	DisplayTotals( TOTALS );
	ListView1.Cursor := crDefault;

END;}
{-------------------------------------------------------------}
(* All ZipTV components on dropped on this form have access to this
   procedure via their OnError events.

   See Err_Msgs.PAS & Err_Msgs.RC for error codes *)
PROCEDURE TfrmMain.ZipTV1Error(Sender: TObject; FN, VolumeID: STRING; ECode: Integer);

VAR
	Msg: STRING;

CONST
	CRLF = #13#13;

BEGIN

	Msg := FN + CRLF;

	CASE ECode OF

		E_BASE..E_WIN32:  (* All error codes *)
				Msg := Msg + 'Error#: ' + IntToStr( Ecode ) + CRLF +
								 'Error: ' + LoadStr( Ecode ) + CRLF;

		M_BASE..M_PASSWORDFAILED:  (* All message codes *)
				Msg := Msg + 'General Message: ' + IntToStr( Ecode ) + CRLF +
								 'Messge: ' + LoadStr( Ecode ) + CRLF;
	ELSE
		EXIT;		(* <- no msg *)
	END;

	Msg := Msg + 'OnError event...';
	MessageDlg( Msg, mtInformation, [mbOk], 0 );

END;
{-------------------------------------------------------------}
(* OnProgress Event Procedure

	-The type of the Sender parameter in the OnProgress event is either
   TUnBase or TUnComp, since this event procedure is used by both
   compression as well as decompression component types.

  -The 'Cancel' property is a property of TZipCommon. TUnBase & TCompBase
   inherit Cancel from TZipCommon. Since both compression and decompression
   components access the below procedure (via OnProgress events), we
   typecast 'Sender' as type TZipCommon to check the value of the Cancel
   property and set the Led color accordingly.

   When Cancel is set as true, the compression / decompression is not
   interrupted until the current file is completed.  Therefore, the components
   will activate their OnProgress events until the current file is either
   compressed or decompressed whichever process is active.  If Cancel is true,
   we circle through colors for the LED to let the user know that Cancel their
   Cancel request will be processed after the operation is completed on the
   current file.

  -All compression & decompression components access this procedure

*)
PROCEDURE TfrmMain.UnZIP1Progress(Sender: TObject; FN: STRING; Progress: Byte);

VAR
	NewColor: TColor;

BEGIN

  IF TZipCommon( Sender ).Cancel THEN
  BEGIN
  	NewColor := Brush.Color + 1;
     SetLedColor( NewColor );
  END;

	gauProgress.Progress := Progress;
	Application.ProcessMessages;

END;
{-------------------------------------------------------------}
(* OnFileExists Event Procedure (all decompression components
   access this procedure) *)
PROCEDURE TfrmMain.UnZIP1FileExists(Sender: TObject; FN: STRING; FileDate: TDateTime; VAR Accept: BOOLEAN);
BEGIN
	//MessageDlg( FormatDateTime( 'mm-dd-yy hh:mm', FileDate ), mtInformation, [ mbOK ], 0 );
	IF MessageDlg( FN + ' already exists, overwrite?' + #13#13'OnFileExists Event...', mtInformation, [mbYes,mbNo], 0 ) = mrNo THEN
		Accept := FALSE;
END;
{-------------------------------------------------------------}
(* OnBegin Event Procedure (all decompression components access
   this procedure) *)
PROCEDURE TfrmMain.UnZIP1Begin(Sender: TObject; FN: STRING; RecNum: Integer; VAR Extract: BOOLEAN);
BEGIN
	StatusBar1.SimpleText := 'Extracting: ' + FN;
END;
{-------------------------------------------------------------}
(* OnEnd Event Procedure (all decompression components access
   this procedure) *)
PROCEDURE TfrmMain.UnZIP1End(Sender: TObject; FN: STRING; CRC_PASS: BOOLEAN);
BEGIN

	StatusBar1.SimpleText := '';
	gauProgress.Progress := 0;

	IF NOT CRC_PASS THEN
		MessageDlg( FN + #13'Extract failed'#13#13'OnEnd Event...',
				mtInformation,
				[mbOk], 0 );

END;
{-------------------------------------------------------------}
(* OnGetPassword Event Procedure (all compression & decompression
   components access this procedure) *)
PROCEDURE TfrmMain.UnZIP1GetPassword(Sender: TObject; FN: STRING; VAR Password: STRING; VAR TryAgain: BOOLEAN);
BEGIN

	PasswordDlg.Caption := ExtractFilename( FN );

	IF PasswordDlg.ShowModal = mrOK THEN
		Password := PasswordDlg.edtPassword.Text
	ELSE
		TryAgain := FALSE;

END;
{-------------------------------------------------------------}
PROCEDURE TfrmMain.FileSpec1Click(Sender: TObject);
BEGIN

	fFileSpec.Edit1.text := ZipTV1.Filespec;

	IF fFileSpec.ShowModal = mrOK THEN
	BEGIN

		IF fFileSpec.Edit1.Text <> '' THEN
		BEGIN

			ZipTV1.FileSpec := fFileSpec.Edit1.Text;

			IF ZipTV1.ArchiveFile <> '' THEN
			BEGIN

           SetLedColor( clRed );
           ListView1.Cursor := crHourGlass;

				ListView1.Items.Clear;
				ZipTV1.ArchiveFile := ZipTV1.ArchiveFile;
				ZipTV1.Activate;

           DisplayTotals( TOTALS );
           SetLedColor( clGreen );
           ListView1.Cursor := crDefault;

			END;

		END;

	END;

END;
{-------------------------------------------------------------}



END.
