//  DBExplorer 1.0.2
unit DBExplor;
//========================================================================
interface
{$I dbex_ver.inc}
uses
  Windows, Messages, SysUtils, Classes, CommCtrl, Graphics, Controls,
  Forms, Dialogs, ComCtrls, DB, DBTables, DsgnIntf,
  Menus;

const
    etnMaxKeyFieldLen = 50;
    MaxBufSize = 100;

type
TDBExNode = class;
TDBExDataNode = class;
TDBExNodes = class;
TCustomDBExplorer = class;
TDBExMetaDispatcher = class;
TDBExDataDispatcher = class;
TDBExDataLink = class;
TCustomDBExDataSetNode = class;
TDBExDispatcher = class;
TDBExNodeType = (ntNullNode, ntDataSetNode, ntLinkNode, ntCustomNode);
TDBExLinkType = (ltMaster, ltParam, ltFilter, ltUnknown);

//------------------------------------------------------------------------
TDBExNodeLinks = class;


TDBExNodeLink = class( TCollectionItem )
private
    FMasterNode  : TDBExDataNode;
    FMasterField : String;
    FDetailField : String;
    FLinkType : TDBExLinkType;
protected
public

    constructor Create(Collection: TCollection);override;
    procedure Assign(Source: TPersistent); override;
published
    property MasterNode : TDBExDataNode read FMasterNode write FMasterNode;
    property MasterField : String read FMasterField write FMasterField; 
    property DetailField : String read FDetailField write FDetailField; 
    property LinkType : TDBExLinkType read FLinkType write FLinkType; 
end;

//------------------------------------------------------------------------

TDBExNodeLinks = class( TCollection )
private
    FOwnerNode : TDBExDataNode;

    function GetItem( Index : integer ) : TDBExNodeLink;
    procedure SetItem( Index : integer; Value : TDBExNodeLink);
protected

public
    function Add : TDBExNodeLink;
    constructor Create( AOwnerNode : TDBExDataNode );
    property Items[ Index : integer ]:TDBExNodeLink read GetItem write SetItem;
    property OwnerNode : TDBExDataNode read FOwnerNode;
end;

//------------------------------------------------------------------------

TDBExLinkValue = class( TObject )
    MasterField : string;
    MasterValue : string;
end;

//------------------------------------------------------------------------

TDBExLinkValues = class( TObject )
private
    FValues : TList;
    FDispatcher : TDBExDispatcher;

    function GetCount:integer;
    function GetValue( Index : integer ): TDBExLinkValue;
public
    constructor Create( Dispatcher : TDBExDispatcher); //override;
    destructor Destroy; override;
    procedure Clear;
    procedure Add( Value : TDBExlinkValue );
    function Find( Link : TDBExNodeLink ): TDBExLinkValue;

    property Value[ Index : integer ] : TDBExLinkValue read GetValue;
    property Count : integer read GetCount;
end;

//------------------------------------------------------------------------


TDBExDispatcher = class( TObject )
private
    FTreeNode : TTreeNode;                   
    FParentDispatcher : TDBExDispatcher;     
    FTopicNode : TDBExNode;                  
    FKeyValue  : string;                     
    FText      : string;                     
    FActiveItem : TDBExDispatcher;           
    FItemIndex   : integer;                  
    FSelectedItem : integer;                 
    FFirstItem : integer;
    FLastItem : integer;                     
    FNodes : TDBExNodes;                     
                                             
    FDispatchers : TList;            
    FExpanded : boolean;                     
    FLinkValues : TDBExLinkValues;           
    FInvisibleDisp : TDBExDispatcher;        
                                             
    FExplorer : TCustomDBExplorer;

    procedure SetTreeNode( Value : TTreeNode );
    function GetTreeNode : TTreeNode;
    procedure SetText( Value : string ); virtual;
    function GetText:string;
    procedure SetItemIndex( Value : integer);
    procedure EndEdit; virtual;abstract;
    procedure SetActiveItem( Value : TDBExDispatcher ); virtual;
    function GetExpanded : boolean;
    function CanHasChildren: boolean ; virtual; abstract;
    procedure ClearRest;
    function GetDispatchers(Index:integer):TDBExDispatcher;
    function GetDispatcherCount:integer;
protected
    FTop  : integer;
    FBottom : integer;
    FNextTopicVisible : boolean;
    procedure AttachTreeNode(Items : TTreeNodes; Master : TDBExDispatcher;
                             Before : TTreeNode; Index : integer);

    procedure OnCollapse; virtual;
    procedure BeforeExpand; virtual;
    procedure AfterExpand; virtual;
    procedure Deselect; virtual;
    procedure Select; virtual; abstract;
    procedure FillDown; virtual; abstract;
    function AddBefore( Node : TTreeNode; DoExpand : boolean ):TDBExDispatcher;
                                               virtual; abstract;
    function GetTopicNode : TDBExNode;
    function CanEdit:boolean; virtual; abstract;
    function GetKey : string; virtual; abstract;
    procedure NotifyExpand( Child : TDBExDispatcher ); virtual;
    procedure NotifyCollapse( Child : TDBExDispatcher ); virtual;
    procedure SaveLinkValues;
    procedure ClearSubtree;
public
    constructor Create( TopicNode : TDBExNode ); virtual;
    destructor Destroy; override;


    property TreeNode : TTreeNode read GetTreeNode write SetTreeNode;
    property Text:string read GetText write SetText;
    property KeyValue : string read FKeyValue write FKeyValue;
    property ParentDispatcher : TDBExDispatcher
             read FParentDispatcher write FParentDispatcher;
    property ItemIndex : integer read FItemIndex;
    property ActiveItem : TDBExDispatcher
        read FActiveItem write SetActiveItem;
    property Expanded : boolean read GetExpanded;
    property Dispatchers [Index:integer]:TDBExDispatcher read GetDispatchers;
    property DispatcherCount:integer read GetDispatcherCount;
end;

//------------------------------------------------------------------------

TDBExMetaDispatcher = class (TDBExDispatcher)
private
    FPrevActive : TDBExMetaDispatcher;

    procedure SetText( Value : string ); override;
    function CreateDispatcher( Node : TDBExNode; Before : TTreeNode ;
                               DoExpand : boolean;
                               Index : integer) : TDBExDispatcher;
    function CanHasChildren: boolean ; override;
    procedure EndEdit; override;
protected
    function CanEdit : boolean; override;
    function GetKey : string; override;
    procedure Select; override;
    procedure Deselect; override;
    procedure FillDown; override;
    function AddBefore( Node : TTreeNode;
                        DoExpand : boolean ):TDBExDispatcher; override;
public
    constructor Create( TopicNode : TDBExNode ); override;
    destructor Destroy; override;

end;

//------------------------------------------------------------------------

TDBExDataDispatcher = class(TDBExDispatcher)
private
    FDataTopicNode : TDBExDataNode;     
                                        
    FLinkTemplate : TDBExLinkValues;
    FSavedActiveDispatcher : TDBExDispatcher;
    FSavedKeyValue : string;

    procedure SetActiveItem( Value : TDBExDispatcher ); override;
    function CreateDispatcher(Before :  TTreeNode;DoExpand : boolean;
                              Index : integer) : TDBExDispatcher;
    function CanHasChildren: boolean ; override;
    procedure EndEdit; override;
protected
    procedure ApplyFilter; virtual;
    procedure SaveDataState;
    procedure RestoreDataState;

    procedure LabelChanged( Value : string );
    procedure KeyChanged( Value : string );
    procedure MoveFirst;
    procedure MoveLast;
    procedure Scroll( Distance : integer );

    procedure SelectionMoved( DFrom, DTo : TDBExDataDispatcher);
    procedure InternalRefresh; //override;
    procedure FillLinkFields( DataSet : TDataSet );
    function CanEdit : boolean; override;
    procedure BeforeExpand; override;
    procedure AfterExpand; override;
    function GetKey : string; override;
    procedure OnCollapse; override;
    procedure Select; override;
    procedure Deselect; override;
    procedure FillDown; override;
    function AddBefore( Node : TTreeNode;
                        DoExpand : boolean ):TDBExDispatcher; override;
public
    constructor Create( TopicNode : TDBExNode ); override;
    destructor Destroy; override;

    procedure Refresh;
end;


//------------------------------------------------------------------------

TDBExDataLink = class (TDataLink )
private
    FNode : TCustomDBExDataSetNode;
    PrevDataSet:TDataSet;
    FModified:Boolean;
protected
    procedure ActiveChanged; override;
    procedure CheckBrowseMode; override;
    procedure DataSetChanged; override;
    procedure DataSetScrolled(Distance: Integer); override;
    procedure FocusControl(Field: TFieldRef); override;
    procedure EditingChanged; override;
    procedure LayoutChanged; override;
    procedure RecordChanged(Field: TField); override;
    procedure UpdateData; override;
public
    constructor Create( ANode : TCustomDBExDataSetNode );
end;


EDBExpError = class (Exception);
TTextChangeEvent = procedure(Sender : TComponent) of object;
TNodeActivateEvent = procedure(Sender : TComponent) of object;
TNodeSelectEvent=procedure(Sender : TComponent; IsTopic : boolean) of object;

//------------------------------------------------------------------------

TDBExNode = class (TComponent)
private
    FText        : string;                
    FOwnerNodes  :TDBExNodes;
    FTopicImageIndex : integer;           
    FTopicSelectedIndex : integer;        
    FNodeType        : TDBExNodeType;     
    FTopicStateIndex : integer;           
    FExpanded        : boolean;
    FExplorer        : TCustomDBExplorer;

    FPopupMenu       :TPopupMenu;
    FFreezed          : boolean;          

    FOnActivate       : TNodeActivateEvent;
    FOnDeactivate     : TNodeActivateEvent;
    FOnSelect         : TNodeSelectEvent;


    function CreateNode(ANodeType : TDBExNodeType;
                           AOwner : TComponent ): TDBExNode;
    procedure DriveDataSet; virtual;
    procedure DriveTopicDataSet; virtual;
    procedure SetActiveTopic(Value : TDBExDataDispatcher);virtual;
    function GetItem( Index : integer ) : TDBExNode; virtual;
    procedure SetItemLabel( Value : String ); virtual;

    function CheckMenuPopup(const Pos: TSmallPoint):Boolean;
    procedure SetPopupMenu(Value: TPopupMenu);
    function GetPopupMenu: TPopupMenu;
    procedure SetParentNode( Value : TDBExNode );
protected
    FActiveTopic  : TDBExDataDispatcher;
    FTopicDispatchers  : TList;

    function CreateDispatcher( Master : TDBExDispatcher;
                               Before : TTreeNode ) : TDBExDispatcher;
    function CreateTopicDispatcher(Master : TDBExDispatcher;
                            Before : TTreeNode):TDBExDispatcher; virtual;
    function CreateItemDispatcher(Master : TDBExDispatcher;
                            Before : TTreeNode):TDBExDispatcher; virtual;
    function CanEdit:boolean; virtual;


    function Add( ANodeType : TDBExNodeType; S : string ):TDBExNode;
    function AddChild( ANodeType : TDBExNodeType; S : string ):TDBExNode;
    function AddChildFirst( ANodeType : TDBExNodeType;
                                    S : string ):TDBExNode;
    function AddFirst( ANodeType : TDBExNodeType; S : string ):TDBExNode;
    procedure RemoveChild( Node : TDBExNode );
    procedure SetText( Value : string );
    procedure EndEdit; virtual;
    procedure Notification(AComponent: TComponent; Operation: TOperation);override;
    function GetParentComponent : TComponent; override;
    function HasParent: Boolean; override;
    procedure ReadState(Reader: TReader); override;
    procedure SetParentComponent(AParent: TComponent); override;
    procedure DeleteDispatchers;
{$IFDEF _DELPHI3_}
    procedure GetChildren(Proc: TGetChildProc; Root: TComponent); override;
{$ELSE}
    procedure GetChildren(Proc: TGetChildProc); override;
{$ENDIF}

public
    FChildren    : TList;                 
    FParentNode   : TDBExNode;
    DebugDIsp : TDBExDispatcher;

    constructor Create( AOwner : TComponent );override;
    destructor Destroy;override;
    function GetFirstChild: TDBExNode; virtual;
    function GetLastChild: TDBExNode;  virtual;
    function GetNext: TDBExNode; virtual;
    function GetNextAfter: TDBExNode; virtual;
    function GetNextChild(Value: TDBExNode): TDBExNode; virtual;
    function GetNextSibling: TDBExNode; virtual;

    function GetPrevChild(Value: TDBExNode): TDBExNode; virtual;
    function GetPrevSibling: TDBExNode; virtual;

    function ChildrenCount : integer; virtual;

    procedure Clear;virtual;
    function ParentNode: TDBExNode;
    property OwnerNodes : TDBExNodes read FOwnerNodes;
    property ActiveTopic : TDBExDataDispatcher read FActiveTopic
                                     write SetActiveTopic;
    property Item[index : integer]:TDBExNode read GetItem;
published
    property PopupMenu : TPopupMenu read GetPopupMenu
                                    write SetPopupMenu;
    property TopicImageIndex : integer read FTopicImageIndex
                                      write FTopicImageIndex;
    property TopicSelectedIndex : integer read FTopicSelectedIndex
                                         write FTopicSelectedIndex;
    property TopicStateIndex : integer read FTopicStateIndex
                                      write FTopicStateIndex;
    property Text : string read FText write SetText;
    property NodeType : TDBExNodeType read FNodeType;
    property OnActivate : TNodeActivateEvent read FOnActivate
                                            write FOnActivate;
    property OnDeactivate : TNodeActivateEvent read FOnDeactivate
                                            write FOnDeactivate;
    property OnSelect : TNodeSelectEvent read FOnSelect
                                            write FOnSelect;
end;

TCreateItemEvent = procedure( Sender : TDBExDataNode; Node :TTreeNode ) of object;

TDBExLinkTypeSet = set of TDBExLinkType;
//------------------------------------------------------------------------

TDBExDataNode = class (TDBExNode)
private
    FItemImageIndex  : integer;             
    FItemSelectedIndex  : integer;          
    FItemStateIndex  : integer;             
    FDriveDataSets   : boolean;             
    FCurrItem        : integer;             
    FFreezeCount     : integer;             
    FTopicVisible    : boolean;             
    FPossibleLinks   : TDBExLinkTypeSet;    
    FLinks           : TDBExNodelinks;
    FCurrentFilter   : TDBExDispatcher;
    FOnCreateItem    : TCreateItemEvent;

    procedure SetLinks( Value : TDBExNodeLinks );
    procedure SetTopicVIsible( Value : boolean );
protected
    FKeyValue         : string;
    FCurrDataItem   : integer;

    procedure FilterByMaster(MasterField:string;
                             MasterValue : string ); virtual; abstract;
    procedure FilterByParam(DetailField:string;
                             MasterValue : string); virtual; abstract;
    procedure FilterByFilter(DetailField:string;
                             MasterValue : string); virtual; abstract;
    function CreateTopicDispatcher(Master : TDBExDispatcher;
                             Before : TTReeNode):TDBExDispatcher; override;
    function CreateItemDispatcher(Master : TDBExDispatcher;
                             Before : TTReeNode):TDBExDispatcher; override;
    procedure SetFilter( Disp:TDBExDispatcher);virtual;
    function CanEdit:boolean; override;

    function DataActive : boolean; virtual; abstract;
    procedure OpenData; virtual; abstract;
    procedure CloseData; virtual; abstract;
    function GetDataItem : string; virtual;
    function GetKeyValue : string; virtual; abstract;
    procedure First;virtual;
    procedure Next;virtual;
    function MoveBy( By : integer ):integer; virtual;
    function RecordCount : integer; virtual;
    function EndOfData : boolean; virtual; abstract;
    function BeginOfData : boolean; virtual;
    procedure FindDependentFields( LinkValues : TDBExLinkValues ); virtual;
    function GetItemImageIndex:Integer;virtual;
    function GetItemSelectedIndex:Integer;virtual;
    function GetItemStateIndex:Integer;virtual;
public

    constructor Create( AOwner : TComponent );override;
    destructor Destroy; override;

    procedure Freeze;virtual;
    procedure UnFreeze;virtual;
    function IsFreezed:boolean;

    function GetField( FieldName : string ) : string; virtual;
    function ParamCount : integer; virtual;
    function ParamName( Index : integer ) : string; virtual;
    function FieldCount : integer; virtual;
    function FieldName( Index : integer ) : string; virtual;
    procedure GetPossibleLinks; virtual;
    property Links : TDBExNodeLinks read FLinks write SetLinks;
    property PossibleLinks : TDBExLinkTypeSet read FPossibleLinks;
    property ItemImageIndex : integer read GetItemImageIndex
                                     write FItemImageIndex;
    property ItemSelectedIndex : integer read GetItemSelectedIndex
                                        write FItemSelectedIndex;
    property ItemStateIndex : integer read GetItemStateIndex
                                     write FItemStateIndex;
    property DriveDataSets : boolean read FDriveDataSets
                                    write FDriveDataSets;
    property TopicVisible : boolean read FTopicVisible
                                    write SetTopicVisible default True;
    property OnCreateItem : TCreateItemEvent read FOnCreateItem write FOnCreateItem;
end;

TDBExCustomNode = class;
TCDBExNodeActive = procedure( Sender : TDBExCustomNode;
                          var Active : boolean )of object;
TCDBExNodeGetData = procedure( Sender : TDBExCustomNode;
                       var Key, ParentKey, LabelValue : string )of object;
TCDBExNodeEndCheck = procedure( Sender : TDBExCustomNode;
                             var IsEnd : boolean )of object;
TCDBExNodeMove = procedure( Sender : TDBExCustomNode)of object;
TCDBExNodeMoveBy = procedure( Sender : TDBExCustomNode;
                         var By : integer)of object;

//------------------------------------------------------------------------



TDBExCustomNode = class( TDBExDataNode )
private
    FActive : boolean;
    FFiltered : boolean;
    FData : pointer;

    FOnActiveChanged : TCDBExNodeActive;
    FOnGetActive : TCDBExNodeActive;
    FOnGetData : TCDBExNodeGetData;
    FOnNext : TCDBExNodeMove;
    FOnFirst : TCDBExNodeMove;
    FOnMoveBy : TCDBExNodeMoveBy;
    FOnEndCheck : TCDBExNodeEndCheck;

    FCurrParentKeyValue : string;
    FCurrKeyValue : string;
    FCurrFilterValue : string;

    FDrivenByMaster : boolean;

    procedure DriveDataset;override;

     procedure FilterByMaster(MasterField:string;
                              MasterValue : string ); override;
     procedure FilterByParam(DetailField:string;
                              MasterValue : string); override;
     procedure FilterByFilter(DetailField:string;
                              MasterValue : string); override;

protected
    function DataActive : boolean; override;
    procedure OpenData; override;
    procedure CloseData;override;
    function GetDataItem : string; override;
    procedure First; override;
    procedure Next; override;
    function MoveBy( By : integer ):integer;override;
    function GetKeyValue : string; override;
    function EndOfData : boolean; override;
    function CanEdit:boolean; override;
public
    constructor Create( AOwner : TComponent );override;

    function FieldCount:Integer;override;
    function GetField( FieldName : string ) : string;override;
    function FieldName( Index : integer ) : string; override;

    property FilterValue : String read FCurrFilterValue;
    property Data : pointer read FData write FData;
published
    property Links;
    property OnActiveChanged : TCDBExNodeActive read FOnActiveChanged
                                               write FOnActiveChanged;
    property OnGetActive : TCDBExNodeActive read FOnGetActive
                                           write FOnGetActive;
    property OnGetData : TCDBExNodeGetData read FOnGetData
                                           write FOnGetData;
    property OnNext : TCDBExNodeMove read FOnNext
                                          write FOnNext;
    property OnFirst : TCDBExNodeMove read FOnFirst
                                          write FOnFirst;
    property OnMoveBy : TCDBExNodeMoveBy read FOnMoveBy
                                          write FOnMoveBy;
    property OnEndCheck : TCDBExNodeEndCheck read FOnEndCheck
                                            write FOnEndCheck;
    property ItemImageIndex;
    property ItemSelectedIndex;
    property ItemStateIndex;
    property OnCreateItem;
end;

//------------------------------------------------------------------------


TCustomDBExDataSetNode = class( TDBExDataNode )
private
     FLabelField : string;           
     FKeyField   : string;           
     FDataLink   : TDBExDataLink;
     FTopicDataSource : TDataSource; 
     FOriginalFilter : String;       
     FCheckInsert : boolean;
     FPostLoading : boolean;
     FDataSet    : TDataSet;
     FReadOnly   : boolean;

     procedure SetDataSource( value : TDataSource );
     function  GetDataSource : TDataSource;
     procedure DriveDataset;override;
     procedure DriveTopicDataset;override;
     procedure SetFilter( Disp:TDBExDispatcher);override;
     procedure SetItemLabel( Value : String ); override;
     procedure SetupNewDataSet( Value : TDataSet ); 
protected
     procedure FilterByMaster(MasterField:string;
                              MasterValue : string ); override;
     procedure FilterByParam(DetailField:string;
                              MasterValue : string); override;
     procedure FilterByFilter(DetailField:string;
                              MasterValue : string); override;
     function DataActive : boolean; override;
     procedure OpenData; override;
     procedure CloseData; override;
     function GetDataItem : string; override;
     procedure First; override;
     procedure Next; override;
     function MoveBy( By : integer ):integer; override;
     function RecordCount : integer; override;
     function GetKeyValue : string;override;
     function EndOfData : boolean; override;
     function BeginOfData : boolean; override;
     function CanEdit:boolean; override;

     procedure LinkActive( Value : boolean );
     procedure RecordChanged( Field : TField );
     procedure DataSetScrolled( Distance : integer );
     procedure DataSetChanged;
     procedure UpdateData;
     procedure EndEdit; override;
     procedure CheckNewRecord;
     procedure GetPossibleLinks; override;
     procedure Notification(AComponent: TComponent;
      Operation: TOperation);override;
public
     constructor Create( AOwner : TComponent );override;
     destructor Destroy; override;


     function GetField( FieldName : string ) : string; override;
     function ParamCount : integer; override;
     function ParamName( Index : integer ) : string; override;
     function FieldCount : integer; override;
     function FieldName( Index : integer ) : string; override;
     procedure Freeze;override;
     procedure UnFreeze;override;

     property ReadOnly : boolean read FReadOnly write FReadOnly;
     property ItemImageIndex;
     property DataSource : TDataSource read GetDataSource
                                      write SetDataSource;
     property LabelField : string read FLabelField write FLabelField;
     property KeyField : string read FKeyField write FKeyField;
     property CheckInsert : boolean read FCheckInsert write FCheckInsert default true;
published
end;

//------------------------------------------------------------------------

TDBExDataSetNode = class (TCustomDBExDataSetNode)
published
    property ItemImageIndex;
    property ItemSelectedIndex;
    property ItemStateIndex;
    property DataSource;
    property LabelField;
    property KeyField;
    property DriveDataSets;
    property Links;
    property CheckInsert;
    property ReadOnly;
    property TopicVisible;
    property OnCreateItem;
end;

//------------------------------------------------------------------------

TDBExLinkNode = class( TDBExDataNode )
private
    FMasterNode       : TCustomDBExDataSetNode;   
    FMasterNodeName   : string;

    procedure SetMasterNode( Value : TCustomDBExDataSetNode);
    procedure DriveDataset; override;
    procedure DriveTopicDataset; override;
    procedure SetActiveTopic(Value : TDBExDataDispatcher);override;
    function GetItem( Index : integer ) : TDBExNode; override;

    function GetItemImageIndex:Integer;override;
    function GetItemSelectedIndex:Integer;override;
    function GetItemStateIndex:Integer;override;

protected
    procedure SetFIlter( Disp:TDBExDispatcher ); override;
    procedure FilterByMaster(MasterField:string;
                             MasterValue : string ); override;
    procedure FilterByParam(DetailField:string;
                             MasterValue : string); override;
    procedure FilterByFilter(DetailField:string;
                             MasterValue : string); override;
    function CanEdit:boolean; override;
    procedure Notification(AComponent: TComponent;
                       Operation: TOperation);override;
    procedure FindDependentFields( LinkValues : TDBExLinkValues ); override;
{$IFDEF _DELPHI3_}
    procedure GetChildren(Proc: TGetChildProc; Root: TComponent); override;
{$ELSE}
    procedure GetChildren(Proc: TGetChildProc); override;
{$ENDIF}


public
    constructor Create( AOwner : TComponent );override;
    destructor Destroy; override;

    function GetField( FieldName : string ) : string; override;
    function ParamCount : integer; override;
    function ParamName( Index : integer ) : string; override;
    function FieldCount : integer; override;
    function FieldName( Index : integer ) : string; override;

    function GetFirstChild: TDBExNode; override;
    function GetLastChild: TDBExNode;  override;
    function GetNextChild(Value: TDBExNode): TDBExNode; override;
    procedure Clear; override;
    function ChildrenCount : integer; override;

    procedure Freeze;override;
    procedure UnFreeze;override;

    function DataActive : boolean; override;
    procedure OpenData; override;
    procedure CloseData; override;
    function GetDataItem : string; override;
    procedure First; override;
    procedure Next; override;
    function MoveBy( By : integer ):integer; override;
    function RecordCount : integer; override;
    function GetKeyValue : string; override;
    function EndOfData : boolean; override;
    function BeginOfData : boolean; override;

published
    property Links;
    property MasterNode : TCustomDBExDataSetNode read FMasterNode
                                                write SetMasterNode;
    property TopicVisible;
    property OnCreateItem;
end;

//------------------------------------------------------------------------

TDBExNodes = class (TPersistent)
private
    FRootNode : TDBExNode;            
    FRootDispatcher : TDBExDIspatcher;
    FOwnerEx  : TCustomDBExplorer;    
    FDrivenDS : TDataSource;
    FNodeHeight : integer;            
                                      

    function GetItems : TTreeNodes;
    function GetNextVisible( Node : TTreeNode ) : TTreeNode;
protected
    FEditorForm : TForm;              
    FLocked   : boolean;              
    FLockSelect   : boolean;
    FLockExpandCollapse : boolean;
    FFirstDisplayedDisp : TDBExDispatcher;

    procedure BeforeExpand( Node : TTReeNode );
    procedure AfterExpand( Node : TTReeNode );
    procedure BeforeCollapse( Node : TTreeNode );
    procedure ClearNode( Disp : TDBExDispatcher );
    function AddTreeNode( Parent : TTreeNode;
                          Disp : TDBExDispatcher;
                          Before : TTreeNode; NodeText : string):TTreeNode;
    procedure PerformExpand( Node : TTreeNode );
    property DrivenDS : TDataSource read FDrivenDS write FDrivenDS;
    property Items : TTreeNodes read GetItems;
public
    FOnTextChange : TTextChangeEvent;
    DisableLinkNodes : boolean;

    constructor Create( AOwner : TComponent );
    destructor Destroy; override;
    procedure Clear;

    function Add( ANodeType : TDBExNodeType; Node:TDBExNode;
                          S : string ):TDBExNode;
    function AddChild( ANodeType : TDBExNodeType; Node:TDBExNode;
                               S : string ):TDBExNode;
    function AddChildFirst( ANodeType : TDBExNodeType; Node:TDBExNode;
                                    S : string ):TDBExNode;
    function AddFirst( ANodeType : TDBExNodeType; Node:TDBExNode;
                               S : string ):TDBExNode;
    function GetFirstNode:TDBExNode;
    procedure Remove( Node : TDBExNode );
    property OwnerEx : TCustomDBExplorer read FOwnerEx write FOwnerEx;
    function UniqueName( ClassName : string):string;
    procedure InitShow;
    procedure CloseEditor;
    property RootNode : TDBExNode read FRootNode;
published
end;


TCreateNewKeyValue = procedure(Sender: TObject; var NewKeyValue: Variant) of Object;
TDragDropNode = procedure(Destination, Source : TDBExDispatcher;
                    Var Accept,DoCopy : Boolean) of object;
//------------------------------------------------------------------------
TCustomDBExplorer = class( TCustomTreeView )
private
    FMayDrop : boolean;
    FNodes : TDBExNodes;
    FStructDone : boolean;
    FPrevScrollPos : integer;
    FLockOnLoad : boolean;
    FullDestroy: Boolean;
    FDragCursorCopy:TCursor;
    FDragTreeNode:TTreeNode;
    FOnDragDropNode:TDragDropNode;

    FScrollTimerID:Integer;
    FDragObject:TDragObject;
    procedure SetDrivenDS( Value : TDataSource );
    function GetDrivenDS : TDataSource;
    procedure PassSelection(DFrom, DTo : TDBExDispatcher);

protected
    FWinSize : integer;
    FBlockPaint : boolean;

    procedure WMDestroy(var Message : TWMDestroy); message WM_DESTROY;
    procedure WMRButtonUp(var Message: TWMRButtonUp); message WM_RButtonUp;
    procedure WMMouseMove(var Message: TWMMouseMove); message WM_MOUSEMOVE;
    procedure CreateWnd; override;

    procedure CMDrag(var Message:TCMDrag); message CM_Drag;
    procedure Edit(const Item: TTVItem); override;
    function CanEdit(Node: TTreeNode): Boolean;override;
    function CanExpand(Node: TTreeNode): Boolean;override;
    function CanChange(Node: TTreeNode): Boolean;override;
    function CanCollapse(Node: TTreeNode): Boolean;override;
    procedure Expand(Node: TTreeNode);override;
    procedure Loaded;override;
    procedure WndProc(var Message: TMessage);override;

    procedure DoStartDrag(var DragObject: TDragObject);override;
    procedure DragOver(Source: TObject; X, Y: Integer; State: TDragState;var Accept: Boolean);override;
    procedure DragDrop(Source:TObject; X,Y:Integer);override;

{$IFDEF _DELPHI3_}
    procedure GetChildren(Proc: TGetChildProc; Root: TComponent); override;
{$ELSE}
    procedure GetChildren(Proc: TGetChildProc); override;
{$ENDIF}
    procedure SetName(const Value: TComponentName); override;

public
    constructor Create( AOwner : TComponent ); override;
    destructor Destroy; override;

    property Nodes : TDBExNodes read FNodes write FNodes;
    property DrivenDS : TDataSource read GetDrivenDS write SetDrivenDS;
    property Items stored False;
    property OnDragDropNode: TDragDropNode read FOnDragDropNode write FOnDragDropNode;



end;

//------------------------------------------------------------------------
TDBExplorer = class( TCustomDBExplorer )
published
    property ShowButtons;
    property BorderStyle;
    property DragCursor;
    property ShowLines;
    property ShowRoot;
    property ReadOnly;
    property DragMode;
    property HideSelection;
    property Indent;
    property OnEditing;
    property OnEdited;
    property OnExpanding;
    property OnExpanded;
    property OnCollapsing;
    property OnCompare;
    property OnCollapsed;

    property OnChanging;
    property OnChange;
    property OnDeletion;
    property OnDragDropNode;
    property OnGetImageIndex;
    property OnGetSelectedIndex;
    property Align;
    property Enabled;
    property Font;
    property Color;
    property ParentColor;
    property ParentCtl3D;
    property Ctl3D;
    property TabOrder;
    property TabStop default True;
    property Visible;
    property OnClick;
    property OnEnter;
    property OnExit;
    property OnDragDrop;
    property OnDragOver;
    property OnStartDrag;
    property OnEndDrag;
    property OnMouseDown;
    property OnMouseMove;
    property OnMouseUp;
    property OnDblClick;
    property OnKeyDown;
    property OnKeyPress;
    property OnKeyUp;
    property PopupMenu;
    property ParentFont;
    property ParentShowHint;
    property ShowHint;
    property Images;
    property StateImages;
    property Nodes;
    property DrivenDS;
end;

procedure Register;




//========================================================================

var
  TimerMessage:Boolean;

implementation
{$BOOLEVAL OFF}
{$R DBEXPLOR.RES}
uses  DBExEd, DBExNLEd, TypInfo;

var
  LoadedObjects : integer;
  Time1, Time2, Time3 : integer;
  hInspector : hWnd;
  Unregistered : boolean;
//  ProccesingExpand:Boolean;


// Property editors
//------------------------------------------------------------------------
procedure ExplorerEdit( Nodes : TDBExNodes; Designer : TFormDesigner );
var
    DBExEditorForm : TDBExEditorForm;
begin
    if Nodes.FEditorForm<>nil then      
        Nodes.FEditorForm.BringToFront
    else
    begin
        DBExEditorForm:=TDBExEditorForm.Create(Application);
        DBExEditorForm.DBEx:=TCustomDBExplorer(Nodes.FOwnerEx);
        Nodes.FEditorForm :=DBExEditorForm;
        DBExEditorForm.FormDesigner:=Designer;
        DBExEditorForm.Show;
    end;
end;

procedure LinksEdit(Links:TDBExNodeLinks);
var
    NodeLinksEditorForm : TNodeLinksEditorForm;
begin
    NodeLinksEditorForm:=TNodeLinksEditorForm.Create(Application);
    NodeLinksEditorForm.SetNode(Links.OwnerNode);
    NodeLinksEditorForm.ShowModal;
end;


type
TDBExNodesEditor = class( TClassProperty)
public
   procedure Edit; override;
   function GetAttributes: TPropertyAttributes; override;
end;

//------------------------------------------------------------------------
procedure TDBExNodesEditor.Edit;
begin
    ExplorerEdit(TDBExNodes(GetOrdValue),Designer);
end;

//------------------------------------------------------------------------
function TDBExNodesEditor.GetAttributes: TPropertyAttributes;
begin
    Result:=[paDialog,paReadOnly];
end;

type
//------------------------------------------------------------------------
TDBExNodeLinksEditor = class( TClassProperty)
public
   procedure Edit; override;
   function GetAttributes: TPropertyAttributes; override;
end;

//------------------------------------------------------------------------
procedure TDBExNodeLinksEditor.Edit;
begin
    LinksEdit(TDBExNodeLinks(GetOrdValue));
end;

//------------------------------------------------------------------------
function TDBExNodeLinksEditor.GetAttributes: TPropertyAttributes;
begin
    Result:=[paDialog,paReadOnly];
end;

//------------------------------------------------------------------------
{TFieldProperty}
type
  TFieldProperty = class(TStringProperty)
  public
    function GetAttributes: TPropertyAttributes; override;
    procedure GetValueList(List: TStrings); virtual;
    procedure GetValues(Proc: TGetStrProc); override;
  end;

//------------------------------------------------------------------------
function TFieldProperty.GetAttributes: TPropertyAttributes;
begin
  Result := [paValueList, paSortList, paMultiSelect];
end;

//------------------------------------------------------------------------
procedure TFieldProperty.GetValues(Proc: TGetStrProc);
var
  I: Integer;
  Values: TStringList;
begin
  Values := TStringList.Create;
  try
    GetValueList(Values);
    for I := 0 to Values.Count - 1 do Proc(Values[I]);
  finally
    Values.Free;
  end;
end;

//------------------------------------------------------------------------
procedure TFieldProperty.GetValueList(List: TStrings);
var
  Instance: TComponent;
  PropInfo: PPropInfo;
  DataSource: TDataSource;
begin
  Instance := TComponent(GetComponent(0));
  PropInfo := TypInfo.GetPropInfo(Instance.ClassInfo, 'DataSource');
  if (PropInfo <> nil) and (PropInfo^.PropType^.Kind = tkClass) then
  begin
    DataSource := TObject(GetOrdProp(Instance, PropInfo)) as TDataSource;
    if (DataSource <> nil) and (DataSource.DataSet <> nil) then
      DataSource.DataSet.GetFieldNames(List);
  end;
end;

//------------------------------------------------------------------------
{TMasterNodeProperty}
type
  TMasterNodeProperty = class(TPropertyEditor)
  private
    FNodeName : string;
  public
    function GetAttributes: TPropertyAttributes; override;
    procedure GetValues(Proc: TGetStrProc); override;
    function GetValue : string; override;
    procedure SetValue( const Value : string); override;
  end;

//------------------------------------------------------------------------
function TMasterNodeProperty.GetValue : string;
var
    ThisNode : TDBExLinkNode;
begin
    ThisNode := TDBExLinkNode(GetComponent(0));
    if ThisNode.MasterNode=nil then
        Result:=''
    else
        Result:=ThisNode.MasterNode.Name;
end;

//------------------------------------------------------------------------
procedure TMasterNodeProperty.SetValue( const Value : string);
var
    ThisNode : TDBExLinkNode;
begin
    ThisNode := TDBExLinkNode(GetComponent(0));
    if Value='' then
        ThisNode.MasterNode:=nil
    else
        ThisNode.MasterNode:=
          TCustomDBExDataSetNode(ThisNode.Owner.FindComponent(Value));
end;

//------------------------------------------------------------------------
function TMasterNodeProperty.GetAttributes: TPropertyAttributes;
begin
  Result := [paValueList, paSortList, paMultiSelect];
end;

//------------------------------------------------------------------------
procedure TMasterNodeProperty.GetValues(Proc: TGetStrProc);
var
    ThisNode : TDBExLinkNode;
    Node : TDBExNode;
begin
    ThisNode := TDBExLinkNode(GetComponent(0));
    Node:= ThisNode.OwnerNodes.GetFirstNode;
    while Node <> nil do
    begin
        if (Node is TCustomDBExDataSetNode) and
           (Node <> ThisNode) then
            Proc( Node.Name );
        Node:=Node.GetNext;
    end;
end;

//------------------------------------------------------------------------
type
  TDBExplorerEditor = class(TComponentEditor)
    procedure ExecuteVerb(Index: Integer); override;
    function GetVerb(Index: Integer): string; override;
    function GetVerbCount: Integer; override;
  end;

//------------------------------------------------------------------------
procedure TDBExplorerEditor.ExecuteVerb(Index: Integer);
begin
  case Index of
    0:     ExplorerEdit(TDBExplorer(Component).Nodes,Designer);
  end;
end;

//------------------------------------------------------------------------
function TDBExplorerEditor.GetVerb(Index: Integer): string;
begin
  case Index of
    0: Result := 'Edit Nodes...';
  end;
end;

//------------------------------------------------------------------------
function TDBExplorerEditor.GetVerbCount: Integer;
begin
  Result := 1;
end;


//------------------------------------------------------------------------
procedure Register;
begin
  RegisterComponents('RSD Data Controls', [TDBExplorer]);

  RegisterClasses( [TDBExNodes,TDBExNode, TDBExDataSetNode,
         TDBExLinkNode, TDBExDataNode,
         TCustomDBExDataSetNode]);

  RegisterNoIcon([TDBExNode,TDBExDataSetNode,
         TDBExLinkNode, TDBExDataNode,
         TCustomDBExDataSetNode]);

  RegisterPropertyEditor(TypeInfo(TDBExNodes), nil, '', TDBExNodesEditor);
  RegisterPropertyEditor(TypeInfo(TDBExNodeLinks),TDBExDataNode,'Links',TDBExNodeLinksEditor);
  RegisterPropertyEditor(TypeInfo(string),
                     TCustomDBExDataSetNode, 'LabelField', TFieldProperty);
  RegisterPropertyEditor(TypeInfo(string),
                     TCustomDBExDataSetNode, 'KeyField', TFieldProperty);
  RegisterPropertyEditor(TypeInfo(TCustomDBExDataSetNode),
                     TDBExLinkNode, 'MasterNode', TMasterNodeProperty);

  RegisterComponentEditor(TDBExplorer,TDBExplorerEditor);
  end;

type
TAutoDragObject = class(TDragControlObject)
protected
  function GetDragCursor(Accepted: Boolean; X, Y: Integer): TCursor; override;
end;

function TAutoDragObject.GetDragCursor(Accepted: Boolean; X, Y: Integer): TCursor;
begin
  if Accepted then begin
    if Not (GetKeyState(VK_CONTROL) < 0) then
      Result := TCustomDBExplorer(Control).DragCursor;
   end else Result := crNoDrop;
end;

procedure ScrollTreeViewTimerProc(Wnd: HWnd; Msg, TimerID, SysTime: Longint); stdcall;
Var  p : TPoint;
begin
  TimerMessage:=True;
  GetCursorPos(p);
  SendMessage(Wnd, LongInt(WM_MOUSEMOVE), MK_LBUTTON, MAKELONG(p.X, p.Y));
  TimerMessage:=False;
end;


//========================================================================
// TDBExDispatcher
//========================================================================
constructor TDBExDispatcher.Create( TopicNode : TDBExNode );
begin
    inherited Create;
    FTopicNode:=TopicNode;
    FExplorer:=nil;
    FTreeNode:=nil;
    FFirstItem := -1;
    FLastItem := -1;
    FSelectedItem := -1;
    FNodes := Topicnode.OwnerNodes;
    FDispatchers:=TList.Create;
    FTop := 0;
    FBottom := 0;
    FExpanded := False;
    FLinkValues := TDBExLinkValues.Create( Self );
    FNextTopicVisible := True;
    FInvisibleDisp := nil;
end;

//------------------------------------------------------------------------
destructor TDBExDispatcher.Destroy;
var
    P,N : TTreeNode;
begin
    while DispatcherCount>0 do
        Dispatchers[DispatcherCount-1].Free;
    if FParentDispatcher<>nil then
        FParentDispatcher.FDispatchers.Remove( Self );

    if FTopicNode<>nil then
    begin
        FTopicNode.FTopicDispatchers.Remove( Self );
        if (FTopicNode.ActiveTopic=self) and (not FExplorer.FullDestroy) then
            FTopicNode.ActiveTopic:=nil;
    end;


    if FTreeNode<>nil then
        FTreeNode.Delete;

    FLinkValues.Free;
    FDispatchers.Free;
    inherited Destroy;
end;

//------------------------------------------------------------------------
procedure TDBExDispatcher.BeforeExpand;
begin
    if FTopicNode=nil then Exit;
    if FParentDispatcher<>nil then   
        FParentDispatcher.NotifyExpand( Self );
    FExpanded := True;
    FFirstItem := -1;
    FLastItem := -1;
    ClearSubtree;
end;

//------------------------------------------------------------------------
procedure TDBExDispatcher.AfterExpand;
begin

end;

//------------------------------------------------------------------------
procedure TDBExDispatcher.SetTreeNode( Value : TTreeNode );
begin
    if FTreeNode=Value then Exit;

    if Value<>nil then
        Value.Data:=Self
    else
        FTreeNode.Data:=nil;
    FTreeNode := Value;
end;

//------------------------------------------------------------------------

function TDBExDispatcher.GetTreeNode : TTreeNode;
begin
    if (FTreeNode<>nil) then
        Result := FTreeNode
    else
        if (ParentDispatcher<> nil) then
            Result := ParentDispatcher.FtreeNode
        else
            Result := nil;
end;

//------------------------------------------------------------------------
procedure TDBExDispatcher.SetText( Value : string );
begin
    FText:=Value;
    if FTreeNode<>nil then
        FTreeNode.Text:=Value;
end;

//------------------------------------------------------------------------
function TDBExDispatcher.GetText:string;
begin
        Result:=FText;
end;

//------------------------------------------------------------------------
function TDBExDispatcher.GetTopicNode : TDBExNode;
begin
    if FTopicNode is TDBExLinkNode then
        Result := TDBExLinkNode(FTopicNode).MasterNode
    else
        Result := FTopicNode;
end;

//------------------------------------------------------------------------
procedure TDBExDispatcher.ClearSubtree;
begin
    while DispatcherCount>0 do
        Dispatchers[DispatcherCount-1].Free;
    if FTreeNode <> nil then
        FTreeNode.DeleteChildren;
end;

//------------------------------------------------------------------------
procedure TDBExDispatcher.OnCollapse;
begin
    if FInvisibleDisp<>nil then   
        FInvisibleDIsp.OnCollapse;
    if FParentDispatcher<>nil then
        FParentDispatcher.NotifyCollapse( Self );









    ClearSubtree;
    FExpanded := False;
    if FTreeNode<>nil then        
        FTreeNode.Owner.AddChild(FtreeNode, 'NONE');
    Select;
end;

//------------------------------------------------------------------------
procedure TDBExDispatcher.NotifyExpand( Child : TDBExDispatcher );
begin
end;

//------------------------------------------------------------------------
procedure TDBExDispatcher.NotifyCollapse( Child : TDBExDispatcher );
begin
end;

//------------------------------------------------------------------------
procedure TDBExDispatcher.SetActiveItem( Value : TDBExDispatcher );
begin
    if Value<>nil then
    begin
        FActiveItem:=Value;
        FSelectedItem := Value.FItemIndex;
    end
    else
        FSelectedItem := -1;
end;

//------------------------------------------------------------------------
procedure TDBExDispatcher.SetItemIndex( Value : integer);
begin
    FItemIndex:=Value;
end;

//------------------------------------------------------------------------
procedure TDBExDispatcher.AttachTreeNode(Items : TTreeNodes;       
                                         Master : TDBExDispatcher; 
                                         Before : TTreeNode;       
                                         Index : integer);         
var
    N : TTreeNode;
begin
    if Master = nil then Exit;
    if FTreeNode=nil then 
    begin
        N:=Master.FtreeNode;
        if (N=nil) and (Master.ParentDispatcher<>nil) then
            N:=Master.ParentDispatcher.FTreeNode;
        FTreeNode:=FNodes.AddTreeNode(N,Self,Before, FText);
    end;
    FNodes.OwnerEx.FBlockPaint := True;
    if (CanHasChildren) and (FTreeNode.GetFirstChild=nil) then
    begin
        FTreeNode.Owner.AddChild(FTreeNode,'NONE');
    end;
    if Master is TDBExDataDispatcher then
    begin
        FTreeNode.ImageIndex:=TDBExDataNode(Master.FTopicNode).ItemImageIndex;
        FTreeNode.SelectedIndex:=TDBExDataNode(Master.FTopicNode).ItemSelectedIndex;
        FTreeNode.StateIndex:=TDBExDataNode(Master.FTopicNode).ItemStateIndex;
    end
    else
    begin
        FTreeNode.ImageIndex:=FTopicNode.TopicImageIndex;
        FTreeNode.SelectedIndex:=FTopicNode.TopicSelectedIndex;
        FTreeNode.StateIndex:=FTopicNode.TopicStateIndex;
    end;
    FNodes.OwnerEx.FBlockPaint := False;
end;

//------------------------------------------------------------------------
function TDBExDispatcher.GetExpanded : boolean;
begin
    if (FTreeNode <> nil)  then
         Result := FTreeNode.Expanded
    else
         Result := FExpanded;
end;

//------------------------------------------------------------------------
procedure TDBExDispatcher.ClearRest;
var
    TN : TTreeNode;
begin
    TN:=treeNode.GetFirstChild;
    while (TN<>nil)
      and (TN.Data=nil) do
    begin
        TN.Delete;
        TN:=treeNode.GetFirstChild;
    end;
end;

//------------------------------------------------------------------------
procedure TDBExDispatcher.Deselect;
begin
    FSelectedItem := -1;
end;

//------------------------------------------------------------------------
procedure TDBExDispatcher.SaveLinkValues;
var
    DataNode : TDBExDataNode;
    i : integer;
    LV : TDBExLinkValue;
begin
    if (ParentDispatcher <> nil) and
       (ParentDispatcher is TDBExDataDispatcher) and
       (ParentDispatcher.FTopicNode <> nil) then
    begin
        DataNode:=TDBExDataNode(ParentDispatcher.FTopicNode);
        for i:=0 to
            TDBExDataDispatcher(ParentDispatcher).FLinkTemplate.Count-1 do
        begin
            LV :=TDBExLinkValue.Create;
            LV.MasterField := TDBExDataDispatcher(
                         ParentDispatcher).FLinkTemplate.Value[i].MasterField;
            LV.MasterValue := DataNode.GetField( LV.MasterField );
            FLinkValues.Add( LV );
        end;
    end;
end;

function TDBExDispatcher.GetDispatchers(Index : integer):TDBExDispatcher;
begin
    Result:=TDBExDispatcher( FDispatchers[Index] );
end;

function TDBExDispatcher.GetDispatcherCount:integer;
begin
    Result := FDispatchers.Count;
end;

//========================================================================
// TDBExMetaDispatcher
//========================================================================
constructor TDBExMetaDispatcher.Create( TopicNode : TDBExNode );
begin
    inherited Create( TopicNode );
end;

destructor TDBExMetaDispatcher.Destroy;
begin
    inherited Destroy;
end;

//------------------------------------------------------------------------
function TDBExMetaDispatcher.CreateDispatcher(Node:TDBExNode;
   Before : TTreeNode;DoExpand : boolean;Index:integer):TDBExDispatcher;
begin
    Result:=Node.CreateDispatcher(Self, Before);
    if FNextTopicVisible then
        Result.AttachTreeNode(FNodes.OwnerEx.Items,
                             Self,Before, Index-FFirstItem);
    FDispatchers.Add(Result);
    Result.ParentDispatcher:=Self;
    Result.SetItemIndex(Index);
end;

//------------------------------------------------------------------------
procedure TDBExMetaDispatcher.FillDown;
var
    N : TDBExNode;
    Disp : TDBExDispatcher;
    i : integer;
begin
    if (FTopicNode.ChildrenCount=1)
         and (FTopicNode.GetFirstChild is TDBExDataNode)
         and(not TDBExDataNode(FTopicNode.GetFirstChild).TopicVisible) then
    begin
        FNextTopicVisible := False;
        Disp := CreateDispatcher(FTopicNode.GetFirstChild, nil, True,0 );
        Disp.BeforeExpand;
        Disp.FillDown;
        Disp.AfterExpand;
        FInvisibleDisp:=Disp;
    end
    else
    begin
        i:=FLastItem;
        Inc(i);
        N:=FTopicNode.Item[i];
        if N<>nil then
        repeat
            Disp:=CreateDispatcher(N,nil, True,i);
            Inc(i);
            N:=N.GetNextSibling;
        until (N=nil);
        FLastItem := i-1;
    end;
end;

//------------------------------------------------------------------------
function TDBExMetaDispatcher.AddBefore( Node : TTreeNode;
                                      DoExpand : boolean ):TDBExDispatcher;
var
    BeforePos : integer;
begin
    if Node<>nil then
        BeforePos := TDBExDispatcher(Node.Data).ItemIndex
    else
        BeforePos := FTreeNode.Count;
    if BeforePos<=0 then
        Result := nil
    else
        Result := CreateDispatcher( FTopicNode.Item[BeforePos-1],
                                               Node ,DoExpand,BeforePos-1);
end;

//------------------------------------------------------------------------
procedure TDBExMetaDispatcher.EndEdit;
begin
    if FTopicNode<>nil then
        FTopicNode.EndEdit;
end;

//------------------------------------------------------------------------
function TDBExMetaDispatcher.CanEdit:boolean;
begin
    if FTopicNode<>nil then
        Result:=FTopicNode.CanEdit
    else
        Result := False;
end;

//------------------------------------------------------------------------
procedure TDBExMetaDispatcher.Select;
begin
    if (FTopicNode <> nil) and
       (not FTopicNode.FOwnerNodes.FLockSelect) then
    begin
       if FParentDispatcher<>nil then
       begin
            if FParentDispatcher is TDBExDataDispatcher then
            begin
                TDBExDataDispatcher(FParentDispatcher).ApplyFilter;
                FTopicNode.ActiveTopic:=TDBExDataDispatcher(FParentDispatcher);
            end;
            FParentDispatcher.ActiveItem:=Self;
       end;
       FTopicNode.DriveDataSet;
       if Assigned( FTopicNode.FOnSelect ) then
           FTopicNode.FOnSelect( FTopicNode, False );
    end;
end;

//------------------------------------------------------------------------
procedure TDBExMetaDispatcher.Deselect;
begin
    inherited Deselect;
    if FParentDispatcher<>nil then
    begin
        FParentDispatcher.ActiveItem:=nil;
        FTopicNode.ActiveTopic:=nil;
    end;
end;

//------------------------------------------------------------------------
function TDBExMetaDispatcher.GetKey:string;
begin
    Result:=FKeyValue;
end;

//------------------------------------------------------------------------
procedure TDBExMetaDispatcher.SetText( Value : string );
begin
    inherited SetText( Value );
    if (FTopicNode<>nil)  then
        FTopicNode.SetItemLabel( Value );
end;

//------------------------------------------------------------------------
function TDBExMetaDispatcher.CanHasChildren: boolean ;
begin
    Result := (FTopicNode.GetFirstChild<>nil);
end;


//========================================================================
// TDBExDataDispatcher
//========================================================================
constructor TDBExDataDispatcher.Create( TopicNode : TDBExNode );
begin
    inherited Create( TopicNode );
    FDataTopicNode:= TDBExDataNode(TopicNode);
    FLinkTemplate := TDBExLinkValues.Create( Self );
    FSavedActiveDispatcher:=nil;
end;

//------------------------------------------------------------------------
destructor TDBExDataDispatcher.Destroy;
begin
    FLinkTemplate.Free;
    inherited Destroy;
end;

//------------------------------------------------------------------------
procedure TDBExDataDispatcher.BeforeExpand;
begin
    try
        inherited BeforeExpand;
        FDataTopicNode.Freeze;
        SaveDataState;
        ApplyFilter;
        FDataTopicNode.First;
        FDataTopicNode.FindDependentFields( FLinkTemplate );
    except
        FDataTopicNode.UnFreeze;
        raise
    end;
end;

procedure TDBExDataDispatcher.AfterExpand;
begin
    FDataTopicNode.First;
    if treeNode <> nil then
    begin
        if treeNode.Count<>0 then
            FActiveItem:=TDBExDispatcher(TreeNode.GetFirstChild.Data);
    end
    else
    begin
        if (FExplorer.Items.GetFirstNode<>nil) then
            FActiveItem:=TDBExDispatcher(FExplorer.Items.GetFirstNode.Data)
        else
            FActiveItem := nil;
    end;
    RestoreDataState;
    FDataTopicNode.Unfreeze;
end;

//------------------------------------------------------------------------
procedure TDBExDataDispatcher.MoveFirst;
begin
    if (not FTreeNode.Expanded) or (FTopicNode=nil) then Exit;
end;

//------------------------------------------------------------------------
procedure TDBExDataDispatcher.MoveLast;
begin
    if (not FTreeNode.Expanded) or (FTopicNode=nil) then Exit;
end;

//------------------------------------------------------------------------
procedure TDBExDataDispatcher.EndEdit;
begin

end;


//------------------------------------------------------------------------
function TDBExDataDispatcher.CanEdit:boolean;
begin
    if FTopicNode<>nil then
        Result:=FTopicNode.CanEdit
    else
        Result := False;
end;

//------------------------------------------------------------------------
procedure TDBExDataDispatcher.ApplyFilter;
begin
    if FDataTopicNode<>nil then
    begin
        FDataTopicNode.Freeze;
        if FDataTopicNode.DriveDataSets then
            FDataTopicNode.OpenData;
        if FParentDispatcher <> nil  then
            FDataTopicNode.SetFilter(
                   FParentDispatcher);  
        FDataTopicNode.Unfreeze;
    end;
end;

//------------------------------------------------------------------------
procedure TDBExDataDispatcher.OnCollapse;
begin
    if (csDestroying in FtopicNode.ComponentState) then Exit;
    try
        FExplorer.Items.BeginUpdate;
        FDataTopicNode.FOwnerNodes.FLockSelect := True;
        FDataTopicNode.Freeze;
        FDataTopicNode.CloseData;
        inherited OnCollapse;
    finally
        FDataTopicNode.UnFreeze;
        FExplorer.Items.EndUpdate;
        FDataTopicNode.FOwnerNodes.FLockSelect := False;
        if (FSelectedItem <> -1)and (TreeNode<>nil) then
            TreeNode.Selected:=True;
    end;
end;

//------------------------------------------------------------------------
function TDBExDataDispatcher.CreateDispatcher(Before : TTreeNode;
                     DoExpand : boolean; Index : integer):TDBExDispatcher;
var
    SaveData : Pointer;
begin
    Result:=FDataTopicNode.CreateDispatcher(Self, Before);
    Result.AttachTreeNode(FNodes.OwnerEx.Items,Self,Before,  Index-FFirstItem);
    FDispatchers.Add(Result);
    Result.ParentDispatcher:=Self;
    Result.SetItemIndex(Index);
    Result.SaveLinkValues;
    if (Result.FTreeNode<>nil) and
       (Assigned(FDataTopicNode.FOnCreateItem)) then
       begin
         SaveData:=Result.FtreeNode.Data;
         FDataTopicNode.FOnCreateItem(FDataTopicNode,Result.FtreeNode);
         Result.FtreeNode.Data:=SaveData;
       end;
end;

//------------------------------------------------------------------------
procedure TDBExDataDispatcher.FillDown;
var
    Disp : TDBExDispatcher;
    i : integer;
begin
    i:=FLastItem+1;
    FSelectedItem:=-1;
    FDataTopicNode.First;
    FDataTopicNode.MoveBy(i);
    while (not FDataTopicNode.EndOfData) do
    begin
        Disp := CreateDispatcher(nil,False,i);
        if (FDataTopicNode.EndOfData) then
        begin
          Disp.free;
          continue;
        end;
        if Disp.FKeyValue=FSavedKeyValue then
            FSelectedItem := i;
        Inc(i);
        FDataTopicNode.Next;
    end;
    FLastItem := i-1;
end;

//------------------------------------------------------------------------
function TDBExDataDispatcher.AddBefore( Node : TTreeNode ;
                                    DoExpand : boolean):TDBExDispatcher;
var
    BeforePos : integer;
begin
    if Node<>nil then
        BeforePos := TDBExDispatcher(Node.Data).ItemIndex
    else
    begin
        BeforePos := FDataTopicNode.RecordCount;
        FLastItem := BeforePos-1;
    end;
    if BeforePos<=0 then
        Result := nil
    else
    begin
        FDataTopicNode.First;
        FDataTopicNode.MoveBy(BeforePos-1);
        Result := CreateDispatcher( Node , DoExpand,BeforePos-1);
    end;
end;

//------------------------------------------------------------------------
procedure TDBExDataDispatcher.Refresh;
begin
    InternalRefresh;
end;

//------------------------------------------------------------------------
procedure TDBExDataDispatcher.InternalRefresh;
var
    P,N : TTreeNode;
    HasSelection : boolean;
    SaveExpanded : boolean;
begin
    if (TreeNode=nil) or
       (TreeNode.Owner=nil) or
       (csDesigning in TreeNode.Owner.Owner.ComponentState) or
       (FDataTopicNode=nil) or
       (FDataTopicNode.FOwnerNodes=nil) then Exit;

    TreeNode.Owner.BeginUpdate;
    FDataTopicNode.FOwnerNodes.FLockSelect := True;
    FDataTopicNode.FOwnerNodes.FLockExpandCollapse := True;
    FDataTopicNode.Freeze;
    FActiveItem:=nil;

    SaveExpanded:=TreeNode.Expanded;
    HasSelection := (FSelectedItem<>-1);
    N:=TreeNode.GetFIrstChild;
    while (N<>nil) and (N.Data<>nil) do
    begin
        P:=N.GetNextSibling;
        FNodes.ClearNode( TDBExDispatcher(N.Data));
        N:=P;
    end;
    SaveDataState;
    FFirstItem:=0;
    FLastItem:=-1;
    FSavedKeyValue := FDataTopicNode.GetKeyValue;
    ApplyFilter;
    FillDown;
    ClearRest;
    RestoreDataState;
    FDataTopicNode.First;
    FDataTopicNode.MoveBy(FSelectedItem);
    TreeNode.Expanded:=SaveExpanded;
    if HasSelection and
       (FSelectedItem <=FLastItem) and
       (FSelectedItem >=FFirstItem)   then
    begin
        N:=TreeNode.Item[FSelectedItem-FFirstItem];
        N.Selected:=True;
        FParentDispatcher.ActiveItem:=TDBExDispatcher(N.Data);
    end
    else
        FSelectedItem := -1;
    FDataTopicNode.UnFreeze;
    TreeNode.Owner.EndUpdate;
    FDataTopicNode.FOwnerNodes.FLockExpandCollapse := False;
    FDataTopicNode.FOwnerNodes.FLockSelect := False;
end;
//------------------------------------------------------------------------
procedure TDBExDataDispatcher.LabelChanged( Value : string );
begin
    if FActiveItem<>nil then
       FActiveItem.FText := Value;
end;

//------------------------------------------------------------------------
procedure TDBExDataDispatcher.KeyChanged( Value : string );
begin
    if FActiveItem<>nil then
    begin
        FActiveItem.KeyValue := Value;
        FDataTopicNode.FOwnerNodes.FLockSelect := True;
        if FActiveItem.Expanded then
            FactiveItem.OnCollapse;
        FDataTopicNode.FOwnerNodes.FLockSelect := False;
    end;
end;

//------------------------------------------------------------------------
procedure TDBExDataDispatcher.Scroll( Distance : integer );
begin
    if FSelectedItem <>-1 then
    begin
        FSelectedItem := FSelectedItem + Distance;
        if FSelectedItem < FFirstItem then
            FSelectedItem := FFirstItem
        else
            if FSelectedItem > FLastItem then
                FSelectedItem := FLastItem;
        FDataTopicNode.FOwnerNodes.FLockSelect := True;
        try
            TreeNode.Item[FSelectedItem].Selected:=True;
        finally
            FDataTopicNode.FOwnerNodes.FLockSelect := False;
        end;
    end;
end;

//------------------------------------------------------------------------
procedure TDBExDataDispatcher.SelectionMoved( DFrom, DTo : TDBExDataDispatcher);
begin
    if FDataTopicNode.IsFreezed then Exit;

    FDataTopicNode.Freeze;
    try
        if (FDataTopicNode.DataActive) then
        begin
            FDataTopicNode.MoveBy( DTo.FItemIndex - DFrom.FItemIndex);
        end;
        ActiveItem := DTo;
    finally
        FDataTopicNode.UnFreeze;
    end;
end;

//------------------------------------------------------------------------
function TDBExDataDispatcher.GetKey:string;
begin
    if FParentDispatcher<>nil then
        Result:=FParentDispatcher.GetKey
    else
        Result:='';
end;

//------------------------------------------------------------------------
procedure TDBExDataDispatcher.Deselect;
begin
    inherited Deselect;
    if FDataTopicNode<>nil then
        FDataTopicNode.ActiveTopic:=nil;
end;

//------------------------------------------------------------------------
procedure TDBExDataDispatcher.Select;
begin
    if (FDataTopicNode <> nil) and
       (not FDataTopicNode.FOwnerNodes.FLockSelect) then
    begin
       FDataTopicNode.DriveDataSet;
       FDataTopicNode.ActiveTopic:=Self;
       ApplyFilter;
       if Assigned( FTopicNode.FOnSelect ) then
           FTopicNode.FOnSelect( FTopicNode, True );
    end;
end;

//------------------------------------------------------------------------
procedure TDBExDataDispatcher.SaveDataState;
begin
    if (FDataTopicNode.ActiveTopic<>nil)and
       (FDataTopicNode.ActiveTopic.FParentDispatcher<>FParentDispatcher)
        then
        FSavedActiveDispatcher:=FDataTopicNode.ActiveTopic;

end;

//------------------------------------------------------------------------
procedure TDBExDataDispatcher.RestoreDataState;
begin
    if FSavedActiveDispatcher<>nil then
        FDataTopicNode.SetFilter( FSavedActiveDispatcher.FParentDispatcher );
    FSavedActiveDispatcher:=nil;

end;

//------------------------------------------------------------------------
procedure TDBExDataDispatcher.SetActiveItem(  Value : TDBExDispatcher );
begin
    inherited SetActiveItem( Value );
    if Value<>nil then
    begin
        FDataTopicNode.Freeze;

        FDataTopicNode.First;
        FDataTopicNode.MoveBy( Value.FItemIndex );
        FDataTopicNode.Unfreeze;
        FDataTopicNode.FActiveTopic:=Self;
    end;
end;

//------------------------------------------------------------------------
function TDBExDataDispatcher.CanHasChildren;
begin
    Result := True;
end;

//------------------------------------------------------------------------
procedure TDBExDataDispatcher.FillLinkFields( DataSet : TDataSet );
var
    Links : TDBExNodeLinks;
    LV : TDBExLinkValue;
    F : TField;
    i : integer;
begin
    Links := FDataTopicNode.Links;
    if Links <> nil then
    begin
        for i:=0 to Links.Count-1 do
        begin
            LV := ParentDispatcher.FLinkValues.Find( Links.Items[i] );
            F := DataSet.FindField( Links.Items[i].DetailField );
            if (LV<>nil) and (F<>nil) then
                F.AsString := LV.MasterValue;
        end;
    end;
end;


//========================================================================
// TDBExNodeLink
//========================================================================
constructor TDBExNodeLink.Create(Collection: TCollection);
begin
    inherited Create( Collection );
    FMasterNode := nil;
//    MasterNodeName := '';
    LinkType := ltUnknown;
end;

procedure TDBExNodeLink.Assign(Source: TPersistent);
begin
    if Source is TDBExNodeLink then
    begin
        MasterNode := TDBExNodeLink(Source).MasterNode;
        MasterField := TDBExNodeLink(Source).MasterField;
        DetailField := TDBExNodeLink(Source).DetailField;
        LinkType := TDBExNodeLink(Source).LinkType;
        Exit;

    end;
    inherited Assign( Source );
end;

//========================================================================
// TDBExNodeLinks
//========================================================================

constructor TDBExNodeLinks.Create( AOwnerNode : TDBExDataNode);
begin
    inherited Create( TDBExNodeLink);
    FOwnerNode := AOwnerNode;
end;

//------------------------------------------------------------------------
function TDBExNodeLinks.Add : TDBExNodeLink;
begin
    Result := TDBExNodeLink( inherited Add );
end;

//------------------------------------------------------------------------
function TDBExNodeLinks.GetItem( Index : integer ):TDBExNodeLink;
begin
  Result := TDBExNodeLink(inherited GetItem(Index));
end;

//------------------------------------------------------------------------
procedure TDBExNodeLinks.SetItem( Index : integer; Value : TDBExNodeLink);
begin
  inherited SetItem(Index, Value);
end;

//========================================================================
// TDBExLinkValues
//========================================================================
constructor TDBExLinkValues.Create( Dispatcher : TDBExDispatcher );
begin
    inherited Create;
    FValues := TList.Create;
    FDispatcher := Dispatcher;
end;

//------------------------------------------------------------------------
destructor TDBExLinkValues.Destroy;
begin
    Clear;
    FValues.Free;
    inherited Destroy;
end;

//------------------------------------------------------------------------
function TDBExLinkValues.GetCount : integer;
begin
    Result := FValues.Count;
end;

//------------------------------------------------------------------------
function TDBExLinkValues.GetValue( Index : integer ): TDBExLinkValue;
begin
    Result := TDBExLinkValue( FValues[Index] );
end;

//------------------------------------------------------------------------
procedure TDBExLinkValues.Add( Value : TDBExLinkValue );
begin
    FValues.Add( Value );
end;

//------------------------------------------------------------------------
function TDBExLinkValues.Find( Link : TDBExNodeLink ) : TDBExLinkValue;
var
    i : integer;
    LValue : TDBExLinkValue;
begin
    Result := nil;
    if (Link.MasterNode = FDispatcher.GetTopicNode) then
    begin
        for i:=0 to FValues.Count-1 do
        begin
            LValue := GetValue( i );
            if LValue.MasterField = Link.MasterField then
            begin
                Result := LValue;
                Exit;
            end;
        end;
    end
    else
        if (FDispatcher.ParentDispatcher<>nil) and
           (FDispatcher.ParentDispatcher.ParentDispatcher<>nil) then
            Result := FDispatcher.ParentDispatcher.ParentDispatcher.FLinkValues.Find( Link );
end;

//------------------------------------------------------------------------
procedure TDBExLinkValues.Clear;
begin
    while FValues.Count <>0 do
    begin
        TDBExLinkValue(FValues[0]).Free;
        FValues.Delete(0);
    end;
end;


//========================================================================
// TDBExNode
//========================================================================
constructor TDBExNode.Create( AOwner : TComponent );
begin
     inherited Create( AOwner );
     FChildren := TList.Create;
     FParentNode:=nil;
     FNodeType:=ntNullNode;
     FActiveTopic := nil;
     FTopicDispatchers:=TList.Create;
     FOnActivate := nil;
     FOnDeactivate := nil;
end;

//------------------------------------------------------------------------
destructor TDBExNode.Destroy;
begin
     FChildren.Free;
     FTopicDispatchers.Free;
     inherited Destroy;
end;


//------------------------------------------------------------------------
procedure TDBExNode.Clear;
var
    aNode : TDBExNode;
begin
   while FChildren.Count>0 do
   begin
      aNode:=TDBExNode(FChildren[FChildren.Count-1]);
      FChildren.Delete(FChildren.Count-1);
      aNode.Free;
   end;
end;

//------------------------------------------------------------------------
function TDBExNode.GetParentComponent: TComponent;
begin
    if FParentNode=OwnerNodes.RootNode then
        Result := FExplorer
    else
        Result := FParentNode;
end;

//------------------------------------------------------------------------
function TDBExNode.HasParent: Boolean;
begin
  HasParent := True;
end;

//------------------------------------------------------------------------
procedure TDBExNode.ReadState(Reader: TReader);
begin
  inherited ReadState(Reader);
  if Reader.Parent is TCustomDBExplorer then
      FExplorer := TCustomDBExplorer(Reader.Parent);
end;

//------------------------------------------------------------------------
procedure TDBExNode.SetParentComponent(AParent: TComponent);
begin
//  if not (csLoading in ComponentState) then
     if AParent is TCustomDBExplorer then
     begin
         FExplorer := TCustomDBExplorer(AParent);
         SetParentNode(FExplorer.Nodes.RootNode);
         FOwnerNodes:=TCustomDBExplorer(FExplorer).Nodes;
     end
     else
     begin
         SetParentNode(TDBExNode(AParent));
         FExplorer:=FParentNode.FOwnerNodes.OwnerEx;
         FOwnerNodes:=TCustomDBExplorer(FExplorer).Nodes;
     end;
end;

//------------------------------------------------------------------------
procedure TDBExNode.DeleteDispatchers;
begin
    while FTopicDispatchers.Count>0 do
        TDBExDispatcher(FTopicDispatchers[0]).Free;
end;

{$IFDEF _DELPHI3_}
procedure TDBExNode.GetChildren(Proc: TGetChildProc; Root: TComponent);
var
  N : TDBExNode;
begin
  N:=GetFirstChild;
  while N <> nil do
  begin
    if N.Owner = Root then Proc(N);
    N := N.GetNextSibling;
  end;
end;
{$ELSE}
procedure TDBExNode.GetChildren(Proc: TGetChildProc);
var
  N : TDBExNode;
begin
  N:=GetFirstChild;
  while N <> nil do
  begin
    if N.Owner = OwnerNodes.OwnerEx.Owner then Proc(N);
    N := N.GetNextSibling;
  end;
end;
{$ENDIF}
//------------------------------------------------------------------------
function TDBExNode.GetFirstChild: TDBExNode;
begin
    if FChildren.Count <> 0 then
        Result:=FChildren[0]
    else
        Result:=nil;
end;

//------------------------------------------------------------------------
function TDBExNode.GetLastChild: TDBExNode;
begin
    if FChildren.Count <> 0 then
        Result:=FChildren[FChildren.Count-1]
    else
        Result:=nil;
end;

//------------------------------------------------------------------------
function TDBExNode.ChildrenCount : integer;
begin
    Result := FChildren.Count;
end;

//------------------------------------------------------------------------
function TDBExNode.GetNext: TDBExNode;
begin
    Result:=nil;
    if FChildren.Count<>0 then
        Result:=FChildren[0]
    else
        if FParentNode<>nil then
           Result := GetNextAfter;
end;

//------------------------------------------------------------------------
function TDBExNode.GetNextAfter: TDBExNode;
var
    i : integer;
begin
    Result := nil;
    if Self=FOwnerNodes.FRootNode then Exit;
    if FParentNode<>nil then
    begin
       i:=FParentNode.FChildren.IndexOf( Self );
       if (i<>-1) then
       begin
           if i<FParentNode.FChildren.Count-1 then
               Result:=FParentNode.FChildren[i+1]
           else
               Result:=FParentNode.GetNextAfter;
       end;
    end;
end;

//------------------------------------------------------------------------
function TDBExNode.GetNextChild(Value: TDBExNode): TDBExNode;
var
    I : integer;
begin
    Result:=nil;
    i:=FChildren.IndexOf( Value );
    if (i<>-1) AND (i<FChildren.Count-1) then
        Result:=FChildren[i+1];
end;

//------------------------------------------------------------------------
function TDBExNode.GetNextSibling: TDBExNode;
begin
    if FParentNode<>nil then
        Result:=FParentNode.GetNextChild(Self)
    else
        Result:=nil;
end;

//------------------------------------------------------------------------
function TDBExNode.GetPrevChild(Value: TDBExNode): TDBExNode;
var
    I : integer;
begin
    Result:=nil;
    i:=FChildren.IndexOf( Value );
    if (i<>-1) AND (i>0) then
        Result:=FChildren[i-1];
end;

//------------------------------------------------------------------------
function TDBExNode.GetPrevSibling: TDBExNode;
begin
    if FParentNode<>nil then
        Result:=FParentNode.GetPrevChild(Self)
    else
        Result:=nil;
end;

//------------------------------------------------------------------------
function TDBExNode.GetItem( Index : integer ): TDBExNode;
begin
    if Index>=FChildren.Count then
        Result := nil
    else
        Result := TDBExNode( FChildren[Index] );
end;

//------------------------------------------------------------------------
function TDBExNode.CreateNode(ANodeType : TDBExNodeType;
                              AOwner : TComponent ): TDBExNode;
begin
    Result:=nil;
    case ANodeType of
        ntDataSetNode : Result := TDBExDataSetNode.Create( AOwner );
        ntLinkNode : Result := TDBExLinkNode.Create( AOwner );
        ntNullNode : Result := TDBExNode.Create( AOwner );
        ntCustomNode : Result := TDBExCustomNode.Create( AOwner );
    else
        raise EDBExpError.Create('Bad NodeType');
    end;
    Result.FOwnerNodes:=FOwnerNodes;
    Result.Name:=FOwnerNodes.UniqueName( Result.ClassName );
    Result.FExplorer:=FExplorer;
end;

//------------------------------------------------------------------------
function TDBExNode.Add( ANodeType : TDBExNodeType; S : string ):TDBExNode;
var
    i : integer;
    N : TDBExNode;
begin
    Result := CreateNode( ANodeType, FOwnerNodes.OwnerEx.Owner );
    if Result <> nil then
    begin
        if FParentNode.ChildrenCount>0 then
        begin
            N:=FParentNode.GetFirstChild;
            if N is TDBExDataNode then
                TDBExDataNode(N).TopicVisible := True;
        end;
        Result.FParentNode:=FParentNode;
        Result.Text:=S;
        if FParentNode<>nil then
        begin
            i:=FParentNode.FChildren.IndexOf( Self );
            FParentNode.FChildren.Insert( i+1, Result);
        end
        else
            raise EDBExpError.Create('Not member of Nodes');
    end;
end;

//------------------------------------------------------------------------
function TDBExNode.AddChild( ANodeType : TDBExNodeType; S : string ):TDBExNode;
var
    N : TDBExNode;
begin
    Result := CreateNode( ANodeType, FOwnerNodes.OwnerEx.Owner );
    if Result <> nil then
    begin
        Result.FParentNode:=Self;
        Result.Text:=S;
        FChildren.Add( Result );
        if FChildren.Count > 1 then
        begin
            N:=GetFirstChild;
            if N is TDBExDataNode then
                TDBExDataNode(N).TopicVisible := True;
        end;
    end;
end;

//------------------------------------------------------------------------
function TDBExNode.AddChildFirst( ANodeType : TDBExNodeType;
                                  S : string ):TDBExNode;
begin
    Result := CreateNode( ANodeType, FOwnerNodes.OwnerEx.Owner );
    Result.FParentNode:=Self;
    Result.Text:=S;
    FChildren.Insert( 0, Result );
end;

//------------------------------------------------------------------------
function TDBExNode.AddFirst( ANodeType : TDBExNodeType; S : string ):TDBExNode;
var
    N : TDBExNode;
begin
    Result := CreateNode( ANodeType, FOwnerNodes.OwnerEx.Owner );
    if Result <> nil then
    begin
        if FParentNode.ChildrenCount>0 then
        begin
            N:=FParentNode.GetFirstChild;
            if N is TDBExDataNode then
                TDBExDataNode(N).TopicVisible := True;
        end;
        Result.FParentNode:=FParentNode;
        Result.Text:=S;
        if FParentNode<>nil then
            FParentNode.FChildren.Insert( 0, Result)
        else
            raise EDBExpError.Create('Not member of Nodes');
    end;
end;

//------------------------------------------------------------------------
procedure TDBExNode.RemoveChild( Node : TDBExNode );
begin
    FChildren.Remove( Node );
end;

//------------------------------------------------------------------------
procedure TDBExNode.DriveDataset;
begin
    if FOwnerNodes.DrivenDS<> nil then
       FOwnerNodes.DrivenDS.DataSet:=nil;
end;

//------------------------------------------------------------------------
procedure TDBExNode.DriveTopicDataset;
begin
    DriveDataSet;
end;

//------------------------------------------------------------------------
function TDBExNode.ParentNode : TDBExNode;
begin
    Result:=FParentNode;
end;

//------------------------------------------------------------------------
procedure TDBExNode.SetParentNode( Value : TDBExNode );
begin
    if (Value=nil) and (FExplorer<>nil) then
         Value := FExplorer.Nodes.RootNode;
    if  (Value<>FParentNode) then
    begin
        if FParentNode<>nil then
            FParentNode.FChildren.Remove(self);
        if Value<>nil then
            Value.FChildren.Add(self);
        FParentNode:=Value;
    end;
end;

//------------------------------------------------------------------------
procedure TDBExNode.SetText( Value : string );
begin
    FText:=Value;
    if (FOwnerNodes<>nil) and
       (Assigned(FOwnerNodes.FOnTextChange)) then
        FOwnerNodes.FOnTextChange(Self);
end;

//------------------------------------------------------------------------
function TDBExNode.CanEdit : boolean;
begin
    Result := False;
end;

//------------------------------------------------------------------------
function TDBExNode.CreateDispatcher(Master : TDBExDispatcher;
                                    Before : TTreeNode):TDBExDispatcher;
begin
    if Master is TDBExMetaDispatcher then
    begin
            Result := CreateTopicDispatcher( Master, Before );
    end
    else begin
            Result := CreateItemDispatcher( Master, Before );
    end;
    if Result<>nil then
        Result.FExplorer := FExplorer;
end;

//------------------------------------------------------------------------
function TDBExNode.CreateTopicDispatcher(Master : TDBExDispatcher;
                                         Before : TTreeNode):TDBExDispatcher;
begin
    Result := TDBExMetaDispatcher.Create( Self );
    if Result<>nil then
    begin
        FTopicDispatchers.Add( Result );
        TDBExMetaDispatcher(Result).FText := Text;
    end;
end;

//------------------------------------------------------------------------
function TDBExNode.CreateItemDispatcher(Master : TDBExDispatcher;
                                        Before:TTreeNode):TDBExDispatcher;
begin
    Result := TDBExMetaDispatcher.Create( Self );
    TDBExMetaDispatcher(Result).FText := Text;
end;

//------------------------------------------------------------------------
procedure TDBExNode.SetActiveTopic(Value : TDBExDataDispatcher);
begin
    if FActiveTopic=Value then Exit;

    FActiveTopic:=Value;
    if (Value<>nil) and Assigned( FOnActivate ) then
        FOnActivate( Self );
    if (Value=nil) and Assigned( FOnDeactivate ) then
        FOnDeactivate( Self );
end;

//------------------------------------------------------------------------
procedure TDBExNode.EndEdit;
begin

end;


//------------------------------------------------------------------------
procedure TDBExNode.SetItemLabel( Value : string);
begin
end;

//------------------------------------------------------------------------
function TDBExNode.CheckMenuPopup(const Pos: TSmallPoint):Boolean;
var
  PopMenu: TPopupMenu;
  P:TPoint;
begin
  Result:=False;
  if csDesigning in ComponentState then Exit;

  PopMenu := FPopupMenu;
  if (PopMenu <> nil) and PopMenu.AutoPopup then
  begin
    PopMenu.PopupComponent := Self;
    P.X:=Pos.X;
    P.Y:=Pos.Y;
    windows.clienttoscreen(FOwnerNodes.FOwnerEx.WindowHandle,P);
    PopMenu.Popup(P.X, P.Y);
    Result:=True;
    Exit;
  end;
end;

//------------------------------------------------------------------------
procedure TDBExNode.SetPopupMenu(Value: TPopupMenu);
begin
  FPopupMenu := Value;
  if Value <> nil then Value.FreeNotification(Self);
end;

//------------------------------------------------------------------------
function TDBExNode.GetPopupMenu: TPopupMenu;
begin
  Result := FPopupMenu;
end;

procedure TDBExNode.Notification(AComponent: TComponent; Operation: TOperation);
begin
  Inherited Notification(AComponent,Operation);
  If (Operation=opRemove) and (AComponent = FPopupMenu) then
    FPopupMenu:=Nil;
end;

//========================================================================
// TDBExDataNode
//========================================================================

//------------------------------------------------------------------------
constructor TDBExDataNode.Create( AOwner : TComponent );
begin
     inherited Create( AOwner );
     FFreezeCount := 0;
     FTopicVisible := True;
     FDriveDataSets := True;
     FLinks := TDBEXNodeLinks.Create( Self );
     FCurrentFilter := nil;
end;

//------------------------------------------------------------------------
destructor TDBExDataNode.Destroy;
begin
    FLinks.Free;
    inherited Destroy;
end;

procedure TDBexDataNode.SetLinks( Value : TDBExNodeLinks );
begin
    FLinks.Assign( Value );
end;

//------------------------------------------------------------------------
function TDBexDataNode.GetDataItem : string;
begin
    FCurrDataItem := 0;
    Result:='';
end;

//------------------------------------------------------------------------
function TDBexDataNode.MoveBy( By : integer ):integer;
begin
    FCurrDataItem := FCurrDataItem + By;
    Result := By;
end;

//------------------------------------------------------------------------
function TDBexDataNode.RecordCount:integer;
var
    By : integer;
begin
    Result := 0;
    By := 1000;
    while By=1000 do
    begin
        By := MoveBy( By );
        Result := Result+By;
    end;
    First;
    MoveBy( FCurrDataItem );
end;

//------------------------------------------------------------------------
procedure TDBexDataNode.First;
begin
    FCurrDataItem := 0;
end;

//------------------------------------------------------------------------
procedure TDBexDataNode.Next;
begin
    if not EndOfData then
        Inc(FcurrDataItem)
end;

//------------------------------------------------------------------------
procedure TDBExDataNode.Freeze;
begin
    Inc(FFreezeCount);
end;

//------------------------------------------------------------------------
procedure TDBExDataNode.Unfreeze;
begin
    Dec(FFreezeCount);
end;

//------------------------------------------------------------------------
function TDBExDataNode.IsFreezed:boolean;
begin
    Result:=(FFreezeCount>0);
end;

//------------------------------------------------------------------------
function TDBExDataNode.CanEdit : boolean;
begin
    Result := False;
end;

//------------------------------------------------------------------------
function TDBExDataNode.CreateTopicDispatcher(Master : TDBExDispatcher;
                                        Before : TTReeNode):TDBExDispatcher;
begin
    Result := TDBExDataDispatcher.Create( Self);
    if Result<>nil then
    begin
        FTopicDispatchers.Add( Result );
        TDBExDataDispatcher(Result).FText := Text;

        Result.SaveLinkValues;
    end;
end;

//------------------------------------------------------------------------
function TDBExDataNode.CreateItemDispatcher(Master : TDBExDispatcher;
                                         Before : TTreeNode):TDBExDispatcher;
begin
    Result := inherited CreateItemDispatcher( Master, Before );
    TDBExMetaDispatcher(Result).FText := GetDataItem;
    TDBExMetaDispatcher(Result).KeyValue := GetKeyValue;
end;

//------------------------------------------------------------------------
procedure TDBExDataNode.GetPossibleLinks;
begin
    FPossibleLinks := [ltFilter];
end;

//------------------------------------------------------------------------
procedure TDBExDataNode.FindDependentFields( LinkValues : TDBExLinkValues );
    procedure CheckNode( N : TDBExNode );
    var
        MValue : TDBExNodeLink;
        CNode : TDBExNode;
        i : integer;
        LValue : TDBExLinkValue;
    begin
        if N is TDBExDataNode then
        begin
            for i:=0 to TDBExDataNode(N).FLinks.Count-1 do
            begin
                if TDBExDataNode(N).FLinks.Items[i].MasterNode=self then
                begin
                    MValue:=TDBExDataNode(N).FLinks.Items[i];
                    if LinkValues.Find( MValue )=nil then
                    begin
                        LValue := TDBExLinkValue.Create;
                        LValue.MasterFIeld:=MValue.MasterField;
                        LinkValues.Add(LValue);
                    end;
                end;
            end;
        end;
        CNode := N.GetFirstChild;
        while CNode <> nil do
        begin
            CheckNode( CNode );
            CNode:=CNode.GetNextSibling;
        end;
    end;

var
    N : TDBExNode;
begin
    if LinkValues=nil then Exit;
    LinkValues.Clear;
    OwnerNodes.DisableLinkNodes:=True;
    N := GetFirstChild;
    while N <> nil do
    begin
        CheckNode( N );
        N:=N.GetNextSibling;
    end;
    OwnerNodes.DisableLinkNodes:=False;
end;

//------------------------------------------------------------------------
procedure TDBExDataNode.SetFilter(Disp:TDBExDispatcher);
var
    i : integer;
    MasterValue : string;
    LValue : TDBExLinkValue;
    N : TDBExDataNode;
    HasFilter : boolean;
    IsMasterValue : boolean;
begin


    FCurrentFilter:=Disp;
    HasFilter := false;
    for i:=0 to FLinks.Count-1 do
    begin
        LValue := Disp.FLinkValues.Find( FLinks.Items[i] );
        if LValue<>nil then
        begin
            MasterValue := LValue.MasterValue;
            if FLinks.Items[i].LinkType = ltMaster then
            begin
                 N := FLinks.Items[i].MasterNode;
                 N.Freeze;
                 FilterByMaster( FLinks.Items[i].MasterField, MasterValue );
                 HasFilter := true;
                 N.UnFreeze;
            end;
        end;
    end;
    for i:=0 to FLinks.Count-1 do
    begin
        IsMasterValue := false;
        if FLinks.Items[i].MasterNode<>nil then
        begin
            LValue := Disp.FLinkValues.Find( FLinks.Items[i] );
            if LValue<>nil then
            begin
                MasterValue := LValue.MasterValue;
                IsMasterValue := true;
            end;
        end
        else
        begin
            MasterValue := FLinks.Items[i].MasterField;
            IsMasterValue := true;
        end;
        if IsMasterValue then
        begin
            case FLinks.Items[i].LinkType of
            ltParam:
                FilterByParam( FLinks.Items[i].DetailField, MasterValue );
            ltFilter:
                FilterByFilter( FLinks.Items[i].DetailField, MasterValue );
            end;
            HasFilter := true;
        end;
    end;
    if not HasFilter then
        FilterByFilter('','');
end;

//------------------------------------------------------------------------
function TDBExDataNode.BeginOfData : boolean;
var
    Distance : integer;
begin
    Distance := MoveBy( -1 );
    Result := (Distance = 0);
    if Distance <> 0 then
        MoveBy( - Distance );
end;

//------------------------------------------------------------------------
function TDBExDataNode.GetField( FieldName : string ) : string;
begin
    Result := '';
end;

//------------------------------------------------------------------------
function TDBExDataNode.ParamCount : integer;
begin
    Result := 0;
end;

//------------------------------------------------------------------------
function TDBExDataNode.ParamName( Index : integer ) : string;
begin
    Result := '';
end;

//------------------------------------------------------------------------
function TDBExDataNode.FieldCount : integer;
begin
    Result := 0;
end;

//------------------------------------------------------------------------
function TDBExDataNode.FieldName( Index : integer ) : string;
begin
  Result := '';
end;

//------------------------------------------------------------------------
function TDBExDataNode.GetItemImageIndex:Integer;
begin
    Result := FItemImageIndex;
end;

//------------------------------------------------------------------------
function TDBExDataNode.GetItemSelectedIndex:Integer;
begin
    Result := FItemSelectedIndex;
end;

//------------------------------------------------------------------------
function TDBExDataNode.GetItemStateIndex:Integer;
begin
    Result := FItemStateIndex;
end;
//------------------------------------------------------------------------
procedure TDBExDataNode.SetTopicVisible( Value : boolean );
begin
    if (ParentNode.ChildrenCount = 1) then
        FTopicVisible := Value
    else
        FTopicVisible := True;
    if (not TopicVisible) and
       (FParentNode = FOwnerNodes.RootNode) then
        DriveDataSets := False;
end;

//========================================================================
// TDBExCustomNode
//========================================================================
constructor TDBExCustomNode.Create( AOwner : TComponent );
begin
    inherited Create( AOwner );
    FNodeType := ntCustomNode;
    FFiltered := False;
end;

function TDBExCustomNode.GetField( FieldName : string ) : string;
begin
  Result:='';
  if FieldName='ParentKey' then Result:=FCurrFilterValue;
  if FieldName='Key' then Result:=FCurrKeyValue;
  if FieldName='Label' then Result:=FText;
end;

function TDBExCustomNode.FieldCount:Integer;
begin
  Result:=1;
end;

function TDBExCustomNode.FieldName( Index : integer ) : string;
begin
  case index of
    0: Result:='ParentKey';
    1: Result:='Key';
    2: Result:='Label';
  else
    Result:='';
  end;
end;

//------------------------------------------------------------------------
function TDBExCustomNode.DataActive : boolean;
begin
    Result := True;
    if Assigned( FOnGetActive ) then
    begin
        FOnGetActive( Self, FActive );
        Result:=FActive;
    end;
end;

//------------------------------------------------------------------------
procedure TDBExCustomNode.OpenData;
begin
    if Assigned( FOnActiveChanged ) then
    begin
        FActive := True;
        FOnActiveChanged( Self, FActive );
    end;
end;

//------------------------------------------------------------------------
procedure TDBExCustomNode.CloseData;
begin
    if not FDriveDatasets then Exit;

    if Assigned( FOnActiveChanged ) then
    begin
        FActive := False;
        FOnActiveChanged( Self, FActive );
    end;
end;

//------------------------------------------------------------------------
function TDBExCustomNode.GetDataItem : string;
begin
    Result:=inherited GetDataItem;

    if Assigned( FOnGetData ) then
    begin
        FOnGetData( Self, FCurrKeyValue, FCurrParentKeyValue, Result);
    end;
end;

//------------------------------------------------------------------------
procedure TDBExCustomNode.Next;
begin


    inherited Next;
    if Assigned( FOnNext ) then
        FOnNext( Self );
    if FFiltered and not EndOfData then
    begin
        GetDataItem;
        while (FCurrParentKeyValue<>FilterValue) and (not EndOfData) do
        begin
            Next;
            if not EndOfData then
                GetDataItem;
        end;
    end;
end;

//------------------------------------------------------------------------
procedure TDBExCustomNode.First;
begin
    inherited First;
    if Assigned( FOnFirst ) then
        FOnFirst( Self );
    if FFiltered then GetDataItem;
    if FFiltered and (FCurrParentKeyValue<>FilterValue) then
        Next;
end;

//------------------------------------------------------------------------
function TDBExCustomNode.MoveBy( By : integer ):integer;
var
    NewPos : integer;
begin
    Result := inherited MoveBy( By );
    if By = 0 then Exit;
    NewPos:=FCurrDataItem + By;
    First;
    By := 0;
    while (FCurrDataItem<>NewPos) and (not EndOfData ) do
    begin
        Next;
        Inc(By);
    end;
    Result := By;
end;

//------------------------------------------------------------------------
function TDBExCustomNode.GetKeyValue : string;
begin
    Result := FCurrKeyValue;
end;

//------------------------------------------------------------------------
function TDBExCustomNode.EndOfData : boolean;
begin
    Result := True;
    if Assigned( FOnEndCheck ) then
    begin
        FOnEndCheck( Self, Result );
    end;
end;

procedure TDBExCustomNode.FilterByFilter(DetailField:string;
                    MasterValue : string);

begin
  FFiltered := True;
  FCurrFilterValue:=MasterValue;
end;

procedure TDBExCustomNode.FilterByMaster(MasterField:string;
                              MasterValue : string );
begin

end;

procedure TDBExCustomNode.FilterByParam(DetailField:string;
                              MasterValue : string);
begin

end;

//------------------------------------------------------------------------
procedure TDBExCustomNode.DriveDataset;
begin
    if FOwnerNodes.DrivenDS<> nil then
       FOwnerNodes.DrivenDS.DataSet:=nil;
end;

//------------------------------------------------------------------------
function TDBExCustomNode.CanEdit : boolean;
begin
    Result := False;
end;

//========================================================================
// TDBExDataLink
//========================================================================
constructor TDBExDataLink.Create( Anode : TCustomDBExDataSetNode );
begin
    inherited Create;
    PrevDataSet:=Nil;
    FNode := ANode;
end;

//------------------------------------------------------------------------
procedure TDBExDataLink.ActiveChanged;
var SField:TStringList;
begin
  if (Assigned( FNode )) then
  begin
    If (PrevDataSet<>DataSet)
      and(not((csLoading in TCustomDBExDataSetNode(FNode).ComponentState)
        or (csDestroying in TCustomDBExDataSetNode(FNode).ComponentState)))
    then
    begin
      PrevDataSet:=DataSet;
      if TCustomDBExDataSetNode(FNode).FKeyField<>'' then
      begin
        if DataSet<>Nil then
        begin
          SField:=TStringList.Create;
          DataSet.GetFieldNames(SField);
          if SField.IndexOf(TCustomDBExDataSetNode(FNode).FKeyField)=-1 then
               raise EDBExpError.Create('Not found KeyField: '''+TCustomDBExDataSetNode(FNode).FKeyField+
               '''. Property Links in descendant Node will be incorrect');
          SField.Free;
        end;
      end;
    end;
    FNode.LinkActive( Active );
  end;
end;

//------------------------------------------------------------------------
procedure TDBExDataLink.CheckBrowseMode;
begin
end;

//------------------------------------------------------------------------
procedure TDBExDataLink.DataSetChanged;
begin
    if (Assigned( FNode )) then
         FNode.DataSetChanged;
     FModified:=False;
end;

//------------------------------------------------------------------------
procedure TDBExDataLink.DataSetScrolled(Distance: Integer);
begin
    if (Assigned( FNode )) then
         FNode.DataSetScrolled( Distance );
end;

//------------------------------------------------------------------------
procedure TDBExDataLink.FocusControl(Field: TFieldRef);
begin
end;

//------------------------------------------------------------------------
procedure TDBExDataLink.EditingChanged;
begin
end;

//------------------------------------------------------------------------
procedure TDBExDataLink.LayoutChanged;
begin
end;

//------------------------------------------------------------------------
procedure TDBExDataLink.RecordChanged(Field: TField);
begin
    if (Assigned( FNode )) then
         Fnode.RecordChanged( Field );
    FModified := False;
end;

//------------------------------------------------------------------------
procedure TDBExDataLink.UpdateData;
begin
  try
    if FModified and (Assigned( FNode ))then
        FNode.UpdateData;
    FModified := False;
  finally
  end;
end;

//========================================================================
// TCustomDBExDataSetNode
//========================================================================
procedure TCustomDBExDataSetNode.SetFilter(Disp:TDBExDispatcher);
begin

if FCurrentFilter<>nil then

    if (FDataLink.DataSource=nil) or
       (FDataLink.DataSource.DataSet=nil) then Exit;
    if Disp=FCurrentFIlter then Exit;
    inherited SetFilter( Disp );
end;

//------------------------------------------------------------------------
procedure TCustomDBExDataSetNode.CheckNewRecord;
begin
    if (FDataLink.DataSource=nil) or
       (FDataLink.DataSource.DataSet=nil) or
       (FActiveTopic=nil) then Exit;
    if FCheckInsert then
        FActiveTopic.FillLinkFields(FDataLink.DataSource.DataSet);
end;


//------------------------------------------------------------------------
procedure TCustomDBExDataSetNode.FilterByMaster(MasterField:string;
                                                MasterValue : string );
var
    MasterDS : TDataSet;
begin
    MasterDS:=nil;

    if FDataLink.DataSet is TTable then
        MasterDS:=TTable(FDataLink.DataSet).MasterSource.DataSet
    else
        if FDataLink.DataSet is TQuery then
            MasterDS:=TQuery(FDataLink.DataSet).DataSource.DataSet;
    if (MasterDS <> nil) then
        MasterDS.Locate(MasterField, MasterValue, [loCaseInsensitive]);
end;

//------------------------------------------------------------------------
procedure TCustomDBExDataSetNode.FilterByParam(DetailField:string;
                                               MasterValue : string);
var
    Param : TParam;
begin
    if FDataLink.DataSet is TQuery then
    begin
        Param := TQuery(FDataLink.DataSet).ParamByName( DetailField );
        if Param<>nil then
            Param.Text := MasterValue;
        FDataLink.DataSet.Close;
        FDataLink.DataSet.Open;
    end;
end;

//------------------------------------------------------------------------
procedure TCustomDBExDataSetNode.FilterByFilter(DetailField:string;
                                                MasterValue : string);
var
    Suffix, Prefix : string;
begin
    if FDataLink.DataSet=nil then Exit; // <<<<<

    if (DetailField <> '') then
    begin
        if FOriginalFilter = '' then
        begin
            Prefix := '';
            Suffix := '';
        end
        else
        begin
            Prefix:='(';
            Suffix := ') AND ';
        end;
        FDataLink.DataSet.Filter:=Prefix+
            FOriginalFilter+Suffix+'('+DetailField+'='''+MasterValue+''')';
    end
    else
        FDataLink.DataSet.Filter:=FOriginalFilter;
    FDataLink.DataSet.Filtered:=True;
end;

//------------------------------------------------------------------------
procedure TCustomDBExDataSetNode.Freeze;
begin
    if (not IsFreezed) and (FDataLink.DataSet<>nil) then
        FDataLink.DataSet.DisableControls;
    inherited Freeze;
end;

//------------------------------------------------------------------------
procedure TCustomDBExDataSetNode.UnFreeze;
begin

    if (FFreezeCount=1) and (FDataLink.DataSet<>nil) then
        FDataLink.DataSet.EnableControls;
    inherited UnFreeze;
end;

//------------------------------------------------------------------------
constructor TCustomDBExDataSetNode.Create( AOwner : TComponent );
begin
     inherited Create( AOwner );
     FKeyField     := '';
     FDataLink     := TDBExDataLink.Create( Self );
     FNodeType     := ntDataSetNode;
     FLabelField   := '';
     FTopicDataSource := nil;
     FCheckInsert:=True;
     FPostLoading := False;
     FDataSet := nil;
end;

//------------------------------------------------------------------------
destructor TCustomDBExDataSetNode.Destroy;
begin
     Destroying;
     FDataLink.Free;
     FDataLink:=nil;
     inherited Destroy;
end;

//------------------------------------------------------------------------
function TCustomDBExDataSetNode.DataActive : boolean;
begin
     Result:=false;
     if FDataLink.DataSet<>nil then
         Result:=FDataLink.DataSet.Active;
end;

//------------------------------------------------------------------------
procedure TCustomDBExDataSetNode.OpenData;
begin
    if FDataLink.DataSet<>nil then
        FDataLink.DataSet.Open;
end;

//------------------------------------------------------------------------
procedure TCustomDBExDataSetNode.CloseData;
begin
    if (FDataLink.DataSet=nil) or (not FDriveDatasets) then
        Exit
    else
        FDataLink.DataSet.Close;
end;

//------------------------------------------------------------------------
procedure TCustomDBExDataSetNode.First;
begin
    inherited First;
    if DataActive then
        FDataLink.DataSet.First;
end;

//------------------------------------------------------------------------
procedure TCustomDBExDataSetNode.Next;
begin
    inherited Next;
    if DataActive then
        FDataLink.DataSet.Next;
end;

//------------------------------------------------------------------------
function TCustomDBExDataSetNode.MoveBy( By : integer ):integer;
begin
    Result := inherited MoveBy( By );
    if DataActive then
        Result := FDataLink.DataSet.MoveBy( By );
end;

//------------------------------------------------------------------------
function TCustomDBExDataSetNode.RecordCount:integer;
begin
    Result := 0;
    if DataActive then
        Result := FDataLink.DataSet.RecordCount;
end;

//------------------------------------------------------------------------
function TCustomDBExDataSetNode.GetDataItem : string;
begin
    Result:=inherited GetDataItem;
    if DataActive then
    begin
        try
             Result := FDataLink.DataSet.FieldByName( FLabelField ).AsString;
        except
            raise EDBExpError.Create('Bad LabelField name');
        end;
    end;
end;


//------------------------------------------------------------------------
function TCustomDBExDataSetNode.GetKeyValue : string;
begin
    if DataActive then
    begin
     try
        if FKeyField<> '' then
            Result := FDataLink.DataSet.FieldByName( FKeyField ).AsString
        else
            Result := '';
     except
        Result := '';
     end;
    end;
end;

//------------------------------------------------------------------------
function TCustomDBExDataSetNode.EndOfData : boolean;
begin
    Result := True;
    if DataActive then
    begin
        Result := FDataLink.DataSet.EOF;
    end;
end;

//------------------------------------------------------------------------
function TCustomDBExDataSetNode.BeginOfData : boolean;
begin
    Result := True;
    if DataActive then
    begin
        Result := FDataLink.DataSet.BOF;
    end;
end;
//------------------------------------------------------------------------
procedure TCustomDBExDataSetNode.SetupNewDataSet(Value : TDataSet);
var
  i : integer;
  MasterSource : TDataSource;
  L : TDBExNodeLink;
  N, LinkedNode : TDBExNode;
  FList : TList;

  Done : boolean;
begin
    if (not (csLoading in ComponentState)) and
       (not (csLoading in Value.ComponentState)) and
       (not FPostLoading){ and (FDatalink.Datasource<>nil)} then
        FLinks.Clear;
    if Value <> nil then
    begin
        Freeze;
        Value.FreeNotification(Self);
        FDataLink.DataSet.Filter:='';
        FOriginalFilter:=FDataLink.DataSet.Filter;

        MasterSource := nil;
        if (Value is TTable) then
            MasterSource :=TTable(Value).MasterSource
        else
            if (Value is TQuery) then
                MasterSource := TQuery(Value).DataSource;
        if (MasterSource <> nil) and
           (MasterSource.DataSet <> nil) then
        begin
            N:=ParentNode;
            Done := False;
            LinkedNode := nil;
            while (N<>nil) and (not Done) do
            begin
                if (N is TCustomDBExDataSetNode) and
                   (TCustomDBExDataSetNode(N).DataSource = MasterSource) then
                begin
                    LinkedNode := N;
                    Done := True;
                end;
                N := N.ParentNode;
            end;
            if LinkedNode <> nil then
            begin
                for i:=FLinks.Count -1 downto 0  do
                begin
                    if Flinks.Items[i].LinkType=ltMaster then

                        Flinks.Items[i].Free;


                end;
                if (Value is TTable) then
                begin
                    FList := TList.Create;
                    MasterSource.DataSet.GetFieldList(
                                FList, TTable( Value ).MasterFields );
                    for i:=0 to FList.Count-1 do
                    begin
                        L := FLinks.Add;
                        L.MasterNode := TDBExDataNode(LinkedNode);
                        L.LinkType := ltMaster;
                        L.MasterField := TField(FList[i]).FieldName;

                    end;
                    Flist.Free;
                end;
                if (Value is TQuery) then
                begin
                    for i:=0 to TQuery( Value ).ParamCount-1 do
                    begin
                        L := Flinks.Add;
                        L.MasterNode := TDBExDataNode(LinkedNode);
                        L.LinkType := ltMaster;
                        L.MasterField := TQuery( Value ).Params[i].Name;

                    end;
                end;
            end;
        end;
        UnFreeze;
    end;
    FDataSet:=Value;
end;
//------------------------------------------------------------------------
procedure TCustomDBExDataSetNode.SetDataSource( Value : TDataSource );
begin
    if Value = FDatalink.Datasource then Exit;

    FDataLink.DataSource := Value;
    if Value <> nil then
    begin
         Value.FreeNotification(Self);
         SetupNewDataSet(Value.DataSet);
    end;
    LinkActive(FDataLink.Active);
end;

//------------------------------------------------------------------------
function TCustomDBExDataSetNode.GetDataSource:TDataSource;
begin
    Result:=FDataLink.DataSource;
end;

//------------------------------------------------------------------------
procedure TCustomDBExDataSetNode.LinkActive( Value : boolean );
var
  i : integer;
begin
  if (IsFreezed) or (csDestroying in ComponentState) then Exit;
  if (OwnerNodes.OwnerEx<>nil) and (not Value) then
  begin
      if (FTopicDispatchers.Count>0)and
         (TDBExDispatcher(FTopicDispatchers[0]).TreeNode=nil) and
         (TDBExDispatcher(FTopicDispatchers[0]).FParentDispatcher = FOwnerNodes.FRootDispatcher)
      then
      begin
          FOwnerNodes.InitShow;
      end
      else
      begin
          i:=0;
          while i < FTopicDispatchers.Count do
          begin
              TDBExDispatcher(FTopicDispatchers[i]).OnCollapse;
              inc( i );
          end;
      end;
  end
  else
  begin
      if (FDataLink.DataSet<>FDataSet) then
      begin
          FPostLoading:=True;
          SetupNewDataSet(FDataLink.DataSet);
          FPostLoading:=False;
      end;
      if (not FTopicVisible) and (FParentNode=FOwnerNodes.RootNode) then
      begin
          FCurrentFilter:=nil;
          OwnerNodes.InitShow;
      end;
  end;
end;

//------------------------------------------------------------------------
procedure TCustomDBExDataSetNode.DataSetChanged;
begin
    if IsFreezed then Exit;

    if (FDataLink.DataSet<>FDataSet) then
    begin
        SetupNewDataSet(FDataLink.DataSet);
    end
    else
    begin
        if (FActiveTopic<>nil) then
            if (FDataLink.DataSet.State=dsInsert) then
                CheckNewRecord
            else
                TDBExDataDispatcher(FActiveTopic).InternalRefresh;
    end;
end;

//------------------------------------------------------------------------
procedure TCustomDBExDataSetNode.DataSetScrolled( Distance : integer );
begin
    if IsFreezed then Exit;

    if FActiveTopic<>nil then
        TDBExDataDispatcher(FActiveTopic).Scroll(Distance);
end;

//------------------------------------------------------------------------
procedure TCustomDBExDataSetNode.RecordChanged( Field : TField );
begin
    if (IsFreezed) or (Field=nil) then Exit;

    if FActiveTopic<>nil then
    begin
        if UpperCase(Field.FieldName) = UpperCase(FLabelField) then
            TDBExDataDispatcher(
            FActiveTopic).LabelChanged( Field.AsString)
        else
            if UpperCase(Field.FieldName) = UpperCase(FKeyField) then
                TDBExDataDispatcher(
                FActiveTopic).KeyChanged( Field.AsString);
    end;
end;

//------------------------------------------------------------------------
procedure TCustomDBExDataSetNode.UpdateData;
begin
    if IsFreezed then Exit;

    if FActiveTopic<>nil then
       TDBExDataDispatcher(FActiveTopic).InternalRefresh;
end;

//------------------------------------------------------------------------
procedure TCustomDBExDataSetNode.Notification(AComponent: TComponent;
                                            Operation: TOperation);
var
    i : integer;
begin
    if (not (csDestroying in ComponentState))
        and (Operation = opRemove)
        and (FExpanded) then
    begin
        if (FDataLink<>nil) and
           (FDataLink.DataSource<>nil)

           then
        begin
            if FDataLink.DataSource=AComponent then
                FDataLink.DataSource:=nil
            else
                if FDataLink.DataSet=AComponent then
                    FDataLink.DataSource.DataSet:=nil;
            for i:=0 to FTopicDispatchers.Count-1 do
                TDBExDispatcher(FTopicDispatchers[i]).Select;
        end;
        if FTopicDataSource=AComponent then
            FTopicDataSource := nil;
    end;
    inherited Notification( AComponent, Operation);
end;

//------------------------------------------------------------------------
procedure  TCustomDBExDataSetNode.DriveDataset;
begin
    if (FDataLink<>nil) and
       (FdataLink.DataSource<>nil) and
       (FDataLink.DataSource.DataSet<>nil) and
       (FOwnerNodes.DrivenDS<> nil) and
       (FOwnerNodes.DrivenDS.DataSet<>FDataLink.DataSet) then
           FOwnerNodes.DrivenDS.DataSet:= FDataLink.DataSet;
end;

//------------------------------------------------------------------------
procedure  TCustomDBExDataSetNode.DriveTopicDataset;
begin
    if (FDataLink<>nil) and
       (FdataLink.DataSource<>nil) and
       (FDataLink.DataSource.DataSet<>nil) then
    begin
        if FOwnerNodes.DrivenDS<> nil then
        begin
            if FTopicDataSource <> nil then
                FOwnerNodes.DrivenDS.DataSet:= FTopicDataSource.DataSet
            else
                FOwnerNodes.DrivenDS.DataSet:= FDataLink.DataSet;
        end;
    end;
end;

//------------------------------------------------------------------------
procedure TCustomDBExDataSetNode.SetItemLabel( Value : string);
begin
    inherited SetItemLabel( Value );
    try
        DataSource.DataSet.FieldByName(LabelField).AsString := Value;
    except
    end;
end;

//------------------------------------------------------------------------
procedure TCustomDBExDataSetNode.EndEdit;
begin
    inherited EndEdit;
    try
        if FDataLink.DataSet<>nil then
            FDataLink.DataSet.Post;
    except
    end;
end;


//------------------------------------------------------------------------
function TCustomDBExDataSetNode.CanEdit : boolean;
begin
    Result := False;
    if not FReadOnly then
    begin
        if (FDataLink <> nil) and
           (FDataLink.DataSet <> nil) then
        begin
            Result:= (FDataLink.DataSet.CanModify);
            if Result then
                FDataLink.Edit;
        end;
    end;
end;

//------------------------------------------------------------------------
procedure TCustomDBExDataSetNode.GetPossibleLinks;
begin
    inherited GetPossibleLinks;
    if FDataLink.DataSet<>nil then
    begin
       if (FDataLink.DataSet is TTable) and
          (TTable(FDataLink.DataSet).MasterSource<>nil) and
          (TTable(FDataLink.DataSet).MasterSource.DataSet<>nil) then
            Include( FPossibleLinks, ltMaster )
       else
           if (FDataLink.DataSet is TQuery) and
              (TQuery(FDataLink.DataSet).DataSource<>nil) and
              (TQuery(FDataLink.DataSet).DataSource.DataSet<>nil) then
                Include( FPossibleLinks, ltMaster )
           else
           begin
               if ((FDataLink.DataSet is TQuery) and
                   (TQuery(FDataLink.DataSet).ParamCount>0))
                  or
                  ((FDataLink.DataSet is TStoredProc) and
                   (TStoredProc(FDataLink.DataSet).ParamCount>0)) then
                   Include( FPossibleLinks, ltParam )
           end;
    end;
end;

//------------------------------------------------------------------------
function TCustomDBExDataSetNode.GetField( FieldName : string ) : string;
begin
    try
        Result := FDataLink.DataSet.FieldByName(FieldName).AsString;
    except
        Result := '';
    end;
end;

//------------------------------------------------------------------------
function TCustomDBExDataSetNode.ParamCount : integer;
begin
    try
        if FDataLink.DataSet is TQuery then
            Result := TQuery(FDataLink.DataSet).ParamCount
        else
            Result := 0;
    except
        Result := 0;
    end;
end;

//------------------------------------------------------------------------
function TCustomDBExDataSetNode.ParamName( Index : integer ) : string;
begin
    try
        if FDataLink.DataSet is TQuery then
            Result := TQuery(FDataLink.DataSet).Params[Index].Name
        else
            Result := '';
    except
        Result := '';
    end;
end;

//------------------------------------------------------------------------
function TCustomDBExDataSetNode.FieldCount : integer;
var L:TStringList;
begin
    try
      if FDataLink.DataSet.FieldCount<>0
      then
      begin
        Result := FDataLink.DataSet.FieldCount;
      end
      else
      begin
        L:=TStringList.Create;
        FDataLink.DataSet.getfieldnames(L);
        Result:=L.Count;
        L.Free;
      end;
    except
        Result := 0;
    end;
end;

//------------------------------------------------------------------------
function TCustomDBExDataSetNode.FieldName( Index : integer ) : string;
var L:TStringList;
begin
    try
      if FDataLink.DataSet.FieldCount<>0
      then
      begin
        Result := FDataLink.DataSet.Fields[Index].FieldName;
      end
      else
      begin
        L:=TStringList.Create;
        FDataLink.DataSet.getfieldnames(L);
        if (Index>=0) and (Index<=L.Count-1) then
          Result:=L.Strings[Index];
        L.Free;
      end;
    except
        Result := '';
    end;
end;

//========================================================================
// TDBExLinkNode
//========================================================================
constructor TDBExLinkNode.Create( AOwner : TComponent );
begin
    inherited Create( AOwner );
    FMasterNode := nil;
    FNodeType:=ntLinkNode;
end;

//------------------------------------------------------------------------
destructor TDBExLinkNode.Destroy;
begin
   Destroying;
   inherited Destroy;
end;


//------------------------------------------------------------------------
function TDBExLinkNode.GetItem( Index : integer ): TDBExNode;
begin
    if Assigned( FMasterNode ) then
        Result:=FMasterNode.GetItem( Index )
    else
        Result:=nil;
end;

//------------------------------------------------------------------------
function TDBExLinkNode.DataActive : boolean;
begin
    if Assigned( FMasterNode ) then
        Result:=FMasterNode.DataActive
    else
        Result:=False;
end;

//------------------------------------------------------------------------
procedure TDBExLinkNode.OpenData;
begin
    if Assigned( FMasterNode ) then
        FMasterNode.OpenData;
end;

//------------------------------------------------------------------------
procedure TDBExLinkNode.CloseData;
begin
    if (FDriveDataSets) and (Assigned( FMasterNode )) then
        FMasterNode.CloseData;
end;

//------------------------------------------------------------------------
function TDBExLinkNode.GetDataItem : string;
begin
    if Assigned( FMasterNode ) then
        Result:=FMasterNode.GetDataItem
    else
        Result:='';
end;

//------------------------------------------------------------------------
procedure TDBExLinkNode.Next;
begin
    inherited Next;
    if Assigned( FMasterNode ) then
        FMasterNode.Next;
end;

//------------------------------------------------------------------------
procedure TDBExLinkNode.First;
begin
  inherited First;
    if Assigned( FMasterNode ) then
        FMasterNode.First;
end;

//------------------------------------------------------------------------
function TDBExLinkNode.MoveBy( By : integer):integer;
begin
    Result := inherited MoveBy( By );
    if Assigned( FMasterNode ) then
        Result := FMasterNode.MoveBy( By );
end;

//------------------------------------------------------------------------
function TDBExLinkNode.RecordCount:integer;
begin
    Result := 0;
    if Assigned( FMasterNode ) then
        Result := FMasterNode.RecordCOunt;;
end;

//------------------------------------------------------------------------
function TDBExLinkNode.GetKeyValue : string;
begin
    if Assigned( FMasterNode ) then
        Result:=FMasterNode.GetKeyValue
    else
        Result:='';
end;

//------------------------------------------------------------------------
function TDBExLinkNode.EndOfData : boolean;
begin
    if Assigned( FMasterNode ) then
        Result:=FMasterNode.EndOfData
    else
        Result:=true;
end;

//------------------------------------------------------------------------
function TDBExLinkNode.BeginOfData : boolean;
begin
    if Assigned( FMasterNode ) then
        Result:=FMasterNode.BeginOfData
    else
        Result:=true;
end;

procedure TDBExLinkNode.SetMasterNode( Value : TCustomDBExDataSetNode );
begin
    if FMasterNode<> Value then
    begin
        FMasterNode:=Value;

    end;
end;

//------------------------------------------------------------------------
function TDBExLinkNode.GetFirstChild: TDBExNode;
begin
    Result := nil;
    if not FOwnerNodes.DisableLinkNodes then
        if FMasterNode<> nil then
            Result:=FMasterNode.GetFirstChild;
end;

//------------------------------------------------------------------------
function TDBExLinkNode.GetLastChild: TDBExNode;
begin
    Result := nil;
    if not FOwnerNodes.DisableLinkNodes then
        if FMasterNode<> nil then
            Result:=FMasterNode.GetLastChild;
end;

//------------------------------------------------------------------------
function TDBExLinkNode.ChildrenCount: integer;
begin
    Result := 0;
    if not FOwnerNodes.DisableLinkNodes then
        if FMasterNode<> nil then
            Result:=FMasterNode.ChildrenCount;
end;

//------------------------------------------------------------------------
function TDBExLinkNode.GetNextChild(Value: TDBExNode): TDBExNode;
begin
    Result := nil;
    if not FOwnerNodes.DisableLinkNodes then
        if FMasterNode<> nil then
            Result:=FMasterNode.GetNextChild(Value);
end;

{$IFDEF _DELPHI3_}
procedure TDBExLinkNode.GetChildren(Proc: TGetChildProc; Root: TComponent);
begin
end;
{$ELSE}
procedure TDBExLinkNode.GetChildren(Proc: TGetChildProc);
begin
end;
{$ENDIF}

//------------------------------------------------------------------------
procedure TDBExLinkNode.SetFIlter( Disp:TDBExDispatcher );
begin

if FCurrentFilter<>nil then


    if FMasterNode=nil then Exit;

    if Disp=FMasterNode.FCurrentFilter then Exit;
    FMasterNode.FCurrentFilter:=Disp;
    inherited SetFilter( Disp );
end;

//------------------------------------------------------------------------
procedure TDBExLinkNode.FilterByMaster(MasterField:string;
                                       MasterValue : string );
begin
    if FMasterNode=nil then Exit;
    MasterNode.FilterByMaster( MasterField, MasterValue );
end;

//------------------------------------------------------------------------
procedure TDBExLinkNode.FilterByParam(DetailField:string;
                                      MasterValue : string);
begin
    if FMasterNode=nil then Exit;
    MasterNode.FilterByParam( DetailField, MasterValue );
end;

//------------------------------------------------------------------------
procedure TDBExLinkNode.FilterByFilter(DetailField:string;
                                       MasterValue : string);
begin
    if FMasterNode=nil then Exit;
    MasterNode.FilterByFilter( DetailField, MasterValue );
end;

//------------------------------------------------------------------------
procedure TDBExLinkNode.Clear;
begin
    if not (csDestroying in ComponentState) then
        MasterNode:=nil;
end;

//------------------------------------------------------------------------
procedure TDBExLinkNode.Notification(AComponent: TComponent;
                                      Operation: TOperation);
begin
    inherited Notification(AComponent, Operation);
    if Operation = opRemove then
    begin
        if AComponent=FMasterNode then
            MasterNode:=nil;
    end;
end;

//------------------------------------------------------------------------
procedure TDBExLinkNode.DriveDataSet;
begin
    if (FOwnerNodes.DrivenDS<>nil)and(FMasterNode<> nil) and
       (FMasterNode.DataSource<>nil) and
       (FMasterNode.DataSource.DataSet<>nil) then
    begin
        FOwnerNodes.DrivenDS.DataSet:= FMasterNode.DataSource.DataSet;
    end;
end;

//------------------------------------------------------------------------
procedure TDBExLinkNode.DriveTopicDataSet;
begin
    if (FMasterNode<> nil) and
       (FMasterNode.DataSource<>nil) and
       (FMasterNode.DataSource.DataSet<>nil) then
    begin
        if FMasterNode.FTopicDataSource<>nil then
            FOwnerNodes.DrivenDS.DataSet:= FMasterNode.FTopicDataSource.DataSet
        else
            FOwnerNodes.DrivenDS.DataSet:= FMasterNode.DataSource.DataSet;
    end;
end;

//------------------------------------------------------------------------
procedure TDBExLinkNode.Freeze;
begin
    inherited Freeze;
    if FMasterNode<>nil then
        FMasterNode.Freeze;
end;

//------------------------------------------------------------------------
procedure TDBExLinkNode.UnFreeze;
begin
    if FMasterNode<>nil then
        FMasterNode.Unfreeze;
    inherited UnFreeze;
end;

//------------------------------------------------------------------------
function TDBExLinkNode.CanEdit : boolean;
begin
    Result := False;
    if FMasterNode <> nil then
        Result:=FMasterNode.CanEdit;
end;

//------------------------------------------------------------------------
procedure TDBExLinkNode.SetActiveTopic(Value : TDBExDataDispatcher);
begin
    inherited SetActiveTopic( Value );
    if FMasterNode <> nil then
        FMasterNode.ActiveTopic:=Value;
end;

//------------------------------------------------------------------------
function TDBExLinkNode.GetField( FieldName : string ) : string;
begin
    Result := '';
    if FMasterNode <> nil then
        Result := FMasterNode.GetField( FieldName );
end;

//------------------------------------------------------------------------
function TDBExLinkNode.ParamCount : integer;
begin
    Result := 0;
    if FMasterNode <> nil then
        Result := FMasterNode.ParamCount;
end;

//------------------------------------------------------------------------
function TDBExLinkNode.ParamName( Index : integer ) : string;
begin
    Result := '';
    if FMasterNode <> nil then
        Result := FMasterNode.ParamName( Index );
end;

//------------------------------------------------------------------------
function TDBExLinkNode.FieldCount : integer;
begin
    Result := 0;
    if FMasterNode <> nil then
        Result := FMasterNode.FieldCount;
end;

//------------------------------------------------------------------------
function TDBExLinkNode.FieldName( Index : integer ) : string;
begin
    Result := '';
    if FMasterNode <> nil then
        Result := FMasterNode.FieldName( Index );
end;

//------------------------------------------------------------------------
procedure TDBExLinkNode.FindDependentFields( LinkValues : TDBExLinkValues );
begin
    if FMasterNode <> nil then
        FMasterNode.FindDependentFields( LinkValues );
end;

//------------------------------------------------------------------------
function TDBExLinkNode.GetItemImageIndex: integer;
begin
    Result := -1;
    if FMasterNode <> nil then
        Result := FMasterNode.ItemImageIndex;
end;

//------------------------------------------------------------------------
function TDBExLinkNode.GetItemSelectedIndex: integer;
begin
    Result := -1;
    if FMasterNode <> nil then
        Result := FMasterNode.ItemSelectedIndex;
end;

//------------------------------------------------------------------------
function TDBExLinkNode.GetItemStateIndex: integer;
begin
    Result := -1;
    if FMasterNode <> nil then
        Result := FMasterNode.ItemStateIndex;
end;

//========================================================================
// TDBExNodes
//========================================================================
constructor TDBExNodes.Create( AOwner : TComponent );
begin
    inherited Create;
    FOwnerEx:=TCustomDBExplorer(AOwner);
    FRootNode:=TDBExNode.Create(AOwner);
    FRootNode.FExplorer:=FOwnerEx;
    FRootNode.Text:='$$$$ ROOTNODE $$$$';
    FRootNode.FOwnerNodes:=self;
    DisableLinkNodes:=false;
    FEditorForm := nil;
    FLocked:=False;
    FLockSelect:=False;
    FFirstDisplayedDisp := nil;
    FLockExpandCollapse:=False;
end;

//------------------------------------------------------------------------
destructor TDBExNodes.Destroy;
begin
    FRootDispatcher.Free;
    FRootNode.Clear;
    FRootNode.Free;
    inherited Destroy;
end;

//------------------------------------------------------------------------
function TDBExNodes.GetItems : TTreeNodes;
begin
    Result := nil;
    if OwnerEx<>nil then
        Result := OwnerEx.Items;
end;


//------------------------------------------------------------------------
function TDBExNodes.Add( ANodeType : TDBExNodeType;
                         Node:TDBExNode; S : string ): TDBExNode;
begin
    if (Node = nil) then
        Result := FRootNode.AddChild( ANodeType,S )
    else
        Result := Node.Add( ANodeType,S );
    if FOwnerEx<> nil then InitShow;
end;

//------------------------------------------------------------------------
function TDBExNodes.AddChild( ANodeType : TDBExNodeType;
                              Node:TDBExNode; S : string ): TDBExNode;
begin
    if (Node = nil) then
        Result := FRootNode.AddChild( ANodeType,S )
    else
        Result := Node.AddChild( ANodeType,S );
end;

//------------------------------------------------------------------------
function TDBExNodes.AddChildFirst( ANodeType : TDBExNodeType;
                                   Node:TDBExNode; S : string ): TDBExNode;
begin
    if (Node = nil) then
        Result := FRootNode.AddChildFirst( ANodeType,S )
    else
        Result := Node.AddChildFirst( ANodeType,S );
end;

//------------------------------------------------------------------------
function TDBExNodes.AddFirst( ANodeType : TDBExNodeType;
                              Node:TDBExNode; S : string ): TDBExNode;
begin
    if (Node = nil) then
        Result := FRootNode.AddChildFirst( ANodeType, S )
    else
        Result := Node.AddFirst( ANodeType,S );
    if FOwnerEx<> nil then InitShow;
end;

//------------------------------------------------------------------------
procedure TDBExNodes.Remove( Node : TDBExNode );
var
    TN, TN1 : TTreeNode;
begin
    if (Node <> nil) AND (Node.ParentNode<>nil) then
    begin
        if (FOwnerEx<> nil) and (FEditorForm=nil) then
        begin
{            TN:=FOwnerEx.Items.GetFirstNode;
            while TN<>nil do
            begin
                TN1:=TN;
                TN:=TN.GetNext;
                if TObject(TN1.Data) is TDBExDispatcher then
                    TDBExDispatcher( TN1.Data ).ClearSubtree;
            end;}
            Node.DeleteDispatchers;
        end;
        Node.ParentNode.RemoveChild( Node );
    end;
end;

//------------------------------------------------------------------------
procedure TDBExNodes.Clear;
begin
    FRootNode.Free;
    FRootNode:=TDBExNode.Create(OwnerEx);
    FRootNode.Text:='$$$$ ROOTNODE $$$$';
    FRootNode.FOwnerNodes:=self;
    if FOwnerEx<>nil then
        TCustomDBExplorer(FOwnerEx).Items.Clear;
end;

//------------------------------------------------------------------------
function TDBExNodes.GetFirstNode:TDBExNode;
begin
if FRootNode=nil then ;
    Result:=FRootNode.GetFirstChild;
end;

//------------------------------------------------------------------------
function TDBExNodes.UniqueName( ClassName : string):string;
var
    i : integer;
    S : string;
begin
    i:=0;
    repeat
        i := i+1;
        s:=ClassName+IntToStr(i);
        Result := OwnerEx.Name+Copy(s,2,Length(s)-1);
    until OwnerEx.Owner.FindComponent( Result)=nil;
end;

//------------------------------------------------------------------------
procedure TDBExNodes.CloseEditor;
begin
  FEditorForm := nil;
end;

//------------------------------------------------------------------------
procedure TDBExNodes.BeforeCollapse( Node : TTreeNode );
begin
    if FLockExpandCollapse then Exit;
    if TObject(Node.Data) is TDBExDispatcher then
    begin
        Items.BeginUpdate;
        TDBExDispatcher(Node.Data).OnCollapse;
        Items.EndUpdate;
    end;
end;

//------------------------------------------------------------------------
procedure TDBExNodes.BeforeExpand( Node : TTreeNode );
begin
    if FLockExpandCollapse then Exit;
    if TObject(Node.Data) is TDBExDispatcher then
    begin
        FLocked:=True;
        Items.BeginUpdate;
        TDBExDispatcher(Node.Data).BeforeExpand;
        TDBExDispatcher(Node.Data).FillDown;
        if Node.Count=0 then
        begin
            TDBExDispatcher(Node.Data).AfterExpand;
            Items.EndUpdate;
            FLocked:=False;
        end;
    end;
end;

//------------------------------------------------------------------------
procedure TDBExNodes.AfterExpand( Node : TTreeNode );
begin
    if FLockExpandCollapse then Exit;
    if TObject(Node.Data) is TDBExDispatcher then
    begin
        TDBExDispatcher(Node.Data).AfterExpand;
        Items.EndUpdate;
        FLocked:=False;
    end;
end;

//------------------------------------------------------------------------
procedure TDBExNodes.ClearNode( Disp : TDBExDispatcher );
var
    Node:TTreeNode;
begin
    Node:=Disp.FTreeNode;
    Node.Data:=nil;
    Node.DeleteChildren;
    Disp.Free;
end;

//------------------------------------------------------------------------
procedure TDBExNodes.InitShow;
begin
    if FOwnerEx=nil then Exit;

    Items.BeginUpdate;
    if FRootDispatcher <> nil then
        FRootDispatcher.Free;
    if Unregistered then
    begin
        Items.Add(nil,'Unregistered Version');
        Items.Add(nil,'not working without IDE');
    end
    else
    begin
        FRootDispatcher := FRootNode.CreateDispatcher( nil,nil );
        FRootDispatcher.BeforeExpand;
        FRootDispatcher.FillDown;
    end;
    Items.EndUpdate;
end;

//------------------------------------------------------------------------
function TDBExNodes.GetNextVisible( Node : TTreeNode ) : TTreeNode;
begin
        Result := Node.GetNextSibling;
    while (Node<>nil) and
          ((Result=nil) )  do
    begin
        Node:=Node.Parent;
        if Node<>nil then
            Result:=Node.GetNextSibling;
    end;
end;

//------------------------------------------------------------------------
function TDBExNodes.AddTreeNode( Parent : TTreeNode; Disp : TDBExDispatcher;
                                 Before : TTreeNode;
                                 NodeText:string):TTreeNode;
begin
    if Parent<>nil then
    begin
        if Before=nil then
        begin
            Result := nil;
           if Result=nil then
                Result:=Items.AddChildObject(Parent, NodeText, Disp)
            else
                Result.Data:=Disp;
            Before := GetNextVisible( Parent );
        end
        else
            Result:=Items.InsertObject(Before, NodeText, Disp);
    end
    else
    begin
        if Before=nil then
            Result:=Items.AddObject( Before,NodeText,  Disp)
        else
        begin
            Result:=Items.InsertObject( Before,NodeText, Disp);
        end;
    end;
end;

//------------------------------------------------------------------------
procedure TDBExNodes.PerformExpand( Node : TTreeNode );
begin
    if not Node.Expanded then
    begin
        FLockExpandCollapse := True;
        Node.Expand( False );
        FLockExpandCollapse := False;
    end;
end;

//========================================================================
// TCustomDBExplorer
//========================================================================
constructor TCustomDBExplorer.Create(AOwner: TComponent);
begin
  FScrollTimerID :=-1;
  inherited Create(AOwner);
  FNodes := TDBExNodes.Create(self);
  FStructDone := False;
  FPrevScrollPos := 0;
  FBlockPaint := false;
  FLockOnLoad := False;
  FullDestroy:=False;
end;

//------------------------------------------------------------------------
destructor TCustomDBExplorer.Destroy;
begin

    if FNodes<>Nil then
    begin
      if  FNodes.FEditorForm<>Nil then FNodes.FEditorForm.Close;
      FNodes.Free;
    end;
    inherited Destroy;
end;

//------------------------------------------------------------------------
procedure TCustomDBExplorer.WMDestroy(var Message: TWMDestroy);
begin
    FullDestroy:=True;
    inherited;
end;

//------------------------------------------------------------------------
procedure TCustomDBExplorer.SetDrivenDS( Value : TDataSource );
begin
    FNodes.DrivenDS:=Value;
end;

//------------------------------------------------------------------------
function TCustomDBExplorer.GetDrivenDS : TDataSource;
begin
    Result := FNodes.DrivenDS;
end;

//------------------------------------------------------------------------
procedure TCustomDBExplorer.Loaded;
begin
    FLockOnLoad := True;
    FStructDone:=True;
    inherited Loaded;
    FNodes.InitShow;
    FLockOnLoad := False;
end;

procedure TCustomDBExplorer.CreateWnd;
var
    N : TTreeNode;
begin
    inherited CreateWnd;
    N:=Items.GetFirstNode;
    while N<>nil do
    begin
        if N.Data<>nil then
            TDBExDispatcher( N.Data ).FTreeNode := N;
        N := N.GetNext;
    end;
end;

{$IFDEF _DELPHI3_}
procedure TCustomDBExplorer.GetChildren(Proc: TGetChildProc; Root: TComponent);
var
  N : TDBExNode;
begin
  N:=FNodes.GetFirstNode;
  while N <> nil do
  begin
    if N.Owner = Root then Proc(N);
    N := N.GetNextSibling;
  end;
end;
{$ELSE}
procedure TCustomDBExplorer.GetChildren(Proc: TGetChildProc);
var
  N : TDBExNode;
begin
  N:=FNodes.GetFirstNode;
  while N <> nil do
  begin
    if N.Owner = Owner then Proc(N);
    N := N.GetNextSibling;
  end;
end;
{$ENDIF}

procedure TCustomDBExplorer.SetName(const Value: TComponentName);
var
  N : TDBExNode;
  OldName, NodeName, NamePrefix: TComponentName;
begin
  OldName := Name;
  inherited SetName(Value);

  begin
    if FNodes=nil then
        N:=nil
    else
    begin
        N:=FNodes.GetFirstNode;
    end;
    while N <> nil do
    begin
      if N.Owner = Owner then begin
        NodeName := N.Name;
        NamePrefix := NodeName;
        if Length(NamePrefix) > Length(OldName) then begin
          SetLength(NamePrefix, Length(OldName));
          if CompareText(OldName, NamePrefix) = 0 then begin
            System.Delete(NodeName, 1, Length(OldName));
            System.Insert(Value, NodeName, 1);
            try
              N.Name := NodeName;
            except
              on EComponentError do
            end;
          end;
        end;
      end;
      N:=N.GetNext;
    end;
  end;
end;


//------------------------------------------------------------------------
function TCustomDBExplorer.CanCollapse(Node: TTreeNode): Boolean;
begin
   Result := inherited CanCollapse(Node);
   if (FNodes<>nil) and Result then
       FNodes.BeforeCollapse( Node );
end;

//------------------------------------------------------------------------
function TCustomDBExplorer.CanExpand(Node: TTreeNode): Boolean;
begin
   if FLockOnLoad then
       Result := False
   else
   begin
       Result := inherited CanExpand(Node);
       if (FNodes<>nil) and Result then
           FNodes.BeforeExpand( Node );
   end;
end;

//------------------------------------------------------------------------
procedure TCustomDBExplorer.Expand(Node: TTreeNode);
begin
    inherited Expand( Node );
    if FNodes<>nil then
        FNodes.AfterExpand( Node );
end;

//------------------------------------------------------------------------
function TCustomDBExplorer.CanChange(Node: TTreeNode): Boolean;
begin
   Result := True;
   if FNodes.FLockSelect then Exit;
   Result := inherited CanChange(Node);
   if Result then
   begin
       if (Selected<>nil) then
       begin
           if (Selected.Data<>nil) and
              (TObject(Selected.Data) is TDBExDispatcher)
              and (TObject(Node.Data) is TDBExDispatcher) then
              PassSelection(TDBExDispatcher(Selected.Data),
                         TDBExDispatcher(Node.Data))
           else
           begin
              if (TObject(Selected.Data) is TDBExDispatcher) then
                  TDBExDispatcher(Selected.Data).Deselect;
              if (TObject(Node.Data) is TDBExDispatcher) then
                  TDBExDispatcher(Node.Data).Select;
           end;
       end
       else
           if (TObject(Node.Data) is TDBExDispatcher) then
              TDBExDispatcher(Node.Data).Select;

   end;
end;

//------------------------------------------------------------------------
procedure TCustomDBExplorer.PassSelection(DFrom, DTo : TDBExDispatcher);
begin
    if (DFrom.FParentDispatcher = DTo.FParentDispatcher) and
       (DFrom.FParentDispatcher is TDBExDataDispatcher) then
        TDBExDataDispatcher(DFrom.FParentDispatcher).SelectionMoved(
             TDBExDataDispatcher(DFrom), TDBExDataDispatcher(DTo) )
    else
    begin
        DFrom.Deselect;
        DTo.Select;
    end;
end;

//------------------------------------------------------------------------
function TCustomDBExplorer.CanEdit(Node: TTreeNode): Boolean;
begin
    Result := inherited CanEdit( Node );
    if result and (TObject(Node.Data) is TDBExDispatcher) then
          Result := TDBExDispatcher(Node.Data).CanEdit;
end;

//------------------------------------------------------------------------
procedure TCustomDBExplorer.Edit(const Item: TTVItem);
begin
    inherited Edit( Item );

    if (Item.pszText<>nil)and(TObject(Selected.Data) is TDBExDispatcher) then
          TDBExDispatcher(Selected.Data).Text:= StrPas( Item.pszText );
    TDBExDispatcher(Selected.Data).EndEdit;
end;

//------------------------------------------------------------------------
procedure TCustomDBExplorer.WndProc(var Message: TMessage);
var P:TPoint;
    Id : Integer;
begin
  if (FScrollTimerID = -1) And Dragging  then begin
     FScrollTimerID := SetTimer(Handle, 1, 200, @ScrollTreeViewTimerProc);
  end;
  if (FScrollTimerID <> -1) And Not Dragging then begin
    Id := FScrollTimerID;
    FScrollTimerID := -1;
    KillTimer(Handle, Id);
  end;

    if Dragging And ((Message.Msg=WM_KEYDOWN) or (Message.Msg=WM_KEYUP)) then
    begin
      if (Message.wParam=VK_CONTROL) then
      begin
        getcursorpos(p);
        Windows.ScreenToClient(GetCapture, p);
        SendMessage(GetCapture, LongInt(WM_MOUSEMOVE), MK_LBUTTON, MAKELONG(p.X, p.Y));
      end;
    end;

    if not ((FBlockPaint) and ((Message.Msg=WM_PAINT))) then
        inherited WndProc(Message);	
end;

//------------------------------------------------------------------------
procedure TCustomDBExplorer.WMRButtonUp(var Message: TWMRButtonUp);
var Node:TTreeNode;
begin
  Node:=GetNodeAt(Message.Pos.x,Message.Pos.y);
  if (Node<>Nil) and (Node.Data<>Nil)
  then
  begin
    if (TDBExDispatcher(Node.Data).FTopicNode<>Nil) then
    begin
      if not TDBExNode(TDBExDispatcher(Node.Data).FTopicNode).
        CheckMenuPopup(Message.Pos) then
          Inherited;
      Exit;
    end;
  end;
  inherited;
end;

//------------------------------------------------------------------------
procedure TCustomDBExplorer.CMDrag(var Message:TCMDrag);
begin
  with Message, DragRec^ do
  begin
    case DragMessage of
      dmDragEnter:
      begin
      end;
      dmDragMove:
      begin

      end;
      dmDragLeave:
      begin
        DropTarget:=Nil;
      end;
      dmDragCancel:
      begin
        DropTarget:=Nil;
      end;
    end;
  end;
  Inherited;

end;

//------------------------------------------------------------------------
procedure TCustomDBExplorer.WMMouseMove(var Message: TWMMouseMove);
var
   Node:TTreeNode;
   Rect:TRect;
   wParam:Word;
   lParam:LongInt;
   p,Pos:TPoint;

const AFree:Boolean=True;
begin
  if AFree And (FScrollTimerID <> -1) And Dragging then
  begin
    GetCursorPos(P);
    Windows.ScreenToClient(WindowHandle,P);
    if (P.X>=0) and (P.X<=Width) then
    begin
      Node := Items.GetFirstNode;
      if (Node<>nil) then
      begin
        Rect:=Node.DisplayRect(False);
        Node:=GetNodeAt(P.X,P.Y);
        if (P.Y>=0) and
           (P.Y<=2*(Rect.Bottom-Rect.Top)) and
           (Node<>Nil)
        then
        begin
          AFree:=False;
          if FDragObject<>Nil then
            FDragObject.HideDragImage;
          LoWord(wParam):=SB_LINEUP;
          Perform(WM_VSCROLL,wParam,lParam);

          if FDragObject<>Nil then
            FDragObject.ShowDragImage;
          WMMouseMove(Message);
          AFree:=True;
        end;
        if (P.Y<=Height) and
           (P.Y>=Height-3*(Rect.Bottom-Rect.Top)) and
           (Node<>Nil)
        then
        begin
          AFree:=False;
          if FDragObject<>Nil then
            FDragObject.HideDragImage;
          LoWord(wParam):=SB_LINEDOWN;
          Perform(WM_VSCROLL,wParam,lParam);
          if FDragObject<>Nil then
            FDragObject.ShowDragImage;
          WMMouseMove(Message);
          AFree:=True;
        end;
      end;
    end;
  end
  else
    Inherited;
end;

//------------------------------------------------------------------------
procedure TCustomDBExplorer.DoStartDrag(var DragObject: TDragObject);
begin
  inherited;
  If DragObject=nil then
    DragObject := TAutoDragObject.Create(self);
  FDragObject:=DragObject;
  FDragTreeNode:=Selected;
end;

//------------------------------------------------------------------------
procedure TCustomDBExplorer.DragOver(Source: TObject; X, Y: Integer; State: TDragState;var Accept: Boolean);
var Node:TTreeNode;
begin
  Inherited;
  Node:=GetNodeAt(X,Y);
  If Node<>Nil then
  begin
    if (not Assigned(OnDragOver)) and
    (TObject(FDragTreeNode.Data) is TDBExMetaDispatcher) then
    begin
      Accept:= (  TDBExDispatcher(Node.data).GetTopicNode =
                  TDBExDispatcher(FDragTreeNode.data).GetTopicNode
               )
              and
               (  ( TDBExDispatcher(Node.data).FParentDispatcher <>
                    TDBExDispatcher(FDragTreeNode.data).FParentDispatcher)
               )
              and
                (TDBExDispatcher(Node.data).CanEdit)
              and
                (TDBExDispatcher(FDragTreeNode.data).CanEdit)
    end;
  end;
  FMayDrop := Accept;
end;

//------------------------------------------------------------------------
procedure TCustomDBExplorer.DragDrop(Source:TObject; X,Y:Integer);
var Accept,DoCopy:Boolean;
    Dest:TTreeNode;
    MaxKeyFieldValue : Variant;
    DragDataSet:TDataSet;
    DragNode:TCustomDBExDataSetNode;
    DestDisp,DragDisp : TDBExDispatcher;
begin
  Accept:=True;
  DoCopy := False;
  inherited;
  Dest:=GetNodeAt(X,Y);
  if ((Dest<>nil)and(Dest.Data<>nil)) then
      DestDisp:= TDBExDispatcher(Dest.data);
  if ((FDragTreeNode<>nil)and(FDragTreeNode.Data<>nil)) then
      DragDisp:= TDBExDispatcher(FDragTreeNode.data);
  if Assigned(FOnDragDropNode) then
  begin
      FOnDragDropNode(DestDisp,DragDisp,Accept,DoCopy);
  end;
  If Accept then
  begin
    If Not DoCopy then
    begin
    // ****************************************
    //  MOVE
    // ****************************************
      Items.BeginUpdate;
      if (DestDisp.FTopicNode is TCustomDBExDataSetNode) or
         (DestDisp.FTopicNode is TDBExLinkNode) then
      begin
        if TDBExNode(DragDisp.FTopicNode)
            is TDBExLinkNode then
            DragNode := TDBExLinkNode(DragDisp.FTopicNode).MasterNode
        else
            DragNode:=TCustomDBExDataSetNode(DragDisp.FTopicNode);
        if (DragNode<>nil) then
            DragDataSet:=DragNode.FDataLink.DataSet
        else
            DragDataSet:=nil;
        if (DragDataSet <> Nil) and (DragDataSet.Active) then
        begin
          If not (DragDataSet.State in [dsInsert,dsEdit]) then DragDataSet.Edit;
          if DestDisp is TDBExDataDispatcher then
              TDBExDataDispatcher(DestDisp).FillLinkFields( DragDataSet )
          else
              TDBExDataDispatcher(DestDisp.ParentDispatcher).FillLinkFields( DragDataSet );

          DragDataSet.Post;
          if DestDisp is TDBExDataDispatcher then
              TDBExDataDispatcher(DestDisp).Refresh
          else
              TDBExDataDispatcher(DestDisp.ParentDispatcher).Refresh;
        end;
      end;
      Items.EndUpdate;
    end
    else
    begin
    // ****************************************
    //  MOVE
    // ****************************************
      MaxKeyFieldValue:=0;
    end;
  end;
end;


//========================================================================
Initialization
  RegisterClasses( [TDBExNodes,TDBExNode, TDBExDataSetNode,
         TDBExLinkNode, TDBExDataNode,TDBExCustomNode,
         TCustomDBExDataSetNode,
         TCustomDBExplorer,TDBExplorer,
         TDBExNodeLink,TDBExNodeLinks]);

end.
