/*
 *
 *	This example application demonstrates how to extend bCAD functionality
 *	using Java API. This program plots 3D surfaces (like landscapes) by
 *	reading data from a file, which contains coordinates of regular mesh vertices 
 * 
 *	Data format is simple: data file is an ASCII file with .xyz extension.
 *	First line contains a prefix XYZ as a format identifier and two integer
 *	numbers defining mesh dimensions. Other lines contain vertices coordinates -
 *	one per line.
 *
 *	Example:
 *	XYZ 2 2
 *	0 0 0
 *	0 1 0
 *	1 0 0
 *	1 1 0
 *
 * 
 * Copyright 1997 ProPro Group. All Rights Reserved.
 *
 * Developed by Alexey Balagansky 
 *
 *
 */

import java.awt.*;
import java.io.*;
import java.util.*;
import bCAD;

/* Class Lanscape is an a subclass of bCAD class. 
   It provides main functionality of our application.
 */

class Landscape extends bCAD
{
	// a Frame object which will be aparent for dialog box.
	private MainFrame mainfrm;					

	// prefix which indicates correct data format. We declare it final
	// because it is a constant and will never be changed.
	public final static String header = "XYZ";

	// Variables to store the data. Their names are quite descriptive.
	public int Width;
	public int Height;
	public float [] dataArray;

	
	public Landscape()   // constructor
	{
	}/*Landscape*/

	public static void main(String []argv)
	{
		//Inform user, if he tries to run this program outside bCAD

		System.out.println("This program is an extension module for bCAD.");
		System.out.println("It will not run as a stand-alone application.");
	}/*main*/


	public void bcadmain(long ctxt, float x, float y, float z) //Entry point
	{
		//We create an instance of Landscape class
		Landscape land = new Landscape();
		// and make him do the job
		land.run(ctxt);
	}/*bcadmain*/


	public void run(long ctxt)
	{
		mainfrm = new MainFrame(this);
		// get file name that user has selected
		String name = mainfrm.run();

		if (name !=null) //he selected something
		{
			traceMessage("You selected "+name);
			if (loadFile(name)!=true) //error occured while loading data from file
			{
				// inform user and stop executing
				MessageBox mb = new MessageBox("Error","Error loading file");
				mb.display();
				return;
			}
			
		}
		else
		{
			// user didn't select any file
			traceMessage("User cancelled");
			return;
		}

		//plot the mesh using loaded data
		createMesh(ctxt, dataArray, Width, Height);
		fixState(ctxt);

		//inform user 
		MessageBox mb = new MessageBox("Success","Successfully processed file");
		mb.display();

				
	}/*run*/

	public boolean loadFile(String name)	//load data from file
	{
		DataInputStream f;	// stream used for input
		StringTokenizer st; // class used to split string into fields
		String s;
		
		try
		{
			// open file for reading
			f = new DataInputStream(new FileInputStream(name));
			s = f.readLine();
		}
		catch (IOException e) 
		{
			// catch i/o error
			traceMessage("IOException "+e);
			return false;
		}

		st = new StringTokenizer(s," ");
		if (st.nextToken().compareTo(header)==0) //header is present
		{
			try
			{
				// determine mesh dimensions
				Width=Integer.parseInt(st.nextToken());
				Height=Integer.parseInt(st.nextToken());
			}
			catch (NumberFormatException e)
			{
				// dimensions defined incorrectly
				traceMessage("Array dimensions are wrong or not set");
				return false;
			}
				
		}
		else	// header not found
		{
			traceMessage("Bad file header");
			return false;
		}
		
		// create array for data storage
		dataArray = new float[3*Width*Height];
		int i = 0;
		int j = 0;

		// skip all blank lines between header and data
		s = "dummy";
		while (s.compareTo("")==0)
		{
			try
			{
				s = f.readLine();
			}
			catch (IOException e)
			{ 
				traceMessage("I/O error" + e);
				return false;
			}
		}

		// load vertices coordinates
		for (j=0; j<Width*Height; j++)
		{
			try
			{
				// read next line
				s = f.readLine();
			}
			catch (IOException e)
			{ 
				traceMessage("I/O error" + e);
				return false;
			}
			
			st = new StringTokenizer(s," ");
			try
			{
				// split line into 3 fields containing x,y and z
				dataArray[i] = Float.valueOf(st.nextToken()).floatValue();
				dataArray[i+1] = Float.valueOf(st.nextToken()).floatValue();
				dataArray[i+2] = Float.valueOf(st.nextToken()).floatValue();
			}
			catch (NumberFormatException e) 
			{
				// ignore incorrect numbers
				dataArray[i]=0;
				dataArray[i+1]=0;
				dataArray[i+2]=0;
			};
			i += 3;
		} 
		
		// a debuggung procedure which dumps array content into trace window
		// uncomment it if you want
		// dumpArray();

		
		return true;
	}/*loadFile*/

	public void dumpArray() // dump array content into trace window
	{
		traceMessage(Width*Height + " lines of data successfully loaded");
		int i = 0;
		for (int j = 0; j<Width*Height; j++)
		{
			traceMessage("Line :"+dataArray[i]+" "+dataArray[i+1]+" "+dataArray[i+2]);
			i += 3;
		}
	}/*dumpArray*/


}/*class Landscape*/

/* Class MainFrame implements a hidden frame which is a parent for dialog
 boxes. Since every dialog box needs to have a "parent", we must have a
 frame even it is never displayed.
 */

class MainFrame extends Frame
{
	// Dialog window which allows to select a file from list.
	private FileDialog fd;
						
	public MainFrame(Landscape l)	// constructor
	{
		//creating the frame
		super("Main frame");
		setResizable(false);
		resize(100,100);

		//creating dialog box
		fd = new FileDialog(this, "Select file");
		fd.setFile("*.xyz");

	}/*MainFrame*/

	public String run()	//returns file name if user selected any or null otherwise
	{
		fd.show();
		if (fd.getFile()!=null) //user selected something
		{
			return fd.getDirectory()+fd.getFile();
		}
		else return null;
		
	}/*run*/


}/*class MainFrame*/

/*
 Class MessageBox implements a dialog box which is intended to 
 display a message for user.
*/

class MessageBox extends Frame
{
	private boolean stop = false;
	public Button OK;
	public Label label;

	public MessageBox(String title, String text)	// constructor
	{
		// creating a frame
		super(title);

		// setting layout manager
		GridBagLayout gbl = new GridBagLayout();
		GridBagConstraints gbc = new GridBagConstraints();
		gbc.weightx = 1.0;
		gbc.weighty = 1.0;
		gbc.gridwidth = GridBagConstraints.REMAINDER;

		// customizing window
		setFont(new Font("MS Sans serif", Font.PLAIN, 12));
		setLayout(gbl);
		resize(200,100);
		setResizable(false);
		
		//adding controls
		label = new Label (text, Label.CENTER);
		gbl.setConstraints(label, gbc);
		add(label);
				
		OK = new Button ("OK");
		gbl.setConstraints(OK, gbc);
		add(OK);
	
					
	}/*MessageBox*/

	public synchronized boolean handleEvent(Event e)	// event handler
	{

		// close window if user clicks on "close" button
		if (e.id == Event.WINDOW_DESTROY)
		{
			stop = true;
			return true;
		}
		// close window if user  presses "OK" button
		if (e.id == Event.ACTION_EVENT)
		{
			if (e.target instanceof Button)
			{
				stop = true;
				return true;
			}
		}
		return super.handleEvent(e);

	}/*handleEvent*/

	public void display()	// show window and wait for user to close it
	{
		show();
		try
		{
			while (stop != true) { Thread.sleep(300);};
		}
		catch (InterruptedException e) {};

		dispose();
	}/*display*/

}