/*
 * (c) Copyright 1997, KL GROUP INC.
 * ALL RIGHTS RESERVED
 *
 * This file is provided for demonstration and educational uses only.
 * Permission to use, copy, modify and distribute this file for
 * any purpose and without fee is hereby granted, provided that the
 * above copyright notice and this permission notice appear in all
 * copies, and that the name of KL Group not be used in advertising
 * or publicity pertaining to this material without the specific,
 * prior written permission of an authorized representative of
 * KL Group.
 *
 * KL GROUP MAKES NO REPRESENTATIONS AND EXTENDS NO WARRANTIES, EX-
 * PRESS OR IMPLIED, WITH RESPECT TO THE SOFTWARE, INCLUDING, BUT
 * NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR ANY PARTICULAR PURPOSE, AND THE WARRANTY AGAINST IN-
 * FRINGEMENT OF PATENTS OR OTHER INTELLECTUAL PROPERTY RIGHTS.  THE
 * SOFTWARE IS PROVIDED "AS IS", AND IN NO EVENT SHALL KL GROUP OR
 * ANY OF ITS AFFILIATES BE LIABLE FOR ANY DAMAGES, INCLUDING ANY
 * LOST PROFITS OR OTHER INCIDENTAL OR CONSEQUENTIAL DAMAGES
 * RELATING TO THE USE OF THIS SOFTWARE.
 */

//   RCSID -- $RCSfile: JCDialog.java $ $Revision: 1.2 $
//            $Date: 1997/03/14 21:15:36 $  $Locker: $  KL Group Inc.

package jclass.contrib;
import java.awt.*;

/**
 * JCDialog implements an OK/CANCEL dialog.
 * By default, JCDialog is made up of a drawArea and a buttonArea
 * containing an okButton, an applyButton, and a cancelButton.
 * These members are accessible by all. <P>
 * JCDialogCB supports two callback methods. The first method is
 * to subclass JCDialog and override the buttonCB() member function.
 * The second methos is to implement the JCDialogCB interface and provide
 * an implementation of the same buttonCB() member function, and then
 * use the setButtonCB() member to register the receiver with the
 * JCDialog class. In both cases, a return of false enables the
 * default callback failure (usually unmapping or hiding).<P>
 * Adding items to JCDialog is done through the drawArea member. It is
 * an instance of JCOutlinePanel, and is derived from the AWT panel.<P>
 * The following is an example of how to use JCDialog:<P>
 <PRE>
   class QuitBox extends JCDialog {
   public QuitBox(Frame fg) {
    super(fg, "Quit Application?");

    drawArea.setLayout(new JCRowColLayout(1, 1));
    drawArea.add(new Label("Quit Application?"));
    okButton.setLabel("Yes");
    cancelButton.setLabel("No");
   }
   public boolean buttonCB(JCDialog d, Button b) {
    if (b == okButton)
        System.exit(0);
    return handleEvent(new Event(this, Event.WINDOW_DESTROY, null));
   }
   }
 </PRE>
 * In this example, the "override" technique is used.  Note that the
 * button labels are changed to "Yes" and "No".<P>
 * JCDialog does not support modal dialogs.
 */
public class JCDialog extends Frame implements JCDialogCB {

    /** OK button. */
    public Button okButton = null;

	/** Apply button. */
    public Button applyButton = null;

	/** Cancel button. */
    public Button cancelButton = null;

	/** Main draw area, which is the body of the dialog. */
    public JCOutlinePanel drawArea;

	/** TButton panel. Add custom buttons here. */
    public JCOutlinePanel buttonArea;

	/** Registered callback object. */
    private JCDialogCB _cb = null;

	/** Parent frame. This is used to position the dialog with the parent */
    private Component _parent = null;

/**
 * Constructs a JCDialog instance for the OK and Cancel buttons.
 * @param parent the parent component, used to position the dialog; if not a Frame, the component's parent frame is found
 * @param title the title of the dialog box
 */
public JCDialog(Component parent, String title) {
    super(title);
    init(parent, title, false);
}

/**
 * Constructs a JCDialog instance for the OK, Apply, and Cancel buttons.
 * @param parent the parent component, used to position the dialog; 
 * if not a Frame, the component's parent frame is found
 * @param title the title of the dialog box
 * @param apply if true, the Apply button is added
 */
public JCDialog(Component parent, String title, boolean apply) {
    super(title);
    init(parent, title, apply);
}

private static Frame
getFrame(Component comp) {
	Component tmp = comp, parent = comp;
	while (true) {
		if (parent == null || parent instanceof Frame)
			break;
		parent = tmp;
		tmp = tmp.getParent();
	}
	return (Frame) parent;
}

/**
 * A utility function that performs the dialog construction.
 * @param parent the parent frame
 * @param title the title of the dialog box
 * @param apply if true, the Apply button is added
 */
private void init(Component parent, String title, boolean apply) {
    _parent = parent;

	GridBagLayout gb = new GridBagLayout();
	GridBagConstraints c = new GridBagConstraints();
	setLayout(gb);
    drawArea = new JCOutlinePanel();
    add(drawArea);

	c.fill = GridBagConstraints.BOTH;
	c.weightx = 1.0;
	c.weighty = 1.0;
	gb.setConstraints(drawArea, c);

    buttonArea = new JCOutlinePanel();
    buttonArea.setLayout(new FlowLayout());
    add(buttonArea);
    buttonArea.add((okButton = new Button("OK")));
    if (apply)
        buttonArea.add((applyButton = new Button("Apply")));
    buttonArea.add((cancelButton = new Button("Cancel")));
	c.gridy = 1;
	c.weighty = 0.0;
	gb.setConstraints(buttonArea, c);
}

/**
 * Gets the parent frame.
 */
public Frame getParentFrame() {
	return getFrame(_parent);
}

/**
 * The Overridden Component.show() member. It guarantees
 * the peer has been allocated and attempts to position the dialog
 * relative to the parent dialog.
 */
public synchronized void show() {
	if (getPeer() == null)
		addNotify();
	pack();
    resize(preferredSize().width+insets().left+insets().right,
		   preferredSize().height+insets().top+insets().bottom);

	Frame frame = getFrame(_parent);
    if (frame != null) {
        // ATTN This doesn't work!
        Rectangle bounds = frame.bounds(), abounds = bounds();
		int x = bounds.x, y = bounds.y;

		if (abounds.width > bounds.width)
			x += bounds.width/2;
		else
			x += (bounds.width-abounds.width)/2;

		if (abounds.height > bounds.height)
			y += bounds.height/2;
		else
			y += (bounds.height - abounds.height)/2;

        if (x < 0) x = 0;
        if (y < 0) y = 0;
        move(x,y);
    }
	else {
		move(preferredSize().width/2, preferredSize().height/2);

	}
    super.show();
}

/**
 * Registers a callback recipient for the dialog. There can only
 * be one callback recipient per dialog. Recipient must
 * implement the JCDialogCB interface.
 * @param target the instance of an object implementing the JCDialogCB interface
 */
public void setButtonCB(JCDialogCB target) {
    // Avoid circular reference
    if (target != this)
        _cb = target;
}

/**
 * Default implementation of a member to clear the dialog.
 */
public void clear() {
}

/**
 * Default implementation of the button callback.
 * @param d the dialog in which the callback occurred
 * @param b the button that was pressed
 */
public boolean buttonCB(JCDialog d, Button b) {
    return false;
}

/**
 * Local event handler. It sends button events to the
 * registered callback recipient. If none were registered, the
 * buttonCB() member of the current object is called instead.
 * Note that if the button callbacks return false, the default handling
 * is performed.
 */
public boolean handleEvent(Event event) {
    if (event.id == Event.WINDOW_DESTROY) {
        hide();
        return true;
    }
    else if (event.id == Event.KEY_PRESS && event.key == 10) {
		if (okButton != null)
			okButton.postEvent(new Event(okButton, Event.ACTION_EVENT, null));
	}
	else if (event.id == Event.ACTION_EVENT) {
		if (event.target instanceof Button) {
            if (_cb != null && _cb.buttonCB(this, (Button)event.target))
                return true;
            else if (buttonCB(this, (Button)event.target))
                return true;
            hide();
            return true;
        }
    }
    return super.handleEvent(event);
}

}

