/*
 *  Author : Stanley Poon
 *  Version: 1.0, 12 Feb 1996
 *
 *  Base on:
 *    Daniel Wyszynski 
 *    Center for Applied Large-Scale Computing (CALC) 
 *    04-12-95 
 *    Test of text animation.
 *    kwalrath: Changed string; added myThread suspension. 5-9-95
 */
import java.applet.Applet;
import java.awt.*;
import java.lang.*;
import java.util.*;
import java.net.URL;
import java.net.MalformedURLException;


class StringInfo extends Applet {
    char separated[];  // store string as character array
    int  charWidth[];  // width of each character
    int  length;       // length of string
    int  width;        // width of window to fit the dancing string
    int  height;       // height of window to fit the dancing string
    int  ascent;       // ascent of font


    public void putInfo(String str, Font font, int charSpace, int xOff, int yOff) {

        length = str.length();  // cache string length

        FontMetrics fontMet = getFontMetrics(font);
         
        // Values return from FontMetrics is not very accurate in PC,
        // *1.02 to height & width to make sure it does cover the whole string.
        width  = fontMet.stringWidth(str) + charSpace * length + xOff;
        width  = (int)(width * 1.02);
        height = fontMet.getHeight() + yOff;
        height = (int)(height * 1.02);
        ascent = fontMet.getAscent();

        // store string into character array for easy access
	  separated = new char[length];
        str.getChars(0, length, separated, 0);

        // cache width of each character
        charWidth = new int[length];
	  for (int i = 0; i < length; i++) {
            charWidth[i] = fontMet.charWidth(separated[i]) + charSpace;
        }
    }  // putInfo()

}  // StringInfo



class ColorInfo extends Applet {
    Color baseColor, shadowColor, textColor[];

    // Returns Color object form a string
    // string contain either red,green,blue or hhhhhh
    // no error checking to miminize applet size
    public Color getColor(String str) {
        if (str.indexOf(',') > 0) {
            StringTokenizer t = new StringTokenizer(str, ",");
            int r = Integer.parseInt(t.nextToken());
            int g = Integer.parseInt(t.nextToken());
            int b = Integer.parseInt(t.nextToken());
            return new Color(r, g, b);
        } else
            return new Color(Integer.parseInt(str,16));
    }  // getColor()


    // Calculate the color value for each character
    // store them in textColor[]
    public void calcColor(String s, int length) {
        textColor = new Color[length];  // create the color array

        // Put every color value inputted by the user in the correct position of the array
        int i, j;
        if (s != null) {
            for (StringTokenizer t = new StringTokenizer(s,"|"); t.hasMoreTokens(); ) {
                String str = t.nextToken();
                if ((i = str.indexOf('-')) > 0) {
                    j = Integer.parseInt(str.substring(0, i));
                    if (j < length)
                        textColor[j] = getColor(str.substring(i + 1));
                } else 
                    textColor[0] = getColor(str);  // single color
            }
        }
        else textColor[0] = Color.black;  // user didn't input textColor parameter, use default - black
	  
        // apply linear gradient between known color value
        int c0 = -1, c1 = -1, diff;
        for (i = 0; i < length; i++) {
            if (textColor[i] != null) {
                if (i == 0) c0 = 0;
                else {
                    c1 = i;
                    if (c0 == -1) {  // users didn't specify color value for the first char.
                        c0 = 0;
                        textColor[c0] = textColor[c1];  // copy first known color to front
                    }
                }
            }
            // end of string reach and no color value found there?
            if ((i == (length - 1)) && (textColor[i] == null)) {
                c1 = i;  // same color for last known color to the end of string
                textColor[c1] = textColor[c0];  // copy last known color to end
            }

            if ((c1 != -1) && ((diff = c1 - c0) > 0)) {  // found empty region in color array
                int rStart, gStart, bStart, rDiff, gDiff, bDiff;

                rStart = textColor[c0].getRed();
                gStart = textColor[c0].getGreen();
                bStart = textColor[c0].getBlue();

                // calculate color difference on each step
                rDiff = (int)((textColor[c1].getRed() - rStart) / diff);
                gDiff = (int)((textColor[c1].getGreen() - gStart) / diff);
                bDiff = (int)((textColor[c1].getBlue() - bStart) / diff);

                // generate and store the new color
                for (j = c0 + 1; j < c1; j++) {
                    rStart += rDiff;
                    gStart += gDiff;
                    bStart += bDiff;
                    textColor[j] = new Color(rStart, gStart, bStart);
                }
                c0 = c1;
                c1 = -1;
            }  // if  
        }  // for
    }  // calcColor()

}  // ColorInfo



public class DancingText extends Applet implements Runnable {
    StringInfo strInfo = new StringInfo();
    ColorInfo  color = new ColorInfo();
    Image      offscreen;
    Graphics   offgraphics;
    Thread     myThread = null;
    Font       font;
    boolean    shadow, mouse;
    int        charSpace, maxXOff, maxYOff, shaXOff = 0, shaYOff = 0, speed;
    URL        url;
    Dimension  dim;
    String     target;

    public void init() {
        String s;

        int fontStyle = 0;
        if ((s = getParameter("fontStyle")) != null) {
	      for (StringTokenizer t = new StringTokenizer(s, "|"); t.hasMoreTokens(); ) {
                String str = t.nextToken();
                if (str.equals("PLAIN"))
                    fontStyle += Font.PLAIN;
                else if (str.equals("BOLD"))
                    fontStyle += Font.BOLD;
                else if (str.equals("ITALIC"))
                    fontStyle += Font.ITALIC;
            }
        } else
            fontStyle = Font.PLAIN;  // default

        int fontSize;
        if ((s = getParameter("fontSize")) != null)
            fontSize = Integer.parseInt(s);
        else
            fontSize = 36;  // default

        String fontName;
        if ((s = getParameter("fontName")) != null)
            fontName = s;
        else
            fontName = "TimesRoman";  // default

        font = new Font(fontName, fontStyle, fontSize);

        if ((s = getParameter("charSpace")) != null)
            charSpace = Integer.parseInt(s);
        else
            charSpace = 2;  // default

        if ((s = getParameter("maxXOffset")) != null)
            maxXOff = Integer.parseInt(s);
        else
            maxXOff = 2;  // default

        if ((s = getParameter("maxYOffset")) != null)
            maxYOff = Integer.parseInt(s);
        else
            maxYOff = 4;  // default

        if ((s = getParameter("shadowXOffset")) != null)
            shaXOff = Integer.parseInt(s);
        else
            shaXOff = 3;  // default

        if ((s = getParameter("shadowYOffset")) != null)
            shaYOff = Integer.parseInt(s);
        else
            shaYOff = 3;  // default

        if ((s = getParameter("text")) == null) s = "DancingText";
        strInfo.putInfo(s, font, charSpace, maxXOff + Math.abs(shaXOff), maxYOff + Math.abs(shaYOff));

        resize(strInfo.width, strInfo.height);

        if ((s = getParameter("shadow")) != null)
            shadow = new Boolean(s).booleanValue();
        else
            shadow = true;  // default

        if (shadow && ((s = getParameter("shadowColor")) != null))
            color.shadowColor = color.getColor(s);
        else
            color.shadowColor = new Color(128, 128, 128);  // default

        s = getParameter("textColor");
        color.calcColor(s, strInfo.length);
      
        if ((s = getParameter("baseColor")) != null)
            color.baseColor = color.getColor(s);
        else
            color.baseColor = getBackground();  // default

        if ((s = getParameter("speed")) != null)
            speed = Integer.parseInt(s);
        else
            speed = 100;  // default

        if ((s = getParameter("target")) != null)
            target = s;

        if ((s = getParameter("URL")) != null) {
            try { 
                url = new URL(s);
            } catch (MalformedURLException e) {
                System.out.println("Malformed URL: Check Applet parameter - URL");
            }
        }

        dim = size();

        offscreen = createImage(dim.width, dim.height);
        offgraphics = offscreen.getGraphics();
        offgraphics.setFont(font);
    }  // init()


    public void start() {
        if (myThread == null) {
            myThread = new Thread(this);
            myThread.start();
        }
    }  // start()


    public void stop() {
        myThread.stop();
        myThread = null;
    }  // stop()


    public void run() {
        while (myThread != null) {
            try {
                Thread.sleep(speed);
            } catch (InterruptedException e) {}
            repaint();
        }
    }  // run()


    public void update(Graphics g) {
        offgraphics.setColor(color.baseColor);
        offgraphics.fillRect(0, 0, dim.width, dim.height);

        int x_off = (shaXOff < 0) ? (-shaXOff + 4) : 4;
        int y_off = 1 + strInfo.ascent - ((shaYOff < 0) ? shaYOff : 0);

        for (int i = 0; i < strInfo.length; i++) {
            int x = x_off + (int)(Math.random() * maxXOff);
            int y = y_off + (int)(Math.random() * maxYOff);
            if (shadow) {
                offgraphics.setColor(color.shadowColor);
                offgraphics.drawChars(strInfo.separated, i, 1, x + shaXOff, y + shaYOff);
            }
            offgraphics.setColor(color.textColor[i]);
            offgraphics.drawChars(strInfo.separated, i, 1, x, y);
            x_off += strInfo.charWidth[i];
	  }

	  g.drawImage(offscreen, 0, 0, null);
    }  // update()

 
    public void paint(Graphics g) {
	  g.drawImage(offscreen, 0, 0, null);
    }  // paint()


    public boolean mouseEnter(Event e, int x, int y){
        if (url != null)
            showStatus(url.toString());

        return true;
    }  // mouseEnter()


    public boolean mouseExit(Event e, int x, int y){
        if(url != null)
            showStatus("");

        return true;
    }  // mouseExit()


    public boolean mouseDown(java.awt.Event evt, int x, int y) {
        if (url != null) {
            if (target != null)
                getAppletContext().showDocument(url, target);
            else
                getAppletContext().showDocument(url);

            showStatus("Opening document, please wait.");
        }

        return true;
    }  // mouseDown()


    public String getAppletInfo() {
        return "DancingText 1.0 by Stanley Poon - 12 Feb 1996";
    }

}  // DancingText



// DancingText.java
