//tabs=4
//-----------------------------------------------------------------------------
//  LED.java  -- LED Sign V3.1
//
//  The main for the LED Sign applet. This applet mimics
//  an LED sign that you typically see displaying messages
//  at airport terminals and the such.
//
//  Revisions:
//	V3.1	Converted to package
//			Modified 11-Aug-96 by Robert B. Denny
//
//	V3.0:	Incorporate my changes to 2.5. Add separate border
//			color when current message has an associated URL.
//			Normally in blue (hyperlink), the color can be changed
//			by the hot-bordercolor parameter. Also add new parameter
//			smooth_leds to draw the LEDs in sizes 3 and 4 as real
//			circles. Slower but much prettier. Brighten up the "red"
//			color a bit so it shows better on red-weak displays.
//			the "off" color is a bit dimmer too, it is too bright
//			on some LCD displays. Correct value in 'ht' parameter
//			warning when not set to pixel hight of selected font.
//			Was multiplied by ledsize. leading to huge suggestion.
//			Substantial cleanup of error handling to reduce boolean
//			returns (were ints) and use exceptions. NOTE: I chose
//			to use the generic RuntimeException because I did not
//			want to create another class to be loaded across the net,
//			further slowing the startup of this applet.
//
//	V2.7:	Basically a "performance enhanced" version of 2.5.
//			Loads quicker. Sleep delay "quirk" killed for good.
//			Added frame targets and a more complete error/warning
//			feature.
//			Thanks again to Robert B. Denny (rdenny@dc3.com)
//			for code ideas and input.
//			Modified Mar 14-22, 1996
//
//	V2.5:	Fixed all known bugs in previous versions!	Added
//			the new feature of ledsize, which allows the user
//			to specify in pixels how big the LED's (1-4).
//			Thanks to Robert B. Denny (rdenny@dc3.com) for
//			code and input!
//			Modified Dec 20-26, 1995
//
//	V2.0beta:
//			Modified V1.0 to comply with Pre-Beta java.
//			A problem with delay causes a jerky display.
//			Modified Oct 20 - 29, 1995
//
//	V1.0:	Written July 10 - August 6, 1995
//
//  By Darrick Brown
// 	 dbrown@cs.hope.edu
// 	 http://www.cs.hope.edu/~dbrown/
//
//   Copyright 1995
//-----------------------------------------------------------------------------

package com.dc3.applet.LEDSign;

import java.awt.*;
import java.io.*;
import java.net.*;

// Just a small struct
// used in randomizing the pixels in the "Pixel" function
class Pixelize
{
	int x;
	int y;
}

//-----------------------------------------------------------------------------
// The java.applet.Applet
//-----------------------------------------------------------------------------
public class LED extends java.applet.Applet implements Runnable
{
	// my #defines
	int WIDTH = 400;
	int HEIGHT = 30;
	String appletName = "LED Sign V3.1";

	Script scr; 					// The class that takes care of the script
	FuncInfo fi;					// All the info for any funtion/transition
	Letters let;					// The class that contains all the letters
	int ledsize;					// The size of the LEDs

	Color highlight;				// Color used for highlight on large LEDs
	Color colors[];					// The array of possible colors
	LEDMessage msg;					// The class that takes care of the message to be displayed
	Color bhilite; 					// Normal border hilite color
	Color bcolor;					// Normal border color
	Color bshadow; 					// Normal border shadow color
	Color h_bhilite;				// Border hilite if message has URL
	Color h_bcolor;					// Border color if message has URL
	Color h_bshadow;				// Border shadow if message has URL
	boolean smooth_leds = true;		// true if want real circular LEDs in larger sizes

	Thread led = null;				// Self-thread object
	String scrpt,endspace,fnt;		// "command line" arguments
	String text;					// the current message
	String currurl = appletName; 	// The current url that are set in the script
	URL currURL = null;
	URL lastURL = null;
	String target = new String("");	// Target for frames
	int place;						// The place where we are in each transition.  How we know when we are done.
	int border; 					// The border width
	int offset; 					// The offset for the sign from the upper left
	int w,h; 						// Width & Height in LEDs
	int swidth; 					// The width of the space character.  Settable in the HTML command line.
	boolean beginning = false;		// make sure we init certain stuff only once
	boolean init = false;			// used to make sure "getinfo" is called only once.
	boolean inapplet; 				// Is the mouse cursor in the applet?  (used to display status messages)
	boolean done = false;			// Is the transition done?
	Image pixmapimg,offimg,tmpimg;	// The pixmaps!!  -- These are what make this program possible
	Graphics pixmap,offmap,tmpmap;	// Graphics for the pixmaps
	Pixelize pix[];					// Array of "pixels" used during the Pixel transition

	//--------------------------------------------------------------------------
	// PRIVATE void getAttrs() - Get the command arguments from the HTML
	//--------------------------------------------------------------------------
	private void getAttrs()
			throws RuntimeException 			// Not really needed, but...
	{
		String s;
		int r,g,b;
		Graphics gr;

		if(getParameter("script") !=	null) {
			scrpt = new String(getParameter("script"));
		} else {
			throw(new RuntimeException("No script specified in HTML."));
		}

		if(getParameter("font") !=  null)
		{
			fnt = new String(getParameter("font"));
		} else {
			throw(new RuntimeException("No font specified in HTML."));
		}

		if(getParameter("spacewidth") !=  null)
		{
			swidth = (new Integer(new String(getParameter("spacewidth")))).intValue();
		}
		else
			swidth = 3;

		if(getParameter("ledsize") !=  null)
		{
			ledsize = new Integer(new String(getParameter("ledsize"))).intValue();

			// A little error trapping
			if(ledsize < 1)
				ledsize = 1;
			else if(ledsize > 4)
				ledsize = 4;

			ledsize++;	  // The user enters 1-4, the applet needs 2-5
		}
		else
			ledsize = 4;

		if(getParameter("ht") != null)
		{
			HEIGHT = ledsize*(new Integer(new String(getParameter("ht")))).intValue();
			h = HEIGHT/ledsize;
		}
		else
		{
			System.out.println("LED Sign Warning: parameter \"ht\" not specified");
			HEIGHT = ledsize*9;
			h = 9;
		}

		if(getParameter("wth") != null)
		{
			WIDTH = ledsize*(new Integer(new String(getParameter("wth")))).intValue();
			if(WIDTH/ledsize%2 == 1)
				WIDTH += ledsize;  // It must be even!!!

			w = WIDTH/ledsize;
		}
		else
		{
			System.out.println("LED Sign Warning: parameter \"wth\" not specified");
			WIDTH = 60*ledsize;
			w = 60;
		}

		if(getParameter("border") != null)
		{
			border = new Integer(new String(getParameter("border"))).intValue();
		}
		else
			border = 0;

		if(getParameter("bordercolor") != null)
		{
			// -----------------------------------------------------------
			// I normally _never_ cut and paste code. This and the code
			// for hot_bordercolor probably be pulled out into a method
			// -----------------------------------------------------------
			// User specified border color!!
			s = new String(getParameter("bordercolor"));
			s = s.trim();
			r = new Integer(s.substring(0,s.indexOf(","))).intValue();
			s = s.substring(s.indexOf(",")+1);
			g = new Integer(s.substring(0,s.indexOf(","))).intValue();
			s = s.substring(s.indexOf(",")+1);
			b = new Integer(s).intValue();

			// Forgive the "if" syntax, I didn't want to bother typing the
			// "normal" ifs for this small part. :)
			bhilite = new Color(r+40<256?r+40:255, g+40<256?g+40:255, b+40<256?b+40:255);
			bcolor = new Color(r,g,b);
			bshadow = new Color(r-40>=0?r-40:0, g-40>=0?g-40:0, b-40>=0?b-40:0);
		}
		else
		{
			// The default gray for normal border
			bhilite = Color.white;
			bcolor = Color.lightGray;
			bshadow = Color.darkGray;
		}

		if(getParameter("hot_bordercolor") != null)
		{
			// User specified hot link border color!!
			s = new String(getParameter("hot_bordercolor"));
			s = s.trim();
			r = new Integer(s.substring(0,s.indexOf(","))).intValue();
			s = s.substring(s.indexOf(",")+1);
			g = new Integer(s.substring(0,s.indexOf(","))).intValue();
			s = s.substring(s.indexOf(",")+1);
			b = new Integer(s).intValue();

			// Forgive the "if" syntax, I didn't want to bother typing the
			// "normal" ifs for this small part. :)
			h_bhilite = new Color(r+40<256?r+40:255, g+40<256?g+40:255, b+40<256?b+40:255);
			h_bcolor = new Color(r,g,b);
			h_bshadow = new Color(r-40>=0?r-40:0, g-40>=0?g-40:0, b-40>=0?b-40:0);
		}
		else
		{
			// The default blue for live URL in message
			h_bhilite = new Color(128, 128, 255);
			h_bcolor = new Color(0, 0, 255);
			h_bshadow = new Color(0,0, 128);
		}

		if(getParameter("smooth_leds") != null)
		{
			s = new String(getParameter("smooth_leds"));
			s = s.trim();
			if(s.equals("true")) {
				smooth_leds = true;
			} else {
				smooth_leds = false;
			}
		}
		else
		{
			smooth_leds = true;						// Default to smooth LEDs
		}

	} // end getAttrs()

	//--------------------------------------------------------------------------
	// PUBLIC init() - The initialization entry for the applet
	//
	// Abbreviated. Do the minimum needed to draw the blank screen as soon
	// as possible. After that runInit() is called to complete the startup.
	//--------------------------------------------------------------------------
	public void init()
	{
		boolean ok = true;

		// Set up the different colors for the sign
		highlight = new Color(100,100,100);
		colors = new Color[9];
		colors[0] = new Color(64,64,64);  // off color (dimmer than Darrick's)
		colors[1] = new Color(255,64,64); // Default red (brighter than Darrick's)
		colors[2] = new Color(130,255,0); // green
		colors[3] = new Color(0,130,255); // blue
		colors[4] = new Color(255,255,0); // yellow
		colors[5] = new Color(255,160,0); // orange
		colors[6] = new Color(255,0,255); // purple
		colors[7] = new Color(255,255,255); // white
		colors[8] = new Color(0,255,255); // cyan

		// If the script and/or font are not specified, then stop!
		try {
			getAttrs();
		} catch(RuntimeException e) {
			currurl = new String("LED Sign Error: " + e.getMessage());
			System.out.println(currurl);
			stop();
		}
		offset = 3*border;
		beginning = true;
		init = true;						// tell run() to do 2nd phase init
	}	// End Init


	//--------------------------------------------------------------------------
	// PRIVATE void runInit() - Second-phase initialization
	//
	// This is called from the run procedure.  This is to allow the
	// init procedure to finish as fast as possible, thus allowing
	// it to draw the blank sign to the screen sooner.
	//--------------------------------------------------------------------------
	private void runInit() throws RuntimeException
	{
		Rectangle r1, r2;

		pix = new Pixelize[1];	// load this class now!
		let = new Letters(getDocumentBase(),fnt,swidth);
		if(HEIGHT != let.height()*ledsize)
		{
			System.out.println("LED Sign Warning: parameter \"ht\" should be set to " + let.height() + ".");
		}

		// now that we have the dimensions of the applet, draw it now!
		// This will make the applet *seem* to load faster.
		///////// paint(getGraphics());

		msg = new LEDMessage(h,w,let);
		scr = new Script(getDocumentBase(),scrpt);
		fi = new FuncInfo();
		nextFunc(); 									// Prime the pump

		//
		// Be nice to the user, if resize actually changes the dimensions
		// of the applet, write the new width and height to System.out so
		// the user doesn't have to guess at the currect dimensions.
		//
		r1 = bounds();
		resize(WIDTH+2*(offset),HEIGHT+2*(offset));	// Set the applet size
		r2 = bounds();
		if((r1.width != r2.width) || (r1.height != r2.height))
			System.out.println("Applet resized to w=" + r2.width + " h=" + r2.height);
		init = false;

	}	// End getinfo()

	//--------------------------------------------------------------------------
	// Start the applet running and thread the process
	//--------------------------------------------------------------------------
	public void start()
	{
		if(led == null)
		{
			led = new Thread(this);  // Start the applet running
			led.start();				 // Calls run() in thread context
		}
	}

	//////////////////////////////////////////////////////////////////
	// Stop the thread
	public void stop()
	{
		if(led != null)
		{
			led.stop();
			led = null;
		}
	}

	//--------------------------------------------------------------------------
	// The run loop
	//--------------------------------------------------------------------------
	public void run()
	{
		if(init) {
			try {
				runInit();
			} catch(RuntimeException e) {
				currurl = new String("LED Sign Error: " + e.getMessage());
				System.out.println(currurl);
				showStatus(currurl);
				stop();
			}
		}

		while(led != null)
		{
			repaint();

			try {
				led.sleep(fi.delay);
			} catch (InterruptedException e) { }

			// If we are done with the current transition, then get the
			// next transition (function).
			if(done)
			{
				try {
					nextFunc();
				} catch(RuntimeException e) {
					currurl = "LED Sign error: " + e.getMessage();
					System.out.println(currurl);
					showStatus(currurl);
					stop();
				}

				if(fi == null) {
					currurl = "Script finished";
					System.out.println(currurl);
					showStatus(currurl);
					stop();
				}
				done = false;
			}
		}
	}

	//--------------------------------------------------------------------------
	// The HTML tag parameter information
	//--------------------------------------------------------------------------
	public String[][] getParameterInfo() {
		String[][] info = {
			{"script         ","URL        ", "LED script to use (Required)"},
			{"font           ","URL        ", "Font to use (Required)"},
			{"spacewidth     ","int        ", "Width of space in columns, default=3 )"},
			{"wth            ","int        ", "Width of live display (cols, default=60)"},
			{"ht             ","int        ", "Height of live display (rows, default=9)"},
			{"border         ","int        ", "Width of display border (pix, default=0)"},
			{"bordercolor    ","int,int,int", "Color of border (n,n,n default=lightGray)"},
			{"hot_bordercolor","int,int,int", "Color of hot border (n,n,n default=blue)"},
			{"ledsize        ","int        ", "Diameter of LEDs pixels (1-4), default=3)"},
			{"smooth_leds    ","boolean    ", "True circles for larger size LEDs (default=false)"}
		};
		return info;
	}

	//--------------------------------------------------------------------------
	// The "about" stuff.
	//--------------------------------------------------------------------------
	public String getAppletInfo() {
		return(appletName + " by Darrick Brown and Bob Denny");
	}


	//--------------------------------------------------------------------------
	// Trap for a mouse click on the applet to check to see if they
	// want to go to another page.
	//--------------------------------------------------------------------------
	public boolean mouseDown(java.awt.Event evt, int x, int y)
	{
		if (currURL != null)
		{
			if(target.length() > 0)  {	// They have specified a target
				getAppletContext().showDocument(currURL,target);
			} else {
				getAppletContext().showDocument(currURL);
			}
		}

		return true;
	}

	//--------------------------------------------------------------------------
	// If the mouse cursor enters the applet, then display something
	// in the status bar of the browser.
	//--------------------------------------------------------------------------
	public boolean mouseEnter(java.awt.Event evt, int x, int y)
	{
		inapplet = true;

		showStatus(currurl);

		return true;
	}

	//--------------------------------------------------------------------------
	// If the mouse cursor exits the applet, then clear the status
	// bar.
	//--------------------------------------------------------------------------
	public boolean mouseExit(java.awt.Event evt, int x, int y)
	{
		inapplet = false;

		showStatus(" ");

		return true;
	}

	//--------------------------------------------------------------------------
	// PRIVATE void nextVunc() - Set the next function
	//
	// This function is only called when the previous function/transition has
	// finished. DO NOT DRAW HERE, The Graphics coordinates are off by one in
	// the current (1.01) version of Java!
	//--------------------------------------------------------------------------
	private void nextFunc()
	{
		int i,j;
		Pixelize temp;
		int rand;

		// get the next function
		fi = scr.nextFunc();

		if(fi == null) {			// Prevent null pointer exceptions
			return;
		}

		// Parse the text line to expand any time/date tags
		fi = scr.parseLine(fi);

		// Create the message in LED format (boolean)
		msg.setmsg(fi);

		if(fi.url != null)
		{
			currurl = fi.url.toString();
			currURL = fi.url;
			target = fi.target;
		}
		else
		{
			currurl = appletName;
			currURL = null;
			target = new String("");
		}

		if(inapplet)
		{
			showStatus(currurl);
		}

		// Set up some initial stuff for each of the transitions
		switch(fi.func)
		{
			case 0:
				place = 0;
				break;
			case 1:
				place = 0;
				break;
			case 2:
				place = 0;
				break;
			case 3:
				place = msg.length()-1;
				break;
			case 4:
				place = 0;
				break;
			case 5:
				place = h-1;
				break;
			case 6:
				place = 0;

				// This randomizes the "LEDs" for the
				// Pixel function.

				pix = new Pixelize[w*h];

				for(i=0;i<w;i++)
				{
					for(j=0;j<h;j++)
					{
						pix[h*i+j] = new Pixelize();
						pix[h*i+j].x = i;
						pix[h*i+j].y = j;
					}
				}

				// Randomly sort all the LED's so all we have to do
				// is draw them in "order" and they come out all pixelly
				for(i=0;i<WIDTH/ledsize*h;i++)
				{
						rand = (int)(Math.random()*(double)(WIDTH/ledsize)*(double)h);
						temp = pix[i];
						pix[i] = pix[rand];
						pix[rand] = temp;
				}
				break;
			case 7:
				place = fi.times*2;	// on AND off
				break;
			case 8:
				place = 0;
				break;
			case 9:
				place = 0;
				break;
			case 10:
				place = 0;
				break;
			case 11:
				place = w;
				break;
			case 12:
				place = h-1;
				break;
			case 13:
				place = 0;
				break;
		}
	}

	//--------------------------------------------------------------------------
	// Draw a pretty little LED
	//--------------------------------------------------------------------------
	private void drawLED(int x, int y, boolean on, int col, Graphics gr)
	{
		if(on)
		{
			gr.setColor(colors[col]);
		}
		else	// its off
		{
			gr.setColor(colors[0]);
		}

		switch(ledsize)
		{
			case 2:	  // Just a pixel
				gr.drawLine(x,y,x,y);
				break;

			case 3:	  // A 2x2 rectangle
				gr.fillRect(x,y,2,2);
				break;

			case 4:	 // A 3x3 '+' or a real circle
				if(smooth_leds) {
					gr.fillOval(x, y, 3, 3);
				} else {
					gr.drawLine(x,y+1,x+2,y+1);
					gr.drawLine(x+1,y,x+1,y+2);
				}
				break;

			case 5:	 // The original shape or a real circle
				if(smooth_leds) {
					gr.fillOval(x, y, 4, 4);
				} else {
					gr.fillRect(x+1,y,2,4);
					gr.fillRect(x,y+1,4,2);
				}
				break;
		}

		if(ledsize == 5 && !on && !smooth_leds)
		{
			gr.setColor(highlight);
			gr.drawLine(x+1,y+1,x+1,y+1);  // the cool little highlight
		}
	}

	//--------------------------------------------------------------------------
	// PRIVATE void draw3DRect() - Draw a 3D rect with variable line width
	//--------------------------------------------------------------------------
	private void draw3DRect(Graphics gr, int x, int y, int lx, int ly,
							int width, boolean raised)
	{
		int i;

		for(i=0; i<width; i++)
		{
			Color c1 = (currURL != null) ? h_bhilite : bhilite;
			Color c2 = (currURL != null) ? h_bshadow : bshadow;

			if(raised)
				gr.setColor(c1);
			else
				gr.setColor(c2);

			gr.drawLine(x+i,y+i,lx-i,y+i);
			gr.drawLine(x+i,y+i,x+i,ly-i);

			if(raised)
				gr.setColor(c2);
			else
				gr.setColor(c1);

			gr.drawLine(lx-i,y+i,lx-i,ly-i);
			gr.drawLine(x+i,ly-i,lx-i,ly-i);
		}
	}

	//--------------------------------------------------------------------------
	// PRIVATE void drawWideRect() - Draw a rectangle with variable line width
	//
	// Used by drawFrame to draw the ledge.
	//--------------------------------------------------------------------------
	private void drawWideRect(Graphics gr, int x, int y, int lx, int ly, int width)
	{
		int i;

		gr.setColor((currURL != null) ? h_bcolor : bcolor);
		for(i=0; i<width; i++)
		{
			gr.drawLine(x+i,y+i,lx-i,y+i);
			gr.drawLine(x+i,y+i,x+i,ly-i);
			gr.drawLine(lx-i,y+i,lx-i,ly-i);
			gr.drawLine(x+i,ly-i,lx-i,ly-i);
	  }
	}

	//--------------------------------------------------------------------------
	// PRIVATE void drawFrame() - Draw the frame
	//--------------------------------------------------------------------------
	private void drawFrame(Graphics gr)
	{
		if(border > 0) {
			draw3DRect(gr,0,0,WIDTH+2*offset-1,HEIGHT+2*offset-1,border,true);
			drawWideRect(gr,border,border,WIDTH+5*border-1,HEIGHT+5*border-1, border);
			draw3DRect(gr,2*border,2*border,WIDTH+4*border-1,HEIGHT+4*border-1,border,false);
		}
	}

	//--------------------------------------------------------------------------
	//--------------------------------------------------------------------------
	public void paint(Graphics gr)
	{
		int i,j;
		int p,p2;

		// don't do any of this if the thread is null
		if(led != null)
		{
			drawFrame(gr); 						// First, draw the frame

			// If the applet has just start, set up the pixmaps
			// and draw all the LEDs off
			if(beginning)
			{
				// OK, lets quickly set up the "offimage" (has all LED's turned
				// off) so that we can draw it to the screen quicker when the
				// applet first starts.
				offimg = createImage(WIDTH, HEIGHT);
				offmap = offimg.getGraphics();
				offmap.setColor(Color.black);
				offmap.fillRect(0,0,WIDTH,HEIGHT);

				for(i=0;i<HEIGHT;i+=ledsize)
					for(j=0;j<WIDTH;j+=ledsize)
					{
						drawLED(j,i,false,1,offmap);
					}

				gr.drawImage(offimg,offset,offset, this);

				// Now that we at least have the initial image up, create the other
				// pixmaps we need.
				pixmapimg = createImage(WIDTH, HEIGHT);
				tmpimg = createImage(WIDTH, HEIGHT);

				pixmap = pixmapimg.getGraphics();
				tmpmap = tmpimg.getGraphics();

				pixmap.setColor(Color.black);
				pixmap.fillRect(0,0,WIDTH,HEIGHT);

				for(i=0;i<HEIGHT;i+=ledsize)
					for(j=0;j<WIDTH;j+=ledsize)
					{
						drawLED(j,i,false,1,pixmap);
					}

				beginning = false;
			}
			else
			{
				gr.drawImage(pixmapimg,offset,offset, this);
			}
		}
	}


	//--------------------------------------------------------------------------
	// This procedure contains all the different transitions
	// Each transition does one iteration and returns to the
	// "run" procedure to use its delay.  This also allows
	// the applet to be redrawn (if needed) more quickly.
	//--------------------------------------------------------------------------
	public void update(Graphics gr)
	{
		int i,j;
		int count;

		if(done)
			return;

		// if we have not initialized our applet, don't do anything here.
		// If the script has stopped (fi == null) don't do anything either!
		if( (led != null) && (fi != null) && (pixmap != null) && (offmap != null) && (tmpmap != null))
		{
			//
			// If we went from not having a current URL to having one,
			// or vice-versa, force-draw the border to reflect the
			// presence or lack of a hot-link
			//
			if(((currURL != null) && (lastURL == null)) ||
					((currURL == null) && (lastURL != null)))
			{
				drawFrame(gr);
				lastURL = currURL;
			}

			switch(fi.func)
			{
				case 0:	// Appear
					if(fi.text == null)
					{
						gr.drawImage(offimg,offset,offset, this);  // Turn all the LEDs off
					}
					else
					{
						for(i=0;i<w;i++)
							for(j=0;j<h;j++)
								drawLED(i*ledsize,j*ledsize,msg.getLED(i,j),msg.getColor(i),pixmap);

						gr.drawImage(pixmapimg,offset,offset, this);
					}

					done = true;

					break;

				case 1:	// Sleep
					done = true;  // We don't do anything here

					break;

				case 2:	// ScrollLeft
					pixmap.copyArea(ledsize,0,WIDTH-ledsize,HEIGHT,-ledsize,0);

					for(i=0;i<HEIGHT;i+=ledsize)
						drawLED(WIDTH-ledsize,i,msg.getLED(place,i/ledsize),msg.getColor(place),pixmap);

					gr.drawImage(pixmapimg,offset,offset, this);

					place++;

					if(!msg.inRange(place))
						done = true;

					break;

				case 3:	// ScrollRight
					pixmap.copyArea(0,0,WIDTH-ledsize,HEIGHT,ledsize,0);

					for(i=0;i<HEIGHT;i+=ledsize)
						drawLED(0,i,msg.getLED(place,i/ledsize),msg.getColor(place),pixmap);

					gr.drawImage(pixmapimg,offset,offset, this);

					place--;

					if(place < 0)
						done = true;

					break;

				case 4:	// ScrollUp
					pixmap.copyArea(0,ledsize,WIDTH,HEIGHT-ledsize,0,-ledsize);

					for(i=0;i<WIDTH;i+=ledsize)
						if(msg.inRange(i/ledsize))
							drawLED(i,HEIGHT-ledsize,msg.getLED(i/ledsize,place),msg.getColor(i/ledsize),pixmap);
						else
							drawLED(i,HEIGHT-ledsize,false,1,pixmap);

					gr.drawImage(pixmapimg,offset,offset, this);

					place++;

					if(place >= h)
						done = true;

					break;

				case 5:	// ScrollDown
					pixmap.copyArea(0,0,WIDTH,HEIGHT-ledsize,0,ledsize);

					for(i=0;i<WIDTH;i+=ledsize)
						if(msg.inRange(i/ledsize))
						{
							drawLED(i,0,msg.getLED(i/ledsize,place),msg.getColor(i/ledsize),pixmap);
						}
						else
						{
							drawLED(i,0,false,1,pixmap);
						}

					gr.drawImage(pixmapimg,offset,offset, this);

					place--;

					if(place < 0)
						done = true;

					break;

				case 6: // Pixel
					i = place + fi.times;
					while(place < WIDTH/ledsize*h && place < i)
					{
						if(msg.inRange(pix[place].x))
						{
							drawLED(pix[place].x*ledsize,pix[place].y*ledsize,msg.getLED(pix[place].x,pix[place].y),msg.getColor(pix[place].x),pixmap);
						}
						else
						{
							drawLED(pix[place].x*ledsize,pix[place].y*ledsize,false,1,pixmap);
						}

						place++;
					}
					gr.drawImage(pixmapimg,offset,offset, this);

					if(place >= w*h)
						done = true;

					break;

				case 7:	// Blink
					if(place%2 == 0)
						gr.drawImage(offimg,offset,offset, this);
					else
						gr.drawImage(pixmapimg,offset,offset, this);

					place--;

					if(place == 0)
						done = true;

					break;

				case 8:	// OverRight
					if(msg.inRange(place))
						for(i=0;i<h;i++)
							drawLED(place*ledsize,i*ledsize,msg.getLED(place,i),msg.getColor(place),pixmap);
					else
						for(i=0;i<h;i++)
							drawLED(place*ledsize,i*ledsize,false,1,pixmap);

					gr.drawImage(pixmapimg,offset,offset, this);

					place++;

					if(place >= w)
						done = true;

					break;

				case 9:	// ScrollCenter
					// The right side
					if(w >= place*2)
					{
						pixmap.copyArea(WIDTH/2,0,WIDTH/2-ledsize,HEIGHT,ledsize,0);
						for(i=0;i<h;i++)
							if(msg.inRange(w-place))
								drawLED(WIDTH/2,i*ledsize,msg.getLED(w-place,i),msg.getColor(w-place),pixmap);
							else
								drawLED(WIDTH/2,i*ledsize,false,1,pixmap);
					}

					if(place < w/2)
					{
						pixmap.copyArea(ledsize,0,WIDTH/2-ledsize,HEIGHT,-ledsize,0);
						for(i=0;i<h;i++)
							if(msg.inRange(place))
								drawLED(WIDTH/2-ledsize,i*ledsize,msg.getLED(place,i),msg.getColor(place),pixmap);
							else
								drawLED(WIDTH/2-ledsize,i*ledsize,false,1,pixmap);
					}

					gr.drawImage(pixmapimg,offset,offset, this);

					place++;

					if(place >= w/2 && place*2 > w)
						done = true;

					break;

				case 10:  // OverCenter
					// The right side
					if(w >= place+w/2)
					{
						for(i=0;i<h;i++)
							if(msg.inRange(w/2+place+1))
								drawLED(WIDTH/2+place*ledsize+ledsize,i*ledsize,msg.getLED(w/2+place+1,i),msg.getColor(w/2+place+1),pixmap);
							else
								drawLED(WIDTH/2+place*ledsize+ledsize,i*ledsize,false,1,pixmap);
					}

					if(place < w/2)
					{
						for(i=0;i<h;i++)
							if(msg.inRange(w/2-place))
								drawLED(WIDTH/2-place*ledsize,i*ledsize,msg.getLED(w/2-place,i),msg.getColor(w/2-place),pixmap);
							else
								drawLED(WIDTH/2-place*ledsize,i*ledsize,false,1,pixmap);
					}

					gr.drawImage(pixmapimg,offset,offset, this);

					place++;

					if(w < w/2+place && place >= w/2)
						done = true;

					break;

				case 11:  // OverLeft
					if(msg.inRange(place))
						for(i=0;i<h;i++)
							drawLED(place*ledsize,i*ledsize,msg.getLED(place,i),msg.getColor(place),pixmap);
					else
						for(i=0;i<h;i++)
							drawLED(place*ledsize,i*ledsize,false,1,pixmap);

					gr.drawImage(pixmapimg,offset,offset, this);

					place--;

					if(place == 0)
						done = true;

					break;

				case 12:  // OverUp
					for(i=0;i<w;i++)
					{
						if(msg.inRange(i))
							drawLED(i*ledsize,place*ledsize,msg.getLED(i,place),msg.getColor(i),pixmap);
						else
							drawLED(i*ledsize,place*ledsize,false,1,pixmap);
					}

					gr.drawImage(pixmapimg,offset,offset, this);

					place--;

					if(place < 0)
						done = true;

					break;

				case 13:  // OverDown
					for(i=0;i<w;i++)
					{
						if(msg.inRange(i))
							drawLED(i*ledsize,place*ledsize,msg.getLED(i,place),msg.getColor(i),pixmap);
						else
							drawLED(i*ledsize,place*ledsize,false,1,pixmap);
					}

					gr.drawImage(pixmapimg,offset,offset, this);

					place++;

					if(place >= h)
						done = true;

					break;
			}	// End switch() statement
		}	// End if(led != null)

		return;

	}	// End update()
}	// End LED class
